﻿#include	"AjcInternal.h"

#define	INST_ID			0xBB55C7A2
#define	IS_MY_INST(P)	(P != NULL && P->InstID == INST_ID)

//**************************************************************************************************************//
//																												//
//	高速フーリエ変換																							//
//																												//
//**************************************************************************************************************//

static	BOOL	IsPowerOf2(int n);
static	VO		MakeSinTbl(double *pSinTbl, int n);
static	VO		MakeRevTbl(int    *pRevTbl, int n);

//==============================================================================================================//
//	ＦＦＴ生成																									//
//																												//
//	引  数	：	n				- サンプリングデータの個数（２のｎ乗）											//
//																												//
//	戻り値	：	≠ NULL	：正常（インスタンスハンドル）															//
//				＝ NULL ：エラー																				//
//==============================================================================================================//
AJCEXPORT	HAJCFFT	WINAPI	AjcFftCreate(int n)
{
	HAJCFFT		rc = NULL;
	HAJCFFT		pW = NULL;

	if (IsPowerOf2(n)) {
		if (pW = (HAJCFFT)AJCMEM(sizeof(AJCFFT))) {
			memset(pW, 0, sizeof(AJCFFT));
			pW->InstID = INST_ID;
			pW->pSinTbl = (double *)AJCMEM(sizeof(double) * (n + n / 4));
			pW->pRevTbl = (int *   )AJCMEM(sizeof(int)    * n);
			if (pW->pSinTbl != NULL && pW->pRevTbl != NULL) {
				pW->n = n;
				AjcFftMakeTable(pW->pSinTbl, pW->pRevTbl, n);
				rc = pW;
			}
			else {
				AjcFftDelete(pW);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	ＦＦＴ消去																									//
//																												//
//	引  数	：	n				- サンプリングデータの個数（２のｎ乗）											//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcFftDelete(HAJCFFT pW)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		if (pW->pSinTbl != NULL) free(pW->pSinTbl);
		if (pW->pRevTbl != NULL) free(pW->pRevTbl);
		free(pW);
		rc = TRUE;
	}
	return rc;
}

//==============================================================================================================//
//	ＦＦＴ計算																									//
//																												//
//	引  数	：	pData			- サンプリングデータ（複素数）配列												//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcFftCalc(HAJCFFT pW, PAJCCOMPLEX pData, BOOL fInverse)
{
	BOOL	rc = FALSE;
	int		i, j, k, ik, h, d, k2, n4;
	double	t, s, c, dx, dy;

	if (IS_MY_INST(pW)) {
		n4 = pW->n /4;
		for (i = 0; i < pW->n; i++) {
			j = pW->pRevTbl[i];
			if (i < j) {
				t = pData[i].re; pData[i].re = pData[j].re; pData[j].re = t;
				t = pData[i].im; pData[i].im = pData[j].im; pData[j].im = t;
			}
		}
		for (k = 1; k < pW->n; k = k2) {
			h = 0; k2 = k + k; d = pW->n / k2;
			for (j = 0; j < k; j++) {
				c = pW->pSinTbl[h + n4];
				if (fInverse) s = -pW->pSinTbl[h];
				else          s =  pW->pSinTbl[h];
				for (i = j; i < pW->n; i += k2) {
					ik = i + k;
					dx = s * pData[ik].im + c * pData[ik].re;
					dy = c * pData[ik].im - s * pData[ik].re;
					pData[ik].re = pData[i].re - dx; pData[i].re += dx;
					pData[ik].im = pData[i].im - dy; pData[i].im += dy;
				}
				h += d;
			}
		}
		if (!fInverse) {
			for (i = 0; i < pW->n; i++) {
				pData[i].re /= pW->n;
				pData[i].im /= pW->n;
			}
		}
		rc = TRUE;
	}
	return rc;
}

//==============================================================================================================//
//	ＦＦＴ計算用三角関数テーブル，ビット反転テーブル作成														//
//																												//
//	引  数	：	pSinTbl - 作成する三角関数表  （配列）のアドレス												//
//				pRevTbl - 作成するビット反転表（配列）のアドレス												//
//				n 		- 配列の要素数（サンプリングデータ数，２のｎ乗）										//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcFftMakeTable(double *pSinTbl, int *pRevTbl, int n)
{
	BOOL	rc = FALSE;

	if (pSinTbl != NULL && pRevTbl != NULL && IsPowerOf2(n)) {
		MakeSinTbl(pSinTbl, n);
		MakeRevTbl(pRevTbl, n);
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	ＦＦＴ計算（テーブル指定）																					//
//																												//
//	引  数	：	pData	- サンプリングデータ（複素数）配列														//
//				n 		- 配列の要素数（サンプリングデータ数，２のｎ乗）										//
//				pSinTbl - 作成する三角関数表  （配列）のアドレス												//
//				pRevTbl - 作成するビット反転表（配列）のアドレス												//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcFftCalcByTbl(PAJCCOMPLEX pData, int n, const double *pSinTbl, const int *pRevTbl, BOOL fInverse)
{
	BOOL	rc = FALSE;
	int		i, j, k, ik, h, d, k2, n4;
	double	t, s, c, dx, dy;

	if (pData != NULL && pSinTbl != NULL && pRevTbl != NULL && IsPowerOf2(n)) {
		n4 = n /4;
		for (i = 0; i < n; i++) {
			j = pRevTbl[i];
			if (i < j) {
				t = pData[i].re; pData[i].re = pData[j].re; pData[j].re = t;
				t = pData[i].im; pData[i].im = pData[j].im; pData[j].im = t;
			}
		}
		for (k = 1; k < n; k = k2) {
			h = 0; k2 = k + k; d = n / k2;
			for (j = 0; j < k; j++) {
				c = pSinTbl[h + n4];
				if (fInverse) s = -pSinTbl[h];
				else          s =  pSinTbl[h];
				for (i = j; i < n; i += k2) {
					ik = i + k;
					dx = s * pData[ik].im + c * pData[ik].re;
					dy = c * pData[ik].im - s * pData[ik].re;
					pData[ik].re = pData[i].re - dx; pData[i].re += dx;
					pData[ik].im = pData[i].im - dy; pData[i].im += dy;
				}
				h += d;
			}
		}
		if (!fInverse) {
			for (i = 0; i < n; i++) {
				pData[i].re /= n;
				pData[i].im /= n;
			}
		}
		rc = TRUE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	２のｎ乗チェック																							//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	IsPowerOf2(int n)
{
	int			msk, bn;

	for (msk = 0x40000000, bn = 0; msk >= 0x00000004; msk >>= 1) {
		if (n & msk) bn++;
	}
	return (bn == 1);
}

//--------------------------------------------------------------------------------------------------------------//
//	ＦＦＴ計算用三角関数テーブル作成																			//
//--------------------------------------------------------------------------------------------------------------//
static	VO	MakeSinTbl(double *pSinTbl, int n)
{
	int		i, n2, n4, n8;
	double	c, s, dc, ds, t;

	n2 = n / 2;
	n4 = n / 4;
	n8 = n / 8;
	t = sin(AJC_PAI / n);
	dc = 2 * t * t; ds = sqrt(dc * (2 - dc));
	t = 2 * dc; c = pSinTbl[n4] = 1;  s = pSinTbl[0] = 0;
	for (i = 1; i < n8; i++) {
		c -= dc; dc += t * c;
		s += ds; ds -= t * s;
		pSinTbl[i] = s; pSinTbl[n4 - i] = c;
	}
	if (n8 != 0) {
		pSinTbl[n8] = sqrt(0.5);
	}
	for (i = 0; i < n4; i++) {
		pSinTbl[n2 - i] = pSinTbl[i];
	}
	for (i = 0; i < n2 + n4; i++) {
		pSinTbl[i + n2] = -pSinTbl[i];
	}
}
//--------------------------------------------------------------------------------------------------------------//
//	ＦＦＴ計算用ビット反転テーブル作成																			//
//--------------------------------------------------------------------------------------------------------------//
static	VO	MakeRevTbl(int *pRevTbl, int n)
{
	int		i, j, k, n2;

	n2 = n / 2;
	for (i = j = 0; i < n;) {
		pRevTbl[i++] = j;
		if (i < n) {
			k = n2;
			while (k <= j) {
				j -= k;
				k /= 2;
			}
			j += k;
		}
	}
}

