﻿#include	"AjcInternal.h"

//==============================================================================================================//
//	フォルダ構造のコピー																						//
//																												//
//	引　数：	pPathFrom	- コピー元先頭フォルダパス名のアドレス												//
//				pPathTo		- コピー先先頭フォルダパス名のアドレス												//
//				opt			- オプションフラグ																	//
//				cbp 		- コールバックパラメタ																//
//				cbNotify	- フォルダコピー結果通知用コールバック関数（不要時はNULL)							//
//				cbQuery		- フォルダコピー／名称変更の問い合わせ用コールバック関数（不要時はNULL)				//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：失敗																						//
//==============================================================================================================//
//	インスタンスワーク
typedef struct {
	BOOL		rc;							//	戻り値
	BOOL		fUnicode;					//	ASCII / UNICODE 識別
	UI			opt;						//	オプション
	UX			cbp; 						//	コールバックパラメタ
	union {									//	コールバック関数
		//	ASCII
		struct {
			BOOL (CALLBACK *cbNotify)(C_BCP pPathFrom , C_BCP pPathTo, EAJCCFS ntc, UX cbp);
			BCP  (CALLBACK *cbQuery )(C_BCP pPathFront,   BCP pNewDir,				UX cbp);
		} asc;
		//	UNICODE
		struct {
			BOOL (CALLBACK *cbNotify)(C_WCP pPathFrom , C_WCP pPathTo, EAJCCFS ntc, UX cbp);
			WCP  (CALLBACK *cbQuery )(C_WCP pPathFront,   WCP pNewDir,				UX cbp);
		} uni;
		//	共通
		struct {
			VOP		vpNotify;
			VOP		vpQuery;
		} vp;
	} u;
} CFSWK, *PCFSWK;

//	サブ関数
static	BOOL	SubCopyFolderStructExW	(PCFSWK pW, C_WCP pPathFrom, WCP pPathTo);
static	BOOL	SubMakeFolderW			(PCFSWK pW, C_WCP pPathFrom, WCP pPathTo, WCP pName, BOOL *pfExist);
static	BOOL	SubCopyAttrAndTimeW		(PCFSWK pW, C_UTP pFrom, C_UTP pTo, BOOL fExist);


//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcCopyFolderStructA(C_BCP pPathFrom, C_BCP pPathTo, UI opt)
{
	return AjcCopyFolderStructExA(pPathFrom, pPathTo, opt, 0, NULL, NULL);

}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	BOOL	WINAPI	AjcCopyFolderStructExA(C_BCP pPathFrom, C_BCP pPathTo, UI opt, UX cbp, 
													BOOL (CALLBACK *cbNotify)(C_BCP pPathFrom , C_BCP pPathTo, EAJCCFS ntc, UX cbp),
													BCP  (CALLBACK *cbQuery )(C_BCP pPathFront,   BCP pNewDir,				UX cbp))
{
	BOOL	rc	   = FALSE;
	CFSWK	wrk    = {0};
	BOOL	fExist = FALSE;
	WC		PathFrom[MAX_PATH];
	WC		PathTo	[MAX_PATH];
	WC		tail	[MAX_PATH];
	WC		fname	[_MAX_FNAME];
	WC		fext	[_MAX_EXT];

	if (pPathFrom != NULL && pPathTo != NULL && AjcPathIsDirectoryA(pPathFrom) && AjcPathIsDirectoryA(pPathTo)) {
		//	ワーク初期化
		wrk.rc				= TRUE;
		wrk.fUnicode		= FALSE;
		wrk.opt				= opt;
		wrk.cbp				= cbp;
		wrk.u.asc.cbNotify	= cbNotify;
		wrk.u.asc.cbQuery	= cbQuery;
		//	パスをワイド文字に変換
		MultiByteToWideChar(CP_ACP, 0, pPathFrom, -1, PathFrom, MAX_PATH);
		MultiByteToWideChar(CP_ACP, 0, pPathTo	, -1, PathTo  , MAX_PATH);
		//	コピー先 先頭フォルダ作成
		if (opt & AJCFSC_MAKETOP) {
			//	コピー元の末尾フォルダ名作成
			AjcSnPrintFW(tail, MAX_PATH, L"%s", PathFrom);
			PathRemoveBackslashW(tail);
			_wsplitpath(tail, NULL, NULL, fname, fext);
			_wmakepath (tail, NULL, NULL, fname, fext);
			//	先頭フォルダ作成
			SubMakeFolderW(&wrk, PathFrom, PathTo, tail, &fExist);
			//	先頭フォルダに末尾名を追加
			AjcPathCatW(PathTo, tail, MAX_PATH);
		}
		//	コピー先フォルダパス設定
		else {
			AjcSnPrintFW(PathTo, MAX_PATH, L"%s", PathTo);
		}
		//	フォルダ構造作成
		SubCopyFolderStructExW(&wrk, PathFrom, PathTo);
		if (rc = wrk.rc) {
			//	コピー先 先頭フォルダの属性とタイムスタンプ設定
			if (opt & AJCFSC_MAKETOP) {
				rc = SubCopyAttrAndTimeW(&wrk, PathFrom, PathTo, fExist);
			}
			//	エラー正常設定
			if (rc) {
				SetLastError(ERROR_SUCCESS);
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcCopyFolderStructW(C_WCP pPathFrom, C_WCP pPathTo, UI opt)
{
	return AjcCopyFolderStructExW(pPathFrom, pPathTo, opt, 0, NULL, NULL);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	BOOL	WINAPI	AjcCopyFolderStructExW(C_WCP pPathFrom, C_WCP pPathTo, UI opt, UX cbp, 
												BOOL (CALLBACK *cbNotify)(C_WCP pPathFrom , C_WCP pPathTo, EAJCCFS opt, UX cbp),
												WCP  (CALLBACK *cbQuery )(C_WCP pPathFront,   WCP pNewDir,				UX cbp))
{
	BOOL	rc	   = FALSE;
	CFSWK	wrk    = {0};
	BOOL	fExist = FALSE;
	WC		PathTo[MAX_PATH];
	WC		tail  [MAX_PATH];
	WC		fname [_MAX_FNAME];
	WC		fext  [_MAX_EXT];

	if (pPathFrom != NULL && pPathTo != NULL && AjcPathIsDirectoryW(pPathFrom) && AjcPathIsDirectoryW(pPathTo)) {
		//	ワーク初期化
		wrk.rc				= TRUE;
		wrk.fUnicode		= TRUE;
		wrk.opt				= opt;
		wrk.cbp				= cbp;
		wrk.u.uni.cbNotify	= cbNotify;
		wrk.u.uni.cbQuery	= cbQuery;
		//	コピー先先頭フォルダ作成
		if (opt & AJCFSC_MAKETOP) {
			//	コピー元の末尾フォルダ名作成
			AjcSnPrintFW(tail, MAX_PATH, L"%s", pPathFrom);
			PathRemoveBackslashW(tail);
			_wsplitpath(tail, NULL, NULL, fname, fext);
			_wmakepath (tail, NULL, NULL, fname, fext);
			//	転送先フォルダ名設定
			AjcSnPrintFW(PathTo, MAX_PATH, L"%s", pPathTo);
			//	先頭フォルダ作成
			SubMakeFolderW(&wrk, pPathFrom, PathTo, tail, &fExist);
			//	先頭フォルダに末尾名を追加
			AjcPathCatW(PathTo, tail, MAX_PATH);
		}
		//	コピー先フォルダパス設定
		else {
			AjcSnPrintFW(PathTo, MAX_PATH, L"%s", pPathTo, fExist);
		}
		//	フォルダ構造作成
		SubCopyFolderStructExW(&wrk, pPathFrom, PathTo);
		if (rc = wrk.rc) {
			//	コピー先 先頭フォルダの属性とタイムスタンプ設定
			if (opt & AJCFSC_MAKETOP) {
				rc = SubCopyAttrAndTimeW(&wrk, pPathFrom, PathTo, fExist);
			}
			//	エラー正常設定
			if (rc) {
				SetLastError(ERROR_SUCCESS);
			}
		}
	}
	return rc;
}
//----- 共通サブ -----------------------------------------------------------------------------------------------//
static	BOOL	SubCopyFolderStructExW(PCFSWK pW, C_WCP pPathFrom, WCP pPathTo)
{
	HANDLE				hFind;
	WIN32_FIND_DATAW	fd;
	WC					FromTop [MAX_PATH] = {0};
	WC					FromWild[MAX_PATH] = {0};
	WC					FromFind[MAX_PATH] = {0};
	WC					ToTop	[MAX_PATH] = {0};
	WC					ToFind	[MAX_PATH] = {0};

	if (pW->rc) {
		if (pPathFrom != NULL && pPathTo != NULL && AjcPathIsDirectoryW(pPathFrom)) {
			AjcSnPrintFW(FromTop , MAX_PATH, L"%s", pPathFrom);
			AjcSnPrintFW(FromWild, MAX_PATH, L"%s", pPathFrom);
			AjcPathCatW (FromWild, L"*.*", MAX_PATH);
			AjcSnPrintFW(ToTop	 , MAX_PATH, L"%s", pPathTo	);
			if ((hFind = FindFirstFileW(FromWild, &fd)) != INVALID_HANDLE_VALUE) {
				do {
					//	パスはディレクトリ・・
					if ((fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) &&
								wcscmp(fd.cFileName, L".") != 0 && wcscmp(fd.cFileName, L"..") != 0) {
						// コピー元ディレクトリ名設定
						AjcSnPrintFW(FromFind, MAX_PATH, L"%s", FromTop);
						AjcPathCatW (FromFind, fd.cFileName, MAX_PATH);
						// コピー先ディレクトリ名設定
						AjcSnPrintFW(ToFind, MAX_PATH, L"%s", ToTop);
						//	ディレクトリ作成
						SubMakeFolderW(pW, FromFind, ToFind, fd.cFileName, NULL);
						// 下の階層へ（再帰）
						AjcPathCatW(ToFind, fd.cFileName, MAX_PATH);
						SubCopyFolderStructExW(pW, FromFind, ToFind);
					}
				} while (pW->rc && FindNextFileW(hFind, &fd));
				FindClose(hFind);
			}
		}
	}
	return pW->rc;
}
//	フォルダ作成
static	BOOL	SubMakeFolderW(PCFSWK pW, C_WCP pPathFrom, WCP pPathTo, WCP pName, BOOL *pfExist)
{
	WC		ToFind	[MAX_PATH];
	WC		ToDir	[MAX_PATH];

	if (pW->rc) {
		do {
			//	コピー先ディレクトリ名変更の問い合わせ
			AjcSnPrintFW(ToDir, MAX_PATH, L"%s", pName);
			if (pW->u.vp.vpQuery != NULL) {
				//	コールバック(UNICODE)
				if (pW->fUnicode) {
					WCP		pDir;
					WC		PathTo[MAX_PATH * 2 + 1] = {0};
					UI		stl;
					//	多重文字列で、「コピー先パス」と「コピー元パス」作成
					MAjcStrCpy(PathTo, MAX_PATH, pPathTo);
					stl = (UI)wcslen(PathTo);
					MAjcStrCpy(&PathTo[stl + 1], MAX_PATH, pPathFrom);
					//	コールバック
					if ((pDir = pW->u.uni.cbQuery(PathTo, ToDir, pW->cbp)) != NULL) {
						if (pDir != ToDir) {
							AjcSnPrintFW(ToDir, MAX_PATH, L"%s", pDir);
						}
						//	変更した名称を返す
						AjcSnPrintFW(pName, MAX_PATH, L"%s", ToDir);
					}
					else ToDir[0] = 0;
				}
				//	コールバック(ASCII)
				else {
					BCP		pDir;
					BC		PathTo[MAX_PATH * 2 + 1] = {0};
					BC		to	 [MAX_PATH];
					UI		stl;
					//	バイト文字に変換
					WideCharToMultiByte(CP_ACP, 0, pPathTo, -1, PathTo, MAX_PATH, NULL, NULL);	//	コピー先パス
					WideCharToMultiByte(CP_ACP, 0, ToDir  , -1, to	  , MAX_PATH, NULL, NULL);	//	新フォルダ名
					//	多重文字列で、「コピー先パス」と「コピー元パス」作成
					stl = (UI)strlen(PathTo);
					WideCharToMultiByte(CP_ACP, 0, pPathFrom, -1, &PathTo[stl + 1], MAX_PATH, NULL, NULL);	//	コピー元パス
					//	コールバック
					if ((pDir = pW->u.asc.cbQuery(PathTo, to, pW->cbp)) != NULL) {
						if (pDir == to) MultiByteToWideChar(CP_ACP, 0, to  , -1, ToDir, MAX_PATH);
						else			MultiByteToWideChar(CP_ACP, 0, pDir, -1, ToDir, MAX_PATH);
						//	変更した名称を返す
						AjcSnPrintFW(pName, MAX_PATH, L"%s", ToDir);
					}
					else ToDir[0] = 0;
				}
			}
			//	ディレクトリを作成する場合・・
			if (ToDir[0] != 0) {
				// コピー先ディレクトリ名設定
				AjcSnPrintFW(ToFind, MAX_PATH, L"%s", pPathTo);
				AjcPathCatW (ToFind, ToDir			, MAX_PATH);
				//	パスが存在しない場合、新たなディレクトリ作成
				if (!PathFileExistsW(ToFind)) {
					BOOL rsu;
					//	フォルダ無しの旨、フラグ設定
					if (pfExist != NULL) *pfExist = FALSE;
					// コピー先ディレクトリ作成
					rsu = (AjcCreateDirectoryW(ToFind, TRUE) != 0);
					//	ディレクトリ作成成功ならば、成功通知
					if (rsu) {
						if (pW->u.vp.vpNotify != NULL) {
							//	ディレクトリ作成通知（UNICODE）
							if (pW->fUnicode) {
								if (!pW->u.uni.cbNotify(pPathFrom, ToFind, AJCCFS_SUCCESS, pW->cbp)) {
									pW->rc = FALSE;
									break;
								}
							}
							//	ディレクトリ作成通知（ASCII）
							else {
								BC	from[MAX_PATH];
								BC	to	[MAX_PATH];
								WideCharToMultiByte(CP_ACP, 0, pPathFrom, -1, from, MAX_PATH, NULL, NULL);
								WideCharToMultiByte(CP_ACP, 0, ToFind	, -1, to  , MAX_PATH, NULL, NULL);
								if (!pW->u.asc.cbNotify(from, to, AJCCFS_SUCCESS, pW->cbp)) {
									pW->rc	= FALSE;
									break;
								}
							}
						}
						//	ディレクトリの属性とタイムスタンプをコピー元に合わせる
						pW->rc = SubCopyAttrAndTimeW(pW, pPathFrom, ToFind, FALSE);
					}
					//	ディレクトリ作成失敗ならば、失敗通知
					else {
						//	失敗通知
						if (pW->u.vp.vpNotify != NULL) {
							//	失敗通知（UNICODE）
							if (pW->fUnicode) {
								pW->u.uni.cbNotify(pPathFrom, ToFind, AJCCFS_SUCCESS, pW->cbp);
							}
							//	失敗通知（ASCII）
							else {
								BC	from[MAX_PATH];
								BC	to	[MAX_PATH];
								WideCharToMultiByte(CP_ACP, 0, pPathFrom, -1, from, MAX_PATH, NULL, NULL);
								WideCharToMultiByte(CP_ACP, 0, ToFind	, -1, to  , MAX_PATH, NULL, NULL);
								pW->u.asc.cbNotify(from, to, AJCCFS_SUCCESS, pW->cbp);
							}
						}
						//	戻り値＝エラー
						pW->rc = FALSE;
					}
				}
				//	既にパスが存在する場合・・
				else {
					//	フォルダ既存の旨、フラグ設定
					if (pfExist != NULL) *pfExist = TRUE;
					//	同一フォルダがあってもエラーとしない場合
					if (pW->opt & AJCFSC_ALLOWEXIST) {
						//	「同一パスあり」を通知
						if (pW->u.vp.vpNotify != NULL) {
							if (pW->fUnicode) {
								if (!pW->u.uni.cbNotify(pPathFrom, ToFind, AJCCFS_EXIST, pW->cbp)) {
									pW->rc	= FALSE;
									break;
								}
							}
							else {
								BC	from[MAX_PATH];
								BC	to	[MAX_PATH];
								WideCharToMultiByte(CP_ACP, 0, pPathFrom, -1, from, MAX_PATH, NULL, NULL);
								WideCharToMultiByte(CP_ACP, 0, ToFind	, -1, to  , MAX_PATH, NULL, NULL);
								if (!pW->u.asc.cbNotify(from, to, AJCCFS_EXIST, pW->cbp)) {
									pW->rc	= FALSE;
									break;
								}
							}
						}
						//	パスがフォルダならば、タイムスタンプ設定
						if (AjcPathIsDirectory(ToFind)) {
							//	ディレクトリの属性とタイムスタンプをコピー元に合わせる
							pW->rc = SubCopyAttrAndTimeW(pW, pPathFrom, ToFind, TRUE);
						}
						//	パスがフォルダ以外ならばエラー
						else {
							pW->rc = FALSE;
						}
					}
					//	同一フォルダがあるとエラーとする場合
					else {
						pW->rc = FALSE;
					}
				}
			}
		} while(0);
	}
	return pW->rc;
}
//	フォルダの属性とタイムスタンプコピー
static	BOOL	SubCopyAttrAndTimeW(PCFSWK pW, C_UTP pFrom, C_UTP pTo, BOOL fExist)
{
	BOOL		rc = TRUE;
	AJCFTIMES	ft = {0};

	//	タイムスタンプコピー
	if ((pW->opt & AJCFSC_FORCETIME) || (!fExist && (pW->opt & AJCFSC_SAMETIME))) {
		if (AjcGetFileTimeW(pFrom, &ft, NULL, FALSE)) {
			AjcSetFileTimeW(pTo, &ft, FALSE);	//	フォルダのタイムスタンプコピー結果は無視する
		}
	}
	//	属性コピー
	if ((pW->opt & AJCFSC_FORCEATTR) || (!fExist && (pW->opt & AJCFSC_SAMEATTR))) {
		UI	att;
		if ((att = GetFileAttributesW(pFrom)) != 0) {
			rc &= (SetFileAttributesW(pTo, att) != 0);
		}
	}
	return rc;
}
//==============================================================================================================//
//	フォルダ内ファイル群コピー																					//
//																												//
//	引　数：	pPathFrom	  - コピー元頭フォルダパス名＋ワイルドカードのアドレス								//
//				pPathTo		  - コピー先頭フォルダパス名のアドレス												//
//				pWildCard	  - コピーするファイルのワイルドカード（NULL指定時は、"*.*"を仮定）					//
//				opt			  - オプション(AJCCPF_XXXXX)														//
//				cbp			  - コールバックパラメタ															//
//				cbNotify	  - ファイルコピー結果通知用コールバック関数（不要時はNULL)							//
//				cbQuery		  - ファイルコピー／名称変更の問い合わせ用コールバック関数（不要時はNULL)			//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：失敗																						//
//==============================================================================================================//
//	ワーク
typedef struct {
	UX		cbp;
	BOOL (CALLBACK *cbNotify)(C_BCP pFileFrom, C_BCP pFileTo  , EAJCCFS ntc , UX cbp);
	BCP  (CALLBACK *cbQuery )(C_BCP pFileFrom, C_BCP pFileTo, BCP pFileName, UIP pAtt, UX cbp);
} CPFSWRK, *PCPFSWRK;

static	BOOL CALLBACK cbCpfsNotify(C_WCP pFileFrom, C_WCP pFileTo  , EAJCCFS ntc , UX cbp);
static	WCP  CALLBACK cbCpfsQuery (C_WCP pFileFrom, C_WCP pFileTo, WCP pFileName, UIP pAtt, UX cbp);

//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI AjcCopyFilesA(C_BCP pPathFrom, C_BCP pPathTo, C_BCP pWildCard, UI opt)
{
	return AjcCopyFilesExA(pPathFrom, pPathTo, pWildCard, opt, 0, NULL, NULL, NULL);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	BOOL	WINAPI AjcCopyFilesExA(C_BCP pPathFrom, C_BCP pPathTo, C_BCP pWildCard, UI opt, UX cbp,
											BOOL (CALLBACK *cbNotify)(C_BCP pFileFrom, C_BCP pFileTo  , EAJCCFS ntc , UX cbp),
											BCP  (CALLBACK *cbQuery )(C_BCP pFileFrom, C_BCP pFileTo, BCP pFileName, UIP pAtt, UX cbp),
											BOOL (CALLBACK *cbProgress)(ULL FileSize, ULL Copied, UX cbp))
{
	BOOL				rc	 = FALSE;
	CPFSWRK				wrk  = {0};
	BOOL				fCan = FALSE;
	WC					PathFrom[MAX_PATH];
	WC					PathTo 	[MAX_PATH];
	WC					WildCard[MAX_PATH];

	if (pPathFrom != NULL && pPathTo != NULL) {
		//	ワーク初期化
		wrk.cbp			= cbp;
		wrk.cbNotify	= cbNotify;
		wrk.cbQuery		= cbQuery;
		//	ワイド文字化
		MultiByteToWideChar(CP_ACP, 0, pPathFrom, -1, PathFrom, MAX_PATH);
		MultiByteToWideChar(CP_ACP, 0, pPathTo	, -1, PathTo  , MAX_PATH);
		//	ワイルドカード指定あり
		if (pWildCard != NULL) {
			MultiByteToWideChar(CP_ACP, 0, pWildCard, -1, WildCard, MAX_PATH);
			//	ファイル群コピー
			cbp = (UX)&wrk;
			rc = AjcCopyFilesExW(PathFrom, PathTo, WildCard, opt, cbp, cbCpfsNotify, cbCpfsQuery, cbProgress);
		}
		//	ワイルドカード指定なし
		else {
			//	ファイル群コピー
			cbp = (UX)&wrk;
			rc = AjcCopyFilesExW(PathFrom, PathTo, NULL    , opt, cbp, cbCpfsNotify, cbCpfsQuery, cbProgress);
		}
	}
	return rc;
}
//- - - コールバック(cbNotify) - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -//
static	BOOL CALLBACK cbCpfsNotify(C_WCP pFileFrom, C_WCP pFileTo  , EAJCCFS ntc , UX cbp)
{
	BOOL		rc;
	PCPFSWRK	p = (PCPFSWRK)cbp;
	BC			FileFrom[MAX_PATH];
	BC			FileTo	[MAX_PATH];

	if (p->cbNotify != NULL) {
		WideCharToMultiByte(CP_ACP, 0, pFileFrom, -1, FileFrom, MAX_PATH, NULL, NULL);
		WideCharToMultiByte(CP_ACP, 0, pFileTo	, -1, FileTo  , MAX_PATH, NULL, NULL);
		rc = p->cbNotify(FileFrom, FileTo, ntc, p->cbp);
	}
	return rc;
}
//- - - コールバック(cbQueryy) - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -//
static	WCP CALLBACK cbCpfsQuery(C_WCP pFileFrom, C_WCP pFileTo, WCP pFileName, UIP pAtt, UX cbp)
{
	WCP			rc = pFileName;
	BCP			ret;
	PCPFSWRK	p = (PCPFSWRK)cbp;
	BC			FileFrom[MAX_PATH];
	BC			FileTo	[MAX_PATH];
	BC			FileName[MAX_PATH];

	if (p->cbQuery != NULL) {
		WideCharToMultiByte(CP_ACP, 0, pFileFrom, -1, FileFrom, MAX_PATH, NULL, NULL);
		WideCharToMultiByte(CP_ACP, 0, pFileTo	, -1, FileTo  , MAX_PATH, NULL, NULL);
		WideCharToMultiByte(CP_ACP, 0, pFileName, -1, FileName, MAX_PATH, NULL, NULL);
		if (ret = p->cbQuery(FileFrom, FileTo, FileName, pAtt, p->cbp)) {
			MultiByteToWideChar(CP_ACP, 0, FileName, -1, pFileName, MAX_PATH);
		}
		else rc = NULL;
	}
	return rc;
}

//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI AjcCopyFilesExW	(C_WCP pPathFrom, C_WCP pPathTo, C_WCP pWildCard, UI opt, UX cbp,
								BOOL (CALLBACK *cbNotify)(C_WCP pFileFrom, C_WCP pFileTo  , EAJCCFS ntc , UX cbp),
								WCP  (CALLBACK *cbQuery )(C_WCP pFileFrom, C_WCP pFileTo, WCP pFileName, UIP pAtt, UX cbp),
								BOOL (CALLBACK *cbProgress)(ULL FileSize, ULL Copied, UX cbp))
{
	BOOL				rc	 = FALSE;
	BOOL				fCan = FALSE;
	HANDLE				hFind;
	WIN32_FIND_DATAW	fd;
	WC					WildPath[MAX_PATH];		//	パス＋ワイルドカード
	WC					src 	[MAX_PATH];		//	コピー元ファイルパス
	WC					dst 	[MAX_PATH];		//	コピー先ファイルパス
	WC					fname	[MAX_PATH];		//	コピー先ファイル名


	if (pPathFrom != NULL && pPathTo != NULL) {
		rc = TRUE;
		AjcSnPrintFW(WildPath, MAX_PATH, L"%s", pPathFrom);
		AjcPathCatW(WildPath, L"*.*"   , MAX_PATH);
		if ((hFind = FindFirstFileW(WildPath, &fd)) != INVALID_HANDLE_VALUE) {
			do {
				// ファイルならばコピー
				if (!(fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
					//	ワイルドカードチェック
					if (pWildCard == NULL ||
						(!(opt & AJCCPF_WILDEXC) &&  AjcPathMatchSpecW(fd.cFileName, pWildCard)) ||
						( (opt & AJCCPF_WILDEXC) && !AjcPathMatchSpecW(fd.cFileName, pWildCard))) {
						UI	attSrc, attDst;
						//	コピー元ファイルパス設定
						AjcSnPrintFW(src, MAX_PATH, L"%s", pPathFrom);	AjcPathCatW(src, fd.cFileName, MAX_PATH);
						//	コピー先ファイルパス設定
						AjcSnPrintFW(dst, MAX_PATH, L"%s", pPathTo	);	AjcPathCatW(dst, fd.cFileName, MAX_PATH);
						//	コピー元ファイル属性設定
						attSrc = GetFileAttributesW(src);
						attDst = attSrc;
						//	コピー先ファイル名設定
						AjcSnPrintFW(fname, MAX_PATH, L"%s", fd.cFileName);
						//	ファイルコピー可否／ファイル名変更問い合わせ
						if (cbQuery != NULL) {
							WCP		pFile;
							if ((pFile = cbQuery(src, dst, fname, &attDst, cbp)) != NULL) {
								if (pFile != fname) {
									AjcSnPrintFW(fname, MAX_PATH, L"%s", pFile);
								}
							}
							else fname[0] = 0;
						}
						if (fname[0] != 0) {
							//	コピー先ファイルパス再設定
							AjcSnPrintFW(dst, MAX_PATH, L"%s", pPathTo);
							AjcPathCatW (dst, fname, MAX_PATH);
							//	ファイル上書きあり／コピー先ファイル無しならばコピー実行
							if ((opt & AJCCPF_CREATEALWAYS) || !AjcPathExistsW(dst)) {
								BOOL	rsu;
								UI		attTmp;
								//	ファイル上書きならば、特殊なファイル属性を解除
								if (opt & AJCCPF_CREATEALWAYS) {
									if ((attTmp = GetFileAttributesW(dst)) != INVALID_FILE_ATTRIBUTES) {
										SetFileAttributesW(dst, attTmp & ~(FILE_ATTRIBUTE_HIDDEN | FILE_ATTRIBUTE_READONLY | FILE_ATTRIBUTE_SYSTEM));
									}
								}
								//	ファイルコピー
								if (rsu = AjcCopyFileExW(src, dst, opt, &fCan, cbp, cbProgress)) {
									//	ファイル属性設定
									if (attSrc != INVALID_FILE_ATTRIBUTES && attSrc != attDst) {
										SetFileAttributesW(dst, attDst);
									}
								}
								//	ファイルコピーキャンセルならば終了
								if (fCan) {
									rc = FALSE;
									break;
								}
								//	結果通知
								if (rsu) {
									if (cbNotify != NULL) {
										if (!cbNotify(src, dst, AJCCFS_SUCCESS, cbp)) break;
									}
								}
								else {
									rc = FALSE;
									if (cbNotify != NULL) {
										if (!cbNotify(src, dst, AJCCFS_FAILURE, cbp)) break;
									}
								}
							}
							//	ファイル上書きなし＆コピー先ファイル有りならばコピーしない旨を通知
							else {
								if (cbNotify != NULL) {
									if (!cbNotify(src, dst, AJCCFS_EXIST, cbp)) break;
								}
							}
						}
					}
				}
			} while (FindNextFileW(hFind, &fd));
			FindClose(hFind);
		}
	}
	return rc;
}
//==============================================================================================================//
//	ファイルコピー																								//
//																												//
//	引　数	：	pFrom			- コピー元ファイルパス名へのポインタ											//
//				pTo				- コピー先ファイルパス名へのポインタ											//
//				opt				- オプション																	//
//				pfCancel		- 中止を示すフラグを格納するバッファポインタ（不要時はNULL）					//
//				cbp				- コールバックパラメタ															//
//				cbProgress		- ファイルコピー状況通知用コールバック関数										//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - 失敗																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcCopyFileA(C_BCP pFrom, C_BCP pTo, UI opt, UX cbp, BOOL (CALLBACK *cbProgress)(ULL FileSize, ULL Copied, UX cbp))
{
	return	AjcCopyFileExA(pFrom,pTo, opt, NULL, cbp, cbProgress);
}
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcCopyFileExA(C_BCP pFrom, C_BCP pTo, UI opt, BOOL *pfCancel, UX cbp, BOOL (CALLBACK *cbProgress)(ULL FileSize, ULL Copied, UX cbp))
{
	BOOL	rc = FALSE;
	WC		PathFrom[MAX_PATH] = {0};
	WC		PathTo	[MAX_PATH] = {0};

	if (pFrom != NULL && pTo != NULL) {
		AjcByteCharToWideChar(pFrom, PathFrom, MAX_PATH);
		AjcByteCharToWideChar(pTo  , PathTo  , MAX_PATH);
		rc = AjcCopyFileW(PathFrom, PathTo, opt, cbp, cbProgress);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcCopyFileW(C_WCP pFrom, C_WCP pTo, UI opt, UX cbp, BOOL (CALLBACK *cbProgress)(ULL FileSize, ULL Copied, UX cbp))
{
	return	AjcCopyFileExW(pFrom,pTo, opt, NULL, cbp, cbProgress);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	BOOL	WINAPI	AjcCopyFileExW(C_WCP pFrom, C_WCP pTo, UI opt, BOOL *pfCancel, UX cbp, BOOL (CALLBACK *cbProgress)(ULL FileSize, ULL Copied, UX cbp))
{
	BOOL		rc			= FALSE;
	HANDLE		hInp		= INVALID_HANDLE_VALUE;
	HANDLE		hOut		= INVALID_HANDLE_VALUE;
	UI			inp_share	= 0;
	UI			out_share	= 0;
	UL			att			= 0;
	UL			RBytes		= 0;
	UL			WBytes		= 0;
	ULL			size		= 0;
	UL			count		= 0;
	FILETIME	ftCreate	= {0};
	FILETIME	ftLastAcc	= {0};
	FILETIME	ftLastWrt	= {0};
	WC			pathTo[MAX_PATH] = {0};
	UB			buf[4096];

	if (pFrom != NULL && pTo != NULL) {
		do {
			//	キャンセルフラグクリアー
			if (pfCancel != NULL) {
				*pfCancel = FALSE;
			}
			//	共有モード設定
			if (opt & AJCCPF_INP_SHARE_READ ) inp_share |= FILE_SHARE_READ;
			if (opt & AJCCPF_INP_SHARE_WRITE) inp_share |= FILE_SHARE_WRITE;
			if (opt & AJCCPF_OUT_SHARE_READ ) out_share |= FILE_SHARE_READ;
			if (opt & AJCCPF_OUT_SHARE_WRITE) out_share |= FILE_SHARE_WRITE;
			//	コピー元ファイル存在チェック
			if (!AjcPathIsFile(pFrom  )) {
				SetLastError(ERROR_FILE_NOT_FOUND);
				break;
			}
			//	コピー先パス名がディレクトリならば、ファイル名を付加
			MAjcStrCpy(pathTo, MAX_PATH, pTo);
			if (AjcPathIsDirectory(pTo)) {
				UT	fname	[_MAX_FNAME];
				UT	fext	[_MAX_EXT];
				UT	FileName[MAX_PATH];
				MAjcSplitPath(pFrom   , NULL, NULL, fname, fext);
				MAjcMakePath (FileName, NULL, NULL, fname, fext);
				AjcPathCat(pathTo, FileName, MAX_PATH);
			}
			//	コピー先ファイル存在チェック
			if (AjcPathIsFile(pathTo )) {
				//	「既に存在する場合上書き」ならばファイル削除
				if (opt & AJCCPF_CREATEALWAYS) {
					SetFileAttributesW(pathTo, FILE_ATTRIBUTE_NORMAL);
					DeleteFile(pathTo);
				}
				//	「既に存在する場合コピーしない」ならば、終了
				else {
					SetLastError(ERROR_ALREADY_EXISTS);
					break;
				}
			}
			//	ファイルサイズ取得
			if ((size = AjcGetFileSize(pFrom)) == -1) break;
			//	ファイルオープン
			if ((hInp = CreateFileW(pFrom , GENERIC_READ , inp_share, NULL, OPEN_EXISTING, 0, NULL)) == INVALID_HANDLE_VALUE) break;
			if ((hOut = CreateFileW(pathTo, GENERIC_WRITE, out_share, NULL, CREATE_ALWAYS, 0, NULL)) == INVALID_HANDLE_VALUE) break;
			//	ファイルコピー
			while (ReadFile(hInp, buf, sizeof buf, &RBytes, NULL) && RBytes != 0) {
				//	ファイル書き込み
				if (WriteFile(hOut, buf, RBytes, &WBytes, NULL) == 0 || RBytes != WBytes) {
					goto cf_exit;
				}
				count += WBytes;
				//	コールバック
				if (cbProgress != NULL) {
					if (!cbProgress(size, count, cbp)) {
						if (pfCancel != NULL) {
							*pfCancel = TRUE;
						}
						goto cf_exit;
					}
				}
			}
			//	タイムスタンプ設定
			if (!GetFileTime(hInp, &ftCreate, &ftLastAcc, &ftLastWrt)) break;
			if (!SetFileTime(hOut, &ftCreate, &ftLastAcc, &ftLastWrt)) break;
			//	ファイル属性設定
			if ((att  = GetFileAttributesW(pFrom)) == INVALID_FILE_ATTRIBUTES) break;
			if (SetFileAttributesW(pathTo, att	 ) == 0 					 ) break;
			//	戻り値＝ＯＫ
			SetLastError(ERROR_SUCCESS);
			rc = TRUE;
		} while(0);
	}
cf_exit:;
	//	ファイルクローズ
	if (hInp != INVALID_HANDLE_VALUE) CloseHandle(hInp);
	if (hOut != INVALID_HANDLE_VALUE) CloseHandle(hOut);

	return rc;
}
//==============================================================================================================//
//	フォルダ削除																								//
//																												//
//	引　数：	pPath		- 削除するフォルダパス名のアドレス													//
//				cbp 		- コールバックパラメタ																//
//				cbNotify	- 削除したフォルダ／ファイル名を通知するコールバック関数（不要時はNULL)				//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：失敗																						//
//==============================================================================================================//
//	ワーク
typedef struct {
	UX		cbp;
	BOOL (CALLBACK *cbNotify)(C_BCP pPathRemoved, UI ntc, UX cbp);
} RMVFWRK, *PRMVFWRK;

static	BOOL CALLBACK cbRmvfNotify(C_WCP pPathRemoved, UI ntc, UX cbp);

//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcRemoveFolderA(C_BCP pPath, UX cbp, BOOL (CALLBACK *cbNotify)(C_BCP pPathRemoved, UI ntc, UX cbp))
{
	BOOL	rc	 = FALSE;
	RMVFWRK	wrk;
	WC		Path[MAX_PATH];

	if (pPath != NULL && AjcPathIsDirectoryA(pPath)) {
		wrk.cbp 	 = cbp;
		wrk.cbNotify = cbNotify;
		MultiByteToWideChar(CP_ACP, 0, pPath, -1, Path, MAX_PATH);
		rc = AjcRemoveFolderW(Path, (UX)&wrk, cbRmvfNotify);
	}
	return rc;
}
//- - - - コールバック(cbNotify) - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	BOOL CALLBACK cbRmvfNotify(C_WCP pPathRemoved, UI ntc, UX cbp)
{
	BOOL		rc;
	PRMVFWRK	p = (PRMVFWRK)cbp;
	BC			PathRemoved[MAX_PATH];

	if (p->cbNotify != NULL) {
		WideCharToMultiByte(CP_ACP, 0, pPathRemoved, -1, PathRemoved, MAX_PATH, NULL, NULL);
		rc = p->cbNotify(PathRemoved, ntc, p->cbp);
	}
	return rc;
}

//----- ワイド文字 ---------------------------------------------------------------------------------------------//
static	VO			SubRemoveFolderW(C_WCP pPath, BOOL *rc, BOOL *fEnd, UX cbp, BOOL (CALLBACK *cbNotify)(C_WCP pPathRemoved, UI ntc, UX cbp));

AJCEXPORT	BOOL	WINAPI	AjcRemoveFolderW(C_WCP pPath, UX cbp, BOOL (CALLBACK *cbNotify)(C_WCP pPathRemoved, UI ntc, UX cbp))
{
	BOOL	rc	 = FALSE;
	BOOL	fEnd = FALSE;

	if (pPath != NULL && AjcPathIsDirectoryW(pPath)) {
		rc = TRUE;
		SubRemoveFolderW(pPath, &rc, &fEnd, cbp, cbNotify);
	}
	return rc;
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	VO			SubRemoveFolderW(C_WCP pPath, BOOL *rc, BOOL *fEnd, UX cbp, BOOL (CALLBACK *cbNotify)(C_WCP pPathRemoved, UI ntc, UX cbp))
{
	HANDLE				hFind;
	WIN32_FIND_DATAW	fd;
	int					att;
	BOOL				rsu;
	WC					PathTop [MAX_PATH];
	WC					PathWild[MAX_PATH];
	WC					PathFind[MAX_PATH];

	if (!(*fEnd)) {
		AjcSnPrintFW(PathTop , MAX_PATH, L"%s", pPath);
		AjcSnPrintFW(PathWild, MAX_PATH, L"%s", pPath);
		AjcPathCatW(PathWild, L"*.*", MAX_PATH);
		if ((hFind = FindFirstFileW(PathWild, &fd)) != INVALID_HANDLE_VALUE) {
			do {
				if(wcscmp(fd.cFileName, L".") != 0 && wcscmp(fd.cFileName, L"..") != 0) {
					AjcSnPrintFW(PathFind, MAX_PATH, L"%s", PathTop);
					AjcPathCatW (PathFind, fd.cFileName, MAX_PATH);
					//	特殊な属性を解除
					if ((att = GetFileAttributesW(PathFind)) != -1) {
						SetFileAttributesW(PathFind, att & ~(FILE_ATTRIBUTE_HIDDEN | FILE_ATTRIBUTE_READONLY | FILE_ATTRIBUTE_SYSTEM));
					}
					// ディレクトリ
					if (fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
						//	下位の階層へ
						SubRemoveFolderW(PathFind, rc, fEnd, cbp, cbNotify);
					}
					//	ファイル
					else {
						//	ファイル削除
						SetFileAttributesW(PathFind, FILE_ATTRIBUTE_NORMAL);
						rsu = DeleteFileW(PathFind);
						//	コールバック
						if (cbNotify != NULL) {
							*fEnd = !cbNotify(PathFind, rsu ? AJCRMV_FILE_OK : AJCRMV_FILE_ERR, cbp);
						}
						//	戻り値設定(TRUE:OK, FALSE:ERR)
						*rc &= (rsu != 0);
					}
				}
			} while (!(*fEnd) && FindNextFileW(hFind, &fd));
			FindClose(hFind);
		}
		//	ディレクトリ削除
		SetFileAttributesW(pPath, FILE_ATTRIBUTE_DIRECTORY);
		rsu = RemoveDirectoryW(pPath);
		//	コールバック
		if (cbNotify != NULL) {
			*fEnd = !cbNotify(pPath, rsu ? AJCRMV_DIR_OK : AJCRMV_DIR_ERR, cbp);
		}
		//	戻り値設定(TRUE:OK, FALSE:ERR)
		*rc &= (rsu != 0);
	}
}

//==============================================================================================================//
//	フォルダ下の全ファイルとディレクトリ削除																	//
//																												//
//	引　数：	pPath		- 削除するフォルダパス名のアドレス													//
//				cbNotify	- 削除したフォルダ／ファイル名を通知するコールバック関数（不要時はNULL)				//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：失敗																						//
//==============================================================================================================//
//	ワーク
typedef struct {
	UX		cbp;
	BOOL (CALLBACK *cbNotify)(C_BCP pPathRemoved, UI ntc, UX cbp);
} CLNFWRK, *PCLNFWRK;

static	BOOL CALLBACK cbClnfNotify(C_WCP path, UI ntc, UX cbp);

//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcCleanFolderA(C_BCP pPath, UX cbp, BOOL (CALLBACK *cbNotify)(C_BCP path, UI ntc, UX cbp))
{
	BOOL	rc	 = FALSE;
	CLNFWRK	wrk;
	WC		Path[MAX_PATH];

	if (pPath != NULL && AjcPathIsDirectoryA(pPath)) {
		wrk.cbp 	 = cbp;
		wrk.cbNotify = cbNotify;
		MultiByteToWideChar(CP_ACP, 0, pPath, -1, Path, MAX_PATH);
		rc = AjcCleanFolderW(Path, (UX)&wrk, cbClnfNotify);
	}
	return rc;
}
//- - - - コールバック(cbNotify) - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -//
static	BOOL CALLBACK cbClnfNotify(C_WCP pPathRemoved, UI ntc, UX cbp)
{
	BOOL		rc = TRUE;
	PCLNFWRK	p = (PCLNFWRK)cbp;
	BC			PathRemoved[MAX_PATH];

	if (p->cbNotify != NULL) {
		WideCharToMultiByte(CP_ACP, 0, pPathRemoved, -1, PathRemoved, MAX_PATH, NULL, NULL);
		rc = p->cbNotify(PathRemoved, ntc, p->cbp);
	}
	return rc;
}

//----- ワイド文字 ---------------------------------------------------------------------------------------------//
static		VO		SubCleanFolderW(C_WCP pPath, BOOL *rc, BOOL *fEnd, UX cbp, BOOL (CALLBACK *cbNotify)(C_WCP path, UI ntc, UX cbp));

AJCEXPORT	BOOL	WINAPI	AjcCleanFolderW(C_WCP pPath, UX cbp, BOOL (CALLBACK *cbNotify)(C_WCP path, UI ntc, UX cbp))
{
	BOOL	rc	 = FALSE;
	BOOL	fEnd = FALSE;

	if (pPath != NULL && AjcPathIsDirectoryW(pPath)) {
		rc = TRUE;
		SubCleanFolderW(pPath, &rc, &fEnd, cbp, cbNotify);
	}
	return rc;
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static		VO		SubCleanFolderW(C_WCP pPath, BOOL *rc, BOOL *fEnd, UX cbp, BOOL (CALLBACK *cbNotify)(C_WCP path, UI ntc, UX cbp))
{
	HANDLE				hFind;
	WIN32_FIND_DATAW	fd;
	int					att;
	BOOL				rsu;
	WC					PathTop [MAX_PATH];
	WC					PathWild[MAX_PATH];
	WC					PathFind[MAX_PATH];

	if (!(*fEnd)) {
		AjcSnPrintFW(PathTop , MAX_PATH, L"%s", pPath);
		AjcSnPrintFW(PathWild, MAX_PATH, L"%s", pPath);
		AjcPathCatW (PathWild, L"*.*", MAX_PATH);
		if ((hFind = FindFirstFileW(PathWild, &fd)) != INVALID_HANDLE_VALUE) {
			do {
				if(wcscmp(fd.cFileName, L".") != 0 && wcscmp(fd.cFileName, L"..") != 0) {
					AjcSnPrintFW(PathFind, MAX_PATH, L"%s", PathTop);
					AjcPathCatW (PathFind, fd.cFileName, MAX_PATH);
					//	特殊な属性を解除
					if ((att = GetFileAttributesW(PathFind)) != -1) {
						SetFileAttributesW(PathFind, att & ~(FILE_ATTRIBUTE_HIDDEN | FILE_ATTRIBUTE_READONLY | FILE_ATTRIBUTE_SYSTEM));
					}
					// ディレクトリ
					if (fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
						SubRemoveFolderW(PathFind, rc, fEnd, cbp, cbNotify);
					}
					//	ファイル
					else {
						//	ファイル削除
						SetFileAttributesW(PathFind, FILE_ATTRIBUTE_NORMAL);
						rsu = DeleteFileW(PathFind);
						//	コールバック
						if (cbNotify != NULL) {
							*fEnd = !cbNotify(PathFind, rsu ? AJCRMV_FILE_OK : AJCRMV_FILE_ERR, cbp);
						}
						//	戻り値設定(TRUE:OK, FALSE:ERR)
						*rc &= (rsu != 0);
					}
				}
			} while (FindNextFileW(hFind, &fd));
			FindClose(hFind);
		}
	}
}
//==============================================================================================================//
//	ファイル群の突合せリスト作成																				//
//																												//
//	引　数	：	pParh1	- パス名へのポインタ																	//
//				pPath2	- ワイルドカード（複数指定時はセミコロン(;)で区切る，NULLを指定時はすべて一致とする）	//
//				opt		- オプション																			//
//				cbp		- コールバックパラメタ																	//
//				cbNtc	- ファイル通知用コールバック関数（不要時はNULL)											//
//																												//
//	戻り値	：	≠NULL - 成功																					//
//				＝NULL - 失敗																					//
//==============================================================================================================//
//	ＡＶＬノード（バイト文字）
typedef struct {
	struct {
		BOOL		fValid;					//	有効フラグ
		AJCFMLHD	hd;						//	共通ヘッダ
		C_BCP		pDrv;					//	ドライブ名(ex. "d:")
		C_BCP		pTop;					//	フォルダパス名（先頭パス）
		C_BCP		pSub;					//	フォルダパス名（後続パス）
		C_BCP		pFName;					//	ファイル名
		C_BCP		pFExt;					//	ファイル拡張子
	} s[2];
} AJCFMLNODEA, *PAJCFMLNODEA;
typedef const AJCFMLNODEA *PCAJCFMLNODEA;

//	ＡＶＬノード（ワイド文字）
typedef struct {
	struct {
		BOOL		fValid;					//	有効フラグ
		AJCFMLHD	hd;						//	共通ヘッダ
		C_WCP		pDrv;					//	ドライブ名(ex. "d:")
		C_WCP		pTop;					//	フォルダパス名（先頭パス）
		C_WCP		pSub;					//	フォルダパス名（後続パス）
		C_WCP		pFName;					//	ファイル名
		C_WCP		pFExt;					//	ファイル拡張子
	} s[2];
} AJCFMLNODEW, *PAJCFMLNODEW;
typedef const AJCFMLNODEW *PCAJCFMLNODEW;

//	インスタンスワーク
typedef struct {
	int			err;						//	エラーコード
	WC			TopPath[2][MAX_PATH];		//	２つの先頭パス
	HAJCAVL		hAvl;						//	ＡＶＬハンドル
	HAJCSPL		hSpl;						//	ＳＰＬハンドル
	BOOL		fUnicode;					//	ASCII / UNICODE 識別
	UI			opt;						//	オプション(AJCFML_XXXXX)
	UX			cbp;						//	コールバックパラメタ
	union {									//	コールバック関数
		//	ASCII
		struct {
			BOOL (CALLBACK *cbQuery)(UI id, C_BCP pPath, UI att, BOOL *pfDiscard, UX cbp);
			BOOL (CALLBACK *cbEnum )(AJCFMLITEMA item[2], UX cbp);
		} asc;
		//	UNICODE
		struct {
			BOOL (CALLBACK *cbQuery)(UI id, C_WCP pPath, UI att, BOOL *pfDiscard, UX cbp);
			BOOL (CALLBACK *cbEnum )(AJCFMLITEMW item[2], UX cbp);
		} uni;
		//	共通
		struct {
			VOP		vpQuery;
			VOP		vpEnum;
		} vp;
	} u;
} AJCFML, *PAJCFML;
typedef const AJCFML *PCAJCFML;

static	BOOL SubEnumFilesMatchingList(PAJCFML pW);

//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	int WINAPI AjcEnumFilesMatchingListA(C_BCP pPath1, C_BCP pPath2, UI opt, UX cbp, BOOL (CALLBACK *cbQuery)(UI id, C_BCP pPath, UI att, BOOL *pfDiscard, UX cbp),
																							 BOOL (CALLBACK *cbEnum)(AJCFMLITEMA item[2], UX cbp))
{
	int			rc = 0;
	PAJCFML		pW = NULL;

	if (pPath1 != NULL || pPath2 != NULL) {
		if (pW = (PAJCFML)AJCMEM(sizeof(AJCFML))) {
			memset(pW, 0, sizeof(AJCFML));
			do {
				//	UNICODE識別設定
				pW->fUnicode = FALSE;
				//	２つのパス設定
				if (pPath1 != NULL) {MultiByteToWideChar(CP_ACP, 0, pPath1, -1, pW->TopPath[0], MAX_PATH);}
				if (pPath2 != NULL) {MultiByteToWideChar(CP_ACP, 0, pPath2, -1, pW->TopPath[1], MAX_PATH);}
				//	オプション設定
				pW->opt = opt;
				//	コールバック設定
				pW->cbp = cbp;
				pW->u.asc.cbQuery = cbQuery;
				pW->u.asc.cbEnum  = cbEnum;
				//	ファイル群の突合せリスト作成
				SubEnumFilesMatchingList(pW);
			} while(0);
			//	戻り値設定
			if (pW->err == AJCFMLERR_OK) rc = AjcAvlGetCount(pW->hAvl);
			else						 rc = pW->err;
			//	リソース開放
			if (pW->hAvl != NULL) {AjcAvlDelete(pW->hAvl); pW->hAvl = NULL;}
			if (pW->hSpl != NULL) {AjcSplDelete(pW->hSpl); pW->hSpl = NULL;}
			free(pW);
		}
	}
	else {
		rc = AJCFMLERR_NULL;
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	int WINAPI AjcEnumFilesMatchingListW(C_WCP pPath1, C_WCP pPath2, UI opt, UX cbp, BOOL (CALLBACK *cbQuery)(UI id, C_WCP pPath, UI att, BOOL *pfDiscard, UX cbp),
																							 BOOL (CALLBACK *cbEnum)(AJCFMLITEMW item[2], UX cbp))
{
	int			rc = FALSE;
	PAJCFML		pW = NULL;

	if (pPath1 != NULL || pPath2 != NULL) {
		if (pW = (PAJCFML)AJCMEM(sizeof(AJCFML))) {
			memset(pW, 0, sizeof(AJCFML));
			do {
				//	UNICODE識別設定
				pW->fUnicode = TRUE;
				//	２つのパス設定
				if (pPath1 != NULL) {MAjcStrCpyW(pW->TopPath[0], MAX_PATH, pPath1);}
				if (pPath2 != NULL) {MAjcStrCpyW(pW->TopPath[1], MAX_PATH, pPath2);}
				//	オプション設定
				pW->opt = opt;
				//	コールバック設定
				pW->cbp 	= cbp;
				pW->u.uni.cbQuery = cbQuery;
				pW->u.uni.cbEnum  = cbEnum;
				//	ファイル群の突合せリスト作成
				SubEnumFilesMatchingList(pW);
			} while(0);
			//	戻り値設定
			if (pW->err == AJCFMLERR_OK) rc = AjcAvlGetCount(pW->hAvl);
			else						 rc = pW->err;
			//	リソース開放
			if (pW->hAvl != NULL) {AjcAvlDelete(pW->hAvl); pW->hAvl = NULL;}
			if (pW->hSpl != NULL) {AjcSplDelete(pW->hSpl); pW->hSpl = NULL;}
			free(pW);
		}
	}
	else {
		rc = AJCFMLERR_NULL;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	共通サブ																									//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL			SubFmlFindDeep(PAJCFML pW, UI ix, WCP pFind);
static	C_WCP			SubFmlSetNode (PAJCFML pW, UI ix, C_WCP pPath, PWIN32_FIND_DATAW pfd, PAJCFMLNODEW pNode);
static	int  CALLBACK	cbCbfComp(UX key1, UX key2, UX cbp);
static	BOOL CALLBACK	cbFmlNtcAvlNode(UX key, VOP pNodeData, UI len, UI nest, UX cbp);

//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	BOOL SubEnumFilesMatchingList(PAJCFML pW)
{
	BOOL		rc = FALSE;
	UI			ix;
	WC			CurDir[MAX_PATH];

	//	エラーコードリセット
	pW->err = AJCFMLERR_OK;

	do {
		//	パスを正規化
		for (ix = 0; ix < 2; ix++) {
			if (pW->TopPath[ix][0] != 0) {
				//	相対パスの場合は、カレントディレクトリからの相対とする
				if (PathIsRelativeW(pW->TopPath[ix])) {
					GetCurrentDirectoryW(AJCTSIZE(CurDir), CurDir);
					AjcPathCatW(CurDir, pW->TopPath[ix], AJCTSIZE(CurDir));
					PathCanonicalizeW(pW->TopPath[ix], CurDir);
				}
				//	絶対パスの場合は、パスを正規化する
				else {
					WC	temp[MAX_PATH];
					MAjcStrCpyW(temp, MAX_PATH, pW->TopPath[ix]);
					PathCanonicalizeW(pW->TopPath[ix], temp);
				}
			}
		}
		//	パスチェック
		if (pW->TopPath[0][0] != 0 && !AjcPathIsDirectoryW(pW->TopPath[0])) {pW->err = AJCFMLERR_PATH1; break;}
		if (pW->TopPath[1][0] != 0 && !AjcPathIsDirectoryW(pW->TopPath[1])) {pW->err = AJCFMLERR_PATH2; break;}
		//	ＡＶＬ，ＳＰＬインスタンス生成
		if ((pW->hAvl = AjcAvlCreate ((UX)pW, cbCbfComp, NULL)) == NULL) {pW->err = AJCFMLERR_MEM; break;}
		if ((pW->hSpl = AjcSplCreateW(AJCCMP_IGNORE_WIDTH)	  ) == NULL) {pW->err = AJCFMLERR_MEM; break;}
		//	ファイルリスト作成
		for (ix = 0; ix < 2; ix++) {
			if (pW->TopPath[ix][0] != 0) {
				SubFmlFindDeep(pW, ix, pW->TopPath[ix]);
			}
			if (pW->err != AJCFMLERR_OK) break;
		}
	} while(0);

	//	突き合せリストの列挙
	if (pW->u.vp.vpEnum != NULL && pW->err == AJCFMLERR_OK) {
		AjcAvlEnumNodes(pW->hAvl, cbFmlNtcAvlNode, FALSE);
	}
	//	戻り値設定
	if (pW->err == AJCFMLERR_OK) rc = TRUE;
	else						 rc = FALSE;

	return rc;
}
//- - - サブディレクトリ検索 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -//
static	BOOL	SubFmlFindDeep(PAJCFML pW, UI ix, WCP pFind)
{
	HANDLE				hFind = INVALID_HANDLE_VALUE;
	C_WCP				pTail = NULL;
	WIN32_FIND_DATAW	fd;
	WC					wild[MAX_PATH];
	WC					find[MAX_PATH];
	AJCFMLNODEW			node  = {0};
	PAJCFMLNODEW		pNode = NULL;

	MAjcStrCpyW(wild, MAX_PATH, pFind);
	AjcPathCatW(wild, L"*.*", MAX_PATH);
	if ((hFind = FindFirstFileW(wild, &fd)) != INVALID_HANDLE_VALUE) {
		do {
			BOOL	fDiscard = FALSE;
			if (wcscmp(fd.cFileName, L".") != 0 && wcscmp(fd.cFileName, L"..") != 0) {
				//	見つかったパス設定
				MAjcStrCpyW(find, MAX_PATH, pFind);
				AjcPathCatW(find, fd.cFileName, MAX_PATH);
				//	ファイル，ディレクトリ選別
				if ((!(fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) && (pW->opt & AJCFML_FILE)) ||
					( (fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) && (pW->opt & AJCFML_DIR ))) {
					//	コールバック
					if (pW->u.vp.vpQuery != NULL) {
						//	UNICODE
						if (pW->fUnicode) {
							if (!pW->u.uni.cbQuery(ix, find, fd.dwFileAttributes, &fDiscard, pW->cbp)) {
								pW->err = AJCFMLERR_STOP_BY_USER;
							}
						}
						//	ASCII
						else {
							BC	findA[MAX_PATH];
							WideCharToMultiByte(CP_ACP, 0, find, -1, findA, MAX_PATH, NULL, NULL);
							if (!pW->u.asc.cbQuery(ix, findA, fd.dwFileAttributes, &fDiscard, pW->cbp)) {
								pW->err = AJCFMLERR_STOP_BY_USER;
							}
						}
					}
					//	ノード登録、サブディレクトリ検索
					if (!fDiscard) {
						//	ノード作成
						if ((pTail = SubFmlSetNode(pW, ix, find, &fd, &node)) == NULL) {pW->err = AJCFMLERR_MEM; break;}
						//	既存ノードがある場合、ノードデータをマージする
						if (pNode = (PAJCFMLNODEW)AjcAvlGetNodePtr(pW->hAvl, (UX)&node, NULL)) {
							memcpy(&pNode->s[ix], &node.s[ix], sizeof node.s[0]);
						}
						//	既存ノードが無い場合、ノードデータ登録
						else {
							if (AjcAvlInsDataNode(pW->hAvl, &node, sizeof node) == 0) {pW->err = AJCFMLERR_MEM; break;}
						}
						//	サブディレクトリ検索
						if (fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
							SubFmlFindDeep(pW, ix, find);
							if (pW->err != AJCFMLERR_OK) break;
						}
					}
				}
			}
		} while (FindNextFileW(hFind, &fd) && pW->err == AJCFMLERR_OK);
		FindClose(hFind);
	}
	return !pW->err;
}
//- - - ノード設定 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -//
static	C_WCP	SubFmlSetNode(PAJCFML pW, UI ix, C_WCP pPath, PWIN32_FIND_DATAW pfd, PAJCFMLNODEW pNode)
{
	C_WCP	rc	  = NULL;
	int		stl;
	C_WCP	pTail = NULL;
	WC		front[MAX_PATH	] = {0};
	WC		drv  [_MAX_DRIVE] = {0};
	WC		dir  [_MAX_DIR	] = {0};
	WC		fname[_MAX_FNAME] = {0};
	WC		fext [_MAX_EXT	] = {0};

	if (pW->err == AJCFMLERR_OK) {
		do {
			SYSTEMTIME	st;
			union {
				struct {UL l, h;} s;
				ULL				  ull;
			} u;
			//	ノードクリアー
			memset(pNode, 0, sizeof(AJCFMLNODEW));
			//	有効フラグ設定
			pNode->s[ix].fValid = TRUE;
			//	ファイルタイム設定
			FileTimeToSystemTime(&pfd->ftLastWriteTime, &st);
			pNode->s[ix].hd.utc = AjcSysTimeToTime1970(&st);
			//	ファイル属性設定
			pNode->s[ix].hd.att = pfd->dwFileAttributes;
			//	ファイルサイズ設定
			u.s.l = pfd->nFileSizeLow;
			u.s.h = pfd->nFileSizeHigh;
			pNode->s[ix].hd.size = u.ull;
			//	先頭パスから、ドライブと前部ディレクトリ設定
			MAjcStrCpyW(front, MAX_PATH, pW->TopPath[ix]);
			PathAddBackslashW(front);	// ディレクトリパスを取得するために、末尾を「\」とする
			MAjcSplitPathW(front, drv, dir, NULL, NULL);
			if ((pNode->s[ix].pDrv	 = AjcSplRegistW(pW->hSpl, drv	)) == NULL) {pW->err = AJCFMLERR_MEM; break;}
			if ((pNode->s[ix].pTop	 = AjcSplRegistW(pW->hSpl, dir	)) == NULL) {pW->err = AJCFMLERR_MEM; break;}
			//	後続パスから、後部ディレクトリとファイル名設定
			stl = (int)wcslen(pW->TopPath[ix]);
			pTail = pPath + stl;
			if (*pTail == L'\\') pTail++;
			MAjcSplitPathW(pTail, NULL, dir, fname, fext);
			if ((pNode->s[ix].pSub	 = AjcSplRegistW(pW->hSpl, dir	)) == NULL) {pW->err = AJCFMLERR_MEM; break;}
			if ((pNode->s[ix].pFName = AjcSplRegistW(pW->hSpl, fname)) == NULL) {pW->err = AJCFMLERR_MEM; break;}
			if ((pNode->s[ix].pFExt  = AjcSplRegistW(pW->hSpl, fext )) == NULL) {pW->err = AJCFMLERR_MEM; break;}
			//	戻り値＝ＯＫ
			rc = pTail;
		} while(0);
	}
	return rc;
}
//- - - ノード大小比較 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -//
static	int CALLBACK	cbCbfComp(UX key1, UX key2, UX cbp)
{
	int				rc		= 0;
	int				ix		= 0;
	PAJCFMLNODEW	pNode1	= (PAJCFMLNODEW)key1;
	PAJCFMLNODEW	pNode2	= (PAJCFMLNODEW)key2;
	WC				path1[MAX_PATH];
	WC				path2[MAX_PATH];

	//	key1側の後部パス設定
	ix = pNode1->s[0].fValid ? 0 : 1;
	MAjcMakePathW(path1, NULL, pNode1->s[ix].pSub, pNode1->s[ix].pFName, pNode1->s[ix].pFExt);
	//	key2側の後部パス設定
	ix = pNode2->s[0].fValid ? 0 : 1;
	MAjcMakePathW(path2, NULL, pNode2->s[ix].pSub, pNode2->s[ix].pFName, pNode2->s[ix].pFExt);

	rc = wcsicmp(path1, path2);

	return rc;
}
//----- ＡＶＬ列挙コールバック ---------------------------------------------------------------------------------//
static BOOL CALLBACK cbFmlNtcAvlNode(UX key, VOP pNodeData, UI len, UI nest, UX cbp)
{
	BOOL		rc	   = TRUE;
	PAJCFML		pW	   = (PAJCFML)cbp;
	PAJCFMLNODEW pNode = (PAJCFMLNODEW)pNodeData;
	UI			ix;
	union {
		AJCFMLITEMW	itemW[2];
		AJCFMLITEMA	itemA[2];
	} u;

	memset(&u, 0, sizeof u);

	for (ix = 0; ix < 2; ix++) {
		//	UNICODE
		if (pW->fUnicode) {
			WC			tail[MAX_PATH];
			if (u.itemW[ix].fValid = pNode->s[ix].fValid) {
				memcpy(&u.itemW[ix].hd, &pNode->s[ix].hd, sizeof(AJCFMLHD));
				MAjcMakePathW(tail, NULL, pNode->s[ix].pSub, pNode->s[ix].pFName, pNode->s[ix].pFExt);
				MAjcStrCpyW(u.itemW[ix].path, MAX_PATH, pW->TopPath[ix]);
				AjcPathCatW(u.itemW[ix].path, tail, MAX_PATH);
				MAjcMakePathW(u.itemW[ix].name, NULL, NULL, pNode->s[ix].pFName, pNode->s[ix].pFExt);
				MAjcStrCpyW(u.itemW[ix].tail, MAX_PATH, tail);
			}
		}
		//	ASCII
		else {
			BC			tail [MAX_PATH];
			BC			sub  [MAX_PATH];
			BC			fname[MAX_PATH];
			BC			fext [MAX_PATH];
			if (u.itemA[ix].fValid = pNode->s[ix].fValid) {
				memcpy(&u.itemA[ix].hd, &pNode->s[ix].hd, sizeof(AJCFMLHD));
				WideCharToMultiByte(CP_ACP, 0, pNode->s[ix].pSub  , -1, sub  , MAX_PATH, NULL, NULL);
				WideCharToMultiByte(CP_ACP, 0, pNode->s[ix].pFName, -1, fname, MAX_PATH, NULL, NULL);
				WideCharToMultiByte(CP_ACP, 0, pNode->s[ix].pFExt , -1, fext , MAX_PATH, NULL, NULL);
				MAjcMakePathA(tail, NULL, sub, fname, fext);
				WideCharToMultiByte(CP_ACP, 0, pW->TopPath[ix], -1, u.itemA[ix].path, MAX_PATH, NULL, NULL);
				AjcPathCatA(u.itemA[ix].path, tail, MAX_PATH);
				MAjcMakePathA(u.itemA[ix].name, NULL, NULL, fname, fext);
				MAjcStrCpyA(u.itemA[ix].tail, MAX_PATH, tail);
			}
		}
	}
	//	突合せリストのコールバック
	if (pW->u.vp.vpEnum != NULL) {
		if (pW->fUnicode) rc = pW->u.uni.cbEnum(u.itemW, pW->cbp);
		else			  rc = pW->u.asc.cbEnum(u.itemA, pW->cbp);
	}
	return rc;
}
//==============================================================================================================//
//	親ディレクトリパス取得																						//
//																												//
//	引　数	：	pParh		- ディレクトリパス名																//
//				pBuf		- 親ディレクトリのパスを格納するバッファ（不要時はNULL）							//
//				lBuf		- 親ディレクトリのパスを格納するバッファ長											//
//				pfRoot		- 親ディレクトリがルートであることを示すフラグのバッファ（不要時はNULL）			//
//																												//
//	戻り値	：	TRUE  - 正常																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI AjcGetParentDirectoryA(C_BCP pPath, BCP pBuf, UI lBuf, BOOL *pfRoot)
{
	BOOL	rc = FALSE;
	WC		path[MAX_PATH] = {0};
	WC		buf [MAX_PATH];

	if (MultiByteToWideChar(CP_ACP, 0, pPath, -1, path, MAX_PATH) != 0) {
		if (rc = AjcGetParentDirectoryW(path, buf, MAX_PATH, pfRoot)) {
			WideCharToMultiByte(CP_ACP, 0, buf, -1, pBuf, lBuf, NULL, NULL);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI AjcGetParentDirectoryW(C_WCP pPath, WCP pBuf, UI lBuf, BOOL *pfRoot)
{
	BOOL	rc = FALSE;
	WC		temp[MAX_PATH];
	WC		buf [MAX_PATH];

	if (pPath != NULL) {
		do {
			//	指定パスがルートならばエラー
			AjcSnPrintFW(temp, MAX_PATH, L"%s", pPath);
			if (!PathCanonicalizeW(buf, temp)) break;
			if (PathIsRootW(buf)) break;
			//	親ディレクトリパス取得
			AjcPathCatW(temp, L"..", MAX_PATH);
			if (!PathCanonicalizeW(buf, temp)) break;
			//	親パス設定
			if (pBuf != NULL && lBuf != 0) {
				AjcSnPrintF(pBuf, lBuf, L"%s", buf);
			}
			//	親パスのルートチェック
			if (pfRoot != NULL) {
				*pfRoot = PathIsRootW(buf);
			}
			rc = TRUE;
		} while(0);
	}
	return rc;
}
//==============================================================================================================//
//	ディレクトリ作成																							//
//																												//
//	引　数	：	lpTemplateDirectory		- テンプレートディレクトリパス名										//
//				lpNewDirectory			- 生成するディレクトリパス名											//
//				lpSecurityAttributes	- SECURITY_ATTRIBUTES構造体へのポインター								//
//				fSaveParentDirTime		- TRUE	: 親ディレクトリの他有無スタンプを保持する						//
//										  FALSE : 親ディレクトリの他有無スタンプを保持しない					//
//																												//
//	戻り値	：	TRUE  - 正常																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//]
AJCEXPORT	BOOL	WINAPI AjcCreateDirectoryA(C_BCP lpNewDirectory, BOOL fSaveParentDirTime)
{
	return AjcCreateDirectoryExA(NULL, lpNewDirectory, NULL, fSaveParentDirTime);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	BOOL	WINAPI AjcCreateDirectoryExA(C_BCP lpTemplateDirectory, C_BCP lpNewDirectory, LPSECURITY_ATTRIBUTES lpSecurityAttributes, BOOL fSaveParentDirTime)
{
	BOOL	rc = FALSE;
	WC		tmpDir[MAX_PATH] = {0};
	WC		newDir[MAX_PATH] = {0};

	if (lpTemplateDirectory != NULL) MultiByteToWideChar(CP_ACP, 0, lpTemplateDirectory, -1, tmpDir, MAX_PATH);
	if (lpNewDirectory		!= NULL) MultiByteToWideChar(CP_ACP, 0, lpNewDirectory	   , -1, newDir, MAX_PATH);

	if (lpNewDirectory != NULL) {
		if (lpTemplateDirectory != NULL) {
			rc = AjcCreateDirectoryExW(tmpDir, newDir, lpSecurityAttributes, fSaveParentDirTime);
		}
		else {
			rc = AjcCreateDirectoryExW(NULL  , newDir, lpSecurityAttributes, fSaveParentDirTime);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI AjcCreateDirectoryW(C_WCP lpNewDirectory, BOOL fSaveParentDirTime)
{
	return AjcCreateDirectoryExW(NULL, lpNewDirectory, NULL, fSaveParentDirTime);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	BOOL	WINAPI AjcCreateDirectoryExW(C_WCP lpTemplateDirectory, C_WCP lpNewDirectory, LPSECURITY_ATTRIBUTES lpSecurityAttributes, BOOL fSaveParentDirTime)
{
	BOOL		rc				 = FALSE;
	BOOL		fRoot			 = FALSE;
	AJCFTIMES	ft				 = {0};
	WC			parent[MAX_PATH] = {0};

	if (lpNewDirectory != NULL) {
		do {
			//	親ディレクトリのタイムスタンプ退避
			if (fSaveParentDirTime) {
				if (!AjcGetParentDirectoryW(lpNewDirectory, parent, MAX_PATH, &fRoot)) break;
				if (!AjcGetFileTimeW(parent, &ft, NULL, FALSE)) break;
			}
			//	新ディレクトリ作成
			if (lpTemplateDirectory != NULL) {
				if (!CreateDirectoryExW(lpTemplateDirectory, lpNewDirectory, lpSecurityAttributes)) break;
			}
			else {
				if (!CreateDirectoryW(lpNewDirectory, lpSecurityAttributes)) break;
			}
			//	親ディレクトリのタイムスタンプ回復
			if (fSaveParentDirTime) {
				if (!AjcSetFileTime(parent, &ft, FALSE)) break;
			}
			rc = TRUE;
		} while(0);
	}
	return rc;
}
//==============================================================================================================//
//	ディレクトリ構造の作成																						//
//																												//
//	引　数	：	pParh					- ディレクトリパス名													//
//				lpSecurityAttributes	- SECURITY_ATTRIBUTES構造体へのポインター								//
//																												//
//	戻り値	：	TRUE  - 正常																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI AjcCreateDirectoryStructA(C_BCP pPath)
{
	return AjcCreateDirectoryStructExA(pPath, NULL);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	BOOL	WINAPI AjcCreateDirectoryStructExA(C_BCP pPath, LPSECURITY_ATTRIBUTES lpSecurityAttributes)
{
	BOOL	rc = FALSE;
	WC		path[MAX_PATH];

	if (MultiByteToWideChar(CP_ACP, 0, pPath, -1, path, MAX_PATH) != 0) {
		rc = AjcCreateDirectoryStructExW(path, lpSecurityAttributes);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI AjcCreateDirectoryStructW(C_WCP pPath)
{
	return AjcCreateDirectoryStructExW(pPath, NULL);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	BOOL	WINAPI AjcCreateDirectoryStructExW(C_WCP pPath, LPSECURITY_ATTRIBUTES lpSecurityAttributes)
{
	BOOL	rc = FALSE;
	WCP		p;
	int		stl;
	WC		relp[MAX_PATH	 ] = {0};
	WC		absp[MAX_PATH + 1] = {0};
	WC		path[MAX_PATH	 ] = {0};

	if (pPath != NULL) {
		rc = TRUE;
		//	相対パスの場合は、カレントディレクトリからの相対とする
		if (PathIsRelative(pPath)) {
			GetCurrentDirectory(MAX_PATH, relp);
			AjcPathCat (relp, pPath, MAX_PATH);
			PathCanonicalize(absp, relp);
		}
		//	絶対パスの場合は、パスを正規化する
		else {
			PathCanonicalize(absp, pPath);
		}

		//	パスを多重文字列化
		PathRemoveBackslash(absp);
		for (p = absp; *p != 0; p++) {
			if (*p == L'\\') *p = 0;
		}

		//	パスを再構成し、ディレクトリ作成
		wcscpy_s(path, MAX_PATH, absp);
		p = &absp[wcslen(absp) + 1];
		while (*p != 0) {
			//	パス項目を１つ追加
			stl = (int)wcslen(p);
			AjcPathCat(path, p, MAX_PATH);
			p += (stl + 1);
			//	ディレクトリ以外のパスが存在する場合は、エラー
			if (AjcPathExists(path)) {
				if (!AjcPathIsDirectory(path)) {
					rc = FALSE;
					break;
				}
			}
			//	パスが存在しなければ、ディレクトリ作成
			else {
				if (!CreateDirectory(path, lpSecurityAttributes)) {
					rc = FALSE;
					break;
				}
			}
		}
	}
	return rc;
}

//==============================================================================================================//
//	パスがワイルドカードの指定に一致するかチェックする															//
//																												//
//	引　数	：	pParh	- パス名へのポインタ																	//
//				pWild	- ワイルドカード（複数指定時はセミコロン(;)で区切る，NULLを指定時はすべて一致とする）	//
//																												//
//	戻り値	：	TRUE  - 一致																					//
//				FALSE - 不一致																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPathMatchSpecA(C_BCP pPath, C_BCP pWild)
{
	BOOL	rc = FALSE;
	int		i;
	C_BCP	p  = pWild;
	BC		wild[MAX_PATH];

	if (pPath != NULL && pWild != NULL) {
		while (*p != 0) {
			//	ワイルドカードを１つ設定
			for (i = 0; i < MAX_PATH - 1 && *p != 0 && *p != ';' && *p != '/'; i++, p ++) {
				wild[i] = *p;
			}
			wild[i] = 0;
			//	両端空白除去
			AjcStrTrimA(wild, wild, MAX_PATH);
			//	空文字列以外ならば、ワイルドカードチェック
			if (wild[0] != 0) {
				if (PathMatchSpecA(pPath, wild)) {
					rc = TRUE;
					break;
				}
				else {
					BC	fname[_MAX_FNAME];
					BC	fext [_MAX_EXT];
					BC	name [MAX_PATH];
					MAjcSplitPathA(pPath, NULL, NULL, fname, fext);
					MAjcMakePathA (name , NULL, NULL, fname, fext);
					if (stricmp(name, wild) == 0) {
						rc = TRUE;
						break;
					}
				}
			}
			//	MAX_PATH以上の長さならば、セミコロン(;)以外をスキップ
			while (*p != ';' && *p != '/' && *p != 0) {
				p++;
			}
			//	セミコロンスキップ
			while (*p == ';' || *p == '/') {
				p++;
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPathMatchSpecW(C_WCP pPath, C_WCP pWild)
{
	BOOL	rc = FALSE;
	int		i;
	C_WCP	p  = pWild;
	WC		wild[MAX_PATH];

	if (pPath != NULL && pWild != NULL) {
		while (*p != 0) {
			//	ワイルドカードを１つ設定
			for (i = 0; i < MAX_PATH - 1 && *p != 0 && *p != L';' && *p != L'/'; i++, p ++) {
				wild[i] = *p;
			}
			wild[i] = 0;
			//	両端空白除去
			AjcStrTrimW(wild, wild, MAX_PATH);
			//	空文字列以外ならば、ワイルドカードチェック
			if (wild[0] != 0) {
				if (PathMatchSpecW(pPath, wild)) {
					rc = TRUE;
					break;
				}
				else {
					WC	fname[_MAX_FNAME];
					WC	fext [_MAX_EXT];
					WC	name [MAX_PATH];
					MAjcSplitPathW(pPath, NULL, NULL, fname, fext);
					MAjcMakePathW (name , NULL, NULL, fname, fext);
					if (wcsicmp(name, wild) == 0) {
						rc = TRUE;
						break;
					}
				}
			}
			//	MAX_PATH以上の長さならば、セミコロン(;)以外をスキップ
			while (*p != L';' && *p != L'/' && *p != 0) {
				p++;
			}
			//	セミコロンスキップ
			while (*p == L';' || *p == L'/') {
				p++;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	パスが指定文字列を含むかチェックする																		//
//																												//
//	引　数	：	pPath	- パス名へのポインタ																	//
//				pStr	- チェックする文字列(複数指定時はセミコロン(;)で区切る，NULLを指定時はすべて一致とする)	//
//																												//
//	戻り値	：	TRUE  - 一致																					//
//				FALSE - 不一致																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPathMatchStrA(C_BCP pPath, C_BCP pStr)
{
	BOOL	rc = FALSE;
	BC		path[MAX_PATH];

	if (pPath != NULL) {
		//	パスコピー
		MAjcStrCpyA(path, MAX_PATH, pPath);
		//	ディレクトリならば末尾に '\' 確保
		if (AjcPathIsDirectoryA(path)) {
			AjcPathCatA(path, "", MAX_PATH);
		}
		//	パス中文字列チェック
		rc = (AjcStrIFindExA(pPath, pStr, "/;", "*", NULL, TRUE, 0, NULL) >= 0);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPathMatchStrW(C_WCP pPath, C_WCP pStr)
{
	BOOL	rc = FALSE;
	WC		path[MAX_PATH];

	if (pPath != NULL) {
		//	パスコピー
		MAjcStrCpyW(path, MAX_PATH, pPath);
		//	ディレクトリならば末尾に '\' 確保
		if (AjcPathIsDirectoryW(path)) {
			AjcPathCatW(path, L"", MAX_PATH);
		}
		//	パス中文字列チェック
		rc = (AjcStrIFindExW(pPath, pStr, L"/;", L"*", NULL, TRUE, 0, NULL) >= 0);
	}
	return rc;
}
//==============================================================================================================//
//	パスの末尾が「\」かチェックする																				//
//																												//
//	引　数	：	pParh	- パス名へのポインタ																	//
//																												//
//	戻り値	：	TRUE  - パスの末尾は「\」である																	//
//				FALSE - パスの末尾は「\」以外																	//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcIsPathBackslashA(C_BCP pPath)
{
	BOOL	rc = FALSE;
	int		stl;

	if (pPath != NULL) {
		stl = (int)strlen(pPath);
		if (stl != 0 && (stl >= 2 && mbsbtype(pPath, stl - 1) != _MBC_TRAIL) && pPath[stl -1] == '\\') {
			rc = TRUE;
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcIsPathBackslashW(C_WCP pPath)
{
	BOOL	rc = FALSE;
	int		stl;

	if (pPath != NULL) {
		stl = (int)wcslen(pPath);
		if (stl != 0 && pPath[stl - 1] == L'\\') {
			rc = TRUE;
		}
	}
	return rc;
}

//==============================================================================================================//
//	パスの末尾が「\」or「/」かチェックする																		//
//																												//
//	引　数	：	pParh	- パス名へのポインタ																	//
//																												//
//	戻り値	：	TRUE  - パスの末尾は「\」or「/」である															//
//				FALSE - パスの末尾は「\」or「/」以外															//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcIsPathBackslashExA(C_BCP pPath)
{
	BOOL	rc = FALSE;
	int		stl;

	if (pPath != NULL) {
		stl = (int)strlen(pPath);
		if (stl != 0 && (stl >= 2 && mbsbtype(pPath, stl - 1) != _MBC_TRAIL) && (pPath[stl -1] == '\\' || pPath[stl -1] == '/')) {
			rc = TRUE;
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcIsPathBackslashExW(C_WCP pPath)
{
	BOOL	rc = FALSE;
	int		stl;

	if (pPath != NULL) {
		stl = (int)wcslen(pPath);
		if (stl == 0 && (pPath[stl - 1] == L'\\' || pPath[stl - 1] == L'/')) {
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	２つのパスの結合（２つの文字列の間に、必ず１つの「￥」が挿入された文字列を作成する）						//
//																												//
//	引　数	：	pTop  - パスが付加される文字列のアドレス														//
//				pTail - 付加する文字列のアドレス																//
//				lBuf  - パスが付加される文字列バッファの総文字数												//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPathCatA(BCP pTop, C_BCP pTail, int lBuf)
{
	BOOL	rc = FALSE;
	BC		cTop, cTail;
	int		lTop;
	int		lTail;

	if (pTop != NULL && pTail != NULL) {
		lTop  = (int)strlen(pTop);
		lTail = (int)strlen(pTail);
		if (lTop == 0 || (lTop >= 2 && mbsbtype(pTop, lTop - 1) == _MBC_TRAIL)) cTop = 0;
		else																	cTop = pTop[lTop - 1];
		cTail = pTail[0];
		if		((cTop == '\\' || cTop == '/')	&&	(cTail == '\\' || cTail == '/')) {
			if (lTop + lTail <= lBuf) {
				strcat(pTop, pTail + 1);
				rc = TRUE;
			}
		}
		else if ((cTop == '\\' || cTop == '/')	||	(cTail == '\\' || cTail == '/')) {
			if (lTop + lTail <	lBuf) {
				strcat(pTop, pTail);
				rc = TRUE;
			}
		}
		else {
			if (lTop + lTail < lBuf - 1) {
				strcat(pTop, "\\");
				strcat(pTop, pTail);
				rc = TRUE;
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPathCatW(WCP pTop, C_WCP pTail, int lBuf)
{
	BOOL	rc = FALSE;
	UW		cTop, cTail;
	int		lTop;
	int		lTail;

	if (pTop != NULL && pTail != NULL) {
		lTop  = (int)wcslen(pTop);
		lTail = (int)wcslen(pTail);
		if (lTop == 0) cTop = 0;
		else		   cTop  = pTop[lTop - 1];
		cTail = pTail[0];
		if		((cTop == L'\\' || cTop == L'/')  &&  (cTail == L'\\' || cTail == L'/')) {
			if (lTop + lTail <= lBuf) {
				wcscat(pTop, pTail + 1);
				rc = TRUE;
			}
		}
		else if ((cTop == L'\\' || cTop == L'/')  ||  (cTail == L'\\' || cTail == L'/')) {
			if (lTop + lTail <	lBuf) {
				wcscat(pTop, pTail);
				rc = TRUE;
			}
		}
		else {
			if (lTop + lTail < lBuf - 1) {
				wcscat(pTop, L"\\");
				wcscat(pTop, pTail);
				rc = TRUE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	２つのパスの比較（文字列末尾の「￥」を削除して比較）														//
//																												//
//	引　数	：	pPath1 - 較元パス１																				//
//				pPath2 - 較先パス２																				//
//																												//
//	戻り値	：	＝０： 等しい																					//
//				＜０：path1 < path2																				//
//				＞０：path1 > path2																				//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	int		WINAPI	AjcPathCmpA(C_BCP pPath1, C_BCP pPath2)
{
	BOOL	rc = 0;
	BC		path1[MAX_PATH];
	BC		path2[MAX_PATH];

	if (pPath1 != NULL && pPath2 != NULL) {
		//	PATH1の末尾の「￥」を除去
		AjcSnPrintFA(path1, MAX_PATH, "%s", pPath1);
		PathRemoveBackslashA(path1);

		//	PATH2の末尾の「￥」を除去
		AjcSnPrintFA(path2, MAX_PATH, "%s", pPath2);
		PathRemoveBackslashA(path2);

		rc = mbsicmp(path1, path2);
	}
	else if (pPath1 == NULL) {
		rc = -1;
	}
	else {
		rc = 1;
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	int		WINAPI	AjcPathCmpW(C_WCP pPath1, C_WCP pPath2)
{
	BOOL	rc = 0;
	WC		path1[MAX_PATH];
	WC		path2[MAX_PATH];

	if (pPath1 != NULL && pPath2 != NULL) {
		//	PATH1の末尾の「￥」を除去
		AjcSnPrintFW(path1, MAX_PATH, L"%s", pPath1);
		PathRemoveBackslashW(path1);

		//	PATH2の末尾の「￥」を除去
		AjcSnPrintFW(path2, MAX_PATH, L"%s", pPath2);
		PathRemoveBackslashW(path2);

		rc = wcsicmp(path1, path2);
	}
	else if (pPath1 == NULL) {
		rc = -1;
	}
	else {
		rc = 1;
	}
	return rc;
}
//==============================================================================================================//
//	ファイル名に使用できない文字を特定の文字に変換する															//
//																												//
//	引　数	：	pFName - パスが付加される文字列のアドレス														//
//				c	  - 付加する文字列のアドレス																//
//																												//
//	戻り値	：	>= 0  - 変更した文字数																			//
//				-1	  - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
static	const BC	InvCharsA[] = "\\/:*?\"<>|";
AJCEXPORT	int	WINAPI	AjcChangeFNameToCorrectA(BCP pFName, BC c)
{
	int		rc = -1;
	BCP		p = pFName;

	if (pFName != NULL && strchr(InvCharsA, c) == NULL) {
		rc = 0;
		while ((p = (BCP)_mbspbrk((UBP)pFName, (UBP)InvCharsA)) != NULL) {
			*p = c;
			rc++;
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
static	const WC	InvCharsW[] = L"\\/:*?\"<>|";
AJCEXPORT	int	WINAPI	AjcChangeFNameToCorrectW(WCP pFName, WC c)
{
	int		rc = -1;
	WCP		p = pFName;

	if (pFName != NULL && wcschr(InvCharsW, c) == NULL) {
		rc = 0;
		while ((p = wcspbrk(pFName, InvCharsW)) != NULL) {
			*p = c;
			rc++;
		}
	}
	return rc;
}

//==============================================================================================================//
//	パスが存在するかチェックする																				//
//																												//
//	引　数	：	pParh	- パス名へのポインタ																	//
//																												//
//	戻り値	：	TRUE  - パスは存在する																			//
//				FALSE - パスは存在しない																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPathExistsA(C_BCP pPath)
{
	BOOL	rc = FALSE;

	if (pPath != NULL) {
		rc = PathFileExistsA(pPath);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPathExistsW(C_WCP pPath)
{
	BOOL	rc = FALSE;

	if (pPath != NULL) {
		rc = PathFileExistsW(pPath);
	}
	return rc;
}

//==============================================================================================================//
//	パスがディレクトリかチェックする																			//
//																												//
//	引　数	：	pParh	- パス名へのポインタ																	//
//																												//
//	戻り値	：	TRUE  - パスはディレクトリである																//
//				FALSE - パスはディレクトリ以外																	//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPathIsDirectoryA(C_BCP pPath)
{
	BOOL	rc = FALSE;

	if (pPath != NULL) {
		rc = (PathIsDirectoryA(pPath) != 0);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPathIsDirectoryW(C_WCP pPath)
{
	BOOL	rc = FALSE;

	if (pPath != NULL) {
		rc = (PathIsDirectoryW(pPath) != 0);
	}
	return rc;
}

//==============================================================================================================//
//	空のディレクトリかチェックする																				//
//																												//
//	引　数	：	pParh	- ディレクトリパス名へのポインタ														//
//				cbp		- コールバックパラメタ																	//
//				cbQuery	- コールバック関数（不要時はNULL)														//
//																												//
//	戻り値	：	TRUE  - パスは空のディレクトリである															//
//				FALSE - パスは空のディレクトリ以外																//
//==============================================================================================================//
static		VO		SubIsEmptyDirW(C_WCP pPath, BOOL *rc, BOOL *pfEnd, UX cbp, BOOL (CALLBACK *cbQuery)(C_WCP pPath, UI att, BOOL *pfIgnore, UX cbp));

//----- バイト文字 ---------------------------------------------------------------------------------------------//
typedef struct {
	UX		cbp;
	BOOL (CALLBACK *cbQuery)(C_BCP pPath, UI att, BOOL *pfIgnore, UX cbp);
} IsEmpPrm, *PIsEmpPrm;

static	BOOL	CALLBACK cbIsEmpQuery(C_WCP pPath, UI att, BOOL *pfIgnore, UX cbp);
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	BOOL	WINAPI	AjcPathIsEmptyDirectoryA(C_BCP pPath, UX cbp, BOOL (CALLBACK *cbQuery)(C_BCP pPath, UI att, BOOL *pfIgnore, UX cbp))
{
	BOOL		rc = FALSE;
	WC			pathW[MAX_PATH];
	IsEmpPrm	prm;

	if (pPath != NULL) {
		prm.cbp 	= cbp;
		prm.cbQuery = cbQuery;
		AjcByteCharToWideChar(pPath, pathW, MAX_PATH);
		rc = AjcPathIsEmptyDirectoryW(pathW, (UX)&prm, cbIsEmpQuery);
	}
	return rc;
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	BOOL	CALLBACK cbIsEmpQuery(C_WCP pPath, UI att, BOOL *pfIgnore, UX cbp)
{
	BOOL		rc = TRUE;
	PIsEmpPrm	p = (PIsEmpPrm)cbp;
	BC			path[MAX_PATH];

	WideCharToMultiByte(CP_ACP, 0, pPath, -1, path, MAX_PATH, NULL, NULL);
	if (p->cbQuery != NULL) {
		rc = p->cbQuery(path, att, pfIgnore, p->cbp);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPathIsEmptyDirectoryW(C_WCP pPath, UX cbp, BOOL (CALLBACK *cbQuery)(C_WCP pPath, UI att, BOOL *pfIgnore, UX cbp))
{
	BOOL	rc	 = FALSE;
	BOOL	fEnd = FALSE;

	if (pPath != NULL && AjcPathIsDirectoryW(pPath)) {
		rc	 = TRUE;
		fEnd = FALSE;
		SubIsEmptyDirW(pPath, &rc, &fEnd, cbp, cbQuery);
	}
	return rc;
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static		VO		SubIsEmptyDirW(C_WCP pPath, BOOL *rc, BOOL *pfEnd, UX cbp, BOOL (CALLBACK *cbQuery)(C_WCP pPath, UI att, BOOL *pfIgnore, UX cbp))
{
	HANDLE				hFind;
	WIN32_FIND_DATAW	fd;
	BOOL				fIgnore = FALSE;
	WC					PathTop [MAX_PATH];
	WC					PathWild[MAX_PATH];
	WC					PathFind[MAX_PATH];

	if (!(*pfEnd)) {
		AjcSnPrintFW(PathTop , MAX_PATH, L"%s", pPath);
		AjcSnPrintFW(PathWild, MAX_PATH, L"%s", pPath);
		AjcPathCatW (PathWild, L"*.*", MAX_PATH);
		if ((hFind = FindFirstFileW(PathWild, &fd)) != INVALID_HANDLE_VALUE) {
			do {
				if(wcscmp(fd.cFileName, L".") != 0 && wcscmp(fd.cFileName, L"..") != 0) {
					AjcSnPrintFW(PathFind, MAX_PATH, L"%s", PathTop);
					AjcPathCatW (PathFind, fd.cFileName, MAX_PATH);
					//	除外フラグクリアー
					fIgnore = FALSE;
					//	コールバック
					if (cbQuery != NULL) {
						*pfEnd = !cbQuery(PathFind, fd.dwFileAttributes, &fIgnore, cbp);
					}
					//	中止の場合、戻り値＝FALSE（空ディレクトリ以外）設定
					if (*pfEnd) {
						*rc = FALSE;
					}
					//	継続の場合・・・
					else if (!fIgnore) {	//	無視しない？
						//	ディレクトリならば、サブディレクトリ検索
						if (fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
							SubIsEmptyDirW(PathFind, rc, pfEnd, cbp, cbQuery);
						}
						//	ファイルならば、戻り値＝FALSE（空ディレクトリ以外）設定し、終了
						else {
							*rc    = FALSE;		//	戻り値＝FALSE（空ディレクトリ以外）
							*pfEnd = TRUE;		//	検索終了
						}
					}
				}
			} while (!(*pfEnd) && FindNextFileW(hFind, &fd));
			FindClose(hFind);
		}
	}
}
//==============================================================================================================//
//	パスがファイルかチェックする																				//
//																												//
//	引　数	：	pParh	- パス名へのポインタ																	//
//																												//
//	戻り値	：	TRUE  - パスはファイルである																	//
//				FALSE - パスはファイル以外																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPathIsFileA(C_BCP pPath)
{
	BOOL	rc = FALSE;

	if (pPath != NULL) {
		rc = (PathFileExistsA(pPath) && !PathIsDirectoryA(pPath));
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPathIsFileW(C_WCP pPath)
{
	BOOL	rc = FALSE;

	if (pPath != NULL) {
		rc = (PathFileExistsW(pPath) && !PathIsDirectoryW(pPath));
	}
	return rc;
}

//==============================================================================================================//
//	ファイルサイズ取得																							//
//																												//
//	引　数	：	pPath	- ファイルのパス名																		//
//																												//
//	戻り値	：	≠-1 - ファイルサイズ																			//
//				＝-1 - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	ULL	WINAPI	AjcGetFileSizeA(C_BCP pPath)
{
	ULL		rc = -1;
	struct _stati64 st;

	if (pPath != NULL) {
		if (_stati64(pPath, &st) == 0 && !(st.st_mode & _S_IFDIR)) {
			rc = st.st_size;
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	ULL	WINAPI	AjcGetFileSizeW(C_WCP pPath)
{
	ULL		rc = -1;
	struct _stati64 st;

	if (pPath != NULL) {
		if (_wstati64(pPath, &st) == 0 && !(st.st_mode & _S_IFDIR)) {
			rc = st.st_size;
		}
	}
	return rc;
}
//==============================================================================================================//
//	ファイルタイム取得																							//
//																												//
//	引　数	：	pPath	- ファイルのパス名																		//
//																												//
//	戻り値	：	≠-1 - ファイルタイム																			//
//				＝-1 - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcGetFileTime1970A(C_BCP pPath)
{
	UL		rc = -1;
	struct _stati64 st;

	if (pPath != NULL) {
		if (_stati64(pPath, &st) == 0 && !(st.st_mode & _S_IFDIR)) {
			if (st.st_mtime <= 0xFFFFFFFE) {
				rc = (UI)st.st_mtime;
			}
			else {
				rc = 0xFFFFFFFE;
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcGetFileTime1970W(C_WCP pPath)
{
	UL		rc = -1;
	struct _stat st;

	if (pPath != NULL) {
		if (_wstat(pPath, &st) == 0 && !(st.st_mode & _S_IFDIR)) {
			if (st.st_mtime <= 0xFFFFFFFE) {
				rc = (UI)st.st_mtime;
			}
			else {
				rc = 0xFFFFFFFE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	ファイル比較																								//
//																												//
//	引　数	：	pFile1, pFile2	- 比較する２つのファイルパス													//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - 不一致／エラー																			//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcFileCompareA(C_BCP pFile1, C_BCP pFile2)
{
	BOOL	rc = TRUE;
	HANDLE	hF1 = INVALID_HANDLE_VALUE;
	HANDLE	hF2 = INVALID_HANDLE_VALUE;
	UL		bytes1;
	UL		bytes2;
	UB		buf1[2048];
	UB		buf2[2048];

	do {
		if ((hF1 = CreateFileA(pFile1, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, 0, NULL)) == INVALID_HANDLE_VALUE) {rc = FALSE; break;}
		if ((hF2 = CreateFileA(pFile2, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, 0, NULL)) == INVALID_HANDLE_VALUE) {rc = FALSE; break;}

		if (!ReadFile(hF1, buf1, sizeof buf1, &bytes1, NULL)) {rc = FALSE; break;}
		if (!ReadFile(hF2, buf2, sizeof buf2, &bytes2, NULL)) {rc = FALSE; break;}
		while (bytes1 == bytes2 && bytes1 != 0) {
			if (memcmp(buf1, buf2, bytes1) != 0) {
				rc = FALSE;
				break;
			}
			if (!ReadFile(hF1, buf1, sizeof buf1, &bytes1, NULL)) {rc = FALSE; break;}
			if (!ReadFile(hF2, buf2, sizeof buf2, &bytes2, NULL)) {rc = FALSE; break;}
		}
		if (bytes1 != bytes2) rc = FALSE;
	} while(0);

	if (hF1 != INVALID_HANDLE_VALUE) CloseHandle(hF1);
	if (hF2 != INVALID_HANDLE_VALUE) CloseHandle(hF2);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcFileCompareW(C_WCP pFile1, C_WCP pFile2)
{
	BOOL	rc = TRUE;
	HANDLE	hF1 = INVALID_HANDLE_VALUE;
	HANDLE	hF2 = INVALID_HANDLE_VALUE;
	UL		bytes1;
	UL		bytes2;
	UB		buf1[2048];
	UB		buf2[2048];

	do {
		if ((hF1 = CreateFileW(pFile1, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, 0, NULL)) == INVALID_HANDLE_VALUE) {rc = FALSE; break;}
		if ((hF2 = CreateFileW(pFile2, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, 0, NULL)) == INVALID_HANDLE_VALUE) {rc = FALSE; break;}

		if (!ReadFile(hF1, buf1, sizeof buf1, &bytes1, NULL)) {rc = FALSE; break;}
		if (!ReadFile(hF2, buf2, sizeof buf2, &bytes2, NULL)) {rc = FALSE; break;}
		while (bytes1 == bytes2 && bytes1 != 0) {
			if (memcmp(buf1, buf2, bytes1) != 0) {
				rc = FALSE;
				break;
			}
			if (!ReadFile(hF1, buf1, sizeof buf1, &bytes1, NULL)) {rc = FALSE; break;}
			if (!ReadFile(hF2, buf2, sizeof buf2, &bytes2, NULL)) {rc = FALSE; break;}
		}
		if (bytes1 != bytes2) rc = FALSE;
	} while(0);

	if (hF1 != INVALID_HANDLE_VALUE) CloseHandle(hF1);
	if (hF2 != INVALID_HANDLE_VALUE) CloseHandle(hF2);

	return rc;
}
//==============================================================================================================//
//	ファイル／ディレクトリのタイムスタンプ取得																	//
//																												//
//	引　数	：	pPath		- ファイル／ディレクトリのパス名													//
//				pFTimes		- 取得したタイムスタンプ（通算）を格納するバッファ（不要時はNULL）					//
//				pSTimes		- 取得したタイムスタンプ（日時）を格納するバッファ（不要時はNULL）					//
//				fLocalTime	- TRUE	: ローカルタイムで取得														//
//							  FALSE : ＵＴＣで取得																//
//																												//
//	戻り値	：	TRUE  : 正常																					//
//				FALSE : エラー																					//
//==============================================================================================================//
static	BOOL	SubGetTimeStamp(HANDLE hFile, PAJCFTIMES pFTimes, PAJCSTIMES pSTimes, BOOL fLocalTime);
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetFileTimeA(C_BCP pPath, PAJCFTIMES pFTimes, PAJCSTIMES pSTimes, BOOL fLocalTime)
{
	BOOL	rc = FALSE;
	HANDLE	hFile;

	if (pPath != NULL) {
		if ((hFile = CreateFileA(pPath,
								 GENERIC_READ,
								 FILE_SHARE_READ | FILE_SHARE_WRITE,
								 NULL,
								 OPEN_EXISTING,
								 AjcPathIsDirectoryA(pPath) ? FILE_FLAG_BACKUP_SEMANTICS : FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE) {
			rc = SubGetTimeStamp(hFile, pFTimes, pSTimes, fLocalTime);
			CloseHandle(hFile);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetFileTimeW(C_WCP pPath, PAJCFTIMES pFTimes, PAJCSTIMES pSTimes, BOOL fLocalTime)
{
	BOOL	rc = FALSE;
	HANDLE	hFile;

	if (pPath != NULL) {
		if ((hFile = CreateFileW(pPath,
								 GENERIC_READ,
								 FILE_SHARE_READ | FILE_SHARE_WRITE,
								 NULL,
								 OPEN_EXISTING,
								 AjcPathIsDirectoryW(pPath) ? FILE_FLAG_BACKUP_SEMANTICS : FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE) {
			rc = SubGetTimeStamp(hFile, pFTimes, pSTimes, fLocalTime);
			CloseHandle(hFile);
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	SubGetTimeStamp(HANDLE hFile, PAJCFTIMES pFTimes, PAJCSTIMES pSTimes, BOOL fLocalTime)
{
	BOOL	rc = FALSE;
	AJCUFT	lfmk, lfac, lfup;		//	ファイルタイム（ローカルタイム）
	AJCUFT	sfmk, sfac, sfup;		//	ファイルタイム（ＵＴＣ）

	//	タイムスタンプ取得
	if (GetFileTime(hFile, &sfmk.ft, &sfac.ft, &sfup.ft)) {
		//	ローカルタイム取得
		if (fLocalTime) {
			//	ローカルタイムへ変換
			FileTimeToLocalFileTime(&sfmk.ft, &lfmk.ft);
			FileTimeToLocalFileTime(&sfac.ft, &lfac.ft);
			FileTimeToLocalFileTime(&sfup.ft, &lfup.ft);
			//	戻り情報(ローカル通算時刻)設定
			if (pFTimes != NULL) {
				pFTimes->mk.v = lfmk.v;
				pFTimes->ac.v = lfac.v;
				pFTimes->up.v = lfup.v;
			}
			//	戻り情報(ローカル日時)設定
			if (pSTimes != NULL) {
				FileTimeToSystemTime(&lfmk.ft, &pSTimes->mk);
				FileTimeToSystemTime(&lfac.ft, &pSTimes->ac);
				FileTimeToSystemTime(&lfup.ft, &pSTimes->up);
			}
		}
		//	ＵＴＣ取得
		else {
			//	戻り情報(ＵＴＣ通算時刻)設定
			if (pFTimes != NULL) {
				pFTimes->mk.v = sfmk.v;
				pFTimes->ac.v = sfac.v;
				pFTimes->up.v = sfup.v;
			}
			//	戻り情報(ＵＴＣ日時)設定
			if (pSTimes != NULL) {
				FileTimeToSystemTime(&sfmk.ft, &pSTimes->mk);
				FileTimeToSystemTime(&sfac.ft, &pSTimes->ac);
				FileTimeToSystemTime(&sfup.ft, &pSTimes->up);
			}
		}
		rc = TRUE;
	}
	return rc;
}

//==============================================================================================================//
//	ファイル／ディレクトリのタイムスタンプ設定（通算時刻で設定）												//
//																												//
//	引　数	：	pPath		- ファイル／ディレクトリのパス名													//
//				pFTimes		- 設定するタイムスタンプ（通算時刻）												//
//				fLocalTime	- TRUE	: ローカルタイムで設定														//
//							  FALSE : ＵＴＣで設定																//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
static	BOOL	SubSetTimeStamp(HANDLE hFile, PCAJCFTIMES pFTimes, BOOL fLocalTime);
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcSetFileTimeA(C_BCP pPath, PCAJCFTIMES pFTimes, BOOL fLocalTime)
{
	BOOL	rc = FALSE;
	HANDLE	hFile;

	if (pPath != NULL && pFTimes != NULL) {
		if ((hFile = CreateFileA(pPath,
								 GENERIC_WRITE,
								 FILE_SHARE_READ | FILE_SHARE_WRITE,
								 NULL,
								 OPEN_EXISTING,
								 AjcPathIsDirectoryA(pPath) ? FILE_FLAG_BACKUP_SEMANTICS : FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE) {
			rc = SubSetTimeStamp(hFile, pFTimes, fLocalTime);
			CloseHandle(hFile);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcSetFileTimeW(C_WCP pPath, PCAJCFTIMES pFTimes, BOOL fLocalTime)
{
	BOOL	rc = FALSE;
	HANDLE	hFile;

	if (pPath != NULL) {
		if ((hFile = CreateFileW(pPath,
								 GENERIC_WRITE,
								 FILE_SHARE_READ | FILE_SHARE_WRITE,
								 NULL,
								 OPEN_EXISTING,
								 AjcPathIsDirectoryW(pPath) ? FILE_FLAG_BACKUP_SEMANTICS : FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE) {
			rc = SubSetTimeStamp(hFile, pFTimes, fLocalTime);
			CloseHandle(hFile);
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	SubSetTimeStamp(HANDLE hFile, PCAJCFTIMES pFTimes, BOOL fLocalTime)
{
	BOOL	rc = FALSE;
	AJCUFT	sfmk, sfac, sfup;		//	設定ファイルタイム（ＵＴＣ）

	if (fLocalTime) {
		//	設定するファイルタイム（ローカルタイム→ＵＴＣ変換）
		LocalFileTimeToFileTime(&pFTimes->mk.ft, &sfmk.ft);
		LocalFileTimeToFileTime(&pFTimes->ac.ft, &sfac.ft);
		LocalFileTimeToFileTime(&pFTimes->up.ft, &sfup.ft);
	}
	else {
		//	設定するファイルタイム
		sfmk.v = pFTimes->mk.v;
		sfac.v = pFTimes->ac.v;
		sfup.v = pFTimes->up.v;
	}
	//	ファイルタイム設定
	rc = ((SetFileTime(hFile, &sfmk.ft, &sfac.ft, &sfup.ft)) != 0);

	return rc;
}
//==============================================================================================================//
//	ファイル／ディレクトリのタイムスタンプ設定（日時で設定）													//
//																												//
//	引　数	：	pPath		- ファイル／ディレクトリのパス名													//
//				pSTime		- 設定するタイムスタンプ（日時）													//
//				fLocalTime	- TRUE	: ローカルタイムで設定														//
//							  FALSE : ＵＴＣで設定																//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
static	BOOL	SubSetTimeStampBySysTime(HANDLE hFile, LPSYSTEMTIME pSTime, BOOL fLocalTime);
static	VO		SubAdjustUru(LPSYSTEMTIME pSTime);
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcSetFileTimeBySysTimeA(C_BCP pPath, LPSYSTEMTIME pSTime, BOOL fLocalTime)
{
	BOOL	rc = FALSE;
	HANDLE	hFile;

	if (pPath != NULL && pSTime != NULL) {
		if ((hFile = CreateFileA(pPath,
								 GENERIC_READ | GENERIC_WRITE,
								 FILE_SHARE_READ | FILE_SHARE_WRITE,
								 NULL,
								 OPEN_EXISTING,
								 AjcPathIsDirectoryA(pPath) ? FILE_FLAG_BACKUP_SEMANTICS : FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE) {
			rc = SubSetTimeStampBySysTime(hFile, pSTime, fLocalTime);
			CloseHandle(hFile);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcSetFileTimeBySysTimeW(C_WCP pPath, LPSYSTEMTIME pSTime, BOOL fLocalTime)
{
	BOOL	rc = FALSE;
	HANDLE	hFile;

	if (pPath != NULL && pSTime != NULL) {
		if ((hFile = CreateFileW(pPath,
								 GENERIC_READ | GENERIC_WRITE,
								 FILE_SHARE_READ | FILE_SHARE_WRITE,
								 NULL,
								 OPEN_EXISTING,
								 AjcPathIsDirectoryW(pPath) ? FILE_FLAG_BACKUP_SEMANTICS : FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE) {
			rc = SubSetTimeStampBySysTime(hFile, pSTime, fLocalTime);
			CloseHandle(hFile);
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	SubSetTimeStampBySysTime(HANDLE hFile, LPSYSTEMTIME pSTime, BOOL fLocalTime)
{
	BOOL		rc = FALSE;
	AJCSTIMES	stm;					//	元ファイルのタイムスタンプ
	AJCSTIMES	dtm;					//	設定するタイムスタンプ
	AJCFTIMES	ftm;					//	ファイルタイム（指定値）
	AJCFTIMES	utc;					//	ファイルタイム（ＵＴＣ）

	do {
		if (!SubGetTimeStamp(hFile, NULL, &stm, fLocalTime)) break;
		//	設定する日時情報をコピー
		memcpy(&dtm.mk, pSTime, sizeof dtm.mk);
		memcpy(&dtm.ac, pSTime, sizeof dtm.ac);
		memcpy(&dtm.up, pSTime, sizeof dtm.up);
		//	設定日時情報作成（-1の項目は元ファイルの値を採用）
		if (pSTime->wYear		  == 0xFFFF) {dtm.mk.wYear		   = stm.mk.wYear;		   dtm.ac.wYear 		= stm.ac.wYear; 		dtm.up.wYear		 = stm.up.wYear;		}
		if (pSTime->wMonth		  == 0xFFFF) {dtm.mk.wMonth 	   = stm.mk.wMonth; 	   dtm.ac.wMonth		= stm.ac.wMonth;		dtm.up.wMonth		 = stm.up.wMonth;		}
		if (pSTime->wDay		  == 0xFFFF) {dtm.mk.wDay		   = stm.mk.wDay;		   dtm.ac.wDay			= stm.ac.wDay;			dtm.up.wDay 		 = stm.up.wDay; 		}
		if (pSTime->wHour		  == 0xFFFF) {dtm.mk.wHour		   = stm.mk.wHour;		   dtm.ac.wHour 		= stm.ac.wHour; 		dtm.up.wHour		 = stm.up.wHour;		}
		if (pSTime->wMinute 	  == 0xFFFF) {dtm.mk.wMinute	   = stm.mk.wMinute;	   dtm.ac.wMinute		= stm.ac.wMinute;		dtm.up.wMinute		 = stm.up.wMinute;		}
		if (pSTime->wSecond 	  == 0xFFFF) {dtm.mk.wSecond	   = stm.mk.wSecond;	   dtm.ac.wSecond		= stm.ac.wSecond;		dtm.up.wSecond		 = stm.up.wSecond;		}
		if (pSTime->wMilliseconds == 0xFFFF) {dtm.mk.wMilliseconds = stm.mk.wMilliseconds; dtm.ac.wMilliseconds = stm.ac.wMilliseconds; dtm.up.wMilliseconds = stm.up.wMilliseconds;}
		//	うるう年以外の場合、２月２９日 → ２月２８日
		SubAdjustUru(&dtm.mk);
		SubAdjustUru(&dtm.ac);
		SubAdjustUru(&dtm.up);
		//	構造体をファイルタイムに変換
		if (!SystemTimeToFileTime(&dtm.mk, &ftm.mk.ft)) break;
		if (!SystemTimeToFileTime(&dtm.ac, &ftm.ac.ft)) break;
		if (!SystemTimeToFileTime(&dtm.up, &ftm.up.ft)) break;
		//	ファイルタイムをＵＴＣに変換
		if (fLocalTime) {
			if (!LocalFileTimeToFileTime(&ftm.mk.ft, &utc.mk.ft)) break;
			if (!LocalFileTimeToFileTime(&ftm.ac.ft, &utc.ac.ft)) break;
			if (!LocalFileTimeToFileTime(&ftm.up.ft, &utc.up.ft)) break;
		}
		else {
			memcpy(&utc, &ftm, sizeof utc);
		}
		//	ファイルタイムの矛盾を補正
		if (utc.ac.v < utc.mk.v) utc.ac.v = utc.mk.v;	//	アクセス ＜ 作成日時 ならば、アクセス ＝ 作成日時
		if (utc.up.v < utc.mk.v) utc.up.v = utc.mk.v;	//	更新日時 ＜ 作成日時 ならば、更新日時 ＝ 作成日時
		if (utc.ac.v < utc.up.v) utc.ac.v = utc.up.v;	//	アクセス ＜ 更新日時 ならば、アクセス ＝ 更新日時
		//	ファイルタイム設定
		if (!SetFileTime(hFile, &utc.mk.ft, &utc.ac.ft, &utc.up.ft)) break;
		//	戻り値＝正常
		rc = TRUE;
	} while (0);
	return rc;
}
//----- うるう年以外の場合、２月２９日 → ２月２８日 -----------------------------------------------------------//
static	VO	SubAdjustUru(LPSYSTEMTIME pSTime)
{
	if (!((pSTime->wYear % 4) == 0 && ((pSTime->wYear % 100) != 0 || (pSTime->wYear % 400) == 0))) {
		if (pSTime->wMonth == 2 && pSTime->wDay == 29) pSTime->wDay = 28;
	}
}
//==============================================================================================================//
//	２つのパスが同一かチェックする																				//
//																												//
//	引　数	：	pBase			- ベースパス名（NULLの場合はカレントディレクトリ）								//
//				pRel1, pRel2	- ２つのファイル／ディレクトリのパス名											//
//																												//
//	戻り値	：	≠NULL	- 一致（実際のパス名へのポインタ）														//
//				＝NULL	- 不一致																				//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	C_BCP	WINAPI	AjcIsSamePathA(C_BCP pBase, C_BCP pRel1, C_BCP pRel2)
{
		BCP		rc = NULL;
		int		stl;
		BC		szBase[MAX_PATH]	 = {0};
		BC		szRel1[MAX_PATH * 2] = {0};
		BC		szRel2[MAX_PATH * 2] = {0};
static	BC		szAbs1[MAX_PATH * 2] = {0};
		BC		szAbs2[MAX_PATH * 2] = {0};

	if (pRel1 != NULL && pRel2 != NULL) {
		memset(szAbs1, 0, sizeof(szAbs1));
		//	２つのパス名退避
		AjcSnPrintFA(szRel1, AJCTSIZE(szRel1), "%s", pRel1);
		AjcSnPrintFA(szRel2, AJCTSIZE(szRel2), "%s", pRel2);
		//	２つのパスの末尾「\」を合わせる
		stl = (int)strlen(pRel1);
		if (stl >= 1 && szRel1[stl - 1] == '\\') AjcPathCatA		 (szRel2, "", AJCTSIZE(szRel2));
		else									 PathRemoveBackslashA(szRel2);
		do {
			//	ベースパス設定
			if (pBase != NULL) AjcSnPrintFA(szBase, AJCTSIZE(szBase), "%s", pBase);
			else			   GetCurrentDirectoryA(AJCTSIZE(szBase), szBase);
			//	絶対パス１作成
			if (PathCombineA(szAbs1, szBase, szRel1) == NULL) {
				break;
			}
			//	絶対パス２作成
			if (PathCombineA(szAbs2, szBase, szRel2) == NULL) {
				break;
			}
			//	比較
			if (stricmp(szAbs1, szAbs2) == 0) {
				rc = szAbs1;
			}
		} while(0);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	C_WCP	WINAPI	AjcIsSamePathW(C_WCP pBase, C_WCP pRel1, C_WCP pRel2)
{
		WCP		rc = NULL;
		int		stl;
		WC		szBase[MAX_PATH]	 = {0};
		WC		szRel1[MAX_PATH * 2] = {0};
		WC		szRel2[MAX_PATH * 2] = {0};
static	WC		szAbs1[MAX_PATH * 2] = {0};
		WC		szAbs2[MAX_PATH * 2] = {0};

	if (pRel1 != NULL && pRel2 != NULL) {
		memset(szAbs1, 0, sizeof(szAbs1));
		//	２つのパス名退避
		AjcSnPrintFW(szRel1, AJCTSIZE(szRel1), L"%s", pRel1);
		AjcSnPrintFW(szRel2, AJCTSIZE(szRel2), L"%s", pRel2);
		//	２つのパスの末尾「\」を合わせる
		stl = (int)wcslen(pRel1);
		if (stl >= 1 && szRel1[stl - 1] == L'\\') AjcPathCatW		  (szRel2, L"", AJCTSIZE(szRel2));
		else									  PathRemoveBackslashW(szRel2);
		do {
			//	ベースパス設定
			if (pBase != NULL) AjcSnPrintFW(szBase, AJCTSIZE(szBase), L"%s", pBase);
			else			   GetCurrentDirectoryW(AJCTSIZE(szBase), szBase);
			//	絶対パス１作成
			if (PathCombineW(szAbs1, szBase, szRel1) == NULL) {
				break;
			}
			//	絶対パス２作成
			if (PathCombineW(szAbs2, szBase, szRel2) == NULL) {
				break;
			}
			//	比較
			if (wcsicmp(szAbs1, szAbs2) == 0) {
				rc = szAbs1;
			}
		} while(0);
	}
	return rc;
}
//==============================================================================================================//
//	２つのパスで、片方がサブパスであるかをチェックする															//
//																												//
//	引　数	：	pBase		- ベースパス（NULL / 絶対パス）」													//
//				pUpperPath	- 上位パス							（pBase=NULL時は絶対パス）						//
//				pUnderPath	- 下位パスかチェックするフォルダパス（pBase=NULL時は絶対パス）						//
//																												//
//	戻り値	：	TRUE	- 下位パスは上位パスのサブパスである（同一パスを含む）									//
//				FALSE	- 下位パスは上位パスのサブフパスでない													//
//				-1		- エラー（絶対パスでない場合を含む）													//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcIsUnderPathA(C_BCP pBase, C_BCP pUpperPath, C_BCP pUnderPath)
{
	BOOL	rc = -1;
	int		stlUpper;
	int		stlUnder;
	BC		szUpper[MAX_PATH];
	BC		szUnder[MAX_PATH];

	if (pUpperPath != NULL && pUnderPath != NULL) {
		do {
			//	パスを正規化
			if (PathCombineA(szUpper, pBase, pUpperPath) == NULL) break;
			if (PathCombineA(szUnder, pBase, pUnderPath) == NULL) break;
			//	上位パスの末尾の「\」削除
			PathRemoveBackslashA(szUpper);
			//	パスの文字列長設定
			stlUpper = (int)strlen(szUpper);
			stlUnder = (int)strlen(szUnder);
			//	上位パス長が短ければ、重複チェック
			if (stlUpper <= stlUnder) {
				//	パスの先頭部分が一致＆下位パスの次文字が'\'/0ならば重複あり
				if (strnicmp(szUpper, szUnder, stlUpper) == 0 && (szUnder[stlUpper] == '\\' || szUnder[stlUpper] == 0)) {
					rc = TRUE;
				}
				//	パスの先頭部分が不一致ならば、重複なし
				else {
					rc = FALSE;
				}
			}
			//	下位パス長が短い場合は、重複なし
			else {
				rc = FALSE;
			}
		} while(0);
	}

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcIsUnderPathW(C_WCP pBase, C_WCP pUpperPath, C_WCP pUnderPath)
{
	BOOL	rc = -1;
	int		stlUpper;
	int		stlUnder;
	WC		szUpper[MAX_PATH];
	WC		szUnder[MAX_PATH];

	if (pUpperPath != NULL && pUnderPath != NULL) {
		do {
			//	パスを正規化
			if (PathCombineW(szUpper, pBase, pUpperPath) == NULL) break;
			if (PathCombineW(szUnder, pBase, pUnderPath) == NULL) break;
			//	上位パスの末尾の「\」削除
			PathRemoveBackslashW(szUpper);
			//	パスの文字列長設定
			stlUpper = (int)wcslen(szUpper);
			stlUnder = (int)wcslen(szUnder);
			//	上位パス長が短ければ、重複チェック
			if (stlUpper <= stlUnder) {
				//	パスの先頭部分が一致＆下位パスの次文字が'\'/0ならば重複あり
				if (wcsnicmp(szUpper, szUnder, stlUpper) == 0 && (szUnder[stlUpper] == L'\\' || szUnder[stlUpper] == 0)) {
					rc = TRUE;
				}
				//	パスの先頭部分が不一致ならば、重複なし
				else {
					rc = FALSE;
				}
			}
			//	下位パス長が短い場合は、重複なし
			else {
				rc = FALSE;
			}
		} while(0);
	}

	return rc;
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
//																												//
//	ＭＳＩＬ用サポートコード																					//
//																												//
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
typedef struct {
	UX	cbp;
	BOOL (CALLBACK *cbEnum )(BOOL fValid1, UI utc1, UI att1, ULL size1, C_WCP pPath1, C_WCP pName1, C_WCP pTail1,
							 BOOL fValid2, UI utc2, UI att2, ULL size2, C_WCP pPath2, C_WCP pName2, C_WCP pTail2, UX cbp);
} FMLPARAM, *PFMLPARAM;

//	コールバック
static	BOOL	CALLBACK cbFmlEnum(AJCFMLITEMW item[2], UX cbp)
{
	BOOL		rc = TRUE;
	PFMLPARAM	p  = (PFMLPARAM)cbp;

	if (p->cbEnum != NULL) {
		rc = p->cbEnum(item[0].fValid, item[0].hd.utc, item[0].hd.att, item[0].hd.size, item[0].path, item[0].name, item[0].tail,
					   item[1].fValid, item[1].hd.utc, item[1].hd.att, item[1].hd.size, item[1].path, item[1].name, item[1].tail, cbp);
	}
	return rc;
}
//	ＡＰＩ
AJCEXPORT	BOOL	WINAPI	AjcEnumFileMatchingListMsil(C_WCP pPath1, C_WCP pPath2, UI opt, UX cbp, 
					BOOL (CALLBACK *cbQuery)(UI id, C_WCP pPath, UI att, BOOL *fDiscard, UX cbp),
					BOOL (CALLBACK *cbEnum )(BOOL fValid1, UI utc1, UI att1, ULL size1, C_WCP pPath1, C_WCP pName1, C_WCP pTail1,
											 BOOL fValid2, UI utc2, UI att2, ULL size2, C_WCP pPath2, C_WCP pName2, C_WCP pTail2, UX cbp))
{
	BOOL		rc = FALSE;
	FMLPARAM	prm;
	prm.cbp 	= cbp;
	prm.cbEnum	= cbEnum;

	rc = AjcEnumFilesMatchingListW(pPath1, pPath2, opt, (UX)&prm, cbQuery, cbFmlEnum);

	return rc;
}


