﻿#include	"AjcInternal.h"

//--------------------------------------------------------------------------------------------------------------//
//	フォントの高さの最大値，最小値																				//
//--------------------------------------------------------------------------------------------------------------//
#define		MAX_FONT_PIXELS		192
#define		MIN_FONT_PIXELS		8

#define		MAX_FONT_POINTS		144
#define		MIN_FONT_POINTS		6


#define		WM_AJCSHOWPREVIEWTEXT	(WM_APP + 10)

//--------------------------------------------------------------------------------------------------------------//
//	フォント設定ダイアログワーク																				//
//--------------------------------------------------------------------------------------------------------------//
typedef struct {
	//	引数
	UI				opt;
	PAJCCFFONTINFOW	pBuf;
	PAJCCFPERMINFOW	pPerm;
	UX				cbp;
	VO	 (CALLBACK *cbInit )(HWND hDlg, UX cbp);
	BOOL (CALLBACK *cbApply)(HFONT hFont, UI lsp, UX cbp);
	//	永続化情報
	AJCCFPERMINFOW	Perm;
	//	ワーク
	HWND		hDlg;						//	ダイアログハンドル
	HWND		hCbo;						//	コンボボックス・ハンドル
	HWND		hVth;						//	プレビューテキストハンドル
	int			SvCboIx;					//	コンボボックス・インデクス退避
	int			DlgMinW;					//	ダイアログ最小幅
	int			DlgMinH;					//	ダイアログ最小高
	int			VthTop;						//	ＶＴＨの上位置
	BOOL		fCtrl;						//	CTRLキー状態
	BOOL		fShowPreview;				//	プレビュー表示フラグ
} DLGFONTWK, *PDLGFONTWK;

//--------------------------------------------------------------------------------------------------------------//
//	プレビューテキスト																							//
//--------------------------------------------------------------------------------------------------------------//
static	const WC	szPreView[] =	L"abcdefghijklmnopqrstuvwxyz\n"
									L"ABCDEFGHIJKLMNOPQRSTUVWXYZ\n"
									L"ａｂｃｄｅｆｇｈｉｊｋｌｍｎｏｐｑｒｓｔｕｖｗｘｙｚ\n"
									L"ＡＢＣＤＥＦＧＨＩＪＫＬＭＮＯＰＱＲＳＴＵＶＷＸＹＺ\n"
									L"This is sample text. Please refer to the font settings.\n"
									L"これはサンプルテキストです。フォント設定の参考にしてください。\n"
									L"이것은 샘플 텍스트입니다. 폰트 설정의 참고로 해 주세요.\n"
									L"Ceci est un exemple de texte. Veuillez vous y référer lors de la définition de la police.";

static	WCP			pPreView = NULL;		//	プレビューテキストへのポインタ
static	int			MsgShowPreview = 0;		//	プレビューテキスト表示メッセージ

//--------------------------------------------------------------------------------------------------------------//
//	作業領域																									//
//--------------------------------------------------------------------------------------------------------------//
static	int			DPI         = 96;			//	論理インチあたりのピクセル数
static	const int	PPI			= 72;			//	論理インチあたりのポイント数
static	HAJCAVS		hAvsFontList = NULL;		//	フォント情報格納ＡＶＳ

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
AJC_DLGPROC_DEF(SetFontDlg);

extern	UI				CorrectAllFonts(HAJCAVS hAvs);

static	VO				MoveVthWindow(HWND hDlg);

static	BOOL CALLBACK	SfgNtcFont(C_WCP pFaceName, UB PitchAndFamily, UB CharSet, UX cbp);
static	BOOL			SfgGetFontInfo(HWND hDlg, LPLOGFONTW pLf);
static	VO				SfgSelFontInCbo(HWND hDlg);
static	VO				SfgSaveFontInfo(HWND hDlg, LPLOGFONTW pLf);
static	VO				SfgShowKindOfPitch(HWND hDlg);
static	VO				SfgSaveToProfile(HWND hDlg);
static	VO				SfgSetPreviewFont(HWND hDlg);
static	VO				SfgShowPreview(HWND hDlg);
static	VO				SfgPostShowPreviewMsg(VO);

static	VO				SubFaceCpyA(C_BCP pS, BCP pD, UI len);
static	VO				SubFaceCpyW(C_WCP pS, WCP pD, UI len);

//==============================================================================================================//
//	起動時初期設定																								//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
BOOL	AjcCfInit(VO)
{
	BOOL		rc	= FALSE;
	UI			stl = 0;
	HDC			hic;

	do {
		//	ＤＰＩ値設定
		hic = CreateICA("DISPLAY", NULL, NULL, NULL);
		DPI = GetDeviceCaps(hic, LOGPIXELSX);
		DeleteDC(hic);

		//	初期のプレビューテキスト設定
		stl = (UI)wcslen(szPreView);
		if ((pPreView = AjcTAllocW(stl + 1)) == NULL) break;
		wcscpy(pPreView, szPreView);

		rc = TRUE;

	} while(0);

	return rc;
}
//==============================================================================================================//
//	終了時後処理																								//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
VO		AjcCfEnd (VO)
{
	if (pPreView     != NULL) {AjcTFree(pPreView);			pPreView     = NULL;}
	if (hAvsFontList != NULL) {AjcAvsDelete(hAvsFontList);	hAvsFontList = NULL;}
}
//==============================================================================================================//
//	英字名を含む全フォント情報の生成																			//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT HAJCAVS	WINAPI	AjcCfCreateAllFontInfo(VO)
{
	HAJCAVS	rc = NULL;

	//	未生成ならば、生成する
	if (hAvsFontList == NULL) {
		if ((hAvsFontList = AjcAvsCreate(AJCCMP_IGNORE_WIDTH, 0, NULL)) != NULL) {
			if (CorrectAllFonts(hAvsFontList) != 0) {
				rc = hAvsFontList;
			}
			else {
				AjcAvsDelete(hAvsFontList);
				hAvsFontList = NULL;
			}
		}
	}
	//	生成済ならば、戻り値＝正常
	else {
		rc = hAvsFontList;
	}
	return rc;
}

//==============================================================================================================//
//	英字名を含む全フォント情報の解放																			//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT VO	WINAPI	AjcCfReleaseAllFontInfo(VO)
{
	if (hAvsFontList != NULL) {
		AjcAvsDelete(hAvsFontList);
		hAvsFontList = NULL;
	}
}
//==============================================================================================================//
//	ポイント数→ピクセル数変換																					//
//																												//
//	引　数	：	points	- ポイント数																			//
//																												//
//	戻り値	：	ピクセル数																						//
//==============================================================================================================//
AJCEXPORT int	WINAPI	AjcCfPointsToPixels(int points)
{
	int		pixels = 0;

	if (points > 0) {
		pixels = points * DPI / PPI;
	}
	return pixels;
}
//==============================================================================================================//
//	ピクセル数→ポイント数変換																					//
//																												//
//	引　数	：	pixels	- ピクセル数数（負数も可）																//
//																												//
//	戻り値	：	ピクセル数																						//
//==============================================================================================================//
AJCEXPORT int	WINAPI	AjcCfPixelsToPoints(int pixels)
{
	int		points = 0;
	int		pixs   = abs(pixels);

	if (pixs > 0) {
		points = (pixs * PPI / DPI) + (((pixs * PPI) % DPI) >= (DPI / 2));
	}
	return points;
}

//==============================================================================================================//
//	プレビューテキスト設定																						//
//																												//
//	引　数	：	pText	- プレビューテキストへのポインタ														//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	WINAPI	AjcCfSetPreviewTextA(C_BCP pText)
{
	BOOL	rc	 = FALSE;
	WCP		pTmp = NULL;
	UI		len;

	do {
		if (pText != NULL) {
			len = MultiByteToWideChar(CP_ACP, 0, pText, -1, NULL, 0);
			if (pTmp = AjcTAlloc(len)) {
				MultiByteToWideChar(CP_ACP, 0, pText, -1, pTmp, len);
			}
		}
		AjcCfSetPreviewTextW(pTmp);
	} while(0);

	if (pTmp != NULL) AjcTFree(pTmp);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	WINAPI	AjcCfSetPreviewTextW(C_WCP pText)
{
	BOOL	rc = FALSE;
	WCP		p = NULL;
	UI		stl;

	if (pText != NULL) {
		stl = (UI)wcslen(pText);
		if (p = AjcTAllocW(stl + 1)) {
			//	旧テキスト解放
			if (pPreView != NULL) {
				AjcTFree(pPreView);
			}
			//	新テキスト設定
			pPreView = p;
			wcscpy(pPreView, pText);
			//	テキスト表示メッセージ送出
			SfgPostShowPreviewMsg();
			rc = TRUE;
		}
	}
	else {
		if (pPreView != NULL) {
			AjcTFree(pPreView);
			pPreView = NULL;
		}
	}
	return rc;
}

//==============================================================================================================//
//	フォント名検索																								//
//																												//
//	引　数	：	pFaceName	- 検索するフォントフェース名へのポインタ											//
//				pBuf		- ピッチ＆ファミリ，キャラクタセット情報を格納するバッファのアドレス				//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - 見つからない																			//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
typedef struct {
	BOOL		rc;
	C_BCP		pFaceName;
	PAJCCFATT	pBuf;
} CBFINDPRMA, *PCBFINDPRMA;
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	BOOL CALLBACK cbFindNtcFontA(C_BCP pFaceName, UB CharSet, UB PitchAndFamily, UX cbp)
{
	BOOL		rc = TRUE;
	PCBFINDPRMA	p  = (PCBFINDPRMA)cbp;

	if (_mbsicmp(pFaceName, p->pFaceName) == 0) {
		p->rc = TRUE;
		p->pBuf->s.PitchAndFamily = PitchAndFamily;
		p->pBuf->s.CharSet		  = CharSet;
		rc = FALSE;
	}
	return rc;
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT BOOL	WINAPI	AjcCfFindFontA(C_BCP pFaceName, PAJCCFATT pBuf)
{
	CBFINDPRMA	prm;

	prm.rc			= FALSE;
	prm.pFaceName	= pFaceName;
	prm.pBuf		= pBuf;
	AjcCfEnumFontsA((UX)&prm, cbFindNtcFontA);

	return prm.rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
typedef struct {
	BOOL		rc;
	C_WCP		pFaceName;
	PAJCCFATT	pBuf;
} CBFINDPRMW, *PCBFINDPRMW;
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	BOOL CALLBACK cbFindNtcFontW(C_WCP pFaceName, UB CharSet, UB PitchAndFamily, UX cbp)
{
	BOOL		rc = TRUE;
	PCBFINDPRMW	p  = (PCBFINDPRMW)cbp;

	if (wcsicmp(pFaceName, p->pFaceName) == 0) {
		p->rc = TRUE;
		p->pBuf->s.PitchAndFamily = PitchAndFamily;
		p->pBuf->s.CharSet		  = CharSet;
		rc = FALSE;
	}
	return rc;
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT BOOL	WINAPI	AjcCfFindFontW(C_WCP pFaceName, PAJCCFATT pBuf)
{
	CBFINDPRMW	prm;
	prm.rc			= FALSE;
	prm.pFaceName	= pFaceName;
	prm.pBuf		= pBuf;
	AjcCfEnumFontsW((UX)&prm, cbFindNtcFontW);

	return prm.rc;
}
//==============================================================================================================//
//	フォント名検索（英字名を含む全フォント検索）																//
//																												//
//	引　数	：	pFaceName	- 検索するフォントフェース名へのポインタ											//
//																												//
//	戻り値	：	≠NULL  - ＯＫ（フォント属性情報へのポインタ）													//
//				＝NULL  - 見つからない																			//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT PCAJCFIFNODE	WINAPI	AjcCfFindFontExA(C_BCP pFaceName)
{
	PCAJCFIFNODE	rc = NULL;
	WCP				pTxtW;
	UI				lTxtW;

	if (pFaceName != NULL) {
		lTxtW = AjcByteCharToWideChar(pFaceName, NULL, 0);
		if (pTxtW = AjcTAlloc(lTxtW)) {
			AjcByteCharToWideChar(pFaceName, pTxtW, lTxtW);
			rc = AjcCfFindFontExW(pTxtW);
			free(pTxtW);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT PCAJCFIFNODE	WINAPI	AjcCfFindFontExW(C_WCP pFaceName)
{
	PCAJCFIFNODE	rc = NULL;

	if (AjcCfCreateAllFontInfo() != NULL) {
		if (pFaceName != NULL) {
			rc = AjcAvsGetNodePtrW(hAvsFontList, pFaceName, NULL);
		}
	}
	return rc;
}
//==============================================================================================================//
//	フォント名列挙																								//
//																												//
//	引　数	：	cbp			- コールバックパラメタ																//
//				cbNtcFont	- フォント情報通知用コールバック関数												//
//																												//
//	戻り値	：	列挙したフォント数																				//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
typedef struct {
	UI		cnt;
	UX		cbp;
	BOOL (CALLBACK *cbNtcFontA)(C_BCP pFaceName, UB CharSet, UB PitchAndFamily, UX cbp);
} CBEFPRMA, *PCBEFPRMA;
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	int CALLBACK cbEnumFontFamExProcA(ENUMLOGFONTEXA *lpelfe, NEWTEXTMETRICEXA *lpntme, DWORD FontType, LPARAM lParam)
{
	PCBEFPRMA	p = (PCBEFPRMA)lParam;
	BOOL		rc;

	p->cnt++;
	rc = p->cbNtcFontA(lpelfe->elfLogFont.lfFaceName, lpelfe->elfLogFont.lfCharSet, lpelfe->elfLogFont.lfPitchAndFamily, p->cbp);

	return rc;
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT UI	WINAPI	AjcCfEnumFontsA(UX cbp, BOOL (CALLBACK *cbNtcFont)(C_BCP pFaceName, UB CharSet, UB PitchAndFamily, UX cbp))
{
	LOGFONTA	lf;
	CBEFPRMA	fwk = {0};
	HDC			hdc;

	hdc = GetDC(NULL);

	fwk.cbp		   = cbp;
	fwk.cbNtcFontA = cbNtcFont;

	memset(&lf, 0, sizeof lf);
	lf.lfCharSet   = DEFAULT_CHARSET;
	EnumFontFamiliesExA(hdc, &lf, (FONTENUMPROCA)cbNtcFont, (LPARAM)&fwk, 0);

	if (hdc != NULL) ReleaseDC(NULL, hdc);

	return fwk.cnt;
}

//----- ワイド文字 ---------------------------------------------------------------------------------------------//
typedef struct {
	UI		cnt;
	UX		cbp;
	BOOL (CALLBACK *cbNtcFontW)(C_WCP pFace, UB CharSet, UB PitchAndFamily, UX cbp);
} CBEFPRMW, *PCBEFPRMW;
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	int CALLBACK cbEnumFontFamExProcW(ENUMLOGFONTEXW *lpelfe, NEWTEXTMETRICEXW *lpntme, DWORD FontType, LPARAM lParam)
{
	PCBEFPRMW	p = (PCBEFPRMW)lParam;
	BOOL		rc;

	p->cnt++;
	rc = p->cbNtcFontW(lpelfe->elfLogFont.lfFaceName, lpelfe->elfLogFont.lfCharSet, lpelfe->elfLogFont.lfPitchAndFamily, p->cbp);

	return rc;
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT UI	WINAPI	AjcCfEnumFontsW(UX cbp, BOOL (CALLBACK *cbNtcFont)(C_WCP pFaceName, UB CharSet, UB PitchAndFamily, UX cbp))
{
	LOGFONTW	lf;
	CBEFPRMW	fwk = {0};

	HDC			hdc;

	hdc = GetDC(NULL);

	fwk.cbp		   = cbp;
	fwk.cbNtcFontW = cbNtcFont;

	memset(&lf, 0, sizeof lf);
	lf.lfCharSet   = DEFAULT_CHARSET;
	EnumFontFamiliesExW(hdc, &lf, (FONTENUMPROCW)cbEnumFontFamExProcW, (LPARAM)&fwk, 0);

	if (hdc != NULL) ReleaseDC(NULL, hdc);

	return fwk.cnt;
}
//==============================================================================================================//
//	フォント名列挙（英字名を含む全フォントの列挙）																//
//																												//
//	引　数	：	cbp			- コールバックパラメタ																//
//				cbNtcFont	- フォント情報通知用コールバック関数												//
//																												//
//	戻り値	：	列挙したフォント数																				//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
typedef struct {
	UI		cnt;
	UX		cbp;
	BOOL (CALLBACK *cbNtcFontA)(C_BCP pFaceName, PCAJCFIFNODE pNode, UX cbp);
} CBEFEXPRMA, *PCBEFEXPRMA;
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	BOOL	CALLBACK cbNtcNodeExA(C_WCP pKey, VOP pNodeData, UI len, UI nest, UX cbp)
{
	BOOL		rc = FALSE;
	PCBEFEXPRMA	p = (PCBEFEXPRMA)cbp;
	BCP			pTxt;
	UI			lTxt;

	lTxt = WideCharToMultiByte(CP_ACP, 0, pKey, -1, NULL, 0, NULL, NULL);
	if (lTxt != 0 && (pTxt = (BCP)AJCMEM(lTxt))) {
		WideCharToMultiByte(CP_ACP, 0, pKey, -1, pTxt, lTxt, NULL, NULL);
		p->cnt++;
		rc = p->cbNtcFontA(pTxt, (PCAJCFIFNODE)pNodeData, p->cbp);
		free(pTxt);
	}
	return rc;
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT UI	WINAPI	AjcCfEnumFontsExA(UX cbp, BOOL (CALLBACK *cbNtcFontA)(C_BCP pFaceName, PCAJCFIFNODE pNode, UX cbp))
{
	CBEFEXPRMA	fwk = {0};

	fwk.cnt			= 0;
	fwk.cbp			= cbp;
	fwk.cbNtcFontA  = cbNtcFontA;

	if (AjcCfCreateAllFontInfo() != NULL) {
		AjcAvsEnumNodesExW(hAvsFontList, (UX)&fwk, cbNtcNodeExA, FALSE);
	}
	return fwk.cnt;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
typedef struct {
	UI		cnt;
	UX		cbp;
	BOOL (CALLBACK *cbNtcFontW)(C_WCP pFaceName, PCAJCFIFNODE pNode, UX cbp);
} CBEFEXPRMW, *PCBEFEXPRMW;
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	BOOL	CALLBACK cbNtcNodeExW(C_WCP pKey, VOP pNodeData, UI len, UI nest, UX cbp)
{
	BOOL		rc = FALSE;
	PCBEFEXPRMW	p = (PCBEFEXPRMW)cbp;

	p->cnt++;
	rc = p->cbNtcFontW(pKey, (PCAJCFIFNODE)pNodeData, p->cbp);

	return rc;
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT UI	WINAPI	AjcCfEnumFontsExW(UX cbp, BOOL (CALLBACK *cbNtcFontW)(C_WCP pFaceName, PCAJCFIFNODE pNode, UX cbp))
{
	CBEFEXPRMW	fwk = {0};
	fwk.cnt			= 0;
	fwk.cbp			= cbp;
	fwk.cbNtcFontW  = cbNtcFontW;

	if (AjcCfCreateAllFontInfo() != NULL) {
		AjcAvsEnumNodesExW(hAvsFontList, (UX)&fwk, cbNtcNodeExW, FALSE);
	}
	return fwk.cnt;
}
//==============================================================================================================//
//	フォント情報初期化																							//
//																												//
//	引　数	：	pFace	- フェース名へのポインタ（不要時はNULL)													//
//				pFont	- 設定するフォント情報へのポインタ														//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	WINAPI	AjcCfInitFontInfoA(C_BCP pFace, PAJCCFFONTINFOA pFont)
{
	BOOL		rc = FALSE;
	AJCCFATT	att;

	if (pFont != NULL) {
		//	フォント情報初期化
		memset(&pFont->lf, 0, sizeof pFont->lf);
		pFont->lf.lfHeight		   = 12;
		pFont->lf.lfCharSet 	   = DEFAULT_CHARSET;
		pFont->lf.lfPitchAndFamily = FIXED_PITCH;
		strcpy(pFont->lf.lfFaceName, LNGSEL("ＭＳ ゴシック", "MS Gothic"));
		pFont->lsp = 0;
		//	フォント検索＆設定
		if (pFace != NULL && AjcCfFindFontA(pFace, &att)) {
			memset(&pFont->lf, 0, sizeof pFont->lf);
			pFont->lf.lfHeight		   = 12;
			pFont->lf.lfCharSet 	   = att.s.CharSet;
			pFont->lf.lfPitchAndFamily = att.s.PitchAndFamily;
			AjcSnPrintFA(pFont->lf.lfFaceName, LF_FACESIZE, "%s", pFace);
		}
		rc = TRUE;
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	WINAPI	AjcCfInitFontInfoW(C_WCP pFace, PAJCCFFONTINFOW pFont)
{
	BOOL		rc = FALSE;
	AJCCFATT	att;

	if (pFont != NULL) {
		//	フォント情報初期化
		memset(&pFont->lf, 0, sizeof pFont->lf);
		pFont->lf.lfHeight		   = 12;
		pFont->lf.lfCharSet 	   = DEFAULT_CHARSET;
		pFont->lf.lfPitchAndFamily = FIXED_PITCH;
		wcscpy(pFont->lf.lfFaceName, LNGSEL(L"ＭＳ ゴシック", L"MS Gothic"));
		pFont->lsp = 0;
		//	フォント検索＆設定
		if (pFace != NULL && AjcCfFindFontW(pFace, &att)) {
			memset(&pFont->lf, 0, sizeof pFont->lf);
			pFont->lf.lfHeight		   = 12;
			pFont->lf.lfCharSet 	   = att.s.CharSet;
			pFont->lf.lfPitchAndFamily = att.s.PitchAndFamily;
			AjcSnPrintFW(pFont->lf.lfFaceName, LF_FACESIZE, L"%s", pFace);
		}
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	ダイアログ設定値（永続化情報）初期化																		//
//																												//
//	引　数	：	pFixFace	- 固定ピッチフェース名へのポインタ（不要時はNULL)									//
//				pAnyFace	- 任意ピッチフェース名へのポインタ（不要時はNULL)									//
//				pPerm		- 設定するフォント選択ダイアログ情報へのポインタ									//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	WINAPI	AjcCfInitPermInfoA(C_BCP pFixFace, C_BCP pAnyFace, PAJCCFPERMINFOA pPerm)
{
	BOOL	rc = FALSE;
	AJCCFATT	att;

	if (pPerm != NULL) {
		//	フォント選択ダイアログ情報初期化
		pPerm->hd.id = AJCCFPERMID;
		pPerm->hd.Pixels = 12;
		pPerm->hd.Points = 12;
		//	固定ピッチフォント初期化
		pPerm->hd.FixAttr.s.CharSet		   = DEFAULT_CHARSET;
		pPerm->hd.FixAttr.s.PitchAndFamily = FIXED_PITCH;
		strcpy(pPerm->FixFace, LNGSEL("ＭＳ ゴシック", "MS Gothic"));
		//	任意ピッチフォント初期化
		pPerm->hd.AnyAttr.s.CharSet		   = DEFAULT_CHARSET;
		pPerm->hd.AnyAttr.s.PitchAndFamily = FF_MODERN | VARIABLE_PITCH;
		strcpy(pPerm->AnyFace, "MS UI Gothic");
		//	固定ピッチフォント検索＆設定
		if ((pFixFace != NULL) && AjcCfFindFontA(pFixFace, &att) && ((att.s.PitchAndFamily & 3) == 1)) {
			pPerm->hd.FixAttr.s.CharSet 	   = att.s.CharSet;
			pPerm->hd.FixAttr.s.PitchAndFamily = att.s.PitchAndFamily;
			AjcSnPrintFA(pPerm->FixFace, LF_FACESIZE, "%s", pFixFace);
			rc = TRUE;
		}
		//	任意ピッチフォント検索＆設定
		if (pAnyFace != NULL && AjcCfFindFontA(pAnyFace, &att)) {
			pPerm->hd.AnyAttr.s.CharSet 	   = att.s.CharSet;
			pPerm->hd.AnyAttr.s.PitchAndFamily = att.s.PitchAndFamily;
			AjcSnPrintFA(pPerm->AnyFace, LF_FACESIZE, "%s", pAnyFace);
			rc = TRUE;
		}
	}
	

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	WINAPI	AjcCfInitPermInfoW(C_WCP pFixFace, C_WCP pAnyFace, PAJCCFPERMINFOW pPerm)
{
	BOOL	rc = FALSE;
	AJCCFATT	att;

	if (pPerm != NULL) {
		//	フォント選択ダイアログ情報初期化
		pPerm->hd.id = AJCCFPERMID;
		pPerm->hd.Pixels = 12;
		pPerm->hd.Points = 12;
		//	固定ピッチフォント初期化
		pPerm->hd.FixAttr.s.CharSet		   = DEFAULT_CHARSET;
		pPerm->hd.FixAttr.s.PitchAndFamily = FIXED_PITCH;
		wcscpy(pPerm->FixFace, LNGSEL(L"ＭＳ ゴシック", L"MS Gothic"));
		//	任意ピッチフォント初期化
		pPerm->hd.AnyAttr.s.CharSet		   = DEFAULT_CHARSET;
		pPerm->hd.AnyAttr.s.PitchAndFamily = FF_MODERN | VARIABLE_PITCH;
		wcscpy(pPerm->AnyFace, L"MS UI Gothic");
		//	固定ピッチフォント検索＆設定
		if ((pFixFace != NULL) && AjcCfFindFontW(pFixFace, &att) && ((att.s.PitchAndFamily & 3) == 1)) {
			pPerm->hd.FixAttr.s.CharSet 	   = att.s.CharSet;
			pPerm->hd.FixAttr.s.PitchAndFamily = att.s.PitchAndFamily;
			AjcSnPrintFW(pPerm->FixFace, LF_FACESIZE, L"%s", pFixFace);
			rc = TRUE;
		}
		//	任意ピッチフォント検索＆設定
		if (pAnyFace != NULL && AjcCfFindFontW(pAnyFace, &att)) {
			pPerm->hd.AnyAttr.s.CharSet 	   = att.s.CharSet;
			pPerm->hd.AnyAttr.s.PitchAndFamily = att.s.PitchAndFamily;
			AjcSnPrintFW(pPerm->AnyFace, LF_FACESIZE, L"%s", pAnyFace);
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	ダイアログ設定値（永続化情報）→文字列 変換																	//
//																												//
//	引　数	：	pFont	- フォント情報へのポインタ																//
//				pPerm	- 永続化情報へのポインタ																//
//																												//
//	戻り値	：	≠０：格納した文字列長																			//
//				＝０：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT UI	WINAPI	AjcCfInfoToTextA(PCAJCCFPERMINFOA pPerm, BC pBuf[AJCCFMAX_TEXT])
{
	return AjcCfInfoToTextExA(NULL, pPerm, pBuf);
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT UI	WINAPI	AjcCfInfoToTextW(PCAJCCFPERMINFOW pPerm, WC pBuf[AJCCFMAX_TEXT])
{
	return AjcCfInfoToTextExW(NULL, pPerm, pBuf);
}
//==============================================================================================================//
//	ダイアログ設定値（永続化情報）→文字列 変換		``（拡張版）												//
//																												//
//	引　数	：	pFont	- フォント情報へのポインタ																//
//				pPerm	- 永続化情報へのポインタ（不要時はNULL)													//
//				pBuf	- 文字列を格納するバッファへのポインタ（不要時はNULL)									//
//																												//
//	戻り値	：	≠０：格納した文字列長																			//
//				＝０：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT UI	WINAPI	AjcCfInfoToTextExA(PCAJCCFFONTINFOA pFont, PCAJCCFPERMINFOA pPerm, BC pBuf[AJCCFMAX_TEXT])
{
	UI		rc = 0;
	BC		szFont[AJCCFMAX_TEXT] = {0};
	BC		szLsp [AJCCFMAX_TEXT] = {0};
	BC		szPerm[AJCCFMAX_TEXT] = {0};

	if (pBuf != NULL) {
		*pBuf = 0;
		//	フォント情報文字列化
		if (pFont != NULL) {
			AjcLogFontToTextA((const LOGFONTA *)&pFont->lf, szFont, AJCCFMAX_TEXT, TRUE);
			AjcSnPrintFA	 (szLsp, AJCCFMAX_TEXT, "lsp=%d", pFont->lsp);
		}
		//	永続化情報文字列化
		if (pPerm != NULL) {
			AjcSnPrintFA(szPerm, AJCCFMAX_TEXT, 
				"id=0x%08X,fP=%d,fI=%d,fB=%d,fF=%d,Px=%d,Pt=%d,Ls=%d,FA=%d,AA=%d,Fix=%s,Any=%s", 
				pPerm->hd.id			,
				pPerm->hd.fPixel  != 0	,
				pPerm->hd.fItalic != 0	,
				pPerm->hd.fBold   != 0	,
				pPerm->hd.fFixed  != 0	,
				pPerm->hd.Pixels		,
				pPerm->hd.Points		,
				pPerm->hd.LSpace		,
				pPerm->hd.FixAttr.v		,
				pPerm->hd.AnyAttr.v		,
				pPerm->FixFace			,
				pPerm->AnyFace
			);
		}
		if		(pFont != NULL && pPerm != NULL) AjcSnPrintFA(pBuf, AJCCFMAX_TEXT, "F<%s>,%s,D<%s>", szFont, szLsp, szPerm);
		else if (pFont == NULL) 				 AjcSnPrintFA(pBuf, AJCCFMAX_TEXT, "D<%s>", szPerm);
		else									 *pBuf = 0;
		rc = (UI)strlen(pBuf);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT UI	WINAPI	AjcCfInfoToTextExW(PCAJCCFFONTINFOW pFont, PCAJCCFPERMINFOW pPerm, WC pBuf[AJCCFMAX_TEXT])
{
	UI		rc = 0;
	WC		szFont[AJCCFMAX_TEXT] = {0};
	WC		szLsp [AJCCFMAX_TEXT] = {0};
	WC		szPerm[AJCCFMAX_TEXT] = {0};

	if (pBuf != NULL) {
		*pBuf = 0;
		//	フォント情報文字列化
		if (pFont != NULL) {
			AjcLogFontToTextW((const LOGFONTW *)&pFont->lf, szFont, AJCCFMAX_TEXT, TRUE);
			AjcSnPrintFW	 (szLsp, AJCCFMAX_TEXT, L"lsp=%d", pFont->lsp);
		}
		//	永続化情報文字列化
		if (pPerm != NULL) {
			AjcSnPrintFW(szPerm, AJCCFMAX_TEXT, 
				L"id=0x%08X,fP=%d,fI=%d,fB=%d,fF=%d,Px=%d,Pt=%d,Ls=%d,FA=%d,AA=%d,Fix=%s,Any=%s", 
				pPerm->hd.id			,
				pPerm->hd.fPixel  != 0	,
				pPerm->hd.fItalic != 0	,
				pPerm->hd.fBold   != 0	,
				pPerm->hd.fFixed  != 0	,
				pPerm->hd.Pixels		,
				pPerm->hd.Points		,
				pPerm->hd.LSpace		,
				pPerm->hd.FixAttr.v		,
				pPerm->hd.AnyAttr.v		,
				pPerm->FixFace			,
				pPerm->AnyFace
			);
		}
		if		(pFont != NULL && pPerm != NULL) AjcSnPrintFW(pBuf, AJCCFMAX_TEXT, L"F<%s>,%s,D<%s>", szFont, szLsp, szPerm);
		else if (pFont == NULL) 				 AjcSnPrintFW(pBuf, AJCCFMAX_TEXT, L"D<%s>", szPerm);
		else									 *pBuf = 0;
		rc = (UI)wcslen(pBuf);
	}
	return rc;
}
//==============================================================================================================//
//	文字列 → ダイアログ設定値（永続化情報） 変換																//
//																												//
//	引　数	：	pTxt	- ダイアログ設定値を表す文字列															//
//				pPerm	- 永続化情報を格納するバッファへのポインタ												//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	WINAPI	AjcCfTextToInfoA(C_BCP pTxt, PAJCCFPERMINFOA pPerm)
{
	return AjcCfTextToInfoExA(pTxt, NULL, pPerm);
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	WINAPI	AjcCfTextToInfoW(C_WCP pTxt, PAJCCFPERMINFOW pPerm)
{
	return AjcCfTextToInfoExW(pTxt, NULL, pPerm);
}
//==============================================================================================================//
//	文字列 → ダイアログ設定値（永続化情報） 変換	（拡張版）													//
//																												//
//	引　数	：	pTxt	- ダイアログ設定値を表す文字列															//
//				pFont	- フォント情報を格納するバッファへのポインタ（不要時はNULL)								//
//				pPerm	- 永続化情報を格納するバッファへのポインタ（不要時はNULL)								//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	WINAPI	AjcCfTextToInfoExA(C_BCP pTxt, PAJCCFFONTINFOA pFont, PAJCCFPERMINFOA pPerm)
{
	BOOL	rc = FALSE;
	C_BCP	p;

	if (pTxt != NULL) {
		//	フォント情報
		if (pFont != NULL) {
			AjcTextToLogFontA(pTxt, &pFont->lf);
			if ((p = strstr(pTxt, "lsp=")) != NULL) pFont->lsp = AjcAscToIntA(p + 4);
		}
		//	永続化情報
		if (pPerm != NULL) {
			if ((p = strstr(pTxt, "id=" )) != NULL) pPerm->hd.id		= AjcAscToIntA(p + 3);						//	id
			if ((p = strstr(pTxt, "fP=" )) != NULL) pPerm->hd.fPixel	= AjcAscToIntA(p + 3);						//	fPixel
			if ((p = strstr(pTxt, "fI=" )) != NULL) pPerm->hd.fItalic	= AjcAscToIntA(p + 3);						//	fItalic
			if ((p = strstr(pTxt, "fB=" )) != NULL) pPerm->hd.fBold 	= AjcAscToIntA(p + 3);						//	fBold
			if ((p = strstr(pTxt, "fF=" )) != NULL) pPerm->hd.fFixed	= AjcAscToIntA(p + 3);						//	fFixed
			if ((p = strstr(pTxt, "Px=" )) != NULL) pPerm->hd.Pixels	= AjcAscToIntA(p + 3);						//	Pixels
			if ((p = strstr(pTxt, "Pt=" )) != NULL) pPerm->hd.Points	= AjcAscToIntA(p + 3);						//	Points
			if ((p = strstr(pTxt, "Ls=" )) != NULL) pPerm->hd.LSpace	= AjcAscToIntA(p + 3);						//	LSpace
			if ((p = strstr(pTxt, "FA=" )) != NULL) pPerm->hd.FixAttr.v	= AjcAscToIntA(p + 3);						//	FixAttr
			if ((p = strstr(pTxt, "AA=" )) != NULL) pPerm->hd.AnyAttr.v	= AjcAscToIntA(p + 3);						//	AnyAttr
			if ((p = strstr(pTxt, "Fix=")) != NULL) SubFaceCpyA (p + 4, pPerm->FixFace, AJCTSIZE(pPerm->FixFace));	//	FixFace
			if ((p = strstr(pTxt, "Any=")) != NULL) SubFaceCpyA (p + 4, pPerm->AnyFace, AJCTSIZE(pPerm->AnyFace));	//	AnyFace
		}
		rc = TRUE;
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	WINAPI	AjcCfTextToInfoExW(C_WCP pTxt, PAJCCFFONTINFOW pFont, PAJCCFPERMINFOW pPerm)
{
	BOOL	rc = FALSE;
	C_WCP	p;

	if (pTxt != NULL) {
		//	フォント情報
		if (pFont != NULL) {
			AjcTextToLogFontW(pTxt, &pFont->lf);
			if ((p = wcsstr(pTxt, L"lsp=")) != NULL) pFont->lsp = AjcAscToIntW(p + 4);
		}
		//	永続化情報
		if (pPerm != NULL) {
			if ((p = wcsstr(pTxt, L"id=" )) != NULL) pPerm->hd.id		 = AjcAscToIntW(p + 3);						//	id
			if ((p = wcsstr(pTxt, L"fP=" )) != NULL) pPerm->hd.fPixel	 = AjcAscToIntW(p + 3);						//	fPixel
			if ((p = wcsstr(pTxt, L"fI=" )) != NULL) pPerm->hd.fItalic	 = AjcAscToIntW(p + 3);						//	fItalic
			if ((p = wcsstr(pTxt, L"fB=" )) != NULL) pPerm->hd.fBold 	 = AjcAscToIntW(p + 3);						//	fBold
			if ((p = wcsstr(pTxt, L"fF=" )) != NULL) pPerm->hd.fFixed	 = AjcAscToIntW(p + 3);						//	fFixed
			if ((p = wcsstr(pTxt, L"Px=" )) != NULL) pPerm->hd.Pixels	 = AjcAscToIntW(p + 3);						//	Pixels
			if ((p = wcsstr(pTxt, L"Pt=" )) != NULL) pPerm->hd.Points	 = AjcAscToIntW(p + 3);						//	Points
			if ((p = wcsstr(pTxt, L"Ls=" )) != NULL) pPerm->hd.LSpace	 = AjcAscToIntW(p + 3);						//	LSpace
			if ((p = wcsstr(pTxt, L"FA=" )) != NULL) pPerm->hd.FixAttr.v = AjcAscToIntW(p + 3);						//	FixAttr
			if ((p = wcsstr(pTxt, L"AA=" )) != NULL) pPerm->hd.AnyAttr.v = AjcAscToIntW(p + 3);						//	AnyAttr
			if ((p = wcsstr(pTxt, L"Fix=")) != NULL) SubFaceCpyW (p + 4, pPerm->FixFace, AJCTSIZE(pPerm->FixFace));	//	FixFace
			if ((p = wcsstr(pTxt, L"Any=")) != NULL) SubFaceCpyW (p + 4, pPerm->AnyFace, AJCTSIZE(pPerm->AnyFace));	//	AnyFace
		}
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	指定フォント，行間スペースをダイアログ設定値に組み込む														//
//																												//
//	引　数	：	pFont	- フォント情報へのポインタ																//
//				pPerm	- ダイアログ設定値へのポインタ															//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	WINAPI	AjcCfMergeToInfoA(PCAJCCFFONTINFOA pFont, PAJCCFPERMINFOA pPerm)
{
	BOOL		rc = FALSE;

	if (pFont != NULL && pPerm != NULL && pPerm->hd.id == AJCCFPERMID) {
		//	フォントフェース名，ピッチ＆ファミリ，キャラセット
		if (((pFont->lf.lfPitchAndFamily & 3) == 1) && pPerm->hd.fFixed) {
			AjcSnPrintFA(pPerm->FixFace, AJCTSIZE(pPerm->FixFace), "%s", pFont->lf.lfFaceName);
			pPerm->hd.FixAttr.s.PitchAndFamily = pFont->lf.lfPitchAndFamily;
			pPerm->hd.FixAttr.s.CharSet 	   = pFont->lf.lfCharSet;
		}
		else {
			AjcSnPrintFA(pPerm->AnyFace, AJCTSIZE(pPerm->AnyFace), "%s", pFont->lf.lfFaceName);
			pPerm->hd.AnyAttr.s.PitchAndFamily = pFont->lf.lfPitchAndFamily;
			pPerm->hd.AnyAttr.s.CharSet 	   = pFont->lf.lfCharSet;
			//	「固定ピットのみ」のチェックを外す
			pPerm->hd.fFixed = FALSE;
		}
		//	高さ
		if (pFont->lf.lfHeight > 0) {
			pPerm->hd.fPixel = TRUE;
			pPerm->hd.Pixels = pFont->lf.lfHeight;
		}
		else {
			int	h;
			pPerm->hd.fPixel = FALSE;
			h = (-pFont->lf.lfHeight);
			pPerm->hd.Points = AjcCfPixelsToPoints(h);
		}
		//	斜字
		if (pFont->lf.lfItalic == 1  ) pPerm->hd.fItalic = TRUE;
		//	太字
		if (pFont->lf.lfWeight == 700) pPerm->hd.fBold	 = TRUE;
		//	行間スペース
		pPerm->hd.LSpace = pFont->lsp;
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	WINAPI	AjcCfMergeToInfoW(PCAJCCFFONTINFOW pFont, PAJCCFPERMINFOW pPerm)
{
	BOOL			rc = FALSE;

	if (pFont != NULL && pPerm != NULL && pPerm->hd.id == AJCCFPERMID) {
		//	フォントフェース名，ピッチ＆ファミリ，キャラセット
		if (((pFont->lf.lfPitchAndFamily & 3) == 1) && pPerm->hd.fFixed) {
			AjcSnPrintFW(pPerm->FixFace, AJCTSIZE(pPerm->FixFace), L"%s", pFont->lf.lfFaceName);
			pPerm->hd.FixAttr.s.PitchAndFamily = pFont->lf.lfPitchAndFamily;
			pPerm->hd.FixAttr.s.CharSet		   = pFont->lf.lfCharSet;
		}
		else {
			AjcSnPrintFW(pPerm->AnyFace, AJCTSIZE(pPerm->AnyFace), L"%s", pFont->lf.lfFaceName);
			pPerm->hd.AnyAttr.s.PitchAndFamily = pFont->lf.lfPitchAndFamily;
			pPerm->hd.AnyAttr.s.CharSet		   = pFont->lf.lfCharSet;
			//	「固定ピッチのみ」のチェックを外す
			pPerm->hd.fFixed = FALSE;
		}
		//	高さ
		if (pFont->lf.lfHeight > 0) {
			pPerm->hd.fPixel = TRUE;
			pPerm->hd.Pixels = pFont->lf.lfHeight;
		}
		else {
			int	h;
			pPerm->hd.fPixel = FALSE;
			h = (-pFont->lf.lfHeight);
			pPerm->hd.Points = AjcCfPixelsToPoints(h);
		}
		//	斜字
		if (pFont->lf.lfItalic == 1  ) pPerm->hd.fItalic = TRUE;
		//	太字
		if (pFont->lf.lfWeight == 700) pPerm->hd.fBold	 = TRUE;
		//	行間スペース
		pPerm->hd.LSpace = pFont->lsp;
	}
	return rc;
}
//==============================================================================================================//
//	ダイアログボックスによるフォント選択																		//
//																												//
//	引　数	：	hOwner		- オーナーウインドのハンドル（不要時はNULL)											//
//				opt			- オプションフラグ（AJCCFF_XXXXX）													//
//				pBuf		- フォント情報へのポインタ															//
//				pPerm		- 永続化情報へのポインタ															//
//				cbp			- コールバックパラメタ																//
//				cbInit		- コールバック関数（初期化）へのポインタ											//
//				cbApply		- コールバック関数（適用）へのポインタ												//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - キャンセル																				//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	WINAPI	AjcCfFontDlgA(HWND hOwner, UI opt, PAJCCFFONTINFOA pBuf, PAJCCFPERMINFOA pPerm, UX cbp,
								VO	 (CALLBACK *cbInit )(HWND hDlg, UX cbp),
								BOOL (CALLBACK *cbApply)(HFONT hFont, UI lsp, UX cbp))
{
	PAJCCFFONTINFOW	pBufW		= NULL;	
	PAJCCFPERMINFOW	pPermW		= NULL;

	BOOL	rc = FALSE;

	do {
		//----- フォント名をワイド文字化 -----------------------------------------------------------------------//
		if (pBuf != NULL) {
			if (pBufW = (PAJCCFFONTINFOW)malloc(sizeof(AJCCFFONTINFOW))) {
				memcpy(&pBufW->lf, &pBuf->lf, sizeof(pBuf->lf));
				MultiByteToWideChar(CP_ACP, 0, pBuf->lf.lfFaceName, -1, pBufW->lf.lfFaceName, AJCTSIZE(pBufW->lf.lfFaceName));
			}
			else break;
		}
		//----- 永続化情報中のフォント名をワイド文字化 ---------------------------------------------------------//
		if (pPerm != NULL) {
			if (pPermW = (PAJCCFPERMINFOW)malloc(sizeof(AJCCFPERMINFOW))) {
				//	ヘッダ部コピー
				memcpy(&pPermW->hd, &pPerm->hd, sizeof(AJCCFPERMHD));
				//	固定ピッチフェース名
				MultiByteToWideChar(CP_ACP, 0, pPerm->FixFace, -1, pPermW->FixFace, AJCTSIZE(pPermW->FixFace));
				//	任意ピッチフェース名
				MultiByteToWideChar(CP_ACP, 0, pPerm->AnyFace, -1, pPermW->AnyFace, AJCTSIZE(pPermW->AnyFace));
			}
			else break;
		}
		//----- ダイアログによるフォント選択 -------------------------------------------------------------------//
		if (rc = AjcCfFontDlgW(hOwner, opt, pBufW, pPermW, cbp, cbInit, cbApply)) {
			if (pBuf != NULL) {
				//	選択フォント情報をバイト文字化
				memcpy(&pBuf->lf, &pBufW->lf, sizeof(pBuf->lf));
				WideCharToMultiByte(CP_ACP, 0, pBufW->lf.lfFaceName, -1, pBuf->lf.lfFaceName, AJCTSIZE( pBuf->lf.lfFaceName), NULL, NULL);
				//	行間スペースを返す
				pBuf->lsp = pBufW->lsp;
			}
			if (pPerm != NULL) {
				//	永続情報をコピー（ヘッダ部）
				memcpy(&pPerm->hd, &pPermW->hd, sizeof(AJCCFPERMHD));
				//	永続情報をワイド文字化（固定ピッチフェース名）
				WideCharToMultiByte(CP_ACP, 0, pPermW->FixFace, -1, pPerm->FixFace, AJCTSIZE(pPerm->FixFace), NULL, NULL);
				//	永続情報をワイド文字化（任意ピッチフェース名）
				WideCharToMultiByte(CP_ACP, 0, pPermW->AnyFace, -1, pPerm->AnyFace, AJCTSIZE(pPerm->AnyFace), NULL, NULL);
			}
		}
	} while(0);
	//	メモリ開放
	if (pBufW	!= NULL) free(pBufW	);
	if (pPermW	!= NULL) free(pPermW);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	WINAPI	AjcCfFontDlgW(HWND hOwner, UI opt, PAJCCFFONTINFOW pBuf, PAJCCFPERMINFOW pPerm, UX cbp,
								VO	 (CALLBACK *cbInit )(HWND hDlg, UX cbp),
								BOOL (CALLBACK *cbApply)(HFONT hFont, UI lsp, UX cbp))
{
	BOOL		rc	= FALSE;
	DLGFONTWK	DWk = {0};

	if (pBuf != NULL) {
		DWk.opt			= opt;
		DWk.pBuf		= pBuf;
		DWk.pPerm		= pPerm;
		DWk.cbp			= cbp;
		DWk.cbInit		= cbInit;
		DWk.cbApply		= cbApply;
		if (DialogBoxParam(hDllInst, MAKEINTRESOURCE(IDD_FONTDLG), hOwner, AJC_DLGPROC_NAME(SetFontDlg), (LPARAM)&DWk) == IDOK) {
			rc = TRUE;
		}
		else {
			rc = FALSE;
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	フォント設定ダイアログプロシージャ 																			//
//																												//
//--------------------------------------------------------------------------------------------------------------//
#define		TID_CBOSEL		1				/*	コンボボックス選択監視タイマ									*/
//----- ダイアログ初期化 ---------------------------------------------------------------------------------------//
AJC_DLGPROC(SetFontDlg, WM_INITDIALOG )
{
	PDLGFONTWK		pDWk	 = (PDLGFONTWK)lParam;
	RECT			r;

	//----- ダイアログにワークポインタを関連付け ---------------------------------------------------------------//
	SetProp(hDlg, L"AjcFontDlgWk", (HANDLE)lParam);

	//----- アイコン設定 ---------------------------------------------------------------------------------------//
	SendMessage(hDlg, WM_SETICON, ICON_SMALL, (LPARAM)hIcoApp);

	//----- ハンドル設定 ---------------------------------------------------------------------------------------//
	pDWk->hDlg = hDlg;
	pDWk->hCbo = GetDlgItem(hDlg, IDC_CBO_NAME);
	pDWk->hVth = GetDlgItem(hDlg, IDC_VTH_PREVIEW);

	//----- ＶＴＨ右クリック禁止，キャレット非表示 -------------------------------------------------------------//
	AjcVthSetNtcRClk(pDWk->hVth, TRUE, 0, 0);
	AjcVthShowCaret(pDWk->hVth, FALSE);

	//----- 日英テキスト ---------------------------------------------------------------------------------------//
	SetWindowText (hDlg, LNGSEL(L"フォント設定", L"Set Font"));
	SetDlgItemText(hDlg, IDC_LBL_NAME	, LNGSEL(L"フォント名"		, L"Font Name"	));
	SetDlgItemText(hDlg, IDC_CHK_ITALIC	, LNGSEL(L"斜体"			, L"Italic"		));
	SetDlgItemText(hDlg, IDC_CHK_BOLD	, LNGSEL(L"太字"			, L"Bold"		));
	SetDlgItemText(hDlg, IDC_CHK_FIXED	, LNGSEL(L"固定ピッチのみ"	, L"Fixed pitch only"));
	SetDlgItemText(hDlg, IDC_GRP_HEIGHT	, LNGSEL(L"高さ"			, L"Height"		));
	SetDlgItemText(hDlg, IDC_RBT_PIXELS	, LNGSEL(L"ピクセル数"		, L"Pixels"		));
	SetDlgItemText(hDlg, IDC_RBT_POINTS	, LNGSEL(L"ポイント数"		, L"Points"		));
	SetDlgItemText(hDlg, IDC_LBL_LSPACE	, LNGSEL(L"行間スペース"	, L"Line space"	));
	SetDlgItemText(hDlg, IDC_LBL_LSPACE2, LNGSEL(L"ピクセル"		, L"Pixels"		));
	SetDlgItemText(hDlg, IDC_CMD_APPLY	, LNGSEL(L"適 用"			, L"Apply"		));
	SetDlgItemText(hDlg, IDOK			, LNGSEL(L"ＯＫ"			, L"OK"			));
	SetDlgItemText(hDlg, IDCANCEL		, LNGSEL(L"キャンセル"		, L"Cancel"		));
	SetDlgItemText(hDlg, IDC_LBL_PITCH, L"");

	//----- ツールチップ設定 -----------------------------------------------------------------------------------//
	AjcTipTextAdd (pDWk->hVth,LNGSEL(L"\n"
									 L" ここにテキストファイルをドロップするとファイルの内容を表示します。\n"
									 L" CTRL+Vキー押下でクリップボード・テキストを表示します。\n"
									 L" テキストは最大5000行まで表示可能です。\n"
										,
									 L"\n"
									 L" Dropping a text file here displays the contents of the file. \n"
									 L" Display clipboard text with CTRL+V. \n"
									 L" Up to 5000 lines of text can be displayed. \n"));

	//----- ダイアログ最小サイズ設定 ---------------------------------------------------------------------------//
	GetWindowRect(hDlg, &r);
	pDWk->DlgMinW = r.right - r.left;
	pDWk->DlgMinH = r.bottom - r.top;

	//----- ＶＴＨ上位置設定 -----------------------------------------------------------------------------------//
	GetWindowRect(pDWk->hVth, &r);
	MapWindowPoints(NULL, hDlg, (LPPOINT)&r, 2);
	pDWk->VthTop = r.top;
	//	ＶＴＨ配置
	MoveVthWindow(hDlg);

	//----- デバッグ時、フォント情報表示 -----------------------------------------------------------------------//
	#ifdef _DEBUG
		ShowWindow(GetDlgItem(hDlg, IDC_TXT_FONTINFO), SW_HIDE);
	#else
		ShowWindow(GetDlgItem(hDlg, IDC_TXT_FONTINFO), SW_HIDE);
	#endif

	//----- フォント高さの入力値範囲設定 -----------------------------------------------------------------------//
	SendDlgItemMessage(hDlg, IDC_INP_PIXELS, AJCIVM_I_SETRANGE, MIN_FONT_PIXELS, MAX_FONT_PIXELS);
	SendDlgItemMessage(hDlg, IDC_INP_POINTS, AJCIVM_I_SETRANGE, MIN_FONT_POINTS, MAX_FONT_POINTS);

	//----- オプションによるダイアログ項目無効化 ---------------------------------------------------------------//
	if (pDWk->opt & AJCCFF_NOBOLD  ) AjcEnableDlgItem (hDlg, IDC_CHK_BOLD  , FALSE);
	if (pDWk->opt & AJCCFF_NOITALIC) AjcEnableDlgItem (hDlg, IDC_CHK_ITALIC, FALSE);
	if (pDWk->opt & AJCCFF_NOFIXED ) AjcEnableDlgItem (hDlg, IDC_CHK_FIXED , FALSE);
	if (pDWk->opt & AJCCFF_NOLSPACE) AjcEnableDlgGroup(hDlg, IDC_GRP_LSPACE, FALSE, FALSE);

	//----- オプションによるダイアログ項目非表示 ---------------------------------------------------------------//
	if (pDWk->opt & AJCCFF_HIDEBOLD  ) AjcShowDlgItem (hDlg, IDC_CHK_BOLD  , FALSE);
	if (pDWk->opt & AJCCFF_HIDEITALIC) AjcShowDlgItem (hDlg, IDC_CHK_ITALIC, FALSE);
	if (pDWk->opt & AJCCFF_HIDEFIXED ) AjcShowDlgItem (hDlg, IDC_CHK_FIXED , FALSE);
	if (pDWk->opt & AJCCFF_HIDELSPACE) AjcShowDlgGroup(hDlg, IDC_GRP_LSPACE, FALSE);

	//----- 適用ボタンアクション無しの場合は、適用ボタン無効化 -------------------------------------------------//
	if (pDWk->cbApply == NULL) {
		AjcEnableDlgItem(hDlg, IDC_CMD_APPLY, FALSE);
	}

	//----- ダイアログ項目初期化 -------------------------------------------------------------------------------//
	AjcSetDlgItemChk (hDlg, IDC_RBT_PIXELS, TRUE);
	AjcSetDlgItemSInt(hDlg, IDC_INP_PIXELS, 12);
	AjcSetDlgItemSInt(hDlg, IDC_INP_POINTS, 12);

	//----- 永続化情報コピー／初期化 ---------------------------------------------------------------------------//
	if (pDWk->pPerm != NULL && pDWk->pPerm->hd.id == AJCCFPERMID) {
		memcpy(&pDWk->Perm, pDWk->pPerm, sizeof(pDWk->Perm));
	}
	else {
		memset(&pDWk->Perm, 0, sizeof(AJCCFPERMINFOW));
		pDWk->Perm.hd.id		= AJCCFPERMID;
		pDWk->Perm.hd.fPixel	= TRUE;
		pDWk->Perm.hd.Pixels	= 12;
		pDWk->Perm.hd.Points	= 12;
		//	固定ピッチフォント
		pDWk->Perm.hd.FixAttr.s.CharSet		   = DEFAULT_CHARSET;
		pDWk->Perm.hd.FixAttr.s.PitchAndFamily = FIXED_PITCH;
		wcscpy(pDWk->Perm.FixFace, LNGSEL(L"ＭＳ ゴシック", L"MS Gothic"));
		//	任意ピッチフォント
		pDWk->Perm.hd.AnyAttr.s.CharSet		   = DEFAULT_CHARSET;
		pDWk->Perm.hd.AnyAttr.s.PitchAndFamily = FF_MODERN | VARIABLE_PITCH;
		wcscpy(pDWk->Perm.AnyFace, L"MS UI Gothic");
	}

	//----- 永続化情報によるダイアログ項目設定 -----------------------------------------------------------------//
	if (pDWk->pPerm != NULL && pDWk->pPerm->hd.id == AJCCFPERMID) {
		AjcSetDlgItemSInt(hDlg, IDC_INP_PIXELS, pDWk->Perm.hd.Pixels);
		AjcSetDlgItemSInt(hDlg, IDC_INP_POINTS, pDWk->Perm.hd.Points);
		AjcSetDlgItemUInt(hDlg, IDC_INP_LSPACE, pDWk->Perm.hd.LSpace);
		AjcSetDlgItemChk (hDlg, IDC_CHK_ITALIC, pDWk->Perm.hd.fItalic);
		AjcSetDlgItemChk (hDlg, IDC_CHK_BOLD  , pDWk->Perm.hd.fBold  );
		AjcSetDlgItemChk (hDlg, IDC_CHK_FIXED , pDWk->Perm.hd.fFixed );
		if (pDWk->Perm.hd.fPixel) {
			AjcSetDlgItemChk(hDlg, IDC_RBT_PIXELS, TRUE );
			AjcSetDlgItemChk(hDlg, IDC_RBT_POINTS, FALSE);
		}
		else {
			AjcSetDlgItemChk(hDlg, IDC_RBT_PIXELS, FALSE);
			AjcSetDlgItemChk(hDlg, IDC_RBT_POINTS, TRUE );
		}
	}
	//----- オプションで「固定ピッチのみ」指定の反映 -----------------------------------------------------------//
	if (pDWk->opt & AJCCFF_FIXEDONLY) {
		AjcSetDlgItemChk(hDlg, IDC_CHK_FIXED, TRUE);
	}

	//	ラジオボタン，チェックボックス擬似クリック
	if (AjcGetDlgItemChk(hDlg, IDC_RBT_PIXELS)) SendMessage(hDlg, WM_COMMAND, MAKELONG(IDC_RBT_PIXELS, BN_CLICKED), (LPARAM)GetDlgItem(hDlg, IDC_RBT_PIXELS));
	else										SendMessage(hDlg, WM_COMMAND, MAKELONG(IDC_RBT_POINTS, BN_CLICKED), (LPARAM)GetDlgItem(hDlg, IDC_RBT_POINTS));
	SendMessage(hDlg, WM_COMMAND, MAKELONG(IDC_CHK_ITALIC, BN_CLICKED), (LPARAM)GetDlgItem(hDlg, IDC_CHK_ITALIC));
	SendMessage(hDlg, WM_COMMAND, MAKELONG(IDC_CHK_BOLD  , BN_CLICKED), (LPARAM)GetDlgItem(hDlg, IDC_CHK_BOLD  ));
	SendMessage(hDlg, WM_COMMAND, MAKELONG(IDC_CHK_FIXED , BN_CLICKED), (LPARAM)GetDlgItem(hDlg, IDC_CHK_FIXED ));

	//----- フォント名選択 -------------------------------------------------------------------------------------//
	SfgSelFontInCbo(hDlg);

	//----- フォントピッチ種別表示 -----------------------------------------------------------------------------//
	SfgShowKindOfPitch(hDlg);

	//----- プレビューテキストありならば、プレビューテキスト表示 -----------------------------------------------//
	if (pPreView != NULL) {
		pDWk->fShowPreview = TRUE;
		SfgShowPreview(hDlg);
	}
	//----- プレビューテキスト無しならば、プレビューテキスト非表示 ---------------------------------------------//
	else {
		RECT	rd, rv;
		int		h, sty;
		pDWk->fShowPreview = FALSE;
		sty = (int)MAjcGetWindowLong(hDlg, GWL_STYLE);
		MAjcSetWindowLong(hDlg, GWL_STYLE, sty & ~(WS_THICKFRAME | WS_MAXIMIZEBOX));
		GetWindowRect(hDlg		, &rd);
		GetWindowRect(pDWk->hVth, &rv);
		h = rv.top - rd.top;
		SetWindowPos(hDlg, NULL, 0, 0, rd.right - rd.left, h, SWP_NOMOVE);
	}

	//----- 初期化コールバック ---------------------------------------------------------------------------------//
	if (pDWk->cbInit != NULL) {
		pDWk->cbInit(hDlg, pDWk->cbp);
	}

	return TRUE;
}
//----- WM_SIZING ----------------------------------------------------------------------------------------------//
AJC_DLGPROC(SetFontDlg, WM_SIZING)
{
	PDLGFONTWK	pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");
	LPRECT		p = (LPRECT)lParam;
	int			w = p->right - p->left;
	int			h = p->bottom - p->top;

	if (w < pDWk->DlgMinW) p->right  = p->left + pDWk->DlgMinW;
	if (h < pDWk->DlgMinH) p->bottom = p->top  + pDWk->DlgMinH;

	return TRUE;
}
//----- WM_SIZE ------------------------------------------------------------------------------------------------//
AJC_DLGPROC(SetFontDlg, WM_SIZE  )
{
	MoveVthWindow(hDlg);
	return TRUE;
}
//----- WM_TIMER -----------------------------------------------------------------------------------------------//
AJC_DLGPROC(SetFontDlg, WM_TIMER )
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");

	switch (wParam) {
		case TID_CBOSEL:									//●コンボボックス選択監視
		{	int		ix = (int)SendDlgItemMessage(hDlg, IDC_CBO_NAME, CB_GETCURSEL, 0, 0);
			if (ix != CB_ERR && ix != pDWk->SvCboIx) {
				SfgShowKindOfPitch(hDlg);	//	フォントのピッチ種別表示
				SfgSetPreviewFont(hDlg);	//	サンプルテキストのフォント設定
				pDWk->SvCboIx = ix;
			}
			break;
		}
	}
	return TRUE;
}
//----- WM_AJCSHOWPREVIEWTEXT ----------------------------------------------------------------------------------//
AJC_DLGPROC(SetFontDlg, WM_AJCSHOWPREVIEWTEXT)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");

	if (pPreView != NULL) {
		SfgShowPreview(hDlg);
	}

	return TRUE;
}
//----- イタリック・チェックボックス ---------------------------------------------------------------------------//
AJC_DLGPROC(SetFontDlg, IDC_CHK_ITALIC	)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");

	if (HIWORD(wParam) == BN_CLICKED) {
		pDWk->Perm.hd.fItalic = AjcGetDlgItemChk(hDlg, IDC_CHK_ITALIC);
		SfgSetPreviewFont(hDlg);	//	サンプルテキストのフォント設定
	}
	return TRUE;
}
//----- ボールド・チェックボックス -----------------------------------------------------------------------------//
AJC_DLGPROC(SetFontDlg, IDC_CHK_BOLD	)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");

	if (HIWORD(wParam) == BN_CLICKED) {
		pDWk->Perm.hd.fBold = AjcGetDlgItemChk(hDlg, IDC_CHK_BOLD);
		SfgSetPreviewFont(hDlg);	//	サンプルテキストのフォント設定
	}
	return TRUE;
}
//----- 固定ピッチ・チェックボックス ---------------------------------------------------------------------------//
AJC_DLGPROC(SetFontDlg, IDC_CHK_FIXED	)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");

	if (HIWORD(wParam) == BN_CLICKED) {
		pDWk->Perm.hd.fFixed = AjcGetDlgItemChk(hDlg, IDC_CHK_FIXED);
		//----- コンボボックス・リセット -----------------------------------//
		AjcSetDlgItemCboReset(hDlg, IDC_CBO_NAME);
		//----- フォントフェースの収集（コンボボックスに設定）--------------//
		AjcCfEnumFontsW((UX)pDWk, SfgNtcFont);
		//----- フォント名選択 ---------------------------------------------//
		SfgSelFontInCbo(hDlg);
		//----- フォントピッチ種別表示 -------------------------------------//
		SfgShowKindOfPitch(hDlg);
		//----- サンプルテキストのフォント設定 -----------------------------//
		SfgSetPreviewFont(hDlg);
	}

	return TRUE;
}
//----- ピクセル数ラジオボタン ---------------------------------------------------------------------------------//
AJC_DLGPROC(SetFontDlg, IDC_RBT_PIXELS	)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");

	if (HIWORD(wParam) == BN_CLICKED) {
		pDWk->Perm.hd.fPixel = AjcGetDlgItemChk(hDlg, IDC_RBT_PIXELS);
		AjcEnableDlgItem(hDlg, IDC_INP_PIXELS, AjcGetDlgItemChk(hDlg, IDC_RBT_PIXELS));
		AjcEnableDlgItem(hDlg, IDC_INP_POINTS, AjcGetDlgItemChk(hDlg, IDC_RBT_POINTS));
		//----- サンプルテキストのフォント設定 -----------------------------//
		SfgSetPreviewFont(hDlg);
	}

	return TRUE;
}
//----- ポイント数ラジオボタン ---------------------------------------------------------------------------------//
AJC_DLGPROC(SetFontDlg, IDC_RBT_POINTS	)
{
	SendMessage(hDlg, WM_COMMAND, MAKELONG(IDC_RBT_PIXELS, BN_CLICKED), 0);

	return TRUE;
}
//----- フォント名コンボボックス -------------------------------------------------------------------------------//
AJC_DLGPROC(SetFontDlg, IDC_CBO_NAME	)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");
	LOGFONTW		lf;

	switch (HIWORD(wParam)) {
		//----- ドロップダウン開始 --------------------------------------------------------//
		case CBN_DROPDOWN:
			pDWk->SvCboIx = CB_ERR;
			SetTimer(hDlg, TID_CBOSEL, 100, NULL);	//	タイマ起動
			break;
		//----- ドロップダウン終了 --------------------------------------------------------//
		case CBN_CLOSEUP:
			KillTimer(hDlg, TID_CBOSEL);			//	タイマ停止
			SfgSetPreviewFont(hDlg);				//	サンプルテキストのフォント設定
			break;
		//----- 選択確定／選択キャンセル／項目削除 ----------------------------------------//
		case CBN_SELENDOK:
			SfgGetFontInfo (hDlg, &lf);				//	選択されているＣＢＯ項目のフォント情報取得
			SfgSaveFontInfo(hDlg, &lf);				//	フォント情報退避
			SfgSetPreviewFont(hDlg);				//	サンプルテキストのフォント設定
			//	continue
		case CBN_SELENDCANCEL:
			//	continue
		case AJCCBN_REMOVED:
			SfgShowKindOfPitch(hDlg);				//	選択されているＣＢＯ項目のピッチ情報表示
			break;
	}
	return TRUE;
}
//----- ピクセル数 ---------------------------------------------------------------------------------------------//
AJC_DLGPROC(SetFontDlg, IDC_INP_PIXELS	)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");

	if (HIWORD(wParam) == AJCIVN_INTVALUE) {
		pDWk->Perm.hd.Pixels = AjcGetDlgItemUInt(hDlg, IDC_INP_PIXELS);
		SfgSetPreviewFont(hDlg);				//	サンプルテキストのフォント設定
	}

	return TRUE;
}
//----- ポイント数 ---------------------------------------------------------------------------------------------//
AJC_DLGPROC(SetFontDlg, IDC_INP_POINTS	)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");

	if (HIWORD(wParam) == AJCIVN_INTVALUE) {
		pDWk->Perm.hd.Points = AjcGetDlgItemUInt(hDlg, IDC_INP_POINTS);
		SfgSetPreviewFont(hDlg);				//	サンプルテキストのフォント設定
	}

	return TRUE;
}
//----- 行間スペース -------------------------------------------------------------------------------------------//
AJC_DLGPROC(SetFontDlg, IDC_INP_LSPACE	)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");

	if (HIWORD(wParam) == AJCIVN_INTVALUE) {
		pDWk->Perm.hd.LSpace = AjcGetDlgItemUInt(hDlg, IDC_INP_LSPACE);
		SfgSetPreviewFont(hDlg);				//	サンプルテキストのフォント設定
	}

	return TRUE;
}
//----- 適用ボタン ---------------------------------------------------------------------------------------------//
AJC_DLGPROC(SetFontDlg, IDC_CMD_APPLY	)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");
	LOGFONTW		lf;
	HFONT			hFont;

	if (HIWORD(wParam) == BN_CLICKED) {
		if (pDWk->cbApply != NULL && SfgGetFontInfo(hDlg, &lf)) {
			//	適用コールバック
			if (hFont = CreateFontIndirect(&lf)) {
				//	コールバック
				if (pDWk->cbApply(hFont, AjcGetDlgItemUInt(hDlg, IDC_INP_LSPACE), pDWk->cbp)) {
					DeleteObject(hFont);
				}
			}
		}
	}
	return TRUE;
}
//----- ＶＴＨ（プレビュー） -----------------------------------------------------------------------------------//
AJC_DLGPROC(SetFontDlg, IDC_VTH_PREVIEW	)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");

	switch (HIWORD(wParam)) {
		case AJCVTHN_VKEYIN:							//	●キー押下
			//	CTRL+V：貼り付け
			if (lParam == TEXT('V') && pDWk->fCtrl) {
				UTP	pTxt;
				if (pTxt = AjcCreateClipboardText()) {
					AjcVthClear(pDWk->hVth);
					AjcVthPutText(pDWk->hVth, pTxt, -1);
					AjcReleaseClipboardText(pTxt);
				}
			}
			//	CTRLキー押下処理
			if (lParam == VK_CONTROL) pDWk->fCtrl = TRUE;
			break;

		case AJCVTHN_VKEYOUT:							//	●キー離し
			//	CTRLキー離し処理
			if (lParam == VK_CONTROL) pDWk->fCtrl = FALSE;
			break;

		case AJCVTHN_DROPFILE:							//	●ファイル ドロップ（ファイルの内容を描画）
		{	WC		path[MAX_PATH];
			while (AjcVthGetDroppedFile(pDWk->hVth, path)) {
				HAJCFILE	hFile;
				WC			wbuf[1024];
				if (hFile = AjcFOpenW(path, AJCTEC_AUTO)) {
					AjcVthClear(pDWk->hVth);
					while (AjcFGetSW(hFile, wbuf, 1024)) {
						AjcVthPutTextW(pDWk->hVth, wbuf, -1);
					}
					AjcFClose(hFile);
				}
			}
			break;
		}
	}

	return TRUE;
}
//----- ＯＫボタン ---------------------------------------------------------------------------------------------//
AJC_DLGPROC(SetFontDlg, IDOK			)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");

	if (HIWORD(wParam) == BN_CLICKED) {
		//	行間スペースを返す
		pDWk->pBuf->lsp = AjcGetDlgItemUInt(hDlg, IDC_INP_LSPACE);
		//	フォント情報を返す
		if (SfgGetFontInfo(hDlg, &pDWk->pBuf->lf)) EndDialog(hDlg, IDOK);
		else									   EndDialog(hDlg, IDCANCEL);
		//	永続化情報を返す
		SfgSaveToProfile(hDlg);

		RemoveProp(hDlg, L"AjcFontDlgWk");
	}

	return TRUE;
}
//----- キャンセル ---------------------------------------------------------------------------------------------//
AJC_DLGPROC(SetFontDlg, IDCANCEL		)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");

	EndDialog(hDlg, IDCANCEL);
	RemoveProp(hDlg, L"AjcFontDlgWk");

	return TRUE;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_DLGMAP_DEF(SetFontDlg)
	AJC_DLGMAP_MSG(SetFontDlg, WM_INITDIALOG		)
	AJC_DLGMAP_MSG(SetFontDlg, WM_SIZING			)
	AJC_DLGMAP_MSG(SetFontDlg, WM_SIZE				)
	AJC_DLGMAP_MSG(SetFontDlg, WM_TIMER				)
	AJC_DLGMAP_MSG(SetFontDlg, WM_AJCSHOWPREVIEWTEXT)

	AJC_DLGMAP_CMD(SetFontDlg, IDC_CHK_ITALIC		)
	AJC_DLGMAP_CMD(SetFontDlg, IDC_CHK_BOLD			)
	AJC_DLGMAP_CMD(SetFontDlg, IDC_CHK_FIXED		)
	AJC_DLGMAP_CMD(SetFontDlg, IDC_RBT_PIXELS		)
	AJC_DLGMAP_CMD(SetFontDlg, IDC_RBT_POINTS		)
	AJC_DLGMAP_CMD(SetFontDlg, IDC_CBO_NAME			)
	AJC_DLGMAP_CMD(SetFontDlg, IDC_INP_PIXELS		)
	AJC_DLGMAP_CMD(SetFontDlg, IDC_INP_POINTS		)
	AJC_DLGMAP_CMD(SetFontDlg, IDC_INP_POINTS		)
	AJC_DLGMAP_CMD(SetFontDlg, IDC_INP_LSPACE		)
	AJC_DLGMAP_CMD(SetFontDlg, IDC_CMD_APPLY		)
	AJC_DLGMAP_CMD(SetFontDlg, IDC_VTH_PREVIEW		)
	AJC_DLGMAP_CMD(SetFontDlg, IDOK					)
	AJC_DLGMAP_CMD(SetFontDlg, IDCANCEL				)
AJC_DLGMAP_END

//----- ＶＴＨ配置 -----------------------------------------------------------------------------------------//
static	VO	MoveVthWindow(HWND hDlg)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");
	RECT			r;
	int				w, h;

	GetClientRect(hDlg, &r);
	w = r.right - r.left;
	h = r.bottom - r.top;

	SetWindowPos(pDWk->hVth, NULL, 4, pDWk->VthTop, w - 8, h - pDWk->VthTop - 4, SWP_NOZORDER);
}

//----- フォント収集(AjcCfEnumFontsのコールバック ----------------------------------------------------------//
static	BOOL CALLBACK SfgNtcFont(C_WCP pFaceName, UB CharSet, UB PitchAndFamily, UX cbp)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)cbp;
	int				ix;

	if ((pDWk->opt & AJCCFF_INCVERT) || pFaceName[0] != L'@') {	//	縦書き用フォントは除く
		if (!AjcGetDlgItemChk(pDWk->hDlg, IDC_CHK_FIXED) || (PitchAndFamily & 3) == 1) {
			if (SendMessage(pDWk->hCbo, CB_FINDSTRINGEXACT, -1, (LPARAM)pFaceName) == CB_ERR) {
				ix = (int)SendMessage(pDWk->hCbo, CB_ADDSTRING, 0, (LPARAM)pFaceName);
				if (ix >= 0) {
					AJCCFATT	att;
					att.s.CharSet		 = CharSet;
					att.s.PitchAndFamily = PitchAndFamily;
					SendMessage(pDWk->hCbo, CB_SETITEMDATA, ix, att.v);
					#ifdef _DEBUG
					//	AjcTrace(L"FN=%s, %02X, %02X\n", pFaceName, CharSet, PitchAndFamily);
					#endif
				}
			}
		}
	}
	return TRUE;
}
//----- 選択されているフォント情報取得 -------------------------------------------------------------------------//
static	BOOL	SfgGetFontInfo(HWND hDlg, LPLOGFONTW pLf)
{
	BOOL			rc = FALSE;
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");
	int				ix;
	AJCCFATT		att;

	//----- ix -------------------------------------------------------------------------------------------------//
	ix = (int)SendMessage(pDWk->hCbo, CB_GETCURSEL, 0, 0);
	if (ix != CB_ERR) {
		memset(pLf, 0, sizeof(LOGFONTW));
		//----- コンボボックスからフォント名取得 ---------------------------------------------------------------//
		SendMessage(pDWk->hCbo, CB_GETLBTEXT, ix, (LPARAM)pLf->lfFaceName);

		//----- サイズ，イタリック，ボールド設定 ---------------------------------------------------------------//
		if (AjcGetDlgItemChk(hDlg, IDC_RBT_PIXELS)) pLf->lfHeight =   AjcGetDlgItemSInt(hDlg, IDC_INP_PIXELS);
		else										pLf->lfHeight = -(AjcCfPointsToPixels(AjcGetDlgItemSInt(hDlg, IDC_INP_POINTS)));

		if (AjcGetDlgItemChk (hDlg, IDC_CHK_ITALIC)) {pLf->lfItalic = 1;	}
		else										 {pLf->lfItalic = 0;	}

		if (AjcGetDlgItemChk (hDlg, IDC_CHK_BOLD  )) {pLf->lfWeight = 700;	}
		else										 {pLf->lfWeight = 0;	}

		//----- 「CharSet, PitchAndFamily」を設定 --------------------------------------------------------------//
		att.v = (UI)SendMessage(pDWk->hCbo, CB_GETITEMDATA, ix, 0);
		pLf->lfCharSet		  = att.s.CharSet;
		pLf->lfPitchAndFamily = att.s.PitchAndFamily;

		rc = TRUE;
	}
	return rc;
}
//----- フォント情報退避 ---------------------------------------------------------------------------------------//
static	VO		SfgSaveFontInfo(HWND hDlg, LPLOGFONTW pLf)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");

	if (AjcGetDlgItemChk(hDlg, IDC_CHK_FIXED)) {	//	固定ピッチ？
		//	最後に選択された固定ピッチフォント退避
		pDWk->Perm.hd.FixAttr.s.PitchAndFamily = pLf->lfPitchAndFamily;
		AjcSnPrintF(pDWk->Perm.FixFace, AJCTSIZE(pDWk->Perm.FixFace), L"%s", pLf->lfFaceName);
	}
	else {											//	任意ピッチ？
		//	最後に選択された任意ピッチフォント退避
		pDWk->Perm.hd.AnyAttr.s.PitchAndFamily = pLf->lfPitchAndFamily;
		AjcSnPrintF(pDWk->Perm.AnyFace, AJCTSIZE(pDWk->Perm.AnyFace), L"%s", pLf->lfFaceName);
	}

}
//----- フォント選択 -------------------------------------------------------------------------------------------//
static	VO		SfgSelFontInCbo(HWND hDlg)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");
	int				ix, i, n;
	AJCCFATT		at, cd;

	ix = -1;

	//	コンボボックスから最後に選択されたの固定／任意ピッチフォント名検索
	if (ix < 0) {
		if (AjcGetDlgItemChk(hDlg, IDC_CHK_FIXED)) {
			ix = (int)SendMessage(pDWk->hCbo, CB_FINDSTRINGEXACT, -1, (LPARAM)pDWk->Perm.FixFace);
		}
		else {
			ix = (int)SendMessage(pDWk->hCbo, CB_FINDSTRINGEXACT, -1, (LPARAM)pDWk->Perm.AnyFace);
		}
	}
	//	見つからない場合は、「CharSet, PitchAndFamily」で検索
	if (ix < 0) {
		if (AjcGetDlgItemChk(hDlg, IDC_CHK_FIXED)) at.v = pDWk->Perm.hd.FixAttr.v;
		else									   at.v = pDWk->Perm.hd.AnyAttr.v;
		n = (int)SendMessage(pDWk->hCbo, CB_GETCOUNT, 0, 0);
		for (i = 0; i < n; i++) {
			cd.v = (UI)SendMessage(pDWk->hCbo, CB_GETITEMDATA, i, 0);
			if (cd.s.CharSet == at.s.CharSet && cd.s.PitchAndFamily == at.s.PitchAndFamily) {
				ix = i;
				break;
			}
		}
	}
	//	見つからない場合は、「CharSet」で検索
	if (ix < 0) {
		if (AjcGetDlgItemChk(hDlg, IDC_CHK_FIXED)) at.v = pDWk->Perm.hd.FixAttr.v;
		else									   at.v = pDWk->Perm.hd.AnyAttr.v;
		n = (int)SendMessage(pDWk->hCbo, CB_GETCOUNT, 0, 0);
		for (i = 0; i < n; i++) {
			cd.v = (UI)SendMessage(pDWk->hCbo, CB_GETITEMDATA, i, 0);
			if (cd.s.CharSet == at.s.CharSet) {
				ix = i;
				break;
			}
		}
	}
	//	見つからない場合は、インデクス＝０とする
	if (ix < 0) ix = 0;
	SendMessage(pDWk->hCbo, CB_SETCURSEL, ix, 0);
}
//----- 選択されているＣＢＯ項目のフォントのピッチ種別表示 -----------------------------------------------------//
static	VO	SfgShowKindOfPitch(HWND hDlg)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");
	int				ix;
	AJCCFATT		att;

	ix = (int)SendMessage(pDWk->hCbo, CB_GETCURSEL, 0, 0);
	if (ix != CB_ERR) {
		att.v = (UI)SendMessage(pDWk->hCbo, CB_GETITEMDATA, ix, 0);
		switch (att.s.PitchAndFamily & 3) {
			case DEFAULT_PITCH:		AjcSetDlgItemStr(hDlg, IDC_LBL_PITCH, L"DEFAULT_PITCH"	);	break;
			case FIXED_PITCH:		AjcSetDlgItemStr(hDlg, IDC_LBL_PITCH, L"FIXED_PITCH"	);	break;
			case VARIABLE_PITCH:	AjcSetDlgItemStr(hDlg, IDC_LBL_PITCH, L"VARIABLE_PITCH"	);	break;
			default:				AjcSetDlgItemStr(hDlg, IDC_LBL_PITCH, L""				);	break;
		}
		//	フォント情報表示（デバッグ時のみ）
		#ifdef _DEBUG
		{	WC		txt[32];
			AjcSnPrintF(txt, AJCTSIZE(txt), L"CharSet=%d, PitchAndFamily=%d",att.s.CharSet, att.s.PitchAndFamily);
			AjcSetDlgItemStr(hDlg, IDC_TXT_FONTINFO, txt);
		}
		#endif
	}
	else {
		AjcSetDlgItemStr(hDlg, IDC_LBL_PITCH, L"");
	}
}
//----- 永続化情報を返す ---------------------------------------------------------------------------------------//
static	VO	SfgSaveToProfile(HWND hDlg)
{
	PDLGFONTWK		pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");

	if (pDWk->pPerm != NULL) {
		memcpy(pDWk->pPerm, &pDWk->Perm, sizeof(AJCCFPERMINFOW));
	}
}
//----- プレビューテキストのフォント設定 -----------------------------------------------------------------------//
static	VO	SfgSetPreviewFont(HWND hDlg)
{
	PDLGFONTWK	pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");
	AJCVTHPROP	prop;

	//	プレビューフォント設定
	if (pDWk->fShowPreview) {
		AjcVthGetProp(pDWk->hVth, &prop);
		SfgGetFontInfo(hDlg, &prop.LogFont);
		prop.LSpace = AjcGetDlgItemUInt(hDlg, IDC_INP_LSPACE);
		AjcVthSetProp(pDWk->hVth, &prop);
	}
}
//----- プレビューテキスト表示 ---------------------------------------------------------------------------------//
static	VO	SfgShowPreview(HWND hDlg)
{
	PDLGFONTWK	pDWk = (PDLGFONTWK)GetProp(hDlg, L"AjcFontDlgWk");
	WC			szTxt[256];
	AJCVTHPROP	prop;
	if (pDWk->fShowPreview) {
		//	プレビューフォント設定
		AjcVthGetProp(pDWk->hVth, &prop);
		SfgGetFontInfo(hDlg, &prop.LogFont);
		prop.LSpace = AjcGetDlgItemUInt(hDlg, IDC_INP_LSPACE);
		AjcVthSetProp(pDWk->hVth, &prop);
		//	プレビューテキスト表示
		AjcVthClear(pDWk->hVth);
		if ((prop.LogFont.lfPitchAndFamily & 3) == 1) AjcSnPrintFW(szTxt, AJCTSIZE(szTxt), L"%s ( Fixed pitch font )\n" , prop.LogFont.lfFaceName);
		else										  AjcSnPrintFW(szTxt, AJCTSIZE(szTxt), L"%s ( Proportional font )\n", prop.LogFont.lfFaceName);
		AjcVthPrintF(pDWk->hVth, L"%s%s", szTxt, pPreView);
	}
}
//--------------------------------------------------------------------------------------------------------------//
//	プレビューテキスト表示メッセージ送出																		//
//--------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL CALLBACK cbSsspmEnumThreadWindows(HWND hwnd, LPARAM lParam)
{
	PDLGFONTWK	pDWk = (PDLGFONTWK)GetProp(hwnd, L"AjcFontDlgWk");

	if (pDWk != NULL) {
		PostMessage(hwnd, WM_AJCSHOWPREVIEWTEXT, 0, 0);
	}
	return TRUE;
}
//--------------------------------------------------------------------------------------------------------------//
static VO	SfgPostShowPreviewMsg(VO)
{
	EnumThreadWindows(GetCurrentThreadId(), (WNDENUMPROC)cbSsspmEnumThreadWindows, 0);
}
//--------------------------------------------------------------------------------------------------------------//
//	フォントフェース名コピー																					//
//--------------------------------------------------------------------------------------------------------------//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
static	VO		SubFaceCpyA(C_BCP pS, BCP pD, UI len)
{
	if (len != 0) {
		len--;
		while (*pS != ',' && *pS != '>' && *pS != ']' && *pS != 0 && len != 0) {
			*pD++ = *pS++;
			len--;
		}
		*pD = 0;
	}
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
static	VO		SubFaceCpyW(C_WCP pS, WCP pD, UI len)
{
	if (len != 0) {
		len--;
		while (*pS != L',' && *pS != L'>' && *pS != L']' && *pS != 0 && len != 0) {
			*pD++ = *pS++;
			len--;
		}
		*pD = 0;
	}
}
