﻿#include	"AjcInternal.h"

//==============================================================================================================//
//	GetLastErrot()で得たエラーコードが表す内容のテキストを取得します											//
//																												//
//	引　数　：　GetLastError()で得たエラーコード																//
//				pBuf - テキストを格納するバッファのアドレス														//
//				bfl	 - テキストを格納するバッファのバイト数／文字数												//
//																												//
//	戻り値　：　エラーテキストのバイト数／文字数																//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcGetLastErrorTextA(int LastError, BCP pBuf, UI bfl)
{
	UI		stl = 0;
	BCP		pErrTxt = NULL;

	if (pBuf != NULL && bfl != 0) {
		stl = FormatMessageA( FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
				NULL, LastError, 0, (LPSTR)&pErrTxt, 0, NULL);
		if (stl != 0) {
			strncpy(pBuf, pErrTxt, bfl - 1);
			pBuf[bfl - 1] = 0;
		}
		else {
			*pBuf = 0;
		}
		if (pErrTxt != NULL) LocalFree(pErrTxt);
	}
	return stl;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcGetLastErrorTextW(int LastError, WCP pBuf, UI bfl)
{
	UI		stl = 0;
	WCP		pErrTxt = NULL;

	if (pBuf != NULL && bfl != 0) {
		stl = FormatMessageW( FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
				NULL, LastError, 0, (LPWSTR)&pErrTxt, 0, NULL);
		if (stl != 0) {
			wcsncpy(pBuf, pErrTxt, bfl - 1);
			pBuf[bfl - 1] = 0;
		}
		else {
			*pBuf = 0;
		}
		if (pErrTxt != NULL) LocalFree(pErrTxt);
	}
	return stl;
}
//==============================================================================================================//
//	バリアントデータが示すデータのポインタ／バイト数を取得する													//
//																												//
//	引　数　：　pVa - バリアントデータのアドレス																//
//				len - バリアントデータが示すデータのバイト数を格納するバッファのアドレス						//
//																												//
//	戻り値　：　≠ＮＵＬＬ - バリアントデータが示すデータのアドレス（配列の場合は、先頭要素のアドレス）			//
//				＝ＮＵＬＬ - エラー（処理不能なデータ型）														//
//==============================================================================================================//
typedef	 struct	 _VBARRAY {
	UL		f1;					//	不明
	UL		len;				//	１項目のバイト数
	UL		f2;					//	不明
	VOP		vp;					//	配列データへのポインタ
	UL		num;				//	配列要素数
	UL		six;				//	配列の開始インデクス
} VBARRAY, *LPVBARRAY;
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	VOP		WINAPI	AjcGetVariantInfo(const VARIANT* pVa, ULP pLen)
{
	void*		vp = NULL;
	UL			*lp;
	LPVBARRAY	pVBA;

	if (pVa != NULL && pLen != NULL) {

		while (pVa->vt == (VT_BYREF | VT_VARIANT)) {
			pVa = pVa->pvarVal;
		}

		if ((pVa->vt & 0xF000) == 0) {
			switch (pVa->vt & VT_TYPEMASK) {
				case VT_I1:
				case VT_UI1:
							vp = (VO*)&pVa->bVal;
							*pLen = 1;
							break;
				case VT_I2:
				case VT_UI2:
				case VT_BOOL:
							vp = (VO*)&pVa->iVal;
							*pLen = 2;
							break;
				case VT_I4:
				case VT_UI4:
				case VT_R4:
				case VT_INT:
				case VT_UINT:
							vp = (VO*)&pVa->lVal;
							*pLen = 4;
							break;
				case VT_I8:
				case VT_UI8:
				case VT_R8:
				case VT_CY:
				case VT_DATE:
							vp = (VO*)&pVa->lVal;
							*pLen = 8;
							break;
				case VT_BSTR:
							vp = (VO*)pVa->bstrVal;
							lp = (UL*)pVa->bstrVal;
							lp--;
							*pLen = *lp;
							break;
			}
		}
		else if ((pVa->vt & 0xF000) == VT_ARRAY) {
			if ((pVa->vt & VT_TYPEMASK) != VT_BSTR	&&	(pVa->vt & VT_TYPEMASK) != VT_VARIANT) {
				#ifdef _WIN64
					pVBA = (LPVBARRAY)pVa->llVal;
				#else
					pVBA = (LPVBARRAY)((SX)pVa->lVal);
				#endif
				vp = pVBA->vp;
				*pLen = pVBA->len * pVBA->num;
			}
		}
	}
	return(vp);
}
//==============================================================================================================//
//	Function	:	ワード（16ビット）値のエンディアン変換														//
//																												//
//	Argument	:	x - 16ビット値																				//
//																												//
//	Return		:	エンディアン変換した値																		//
//==============================================================================================================//
AJCEXPORT	UW		WINAPI	AjcExcWord (UI x)
{
	union {UW w; UB b[2];}	u;

	u.w = (UW)x;

	u.b[0] ^= u.b[1];
	u.b[1] ^= u.b[0];
	u.b[0] ^= u.b[1];

	return u.w;
}
//==============================================================================================================//
//	Function	:	ロングワード（32ビット）値のエンディアン変換												//
//																												//
//	Argument	:	x - 32ビット値																				//
//																												//
//	Return		:	エンディアン変換した値																		//
//==============================================================================================================//
AJCEXPORT	UL		WINAPI	AjcExcLong (UL x)
{
	union {UL l; UB b[4];}	u;
	UBP						p = (UBP)&x;

	u.b[3] = *p++;
	u.b[2] = *p++;
	u.b[1] = *p++;
	u.b[0] = *p++;

	return u.l;
}
//==============================================================================================================//
//	Function	:	ロング・ロングワード（64ビット）値のエンディアン変換										//
//																												//
//	Argument	:	x - 32ビット値																				//
//																												//
//	Return		:	エンディアン変換した値																		//
//==============================================================================================================//
AJCEXPORT	ULL		WINAPI	AjcExcLLong (ULL x)
{
	union {ULL l; UB b[8];}	u;
	UBP						p = (UBP)&x;

	u.b[7] = *p++;
	u.b[6] = *p++;
	u.b[5] = *p++;
	u.b[4] = *p++;
	u.b[3] = *p++;
	u.b[2] = *p++;
	u.b[1] = *p++;
	u.b[0] = *p++;

	return u.l;
}
//==============================================================================================================//
//	Function	:	実数（単精度）値のエンディアン変換															//
//																												//
//	Argument	:	x - 32ビット値																				//
//																												//
//	Return		:	エンディアン変換した値																		//
//==============================================================================================================//
AJCEXPORT	float		WINAPI	AjcExcFloat (float x)
{
	union {float f; UB b[4];}	u;
	UBP							p = (UBP)&x;

	u.b[3] = *p++;
	u.b[2] = *p++;
	u.b[1] = *p++;
	u.b[0] = *p++;

	return u.f;
}
//==============================================================================================================//
//	Function	:	実数（倍精度）値のエンディアン変換															//
//																												//
//	Argument	:	x - 32ビット値																				//
//																												//
//	Return		:	エンディアン変換した値																		//
//==============================================================================================================//
AJCEXPORT	double		WINAPI	AjcExcDouble (double x)
{
	union {double d; UB b[4];}	u;
	UBP							p = (UBP)&x;

	u.b[7] = *p++;
	u.b[6] = *p++;
	u.b[5] = *p++;
	u.b[4] = *p++;
	u.b[3] = *p++;
	u.b[2] = *p++;
	u.b[1] = *p++;
	u.b[0] = *p++;

	return u.d;
}
//==============================================================================================================//
//	Function	:	任意バイト数の値のエンディアン変換															//
//																												//
//	Argument	:	pVal - 任意バイト数の値の先頭アドレス														//
//					len  - 値のバイト数																			//
//																												//
//	Return		:	なし																						//
//==============================================================================================================//
AJCEXPORT	VO		WINAPI	AjcExcAny (VOP pVal, int len)
{
	UBP		p1, p2;
	int		n, i;

	if (pVal != NULL && len > 1) {
		p1 = ((UBP)pVal);
		p2 = ((UBP)pVal) + (len - 1);
		n  = len / 2;
		for (i = 0; i < n; i++) {
			*p1 ^= *p2;
			*p2 ^= *p1;
			*p1 ^= *p2;
			p1++;
			p2--;
		}
	}
}
//==============================================================================================================//
//	Function	:	メモリブロックの内容交換																	//
//																												//
//	Argument	:	pMem1 - メモリブロック１のアドレス															//
//					pMem2 - メモリブロック２のアドレス															//
//					len	  - メモリブロックのバイト数															//
//																												//
//	Return		:	なし																						//
//==============================================================================================================//
AJCEXPORT	VO		WINAPI	AjcMemSwap(VOP pMem1, VOP pMem2, UI len)
{
	if (pMem1 != NULL && pMem2 != NULL && len > 0) {
		if (((((UX)pMem1) & (sizeof(UI)-1))) == 0  &&  ((((UX)pMem1) & (sizeof(UI)-1))) == 0) {
			while (len >= sizeof(UI)) {
				*((UIP)pMem1) ^= *((UIP)pMem2);
				*((UIP)pMem2) ^= *((UIP)pMem1);
				*((UIP)pMem1) ^= *((UIP)pMem2);
				((UIP)pMem1)++;
				((UIP)pMem2)++;
				len -= sizeof(UI);
			}
		}
		while (len--) {
			*((UBP)pMem1) ^= *((UBP)pMem2);
			*((UBP)pMem2) ^= *((UBP)pMem1);
			*((UBP)pMem1) ^= *((UBP)pMem2);
			((UBP)pMem1)++;
			((UBP)pMem2)++;
		}
	}
}
//==============================================================================================================//
//	Function	:	バッファをデータで埋める																	//
//																												//
//	Argument	:	pBuf  - 指定データで埋めるバッファのアドレス												//
//					dat   - 埋めるデータ																		//
//					len	  - バッファのバイト数/ワード数／ロングワード数											//
//																												//
//	Return		:	なし																						//
//==============================================================================================================//
//----- バイト -------------------------------------------------------------------------------------------------//
AJCEXPORT	VOP		WINAPI	AjcMemSet8(VOP pBuf, UB dat, UI len)
{
	return memset(pBuf, dat, len);
}
//----- ワード -------------------------------------------------------------------------------------------------//
AJCEXPORT	VOP		WINAPI	AjcMemSet16(VOP pBuf, UW dat, UI len)
{
	UI		i;

	if (pBuf != NULL) {
		for (i = 0; i < len; i++) {
			*((UWP)pBuf)++ = dat;
		}
	}
	return pBuf;
}
//----- ロングワード -------------------------------------------------------------------------------------------//
AJCEXPORT	VOP		WINAPI	AjcMemSet32(VOP pBuf, UL dat, UI len)
{
	UI		i;

	if (pBuf != NULL) {
		for (i = 0; i < len; i++) {
			*((ULP)pBuf)++ = dat;
		}
	}
	return pBuf;
}


//==============================================================================================================//
//	Function	:	指定された数値を単位数値の整数倍に調整する													//
//																												//
//	Argument	:	value  - 整数倍に調整する値																	//
//					unit   - 単位数値																			//
//																												//
//	Return		:	単位数値の整数倍に調整した値																//
//==============================================================================================================//
AJCEXPORT	double	WINAPI	AjcAdjustMultValue(double value, double unit)
{
	double	rc = 0.0;
	double	mod, mul;

	if (unit > 0.0) {
		mod = modf(value / unit, &mul);
		if (fabs(mod) >= 0.5) {
			if (mul >= 0.0) mul += 1.0;
			else			mul -= 1.0;
		}
		rc = unit * mul;
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	指定された値の絶対値以上の「１０＾ｎ×Ｎ」値に調整する										//
//																												//
//	Argument	:	n	   - 調整する値																			//
//					pDigit - １０進表現の整数桁数を格納するバッファのアドレス（不要時はＮＵＬＬ）				//
//					pPrec  - １０進表現の小数部桁数を格納するバッファのアドレス（不要時はＮＵＬＬ）				//
//					fEven  - TRUE：乗数（Ｎ）を偶数とする														//
//																												//
//	Return		:	調整した値（絶対値）																		//
//==============================================================================================================//
AJCEXPORT	double	WINAPI	AjcAdjustMultPow10(double n, UIP pDigit, UIP pPrec, BOOL fEven)
{
	double	m, mul, rc;
	UI		digit, prec;

	n = fabs(n);
	m = AjcAdjustPow10(n, &digit, &prec);
	if (m >= n) {
		rc = m;
	}
	else {
		mul = (n / m);
		mul -= fmod(mul, 1.0);
		mul++;
		if (fEven && (((int)mul) & 1)) {
			mul++;
		}
		rc = m * mul;
	}
	if (pPrec  != NULL) *pPrec = prec;
	if (pDigit != NULL) *pDigit = digit;

	return rc;
}
//==============================================================================================================//
//	Function	:	指定された値の絶対値以上の「１０＾ｎ×Ｎ」値（但し、末桁は 5 or 0）に調整する				//
//																												//
//	Argument	:	n	   - 調整する値																			//
//					pDigit - １０進表現の整数桁数を格納するバッファのアドレス（不要時はＮＵＬＬ）				//
//					pPrec  - １０進表現の小数部桁数を格納するバッファのアドレス（不要時はＮＵＬＬ）				//
//					fEven  - TRUE：乗数（Ｎ）を偶数とする														//
//																												//
//	Return		:	調整した値（絶対値）																		//
//==============================================================================================================//
AJCEXPORT	double	WINAPI	AjcRoundMultPow10(double n, UIP pDigit, UIP pPrec)
{
	double	rc;
	double	step, unit;

	step = AjcAdjustMultPow10(n, pDigit, pPrec, FALSE);
	unit = AjcAdjustPow10(step, NULL, NULL);

	if (step != unit) {
		if (step / unit <= 5.0) {
			rc = unit * 5.0;
		}
		else {
			rc = unit * 10.0;
			if (pDigit != NULL && pPrec != NULL) {
				if (*pPrec >= 1) {
					(*pPrec)--;
				}
				else {
					(*pDigit)++;
				}
			}
		}
	}
	else {
		rc = step;
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	指定された値の絶対値以下の１０のｎ乗値に調整する											//
//																												//
//	Argument	:	n	   - 調整する値																			//
//					pDigit - １０進表現の整数桁数を格納するバッファのアドレス（不要時はＮＵＬＬ）				//
//					pPrec  - １０進表現の小数部桁数を格納するバッファのアドレス（不要時はＮＵＬＬ）				//
//																												//
//	Return		:	調整した値（絶対値）																		//
//==============================================================================================================//
AJCEXPORT	double	WINAPI	AjcAdjustPow10(double n, UIP pDigit, UIP pPrec)
{
	double	rc;
	double	p;

	n = fabs(n);
	p = log10(n);
	
	if (p >= 0.0) {
		p = floor(p);
		if (pDigit != NULL) *pDigit = (int)(p + 1);
		if (pPrec  != NULL) *pPrec	= 0;
	}
	else {
		p = -ceil(fabs(p));
		if (pDigit != NULL) *pDigit = 1;
		if (pPrec  != NULL) *pPrec	= abs((int)p);
	}
	rc = pow(10.0, p);
//	rc -= fmod(10.0, pow(10.0, p - 10));
	return rc;
}
//==============================================================================================================//
//	Function	:	色の明るさ算出																				//
//																												//
//	Argument	:	rgb	  -	ＲＧＢ色値																			//
//																												//
//	Return		:	明るさの度合い（０～２５５，１２５以上が見やすい明るさの差）								//
//==============================================================================================================//
AJCEXPORT	UI		WINAPI	AjcRgbBright(COLORREF rgb)
{
	UI		r = GetRValue(rgb);
	UI		g = GetGValue(rgb);
	UI		b = GetBValue(rgb);
	UI		bright;

	bright = ((r * 299) + (g * 587) + (b * 114)) / 1000;

	return bright;
}
//==============================================================================================================//
//	Function	:	色の輝度算出																				//
//																												//
//	Argument	:	rgb	  -	ＲＧＢ色値																			//
//																												//
//	Return		:	輝度																						//
//==============================================================================================================//
AJCEXPORT	double		WINAPI	AjcRgbIntensity(COLORREF rgb)
{
	double	sR = (double)GetRValue(rgb) / 255.0;
	double	sG = (double)GetGValue(rgb) / 255.0;
	double	sB = (double)GetBValue(rgb) / 255.0;
	double	dR = (sR <= 0.03928) ? sR / 12.92 : pow(((sR + 0.055) / 1.055), 2.4);
	double	dG = (sG <= 0.03928) ? sG / 12.92 : pow(((sG + 0.055) / 1.055), 2.4);
	double	dB = (sB <= 0.03928) ? sB / 12.92 : pow(((sB + 0.055) / 1.055), 2.4);
	double	intensity;

	intensity = dR * 0.2126 + dG * 0.7152 + dB * 0.0722;

	return intensity;
}
//==============================================================================================================//
//	Function	:	輝度比算出																					//
//																												//
//	Argument	:	L1, L2	 -	輝度																			//
//																												//
//	Return		:	輝度比	（５～７以上が見やすい輝度比である）												//
//==============================================================================================================//
AJCEXPORT	double		WINAPI	AjcRgbIntensityRatio(double L1, double L2)
{
	double	hi = __max(L1, L2);
	double	lo = __min(L1, L2);
	double	ratio;

	ratio = (hi + 0.05) / (lo + 0.05);

	return ratio;
}
//==============================================================================================================//
//	Function	:	色の差を算出																				//
//																												//
//	Argument	:	rgb1, rgb2	 -	色の差を算出するＲＧＢ値													//
//																												//
//	Return		:	色の差（０～７６５，５００以上が見やすい色差）												//
//==============================================================================================================//
AJCEXPORT	UI		WINAPI	AjcRgbDeffrence(COLORREF rgb1, COLORREF rgb2)
{
	UI		r1 = GetRValue(rgb1);
	UI		g1 = GetGValue(rgb1);
	UI		b1 = GetBValue(rgb1);
	UI		r2 = GetRValue(rgb2);
	UI		g2 = GetGValue(rgb2);
	UI		b2 = GetBValue(rgb2);
	UI		differ;

	differ = abs(r1 - r2) + abs(g1 - g2) + abs(b1 - b2);

	return differ;
}
//==============================================================================================================//
//	Function	:	補色／反対色を算出																			//
//																												//
//	Argument	:	rgb	 -	補色／反対色を算出するＲＧＢ値														//
//																												//
//	Return		:	補色／反対色																				//
//==============================================================================================================//
AJCEXPORT	COLORREF WINAPI	AjcComplementaryColor(COLORREF rgb)
{
	COLORREF	rc;
	COLORREF	rgb1, rgb2;
	UI			df1 , df2;
	UI			r, g, b;
	UI			cmin, cmax, csum;
	UI			r1, g1, b1;
	UI			r2, g2, b2;

	//	色の成分設定
	r = GetRValue(rgb);
	g = GetGValue(rgb);
	b = GetBValue(rgb);

	//	補色算出
	cmax = __max(r	 , g);	cmin = __min(r	 , g);
	cmax = __max(cmax, b);	cmin = __min(cmin, b);
	csum = cmin + cmax;
	r1 = ((csum - r) % 255);
	g1 = ((csum - g) % 255);
	b1 = ((csum - b) % 255);
	rgb1 = RGB(r1, g1, b1);
	df1  = AjcRgbDeffrence(rgb, rgb1);

	//	反対色算出
	r2 = 255 - r;
	g2 = 255 - g;
	b2 = 255 - b;
	rgb2 = RGB(r2, g2, b2);
	df2 = AjcRgbDeffrence(rgb, rgb2);

	//	戻り値設定
	if (df1 >= df2) rc = rgb1;
	else			rc = rgb2;

	return rc;
}
//==============================================================================================================//
//	Function	:	コマンドライン文字列から先頭パラメタをスキップする											//
//																												//
//	Argument	:	pCmd	- コマンドライン文字列のアドレス													//
//					n		- スキップする引数の個数															//
//																												//
//	Return		:	スキップした引数の次の引数のアドレス														//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	C_BCP		WINAPI	AjcSkipCmdArgsA(C_BCP pCmd, UI n)
{
	C_BCP	rc = NULL;
	UI		i;
	C_BCP	p;

	if (pCmd != NULL) {
		for (i = 0, p = pCmd; i < n && *p != 0; i++) {
			//	引数スキップ
			if (*p == '\"') {
				p++;
				while (*p != 0 && *p != '\"') p++;
				if (*p == '\"') p++;
			}
			else {
				while (*p != 0 && *p != ' ') p++;
			}
			//	空白スキップ
			while (*p == ' ') p++;
		}
		//	戻り値設定
		rc = p;
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	C_WCP		WINAPI	AjcSkipCmdArgsW(C_WCP pCmd, UI n)
{
	C_WCP	rc = NULL;
	UI		i;
	C_WCP	p;

	if (pCmd != NULL) {
		for (i = 0, p = pCmd; i < n && *p != 0; i++) {
			//	引数スキップ
			if (*p == L'\"') {
				p++;
				while (*p != 0 && *p != L'\"') p++;
				if (*p == L'\"') p++;
			}
			else {
				while (*p != 0 && *p != L' ') p++;
			}
			//	空白スキップ
			while (*p == L' ') p++;
		}
		//	戻り値設定
		rc = p;
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	コマンドライン文字列から引数の配列を生成													//
//																												//
//	Argument	:	pCmd	- コマンドライン文字列のアドレス													//
//					pArgC	- 引数の個数を格納するバッファのアドレス											//
//																												//
//	Return		:	引数へのポインタ配列のアドレス																//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BCP	*	WINAPI	AjcParseCmdArgsA(C_BCP pCmd, UIP pArgC)
{
	VOP		rc = NULL;
	UI		argc = 0;
	UI		stl;
	BCP		argv = NULL;
	C_BCP	p;
	BCP		ps;
	BCP		*pv;
	UI		BufSize = 0;

	if (pCmd != NULL) {
		//----- 引数個数カウント --------------------//
		p = pCmd;
		while (MAjcIsBlankA(*p)) p++;
		while (*p != 0) {
			stl = 0;
			while (!MAjcIsBlankA(*p) && *p != 0) {
				if (*p == '"') {
					p++;
					while (*p != '"' && *p != 0) {stl++; p++;}
					if (*p == '"') p++;
				}
				else {
					while (!MAjcIsBlankA(*p) && *p != '"' && *p != 0) {stl++; p++;}
				}
			}
			argc++;
			BufSize += (sizeof(VOP) + stl + 1);
			while (MAjcIsBlankA(*p)) p++;
		}
		BufSize += sizeof(VOP);

		if (rc = (VOP)AJCMEM(BufSize)) {
			pv = (BCP*)rc;
			ps = (UBP)(((UBP*)rc) + (argc + 1));
			//----- 引数配列作成 ---------------------//
			p = pCmd;
			while (MAjcIsBlankA(*p)) p++;
			while (*p != 0) {
				*pv++ = ps;
				while (!MAjcIsBlankA(*p) && *p != 0) {
					if (*p == '"') {
						p++;
						while (*p != '"' && *p != 0) {*ps++ = *p++;}
						if (*p == '"') p++;
					}
					else {
						while (!MAjcIsBlankA(*p) && *p != '"' && *p != 0) {*ps++ = *p++;}
					}
				}
				*ps++ = 0;
				while (MAjcIsBlankA(*p)) p++;
			}
			*pv = NULL;
		}

		if (pArgC != NULL) {
			*pArgC = argc;
		}
	}
	return (BCP *)rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	WCP	*	WINAPI	AjcParseCmdArgsW(C_WCP pCmd, UIP pArgC)
{
	VOP		rc = NULL;
	UI		argc = 0;
	UI		stl;
	WCP		argv = NULL;
	C_WCP	p;
	WCP		ps;
	WCP		*pv;
	UI		BufSize = 0;

	if (pCmd != NULL) {
		//----- 引数個数カウント --------------------//
		p = pCmd;
		while (MAjcIsBlankW(*p)) p++;
		while (*p != 0) {
			stl = 0;
			while (!MAjcIsBlankW(*p) && *p != 0) {
				if (*p == L'"') {
					p++;
					while (*p != L'"' && *p != 0) {stl++; p++;}
					if (*p == L'"') p++;
				}
				else {
					while (!MAjcIsBlankW(*p) && *p != L'"' && *p != 0) {stl++; p++;}
				}
			}
			argc++;
			BufSize += (sizeof(VOP) + (stl + 1) * 2);
			while (MAjcIsBlankW(*p)) p++;
		}
		BufSize += sizeof(VOP);

		if (rc = (VOP)AJCMEM(BufSize)) {
			pv = (WCP*)rc;
			ps = (WCP)(((WCP*)rc) + (argc + 1));
			//----- 引数配列作成 ---------------------//
			p = pCmd;
			while (MAjcIsBlankW(*p)) p++;
			while (*p != 0) {
				*pv++ = ps;
				while (!MAjcIsBlankW(*p) && *p != 0) {
					if (*p == L'"') {
						p++;
						while (*p != L'"' && *p != 0) {*ps++ = *p++;}
						if (*p == L'"') p++;
					}
					else {
						while (!MAjcIsBlankW(*p) && *p != L'"' && *p != 0) {*ps++ = *p++;}
					}
				}
				*ps++ = 0;
				while (MAjcIsBlankW(*p)) p++;
			}
			*pv = NULL;
		}

		if (pArgC != NULL) {
			*pArgC = argc;
		}
	}
	return (WCP *)rc;
}
//==============================================================================================================//
//	Function	:	コマンドライン文字列から引数の配列を生成（拡張）											//
//																												//
//	Argument	:	pCmd		 - コマンドライン文字列のアドレス												//
//					pArgC		 - 引数の個数を格納するバッファのアドレス										//
//					fRemoveQuote - ダブルクゥオート(")を削除するか否かの指定（TRUE:削除する，FALSE:削除しない）	//
//																												//
//	Return		:	引数へのポインタ配列のアドレス																//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BCP	*	WINAPI	AjcParseCmdArgsExA(C_BCP pCmd, UIP pArgC, BOOL fRemoveQuote)
{
	VOP		rc = NULL;
	UI		argc = 0;
	UI		stl;
	BCP		argv = NULL;
	C_BCP	p;
	BCP		ps;
	BCP		*pv;
	UI		BufSize = 0;

	if (pCmd != NULL) {
		//----- 引数個数カウント --------------------//
		p = pCmd;
		while (MAjcIsBlankA(*p)) p++;
		while (*p != 0) {
			stl = 0;
			while (!MAjcIsBlankA(*p) && *p != 0) {
				if (*p == '"') {
					p++; if (!fRemoveQuote) stl++;
					while (*p != '"' && *p != 0) {stl++; p++;}
					if (*p == '"') {p++; if (!fRemoveQuote) stl++;}
				}
				else {
					while (!MAjcIsBlankA(*p) && *p != '"' && *p != 0) {stl++; p++;}
				}
			}
			argc++;
			BufSize += (sizeof(VOP) + stl + 1);
			while (MAjcIsBlankA(*p)) p++;
		}
		BufSize += sizeof(VOP);

		if (rc = (VOP)AJCMEM(BufSize)) {
			pv = (BCP*)rc;
			ps = (UBP)(((UBP*)rc) + (argc + 1));
			//----- 引数配列作成 ---------------------//
			p = pCmd;
			while (MAjcIsBlankA(*p)) p++;
			while (*p != 0) {
				*pv++ = ps;
				while (!MAjcIsBlankA(*p) && *p != 0) {
					if (*p == '"') {
						if (fRemoveQuote) p++;
						else			  *ps++ = *p++;
						while (*p != '"' && *p != 0) {*ps++ = *p++;}
						if (*p == '"') {
							if (fRemoveQuote) p++;
							else			  *ps++ = *p++;
						}
					}
					else {
						while (!MAjcIsBlankA(*p) && *p != '"' && *p != 0) {*ps++ = *p++;}
					}
				}
				*ps++ = 0;
				while (MAjcIsBlankA(*p)) p++;
			}
			*pv = NULL;
		}

		if (pArgC != NULL) {
			*pArgC = argc;
		}
	}
	return (BCP *)rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	WCP	*	WINAPI	AjcParseCmdArgsExW(C_WCP pCmd, UIP pArgC, BOOL fRemoveQuote)
{
	VOP		rc = NULL;
	UI		argc = 0;
	UI		stl;
	WCP		argv = NULL;
	C_WCP	p;
	WCP		ps;
	WCP		*pv;
	UI		BufSize = 0;

	if (pCmd != NULL) {
		//----- 引数個数カウント --------------------//
		p = pCmd;
		while (MAjcIsBlankW(*p)) p++;
		while (*p != 0) {
			stl = 0;
			while (!MAjcIsBlankW(*p) && *p != 0) {
				if (*p == L'"') {
					p++; if (!fRemoveQuote) stl++;
					while (*p != L'"' && *p != 0) {stl++; p++;}
					if (*p == L'"') {p++; if (!fRemoveQuote) stl++;}
				}
				else {
					while (!MAjcIsBlankW(*p) && *p != L'"' && *p != 0) {stl++; p++;}
				}
			}
			argc++;
			BufSize += (sizeof(VOP) + (stl + 1) * 2);
			while (MAjcIsBlankW(*p)) p++;
		}
		BufSize += sizeof(VOP);

		if (rc = (VOP)AJCMEM(BufSize)) {
			pv = (WCP*)rc;
			ps = (WCP)(((WCP*)rc) + (argc + 1));
			//----- 引数配列作成 ---------------------//
			p = pCmd;
			while (MAjcIsBlankW(*p)) p++;
			while (*p != 0) {
				*pv++ = ps;
				while (!MAjcIsBlankW(*p) && *p != 0) {
					if (*p == L'"') {
						if (fRemoveQuote) p++;
						else			  *ps++ = *p++;
						while (*p != L'"' && *p != 0) {*ps++ = *p++;}
						if (*p == L'"') {
							if (fRemoveQuote) p++;
							else			  *ps++ = *p++;
						}
					}
					else {
						while (!MAjcIsBlankW(*p) && *p != L'"' && *p != 0) {*ps++ = *p++;}
					}
				}
				*ps++ = 0;
				while (MAjcIsBlankW(*p)) p++;
			}
			*pv = NULL;
		}

		if (pArgC != NULL) {
			*pArgC = argc;
		}
	}
	return (WCP *)rc;
}
//==============================================================================================================//
//	Function	:	引数の配列を開放する																		//
//																												//
//	Argument	:	pCmd	- コマンドライン文字列のアドレス													//
//					pArgC	- 引数の個数を格納するバッファのアドレス											//
//																												//
//	Return		:	引数へのポインタ配列のアドレス																//
//==============================================================================================================//
AJCEXPORT	VO		WINAPI	AjcReleaseCmdArgs(VOP pArgV)
{
	if (pArgV != NULL) {
		free(pArgV);
	}
}
//==============================================================================================================//
//	ＣＰＵのＩＤ情報取得																						//
//																												//
//	引　数：	pSerialNoH	- プロセッサシリアル番号の上位３２ビットを格納するバッファのアドレス				//
//				pSerialNoL	- プロセッサシリアル番号の下位６４ビットを格納するバッファのアドレス				//
//																												//
//	戻り値：	≠NULL : ベンダ名文字列のアドレス																//
//				＝NULL : ＣＰＵのＩＤ情報取得不可																//
//==============================================================================================================//
#ifndef _WIN64
//---- バイト文字 ----------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcGetCpuIdA(UIP pSignature, UIP pParam, ULLP pSerialNo)
{
	BCP		rc = NULL;
	UI		aftEF, bfrEF, MaxId;
	UI				  Sig, Prm;
	union {
		struct {UI l, h;} s;
		ULL	ll;
	} idL;

	static struct {UI b1, b2, b3, e;} vn;


	Sig	   = 0;
	Prm	   = 0;
	idL.ll = 0;
	memset(&vn, 0, sizeof vn);

	_asm {
		pushfd
		pop		eax
		mov		bfrEF, eax
		xor		eax, 00200000h
		push	eax
		popfd
		pushfd
		pop		eax
		mov		aftEF, eax
	}

	if (aftEF != bfrEF) {
		_asm {
			mov		eax,0
			cpuid
			mov		MaxId, eax;
			mov		vn.b1, ebx
			mov		vn.b2, edx
			mov		vn.b3, ecx
		}
		if (MaxId >= 1) {
			_asm {
				mov		eax, 1
				cpuid
				mov		Sig, eax;
			}
		}
		if (MaxId >= 2) {
			_asm {
				mov		eax, 2
				cpuid
				mov		Prm, eax;
			}
		}
		if (MaxId >= 3) {
			_asm {
				mov		eax, 3
				cpuid
				mov		idL.s.l, ecx;
				mov		idL.s.h, edx;
			}
		}
		rc = (UBP)&vn;
	}

	if (pSignature != NULL) *pSignature = Sig;
	if (pParam	   != NULL) *pParam		= Prm;
	if (pSerialNo  != NULL) *pSerialNo	= idL.ll;

	return rc;
}
//---- ワイド文字 ----------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcGetCpuIdW(UIP pSignature, UIP pParam, ULLP pSerialNo)
{
	WCP		rc;
	BCP		pvn;

	static	UW	str[13];

	if (pvn = AjcGetCpuIdA(pSignature, pParam, pSerialNo)) {
		MultiByteToWideChar(CP_ACP, 0, pvn, -1, str, 13);
		rc = str;
	}
	else {
		rc = NULL;
	}
	return rc;
}
#endif
//==============================================================================================================//
//	２つの角度の差を求める																						//
//																												//
//	引　数：	t1, t2	- 差を求める２つの角度																	//
//																												//
//	戻り値	：	t1～t2の角度差																					//
//==============================================================================================================//
AJCEXPORT	double		WINAPI	AjcDifferenceOfTheta(double t1, double t2)
{
	double	rc;
	double	t, x, y;

	t = t2 - t1;
	t = t / 180.0 * AJC_PAI;
	x = cos(t);
	y = sin(t);
	rc = atan2(y, x);
	rc = rc * 180.0 / AJC_PAI;

	return rc;
}
//==============================================================================================================//
//	ＤＬＬのエクスポート情報取得																				//
//																												//
//	引　数	：	hDll - ＤＬＬのモジュールハンドル																//
//				pNum - エクスポート情報の個数を格納するバッファのアドレス										//
//																												//
//	戻り値	：	≠NULL：ＤＬＬエクスポート情報（配列）のアドレス												//
//				＝NULL：エラー／エクスポート情報なし															//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	PAJCDLLEXPORTINFOA	WINAPI	AjcCreateDllExportInfoA(HMODULE hDll,  UIP pNum)
{
	PAJCDLLEXPORTINFOA	rc = NULL;
	PAJCDLLEXPORTINFOA	p  = NULL;
	UL		nSize;
	UI		i, j;
	ULP		pxFunctionName;
	WCP		pwFunctionOrdinal;
	ULP		pxFuncAddr;
	IMAGE_EXPORT_DIRECTORY*	pImageExportDir;

	do {
		if (hDll == NULL) break;

		pImageExportDir = (IMAGE_EXPORT_DIRECTORY*)ImageDirectoryEntryToData((VOP)hDll,
																			 TRUE,
																			 IMAGE_DIRECTORY_ENTRY_EXPORT,
																			 &nSize);
		if (pImageExportDir == NULL) break;

		pxFunctionName		= (ULP)(pImageExportDir->AddressOfNames		   + (UX)hDll);
		pwFunctionOrdinal	= (WCP)(pImageExportDir->AddressOfNameOrdinals + (UX)hDll);
		pxFuncAddr			= (ULP)(pImageExportDir->AddressOfFunctions	   + (UX)hDll);

		if (pImageExportDir->NumberOfFunctions == 0) break;

		rc = (PAJCDLLEXPORTINFOA)AJCMEM(sizeof(AJCDLLEXPORTINFOA) * (pImageExportDir->NumberOfFunctions + 1));
		if (rc == NULL) break;

		memset(rc, 0, sizeof(AJCDLLEXPORTINFOA) * (pImageExportDir->NumberOfFunctions + 1));

		for(i = 0, p = rc; i < pImageExportDir->NumberOfFunctions; i++, p++)	{
			p->Ordinal = pImageExportDir->Base + i;
			for (j = 0; j < pImageExportDir->NumberOfNames; j++) {
				if (pwFunctionOrdinal[j] == i) {
					p->pFuncName = (C_BCP)(pxFunctionName[j] + (UX)hDll);
					break;
				}
			}
			p->pFuncAddr = (VOP)(pxFuncAddr[i] + (UX)hDll);
		}
		if (pNum != NULL) *pNum = pImageExportDir->NumberOfFunctions;
	} while(0);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	PAJCDLLEXPORTINFOW	WINAPI	AjcCreateDllExportInfoW(HMODULE hDll,  UIP pNum)
{
	PAJCDLLEXPORTINFOW	rc = NULL;
	PAJCDLLEXPORTINFOW	p  = NULL;
	UL		nSize;
	UI		i, j;
	UXP		pxFunctionName;
	WCP		pwFunctionOrdinal;
	UXP		pxFuncAddr;
	IMAGE_EXPORT_DIRECTORY*	pImageExportDir;

	do {
		if (hDll == NULL) break;

		pImageExportDir = (IMAGE_EXPORT_DIRECTORY*)ImageDirectoryEntryToData((VOP)hDll,
																			 TRUE,
																			 IMAGE_DIRECTORY_ENTRY_EXPORT,
																			 &nSize);
		if (pImageExportDir == NULL) break;

		pxFunctionName		= (UXP)(pImageExportDir->AddressOfNames		   + (UX)hDll);
		pwFunctionOrdinal	= (WCP)(pImageExportDir->AddressOfNameOrdinals + (UX)hDll);
		pxFuncAddr			= (UXP)(pImageExportDir->AddressOfFunctions	   + (UX)hDll);

		if (pImageExportDir->NumberOfFunctions == 0) break;

		rc = (PAJCDLLEXPORTINFOW)AJCMEM(sizeof(AJCDLLEXPORTINFOA) * (pImageExportDir->NumberOfFunctions + 1));
		if (rc == NULL) break;

		memset(rc, 0, sizeof(AJCDLLEXPORTINFOA) * (pImageExportDir->NumberOfFunctions + 1));

		for(i = 0, p = rc; i < pImageExportDir->NumberOfFunctions; i++, p++)	{
			p->Ordinal = pImageExportDir->Base + i;
			for (j = 0; j < pImageExportDir->NumberOfNames; j++) {
				if (pwFunctionOrdinal[j] == i) {
					C_BCP	pTxt = (C_BCP)(pxFunctionName[j] + (UX)hDll);
					int		len	 = MultiByteToWideChar(CP_ACP, 0, pTxt, -1, NULL, 0);
					p->pFuncName = (C_WCP)AJCMEM(len * 2);
					if (p->pFuncName == NULL) {
						AjcReleaseDllExportInfoW(rc);
						rc = NULL;
						goto dcdei_exit;
					}
					MultiByteToWideChar(CP_ACP, 0, pTxt, -1, (WCP)p->pFuncName, len);
					break;
				}
			}
			p->pFuncAddr = (VOP)(pxFuncAddr[i] + (UX)hDll);
		}
dcdei_exit:;
		if (pNum != NULL) *pNum = pImageExportDir->NumberOfFunctions;
	} while(0);

	return rc;
}

//==============================================================================================================//
//	ＤＬＬのエクスポート情報の開放																				//
//																												//
//	引　数	：	pInfo - エクスポート情報のアドレス（AjcCreateDllExportInfoの戻り値）							//
//																												//
//	戻り値	：	≠NULL：ＤＬＬエクスポート情報（配列）のアドレス												//
//				＝NULL：エラー／エクスポート情報なし															//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	VO		WINAPI	AjcReleaseDllExportInfoA(PAJCDLLEXPORTINFOA pInfo)
{
	if (pInfo != NULL) {
		free(pInfo);
	}
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	VO		WINAPI	AjcReleaseDllExportInfoW(PAJCDLLEXPORTINFOW pInfo)
{
	PAJCDLLEXPORTINFOW	p = pInfo;

	if (pInfo != NULL) {
		while (p->pFuncAddr != NULL) {
			if (p->pFuncName != NULL) {
				free((VOP)p->pFuncName);
			}
			p++;
		}
		free(pInfo);
	}
}
//==============================================================================================================//
//	モジュールのインポート情報取得																				//
//																												//
//	引　数	：	hMod - モジュールハンドル																		//
//				pNum - インポート情報の個数を格納するバッファのアドレス											//
//																												//
//	戻り値	：	≠NULL：モジュールのインポート情報（配列）のアドレス											//
//				＝NULL：エラー／インポート情報なし																//
//==============================================================================================================//
#ifndef _WIN64
	#define		IMAGE_ORDINAL_FLAG	IMAGE_ORDINAL_FLAG32
#else
	#define		IMAGE_ORDINAL_FLAG	IMAGE_ORDINAL_FLAG64
#endif
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	PAJCMODIMPORTINFOA	WINAPI	AjcCreateModImportInfoA(HMODULE hMod, UIP pNum)
{
	PAJCMODIMPORTINFOA	rc = NULL;
	PAJCMODIMPORTINFOA	p;
	UL	Size, nInfo;
	IMAGE_IMPORT_DESCRIPTOR	*pImpDesc;
	IMAGE_THUNK_DATA		*pThunkINT, *pThunkIAT;
	IMAGE_IMPORT_BY_NAME	*pImpByName;

	do {
		//------------------------------------------------------------------------------------------------------//
		//	インポート情報の個数カウント																		//
		//------------------------------------------------------------------------------------------------------//
		// ベースアドレスからインポートデスクリプタのアドレスを得る
		pImpDesc = (IMAGE_IMPORT_DESCRIPTOR*)ImageDirectoryEntryToData(hMod,
																	   TRUE,
																	   IMAGE_DIRECTORY_ENTRY_IMPORT,
																	   &Size);
		nInfo = 0;
		while (pImpDesc->Name) {
			pThunkINT = (IMAGE_THUNK_DATA*)((BYTE*)hMod + pImpDesc->OriginalFirstThunk);
			while (pThunkINT->u1.Function) {
				nInfo++;
				pThunkINT++;
			}
			pImpDesc++;
		}
		if (nInfo == 0) break;

		//------------------------------------------------------------------------------------------------------//
		//	インポート情報テーブル領域確保																		//
		//------------------------------------------------------------------------------------------------------//
		rc = (PAJCMODIMPORTINFOA)AJCMEM(sizeof(AJCMODIMPORTINFOA) * (nInfo + 1));
		if (rc == NULL) break;
		memset(rc, 0, sizeof(AJCMODIMPORTINFOA) * (nInfo + 1));

		//------------------------------------------------------------------------------------------------------//
		//	インポート情報設定																					//
		//------------------------------------------------------------------------------------------------------//
		pImpDesc = (IMAGE_IMPORT_DESCRIPTOR*)ImageDirectoryEntryToData(hMod,
																	   TRUE,
																	   IMAGE_DIRECTORY_ENTRY_IMPORT,
																	   &Size);
		p = rc;
		while (pImpDesc->Name) {
			BCP	pDllName = (BCP)((BCP)hMod + pImpDesc->Name);

			pThunkINT = (IMAGE_THUNK_DATA*)((BYTE*)hMod + pImpDesc->OriginalFirstThunk);
			pThunkIAT = (IMAGE_THUNK_DATA*)((BYTE*)hMod + pImpDesc->FirstThunk);
			while (pThunkINT->u1.Function) {
				p->pDllName = pDllName;
				p->pFuncAddr = (VOP)pThunkIAT->u1.Function;
				if (pThunkINT->u1.AddressOfData & IMAGE_ORDINAL_FLAG) {		// シンボルが序数情報？
					p->Ordinal	 = (int)(pThunkINT->u1.AddressOfData & (~IMAGE_ORDINAL_FLAG));
					p->pFuncName = NULL;
				}
				else {														//	シンボルが名前情報？
					p->Ordinal	 = 0;
					pImpByName	 = (IMAGE_IMPORT_BY_NAME*)((BYTE*)hMod+ pThunkINT->u1.AddressOfData);
					p->pFuncName = (BCP)(pImpByName->Name);
				}
				pThunkINT++;
				pThunkIAT++;
				p++;
			}
			pImpDesc++;
		}
		if (pNum != NULL) *pNum = nInfo;
	} while(0);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	PAJCMODIMPORTINFOW	WINAPI	AjcCreateModImportInfoW(HMODULE hMod, UIP pNum)
{
	PAJCMODIMPORTINFOW	rc = NULL;
	PAJCMODIMPORTINFOW	p;
	UL	Size, nInfo;
	IMAGE_IMPORT_DESCRIPTOR	*pImpDesc;
	IMAGE_THUNK_DATA		*pThunkINT, *pThunkIAT;
	IMAGE_IMPORT_BY_NAME	*pImpByName;

	do {
		//------------------------------------------------------------------------------------------------------//
		//	インポート情報の個数カウント																		//
		//------------------------------------------------------------------------------------------------------//
		// ベースアドレスからインポートデスクリプタのアドレスを得る
		pImpDesc = (IMAGE_IMPORT_DESCRIPTOR*)ImageDirectoryEntryToData(hMod,
																	   TRUE,
																	   IMAGE_DIRECTORY_ENTRY_IMPORT,
																	   &Size);
		nInfo = 0;
		while (pImpDesc->Name) {
			pThunkINT = (IMAGE_THUNK_DATA*)((BYTE*)hMod + pImpDesc->OriginalFirstThunk);
			while (pThunkINT->u1.Function) {
				nInfo++;
				pThunkINT++;
			}
			pImpDesc++;
		}
		if (nInfo == 0) break;

		//------------------------------------------------------------------------------------------------------//
		//	インポート情報テーブル領域確保																		//
		//------------------------------------------------------------------------------------------------------//
		rc = (PAJCMODIMPORTINFOW)AJCMEM(sizeof(AJCMODIMPORTINFOA) * (nInfo + 1));
		if (rc == NULL) break;
		memset(rc, 0, sizeof(AJCMODIMPORTINFOA) * (nInfo + 1));

		pImpDesc = (IMAGE_IMPORT_DESCRIPTOR*)ImageDirectoryEntryToData(hMod,
																	   TRUE,
																	   IMAGE_DIRECTORY_ENTRY_IMPORT,
																	   &Size);
		p = rc;
		while (pImpDesc->Name) {
			int	len;
			BCP	pTxt;
			WCP	pDllName;

			pTxt = (BCP)((BCP)hMod + pImpDesc->Name);
			len = MultiByteToWideChar(CP_ACP, 0, pTxt, -1, NULL, 0);
			if ((pDllName = (WCP)AJCMEM(len * 2)) == NULL) {
				AjcReleaseModImportInfoW(rc);
				rc = NULL;
				break;
			}
			MultiByteToWideChar(CP_ACP, 0, pTxt, -1, (WCP)p->pDllName, len);

			pThunkINT = (IMAGE_THUNK_DATA*)((BYTE*)hMod + pImpDesc->OriginalFirstThunk);
			pThunkIAT = (IMAGE_THUNK_DATA*)((BYTE*)hMod + pImpDesc->FirstThunk);
			while (pThunkINT->u1.Function) {
				p->pDllName = pDllName;
				p->pFuncAddr = (VOP)pThunkIAT->u1.Function;
				if (pThunkINT->u1.AddressOfData & 0x80000000) {		// シンボルが序数情報？
					p->Ordinal	 = pThunkINT->u1.AddressOfData & ~0x80000000;
					p->pFuncName = NULL;
				}
				else {												//	シンボルが名前情報？
					p->Ordinal	 = 0;
					pImpByName	 = (IMAGE_IMPORT_BY_NAME*)((BYTE*)hMod+ pThunkINT->u1.AddressOfData);
					pTxt		 = (BCP)(pImpByName->Name);
					len			 = MultiByteToWideChar(CP_ACP, 0, pTxt, -1, NULL, 0);
					if ((p->pFuncName = (C_WCP)AJCMEM(len * 2)) == NULL){
						AjcReleaseModImportInfoW(rc);
						rc = NULL;
						goto CmiA_Exit;
					}
					MultiByteToWideChar(CP_ACP, 0, pTxt, -1, (WCP)p->pFuncName, len);
				}
				pThunkINT++;
				pThunkIAT++;
				p++;
			}
			pImpDesc++;
		}
CmiA_Exit:;
		if (pNum != NULL) *pNum = nInfo;
	} while(0);

	return rc;
}
//==============================================================================================================//
//	モジュールのインポート情報開放																				//
//																												//
//	引　数	：	pInfo - インポート情報のアドレス（AjcCreateModImportInfoの戻り値）								//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	VO		WINAPI	AjcReleaseModImportInfoA(PAJCMODIMPORTINFOA pInfo)
{
	if (pInfo != NULL) {
		free(pInfo);
	}
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	VO		WINAPI	AjcReleaseModImportInfoW(PAJCMODIMPORTINFOW pInfo)
{
	PAJCMODIMPORTINFOW	p	   = pInfo;
	C_WCP				pSvDll = NULL;

	if (pInfo != NULL) {
		while (p->pFuncAddr != NULL) {
			if (p->pDllName != NULL && p->pDllName != pSvDll) {
				pSvDll = p->pDllName;
				free((VOP)p->pDllName);
			}
			if (p->pFuncName != NULL) {
				free((VOP)p->pFuncName);
			}
			p++;
		}
		free(pInfo);
	}
}

//==============================================================================================================//
//	Windowsのシャットダウンやリブートを行う																		//
//																												//
//	引　数	：	flag	- 動作フラグ（EWX_LOGOFF, EWX_POWEROFF, EWX_REBOOT, EWX_SHUTDOWN |						//
//									   EWX_FORCE, EWX_FORCEIFHUNG）												//
//				reason	- シャットダウンを開始する理由（詳細はMSDN参照）										//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcExitWindows	 (UI flag)
{
	return AjcExitWindowsEx (flag, SHTDN_REASON_MAJOR_APPLICATION |
								   SHTDN_REASON_MINOR_MAINTENANCE |
								   SHTDN_REASON_FLAG_PLANNED);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	BOOL	WINAPI	AjcExitWindowsEx (UI flag, UI reason)
{
	BOOL	rc = TRUE;
	HINSTANCE	hDll = NULL;
	BOOL	(WINAPI *DllOpenProcessToken)(HANDLE ProcessHandle, DWORD DesiredAccess, PHANDLE TokenHandle) = NULL;
	BOOL	(WINAPI *DllLookupPrivilegeValue)(LPCTSTR lpSystemName, LPCTSTR lpName, PLUID lpLuid) = NULL;
	BOOL	(WINAPI *DllAdjustTokenPrivileges)(HANDLE TokenHandle, BOOL DisableAllPrivileges, PTOKEN_PRIVILEGES NewState, DWORD BufferLength, PTOKEN_PRIVILEGES PreviousState, PDWORD ReturnLength) = NULL;
	HANDLE			 htkThread;
	TOKEN_PRIVILEGES tkPriv;

	do {
		if (!(hDll = LoadLibrary(TEXT("advapi32.dll")))) break;
		if (!(DllOpenProcessToken		= (VOP)GetProcAddress(hDll, "OpenProcessToken"	   ))) break;
		if (!(DllLookupPrivilegeValue	= (VOP)GetProcAddress(hDll, "LookupPrivilegeValueW"))) break;
		if (!(DllAdjustTokenPrivileges	= (VOP)GetProcAddress(hDll, "AdjustTokenPrivileges"))) break;

		//--- プロセスに関連付けられたアクセストークンをオープンする ---------//
		if (rc) rc &= (DllOpenProcessToken(GetCurrentProcess(), TOKEN_ADJUST_PRIVILEGES, &htkThread) != 0);
		//--- 追加（変更）する特権の数 ---------------------------------------//
		tkPriv.PrivilegeCount = 1;
		//--- ローカルシステム内の SE_SHUTDOWN_NAME 特権の LUID を取得する ---//
		if (rc) rc &= (DllLookupPrivilegeValue(NULL, SE_SHUTDOWN_NAME, &tkPriv.Privileges[0].Luid) != 0);
		//--- 特権を使用可能にする指定をする ---------------------------------//
		tkPriv.Privileges[0].Attributes = SE_PRIVILEGE_ENABLED;
		//--- アクセストークンに特権情報を追加する ---------------------------//
		if (rc) rc &= (DllAdjustTokenPrivileges(htkThread, FALSE, &tkPriv, 0, NULL, NULL) != 0);
	} while(0);

	//--- DLL 開放 -------------------------------------------------------//
	if (hDll != NULL) FreeLibrary(hDll);

	//--- リブート等の実行 -----------------------------------------------//
	if (rc) {
		rc = (ExitWindowsEx(flag, reason) != 0);
	}
	return rc;
}

//==============================================================================================================//
//	電源をシャットダウンしてシステムを中断します																//
//																												//
//	引　数	：	fSuspend	- TRUE : 中断，FALSE：休止															//
//				fForce		- TRUE : 強制的に実行，FALSE:通常													//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcSetSystemPowerState (BOOL fSuspend, BOOL fForce)
{
	BOOL	rc = TRUE;
	HINSTANCE	hDll = NULL;
	BOOL	(WINAPI *DllOpenProcessToken)(HANDLE ProcessHandle, DWORD DesiredAccess, PHANDLE TokenHandle) = NULL;
	BOOL	(WINAPI *DllLookupPrivilegeValue)(LPCTSTR lpSystemName, LPCTSTR lpName, PLUID lpLuid) = NULL;
	BOOL	(WINAPI *DllAdjustTokenPrivileges)(HANDLE TokenHandle, BOOL DisableAllPrivileges, PTOKEN_PRIVILEGES NewState, DWORD BufferLength, PTOKEN_PRIVILEGES PreviousState, PDWORD ReturnLength) = NULL;
	HANDLE			 htkThread;
	TOKEN_PRIVILEGES tkPriv;

	do {
		if (!(hDll = LoadLibraryA("advapi32.dll"))) break;
		if (!(DllOpenProcessToken		= (VOP)GetProcAddress(hDll, "OpenProcessToken"	   ))) break;
		if (!(DllLookupPrivilegeValue	= (VOP)GetProcAddress(hDll, "LookupPrivilegeValueA"))) break;
		if (!(DllAdjustTokenPrivileges	= (VOP)GetProcAddress(hDll, "AdjustTokenPrivileges"))) break;

		//--- プロセスに関連付けられたアクセストークンをオープンする ---------//
		if (rc) rc &= (DllOpenProcessToken(GetCurrentProcess(), TOKEN_ADJUST_PRIVILEGES, &htkThread) != 0);
		//--- 追加（変更）する特権の数 ---------------------------------------//
		tkPriv.PrivilegeCount = 1;
		//--- ローカルシステム内の SE_SHUTDOWN_NAME 特権の LUID を取得する ---//
		if (rc) rc &= (DllLookupPrivilegeValue(NULL, SE_SHUTDOWN_NAME, &tkPriv.Privileges[0].Luid) != 0);
		//--- 特権を使用可能にする指定をする ---------------------------------//
		tkPriv.Privileges[0].Attributes = SE_PRIVILEGE_ENABLED;
		//--- アクセストークンに特権情報を追加する ---------------------------//
		if (rc) rc &= (DllAdjustTokenPrivileges(htkThread, FALSE, &tkPriv, 0, NULL, NULL) != 0);
	} while(0);

	//--- DLL 開放 -------------------------------------------------------//
	if (hDll != NULL) FreeLibrary(hDll);

	//--- リブート等の実行 -----------------------------------------------//
	if (rc) {
		rc = SetSystemPowerState(fSuspend, fForce);
	}
	return rc;
}
//==============================================================================================================//
//	現在のユーザのＳＩＤ文字列を取得する																		//
//																												//
//	引　数：	pSidBuf	- ＳＩＤ文字列を格納するバッファのアドレス												//
//				lSidBuf	- ＳＩＤ文字列を格納するバッファの文字数												//
//																												//
//	戻り値	：	≠０：成功（ＳＩＤ文字列の長さ，文字列終端を含む）												//
//				＝０：失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI AjcGetSidStringByCurrentUserA(BCP pSidBuf, UI lSidBuf)
{
	UI				rc = 0;
	BC				UserName[128];
	DWORD			UserNameLen = AJCTSIZE(UserName);

	BC				szDomain[256];
	UI				dwSidLen	= 0;
	UI				dwDomainLen = AJCTSIZE(szDomain);
	SID_NAME_USE	snu;
	PSID			pSid 		= NULL;
	BCP				pStringSid	= NULL;

	do {
		//----- アカウント名（ユーザ名）取得 ---------//
		if (!GetUserNameA(UserName, &UserNameLen)) break;
		//----- SID長取得 ----------------------------//
		LookupAccountNameA(	NULL,							//	コンピュータ名（ＮＵＬＬ：ローカルコンピュータ）
							UserName,						//	アカウント名
							NULL,							//	SID格納バッファ
							&dwSidLen,						//	SID長
							szDomain,						//	ドメイン名格納バッファ
							&dwDomainLen,					//	ドメイン名格納バッファ長
							&snu);							//	アカウントのタイプ
		if (dwSidLen				  == 0	 ) break;
		if ((pSid = (PSID)AJCMEM(dwSidLen)) == NULL) break;
		//----- ＳＩＤ取得 ---------------------------//
		if (!LookupAccountNameA(NULL,						//	コンピュータ名（ＮＵＬＬ：ローカルコンピュータ）
								UserName,					//	アカウント名
								pSid,						//	SID格納バッファ
								&dwSidLen,					//	SID長
								szDomain,					//	ドメイン名格納バッファ
								&dwDomainLen,				//	ドメイン名格納バッファ長
								&snu)) {					//	アカウントのタイプ
			break;
		}
		//----- ＳＩＤを文字列に変換 -----------------//
		ConvertSidToStringSidA(pSid, &pStringSid);
		//----- ＳＩＤをバッファに格納 ---------------//
		if (pSidBuf != NULL) {
			AjcSnPrintFA(pSidBuf, lSidBuf, "%s", pStringSid);
		}
		//----- 戻り値（ＳＩＤ長）設定 ---------------//
		rc = (UI)strlen(pStringSid);
	} while(0);

	//----- リソース解放 -----------------------------//
	if (pSid	   != NULL) free	 (pSid);
	if (pStringSid != NULL) LocalFree(pStringSid);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI AjcGetSidStringByCurrentUserW(WCP pSidBuf, UI lSidBuf)
{
	UI				rc = 0;
	WC				UserName[128];
	DWORD			UserNameLen = AJCTSIZE( UserName);

	WC				szDomain[256];
	UI				dwSidLen	= 0;
	UI				dwDomainLen = AJCTSIZE(szDomain);
	SID_NAME_USE	snu;
	PSID			pSid 		= NULL;
	WCP				pStringSid	= NULL;

	do {
		//----- アカウント名（ユーザ名）取得 ---------//
		if (!GetUserNameW(UserName, &UserNameLen)) break;
		//----- SID長取得 ----------------------------//
		LookupAccountNameW(	NULL,							//	コンピュータ名（ＮＵＬＬ：ローカルコンピュータ）
							UserName,						//	アカウント名
							NULL,							//	SID格納バッファ
							&dwSidLen,						//	SID長
							szDomain,						//	ドメイン名格納バッファ
							&dwDomainLen,					//	ドメイン名格納バッファ長
							&snu);							//	アカウントのタイプ
		if (dwSidLen					  == 0	 ) break;
		if ((pSid = (PSID)AJCMEM(dwSidLen * 2)) == NULL) break;
		//----- ＳＩＤ取得 ---------------------------//
		if (!LookupAccountNameW(NULL,						//	コンピュータ名（ＮＵＬＬ：ローカルコンピュータ）
								UserName,					//	アカウント名
								pSid,						//	SID格納バッファ
								&dwSidLen,					//	SID長
								szDomain,					//	ドメイン名格納バッファ
								&dwDomainLen,				//	ドメイン名格納バッファ長
								&snu)) {					//	アカウントのタイプ
			break;
		}
		//----- ＳＩＤを文字列に変換 -----------------//
		ConvertSidToStringSidW(pSid, &pStringSid);
		//----- ＳＩＤをバッファに格納 ---------------//
		if (pSidBuf != NULL) {
			AjcSnPrintFW(pSidBuf, lSidBuf, L"%s", pStringSid);
		}
		//----- 戻り値（ＳＩＤ長）設定 ---------------//
		rc = (UI)wcslen(pStringSid);
	} while(0);

	//----- リソース解放 -----------------------------//
	if (pSid	   != NULL) free	 (pSid);
	if (pStringSid != NULL) LocalFree(pStringSid);

	return rc;
}
//==============================================================================================================//
//	ＳＩＤ文字列からＲＩＤ値取得																				//
//																												//
//	引　数：	pSid	- ＳＩＤ文字列のアドレス																//
//																												//
//	戻り値	：	≠０：成功（ＲＩＤ値）																			//
//				＝０：失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	int		WINAPI AjcGetRidInSidA(C_BCP pSid)
{
	int		rid  = 0;
	C_BCP	p; 
	C_BCP	pRid = NULL;

	if (pSid != NULL) {
		p = pSid;
		while (p = mbsstr(p, "-")) {
			p++;
			if (_ismbcdigit(*p)) {
				pRid = p;
			}
		}
		if (pRid != NULL) {
			rid = atoi(pRid);
		}
	}
	return rid;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	int		WINAPI AjcGetRidInSidW(C_WCP pSid)
{
	int		rid  = 0;
	C_WCP	p; 
	C_WCP	pRid = NULL;

	if (pSid != NULL) {
		p = pSid;
		while (p = wcsstr(p, L"-")) {
			p++;
			if (MAjcAscIsDigitW(*p)) {
				pRid = p;
			}
		}
		if (pRid != NULL) {
			rid = _wtoi(pRid);
		}
	}
	return rid;
}
//==============================================================================================================//
//	２つの矩形の重なり合う部分を取得する（整数）																//
//																												//
//	引　数	：	pRect1, pRect2 	- ２つの矩形情報																//
//				pDupRect		- ２つの矩形が重なり合う部分の矩形情報を格納するバッファ（不要時はNULL)			//
//																												//
//	戻り値	：	重なり合う部分の面積（ピクセル数）																//
//==============================================================================================================//
AJCEXPORT	UI	WINAPI AjcGetDupRect(LPCRECT pRect1, LPCRECT pRect2, LPRECT pDupRect)
{
	UI		sq = 0;
	RECT	rc1, rc2;
	int		w1, h1, w2, h2;
	int		sx, sy, ex, ey, w, h;

	if (pRect1 != NULL && pRect2 != NULL) {
		rc1.left   = __min(pRect1->left, pRect1->right );
		rc1.right  = __max(pRect1->left, pRect1->right );
		rc1.top    = __min(pRect1->top , pRect1->bottom);
		rc1.bottom = __max(pRect1->top , pRect1->bottom);
		w1		   = rc1.right - rc1.left;
		h1		   = rc1.bottom - rc1.top;

		rc2.left   = __min(pRect2->left, pRect2->right );
		rc2.right  = __max(pRect2->left, pRect2->right );
		rc2.top    = __min(pRect2->top , pRect2->bottom);
		rc2.bottom = __max(pRect2->top , pRect2->bottom);
		w2		   = rc2.right - rc2.left;
		h2		   = rc2.bottom - rc2.top;

		sx = __max(rc1.left, rc2.left);
		sy = __max(rc1.top , rc2.top );
		ex = __min(rc1.left + w1, rc2.left + w2);
		ey = __min(rc1.top	+ h1, rc2.top  + h2);
		w  = ex - sx;
		h  = ey - sy;
		if (w > 0 && h > 0) {
			sq	= (w * h);
			if (pDupRect != NULL) {
				pDupRect->left	 = sx;
				pDupRect->right  = sx + w;
				pDupRect->top	 = sy;
				pDupRect->bottom = sy + h;
			}
		}
		else {
			sq = 0;
			if (pDupRect != NULL) {
				pDupRect->left	 = 0;
				pDupRect->right  = 0;
				pDupRect->top	 = 0;
				pDupRect->bottom = 0;
			}
		}
	}
	return sq;
}
//==============================================================================================================//
//	アスペクト比を維持し拡大／縮小したサイズを求める															//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
//	引　数	：	pRectFrom - 入力側矩形情報																		//
//				pRectRef  - 出力側矩形情報																		//
//				pRectTo   - アスペクト比を維持し拡大／縮小した矩形情報を格納するバッファのアドレス				//
//																												//
AJCEXPORT	BOOL	WINAPI	AjcAspGetZoomedRect(LPCRECT pRectSrc, LPCRECT pRectRef, LPRECT pRectBuf)
{
	BOOL	rc = FALSE;
	SIZE	szSrc, szRef, szBuf;

	if (pRectSrc != NULL && pRectRef != NULL && pRectBuf != NULL) {
		szSrc.cx = abs(pRectSrc->right	- pRectSrc->left);
		szSrc.cy = abs(pRectSrc->bottom - pRectSrc->top );
		szRef.cx = abs(pRectRef->right	- pRectRef->left);
		szRef.cy = abs(pRectRef->bottom - pRectRef->top );
		if (rc = AjcAspGetZoomedSize(&szSrc, &szRef, &szBuf)) {
			pRectBuf->left	 = pRectRef->left;
			pRectBuf->top	 = pRectRef->top;
			pRectBuf->right  = pRectRef->left + szBuf.cx;
			pRectBuf->bottom = pRectRef->top  + szBuf.cy;
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	引　数	：	pSizeFrom - 入力側サイズ情報																	//
//				pSizeRef  - 出力側サイズ情報																	//
//				pSizeTo   - アスペクト比を維持し拡大／縮小したサイズ情報を格納するバッファのアドレス			//
//																												//
AJCEXPORT	BOOL	WINAPI	AjcAspGetZoomedSize(const SIZE *pSizeSrc, const SIZE *pSizeRef, LPSIZE pSizeBuf)
{
	BOOL	rc = FALSE;
	double	zm, zmX, zmY;

	if (pSizeSrc != NULL && pSizeRef != NULL && pSizeBuf != NULL) {
		//	横倍率
		zmX = (double)pSizeRef->cx / (double)pSizeSrc->cx;
		//	縦倍率
		zmY = (double)pSizeRef->cy / (double)pSizeSrc->cy;
		//	倍率設定
		zm = __min(zmX, zmY);
		//	サイズ情報設定
		pSizeBuf->cx = (int)((double)pSizeSrc->cx * zm);
		pSizeBuf->cy = (int)((double)pSizeSrc->cy * zm);

		rc = TRUE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	引　数	：	cx		  - 入力側の幅																			//
//				cy		  - 出力側の幅																			//
//				MaxLen	  - 一辺の最大長																		//
//				pRectTo   - アスペクト比を維持し拡大／縮小した矩形情報を格納するバッファのアドレス				//
//																												//
AJCEXPORT	BOOL	WINAPI	AjcAspGetZoomedInfo(int cx, int cy, int MaxLen, LPSIZE pSizeBuf)
{
	BOOL	rc = FALSE;

	if (cx > 0 && cy > 0 && MaxLen > 0 && pSizeBuf != NULL) {
		//	入力側に合わせたサイズ設定
		if (cy >= cx) {pSizeBuf->cx = MaxLen * cx / cy; pSizeBuf->cy = MaxLen;			}		//	縦長
		else		  {pSizeBuf->cx = MaxLen;			pSizeBuf->cy = MaxLen * cy / cx;}		//	横長
		rc = TRUE;
	}
	return rc;
}

//==============================================================================================================//
//	Function	:	ベル音																						//
//																												//
//	Argument	:	なし																						//
//																												//
//	Return		:	なし																						//
//==============================================================================================================//
AJCEXPORT	VO	WINAPI	AjcBell(VO)
{
	PlaySound(L"Bell.wav", NULL, SND_FILENAME | SND_ASYNC);
}
//==============================================================================================================//
//	Function	:	親プロセスの取得																			//
//																												//
//	Argument	:	pid		- 子プロセスＩＤ（０の場合は、自プロセスＩＤを仮定）								//
//					pParent	- 親プロセスＩＤを格納するバッファのアドレス（不要時はＮＵＬＬ）					//
//					pBuf	- 親プロセスの実行ファイル名を格納するバッファのアドレス							//
//					lBuf	- 親プロセスの実行ファイル名を格納するバッファのバイト数／文字数					//
//																												//
//	Return		:	TRUE  : 成功（親プロセスの実行ファイル名を設定した）										//
//					FALSE : 親プロセスが見つからない															//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetParentProcessA(UL pid, ULP pParent, BCP pBuf, UI lBuf)
{
	BOOL	rc = FALSE;
	WC		path[MAX_PATH] = {0};

	if (rc = AjcGetParentProcess(pid, pParent, path, MAX_PATH)) {
		AjcWideCharToByteChar(path, pBuf, lBuf);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetParentProcessW(UL pid, ULP pParent, WCP pBuf, UI lBuf)
{
	BOOL 	rc = 0;

	HANDLE			hss = INVALID_HANDLE_VALUE;
	PROCESSENTRY32	pe;
	UL				parent = 0;

	do {
		//	引数チェック
		if (pBuf == NULL || lBuf == 0) {
			break;
		}

		//	プロセス・スナップショット生成
		if ((hss = CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS, 0)) == INVALID_HANDLE_VALUE) {
			break;
		}

		memset(&pe, 0, sizeof(PROCESSENTRY32));
		pe.dwSize = sizeof(PROCESSENTRY32);

		//	自プロセスＩＤ設定
		if (pid == 0) {
			pid = GetCurrentProcessId();
		}

		//	自プロセスエントリ検索
		if (Process32First(hss, &pe)) {
			do {
				if (pe.th32ProcessID == pid) {
					parent = pe.th32ParentProcessID;
					if (pParent != NULL) {
						*pParent = parent;
					}
					rc = TRUE;
					break;
				}
			} while (Process32Next(hss, &pe));
		}
		//	親プロセスエントリ検索（親実行ファイル名設定）
		if (parent != 0 && pBuf != NULL && lBuf != 0) {
			rc = FALSE;
			if (Process32First(hss, &pe)) {
				do {
					if (pe.th32ProcessID == parent) {
						AjcSnPrintF(pBuf, lBuf, L"%s", pe.szExeFile);
						rc = TRUE;
						break;
					}
				} while (Process32Next(hss, &pe));
			}
		}
	} while(0);

	//	スナップショット消去
	if (hss != INVALID_HANDLE_VALUE) {
		CloseHandle(hss);
	}

	return rc;
}
//==============================================================================================================//
//	Function	:	ストックフォント取得																		//
//																												//
//	Argument	:	fid		- フォント種別（AJCFID_VAR10 ～ AJCFID_VAR24 / AJCFID_FIX10 ～ AJCFID_FIX24）		//
//																												//
//	Return		:	≠NULL : 成功（フォントハンドル）															//
//					＝NULL : 失敗																				//
//==============================================================================================================//
AJCEXPORT	HFONT	WINAPI	AjcGetStockFont(AJCFONTID fid)
{
	HFONT	rc = NULL;

	switch (fid) {
		case AJCFID_FIX10:	rc = hDefFontFix10; break;
		case AJCFID_FIX12:	rc = hDefFontFix12; break;
		case AJCFID_FIX14:	rc = hDefFontFix14; break;
		case AJCFID_FIX16:	rc = hDefFontFix16; break;
		case AJCFID_FIX18:	rc = hDefFontFix18; break;
		case AJCFID_FIX20:	rc = hDefFontFix20; break;
		case AJCFID_FIX22:	rc = hDefFontFix22; break;
		case AJCFID_FIX24:	rc = hDefFontFix24; break;

		case AJCFID_VAR10:	rc = hDefFontVar10; break;
		case AJCFID_VAR12:	rc = hDefFontVar12; break;
		case AJCFID_VAR14:	rc = hDefFontVar14; break;
		case AJCFID_VAR16:	rc = hDefFontVar16; break;
		case AJCFID_VAR18:	rc = hDefFontVar18; break;
		case AJCFID_VAR20:	rc = hDefFontVar20; break;
		case AJCFID_VAR22:	rc = hDefFontVar22; break;
		case AJCFID_VAR24:	rc = hDefFontVar24; break;
	}

	return rc;
}
//==============================================================================================================//
//	Function	:	フォント生成，訂正																			//
//																												//
//	Argument	:	pLf		- フォント情報																		//
//																												//
//	Return		:	≠NULL : 成功（フォントハンドル）															//
//					＝NULL : 失敗																				//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	HFONT	WINAPI	AjcCreateFontA(LPLOGFONTA pLf)
{
	HFONT		rc = NULL;
	HDC			hdc, mdc;
	HFONT		hFSv;
	AJCCFATT	att;

	//	ＤＣ作成
	hdc  = GetDC(NULL);
	mdc  = CreateCompatibleDC(hdc);

	//	フォントハンドル生成
	if ((rc = CreateFontIndirectA(pLf)) == NULL) {
		LOGFONTA	lf;
		GetObject((HFONT)GetStockObject(SYSTEM_FONT), sizeof lf, &lf);
		rc = CreateFontIndirectA(&lf);
	}
	//	フォント情報訂正
	if (pLf != NULL && rc != NULL) {
		//	フォント割り当て
		hFSv = SelectObject(mdc, rc);
		//	フォント名設定（システムが認識できる名称に変更(ex. MS Gothic -> ＭＳ ゴシック)）
		GetTextFaceA(mdc, AJCTSIZE(pLf->lfFaceName), pLf->lfFaceName);
		//	フォントフェース名から、ピッチ＆ファミリ，キャラセット設定
		if (AjcCfFindFontA(pLf->lfFaceName, &att)) {
			pLf->lfPitchAndFamily = att.s.PitchAndFamily;
			pLf->lfCharSet		  = att.s.CharSet;
		}
		//	フォント解除
		SelectObject(mdc, hFSv);
	}
	//	ＤＣ解放
	DeleteDC(mdc);
	ReleaseDC(NULL, hdc);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	HFONT	WINAPI	AjcCreateFontW(LPLOGFONTW pLf)
{
	HFONT		rc = NULL;
	HDC			hdc, mdc;
	HFONT		hFSv;
	AJCCFATT	att;

	//	ＤＣ作成
	hdc  = GetDC(NULL);
	mdc  = CreateCompatibleDC(hdc);

	//	フォントハンドル生成
	if ((rc = CreateFontIndirectW(pLf)) == NULL) {
		LOGFONTW	lf;
		GetObject((HFONT)GetStockObject(SYSTEM_FONT), sizeof lf, &lf);
		rc = CreateFontIndirectW(&lf);
	}
	//	フォント情報訂正
	if (pLf != NULL && rc != NULL) {
		//	フォント割り当て
		hFSv = SelectObject(mdc, rc);
		//	フォント名設定（システムが認識できる名称に変更(ex. MS Gothic -> ＭＳ ゴシック)）
		GetTextFaceW(mdc, AJCTSIZE(pLf->lfFaceName), pLf->lfFaceName);
		//	フォントフェース名から、ピッチ＆ファミリ，キャラセット設定
		if (AjcCfFindFontW(pLf->lfFaceName, &att)) {
			pLf->lfPitchAndFamily = att.s.PitchAndFamily;
			pLf->lfCharSet		  = att.s.CharSet;
		}
		//	フォント解除
		SelectObject(mdc, hFSv);
	}
	//	ＤＣ解放
	DeleteDC(mdc);
	ReleaseDC(NULL, hdc);

	return rc;
}


//==============================================================================================================//
//	バージョン情報の取得																						//
//																												//
//	引　数	：	pPath 		- ファイルパス名（.exe / .dll）														//
//				pInfoName	- 取得する情報の名称																//
//				pInfoBuf	- 取得したバージョン情報を格納するバッファのアドレス（不要時はNULL)					//
//				lInfoBuf	- 取得したバージョン情報を格納するバッファのサイズ（バイト数／文字数）				//
//																												//
//	戻り値	：	≠０：成功（文字列を格納するのに必要なサイズ（バイト数／文字数））								//
//				＝０：失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcGetVerInfoA(C_BCP pPath, C_BCP pInfoName, BCP pInfoBuf, int lInfoBuf)
{
	UI		rc = 0;
	UL		dummy;
	UI		i;
	VOP		pRes = NULL;			//	バージョンリソースバッファのポインタ
	UI		lRes = 0;				//	バージョンリソースバッファのサイズ
	UI		defCodePage;			//	既定のコードページ情報
	UI		savCodePage = -1;		//	既定のコードページ検索結果
	UI		lCodePage;				//	コードページ情報のサイズ
	UI		nCodePage;				//	コードページ情報の個数
	UIP		pCodePage, pCp;			//	コードページ情報へのポインタ
	BCP		pInfo;					//	取得したバージョン情報へのポインタ
	UI		uLen;					//	取得したバージン情報のサイズ
	BC		szKey[256];				//	バージョン情報取得用キー文字列

	do {
		if ((lRes = GetFileVersionInfoSizeA(pPath, &dummy)) == 0) break;
		if ((pRes = (BCP)malloc(lRes)) == NULL) break;
		if (!GetFileVersionInfoA(pPath, 0, lRes, pRes)) break;
		if (!VerQueryValueA(pRes, "\\VarFileInfo\\Translation", (LPVOID*)&pCodePage, &lCodePage)) break;

		//	デフォルトコードページと同一コードページの情報インデクスを検索
		defCodePage = GetThreadLocale();
		nCodePage = lCodePage / sizeof(UI);
		for (i = 0, pCp = pCodePage; i < nCodePage; i++, pCp++) {
			if (LOWORD(*pCp) == defCodePage) {
				savCodePage = *pCp;
				break;
			}
		}
		//	デフォルトコードページと同一コードページがあれば、当該情報取得
		if (savCodePage != -1) {
			AjcSnPrintFA(szKey, AJCTSIZE(szKey), "\\StringFileInfo\\%04x%04x\\%s", LOWORD(savCodePage), HIWORD(savCodePage), pInfoName);
			if (VerQueryValueA(pRes, szKey, (LPVOID*)&pInfo, &uLen)) {
				rc = (UI)strlen(pInfo) + 1;
				if (pInfoBuf != NULL && lInfoBuf > 0) {
					AjcSnPrintFA(pInfoBuf, lInfoBuf, "%s", pInfo);
				}
				break;
			}
		}
		//	最初に見つかったバージョン情報を設定
		for (i = 0, pCp = pCodePage; i < nCodePage; i++, pCp++) {
			AjcSnPrintFA(szKey, AJCTSIZE(szKey), "\\StringFileInfo\\%04x%04x\\%s", LOWORD(*pCp), HIWORD(*pCp), pInfoName);
			if (VerQueryValueA(pRes, szKey, (LPVOID*)&pInfo, &uLen)) {
				rc = (UI)strlen(pInfo) + 1;
				if (pInfoBuf != NULL && lInfoBuf > 0) {
					AjcSnPrintFA(pInfoBuf, lInfoBuf, "%s", pInfo);
				}
				break;
			}
		}
	} while(0);

	//	リソース情報バッファ解放
	if (pRes != NULL) free(pRes);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcGetVerInfoW(C_WCP pPath, C_WCP pInfoName, WCP pInfoBuf, int lInfoBuf)
{
	UI		rc = 0;
	UL		dummy;
	UI		i;
	VOP		pRes = NULL;			//	バージョンリソースバッファのポインタ
	UI		lRes = 0;				//	バージョンリソースバッファのサイズ
	UI		defCodePage;			//	既定のコードページ情報
	UI		savCodePage = -1;		//	既定のコードページ検索結果
	UI		lCodePage;				//	コードページ情報のサイズ
	UI		nCodePage;				//	コードページ情報の個数
	UIP		pCodePage, pCp;			//	コードページ情報へのポインタ
	WCP		pInfo;					//	取得したバージョン情報へのポインタ
	UI		uLen;					//	取得したバージン情報のサイズ
	WC		szKey[256];				//	バージョン情報取得用キー文字列

	do {
		if ((lRes = GetFileVersionInfoSizeW(pPath, &dummy)) == 0) break;
		if ((pRes = (WCP)malloc(lRes)) == NULL) break;
		if (!GetFileVersionInfoW(pPath, 0, lRes, pRes)) break;
		if (!VerQueryValueW(pRes, L"\\VarFileInfo\\Translation", (LPVOID*)&pCodePage, &lCodePage)) break;

		//	デフォルトコードページと同一コードページの情報インデクスを検索
		defCodePage = GetThreadLocale();
		nCodePage = lCodePage / sizeof(UI);
		for (i = 0, pCp = pCodePage; i < nCodePage; i++, pCp++) {
			if (LOWORD(*pCp) == defCodePage) {
				savCodePage = *pCp;
				break;
			}
		}
		//	デフォルトコードページと同一コードページがあれば、当該情報取得
		if (savCodePage != -1) {
			AjcSnPrintFW(szKey, AJCTSIZE(szKey), L"\\StringFileInfo\\%04x%04x\\%s", LOWORD(savCodePage), HIWORD(savCodePage), pInfoName);
			if (VerQueryValueW(pRes, szKey, (LPVOID*)&pInfo, &uLen)) {
				rc = (UI)wcslen(pInfo) + 1;
				if (pInfoBuf != NULL && lInfoBuf > 0) {
					AjcSnPrintFW(pInfoBuf, lInfoBuf, L"%s", pInfo);
				}
				break;
			}
		}
		//	最初に見つかったバージョン情報を設定
		for (i = 0, pCp = pCodePage; i < nCodePage; i++, pCp++) {
			AjcSnPrintFW(szKey, AJCTSIZE(szKey), L"\\StringFileInfo\\%04x%04x\\%s", LOWORD(*pCp), HIWORD(*pCp), pInfoName);
			if (VerQueryValueW(pRes, szKey, (LPVOID*)&pInfo, &uLen)) {
				rc = (UI)wcslen(pInfo) + 1;
				if (pInfoBuf != NULL && lInfoBuf > 0) {
					AjcSnPrintFW(pInfoBuf, lInfoBuf, L"%s", pInfo);
				}
				break;
			}
		}
	} while(0);

	//	リソース情報バッファ解放
	if (pRes != NULL) free(pRes);

	return rc;
}
//==============================================================================================================//
//	クリップボードテキストの取得																				//
//																												//
//	引　数	：	pBuf		- クリップボードテキストを格納するバッファへのポインタ								//
//				lBuf		- クリップボードテキストを格納するバッファの文字数									//
//																												//
//	戻り値	：	≠０：テキストの文字数（終端(\0)を含む）														//
//				＝０：テキスト無し／失敗																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcGetClipboardTextA(BCP pBuf, UI lBuf)
{
	UI		rc = 0;

	if (IsClipboardFormatAvailable(CF_TEXT)) {
		HGLOBAL	hGlobal;
		BCP		pBlk;
		if (OpenClipboard(NULL)) {
			if (hGlobal = GetClipboardData(CF_TEXT)) {
				if (pBlk = (BCP)GlobalLock(hGlobal)) {
					if (pBuf != NULL && lBuf != 0) {
						strncpy(pBuf, pBlk, lBuf);
						pBuf[lBuf - 1] = 0;
					}
					rc = (UI)strlen(pBlk) + 1;
					GlobalUnlock(hGlobal);
				}
			}
			CloseClipboard();
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcGetClipboardTextW(WCP pBuf, UI lBuf)
{
	UI		rc = 0;

	if (IsClipboardFormatAvailable(CF_UNICODETEXT)) {
		HGLOBAL	hGlobal;
		WCP		pBlk;
		if (OpenClipboard(NULL)) {
			if (hGlobal = GetClipboardData(CF_UNICODETEXT)) {
				if (pBlk = (WCP)GlobalLock(hGlobal)) {
					if (pBuf != NULL && lBuf != 0) {
						wcsncpy(pBuf, pBlk, lBuf);
						pBuf[lBuf - 1] = 0;
					}
					rc = (UI)wcslen(pBlk) + 1;
					GlobalUnlock(hGlobal);
				}
			}
			CloseClipboard();
		}
	}
	return rc;
}

//==============================================================================================================//
//	クリップボードテキストの生成																				//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	≠NULL：生成したクリップボードテキストへのポインタ												//
//				＝NULL：テキスト無し／失敗																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcCreateClipboardTextA(VO)
{
	BCP		rc = NULL;

	if (IsClipboardFormatAvailable(CF_TEXT)) {
		HGLOBAL	hGlobal;
		BCP		pBlk, pMem;
		if (OpenClipboard(NULL)) {
			if (hGlobal = GetClipboardData(CF_TEXT)) {
				if (pBlk = (BCP)GlobalLock(hGlobal)) {
					if (pMem = AjcTAllocA((UI)strlen(pBlk) + 1)) {
						strcpy(pMem, pBlk);
						rc = pMem;
					}
					GlobalUnlock(hGlobal);
				}
			}
			CloseClipboard();
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcCreateClipboardTextW(VO)
{
	WCP		rc = NULL;

	if (IsClipboardFormatAvailable(CF_UNICODETEXT)) {
		HGLOBAL	hGlobal;
		WCP		pBlk, pMem;
		if (OpenClipboard(NULL)) {
			if (hGlobal = GetClipboardData(CF_UNICODETEXT)) {
				if (pBlk = (WCP)GlobalLock(hGlobal)) {
					if (pMem = AjcTAllocW((UI)wcslen(pBlk) + 1)) {
						wcscpy(pMem, pBlk);
						rc = pMem;
					}
					GlobalUnlock(hGlobal);
				}
			}
			CloseClipboard();
		}
	}
	return rc;
}

//==============================================================================================================//
//	クリップボードテキストの解放																				//
//																												//
//	引　数	：	pBuf	- AjcCreateClipboardText()で生成したクリップボードテキストへのポインタ					//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcReleaseClipboardTextA(BCP pTxt)
{
	BOOL	rc = FALSE;

	if (pTxt != NULL) {
		AjcTFree(pTxt);
		rc = TRUE;
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcReleaseClipboardTextW(WCP pTxt)
{
	BOOL	rc = FALSE;

	if (pTxt != NULL) {
		AjcTFree(pTxt);
		rc = TRUE;
	}
	return rc;
}

//==============================================================================================================//
//	クリップボードへテキスト格納																				//
//																												//
//	引　数	：	pTxt - クリップボードへ格納するテキストへのポインタ												//
//				lTxt - クリップボードへ格納するテキストの文字数（-1：自動設定）									//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPutClipboardTextA(C_BCP pTxt, UI lTxt)
{
	BOOL	rc = FALSE;
	HGLOBAL	hGlobal;
	BCP		pGbl;
	UI		lGbl;

	if (pTxt != NULL && lTxt != 0) {

		if (lTxt == -1) lGbl = (UI)strlen(pTxt) + 1;
		else			lGbl = lTxt + 1;

		if (hGlobal = GlobalAlloc(GHND | GMEM_SHARE, lGbl)) {
			pGbl = (BCP)GlobalLock(hGlobal);
			strncpy(pGbl, pTxt, lGbl);
			pGbl[lGbl - 1] = 0;
			GlobalUnlock(hGlobal);
			OpenClipboard(NULL);
			EmptyClipboard();
			SetClipboardData(CF_TEXT, hGlobal);
			CloseClipboard();
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPutClipboardTextW(C_WCP pTxt, UI lTxt)
{
	BOOL	rc = FALSE;
	HGLOBAL	hGlobal;
	WCP		pGbl;
	UI		lGbl;

	if (pTxt != NULL && lTxt != 0) {

		if (lTxt == -1) lGbl = (UI)wcslen(pTxt) + 1;
		else			lGbl = lTxt + 1;

		if (hGlobal = GlobalAlloc(GHND | GMEM_SHARE, lGbl * 2)) {
			pGbl = (WCP)GlobalLock(hGlobal);
			wcsncpy(pGbl, pTxt, lGbl);
			pGbl[lGbl - 1] = 0;
			GlobalUnlock(hGlobal);
			OpenClipboard(NULL);
			EmptyClipboard();
			SetClipboardData(CF_UNICODETEXT, hGlobal);
			CloseClipboard();
		}
	}
	return rc;
}
//==============================================================================================================//
//	メニューアイテムの列挙																						//
//																												//
//	引　数：	hMenu	  - メニューハンドル																	//
//				fSubMenu  - サブメニュー列挙フラグ（TRUE:サブメニューも列挙する）								//
//				cbp		  - コールバックパラメタ																//
//				cb		  - 個々のメニューアイテム通知用コールバック関数										//
//																												//
//	戻り値：	列挙したメニューアイテムの個数																	//
//==============================================================================================================//
typedef struct {
	BOOL	fSubMenu;
	UI		cnt;
	UX		cbp;
	BOOL (CALLBACK *cb)(HMENU hMenu, UI ix, UX id, UI type, UI state, HMENU hSubMenu, UX cbp);
	BOOL	fEnd;
} EMIWK, *PEMIWK;
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	VO	SubEnumMenuItem(HMENU hMenu, PEMIWK pWk)
{
	if (IsMenu(hMenu) && !pWk->fEnd) {
		int		n;
		//	メニュー項目数取得
		if ((n = GetMenuItemCount(hMenu)) > 0) {
			MENUITEMINFO	miif;
			int				i;
			//	メニュー項目数ループ
			for (i = 0; i < n; i++) {
				//	メニューのＩＤ，タイプ，状態，サブメニュー取得
				memset(&miif, 0, sizeof miif);
				miif.cbSize = sizeof(MENUITEMINFO);
				miif.fMask	= (MIIM_ID | MIIM_FTYPE | MIIM_STATE | MIIM_SUBMENU);
				if (GetMenuItemInfo(hMenu, i, TRUE, &miif)) {
					//	コールバック有ならば、列挙
					if (pWk->cb != NULL) {
						//	列挙数更新
						pWk->cnt++;
						//	コールバック，列挙継続？
						if (pWk->cb(hMenu, i, miif.wID, miif.fType, miif.fState, miif.hSubMenu, pWk->cbp)) {
							//	サブメニューも列挙 ＆ サブメニュー有り ならば、再帰呼び出し
							if (pWk->fSubMenu && miif.hSubMenu != NULL) {
								SubEnumMenuItem(miif.hSubMenu, pWk);
							}
						}
						//	列挙中止？
						else {
							pWk->fEnd = TRUE;
							break;
						}
					}
				}
				//	メニュー項目取得失敗？
				else {
					pWk->fEnd = TRUE;
					break;
				}
			}
		}
	}
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT UI	 WINAPI AjcEnumMenuItems(HMENU hMenu, BOOL fSubMenu, UX cbp, BOOL (CALLBACK *cbEnumMenu)(HMENU hMenu, UI ix, UX id, UI type, UI state, HMENU hSubMenu, UX cbp))
{
	BOOL	rc	= FALSE;
	EMIWK	wk = {0};

	if (IsMenu(hMenu)) {
		wk.fSubMenu = fSubMenu;
		wk.cnt		= 0;
		wk.cbp		= cbp;
		wk.cb		= cbEnumMenu;
		wk.fEnd 	= FALSE;
		SubEnumMenuItem(hMenu, &wk);
	}
	return wk.cnt;
}


