﻿#include	"AjcInternal.h"

//**************************************************************************************************************//
//																												//
//	ファイル名／フォルダ名取得																					//
//																												//
//**************************************************************************************************************//

//==============================================================================================================//
//	セーブファイル取得ダイアログ（標準的な指定）																//
//																												//
//	引　数　：　hWnd	 - オーナーウインドハンドル（不要時はＮＵＬＬ）											//
//				pTitle	 - ウィンドタイトルバーに表示するテキストのアドレス（不要時は、NULL)					//
//				pFilter	 - フィルタ文字列（不要時はＮＵＬＬ)													//
//						   フィルタは（／）で区切って、、"タイトル／ワイルドカード"のペアで複数指定可能。		//
//						   ex. "AllFiles { *.* }/*.*/TextFiles { *.txt }/*.txt"									//
//				pDefExt	 - デフォルトの拡張子（ピリオドは指定しない(ex. "txt")，不要時はＮＵＬＬ）				//
//				pBuf	 - 取得したファイルパス名を格納するバッファのアドレス									//
//				lBuf	 - 取得したファイルパス名を格納するバッファのサイズ（バイト数／文字数）					//
//																												//
//	戻り値　：　TRUE  - ＯＫボタンで終了																		//
//				FALSE - キャンセル／エラー																		//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetSaveFileA(HWND hWnd, C_BCP pTitle, C_BCP pFilter, C_BCP pDefExt, BCP pBuf, UI lBuf)
{
	BOOL			rc = FALSE;
	OPENFILENAMEA	ofn;
	BCP				p1, p2;
	UI				ovrp = 0;

	do {
		//	NULLポインタチェック
		if (pBuf == NULL) break;
		if (lBuf == 0	) break;
		//	フィルタ多重文字列作成と、ファイル上書き確認属性設定
		if (pFilter) {
			if (p1 = p2 = AjcTAllocA((UI)strlen(pFilter) + 2)) {
				if (*pFilter == '@') {
					ovrp = OFN_OVERWRITEPROMPT;
					pFilter++;
				}
				while (*pFilter) {
					if (*pFilter == '/') *p2 = 0;
					else				 *p2 = *pFilter;
					pFilter++;
					p2++;
				}
				*p2++ = 0;
				*p2	  = 0;
			}
			else break;
		}
		else p1 = p2 = NULL;

		ofn.lStructSize		  = sizeof(OPENFILENAMEA);
		ofn.hwndOwner		  = hWnd;
		ofn.hInstance		  = NULL;
		ofn.lpstrFilter		  = p1;
		ofn.lpstrCustomFilter = NULL;
		ofn.nMaxCustFilter	  = 0;
		ofn.nFilterIndex	  = 1;
		ofn.lpstrFile		  = pBuf;
		ofn.nMaxFile		  = lBuf;
		ofn.lpstrFileTitle	  = NULL;
		ofn.nMaxFileTitle	  = 0;
		ofn.lpstrInitialDir	  = NULL;
		ofn.lpstrTitle		  = pTitle;
		ofn.Flags			  = OFN_HIDEREADONLY | OFN_NOREADONLYRETURN | OFN_EXPLORER | ovrp;
		ofn.nFileOffset		  = 0;
		ofn.nFileExtension	  = 0;
		ofn.lpstrDefExt		  = pDefExt;
		ofn.lCustData		  = 0L;
		ofn.lpfnHook		  = NULL;
		ofn.lpTemplateName	  = NULL;

		if (!(rc = GetSaveFileNameA(&ofn))) {
			if (CommDlgExtendedError() == FNERR_INVALIDFILENAME) {
				*pBuf = 0;
				rc = GetSaveFileNameA(&ofn);
			}
		}
		if (p1) free(p1);

	} while(0);

	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetSaveFileW(HWND hWnd, C_WCP pTitle, C_WCP pFilter, C_WCP pDefExt, WCP pBuf, UI lBuf)
{
	BOOL			rc = FALSE;
	OPENFILENAMEW	ofn;
	WCP				p1, p2;
	UI				ovrp = 0;

	do {
		//	NULLポインタチェック
		if (pBuf == NULL) break;
		if (lBuf == 0	) break;
		//	フィルタ多重文字列作成と、ファイル上書き確認属性設定
		if (pFilter) {
			if (p1 = p2 = AjcTAllocW((UI)wcslen(pFilter) + 2)) {
				if (*pFilter == L'@') {
					ovrp = OFN_OVERWRITEPROMPT;
					pFilter++;
				}
				while (*pFilter) {
					if (*pFilter == L'/') *p2 = 0;
					else				  *p2 = *pFilter;
					pFilter++;
					p2++;
				}
				*p2++ = 0;
				*p2	  = 0;
			}
			else break;
		}
		else p1 = p2 = NULL;

		ofn.lStructSize		  = sizeof(OPENFILENAMEW);
		ofn.hwndOwner		  = hWnd;
		ofn.hInstance		  = NULL;
		ofn.lpstrFilter		  = p1;
		ofn.lpstrCustomFilter = NULL;
		ofn.nMaxCustFilter	  = 0;
		ofn.nFilterIndex	  = 1;
		ofn.lpstrFile		  = pBuf;
		ofn.nMaxFile		  = lBuf;
		ofn.lpstrFileTitle	  = NULL;
		ofn.nMaxFileTitle	  = 0;
		ofn.lpstrInitialDir	  = NULL;
		ofn.lpstrTitle		  = pTitle;
		ofn.Flags			  = OFN_HIDEREADONLY | OFN_NOREADONLYRETURN | OFN_EXPLORER | ovrp;
		ofn.nFileOffset		  = 0;
		ofn.nFileExtension	  = 0;
		ofn.lpstrDefExt		  = pDefExt;
		ofn.lCustData		  = 0L;
		ofn.lpfnHook		  = NULL;
		ofn.lpTemplateName	  = NULL;

		if (!(rc = GetSaveFileNameW(&ofn))) {
			if (CommDlgExtendedError() == FNERR_INVALIDFILENAME) {
				*pBuf = 0;
				rc = GetSaveFileNameW(&ofn);
			}
		}

		if (p1) free(p1);

	} while(0);

	return rc;
}
//==============================================================================================================//
//	オープンファイル取得ダイアログ（標準的な指定，１ヶのファイル名）											//
//																												//
//	引　数　：　hWnd	 - オーナーウインドハンドル（不要時はＮＵＬＬ）											//
//				pTitle	 - ウィンドタイトルバーに表示するテキストのアドレス（不要時は、NULL)					//
//				pFilter	 - フィルタ文字列（不要時はＮＵＬＬ)													//
//						   フィルタは（／）で区切って、、"タイトル／ワイルドカード"のペアで複数指定可能。		//
//						   ex. "AllFiles { *.* }/*.*/TextFiles { *.txt }/*.txt"									//
//				pDefExt	 - デフォルトの拡張子（ピリオドは指定しない(ex. "txt")，不要時はＮＵＬＬ）				//
//				pBuf	 - 取得したファイルパス名を格納するバッファのアドレス									//
//				lBuf	 - 取得したファイルパス名を格納するバッファのサイズ（バイト数／文字数）					//
//																												//
//	戻り値　：　TRUE  - ＯＫボタンで終了																		//
//				FALSe - キャンセル／エラー																		//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetOpenFileA(HWND hWnd, C_BCP pTitle, C_BCP pFilter, C_BCP pDefExt, BCP pBuf, UI lBuf)
{
	BOOL	rc = FALSE;
	BCP		*ppName = NULL;

	if (ppName = AjcGetOpenFilesA(hWnd, pBuf, pTitle, pFilter, pDefExt, FALSE, NULL)) {
		strncpy(pBuf, *ppName, lBuf);
		pBuf[lBuf - 1] = 0;
		AjcReleaseOpenedFilesArray(ppName);
		rc = TRUE;
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetOpenFileW(HWND hWnd, C_WCP pTitle, C_WCP pFilter, C_WCP pDefExt, WCP pBuf, UI lBuf)
{
	BOOL	rc = FALSE;
	WCP		*ppName = NULL;

	if (ppName = AjcGetOpenFilesW(hWnd, pBuf, pTitle, pFilter, pDefExt, FALSE, NULL)) {
		wcsncpy(pBuf, *ppName, lBuf);
		pBuf[lBuf - 1] = 0;
		AjcReleaseOpenedFilesArray(ppName);
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	オープンファイル群取得ダイアログ（標準的な指定，複数選択可）												//
//																												//
//	引　数　：　hWnd		 - オーナーウインドハンドル（不要時はＮＵＬＬ）										//
//				pInitialPath - 初期パス名へのポインタ（不要時はNULL)											//
//				pTitle		 - ウィンドタイトルバーに表示するテキストのアドレス（不要時は、NULL)				//
//				pFilter		 - フィルタ文字列（不要時はＮＵＬＬ)												//
//							   フィルタは（／）で区切って、、"タイトル／ワイルドカード"のペアで複数指定可能。	//
//							   ex. "AllFiles { *.* }/*.*/TextFiles { *.txt }/*.txt"								//
//				pDefExt		 - デフォルトの拡張子（ピリオドは指定しない(ex. "txt")，不要時はＮＵＬＬ）			//
//				fMulti		 - FALSE:１ヶのファイル選択，TRUE:複数ファイル選択）								//
//				pNum		 - 選択されたファイルの個数を格納するバッファのアドレス（不要時は、ＮＵＬＬ）		//
//																												//
//	戻り値　：　≠ＮＵＬＬ：変換した絶対パス名群へのポインタ配列の先頭アドレス									//
//							このバッファは、後で、ReleaseOpenedFilesArray 関数により開放しなければなりません。	//
//				＝ＮＵＬＬ：キャンセル／メモリ確保失敗															//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BCP *	WINAPI	AjcGetOpenFilesA(HWND hWnd, C_BCP pInitialPath, C_BCP pTitle, C_BCP pFilter, C_BCP pDefExt, BOOL fMulti, UIP pNum)
{
	BC		buf[2048];

	if (pInitialPath) {strncpy(buf, pInitialPath, AJCTSIZE(buf)); buf[AJCTSIZE(buf) - 1] = 0;}
	else			  {buf[0] = 0;}

	return	AjcGetOpenFilesExA(hWnd, pTitle, pFilter, pDefExt, 
				OFN_FILEMUSTEXIST | OFN_HIDEREADONLY | OFN_NONETWORKBUTTON |
				OFN_EXPLORER | (fMulti ? OFN_ALLOWMULTISELECT : 0),
				buf, sizeof buf, pNum);
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	WCP *	WINAPI	AjcGetOpenFilesW(HWND hWnd, C_WCP pInitialPath, C_WCP pTitle, C_WCP pFilter, C_WCP pDefExt, BOOL fMulti, UIP pNum)
{
	UW		buf[2048];

	if (pInitialPath) {wcsncpy(buf, pInitialPath, AJCTSIZE(buf)); buf[AJCTSIZE(buf) - 1] = 0;}
	else			  {buf[0] = 0;}

	return	AjcGetOpenFilesExW(hWnd, pTitle, pFilter, pDefExt, 
				OFN_FILEMUSTEXIST | OFN_HIDEREADONLY | OFN_NONETWORKBUTTON |
				OFN_EXPLORER | (fMulti ? OFN_ALLOWMULTISELECT : 0),
				buf, AJCTSIZE(buf), pNum);
}
//==============================================================================================================//
//	オープンファイル取得ダイアログ（オプションフラグ指定）														//
//																												//
//	引　数　：　hWnd	- オーナーウインドハンドル（不要時はＮＵＬＬ）											//
//				pTitle	- ウィンドタイトルバーに表示するテキストのアドレス（不要時は、NULL)						//
//				pFilter	 - フィルタ文字列（不要時はＮＵＬＬ)													//
//						   フィルタは（／）で区切って、、"タイトル／ワイルドカード"のペアで複数指定可能。		//
//						   ex. "AllFiles { *.* }/*.*/TextFiles { *.txt }/*.txt"									//
//				pDefExt - デフォルトの拡張子（ピリオドは指定しない(ex. "txt")，不要時はＮＵＬＬ）				//
//				flags	- オプションフラグ(OFN_・・・)															//
//				pBuf	- GetOpenFileNameに与えるバッファアドレス												//
//				lBuf	- GetOpenFileNameに与えるバッファサイズ													//
//				pNum	- 選択されたファイルの個数を格納するバッファのアドレス（不要時は、NULL)					//
//																												//
//	戻り値　：　≠ＮＵＬＬ：変換した絶対パス名群へのポインタ配列の先頭アドレス									//
//							このバッファは、後で、ReleaseOpenedFilesArray 関数により開放しなければなりません。	//
//				＝ＮＵＬＬ：キャンセル／メモリ確保失敗															//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BCP *	WINAPI	AjcGetOpenFilesExA(HWND hWnd, C_BCP pTitle, C_BCP pFilter, C_BCP pDefExt,
													 UI flags, BCP pBuf, UI lBuf, UIP pNum)
{
	BCP					*rc = NULL;
	OPENFILENAMEA		ofn;
	BCP					p1, p2;

	do {
		//	NULLポインタチェック
		if (pBuf == NULL) break;
		if (lBuf == 0	) break;
		//	フィルタ多重文字列作成
		if (pFilter) {
			if (p1 = p2 = AjcTAllocA((UI)strlen(pFilter) + 2)) {
				while (*pFilter) {
					if (*pFilter == '/') *p2 = 0;
					else				 *p2 = *pFilter;
					pFilter++;
					p2++;
				}
				*p2++ = 0;
				*p2	  = 0;
			}
			else break;
		}
		else p1 = p2 = NULL;

		ofn.lStructSize		  = sizeof(OPENFILENAMEA);
		ofn.hwndOwner		  = hWnd;
		ofn.hInstance		  = NULL;
		ofn.lpstrFilter		  = p1;
		ofn.lpstrCustomFilter = NULL;
		ofn.nMaxCustFilter	  = 0;
		ofn.nFilterIndex	  = 1;
		ofn.lpstrFile		  = pBuf;
		ofn.nMaxFile		  = lBuf;
		ofn.lpstrFileTitle	  = NULL;
		ofn.nMaxFileTitle	  = 0;
		ofn.lpstrInitialDir	  = NULL;
		ofn.lpstrTitle		  = pTitle;
		ofn.Flags			  = flags;
		ofn.nFileOffset		  = 0;
		ofn.nFileExtension	  = 0;
		ofn.lpstrDefExt		  = pDefExt;
		ofn.lCustData		  = 0L;
		ofn.lpfnHook		  = NULL;
		ofn.lpTemplateName	  = NULL;

		if (GetOpenFileNameA(&ofn)) {
			rc = AjcOpenedFilesArrayA(pBuf, pNum);
		}
		else if (CommDlgExtendedError() == FNERR_INVALIDFILENAME) {
			*pBuf = 0;
			rc = AjcOpenedFilesArrayA(pBuf, pNum);
		}

		if (p1) free(p1);

	} while(0);

	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	WCP *	WINAPI	AjcGetOpenFilesExW(HWND hWnd, C_WCP pTitle, C_WCP pFilter, C_WCP pDefExt,
													 UI flags, WCP pBuf, UI lBuf, UIP pNum)
{
	WCP					*rc = NULL;
	OPENFILENAMEW		ofn;
	WCP					p1, p2;

	do {
		//	NULLポインタチェック
		if (pBuf == NULL) break;
		if (lBuf == 0	) break;
		//	フィルタ多重文字列作成
		if (pFilter) {
			if (p1 = p2 = AjcTAllocW((UI)wcslen(pFilter) + 2)) {
				while (*pFilter) {
					if (*pFilter == '/') *p2 = 0;
					else				 *p2 = *pFilter;
					pFilter++;
					p2++;
				}
				*p2++ = 0;
				*p2	  = 0;
			}
			else break;
		}
		else p1 = p2 = NULL;

		ofn.lStructSize		  = sizeof(OPENFILENAMEW);
		ofn.hwndOwner		  = hWnd;
		ofn.hInstance		  = NULL;
		ofn.lpstrFilter		  = p1;
		ofn.lpstrCustomFilter = NULL;
		ofn.nMaxCustFilter	  = 0;
		ofn.nFilterIndex	  = 1;
		ofn.lpstrFile		  = pBuf;
		ofn.nMaxFile		  = lBuf;
		ofn.lpstrFileTitle	  = NULL;
		ofn.nMaxFileTitle	  = 0;
		ofn.lpstrInitialDir	  = NULL;
		ofn.lpstrTitle		  = pTitle;
		ofn.Flags			  = flags;
		ofn.nFileOffset		  = 0;
		ofn.nFileExtension	  = 0;
		ofn.lpstrDefExt		  = pDefExt;
		ofn.lCustData		  = 0L;
		ofn.lpfnHook		  = NULL;
		ofn.lpTemplateName	  = NULL;

		if (GetOpenFileNameW(&ofn)) {
			rc = AjcOpenedFilesArrayW(pBuf, pNum);
		}
		else if (CommDlgExtendedError() == FNERR_INVALIDFILENAME) {
			*pBuf = 0;
			rc = AjcOpenedFilesArrayW(pBuf, pNum);
		}

		if (p1) free(p1);

	} while(0);

	return rc;
}

/*==============================================================================================================*/
/*	API(GetOpenFileName)で取得したファイル情報から、絶対パス名群へのポインタ配列を作成する						*/
/*																												*/
/*	引　数　：　pFileStrings - GetOpenFileNameで取得したファイル情報群											*/
/*				pNumber		 - 絶対パス名の個数を格納するバッファのアドレス（不要時は、NULL)					*/
/*																												*/
/*	戻り値　：　≠ＮＵＬＬ：変換した絶対パス名群へのポインタ配列の先頭アドレス									*/
/*							このバッファは、後で、ReleaseOpenedFilesArray 関数により開放しなければなりません。	*/
/*				＝ＮＵＬＬ：エラー（メモリ確保失敗）															*/
/*																												*/
/*	　「GetOpenFileName」では、複数ファイルを選択した場合、先頭文字列がディレクトリ名を示し、後続の文字列が		*/
/*	ディレクトリを含まないファイル名の羅列で構成されます。														*/
/*	　本関数は、これを、絶対ファイルパス名へのポインタ配列に変換します。										*/
/*																												*/
/*	（例）	［変換前］	"C:\\TEMP\0" "A.TXT\0" "B.TXT\0" "C.TXT\0"												*/
/*			［変換後］	p[0]="C:\\TEMP\\A.TXT", p[1]="C:\\TEMP\\B.TXT", p[2]="C:\\TEMP\\C.TXT"					*/
/*																												*/
/*	　なお、「GetOpenFileName」で、ファイルを１つだけ選択した場合は、絶対パス名の文字列が１つだけ設定されますが	*/
/*	本関数では、この場合も正常にポインタ配列を作成します。														*/
/*																												*/
/*==============================================================================================================*/
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BCP * WINAPI	AjcOpenedFilesArrayA(C_BCP pFileStrings, UIP pNumber)
{
	BCP		 pBuf;
	BCP		*pArr = NULL;
	BCP		 pDir, pDes;
	C_BCP	 pSrc;
	UI		 i, stl, DirLen;
	UI		 BufSize = 0;

	if (pNumber) *pNumber = 0;

	if (AjcPathExistsA(pFileStrings)) {												//	先頭は有効なパス？
		if (!AjcPathIsDirectoryA(pFileStrings)) {									//	　先頭文字列はファイルパス名？
			stl = (UI)strlen(pFileStrings);											//	　　バッファを確保し
			if (pBuf = (BCP)AJCMEM(sizeof(VO *) + stl + 1)) {						//	　　バッファ割り当て成功？
				pArr = (BCP *)pBuf;													//	　　　ポインタ配列アドレス設定
				pDes = pBuf + sizeof(VO *);											//	　　　文字列バッファアドレス設定
				pArr[0] = pDes;														//	　　　文字列ポインタ設定
				memcpy(pDes, pFileStrings, stl);									//	　　　パス名文字列設定
				*(pDes + stl) = 0;													//	　　　文字列終端設定
				if (pNumber) *pNumber = 1;											//	　　　パス名の個数設定
			}
		}
		else {																		//　　先頭文字列はディレクトリパス名？
			DirLen = (UI)strlen(pFileStrings);										//　　　ＤＩＲ長設定
			if (pDir = (BCP)AJCMEM(DirLen + 2)) {									//　　　ＤＩＲ名バッファ確保，成功？
				memset(pDir, 0, DirLen + 2);										//　　　　ＤＩＲ文字列設定
				strcpy(pDir, pFileStrings);											//　　　　・
				if (*(pDir + (DirLen - 1)) != '\\') {								//　　　　末尾「￥」追加
					*(pDir + (DirLen	))	= '\\';									//　　　　・
					DirLen++;														//　　　　・
				}
				BufSize = sizeof(VO *);												//　　　　バッファサイズ初期化
				for (i = 0, pSrc = pFileStrings + (strlen(pFileStrings) + 1);		//　　　　後続の全ファイル名ループ
															 *pSrc; i++) {			//　　　　・
					BufSize += (DirLen + (UI)strlen(pSrc) + 1);						//　　　　　バッファサイズ更新
					pSrc += ((UI)strlen(pSrc) + 1);									//　　　　　次のファイル名のアドレス設定
				}
				BufSize += (sizeof(VO *) * i);										//　　　　ポインタ配列分加算
				if (pBuf = (BCP)AJCMEM(BufSize)) {									//　　　　バッファ確保，成功？
					pArr = (BCP *)pBuf;												//　　　　　バッファポインタ初期化
					pDes = pBuf + (sizeof(VO *) * (i + 1));							//　　　　　・
					for (i = 0, pSrc = pFileStrings + (strlen(pFileStrings) + 1);	//　　　　　後続の全ファイル名ループ
																 *pSrc; i++) {		//　　　　　・
						pArr[i] = pDes;												//　　　　　　ポインタ配列設定
						strcpy(pDes, pDir);											//　　　　　　ＤＩＲ名コピー
						strcat(pDes, pSrc);											//　　　　　　ファイル名追加
						pDes += ((UI)strlen(pDes) + 1);								//　　　　　　次のバッファアドレス
						pSrc += ((UI)strlen(pSrc) + 1);								//　　　　　　次のファイル名アドレス
					}
					pArr[i] = NULL;
					if (pNumber) *pNumber = i;										//　　　　　パス名の個数設定
				}
				free(pDir);															//　　　　ＤＩＲバッファ開放
			}
		}
	}
	return pArr;
}

//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	WCP	* WINAPI	AjcOpenedFilesArrayW(C_WCP pFileStrings, UIP pNumber)
{
	WCP		 pBuf;
	WCP		*pArr = NULL;
	WCP		 pDir, pDes;
	C_WCP	 pSrc;
	UI		 i, stl, DirLen;
	UI		 BufSize = 0;

	if (pNumber) *pNumber = 0;

	if (AjcPathExistsW(pFileStrings)) {												//	先頭は有効なパス？
		if (!AjcPathIsDirectoryW(pFileStrings)) {									//	　先頭文字列はファイルパス名？
			stl = (UI)wcslen(pFileStrings);											//	　　バッファを確保し
			if (pBuf = (WCP)AJCMEM(sizeof(VO *) + (stl + 1) * 2)) {					//	　　バッファ割り当て成功？
				pArr = (WCP *)pBuf;													//	　　　ポインタ配列アドレス設定
				pDes = (WCP)(((BCP)pBuf) + sizeof(VO *));							//	　　　文字列バッファアドレス設定
				pArr[0] = pDes;														//	　　　文字列ポインタ設定
				memcpy(pDes, pFileStrings, (stl * 2));								//	　　　パス名文字列設定
				*(pDes + stl) = 0;													//	　　　文字列終端設定
				if (pNumber) *pNumber = 1;											//	　　　パス名の個数設定
			}
		}
		else {																		//	文先頭文字列はディレクトリパス名？
			DirLen = (UI)wcslen(pFileStrings);										//	　ＤＩＲ名長設定
			if (pDir = (WCP)AJCMEM((DirLen + 2) * 2)) {								//	　ＤＩＲ名バッファ確保，成功？
				memset(pDir, 0, (DirLen + 2) * 2);									//	　　ＤＩＲ文字列設定
				wcscpy(pDir, pFileStrings);											//	　　・
				if (*(pDir + (DirLen - 1)) != L'\\') {								//	　　末尾「￥」追加
					*(pDir + (DirLen	))	= L'\\';								//	　　・
					DirLen++;														//	　　・
				}
				BufSize = sizeof(VO *);												//	　　バッファサイズ初期化
				for (i = 0, pSrc = pFileStrings + (wcslen(pFileStrings) + 1);		//	　　後続の全ファイル名ループ
																 *pSrc; i++) {		//	　　・
					BufSize += ((DirLen + (UI)wcslen(pSrc) + 1) * 2);				//	　　　バッファサイズ更新
					pSrc += (wcslen(pSrc) + 1);										//	　　　次のファイル名のアドレス設定
				}
				BufSize += (sizeof(VO *) * i);										//	　　ポインタ配列分加算
				if (pBuf = (WCP)AJCMEM(BufSize)) {									//	　　バッファ確保，成功？
					pArr = (WCP *)pBuf;												//	　　　バッファポインタ初期化
					pDes = (WCP)(((BCP)pBuf) + (sizeof(VO *) * (i + 1)));			//	　　　・
					for (i = 0, pSrc = pFileStrings + (wcslen(pFileStrings) + 1);	//	　　　後続の全ファイル名ループ
																	 *pSrc; i++) {	//	　　　・
						pArr[i] = pDes;												//	　　　　ポインタ配列設定
						wcscpy(pDes, pDir);											//	　　　　ＤＩＲ名コピー
						wcscat(pDes, pSrc);											//	　　　　ファイル名追加
						pDes += (wcslen(pDes) + 1);									//	　　　　次のバッファアドレス
						pSrc += (wcslen(pSrc) + 1);									//	　　　　次のファイル名アドレス
					}
					pArr[i] = NULL;
					if (pNumber) *pNumber = i;										//	　　　文字列数設定
				}
				free(pDir);															//	　　ＤＩＲバッファ開放
			}
		}
	}
	return pArr;
}

/*==============================================================================================================*/
/*	GetOpenFileで取得したポインタ配列中、インデクスで指定した項目の文字列長を得る								*/
/*																												*/
/*	引　数　：　pArr - GetOpenFileで取得したポインタ配列のアドレス												*/
/*				ix	 - インデクス																				*/
/*																												*/
/*	戻り値　：　インデクスで指定した項目の文字列長																*/
/*==============================================================================================================*/
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcGetFilesArrayItemLenA(BCP pArr[], UI ix)
{
	return (UI)strlen(pArr[ix]);
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcGetFilesArrayItemLenW(WCP pArr[], UI ix)
{
	return (UI)wcslen(pArr[ix]);
}
/*==============================================================================================================*/
/*	GetOpenFileで取得したポインタ配列中、インデクスで指定した項目の文字列を得る									*/
/*																												*/
/*	引　数　：　pArr - GetOpenFileで取得したポインタ配列のアドレス												*/
/*				ix	 - インデクス																				*/
/*				pBuf - インデクスで指定した項目の文字列を格納するバッファのアドレス								*/
/*				lBuf - インデクスで指定した項目の文字列を格納するバッファのバイト数／文字数						*/
/*																												*/
/*	戻り値　：　なし																							*/
/*==============================================================================================================*/
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	VO	WINAPI	AjcGetFilesArrayItemStrA(BCP pArr[], UI ix, BCP pBuf, UI lBuf)
{
	strncpy(pBuf, pArr[ix], lBuf - 1);
	pBuf[lBuf - 1] = 0;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	VO	WINAPI	AjcGetFilesArrayItemStrW(WCP pArr[], UI ix, WCP pBuf, UI lBuf)
{
	wcsncpy(pBuf, pArr[ix], lBuf - 1);
	pBuf[lBuf - 1] = 0;
}
/*==============================================================================================================*/
/*	OpendFilesArrayで取得したワークバッファを開放する															*/
/*																												*/
/*	引　数　：　pArr - （絶対パス名群へのポインタ配列の先頭アドレス＝OpendFilesArrayの戻り値）					*/
/*																												*/
/*	戻り値　：　なし																							*/
/*==============================================================================================================*/
AJCEXPORT	VO	WINAPI	AjcReleaseOpenedFilesArray(VOP pArr)
{
	if (pArr)
		free(pArr);
}
/*==============================================================================================================*/
/*	フォルダ名取得ダイアログ（コンピュータ上のフォルダ名選択）													*/
/*																												*/
/*	引　数　：　hWnd			 - オーナーウインドハンドル（不要時はＮＵＬＬ）									*/
/*				pWndTtl			 - ウィンドタイトルバーに表示するテキストのアドレス			（不要時はNULL）	*/
/*				pBoxTtl			 - ダイアログボックス内に表示するタイトルテキストのアドレス	（不要時はNULL）	*/
/*				pBuf			 - フォルダ名を格納するバッファのアドレス										*/
/*				fTailIsDelimiter - TRUE	 : 常に末尾が、区切り記号「￥」となるようにする。						*/
/*								   FALSE : ルートディレクトリの場合を除いて、末尾に「￥」を付加しないようにする	*/
/*																												*/
/*	戻り値　：　TRUE  - ＯＫ																					*/
/*				FALSE - キャンセル																				*/
/*==============================================================================================================*/
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetFolderNameA(HWND hWnd, C_BCP pWndTtl, C_BCP pBoxTtl, BC pBuf[MAX_PATH], BOOL fTailIsDelimiter)
{
	return	AjcGetFolderNameExA(hWnd, pWndTtl, pBoxTtl, pBuf, fTailIsDelimiter, 
							0
						|	BIF_BROWSEFORCOMPUTER
					//	|	BIF_BROWSEFORPRINTER
					//	|	BIF_BROWSEINCLUDEFILES
						|	BIF_DONTGOBELOWDOMAIN
					//	|	BIF_EDITBOX
					//	|	BIF_RETURNFSANCESTORS
						|	BIF_RETURNONLYFSDIRS
						|	BIF_STATUSTEXT
					//	|	BIF_VALIDATE
			);
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetFolderNameW(HWND hWnd, C_WCP pWndTtl, C_WCP pBoxTtl, WC pBuf[MAX_PATH], BOOL fTailIsDelimiter)
{
	return	AjcGetFolderNameExW(hWnd, pWndTtl, pBoxTtl, pBuf, fTailIsDelimiter,
							0
						|	BIF_BROWSEFORCOMPUTER
					//	|	BIF_BROWSEFORPRINTER
					//	|	BIF_BROWSEINCLUDEFILES
						|	BIF_DONTGOBELOWDOMAIN
					//	|	BIF_EDITBOX
					//	|	BIF_RETURNFSANCESTORS
						|	BIF_RETURNONLYFSDIRS
						|	BIF_STATUSTEXT
					//	|	BIF_VALIDATE
			);
}
/*==============================================================================================================*/
/*	フォルダ名取得ダイアログ（オプションフラグ指定）															*/
/*																												*/
/*	引　数　：　hWnd			 - オーナーウインドハンドル（不要時はＮＵＬＬ）									*/
/*				pWndTtl			 - ウィンドタイトルバーに表示するテキストのアドレス			（不要時はNULL）	*/
/*				pBoxTtl			 - ダイアログボックス内に表示するタイトルテキストのアドレス	（不要時はNULL）	*/
/*				pBuf			 - フォルダ名を格納するバッファのアドレス										*/
/*				fTailIsDelimiter - TRUE	 : 常に末尾が、区切り記号「￥」となるようにする。						*/
/*								   FALSE : ルートディレクトリの場合を除いて、末尾に「￥」を付加しないようにする	*/
/*				flags			 - オプションフラグ																*/
/*																												*/
/*	戻り値　：　TRUE  - ＯＫ																					*/
/*				FALSE - キャンセル																				*/
/*==============================================================================================================*/
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetFolderNameExA(HWND hWnd, C_BCP pWndTtl, C_BCP pBoxTtl, BC pBuf[MAX_PATH], BOOL fTailIsDelimiter, UI flags)
{
	BOOL	rc = FALSE;
	UI		lTmpWndTtl, lTmpBoxTtl;
	WCP		pTmpWndTtl = NULL;
	WCP		pTmpBoxTtl = NULL;
	WC		buf[MAX_PATH];

	if (pBuf != NULL) {
		if (pWndTtl != NULL && pBoxTtl != NULL) {
			pTmpWndTtl = AjcTAlloc(lTmpWndTtl = MultiByteToWideChar(CP_ACP, 0, pWndTtl , -1, NULL, 0));
			pTmpBoxTtl = AjcTAlloc(lTmpBoxTtl = MultiByteToWideChar(CP_ACP, 0, pBoxTtl , -1, NULL, 0));
			if (pTmpWndTtl && pTmpBoxTtl) {
				MultiByteToWideChar(CP_ACP, 0, pWndTtl , -1, pTmpWndTtl, lTmpWndTtl);
				MultiByteToWideChar(CP_ACP, 0, pBoxTtl , -1, pTmpBoxTtl, lTmpBoxTtl);
				rc = AjcGetFolderNameExW(hWnd, pTmpWndTtl, pTmpBoxTtl, buf, fTailIsDelimiter, flags);
				if (rc == TRUE) {
					WideCharToMultiByte(CP_ACP, 0, buf, -1, pBuf, MAX_PATH, NULL, NULL);
				}
			}

			if (pTmpWndTtl) free(pTmpWndTtl);
			if (pTmpBoxTtl) free(pTmpBoxTtl);
		}
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
//	コールバックパラメタ
typedef struct {
	C_WCP	pWndTtl;
	C_WCP	pBoxTtl;
	WCP		pPath;
} GETFOLPARAM, *PGETFOLPARAM;
//	コールバック関数
static	int CALLBACK BrowseCallbackProcW(HWND hWnd, UINT uMsg, LPARAM lParam, LPARAM lpData);

AJCEXPORT	BOOL	WINAPI	AjcGetFolderNameExW(HWND hWnd, C_WCP pWndTtl, C_WCP pBoxTtl, WC pBuf[MAX_PATH], BOOL fTailIsDelimiter, UI flags)
{
	BOOL			rc = FALSE;
	BROWSEINFOW		bri;
	UI				stl;
	LPITEMIDLIST	lpid;
	GETFOLPARAM		gfp;

	gfp.pWndTtl   = pWndTtl;
	gfp.pBoxTtl   = pBoxTtl;
	gfp.pPath	  = pBuf;

	if (pBuf != NULL) {
		memset(&bri, 0, sizeof bri);
		bri.hwndOwner = hWnd;
		bri.lpszTitle = pBoxTtl;
		bri.pidlRoot  = NULL;
		bri.lpfn	  = BrowseCallbackProcW;
		bri.ulFlags	  = flags;
		bri.lParam	  = (LPARAM)&gfp;

		if (rc = ((lpid = SHBrowseForFolderW(&bri)) != NULL)) {	//	シェルフォルダの選択ダイアログボックスを表示
			SHGetPathFromIDListW(lpid, pBuf);					//	選択結果をファイルシステムパスに変換
			CoTaskMemFree(lpid);								//	ITEMIDLIST解放

			if (fTailIsDelimiter) {								//	末尾に「￥」を付加要？
				if (stl = (UI)wcslen(pBuf)) {					//		末尾に「￥」を付加
					if (*(pBuf + (stl - 1)) != L'\\') {			//		・
						wcscat(pBuf, L"\\");					//		・
			}	}	}
		}
	}

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
static C_WCP CALLBACK cbBcpGetText(HWND hwnd, WCP pBuf, UI lBuf, UX cbp)
{
	GetWindowText(hwnd, pBuf, lBuf);
	return pBuf;
}
//--------------------------------------------------------------------------------------------------------------//

static	BOOL CALLBACK cbBcpEnumChild(HWND hwnd, LPARAM lParam)
{
	BOOL			rc = TRUE;
	PGETFOLPARAM	gfp = (PGETFOLPARAM)lParam;
	int				id;
	WC				txt[MAX_PATH + 64];

	id = (int)MAjcGetWindowLong(hwnd, GWL_ID);
	GetClassName(hwnd, txt, AJCTSIZE(txt));
	if (wcsicmp(txt, L"Static") == 0) {
		GetWindowText(hwnd, txt, AJCTSIZE(txt));
		if	   (id == 0x3742) {
			HFONT	hFont = (HFONT)SendMessage(hwnd, WM_GETFONT, 0, 0);
			//	チップテキスト設定
			MAjcSetWindowLong(hwnd, GWL_STYLE, (UI)MAjcGetWindowLong(hwnd, GWL_STYLE) | SS_NOTIFY);
			AjcTipTextAddEx  (hwnd, gfp->pBoxTtl, 500, 10000, hFont, -1, -1, -1);
		}
		else if (id == 0x3743){
		//	RECT	r;
			HFONT	hFont = NULL;
			//	ラベルコントロールを下へ拡張
		//	GetWindowRect(hwnd, &r);
		//	MapWindowPoints(NULL, GetParent(hwnd), (LPPOINT)&r, 2);
		//	SetWindowPos(hwnd, NULL, 0, 0, r.right - r.left, r.bottom - r.top + 7, SWP_NOMOVE);
			//	ツールチップ（コールバック）設定
			hFont = (HFONT)SendMessage(hwnd, WM_GETFONT, 0, 0);
			MAjcSetWindowLong(hwnd, GWL_STYLE, (UI)MAjcGetWindowLong(hwnd, GWL_STYLE) | SS_NOTIFY);
			AjcTipTextAddEx		 (hwnd, L"", 500, 10000, hFont, -1, -1, -1);
			AjcTipTextSetCallBack(hwnd, (UX)hwnd, NULL, cbBcpGetText);
		}
	}
	else if (wcsicmp(txt, L"SysTreeView32") == 0) {
		//	ツリーコントロールを下へ移動
	//	RECT	r;
	//	GetWindowRect(hwnd, &r);
	//	MapWindowPoints(NULL, GetParent(hwnd), (LPPOINT)&r, 2);
	//	SetWindowPos(hwnd, NULL, r.left, r.top + 7, 0, 0, SWP_NOSIZE);
	}

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
static	int CALLBACK BrowseCallbackProcW(HWND hWnd, UINT uMsg, LPARAM lParam, LPARAM lpData)
{
	int		rc = 0;
	WC		buf[MAX_PATH];

	switch (uMsg) {
		case BFFM_INITIALIZED:
			if (lpData) {
				PGETFOLPARAM	gfp = (PGETFOLPARAM)lpData;
				SetWindowTextW(hWnd, (LPWSTR)gfp->pWndTtl);
				SendMessage(hWnd, BFFM_SETSELECTION, (WPARAM)TRUE, (LPARAM)gfp->pPath);
				//	ボックスタイトルへツールチップ設定
				EnumChildWindows(hWnd, cbBcpEnumChild, (LPARAM)lpData);
			}
			break;

		case BFFM_SELCHANGED:
			SHGetPathFromIDListW((LPITEMIDLIST)lParam, buf);
			SendMessage(hWnd, BFFM_SETSTATUSTEXT, 0, (LPARAM)buf);
			break;

		case BFFM_VALIDATEFAILED:
			rc = 1;
			break;
  }
  return rc;
}
//==============================================================================================================//
//	起動したプログラムが格納されているフォルダの絶対パス名を取得します。										//
//																												//
//	引　数　：　pBuf - 絶対パス名を格納するバッファのアドレス（文字列の末尾は常に「￥」となります）				//
//				bfl	 - 絶対パス名を格納するバッファのバイト数／文字数											//
//																												//
//	戻り値　：　TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetAppPathA(BCP pBuf, UI bfl)
{
	BOOL	rc = FALSE;
	BC		mod[MAX_PATH];
	BC		buf[MAX_PATH];
	BC		abs[MAX_PATH];
	BC		drv[_MAX_DRIVE], dir[_MAX_DIR];

	if (pBuf != NULL && bfl > 0) {
		if (rc = GetModuleFileNameA(NULL, mod, MAX_PATH)) {
			if (GetFullPathNameA(mod, MAX_PATH, buf, NULL) == 0) {
				AjcSnPrintFA(buf, MAX_PATH, "%s", mod);
			}
			_splitpath(buf, drv, dir, NULL, NULL);
			_makepath (buf, drv, dir, NULL, NULL);
			if (PathCanonicalizeA(abs, buf)) {
				_splitpath(abs, drv, dir, NULL, NULL);
			}
			if (bfl > strlen(drv) + strlen(dir) + 2) {
				_makepath (pBuf, drv, dir, NULL, NULL);
				rc = TRUE;
			}
			else rc = FALSE;
		}
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetAppPathW(WCP pBuf, UI bfl)
{
	BOOL	rc = FALSE;
	WC		mod[MAX_PATH];
	WC		buf[MAX_PATH];
	WC		abs[MAX_PATH];
	WC		drv[_MAX_DRIVE], dir[_MAX_DIR];

	if (pBuf != NULL && bfl > 0) {
		if (rc = GetModuleFileNameW(NULL, mod, AJCTSIZE(buf))) {
			if (GetFullPathNameW(mod, MAX_PATH, buf, NULL) == 0) {
				AjcSnPrintFW(buf, MAX_PATH, L"%s", mod);
			}
			_wsplitpath(buf, drv, dir, NULL, NULL);
			_wmakepath (buf, drv, dir, NULL, NULL);
			if (PathCanonicalizeW(abs, buf)) {
				_wsplitpath(abs, drv, dir, NULL, NULL);
			}
			if (bfl > wcslen(drv) + wcslen(dir) + 2) {
				_wmakepath (pBuf, drv, dir, NULL, NULL);
				rc = TRUE;
			}
			else rc = FALSE;
		}
	}
	return rc;
}

//==============================================================================================================//
//	起動したプログラム名を取得します。																			//
//																												//
//	引　数　：　pBuf - プログラム名を格納するバッファのアドレス													//
//				bfl	 - プログラム名を格納するバッファのバイト数／文字数											//
//																												//
//	戻り値　：　TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetAppNameA(BCP pBuf, UI bfl)
{
	BOOL	rc = FALSE;
	BC		buf[MAX_PATH];
	BC		fname[_MAX_FNAME] = {0};
	BC		fext [_MAX_EXT	] = {0};

	if (pBuf != NULL && bfl > 0) {
		if (rc = GetModuleFileNameA(NULL, buf, AJCTSIZE(buf))) {
			_splitpath(buf, NULL, NULL, fname, fext);
			_makepath (buf, NULL, NULL, fname, fext);
			if (bfl >= strlen(buf) + 1) {
				strcpy(pBuf, buf);
				rc = TRUE;
			}
			else rc = FALSE;
		}
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetAppNameW(WCP pBuf, UI bfl)
{
	BOOL	rc = FALSE;
	WC		buf[MAX_PATH];
	WC		fname[_MAX_FNAME] = {0};
	WC		fext [_MAX_EXT	] = {0};

	if (pBuf != NULL && bfl > 0) {
		if (rc = GetModuleFileNameW(NULL, buf, AJCTSIZE(buf))) {
			_wsplitpath(buf, NULL, NULL, fname, fext);
			_wmakepath (buf, NULL, NULL, fname, fext);
			if (bfl > wcslen(buf) + + 1) {
				wcscpy (pBuf, buf);
				rc = TRUE;
			}
			else rc = FALSE;
		}
	}
	return rc;
}

//==============================================================================================================//
//	指定したウインド（プロセス）の絶対パス名を取得します。														//
//																												//
//	引　数　：　hWnd - プロセスが所有するウインド																//
//				pBuf - 絶対パス名を格納するバッファのアドレス													//
//				bfl	 - 絶対パス名を格納するバッファのバイト数／文字数											//
//																												//
//	戻り値　：　TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetAppPathByHWndA(HWND hWnd, BCP pBuf, UI bfl)
{
	BOOL		rc = FALSE;
	HINSTANCE	hI	= NULL;
	UL			pid = 0;
	UL			tid = 0;
	HANDLE		hProc = NULL;

	if (hI = (HINSTANCE)MAjcGetWindowLong(hWnd, GWL_HINSTANCE)) {
		tid = GetWindowThreadProcessId(hWnd, &pid);
		if(tid && pid) {
			if (hProc = OpenProcess(PROCESS_ALL_ACCESS, FALSE, pid)) {
				if (GetModuleFileNameExA(hProc, hI, pBuf, bfl)) {
					rc = TRUE;
				}
				CloseHandle(hProc);
			}
		}
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetAppPathByHWndW(HWND hWnd, WCP pBuf, UI bfl)
{
	BOOL		rc = FALSE;
	HINSTANCE	hI	= NULL;
	UL			pid = 0;
	UL			tid = 0;
	HANDLE		hProc = NULL;

	if (hI = (HINSTANCE)MAjcGetWindowLong(hWnd, GWL_HINSTANCE)) {
		tid = GetWindowThreadProcessId(hWnd, &pid);
		if(tid && pid) {
			if (hProc = OpenProcess(PROCESS_ALL_ACCESS, FALSE, pid)) {
				if (GetModuleFileNameExW(hProc, hI, pBuf, bfl)) {
					rc = TRUE;
				}
				CloseHandle(hProc);
			}
		}
	}
	return rc;
}
