﻿#include	"AjcInternal.h"

//**************************************************************************************************************//
//																												//
//	インテル／モトローラＨＥＸデータ																			//
//																												//
//**************************************************************************************************************//

/*==============================================================================================================*/
/*	Function	:	インテルＨＥＸ／モトローラＳ形式のデータをバイナリ化し、メモリ上に展開します。				*/
/*																												*/
/*	Argument	:	pBuf    - プログラムオブジェクトを格納するバッファのアドレス（※１）						*/
/*					nBuf    - プログラムオブジェクトを格納するバッファのバイト数（※１）						*/
/*					ofs     - バッファ先頭のオフセット値（この値以降のオブジェクトがバッファにロードされる）	*/
/*					pSrtAdr - スタートアドレスを格納するバッファのアドレス（不要時は、ＮＵＬＬ）				*/
/*							  但し、スタートアドレスが無い場合は、0xFFFFFFFFを格納します						*/
/*					xp      - コールバックパラメタ																*/
/*					cbGetRec- インテルＨＥＸ／モトローラＳ形式テキストレコード取得用コールバック関数			*/
/*																												*/
/*	Return		:	＝０：ＯＫ																					*/
/*					≠０：エラー																				*/
/*																												*/
/*	※１：	　このバッファは、（複数のＨＥＸファイルをマージ可能とするために）ＦＦｈ等のデータによるクリアー	*/
/*			は行いません。従って、オブジェクトのロードされないアドレスの内容は、実行前のままとなります。		*/
/*																												*/
/*==============================================================================================================*/
#define	LDPTXTBUFSIZE	512
#define	LDPBINBUFSIZE	512
typedef  struct  {
	UB		Typ;			//	レコードタイプ
	UL		Adr;			//	アドレス情報
	UI		Len;			//	バイナリデータバイト数
	UL		Base;			//	ベ-スアドレス情報
	UL		LimL;			//	バイナリイメージの低位オフセット
	UL		LimH;			//	バイナリイメージの高位オフセット
	UBP		pBinBuf;		//	１レコード分のバイナリデータバッファアドレス
	UBP		pBuf;			//	バイナリイメージ・バッファ
	UL		nBuf;			//	バイナリ・イメージバッファのバイト数
	ULP		pSrtAdr;		//	スタートアドレス
} LDPWRK, *LPLDPWRK;

static	UI	lposub_StoreData(LPLDPWRK pW);

/*----- バイト文字用 -------------------------------------------------------------------------------------------*/
AJCEXPORT	UI	WINAPI	AjcLoadHexDataA(UBP pBuf, UL nBuf, UL ofs, ULP pSrtAdr,
										UX xp, BOOL (CALLBACK *cbGetRec)(UX xp, BCP pRec, UI nRec))
{
	UI		rc = AJCIHERR_NOEND;
	UI		rsu;
	LDPWRK	sW;
	BC		TxtBuf[LDPTXTBUFSIZE];
	UB		BinBuf[LDPBINBUFSIZE];

	sW.Typ     = 0xFF;
	sW.Base    = 0;
	sW.LimL    = ofs;
	sW.LimH    = ofs + nBuf;
	sW.pBinBuf = BinBuf;
	sW.pBuf    = pBuf;
	sW.nBuf    = nBuf;
	sW.pSrtAdr = pSrtAdr;

	if (sW.pSrtAdr) *sW.pSrtAdr = 0xFFFFFFFF;

	while (cbGetRec(xp, TxtBuf, LDPTXTBUFSIZE)) {
		if (TxtBuf[0] == ':') rsu = AjcIHexToBinA(TxtBuf, sW.pBinBuf, LDPBINBUFSIZE, &sW.Typ, &sW.Adr, &sW.Len);
		else                  rsu = AjcSMotToBinA(TxtBuf, sW.pBinBuf, LDPBINBUFSIZE, &sW.Typ, &sW.Adr, &sW.Len);
		if (rsu != AJCIHERR_OK) {
			rc = rsu;
			break;
		}
		rsu = lposub_StoreData(&sW);
		if (rsu != 0xFFFF) {
			rc = rsu;
			break;
		}
	}
	return rc;
}
/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
static	UI	lposub_StoreData(LPLDPWRK pW)
{
	UI		rc = 0xFFFF;
	UL		CurL, CurH;

	if      (pW->Typ == 0x00 || pW->Typ == '2' || pW->Typ == '3') {		/*	●データレコード（iHEX/motS）		*/
		if (pW->Typ == 0x00) CurL = pW->Base + pW->Adr;
		else                 CurL = pW->Adr;
		CurH = CurL + pW->Len;
		if      (CurL >= pW->LimL && CurH <= pW->LimH)
			memcpy(pW->pBuf + (CurL - pW->LimL), pW->pBinBuf                    , (UI)(pW->Len        ));
		else if (CurL <= pW->LimL && CurH >= pW->LimH)
			memcpy(pW->pBuf                    , pW->pBinBuf + (pW->LimL - CurL), (UI)(pW->nBuf       ));
		else if (CurL <= pW->LimL && CurH >  pW->LimL)
			memcpy(pW->pBuf                    , pW->pBinBuf + (pW->LimL - CurL), (UI)(CurH - pW->LimL));
		else if (CurL <  pW->LimH && CurH >= pW->LimH)
			memcpy(pW->pBuf + (CurL - pW->LimL), pW->pBinBuf                    , (UI)(pW->LimH - CurL));
	}
	else if (pW->Typ == 0x01) {											/*	●ＥＮＤレコード（iHEX）			*/
		rc = AJCIHERR_OK;
	}
	else if (pW->Typ == 0x02 || pW->Typ == 0x04) {						/*	●拡張アドレスレコード（iHEX）		*/
		pW->Base = pW->Adr;
	}
	else if (pW->Typ == 0x03) {											/*	●スタートアドレスレコード（iHEX）	*/
		if (pW->pSrtAdr)
			*pW->pSrtAdr = pW->Adr;
	}
	else if (pW->Typ == '7' || pW->Typ == '8') {						/*	●ＥＮＤレコード（motS）			*/
		if (pW->pSrtAdr)
			*pW->pSrtAdr = pW->Adr;
		rc = AJCIHERR_OK;
	}
	return rc;
}
/*----- ワイド文字用 -------------------------------------------------------------------------------------------*/
AJCEXPORT	UI	WINAPI	AjcLoadHexDataW(UBP pBuf, UL nBuf, UL ofs, ULP pSrtAdr,
										UX xp, BOOL (CALLBACK *cbGetRec)(UX xp, WCP pRec, UI nRec))
{
	UI		rc = AJCIHERR_NOEND;
	UI		rsu;
	LDPWRK	sW;
	WC		TxtBuf[LDPTXTBUFSIZE];
	UB		BinBuf[LDPBINBUFSIZE];

	sW.Typ     = 0xFF;
	sW.Base    = 0;
	sW.LimL    = ofs;
	sW.LimH    = ofs + nBuf;
	sW.pBinBuf = BinBuf;
	sW.pBuf    = pBuf;
	sW.nBuf    = nBuf;
	sW.pSrtAdr = pSrtAdr;

	if (sW.pSrtAdr) *sW.pSrtAdr = 0xFFFFFFFF;

	while (cbGetRec(xp, TxtBuf, LDPTXTBUFSIZE)) {
		if (TxtBuf[0] == L':') rsu = AjcIHexToBinW(TxtBuf, sW.pBinBuf, LDPBINBUFSIZE, &sW.Typ, &sW.Adr, &sW.Len);
		else                   rsu = AjcSMotToBinW(TxtBuf, sW.pBinBuf, LDPBINBUFSIZE, &sW.Typ, &sW.Adr, &sW.Len);
		if (rsu != 0) {
			rc = rsu;
			break;
		}
		rsu = lposub_StoreData(&sW);
		if (rsu != 0xFFFF) {
			rc = rsu;
			break;
		}
	}
	return rc;
}
/*==============================================================================================================*/
/*	Function	:	インテルＨＥＸ／モトローラＳ形式のファイルをバイナリ形式に変換し、バッファにロードする		*/
/*																												*/
/*	Argument	:	pFile	- ファイル名文字列のアドレス（インテルＨＥＸ／モトローラＳファイル）				*/
/*					pBuf	- バイナリ形式に変換したデータを格納するバッファのアドレス（※１）					*/
/*					nBuf	- バイナリ形式に変換したデータを格納するバッファのバイト数（※１）					*/
/*					ofs		- バッファ先頭のオフセット値（この値以降のオブジェクトがバッファにロードされる）	*/
/*					pSrtAdr - スタートアドレスを格納するバッファのアドレス（不要時は、ＮＵＬＬ）				*/
/*							  但し、スタートアドレスが無い場合は、0xFFFFFFFFを格納します						*/
/*																												*/
/*	Return		:	＝０：ＯＫ																					*/
/*					≠０：エラー																				*/
/*																												*/
/*	※１：	　このバッファは、（複数のＨＥＸファイルをマージ可能とするために）ＦＦｈ等のデータによるクリアー	*/
/*			は行いません。従って、オブジェクトのロードされないアドレスの内容は、実行前のままとなります。		*/
/*																												*/
/*==============================================================================================================*/
//----- バイト文字用 -------------------------------------------------------------------------------------------//
static	BOOL  CALLBACK	lpfsub_GetRecA(UX xp, BCP pRec, UI nRec)
{
	return fgets(pRec, nRec, (FILE*)xp) != NULL;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcLoadHexFileA(C_BCP pFile, UBP pBuf, UL nBuf, UL ofs, ULP pSrtAdr)
{
	UI		rc;
	FILE	*pFh;

	if (pFh = fopen(pFile, "rt")) {
		rc = AjcLoadHexDataA(pBuf, nBuf, ofs, pSrtAdr, (UX)pFh, lpfsub_GetRecA);
		fclose(pFh);
	}
	else {
		rc = AJCIHERR_FOPEN;
	}

	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
static	BOOL  CALLBACK	lpfsub_GetRecW(UX xp, WCP pRec, UI nRec)
{
	return fgetws(pRec, nRec, (FILE*)xp) != NULL;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcLoadHexFileW(C_WCP pFile, UBP pBuf, UL nBuf, UL ofs, ULP pSrtAdr)
{
	UI		rc;
	FILE	*pFh;

	if (pFh = _wfopen(pFile, L"rt")) {
		rc = AjcLoadHexDataW(pBuf, nBuf, ofs, pSrtAdr, (UX)pFh, lpfsub_GetRecW);
		fclose(pFh);
	}
	else {
		rc = AJCIHERR_FOPEN;
	}

	return rc;
}
/*==============================================================================================================*/
/*	Function	:	インテルＨＥＸ形式のレコード（テキスト形式）をバイナリ形式に変換する						*/
/*																												*/
/*	Argument	:	pRec - インテルＨＥＸレコードの先頭アドレス													*/
/*					pBuf - バイナリ形式に変換したデータを格納するバッファのアドレス				（※１）		*/
/*					nBuf - バッファのバイト数																	*/
/*					pTyp - レコードタイプ（0x00/0x01/0x02/0x04）を格納するバッファのアドレス					*/
/*					pAdr - アドレス情報を格納するバッファのアドレス								（※２）		*/
/*					pLen - 格納したデータバイト数を格納するバッファのアドレス					（※１）		*/
/*																												*/
/*					※１：　データレコードの実データ部分が「pBuf」で示されるバッファに格納され、そのバイト数	*/
/*						  が「pLen」で示されるバッファに設定されます。											*/
/*							但し、「nBuf」で示されるバッファサイズが、実データバイト数よりも小さい場合は、		*/
/*						　後部の実データはバッファに格納されません。											*/
/*							データレコード以外の場合は、このバッファには何も設定されません。					*/
/*																												*/
/*					※２：　レコードタイプにより、以下の情報が設定されます。									*/
/*																												*/
/*							・データレコード ------------ データのロケーションアドレス（0～0xFFFF）				*/
/*							・ＥＮＤレコード ------------ ０が設定される										*/
/*							・拡張アドレスレコード ------ ベースアドレス（バイト単位に変換したアドレス値）		*/
/*							・スタートアドレスレコード -- 開始アドレス（0～0xFFFFFFFF）							*/
/*																												*/
/*	Return		:	＝０：ＯＫ																					*/
/*					≠０：エラー																				*/
/*==============================================================================================================*/
/*----- バイト文字用 -------------------------------------------------------------------------------------------*/
AJCEXPORT	UI	WINAPI	AjcIHexToBinA(C_BCP pRec, UBP pBuf, UI nBuf, UBP pTyp, ULP pAdr, UIP pLen)
{
	UI		rc = AJCIHERR_OK;
	UB		sum, cs, datl;

	*pLen = 0;
	do {
		if (*pRec == ':') {
			sum = 0;
			pRec++;
			if (!AjcHexStrToUBA(pRec, 2, &datl)) {rc = AJCIHERR_DIGIT;  break;}
			sum += datl;  pRec += 2;
			if (!AjcHexStrToUIA(pRec, 4, pAdr )) {rc = AJCIHERR_DIGIT;  break;}
			sum += (UB)((*pAdr) >> 8);  sum += (UB)((*pAdr) & 255);  pRec += 4;
			if (!AjcHexStrToUBA(pRec, 2, pTyp )) {rc = AJCIHERR_DIGIT;  break;}
			sum += *pTyp; pRec += 2;
			switch (*pTyp) {
				case 0x00:									/*	●データレコード								*/

					while (*pLen < nBuf  &&  *pLen < datl) {
						if (!AjcHexStrToUBA(pRec, 2, pBuf)) {rc = AJCIHERR_DIGIT; break;}
						pRec += 2;
						sum += *pBuf;
						pBuf++; (*pLen)++;
					}
					break;

				case 0x01:									/*	●ＥＮＤレコード								*/

					break;

				case 0x02:									/*	●拡張アドレスレコード（パラグラフ単位）		*/
				case 0x04:									/*	●拡張アドレスレコード（６４ＫＢ  単位）		*/

					if (!AjcHexStrToUIA(pRec, 4, pAdr)) {rc = AJCIHERR_DIGIT; break;}
					sum += (UB)(((*pAdr) >>  8) & 0xFF);
					sum += (UB)( (*pAdr)        & 0xFF);
					if (*pTyp == 0x02) *pAdr <<= 4;
					else               *pAdr <<= 16;
					pRec += 4;
					break;

				case 0x03:									/*	●スタートアドレスレコード						*/

					if (!AjcHexStrToUIA(pRec, 8, pAdr)) {rc = AJCIHERR_DIGIT; break;}
					pRec += 8;
					sum += (UB)(((*pAdr) >> 24) & 0xFF);
					sum += (UB)(((*pAdr) >> 16) & 0xFF);
					sum += (UB)(((*pAdr) >>  8) & 0xFF);
					sum += (UB)( (*pAdr)        & 0xFF);
					break;

				default:									/*	●その他										*/

					rc = AJCIHERR_TYPE;
					break;
			}
			if (rc == AJCIHERR_OK) {
				if (!AjcHexStrToUBA(pRec, 2, &cs)) 	rc = AJCIHERR_DIGIT;
				if (((sum + cs) & 255) != 0) 		rc = AJCIHERR_SUM;
			}
		}
		else {
			rc = AJCIHERR_NOMARK;
		}
	} while(0);

	return rc;
}
/*----- ワイド文字用 -------------------------------------------------------------------------------------------*/
AJCEXPORT	UI	WINAPI	AjcIHexToBinW(C_WCP pRec, UBP pBuf, UI nBuf, UBP pTyp, ULP pAdr, UIP pLen)
{
	UI		rc = AJCIHERR_OK;
	UB		sum, cs, datl;

	*pLen = 0;
	do {
		if (*pRec == ':') {
			sum = 0;
			pRec++;
			if (!AjcHexStrToUBW(pRec, 2, &datl)) {rc = AJCIHERR_DIGIT;  break;}
			sum += datl;  pRec += 2;
			if (!AjcHexStrToUIW(pRec, 4, pAdr )) {rc = AJCIHERR_DIGIT;  break;}
			sum += (UB)((*pAdr) >> 8);  sum += (UB)((*pAdr) & 255);  pRec += 4;
			if (!AjcHexStrToUBW(pRec, 2, pTyp )) {rc = AJCIHERR_DIGIT;  break;}
			sum += *pTyp; pRec += 2;
			switch (*pTyp) {
				case 0x00:									/*	●データレコード								*/

					while (*pLen < nBuf  &&  *pLen < datl) {
						if (!AjcHexStrToUBW(pRec, 2, pBuf)) {rc = AJCIHERR_DIGIT; break;}
						pRec += 2;
						sum += *pBuf;
						pBuf++; (*pLen)++;
					}
					break;

				case 0x01:									/*	●ＥＮＤレコード								*/

					break;

				case 0x02:									/*	●拡張アドレスレコード（パラグラフ単位）		*/
				case 0x04:									/*	●拡張アドレスレコード（６４ＫＢ  単位）		*/

					if (!AjcHexStrToUIW(pRec, 4, pAdr)) {rc = AJCIHERR_DIGIT; break;}
					sum += (UB)(((*pAdr) >>  8) & 0xFF);
					sum += (UB)( (*pAdr)        & 0xFF);
					if (*pTyp == 0x02) *pAdr <<= 4;
					else               *pAdr <<= 16;
					pRec += 4;
					break;

				case 0x03:									/*	●スタートアドレスレコード						*/

					if (!AjcHexStrToUIW(pRec, 8, pAdr)) {rc = AJCIHERR_DIGIT; break;}
					pRec += 8;
					sum += (UB)(((*pAdr) >> 24) & 0xFF);
					sum += (UB)(((*pAdr) >> 16) & 0xFF);
					sum += (UB)(((*pAdr) >>  8) & 0xFF);
					sum += (UB)( (*pAdr)        & 0xFF);
					break;

				default:									/*	●その他										*/

					rc = AJCIHERR_TYPE;
					break;
			}
			if (rc == AJCIHERR_OK) {
				if (!AjcHexStrToUBW(pRec, 2, &cs)) 	rc = AJCIHERR_DIGIT;
				if (((sum + cs) & 255) != 0) 		rc = AJCIHERR_SUM;
			}
		}
		else {
			rc = AJCIHERR_NOMARK;
		}
	} while(0);

	return rc;
}
/*==============================================================================================================*/
/*	Function	:	モトローラＳ形式のレコード（テキスト形式）をバイナリ形式に変換する							*/
/*																												*/
/*	Argument	:	pRec - モトローラＳレコードの先頭アドレス													*/
/*					pBuf - バイナリ形式に変換したデータを格納するバッファのアドレス				（※１）		*/
/*					nBuf - バッファのバイト数																	*/
/*					pTyp - レコードタイプ（'0'/'2'/'3'/'7'/'8'）を格納するバッファのアドレス					*/
/*					pAdr - アドレス情報を格納するバッファのアドレス								（※２）		*/
/*					pLen - 格納したデータバイト数を格納するバッファのアドレス					（※１）		*/
/*																												*/
/*					※１：　S2/S3レコードの実データ部分が「pBuf」で示されるバッファに格納され、そのバイト数		*/
/*						  が「pLen」で示されるバッファに設定されます。											*/
/*							但し、「nBuf」で示されるバッファサイズが、実データバイト数よりも小さい場合は、		*/
/*						　後部の実データはバッファに格納されません。											*/
/*							S2/S3レコード以外の場合は、このバッファには何も設定されません。						*/
/*																												*/
/*					※２：　レコードタイプにより、以下の情報が設定されます。									*/
/*																												*/
/*							・Ｓ０ -------- 何も設定されない													*/
/*							・Ｓ２／Ｓ３ -- データのアドレス（0～0xFFFFFFFF）									*/
/*							・Ｓ７／Ｓ８ -- 開始アドレス    （0～0xFFFFFFFF）									*/
/*																												*/
/*	Return		:	＝０：ＯＫ																					*/
/*					≠０：エラー																				*/
/*==============================================================================================================*/
/*----- バイト文字用 -------------------------------------------------------------------------------------------*/
AJCEXPORT	UI	WINAPI	AjcSMotToBinA(C_BCP pRec, UBP pBuf, UI nBuf, UBP pTyp, ULP pAdr, UIP pLen)
{
	UI		rc = AJCIHERR_OK;
	UB		sum, cs, al, recl, datl;

	*pLen = 0;

	do {
		if (*pRec++ == 'S') {
			*pTyp = *pRec++;
			if (!AjcHexStrToUBA(pRec, 2, &recl)) {rc = AJCIHERR_DIGIT;  break;}
			sum = recl;
			pRec += 2;
			datl = recl - 1;
			if      (*pTyp == '0')	{al = 0; datl = recl - 1;}
			else if (*pTyp == '2')	{al = 6; datl = recl - 4;}
			else if (*pTyp == '3')	{al = 8; datl = recl - 5;}
			else if (*pTyp == '7')	{al = 8; datl = 0;		 }
			else if (*pTyp == '8')	{al = 6; datl = 0;		 }
			else {rc = AJCIHERR_TYPE; break;}
			if (al != 0) {
				if (!AjcHexStrToUIA(pRec, al, pAdr)) {rc = AJCIHERR_DIGIT;  break;}
				pRec += al;
				sum += (UB)(((*pAdr) >> 24) & 0xFF);
				sum += (UB)(((*pAdr) >> 16) & 0xFF);
				sum += (UB)(((*pAdr) >>  8) & 0xFF);
				sum += (UB)( (*pAdr)        & 0xFF);
			}
			if (datl != 0) {
				while (*pLen < nBuf  &&  *pLen < datl) {
					if (!AjcHexStrToUBA(pRec, 2, pBuf)) {rc = AJCIHERR_DIGIT; break;}
					pRec += 2;
					sum += *pBuf;
					pBuf++; (*pLen)++;
				}
			}
			if (rc == AJCIHERR_OK) {
				if (!AjcHexStrToUBA(pRec, 2, &cs)) 	rc = AJCIHERR_DIGIT;
				if (((sum + cs) & 255) != 0xFF) 	rc = AJCIHERR_SUM;
			}
		}
		else {
			rc = AJCIHERR_NOMARK;
		}

	} while (0);

	return rc;
}
/*----- ワイド文字用 -------------------------------------------------------------------------------------------*/
AJCEXPORT	UI	WINAPI	AjcSMotToBinW(C_WCP pRec, UBP pBuf, UI nBuf, UBP pTyp, ULP pAdr, UIP pLen)
{
	UI		rc = AJCIHERR_OK;
	UB		sum, cs, al, recl, datl;

	*pLen = 0;

	do {
		if (*pRec++ == 'S') {
			*pTyp = (UB)(*pRec++);
			if (!AjcHexStrToUBW(pRec, 2, &recl)) {rc = AJCIHERR_DIGIT;  break;}
			sum = recl;
			pRec += 2;
			if      (*pTyp == '0')	{al = 0; datl = recl - 1;}
			else if (*pTyp == '2')	{al = 6; datl = recl - 4;}
			else if (*pTyp == '3')	{al = 8; datl = recl - 5;}
			else if (*pTyp == '7')	{al = 8; datl = 0;		 }
			else if (*pTyp == '8')	{al = 6; datl = 0;		 }
			else {rc = AJCIHERR_TYPE; break;}

			if (al != 0) {
				if (!AjcHexStrToUIW(pRec, al, pAdr)) {rc = AJCIHERR_DIGIT;  break;}
				pRec += al;
				sum += (UB)(((*pAdr) >> 24) & 0xFF);
				sum += (UB)(((*pAdr) >> 16) & 0xFF);
				sum += (UB)(((*pAdr) >>  8) & 0xFF);
				sum += (UB)( (*pAdr)        & 0xFF);
			}
			if (datl != 0) {
				while (*pLen < nBuf  &&  *pLen < datl) {
					if (!AjcHexStrToUBW(pRec, 2, pBuf)) {rc = AJCIHERR_DIGIT; break;}
					pRec += 2;
					sum += *pBuf;
					pBuf++; (*pLen)++;
				}
			}
			if (rc == AJCIHERR_OK) {
				if (!AjcHexStrToUBW(pRec, 2, &cs)) 	rc = AJCIHERR_DIGIT;
				if (((sum + cs) & 255) != 0xFF) 	rc = AJCIHERR_SUM;
			}
		}
		else {
			rc = AJCIHERR_NOMARK;
		}

	} while (0);

	return rc;
}

