﻿#include	"AjcInternal.h"

//**************************************************************************************************************//
//																												//
//	ＩＮＩファイルアクセス																						//
//																												//
//**************************************************************************************************************//
//--------------------------------------------------------------------------------------------------------------//
//	作業領域																									//
//--------------------------------------------------------------------------------------------------------------//
//	バイナリデータ格納ディレクトリパス
static	BC		BinDataDirA[MAX_PATH] = {0};
static	WC		BinDataDirW[MAX_PATH] = {0};

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
extern	BCP		SubProfileGenerateBinFileA(BOOL fVolatile, C_BCP pTail, BC full[MAX_PATH], C_BCP pBinDir);
extern	WCP		SubProfileGenerateBinFileW(BOOL fVolatile, C_WCP pTail, UW full[MAX_PATH], C_WCP pBinDir);

extern	BOOL	SubProfileDeleteBinFileA(C_BCP pTail, C_BCP pBinDir);
extern	BOOL	SubProfileDeleteBinFileW(C_WCP pTail, C_WCP pBinDir);

static	BOOL	SubDelBinFilesA(C_BCP pIniFile, C_BCP pSecName, C_BCP pBinDir);
static	BOOL	SubDelBinFilesW(C_WCP pIniFile, C_WCP pSecName, C_WCP pBinDir);

static	BOOL	SubDelAllKeysA(C_BCP pIniFile, C_BCP pSecName);
static	BOOL	SubDelAllKeysW(C_WCP pIniFile, C_WCP pSecName);

static	BOOL	CreateIniFileA(C_BCP pIniFile);
static	BOOL	CreateIniFileW(C_WCP pIniFile);

//==============================================================================================================//
//	バイナリデータ格納ディレクトリパス設定																		//
//																												//
//	引　数	：	pBinDir	   - バイナリデータ格納ディレクトリパス（NULLの場合は、保存先解除）						//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	VO		WINAPI	AjcSetIniBinDirA(C_BCP pBinDir)
{
	if (pBinDir != NULL && *pBinDir != 0) {
		AjcSnPrintFA(BinDataDirA, MAX_PATH, "%s", pBinDir);
		AjcPathCatA(BinDataDirA, "", MAX_PATH);
		BinDataDirA[MAX_PATH - 1] = 0;
	}
	else {
		AjcGetAppPathA(BinDataDirA, MAX_PATH);
	}
	MultiByteToWideChar(CP_ACP, 0, BinDataDirA, -1, BinDataDirW, MAX_PATH);
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	VO		WINAPI	AjcSetIniBinDirW(C_WCP pBinDir)
{
	if (pBinDir != NULL && *pBinDir != 0) {
		AjcSnPrintFW(BinDataDirW, MAX_PATH, L"%s", pBinDir);
		AjcPathCatW(BinDataDirW, L"", MAX_PATH);
		BinDataDirW[MAX_PATH - 1] = 0;
	}
	else {
		AjcGetAppPathW(BinDataDirW, MAX_PATH);
	}
	WideCharToMultiByte(CP_ACP, 0, BinDataDirW, -1, BinDataDirA, MAX_PATH, NULL, NULL);
}
//==============================================================================================================//
//	バイナリデータ格納ディレクトリパス取得																		//
//																												//
//	引　数	：	pBuf	- バイナリデータ格納ディレクトリパスを格納するバッファのアドレス						//
//				lBuf	- バイナリデータ格納ディレクトリパスを格納するバッファの文字数							//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	VO		WINAPI	AjcGetIniBinDirA(BCP pBuf, UI lBuf)
{
	if (pBuf != NULL  &&  lBuf > 0) {
		AjcSnPrintFA(pBuf, lBuf, "%s", BinDataDirA);
	}
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	VO		WINAPI	AjcGetIniBinDirW(WCP pBuf, UI lBuf)
{
	if (pBuf != NULL  &&  lBuf > 0) {
		AjcSnPrintFW(pBuf, lBuf, L"%s", BinDataDirW);
	}
}

//==============================================================================================================//
//	ＩＮＩファイル読み出し（数値）																				//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//				pKeyName	- キー名のアドレス																	//
//				defValue	- デフォルト値																		//
//																												//
//	戻り値：	ＩＮＩファイルから読み出した当該キー値／デフォルト値											//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
#define	M_GSGETPROFILEN_A(_T1_, _T2_, _BS_, _CV_)																\
AJCEXPORT _T1_ 	WINAPI AjcGetIniFile##_T2_##A(C_BCP pIniFile, C_BCP pSecName, C_BCP pKeyName, _T1_ defValue)	\
{	_T1_	rc = defValue;																						\
	BOOL	fHex;																								\
	BCP		p;																									\
	BC	buf[_BS_];																								\
	if (pIniFile != NULL && pSecName != NULL && pKeyName != NULL) {												\
		GetPrivateProfileStringA(pSecName, pKeyName, "$$UNDEF$$", buf, _BS_, pIniFile);							\
		if (mbscmp(buf, "$$UNDEF$$") == 0) {																	\
			rc = defValue;																						\
		}																										\
		else {																									\
			for (p=buf; MAjcIsBlankA(*p); p++);																	\
			fHex = (strnicmp(p, "0x", 2) == 0);																	\
			rc = (_T1_)_CV_;																					\
		}																										\
	}																											\
	return rc;																									\
}
M_GSGETPROFILEN_A(UI	, UInt,  32, fHex ? AjcHexToUIA(p) : AjcDecToUIA(p))
M_GSGETPROFILEN_A(SI	, SInt,  32, fHex ? AjcHexToUIA(p) : atoi(p))
M_GSGETPROFILEN_A(UI	, Hex ,  32, fHex ? AjcHexToUIA(p) : AjcDecToUIA(p))
M_GSGETPROFILEN_A(double, Real, 512, AjcAscToRealA(p))
M_GSGETPROFILEN_A(ULL	, UI64,  64, fHex ? AjcHexToULLA(p) : AjcDecToULLA(p))
M_GSGETPROFILEN_A(SLL	, SI64,  64, fHex ? AjcHexToULLA(p) : _atoi64(p))
M_GSGETPROFILEN_A(ULL	, H64 ,  64, fHex ? AjcHexToULLA(p) : AjcDecToULLA(p))

//----- ワイド文字用 -------------------------------------------------------------------------------------------//
#define	M_GSGETPROFILEN_W(_T1_, _T2_, _BS_, _CV_)																\
AJCEXPORT _T1_ 	WINAPI AjcGetIniFile##_T2_##W(C_WCP pIniFile, C_WCP pSecName, C_WCP pKeyName, _T1_ defValue)	\
{	_T1_	rc = defValue;																						\
	BOOL	fHex;																								\
	WCP		p;																									\
	WC	buf[_BS_];																								\
	if (pIniFile != NULL && pSecName != NULL && pKeyName != NULL) {												\
		GetPrivateProfileStringW(pSecName, pKeyName, L"$$UNDEF$$", buf, _BS_, pIniFile);						\
		if (wcscmp(buf, L"$$UNDEF$$") == 0) {																	\
			rc = defValue;																						\
		}																										\
		else {																									\
			for (p=buf; iswspace(*p); p++);																		\
			fHex = (wcsnicmp(buf, L"0x", 2) == 0);																\
			rc = (_T1_)_CV_;																					\
		}																										\
	}																											\
	return rc;																									\
}
M_GSGETPROFILEN_W(UI	, UInt,  32, fHex ? AjcHexToUIW(p) : AjcDecToUIW(p))
M_GSGETPROFILEN_W(SI	, SInt,  32, fHex ? AjcHexToUIW(p) : _wtoi(p))
M_GSGETPROFILEN_W(UI	, Hex ,  32, fHex ? AjcHexToUIW(p) : AjcDecToUIW(p))
M_GSGETPROFILEN_W(double, Real, 512, AjcAscToRealW(p))
M_GSGETPROFILEN_W(ULL	, UI64,  64, fHex ? AjcHexToULLW(p) : AjcDecToULLW(p))
M_GSGETPROFILEN_W(SLL	, SI64,  64, fHex ? AjcHexToULLW(p) : _wtoi64(p))
M_GSGETPROFILEN_W(ULL	, H64 ,  64, fHex ? AjcHexToULLW(p) : AjcDecToULLW(p))

//==============================================================================================================//
//	ＩＮＩファイル読み出し（整数配列）																			//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//				pKeyName	- キー名のアドレス																	//
//				pArray		- 読み出した値を格納する整数配列のアドレス											//
//				lArray		- 整数配列の要素サイズ（１，２，４ or ８）											//
//				nArray		- 整数配列の要素数																	//
//																												//
//	戻り値：	ＩＮＩファイルから読み出した整数値の個数														//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcGetIniFileArrA(C_BCP pIniFile, C_BCP pSecName, C_BCP pKeyName, VOP pArray, UI lArray, UI nArray)
{
	UI		rc = 0;
	UI		lTmpIniFile, lTmpSecName, lTmpKeyName;
	WCP		pTmpIniFile;
	WCP		pTmpSecName;
	WCP		pTmpKeyName;

	if (pIniFile != NULL && pSecName != NULL && pKeyName != NULL) {
		pTmpIniFile	= AjcTAllocW(lTmpIniFile = MultiByteToWideChar(CP_ACP, 0, pIniFile, -1, NULL, 0));
		pTmpSecName	= AjcTAllocW(lTmpSecName = MultiByteToWideChar(CP_ACP, 0, pSecName, -1, NULL, 0));
		pTmpKeyName	= AjcTAllocW(lTmpKeyName = MultiByteToWideChar(CP_ACP, 0, pKeyName, -1, NULL, 0));
		if (pTmpIniFile && pTmpSecName && pTmpKeyName) {
			MultiByteToWideChar(CP_ACP, 0, pIniFile, -1, pTmpIniFile, lTmpIniFile);
			MultiByteToWideChar(CP_ACP, 0, pSecName, -1, pTmpSecName, lTmpSecName);
			MultiByteToWideChar(CP_ACP, 0, pKeyName, -1, pTmpKeyName, lTmpKeyName);
			rc = AjcGetIniFileArrW(pTmpIniFile, pTmpSecName, pTmpKeyName, pArray, lArray, nArray);
		}

		if (pTmpIniFile) AjcTFree(pTmpIniFile);
		if (pTmpSecName) AjcTFree(pTmpSecName);
		if (pTmpKeyName) AjcTFree(pTmpKeyName);
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcGetIniFileArrW(C_WCP pIniFile, C_WCP pSecName, C_WCP pKeyName, VOP pArray, UI lArray, UI nArray)
{
	UI		rc = 0, n = 0;
	UI		lBuf;
	WCP		pBuf, pTxt;
	union {
		VOP		pVO;
		UBP		pUB;
		WCP		pUW;
		ULP		pUL;
		ULLP	pULL;
	} u;

	if (pIniFile  &&  pSecName	&&	pKeyName  &&  pArray  &&  (lArray == 1 || lArray == 2 || lArray == 4 || lArray == 8)) {
		lBuf = (((lArray * 2) + 1) * nArray) + 1;
		if (pBuf = AjcTAllocW(lBuf)) {
			*pBuf = 0;
			GetPrivateProfileStringW(pSecName, pKeyName, L"", pBuf, lBuf, pIniFile);
			for (pTxt = pBuf; MAjcIsBlankW(*pTxt); pTxt++);
			u.pVO = pArray;
			while (*pTxt  &&  n < nArray) {
				n++;
				switch (lArray) {
					case 1: *u.pUB++  = (UB)AjcHexToUIW (pTxt); break;
					case 2: *u.pUW++  = (UW)AjcHexToUIW (pTxt); break;
					case 4: *u.pUL++  = 	AjcHexToUIW (pTxt); break;
					case 8: *u.pULL++ = 	AjcHexToULLW(pTxt); break;
				}
				while (*pTxt != L',' && *pTxt != 0) pTxt++;
				if (*pTxt == L',') pTxt++;
				while (*pTxt == L' ' || *pTxt == L'\t') pTxt++;
			}
			rc = n;
			free(pBuf);
		}
	}

	return rc;
}
//==============================================================================================================//
//	ＩＮＩファイル読み出し（文字列）																			//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//				pKeyName	- キー名のアドレス																	//
//				pDefStr		- デフォルト値（不要時はNULL）														//
//				pBuf		- 読み出した文字列を格納するバッファのアドレス	（長さを求める場合はNULL）			//
//				lBuf		- 読み出した文字列を格納するバッファの文字数	（長さを求める場合は０）			//
//																												//
//	戻り値：	読み出したキー値の文字長																		//
//==============================================================================================================//
#define	MAX_INISTRBUF	1024
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcGetIniFileStrA(C_BCP pIniFile, C_BCP pSecName, C_BCP pKeyName, C_BCP pDefStr, BCP pBuf, UI lBuf)
{
	UI		rc = 0;

	if (pIniFile != NULL && pSecName != NULL && pKeyName != NULL) {
		if (pBuf != NULL && lBuf != 0) {
			rc = GetPrivateProfileStringA(pSecName, pKeyName, pDefStr, pBuf, lBuf		 , pIniFile);
		}
		else {
			BC	buf[MAX_INISTRBUF];
			rc = GetPrivateProfileStringA(pSecName, pKeyName, pDefStr, buf, MAX_INISTRBUF, pIniFile);
		}
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcGetIniFileStrW(C_WCP pIniFile, C_WCP pSecName, C_WCP pKeyName, C_WCP pDefStr, WCP pBuf, UI lBuf)
{
	UI		rc = 0;

	if (pIniFile != NULL && pSecName != NULL && pKeyName != NULL) {
		if (pBuf != NULL && lBuf != 0) {
			rc = GetPrivateProfileStringW(pSecName, pKeyName, pDefStr, pBuf, lBuf		 , pIniFile);
		}
		else {
			WC	buf[MAX_INISTRBUF];
			rc = GetPrivateProfileStringW(pSecName, pKeyName, pDefStr, buf, MAX_INISTRBUF, pIniFile);
		}
	}
	return rc;
}
//==============================================================================================================//
//	ＩＮＩファイル読み出し（バイナリデータ）																	//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//				pKeyName	- キー名のアドレス																	//
//				pBuf		- 読み出したバイナリデータを格納するバッファのアドレス（バイト数だけ取得時はNULL)	//
//				lBuf		- 読み出したバイナリデータを格納するバッファのバイト数								//
//																												//
//	戻り値：	読み出したデータのバイト数（エラー時は０）														//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcGetIniFileBinA(C_BCP pIniFile, C_BCP pSecName, C_BCP pKeyName, VOP pBuf, UI lBuf)
{
	return AjcGetIniFileBinExA(pIniFile, pSecName, pKeyName, pBuf, lBuf, BinDataDirA);
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcGetIniFileBinW(C_WCP pIniFile, C_WCP pSecName, C_WCP pKeyName, VOP pBuf, UI lBuf)
{
	return AjcGetIniFileBinExW(pIniFile, pSecName, pKeyName, pBuf, lBuf, BinDataDirW);
}
//==============================================================================================================//
//	ＩＮＩファイル読み出し（バイナリデータ，バイナリデータフォルダ指定）										//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//				pKeyName	- キー名のアドレス																	//
//				pBuf		- 読み出したバイナリデータを格納するバッファのアドレス（バイト数だけ取得時はNULL)	//
//				lBuf		- 読み出したバイナリデータを格納するバッファのバイト数								//
//				pBinDir		- バイナリデータフォルダ															//
//																												//
//	戻り値：	読み出したデータのバイト数（エラー時は０）														//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcGetIniFileBinExA(C_BCP pIniFile, C_BCP pSecName, C_BCP pKeyName, VOP pBuf, UI lBuf, C_BCP pBinDir)
{
	UI		rc = 0;
	UI		len;
	HANDLE	hFile;
	BC		tail[MAX_PATH];
	BC		path[MAX_PATH];

	if (pIniFile != NULL && pSecName != NULL && pKeyName != NULL) {
		if (AjcGetIniFileStrA(pIniFile, pSecName, pKeyName, "", tail, MAX_PATH) != 0) {
			if (SubProfileGenerateBinFileA(0, tail, path, pBinDir) != NULL) {
				if ((hFile = CreateFileA(path, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE) {
					if (pBuf != NULL) {
						ReadFile(hFile, pBuf, lBuf, &rc, NULL);
					}
					else {
						if ((len = GetFileSize(hFile, NULL)) != 0xFFFFFFFF) {
							rc = len;
						}
					}
					CloseHandle(hFile);
				}
			}
		}
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcGetIniFileBinExW(C_WCP pIniFile, C_WCP pSecName, C_WCP pKeyName, VOP pBuf, UI lBuf, C_WCP pBinDir)
{
	UI		rc = 0;
	UI		len;
	HANDLE	hFile;
	UW		tail[MAX_PATH];
	UW		path[MAX_PATH];

	if (pIniFile != NULL && pSecName != NULL && pKeyName != NULL) {
		if (AjcGetIniFileStrW(pIniFile, pSecName, pKeyName, L"", tail, MAX_PATH) != 0) {
			if (SubProfileGenerateBinFileW(0, tail, path, pBinDir) != NULL) {
				if ((hFile = CreateFileW(path, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE) {
					if (pBuf != NULL) {
						ReadFile(hFile, pBuf, lBuf, &rc, NULL);
					}
					else {
						if ((len = GetFileSize(hFile, NULL)) != 0xFFFFFFFF) {
							rc = len;
						}
					}
					CloseHandle(hFile);
				}
			}
		}
	}
	return rc;
}

//==============================================================================================================//
//	ＩＮＩファイル書き込み（数値）																				//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//				pKeyName	- キー名のアドレス																	//
//				Value		- 書き込み値																		//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
#define	M_GSPUTPROFILEN_A(_T1_, _T2_, _BS_, _CV_)																\
AJCEXPORT BOOL WINAPI AjcPutIniFile##_T2_##A(C_BCP pIniFile, C_BCP pSecName, C_BCP pKeyName, _T1_ Value)		\
{																												\
	BOOL	rc = FALSE;																							\
	BC		buf[_BS_];																							\
	if (pIniFile != NULL && pSecName != NULL && pKeyName != NULL) {												\
		CreateIniFileA(pIniFile);																				\
		AjcSnPrintFA(buf, _BS_, _CV_, Value);																	\
		rc = WritePrivateProfileStringA(pSecName, pKeyName, buf, pIniFile);										\
	}																											\
	return rc;																									\
}
M_GSPUTPROFILEN_A(UI	, UInt,  32, "%u"  )
M_GSPUTPROFILEN_A(SI	, SInt,  32, "%d"  )
M_GSPUTPROFILEN_A(UI	, Hex ,  32, "0x%X")
M_GSPUTPROFILEN_A(double, Real, 512, "%.20G"  )
M_GSPUTPROFILEN_A(ULL	, UI64,  64, "%I64u"  )
M_GSPUTPROFILEN_A(SLL	, SI64,  64, "%I64d"  )
M_GSPUTPROFILEN_A(ULL	, H64 ,  64, "0x%I64X")

//----- ワイド文字用 -------------------------------------------------------------------------------------------//
#define	M_GSPUTPROFILEN_W(_T1_, _T2_, _BS_, _CV_)																\
AJCEXPORT BOOL WINAPI AjcPutIniFile##_T2_##W(C_WCP pIniFile, C_WCP pSecName, C_WCP pKeyName, _T1_ Value)		\
{																												\
	BOOL	rc = FALSE;																							\
	WC		buf[_BS_];																							\
	if (pIniFile != NULL && pSecName != NULL && pKeyName != NULL) {												\
		CreateIniFileW(pIniFile);																				\
		AjcSnPrintFW(buf, _BS_, _CV_, Value);																	\
		rc = WritePrivateProfileStringW(pSecName, pKeyName, buf, pIniFile);										\
	}																											\
	return rc;																									\
}
M_GSPUTPROFILEN_W(UI	, UInt,  32, L"%u"	)
M_GSPUTPROFILEN_W(SI	, SInt,  32, L"%d"	)
M_GSPUTPROFILEN_W(UI	, Hex ,  32, L"0x%X")
M_GSPUTPROFILEN_W(double, Real, 512, L"%.20G"  )
M_GSPUTPROFILEN_W(ULL	, UI64,  64, L"%I64u"  )
M_GSPUTPROFILEN_W(SLL	, SI64,  64, L"%I64d"  )
M_GSPUTPROFILEN_W(ULL	, H64 ,  64, L"0x%I64X")

//==============================================================================================================//
//	ＩＮＩファイル書き込み（整数配列）																			//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//				pKeyName	- キー名のアドレス																	//
//				pArray		- 書き込み値が格納された整数配列のアドレス											//
//				lArray		- 整数配列の要素サイズ（１，２ or ４）												//
//				nArray		- 整数配列の要素数																	//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPutIniFileArrA(C_BCP pIniFile, C_BCP pSecName, C_BCP pKeyName, VOP pArray, UI lArray, UI nArray)
{
	BOOL	rc = FALSE;
	UI		lTmpIniFile, lTmpSecName, lTmpKeyName;
	WCP		pTmpIniFile = NULL;
	WCP		pTmpSecName	= NULL;
	WCP		pTmpKeyName	= NULL;

	if (pIniFile != NULL && pSecName != NULL && pKeyName != NULL) {
		pTmpIniFile	= AjcTAllocW(lTmpIniFile = MultiByteToWideChar(CP_ACP, 0, pIniFile, -1, NULL, 0));
		pTmpSecName	= AjcTAllocW(lTmpSecName = MultiByteToWideChar(CP_ACP, 0, pSecName, -1, NULL, 0));
		pTmpKeyName	= AjcTAllocW(lTmpKeyName = MultiByteToWideChar(CP_ACP, 0, pKeyName, -1, NULL, 0));
		if (pTmpIniFile && pTmpSecName && pTmpKeyName) {
			MultiByteToWideChar(CP_ACP, 0, pIniFile, -1, pTmpIniFile, lTmpIniFile);
			MultiByteToWideChar(CP_ACP, 0, pSecName, -1, pTmpSecName, lTmpSecName);
			MultiByteToWideChar(CP_ACP, 0, pKeyName, -1, pTmpKeyName, lTmpKeyName);
			rc = AjcPutIniFileArrW(pTmpIniFile, pTmpSecName, pTmpKeyName, pArray, lArray, nArray);
		}

		if (pTmpIniFile) AjcTFree(pTmpIniFile);
		if (pTmpSecName) AjcTFree(pTmpSecName);
		if (pTmpKeyName) AjcTFree(pTmpKeyName);
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPutIniFileArrW(C_WCP pIniFile, C_WCP pSecName, C_WCP pKeyName, VOP pArray, UI lArray, UI nArray)
{
	BOOL	rc = FALSE;
	UI		lBuf, i;
	WCP		pBuf, pTxt;
	union {
		VOP		pVO;
		UBP		pUB;
		UWP		pUW;
		ULP		pUL;
		ULLP	pULL;
	} u;

	if (pIniFile  &&  pSecName	&&	pKeyName  &&  pArray  &&  (lArray == 1 || lArray == 2 || lArray == 4 || lArray == 8)) {
		CreateIniFileW(pIniFile);
		lBuf = (((lArray * 2) + 1) * nArray) + 1;
		if (pBuf = AjcTAllocW(lBuf)) {
			pTxt = pBuf;
			u.pVO = pArray;
			for (i = 0; i < nArray; i++) {
				switch (lArray) {
					case 1: _itow  (*u.pUB++ , pTxt, 16); break;
					case 2: _itow  (*u.pUW++ , pTxt, 16); break;
					case 4: _itow  (*u.pUL++ , pTxt, 16); break;
					case 8: _i64tow(*u.pULL++, pTxt, 16); break;
				}
				while (*pTxt != 0) pTxt++;
				if (i + 1 < nArray) *pTxt++ = L',';
			}
			WritePrivateProfileStringW(pSecName, pKeyName, pBuf, pIniFile);
			free(pBuf);
			rc = TRUE;
		}
	}
	return rc;
}

//==============================================================================================================//
//	ＩＮＩファイル書き込み（文字列）																			//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//				pKeyName	- キー名のアドレス																	//
//				pStr		- 書き込む文字列のアドレス															//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPutIniFileStrA(C_BCP pIniFile, C_BCP pSecName, C_BCP pKeyName, C_BCP pStr)
{
	BOOL	rc = FALSE;

	if (pIniFile != NULL && pSecName != NULL && pKeyName != NULL && pStr != NULL) {
		CreateIniFileA(pIniFile);
		rc = WritePrivateProfileStringA(pSecName, pKeyName, pStr, pIniFile);
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPutIniFileStrW(C_WCP pIniFile, C_WCP pSecName, C_WCP pKeyName, C_WCP pStr)
{
	BOOL	rc = FALSE;

	if (pIniFile != NULL && pSecName != NULL && pKeyName != NULL && pStr != NULL) {
		CreateIniFileW(pIniFile);
		rc = WritePrivateProfileStringW(pSecName, pKeyName, pStr, pIniFile);
	}
	return rc;
}

//==============================================================================================================//
//	ＩＮＩファイル書き込み（バイナリデータ）																	//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//				pKeyName	- キー名のアドレス																	//
//				pDat		- 書き込むバイナリデータのアドレス													//
//				lDat		- 書き込むバイナリデータのバイト数													//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPutIniFileBinA(C_BCP pIniFile, C_BCP pSecName, C_BCP pKeyName, C_VOP pDat, UI lDat)
{
	return AjcPutIniFileBinExA(pIniFile, pSecName, pKeyName, pDat, lDat, BinDataDirA);
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPutIniFileBinW(C_WCP pIniFile, C_WCP pSecName, C_WCP pKeyName, C_VOP pDat, UI lDat)
{
	return AjcPutIniFileBinExW(pIniFile, pSecName, pKeyName, pDat, lDat, BinDataDirW);
}
//==============================================================================================================//
//	ＩＮＩファイル書き込み（バイナリデータ，バイナリデータフォルダ指定）										//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//				pKeyName	- キー名のアドレス																	//
//				pDat		- 書き込むバイナリデータのアドレス													//
//				lDat		- 書き込むバイナリデータのバイト数													//
//				pBinDir		- バイナリデータフォルダ															//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPutIniFileBinExA(C_BCP pIniFile, C_BCP pSecName, C_BCP pKeyName, C_VOP pDat, UI lDat, C_BCP pBinDir)
{
	BOOL	rc = FALSE;
	HANDLE	hFile;
	UL		bytes;
	BCP		pTail;
	BC		tail[MAX_PATH];
	BC		path[MAX_PATH];

	do {
		//----- NULLチェック -----------------------------------------------------------------------------------//
		if (pIniFile == NULL) break;
		if (pSecName == NULL) break;
		if (pKeyName == NULL) break;
		if (pDat	 == NULL) break;
		//----- INIファイルが無い場合、ＢＯＭで初期化 ----------------------------------------------------------//
		CreateIniFileA(pIniFile);
		//----- バイナリファイル名設定 -------------------------------------------------------------------------//
		if (AjcGetIniFileStrA(pIniFile, pSecName, pKeyName, "", tail, MAX_PATH) != 0) {				//	キーあり？
			SubProfileDeleteBinFileA(tail, pBinDir);												//		既存バイナリファイル削除
			if ((pTail = SubProfileGenerateBinFileA(FALSE, tail, path, pBinDir)) == NULL) break;	//		バイナリファイル名生成
		}
		else {																						//	キー無し？
			if ((pTail = SubProfileGenerateBinFileA(FALSE, NULL, path, pBinDir)) == NULL) break;	//		バイナリファイル名生成
			if (!AjcPutIniFileStrA(pIniFile, pSecName, pKeyName, pTail)) break;						//		キー作成
		}
		//----- バイナリファイル書き込み -----------------------------------------------------------------------//
		hFile = CreateFileA(path, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
		if (hFile != INVALID_HANDLE_VALUE) {
			WriteFile(hFile, pDat, lDat, &bytes, NULL);
			CloseHandle(hFile);
			if (bytes == lDat) {
				rc = TRUE;
			}
			else {
				DeleteFileA(path);
			}
		}
		if (rc == FALSE) {
			AjcDelIniKeyA(pIniFile, pSecName, pKeyName);
		}
	} while(0);

	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcPutIniFileBinExW(C_WCP pIniFile, C_WCP pSecName, C_WCP pKeyName, C_VOP pDat, UI lDat, C_WCP pBinDir)
{
	BOOL	rc = FALSE;
	HANDLE	hFile;
	UL		bytes;
	WCP		pTail;
	WC		tail[MAX_PATH];
	WC		path[MAX_PATH];

	do {
		//----- NULLチェック -----------------------------------------------------------------------------------//
		if (pIniFile == NULL) break;
		if (pSecName == NULL) break;
		if (pKeyName == NULL) break;
		if (pDat	 == NULL) break;
		//----- INIファイルが無い場合、ＢＯＭで初期化 ----------------------------------------------------------//
		CreateIniFileW(pIniFile);
		//----- バイナリファイル名設定 -------------------------------------------------------------------------//
		if (AjcGetIniFileStrW(pIniFile, pSecName, pKeyName, L"", tail, MAX_PATH) != 0) {			//	キーあり？
			SubProfileDeleteBinFileW(tail, pBinDir);												//		既存バイナリファイル削除
			if ((pTail = SubProfileGenerateBinFileW(FALSE, tail, path, pBinDir)) == NULL) break;	//		バイナリファイル名生成
		}
		else {																						//	キー無し？
			if ((pTail = SubProfileGenerateBinFileW(FALSE, NULL, path, pBinDir)) == NULL) break;	//		バイナリファイル名生成
			if (!AjcPutIniFileStrW(pIniFile, pSecName, pKeyName, pTail)) break;						//		キー作成
		}
		//----- バイナリファイル書き込み -----------------------------------------------------------------------//
		hFile = CreateFileW(path, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
		if (hFile != INVALID_HANDLE_VALUE) {
			WriteFile(hFile, pDat, lDat, &bytes, NULL);
			CloseHandle(hFile);
			if (bytes == lDat) {
				rc = TRUE;
			}
			else {
				DeleteFileW(path);
			}
		}
		if (rc == FALSE) {
			AjcDelIniKeyW(pIniFile, pSecName, pKeyName);
		}
	} while(0);

	return rc;
}
//==============================================================================================================//
//	ＩＮＩファイル・セクション削除																				//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI	AjcDelIniSectA	  (C_BCP pIniFile, C_BCP pSecName)
{
	return AjcDelIniSectExA(pIniFile, pSecName, BinDataDirA);
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI	AjcDelIniSectW	  (C_WCP pIniFile, C_WCP pSecName)
{
	return AjcDelIniSectExW(pIniFile, pSecName, BinDataDirW);
}
//==============================================================================================================//
//	ＩＮＩファイル・セクション削除（バイナリデータフォルダ指定）												//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//				pBinDir		- バイナリデータフォルダ															//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI	AjcDelIniSectExA  (C_BCP pIniFile, C_BCP pSecName, C_BCP pBinDir)
{
	BOOL	rc = FALSE;
	UI		len = 4096;

	if (pIniFile != NULL && pSecName != NULL) {
		//	セクション内のバイナリファイル削除
		if (rc = SubDelBinFilesA(pIniFile, pSecName, pBinDir)) {
			//	セクション削除
			rc = WritePrivateProfileStringA(pSecName, NULL, NULL, pIniFile);
		}
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI	AjcDelIniSectExW  (C_WCP pIniFile, C_WCP pSecName, C_WCP pBinDir)
{
	BOOL	rc = FALSE;
	UI		len = 8192;

	if (pIniFile != NULL && pSecName != NULL) {
		//	セクション内のバイナリファイル削除
		if (rc = SubDelBinFilesW(pIniFile, pSecName, pBinDir)) {
			//	セクション削除
			rc = WritePrivateProfileStringW(pSecName, NULL, NULL, pIniFile);
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	セクション内のバイナリファイル削除																			//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//--------------------------------------------------------------------------------------------------------------//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
static BOOL		SubDelBinFilesA(C_BCP pIniFile, C_BCP pSecName, C_BCP pBinDir)
{
	BOOL	rc = FALSE;
	UI		len = 4096;
	BCP		pTop, pCur, pNxt, pStr;

	//	全キー読み出し
	pTop = pNxt = (BCP)AJCMEM(len);
	while (pNxt != NULL && GetPrivateProfileSectionA(pSecName, pTop, len, pIniFile) == len - 2) {
		len += 4096;
		if ((pNxt = realloc(pTop, len)) != NULL) {
			pTop = pNxt;
		}
		else {
			free(pTop);
			pTop = NULL;
			break;
		}
	}
	//	セクション内のバイナリファイル削除
	if (pTop != NULL) {
		pCur = pTop;
		while (*pCur != 0) {
			pNxt = pCur + (strlen(pCur) + 1);
			if (mbstok(pCur, "=")) {
				pStr = pCur + (strlen(pCur) + 1);
				if (mbsstr(pStr, "\\Ajr")) {
					SubProfileDeleteBinFileA(pStr, pBinDir);
				}
			}
			pCur = pNxt;
		}
		free(pTop);
		rc = TRUE;
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
static	BOOL	SubDelBinFilesW(C_WCP pIniFile, C_WCP pSecName, C_WCP pBinDir)
{
	BOOL	rc = FALSE;
	UI		len = 8192;
	WCP		pTop, pCur, pNxt, pStr;

	//	全キー読み出し
	pTop = pNxt = (WCP)AJCMEM(len);
	while (pNxt != NULL && GetPrivateProfileSectionW(pSecName, pTop, (len / 2), pIniFile) == len / 2 - 2) {
		len += 8192;
		if ((pNxt = realloc(pTop, len)) != NULL) {
			pTop = pNxt;
		}
		else {
			free(pTop);
			pTop = NULL;
			break;
		}
	}
	//	セクション内のバイナリファイル削除
	if (pTop != NULL) {
		pCur = pTop;
		while (*pCur != 0) {
			pNxt = pCur + (wcslen(pCur) + 1);
			if (AjcStrTokW(pCur, L"=")) {
				pStr = pCur + (wcslen(pCur) + 1);
				if (wcsstr(pStr, L"\\Ajr")) {
					SubProfileDeleteBinFileW(pStr, pBinDir);
				}
			}
			pCur = pNxt;
		}
		free(pTop);
		rc = TRUE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	セクション内の全キー削除																					//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//--------------------------------------------------------------------------------------------------------------//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
static BOOL	SubDelAllKeysA(C_BCP pIniFile, C_BCP pSecName)
{
BOOL		rc = FALSE;
	UI		len = 4096;
	BCP		pTop, pCur, pNxt;

	//	全キー読み出し
	pTop = pNxt = (BCP)AJCMEM(len);
	while (pNxt != NULL && GetPrivateProfileSectionA(pSecName, pTop, len, pIniFile) == len - 2) {
		len += 4096;
		if ((pNxt = realloc(pTop, len)) != NULL) {
			pTop = pNxt;
		}
		else {
			free(pTop);
			pTop = NULL;
			break;
		}
	}
	//	全キー削除
	if (pTop != NULL) {
		pCur = pTop;
		while (*pCur != 0) {
			pNxt =	pCur + (strlen(pCur) + 1);
			MAjcStrTokA(pCur, "=");
			rc = WritePrivateProfileStringA(pSecName, pCur, NULL, pIniFile);
			pCur = pNxt;
		}
		free(pTop);
		rc = TRUE;
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
static	BOOL	SubDelAllKeysW(C_WCP pIniFile, C_WCP pSecName)
{
	BOOL	rc	= FALSE;
	UI		len = 8192;
	WCP		pTop, pCur, pNxt;

	//	全キー読み出し
	pTop = pNxt = (WCP)AJCMEM(len);
	while (pNxt != NULL && GetPrivateProfileSectionW(pSecName, pTop, (len / 2), pIniFile) == len / 2 - 2) {
		len += 8192;
		if ((pNxt = realloc(pTop, len)) != NULL) {
			pTop = pNxt;
		}
		else {
			free(pTop);
			pTop = NULL;
			break;
		}
	}
	//	全キー削除
	if (pTop != NULL) {
		pCur = pTop;
		while (*pCur != 0) {
			pNxt = pCur + (wcslen(pCur) + 1);
			MAjcStrTokW(pCur, L"=");
			rc = WritePrivateProfileStringW(pSecName, pCur, NULL, pIniFile);
			pCur = pNxt;
		}
		free(pTop);
		rc = TRUE;
	}
	return rc;
}


//==============================================================================================================//
//	ＩＮＩファイル・キー削除																					//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI	AjcDelIniKeyA	  (C_BCP pIniFile, C_BCP pSecName, C_BCP pKeyName)
{
	return AjcDelIniKeyExA(pIniFile, pSecName, pKeyName, BinDataDirA);
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI	AjcDelIniKeyW	  (C_WCP pIniFile, C_WCP pSecName, C_WCP pKeyName)
{
	return AjcDelIniKeyExW(pIniFile, pSecName, pKeyName, BinDataDirW);
}
//==============================================================================================================//
//	ＩＮＩファイル・キー削除（バイナリデータフォルダ指定）														//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//				pBinDir		- バイナリデータフォルダ															//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI	AjcDelIniKeyExA	  (C_BCP pIniFile, C_BCP pSecName, C_BCP pKeyName, C_BCP pBinDir)
{
	BOOL	rc = FALSE;
	BC		tail[MAX_PATH];

	if (pIniFile != NULL && pSecName != NULL && pKeyName != NULL) {
		//----- バイナリファイル削除 -----//
		AjcGetIniFileStrA(pIniFile, pSecName, pKeyName, "", tail, MAX_PATH);
		SubProfileDeleteBinFileA(tail, pBinDir);

		//----- キー削除 -----------------//
		rc = WritePrivateProfileStringA(pSecName, pKeyName, NULL, pIniFile);
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI	AjcDelIniKeyExW	  (C_WCP pIniFile, C_WCP pSecName, C_WCP pKeyName, C_WCP pBinDir)
{
	BOOL	rc = FALSE;
	UW		tail[MAX_PATH];

	if (pIniFile != NULL && pSecName != NULL && pKeyName != NULL) {
		//----- バイナリファイル削除 -----//
		AjcGetIniFileStrW(pIniFile, pSecName, pKeyName, L"", tail, MAX_PATH);
		SubProfileDeleteBinFileW(tail, pBinDir);

		//----- キー削除 -----------------//
		rc = WritePrivateProfileStringW(pSecName, pKeyName, NULL, pIniFile);
	}
	return rc;
}
//==============================================================================================================//
//	ＩＮＩファイル・セクション消去																				//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI	AjcRemoveIniSectA  (C_BCP pIniFile, C_BCP pSecName)
{
	return AjcDelIniSectExA(pIniFile, pSecName, BinDataDirA);
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI	AjcRemoveIniSectW  (C_WCP pIniFile, C_WCP pSecName)
{
	return AjcDelIniSectExW(pIniFile, pSecName, BinDataDirW);
}

//==============================================================================================================//
//	ＩＮＩファイル・セクションクリーンアップ																	//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI	AjcCleanupIniSectA (C_BCP pIniFile, C_BCP pSecName)
{
	BOOL	rc = FALSE;

	if (pIniFile != NULL && pSecName != NULL) {
		//	セクション内のバイナリファイル削除
		if (rc = SubDelBinFilesA(pIniFile, pSecName, BinDataDirA)) {
			//	全キー消去
			rc = SubDelAllKeysA(pIniFile, pSecName);
		}
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI	AjcCleanupIniSectW (C_WCP pIniFile, C_WCP pSecName)
{
	BOOL	rc = FALSE;

	if (pIniFile != NULL && pSecName != NULL) {
		//	セクション内のバイナリファイル削除
		if (rc = SubDelBinFilesW(pIniFile, pSecName, BinDataDirW)) {
			//	全キー削除
			rc = SubDelAllKeysW(pIniFile, pSecName);
		}
	}
	return rc;
}

//==============================================================================================================//
//	ＩＮＩファイル・セクション名収集																			//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				cbp			- コールバックパラメタ																//
//				cb			- セクション名を通知するコールバック関数（不要時はNULL)								//
//																												//
//	戻り値：	収集したセクション名の個数																		//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT int	 WINAPI	AjcEnumIniSectA	  (C_BCP pIniFile, UX cbp, BOOL (CALLBACK *cb)(C_BCP pSect, UX cbp))
{
	int		rc	 = 0;
	BCP		pBuf = NULL;
	BCP		p;

	if (AjcPathExistsA(pIniFile)) {
		if (pBuf = (BCP)AJCMEM(32768)) {
			GetPrivateProfileSectionNamesA(pBuf, 32768, pIniFile);
			for (p = pBuf; *p != 0; p += (strlen(p) + 1)) {
				rc++;
				if (cb != NULL) {
					if (!cb(p, cbp)) {
						break;
					}
				}
			}
			free(pBuf);
		}
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT int	 WINAPI	AjcEnumIniSectW	  (C_WCP pIniFile, UX cbp, BOOL (CALLBACK *cb)(C_WCP pSect, UX cbp))
{
	int		rc	 = 0;
	WCP		pBuf = NULL;
	WCP		p;

	if (AjcPathExistsW(pIniFile)) {
		if (pBuf = (WCP)AJCMEM(32768 * 2)) {
			GetPrivateProfileSectionNamesW(pBuf, 32768, pIniFile);
			for (p = pBuf; *p != 0; p += (wcslen(p) + 1)) {
				rc++;
				if (cb != NULL) {
					if (!cb(p, cbp)) {
						break;
					}
				}
			}
			free(pBuf);
		}
	}
	return rc;
}
//==============================================================================================================//
//	ＩＮＩファイル・キー収集																					//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//				pSecName	- セクション名のアドレス															//
//				cbp			- コールバックパラメタ																//
//				cb			- キーを通知するコールバック関数（不要時はNULL)										//
//																												//
//	戻り値：	収集したセクション名の個数																		//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT int	 WINAPI	AjcEnumIniKeyA	  (C_BCP pIniFile, C_BCP pSecName, UX cbp, BOOL (CALLBACK *cb)(C_BCP pKey, C_BCP pVal, UX cbp))
{
	int		rc	 = 0;
	BCP		pBuf = NULL;
	BCP		p, pNxt, pKey, pVal;

	if (AjcPathExistsA(pIniFile) && pSecName != NULL) {
		if (pBuf = (BCP)AJCMEM(65536)) {
			GetPrivateProfileSectionA(pSecName, pBuf, 65536, pIniFile);
			for (p = pBuf; *p != 0; p = pNxt) {
				pNxt = p + (strlen(p) + 1);
				rc++;
				if (cb != NULL) {
					pKey = mbstok(p   , "=" );
					pVal = mbstok(NULL, "\0");
					if (!cb(pKey, (pVal ? pVal : ""), cbp)) {
						break;
					}
				}
			}
			free(pBuf);
		}
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT int	 WINAPI	AjcEnumIniKeyW	  (C_WCP pIniFile, C_WCP pSecName, UX cbp, BOOL (CALLBACK *cb)(C_WCP pKey, C_WCP pVal, UX cbp))
{
	int		rc	 = 0;
	WCP		pBuf = NULL;
	WCP		p, pNxt, pKey, pVal;

	if (AjcPathExistsW(pIniFile) && pSecName != NULL) {
		if (pBuf = (WCP)AJCMEM(65536 * 2)) {
			GetPrivateProfileSectionW(pSecName, pBuf, 65536, pIniFile);
			for (p = pBuf; *p != 0; p = pNxt) {
				pNxt = p + (wcslen(p) + 1);
				rc++;
				if (cb != NULL) {
					pKey = AjcStrTokW(p   , L"=" );
					pVal = AjcStrTokW(NULL, L"\0");
					if (!cb(pKey, (pVal ? pVal : L""), cbp)) {
						break;
					}
				}
			}
			free(pBuf);
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	ＩＮＩが無い場合、ＢＯＭで初期化																			//
//																												//
//	引　数：	pIniFile	- ＩＮＩファイル名のアドレス														//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//--------------------------------------------------------------------------------------------------------------//
static	const UB u16bom[2] = {0xFF, 0xFE};
//----- バイト文字 ---------------------------------------------------------------------------------------------//
static	BOOL	CreateIniFileA(C_BCP pIniFile)
{
	BOOL	rc		= FALSE;
	HANDLE	hFile	= INVALID_HANDLE_VALUE;
	UL		bytes	= 0;

	if (!AjcPathExistsA(pIniFile)) {
		if ((hFile = CreateFileA(pIniFile,
								 GENERIC_WRITE,
								 FILE_SHARE_WRITE,
								 NULL,
								 CREATE_NEW,
								 FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE) {
			if (WriteFile(hFile, u16bom, sizeof(u16bom), &bytes, NULL) != 0 && bytes == sizeof(u16bom)) {
				rc = TRUE;
			}
			CloseHandle(hFile);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
static	BOOL	CreateIniFileW(C_WCP pIniFile)
{
	BOOL	rc		= FALSE;
	HANDLE	hFile	= INVALID_HANDLE_VALUE;
	UL		bytes	= 0;

	if (!AjcPathExistsW(pIniFile)) {
		if ((hFile = CreateFileW(pIniFile,
								 GENERIC_WRITE,
								 FILE_SHARE_WRITE,
								 NULL,
								 CREATE_NEW,
								 FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE) {
			if (WriteFile(hFile, u16bom, sizeof(u16bom), &bytes, NULL) != 0 && bytes == sizeof(u16bom)) {
				rc = TRUE;
			}
			CloseHandle(hFile);
		}
	}
	return rc;
}
//==============================================================================================================//
//	INIファイルをＵＴＦ－１６に変更																				//
//																												//
//	引　数	：	pPath	- 変換する初期化ファイルのパス															//
//				fMkBkup	- オリジナルファイルをバックアップする旨のフラグ										//
//																												//
//	戻り値	：	TRUE  - 成功（INIファイルなし，既にUNICODEファイルを含む）										//
//				FALSE - 失敗																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI	AjcIniFileToUnicodeA(C_BCP pIniFile, BOOL fMkBkup)
{
	BOOL	rc = FALSE;

	if (pIniFile != NULL) {
		UI		len;
		WCP		pTmp = NULL;
		len = MultiByteToWideChar(CP_ACP, 0, pIniFile, -1, NULL, 0);
		if (len != 0 && (pTmp = AjcTAllocW(len))) {
			MultiByteToWideChar(CP_ACP, 0, pIniFile, -1, pTmp, len);
			rc = AjcIniFileToUnicodeW(pTmp, fMkBkup);
		}
		if (pTmp != NULL) free(pTmp);
	}
	else {
		rc = AjcIniFileToUnicodeW(NULL, fMkBkup);
	}

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI	AjcIniFileToUnicodeW(C_WCP pIniFile, BOOL fMkBkup)
{
	BOOL		rc = FALSE;
	HAJCFILE	hInp = NULL;
	HAJCFILE	hOut = NULL;
	EAJCTEC		tec;
	WC			bkup[MAX_PATH] = {0};
	WC			drv[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], fext[_MAX_EXT];
	WC			buf[1024];

	//	INIファイル指定無ならば、規定のINIファイルパスとする
	if (pIniFile != NULL) {
		//	ファイルエンコード取得
		if (hInp = AjcFOpen(pIniFile, AJCTEC_AUTO)) {
			tec = AjcFGetTec(hInp);
			AjcFClose(hInp);
			//	UTF-16以外ならば変換
			if (tec != AJCTEC_UTF_16LE) {
				//	バックアップ先設定
				if (fMkBkup) {
					_wsplitpath(pIniFile, drv, dir, fname, fext);
					AjcPathCat(dir, L"bkup", AJCTSIZE(dir));
					_wmakepath (bkup, drv, dir, NULL, NULL);
					CreateDirectory(bkup, NULL);
					wcscat_s(fname, _MAX_FNAME, L"_bkup");
					_wmakepath (bkup, drv, dir, fname, fext);
				}
				else {
					GetTempPath(MAX_PATH, bkup);
					AjcPathCat(bkup, fname, MAX_PATH);
					wcscat_s  (bkup, MAX_PATH, fext );
				}
				//	オリジナルファイルをコピー
				rc = (CopyFile(pIniFile, bkup, FALSE) != 0);
				//	ＵＴＦ－１６へ変換
				if (rc) {
					//	出力ファイル（UNICODEファイル）作成
					if (hOut = AjcFCreate(pIniFile, AJCTEC_UTF_16LE, TRUE)) {
						//	入力ファイル（オリジナルファイル）オープン
						if (hInp = AjcFOpen(bkup, AJCTEC_AUTO)) {
							//	UTF-16へ変換しコピー
							while (rc && AjcFGetS(hInp, buf, AJCTSIZE(buf))) {
								rc = AjcFPutS(hOut, buf, -1);
							}
							AjcFClose(hInp);
						}
						AjcFClose(hOut);
					}
				}
			}
			else rc = TRUE;		//	初期化ファイルは既にUTF-16である！
		}
		else rc = TRUE;			//	初期化ファイル無し（プロファイルさ書き込み時にUTF-16で作成されるのでＯＫ）
	}
	return rc;
}

