﻿#include	"AjcInternal.h"

//--------------------------------------------------------------------------------------------------------------//
//	定 数																										//
//--------------------------------------------------------------------------------------------------------------//
#define		SDIC_SIZE	8192			//	スライド辞書サイズ（バイト数）										//

//--------------------------------------------------------------------------------------------------------------//
//	ＲＯＭ定数																									//
//--------------------------------------------------------------------------------------------------------------//
//----- 各月の日数テーブル -------------------------------------------------------------------------------------//
//									  1   2   3   4   5   6   7   8   9   10  11  12							//
static	const UB	 DaysTable[13] = {0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

//----- ビットストリーム抽出マスク -----------------------------------------------------------------------------//
static const UW		 BitMsk16[17] = {	0x0000,
									0x8000, 0xC000, 0xE000, 0xF000,
									0xF800, 0xFC00, 0xFE00, 0xFF00,
									0xFF80, 0xFFC0, 0xFFE0, 0xFFF0,
									0xFFF8, 0xFFFC, 0xFFFE, 0xFFFF,
};
//----- 単進符号変換テーブル -----------------------------------------------------------------------------------//
static	const UB	 NCodeTable[256] = {
//		   00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F 10 11 12 13 14 15 16 17 18 19 1A 1B 1C 1D 1E 1F
/* 00 */	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 20 */	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 40 */	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 60 */	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 80 */	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 ,1, 1, 1,
/* A0 */	1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 ,1, 1, 1,
/* C0 */	2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2 ,2, 2, 2,
/* E0 */	3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 6 ,6, 7, 8
};
//--------------------------------------------------------------------------------------------------------------//
//	ビット左シフト　マクロ																						//
//--------------------------------------------------------------------------------------------------------------//
#define		SHL(_X_, _N_)	if (sizeof(int) >= 4) {								\
								_X_.ul <<= (_N_);								\
							}													\
							else {												\
								switch ((_N_) >> 3) {							\
									case 1: _X_.ub.h  = _X_.ub.mh;				\
											_X_.ub.mh = _X_.ub.ml;				\
											_X_.ub.ml = _X_.ub.l;				\
											_X_.ub.l  = 0;			break;		\
									case 2: _X_.uw.h  = _X_.uw.l;				\
											_X_.uw.l  = 0;			break;		\
									case 3: _X_.ub.h  = _X_.ub.l;				\
											_X_.ub.mh = 0;						\
											_X_.uw.l  = 0;			break;		\
									case 4: _X_.ul	  = 0;			break;		\
								}												\
								_X_.ul <<= ((_N_) & 7);							\
							}


//--------------------------------------------------------------------------------------------------------------//
//	書庫アクセスエラーコード																					//
//--------------------------------------------------------------------------------------------------------------//
__declspec(thread) static	AJCLZHERR	LastErr = 0;	//	書庫アクセスエラーコード

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
static	UI	SubFind(HAJCLZH pW);
static	AJCLZHERR	ReadHuff(HAJCLZH pW, UBP pBuf, UI len, UIP pBytes);
static	AJCLZHERR	ReadNude(HAJCLZH pW, UBP pBuf, UI len, UIP pBytes);
static	AJCLZHERR	InitDecodeInfo(HAJCLZH pW);

static	UB			GetByte   (HAJCLZH pW);
static	VO			SetBytePos(HAJCLZH pW);
static	UI			GetNCode  (HAJCLZH pW);
static	UI			HuffDecode(HAJCLZH pW, PAJCLZHTBL pTbl);
static	UI			BltToCtb  (PAJCLZHTBL pTbl);
static	VO			SortCtb   (PAJCLZHTBL pTbl);
static	PAJCLZHCTB	SearchCtb(UI bptn, PAJCLZHTBL pTbl);

static	UW		PeekBits(HAJCLZH pW, UI n);
static	VO		SkipBits(HAJCLZH pW, UI n);
static	UW		GetBNum (HAJCLZH pW, UI n);

static	UI		ReadHeaderInfo(HAJCLZH pW);
static	VO		ReadH0		  (HAJCLZH pW, UBP s);
static	VO		ReadH1		  (HAJCLZH pW, UBP s);
static	VO		ReadH2		  (HAJCLZH pW, UBP s);
static	UI		ReadHEx 	  (HAJCLZH pW);
static	UW		GetWord 	  (HAJCLZH pW);
static	VO		GetString	  (HAJCLZH pW, UI stl, BCP pBuf, UI bfl);
static	UW		StrWord   (BCP pStr);
static	UL		StrLong   (BCP pStr);
static	VO		SplitPath(C_BCP pPath, BC pDir[MAX_PATH], BC pFile[MAX_PATH]);
static	BOOL	CmpDirName(C_BCP pDirName, C_BCP pWildDir, BOOL fSubDir);
static	BOOL	CmpWildStr(C_BCP pPath, C_BCP pWild);
static	UL		DosTimeToFileTime(UL DosTime);

//==============================================================================================================//
//	書庫内のファイル検索（初回）																				//
//																												//
//	引　数	：	pW		   - 検索用ワークバッファのアドレス														//
//				pWild	   - 検索ワイルドカード																	//
//				flag	   - 動作指定フラグ（AJCLZHFLG_XXXXX）													//
//				xp		   - コールバックパラメタ																//
//				cbRead	   - ストリーム入力用コールバック関数のアドレス											//
//				cbSeek	   - バイト位置設定用コールバック関数のアドレス											//
//																												//
//	戻り値	：	≠ＮＵＬＬ：正常（ＬＺＨデコードハンドル）														//
//				＝ＮＵＬＬ：ファイルなし／エラー																//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	HAJCLZH		WINAPI	AjcLzhFindFirstA(C_BCP pWild, UI flag, UX xp,
												UI (CALLBACK *cbRead)(UX xp, VOP pBuf, UI len),
												VO (CALLBACK *cbSeek)(UX xp, UL ofs))
{
	HAJCLZH	pW = NULL;

	LastErr = AJCLZHERR_OK;

	if (pWild != NULL && cbRead != NULL && cbSeek != NULL) {
		if (pW = (HAJCLZH)AJCMEM(sizeof(AJCLZH))) {
			memset(pW, 0, sizeof *pW);							//	インスタンスワーククリアー
			pW->mode   = AJCLZHMODE_SEARCH;						//	書庫内ファイル名検索モード
			pW->flg    = (flag & AJCLZHF_MSK_FIND);				//	動作フラグ
			pW->cbRead = cbRead;								//	コールバック関数設定
			pW->cbSeek = cbSeek;								//	・
			pW->xp	   = xp;									//	コールバックパラメタ
			//	指定ディレクトリ，ワイルドスペック設定
			SplitPath(pWild, pW->WildDir, pW->WildSpec);
			//	ファイル検索
			if ((LastErr = SubFind(pW)) != AJCLZHERR_OK) {		//	ファイル検索失敗
				free(pW);
				pW = NULL;
			}
		}
		else LastErr = 	AJCLZHERR_MEMORY;
	}
	else LastErr = AJCLZHERR_NULLPTR;

	return pW;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	HAJCLZH		WINAPI	AjcLzhFindFirstW(C_WCP pWild, UI flag, UX xp,
												UI (CALLBACK *cbRead)(UX xp, VOP pBuf, UI len),
												VO (CALLBACK *cbSeek)(UX xp, UL ofs))
{
	HAJCLZH	pW = NULL;
	UI		len;
	BCP		pTmp = NULL;

	if (pWild != NULL) {
		len = WideCharToMultiByte(CP_ACP, 0, pWild, -1, NULL, 0, NULL, NULL);
		if (len != 0 && (pTmp = (UBP)AJCMEM(len))) {
			WideCharToMultiByte(CP_ACP, 0, pWild, -1, pTmp, len, NULL, NULL);
			pW = AjcLzhFindFirstA(pTmp, flag, xp, cbRead, cbSeek);
		}
		else LastErr = 	AJCLZHERR_MEMORY;

		if (pTmp != NULL) free(pTmp);
	}
	else LastErr = AJCLZHERR_NULLPTR;

	return pW;
}
//==============================================================================================================//
//	書庫内のファイル検索（２回目以降）																			//
//																												//
//	引　数	：	pW	  - 検索用ワークバッファのアドレス															//
//																												//
//	戻り値	：	TRUE ：ＥＯＦ以外																				//
//				FALSE：ＥＯＦ／エラー																			//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcLzhFindNext (HAJCLZH pW)
{
	if (pW->mode == AJCLZHMODE_SEARCH) {		//	書庫内ファイル名検索モード？
		pW->ofs += pW->BlkSize;					//		次のヘッダ位置設定
		LastErr = SubFind(pW);					//		ファイル検索
	}
	else {										//	書庫内ファイル名検索モード以外？
		LastErr = AJCLZHERR_INVALIDMODE;		//		エラーコード設定
	}

	return (LastErr == AJCLZHERR_OK);
}
//--------------------------------------------------------------------------------------------------------------//
static	UI	SubFind(HAJCLZH pW)
{
	UI		rsu;

	LastErr = AJCLZHERR_NOTFOUND;

	SetBytePos(pW);																		//	次のヘッダ位置へ移動
	rsu = ReadHeaderInfo(pW);
	while (rsu == 0) {																	//	次のヘッダあり・の間ループ
		if (((pW->Sig == '0' || pW->Sig == '5') && !(pW->flg & AJCLZHF_NOSRH_FILE)) ||	//		ファイル情報 && 非ファイル検索指定無し or
			((pW->Sig == 'd') && (pW->flg & AJCLZHF_SRH_DIR))) {						//		ＤＩＲ情報 && ＤＩＲ検索指定あり？
			if (CmpDirName(pW->DirName, pW->WildDir, (pW->flg & AJCLZHF_SUBDIR)) &&		//			ワイルドカードのＤＩＲ部分一致 &&
				CmpWildStr(pW->PathName, pW->WildSpec)) {								//			ワイルドカード一致？
				LastErr = AJCLZHERR_OK;													//				エラーコード＝ＯＫ
				break;																	//				→終了
			}
		}
		pW->ofs += pW->BlkSize;															//			次のヘッダ位置設定
		SetBytePos(pW);																	//			次のヘッダ位置へ移動
		rsu = ReadHeaderInfo(pW);														//			ヘッダ情報読み出し
	}
	if (rsu != 0xFFFF)																	//	ＥＯＦ以外？
		LastErr = (AJCLZHERR)rsu;														//		エラーコード設定

	return LastErr;
}
//==============================================================================================================//
//	書庫内ファイルのオープン																					//
//																												//
//	引　数	：	pW		  - デコードインスタンスワークのアドレス												//
//				pFilePath - 書庫内のファイルパス名																//
//				flag	  - 動作指定フラグ（AJCLZHFLG_XXXXX）													//
//				xp		  - コールバックパラメタ																//
//				cbRead	  - ストリーム入力用コールバック関数のアドレス											//
//				cbSeek	  - バイト位置設定用コールバック関数のアドレス											//
//																												//
//	戻り値	：	≠ＮＵＬＬ：ＬＺＨデコードハンドル																//
//				＝ＮＵＬＬ：エラー																				//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	HAJCLZH		WINAPI	AjcLzhOpenA(C_BCP pFilePath, UI flag, UX xp,
											 UI (CALLBACK *cbRead)(UX xp, VOP pBuf, UI len),
											 VO (CALLBACK *cbSeek)(UX xp, UL ofs))
{
	HAJCLZH		pW;

	LastErr = AJCLZHERR_OK;

	if (pFilePath != NULL && cbRead != NULL && cbSeek != NULL) {
		if (pW = (HAJCLZH)AJCMEM(sizeof(AJCLZH))) {
			//	インスタンス初期化
			memset(pW, 0, sizeof *pW);											//	デコードインスタンスクリアー
			pW->mode = AJCLZHMODE_READ;											//	書庫内ファイル読み出しモード
			pW->flg    = (flag & AJCLZHF_MSK_OPEN);								//	動作フラグ
			pW->cbRead = cbRead;												//	ストリーム入力用コールバック
			pW->cbSeek = cbSeek;												//	バイト位置設定用コールバック
			pW->xp	   = xp;													//	コールバックパラメタ

			//	バッファアドレス初期化
			pW->pDicTop		= pW->pDic = (UBP		)calloc(SDIC_SIZE	, 1);	//	スライド辞書
			pW->BLen.pBlt	= (UBP		 )calloc(					  19, 1);	//	ビット長テーブル
			pW->BLen.pCtb	= (PAJCLZHCTB)calloc(sizeof(AJCLZHCTB) *  19, 1);	//	・
			pW->Deco.pBlt	= (UBP		 )calloc(					 510, 1);	//	デコードテーブル
			pW->Deco.pCtb	= (PAJCLZHCTB)calloc(sizeof(AJCLZHCTB) * 510, 1);	//	・
			pW->Dist.pBlt	= (UBP		 )calloc(					  15, 1);	//	ディスタンステーブル
			pW->Dist.pCtb	= (PAJCLZHCTB)calloc(sizeof(AJCLZHCTB) *  15, 1);	//	・

			if (pW->pDicTop && pW->BLen.pBlt && pW->BLen.pCtb && pW->Deco.pBlt && pW->Deco.pCtb && pW->Dist.pBlt && pW->Dist.pCtb) {
				//	指定ディレクトリ，ワイルドカード設定
				SplitPath(pFilePath, pW->WildDir, pW->WildSpec);
				//	ファイル検索
				if ((LastErr = SubFind(pW)) != AJCLZHERR_OK) {
					AjcLzhClose(pW); pW = NULL;
				}
			}
			else {
				AjcLzhClose(pW); pW = NULL;
				LastErr = AJCLZHERR_MEMORY;
			}
		}
		else LastErr = AJCLZHERR_MEMORY;
	}
	else LastErr = AJCLZHERR_NULLPTR;

	return pW;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	HAJCLZH		WINAPI	AjcLzhOpenW(C_WCP pFilePath, UI flag, UX xp,
											 UI (CALLBACK *cbRead)(UX xp, VOP pBuf, UI len),
											 VO (CALLBACK *cbSeek)(UX xp, UL ofs))
{
	HAJCLZH	pW = NULL;
	UI		len;
	BCP		pTmp = NULL;

	if (pFilePath != NULL) {
		len = WideCharToMultiByte(CP_ACP, 0, pFilePath, -1, NULL, 0, NULL, NULL);
		if (len != 0 && (pTmp = (BCP)AJCMEM(len))) {
			WideCharToMultiByte(CP_ACP, 0, pFilePath, -1, pTmp, len, NULL, NULL);
			pW = AjcLzhOpenA(pTmp, flag, xp, cbRead, cbSeek);
		}
		else LastErr = 	AJCLZHERR_MEMORY;

		if (pTmp != NULL) free(pTmp);
	}
	else LastErr = AJCLZHERR_NULLPTR;

	return pW;
}
//==============================================================================================================//
//	書庫内ファイルの読み出し（デコード）																		//
//																												//
//	引　数	：	pBuf   - 読み出した（解凍した）データを格納するバッファのアドレス								//
//				len    - 読み出す（解凍する）データのバイト数													//
//																												//
//	戻り値	：	実際に読み出したバイト数（０：ＥＯＦ／エラー）													//
//==============================================================================================================//

#define	LR_SET_DIC(_X_)		*pW->pDic++ = (UB)(_X_);		\
							if (++pW->xDic >= SDIC_SIZE) {	\
								pW->pDic  = pW->pDicTop;	\
								pW->xDic = 0;				\
							}

#define	LR_SET_BUF(_X_)		*pBuf++ = (UB)_X_;

//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcLzhRead		(HAJCLZH pW, UBP pBuf, UI len)
{
	UI		rc;
	UI		bytes = 0;

	if (pBuf != NULL) {
		if (pW->mode == AJCLZHMODE_READ) {

			if (pW->Sig == '5') LastErr = ReadHuff(pW, pBuf, len, &bytes);	//	"-lh5-"
			else				LastErr = ReadNude(pW, pBuf, len, &bytes);	//	"-lh0-"

			if (LastErr == AJCLZHERR_OK) {
				rc = bytes;
			}
			else {
				rc = 0;
			}
		}
		else {
			LastErr = AJCLZHERR_INVALIDMODE;
			rc = 0;
		}
	}
	else {
		LastErr = AJCLZHERR_NULLPTR;
		rc = 0;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	シグネチャ＝"-lh5-" : 圧縮コードの復元・読み出し															//
//--------------------------------------------------------------------------------------------------------------//
static	AJCLZHERR	ReadHuff(HAJCLZH pW, UBP pBuf, UI len, UIP pBytes)
{
	AJCLZHERR	rc	= AJCLZHERR_OK;
	UI			cnt = 0;
	UI			bl, code, dist;

	while (cnt < len && pW->FileCnt < pW->FileSize) {					//	読出ｻｲｽﾞ満了／ﾌｧｲﾙｻｲｽﾞｵｰﾊﾞｰまでﾙｰﾌﾟ
		if (pW->MatchLen == 0) {										//		一致文字列処理中以外？
			if (pW->BlkCodes == 0) {									//			残圧縮コードなし？
				rc = InitDecodeInfo(pW);								//				デコード初期化
				if (rc != 0) break;										//				エラーならば終了
			}
			if (pW->Deco.nCtb == 1) {									//			単一パターン？
				code = pW->Deco.pCtb[0].code;							//				コード設定
			}
			else {														//			複数パターンあり？
				code = HuffDecode(pW, &pW->Deco);						//				デコード
			}
			if (code < 256) {											//			単一文字？
				LR_SET_BUF(code)										//				バッファに文字設定
				LR_SET_DIC(code)										//				ｽﾗｲﾄﾞ辞書に文字設定	
				cnt++;													//				読出ｻｲｽﾞ更新
				pW->FileCnt++;											//				ファイルサイズ更新
			}
			else {														//			一致文字列？
				pW->MatchLen = code - 256 + 3;							//				一致文字列長算出
				bl = HuffDecode(pW, &pW->Dist);							//				ビット長デコード
				if (bl == 0) {											//				ビット長＝０？
					dist = 1;											//					距離＝１
				}
				else {													//				ビット長/≠０？
					dist = GetBNum(pW, bl - 1) + (1 << (bl - 1)) + 1;	//					距離算出
				}
				if (dist <= pW->xDic) {									//				一致文字列のアドレス設定
					pW->pMatch = pW->pDic - dist;						//				・
					pW->xMatch = pW->xDic - dist;						//				・
				}														//				・
				else {													//				・
					pW->xMatch = ((pW->xDic - dist) & (SDIC_SIZE - 1));	//				・
					pW->pMatch = pW->pDicTop + pW->xMatch;				//				・
				}
			}
			pW->BlkCodes--;												//			圧縮コード数減算
		}
		else {															//		一致文字列処理中？
			code = *pW->pMatch++;										//			ｽﾗｲﾄﾞ辞書から文字取得
			if (++pW->xMatch >= SDIC_SIZE) {							//			ｽﾗｲﾄﾞ辞書アドレス更新
				pW->pMatch = pW->pDicTop;								//			・
				pW->xMatch = 0;											//			・
			}
			LR_SET_BUF(code)											//			バッファに文字設定
			LR_SET_DIC(code)											//			ｽﾗｲﾄﾞ辞書に文字設定
			cnt++;														//			読出サイズ更新
			pW->FileCnt++;												//			ファイルサイズ更新
			pW->MatchLen--;												//			一致文字列長減算
		}
	}
	*pBytes = cnt;														//	戻り情報(実際に読み出したバイト数）

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	シグネチャ＝"-lh0-" : 非圧縮コードの読み出し																//
//--------------------------------------------------------------------------------------------------------------//
static	AJCLZHERR	ReadNude(HAJCLZH pW, UBP pBuf, UI len, UIP pBytes)
{
	UI	cnt = 0;

	while (cnt < len && pW->FileCnt < pW->FileSize) {		//	読出ｻｲｽﾞ満了／ﾌｧｲﾙｻｲｽﾞｵｰﾊﾞｰまでﾙｰﾌﾟ
		*pBuf++ = (UB)GetByte(pW);							//		バッファにファイルデータ文字設定
		cnt++;												//		読み出しサイズ更新
		pW->FileCnt++;										//		ファイルサイズ更新
	}
	*pBytes = cnt;											//	戻り情報(実際に読み出したバイト数）

	return AJCLZHERR_OK;
}
//==============================================================================================================//
//	書庫内ファイルのクローズ																					//
//																												//
//	引　数	：	pW	   - ＬＺＨデコードハンドル																	//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	VO		WINAPI	AjcLzhClose	(HAJCLZH pW)
{
	LastErr = AJCLZHERR_OK;

	if (pW != NULL) {
		//	バッファ解放
		if (pW->pDicTop		!= NULL) free(pW->pDicTop	);		//	スライド辞書
		if (pW->BLen.pBlt	!= NULL) free(pW->BLen.pBlt	);		//	ビット長テーブル
		if (pW->BLen.pCtb	!= NULL) free(pW->BLen.pCtb	);		//	・
		if (pW->Deco.pBlt	!= NULL) free(pW->Deco.pBlt	);		//	デコードテーブル
		if (pW->Deco.pCtb	!= NULL) free(pW->Deco.pCtb	);		//	・
		if (pW->Dist.pBlt	!= NULL) free(pW->Dist.pBlt	);		//	ディスタンステーブル
		if (pW->Dist.pCtb	!= NULL) free(pW->Dist.pCtb	);		//	・
		free(pW);
	}
}
//==============================================================================================================//
//	ファイル情報の取得																							//
//																												//
//	引　数	：	pW	   - ＬＺＨデコードハンドル																	//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcLzhGetFileInfoA(HAJCLZH pW, PAJCLZHFILEINFOA pInfo)
{
	BOOL	rc;

	if (pW->mode == AJCLZHMODE_SEARCH  ||  pW->mode == AJCLZHMODE_READ) {
		pInfo->HuffSize		= pW->HuffSize;			//	圧縮データサイズ
		pInfo->FileSize		= pW->FileSize;			//	ファイルサイズ
		pInfo->FileTime		= pW->FileTime;			//	ファイルタイム（1970-01-01, 00:00:00からの通算秒数）
		pInfo->fLocalTime	= pW->fLocalTime;		//	TRUE:ローカルタイム．FALSE:ＵＴＣ
		pInfo->crc			= pW->crc;				//	ＣＲＣ
		pInfo->dmy			= 0;					//	Filler
		AjcSnPrintFA(pInfo->szPathName, MAX_PATH, "%s", pW->PathName); // ファイルパス名(ex. "temp\sample.txt")
		AjcSnPrintFA(pInfo->szDirName , MAX_PATH, "%s", pW->DirName ); // ディレクトリ名(ex. "temp\"          )
		AjcSnPrintFA(pInfo->szFileName, MAX_PATH, "%s", pW->FileName); // ファイル名	(ex. "sample"		  )
		pInfo->fDir			= (pW->Sig == 'd');
		LastErr = AJCLZHERR_OK;
		rc = TRUE;
	}
	else {
		LastErr = AJCLZHERR_INVALIDMODE;
		rc = FALSE;
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcLzhGetFileInfoW(HAJCLZH pW, PAJCLZHFILEINFOW pInfo)
{
	BOOL	rc;

	if (pW->mode == AJCLZHMODE_SEARCH  ||  pW->mode == AJCLZHMODE_READ) {
		pInfo->HuffSize		= pW->HuffSize;			//	圧縮データサイズ
		pInfo->FileSize		= pW->FileSize;			//	ファイルサイズ
		pInfo->FileTime		= pW->FileTime;			//	ファイルタイム（1970-01-01, 00:00:00からの通算秒数）
		pInfo->fLocalTime	= pW->fLocalTime;		//	TRUE:ローカルタイム．FALSE:ＵＴＣ
		pInfo->crc			= pW->crc;				//	ＣＲＣ
		pInfo->dmy			= 0;					//	Filler
		MultiByteToWideChar(CP_ACP, 0, pW->DirName , -1, pInfo->szDirName , MAX_PATH);	//	ファイルパス名
		MultiByteToWideChar(CP_ACP, 0, pW->PathName, -1, pInfo->szPathName, MAX_PATH);	//	ディレクトリ名
		MultiByteToWideChar(CP_ACP, 0, pW->FileName, -1, pInfo->szFileName, MAX_PATH);	//	ファイル名
		pInfo->fDir			= (pW->Sig == 'd');
		LastErr = AJCLZHERR_OK;
		rc = TRUE;
	}
	else {
		LastErr = AJCLZHERR_INVALIDMODE;
		rc = FALSE;
	}
	return rc;
}
//==============================================================================================================//
//	エラーコード取得																							//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	最後のファンクション実行時のエラーコード														//
//==============================================================================================================//
AJCEXPORT	AJCLZHERR	WINAPI	AjcLzhGetLastError(VO)
{
	return LastErr;
}
//--------------------------------------------------------------------------------------------------------------//
//	デコード準備（コード表の入力）																				//
//																												//
//	引　数	：	pW	- インスタンスのアドレス																	//
//																												//
//	戻り値	：	＝０ - ＯＫ																						//
//				≠０ - エラー																					//
//--------------------------------------------------------------------------------------------------------------//
static	AJCLZHERR	InitDecodeInfo(HAJCLZH pW)
{
	AJCLZHERR	rc = AJCLZHERR_OK;
	UI			i, bl, z, n, code;
	PAJCLZHTBL	pTbl;

	//----- ブロックサイズ入力 ---------------------------------------------------------------------------------//
	pW->BlkCodes = GetBNum(pW, 16);

	//----- ビット長のビット長テーブル入力 ---------------------------------------------------------------------//
	pTbl = &pW->BLen;												//	テーブル情報アドレス設定
	pTbl->nBlt	= GetBNum(pW, 5);									//	テーブルエントリ数リード
	if (pTbl->nBlt > 19) {rc = AJCLZHERR_BLEN_OVER19; goto idi_ret;}//	２０以上ならばエラー
	if (pTbl->nBlt == 0) {											//	エントリ数＝０？
		i = GetBNum(pW, 5);											//		雄一のコード値リード
		if (i > 19) {rc = AJCLZHERR_BLEN_OVER19; goto idi_ret;}		//		２０以上ならばエラー
		memset(pTbl->pBlt, 0, i);									//		テーブルクリアー
		pTbl->pBlt[i] = 1;											//		雄一のコードに１ビット割り当て
		pTbl->MaxBl   = 1;											//		最大ビット長＝１
		pTbl->nBlt	  = i + 1;										//		有効テーブルエントリ数設定
	}
	else {															//	エントリ数≠０？
		pTbl->MaxBl = 0;											//		最大ビット長クリアー
		for (i=0; i < pTbl->nBlt;) {								//		全エントリループ
			bl = GetNCode(pW);										//			単身符号取得
			if (bl > 16) {rc = bl; goto idi_ret;}					//			１７以上ならばエラー
			pTbl->pBlt[i++] = (UB)bl;								//			ビット長テーブルエントリ設定
			if (bl > pTbl->MaxBl) pTbl->MaxBl = bl;					//			最大ビット長更新
			if (i == 3) {											//			３～６は特別に０の個数
				z = GetBNum(pW, 2);									//			・
				while (z--) pTbl->pBlt[i++] = 0;					//			・
			}
		}
	}
	//----- ビット長テーブル生成用コード表作成 -----------------------------------------------------------------//
	pTbl->nCtb = BltToCtb(pTbl);
	if (pTbl->nCtb > 19) {rc = pTbl->nCtb; goto idi_ret;}
	SortCtb(pTbl);

	//----- ビット長テーブル入力 -------------------------------------------------------------------------------//
	pTbl = &pW->Deco;																//	テーブル情報アドレス設定
	pTbl->MaxBl = 0;																//	最大ビット長クリアー
	pTbl->nBlt	= GetBNum(pW, 9);													//	エントリ数リード
	if (pTbl->nBlt > 510) {rc = AJCLZHERR_BLEN_OVER510; goto idi_ret;}				//	５１１以上ならばエラー
	if (pTbl->nBlt == 0) {															//	エントリ数＝０？
		i = GetBNum(pW, 9);															//		雄一のコード値リード
		if (i > 510) {rc = AJCLZHERR_BLEN_OVER510; goto idi_ret;}					//		５１１以上ならばエラー
		memset(pTbl->pBlt, 0, i);													//		テーブルクリアー
		pTbl->pBlt[i] = 1;															//		雄一のｺｰﾄﾞに１ﾋﾞｯﾄ割当て
		pTbl->MaxBl   = 1;															//		最大ビット長＝１
		pTbl->nBlt	  = i + 1;														//		有効ﾃｰﾌﾞﾘｴﾝﾄﾘ数設定
	}
	else {																			//	エントリ数≠０？
		pTbl->MaxBl = 0;															//		最大ﾋﾞｯﾄ長クリアー
		for (i=0; i<pTbl->nBlt;) {													//		全エントリループ
			switch (code = HuffDecode(pW, &pW->BLen)) {
				case 0:																//			０：当該エントリ＝０
					pTbl->pBlt[i++] = 0;
					break;
				case 1:																//			１：４ビット数＋３
					n = GetBNum(pW, 4) + 3;											//			　　エントリを０
					if ((i + n) > 510) {rc = AJCLZHERR_BLEN_OVER510; goto idi_ret;}
					memset(&pTbl->pBlt[i], 0, n);
					i += n;	
					break;
				case 2:																//			２：9ﾋﾞｯﾄ数+20
					n = GetBNum(pW, 9) + 20;										//			　　エントリを０
					if ((i + n) > 510) {rc = AJCLZHERR_BLEN_OVER510; goto idi_ret;}
					memset(&pTbl->pBlt[i], 0, n);
					i += n;
					break;
				default:															//			その他：
					if ((code - 2) > pTbl->MaxBl) pTbl->MaxBl = (code - 2);			//			　　ｺｰﾄﾞ値設定
					if (pTbl->MaxBl == 2308) {
						int x = 10;
					}
					pTbl->pBlt[i++] = (UB)(code - 2);
					break;
			}
		}
	}
	//----- コード生成用コード表作成 ---------------------------------------------------------------------------//
	pTbl->nCtb = BltToCtb(pTbl);
	if (pTbl->nCtb > 510) {rc = pTbl->nCtb; goto idi_ret;}
	SortCtb(pTbl);

	//----- ディスタンス・ビット長テーブル入力 -----------------------------------------------------------------//
	pTbl = &pW->Dist;												//	テーブル情報アドレス設定
	pTbl->nBlt	= GetBNum(pW, 4);									//	テーブルエントリ数リード
	if (pTbl->nBlt == 0) {											//	エントリ数＝０？
		i = GetBNum(pW, 4);											//		雄一のコード値リード
		memset(pTbl->pBlt, 0, i);									//		テーブルクリアー
		pTbl->pBlt[i] = 1;											//		雄一のコードに１ビット割り当て
		pTbl->MaxBl   = 1;											//		最大ビット長＝１
		pTbl->nBlt	  = i + 1;										//		有効テーブルエントリ数設定
	}
	else {															//	エントリ数≠０？
		pTbl->MaxBl = 0;											//		最大ﾋﾞｯﾄ長クリアー
		for (i=0; i < pTbl->nBlt;) {								//		全エントリループ
			bl = GetNCode(pW);										//			単身符号取得
			if (bl > 16) {rc = bl; goto idi_ret;}					//			１７以上ならばエラー
			pTbl->pBlt[i++] = (UB)bl;								//			ビット長テーブルエントリ設定
			if (bl > pTbl->MaxBl) pTbl->MaxBl = bl;					//			最大ビット長更新
		}
	}
	//----- ディスタンス・コード表作成 -------------------------------------------------------------------------//
	pTbl->nCtb = BltToCtb(pTbl);
	if (pTbl->nCtb > 510) {rc = pTbl->nCtb; goto idi_ret;}
	SortCtb(pTbl);

idi_ret:;

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	書庫から１バイト入力																						//
//																												//
//	引　数	：	pW	- インスタンスワークのアドレス																//
//																												//
//	戻り値	：	入力した値																						//
//--------------------------------------------------------------------------------------------------------------//
static	UB			GetByte   (HAJCLZH pW)
{
	if (pW->dl <= pW->ix) {
		if ((pW->dl = pW->cbRead(pW->xp, pW->RBuf, AJCLZHINPBUFSIZE)) == 0) {
			pW->flg |= AJCLZHF_EOF;
		}
		pW->ix = 0;
	}
	return *(pW->RBuf + pW->ix++);
}
//--------------------------------------------------------------------------------------------------------------//
//	書庫の入力バイト位置設定																					//
//																												//
//	引　数	：	pW	- インスタンスワークのアドレス																//
//																												//
//	戻り値	：	入力した値																						//
//--------------------------------------------------------------------------------------------------------------//
static	VO			SetBytePos(HAJCLZH pW)
{
	pW->cbSeek(pW->xp, pW->ofs);
	pW->dl = pW->ix = 0;
	pW->flg &= ~AJCLZHF_EOF;
}
//--------------------------------------------------------------------------------------------------------------//
//	単進符号取得																								//
//																												//
//	引　数	：	pW	- デコードインスタンスのアドレス															//
//																												//
//	戻り値	：	０～１６ - ビットストリームが表す単進符号値														//
//				その他	 - エラー																				//
//--------------------------------------------------------------------------------------------------------------//
static	UI	GetNCode(HAJCLZH pW)
{
	UI				rc, n;
	AJCLZHUNIWORD	ex;

	rc = GetBNum(pW, 3);
	if (rc == 7) {
		ex.uw = PeekBits(pW, 10);

		if (ex.ub.h == 0xFF)
			n = (8 + NCodeTable[ex.ub.l]);
		else
			n = NCodeTable[ex.ub.h];

		rc += n;
		SkipBits(pW, n + 1);

		if (rc > 16) {
			rc = AJCLZHERR_UNARYCODE_OUTOFRANGE;
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	ハフマン・デコード																							//
//																												//
//	引　数	：	pW	- デコードインスタンスのアドレス															//
//																												//
//	戻り値	：	ビットストリーム値（ＬＳＢ側のｎビットに設定）													//
//--------------------------------------------------------------------------------------------------------------//
static	UI	HuffDecode(HAJCLZH pW, PAJCLZHTBL pTbl)
{
	UI			bptn = PeekBits(pW, pTbl->MaxBl);
	PAJCLZHCTB	pCtb;

	pCtb = SearchCtb(bptn, pTbl);
	SkipBits(pW, pCtb->bl);

	return pCtb->code;
}
//--------------------------------------------------------------------------------------------------------------//
//	ビット長テーブルから、コード表を生成する																	//
//																												//
//	引　数	：	pBlt - ビット長テーブルのアドレス																//
//				max  - 最大ビット長																				//
//				pTbl - コード表のアドレス																		//
//																												//
//	戻り値	：	1～510 - コード表の有効エントリ数																//
//				その他 - エラー																					//
//--------------------------------------------------------------------------------------------------------------//
static	UI		BltToCtb(PAJCLZHTBL pTbl)
{
	UW			frq[17], wgt[17], spt[17];
	UW			ptn, w;
	UI			i, j, bl;
	UI			MaxBl = pTbl->MaxBl;
	UBP			pBlt = pTbl->pBlt;
	PAJCLZHCTB	pCtb = pTbl->pCtb;

	memset(frq, 0, sizeof frq);

	//----- ビット長の頻度を算出 ----------//
	for (i=0; i < pTbl->nBlt; i++) {
		bl = pBlt[i];
		if (bl <= 16) {
			if (bl != 0) frq[bl]++;
		}
		else {
			return AJCLZHERR_ILLEGAL_BITLEN;
		}
	}
	//----- コード表作成 ------------------//
	ptn = 0; w = (1 << (MaxBl - 1));
	for (i = 1; i <= MaxBl; i++) {
		spt[i] = ptn;
		wgt[i] = w;
		ptn += (w * frq[i]);
		w >>= 1;
	}
	for (i=j=0; i<pTbl->nBlt; i++) {
		bl = pBlt[i];
		if (bl != 0) {
			ptn = spt[bl];
			pCtb[j].bl	 = bl;
			pCtb[j].bptn = (ptn << (16 - (MaxBl)));
			pCtb[j].code = i;
			j++;
			spt[bl] += wgt[bl];
		}
	}
	pTbl->nCtb = j;

	return j;
}
//--------------------------------------------------------------------------------------------------------------//
//	コード表ソート																								//
//																												//
//	引　数	：	pTbl - コード表のアドレス																		//
//				n	 - コード表のエントリ数																		//
//																												//
//	戻り値	：	なし																							//
//--------------------------------------------------------------------------------------------------------------//
static	VO	SortCtb(PAJCLZHTBL pTbl)
{
	UI			i, j, k;
	UI			nCtb = pTbl->nCtb;
	PAJCLZHCTB	pCtb = pTbl->pCtb;
	AJCLZHCTB	x;

	if (nCtb >= 2) {
		nCtb--;
		k = nCtb / 2;
		do {
			i = k;
			memcpy(&x, pCtb + i, sizeof(AJCLZHCTB));
			while ((j = 2 * i + 1) <= nCtb) {
				if (j < nCtb && pCtb[j].bptn < pCtb[j+1].bptn) j++;
				if (x.bptn >= pCtb[j].bptn) break;
				memcpy(pCtb + i, pCtb + j, sizeof(AJCLZHCTB));
				i = j;
			}
			memcpy(pCtb + i, &x, sizeof(AJCLZHCTB));
		} while (k-- != 0);

		while (nCtb > 0) {
			memcpy(&x, pCtb + nCtb	, sizeof(AJCLZHCTB));
			memcpy(pCtb + nCtb, pCtb, sizeof(AJCLZHCTB));
			nCtb--;
			i = 0;
			while ((j = 2 * i + 1) <= nCtb) {
				if (j < nCtb && pCtb[j].bptn < pCtb[j+1].bptn) j++;
				if (x.bptn >= pCtb[j].bptn) break;
				memcpy(pCtb + i, pCtb + j, sizeof(AJCLZHCTB));
				i = j;
			}
			memcpy(pCtb + i, &x, sizeof(AJCLZHCTB));
		}
	}
}
//--------------------------------------------------------------------------------------------------------------//
//	コード表検索																								//
//																												//
//	引　数	：	bptn - 検索するビットパターン																	//
//				pTbl - コードのアドレス																			//
//				n	 - コード表のエントリ数																		//
//																												//
//	戻り値	：	当該テーブルエントリのアドレス																	//
//--------------------------------------------------------------------------------------------------------------//
static	PAJCLZHCTB	SearchCtb(UI bptn, PAJCLZHTBL pTbl)
{
	PAJCLZHCTB	pCtb  = pTbl->pCtb;
	UI			l = 0;
	UI			r = pTbl->nCtb;
	UI			m, sft;
	UW			msk;

	while (l < r) {
		m = ((l + r) >> 1);
		if (pCtb[m].bptn < bptn) l = m + 1;
		else					 r = m;
	}
	if		(l == pTbl->nCtb) --l;
	else if (l != 0) {
		msk = BitMsk16[pCtb[l-1].bl];
		sft = pCtb[l].bl - pCtb[l-1].bl;
		if ((((UW)bptn & msk) >> sft) == ((pCtb[l-1].bptn & msk) >> sft)) l--;
	}
	return &pCtb[l];
}
//--------------------------------------------------------------------------------------------------------------//
//	ビットストリーム参照（ビットストリームの先頭ｎビットを覗く）												//
//																												//
//	引　数	：	pW	- デコードインスタンスのアドレス															//
//				n	- ビット数（０～１６）																		//
//																												//
//	戻り値	：	ビットストリーム値（ＭＳＢ側のｎビットに設定）													//
//--------------------------------------------------------------------------------------------------------------//
static	UW	PeekBits(HAJCLZH pW, UI n)
{
	AJCLZHUNIWORD	c;

	for (; pW->rem < n; pW->rem += 8) {						//		指定ビット数以上となるまで
		c.uw = GetByte(pW);									//		書庫ストリーム読み出し
		c.uw <<= (8 - (pW->rem & 7));						//		・
		switch (pW->rem >> 3) {								//		・
			case 0:	pW->u.uw.h	|= c.uw;	break;			//		・
			case 1:	pW->u.ub.mh |= c.ub.h;					//		・
					pW->u.ub.ml |= c.ub.l;	break;			//		・
			case 2: pW->u.uw.l	|= c.uw;	break;			//		・
	}	}
	return (UW)(pW->u.uw.h & BitMsk16[n]);					//		戻り値＝左詰めのビットパターン
}

//--------------------------------------------------------------------------------------------------------------//
//	ビットストリーム破棄（先頭のｎビットを破棄する）															//
//																												//
//	引　数	：	pW	- デコードインスタンスのアドレス															//
//				n	- ビット数（０～２４）																		//
//																												//
//	戻り値	：	なし																							//
//--------------------------------------------------------------------------------------------------------------//
static	VO	SkipBits(HAJCLZH pW, UI n)
{
	SHL(pW->u, n);					//	ＭＳＢ側を破棄															//
	pW->rem -= n;					//	有効ビット数減算														//
}

//--------------------------------------------------------------------------------------------------------------//
//	ビットストリーム取得（ビットストリームが表す数値（右詰のビットバターン）取得）								//
//																												//
//	引　数	：	pW	- デコードインスタンスのアドレス															//
//				n	- ビット数（０～２４）																		//
//																												//
//	戻り値	：	ビットストリーム値（ＬＳＢ側のｎビットに設定）													//
//--------------------------------------------------------------------------------------------------------------//
static	UW	GetBNum(HAJCLZH pW, UI n)
{
	AJCLZHUNIWORD	u;
	UI		sft = 16 - n;

	u.uw = PeekBits(pW, n);
	if (sft >= 8) {
		u.ub.l = u.ub.h;
		u.ub.h = 0;
		sft -= 8;
	}
	u.uw >>= sft;

	SHL(pW->u, n);					//	ＭＳＢ側を破棄	
	pW->rem -= n;					//	有効ビット数減算

	return u.uw;
}
//--------------------------------------------------------------------------------------------------------------//
//	ヘッダ情報読み出し																							//
//																												//
//	引　数	：	pW	- インスタンスヘッダのアドレス																//
//																												//
//	戻り値	：	＝０  ：正常																					//
//				0xFFFF：EOF																						//
//				その他：エラー																					//
//--------------------------------------------------------------------------------------------------------------//
static	UI	ReadHeaderInfo(HAJCLZH pW)
{
	UI		i, rc = 0;
	UB		s[21];

	if (!(pW->flg & AJCLZHF_EOF)) {
		for (i=0; i<21; i++) { 										//	先頭２１バイト読み出し
			s[i] = GetByte(pW);
			if (pW->flg & AJCLZHF_EOF) break;
		}
		if (!(pW->flg & AJCLZHF_EOF)) {
			if (memcmp(&s[2], "-lh0-", 5) == 0 ||					//	lh0 or lh5 or lhd ？
				memcmp(&s[2], "-lh5-", 5) == 0 ||					//	・
				memcmp(&s[2], "-lhd-", 5) == 0) {					//	・
				pW->Sig = s[5];										//	シグネチャ('0' / '5' / 'd')設定
				pW->Lvl = s[20];									//		レベル(0 / 1 / 2)設定
				switch (s[20]) {
					case 0:	ReadH0(pW, s);					break;	//		レベル０ヘッダ読み出し
					case 1:	ReadH1(pW, s);					break;	//		レベル１ヘッダ読み出し
					case 2:	ReadH2(pW, s);					break;	//		レベル２ヘッダ読み出し
					default: rc = AJCLZHERR_ILLEGAL_HDLVL;	break;	//		レベル番号エラー
				}
			}
			else {													//	シグネチャ不正？
				rc = AJCLZHERR_ILLEGAL_FORMAT;						//		エラー
			}
		}
		else {
			rc = 0xFFFF;
		}
	}
	else {
		rc = 0xFFFF;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	ＬＨＡレベル０ヘッダ情報読み出し																			//
//																												//
//	引　数	：	pW	- インスタンスヘッダのアドレス																//
//				s	- ヘッダの先頭部分（２１バイト）のアドレス													//
//																												//
//	戻り値	：	なし																							//
//--------------------------------------------------------------------------------------------------------------//
static	VO	ReadH0(HAJCLZH pW, UBP s)
{
	UB		stl;

	pW->HeadSize	 = s[0];									//	ヘッダサイズ読み出し
	pW->CheckSum	 = s[1];									//	チェックサム読み出し
	pW->HuffSize	 = StrLong(s + 7);							//	圧縮データサイズ読み出し
	pW->FileSize	 = StrLong(s + 11);							//	ファイルサイズ読み出し
	pW->FileTime	 = DosTimeToFileTime(StrLong(s + 15));		//	ファイルタイム読み出し
	pW->fLocalTime	 = TRUE;									//	フラグ＝「ローカルタイム」
	pW->FileAttr	 = s[19];									//	ファイル属性読み出し
	stl = GetByte(pW);											//	stl = ファイル名の長さ
	GetString(pW, stl, pW->PathName,							//	ファイル名読み出し
		(UI)(AJCLZHNAMEBUFSIZE - strlen(pW->PathName)));		//	・
	pW->crc 		 = GetWord(pW);								//	ＣＲＣ読み出し

	pW->BlkSize = pW->HeadSize + pW->HuffSize + 2;				//	ブロックサイズ設定

	SplitPath(pW->PathName, pW->DirName, pW->FileName);
}
//--------------------------------------------------------------------------------------------------------------//
//	ＬＨＡレベル１ヘッダ情報読み出し																			//
//																												//
//	引　数	：	pW	- インスタンスヘッダのアドレス																//
//				s	- ヘッダの先頭部分（２１バイト）のアドレス													//
//																												//
//	戻り値	：	なし																							//
//--------------------------------------------------------------------------------------------------------------//
static	VO	ReadH1(HAJCLZH pW, UBP s)
{
	UW		stl, exl;

	pW->HeadSize	 = s[0];									//	ヘッダサイズ読み出し
	pW->CheckSum	 = s[1];									//	チェックサム読み出し
	pW->HuffSize	 = StrLong(s + 7);							//	圧縮データサイズ読み出し
	pW->FileSize	 = StrLong(s + 11);							//	ファイルサイズ読み出し
	pW->FileTime	 = DosTimeToFileTime(StrLong(s + 15));		//	ファイルタイム読み出し
	pW->fLocalTime	 = TRUE;									//	フラグ＝「ローカルタイム」
	stl = GetByte(pW);											//	stl = ファイル名の長さ
	GetString(pW, stl, pW->PathName, AJCLZHNAMEBUFSIZE);		//	ファイル名読み出し
	pW->crc 		 = GetWord(pW);								//	ＣＲＣ読み出し
	pW->os	= GetByte(pW);										//	ＯＳ種別読み出し

	exl = ReadHEx(pW);											//	拡張ヘッダ読み出し

	pW->BlkSize = pW->HeadSize + pW->HuffSize + 2;				//	ブロックサイズ設定

	SplitPath(pW->PathName, pW->DirName, pW->FileName);
}
//--------------------------------------------------------------------------------------------------------------//
//	ＬＨＡレベル２ヘッダ情報読み出し																			//
//																												//
//	引　数	：	pW	- インスタンスヘッダのアドレス																//
//				s	- ヘッダの先頭部分（２１バイト）のアドレス													//
//																												//
//	戻り値	：	なし																							//
//--------------------------------------------------------------------------------------------------------------//
static	VO	ReadH2(HAJCLZH pW, UBP s)
{
	UI		exl;

	pW->HeadSize	 = StrWord(s + 0);							//	ヘッダサイズ読み出し
	pW->HuffSize	 = StrLong(s + 7);							//	圧縮データサイズ読み出し
	pW->FileSize	 = StrLong(s + 11);							//	ファイルサイズ読み出し
	pW->FileTime	 = StrLong(s + 15);							//	ファイルタイム読み出し
	pW->fLocalTime	 = FALSE;									//	フラグ＝「世界標準時」
	pW->crc 		 = GetWord(pW);								//	ＣＲＣ読み出し
	pW->os			 = GetByte(pW);								//	ＯＳ種別読み出し

	memset(pW->PathName, 0, AJCLZHNAMEBUFSIZE);					//	ファイル名クリアー
	exl = ReadHEx(pW);											//	拡張ヘッダ読み出し

	pW->BlkSize = pW->HeadSize + pW->HuffSize;					//	ブロックサイズ設定

	SplitPath(pW->PathName, pW->DirName, pW->FileName);
}
//--------------------------------------------------------------------------------------------------------------//
//	ＬＨＡ拡張ヘッダ情報読み出し																				//
//																												//
//	引　数	：	pW	- インスタンスヘッダのアドレス																//
//				s	- ヘッダの先頭部分（２１バイト）のアドレス													//
//																												//
//	戻り値	：	拡張ヘッダの総バイト数																			//
//--------------------------------------------------------------------------------------------------------------//
static	UI	ReadHEx(HAJCLZH pW)
{
	UI		rc = 2;
	UI		i;
	UI		len, stl, exl;
	UB		id;

	for (i=0; i < 16 && (exl = GetWord(pW)) != 0; i++) {								//	拡張ヘッダあり間ループ
		if (exl > 3) {
			stl = exl - 3;																//	　stl=拡張データサイズ
			id = GetByte(pW);															//	　ＩＤ読み出し
			switch (id) {
				case 0:																	//	　０：ＣＲＣ読み出し
					pW->crcEx = GetWord(pW);
					break;

				case 1:																	//	　１：ファイル名読み出し
					len = (UI)strlen(pW->PathName);										//		len = DIR名長
					GetString(pW, stl, &pW->PathName[len], AJCLZHNAMEBUFSIZE - len);	//		DIRの後へﾌｧｲﾙ名読出
					break;
				case 2:																	//	　２：ＤＩＲ名読み出し
					len = (UI)(strlen(pW->PathName) + 1);								//		len = ﾌｧｲﾙ名長
					memmove(&pW->PathName[AJCLZHNAMEBUFSIZE - len],						//		ﾌｧｲﾙ名を後部へ移動
							 pW->PathName, len);										//		・
					GetString(pW, stl, pW->PathName, AJCLZHNAMEBUFSIZE - len);			//		DIR読み出し
					strcat(pW->PathName, &pW->PathName[AJCLZHNAMEBUFSIZE - len]);		//		ﾌｧｲﾙ名を連結
					break;
				default:																//	　他：読み飛ばし
					GetString(pW, stl, NULL 	   , 0						);
					break;
			}
			rc += exl;																	//	　拡張ヘッダサイズ更新
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	書庫ストリームからワード情報読み出し																		//
//																												//
//	引　数	：	pW	- インスタンスヘッダのアドレス																//
//																												//
//	戻り値	：	読み出したワード値																				//
//--------------------------------------------------------------------------------------------------------------//
static	UW	GetWord(HAJCLZH pW)
{
	AJCLZHUNIWORD	u;

	u.ub.l = GetByte(pW);
	u.ub.h = GetByte(pW);

	return u.uw;
}
//--------------------------------------------------------------------------------------------------------------//
//	書庫ストリームから文字列情報読み出し																		//
//																												//
//	引　数	：	pW	 - インスタンスヘッダのアドレス																//
//				stl  - 文字列情報のバイト数																		//
//				pBuf - 文字列を格納するバッファのアドレス														//
//				bfl  - 文字列を格納するバッファのバイト数														//
//																												//
//	戻り値	：	なし																							//
//--------------------------------------------------------------------------------------------------------------//
static	VO	GetString(HAJCLZH pW, UI stl, BCP pBuf, UI bfl)
{
	UI	i = 0;
	BC	c;

	if (bfl) {
		for (; i < bfl-1 && i < stl; i++) {
			c = GetByte(pW);
			if (c == 0xFF) *pBuf++ = '\\';
			else		   *pBuf++ = c;
		}
		*pBuf = 0;
	}
	for (; i < stl; i++) {
		GetByte(pW);
	}
}
//--------------------------------------------------------------------------------------------------------------//
//	バイトストリームからワード情報読み出し																		//
//																												//
//	引　数	：	pW	- インスタンスヘッダのアドレス																//
//																												//
//	戻り値	：	読み出したワード値																				//
//--------------------------------------------------------------------------------------------------------------//
static	UW	StrWord(UBP pStr)
{
	AJCLZHUNIWORD	u;

	u.ub.l = *pStr++;
	u.ub.h = *pStr;

	return u.uw;
}
//--------------------------------------------------------------------------------------------------------------//
//	バイトストリームからロングワード情報読み出し																//
//																												//
//	引　数	：	pW	- インスタンスヘッダのアドレス																//
//																												//
//	戻り値	：	読み出したロングワード値																		//
//--------------------------------------------------------------------------------------------------------------//
static	UL	StrLong(UBP pStr)
{
	AJCLZHUNILONG	u;

	u.ub.l	= *pStr++;
	u.ub.ml = *pStr++;
	u.ub.mh = *pStr++;
	u.ub.h	= *pStr;

	return u.ul;
}
//--------------------------------------------------------------------------------------------------------------//
//	パス名分解																									//
//																												//
//	引　数	：	pPath - 分解するパス名のアドレス																//
//				pDir  - ディレクトリ名を格納するバッファのアドレス												//
//				pFile - ファイル名＋拡張子を格納するバッファのアドレス											//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - バッファオーバー																		//
//--------------------------------------------------------------------------------------------------------------//
static	VO	SplitPath(C_BCP pPath, BC pDir[MAX_PATH], BC pFile[MAX_PATH])
{
	BC		fname[_MAX_FNAME], fext[_MAX_EXT];

	_splitpath(pPath, NULL, pDir, fname, fext);
	_makepath (pFile, NULL, NULL, fname, fext);
}

//--------------------------------------------------------------------------------------------------------------//
//	ディレクトリパス名比較																						//
//																												//
//	引　数	：	pDirName  - ディレクトリパス名																	//
//				pWildDir  - ワイルドカードのディレクトリ名部分													//
//				fSubDir   - サブディレクトリ検索フラグ															//
//																												//
//	戻り値	：	TRUE ：一致																						//
//				FALSE：不一致																					//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	CmpDirName(C_BCP pDirName, C_BCP pWildDir, BOOL fSubDir)
{
	BOOL	rc = FALSE;
	int		i, len1, len2;
	BC		d1[MAX_PATH];
	BC		d2[MAX_PATH];

	//	パス名コピー
	AjcSnPrintFA(d1, AJCTSIZE(d1), "%s", pDirName);
	AjcSnPrintFA(d2, AJCTSIZE(d2), "%s", pWildDir);

	//	d1 : '\' -> '/' 変換
	for (i = 0; d1[i] != 0; i++) {
		if (d1[i] == '\\') d1[i] = '/';
	}
	//	d2 : '\' -> '/' 変換
	for (i = 0; d2[i] != 0; i++) {
		if (d2[i] == '\\') d2[i] = '/';
	}

	len1 = (int)strlen(d1);
	len2 = (int)strlen(d2);

	//	サブディレクトリも検索する場合は、先頭部分のみ比較
	if (fSubDir) {
		if (len1 >= len2) {
			rc = (memicmp(d1, d2, len2) == 0);
		}
	}
	//	サブディレクトリを検索しない場合は、全体を比較
	else {
		if (len1 == len2) {
			rc = (mbsicmp(d1, d2) == 0);
		}
	}

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	ワイルドカード比較																							//
//																												//
//	引　数	：	pPath  - パス名																					//
//				pWild  - ワイルドカード																			//
//																												//
//	戻り値	：	TRUE ：一致																						//
//				FALSE：不一致																					//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	CmpWildStr(C_BCP pPath, C_BCP pWild)
{
	BOOL	rc = TRUE;
	BC		path[MAX_PATH];
	BC		fname[_MAX_FNAME], fext[_MAX_EXT];
	BC		file[MAX_PATH];

	//	パスをコピーし末尾の'\'を削除
	AjcSnPrintFA(path, AJCTSIZE(path), "%s", pPath);
	PathRemoveBackslashA(path);
	//	ファイル名／ディレクトリ名一致チェック
	_splitpath(path, NULL, NULL, fname, fext);
	_makepath (file, NULL, NULL, fname, fext);
	if (mbsicmp(file, pWild) != 0) {
		//	ワイルドカード・ヒットチェック
		rc = PathMatchSpecA(path, pWild);
	}
	return rc;
}

//--------------------------------------------------------------------------------------------------------------//
//	ＤＯＳタイムをファイルタイム（1970-01-01 00:00:00からの通算秒数）に変換										//
//																												//
//	引　数	：	DosTime - ＤＯＳタイム																			//
//																												//
//	戻り値	：	ファイルタイム（1970-01-01 00:00:00からの通算秒数）												//
//--------------------------------------------------------------------------------------------------------------//
#define		ISURU(Y)	(((Y % 4) == 0 && (Y % 100) != 0) || (Y % 400) == 0)

static	UL	DosTimeToFileTime(UL DosTime)
{
	UL	ftime = 0;
	UI	m, year, month, day, hour, minute, second;

	year   = (UI)(( DosTime >> 25) + 1980);
	month  = (UI)(((DosTime >> 21) & 0x0F));
	day    = (UI)(((DosTime >> 16) & 0x1F));
	hour   = (UI)(((DosTime >> 11) & 0x1F));
	minute = (UI)(((DosTime >>	5) & 0x3F));
	second = (UI)(((DosTime >>	0) & 0x1F) * 2);

	ftime += (((((UL)year - (UL)1970) * (UL)365) + (((UL)year - (UL)1969) / (UL)4)) * (UL)86400);
	if (year > 2100) 
		ftime -= (UL)86400;
	for (m=1; m < month; m++) 
		ftime += ((DaysTable[m] + (m==2 ? ISURU(year) : 0)) * (UL)86400);
	ftime += ((UL)day - (UL)1) * (UL)86400;
	ftime += ((UL)hour * (UL)3600);
	ftime += ((UL)minute * (UL)60);
	ftime += second;

	return ftime;
}
