﻿//
//	AcsMacro.c - マクロ登録、展開等
//
#include	"AjcInternal.h"
#include	"AjcPp.h"

#define		SYL_VA_ARGS		"__VA_ARGS__"

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
static PAJCPPCTKNNODE MacExpWithArg	(HAJCPPC pW, PAJCPPCTKNNODE *ppBfr, PAJCPPCTKNNODE pTkn, PPPCMACARGS pMac);
static PAJCPPCTKNNODE MacExpNoArg  	(HAJCPPC pW, PAJCPPCTKNNODE *ppBfr, PAJCPPCTKNNODE pTkn, PPPCMACNODE pMac);
static BCP			  NumStr		(HAJCPPC pW, int n);

//--------------------------------------------------------------------------------------------------------------//
//	キーの比較コールバック																						//
//--------------------------------------------------------------------------------------------------------------//
static int CALLBACK cbAvlComp(UX key1, UX key2, UX cbp)
{
	HAJCPPC pW = (HAJCPPC)cbp;

	return(mbscmp((C_BCP)key1, (C_BCP)key2));
}
//--------------------------------------------------------------------------------------------------------------//
//	オブジェクト削除時のコールバック																			//
//--------------------------------------------------------------------------------------------------------------//
static VO CALLBACK cbAvlRemove(UX key, C_VOP pNode, UI len, UI nest, UX cbp)
{
	HAJCPPC			pW = (HAJCPPC)cbp;
	PCPPCMACARGS 	pN = (PCPPCMACARGS)pNode;
	PAJCPPCTKNNODE	pT = pN->pTknBody;

	//----- マクロ名解放 ---------------------------//
	if (pN->pMacName != NULL) PpcMemFree(pW, pN->pMacName);
	//----- マクロ名ノード解放 ---------------------//
	PpcTknFree(pW, pN->pTknName);
	//----- マクロボディ解放 -----------------------//
	while (pT != NULL) {
		pT = PpcRelCurAndGetNextToken(pW, pT);
	}
	//----- 仮引数リスト解放 -----------------------//
	if (pN->fWithArg && pN->hVQueArg != NULL) {
		AjcVQueDelete(pN->hVQueArg);
	}
}

//==============================================================================================================//
//	初期設定																									//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	≠NULL：AVLハンドル																				//
//				＝NULL：エラー																					//
//==============================================================================================================//
HAJCAVL	PpcMacInit(HAJCPPC pW)
{

	if (pW->hAvlMac == NULL) {
		pW->hAvlMac = AjcAvlCreate((UX)pW, cbAvlComp, cbAvlRemove);

		if (pW->hAvlMac == NULL) {
			pW->fMemErr = TRUE;
			PPC_ERR(AJCPPC_ERROR_MEMALLOC, 0, 0, 0);
		}
	}
	else {
		AjcAvlDelAllNodes(pW->hAvlMac);
	}
	return pW->hAvlMac;
}
//==============================================================================================================//
//	終了処理																									//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
VO		PpcMacEnd(HAJCPPC pW)
{
	if (pW->hAvlMac != NULL) {
		AjcAvlDelete(pW->hAvlMac);
		pW->hAvlMac = NULL;
	}
}
//==============================================================================================================//
//	マクロ登録																									//
//																												//
//	引　数	：	pTkn	 - マクロ定義トークン・ストリームのアドレス（先頭トークンはマクロ名）					//
//				fWithArg - TRUE:引数付マクロ，FALSE:引数なしマクロ												//
//				pos		 - 先頭「#」の桁位置																	//
//																												//
//	戻り値	：	TRUE ：OK																						//
//				FALSE：エラー																					//
//																												//
//	備考	：	本関数実行後もpTknで指定したトークンストリームは解放しないでください							//
//==============================================================================================================//
BOOL	PpcMacRegist(HAJCPPC pW, PAJCPPCTKNNODE pTkn, BOOL fWithArg, UI pos)
{
	BOOL	rc = FALSE;
	BCP		pFile = pTkn->pFile;
	UI		lno   = pTkn->lno;

	if (pW->hAvlMac != NULL) {
		//----- 引数付きマクロ ---------------------------------------------------------------------------------//
		if (fWithArg) {
			PPCMACARGS		n_new;
			PPPCMACARGS		p_reg;
			PAJCPPCTKNNODE	pC		 = pTkn;
			UI				stl;
			BCP				pLastArg = "";
			BOOL			fErr	 = FALSE;

			do {
				//----- ノード初期化 -----------------------------------------//
				memset(&n_new, 0, sizeof n_new);
				//----- 先頭「#」の桁位置設定 --------------------------------//
				n_new.pos = pos;
				//----- マクロ名設定 -----------------------------------------//
				n_new.fWithArg = TRUE;
				stl = (UI)strlen(pTkn->pSyl) + 1;
				n_new.pMacName = PpcMemAlloc(pW, stl);
				if (n_new.pMacName == NULL) {
					fErr = pW->fStop = TRUE;
					break;
				}
				strcpy(n_new.pMacName, pTkn->pSyl);
				//----- マクロ名トークン・ノードポインタ設定 -----------------//
				n_new.pTknName = pTkn;
				//----- マクロ名の次のトークンポインタ設定 -------------------//
				if ((pC = pC->pNxt) == NULL) break;
				//----- '('チェック ------------------------------------------//
				if (pC->tkn != EAJCTK_DLM_LSPART) {
					PPC_ERR(AJCPPC_MACERR_NEED_LP, pC->pFile, pC->lno, n_new.pMacName);
					fErr = TRUE;
					break;
				}
				//----- '(' の次のトークンポインタ設定 -----------------------//
				if ((pC = PpcRelCurAndGetNextToken(pW, pC)) == NULL) break;
				//----- 仮引数リスト作成 -------------------------------------//
				if ((n_new.hVQueArg = AjcVQueCreate(0, NULL)) == NULL) break;
				while (pC != NULL && pC->tkn != EAJCTK_DLM_RSPART) {
					//----- 語句文字列退避 -----------------------------------//
					pLastArg = pC->pSyl;
					//----- 可変個引数チェック -------------------------------//
					if (pC->tkn == EAJCTK_DLM_VARG) {
						pC->tkn 	 = EAJCTK_USR_NAME;
						pC->pSyl	 = PpcNStrRegist(pW, SYL_VA_ARGS);
						if (pC->pSyl == NULL) {
							fErr = pW->fStop = TRUE; 
							break;
						}
						n_new.fVaArgs = TRUE;
					}
					//----- 仮引数がシンボル以外ならばエラー -----------------//
					if (!AJCTKIS_SYMBOL(pC->tkn)) break;
					//----- 仮引数リスト追加 ---------------------------------//
					stl = (UI)strlen(pC->pSyl) + 1;
					if (!AjcVQueEnque(n_new.hVQueArg, pC->pSyl, stl)) {
						fErr = pW->fStop = TRUE;
						break;
					}
					//----- 仮引数リストの個数更新 ---------------------------//
					n_new.nTmpArg++;
					//----- 仮引数の次のトークンポインタ設定 -----------------//
					if ((pC = PpcRelCurAndGetNextToken(pW, pC)) == NULL) break;
					//----- ',' or ')' チェック ------------------------------//
					if		(pC->tkn == EAJCTK_DLM_COMMA && !n_new.fVaArgs) {		// ',' && 可変個引数(...)無し？
						//----- ','の次のトークンポインタ設定 ----------------//
						if ((pC = PpcRelCurAndGetNextToken(pW, pC)) == NULL) break;
					}
					else if (pC->tkn == EAJCTK_DLM_RSPART) {						// ')' ？
						// ループを抜けてマクロボディ設定 //
						break;
					}
					else {	  														// ','	')' 以外？
						if (n_new.fVaArgs) {
							PPC_ERR(AJCPPC_MACERR_NEED_RP, pC->pFile, pC->lno, pLastArg);
							fErr = TRUE;
						}
						else {
							PPC_ERR(AJCPPC_MACERR_NEED_RP_C, pC->pFile, pC->lno, pLastArg);
							fErr = TRUE;
						}
						break;
					}
				}
			} while(0);
			//----- マクロボディ設定，マクロ登録 -----------------------------//
			if (!fErr && pC != NULL && pC->tkn == EAJCTK_DLM_RSPART) {	//	最後が右括弧 ')' ?
				//----- 右括弧の次のトークンポインタ設定 ---------------------//
				pC = PpcRelCurAndGetNextToken(pW, pC);
				//----- マクロボディ設定 -------------------------------------//
				n_new.pTknBody = pC;
				//----- マクロ登録 -------------------------------------------//
				p_reg = AjcAvlGetNodePtr(pW->hAvlMac, (UX)n_new.pMacName, NULL);
				if (rc = (p_reg == NULL)) {
					if (rc = AjcAvlInsNode(pW->hAvlMac, (UX)n_new.pMacName, &n_new, sizeof(n_new))) {
					//	AjcTrace("Registered MACRO() %s\n", n_new.pMacName);
					}
					else {
						pW->fMemErr = TRUE;
						pW->fStop = TRUE;
						PPC_ERR(AJCPPC_ERROR_MEMALLOC, 0, 0, 0);
					}
				}
				else {
					if (!PpcCompTokenStream(pW, p_reg->pTknBody, n_new.pTknBody)) {
						PPC_ERR(AJCPPC_MACERR_MULTDEF, pFile, lno, n_new.pMacName);
					}
					cbAvlRemove((UX)n_new.pMacName, &n_new, sizeof n_new, 0, 0);
				}
			}
			//----- リソース解放 --------------------------------------------//
			else {
				//----- マクロボディ・トークンを解放 --------//
				while (pC != NULL) {
					pC = PpcRelCurAndGetNextToken(pW, pC);
				}
				//----- マクロノード解放 --------------------//
				if (n_new.pMacName != NULL) PpcMemFree(pW, n_new.pMacName);
				if (n_new.hVQueArg != NULL) AjcVQueDelete(n_new.hVQueArg);
			}
			//----- 登録内容表示（デバッグ） ---------------------------------//
		}
		//----- 引数なしマクロ ---------------------------------------------------------------------------------//
		else {
			PPCMACNODE		n_new;
			PPPCMACARGS		p_reg;
			PAJCPPCTKNNODE	pC = pTkn;
			UI				stl;

			//----- ノードクリアー ---------------------------------------//
			memset(&n_new, 0, sizeof n_new);
			//----- マクロ名設定 -----------------------------------------//
			n_new.fWithArg = FALSE;
			stl = (UI)strlen(pTkn->pSyl) + 1;
			n_new.pMacName = PpcMemAlloc(pW, stl);
			//----- マクロ名トークン・ノードポインタ設定 -----------------//
			n_new.pTknName = pTkn;

			if (n_new.pMacName != NULL) {
				strcpy(n_new.pMacName, pTkn->pSyl);
				//----- マクロ名の次のトークンポインタ設定 -------------------//
				pC = pC->pNxt;
				//----- マクロボディ設定 -------------------------------------//
				n_new.pTknBody = pC;
				//----- マクロ登録 -------------------------------------------//
				p_reg = AjcAvlGetNodePtr(pW->hAvlMac, (UX)n_new.pMacName, NULL);
				if (rc = (p_reg == NULL)) {
					if (rc = AjcAvlInsNode(pW->hAvlMac, (UX)n_new.pMacName, &n_new, sizeof(n_new))) {
					}
					else {
						pW->fMemErr = TRUE;
						pW->fStop	= TRUE;
						PPC_ERR(AJCPPC_ERROR_MEMALLOC, 0, 0, 0);
					}
				}
				else {
					if (!PpcCompTokenStream(pW,  p_reg->pTknBody, n_new.pTknBody)) {
						PPC_ERR(AJCPPC_MACERR_MULTDEF, pFile, lno, n_new.pMacName);
					}
					cbAvlRemove((UX)n_new.pMacName, &n_new, sizeof n_new, 0, 0);
				}
			}
			else pW->fStop = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	マクロ登録解除																								//
//																												//
//	引　数	：	pName	  - マクロ名のアドレス																	//
//																												//
//	戻り値	：	TRUE ：OK																						//
//				FALSE：エラー																					//
//==============================================================================================================//
BOOL		PpcMacUnRegist(HAJCPPC pW, C_BCP pName)
{
	BOOL	rc = FALSE;

	if (pW->hAvlMac != NULL) {
		rc = AjcAvlDelNode(pW->hAvlMac, (UX)pName);
	}
	return rc;
}
//==============================================================================================================//
//	マクロ情報ノード取得（マクロ登録チェック）																	//
//																												//
//	引　数	：	pName	  - マクロ名のアドレス																	//
//																												//
//	戻り値	：	≠NULL：登録済（マクロ情報ノードのアドレス）													//
//				＝NULL：未登録																					//
//==============================================================================================================//
VOP		PpcMacGetNode(HAJCPPC pW, C_BCP pName)
{
	VOP		rc = NULL;

	if (pW->hAvlMac != NULL) {
		rc = AjcAvlGetNodePtr(pW->hAvlMac, (UX)pName, NULL);
	}
	return rc;
}
//==============================================================================================================//
//	先頭シンボルのマクロ展開（重複展開）																		//
//																												//
//	先頭シンボルがマクロ名である場合、当該マクロ参照を展開形で置き換えます。（重複展開あり）					//
//																												//
//	引　数	：	ppBfr - 直前のトークンから先頭トークンへのポインタのアドレス									//
//																												//
//	戻り値	：	TRUE : マクロ展開済（マクロ参照部分を展開形で置換えた，置き換えるトークンが1つも無い場合も含む）//
//				FALSE: マクロ非展開																				//
//==============================================================================================================//
BOOL	PpcMacExpand(HAJCPPC pW, PAJCPPCTKNNODE *ppBfr)
{
	BOOL			rc		  = FALSE;
	VOP				pMacNode  = NULL;
	PAJCPPCTKNNODE 	pTkn;
	AJCPPCTKNNODE	TknRef;

	if (*ppBfr != NULL) {
		pTkn = *ppBfr;
		if (pTkn->mexp <= PPCMAX_MEXP_NEST) {
			//	マクロ参照先頭トークン退避
			memcpy(&TknRef, pTkn, sizeof(AJCPPCTKNNODE));
			while (pTkn != NULL && AJCTKIS_SYMBOL(pTkn->tkn) && (pMacNode = PpcMacGetNode(pW, pTkn->pSyl))) {
				//	中止チェック
				if (pW->fStop) break;
				//	引数付マクロで、次のトークンが '(' 以外ならば、展開しない（関数マクロの呼び出しではない）
				if (((PPPCMACNODE)pMacNode)->fWithArg && (pTkn->pNxt == NULL || pTkn->pNxt->tkn != EAJCTK_DLM_LSPART)) break;
				//	マクロ展開処理
				if (pTkn->mexp <= PPCMAX_MEXP_NEST) {
					//	マクロ展開
					if (((PPPCMACNODE)pMacNode)->fWithArg) {
						pTkn = MacExpWithArg(pW, ppBfr, pTkn, (PPPCMACARGS)pMacNode);
					}
					else {
						pTkn = MacExpNoArg	(pW, ppBfr, pTkn, (PPPCMACNODE)pMacNode);
					}
					//	マクロ参照通知
					PPC_NTC(AJCPPC_NTC_MACREF, (UX)&TknRef, (UX)pMacNode, 0);
					//	戻り値＝マクロ展開済
					rc = TRUE;
				}
				else {
					PPC_ERR(AJCPPC_MACERR_NEST_OVER, (UX)pTkn->pFile, (UX)pTkn->lno, (UX)(((PPPCMACNODE)pMacNode)->pMacName));
					break;
				}
			}
		}
		else {
			PPC_ERR(AJCPPC_MACERR_NEST_OVER, (UX)pTkn->pFile, (UX)pTkn->lno, (UX)"");
		}
	}

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	引数付きマクロの展開																						//
//																												//
//	引　数	：	ppBfr - マクロ参照文先頭トークンへのポインタのアドレス											//
//				pTkn  - マクロ参照文先頭トークンのアドレス														//
//				pMac  - マクロ定義ノードのアドレス																//
//																												//
//	戻り値	：	≠NULL : マクロ展開トークンあり（展開形の先頭ノードアドレス）									//
//				＝NULL : マクロ展開トークンなし																	//
//--------------------------------------------------------------------------------------------------------------//
//--- 仮引数と実引数の対応リスト ---//
typedef struct {
	BCP			pTmpArg;
	PAJCPPCTKNNODE	pRelArg;
} MEW_LIST, *PMEW_LIST;
typedef const MEW_LIST *PCMEW_LIST;

static	int		SrhTmpArg(HAJCPPC pW, C_BCP pSyl, PCMEW_LIST pArr, int n);

//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static PAJCPPCTKNNODE MacExpWithArg(HAJCPPC pW, PAJCPPCTKNNODE *ppBfr, PAJCPPCTKNNODE pTkn, PPPCMACARGS pMac)
{
	PMEW_LIST		pALArr		 = NULL;					//	引数リスト配列のアドレス
	PAJCPPCTKNNODE	pALTkn		 = NULL;					//	仮引数に対する実引数トークンリスト
	BCP				pFile		 = pTkn->pFile;				//	ファイルパス名設定
	UW				lno 		 = pTkn->lno;				//	行番号初期値＝マクロ名の行番号
	UI				pos			 = pTkn->pos;				//	トークンの位置（タブステップ＝４）
	UB				inst		 = pTkn->inst;				//	#includeのネストレベル
	UB				incf		 = pTkn->incf;				//	#include種別
	UB				mexp		 = pTkn->mexp + 1;			//	マクロ展開ネストレベル
	BCP				pMacName	 = pTkn->pSyl;				//	マクロ名
	BCP				pStr		 = NULL;					//	文字列編集ワーク
	BOOL			fVaArg		 = FALSE;					//	可変個引数マクロ・フラグ
	UI				nTmpArg		 = pMac->nTmpArg;			//	仮引数の個数
	UI				nValArg		 = 0;						//	有効な引数の個数（可変個引数が無い場合は仮引数－１）
	PAJCPPCTKNNODE	pCur;									//	現トークンポインタ・ワーク
	PAJCPPCTKNNODE	pBfr;									//	トークンポインタ退避ワーク
	PAJCPPCTKNNODE	pArg		 = NULL;					//	実引数トークンポインタ
	PAJCPPCTKNNODE	pBdy		 = NULL;					//	マクロボディトークンポインタ
	PAJCPPCTKNNODE	pExpTop		 = NULL;					//	マクロ展開トークンの先頭ポインタ
	PAJCPPCTKNNODE	pExp		 = NULL;					//	マクロ展開トークンポインタ
	PAJCPPCTKNNODE	pSvNxt;									//	マクロ参照文の直後のトークンアドレス（無い場合はNULL）
	int				nest 		= 0;						//	括弧のネストカウンタ
	UB				flg			= 0;						//	マクロ参照の末尾（右括弧）のフラグ情報
	UI				i, stl;

	//----------------------------------------------------------------------------------------------------------//
	//	仮引数と実引数の対応リストテーブル確保し、仮引数名設定													//
	//----------------------------------------------------------------------------------------------------------//
	i = 0;
	pALArr = PpcMemAlloc(pW, sizeof(MEW_LIST) * pMac->nTmpArg);
	if (pALArr != NULL) {
		if (pStr = AjcVQueTopNode(pMac->hVQueArg, NULL)) {
			do {
				pALArr[i].pTmpArg = pStr;
				i++;
			} while(i < pMac->nTmpArg && (pStr = AjcVQueNextNode(pMac->hVQueArg, pStr, NULL)));
		}
		//----------------------------------------------------------------------------------------------------------//
		//	本文からマクロ参照文部分を消去し、テーブルに実引数を設定												//
		//----------------------------------------------------------------------------------------------------------//
		pCur = PpcRelCurAndGetNextToken(pW, pTkn);						//	マクロ名スキップ
		if (pCur != NULL && pCur->tkn == EAJCTK_DLM_LSPART) {			//	次のトークンは左括弧 '('
			if ((pCur = PpcRelCurAndGetNextToken(pW, pCur)) != NULL) {	//	左括弧'('スキップ，次トークン有り？
				nest	= 0;
				nValArg = 0;
				pALTkn	= pCur;
				while (pCur != NULL && pCur->tkn != EAJCTK_DLM_RSPART && nest >= 0) {
					if (pW->fStop) break;
					//----- 実引数トークンリスト設定 ---------------------//
					if (pCur->tkn != EAJCTK_DLM_COMMA && pCur->tkn != EAJCTK_DLM_RSPART) {
						//----- 可変個引数フラグ設定 ---------------------//
						fVaArg = (pMac->fVaArgs && mbscmp(pALArr[nValArg].pTmpArg, SYL_VA_ARGS) == 0);
						//----- 実引数の先頭トークンアドレス設定 ---------//
						pALTkn = pCur;
						//----- 実引数の終端検索 -------------------------//
						nest = 0;
						pBfr = pCur;
						do {
							//----- ネストカウンタ設定 -------------------//
							if		(pCur->tkn == EAJCTK_DLM_LSPART) {
								nest++;
							}
							else if (pCur->tkn == EAJCTK_DLM_RSPART) {
								nest--;
							}
							//----- 次のトークン設定 ---------------------//
							pBfr = pCur;
							pCur = pCur->pNxt;
						} while (pCur != NULL && 
								 !(nest == 0  && ((!fVaArg && pCur->tkn == EAJCTK_DLM_COMMA) || pCur->tkn == EAJCTK_DLM_RSPART)));
						//----- 実引数トークンリストの終端設定 -----------//
						if (pBfr != NULL) {
							pBfr->pNxt = NULL;
						}
						//----- 実引数リスト設定 -------------------------//
						if (nValArg < pMac->nTmpArg) {		//	引数の個数範囲内？
							pALArr[nValArg].pRelArg = pALTkn;
						}
						else {								//	引数個数オーバー
							PpcRelTokenStream(pW, pALTkn);
						}
					}
					else {	// 実引数無し？
						// 実引数トークンポインタ=NULL
					}
					//----- 有効な引数の個数更新 -------------------------//
					nValArg++;
					//----- カンマ(,)スキップ ----------------------------//
					if (pCur != NULL && pCur->tkn == EAJCTK_DLM_COMMA) {
						pCur = PpcRelCurAndGetNextToken(pW, pCur);
						//----- カンマの直後が右括弧ならば、有効な引数（空引数）の個数更新 --//
						if (pCur != NULL && pCur->tkn == EAJCTK_DLM_RSPART) {
							nValArg++;
						}
					}
				}
				//----- 右括弧スキップ -----------------------------------//
				if (pCur != NULL && pCur->tkn == EAJCTK_DLM_RSPART) {
					//	右括弧のフラグ退避
					flg = pCur->flg;
					//	右括弧スキップ
					pCur = PpcRelCurAndGetNextToken(pW, pCur);
				}
			}
		}
		//----- マクロ参照文の次のトークンアドレス設定 -------------------//
		pSvNxt = pCur;

		//----- 実引数の個数エラーチェック -------------------------------//
		if (pMac->fVaArgs) {		//	可変個引数(...)あり？
			if (nValArg < pMac->nTmpArg - 1) {
				PPC_ERR(AJCPPC_MACERR_ARG_LACK, pFile, lno, pMacName);
			}
		}
		else {						//	可変個引数(...)なし？
			if (nValArg != pMac->nTmpArg) {
				if (nValArg < pMac->nTmpArg) {
					PPC_ERR(AJCPPC_MACERR_ARG_LACK, pFile, lno, pMacName);
				}
				else {
					PPC_ERR(AJCPPC_MACERR_ARG_OVER, pFile, lno, pMacName);
				}
			}
		}

		//----------------------------------------------------------------------------------------------------------//
		//	マクロ展開																								//
		//----------------------------------------------------------------------------------------------------------//
		pExp = NULL;
		if ((pBdy = pMac->pTknBody) != NULL) {
			while (pBdy != NULL) {
				if (pW->fStop) break;	//	中止
				if (AJCTKIS_SYMBOL(pBdy->tkn)) {									//	シンボル
					//----- トークン生成 -------------------//
					if ((i = SrhTmpArg(pW, pBdy->pSyl, pALArr, nTmpArg)) != -1) {	//		仮引数と一致？
						if (pALArr[i].pRelArg != NULL) {							//			実引数有り？
							//----- 実引数をコピー ---------//
							pArg = pALArr[i].pRelArg;
							while (pArg != NULL) {
								if (pW->fStop) break;	//	中止
								lno = __max(lno, pArg->lno);
								if (pExpTop == NULL) pExp = pExpTop = PpcTknAlloc(pW);
								else				 pExp = PpcAllocAndLinkNextToken(pW, pExp);
								if (pExp != NULL) {
									pExp->pFile  = pFile;
									pExp->lno	 = lno;
									pExp->pSyl	 = pArg->pSyl;
									pExp->tkn	 = pArg->tkn;
									pExp->kndPP  = AJCPPC_PPK_TOKEN;
									pExp->pos	 = pos;
									pExp->ppIf	 = 0;
									pExp->flg	 = pArg->flg;
									pExp->inst	 = inst;
									pExp->incf	 = incf;
									pExp->mexp	 = mexp;
									pArg = pArg->pNxt;
								}
								else {
									pW->fStop = TRUE;
								}
							}
						}
					}
					else {															//		仮引数以外？
						//----- ボディトークンをコピー -----//
						if (pExpTop == NULL) pExp = pExpTop = PpcTknAlloc(pW);
						else				 pExp = PpcAllocAndLinkNextToken(pW, pExp);
						if (pExp != NULL) {
							pExp->pFile  = pFile;
							pExp->lno	 = lno;
							pExp->pSyl	 = pBdy->pSyl;
							pExp->tkn	 = pBdy->tkn;
							pExp->kndPP  = AJCPPC_PPK_TOKEN;
							pExp->pos	 = pos;
							pExp->ppIf	 = 0;
							pExp->flg	 = (pBdy->flg & (AJCTKF_NXTSPC | AJCTKF_WIDECHAR));
							pExp->inst	 = inst;
							pExp->incf	 = incf;
							pExp->mexp	 = mexp;
						}
						else pW->fStop = TRUE;
					}
				}
				else if (pBdy->tkn == EAJCTK_DLM_SHARP 	&&							// '#'					&&
						 pBdy->pNxt != NULL				&&							//	次のトークン有り	&&
						(i = SrhTmpArg(pW, pBdy->pNxt->pSyl, pALArr, nTmpArg)) != -1) {	//	次のトークンは仮引数
					//----- '#' スキップ -------------------//
					pBdy = pBdy->pNxt;
					//----- 文字列トークン生成 -------------//
					pArg = pArg = pALArr[i].pRelArg;
					if (pArg != NULL) {	// 実引数あり？
						lno  = __max(lno, pArg->lno);
						//	文字列長算出
						stl = 0;
						pArg = pALArr[i].pRelArg;
						while (pArg != NULL) {
							stl += (UI)strlen(pArg->pSyl);
							pArg = pArg->pNxt;
						}
						//	総合文字列生成
						pStr = PpcMemAlloc(pW, stl + 3);
						if (pStr != NULL) {
							strcpy(pStr, "\"");
							pArg = pALArr[i].pRelArg;
							while (pArg != NULL) {
								strcat(pStr, pArg->pSyl);
								pArg = pArg->pNxt;
							}
							strcat(pStr, "\"");
						}
						else pW->fStop = TRUE;
					}
					else {				//	実引数なし？
						pStr = PpcMemAlloc(pW, 3);
						if (pStr != NULL) {
							strcpy(pStr, "\"\"");
						}
						else pW->fStop = TRUE;
					}
					//	文字列トークン生成
					pArg = pALArr[i].pRelArg;
					if (pExpTop == NULL) pExp = pExpTop = PpcTknAlloc(pW);
					else				 pExp = PpcAllocAndLinkNextToken(pW, pExp);
					if (pExp != NULL && pStr != NULL) {
						pExp->pFile  = pFile;
						pExp->lno	 = lno;
						pExp->pSyl	 = PpcNStrRegist(pW, pStr);
						if (pExp->pSyl == NULL) {pW->fStop = TRUE; break;}
						pExp->tkn	 = EAJCTK_STR_QUOTE;
						pExp->kndPP  = AJCPPC_PPK_TOKEN;
						pExp->pos	 = pos;
						pExp->ppIf	 = 0;
						pExp->flg	 = 0;
						pExp->inst	 = inst;
						pExp->incf	 = incf;
						pExp->mexp	 = mexp;
					}
					else pW->fStop = TRUE;
					//	文字列解放
					PpcMemFree(pW, pStr);
				}
				else if (pBdy->tkn == EAJCTK_DLM_SHARP2	&&							//	'##'				&&
						 pBdy->pNxt != NULL) {										//	次のトークン有り
					//----- '##' スキップ ------------------//
					pBdy = pBdy->pNxt;
					if ((i = SrhTmpArg(pW, pBdy->pSyl, pALArr, nTmpArg)) != -1) {	//		仮引数？
						//----- トークン文字列連結 -------------//
						//	語句文字列長算出
						pArg = pALArr[i].pRelArg;
						if (pArg != NULL) {	//	実引数あり？
							stl = (UI)strlen(pExp->pSyl) + (UI)strlen(pArg->pSyl);
							//	新たな語句文字列生成
							pStr = PpcMemAlloc(pW, stl + 1);
							if (pStr != NULL) {
								strcpy(pStr, pExp->pSyl);
								strcat(pStr, pArg->pSyl);
								pExp->pSyl = PpcNStrRegist(pW, pStr);
								if (pExp->pSyl == NULL) {pW->fStop = TRUE; break;}
								//	後続の実引数生成
								pArg = pArg->pNxt;
								while (pArg != NULL) {
									if (pW->fStop) break;	//	中止
									if (pExpTop == NULL) pExp = pExpTop = PpcTknAlloc(pW);
									else				 pExp = PpcAllocAndLinkNextToken(pW, pExp);
									if (pExp != NULL) {
										pExp->pFile  = pFile;
										pExp->lno	 = lno;
										pExp->pSyl	 = pArg->pSyl;
										pExp->tkn	 = pArg->tkn;
										pExp->kndPP  = AJCPPC_PPK_TOKEN;
										pExp->pos	 = pos;
										pExp->ppIf	 = 0;
										pExp->flg	 = pArg->flg;
										pExp->inst	 = inst;
										pExp->incf	 = incf;
										pExp->mexp	 = mexp;
									}
									else pW->fStop = TRUE;
								}
								//	語句文字列解放
								PpcMemFree(pW, pStr);
							}
							else pW->fStop = TRUE;
						}
					}
					else {															//		仮引数以外？
						//	語句文字列長算出
						stl = (UI)strlen(pExp->pSyl) + (UI)strlen(pBdy->pSyl);
						//	新たな語句文字列生成
						pStr = PpcMemAlloc(pW, stl + 1);
						if (pStr != NULL) {
							strcpy(pStr, pExp->pSyl);
							strcat(pStr, pBdy->pSyl);
							pExp->pSyl = PpcNStrRegist(pW, pStr);
							if (pExp->pSyl == NULL) {pW->fStop = TRUE; break;}
							pExp->mexp = mexp;
							//	語句文字列解放
							PpcMemFree(pW, pStr);
						}
						else pW->fStop = TRUE;
					}
				}
				else if (pBdy->tkn == EAJCTK_DLM_COMMA 	&&						//	カンマ(,) 						&&
						 pMac->fVaArgs 					&&						//	可変個引数(...)付マクロ			&&
						 pBdy->pNxt != NULL 			&&						//	次のトークン有					&&
						 mbscmp(pBdy->pNxt->pSyl, SYL_VA_ARGS) == 0 &&			//	次のトークンは「__VA_ARGS__」	&&
						 nValArg < pMac->nTmpArg) {								//	実引数の個数＜仮引数の個数（実引数指定無し）
					//----- カンマをスキップ ---------------//
					pBdy = pBdy->pNxt;
				}
				else {															//	その他
					//----- ボディトークンをコピー ---------//
					if (pExpTop == NULL) pExp = pExpTop = PpcTknAlloc(pW);
					else				 pExp = PpcAllocAndLinkNextToken(pW, pExp);
					if (pExp != NULL) {
						pExp->pFile  = pFile;
						pExp->lno	 = lno;
						pExp->pSyl	 = pBdy->pSyl;
						pExp->tkn	 = pBdy->tkn;
						pExp->kndPP  = AJCPPC_PPK_TOKEN;
						pExp->pos	 = pos;
						pExp->ppIf	 = 0;
						pExp->flg	 = (pBdy->flg & (AJCTKF_NXTSPC | AJCTKF_WIDECHAR));
						pExp->inst	 = inst;
						pExp->incf	 = incf;
						pExp->mexp	 = mexp;
					}
					else pW->fStop = TRUE;
				}
				//----- 次のマクロボディ語句ポインタ設定 -------------//
				if (pBdy != NULL) {
					pBdy = pBdy->pNxt;
				}
			}
		}

		//----- マクロ展開中のマクロを解決する ----------------------//
		if (!pW->fStop) {
			VOP				pNaMac 	 = NULL;
			PAJCPPCTKNNODE	pNaExp	 = pExpTop;
			PAJCPPCTKNNODE	*ppNaBfr = &pExpTop;

			//	引数をマクロ展開
			//
			//	（マクロ展開中のマクロの引数がマクロである場合、この引数マクロを先に展開する必要がある（ＶＣ対応））
			//
			while (pNaExp != NULL) {
				pNaMac = NULL;
				while (pNaExp != NULL && !(AJCTKIS_SYMBOL(pNaExp->tkn) && (pNaMac = PpcMacGetNode(pW, pNaExp->pSyl)) != NULL)) {
					ppNaBfr = &(pNaExp->pNxt);
					pNaExp	=	pNaExp->pNxt;
				}
				if (pNaExp != NULL && pNaMac != NULL) {
					//	引数付マクロで、次のトークンが '(' 以外ならば、展開しない（関数マクロの呼び出しではない）
					if (((PPPCMACNODE)pNaMac)->fWithArg && (pNaExp->pNxt == NULL || pNaExp->pNxt->tkn != EAJCTK_DLM_LSPART)) {
						//	このトークンをスキップ
						ppNaBfr = &(pNaExp->pNxt);
						pNaExp	=	pNaExp->pNxt;
					}
					//	引数無しマクロか、次のトークンが '(' ならば展開する
					else {
						if (pNaExp->mexp < PPCMAX_MEXP_NEST) {
							if (((PPPCMACNODE)pNaMac)->fWithArg) {
								pNaExp = MacExpWithArg(pW, ppNaBfr, pNaExp, (PPPCMACARGS)pNaMac);
							}
							else {
								pNaExp = MacExpNoArg  (pW, ppNaBfr, pNaExp, (PPPCMACNODE)pNaMac);
							}
						}
						else {
							//	マクロ展開ネストオーバー
							PPC_ERR(AJCPPC_MACERR_NEST_OVER, (UX)pNaExp->pFile, (UX)pNaExp->lno, (UX)(((PPPCMACNODE)pNaMac)->pMacName));
							//	このマクロをスキップ（展開しない）
							ppNaBfr = &(pNaExp->pNxt);
							pNaExp	=	pNaExp->pNxt;
						}
					}
				}
			}
		}
		//----- マクロ展開の前後を連結 -------------//
		if (pExpTop != NULL) {				//	マクロ展開有り？
			//	先頭連結
			*ppBfr	   = pExpTop;
			//	defined解決
			PpcDefExp(pW, ppBfr, *ppBfr);
			//	末尾トークン設定
			pExp = pExpTop;
			while(pExp != NULL && pExp->pNxt != NULL) {
				pExp = pExp->pNxt;
			}
			//	末尾連結
			pExp->pNxt = pSvNxt;
			//	末尾トークンの次の空白有無フラグ設定（マクロ参照のトークンと同じにする）
			pExp->flg |= (flg & AJCTKF_NXTSPC);
		}
		else {								//	マクロ展開なし？
			*ppBfr	= pSvNxt;
			pExpTop = pSvNxt;
		}
		//----------------------------------------------------------------------------------------------------------//
		//	引数リスト消去																							//
		//----------------------------------------------------------------------------------------------------------//
		for (i = 0; i < pMac->nTmpArg; i++) {
			PpcRelTokenStream(pW, pALArr[i].pRelArg);
		}
		PpcMemFree(pW, pALArr);
	}
	else pW->fStop = TRUE;

	return pExpTop;
}
//----- 仮引数チェック -----------------------------------------------------------------------------------------//
static	int		SrhTmpArg(HAJCPPC pW, C_BCP pSyl, PCMEW_LIST pArr, int n)
{
	int		rc = -1;
	int		i;

	for (i = 0; i < n; i++) {
		if (mbscmp(pArr[i].pTmpArg, pSyl) == 0) {
			rc = i;
			break;
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	引数無しマクロの展開																						//
//																												//
//	引　数	：	ppBfr - 直前のトークンから現トークン（マクロ名）へのポインタのアドレス							//
//				pTkn  - 現トークン（マクロ名）のアドレス														//
//				pMac  - マクロボディ定義ノードのアドレス														//
//																												//
//	戻り値	：	≠NULL : マクロ展開トークンあり（展開形の先頭ノードアドレス）									//
//				＝NULL : マクロ展開トークンなし																	//
//--------------------------------------------------------------------------------------------------------------//
static PAJCPPCTKNNODE MacExpNoArg  (HAJCPPC pW, PAJCPPCTKNNODE *ppBfr, PAJCPPCTKNNODE pTkn, PPPCMACNODE pMac)
{
	PAJCPPCTKNNODE	pBdy	= pMac->pTknBody;
	PAJCPPCTKNNODE	pExp	= NULL;
	PAJCPPCTKNNODE	pExpTop = NULL;
	PAJCPPCTKNNODE	pSvNxt	= NULL;
	UB				mexp	= pTkn->mexp + 1;	//	マクロ展開ネストレベル
	UB				flg 	= 0;

	//----- マクロ参照トークンのフラグ退避 -------------------//
	if (*ppBfr != NULL) flg = (*ppBfr)->flg;
	//----- マクロ参照の次トークンアドレス退避 ---------------//
	pSvNxt = pTkn->pNxt;
	//----- マクロボディ・トークンを挿入 ---------------------//
	while (pBdy != NULL) {
		if (pExp == NULL) pExp = pExpTop = PpcTknAlloc(pW);
		else			  pExp = PpcAllocAndLinkNextToken(pW, pExp);
		if (pExp != NULL) {
			//----- ボディ・トークンコピー ----//
			pExp->pNxt	 = NULL;
			pExp->pFile  = pTkn->pFile;
			pExp->lno	 = pTkn->lno;
			pExp->pSyl	 = pBdy->pSyl;
			pExp->tkn	 = pBdy->tkn;
			pExp->kndPP  = AJCPPC_PPK_TOKEN;
			pExp->pos	 = pTkn->pos;
			pExp->ppIf	 = 0;
			pExp->flg	 = (pBdy->flg & (AJCTKF_NXTSPC | AJCTKF_WIDECHAR));
			pExp->inst	 = pTkn->inst;
			pExp->incf	 = pTkn->incf;
			pExp->mexp	 = mexp;
		}
		else pW->fStop = TRUE;
		//----- 次のボディ・トークン ------//
		pBdy = pBdy->pNxt;
	}

	//----- マクロ展開の前後を連結 --------//
	if (pExp != NULL) {				//	マクロ展開有り？
		//	先頭連結
		*ppBfr	   = pExpTop;
		//	defined解決
		PpcDefExp(pW, ppBfr, *ppBfr);
		//	末尾トークン設定
		pExp = pExpTop;
		while(pExp != NULL && pExp->pNxt != NULL) {
			pExp = pExp->pNxt;
		}
		//	末尾連結
		pExp->pNxt = pSvNxt;
		//	末尾トークンの次の空白有無フラグ設定（マクロ参照のトークンと同じにする）
		pExp->flg |= (flg & AJCTKF_NXTSPC);
	}
	else {							//	マクロ展開なし？
		*ppBfr	= pSvNxt;
		pExpTop = pSvNxt;
	}

	//----- マクロ参照トークンノード削除 ---------------------//
	PpcTknFree(pW, pTkn);

	return pExpTop;
}
//--------------------------------------------------------------------------------------------------------------//
//	順番を表す文字列生成																						//
//																												//
//	引　数	：	ｎ 		- 順番の値																				//
//																												//
//	戻り値	：	TRUE : 正常																						//
//				FALSE: 展開エラー発生																			//
//--------------------------------------------------------------------------------------------------------------//
static	BCP		NumStr(HAJCPPC pW, int n)
{
	static	BC		txt[32];

	if (AjcGetLangId() == AJCLID_JPN) {
		AjcSnPrintFA(txt, AJCTSIZE(txt), "%d番目", n);
	}
	else {
		switch (n) {
			case 1:		strcpy(txt, "1st");	break;
			case 2:		strcpy(txt, "2nd");	break;
			case 3:		strcpy(txt, "3rd");	break;
			default:	AjcSnPrintFA(txt, AJCTSIZE(txt), "%dth", n); break;
		}
	}
	return txt;
}
