﻿//
//	AjcPpMain.c - 主制御
//
#include	"AjcInternal.h"
#include	"AjcPp.h"
#include	<time.h>


//--------------------------------------------------------------------------------------------------------------//
//	登録オブジェクト																							//
//--------------------------------------------------------------------------------------------------------------//
typedef struct {
	PAJCPPCTKNNODE	pTknGen;				//	トークンストリーム（生成部分）
	PAJCPPCTKNNODE	pTknNoGen;				//	トークンストリーム（非生成部分，「AJCPPC_FLG_GENALL」指定時に作成
} TKNSTREAMS, *PTKNSTREAMS;

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
static	VO				RegistAnsiMacro			(HAJCPPC pW);
static	PAJCPPCTKNNODE	GetTokenOrPreProStream	(HAJCPPC pW, HAJCTK hCtk, C_BCP pFilePath, EAJCTKCODE *pfPpTkn, UIP pNum);
static	VO				GenerateToken			(HAJCPPC pW);
static	PAJCPPCTKNNODE	GenPreProNode			(HAJCPPC pW, PAJCPPCTKNNODE pTop, UI knd);
static	PAJCPPCTKNNODE	GenDefineNode			(HAJCPPC pW, PAJCPPCTKNNODE pTop);

static	VO				PrePro_define			(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num);
static	VO				PrePro_undef			(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num);
static	UB				PrePro_include			(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num);
static	BOOL			PrePro_ifdef			(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num);
static	BOOL			PrePro_ifndef			(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num);
static	BOOL			PrePro_if				(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num);
static	BOOL			PrePro_elif				(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num);
static	BOOL			PrePro_else				(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num);
static	UI				PrePro_endif			(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num);
static	VO				PrePro_others			(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num);

static	VO				Generate_InIf			(HAJCPPC pW);
static	VO				Generate_InElse			(HAJCPPC pW);
static	VO				SkipTo_elif_else_endif	(HAJCPPC pW);
static	VO				SkipTo_endif			(HAJCPPC pW);

static	BOOL CALLBACK 	cbCtkOptSym				(BCP pBuf, UI lBuf, UX xp);
static	BOOL CALLBACK 	cbCtkFile  				(BCP pBuf, UI lBuf, UX xp);

static	BOOL			SearchIncFile			(HAJCPPC pW, C_BCP fname, BCP path, int len);
static	BOOL CALLBACK 	cbASearchIncFile		(UI nest, BCP pPath, BCP pName, UI attrib, UI wtime, UX cbp);

//==============================================================================================================//
//	インスタンス生成																							//
//																												//
//	引　数	：	pBasePath		- #include検索時のベースパス	（不要時はNULL)									//
//				pIncPath		- #include検索パス群			（不要時はNULL)									//
//				pOptSym			- オプション・シンボル群		（不要時はNULL)									//
//				cbp				- コールバックパラメタ															//
//				cbNtc			- イベント通知用コールバック関数（不要時はNULL)									//
//				cbErr	  	 	- エラー通知用コールバック関数	（不要時はNULL)									//
//																												//
//	戻り値	：	≠NULL : 正常（インスタンスハンドル）															//
//				＝NULL : エラー																					//
//==============================================================================================================//
AJCEXPORT	HAJCPPC		WINAPI	AjcPpcCreate(	C_BCP 		 pBasePath,
												PAJCLBXITEMA pIncPath,
												PAJCLBXITEMA pOptSym,
												UX			 cbp,
												VO (CALLBACK *cbNtc)(AJCPPCNOTIFY ntc, UX p1, UX p2, UX p3, UX cbp),
												VO (CALLBACK *cbErr)(AJCPPCERROR  err, UX p1, UX p2, UX p3, UX cbp))
{
	HAJCPPC		pW = NULL;
	BOOL		fErr = FALSE;
	BC			relp[MAX_PATH * 2];

	if (pW = AJCMEM(sizeof(AJCPPC))) {
		memset(pW, 0, sizeof(AJCPPC));
		do {
			//----- インスタンスＩＤ設定 ------------------------------------------//
			pW->InstID = INST_ID_PP;
			//----- コールバック設定 ----------------------------------------------//
			pW->cbp   = cbp;
			pW->cbNtc = cbNtc;
			pW->cbErr = cbErr;
			//----- 管理用ハンドル生成 --------------------------------------------//
			if (PpcIStrInit(pW) == NULL) {fErr = TRUE; break;}		//	文字列プール（大小区別なし）
			if (PpcNStrInit(pW) == NULL) {fErr = TRUE; break;}		//	文字列プール（大小区別あり）
			//----- オプションフラグ設定 ------------------------------------------//
			pW->OptFlag  = AJCPPC_FLG_AUTO_SEARCH | AJCPPC_FLG_ONCE;
			//----- テキストファイルエンコード設定 --------------------------------//
			pW->InpTec = AJCTEC_MBC;
			pW->OutTec = AJCTEC_MBC;
			pW->fBOM   = FALSE;
			pW->InpActTec = (EAJCTEC)-1;
			pW->InpActBOM = 		 -1;
			//----- Includeファイル検索用ベースパス設定 ---------------------------//
			if (pBasePath != NULL && *pBasePath != 0)	{
				//	相対パスの場合は、カレントディレクトリからの相対とする
				if (PathIsRelativeA(pBasePath)) {
					GetCurrentDirectoryA(AJCTSIZE(relp), relp);
					AjcPathCatA		    (relp, pBasePath, AJCTSIZE(relp));
					PathCanonicalizeA(pW->BasePath, relp);
				}
				//	絶対パスの場合は、パスを正規化する
				else {
					PathCanonicalizeA(pW->BasePath, pBasePath);
				}
			}
			//----- インクルードパス群退避（コピー）-------------------------------//
			if (pIncPath != NULL) {
				PAJCLBXITEMA	p = pIncPath;
				PAJCLBXITEMA	q = NULL;
				UI				n = 0;
				UI				lTxt;

				while (p->pStr != NULL) {
					n++;
					p++;
				}
				if (pW->pIncArr = AJCMEM(sizeof(AJCLBXITEMA) * (n + 1))) {
					memset(pW->pIncArr, 0, sizeof(AJCLBXITEMA) * (n + 1));
					p = pIncPath;
					q = pW->pIncArr;
					while (p->pStr != NULL) {
						lTxt = (UI)strlen(p->pStr);
						if (q->pStr = AjcTAllocA(lTxt + 1)) {
							strcpy(q->pStr, p->pStr);
							q->data = p->data;
						}
						else {
							fErr = TRUE;
							goto apca_exit;
						}
						p++;
						q++;
					}
				}
			}
			//----- オプションシンボル群退避（コピー）-----------------------------//
			if (pOptSym != NULL) {
				PAJCLBXITEMA	p = pOptSym;
				PAJCLBXITEMA	q = NULL;
				UI				n = 0;
				UI				lTxt;

				while (p->pStr != NULL) {
					n++;
					p++;
				}
				if (pW->pOptArr = AJCMEM(sizeof(AJCLBXITEMA) * (n + 1))) {
					memset(pW->pOptArr, 0, sizeof(AJCLBXITEMA) * (n + 1));
					p = pOptSym;
					q = pW->pOptArr;
					while (p->pStr != NULL) {
						lTxt = (UI)strlen(p->pStr);
						if (q->pStr = AjcTAllocA(lTxt + 1)) {
							strcpy(q->pStr, p->pStr);
						}
						else {
							fErr = TRUE;
							goto apca_exit;
						}
						p++;
						q++;
					}
				}
			}
		} while(0);

apca_exit:;

		//----- エラーならばリソース解放 ------------------------------------------//
		if (fErr) {
			AjcPpcDelete(pW);
			pW = NULL;
		}
	}

	return pW;
}
//==============================================================================================================//
//	インスタンス消去																							//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcPpcDelete	(HAJCPPC pW)
{
	BOOL	rc = FALSE;

	if (IS_PP_INST(pW)) {
		//	コンパイルオブジェクト 消去
		AjcPpcReleaseObject(pW);

		//	管理用ハンドル解放
		PpcIStrEnd(pW);
		PpcNStrEnd(pW);
		PpcMacEnd (pW);
		PpcIncEnd (pW);
		//	インクルードパス群解放
		if (pW->pIncArr != NULL) {
			PAJCLBXITEMA	p = pW->pIncArr;
			while (p->pStr != NULL) {
				free(p->pStr);
				p++;
			}
			free(pW->pIncArr);
		}
		//	オプションシンボル群解放
		if (pW->pOptArr != NULL) {
			PAJCLBXITEMA	p = pW->pOptArr;
			while (p->pStr != NULL) {
				free(p->pStr);
				p++;
			}
			free(pW->pOptArr);
		}
		//	インスタンスワーク解放
		free(pW);
		//	戻り値＝ＯＫ
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	オプションフラグ設定																						//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				flag		- オプションフラグ(AJCPPC_FLG_XXXXX)												//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcPpcSetOption(HAJCPPC pW, UI opt)
{
	BOOL	rc = FALSE;

	if (IS_PP_INST(pW)) {
		pW->OptFlag = opt;
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	オプションフラグ取得																						//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//																												//
//	戻り値	：	オプションフラグ(AJCPPC_FLG_XXXXX)																//
//==============================================================================================================//
AJCEXPORT	UI	WINAPI	AjcPpcGetOption(HAJCPPC pW)
{
	UI		rc = AJCPPC_FLG_NONE;
	if (IS_PP_INST(pW)) {
		rc = pW->OptFlag;
	}
	return rc;
}
//==============================================================================================================//
//	プリプロセス・コンパイル																					//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				pSrcPath	- ソースプログラムパス																//
//																												//
//	戻り値	：	＝AJCPPCR_OK : ＯＫ																				//
//				≠AJCPPCR_OK : トークンが１つも無い／エラー														//
//==============================================================================================================//
AJCEXPORT	AJCPPCRESULT	WINAPI	AjcPpcCompile(	HAJCPPC pW, C_BCP pSrcPath)
{
	AJCPPCRESULT	rc	= AJCPPCR_PARAM;
	PAJCPPCINCNEST	pTi = NULL;
	EAJCTKCODE		tkn;
	PAJCPPCTKNNODE	pTop;
	BCP				pObjPath = NULL;
	UI				num;
	BOOL			fNoFile = FALSE;
	BC				fname[_MAX_FNAME], fext[_MAX_EXT];
	BC				SrcPath[MAX_PATH];
	BC				relp[MAX_PATH * 2];

	if (IS_PP_INST(pW)) {
		if (pSrcPath != NULL && *pSrcPath != 0 ) {
			do {
				//	前回のプリコンパイルオブジェクト解放
				AjcPpcReleaseObject(pW);

				//	ワーク初期化
				pTi						= &pW->TblInc[0];
				pW->incf				= AJCPPC_INCF_NO;
				pW->IxInc				= 0;
				memset(&pW->TblInc, 0, sizeof pW->TblInc);

				pW->pOptSymText			= NULL;		//	オプションシンボルへのポインタ
				pW->pTknTop 			= NULL;		//	先頭トークンノードアドレス
				pW->pTknLas 			= NULL;		//	最終トークンノードアドレス
				pW->ppMacBfr 			= NULL;		//	マクロ展開済トークンノードポインタのアドレス
				pW->pNoGTop				= NULL;		//	先頭トークンノードアドレス（非生成部分のトークン）
				pW->pNoGLas				= NULL;		//	最終トークンノードアドレス（非生成部分のトークン）
				pW->fMemErr				= FALSE;	//	メモリエラーフラグ
				pW->fStop				= FALSE;	//	解析中止フラグ
				pW->pSrhPath			= NULL;		//	ファイル検索パス

				//	#if～#endifネスト情報初期化
				pTi->IxIF = 0;
				memset(pTi->TblIF, 0, sizeof pTi->TblIF);
				pTi->TblIF[0].knd = AJCPPC_IFK_NONE;
				pTi->TblIF[0].act = AJCPPC_IFA_GEN_NORMAL;

				//	管理用ハンドル生成
				if (PpcMacInit (pW) == NULL) break;			//	マクロ定義
				if (PpcIncInit (pW) == NULL) break;			//	インクルード名
				//	ソースファイルパス設定
				//	相対パスの場合は、ベースパスからの相対とする
				if (PathIsRelativeA(pSrcPath)) {
					AjcSnPrintFA(relp, AJCTSIZE(relp), "%s", pW->BasePath);
					AjcPathCatA (relp, pSrcPath, AJCTSIZE(relp));
					PathCanonicalizeA(SrcPath, relp);
				}
				//	絶対パスの場合は、パスを正規化する
				else {
					PathCanonicalizeA(SrcPath, pSrcPath);
				}
				//	ソースファイル・オープン	
				if ((pTi->fh = AjcFOpenA(SrcPath, pW->InpTec)) == NULL) {
					fNoFile = TRUE;
					PPC_ERR(AJCPPC_ERROR_SRC_OPEN, SrcPath, 0, 0);
					break;
				}
				//	ソースファイルの実際のエンコード，ＢＯＭ設定
				pW->InpActTec = AjcFGetTec(pTi->fh);
				pW->InpActBOM = AjcFGetBOM(pTi->fh);
				//	ソースファイルのエンコード通知
				PPC_NTC(AJCPPC_NTC_SRCTEC, SrcPath, pW->InpActTec, pW->InpActBOM);
				//	字句分解インスタンス生成
				pTi->hCtk = PpcCtkCreate(pW, cbCtkFile);
				if (pTi->hCtk == NULL) {pW->fStop = TRUE; break;}
				//	#includeネスト制御テーブル初期化（先頭エントリにソースファイル設定）
				pTi->pPath = PpcIStrRegist(pW, SrcPath);
				if (pTi->pPath == NULL) {pW->fStop = TRUE; break;}
				_splitpath(SrcPath	, NULL, NULL, fname, fext);
				_makepath (pTi->file, NULL, NULL, fname, fext);
				pTi->lno = 0;
				//	暗黙定義済マクロ登録
				RegistAnsiMacro(pW);
				//	オプションシンボル登録
				if (pW->pOptArr != NULL) {
					PAJCLBXITEMA pos = pW->pOptArr;
					HAJCTK		hctk;
					while (pos->pStr != NULL) {
						if (pW->fStop) break;	//	中止
						pW->pOptSymText = pos->pStr;
						if (hctk = PpcCtkCreate(pW, cbCtkOptSym)) {
							if (pTop = GetTokenOrPreProStream(pW, hctk, "<OptionSymbol>", &tkn, &num)) {
								if (tkn == EAJCTK_RSV_DEFINE) PrePro_define(pW, hctk, pTop, num);
								else						  PrePro_others(pW, hctk, pTop, num);
							}
							AjcCtkDelete(hctk);
						}
						else pW->fStop = TRUE;
						pos++;
					}
				}
				if (pW->fStop) break;

				//	トークンストリーム生成
				GenerateToken(pW);

			} while(0);
		}

		//	残ファイル，リソース解放
		while (pW->IxInc >= 0) {
			pTi = &pW->TblInc[pW->IxInc];
			if (pTi->fh != NULL) {
				AjcFClose(pTi->fh);
				pTi->fh = NULL;
			}
			if (pTi->hCtk != NULL) {
				AjcCtkDelete(pTi->hCtk);
				pTi->hCtk = NULL;
			}
			pW->IxInc--;
		}

		//	戻り値設定
		if		(fNoFile			) rc = AJCPPCR_NOFILE;
		else if (pW->fMemErr		) rc = AJCPPCR_MEMERR;
		else if (pW->fStop			) rc = AJCPPCR_STOP;
		else if (pW->pTknTop == NULL) rc = AJCPPCR_NOTOKEN;
		else						  rc = AJCPPCR_OK;
	}
	return rc;
}
//==============================================================================================================//
//	プリコンパイル済オブジェクトの取得																			//
//																												//
//	引　数	ppNoGen	- 非生成部分のトークンストリームの先頭アドレスを格納するバッファのアドレス（不要時はNULL）	//
//																												//
//	戻り値	：	≠NULL : 生成部分のトークンストリームの先頭アドレス												//
//				＝NULL : トークンが１つも無い／エラー															//
//==============================================================================================================//
AJCEXPORT	PAJCPPCTKNNODE	WINAPI	AjcPpcGetObject			(HAJCPPC pW, PAJCPPCTKNNODE *ppNoGen)
{
	PAJCPPCTKNNODE	rc = NULL;

	if (IS_PP_INST(pW)) {
		if (ppNoGen != NULL) *ppNoGen = pW->pNoGTop;
		rc = pW->pTknTop;
	}
	return rc;
}
//==============================================================================================================//
//	プリコンパイル済オブジェクトの解放																			//
//																												//
//	引　数		なし																							//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	BOOL			WINAPI	AjcPpcReleaseObject		(HAJCPPC pW)
{
	BOOL	rc = FALSE;

	if (IS_PP_INST(pW)) {
		//	生成部分のトークンストリーム
		if (pW->pTknTop != NULL) {PpcRelTokenStream(pW, pW->pTknTop); pW->pTknTop = pW->pTknLas = NULL;}
		//	非生成部分のトークンストリーム
		if (pW->pNoGTop != NULL) {PpcRelTokenStream(pW, pW->pNoGTop); pW->pNoGTop = pW->pNoGLas = NULL;}
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	マクロ名の列挙																								//
//																												//
//	引　数		cbp			 - コールバックパラメタ																//
//				cbNtcMacName - マクロ名通知用コールバック関数（不要時はNULL)									//
//																												//
//	戻り値	：	マクロ名の個数																					//
//==============================================================================================================//
typedef struct {
	UX			cbp;
	BOOL (CALLBACK *cb)(PCAJCPPCMACINFO pMacInfo, UX cbp);
} CBP_ENUM_MACRO, *PCBP_ENUM_MACRO;
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	CALLBACK cbPpcEnumMacro(UX key, VOP pNodeData, UI len, UI nest, UX cbp)
{
	BOOL			rc 		 = TRUE;
	PCBP_ENUM_MACRO	p		 = (PCBP_ENUM_MACRO)cbp;
	PPPCMACARGS		pMacNode = (PPPCMACARGS)pNodeData;
	AJCPPCMACINFO	mi;

	if (p->cb != NULL) {
		memset(&mi, 0, sizeof mi);
		mi.fWithArg		= pMacNode->fWithArg;			//	引数付マクロ・フラグ（FALSE:引数無し，TRUE:引数あり）
		mi.pMacName		= pMacNode->pMacName;			//	マクロ名文字列へのポインタ
		mi.pTknName		= pMacNode->pTknName;			//	マクロ名・トークンノードへのポインタ
		mi.pTknBody		= pMacNode->pTknBody;			//	マクロボディ・トークンノードへのポインタ
		mi.pos			= pMacNode->pos;				//	先頭「#」の桁位置
		if (pMacNode->fWithArg) {						//	引数付きマクロ？
			mi.fVaArgs		= pMacNode->fVaArgs;		//		可変個引数マクロフラグ
			mi.nTmpArg		= pMacNode->nTmpArg;		//		仮引数の個数
			mi.hVQueArg		= pMacNode->hVQueArg;		//		マクロ仮引数名リスト
		}
		rc = p->cb(&mi, p->cbp);
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcPpcEnumMacro	(HAJCPPC pW, UX cbp, BOOL (CALLBACK *cbNtcMacName)(PCAJCPPCMACINFO pMacInfo, UX cbp))
{
	UI				rc = 0;
	CBP_ENUM_MACRO	prm;

	if (IS_PP_INST(pW)) {
		if (pW->hAvlMac != NULL) {
			prm.cbp   = cbp;
			prm.cb	  = cbNtcMacName;
			rc = AjcAvlEnumNodesEx(pW->hAvlMac, (UX)&cbp, cbPpcEnumMacro, FALSE);
		}
	}
	return rc;
}
//==============================================================================================================//
//	マクロ定義の取得																							//
//																												//
//	引　数		pMacName	- マクロ名																			//
//				pMacInfo	- マクロ定義情報を格納するバッファ（不要時はNULL)									//
//																												//
//	戻り値	：	TRUE   : ＯＫ																					//
//				FALSE  : マクロが見つからない																	//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcPpcGetMacroInfo			(HAJCPPC pW, C_BCP pMacName, PAJCPPCMACINFO pMacInfo)
{
	BOOL			rc		 = FALSE;
	PPPCMACARGS		pMacNode = NULL;

	if (IS_PP_INST(pW)) {
		if (pMacNode = (PPPCMACARGS)PpcMacGetNode(pW, pMacName)) {
			if (pMacInfo != NULL) {
				memset(pMacInfo, 0, sizeof(AJCPPCMACINFO));
				pMacInfo->fWithArg		= pMacNode->fWithArg;			//	引数付マクロ・フラグ（FALSE:引数無し，TRUE:引数あり）
				pMacInfo->pMacName		= pMacNode->pMacName;			//	マクロ名文字列へのポインタ
				pMacInfo->pTknName		= pMacNode->pTknName;			//	マクロ名・トークンノードへのポインタ
				pMacInfo->pTknBody		= pMacNode->pTknBody;			//	マクロボディ・トークンノードへのポインタ
				pMacInfo->pos			= pMacNode->pos;				//	先頭「#」の桁位置
				if (pMacNode->fWithArg) {								//	引数付きマクロ？
					pMacInfo->fVaArgs		= pMacNode->fVaArgs;		//		可変個引数マクロフラグ
					pMacInfo->nTmpArg		= pMacNode->nTmpArg;		//		仮引数の個数
					pMacInfo->hVQueArg		= pMacNode->hVQueArg;		//		マクロ仮引数名リスト
				}
			}
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	プリコンパイル中止																							//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI		AjcPpcStop(HAJCPPC pW)
{
	BOOL	rc = FALSE;

	if (IS_PP_INST(pW)) {
		pW->fStop = TRUE;
		rc = TRUE;
	}
	return rc;
}


//==============================================================================================================//
//	トークンストリームのファイル出力																			//
//																												//
//	引　数	：	pOut	- 出力ファイルパス																		//
//				fExpInc	- TRUE : インクルードした内容を展開する													//
//						  FALSE: #include分をそのまま残す（インクルードした内容は展開しない）					//
//				CommOutOfkndPP - コメント出力するプリプロセス文を指定する（AJCPPC_PPK_XXXXX）					//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
//	生成部分のトークンストリームと、非生成部分のトークンストリームをマージ										//
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	VO	SubMerge(HAJCPPC pW)
{
	PAJCPPCTKNNODE	pTknCur;					//	生成部分  のノードポインタ
	PAJCPPCTKNNODE	pTknBfr;					//	 〃 	  の１つ前のノードポインタ
	PAJCPPCTKNNODE	pNoGCur;					//	非生成部分のノードポインタ
	PAJCPPCTKNNODE	pNoGBfr;					//	 〃 	  の１つ前のノードポインタ
	PAJCPPCTKNNODE	pSvCurNxt, pSvNoGBfr;
	UI				seq;

	//	エントリ数表示（デバッグ用）
	#ifdef _DEBUG
//	{	int				i;
//		PAJCPPCTKNNODE	p;
//		AjcTrace("----- TokenEntries ----------------------\n");
//		for (p = pW->pTknTop, i = 0; p != NULL; p = p->pNxt, i++);
//		AjcTrace("Tkn : %d entries\n", i);
//		for (p = pW->pNoGTop, i = 0; p != NULL; p = p->pNxt, i++);
//		AjcTrace("NoG : %d entries\n", i);
//	}
	#endif
	//	生成エントリ表示（デバッグ用）
	#ifdef _DEBUG
//	{	int				i;
//		PAJCPPCTKNNODE	p;
//		AjcTrace("----- Generate entries ----------------------\n");
//		for (p = pW->pTknTop, i = 1; p != NULL; p = p->pNxt, i++) {
//			AjcTrace("seq=%4d : ppIf = %04X, syl = '%s'\n", p->seq, p->ppIf, p->pSyl);
//		}
//	}
	#endif
	//	非生成エントリ表示（デバッグ用）
	#ifdef _DEBUG
//	{	int				i;
//		PAJCPPCTKNNODE	p;
//		AjcTrace("----- No generate entries -------------------\n");
//		for (p = pW->pNoGTop, i = 1; p != NULL; p = p->pNxt, i++) {
//			AjcTrace("seq=%4d : ppIf = %04X, syl = '%s'\n", p->seq, p->ppIf, p->pSyl);
//		}
//	}
	#endif

	pTknCur = pW->pTknTop;
	pNoGCur = pW->pNoGTop;
	pTknBfr = pNoGBfr = NULL;

	while (pNoGCur != NULL) {
		//	生成部分の末尾へ残りの非生成部分を追加
		if (pTknCur == NULL) {
			if (pTknBfr == NULL) pW->pTknTop   = pNoGCur;
			else				 pTknBfr->pNxt = pNoGCur;
			pNoGCur = NULL;
		}
		//	生成部分の途中へ挿入
		else {
			//	生成部分の２つのＳＥＱ＃設定
			if (pTknCur->pNxt == NULL) seq = 0x7FFFFFFF;
			else					   seq = pTknCur->pNxt->seq;
			//	挿入位置ならば、非生成部分を挿入
			if (pNoGCur->seq > pTknCur->seq && pNoGCur->seq < seq) {
				//	非生成部分の直前ポインタ退避
				pSvNoGBfr = pNoGBfr;
				//	生成部分の次のノードポインタ退避
				pSvCurNxt = pTknCur->pNxt;
				//	挿入位置へ非生成部分の先頭を連結
				pTknCur->pNxt = pNoGCur;
				//	挿入範囲終端設定
				while (pNoGCur != NULL && pNoGCur->seq < seq) {
					pNoGBfr = pNoGCur;
					pNoGCur = pNoGCur->pNxt;
				}
				//	挿入部分終端へ生成部分の次ノード設定
				pNoGBfr->pNxt = pSvCurNxt;
				//	次の検索基点ポインタ設定
				pTknBfr = pNoGBfr;
				pTknCur = pTknBfr->pNxt;
				//	挿入部分（非生成部分）の直前ポインタ回復
				pNoGBfr = pSvNoGBfr;
				if (pNoGBfr == NULL) pW->pNoGTop   = pNoGCur;
				else				 pNoGBfr->pNxt = pNoGCur;
			}
			//	挿入位置でなければ、生成部分末尾ポインタ更新
			else {
				pTknBfr = pTknCur;
				pTknCur = pTknCur->pNxt;
			}
		}
	}
	//	生成部分終端ポインタ設定
	pTknCur   = pW->pTknTop;
	while (pTknCur != NULL) {
		pW->pTknLas = pTknCur;
		pTknCur 	= pTknCur->pNxt;
	}
	//	非生成部分無しを設定
	pW->pNoGLas = pW->pNoGLas = NULL;

	//	マージエントリ表示（デバッグ用）
	#ifdef _DEBUG
//	{	int				i;
//		PAJCPPCTKNNODE	p;
//		AjcTrace("----- Marge entries ----------------------\n");
//		for (p = pW->pTknTop, i = 1; p != NULL; p = p->pNxt, i++) {
//			AjcTrace("seq=%4d : ppIf = %04X, syl = '%s'\n", p->seq, p->ppIf, p->pSyl);
//		}
//	}
	#endif

}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
//	生成部分のトークンストリームと、非生成部分のトークンストリームを分離する									//
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	VO	SubSeparate(HAJCPPC pW)
{
	PAJCPPCTKNNODE	pTknCur = pW->pTknTop;
	PAJCPPCTKNNODE	pNoGCur = pW->pNoGTop;
	PAJCPPCTKNNODE	pTknBfr = NULL;
	PAJCPPCTKNNODE	pSvNxt	= NULL;

	pW->pNoGTop = pW->pNoGLas = NULL;

	while (pTknCur != NULL) {
		//	非生成部分のノード検出
		if (pTknCur->ppIf == AJCPPIF_NO_GEN) {
			do {
				//	非生成ノードを外す
				if (pTknBfr == NULL) pW->pTknTop   = pTknCur->pNxt;
				else				 pTknBfr->pNxt = pTknCur->pNxt;
				//	現ノードポインタ退避
				pSvNxt = pTknCur->pNxt;
				//	非生成キュー初回更新
				if (pW->pNoGTop == NULL) {
					pTknCur->pNxt = NULL;
					pW->pNoGTop   = pTknCur;
					pW->pNoGLas   = pTknCur;
				}
				//	非生成キュー2回目以降更新
				else {
					pTknCur->pNxt	  = NULL;
					pW->pNoGLas->pNxt = pTknCur;
					pW->pNoGLas 	  = pTknCur;
				}
				//	現ノードポインタ設定
				pTknCur = pSvNxt;
			} while (pTknCur != NULL && pTknCur->ppIf == AJCPPIF_NO_GEN);
		}
		//	生成部分のノード検出
		else {
			//	ポインタ更新
			pTknBfr 	= pTknCur;
			pTknCur 	= pTknCur->pNxt;
		}
	}

	//	エントリ数表示（デバッグ用）
	#ifdef _DEBUG
//	{	int				i;
//		PAJCPPCTKNNODE	p;
//		AjcTrace("----- TokenEntries ----------------------\n");
//		for (p = pW->pTknTop, i = 0; p != NULL; p = p->pNxt, i++);
//		AjcTrace("Tkn : %d entries\n", i);
//		for (p = pW->pNoGTop, i = 0; p != NULL; p = p->pNxt, i++);
//		AjcTrace("NoG : %d entries\n", i);
//	}
	#endif

}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
//	行頭情報出力（ "/* IncNest; MacNest; FileName; LineNo */ " ）												//
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	VO	WriteLineTopInfo(HAJCPPC pW, PAJCPPCTKNNODE pCur)
{
	UI		stl;
	BC		fname[_MAX_FNAME], fext[_MAX_EXT];
	BC		name[MAX_PATH];

	_splitpath(pCur->pFile, NULL, NULL, fname, fext);					//	ファイル名作成
	_makepath (name 	  , NULL, NULL, fname, fext);					//	・
	stl = (UI)strlen(name);												//	ファイル名長設定
	AjcFPrintFA(pW->fhTsoOut, "/*%2d; %2d; %s;%*s %5d; */ ",			//	ネスト，ファイル名，行番号表示
			pCur->inst, pCur->mexp, name, pW->TsoMaxFName - stl, "", pCur->inst == 0 ? pW->TsoNst0Lno : pCur->lno);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
//	語句を出力 	（行頭ならば、行頭に余白を付加）																//
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	UI	WriteSyl(HAJCPPC pW, PAJCPPCTKNNODE pCur)
{
	UI		rc = 0;		//	戻り値＝出力桁数
	UI		i;

	if (pW->fTsoTop) {																				//	行頭？
		AjcFPrintFA(pW->fhTsoOut, "    ");															//		"    " 出力
		for (i = 0; i < pCur->pos; i++) {AjcFPrintFA(pW->fhTsoOut, " "); rc++;}						//		先頭位置調整
	}
	AjcFPrintFA(pW->fhTsoOut, "%s", pCur->pSyl); rc += (UI)strlen(pCur->pSyl);						//	語句出力
	pW->fTsoTop = FALSE;																			//	「行頭以外」

	return rc;
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
//	語句をコメント出力 ( 行頭ならば、行頭に "//" 付加 )															//
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	UI	WriteLineComment(HAJCPPC pW, PAJCPPCTKNNODE pCur)
{
	UI		rc = 0;		//	戻り値＝出力桁数
	UI		i;

	if (pW->fTsoTop) {																				//	行頭？
		if (pCur->ppIf == AJCPPIF_NO_GEN) AjcFPrintFA(pW->fhTsoOut, "//- ");						//		"//" 出力
		else							  AjcFPrintFA(pW->fhTsoOut, "//  ");
		for (i = 0; i < pCur->pos; i++) {AjcFPrintFA(pW->fhTsoOut, " "); rc++;}						//		先頭位置調整
	}
	AjcFPrintFA(pW->fhTsoOut, "%s", pCur->pSyl); rc += (UI)strlen(pCur->pSyl);						//	語句をコメント出力

	if (pCur->kndPP & (AJCPPC_PPK_ALLIF | AJCPPC_PPK_ELSE)) {										//	#if, #elsif, #ifdef, #ifndef, #else 判定結果表示
		for (i = rc; i < 32; i++) AjcFPrintFA(pW->fhTsoOut, " ");									//		表示位置調整
		if		(pCur->ppIf == AJCPPIF_UNCOND) AjcFPrintFA(pW->fhTsoOut, "  // --> During FALSE cond.");
		else if (pCur->ppIf) 				   AjcFPrintFA(pW->fhTsoOut, "  // --> TRUE");
		else								   AjcFPrintFA(pW->fhTsoOut, "  // --> FALSE");
	}
	else if (pCur->kndPP & AJCPPC_PPK_ENDIF) {														//	#endif に対応する #ifdef.#ifndef/#if の行番号表示
		if (pCur->lno > pCur->ppIf) {
			for (i = rc; i < 32; i++) AjcFPrintFA(pW->fhTsoOut, " ");								//		表示位置調整
			if (pCur->ppIf != 0) AjcFPrintFA(pW->fhTsoOut, "  // --> %d" 			, pCur->lno - pCur->ppIf);
			else				 AjcFPrintFA(pW->fhTsoOut, "  // --> ??? (Before %d)", pCur->lno - 65530 	);
		}
	}

	pW->fTsoTop = FALSE;																			//	「行頭以外」

	return rc;
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
//	マクロ定義を出力																							//
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	UI	SubMacStreamOut(HAJCPPC pW, PAJCPPCTKNNODE pMacTop, BOOL fExpInc, BOOL fComment)
{
	UI				rc	   = 0;
	PAJCPPCTKNNODE	pCur   = NULL;			//	マクロ定義の先頭トークン
	UI				svLno  = pMacTop->lno;	//	行番号退避
	UI				col    = 0;				//	桁位置
	UI				MaxCol = 0;				//	最大桁位置
	UI				stl;

	if (pMacTop != NULL) {
		//	最大桁位置設定
		pCur = pMacTop->pNxt;
		while (pCur != NULL && pCur->kndPP == AJCPPC_PPK_DEFCONT && !pW->fStop) {
			stl = (UI)strlen(pCur->pSyl);
			MaxCol = __max(MaxCol, pCur->pos + stl + 2);
			pCur = pCur->pNxt;
		}
		MaxCol = __min(MaxCol, 132);
		//	マクロ識別ノード(AJCPPC_PPK_DEFINE)スキップ
		pCur = pMacTop->pNxt;
		//	マクロ定義ノード(AJCPPC_PPK_DEFCONT)出力
		while (pCur != NULL && pCur->kndPP == AJCPPC_PPK_DEFCONT && !pW->fStop) {
			//	行番号が異なる場合、改行
			if (svLno != pCur->lno) {
				//	改行出力
				if (fComment) {													//	コメント化？
					for (; col < MaxCol; col++) AjcFPrintFA(pW->fhTsoOut, " ");	//		行末桁位置調整
					AjcFPrintFA(pW->fhTsoOut, " \\ //\n");						//		"\ //" 出力＆改行
				}
				else {															//	非コメント？
					AjcFPrintFA(pW->fhTsoOut, " \\\n");							//		"\"    出力＆改行
				}
				//	インクルード展開モードならば、行頭情報出力
				if (fExpInc) {													//	インクルード内容展開？
					WriteLineTopInfo(pW, pCur);									//		行頭情報出力
				}
				//	ネスト０ソースの行番号設定
				if (pCur->inst == 0) {											//	ネスト０（ソースファイル）？
					pW->TsoNst0Lno = pCur->lno + 1;								//		ネスト０行番号設定
				}
				col = 0;
				pW->fTsoTop = TRUE;												//	「行頭」
				rc++;															//	出力行数更新
			}
			//	行頭ならば、行頭空白と語句を出力
			if (pW->fTsoTop) {
					if (fComment) col = WriteLineComment(pW, pCur);				//	'//' + 語句文字列出力（コメント化時）
					else		  col = WriteSyl		(pW, pCur);				//	語句文字列出力		 （非コメント化）
			}
			//	行頭以外ならば、語句を出力
			else {
				AjcFPrintFA(pW->fhTsoOut, "%s", pCur->pSyl); col += (UI)strlen(pCur->pSyl);
			}
			//	語句直後の空白出力
			if (PpcNeedSpace(pCur)) {
				AjcFPrintFA(pW->fhTsoOut, " "); col++;
			}
			//	行番号退避
			svLno = pCur->lno;
			//	トークンポインタ更新
			pCur = pCur->pNxt;
		}
		//	改行
		AjcFPrintFA(pW->fhTsoOut, " \n");
		pW->fTsoTop = TRUE;
		rc++;
	}
	return rc;		//	戻り値＝マクロ定義の行数
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
//	インクルード内容 展開時のファイル出力																		//
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	VO	SubToFile_ExpInc  (HAJCPPC pW, UI CommOutOfkndPP)
{
	PAJCPPCTKNNODE	pTkn  = pW->pTknTop;
	PAJCPPCTKNNODE	pCur  = pW->pTknTop;
	PAJCPPCTKNNODE	pBfr  = NULL;
	BCP				pFCur = "";
	UI				lno   = 1;
	UI				sv_lno = -1;

	//	ネスト０の行番号（＝ソースファイルの行番号）初期化
	pW->TsoNst0Lno = 1;
	//	注釈出力
	AjcFPrintFA(pW->fhTsoOut, "/* \"%s\" */\n", pTkn != NULL ? pTkn->pFile : "");
	AjcFPrintFA(pW->fhTsoOut, "/* Include-Nest; Macro-Nest; SourceFile; LineNumber; */\n");

	//	プリコンパイルソースファイル出力
	while (pCur != NULL && !pW->fStop) {
		//	新たな行（ファイル名／行番号が異なる）ならば、改行
		if (mbscmp(pCur->pFile, pFCur) != 0  ||  pCur->lno != sv_lno) {					//	ファイル名／行番号が異なる？
			//	行頭以外ならば改行
			if (!pW->fTsoTop) {															//		行頭以外？
				AjcFPrintFA(pW->fhTsoOut, "\n");										//			改行
				pW->fTsoTop = TRUE;														//			行頭の旨フラグセット
			}
			//	行頭情報を出力し、ネスト０ソースの行数を合わせる
			if (pCur->inst == 0) {														//		ネスト０（ソースファイル）？
				while (pW->TsoNst0Lno < pCur->lno) {									//			ダミーの改行出力
					AJCPPCTKNNODE	cur;												//			・
					memcpy(&cur, pCur, sizeof cur);										//			・
					cur.mexp = 0;														//			・
					WriteLineTopInfo(pW, &cur);											//			・
					AjcFPrintFA(pW->fhTsoOut, "\n");									//			・
					pW->TsoNst0Lno++;													//			ネスト０（ソースファイル）の行番号更新
				}
			}
			WriteLineTopInfo(pW, pCur);													//		行頭情報出力
			if (pCur->inst == 0) {														//		ネスト０（ソースファイル）？
				pW->TsoNst0Lno = pCur->lno + 1;											//			ネスト０行番号設定
			}
			pW->fTsoTop = TRUE;															//		「行頭」
			pFCur		= pCur->pFile;													//		ファイル名退避
			sv_lno		= pCur->lno;
		}
		//	通常のトークン出力
		if (pCur->kndPP == AJCPPC_PPK_TOKEN) {											//	通常のトークン？
			if (pW->fTsoTop) {															//		行頭？
				if (pCur->ppIf != AJCPPIF_NO_GEN) WriteSyl		  (pW, pCur);			//			語句文字列出力			  （生成部分）
				else					  		  WriteLineComment(pW, pCur);			//			語句文字列を行コメント出力（非生成部分）
			}
			else {																		//		行頭以外？
				if (PpcNeedSpace(pBfr)) {												//			空白挿入要？
					AjcFPrintFA(pW->fhTsoOut, " ");										//				空白出力
				}
				if (pCur->ppIf != AJCPPIF_NO_GEN) WriteSyl		  (pW, pCur);			//			語句文字列出力			  （生成部分）
				else							  WriteLineComment(pW, pCur);			//			語句文字列を行コメント出力（非生成部分）
			}
		}
		//	プリプロセス出力（#define以外のプリプロセス文は全体を１行で出力する）
		else {																			//	プリプロセス文？
			if (pCur->kndPP == AJCPPC_PPK_OTHERS) {										//		#pragma等？
				WriteSyl(pW, pCur);														//			#pragma等をファイルへ出力
				pW->fTsoTop = FALSE;
			}
			else if (pCur->kndPP == AJCPPC_PPK_INCLUDE) {								//		#include？
				int		i, rc;
				rc = WriteLineComment(pW, pCur);										//			プリプロセス文をコメント出力
				if (pCur->incf & AJCPPC_INCF_DUP) {
					for (i = rc; i <= 32; i++) AjcFPrintFA(pW->fhTsoOut, " ");			//				表示位置調整
					AjcFPrintFA(pW->fhTsoOut, " // Already included.");					//				Already included.
				}
			}
			else if (pCur->kndPP == AJCPPC_PPK_DEFINE) {								//		#define？
				UI	MacLines = SubMacStreamOut(pW, pCur, TRUE, 							//			マクロ定義出力
											CommOutOfkndPP & AJCPPC_PPK_DEFINE);		//			・
			}
			else if (pCur->kndPP & CommOutOfkndPP) {									//		コメント出力するプリプロセス文(#undef, #if等～#endif)？
				WriteLineComment(pW, pCur);												//			プリプロセス文をコメント出力
			}
			else {																		//		そのまま出力するプロセス文(#undef, #if等～#endif)？
				WriteSyl(pW, pCur);														//			プリプロセス文をそのまま出力
			}
		}
		pBfr = pCur;																	//	直前のトークン設定
		pCur = pCur->pNxt;																//	現在のトークン更新
		//	マクロ定義部分スキップ
		while (pCur != NULL && pCur->kndPP == AJCPPC_PPK_DEFCONT) {
			pBfr = pCur;
			pCur = pCur->pNxt;
		}

		PPC_NTC(AJCPPC_NTC_OUTLOOP, 0, 0, 0);											//	トークンストリームをファイルへ出力ループ中通知
	}
	AjcFPrintFA(pW->fhTsoOut, "\n");
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
//	インクルード内容 非展開時のファイル出力																		//
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	VO	SubToFile_NoExpInc(HAJCPPC pW, UI CommOutOfkndPP)
{
	PAJCPPCTKNNODE	pTkn  = pW->pTknTop;
	PAJCPPCTKNNODE	pCur  = pW->pTknTop;
	PAJCPPCTKNNODE	pBfr  = NULL;
	BCP				pFCur = "";
	UI				lno   = 1;
	UI				sv_lno = -1;

	//	プリコンパイルソースファイル出力
	while (pCur != NULL && !pW->fStop) {
		//	新たな行（ファイル名／行番号が異なる）ならば、改行
		if (mbscmp(pCur->pFile, pFCur) != 0  ||  pCur->lno != sv_lno) {					//	ファイル名／行番号が異なる？
			//	改行を行い、元のソースと行番号を合わせる
			if (pCur->inst == 0) {														//		INCLUDE ネスト＝０（元ソース）？
				while (lno < pCur->lno) {												//				改行数を合わせる
					AjcFPrintFA(pW->fhTsoOut, "\n"); lno++;								//				・
				}
			}
			pW->fTsoTop = TRUE;															//		「行頭」
			pFCur		= pCur->pFile;													//		ファイル名退避
			sv_lno		= pCur->lno;													//		行番号退避
		}
		//	通常のトークン出力
		if (pCur->kndPP == AJCPPC_PPK_TOKEN) {											//	通常のトークン？
			if (pCur->inst == 0) {														//		INCLUDE ネスト＝０（元ソース）？
				if (pW->fTsoTop) {														//			行頭？
					if (pCur->ppIf != AJCPPIF_NO_GEN) WriteSyl		  (pW, pCur);		//			語句文字列出力			  （生成部分）
					else							  WriteLineComment(pW, pCur);		//			語句文字列を行コメント出力（非生成部分）
				}
				else {																	//			行頭以外？
					if (PpcNeedSpace(pBfr)) {											//				空白挿入要？
						AjcFPrintFA(pW->fhTsoOut, " ");									//					空白出力
					}
					if (pCur->ppIf != AJCPPIF_NO_GEN) WriteSyl		  (pW, pCur);		//			語句文字列出力			  （生成部分）
					else					  		  WriteLineComment(pW, pCur);		//			語句文字列を行コメント出力（非生成部分）
				}
			}
		}
		//	プリプロセス出力（#define以外のプリプロセス文は全体を１行で出力する）
		else {																			//	プリプロセス文？
			if (pCur->inst == 0) {														//		INCLUDE ネスト＝０（元ソース）？
				if (pCur->kndPP == AJCPPC_PPK_OTHERS) {									//			#pragma等？
						WriteSyl(pW, pCur);												//					#pragma等をファイルへ出力
						pW->fTsoTop = FALSE;
				}
				else if (pCur->kndPP == AJCPPC_PPK_INCLUDE) {							//			#include？
						int		i, rc;
						rc = WriteSyl(pW, pCur);										//					ファイルへ「#include」出力
						if (pCur->incf & AJCPPC_INCF_DUP) {
							for (i = rc; i < 32; i++) AjcFPrintFA(pW->fhTsoOut, " ");	//				表示位置調整
							AjcFPrintFA(pW->fhTsoOut, " // Already included.");		
						}
				}
				else if (pCur->kndPP == AJCPPC_PPK_DEFINE) {							//			#define？
					UI	MacLines = SubMacStreamOut(pW, pCur, FALSE,						//				マクロ定義出力
											CommOutOfkndPP & AJCPPC_PPK_DEFINE);		//				・
					lno += MacLines;													//				行番号更新
				}
				else if (pCur->kndPP & CommOutOfkndPP) {								//			コメント出力するプリプロセス(#undef, #if等～#endif)？
					WriteLineComment(pW, pCur);											//				プリプロセス文をコメント出力
				}
				else {																	//			そのまま出力するプロセス文(#undef, #if等～#endif)？
					WriteSyl(pW, pCur);													//				プリプロセス文をそのまま出力
					pW->fTsoTop = FALSE;												//				行頭以外の旨フラグセット
				}
			}
		}
		pBfr = pCur;																	//	直前のトークン設定
		pCur = pCur->pNxt;																//	現在のトークン更新
		//	マクロ定義部分スキップ
		while (pCur != NULL && pCur->kndPP == AJCPPC_PPK_DEFCONT) {
			pBfr = pCur;
			pCur = pCur->pNxt;
		}

		PPC_NTC(AJCPPC_NTC_OUTLOOP, 0, 0, 0);											//	トークンストリームをファイルへ出力ループ中通知
	}
	AjcFPrintFA(pW->fhTsoOut, "\n");
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcPpcTokenStreamToFile(HAJCPPC pW, C_BCP pOut, BOOL fExpInc, UI CommOutOfkndPP)
{
	BOOL			rc	  = FALSE;
	PAJCPPCTKNNODE	pCur  = NULL;
	UI				stl;
	BC				fname[_MAX_FNAME], fext[_MAX_EXT];
	BC				name   [MAX_PATH];
	BC				OutPath[MAX_PATH];
	BC				relp   [MAX_PATH * 2];

	if (IS_PP_INST(pW)) {
		//	有効なフラグのみとする
		CommOutOfkndPP &= AJCPPC_PPK_ALL;

		//	pW->fStop = FALSE;

		//	出力ファイルパス設定
		//	相対パスの場合は、ベースパスからの相対とする
		if (PathIsRelativeA(pOut)) {
			AjcSnPrintFA(relp, AJCTSIZE(relp), "%s", pW->BasePath);
			AjcPathCatA (relp, pOut, AJCTSIZE(relp));
			PathCanonicalizeA(OutPath, relp);
		}
		//	絶対パスの場合は、パスを正規化する
		else {
			PathCanonicalizeA(OutPath, pOut);
		}

		if (pW->fhTsoOut = AjcFCreateA(OutPath, pW->OutTec, pW->fBOM)) {
			//	非生成部分を出力する場合、生成トークンストリームと、非生成トークンストリームをマージする
			if (pW->OptFlag & AJCPPC_FLG_GENALL) {
				SubMerge(pW);
			}
			//	ファイル名の最大長算出
			pCur = pW->pTknTop;
			while (pCur != NULL) {
				_splitpath(pCur->pFile, NULL, NULL, fname, fext);
				_makepath (name 	  , NULL, NULL, fname, fext);
				stl 			= (UI)strlen(name);
				pW->TsoMaxFName = __max(pW->TsoMaxFName, stl);
				pCur			= pCur->pNxt;
			}

			//	プリコンパイル済ソースファイル出力
			if (fExpInc) {
				SubToFile_ExpInc  (pW, CommOutOfkndPP);
			}
			else {
				SubToFile_NoExpInc(pW, CommOutOfkndPP);
			}
			//	出力ファイルクローズ
			AjcFClose(pW->fhTsoOut);

			//	非生成部分を出力する場合、生成トークンストリームと、非生成トークンストリームを分離する
			if (pW->OptFlag & AJCPPC_FLG_GENALL) {
				SubSeparate(pW);
			}
			//	ファイル出力終了通知
			PPC_NTC(AJCPPC_NTC_OUTLOOP, 1, 0, 0);
			//	戻り値＝ＯＫ
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	トークンストリームをファイル出力する際のテキストエンコード設定												//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				tec		- 入力テキストエンコード																//
//																												//
//	戻り値：	TRUE  - 正常																					//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcPpcSetTecAtTokenStreamToFile(HAJCPPC pW, EAJCTEC tec, BOOL fBOM)
{
	BOOL		rc = FALSE;

	if (IS_PP_INST(pW)) {
		if (MAJCTEC_VALID(tec)) {
			pW->OutTec = tec;
			rc		   = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	ソースファイルエンコード設定																				//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				tec		- テキストエンコード																	//
//																												//
//	戻り値：	TRUE  - 正常																					//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT BOOL	 WINAPI	AjcPpcSetTextEncode(HAJCPPC pW, EAJCTEC tec)
{
	BOOL		rc = FALSE;

	if (IS_PP_INST(pW)) {
		if (MAJCTEC_VALID(tec)) {
			pW->InpTec = tec;
			rc		   = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	ソースファイルエンコード取得																				//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				pTec	- ソースファイルのテキストエンコードを格納するバッファのアドレス	（不要時はNULL）	//
//				pfBOM	- ソースファイルのBOM出力フラグを格納するバッファのアドレス			（不要時はNULL）	//
//																												//
//	戻り値：	TRUE  - 正常																					//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT BOOL	 WINAPI	AjcPpcGetTextEncode(HAJCPPC pW, PEAJCTEC pTec, BOOL *pfBOM)
{
	BOOL	rc = FALSE;

	if (IS_PP_INST(pW)) {
		if (pTec  != NULL) *pTec  = pW->InpActTec;
		if (pfBOM != NULL) *pfBOM = pW->InpActBOM;
		rc = TRUE;
	}
	return rc;
}

//--------------------------------------------------------------------------------------------------------------//
//	暗黙定義マクロ登録																							//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	なし																							//
//--------------------------------------------------------------------------------------------------------------//
static	VO		RegistAnsiMacro(HAJCPPC pW)
{
	time_t 			t;
	EAJCTKCODE		tkn;
	UI				num;
	PAJCPPCTKNNODE	pTop;
	HAJCTK			hctk;
	BC				ts[32], dt[32], tm[32];
	BC				mac[64];

	time(&t);
	// TIMESTAMP 文字列設定
	AjcSnPrintFA(ts, AJCTSIZE(ts), "%s", ctime(&t));
	mbstok(ts, "\n");
	// DATE 文字列設定
	memset(dt, ' ', sizeof dt);
	memcpy(&dt[ 0], &ts[ 4], 7);
	memcpy(&dt[ 7], &ts[20], 4);
	dt[11] = 0;
	// TIME 文字列設定
	_strtime(tm);

	// __DATE_
	AjcSnPrintFA(mac, AJCTSIZE(mac), "__DATE__=\"%s\"", dt);
	pW->pOptSymText = mac;
	if (hctk = PpcCtkCreate(pW, cbCtkOptSym)) {
		pTop = GetTokenOrPreProStream(pW, hctk, "<ANSI-C>", &tkn, &num);
		PrePro_define(pW, hctk, pTop, num);
		AjcCtkDelete(hctk);
	}
	// __TIMESTAMP__
	AjcSnPrintFA(mac, AJCTSIZE(mac), "__TIMESTAMP__=\"%s\"", ts);
	pW->pOptSymText = mac;
	if (hctk = PpcCtkCreate(pW, cbCtkOptSym)) {
		pTop = GetTokenOrPreProStream(pW, hctk, "<ANSI-C>", &tkn, &num);
		PrePro_define(pW, hctk, pTop, num);
		AjcCtkDelete(hctk);
	}
	// __TIME__
	AjcSnPrintFA(mac, AJCTSIZE(mac), "__TIME__=\"%s\"", tm);
	pW->pOptSymText = mac;
	if (hctk = PpcCtkCreate(pW, cbCtkOptSym)) {
		pTop = GetTokenOrPreProStream(pW, hctk, "<ANSI-C>", &tkn, &num);
		PrePro_define(pW, hctk, pTop, num);
		AjcCtkDelete(hctk);
	}
}

//--------------------------------------------------------------------------------------------------------------//
//	トークンストリーム生成																						//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	なし																							//
//--------------------------------------------------------------------------------------------------------------//
static	VO	GenerateToken(HAJCPPC pW)
{
	UI				seq = 0;
	PAJCPPCINCNEST	pTI = NULL;
	EAJCTKCODE		TknKnd;
	PAJCPPCTKNNODE	pTkn, pCur;
	UI				num;
	BC				fname[_MAX_FNAME], fext[_MAX_EXT];
	BC				txt[MAX_PATH + 2];

	if (!pW->fStop) {
		while (pW->IxInc >= 0) {
			if (pW->fStop) break;	//	中止
			pTI = &pW->TblInc[pW->IxInc];
			if ((pTkn = GetTokenOrPreProStream(pW, pTI->hCtk, pTI->pPath, &TknKnd, &num)) != NULL) {
				//----- ファイル名，行番号通知 -----------------------------------------------------------------//
				PPC_NTC(AJCPPC_NTC_FILE_LNO, pTI->file, pTI->lno, pW->IxInc);

				if (TknKnd == 0) {										//	通常の単一トークン（プリプロセス以外）？
					if (pTI->TblIF[pTI->IxIF].act & AJCPPC_IFA_GEN) {	//		生成状態？
						//----- ANSI暗黙マクロ処理 -------------------------------------------------------------//
						if (pTkn->tkn == EAJCTK_USR_NAME) {				//			ユーザ名？
							if		(mbscmp(pTkn->pSyl, "__FILE__") == 0) {				//「__FILE__」？
								pTkn->tkn  = EAJCTK_STR_QUOTE;
								_splitpath(pTkn->pFile, NULL, NULL, fname, fext);
								txt[0] = '"';
								_makepath (&txt[1]	  , NULL, NULL, fname, fext);
								strcat(txt, "\"");
								pTkn->pSyl = PpcNStrRegist(pW, txt);
								if (pTkn->pSyl == NULL) pW->fStop = TRUE;
							}
							else if (mbscmp(pTkn->pSyl, "__LINE__") == 0) {				//「__LINE__」
								pTkn->tkn  = EAJCTK_VAL_DECIMAL;
								AjcSnPrintFA(txt, AJCTSIZE(txt), "%d", pTkn->lno);
								pTkn->pSyl = PpcNStrRegist(pW, txt);
								if (pTkn->pSyl == NULL) pW->fStop = TRUE;
							}
						}
						//----- トークンポインタ設定 -----------------------------------------------------------//
						if (pTkn->pSyl != NULL) {						//					語句ポインタ設定ＯＫ？
							//	トークンを末尾へリンク
							if (pW->pTknTop == NULL) {
								pW->pTknTop  = pTkn;
								pW->pTknLas  = pTkn;
								pCur		 = pTkn;
								pW->ppMacBfr = &pW->pTknTop;
							}
							else {
								pCur->pNxt	 = pTkn;
								pW->pTknLas  = pTkn;
								pCur		 = pCur->pNxt;
							}
						}
						else {											//					語句ポインタ設定失敗？
							PpcTknFree(pW, pTkn);						//						トークン破棄
						}
					}
					else {												//				非生成状態？
						//	非生成部分もファイル出力？
						if (pW->OptFlag & AJCPPC_FLG_GENALL) {
							pTkn->seq = seq++;
							//	トークン・ノードを非生成ストリームへ追加（非生成部分は、常に通常のトークンストリームとする）
							if (pW->pNoGTop == NULL) pW->pNoGTop = pW->pNoGLas = pTkn;
							else			   pW->pNoGLas = pW->pNoGLas->pNxt = pTkn;
							pTkn->ppIf = AJCPPIF_NO_GEN;	//	非生成部分をマーク
						}
						//	非生成部分はファイル出力しない
						else {
							//	トークン・ノード解放
							PpcTknFree(pW, pTkn);
						}
					}
				}
				else {											//	プリプロセス・トークンストリーム？
					//----- プリプロセス文ノード追加 -----------------------------------------------------------//
					{	PAJCPPCTKNNODE	pPre = NULL;
						//----- #if～#endif, #pragma は生成状態に関係なくプリプロセスノードを追加 --------------//
						switch (TknKnd) {
							case EAJCTK_RSV_DEFINE:		break;
							case EAJCTK_RSV_UNDEF:		break;
							case EAJCTK_RSV_INCLUDE:	break;
							case EAJCTK_RSV_IFDEF:		pPre = GenPreProNode(pW, pTkn, AJCPPC_PPK_IFDEF	);	break;
							case EAJCTK_RSV_IFNDEF:		pPre = GenPreProNode(pW, pTkn, AJCPPC_PPK_IFNDEF);	break;
							case EAJCTK_RSV_IF:			pPre = GenPreProNode(pW, pTkn, AJCPPC_PPK_IF	);	break;
							case EAJCTK_RSV_ELIF:		pPre = GenPreProNode(pW, pTkn, AJCPPC_PPK_ELIF	);	break;
							case EAJCTK_RSV_ELSE:		pPre = GenPreProNode(pW, pTkn, AJCPPC_PPK_ELSE	);	break;
							case EAJCTK_RSV_ENDIF:		pPre = GenPreProNode(pW, pTkn, AJCPPC_PPK_ENDIF	);	break;
							default:					break;
						}
						//----- #if～#endif, #pragma 以外は生成状態時のみプリプロセスノードを追加 --------------//
						if (pTI->TblIF[pTI->IxIF].act & AJCPPC_IFA_GEN) {	//		生成状態？
							switch (TknKnd) {
								case EAJCTK_RSV_DEFINE:		pPre = GenDefineNode(pW, pTkn);						break;
								case EAJCTK_RSV_UNDEF:		pPre = GenPreProNode(pW, pTkn, AJCPPC_PPK_UNDEF	);	break;
								case EAJCTK_RSV_INCLUDE:	pPre = GenPreProNode(pW, pTkn, AJCPPC_PPK_INCLUDE);	break;
								case EAJCTK_RSV_IFDEF:		break;
								case EAJCTK_RSV_IFNDEF:		break;
								case EAJCTK_RSV_IF:			break;
								case EAJCTK_RSV_ELIF:		break;
								case EAJCTK_RSV_ELSE:		break;
								case EAJCTK_RSV_ENDIF:		break;
								default:					pPre = GenPreProNode(pW, pTkn, AJCPPC_PPK_OTHERS);	break;
							}
						}
						//----- トークンポインタ設定 --------------------------//
						if (pPre != NULL) {		//	プリプロセスノード生成？
							if (pW->pTknTop == NULL) {
								pW->pTknTop  = pCur = pPre;
								pW->ppMacBfr = &pW->pTknTop;
							}
							else {
								pCur->pNxt = pPre;
								pCur	   = pCur->pNxt;
							}
						}
						//----- ここまで（前回のマクロ展開直後～検出したプリプロセス文の直前）のマクロ展開 ---------//
						if (!pW->fStop) {
							PAJCPPCTKNNODE *ppBfr = NULL;
							PAJCPPCTKNNODE	pMExp = NULL;
							if (pW->ppMacBfr != NULL) {
								//	先頭ノード退避
								ppBfr = pW->ppMacBfr;
								//	マクロ展開
								while (*pW->ppMacBfr != NULL) {
									if (pW->fStop) break;	//	中止
									if (((*pW->ppMacBfr)->flg & AJCTKF_PREPRO) || !PpcMacExpand(pW, pW->ppMacBfr)) {
										pW->ppMacBfr = &((*pW->ppMacBfr)->pNxt);
									}
								}
								//	最終トークンポインタ設定
								if ((pCur = pW->pTknTop) != NULL) {
									if (*pW->ppMacBfr != NULL) pCur = *pW->ppMacBfr;
									while (pCur->pNxt != NULL) {
										pCur = pCur->pNxt;
									}
									pW->pTknLas  = pCur;
									pW->ppMacBfr = &(pCur->pNxt);
								}
								//	ノード番号付与
								pMExp = *ppBfr;
								while (pMExp != NULL) {
									pMExp->seq = seq++;
									pMExp = pMExp->pNxt;
								}
							}
						}
						//----- プリプロセス文の処理 ---------------------------------------------------------------//
						if (pW->pTknLas != NULL) {
							if (pTI->TblIF[pTI->IxIF].act & AJCPPC_IFA_GEN) {	//		生成状態？
								switch (TknKnd) {
									case EAJCTK_RSV_DEFINE:							PrePro_define	(pW, pTI->hCtk, pTkn, num);		break;
									case EAJCTK_RSV_UNDEF:							PrePro_undef	(pW, pTI->hCtk, pTkn, num);		break;
									case EAJCTK_RSV_INCLUDE:	pCur->incf		  =	PrePro_include	(pW, pTI->hCtk, pTkn, num);		break;
									case EAJCTK_RSV_IFDEF:		pW->pTknLas->ppIf = PrePro_ifdef	(pW, pTI->hCtk, pTkn, num);		break;
									case EAJCTK_RSV_IFNDEF:		pW->pTknLas->ppIf = PrePro_ifndef	(pW, pTI->hCtk, pTkn, num);		break;
									case EAJCTK_RSV_IF:			pW->pTknLas->ppIf = PrePro_if		(pW, pTI->hCtk, pTkn, num);		break;
									case EAJCTK_RSV_ELIF:		pW->pTknLas->ppIf = PrePro_elif		(pW, pTI->hCtk, pTkn, num);		break;
									case EAJCTK_RSV_ELSE:		pW->pTknLas->ppIf = PrePro_else		(pW, pTI->hCtk, pTkn, num);		break;
									case EAJCTK_RSV_ENDIF:		pW->pTknLas->ppIf = PrePro_endif	(pW, pTI->hCtk, pTkn, num);		break;
									default:										PrePro_others	(pW, pTI->hCtk, pTkn, num);		break;
								}
							}
							else {												//		非生成状態？
								switch (TknKnd) {
									default:				//	#pragma等その他のプリプロセス
									case EAJCTK_RSV_DEFINE:	//	PpcRelTokenStream(pW, pTkn);	break;
									case EAJCTK_RSV_UNDEF:	//	PpcRelTokenStream(pW, pTkn);	break;
									case EAJCTK_RSV_INCLUDE://	PpcRelTokenStream(pW, pTkn);	break;
										//	非生成部分もファイル出力？
										if (pW->OptFlag & AJCPPC_FLG_GENALL) {
											//	非生成部分は、常に通常のトークンストリームとする
											while (pTkn != NULL) {
												//	SEQ付与
												pTkn->seq = seq++;
												//	Includeネスト付与
												pTkn->inst = pW->IxInc;
												//	プリプロセス文のトークンストリームを非生成ストリームへ追加
												if (pW->pNoGTop == NULL) pW->pNoGTop = pW->pNoGLas = pTkn;
												else			   pW->pNoGLas = pW->pNoGLas->pNxt = pTkn;
												pTkn->ppIf = AJCPPIF_NO_GEN;	//	非生成部分をマーク
												pTkn = pTkn->pNxt;
											}
										}
										//	非生成部分はファイル出力しない
										else {
											//	プリプロセス文のトークンストリーム解放
											PpcRelTokenStream(pW, pTkn);
										}
										break;

									case EAJCTK_RSV_IFDEF:		pW->pTknLas->ppIf = PrePro_ifdef	(pW, pTI->hCtk, pTkn, num);		break;
									case EAJCTK_RSV_IFNDEF:		pW->pTknLas->ppIf = PrePro_ifndef	(pW, pTI->hCtk, pTkn, num);		break;
									case EAJCTK_RSV_IF:			pW->pTknLas->ppIf = PrePro_if		(pW, pTI->hCtk, pTkn, num);		break;
									case EAJCTK_RSV_ELIF:		pW->pTknLas->ppIf = PrePro_elif		(pW, pTI->hCtk, pTkn, num);		break;
									case EAJCTK_RSV_ELSE:		pW->pTknLas->ppIf = PrePro_else		(pW, pTI->hCtk, pTkn, num);		break;
									case EAJCTK_RSV_ENDIF:		pW->pTknLas->ppIf = PrePro_endif	(pW, pTI->hCtk, pTkn, num);		break;
								//	default:										PrePro_others	(pW, pTI->hCtk, pTkn, num);		break;
								}
							}
						}
					}
				}
			}
			else {	// EOF?
				AjcFClose	(pTI->fh  );  pTI->fh	= NULL;
				AjcCtkDelete(pTI->hCtk);  pTI->hCtk = NULL;
				//----- #if～#endif ネストチェック ---------//
				if (pTI->IxIF != 0) {
					PPC_ERR(AJCPPC_ERROR_COND_NOTCLS, pTI->pPath, 1, 0);
				}
				//----- Includeネスト減算 ------------------//
				pW->IxInc--;
				if (pW->IxInc >= 0) {
					pTI = &pW->TblInc[pW->IxInc];
				}
				else  {
					pTI = NULL;
				}
			}
		}
		//----- 末尾部分（最後に検出したプリプロセス文以降）のマクロ展開 ---------------------------------------//
		if (!pW->fStop && pW->ppMacBfr != NULL) {
			PAJCPPCTKNNODE *ppMExp = NULL;		//	マクロ展開直言のノードから展開ノードへのポインタのアドレス
			PAJCPPCTKNNODE   pMExp = NULL;		//	展開ノードポインタ
			//	先頭ノード退避
			ppMExp = pW->ppMacBfr;
			//	マクロ展開
			while (*pW->ppMacBfr != NULL) {
				if (pW->fStop) break;
				PpcMacExpand(pW, pW->ppMacBfr);
				if (*pW->ppMacBfr == NULL) break;
				pW->ppMacBfr = &((*pW->ppMacBfr)->pNxt);
			}
			//	最終トークンポインタ設定
			if ((pCur = pW->pTknTop) != NULL) {
				while (pCur->pNxt != NULL) {
					pCur = pCur->pNxt;
				}
				pW->pTknLas  = pCur;
			}
			//	展開したノード以降にノード番号付与
			pMExp = *ppMExp;
			while (pMExp != NULL) {
				pMExp->seq = seq++;
				pMExp = pMExp->pNxt;
			}
		}
	}
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	プリプロセス文(#define以外)情報ノード出力																	//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	PAJCPPCTKNNODE	GenPreProNode(HAJCPPC pW, PAJCPPCTKNNODE pTop, UI knd)
{
	PAJCPPCTKNNODE	pNode = NULL;
	PAJCPPCINCNEST	pTi   = &pW->TblInc[pW->IxInc];
	PAJCPPCTKNNODE	pCur, pBfr;
	PAJCPPCTKNNODE	pC	, pB  ;
	UI				ix = 0;
	UI				stl;
	UI				lOrg = 0, lPre= 0, lTxt = 0;
	BCP				pOrg = NULL;		//	マクロ展開前の原文テキスト
	BCP				pPre = NULL;		//	プリプロセスの先頭部分（ex. "#pragma "）
	BCP				pTxt = NULL;		//	プリプロセス文展開テキスト

	//----- コンパイルオプションシンボル通知(#if, #elif, #ifdef or #ifndef のみ) -------------------------------//
	if (knd == AJCPPC_PPK_IF || knd == AJCPPC_PPK_ELIF || knd == AJCPPC_PPK_IFDEF || knd == AJCPPC_PPK_IFNDEF) {
		if (pTop != NULL) {	
			PAJCPPCTKNNODE p = pTop;
			while (p != NULL) {
				if (p->tkn == EAJCTK_USR_NAME) {
					PPC_NTC(AJCPPC_NTC_OPTSYM, (UX)p, 0, 0);
				}
				p = p->pNxt;
			}
		}
	}

	//----- マクロ展開前の原文テキストをコメント形式で設定(#if, #elif のみ) ------------------------------------//
	if (knd == AJCPPC_PPK_IF || knd == AJCPPC_PPK_ELIF) {
		//	テキスト長設定
		pCur = pTop;
		pBfr = NULL;
		lOrg = 0;
		while (pCur != NULL) {
			//	空白分加算
			if (PpcNeedSpace(pBfr)) {
				lOrg++;
			}
			//	語句テキスト長加算
			lOrg += (UI)strlen(pCur->pSyl);
			//	トークンポインタ更新
			pBfr = pCur;
			pCur = pCur->pNxt;
		}
		lOrg += 16;		//	" /* (" と ") */" 分
		lOrg++;			//	
		//	原文テキスト設定
		if (pOrg = (BCP)PpcMemAlloc(pW, lOrg)) {
			pCur = pTop;
			pBfr = NULL;
			//	テキストインデクス初期化
			ix = 0;
			//	コメント開始を追加
			strcpy(&pOrg[ix], " /* (");
			ix += 5;
			//	原文テキスト設定
			while (pCur != NULL) {
				//	空白挿入
				if (PpcNeedSpace(pBfr)) {
					pOrg[ix++] = ' ';
				}
				//	語句テキスト挿入
				stl = (UI)strlen(pCur->pSyl);
				strcpy(&pOrg[ix], pCur->pSyl);
				ix += stl;
				//	トークンポインタ更新
				pBfr = pCur;
				pCur = pCur->pNxt;
			}
			//	コメント終端を追加
			strcpy(&pOrg[ix], ") */");
			ix += 4;
			//	テキスト終端
			pOrg[ix] = 0;
		}
	}

	//----- プリプロセスの先頭部分の長さ設定 -------------------------------------------------------------------//
	pCur = pTop;
	lPre  = 0;
	if (pCur != NULL && pCur->pSyl != NULL) {lPre  = (UI)strlen(pCur->pSyl); pCur = pCur->pNxt;}
	if (pCur != NULL && pCur->pSyl != NULL) {lPre += (UI)strlen(pCur->pSyl);				   }
	lPre += 2;	//	空白と終端分
	if (pPre = (BCP)PpcMemAlloc(pW, lPre)) {
		//----- プリプロセスの先頭部分生成 ---------------------------------------------------------------------//
		//	マクロ展開の都合上、pCur = <pre-process-name> で止めておく
		pCur = pTop;
		pBfr = NULL;
		ix	 = 0;
		if (pCur != NULL && pCur->pSyl != NULL) {stl = (UI)strlen(pCur->pSyl); strcpy(&pPre[ix], pCur->pSyl); ix += stl; pCur = pCur->pNxt;}	//	#
		if (pCur != NULL && pCur->pSyl != NULL) {stl = (UI)strlen(pCur->pSyl); strcpy(&pPre[ix], pCur->pSyl); ix += stl; 				   }	//	preprocess symbol
	//	pPre[ix++] = ' ';																														//	' '
		pPre[ix  ] = 0;
		//----- マクロ展開（#include, #if, #elif, #pragma等）---------------------------------------------------//
		if (pCur != NULL && ((knd == AJCPPC_PPK_INCLUDE && pCur->pNxt != NULL && pCur->pNxt->tkn != EAJCTK_DLM_LO) ||	//	#includeの次が '<' 以外 or
							  knd == AJCPPC_PPK_IF || knd == AJCPPC_PPK_ELIF || knd == AJCPPC_PPK_OTHERS)) {			//	#if, #elif or others(#pragma, #error等)
			PAJCPPCTKNNODE	pExp;
			PAJCPPCTKNNODE	*ppBfr = &pCur->pNxt;
			//	'defined(XXX)'の解決
			PpcDefExp(pW, ppBfr, pCur->pNxt);
			//	<pre-process-name>の次以降マクロ展開
			while (*ppBfr != NULL) {
				PpcMacExpand(pW, ppBfr);
				if (*ppBfr == NULL) break;
				ppBfr = &((*ppBfr)->pNxt);
			}
			//	マクロ展開したトークンに「プリプロセス」をマークする
			pExp = pCur;
			while (pExp != NULL) {
				pExp->flg |= AJCTKF_PREPRO;
				pExp = pExp->pNxt;
			}
			//	pCur = <pre-process-name> をスキップする
			pBfr = pCur;
			pCur = pCur->pNxt;
		}
		//----- #if, #elif, #pragma以外の場合、pCur = <pre-process-name> をスキップする-------------------------//
		else if (pCur != NULL) {
			//	pCur = <pre-process-name> をスキップする
			pBfr = pCur;
			pCur = pCur->pNxt;
		}
		//----- プリプロセステキスト長算出 ---------------------------------------------------------------------//
		lTxt = (UI)strlen(pPre) + lOrg;
		pC = pCur;
		pB = pBfr;
		while (pC != NULL) {
			//	空白分加算
			if (PpcNeedSpace(pB)) {
				lTxt++;
			}
			//	語句テキスト長加算
			stl = (UI)strlen(pC->pSyl);
			lTxt += stl;
			//	ポインタ更新
			pB = pC;
			pC = pC->pNxt;
		}
		lTxt++;
		//----- プリプロセステキストメモリ確保 -----------------------------------------------------------------//
		if (pTxt = (BCP)PpcMemAlloc(pW, lTxt)) {
			//----- プリプロセステキスト作成 -------------------------------------------------------------------//
			strcpy(pTxt, pPre);
			ix = (UI)strlen(pTxt);
			while (pCur != NULL) {
				//	空白挿入
				if (PpcNeedSpace(pBfr)) {
					pTxt[ix++] = ' ';
				}
				//	語句テキストコピー
				stl = (UI)strlen(pCur->pSyl);
				strcpy(&pTxt[ix], pCur->pSyl);
				ix += stl;
				//	ポインタ更新
				pBfr = pCur;
				pCur = pCur->pNxt;
			}
			//----- マクロ展開前の原文テキスト追加(#if, #elifのみ) ---------------------------------------------//
			if (pOrg != NULL) {
				stl = (UI)strlen(pOrg);
				strcpy(&pTxt[ix], pOrg);
				ix += stl;
			}
			//	テキスト終端
			pTxt[ix] = 0;
			//----- プリプロセス文情報ノード追加 ---------------------------------------------------------------//
			if (pNode = PpcTknAlloc(pW)) {
				pNode->pFile  = PpcIStrRegist(pW, pTop->pFile);
				pNode->pSyl   = PpcNStrRegist(pW, pTxt);
				pNode->tkn	  = EAJCTK_PREPRO;
				pNode->kndPP  = (UW)knd;
				pNode->lno	  = pTop->lno;
				pNode->pos	  = pTop->pos;
				pNode->ppIf	  = 0;
				pNode->flg	  = 0;
				pNode->inst   = pW->IxInc;
				pNode->incf	  = pTop->incf;
				pNode->mexp   = pTop->mexp;
				//	ファイル名／語句 登録失敗ならばノード破棄
				if (pNode->pFile == NULL || pNode->pSyl == NULL) {
					PpcTknFree(pW, pNode);
					pNode = NULL;
					pW->fStop = TRUE;
				}
			}
			PpcMemFree(pW, pTxt);
		}
		PpcMemFree(pW, pPre);
	}
	else pW->fStop = TRUE;

	//----- マクロ展開前テキスト解放 ----------------------------//
	if (pOrg != NULL) {
		free(pOrg);
	}

	return pNode;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	プリプロセス文(#define)情報ノード出力																		//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	PAJCPPCTKNNODE	GenDefineNode(HAJCPPC pW, PAJCPPCTKNNODE pTop)
{
	PAJCPPCTKNNODE	pNode = NULL;
	PAJCPPCTKNNODE	pCur  = NULL;
	PAJCPPCTKNNODE	pNew  = NULL;
	PAJCPPCTKNNODE	pBfr  = NULL;
	BOOL			fErr = FALSE;

	//----- pCur = マクロ名トークンノード ----------------------------------------------------------------------//
	pCur = pTop;
	pCur = pCur->pNxt;	//	'#' 	 スキップ
	pCur = pCur->pNxt;	//	'define' スキップ
	//----- #define文プリプロセスノード追加 --------------------------------------------------------------------//
	if (pNode = PpcTknAlloc(pW)) {
		pNode->pFile  = PpcIStrRegist(pW, pTop->pFile);
		pNode->pSyl   = PpcNStrRegist(pW, pCur->pSyl);		//	pSyl = "マクロ名"
		pNode->tkn	  = EAJCTK_PREPRO;
		pNode->kndPP  = AJCPPC_PPK_DEFINE;
		pNode->lno	  = pTop->lno;
		pNode->pos	  = pTop->pos;
		pNode->ppIf	  = 0;
		pNode->flg	  = 0;
		pNode->inst   = pW->IxInc;
		pNode->incf	  = pTop->incf;
		pNode->mexp   = pTop->mexp;
		//	pNxt以降にマクロ定義原文のコピーを連結
		pCur = pTop;
		pBfr = NULL;
		while (pCur != NULL) {
			if (pNew = PpcTknAlloc(pW)) {
				memcpy(pNew, pCur, sizeof(AJCPPCTKNNODE));
				pNew->kndPP  = AJCPPC_PPK_DEFCONT;
				if (pBfr == NULL) {
					pNode->pNxt = pNew;
				}
				else {
					pBfr->pNxt	= pNew;
				}
			}
			else {
				fErr = TRUE;
				break;
			}
			pBfr = pNew;
			pCur = pCur->pNxt;
		}
		//	ノード作成失敗ならばノード破棄
		if (fErr || pNode->pFile == NULL || pNode->pSyl == NULL) {
			PpcRelTokenStream(pW, pNode);
			pNode = NULL;
			pW->fStop = TRUE;
		}
	}
	return pNode;
}

//--------------------------------------------------------------------------------------------------------------//
//	トークンノード／プリプロセス文のトークンストリーム取得														//
//																												//
//	引　数	：	hCtk		- 字句分解インスタンス・ハンドル													//
//				pFilePath	- ファイルパス名のアドレス															//
//				pfPpTkn		- トークンノード／プリプロセス文のトークンストリームの識別を格納するバッファアドレス//
//								＝０ - 単一トークンノード（プリプロセス文でない）								//
//								≠０ - プリプロセス文の種類(EAJCTK_DEFINE, EAJCTK_IF ... )						//
//				pNum		- 生成したトークンの個数を格納するバッファのアドレス								//
//																												//
//	戻り値	：	≠NULL : 先頭トークン・ノードアドレス															//
//				＝NULL : トークン終端																			//
//--------------------------------------------------------------------------------------------------------------//
static	PAJCPPCTKNNODE	GetTokenOrPreProStream(HAJCPPC pW, HAJCTK hCtk, C_BCP pFilePath, EAJCTKCODE *pfPpTkn, UIP pNum)
{
	PAJCPPCTKNNODE	pTop = NULL;
	PAJCPPCTKNNODE	pCur = NULL;
	EAJCTKCODE		PpTkn;
	UI				flg;
	UI				n = 0;
	BOOL			rsu;
	BC				syl[1024];

	if (AjcCtkGetTokenA(hCtk, syl, AJCTSIZE(syl))) {
		flg = AJCTK_FLAG(hCtk);
		PPC_NTC(AJCPPC_NTC_TOKEN, AJCTK_TOKEN(hCtk), syl, 0);
		//----- プリプロセスのトークン・ストリーム生成 ---------------------//
		if ((flg & (AJCTKF_PREPRO | AJCTKF_PPTOP)) == (AJCTKF_PREPRO | AJCTKF_PPTOP)) {
			//----- 生成トークン数初期化 -----------------------------------//
			n = 1;
			//----- 仮のプリプロセス・トークン種別設定 ---------------------//
			PpTkn = AJCTK_TOKEN(hCtk);
			//----- 先頭トークンノード生成 ---------------------------------//
			pTop = pCur = PpcTknAlloc(pW);
			if (pTop != NULL) {
				if (PpcSetTokenNode(pW, pTop, pFilePath, hCtk, syl)) {
					//----- 次のトークン種別設定 -----------------------------------//
					AjcCtkPeekTokenA(hCtk, syl, AJCTSIZE(syl));
					flg = AJCTK_FLAG(hCtk);
					while ((flg & AJCTKF_PREPRO) && !(flg & AJCTKF_PPTOP)) {
						if (pW->fStop) break;
						//----- 次のトークン取得 -----------------------------------//
						AjcCtkGetTokenA(hCtk, syl, AJCTSIZE(syl));
						PPC_NTC(AJCPPC_NTC_TOKEN, AJCTK_TOKEN(hCtk), syl, 0);
						//----- 次のトークン・ストリームを生成し連結 ---------------//
						pCur = PpcAllocAndLinkNextToken(pW, pCur);
						if (pW->IxInc == 0) rsu = PpcSetTokenNode	  (pW, pCur, pFilePath, hCtk, syl);
						else				rsu = PpcSetTokenNodeByInc(pW, pCur, pFilePath, hCtk, syl, pW->IxInc, pW->incf);
						if (rsu) {
							//----- 生成トークン数更新 -----------------------------//
							n++;
							//----- プリプロセス・トークン種別設定 -----------------//
							if (n == 2) {
								PpTkn = AJCTK_TOKEN(hCtk);
							}
							//----- 次のトークン種別設定 ---------------------------//
							AjcCtkPeekTokenA(hCtk,  syl, AJCTSIZE(syl));
							flg = AJCTK_FLAG(hCtk);
						}
						else pW->fStop = TRUE;
					}
				}
				else pW->fStop = TRUE;
			}
			else pW->fStop = TRUE;
			//----- プリプロセス文のトークンストリームの旨、フラグ設定 -----//
			if (pfPpTkn != NULL) *pfPpTkn = PpTkn;
		}
		//----- 単一トークン生成 -------------------------------------------//
		else {
			//----- トークンノード生成 -------------------------------------//
			pTop = PpcTknAlloc(pW);
			if (pTop != NULL) {
				if (pW->IxInc == 0) rsu = PpcSetTokenNode	  (pW, pTop, pFilePath, hCtk, syl);
				else				rsu = PpcSetTokenNodeByInc(pW, pTop, pFilePath, hCtk, syl, pW->IxInc, pW->incf);
				if (rsu) {
					//----- 単一トークンの旨、フラグ設定 -----------------------//
					if (pfPpTkn != NULL) *pfPpTkn = 0;
					//----- 生成トークン数＝１ ---------------------------------//
					n = 1;
				}
				else pW->fStop = TRUE;
			}
			else pW->fStop = TRUE;
		}
	}
	//----- 生成したトークンの個数設定 -------------------------------------//
	if (pNum != NULL) *pNum = n;

	return pTop;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	プリプロセス文処理（#define)																				//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	VO		PrePro_define	(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num)
{
	PAJCPPCTKNNODE	pCur  = pTop;
	BCP				pFile = pTop->pFile;
	UI				lno   = pTop->lno;
	UI				pos   = pTop->pos;

	if (num >= 3) {
		pCur = PpcRelCurAndGetNextToken(pW, pCur);			//	'#'		スキップ
		pCur = PpcRelCurAndGetNextToken(pW, pCur);			//	'define'スキップ
		if (pCur->flg & AJCTKF_MACNAME) {
			if (PpcMacRegist(pW, pCur, pCur->flg & AJCTKF_MACWITHARG, pos)) {
				//	マクロ定義通知
				PPPCMACNODE pMacNode = (PPPCMACNODE)PpcMacGetNode(pW, pCur->pSyl);
				if (pMacNode != NULL) {
					PPC_NTC(AJCPPC_NTC_MACDEF, (UX)pCur, (UX)pMacNode, 0);
				}
			}
		}
		else {
			PPC_ERR(AJCPPC_MACERR_INV_NAME, pFile, lno, 0);
			PpcRelTokenStream(pW, pTop);
		}
	}
	else {
		PPC_ERR(AJCPPC_MACERR_NO_NAME, pFile, lno, 0);
		PpcRelTokenStream(pW, pTop);
	}
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	プリプロセス文処理（#undef)																					//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	VO		PrePro_undef	(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num)
{
	PAJCPPCTKNNODE	pCur  = pTop;
	BCP				pFile = pTop->pFile;
	UI				lno   = pTop->lno;

	if (num == 3) {
		pCur = PpcRelCurAndGetNextToken(pW, pCur);		//	'#'		スキップ
		pCur = PpcRelCurAndGetNextToken(pW, pCur);		//	'undef'	スキップ
		PpcMacUnRegist(pW, pCur->pSyl);					//	マクロ定義消去（マクロが存在しない場合、特にエラーとしない）
		PpcRelTokenStream(pW, pCur);					//	後続トークン消去
	}
	else if (num <= 2) {
		PPC_ERR(AJCPPC_MACERR_NO_NAME, pFile, lno, 0);
		PpcRelTokenStream(pW, pTop);
	}
	else {
		PPC_ERR(AJCPPC_MACERR_INV_NAME, pFile, lno, 0);
		PpcRelTokenStream(pW, pTop);
	}
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	プリプロセス文処理（#include)																				//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	UB		PrePro_include	(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num)
{
	UB				rc		= pTop->incf;
	PAJCPPCINCNEST	pTi		= &pW->TblInc[pW->IxInc];
	PAJCPPCTKNNODE	pCur	= pTop;
	BCP				pFile	= pTop->pFile;
	UI				lno 	= pTop->lno;
	BOOL 			fExist	= FALSE;
	PAJCLBXITEMA	pInc	= NULL;
	BC				drv[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], fext[_MAX_EXT];
	BC				IncName[MAX_PATH];
	BC				relp[MAX_PATH * 2];
	BC				path[MAX_PATH];

	do {
		pW->incf = AJCPPC_INCF_NO;
		if ((pCur = PpcRelCurAndGetNextToken(pW, pCur)) == NULL) break;		//	'#'			スキップ
		if ((pCur = PpcRelCurAndGetNextToken(pW, pCur)) == NULL) {			//	'include'	スキップ
			PPC_ERR(AJCPPC_INCERR_NO_FILE, pFile, lno, 0);
			break;
		}
		//----- #include "XXXXX.h" -----------------------------------------------------------------------------//
		if		(pCur->tkn == EAJCTK_STR_QUOTE) {
			pW->incf |= AJCPPC_INCF_LCL;
			//----- ファイル名設定 -------------------------------------------------------//
			AjcSnPrintFA(IncName, MAX_PATH, "%s", pCur->pSyl + 1);
			mbstok(IncName, "\"");
			//----- 自フォルダ検索 -------------------------------------------------------//
			_splitpath(pTi->pPath, drv, dir, NULL, NULL);
			_makepath  (path 	, drv, dir, NULL, NULL);
			AjcPathCatA(path, IncName, MAX_PATH);
			fExist = PathFileExistsA(path);
		}
		//----- #include <XXXXX.h> -----------------------------------------------------------------------------//
		else if (pCur->tkn == EAJCTK_DLM_LO   ) {
			pW->incf |= AJCPPC_INCF_GBL;
			//----- '<' チェック --------------------------------------------------------//
			if ((pCur = PpcRelCurAndGetNextToken(pW, pCur)) == NULL) {		//	'<'	スキップ
				PPC_ERR(AJCPPC_INCERR_NO_FILE, pFile, lno, 0);
				break;
			}
			//----- ファイル名設定 -------------------------------------------------------//
			{	UI	ix, stl;
				ix = 0;
				while (pCur != NULL && pCur->tkn != EAJCTK_DLM_HI) {
					stl = (UI)strlen(pCur->pSyl);
					if (ix + stl + 1 < MAX_PATH) {
						strcpy(&IncName[ix], pCur->pSyl);
						ix += stl;
					}
					pCur = PpcRelCurAndGetNextToken(pW, pCur);
				}
			}
			//----- '>' チェック ---------------------------------------------------------//
			if (pCur == NULL || pCur->tkn != EAJCTK_DLM_HI) {
				PPC_ERR(AJCPPC_INCERR_NOTCLS, pFile, lno, 0);
				break;
			}
		}
		//----- #include 記述ミス ------------------------------------------------------------------------------//
		else {
			PPC_ERR(AJCPPC_INCERR_INV_FILE, pFile, lno, 0);
			break;
		}
		//----- インクルードファイル読み込み済みチェック -------------------------------------------------------//
		if ((pW->OptFlag & AJCPPC_FLG_ONCE) && PpcIncIsRegist(pW, IncName)) {	// ONCE and 読み込み済み？
			//「#include」の「"ファイル名"」トークンに多重インクルードの旨、フラグセット
			pW->incf |= AJCPPC_INCF_DUP;
			break;
		}
		//----- インクルードファイル名登録 ---------------------------------------------------------------------//
		if (pW->hAvlInc != NULL) {
			if (PpcIncRegist(pW, IncName) == NULL) {
				pW->fStop = TRUE;
				break;
			}
		}
		//----- 最優先Includeパス検索 --------------------------------------------------------------------------//
		if (pW->pIncArr != NULL) {
			BC	temp_path[MAX_PATH];
			for (pInc = pW->pIncArr; pInc->pStr != NULL; pInc++) {
				if (pInc->data == AJCPP_INC_TOPPRIORITY) {
					//----- インクルード・ディレクトリ設定 ---------------------//
					if (PathIsRelativeA(pInc->pStr)) {
						AjcSnPrintFA(relp, AJCTSIZE(relp), "%s", pW->BasePath);
						AjcPathCatA (relp, pInc->pStr, AJCTSIZE(relp));
						PathCanonicalizeA(temp_path, relp);
					}
					else {
						PathCanonicalizeA(temp_path, pInc->pStr);
					}
					//----- インクルードファイルの存在チェック -----------------//
					AjcPathCatA(temp_path, IncName, AJCTSIZE(temp_path));
					if (fExist = PathFileExistsA(temp_path)) {
						strcpy(path, temp_path);
						break;
					}
				}
			}
		}
		//----- Includeパス検索 --------------------------------------------------------------------------------//
		if (!fExist) {
			if (pW->pIncArr != NULL) {
				for (pInc = pW->pIncArr; pInc->pStr != NULL && !fExist; pInc++) {
					//----- インクルード・ディレクトリ設定 ---------------------//
					if (PathIsRelativeA(pInc->pStr)) {
						AjcSnPrintFA(relp, AJCTSIZE(relp), "%s", pW->BasePath);
						AjcPathCatA (relp, pInc->pStr, AJCTSIZE(relp));
						PathCanonicalizeA(path, relp);
					}
					else {
						PathCanonicalizeA(path, pInc->pStr);
					}
					//----- インクルードファイルの存在チェック -----------------//
					if (pW->OptFlag & AJCPPC_FLG_AUTO_SEARCH) {
						if (fExist = SearchIncFile(pW, IncName, path, AJCTSIZE(path))) break;
					}
					else {
						AjcPathCatA(path, IncName, AJCTSIZE(path));
						if (fExist = PathFileExistsA(path)) break;
					}
				}
			}
			if (!fExist) {
				//----- インクルード・ディレクトリ設定 -------------------------//
				AjcSnPrintFA(path, AJCTSIZE(path), "%s", pW->BasePath);
				//----- インクルードファイルの存在チェック ---------------------//
				if (pW->OptFlag & AJCPPC_FLG_AUTO_SEARCH) {
					fExist = SearchIncFile(pW, IncName, path, AJCTSIZE(path));
				}
				else {
					AjcPathCatA(path, IncName, AJCTSIZE(path));
					fExist = PathFileExistsA(path);
				}
			}
		}
		//----- Includeファイルが見つかったら、ネストテーブル更新 ----------------------------------------------//
		if (fExist) {
			//----- ネストテーブル更新処理 -----------------------------------------------//
			if (pW->IxInc < AJCPPC_MAX_INC_NEST - 1) {
				pW->IxInc++;
				pTi = &pW->TblInc[pW->IxInc];
				do {
					//----- インクルードファイル・オープン -------------------------------//
					if ((pTi->fh = AjcFOpenA(path, pW->InpTec)) == NULL) {
						PPC_ERR(AJCPPC_ERROR_INC_OPEN, pFile, lno, path);
						pW->IxInc--;
						pTi = &pW->TblInc[pW->IxInc];
						break;
					}
					//----- 字句分解インスタンス生成 -------------------------------------//
					pTi->hCtk = PpcCtkCreate(pW, cbCtkFile);
					if (pTi->hCtk == NULL) {pW->fStop = TRUE; break;}
					//----- #includeネスト制御テーブルエントリにインクルードファイル設定 -//
					pTi->pPath = PpcIStrRegist(pW, path);
					_splitpath(path    , NULL, NULL, fname, fext);
					_makepath (pTi->file, NULL, NULL, fname, fext);
					pTi->lno = 0;
					//----- #if～#endif ネスト情報初期化 ---------------------------------//
					pTi->IxIF = 0;
					memset(pTi->TblIF, 0, sizeof pTi->TblIF);
					pTi->TblIF[0].knd = AJCPPC_IFK_NONE;
					pTi->TblIF[0].act = AJCPPC_IFA_GEN_NORMAL;
				} while(0);
			}
			else {
				PPC_ERR(AJCPPC_INCERR_NEST_OVER, pFile, lno, 0);
				break;
			}
		}
		//----- Includeファイルが見つからない場合 --------------------------------------------------------------//
		else {
			PPC_ERR(AJCPPC_INCERR_NOT_FOUND, pFile, lno, IncName);
			break;
		}
	} while(0);
	//----- 末尾部分のトークン消去 -----------------------------------------------------------------------------//
	PpcRelTokenStream(pW, pCur);

	return pW->incf;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	プリプロセス文処理（#ifdef)																					//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL		PrePro_ifdef	(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num)
{
	BOOL			rc	   = FALSE;		//	戻り値（条件コンパイルの真偽値）
	PAJCPPCINCNEST	pTi    = &pW->TblInc[pW->IxInc];
	PAJCPPCTKNNODE	pCur   = pTop;
	BCP				pFile  = pTop->pFile;
	UI				lno    = pTop->lno;
	AJCPPCIFACT		CurAct = pTi->TblIF[pTi->IxIF].act;

	pCur = PpcRelCurAndGetNextToken(pW, pCur);							//	'#'		スキップ
	pCur = PpcRelCurAndGetNextToken(pW, pCur);							//	'ifdef'	スキップ

	if (pTi->IxIF < AJCPPC_MAX_IF_NEST - 1) {							//	許容ネスト範囲内？
		pTi->IxIF++;													//		ネスト数加算
		pTi->TblIF[pTi->IxIF].knd = AJCPPC_IFK_IFDEF;					//		ＰＰ種別設定
		pTi->TblIF[pTi->IxIF].lno = lno;								//		（対応する）#ifdef, #ifndef, #if の行番号
		if (CurAct & AJCPPC_IFA_GEN) {									//		生成条件中？
			if (pCur != NULL && AJCTKIS_SYMBOL(pCur->tkn)) {			//			シンボル？
				if (PpcMacGetNode(pW, pCur->pSyl) != NULL) {			//				シンボル定義済？
					pTi->TblIF[pTi->IxIF].act = AJCPPC_IFA_GEN_IN_IF;	//					「ＩＦ（真条件）による生成状態」
					rc = TRUE;											//					戻り値＝真
				}
				else {													//				シンボル未定義？
					pTi->TblIF[pTi->IxIF].act = AJCPPC_IFA_SKIP_BY_IF;	//					「ＩＦ（偽条件）によるスキップ状態」
				}
			}
			else {														//		シンボル以外？
				PPC_ERR(AJCPPC_ERROR_NO_SYMBOL, pFile, lno, 0);
				pTi->TblIF[pTi->IxIF].act = AJCPPC_IFA_SKIP_BY_IF;		//			「ＩＦ（偽条件）によるスキップ状態」
			}
		}
		else {															//		スキップ条件中？
			pTi->TblIF[pTi->IxIF].act = AJCPPC_IFA_SKIP_ALL;			//			「無条件スキップ状態（上位の条件が偽）」
			rc = AJCPPIF_UNCOND;
		}
	}
	else {
		PPC_ERR(AJCPPC_ERROR_COND_DEEP, pFile, lno, 0);
	}
	//----- 末尾部分のトークン消去 -----------------------------------------------------------------------------//
	PpcRelTokenStream(pW, pCur);

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	プリプロセス文処理（#ifndef)																				//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL		PrePro_ifndef	(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num)
{
	BOOL			rc	 	= FALSE;		//	戻り値（条件コンパイルの真偽値）
	PAJCPPCINCNEST	pTi		= &pW->TblInc[pW->IxInc];
	PAJCPPCTKNNODE	pCur	= pTop;
	BCP				pFile	= pTop->pFile;
	UI				lno 	= pTop->lno;
	AJCPPCIFACT		CurAct	= pTi->TblIF[pTi->IxIF].act;

	pCur = PpcRelCurAndGetNextToken(pW, pCur);							//	'#' 	 スキップ
	pCur = PpcRelCurAndGetNextToken(pW, pCur);							//	'ifndef' スキップ

	if (pTi->IxIF < AJCPPC_MAX_IF_NEST - 1) {							//	許容ネスト範囲内？
		pTi->IxIF++;													//		ネスト数加算
		pTi->TblIF[pTi->IxIF].knd = AJCPPC_IFK_IFNDEF;					//		ＰＰ種別設定
		pTi->TblIF[pTi->IxIF].lno = lno;								//		（対応する）#ifdef, #ifndef, #if の行番号
		if (CurAct & AJCPPC_IFA_GEN) {									//		生成条件中？
			if (pCur != NULL && AJCTKIS_SYMBOL(pCur->tkn)) {			//			シンボル？
				if (PpcMacGetNode(pW, pCur->pSyl) == NULL) {			//				シンボル未定義？
					pTi->TblIF[pTi->IxIF].act = AJCPPC_IFA_GEN_IN_IF;	//					「ＩＦ（真条件）による生成状態」
					rc = TRUE;											//					戻り値＝真
				}
				else {													//				シンボル定義済？
					pTi->TblIF[pTi->IxIF].act = AJCPPC_IFA_SKIP_BY_IF;	//					「ＩＦ（偽条件）によるスキップ状態」
				}
			}
			else {														//		シンボル以外？
				PPC_ERR(AJCPPC_ERROR_NO_SYMBOL, pFile, lno, 0);
				pTi->TblIF[pTi->IxIF].act = AJCPPC_IFA_SKIP_BY_IF;		//			「ＩＦ（偽条件）によるスキップ状態」
			}
		}
		else {															//		スキップ条件中？
			pTi->TblIF[pTi->IxIF].act = AJCPPC_IFA_SKIP_ALL;			//			「無条件スキップ状態（上位の条件が偽）」
			rc = AJCPPIF_UNCOND;										//			 戻り値＝「偽条件中」
		}
	}
	else {
		PPC_ERR(AJCPPC_ERROR_COND_DEEP, pFile, lno, 0);
	}
	//----- 末尾部分のトークン消去 -----------------------------------------------------------------------------//
	PpcRelTokenStream(pW, pCur);

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	プリプロセス文処理（#if)																					//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	PrePro_if		(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num)
{
	BOOL			rc	   = FALSE;		//	戻り値（条件コンパイルの真偽値）
	PAJCPPCINCNEST	pTi    = &pW->TblInc[pW->IxInc];
	PAJCPPCTKNNODE	pCur   = pTop;
	BCP				pFile  = pTop->pFile;
	UI				lno    = pTop->lno;
	AJCPPCIFACT		CurAct = pTi->TblIF[pTi->IxIF].act;
	int				n	   = 0;

	pCur = PpcRelCurAndGetNextToken(pW, pCur);							//	'#' スキップ
	pCur = PpcRelCurAndGetNextToken(pW, pCur);							//	'if' スキップ

	if (pTi->IxIF < AJCPPC_MAX_IF_NEST - 1) {							//	許容ネスト範囲内？
		pTi->IxIF++;													//		ネスト数加算
		pTi->TblIF[pTi->IxIF].knd = AJCPPC_IFK_IF;						//		ＰＰ種別設定
		pTi->TblIF[pTi->IxIF].lno = lno;								//		（対応する）#ifdef, #ifndef, #if の行番号
		if (CurAct & AJCPPC_IFA_GEN) {									//		生成条件中？
			if (pCur != NULL && PpcFormulaExpression(pW, pCur, &n) == AJCPPC_ERROR_OK) {
																		//			正常な演算式？
				if (n != 0) {											//				TRUE？
					pTi->TblIF[pTi->IxIF].act = AJCPPC_IFA_GEN_IN_IF;	//					「ＩＦ（真条件）による生成状態」
					rc = TRUE;											//					戻り値＝真
				}
				else {													//				FALSE？
					pTi->TblIF[pTi->IxIF].act = AJCPPC_IFA_SKIP_BY_IF;	//					「ＩＦ（偽条件）によるスキップ状態」
				}
			}
			else {														//			不正な演算式？
				pTi->TblIF[pTi->IxIF].act = AJCPPC_IFA_SKIP_BY_IF;		//				「ＩＦ（偽条件）によるスキップ状態」
			}
		}
		else {															//		スキップ条件中？
			pTi->TblIF[pTi->IxIF].act = AJCPPC_IFA_SKIP_ALL;			//			「無条件スキップ状態（上位の条件が偽）」
			rc = AJCPPIF_UNCOND;										//			 戻り値＝「偽条件中」
		}
	}
	else {
		PPC_ERR(AJCPPC_ERROR_COND_DEEP, pFile, lno, 0);
	}
	//----- 末尾部分のトークン消去 -----------------------------------------------------------------------------//
	PpcRelTokenStream(pW, pCur);

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	プリプロセス文処理（#elif)																					//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	PrePro_elif		(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num)
{
	BOOL			rc	   = FALSE;		//	戻り値（条件コンパイルの真偽値）
	PAJCPPCINCNEST	pTi    = &pW->TblInc[pW->IxInc];
	PAJCPPCTKNNODE	pCur   = pTop;
	BCP				pFile  = pTop->pFile;
	UI				lno    = pTop->lno;
	AJCPPCIFKIND	CurKnd = pTi->TblIF[pTi->IxIF].knd;
	AJCPPCIFACT		CurAct = pTi->TblIF[pTi->IxIF].act;
	int				n	   = 0;

	pCur = PpcRelCurAndGetNextToken(pW, pCur);								//	'#' スキップ
	pCur = PpcRelCurAndGetNextToken(pW, pCur);								//	'elif' スキップ

	if (pTi->IxIF > 0) {													//	ネスト中？
		if (CurKnd >= AJCPPC_IFK_IFDEF && CurKnd <= AJCPPC_IFK_ELIF) {		//		「ifdef / ifndef / if / elif」によるネスト中？
			pTi->TblIF[pTi->IxIF].knd = AJCPPC_IFK_ELIF;					//			ＰＰ種別設定
			if (CurAct & AJCPPC_IFA_GEN) {									//			生成条件中？
				pTi->TblIF[pTi->IxIF].act = AJCPPC_IFA_SKIP_ENDIF;			//				「ＥＮＤＩＦまでスキップ状態」
			}
			else if (CurAct == AJCPPC_IFA_SKIP_BY_IF) {						//			ＩＦ（偽条件）によるスキップ状態
				if (pCur != NULL && PpcFormulaExpression(pW, pCur, &n) == AJCPPC_ERROR_OK) {
																			//				正常な演算式？
					if (n != 0) {											//					TRUE？
						pTi->TblIF[pTi->IxIF].act = AJCPPC_IFA_GEN_IN_IF;	//						「ＩＦ（真条件）による生成状態」
						rc = TRUE;											//						戻り値＝真
					}
					else {													//					FALSE？
						pTi->TblIF[pTi->IxIF].act = AJCPPC_IFA_SKIP_BY_IF;	//						「ＩＦ（偽条件）によるスキップ状態」
					}
				}
				else {														//				不正な演算式？
					pTi->TblIF[pTi->IxIF].act = AJCPPC_IFA_SKIP_BY_IF;		//					「ＩＦ（偽条件）によるスキップ状態」
				}
			}
			else  {															//			ＥＮＤＩＦまでスキップ中／無条件キップ中？
				if (pTi->IxIF > 0 && (pTi->TblIF[pTi->IxIF - 1].act & AJCPPC_IFA_SKIP)) {
					rc = AJCPPIF_UNCOND;									//				戻り値＝「偽条件中」
				}
			}
		}
		else {																//		「ELSE」によるネスト中？
			PPC_ERR(AJCPPC_ERROR_ELIF_IN_ELSE, pFile, lno, 0);
		}
	}
	else {																	//	ネストなし？
		PPC_ERR(AJCPPC_ERROR_NOT_IN_IF, pFile, lno, 0);
	}
	//----- 末尾部分のトークン消去 -----------------------------------------------------------------------------//
	PpcRelTokenStream(pW, pCur);

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	プリプロセス文処理（#else)																					//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	PrePro_else		(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num)
{
	BOOL			rc		= FALSE;		//	戻り値（条件コンパイルの真偽値）
	PAJCPPCINCNEST	pTi		= &pW->TblInc[pW->IxInc];
	PAJCPPCTKNNODE	pCur	= pTop;
	BCP				pFile	= pTop->pFile;
	UI				lno 	= pTop->lno;
	AJCPPCIFKIND	CurKnd	= pTi->TblIF[pTi->IxIF].knd;
	AJCPPCIFACT		CurAct	= pTi->TblIF[pTi->IxIF].act;

	if (pTi->IxIF > 0) {													//	ネスト中？
		if (CurKnd >= AJCPPC_IFK_IFDEF && CurKnd <= AJCPPC_IFK_ELIF) {		//		「ifdef / ifndef / if / elif」によるネスト中？
			pTi->TblIF[pTi->IxIF].knd = AJCPPC_IFK_ELSE;					//			ＰＰ種別設定
			if		(CurAct == AJCPPC_IFA_GEN_IN_IF) {						//			ＩＦ（真条件）による生成状態？
				pTi->TblIF[pTi->IxIF].act = AJCPPC_IFA_SKIP_ENDIF;			//				「ＥＮＤＩＦまでスキップ状態」
			}
			else if (CurAct == AJCPPC_IFA_SKIP_BY_IF) {						//			ＩＦ（偽条件）によるスキップ状態？
				pTi->TblIF[pTi->IxIF].act = AJCPPC_IFA_GEN_IN_ELSE;			//				「ＥＬＳＥ条件による生成状態」
				rc = TRUE;													//				戻り値＝真
			}
			else {															//			ＥＮＤＩＦまでスキップ中／無条件キップ中？
				if (pTi->IxIF > 0 && (pTi->TblIF[pTi->IxIF - 1].act & AJCPPC_IFA_SKIP)) {
					rc = AJCPPIF_UNCOND;									//				戻り値＝「偽条件中」
				}
			}
		}
		else {																//		「ELSE」によるネスト中？
			PPC_ERR(AJCPPC_ERROR_ELSE_IN_ELSE, pFile, lno, 0);
		}
	}
	else {																	//	ネストなし？
		PPC_ERR(AJCPPC_ERROR_NOT_IN_IF, pFile, lno, 0);
	}
	//----- トークンストリーム消去 -----------------------------------------------------------------------------//
	PpcRelTokenStream(pW, pCur);

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	プリプロセス文処理（#endif)																					//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	UI		PrePro_endif	(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num)
{
	UI				rc		= 0;		//	戻り値（対応する #ifdef.#ifndef/#if までの相対行番号，63335を超える場合は0）
	PAJCPPCINCNEST	pTi		= &pW->TblInc[pW->IxInc];
	PAJCPPCTKNNODE	pCur	= pTop;
	BCP				pFile	= pTop->pFile;
	UI				lno 	= pTop->lno;
	AJCPPCIFACT		CurAct	= pTi->TblIF[pTi->IxIF].act;

	if (pTi->IxIF > 0) {												//	ネスト中？
		if ((pTop->lno - pTi->TblIF[pTi->IxIF].lno) > 65530) {			//		条件コンパイル行数範囲オーバー？
			rc = 0;														//			戻り値＝０
		}
		else {															//		その他（正常）？
			rc = pTop->lno - pTi->TblIF[pTi->IxIF].lno;					//			戻り値（対応する #ifdef.#ifndef/#if までの相対行番号）
		}
		pTi->TblIF[pTi->IxIF].knd = 0;									//		ネストテーブルエントリクリアー
		pTi->TblIF[pTi->IxIF].act = 0;									//		・
		pTi->TblIF[pTi->IxIF].lno = 0;									//		・
		pTi->IxIF--;													//		ネスト数減算
	}
	else {
		PPC_ERR(AJCPPC_ERROR_NOT_IN_IF, pFile, lno, 0);
	}
	//----- トークンストリーム消去 -----------------------------------------------------------------------------//
	PpcRelTokenStream(pW, pCur);

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	プリプロセス文処理（その他)																					//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	VO		PrePro_others	(HAJCPPC pW, HAJCTK hCtk, PAJCPPCTKNNODE pTop, UI num)
{
	PAJCPPCINCNEST	pTi = &pW->TblInc[pW->IxInc];

	//----- トークンストリーム消去 -----------------------------------------------------------------------------//
	PpcRelTokenStream(pW, pTop);
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	プリプロセス制御（IF条件中のトークン生成)																	//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	VO	Generate_InIf(HAJCPPC pW)
{
	PAJCPPCINCNEST	pTi = &pW->TblInc[pW->IxInc];
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	プリプロセス制御（ELSE条件中のトークン生成)																	//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	VO	Generate_InElse(HAJCPPC pW)
{
	PAJCPPCINCNEST	pTi = &pW->TblInc[pW->IxInc];
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	プリプロセス制御（ELIF / ELSE / ENDIF)までスキップ															//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	VO	SkipTo_elif_else_endif(HAJCPPC pW)
{
	PAJCPPCINCNEST	pTi = &pW->TblInc[pW->IxInc];
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	プリプロセス制御（ENDIF)までスキップ																		//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	VO	SkipTo_endif(HAJCPPC pW)
{
	PAJCPPCINCNEST	pTi = &pW->TblInc[pW->IxInc];
}
//--------------------------------------------------------------------------------------------------------------//
//	オプションシンボル解析の字句分解用コールバック																//
//																												//
//	引　数	：	pBuf		- テキスト格納バッファのアドレス													//
//				pFilePath	- テキスト格納バッファの文字数														//
//				xp			- コールバックパラメタ（インスタンスワーク）										//
//																												//
//	戻り値	：	TRUE  : 有効なテキストを設定した																//
//				FALSE : テキスト終端																			//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL CALLBACK cbCtkOptSym(BCP pBuf, UI lBuf, UX xp)
{
	HAJCPPC			pW	= (HAJCPPC)xp;
	PAJCPPCINCNEST	pTi = &pW->TblInc[pW->IxInc];
	BOOL			rc;
	UI				stl;
	BCP				p1, p2, pTmp;

	if (pW->pOptSymText != NULL) {
		stl = (UI)strlen(pW->pOptSymText);
		pTmp = PpcMemAlloc(pW, stl + 1);
		if (pTmp != NULL) {
			strcpy(pTmp, pW->pOptSymText);
			p1 = mbstok(pTmp, "=");
			p2 = mbstok(NULL, "\0");
			if (p1 != NULL) {
				if (p2 == NULL) AjcSnPrintFA(pBuf, lBuf, "#define %s"   , p1    );
				else			AjcSnPrintFA(pBuf, lBuf, "#define %s %s", p1, p2);
			}
			free(pTmp);
			pW->pOptSymText = NULL;
			rc = TRUE;
		}
		else pW->fStop = TRUE;
	}
	else {
		rc = FALSE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	ソースファイル字句分解用コールバック																		//
//																												//
//	引　数	：	pBuf		- テキスト格納バッファのアドレス													//
//				pFilePath	- テキスト格納バッファの文字数														//
//				xp			- コールバックパラメタ（インスタンスワーク）										//
//																												//
//	戻り値	：	TRUE  : 有効なテキストを設定した																//
//				FALSE : テキスト終端																			//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL CALLBACK cbCtkFile(BCP pBuf, UI lBuf, UX xp)
{
	VOP				rc;
	HAJCPPC			pW	= (HAJCPPC)xp;
	PAJCPPCINCNEST	pTi = &pW->TblInc[pW->IxInc];

	pTi->lno++;
	if ((rc = AjcFGetSA(pTi->fh, pBuf, lBuf)) != NULL) {
	//	SsSetText(pBuf, pTi->lno);
	}

	return (rc != NULL);
}


//--------------------------------------------------------------------------------------------------------------//
//	インクルードファイル検索																					//
//																												//
//	引　数	：	pFName	- インクルードファイル名のアドレス														//
//				pPath	- 検索するディレクトリパス（見つかった場合は当該ファイルパスが設定される）				//
//				len		- pathが示すバッファのサイズ															//
//																												//
//	戻り値	：	TRUE : 当該ファイルが見つかった																	//
//				FALSE: 当該ファイルは見つからない																//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	SearchIncFile(HAJCPPC pW, C_BCP pFName, BCP pPath, int len)
{
	BOOL		rc = FALSE;				//	戻り値
	BC			drv[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], fext[_MAX_EXT];

	PPC_NTC(AJCPPC_NTC_SRH_START, pFName, pPath, 0);									//	検索開始通知
	pW->pSrhPath = pPath;																//	パス名設定バッファアドレス退避

	if (PathIsRelativeA(pFName)) {														//	インクルードファイルは相対パス？
		AjcSetNtcSearchingDir(TRUE);													//		ファイル検索時、検索ディレクトリを通知
		if (strchr(pPath, '*') == NULL && strchr(pPath, '?') == NULL) {					//		ディレクトリパスにワイルドカード無し？
			//	パスとファイル名を再構成（「"d:\incpath", "sys/types.h"」→「"d:\incpath\sys", "types.h"」）
			BC		iPath[MAX_PATH];
			BC		iName[MAX_PATH];
			_splitpath(pFName, drv, dir, fname, fext);
			AjcSnPrintFA(iPath, MAX_PATH, "%s", pPath);
			AjcPathCatA(iPath, dir, MAX_PATH);
			_makepath (iName, NULL, NULL, fname, fext);
			//	インクルードファイル検索
			pW->rcSrhInc = FALSE;
			AjcSearchFilesA(iPath, iName, TRUE , (UX)pW, cbASearchIncFile);				//			インクルードファイル検索処理
			rc = pW->rcSrhInc;
		}
		else {																			//		ディレクトリパスにワイルドカード有り？
			UX	fh;
			struct _finddata_t fd;
			BC		SrhDir[MAX_PATH];
			//	ワイルドカードの全ディレクトリについてインクルードファイル検索
			if (fh = _findfirst(pPath, &fd)) {
				do {
					if (fd.attrib & _A_SUBDIR) {
						if (mbscmp(fd.name, ".") != 0 && mbscmp(fd.name, "..") != 0) {
							//	インクルードファイル検索パス設定
							_splitpath(pPath , drv, dir, NULL, NULL);	//	ワイルドカード除去
							_makepath (SrhDir, drv, dir, NULL, NULL);	//	・
							AjcPathCatA(SrhDir, fd.name, MAX_PATH);		//	検索したディレクトリ付加
							pW->rcSrhInc = FALSE;
							AjcSearchFilesA(SrhDir, pFName, TRUE , (UX)pW, cbASearchIncFile);
							rc = pW->rcSrhInc;
							if (rc) break;
						}
					}
				} while (_findnext(fh, &fd) != -1 && !pW->fStop);
				_findclose(fh);
			}
		}
	}
	else {																				//	インクルードファイルは絶対パス？
		if (AjcPathIsFileA(pFName)) {													//		パスはファイル？
			AjcSnPrintFA(pW->pSrhPath, MAX_PATH, "%s", pFName);							//			インクルードファイルのパス名設定
			rc = TRUE;
		}
	}
	PPC_NTC(AJCPPC_NTC_SRH_END, pFName, rc, 0);											//	検索終了通知

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	インクルードファイル検索のコールバック																		//
//--------------------------------------------------------------------------------------------------------------//
static BOOL CALLBACK cbASearchIncFile(UI nest, BCP pPath, BCP pName, UI attrib, UI wtime, UX cbp)
{
	BOOL	rc = TRUE;
	HAJCPPC	pW = (HAJCPPC)cbp;

	if (nest != -1) {											//	ファイル／ディレクトリ通知？if 
		if (!(attrib & _A_SUBDIR)) {							//		ディレクトリ以外？
			AjcSnPrintFA(pW->pSrhPath, MAX_PATH, "%s", pPath);	//			インクルードファイルのパス名設定
			pW->rcSrhInc = TRUE;								//			インクルードファイル検索結果設定
			rc		     = FALSE;								//			戻り値＝検索終了
		}
		else {
			rc = !pW->fStop;									//		戻り値＝(TRUE:検索継続，FALSE:検索中止)
		}
	}
	else {														//	検索中のディレクトリ通知？
		PPC_NTC(AJCPPC_NTC_SRH_DIR, pPath, 0, 0);				//		検索中のパス名通知
		rc = !pW->fStop;										//		戻り値＝(TRUE:検索継続，FALSE:検索中止)
	}

	return rc;
}

