﻿#include	"AjcInternal.h"

//**************************************************************************************************************//
//																												//
//	可変長キュー（線形リスト）																					//
//																												//
//**************************************************************************************************************//


#define	INST_ID			0x816CD9AF
#define	IS_MY_INST(P)	(P != NULL && P->InstID == INST_ID)


//==============================================================================================================//
//	線形リスト制御（可変長データ）																				//
//==============================================================================================================//
//--------------------------------------------------------------------------------------------------------------//
//	内部マクロ																									//
//--------------------------------------------------------------------------------------------------------------//
#define		AJCVQUE_ENTER		if (pW->fMultiThread) EnterCriticalSection(&pW->cs);
#define		AJCVQUE_LEAVE		if (pW->fMultiThread) LeaveCriticalSection(&pW->cs);


//==============================================================================================================//
//	インスタンス生成																							//
//																												//
//	引　数	：	cbp 	 - コールバックパラメタ																	//
//				cbRemove - ノード消去通知用コールバック（不要時はＮＵＬＬ）										//
//																												//
//	戻り値	：	≠ＮＵＬＬ：ＯＫ（インスタンスハンドル（＝インスタンスワークアドレス））						//
//				＝ＮＵＬＬ：エラー																				//
//==============================================================================================================//
AJCEXPORT HAJCVQUE	WINAPI	AjcVQueCreate(UX cbp, VO (CALLBACK *cbRemove)(VOP pDat, UI len, UX cbp))
{
	HAJCVQUE  pW;

	if ((pW = (HAJCVQUE)AJCMEM(sizeof(AJCVQUE))) != NULL) {
		memset(pW, 0, sizeof *pW);
		pW->InstID	= INST_ID;
		pW->pTop	= NULL;
		pW->pLas	= NULL;
		pW->num		= 0;
		pW->cbp 	= cbp;
		pW->cbRemove= cbRemove;
	}
	return pW;
}
//==============================================================================================================//
//	インスタンス消去																							//
//																												//
//	引　数	：	pW 		  - インスタンスハンドル																//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT BOOL	WINAPI	AjcVQueDelete(HAJCVQUE pW)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		AjcVQuePurge(pW);
		if (pW->fMultiThread) {
			DeleteCriticalSection(&pW->cs);
		}
		free(pW);
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	マルチスレッドの許可／禁止																					//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//				fEnable - FALSE : マルチスレッド禁止															//
//						  TRUE	: マルチスレッド許可															//
//																												//
//	戻り値	：	前回の許可／禁止状態																			//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcVQueEnableMultiThread(HAJCVQUE pW, BOOL fEnable)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		rc = pW->fMultiThread;
		if (fEnable) {
			if (!pW->fMultiThread) {
				InitializeCriticalSection(&pW->cs);
				pW->fMultiThread = TRUE;
			}
		}
		else {
			if (pW->fMultiThread) {
				pW->fMultiThread = FALSE;
				DeleteCriticalSection(&pW->cs);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	末尾へノード追加																							//
//																												//
//	引　数	：	pW 	 - インスタンスハンドル																		//
//				pDat - 末尾へ挿入するノードデータのアドレス														//
//				len  - 末尾へ挿入するノードデータのバイト数														//
//																												//
//	戻り値	：	TRUE  / ≠NULL - 追加したノードのアドレス														//
//				FALSE / ＝NULL - エラー																			//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcVQueEnque	(HAJCVQUE pW, C_VOP pDat, UI len)
{
	return (AjcVQueEnqueEx(pW, pDat, len) != NULL);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	VOP		WINAPI	AjcVQueEnqueEx	(HAJCVQUE pW, C_VOP pDat, UI len)
{
	VOP		rc = FALSE;
	PAJCVQNODE	pNew;

	if (IS_MY_INST(pW) && pDat != NULL) {
		AJCVQUE_ENTER
		if ((pNew = (PAJCVQNODE)AJCMEM(sizeof(AJCVQNODE) + len)) != NULL) {	//	メモリブロック割り当て，成功？
			pNew->pNxt = 0;													//		新ＢＬＫ初期化
			pNew->len = len;												//		・
			memcpy(pNew + 1, pDat, len);									//		・
			if (pW->pLas == NULL) {											//		初回キューイング？
				pW->pTop = pW->pLas = pNew;									//			新ＢＬＫを初キューイング
			}
			else {															//		追加キューイング？
				pW->pLas->pNxt = pNew;										//			新ＢＬＫを末尾へキューイング
				pW->pLas	   = pNew;										//			・
			}
			pW->num++;														//		ノード数更新
			rc = pNew + 1;													//		戻り値＝ＯＫ
		}
		else {																//	メモリブロック割り当て，失敗？
			rc = NULL;														//		戻り値＝エラー
		}
		AJCVQUE_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	先頭へノード挿入																							//
//																												//
//	引　数	：	pW 	 - インスタンスハンドル																		//
//				pDat - 先頭へ挿入するノードデータのアドレス														//
//				len  - 先頭へ挿入するノードデータのバイト数														//
//																												//
//	戻り値	：	≠NULL - 追加したノードのアドレス																//
//				＝NULL - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcVQueEnqTop	(HAJCVQUE pW, C_VOP pDat, UI len)
{
	return (AjcVQueEnqTopEx(pW, pDat, len) != NULL);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	VOP		WINAPI	AjcVQueEnqTopEx	(HAJCVQUE pW, C_VOP pDat, UI len)
{
	VOP			rc = NULL;
	PAJCVQNODE	pNew;

	if (IS_MY_INST(pW) && pDat != NULL) {
		AJCVQUE_ENTER
		if ((pNew = (PAJCVQNODE)AJCMEM(sizeof(AJCVQNODE) + len)) != NULL) {	//	メモリブロック割り当て，成功？
			pNew->len	= len;												//		・
			memcpy(pNew + 1, pDat, len);									//		・
			if (pW->pTop == NULL) {											//		初回キューイング？
				pNew->pNxt = NULL;											//			新ＢＬＫを初キューイング
				pW->pTop = pW->pLas = pNew;									//			・
			}
			else {															//		追加キューイング？
				pNew->pNxt = pW->pTop;										//			新ＢＬＫを先頭へキューイング
				pW->pTop   = pNew;											//			・
			}
			pW->num++;														//		ノード数更新
			rc = pNew + 1;													//		戻り値＝ＯＫ
		}
		else {																//	メモリブロック割り当て，失敗？
			rc = NULL;														//		戻り値＝エラー
		}
		AJCVQUE_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	先頭ノードを取り出す																						//
//																												//
//	引　数	：	pW 	 - インスタンスハンドル																		//
//				pBuf - 取り出したノードのデータを格納するバッファのアドレス（不要時はＮＵＬＬ）					//
//				bfl  - 取り出したノードのデータを格納するバッファの容量(バイト数）								//
//																												//
//	戻り値	：	≠０：取り出したノードデータのバイト数															//
//				＝０：ノードなし																				//
//==============================================================================================================//
AJCEXPORT	UI	WINAPI	AjcVQueDeque	(HAJCVQUE pW, VOP pBuf, UI bfl)
{
	UI			rc = 0;
	PAJCVQNODE	pCur;

	if (IS_MY_INST(pW)) {
		AJCVQUE_ENTER
		if ((pCur = pW->pTop) != NULL) {										//	ノードあり？
			if ((pW->pTop = pCur->pNxt) == NULL) {								//		先頭ｴﾚﾒﾝﾄ取り外し，次のｴﾚﾒﾝﾄなし？
				pW->pLas = NULL;												//			ラストポインタ＝ＮＵＬＬ
			}
			if (pBuf != NULL) memcpy(pBuf, (pCur + 1), __min(bfl, pCur->len));	//		バッファへノードデータコピー
			rc = pCur->len;														//		戻り値＝ノードデータサイズ
			free(pCur);															//		取り外したノードを開放
			pW->num--;															//		ノード数減算
		}
		else {																	//	ノードなし？
			rc = 0;																//		戻り値＝ノードなし
		}
		AJCVQUE_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	先頭ノードを取り出す（新）																					//
//																												//
//	引　数	：	pW 	 - インスタンスハンドル																		//
//				pBuf - 取り出したノードのデータを格納するバッファのアドレス（不要時はＮＵＬＬ）					//
//				bfl  - 取り出したノードのデータを格納するバッファの容量(バイト数）								//
//																												//
//	戻り値	：	≠０：取り出したノードデータのバイト数															//
//				＝-1：ノードなし／エラー																		//
//==============================================================================================================//
AJCEXPORT	UI	WINAPI	AjcVQueDequeEx	(HAJCVQUE pW, VOP pBuf, UI bfl)
{
	UI			rc = -1;
	PAJCVQNODE	pCur;

	if (IS_MY_INST(pW)) {
		AJCVQUE_ENTER
		if ((pCur = pW->pTop) != NULL) {										//	ノードあり？
			if ((pW->pTop = pCur->pNxt) == NULL) {								//		先頭ｴﾚﾒﾝﾄ取り外し，次のｴﾚﾒﾝﾄなし？
				pW->pLas = NULL;												//			ラストポインタ＝ＮＵＬＬ
			}
			if (pBuf != NULL) memcpy(pBuf, (pCur + 1), __min(bfl, pCur->len));	//		バッファへノードデータコピー
			rc = pCur->len;														//		戻り値＝ノードデータサイズ
			free(pCur);															//		取り外したノードを開放
			pW->num--;															//		ノード数減算
		}
		else {																	//	ノードなし？
			rc = -1;															//		戻り値＝ノードなし
		}
		AJCVQUE_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	全ノードを消去する																							//
//																												//
//	引　数	：	pW 		  - インスタンスハンドル																//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	BOOL	  WINAPI	AjcVQuePurge(HAJCVQUE pW)
{
	BOOL		rc = FALSE;
	PAJCVQNODE	pCur, svp;

	if (IS_MY_INST(pW)) {
		AJCVQUE_ENTER
		pCur = pW->pTop;									//	先頭ＢＬＫアドレス設定
		while (pCur) {										//	全ＢＬＫループ
			if (pW->cbRemove) {								//		消去通知コールバックあり？
				pW->cbRemove(pCur + 1, pCur->len, pW->cbp);	//			コールバック
			}
			svp = pCur;										//		ＢＬＫアドレス退避
			pCur = pCur->pNxt;								//		次のＢＬＫアドレス設定
			free(svp);										//		ＢＬＫ開放
		}
		pW->pTop = pW->pLas = NULL;							//	キューチェインポインタ初期化
		pW->num = 0;										//	ノード数＝０
		AJCVQUE_LEAVE
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	現在のノード数を得る																						//
//																												//
//	引　数	：	pW 	 - インスタンスハンドル																		//
//																												//
//	戻り値	：	現在のノード数																					//
//==============================================================================================================//
AJCEXPORT	UI	  WINAPI	AjcVQueGetCount	(HAJCVQUE pW)
{
	UI		rc = 0;

	if (IS_MY_INST(pW)) {
		AJCVQUE_ENTER
		rc = pW->num;
		AJCVQUE_LEAVE
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	UI	  WINAPI	AjcVQueNumber	(HAJCVQUE pW)
{
	return AjcVQueGetCount(pW);
}

//==============================================================================================================//
//	２つのＬＳＴ構造を連結する																					//
//																												//
//	引　数	：	pW 	 - インスタンスハンドル																		//
//				pSrc - 連結するＬＳＴのハンドル（このＬＳＴは空になります）										//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcVQueMerge	(HAJCVQUE pW, HAJCVQUE pSrc)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW) && IS_MY_INST(pSrc)) {
		AJCVQUE_ENTER
		if (pSrc->fMultiThread) EnterCriticalSection(&pSrc->cs);
		if (pW->pLas == 0) {				//	被連結側にノードなし？
			pW->pTop = pSrc->pTop;			//		連結するチェイン情報を、被連結側へコピー
			pW->pLas = pSrc->pLas;			//		・
			pW->num  = pSrc->num;			//		・
		}
		else if (pSrc->num) {				//	被連結側にノードあり & 連結するノードあり？
			pW->pLas->pNxt = pSrc->pTop;	//		連結するチェイン情報を、被連結側の末尾へ追加
			pW->pLas	   = pSrc->pLas;	//		・
			pW->num += pSrc->num;			//		・
		}
		pSrc->pTop = NULL;					//	連結したチェイン情報をリセット
		pSrc->pLas = NULL;					//	・
		pSrc->num  = 0;						//	・
		if (pSrc->fMultiThread) LeaveCriticalSection(&pSrc->cs);
		AJCVQUE_LEAVE
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	先頭ノードデータのアドレスを得る																			//
//																												//
//	引　数	：	pW 	   - インスタンスハンドル																	//
//				pBytes - 先頭ノードデータのバイト数を格納するバッファのアドレス									//
//																												//
//	戻り値	：	≠ＮＵＬＬ：先頭ノードデータのアドレス															//
//				＝ＮＵＬＬ：ノードなし																			//
//==============================================================================================================//
AJCEXPORT	VOP   WINAPI	AjcVQueTopNode	(HAJCVQUE pW, UIP pBytes)
{
	VOP		rc = NULL;

	if (IS_MY_INST(pW)) {
		AJCVQUE_ENTER
		if (pW->pTop != NULL) {								//	先頭ノードあり？
			if (pBytes != NULL) *pBytes = pW->pTop->len;	//		データバイト数設定
			rc = pW->pTop + 1;								//		戻り値＝先頭ノードのデータアドレス
		}
		AJCVQUE_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	次のノードデータのアドレスを得る																			//
//																												//
//	引　数	：	pW 			 - インスタンスハンドル																//
//				pCurrentNode - 現在のノードデータアドレス														//
//				pBytes		 - 次ノードデータのバイト数を格納するバッファのアドレス								//
//																												//
//	戻り値	：	≠ＮＵＬＬ：次のノードデータのアドレス															//
//				＝ＮＵＬＬ：ノードなし																			//
//==============================================================================================================//
AJCEXPORT	VOP   WINAPI	AjcVQueNextNode	(HAJCVQUE pW, C_VOP pCurrentNode, UIP pBytes)
{
	VOP			rc = NULL;
	PAJCVQNODE	pNode;

	if (IS_MY_INST(pW)) {
		AJCVQUE_ENTER
		if (pCurrentNode) {									//	現ノードあり？
			pNode = (PAJCVQNODE)pCurrentNode;				//		現ノードのノード先頭アドレス設定
			pNode--;										//		・
			if (pNode = pNode->pNxt) {						//		次ノードあり？
				if (pBytes != NULL) *pBytes = pNode->len;	//			データバイト数設定
				rc = pNode + 1;								//			戻り値＝次ノードのデータアドレス
			}
		}
		AJCVQUE_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	全ノードへのポインタ配列生成																				//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				pNum		- 配列要素数を格納するバッファのアドレス											//
//																												//
//	戻り値	：	≠NULL : 全ノードへのポインタ配列の先頭アドレス													//
//				＝NULL : エラー																					//
//==============================================================================================================//
AJCEXPORT	PCAJCVQUEPTR  WINAPI AjcVQueCreatePtrArr(HAJCVQUE pW, UIP pNum)
{
	PCAJCVQUEPTR rc    = NULL;
	PAJCVQUEPTR  pArr  = NULL;
	VOP		 	 pNode = NULL;
	UI		 	 ix    = 0;
	UI			 len;

	if (IS_MY_INST(pW)) {
		AJCVQUE_ENTER
		if (pArr  =  (PAJCVQUEPTR)AJCMEM(sizeof(AJCVQUEPTR) * (pW->num + 1))) {
			memset(pArr, 0, sizeof(AJCVQUEPTR) * (pW->num + 1));
			if (pNode = AjcVQueTopNode(pW, &len)) {
				do {
					pArr[ix].pNode = pNode;
					pArr[ix].len   = len;
					ix++;
				} while (pNode = AjcVQueNextNode(pW, pNode, &len));
			}
			if (pNum != NULL) *pNum = ix;
			rc = pArr;
		}
		AJCVQUE_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	全ノードへのポインタ配列解放																				//
//																												//
//	引　数	：	pW			 - インスタンスハンドル																//
//				pArr		 - 全ノードへのポインタ配列の先頭アドレス											//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	VO	 			WINAPI AjcVQueReleasePtrArr(HAJCVQUE pW, PCAJCVQUEPTR pArr)
{
	if (IS_MY_INST(pW)) {
		AJCVQUE_ENTER
		if (pArr != NULL) free((VOP)pArr);
		AJCVQUE_LEAVE
	}
}

