﻿#include	"AjcInternal.h"

//**************************************************************************************************************//
//																												//
//	可変長キュー（双方向リスト）																				//
//																												//
//**************************************************************************************************************//


#define	INST_ID			0x7264198A
#define	IS_MY_INST(P)	(P != NULL && P->InstID == INST_ID)


//--------------------------------------------------------------------------------------------------------------//
//	内部マクロ																									//
//--------------------------------------------------------------------------------------------------------------//
#define		AJCXQUE_ENTER		if (pW->fMultiThread) EnterCriticalSection(&pW->cs);
#define		AJCXQUE_LEAVE		if (pW->fMultiThread) LeaveCriticalSection(&pW->cs);


//==============================================================================================================//
//	インスタンス生成																							//
//																												//
//	引　数	：	cbp 	 - コールバックパラメタ																	//
//				cbRemove - ノード消去通知用コールバック（不要時はＮＵＬＬ）										//
//																												//
//	戻り値	：	≠ＮＵＬＬ：ＯＫ（インスタンスハンドル（＝インスタンスワークアドレス））						//
//				＝ＮＵＬＬ：エラー																				//
//==============================================================================================================//
AJCEXPORT HAJCXQUE	WINAPI	AjcXQueCreate(UX cbp, VO (CALLBACK *cbRemove)(VOP pDat, UI len, UX cbp))
{
	HAJCXQUE  pW;

	if ((pW = (HAJCXQUE)AJCMEM(sizeof(AJCXQUE))) != NULL) {
		memset(pW, 0, sizeof *pW);
		pW->InstID	= INST_ID;
		pW->pTop	= NULL;
		pW->pLas	= NULL;
		pW->num		= 0;
		pW->cbp 	= cbp;
		pW->cbRemove= cbRemove;
	}
	return pW;
}
//==============================================================================================================//
//	インスタンス消去																							//
//																												//
//	引　数	：	pW 		  - インスタンスハンドル																//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT BOOL	WINAPI	AjcXQueDelete(HAJCXQUE pW)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		AjcXQuePurge(pW);
		if (pW->fMultiThread) {
			DeleteCriticalSection(&pW->cs);
		}
		free(pW);
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	マルチスレッドの許可／禁止																					//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//				fEnable - FALSE : マルチスレッド禁止															//
//						  TRUE	: マルチスレッド許可															//
//																												//
//	戻り値	：	前回の許可／禁止状態																			//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcXQueEnableMultiThread(HAJCXQUE pW, BOOL fEnable)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		rc = pW->fMultiThread;
		if (fEnable) {
			if (!pW->fMultiThread) {
				InitializeCriticalSection(&pW->cs);
				pW->fMultiThread = TRUE;
			}
		}
		else {
			if (pW->fMultiThread) {
				pW->fMultiThread = FALSE;
				DeleteCriticalSection(&pW->cs);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	末尾へノード追加																							//
//																												//
//	引　数	：	pW 	 - インスタンスハンドル																		//
//				pDat - 末尾へ挿入するノードデータのアドレス														//
//				len  - 末尾へ挿入するノードデータのバイト数														//
//																												//
//	戻り値	：	≠NULL - 追加したノードのアドレス																//
//				＝NULL - エラー																					//
//==============================================================================================================//
AJCEXPORT	VOP		WINAPI	AjcXQueEnque	(HAJCXQUE pW, C_VOP pDat, UI len)
{
	VOP		rc = FALSE;
	PAJCXQNODE	pNew;

	if (IS_MY_INST(pW)) {
		AJCXQUE_ENTER
		if ((pNew = (PAJCXQNODE)AJCMEM(sizeof(AJCXQNODE) + len)) != NULL) {	//	メモリブロック割り当て，成功？
			pNew->id   = AJCXQUEID;											//		ノードＩＤ
			pNew->len  = len;												//		データ長
			if (pDat != NULL) memcpy(pNew + 1, pDat, len);					//		データ
			else              memset(pNew + 1, 0   , len);					//		・
			if (pW->pLas == NULL) {											//		初回キューイング？
				pNew->pBfr = NULL;											//			新ＢＬＫを初キューイング
				pNew->pNxt = NULL;											//			・
				pW->pTop = pW->pLas = pNew;									//			・
			}
			else {															//		追加キューイング？
				pNew->pBfr     = pW->pLas;									//			新ＢＬＫを末尾へキューイング
				pNew->pNxt     = NULL;										//			・
				pW->pLas->pNxt = pNew;										//			・
				pW->pLas	   = pNew;										//			・
			}
			pW->num++;														//		ノード数更新
			rc = pNew + 1;													//		戻り値＝ＯＫ
		}
		else {																//	メモリブロック割り当て，失敗？
			rc = NULL;														//		戻り値＝エラー
		}
		AJCXQUE_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	先頭へノード挿入																							//
//																												//
//	引　数	：	pW 	 - インスタンスハンドル																		//
//				pDat - 先頭へ挿入するノードデータのアドレス														//
//				len  - 先頭へ挿入するノードデータのバイト数														//
//																												//
//	戻り値	：	≠NULL - 追加したノードのアドレス																//
//				＝NULL - エラー																					//
//==============================================================================================================//
AJCEXPORT	VOP	WINAPI	AjcXQueEnqTop	(HAJCXQUE pW, C_VOP pDat, UI len)
{
	VOP			rc = NULL;
	PAJCXQNODE	pNew;

	if (IS_MY_INST(pW)) {
		AJCXQUE_ENTER
		if ((pNew = (PAJCXQNODE)AJCMEM(sizeof(AJCXQNODE) + len)) != NULL) {	//	メモリブロック割り当て，成功？
			pNew->id  = AJCXQUEID;											//		ノードＩＤ
			pNew->len = len;												//		データ長
			if (pDat != NULL) memcpy(pNew + 1, pDat, len);					//		データ
			else              memset(pNew + 1, 0   , len);					//		・
			if (pW->pTop == NULL) {											//		初回キューイング？
				pNew->pBfr     = NULL;										//			新ＢＬＫを初キューイング
				pNew->pNxt     = NULL;										//			・
				pW->pTop       = pW->pLas = pNew;							//			・
			}
			else {															//		追加キューイング？
				pNew->pBfr     = NULL;										//			新ＢＬＫを先頭へキューイング
				pNew->pNxt     = pW->pTop;									//			・
				pW->pTop->pBfr = pNew;										//			・
				pW->pTop       = pNew;										//			・
			}
			pW->num++;														//		ノード数更新
			rc = pNew + 1;													//		戻り値＝ＯＫ
		}
		else {																//	メモリブロック割り当て，失敗？
			rc = NULL;														//		戻り値＝エラー
		}
		AJCXQUE_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	指定ノードの直前へノード挿入																				//
//																												//
//	引　数	：	pW 	    - インスタンスハンドル																	//
//				pDat    - 先頭へ挿入するノードデータのアドレス													//
//				len     - 先頭へ挿入するノードデータのバイト数													//
//				pInsPtr - 挿入位置ノードのノードデータアドレス（NULL：末尾へ挿入）								//
//																												//
//	戻り値	：	≠NULL - 追加したノードのアドレス																//
//				＝NULL - エラー																					//
//==============================================================================================================//
AJCEXPORT	VOP	WINAPI	AjcXQueInsert	(HAJCXQUE pW, C_VOP pDat, UI len, C_VOP pInsPtr)
{
	VOP			rc = NULL;
	PAJCXQNODE	pNew;
	PAJCXQNODE	pIns;

	if (IS_MY_INST(pW)) {
		//	末尾へ挿入
		if (pInsPtr == NULL) {
			rc = AjcXQueEnque(pW, pDat, len);
		}
		//	指定ノードの直前に挿入
		else {
			pIns = ((PAJCXQNODE)pInsPtr) - 1;
			if (pIns->id == AJCXQUEID) {
				AJCXQUE_ENTER
				if ((pNew = (PAJCXQNODE)AJCMEM(sizeof(AJCXQNODE) + len)) != NULL) {	//	メモリブロック割り当て，成功？
					pNew->id  = AJCXQUEID;											//		ノードＩＤ
					pNew->len = len;												//		データ長
					if (pDat != NULL) memcpy(pNew + 1, pDat, len);					//		データ
					else              memset(pNew + 1, 0   , len);					//		・
					if      (pIns->pBfr == NULL) {									//		挿入ポイントは先頭ノード？
						rc = AjcXQueEnqTop(pW, pDat, len);							//			ノードを先頭へ挿入
					}
					else {															//		挿入ポイントは先頭ノード以外？
						pNew->pBfr       = pIns->pBfr;								//			新ＢＬＫを連結
						pNew->pNxt       = pIns;									//			・
						pNew->pBfr->pNxt = pNew;									//			・
						pNew->pNxt->pBfr = pNew;									//			・
						pW->num++;													//			ノード数更新
						rc = pNew + 1;												//			戻り値＝ＯＫ
					}
				}
				else {																//	メモリブロック割り当て，失敗？
					rc = NULL;														//		戻り値＝エラー
				}
				AJCXQUE_LEAVE
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	指定ノードを削除																							//
//																												//
//	引　数	：	pW 	    - インスタンスハンドル																	//
//				pDelPtr - 削除するノードのデータアドレス														//
//																												//
//	戻り値	：	≠-1：取り出したノードデータのバイト数															//
//				＝-1：ノードなし／エラー																		//
//==============================================================================================================//
AJCEXPORT	UI	WINAPI	AjcXQueRemove	(HAJCXQUE pW, C_VOP pDelPtr)
{
	UI			rc = -1;
	PAJCXQNODE	pDel;

	if (IS_MY_INST(pW) && pDelPtr != NULL) {
		AJCXQUE_ENTER
		pDel = ((PAJCXQNODE)pDelPtr) - 1;
		if (pW->pTop != NULL && pDel->id == AJCXQUEID) {						//	ノードあり？
			//	後方ポインタ変更
			if (pDel->pBfr) {
				pDel->pBfr->pNxt = pDel->pNxt;
			}
			else {
				pW->pTop = pDel->pNxt;
			}
			//	前方ポインタ変更
			if (pDel->pNxt) {
				pDel->pNxt->pBfr = pDel->pBfr;
			}
			else {
				pW->pLas = pDel->pBfr;
			}
			//	ノードＩＤ無効化（コールバックからのノード多重削除防止）
			pDel->id = 0x12345678;
			//	消去通知コールバック
			if (pW->cbRemove) {
				pW->cbRemove(pDel + 1, pDel->len, pW->cbp);
			}
			//	自ノード開放
			free(pDel);															//		取り外したノードを開放
			//	ノード数減算
			pW->num--;															//		
		}
		else {																	//	ノードなし？
			rc = -1;															//		戻り値＝ノードなし
		}
		AJCXQUE_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	全ノードを消去する																							//
//																												//
//	引　数	：	pW 		  - インスタンスハンドル																//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	BOOL	  WINAPI	AjcXQuePurge(HAJCXQUE pW)
{
	BOOL		rc = FALSE;
	PAJCXQNODE	pCur, svp;

	if (IS_MY_INST(pW)) {
		AJCXQUE_ENTER
		pCur = pW->pTop;									//	先頭ＢＬＫアドレス設定
		while (pCur) {										//	全ＢＬＫループ
			pCur->id = 0x12345678;							//		ノードＩＤ無効化
			if (pW->cbRemove) {								//		消去通知コールバックあり？
				pW->cbRemove(pCur + 1, pCur->len, pW->cbp);	//			コールバック
			}
			svp = pCur;										//		ＢＬＫアドレス退避
			pCur = pCur->pNxt;								//		次のＢＬＫアドレス設定
			free(svp);										//		ＢＬＫ開放
		}
		pW->pTop = pW->pLas = NULL;							//	キューチェインポインタ初期化
		pW->num = 0;										//	ノード数＝０
		AJCXQUE_LEAVE
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	先頭ノードを取り出す																						//
//																												//
//	引　数	：	pW 	 - インスタンスハンドル																		//
//				pBuf - 取り出したノードのデータを格納するバッファのアドレス（不要時はＮＵＬＬ）					//
//				bfl  - 取り出したノードのデータを格納するバッファの容量(バイト数）								//
//																												//
//	戻り値	：	≠-1：取り出したノードデータのバイト数															//
//				＝-1：ノードなし／エラー																		//
//==============================================================================================================//
AJCEXPORT	UI	WINAPI	AjcXQueDeque	(HAJCXQUE pW, VOP pBuf, UI bfl)
{
	UI			rc = -1;
	PAJCXQNODE	pCur;

	if (IS_MY_INST(pW)) {
		AJCXQUE_ENTER
		if ((pCur = pW->pTop) != NULL) {										//	ノードあり？
			if ((pW->pTop = pCur->pNxt) == NULL) {								//		先頭ノード取り外し，次のノードなし？
				pW->pLas = NULL;												//			ラストポインタ＝ＮＵＬＬ
			}
			else {																//		次のノードあり？
				pCur->pNxt->pBfr = NULL;										//			新先頭ノードのバックポインタ
			}
			if (pBuf != NULL) memcpy(pBuf, (pCur + 1), __min(bfl, pCur->len));	//		バッファへノードデータコピー
			rc = pCur->len;														//		戻り値＝ノードデータサイズ
			free(pCur);															//		取り外したノードを開放
			pW->num--;															//		ノード数減算
		}
		else {																	//	ノードなし？
			rc = -1;															//		戻り値＝ノードなし
		}
		AJCXQUE_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	現在のノード数を得る																						//
//																												//
//	引　数	：	pW 	 - インスタンスハンドル																		//
//																												//
//	戻り値	：	現在のノード数																					//
//==============================================================================================================//
AJCEXPORT	UI	  WINAPI	AjcXQueGetCount	(HAJCXQUE pW)
{
	UI		rc = 0;

	if (IS_MY_INST(pW)) {
		AJCXQUE_ENTER
		rc = pW->num;
		AJCXQUE_LEAVE
	}
	return rc;
}

//==============================================================================================================//
//	２つのＬＳＴ構造を連結する																					//
//																												//
//	引　数	：	pW 	 - インスタンスハンドル																		//
//				pSrc - 連結するＬＳＴのハンドル（このＬＳＴは空になります）										//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcXQueMerge	(HAJCXQUE pW, HAJCXQUE pSrc)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW) && IS_MY_INST(pSrc)) {
		AJCXQUE_ENTER
		if (pSrc->fMultiThread) EnterCriticalSection(&pSrc->cs);
		if (pW->pLas == 0) {				//	被連結側にノードなし？
			pW->pTop = pSrc->pTop;			//		連結するチェイン情報を、被連結側へコピー
			pW->pLas = pSrc->pLas;			//		・
			pW->num  = pSrc->num;			//		・
		}
		else if (pSrc->num) {				//	被連結側にノードあり & 連結するノードあり？
			pW->pLas->pNxt   = pSrc->pTop;	//		連結するチェイン情報を、被連結側の末尾へ追加
			pW->pLas	     = pSrc->pLas;	//		・
			pSrc->pTop->pBfr = pW->pLas;	//		・
			pW->num += pSrc->num;			//		・
		}
		pSrc->pTop = NULL;					//	連結したチェイン情報をリセット
		pSrc->pLas = NULL;					//	・
		pSrc->num  = 0;						//	・
		if (pSrc->fMultiThread) LeaveCriticalSection(&pSrc->cs);
		AJCXQUE_LEAVE
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	先頭ノードデータのアドレスを得る																			//
//																												//
//	引　数	：	pW 	   - インスタンスハンドル																	//
//				pBytes - 先頭ノードデータのバイト数を格納するバッファのアドレス									//
//																												//
//	戻り値	：	≠ＮＵＬＬ：先頭ノードデータのアドレス															//
//				＝ＮＵＬＬ：ノードなし																			//
//==============================================================================================================//
AJCEXPORT	VOP   WINAPI	AjcXQueTopNode	(HAJCXQUE pW, UIP pBytes)
{
	VOP		rc = NULL;

	if (IS_MY_INST(pW)) {
		AJCXQUE_ENTER
		if (pW->pTop != NULL) {								//	先頭ノードあり？
			if (pBytes != NULL) *pBytes = pW->pTop->len;	//		データバイト数設定
			rc = pW->pTop + 1;								//		戻り値＝先頭ノードのデータアドレス
		}
		AJCXQUE_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	末尾ノードデータのアドレスを得る																			//
//																												//
//	引　数	：	pW 	   - インスタンスハンドル																	//
//				pBytes - 末尾ノードデータのバイト数を格納するバッファのアドレス									//
//																												//
//	戻り値	：	≠ＮＵＬＬ：末尾ノードデータのアドレス															//
//				＝ＮＵＬＬ：ノードなし																			//
//==============================================================================================================//
AJCEXPORT	VOP   WINAPI	AjcXQueLastNode	(HAJCXQUE pW, UIP pBytes)
{
	VOP		rc = NULL;

	if (IS_MY_INST(pW)) {
		AJCXQUE_ENTER
		if (pW->pTop != NULL) {								//	先頭ノードあり？
			if (pBytes != NULL) *pBytes = pW->pLas->len;	//		データバイト数設定
			rc = pW->pLas + 1;								//		戻り値＝先頭ノードのデータアドレス
		}
		AJCXQUE_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	次のノードデータのアドレスを得る																			//
//																												//
//	引　数	：	pW 			 - インスタンスハンドル																//
//				pCurrentNode - 現在のノードデータアドレス														//
//				pBytes		 - 次ノードデータのバイト数を格納するバッファのアドレス								//
//																												//
//	戻り値	：	≠ＮＵＬＬ：次のノードデータのアドレス															//
//				＝ＮＵＬＬ：ノードなし																			//
//==============================================================================================================//
AJCEXPORT	VOP   WINAPI	AjcXQueNextNode	(HAJCXQUE pW, C_VOP pCurrentNode, UIP pBytes)
{
	VOP			rc = NULL;
	PAJCXQNODE	pNode;

	if (IS_MY_INST(pW)) {
		AJCXQUE_ENTER
		if (pCurrentNode) {									//	現ノードあり？
			pNode = (PAJCXQNODE)pCurrentNode;				//		現ノードのノード先頭アドレス設定
			pNode--;										//		・
			if (pNode = pNode->pNxt) {						//		次ノードあり？
				if (pBytes != NULL) *pBytes = pNode->len;	//			データバイト数設定
				rc = pNode + 1;								//			戻り値＝次ノードのデータアドレス
			}
		}
		AJCXQUE_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	直前のノードデータのアドレスを得る																			//
//																												//
//	引　数	：	pW 			 - インスタンスハンドル																//
//				pCurrentNode - 現在のノードデータアドレス														//
//				pBytes		 - 次ノードデータのバイト数を格納するバッファのアドレス								//
//																												//
//	戻り値	：	≠ＮＵＬＬ：直前のノードデータのアドレス														//
//				＝ＮＵＬＬ：ノードなし																			//
//==============================================================================================================//
AJCEXPORT	VOP   WINAPI	AjcXQuePrevNode	(HAJCXQUE pW, C_VOP pCurrentNode, UIP pBytes)
{
	VOP			rc = NULL;
	PAJCXQNODE	pNode;

	if (IS_MY_INST(pW)) {
		AJCXQUE_ENTER
		if (pCurrentNode) {									//	現ノードあり？
			pNode = (PAJCXQNODE)pCurrentNode;				//		現ノードのノード先頭アドレス設定
			pNode--;										//		・
			if (pNode = pNode->pBfr) {						//		次ノードあり？
				if (pBytes != NULL) *pBytes = pNode->len;	//			データバイト数設定
				rc = pNode + 1;								//			戻り値＝次ノードのデータアドレス
			}
		}
		AJCXQUE_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	全ノードへのポインタ配列生成																				//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				pNum		- 配列要素数を格納するバッファのアドレス											//
//																												//
//	戻り値	：	≠NULL : 全ノードへのポインタ配列の先頭アドレス													//
//				＝NULL : エラー																					//
//==============================================================================================================//
AJCEXPORT	PCAJCXQUEPTR  WINAPI AjcXQueCreatePtrArr(HAJCXQUE pW, UIP pNum)
{
	PCAJCXQUEPTR rc    = NULL;
	PAJCXQUEPTR  pArr  = NULL;
	VOP		 	 pNode = NULL;
	UI		 	 ix    = 0;
	UI			 len;

	if (IS_MY_INST(pW)) {
		AJCXQUE_ENTER
		if (pArr  =  (PAJCXQUEPTR)AJCMEM(sizeof(AJCXQUEPTR) * (pW->num + 1))) {
			memset(pArr, 0, sizeof(AJCXQUEPTR) * (pW->num + 1));
			if (pNode = AjcXQueTopNode(pW, &len)) {
				do {
					pArr[ix].pNode = pNode;
					pArr[ix].len   = len;
					ix++;
				} while (pNode = AjcXQueNextNode(pW, pNode, &len));
			}
			if (pNum != NULL) *pNum = ix;
			rc = pArr;
		}
		AJCXQUE_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	全ノードへのポインタ配列解放																				//
//																												//
//	引　数	：	pW			 - インスタンスハンドル																//
//				pArr		 - 全ノードへのポインタ配列の先頭アドレス											//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	VO	 			WINAPI AjcXQueReleasePtrArr(HAJCXQUE pW, PCAJCXQUEPTR pArr)
{
	if (IS_MY_INST(pW)) {
		AJCXQUE_ENTER
		if (pArr != NULL) free((VOP)pArr);
		AJCXQUE_LEAVE
	}
}

