﻿#include	"AjcInternal.h"

//**************************************************************************************************************//
//																												//
//	リングバッファ																								//
//																												//
//**************************************************************************************************************//


#define	INST_ID			0xBD659FA6
#define	IS_MY_INST(P)	(P != NULL && P->InstID == INST_ID)


//==============================================================================================================//
//	内部マクロ																									//
//==============================================================================================================//
//----- リングバッファインデクス更新 ---------------------------------------------------------------------------//
#define		UPD_RINGIDX(_ix_, _len_, _siz_)	_ix_ += _len_;	if (_ix_ >= _siz_) _ix_ -= _siz_

//----- 排他制御 -----------------------------------------------------------------------------------------------//
#define		AJCRNG_ENTER	if (pW->fMultiThread) EnterCriticalSection(&pW->cs);
#define		AJCRNG_LEAVE	if (pW->fMultiThread) LeaveCriticalSection(&pW->cs);

//==============================================================================================================//
//	内部サブ関数																								//
//==============================================================================================================//
static	UI	SubPutData	   (HAJCRNG pW, C_VOP pDat, UI lDat);
static	UI	SubPutDataEx   (HAJCRNG pW, C_VOP pDat, UI lDat);
static	UI	SubGetData	   (HAJCRNG pW, VOP pBuf, UI lDat);
static	UI	SubPeekData    (HAJCRNG pW, VOP pBuf, UI lDat);
static	VO	FreeAllQue	   (LPAJCRNG pW);
static	UI	GetRingDataSize(UI six, UI eix, UI RingBufSize, UIP plDat);
static	UI	GetRingFreeSize(UI six, UI eix, UI RingBufSize, UIP plDat);

//==============================================================================================================//
//	リングバッファ・インスタンス生成																			//
//																												//
//	引　数	：	pStaticBuf- スタティックなリングバッファのアドレス（不要時はNULL)								//
//				lStaticBuf- スタティックなリングバッファのバイト数（２以上，pStaticBuf=NULL時は不要）			//
//				ChunkSize - リングバッファチャンクのバイト数													//
//																												//
//	戻り値	：	≠ＮＵＬＬ - 成功（インスタンスハンドル）														//
//				＝ＮＵＬＬ - エラー																				//
//==============================================================================================================//
AJCEXPORT	HAJCRNG	WINAPI	AjcRngCreate	 (VOP pStaticBuf, UI lStaticBuf, UI ChunkSize)
{
	HAJCRNG	pW = NULL;

	if (pStaticBuf == NULL	||	lStaticBuf >= 2) {
		if ((pW = (HAJCRNG)AJCMEM(sizeof(AJCRNG))) != NULL) {
			memset(pW, 0, sizeof *pW);
			pW->InstID	   = INST_ID;							//	インスタンス識別ＩＤ
			pW->DataSize   = 0;									//	リングバッファに格納されているデータのバイト数
			pW->pStaticBuf = pStaticBuf;						//	スタティックなリングバッファのアドレス
			pW->lStaticBuf = lStaticBuf;						//	スタティックなリングバッファのバイト数
			pW->stSix = pW->stEix = 0;							//	スタティック・リングバッファ・インデクス
			pW->pQTop	= 0;									//	先頭キューデータへのポインタ
			pW->pQEnd	= 0;									//	末尾キューデータへのポインタ
			pW->pQBfr	= 0;									//	末尾キューの直前キューデータへのポインタ

			pW->siz		= (ChunkSize >= (AJCRNGQUE_HEADER_SIZE + 2) ? ChunkSize - AJCRNGQUE_HEADER_SIZE : 0);
			pW->fMultiThread = FALSE;
		}
	}

	return pW;
}
//==============================================================================================================//
//	リングバッファ・インスタンス消去																			//
//																												//
//	引　数	：	pW		  - インスタンスハンドル																//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcRngDelete(HAJCRNG pW)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		AjcRngPurge(pW);
		if (pW->fMultiThread) {
			DeleteCriticalSection(&pW->cs);
		}
		free(pW);
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	マルチスレッドの許可／禁止																					//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//				fEnable - FALSE : マルチスレッド禁止															//
//						  TRUE	: マルチスレッド許可															//
//																												//
//	戻り値	：	前回の許可／禁止状態																			//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcRngEnableMultiThread(HAJCRNG pW, BOOL fEnable)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		rc = pW->fMultiThread;
		if (fEnable) {
			if (!pW->fMultiThread) {
				InitializeCriticalSection(&pW->cs);
				pW->fMultiThread = TRUE;
			}
		}
		else {
			if (pW->fMultiThread) {
				pW->fMultiThread = FALSE;
				DeleteCriticalSection(&pW->cs);
			}
		}
	}
	return rc;
}

//==============================================================================================================//
//	リングバッファへデータ格納（格納しきれない場合は、格納できる分だけ格納する）								//
//																												//
//	引　数	：	pW	 - インスタンスハンドル																		//
//				pDat - 格納するデータのアドレス																	//
//				lDat - 格納するデータのバイト数																	//
//																												//
//	戻り値	：	実際に格納されたデータバイト数																	//
//==============================================================================================================//
AJCEXPORT	UI	WINAPI	AjcRngPutData	 (HAJCRNG pW, C_VOP pDat, UI lDat)
{
	UI	rc = 0;
	UI	i, fre;
	UI	len[2];

	if (IS_MY_INST(pW) && pDat != NULL) {
		AJCRNG_ENTER
		//--- スタティックリングバッファへデータ格納 ----------------------//
		if (pW->pStaticBuf != NULL) {
			fre = GetRingFreeSize(pW->stSix, pW->stEix, pW->lStaticBuf, len);
			if (fre != 0) {
				for (i=0; i<2 && lDat && len[i]; i++) {
					if (len[i] >= lDat) {
						memcpy(&pW->pStaticBuf[pW->stEix], pDat, lDat);
						((C_UBP)pDat) += lDat;
						UPD_RINGIDX(pW->stEix, lDat, pW->lStaticBuf);
						pW->DataSize += lDat;
						rc			 += lDat;
						lDat		  = 0;
					}
					else {
						memcpy(&pW->pStaticBuf[pW->stEix], pDat, len[i]);
						((C_UBP)pDat) += len[i];
						UPD_RINGIDX(pW->stEix, len[i], pW->lStaticBuf);
						pW->DataSize  += len[i];
						rc			  += len[i];
						lDat		  -= len[i];
					}
				}
			}
		}
		//--- 残データをリングバッファキューへ格納 ------------------------//
		if (lDat != 0) {
			if (pW->siz != 0) {
				rc += SubPutData(pW, pDat, lDat);
			}
		}
		AJCRNG_LEAVE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	リングバッファキューへデータ格納（格納しきれない場合は、格納できる分だけ格納する）							//
//																												//
//	引　数	：	pW	 - インスタンスハンドル																		//
//				pDat - 格納するデータのアドレス																	//
//				lDat - 格納するデータのバイト数																	//
//																												//
//	戻り値	：	実際に格納されたデータバイト数																	//
//--------------------------------------------------------------------------------------------------------------//
static	UI	SubPutData	  (HAJCRNG pW, C_VOP pDat, UI lDat)
{
	UI			rc = 0;
	PAJCRNGQUE	pQue;
	UI			fre;
	UI			i;
	UI			len[2];

	do {
		if (lDat == 0) {
			break;
		}

		//----- リングバッファキューが無ければ、1つ確保する ---------------------------------------------------//
		if (pW->pQTop == 0) {
			if ((pW->pQTop = (PAJCRNGQUE)AJCMEM(AJCRNGQUE_HEADER_SIZE + pW->siz)) != 0) {
				pQue = pW->pQTop;
				pQue->pNxt = NULL;
				pQue->six  = pQue->eix = 0;
				pW->pQEnd  = pW->pQTop;
			}
			else {
				break;
			}
		}
		//----- 既にリングバッファキューが存在する場合は、末尾キュを割り当てる ---------------------------------//
		else {
			pQue = pW->pQEnd;
		}
		//----- 現リングバッファにデータを格納 -----------------------------------------------------------------//
		fre = GetRingFreeSize(pQue->six, pQue->eix, pW->siz, len);
		for (i=0; i<2 && lDat && len[i]; i++) {
			if (len[i] >= lDat) {
				memcpy(&pQue->dat[pQue->eix], pDat, lDat);
				((C_UBP)pDat)		+= lDat;
				UPD_RINGIDX(pQue->eix, lDat, pW->siz);
				pW->DataSize		+= lDat;
				rc					+= lDat;
				lDat = 0;
			}
			else {
				memcpy(&pQue->dat[pQue->eix], pDat, len[i]);
				((C_UBP)pDat)		+= len[i];
				UPD_RINGIDX(pQue->eix, len[i], pW->siz);
				pW->DataSize		+= len[i];
				rc					+= len[i];
				lDat				-= len[i];
			}
		}
		//----- 全データがリングバッファに収まった場合は、終了する ---------------------------------------------//
		if (lDat == 0) {
			break;
		}
		//----- 残格納データがある場合は、キューを拡張しながら、データを格納する -------------------------------//
		while (lDat) {
			if ((pQue = (PAJCRNGQUE)AJCMEM(AJCRNGQUE_HEADER_SIZE + pW->siz)) != NULL) {
				//----- 新キューエントリを初期化する -----------------------------------------------------------//
				pQue->pNxt = NULL;
				pQue->six  = pQue->eix = 0;
				//----- 新キューエントリを末尾へ追加 -----------------------------------------------------------//
				pW->pQEnd->pNxt = pQue;
				pW->pQBfr		= pW->pQEnd;
				pW->pQEnd		= pQue;
				//----- 格納データを新キューエントリへ格納 -----------------------------------------------------//
				pQue->eix = __min(pW->siz - 1, lDat);
				memcpy(pQue->dat, pDat, pQue->eix);
				((UBP)pDat) += pQue->eix;
				lDat		-= pQue->eix;
				//----- 全データサイズ更新 ---------------------------------------------------------------------//
				pW->DataSize += pQue->eix;
				rc			 += pQue->eix;
			}
			else {
				break;
			}
		}

	} while(0);

	return (rc);
}
//==============================================================================================================//
//	リングバッファへデータ格納（格納しきれない場合は、すべて格納しない）										//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				pDat	 - 格納するデータのアドレス																//
//				lDat	 - 格納するデータのバイト数																//
//																												//
//	戻り値	：	実際に格納されたデータバイト数																	//
//==============================================================================================================//
AJCEXPORT	UI	WINAPI	AjcRngPutDataEx  (HAJCRNG pW, C_VOP pDat, UI lDat)
{
	UI	rc = 0;
	UI	SvSix, SvEix;
	UL	SvDataSize;
	UI	i, fre, sdl;
	UI	len[2];

	if (IS_MY_INST(pW) && pDat != NULL) {
		AJCRNG_ENTER
		//--- スタティックリングバッファ情報退避 --------------------------//
		SvSix	   = pW->stSix;
		SvEix	   = pW->stEix;
		SvDataSize = pW->DataSize;
		//--- スタティックリングバッファへデータ格納 ----------------------//
		if (pW->pStaticBuf != NULL) {
			fre = GetRingFreeSize(pW->stSix, pW->stEix, pW->lStaticBuf, len);
			if (fre != 0) {
				for (i=0; i<2 && lDat && len[i]; i++) {
					if (len[i] >= lDat) {
						memcpy(&pW->pStaticBuf[pW->stEix], pDat, lDat);
						((C_UBP)pDat) += lDat;
						UPD_RINGIDX(pW->stEix, lDat, pW->lStaticBuf);
						pW->DataSize += lDat;
						rc			 += lDat;
						lDat		  = 0;
					}
					else {
						memcpy(&pW->pStaticBuf[pW->stEix], pDat, len[i]);
						((C_UBP)pDat) += len[i];
						UPD_RINGIDX(pW->stEix, len[i], pW->lStaticBuf);
						pW->DataSize  += len[i];
						rc			  += len[i];
						lDat		  -= len[i];
					}
				}
			}
		}
		//--- 残データをリングバッファキューへ格納 ------------------------//
		if (lDat != 0) {
			sdl = SubPutDataEx(pW, pDat, lDat);
			if (sdl == 0) {
				 pW->stSix	  = SvSix;
				 pW->stEix	  = SvEix;
				 pW->DataSize = SvDataSize;
			}
			else {
				rc += sdl;
			}
		}
		AJCRNG_LEAVE
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	リングバッファキューへデータ格納（格納しきれない場合は、全て格納しない）									//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				pDat	 - 格納するデータのアドレス																//
//				lDat	 - 格納するデータのバイト数																//
//																												//
//	戻り値	：	実際に格納されたデータバイト数																	//
//--------------------------------------------------------------------------------------------------------------//
static	UI	SubPutDataEx  (HAJCRNG pW, C_VOP pDat, UI lDat)
{
	UI	rc;

	UL				DataSize;
	PAJCRNGQUE		pQTop;
	PAJCRNGQUE		pQEnd;
	PAJCRNGQUE		pQBfr;
	UI				six, eix;
	PAJCRNGQUE		pQue;
	PAJCRNGQUE		pNxt;
	PAJCRNGQUE		mp;

	DataSize = pW->DataSize;				//		キュー構成情報退避
	pQTop	 = pW->pQTop;					//		・
	pQEnd	 = pW->pQEnd;					//		・
	pQBfr	 = pW->pQBfr;					//		・
	if (pQEnd != 0) {						//		末尾キューあり？
		pQue = pQEnd;						//			リングバッファ情報退避
		six = pQue->six;					//			・
		eix = pQue->eix;					//			・
	}
	rc = SubPutData(pW, pDat, lDat);		//		データ格納処理
	if (rc != lDat) {						//		格納しきれなかった？
		if (pQEnd != 0) {					//			末尾キューデータあり？
			pQue = pQEnd;					//				追加分の先頭キューポインタを設定
			mp	 = pQue->pNxt;				//				・
		}
		else {								//			末尾キューなし？
			mp = NULL;						//				追加分の先頭キューポインタ＝０（追加キューなし）
		}
		while (mp != NULL) {				//			追加された分の全キューエントリ ループ
			pQue = mp;						//				キューエントリ
			pNxt = pQue->pNxt;				//				次のキューポインタ取得
			free(mp);						//				・
			mp = pNxt;						//				次のキューポインタ設定
		}
		pW->DataSize = DataSize;			//			キュー構成情報を元に戻す
		pW->pQTop	 = pQTop;				//			・
		pW->pQEnd	 = pQEnd;				//			・
		pW->pQBfr	 = pQBfr;				//			・
		if (pQEnd != 0) {					//			末尾キューあり？
			pQue = pQEnd;					//				末尾キュー
			pQue->pNxt = 0;					//				次のキューへのポインタ＝０
			pQue->six = six;				//				リングバッファ情報を元に戻す
			pQue->eix = eix;				//				・
		}
		rc = 0;								//			戻り値（格納済みバイト数）＝０
	}

	return rc;
}
//==============================================================================================================//
//	リングバッファからデータ取り出し（但し、ｐＢｕｆ＝ＮＵＬＬの場合は、空読みする）							//
//																												//
//	引　数	：	pW	 - インスタンスハンドル																		//
//				pBuf - 取り出したデータを格納するバッファのアドレス												//
//				lBuf - 取り出したデータを格納するバッファのサイズ（バイト数）									//
//																												//
//	戻り値	：	取り出したデータのバイト数																		//
//==============================================================================================================//
AJCEXPORT	UI	WINAPI	AjcRngGetData	(HAJCRNG pW, VOP pBuf, UI lBuf)
{
	UI		rc = 0;
	UI		i, dtl, gtl;
	UI		len[2];

	if (IS_MY_INST(pW)) {
		AJCRNG_ENTER
		//--- スタティックリングバッファからデータ取り出し ---//
		if (pW->pStaticBuf != NULL) {
			dtl  = GetRingDataSize(pW->stSix, pW->stEix, pW->lStaticBuf, len);
			if (dtl != 0) {
				for (i=0; i<2 && lBuf && len[i]; i++) {
					if (len[i] >= lBuf) {
						if (pBuf != NULL) {
							memcpy(pBuf, &pW->pStaticBuf[pW->stSix], lBuf);
							((UB*)pBuf) 	+= lBuf;
						}
						UPD_RINGIDX(pW->stSix, lBuf, pW->lStaticBuf);
						pW->DataSize		-= lBuf;
						rc					+= lBuf;
						lBuf				 = 0;
					}
					else {
						if (pBuf != NULL) {
							memcpy(pBuf, &pW->pStaticBuf[pW->stSix], len[i]);
							((UB*)pBuf) 	+= len[i];
						}
						UPD_RINGIDX(pW->stSix, len[i], pW->lStaticBuf);
						pW->DataSize		-= len[i];
						rc					+= len[i];
						lBuf				-= len[i];
					}
				}
			}
		}
		//--- 残データをリングバッファキューから取り出し -----//
		if (lBuf != 0  &&  pW->pQTop != 0) {
			rc += SubGetData(pW, pBuf, lBuf);
		}
		//--- スタティックリングバッファ補充 -----------------//
		if (pW->pStaticBuf != NULL	&&	pW->pQTop != 0) {
			dtl = GetRingFreeSize(pW->stSix, pW->stEix, pW->lStaticBuf, len);
			if (dtl != 0) {
				for (i=0; i<2 && dtl && len[i]; i++) {
					if (len[i] >= dtl) {
						gtl = SubGetData(pW, &pW->pStaticBuf[pW->stEix], dtl);
						UPD_RINGIDX(pW->stEix, gtl, pW->lStaticBuf);
						pW->DataSize += gtl;
						dtl 		  = 0;
					}
					else {
						gtl = SubGetData(pW, &pW->pStaticBuf[pW->stEix], len[i]);
						UPD_RINGIDX(pW->stEix, gtl, pW->lStaticBuf);
						pW->DataSize  += gtl;
						dtl 		  -= gtl;
						if (gtl < len[i]) break;
					}
				}
			}
		}
		AJCRNG_LEAVE
	}
	return (rc);
}
//--------------------------------------------------------------------------------------------------------------//
//	リングバッファキューからデータ取り出し（但し、ｐＢｕｆｆｅｒ＝ＮＵＬＬの場合は、空読みする）				//
//																												//
//	引　数	：	pW	 - インスタンスハンドル																		//
//				pBuf - 取り出したデータを格納するバッファのアドレス												//
//				lBuf - 取り出したデータを格納するバッファのサイズ（バイト数）									//
//																												//
//	戻り値	：	取り出したデータのバイト数																		//
//--------------------------------------------------------------------------------------------------------------//
static	UI	SubGetData	 (HAJCRNG pW, VOP pBuf, UI lBuf)
{
	UI				rc = 0;
	PAJCRNGQUE		pQue;
	UI				i, sdl;
	UI				len[2];

	while (lBuf  &&  pW->pQTop) {
		//----- 先頭キューエントリ設定	------------------------------------------------------------------------//
		pQue = pW->pQTop;
		//----- 先頭キューエントリからデータを取り出す ---------------------------------------------------------//
		sdl  = GetRingDataSize(pQue->six, pQue->eix, pW->siz, len);
		for (i=0; i<2 && lBuf && len[i]; i++) {
			if (len[i] >= lBuf) {
				if (pBuf != NULL) {
					memcpy(pBuf, &pQue->dat[pQue->six], lBuf);
					((UB*)pBuf) 	+= lBuf;
				}
				UPD_RINGIDX(pQue->six, lBuf, pW->siz);
				pW->DataSize		-= lBuf;
				rc					+= lBuf;
				lBuf				 = 0;
			}
			else {
				if (pBuf != NULL) {
					memcpy(pBuf, &pQue->dat[pQue->six], len[i]);
					((UB*)pBuf) 	+= len[i];
				}
				UPD_RINGIDX(pQue->six, len[i], pW->siz);
				pW->DataSize		-= len[i];
				rc					+= len[i];
				lBuf				-= len[i];
			}
		}
		//----- 先頭キューエントリが空ならば、キューから削除する -----------------------------------------------//
		if (pQue->six == pQue->eix) {
			pW->pQTop = pQue->pNxt;
			if		(pW->pQTop == pW->pQEnd) {
				pW->pQBfr = NULL;
			}
			else if (pW->pQTop == NULL) {
				pW->pQEnd = NULL;
			}
			free(pQue);
		}
	}

	return (rc);
}

//==============================================================================================================//
//	リングバッファ先頭部分のデータ覗き見																		//
//																												//
//	引　数	：	pW	 - インスタンスハンドル																		//
//				pBuf - 覗き見するデータを格納するバッファのアドレス												//
//				lBuf - 覗き見するデータを格納するバッファのサイズ（バイト数）									//
//																												//
//	戻り値	：	実際に格納したデータのバイト数																	//
//==============================================================================================================//
AJCEXPORT	UI	WINAPI	AjcRngPeekData	 (HAJCRNG pW, VOP pBuf, UI lBuf)
{
	UI		rc = 0;
	UI		i, dtl, six;
	UI		len[2];

	if (IS_MY_INST(pW) && pBuf != NULL) {
		AJCRNG_ENTER
		//--- スタティックリングバッファからデータ読み出し ---//
		if (pW->pStaticBuf != NULL) {
			dtl  = GetRingDataSize(pW->stSix, pW->stEix, pW->lStaticBuf, len);
			if (dtl != 0) {
				six = pW->stSix;
				for (i=0; i<2 && lBuf && len[i]; i++) {
					if (len[i] >= lBuf) {
						memcpy(pBuf, &pW->pStaticBuf[six], lBuf);
						((UB*)pBuf) 		+= lBuf;
						UPD_RINGIDX(six 	 , lBuf, pW->lStaticBuf);
						rc					+= lBuf;
						lBuf				 = 0;
					}
					else {
						memcpy(pBuf, &pW->pStaticBuf[six], len[i]);
						((UB*)pBuf) 		+= len[i];
						UPD_RINGIDX(six 	 , len[i], pW->lStaticBuf);
						rc					+= len[i];
						lBuf				-= len[i];
					}
				}
			}
		}
		//--- 残データをリングバッファキューから読み出し -----//
		if (lBuf != 0  &&  pW->pQTop != 0) {
			rc += SubPeekData(pW, pBuf, lBuf);
		}
		AJCRNG_LEAVE
	}
	return (rc);
}
//--------------------------------------------------------------------------------------------------------------//
//	リングバッファキュー先頭部分のデータ覗き見																	//
//																												//
//	引　数	：	pW	 - インスタンスハンドル																		//
//				pBuf - 覗き見するデータを格納するバッファのアドレス												//
//				lBuf - 覗き見するデータを格納するバッファのサイズ（バイト数）									//
//																												//
//	戻り値	：	実際に格納したデータのバイト数																	//
//--------------------------------------------------------------------------------------------------------------//
static	UI	SubPeekData   (HAJCRNG pW, VOP pBuf, UI lBuf)
{
	UI				rc = 0;
	PAJCRNGQUE		pQue;
	UI				i, sdl;
	UI				len[2];
	PAJCRNGQUE		pQTop	 = pW->pQTop;
	PAJCRNGQUE		pQEnd	 = pW->pQEnd;
	UL				DataSize = pW->DataSize;
	UI				six;

	while (lBuf  &&  pQTop) {
		//----- 先頭キューエントリアドレス設定 -----------------------------------------------------------------//
		pQue = pQTop;
		//----- 先頭キューエントリからデータを読む -------------------------------------------------------------//
		six  = pQue->six;
		sdl  = GetRingDataSize(pQue->six, pQue->eix, pW->siz, len);
		for (i=0; i<2 && lBuf && len[i]; i++) {
			if (len[i] >= lBuf) {
				memcpy(pBuf, &pQue->dat[six], lBuf);
				((UB*)pBuf) 		+= lBuf;
				UPD_RINGIDX(six 	 , lBuf, pW->siz);
				DataSize			-= lBuf;
				rc					+= lBuf;
				lBuf				 = 0;
			}
			else {
				memcpy(pBuf, &pQue->dat[six], len[i]);
				((UB*)pBuf) 		+= len[i];
				UPD_RINGIDX(six 	 , len[i], pW->siz);
				DataSize			-= len[i];
				rc					+= len[i];
				lBuf				-= len[i];
			}
		}
		//----- 次のキューエントリ設定 -------------------------------------------------------------------------//
		if ((pQTop = pQue->pNxt) == NULL) {
			pQEnd = NULL;
		}
	}

	return (rc);
}

//==============================================================================================================//
//	リングバッファに格納されているデータのバイト数を取得する													//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//																												//
//	戻り値	：	リングバッファに格納されているデータのバイト数													//
//==============================================================================================================//
AJCEXPORT	UL		WINAPI	AjcRngGetDataSize(HAJCRNG pW)
{
	UL	rc = 0;

	if (IS_MY_INST(pW)) {
		AJCRNG_ENTER
		rc = pW->DataSize;
		AJCRNG_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	リングバッファに格納されている全データを破棄する															//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcRngPurge 	 (HAJCRNG pW)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		AJCRNG_ENTER
		pW->stSix = pW->stEix = 0;
		FreeAllQue(pW);
		pW->DataSize = 0;
		AJCRNG_LEAVE
		rc = TRUE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	全キューデータを開放する																					//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//																												//
//	戻り値	：	なし																							//
//--------------------------------------------------------------------------------------------------------------//
static	VO	FreeAllQue(LPAJCRNG pW)
{
	PAJCRNGQUE	mp, pNxt;

	mp = pW->pQTop;
	while (mp != NULL) {
		pNxt = mp->pNxt;
		free(mp);
		mp = pNxt;
	}
	pW->pQTop = pW->pQEnd = pW->pQBfr = NULL;
}
//--------------------------------------------------------------------------------------------------------------//
//	リングバッファ中に格納されているデータバイト数を得る														//
//																												//
//	引　数	：	six, eix	- リングバッファのデータインデクス													//
//				RingBufSize - リングバッファサイズ																//
//				plDat		- バイト数を格納する配列のアドレス													//
//						  	  len[0]=１stメモリブロックのバイト数， len[1] = ２ndメモリブロックのバイト数		//
//																												//
//	戻り値	：	リングバッファ中に格納されているデータの総バイト数												//
//--------------------------------------------------------------------------------------------------------------//
static	UI	GetRingDataSize(UI six, UI eix, UI RingBufSize, UIP plDat)
{
	UI	rc;

	if (six == eix) {
		*plDat++ = 0;
		*plDat	 = 0;
		rc		 = 0;
	}
	else if (six < eix) {
		*plDat++ = rc = eix - six;
		*plDat	 = 0;
	}
	else {
		*plDat++ = rc = RingBufSize - six;
		*plDat	 = eix;
		rc		+= eix;
	}
	return(rc);
}
//--------------------------------------------------------------------------------------------------------------//
//	リングバッファ中の空データバイト数を得る																	//
//																												//
//	引　数	：	six, eix	- リングバッファのデータインデクス													//
//				RingBufSize - リングバッファサイズ																//
//				plDat		- バイト数を格納する配列のアドレス													//
//							  len[0]=１st空メモリブロックのバイト数， len[1] = ２nd空メモリブロックのバイト数	//
//																												//
//	戻り値	：	リングバッファ中の総空バイト数																	//
//--------------------------------------------------------------------------------------------------------------//
static	UI	GetRingFreeSize(UI six, UI eix, UI RingBufSize, UIP plDat)
{
	UI	rc;

	if (six <= eix) {
		if (six == 0) {
			*plDat++ = rc = (RingBufSize - eix - 1);
			*plDat	 = 0;
		}
		else {
			*plDat++ = rc = RingBufSize - eix;
			*plDat	 = six - 1;
			rc		+= *plDat;
		}
	}
	else {
		*plDat++ = rc = (six - eix - 1);
		*plDat	 = 0;
	}
	return(rc);
}
