﻿#include	"AjcInternal.h"

//**************************************************************************************************************//
//																												//
//	ストリームの分離																							//
//																												//
//**************************************************************************************************************//

#define	INST_ID			0x87F65A21
#define	IS_MY_INST(P)	(P != NULL && P->InstID == INST_ID)

//--------------------------------------------------------------------------------------------------------------//
//	内部マクロ																									//
//--------------------------------------------------------------------------------------------------------------//
#define		AJCSSEP_ENTER		if (pW->fMultiThread) EnterCriticalSection(&pW->cs);
#define		AJCSSEP_LEAVE		if (pW->fMultiThread) LeaveCriticalSection(&pW->cs);

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	IsStartTxt (HAJCSSEP pW, BC c);
static	BOOL	IsStartEsc (HAJCSSEP pW, BC c);
static	BOOL	IsStartPkt (HAJCSSEP pW, BC c);
static	VO		SubNtcTxt  (HAJCSSEP pW);
static	VO		SubNtcEsc  (HAJCSSEP pW);
static	VO		SubNtcPkt  (HAJCSSEP pW);
static	VO		SubNtcCtrl (HAJCSSEP pW, BC c);
static	VO		SubNtcNoPkt(HAJCSSEP pW, BC c);
static	VO		SubNtcEvt  (HAJCSSEP pW, UI evt, VOP pDat, UI lDat);

//==============================================================================================================//
//	インスタンス生成																							//
//																												//
//	引　数	：	evt 	 - 生成するイベントコード																//
//				cbp 	 - コールバックパラメタ																	//
//				cbEvent  - イベント通知用コールバック															//
//																												//
//	戻り値	：	≠ＮＵＬＬ：ＯＫ（インスタンスハンドル（＝インスタンスワークアドレス））						//
//				＝ＮＵＬＬ：エラー																				//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	HAJCSSEP	WINAPI	AjcSsepCreateA(UI evt, UX cbp, BOOL (CALLBACK *cbEvent)(UI evt, VOP pDat, UI lDat, UX cbp))
{
	HAJCSSEP  pW;

	if ((pW = (HAJCSSEP)AJCMEM(sizeof(AJCSSEP))) != NULL) {
		memset(pW, 0, sizeof *pW);
		pW->InstID				= INST_ID;						//	インスタンスＩＤ
		pW->Mode 				= AJCSSEP_MD_IDLE;				//	処理モード

		pW->evt					= (evt & AJCSSEP_EV_ALL);		//	生成するイベント

		pW->fUnicode			= FALSE;						//	UNICODEフラグ
		pW->fDle				= FALSE;						//	ＤＬＥの直後を示すフラグ
		pW->TxtIncMap[CC_TAB]	= TRUE;							//	テキスト終端制御コードマップ
		pW->stx					= CC_STX;						//	パケット分離用制御コード
		pW->etx					= CC_ETX;						//	・
		pW->dle 				= CC_DLE;						//	・

		pW->PktTimOut			= 0;							//	パケットデータのタイムアウト値［ms］

		pW->hMbc 				= AjcMbcCreate(256, AJCMBC_SJIS);
		pW->SvMbk				= AJCMBC_SJIS;

		pW->hRng				= AjcRngCreate(&pW->RngBuf, sizeof pW->RngBuf, 4096);

		pW->fMultiThread		= FALSE;

		pW->cbp					= cbp;
		pW->cbEvent				= cbEvent;

		if (pW->hRng == NULL) {
			AjcSsepDelete(pW);
			pW = NULL;
		}
	}
	return pW;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	HAJCSSEP	WINAPI	AjcSsepCreateW(UI evt, UX cbp, BOOL (CALLBACK *cbEvent)(UI evt, VOP pDat, UI lDat, UX cbp))
{
	HAJCSSEP  pW;

	if (pW = AjcSsepCreateA(evt, cbp, cbEvent)) {
		pW->fUnicode = TRUE;
	}
	return pW;
}


//==============================================================================================================//
//	インスタンス消去																							//
//																												//
//	引　数	：	pW 		  - インスタンスハンドル																//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	VO			WINAPI	AjcSsepDelete(HAJCSSEP pW)
{
	if (IS_MY_INST(pW)) {
		if (pW->hMbc != NULL) AjcMbcDelete(pW->hMbc);
		if (pW->hRng != NULL) AjcRngDelete(pW->hRng);
		if (pW->fMultiThread) DeleteCriticalSection(&pW->cs);
		free(pW);
	}
}
//==============================================================================================================//
//	マルチスレッドの許可／禁止																					//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//				fEnable - FALSE : マルチスレッド禁止															//
//						  TRUE	: マルチスレッド許可															//
//																												//
//	戻り値	：	前回の許可／禁止状態																			//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcSsepEnableMultiThread(HAJCSSEP pW, BOOL fEnable)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		rc = pW->fMultiThread;
		if (fEnable) {
			if (!pW->fMultiThread) {
				InitializeCriticalSection(&pW->cs);
				pW->fMultiThread = TRUE;
			}
		}
		else {
			if (pW->fMultiThread) {
				pW->fMultiThread = FALSE;
				DeleteCriticalSection(&pW->cs);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	データ投与																									//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//				pDat	- 投与するデータのアドレス																//
//				lDat	- 投与するデータのバイト数																//
//																												//
//	戻り値	：	現在処理中のモード																				//
//==============================================================================================================//
AJCEXPORT	AJCSSEP_MODE	WINAPI	AjcSsepPutData	(HAJCSSEP pW, C_UBP pDat, UI lDat)
{
	UI			tc;

	if (IS_MY_INST(pW) && pDat != NULL) {
		AJCSSEP_ENTER
		//----- パケットタイムアウトチェック -----------------------------------------------------------------------//
		if (pW->Mode == AJCSSEP_MD_PKT	&&	pW->PktTimOut != 0) {
			tc = GetTickCount();
			if ((tc - pW->PktTimSrt) >= pW->PktTimOut) {
				pW->Mode = AJCSSEP_MD_IDLE;
				pW->fDle = FALSE;
				AjcRngPurge(pW->hRng);
			}
		}
		//----- ストリーム分離 -------------------------------------------------------------------------------------//
		while (lDat--) {
			switch (pW->Mode) {
				//----- アイドル状態 -------------------------------------------------------------------------------//
				case AJCSSEP_MD_IDLE:
					if		(IsStartPkt(pW, *pDat)) {				//	ＰＫＴ開始？
						pW->PktTimSrt = GetTickCount();				//		ＰＫＴ受信開始時刻設定
						pW->Mode = AJCSSEP_MD_PKT;					//		→ＰＫＴモード
					}
					else if (!pW->fDle) {							//	ＤＬＥ直後以外？
						SubNtcNoPkt(pW, *pDat);						//		パケット外バイト通知
						if		(IsStartEsc(pW, *pDat)) {			//		ＥＳＣ開始？
							pW->Mode = AJCSSEP_MD_ESC;				//			→ＥＳＣモード
							AjcRngPutData(pW->hRng, pDat, 1);		//			ＥＳＣコードバッファリング
						}
						else if (IsStartTxt(pW, *pDat)) {			//		ＴＸＴ開始？
							BC	txt[2];
							pW->Mode = AJCSSEP_MD_TXT;				//			→ＴＸＴモード
							txt[0] = *pDat; txt[1] = 0;				//				文字コード判定用データ蓄積
							AjcMbcPutData(pW->hMbc, txt);			//				・
							if (pW->evt & AJCSSEP_EV_TXT) {			//			ＴＸＴ通知要？
								AjcRngPutData(pW->hRng, pDat, 1);	//				テキストデータバッファリング
							}
						}
						else {										//		その他
							SubNtcCtrl(pW, *pDat);					//			制御コード通知
						}
					}
					break;
				//----- テキストモード -----------------------------------------------------------------------------//
				case AJCSSEP_MD_TXT:
					if		(IsStartPkt(pW, *pDat)) {				//	ＰＫＴ開始？
						SubNtcTxt(pW);								//		ＴＸＴ通知
						pW->Mode = AJCSSEP_MD_PKT;					//		→ＰＫＴモード
						pW->PktTimSrt = GetTickCount();				//		ＰＫＴ受信開始時刻設定
					}
					else if (!pW->fDle) {							//	ＤＬＥ直後以外？
						SubNtcNoPkt(pW, *pDat);						//		パケット外バイト通知
						if		(IsStartEsc(pW, *pDat)) {			//		ＥＳＣ開始？
							SubNtcTxt(pW);							//			ＴＸＴ通知
							pW->Mode = AJCSSEP_MD_ESC;				//			→ＥＳＣモード
							AjcRngPutData(pW->hRng, pDat, 1);		//			ＥＳＣデータバッファリング
						}
						else if (iscntrl(*pDat) &&					//		ＴＸＴ終了？
								 !pW->TxtIncMap[*pDat]) {			//		・
							SubNtcTxt(pW);							//			ＴＸＴ通知
							SubNtcCtrl(pW, *pDat);					//			制御コード通知
							pW->Mode = AJCSSEP_MD_IDLE;				//			→アイドル状態
						}
						else {										//		その他？
							BC	txt[2];
							txt[0] = *pDat; txt[1] = 0;				//			文字コード判定用データ蓄積
							AjcMbcPutData(pW->hMbc, txt);			//			・
							if (pW->evt & AJCSSEP_EV_TXT) {			//			ＴＸＴ通知要？
								AjcRngPutData(pW->hRng, pDat, 1);	//				ＴＸＴデータバッファリング
							}
						}
					}
					break;
				//----- ＥＳＣモード -------------------------------------------------------------------------------//
				case AJCSSEP_MD_ESC:
					if		(IsStartPkt(pW, *pDat)) {				//	ＰＫＴ開始？
						AjcRngPurge(pW->hRng);						//		ＥＳＣデータ破棄
						pW->PktTimSrt = GetTickCount();				//		ＰＫＴ受信開始時刻設定
						pW->Mode = AJCSSEP_MD_PKT;					//		→ＰＫＴモード
					}
					else if (!pW->fDle) {							//	ＤＬＥ直後以外？
						SubNtcNoPkt(pW, *pDat);						//		パケット外バイト通知
						if		(iscntrl(*pDat)) {					//		制御コード？
							AjcRngPurge(pW->hRng);					//			ＥＳＣデータ破棄
							SubNtcCtrl(pW, *pDat);					//			制御コード通知
							pW->Mode = AJCSSEP_MD_IDLE;				//			→アイドル状態
						}
						else if (isalpha(*pDat)) {					//		英字？
							AjcRngPutData(pW->hRng, pDat, 1);		//			英字をバッファリング
							SubNtcEsc(pW);							//			ＥＳＣデータ通知
							pW->Mode = AJCSSEP_MD_IDLE;				//			→アイドル状態
						}
						else {										//		その他？
							AjcRngPutData(pW->hRng, pDat, 1);		//			ＥＳＣデータバッファリング
						}
					}
					break;
				//----- ＰＫＴモード -------------------------------------------------------------------------------//
				case AJCSSEP_MD_PKT:
					if (!pW->fDle) {								//	ＤＬＥ直後以外？
						if (*pDat == pW->dle) {						//		ＤＬＥ？
							pW->fDle = TRUE;						//			→ＤＬＥ直後
						}
						else {										//		ＤＬＥ以外？
							AjcRngPutData(pW->hRng, pDat, 1);		//			ＰＫＴデータバッファリング
						}
					}
					else {											//	ＤＬＥ直後？
						pW->fDle = FALSE;							//		ＤＬＥ直後を解除
						if		(*pDat == pW->etx) {				//		ＥＴＸ？
							SubNtcPkt(pW);							//			ＰＫＴデータ通知
							pW->Mode = AJCSSEP_MD_IDLE;				//			→アイドル状態
						}
						else if (*pDat == pW->stx) {				//		ＳＴＸ？
							AjcRngPurge(pW->hRng);					//			ＰＫＴデータ破棄（ＰＫＴ再開始）
							pW->PktTimSrt = GetTickCount();			//			ＰＫＴ受信開始時刻設定
						}
						else {										//		その他？
							AjcRngPutData(pW->hRng, pDat, 1);		//			ＰＫＴデータバッファリング
						}
					}
					break;
			}
			pDat++;
		}
		AJCSSEP_LEAVE
	}
	return pW->Mode;
}
//==============================================================================================================//
//	テキスト投与																								//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//				pTxt	- 投与するテキストデータのアドレス														//
//				lTxt	- 投与するテキストデータの文字数（－１指定時は自動算出）								//
//																												//
//	戻り値	：	現在処理中のモード																				//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	AJCSSEP_MODE	WINAPI	AjcSsepPutTextA	(HAJCSSEP pW, C_BCP pTxt, UI lTxt)
{
	AJCSSEP_MODE	rc;

	if (IS_MY_INST(pW) && pTxt != NULL) {
		AJCSSEP_ENTER
		lTxt = AjcStrAdjustLenA(pTxt, lTxt);
		rc = AjcSsepPutData(pW, pTxt, lTxt);
		AJCSSEP_LEAVE
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	AJCSSEP_MODE	WINAPI	AjcSsepPutTextW	(HAJCSSEP pW, C_WCP pTxt, UI lTxt)
{
	AJCSSEP_MODE	rc	 = pW->Mode;
	BCP				pTmp = NULL;
	UI				len  = 0;

	if (IS_MY_INST(pW) && pTxt != NULL) {
		lTxt = AjcStrAdjustLenW(pTxt, lTxt);
		len = WideCharToMultiByte(CP_ACP, 0, pTxt, lTxt, NULL, 0, NULL, NULL);
		if (len != 0 && (pTmp = (BCP)AJCMEM(len))) {
			WideCharToMultiByte(CP_ACP, 0, pTxt, lTxt, pTmp, len, NULL, NULL);
			rc = AjcSsepPutData(pW, pTmp, len);
		}
		if (pTmp != NULL) free(pTmp);
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	テキスト開始チェック																						//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	IsStartTxt(HAJCSSEP pW, BC c)
{
	BOOL	rc = FALSE;

	if (!iscntrl((UB)c) || pW->TxtIncMap[c]) {
		rc = TRUE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	ＥＳＣコード開始チェック																					//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	IsStartEsc(HAJCSSEP pW, UB c)
{
	BOOL	rc = FALSE;

	if (pW->evt & AJCSSEP_EV_ESC) {
		if (c == CC_ESC) {
			rc = TRUE;
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	パケット開始チェック																						//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	IsStartPkt(HAJCSSEP pW, UB c)
{
	BOOL	rc = FALSE;

	if (pW->evt & AJCSSEP_EV_PKT) {
		if (!pW->fDle) {
			if (c == pW->dle) {
				pW->fDle = TRUE;
			}
		}
		else {
			pW->fDle = FALSE;
			if (c == pW->stx) {
				rc = TRUE;
			}
		}
	}
	return rc;
}

//--------------------------------------------------------------------------------------------------------------//
//	テキスト通知																								//
//--------------------------------------------------------------------------------------------------------------//
static	VO		SubNtcTxt(HAJCSSEP pW)
{
	BCP		pTxt;
	WCP		pWcs;
	UI		len, lWcs;

	if (pW->evt & AJCSSEP_EV_TXT) {
		if (len = AjcRngGetDataSize(pW->hRng)) {
			if (pTxt = AjcTAllocA(len + 1)) {
				AjcRngGetData(pW->hRng, pTxt, len);
				pTxt[len] = 0;
				if (pW->fUnicode) {
					lWcs = MultiByteToWideChar(CP_ACP, 0, pTxt, -1, NULL, 0);
					if (pWcs = AjcTAllocW(lWcs)) {
						MultiByteToWideChar(CP_ACP, 0, pTxt, -1, pWcs, lWcs);
						SubNtcEvt(pW, AJCSSEP_EV_TXT, pWcs, lWcs - 1);
					}
					free(pTxt);
				}
				else {
					SubNtcEvt(pW, AJCSSEP_EV_TXT, pTxt, len);
				}
			}
		}
	}
	AjcRngPurge(pW->hRng);
}
//--------------------------------------------------------------------------------------------------------------//
//	ＥＳＣコード通知																							//
//--------------------------------------------------------------------------------------------------------------//
static	VO		SubNtcEsc(HAJCSSEP pW)
{
	BCP		pTxt;
	WCP		pWcs;
	UI		len, lWcs;

	if (pW->evt & AJCSSEP_EV_ESC) {
		if (len = AjcRngGetDataSize(pW->hRng)) {
			if (pTxt = AjcTAllocA(len + 1)) {
				AjcRngGetData(pW->hRng, pTxt, len);
				pTxt[len] = 0;
				if (pW->fUnicode) {
					lWcs = MultiByteToWideChar(CP_ACP, 0, pTxt, -1, NULL, 0);
					if (pWcs = AjcTAllocW(lWcs)) {
						MultiByteToWideChar(CP_ACP, 0, pTxt, -1, pWcs, lWcs);
						SubNtcEvt(pW, AJCSSEP_EV_ESC, pWcs, lWcs - 1);
					}
					free(pTxt);
				}
				else {
					SubNtcEvt(pW, AJCSSEP_EV_ESC, pTxt, len);
				}
			}
		}
	}
	AjcRngPurge(pW->hRng);
}
//--------------------------------------------------------------------------------------------------------------//
//	パケット通知																								//
//--------------------------------------------------------------------------------------------------------------//
static	VO		SubNtcPkt(HAJCSSEP pW)
{
	VOP		pPkt;
	UI		len;

	if (pW->evt & AJCSSEP_EV_PKT) {
		len = AjcRngGetDataSize(pW->hRng);
		if (pPkt = (VOP)AJCMEM(len)) {
			AjcRngGetData(pW->hRng, pPkt, len);
			SubNtcEvt(pW, AJCSSEP_EV_PKT, pPkt, len);
		}
	}
	AjcRngPurge(pW->hRng);
}
//--------------------------------------------------------------------------------------------------------------//
//	制御コード通知																								//
//--------------------------------------------------------------------------------------------------------------//
static	VO		SubNtcCtrl(HAJCSSEP pW, BC c)
{
	union {BCP pTxt; WCP pWcs;} u;

	if (pW->evt & AJCSSEP_EV_CTRL) {
		if (pW->fUnicode) {
			if (u.pWcs = AjcTAllocW(2)) {
				u.pWcs[0] = c;
				u.pWcs[1] = 0;
				SubNtcEvt(pW, AJCSSEP_EV_CTRL, u.pWcs, 1);
			}
		}
		else {
			if (u.pTxt = AjcTAllocA(2)) {
				u.pTxt[0] = c;
				u.pTxt[1] = 0;
				SubNtcEvt(pW, AJCSSEP_EV_CTRL, u.pTxt, 1);
			}
		}
	}
}
//--------------------------------------------------------------------------------------------------------------//
//	パケット以外のバイト通知																					//
//--------------------------------------------------------------------------------------------------------------//
static	VO		SubNtcNoPkt(HAJCSSEP pW, UB c)
{
	UBP		pByte;

	if (pW->evt & AJCSSEP_EV_NOPKT) {
		if (pByte = (UBP)AJCMEM(2)) {
			pByte[0] = c;
			pByte[1] = 0;
			SubNtcEvt(pW, AJCSSEP_EV_NOPKT, pByte, 1);
		}
	}
}
//--------------------------------------------------------------------------------------------------------------//
//	ユーザＡＰＰにイベント通知																					//
//--------------------------------------------------------------------------------------------------------------//
static	VO		SubNtcEvt(HAJCSSEP pW, UI evt, VOP pDat, UI lDat)
{
	BOOL	rsu = TRUE;

	if (pW->cbEvent) {
		rsu = pW->cbEvent(evt, pDat, lDat, pW->cbp);
	}
	if (rsu && pDat != NULL) {
		free(pDat);
	}
}

//==============================================================================================================//
//	リセット																									//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	VO	 		WINAPI	AjcSsepReset	(HAJCSSEP pW)
{
	if (IS_MY_INST(pW)) {
		AJCSSEP_ENTER
		pW->Mode = AJCSSEP_MD_IDLE;			//	アイドル状態
		pW->fDle = FALSE;					//	先行ＤＬＥなし
		pW->SvMbk = AJCMBC_SJIS;			//	S-JISコードを仮定
		AjcMbcReset(pW->hMbc);				//	自動判別文字コードバッファ初期化
		AjcRngPurge(pW->hRng);			//	リングバッファ・クリアー
		AJCSSEP_LEAVE
	}
}
//==============================================================================================================//
//	生成するイベントコード設定																					//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//				evt		- 生成するイベントコード																//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	VO			WINAPI	AjcSsepSetEvent (HAJCSSEP pW, UI evt)
{
	AJCSSEP_ENTER

	if (IS_MY_INST(pW)) {
		pW->evt = evt;

		pW->Mode = AJCSSEP_MD_IDLE;
		pW->fDle = FALSE;
		AjcRngPurge(pW->hRng);

		AJCSSEP_LEAVE
	}
}
//==============================================================================================================//
//	生成するイベントコード取得																					//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//																												//
//	戻り値	：	生成するイベントコード																			//
//==============================================================================================================//
AJCEXPORT	UI	WINAPI	AjcSsepGetEvent (HAJCSSEP pW)
{
	UI	rc = 0;

	if (IS_MY_INST(pW)) {
		AJCSSEP_ENTER
		rc = pW->evt;
		AJCSSEP_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	テキストに含めるコード設定																					//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//				num		- 制御コードの個数																		//
//				...		- 制御コード群（int）																	//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	VO			WINAPI	AjcSsepSetIncTxt(HAJCSSEP pW, UI num, ...)
{
	UI		i;
	BC		c;
	va_list vls;

	if (IS_MY_INST(pW)) {
		AJCSSEP_ENTER
		va_start(vls, num);
		memset(&pW->TxtIncMap, FALSE, sizeof pW->TxtIncMap);
		for (i=0; i<num; i++) {
			c = (BC)va_arg(vls, UI);
			if (iscntrl((UB)c)) {
				pW->TxtIncMap[c] = TRUE;
			}
		}
		va_end(vls);
		AJCSSEP_LEAVE
	}
}
//==============================================================================================================//
//	パケット制御コード（ＳＴＸ）設定																			//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//				stx		- ＳＴＸコード																			//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	VO			WINAPI	AjcSsepSetStx	(HAJCSSEP pW, UI stx)
{
	if (IS_MY_INST(pW)) {
		AJCSSEP_ENTER
		pW->stx = stx;
		AJCSSEP_LEAVE
	}
}
//==============================================================================================================//
//	パケット制御コード（ＳＴＸ）取得																			//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//																												//
//	戻り値	：	ＳＴＸコード																					//
//==============================================================================================================//
AJCEXPORT	UI			WINAPI	AjcSsepGetStx	(HAJCSSEP pW)
{
	UI	rc = 0;

	if (IS_MY_INST(pW)) {
		AJCSSEP_ENTER
		rc = pW->stx;
		AJCSSEP_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	パケット制御コード（ＥＴＸ）設定																			//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//				etx		- ＥＴＸコード																			//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	VO			WINAPI	AjcSsepSetEtx	(HAJCSSEP pW, UI etx)
{
	if (IS_MY_INST(pW)) {
		AJCSSEP_ENTER
		pW->etx = etx;
		AJCSSEP_LEAVE
	}
}
//==============================================================================================================//
//	パケット制御コード（ＥＴＸ）取得																			//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//																												//
//	戻り値	：	ＥＴＸコード																					//
//==============================================================================================================//
AJCEXPORT	UI			WINAPI	AjcSsepGetEtx	(HAJCSSEP pW)
{
	UI	rc = 0;

	if (IS_MY_INST(pW)) {
		AJCSSEP_ENTER
		rc = pW->etx;
		AJCSSEP_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	パケット制御コード（ＤＬＥ）設定																			//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//				dle		- ＤＬＥコード																			//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	VO			WINAPI	AjcSsepSetDle	(HAJCSSEP pW, UI dle)
{
	if (IS_MY_INST(pW)) {
		AJCSSEP_ENTER
		pW->dle = dle;
		AJCSSEP_LEAVE
	}
}
//==============================================================================================================//
//	パケット制御コード（ＤＬＥ）取得																			//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//																												//
//	戻り値	：	ＤＬＥコード																					//
//==============================================================================================================//
AJCEXPORT	UI			WINAPI	AjcSsepGetDle	(HAJCSSEP pW)
{
	UI	rc = 0;

	if (IS_MY_INST(pW)) {
		AJCSSEP_ENTER
		rc = pW->dle;
		AJCSSEP_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	パケットデータのタイムアウト値設定																			//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//				time	- タイムアウト値［ms］																	//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	VO			WINAPI	AjcSsepSetPktTimeOut(HAJCSSEP pW, UI time)
{
	if (IS_MY_INST(pW)) {
		AJCSSEP_ENTER
		pW->PktTimOut = time;
		AJCSSEP_LEAVE
	}
}
//==============================================================================================================//
//	パケットデータのタイムアウト値取得																			//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//																												//
//	戻り値	：	タイムアウト値［ms］																			//
//==============================================================================================================//
AJCEXPORT	UI			WINAPI	AjcSsepGetPktTimeOut(HAJCSSEP pW)
{
	UI	rc = 0;

	if (IS_MY_INST(pW)) {
		AJCSSEP_ENTER
		rc = pW->PktTimOut;
		AJCSSEP_LEAVE
	}
	return rc;
}

//==============================================================================================================//
//	パケットデータ生成																							//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//				pDat	- パケットデータに変換するバイナリデータのアドレス（空パケット作成時はNULLでもOK)		//
//				lDat	- パケットデータに変換するバイナリデータのバイト数（空パケット作成時は０）				//
//				pBytes	- 生成したパケットデータのバイト数を格納するバッファのアドレス							//
//																												//
//	戻り値	：	≠NULL : 生成したパケットデータのアドレス														//
//				＝NULL : エラー																					//
//==============================================================================================================//
AJCEXPORT	VOP			WINAPI	AjcSsepMakePacket(HAJCSSEP pW, C_VOP pDat, UI lDat, UIP pBytes)
{
	UBP		pPkt;
	C_UBP	p;
	UBP		q;
	UI		i, lPkt;

	if (IS_MY_INST(pW) && (pDat != NULL || lDat == 0)) {
		AJCSSEP_ENTER
		//----- パケットデータイメージのバイト数算出 -----------------------------//
		lPkt = 4;
		for (i=0, p=pDat; i<lDat; i++, p++) {
			if (*p == pW->dle) lPkt += 2;
			else			   lPkt++;
		}
		//----- パケットデータイメージ生成 ---------------------------------------//
		if (pPkt = (UBP)AJCMEM(lPkt)) {
			q = pPkt;
			*q++ = pW->dle;
			*q++ = pW->stx;
			for (i=0, p=pDat; i<lDat; i++, p++) {
				*q++ = *p;
				if (*p == pW->dle) {
					*q++ = pW->dle;
				}
			}
			*q++ = pW->dle;
			*q++ = pW->etx;

			if (pBytes != NULL) {
				*pBytes = lPkt;
			}
		}
		AJCSSEP_LEAVE
	}
	return pPkt;
}
//==============================================================================================================//
//	生成したパケットデータの開放																				//
//																												//
//	引　数	：	pMemBlk	- AjcSsepMakePacketにより生成したパケットデータのアドレス								//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	VO			WINAPI	AjcSsepRelease(VOP pMemBlk)
{
	if (pMemBlk != NULL) {
		free(pMemBlk);
	}
}
//==============================================================================================================//
//	文字コード判定用（半角）テキスト投与																		//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//				pTxt	- 投与する（半角）テキストのアドレス													//
//																												//
//	戻り値	：	なし																							//
//																												//
//	※　通常、文字コード判定用データは、文字列分離の過程で、テキスト部分のデータにて行われるが、AjcSsepPutData	//
//		や、AjcSsepPutTextにてデータが投与されない場合、文字コード判定用データを本関数にて投与する				//
//==============================================================================================================//
AJCEXPORT	VO	WINAPI	AjcSsepPutMbc	(HAJCSSEP pW, C_BCP pTxt)
{
	if (IS_MY_INST(pW)) {
		AJCSSEP_ENTER
		AjcMbcPutData(pW->hMbc, pTxt);
		AJCSSEP_LEAVE
	}
}
//==============================================================================================================//
//	テキストの文字コード種別判定																				//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//																												//
//	Return		:	文字コード種別(AJCMBC_SJIS / EUC / UTF8)													//
//==============================================================================================================//
AJCEXPORT	AJCMBCKIND	WINAPI	AjcSsepGetMbcKind(HAJCSSEP pW)
{
	AJCMBCKIND	mbk = AJCMBC_SJIS;

	if (IS_MY_INST(pW)) {
		AJCSSEP_ENTER
		mbk = AjcMbcCheck(pW->hMbc);
		if (mbk <= 0) {
			mbk = pW->SvMbk;
		}
		pW->SvMbk = mbk;
		AJCSSEP_LEAVE
	}
	return mbk;
}
//==============================================================================================================//
//	最後に判定した文字コード種別取得																			//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//																												//
//	Return		:	文字コード種別(AJCMBC_SJIS / EUC / UTF8)													//
//==============================================================================================================//
AJCEXPORT	AJCMBCKIND	WINAPI	AjcSsepGetLastMbcKind(HAJCSSEP pW)
{
	AJCMBCKIND	mbk = AJCMBC_SJIS;

	if (IS_MY_INST(pW)) {
		AJCSSEP_ENTER
		mbk = pW->SvMbk;
		AJCSSEP_LEAVE
	}
	return mbk;
}
//==============================================================================================================//
//	文字コード判定バッファ リセット																				//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	VO	 		WINAPI	AjcSsepMbcReset	(HAJCSSEP pW)
{
	if (IS_MY_INST(pW)) {
		AJCSSEP_ENTER
		AjcMbcReset(pW->hMbc);				//	自動判別文字コードバッファ初期化
		AJCSSEP_LEAVE
	}
}
