﻿#include	"AjcInternal.h"
#include	"AjcSockClientDef.h"
//**************************************************************************************************************//
//																												//
//	ソケット ( TCP/IP ) クライアント処理						ＡＰＩ											//
//																												//
//**************************************************************************************************************//

#define	IS_MY_INST(P)	(P != NULL && P->InstID == SCT_INST_ID)

//--------------------------------------------------------------------------------------------------------------//
//	グローバルワーク																							//
//--------------------------------------------------------------------------------------------------------------//
ATOM	SctClassForMSIL	= 0;

//==============================================================================================================//
//	起動時初期設定																								//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
BOOL	AjcCtrlSockClientInit(VO)
{
	BOOL		rc = FALSE;
	WNDCLASS	wndclass;

	do {
		//----- ＭＳＩＬコードサポート用 ---------------------//
		wndclass.style			= 0;
		wndclass.lpfnWndProc	= SctWndProcMSIL();
		wndclass.cbClsExtra		= 0;
		wndclass.cbWndExtra		= sizeof(VOP);
		wndclass.hInstance		= hDllInst;
		wndclass.hIcon			= NULL;
		wndclass.hCursor		= LoadCursor(NULL, IDC_ARROW);
		wndclass.hbrBackground	= (HBRUSH)GetStockObject(WHITE_BRUSH);
		wndclass.lpszMenuName	= NULL;
		wndclass.lpszClassName	= L"AjcSctForMSIL";
		if ((SctClassForMSIL = RegisterClass(&wndclass)) == 0) break;

		rc = TRUE;
	} while(0);

	if (rc == FALSE) {
		AjcCtrlSockServerEnd();
	}

	return rc;
}
//==============================================================================================================//
//	終了時後処理																								//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
VO		AjcCtrlSockClientEnd (VO)
{
	if (SctClassForMSIL != 0 ) {UnregisterClass((WCP)SctClassForMSIL, hDllInst); SctClassForMSIL = 0;}
}
//==============================================================================================================//
//	インスタンス生成																							//
//																												//
//	引　数	：	pPort			- TCP/IPポート番号／サービス名文字列											//
//				AddressFamily	- アドレスファミリ(AF_INET or AF_INET6)											//
//				MaxClients		- 最大クライアント数															//
//																												//
//	戻り値	：	インスタンスハンドル																			//
//==============================================================================================================//
AJCEXPORT HAJCSCT	 WINAPI AjcSctCreate(VO)
{
	HAJCSCT		pW		 = NULL;
	BOOL		success	 = FALSE;

	do {
		//	インスタンスワーク確保
		if ((pW = AJCMEM(sizeof(AJCSCT))) == NULL) {
			break;
		}
		memset(pW, 0, sizeof(AJCSCT));
		pW->InstID = SCT_INST_ID;

		//	デフォルトイベントマスク設定
		pW->EvtMask = AJCSCT_EV_ALL;

		//	バイナリチャンクモードを仮定
		pW->ChunkMode		= AJCSCT_CM_BIN;

		//	送受信データ＝SJISを仮定
		pW->RxTxtCode		= AJCSCT_TXT_SJIS;
		pW->TxTxtCode		= AJCSCT_TXT_SJIS;

		//	パケット受信タイムアウト[ms]
		pW->PktTimeout = 3000;

		//	パケットを認識するための制御コード
		pW->stx = 0x02;
		pW->etx = 0x03;
		pW->dle = 0x10;	

		//	ユーザＡＰへのイベント通知用メールボックスハンドル生成
		if ((pW->hFMbxNtc = AjcFMbxCreate(sizeof(PSCTEVTINF), 0, NULL)) == NULL) {
			//	イベント通知用メールボックス生成失敗エラー通知（インスタンス生成時は通知不可）
		//	SctNtcEvtToUser(pW, AJCSCT_EV_ERR, NULL, 0, AJCSCT_ERR_CREMBXNTC);
			break;
		}

		//	送信データ用メールボックス生成
		if ((pW->hVMbxTxD = AjcVMbxCreate(0, NULL)) == NULL) {
			//	送信データ用メールボックス生成失敗エラー通知（インスタンス生成時は通知不可）
		//	SctNtcEvtToUser(pW, AJCSCT_EV_ERR, NULL, 0, AJCSCT_ERR_CREMBXTXD);
			break;
		}

		//	ストリーム分離インスタンス生成
		if (!SctCreSsep(pW)) {
			break;
		}

		//	成功の旨設定
		success = TRUE;

	} while(0);

	//	失敗ならば、リソース解放
	if (!success) {
		if (pW != NULL) {
			if (pW->pServ	 != NULL) {free(pW->pServ); 			pW->pServ	 = NULL;}
			if (pW->pPort	 != NULL) {free(pW->pPort); 			pW->pPort	 = NULL;}
			if (pW->hFMbxNtc != NULL) {AjcFMbxDelete(pW->hFMbxNtc); pW->hFMbxNtc = NULL;}
			AJCFREE(pW);
			pW = NULL;
		}
	}
	return pW;
}
//==============================================================================================================//
//	インスタンス消去																							//
//																												//
//	引　数	：	pW				- インスタンスハンドル															//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT BOOL		 WINAPI AjcSctDelete(HAJCSCT pW)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		//	インスタンス消去中の旨、設定
		pW->fEnding = TRUE;
		//	通信停止
		AjcSctDisconnect(pW);
		//	リソース解放
		if (pW->pServ	 != NULL) {free(pW->pServ); 			pW->pServ	 = NULL;}
		if (pW->pPort	 != NULL) {free(pW->pPort); 			pW->pPort	 = NULL;}
		if (pW->hFMbxNtc != NULL) {AjcFMbxDelete(pW->hFMbxNtc); pW->hFMbxNtc = NULL;}
		if (pW->hVMbxTxD != NULL) {AjcVMbxDelete(pW->hVMbxTxD); pW->hVMbxTxD = NULL;}
		if (pW->hSsep	 != NULL) {AjcSsepDelete(pW->hSsep	 ); pW->hSsep	 = NULL;}
		//	インスタンスワーク解放
		AJCFREE(pW);
		pW = NULL;
	}
	return rc;
}
//==============================================================================================================//
//	回線接続																									//
//																												//
//	引　数	：	pW				- インスタンスハンドル															//
//				pServ			- サーバのホスト名																//
//				pPort			- TCP/IPポート番号／サービス名文字列											//
//				AddressFamily	- アドレスファミリ(AF_INET or AF_INET6)											//
//				hWndNtc			- ユーザＡＰへのイベント通知用ウインドハンドル									//
//				WndMsgNtc		- ユーザＡＰへのイベント通知用ウインドメッセージ								//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
static	 BOOL	 SubConnect(HAJCSCT pW, C_WCP pServ, C_WCP pPort, int AddressFamily, HWND hWndNtc, UI WndMsgNtc, BOOL fUniCode);
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL		 WINAPI AjcSctConnectA(HAJCSCT pW, C_BCP pServ, C_BCP pPort, int AddressFamily, HWND hWndNtc, UI WndMsgNtc)
{
	BOOL	rc = FALSE;
	int		len;
	WCP		pTmpServ = NULL;
	WCP		pTmpPort = NULL;

	if (pServ != NULL && pPort != NULL) {
		do {
			//	サーバ名をワイド文字に変換
			len = MultiByteToWideChar(CP_ACP, 0, pServ, -1, NULL, 0);
			if (len != 0 && (pTmpServ = AjcTAlloc(len))) {
				MultiByteToWideChar(CP_ACP, 0, pServ, -1, pTmpServ, len);
			}
			else break;
			//	ポートをワイド文字に変換
			len = MultiByteToWideChar(CP_ACP, 0, pPort, -1, NULL, 0);
			if (len != 0 && (pTmpPort = AjcTAlloc(len))) {
				MultiByteToWideChar(CP_ACP, 0, pPort, -1, pTmpPort, len);
			}
			else break;
			//	接続処理
			rc = SubConnect(pW, pTmpServ, pTmpPort, AddressFamily, hWndNtc, WndMsgNtc, FALSE);
		} while(0);
		if (pTmpServ != NULL) free(pTmpServ);
		if (pTmpPort != NULL) free(pTmpPort);
	}

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL		 WINAPI AjcSctConnectW(HAJCSCT pW, C_WCP pServ, C_WCP pPort, int AddressFamily, HWND hWndNtc, UI WndMsgNtc)
{
	BOOL	rc = FALSE;

	rc = SubConnect(pW, pServ, pPort, AddressFamily, hWndNtc, WndMsgNtc, TRUE);

	return rc;
}
//----- 共通サブ -----------------------------------------------------------------------------------------------//
static	 BOOL	 SubConnect(HAJCSCT pW, C_WCP pServ, C_WCP pPort, int AddressFamily, HWND hWndNtc, UI WndMsgNtc, BOOL fUniCode)
{
	BOOL		rc = FALSE;
	UI			stl;

	if (IS_MY_INST(pW) && !pW->fConnected) {
		do {
			//	文字コード判定バッファリセット
			AjcSsepMbcReset(pW->hSsep);

			//	送信テキスト半端バイト／ワードリセット
			pW->SvSndA = 0;
			pW->SvSndW = 0;

			//	アドレスファミリ設定
			pW->AddressFamily = AddressFamily;

			//	サーバ名設定
			if (pW->pServ != NULL) {
				free(pW->pServ);
				pW->pServ = NULL;
			}
			stl = (UI)wcslen(pServ);
			if (pW->pServ = AjcTAlloc(stl + 1)) {
				wcscpy(pW->pServ, pServ);
			}
			else break;

			//	ポート／サービス名設定
			if (pW->pPort != NULL) {
				free(pW->pPort);
				pW->pPort = NULL;
			}
			stl = (UI)wcslen(pPort);
			if (pW->pPort = AjcTAlloc(stl + 1)) {
				wcscpy(pW->pPort, pPort);
			}
			else break;

			//	ユーザウインドへの通知情報設定
			pW->hWndNtc = hWndNtc;
			pW->MsgNtc	= WndMsgNtc;

			//	バイト文字／ワイド文字モード設定
			pW->fUnicode = fUniCode;

			//	ユーザＡＰへのイベント通知用メールボックス・リセット
			AjcFMbxPurge(pW->hFMbxNtc);

			//	送信データ用メールボックス・リセット
			AjcVMbxPurge(pW->hVMbxTxD);

			//	ストリーム分離のリセット
			AjcSsepReset(pW->hSsep);

			//	通信スレッドを生成する
			if (!SctStartThreadComm(pW)) {
				//	通信スレッド生成失敗エラー通知
				SctNtcEvtToUser(pW, AJCSCT_EV_ERR, NULL, 0, AJCSCT_ERR_SUBTHREAD);
				break;
			}
			//	接続中状態
			pW->State = AJCSCT_CONNECTING;
			//	成功の旨設定
			rc = TRUE;
		} while(0);

		//	失敗ならば、リソース解放
		if (!rc) {
			if (pW->hSock != 0	 ) {closesocket(pW->hSock); pW->hSock = 0;	 }
			if (pW->hThread != NULL) {
				//	スレッド終了待ち（pW->hSockクローズにより、スレッドはconnect()失敗により終了する）
				WaitForSingleObject(pW->hThread, 10000);
				//	スレッドハンドル クローズ
				CloseHandle(pW->hThread);
				pW->hThread = NULL;
			}
		}
	}
	return rc;
}

//==============================================================================================================//
//	回線切断																									//
//																												//
//	引　数	：	pW				- インスタンスハンドル															//
//				msTimeout		- 回線切断待ち時間[ms]															//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT BOOL		 WINAPI AjcSctDisconnect(HAJCSCT pW)
{
	return AjcSctDisconnectEx(pW, 10000);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL		 WINAPI AjcSctDisconnectEx(HAJCSCT pW, UI msTimeout)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW) && pW->fConnected) {
		//	通信終了中フラグ セット
		pW->fEnding  = TRUE;

		//	回線切断中
		pW->State = AJCSCT_DISCONNECTING;

		//	サーバ名解放
		if (pW->pServ != NULL) {
			free(pW->pServ);
			pW->pServ = NULL;
		}

		//	ポート／サービス名解放
		if (pW->pPort != NULL) {
			free(pW->pPort);
			pW->pPort = NULL;
		}

		//	ソケットクローズ
		if (pW->hSock != 0) {
			closesocket(pW->hSock);
			pW->hSock = 0;
		}

		//	通信スレッド終了を待つ
		pW->fExit = TRUE;
		if (pW->hThread != NULL) {
			if (WaitForSingleObject((HANDLE)pW->hThread, msTimeout) == WAIT_TIMEOUT) {
				SctNtcEvtToUser(pW, AJCSCT_EV_ERR, NULL, 0, AJCSCT_ERR_TIMEOUT);
			}
			CloseHandle(pW->hThread);  pW->hThread = NULL;
		}

		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	回線状態取得																								//
//																												//
//	引　数	：	pW				- インスタンスハンドル															//
//																												//
//	戻り値	：	接続状態																						//
//==============================================================================================================//
AJCEXPORT	AJCSCT_STATE	WINAPI	AjcSctGetState		(HAJCSCT pW)
{
	AJCSCT_STATE	rc = AJCSCT_DISCONNECT;

	if (IS_MY_INST(pW)) {
		rc = pW->State;
	}
	return rc;
}


//==============================================================================================================//
//	チャンクデータの受信モード（テキスト／バイナリ）設定														//
//																												//
//	引　数	：	pW				- インスタンスハンドル															//
//				ChunkMode		- チャンクデータ通知モード														//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcSctSetChunkMode		(HAJCSCT pW, AJCSCT_CHUNKMODE ChunkMode)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		pW->ChunkMode = ChunkMode;
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	チャンクデータの受信モード（テキスト／バイナリ）取得														//
//																												//
//	引　数	：	pW				- インスタンスハンドル															//
//																												//
//	戻り値	：	チャンクデータ通知モード																		//
//==============================================================================================================//
AJCEXPORT	AJCSCT_CHUNKMODE	WINAPI	AjcSctGetChunkMode		(HAJCSCT pW)
{
	AJCSCT_CHUNKMODE	rc = 0;

	if (IS_MY_INST(pW)) {
		rc = pW->ChunkMode;
	}
	return rc;
}
//==============================================================================================================//
//	イベントマスク設定																							//
//																												//
//	引　数	：	pW				- インスタンスハンドル															//
//				EvtMsk			- イベントマスク																//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT BOOL		 WINAPI AjcSctSetEvtMask(HAJCSCT pW, UI EvtMsk)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		pW->EvtMask = EvtMsk;
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	イベントマスク取得																							//
//																												//
//	引　数	：	pW				- インスタンスハンドル															//
//																												//
//	戻り値	：	イベントマスク																					//
//==============================================================================================================//
AJCEXPORT UI		 WINAPI AjcSctGetEvtMask(HAJCSCT pW)
{
	UI		rc = 0;

	if (IS_MY_INST(pW)) {
		rc = pW->EvtMask;
	}
	return rc;
}
//==============================================================================================================//
//	受信テキストの文字コード設定																				//
//																												//
//	引　数		：	pW				- インスタンスハンドル														//
//					code			- 文字コード種別															//
//																												//
//	戻り値		：	TRUE  - OK																					//
//					FALSE - Error																				//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcSctSetRxTextCode(HAJCSCT pW, AJCSCT_TEXTCODE code)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		switch (code) {
			case AJCSCT_TXT_SJIS:				//	シフトＪＩＳ
			case AJCSCT_TXT_EUC:				//	ＥＵＣ　　　
			case AJCSCT_TXT_UTF8:				//	ＵＴＦ－８　
			case AJCSCT_TXT_AUTO:				//	自動判別　　
				//	文字コード判定バッファリセット
				AjcSsepMbcReset(pW->hSsep);
				//	文字コード種別設定
				pW->RxTxtCode = code;
				rc = TRUE;
				break;

			default:
				rc = FALSE;
				break;
		}
	}
	return rc;
}
//==============================================================================================================//
//	受信テキストの文字コード取得																				//
//																												//
//	引　数		：	pW				- インスタンスハンドル														//
//																												//
//	戻り値		：	文字コード種別（AJCSCT_TXT_SJIS / EUC / UTF8 / AUTO）										//
//==============================================================================================================//
AJCEXPORT	AJCSCT_TEXTCODE	WINAPI	AjcSctGetRxTextCode(HAJCSCT pW)
{
	AJCSCT_TEXTCODE	rc = (AJCSCT_TEXTCODE)0;

	if (IS_MY_INST(pW)) {
		rc = pW->RxTxtCode;
	}
	return rc;
}
//==============================================================================================================//
//	実際の受信文字エンコード種別取得																			//
//																												//
//	引　数		：	pW				- インスタンスハンドル														//
//																												//
//	戻り値		：	文字コード種別（AJCSCP_TXT_SJIS / EUC / UTF8）												//
//==============================================================================================================//
AJCEXPORT	AJCSCT_TEXTCODE	WINAPI	AjcSctGetActualRxTextCode	(HAJCSCT pW)
{
	AJCSCT_TEXTCODE	rc = (AJCSCT_TEXTCODE)0;
	AJCMBCKIND		mbk;

	if (IS_MY_INST(pW)) {
		if ((rc = pW->RxTxtCode) == AJCSCT_TXT_AUTO) {
			mbk = AjcSsepGetMbcKind(pW->hSsep);
			switch (mbk) {
				case AJCMBC_SJIS:	rc = AJCSCT_TXT_SJIS;	break;
				case AJCMBC_EUC:	rc = AJCSCT_TXT_EUC	;	break;
				case AJCMBC_UTF8:	rc = AJCSCT_TXT_UTF8;	break;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	送信テキストの文字コード設定																				//
//																												//
//	引　数		：	pW				- インスタンスハンドル														//
//					code			- 文字コード種別															//
//																												//
//	戻り値		：	TRUE  - OK																					//
//					FALSE - Error																				//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcSctSetTxTextCode(HAJCSCT pW, AJCSCT_TEXTCODE code)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		switch (code) {
			case AJCSCT_TXT_SJIS:				//	シフトＪＩＳ
			case AJCSCT_TXT_EUC:				//	ＥＵＣ
			case AJCSCT_TXT_UTF8:				//	ＵＴＦ－８
			case AJCSCT_TXT_AUTO:				//	自動判別
				//	文字コード種別設定
				pW->TxTxtCode = code;
				rc = TRUE;
				break;

			default:
				rc = FALSE;
				break;
		}
	}
	return rc;
}
//==============================================================================================================//
//	送信テキストの文字コード取得																				//
//																												//
//	引　数		：	pW				- インスタンスハンドル														//
//																												//
//	戻り値		：	文字コード種別（AJCSCT_TXT_SJIS / EUC / UTF8 / AUTO）										//
//==============================================================================================================//
AJCEXPORT	AJCSCT_TEXTCODE	WINAPI	AjcSctGetTxTextCode(HAJCSCT pW)
{
	AJCSCT_TEXTCODE	rc = (AJCSCT_TEXTCODE)0;

	if (IS_MY_INST(pW)) {
		rc = pW->TxTxtCode;
	}
	return rc;
}
//==============================================================================================================//
//	実際の送信文字エンコード種別取得																			//
//																												//
//	引　数		：	pW				- インスタンスハンドル														//
//																												//
//	戻り値		：	文字コード種別（AJCSCT_TXT_SJIS / EUC / UTF8）												//
//==============================================================================================================//
AJCEXPORT	AJCSCT_TEXTCODE	WINAPI	AjcSctGetActualTxTextCode(HAJCSCT pW)
{
	AJCSCT_TEXTCODE	rc = (AJCSCT_TEXTCODE)0;

	if (IS_MY_INST(pW)) {
		if ((rc = pW->TxTxtCode) == AJCSCT_TXT_AUTO) {
			rc = AjcSctGetActualRxTextCode(pW);
		}
	}
	return rc;
}
//==============================================================================================================//
//	イベント発生待ち																							//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				pwParam		- イベントコードを格納するバッファのアドレス										//
//				plParam		- イベントデータ情報を格納するバッファのアドレス									//
//				msTime		- イベント発生待ち時間[ms]															//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - タイムアウト																			//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcSctWaitEvent		(HAJCSCT pW, WPARAM *pwParam, LPARAM *plParam, UI msTime)
{
	BOOL		rc = FALSE;
	PSCTEVTINF	pEvtInf;

	if (IS_MY_INST(pW) && pwParam != NULL && plParam != NULL) {
		if (AjcFMbxDeque(pW->hFMbxNtc, &pEvtInf, msTime)) {
			if (pEvtInf != NULL) {
				*pwParam = pEvtInf->Evt;
				*plParam = (LPARAM)pEvtInf;
				rc = TRUE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	イベントデータ取得																							//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				lParam		- イベントデータ情報																//
//				ppDat		- 受信データのアドレスを格納するバッファのアドレス									//
//				plDat		- 受信データのバイト数を格納するバッファのアドレス									//
//				pParam		- パラメタ情報を格納するバッファのアドレス											//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - イベントデータなし																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcSctGetEventData	(HAJCSCT pW, LPARAM lParam, VOP *ppDat, UIP plDat, UIP pParam)
{
	BOOL		rc = FALSE;
	PSCTEVTINF		pEvtInf;

	if (IS_MY_INST(pW) && lParam != 0) {
		pEvtInf = (PSCTEVTINF)lParam;
		if (ppDat  != NULL) *ppDat	= pEvtInf->pDat;
		if (plDat  != NULL) *plDat	= pEvtInf->lDat;
		if (pParam != NULL) *pParam = pEvtInf->param;
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	イベントデータ開放																							//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				lParam		- イベントデータ情報																//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcSctRelEventData	(HAJCSCT pW, LPARAM lParam)
{
	BOOL		rc = FALSE;
	PSCTEVTINF		pEvtInf;

	if (IS_MY_INST(pW) && lParam != 0) {
		pEvtInf = (PSCTEVTINF)lParam;
		//	イベント情報解放（イベントデータ(pDat)は AjcMain::AjcReleaseInstance() -> AjcSockClientSubFunc::cbReleaseEvtInf()経由で解放）
		AjcReleaseInstance(pEvtInf);
		rc = TRUE;
	}
	return rc;
}

//==============================================================================================================//
//	パケットフレームを認識する為の制御コード設定																//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				stx			- ＳＴＸコード値																	//
//				etx			- ＥＴＸコード値																	//
//				dle			- ＤＬＥコード値																	//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcSctSetPktCtrlCode(HAJCSCT pW, UI stx, UI etx, UI dle)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		if (stx != 0) pW->stx = stx;
		if (etx != 0) pW->etx = etx;
		if (dle != 0) pW->dle = dle;
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	パケットフレームを認識する為の制御コード取得																//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				pStx		- ＳＴＸコード値を格納するバッファのアドレス（不要時はＮＵＬＬ）					//
//				pEtx		- ＥＴＸコード値を格納するバッファのアドレス（不要時はＮＵＬＬ）					//
//				pDle		- ＤＬＥコード値を格納するバッファのアドレス（不要時はＮＵＬＬ）					//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcSctGetPktCtrlCode(HAJCSCT pW, UIP pStx, UIP pEtx, UIP pDle)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		if (pStx != NULL) *pStx= pW->stx;
		if (pEtx != NULL) *pEtx= pW->etx;
		if (pDle != NULL) *pDle= pW->dle;
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	パケットフレーム受信タイムアウト値設定																		//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				msTime		- タイムアウト時間[ms]																//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcSctSetPktTimeout (HAJCSCT pW, UI msTime)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		pW->PktTimeout;
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	パケットフレーム受信タイムアウト値取得																		//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				pMsTime		- タイムアウト時間[ms]を格納するバッファのアドレス（不要時はＮＵＬＬ）				//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcSctGetPktTimeout (HAJCSCT pW, UIP pMsTime)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		if (pMsTime != NULL) {
			*pMsTime = pW->PktTimeout;
		}
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	１文字送信																									//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				code		- 送信する文字の文字コード															//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcSctSendCharA		(HAJCSCT pW, BC code)
{
	BOOL	rc = FALSE;
	BC		txt[4];

	if (IS_MY_INST(pW) && pW->State == AJCSCT_CONNECT) {
		//	全角１バイト目退避なし？
		if (pW->SvSndA == 0) {
			if (MAjcIsLeadA(code)) {
				pW->SvSndA = code;
				rc = TRUE;
			}
			else {
				txt[0] = code;
				txt[1] = 0;
				rc = AjcSctSendTextA(pW, txt, -1);
			}
		}
		//	全角１バイト目退避あり？
		else {
			//	２バイト作成し送信
			txt[0] = pW->SvSndA;
			txt[1] = code;
			txt[2] = 0;
			rc = AjcSctSendTextA(pW, txt, -1);
			//	全角１バイト目退避クリアー
			pW->SvSndA = 0;
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcSctSendCharW		(HAJCSCT pW, WC code)
{
	BOOL	rc = FALSE;
	WC		txt[2];

	if (IS_MY_INST(pW) && pW->State == AJCSCT_CONNECT) {
		//	サロゲート１ワード目退避なし？
		if (pW->SvSndW == 0) {
			if (MAjcIsLeadW(code)) {
				pW->SvSndW = code;
				rc = TRUE;
			}
			else {
				txt[0] = code;
				txt[1] = 0;
				rc = AjcSctSendTextW(pW, txt, 1);
			}
		}
		//	サロゲート１ワード目退避あり？
		else {
			//	２ワード作成し送信
			txt[0] = pW->SvSndW;
			txt[1] = code;
			txt[2] = 0;
			rc = AjcSctSendTextW(pW, txt, 2);
			//	サロゲート１ワード目退避クリアー
			pW->SvSndW = 0;
		}
	}
	return rc;
}
//==============================================================================================================//
//	テキストデータ送信																							//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				pTxt		- 送信するテキストデータのアドレス（シフトＪＩＳ／ＵＮＩＣＯＤＥテキスト）			//
//				lTxt		- 送信するテキストデータの文字数（－１の場合は自動算出）							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcSctSendTextA		(HAJCSCT pW, C_BCP pTxt, UI lTxt)
{
	BOOL		rc	= FALSE;
	AJCMBCKIND	mbk = AJCMBC_SJIS;
	BCP			pAlt = NULL;

	if (IS_MY_INST(pW) && pTxt != NULL && pW->State == AJCSCT_CONNECT) {
		rc = TRUE;
		do {
			//	ソーステキスト長設定
			lTxt = AjcStrAdjustLenA(pTxt, lTxt);
			if (lTxt == 0) break;
			//	前回の半端バイトとテキストを動的バッファに格納
			if (pW->SvSndA != 0) {
				if (pAlt = AjcTAllocA(1 + lTxt + 1)) {
					*pAlt = pW->SvSndA;
					strncpy(pAlt + 1, pTxt, lTxt);
					*(pAlt + (1 + lTxt)) = 0;
					//	テキストポインタ切り替え
					pTxt = pAlt;
					lTxt++;
				}
				else {
					rc = FALSE;
				}
				pW->SvSndA = 0;
			}
			if (!rc) break;
			//	テキストの末尾が全角１バイト目ならば、退避して末尾を除外
			if (mbsbtype(pTxt, lTxt - 1) == _MBC_LEAD) {
				pW->SvSndA = pTxt[lTxt - 1];
				lTxt--;
			}
			if (lTxt == 0) break;
			//	送信テキストコード設定
			switch (pW->TxTxtCode) {
				default:
				case AJCSCT_TXT_SJIS:	mbk = AJCMBC_SJIS;						break;	//	シフトＪＩＳ
				case AJCSCT_TXT_EUC:	mbk = AJCMBC_EUC;						break;	//	ＥＵＣ　　　
				case AJCSCT_TXT_UTF8:	mbk = AJCMBC_UTF8;						break;	//	ＵＴＦ－８　
				case AJCSCP_TXT_AUTO:													//	自動判別
										switch (pW->RxTxtCode) {
											case AJCSCP_TXT_SJIS:	mbk = AJCMBC_SJIS;						break;
											case AJCSCP_TXT_EUC:	mbk = AJCMBC_EUC;						break;
											case AJCSCP_TXT_UTF8:	mbk = AJCMBC_UTF8;						break;
											case AJCSCP_TXT_AUTO:	mbk = AjcSsepGetLastMbcKind(pW->hSsep);	break;
										}
										break;
			}
			//	送信コード＝シフトＪＩＳ
			if (mbk == AJCMBC_SJIS) {
				C_BCP	p = pTxt;
				UI		len;
				//	送信テキストエンキュー
				while (lTxt > 0) {
					len = __min(lTxt, AJCSCTMAX_TXBUF);
					if (!(rc = AjcVMbxEnque(pW->hVMbxTxD, p, len))) break;
					p	 += len;
					lTxt -= len;
				}
			}
			//	送信コード＝ＥＵＣ／ＵＴＦ８
			else {
				BCP			pTop = NULL;
				BCP			pTmp = NULL;
				UI			lTmp = 0;
				UI			len;
				do {
					//	文字コード変換
					if (mbk == AJCMBC_EUC) {
						lTmp = AjcSJisToEucEx(pTxt, lTxt, NULL, 0);
						if ((pTop = AJCMEM(lTmp)) == NULL) break;
						AjcSJisToEucEx(pTxt, lTxt, pTop, lTmp);
					}
					else {
						lTmp = AjcMbcToUtf8Ex(pTxt, lTxt, NULL, 0);
						if ((pTop = AJCMEM(lTmp)) == NULL) break;
						AjcMbcToUtf8Ex(pTxt, lTxt, pTop, lTmp);
					}
					//	送信テキストエンキュー
					pTmp = pTop;
					while (lTmp > 0) {
						len = __min(lTmp, AJCSCTMAX_TXBUF);
						if (!(rc = AjcVMbxEnque(pW->hVMbxTxD, pTmp, len))) break;
						pTmp += len;
						lTmp -= len;
					}
					//	変換バッファ解放
					free(pTop);
				} while (0);
			}
		} while(0);
		//	切り替えバッファ解放
		if (pAlt != NULL) free(pAlt);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcSctSendTextW		(HAJCSCT pW, C_WCP pTxt, UI lTxt)
{
	BOOL		rc	= FALSE;
	AJCMBCKIND	mbk = AJCMBC_SJIS;
	WCP			pAlt = NULL;

	if (IS_MY_INST(pW) && pTxt != NULL && pW->State == AJCSCT_CONNECT) {
		rc = TRUE;
		do {
			//	ソーステキスト長設定
			lTxt = AjcStrAdjustLenW(pTxt, lTxt);
			if (lTxt == 0) break;
			//	前回の半端ワードとテキストを動的バッファに格納
			if (pW->SvSndW != 0) {
				if (pAlt = AjcTAllocW(1 + lTxt + 1)) {
					*pAlt = pW->SvSndW;
					wcsncpy(pAlt + 1, pTxt, lTxt);
					*(pAlt + (1 + lTxt)) = 0;
					//	テキストポインタ切り替え
					pTxt = pAlt;
					lTxt++;
				}
				else {
					rc = FALSE;
				}
				pW->SvSndW = 0;
			}
			if (!rc) break;
			//	テキストの末尾がサロゲート１ワード目ならば、退避して末尾を除外
			if (MAjcIsLeadW(pTxt[lTxt - 1])) {
				pW->SvSndW = pTxt[lTxt - 1];
				lTxt--;
			}
			if (lTxt == 0) break;
			//	送信テキストコード設定
			switch (pW->TxTxtCode) {
				default:
				case AJCSCP_TXT_SJIS:	mbk = AJCMBC_SJIS;						break;	//	シフトＪＩＳ
				case AJCSCP_TXT_EUC:	mbk = AJCMBC_EUC;						break;	//	ＥＵＣ
				case AJCSCP_TXT_UTF8:	mbk = AJCMBC_UTF8;						break;	//	ＵＴＦ－８
				case AJCSCP_TXT_AUTO:													//	自動判別
										switch (pW->RxTxtCode) {
											case AJCSCP_TXT_SJIS:	mbk = AJCMBC_SJIS;						break;
											case AJCSCP_TXT_EUC:	mbk = AJCMBC_EUC;						break;
											case AJCSCP_TXT_UTF8:	mbk = AJCMBC_UTF8;						break;
											case AJCSCP_TXT_AUTO:	mbk = AjcSsepGetLastMbcKind(pW->hSsep);	break;
										}
										break;
			}
			//	送信コード＝シフトＪＩＳ
			if (mbk == AJCMBC_SJIS) {
				BCP		pTxd;
				UI		lTxd;
				C_BCP	p;
				UI		len;
				lTxd = WideCharToMultiByte(CP_ACP, 0, pTxt, lTxt, NULL, 0, NULL, NULL);
				if (pTxd = AjcTAllocA(lTxd)) {
					WideCharToMultiByte(CP_ACP, 0, pTxt, lTxt, pTxd, lTxd, NULL, NULL);
					//	送信テキストエンキュー
					p = pTxd;
					while (lTxd > 0) {
						len = __min(lTxd, AJCSCTMAX_TXBUF);
						if (!(rc = AjcVMbxEnque(pW->hVMbxTxD, p, len))) break;
						p	 += len;
						lTxd -= len;
					}
					free(pTxd);
				}
			}
			//	送信コード＝ＥＵＣ／ＵＴＦ８
			else {
				BCP			pTop = NULL;
				BCP			pTmp = NULL;
				UI			lTmp = 0;
				UI			len;
				do {
					//	文字コード変換(UNICODE -> EUC)
					if (mbk == AJCMBC_EUC) {
						BCP		pSJis;
						UI		lSJis;
						lSJis = WideCharToMultiByte(CP_ACP, 0, pTxt, lTxt, NULL, 0, NULL, NULL);
						if ((pSJis = AjcTAllocA(lSJis)) == NULL) break;
						WideCharToMultiByte(CP_ACP, 0, pTxt, lTxt, pSJis, lSJis, NULL, NULL);
						lTmp = AjcSJisToEucEx(pSJis, lSJis, NULL, 0);
						if ((pTop = AjcTAllocA(lTmp)) == NULL) {free(pSJis); break;}
						AjcSJisToEucEx(pSJis, lSJis, pTop, lTmp);
					}
					//	文字コード変換(UNICODE -> UTF-8)
					else {
						lTmp = WideCharToMultiByte(CP_UTF8, 0, pTxt, lTxt, NULL, 0, NULL, NULL);
						if ((pTop = AjcTAllocA(lTmp)) == NULL) break;
						WideCharToMultiByte(CP_UTF8, 0, pTxt, lTxt, pTop, lTmp, NULL, NULL);
					}
					//	送信テキストエンキュー
					pTmp = pTop;
					while (lTmp > 0) {
						len = __min(lTmp, AJCSCTMAX_TXBUF);
						if (!(rc = AjcVMbxEnque(pW->hVMbxTxD, pTmp, len))) break;
						pTmp += len;
						lTmp -= len;
					}
					//	変換バッファ解放
					free(pTop);
				} while (0);
			}
		} while (0);
		//	切り替えバッファ解放
		if (pAlt != NULL) free(pAlt);
	}
	return rc;
}
//==============================================================================================================//
//	書式テキストデータ送信																						//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				pFmt		- 書式テキスト																		//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI		AjcSctSendTextFA	(HAJCSCT pW, C_BCP pFmt, ...)
{
	BOOL	rc = FALSE;
	va_list vls;
	BC		txt[2048];

	if (IS_MY_INST(pW) && pFmt != NULL && pW->State == AJCSCT_CONNECT) {
		va_start(vls, pFmt);
		_vsnprintf(txt, 2048, pFmt, vls);
		txt[2047] = 0;
		va_end	(vls);
		rc = AjcSctSendTextA(pW, txt, -1);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI		AjcSctSendTextFW	(HAJCSCT pW, C_WCP pFmt, ...)
{
	BOOL	rc = FALSE;
	va_list vls;
	WC		txt[2048];

	if (IS_MY_INST(pW) && pFmt != NULL && pW->State == AJCSCT_CONNECT) {
		va_start(vls, pFmt);
		_vsnwprintf(txt, 2048, pFmt, vls);
		txt[2047] = 0;
		va_end	(vls);
		rc = AjcSctSendTextW(pW, txt, -1);
	}
	return rc;
}
//==============================================================================================================//
//	バイナリデータ送信																							//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				pDat		- 送信するバイナリデータのアドレス													//
//				lDat		- 送信するバイナリデータのバイト数													//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcSctSendBinData	(HAJCSCT pW, C_VOP pDat, UI lDat)
{
	BOOL	rc = FALSE;
	C_UBP	p = (C_UBP)pDat;
	UI		len;

	if (IS_MY_INST(pW) && pDat != NULL && pW->State == AJCSCT_CONNECT) {
		rc = TRUE;
		while (lDat > 0) {
			len = __min(lDat, AJCSCTMAX_TXBUF);
			if (!(rc = AjcVMbxEnque(pW->hVMbxTxD, p, len))) break;
			p	 += len;
			lDat -= len;
		}
	}
	return rc;
}
//==============================================================================================================//
//	パケットデータ送信																							//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				pPkt		- 送信するパケットデータのアドレス（空パケット送信時はNUULでもOK)					//
//				lPkt		- 送信するパケットデータのバイト数（空パケット送信時は０)							//
//																												//
//	戻り値	：	４以上	- OK（実際の送信バイト数）																//
//				０		- Error																					//
//==============================================================================================================//
AJCEXPORT	UI		WINAPI	AjcSctSendPacket	(HAJCSCT pW, C_VOP pPkt, UI lPkt)
{
	UI			rc = 0;
	UBP			pImg;
	UI			lImg, len;
	UBP			p;

	if (IS_MY_INST(pW) && (pPkt != NULL || lPkt == 0) && pW->State == AJCSCT_CONNECT) {
		if (pImg = (UBP)AjcSsepMakePacket(pW->hSsep, pPkt, lPkt, &lImg)) {
			rc = lImg;
			p = pImg;
			while (lImg > 0) {
				len = __min(lImg, AJCSCTMAX_TXBUF);
				if (!AjcVMbxEnque(pW->hVMbxTxD, p, len)) {rc = 0; break;}
				p	 += len;
				lImg -= len;
			}
			AjcSsepRelease(pImg);
		}
	}
	return rc;
}
//==============================================================================================================//
//	全受信済データ破棄																							//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcSctPurgeRecvData	(HAJCSCT pW)
{
	BOOL	rc = TRUE;

	if (IS_MY_INST(pW)) {
		AjcFMbxPurge(pW->hFMbxNtc);
	}
	else rc = FALSE;

	return rc;
}
//==============================================================================================================//
//	全送信待ちデータ破棄																						//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcSctPurgeSendData	(HAJCSCT pW)
{
	BOOL	rc = TRUE;

	if (IS_MY_INST(pW)) {
		AjcVMbxPurge(pW->hVMbxTxD);
	}
	else rc = FALSE;

	return rc;
}
//==============================================================================================================//
//	全送受信データ破棄																							//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcSctPurgeAllData	(HAJCSCT pW)
{
	BOOL	rc = TRUE;

	if (IS_MY_INST(pW)) {
		AjcVMbxPurge(pW->hVMbxTxD);
		AjcFMbxPurge(pW->hFMbxNtc);
	}
	else rc = FALSE;

	return rc;
}
