﻿#include	"AjcInternal.h"

//**************************************************************************************************************//
//																												//
//	状態遷移制御																								//
//																												//
//**************************************************************************************************************//

//--------------------------------------------------------------------------------------------------------------//
//	作業領域																									//
//--------------------------------------------------------------------------------------------------------------//
static	ATOM	StcHideClass = 0;

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
static	int					SubExecEvent(HAJCSTC pW, PCAJCSTC_EVT_NF pEvt);
static	VO					SubPreFunc	(HAJCSTC pW, const UB pft[AJCSTC_MAX_PSF]);
static	VO					SubTimStart (HAJCSTC pW, UW tmc);
static	VO					SubTimStop	(HAJCSTC pW, UW tmc);

static	LRESULT CALLBACK 	WndStcHideProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);

//==============================================================================================================//
//	起動時初期設定																								//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
BOOL	AjcStcInit(VO)
{
	WNDCLASS	wndclass;

	wndclass.style			= 0;
	wndclass.lpfnWndProc	= WndStcHideProc;
	wndclass.cbClsExtra		= 0;
	wndclass.cbWndExtra		= sizeof(HAJCSTC);
	wndclass.hInstance		= hDllInst;
	wndclass.hIcon			= NULL;
	wndclass.hCursor		= NULL;
	wndclass.hbrBackground	= (HBRUSH)GetStockObject(LTGRAY_BRUSH);
	wndclass.lpszMenuName	= NULL;
	wndclass.lpszClassName	= L"AjcStcHideClass";
	return ((StcHideClass	= RegisterClass(&wndclass)) != 0);
}
//==============================================================================================================//
//	終了時後処理																								//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
VO		AjcStcEnd (VO)
{
	if (StcHideClass != 0) {
		UnregisterClass(L"AjcStcHideClass", hDllInst);
		StcHideClass = 0;
	}
}

//==============================================================================================================//
//	インスタンス生成																							//
//																												//
//	引　数	：	pCom	- 他の状態から当該状態へ遷移時のアクションテーブル（不要時はＮＵＬＬ）					//
//				pRel	- 当該状態から他の状態へ遷移時のアクションテーブル（不要時はＮＵＬＬ）					//
//				pEvt	- イベントアクションテーブルのアドレス													//
//				nState	- 状態の個数																			//
//				cbp 	- コールバックパラメタ																	//
//				cbPsf	- プリセットファンクションハンドラ														//
//				cbNtc	- 状態遷移制御通知用コールバック														//
//																												//
//	戻り値	：	≠NULL : インスタンスハンドル																	//
//				＝NULL : エラー																					//
//==============================================================================================================//
AJCEXPORT HAJCSTC	WINAPI AjcStcCreate(PCAJCSTC_STSACT pCom,
										PCAJCSTC_STSACT pRel,
										PCAJCSTC_EVT_NF pEvt,
										int 			 nState,
										UX				 cbp ,
										VO (CALLBACK *cbPsf)(UI pno, int sts, UX cbp),
										VO (CALLBACK *cbNtc)(AJCSTC_NTC ntc, UL p1, UL p2, UX cbp))
{
	HAJCSTC	pW = NULL;
	int		i;

	do {
		//----- インスタンスワーク生成 -------------------------------------------------------------------------//
		if (!(pW = (HAJCSTC)AJCMEM(sizeof(AJCSTC)))) break;
		memset(pW, 0, sizeof(AJCSTC));
		//----- ウインド生成 -----------------------------------------------------------------------------------//
		pW->hwnd = CreateWindow(L"AjcStcHideClass",			// window class name
								L"",						// window caption
								0,							// window style
								0,							// initial x position
								0,							// initial y position
								0,							// initial x size
								0,							// initial y size
								NULL,						// parent window handle
								NULL,						// window menu handle
								hDllInst,					// program instance handle
								pW);						// creation parameters
		if (pW->hwnd == NULL) {free(pW); pW = NULL; break;}
		ShowWindow(pW->hwnd, SW_HIDE);
		//----- インスタンスワーク初期化 -----------------------------------------------------------------------//
		pW->fBusy	= 0;							//	イベント処理中フラグ
		pW->CurSts	= 0;							//	現在の状態
		pW->nState	= nState;						//	状態の個数
		pW->pCom	= pCom;							//	他の状態から当該状態へ遷移時のアクションテーブル
		pW->pRel	= pRel;							//	当該状態から他の状態へ遷移時のアクションテーブル
		pW->pEvt	= pEvt;							//	イベントアクションテーブル
		pW->six		= 0;							//	イベントバッファ・インデクス
		pW->eix		= 0;							//	・
		pW->cbp		= cbp;							//	コールバックパラメタ
		pW->cbPsf	= cbPsf;						//	プリセットファンクション通知用コールバック
		pW->cbNtc	= cbNtc;
		//	タイムアウトイベントクリアー
		for (i = 0; i < AJCSTC_MAX_TMINFO; i++) {
			pW->TmInfo[i].EvtCode = -1;
		}
	} while(0);

	return pW;
}

//==============================================================================================================//
//	インスタンス消去																							//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT VO		WINAPI AjcStcDelete(HAJCSTC pW)
{
	if (pW != NULL) {
		if (pW->hwnd != NULL) DestroyWindow(pW->hwnd);
		free(pW);
	}
}

//==============================================================================================================//
//	タイマ情報設定																								//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				tid 	- タイマＩＤ（０～）																	//
//				msTime	- タイマ値[ms]																			//
//				EvtCode - タイムアウト・イベントコード															//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT BOOL		WINAPI AjcStcSetTimerInfo	(HAJCSTC pW, UI tid, UI msTime, int EvtCode)
{
	BOOL	rc = FALSE;

	if (tid <AJCSTC_MAX_TMINFO) {
		pW->TmInfo[tid].msTime	= msTime;
		pW->TmInfo[tid].EvtCode = EvtCode;
		rc = TRUE;
	}

	return rc;
}
//==============================================================================================================//
//	タイマ・スタート																							//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				tid 	- タイマＩＤ（０～）																	//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT BOOL		WINAPI AjcStcTimerStart (HAJCSTC pW, UI tid)
{
	BOOL	rc = FALSE;

	if (tid < AJCSTC_MAX_TMINFO) {
		//	タイマ起動
		SetTimer(pW->hwnd, tid + 1, pW->TmInfo[tid].msTime, NULL);
		//	タイマ起動通知
		if (pW->cbNtc != NULL) {
			pW->cbNtc(AJCSTC_NTC_TIMSRT, tid, pW->TmInfo[tid].msTime, pW->cbp);
		}
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	タイマ・ストップ																							//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				tid 	- タイマＩＤ（０～）																	//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT BOOL		WINAPI AjcStcTimerStop (HAJCSTC pW, UI tid)
{
	BOOL	rc = FALSE;

	if (tid < AJCSTC_MAX_TMINFO) {
		//	タイマ停止
		KillTimer(pW->hwnd, tid + 1);
		//	タイマ停止通知
		if (pW->cbNtc != NULL) {
			pW->cbNtc(AJCSTC_NTC_TIMSTP, tid, 0, pW->cbp);
		}
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	現在の状態コード取得																						//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//																												//
//	戻り値	：	現在の状態コード																				//
//==============================================================================================================//
AJCEXPORT int		WINAPI AjcStcGetCurrentState(HAJCSTC pW)
{
	return pW->CurSts;
}
//==============================================================================================================//
//	現在の状態コード設定																						//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				sts 	- 状態コード																			//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT BOOL	WINAPI AjcStcSetCurrentState(HAJCSTC pW, int sts)
{
	BOOL	rc = FALSE;

	if (((UI)sts) <= AJCSTC_MAX_STS) {
		pW->CurSts = sts;
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	イベント実行																								//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				evt 	- イベントコード																		//
//																												//
//	戻り値	：	遷移先状態コード（状態遷移しない場合は変化なし）												//
//				-1	- Busy																						//
//				-2	- Error																						//
//==============================================================================================================//
AJCEXPORT int		WINAPI AjcStcExecEvent		(HAJCSTC pW, int evt)
{
	int		rc = -1;
	UI		nix;


	do {
		//----- イベントエンキュー（イベントを一旦保留する）----------------------------------------------------//
		nix = ((pW->eix + 1) % AJCSTC_MAX_EVT);
		if (nix != pW->six) {
			pW->EvtBuf[pW->eix] = evt;
			pW->eix = nix;
		}
		else {rc = -2; break;}
		//----- イベント実行（イベント実行中(プリセットファンクション内からのイベント実行)の場合は実行しない ---//
		if (!pW->fBusy) {
			//	イベント実行中の旨、フラグ設定
			pW->fBusy = TRUE;
			//	保留中のイベント実行
			while (pW->six != pW->eix) {
				if (pW->cbNtc != NULL) {
					pW->cbNtc(AJCSTC_NTC_EVT, pW->CurSts, pW->EvtBuf[pW->six], pW->cbp);
				}
				rc = SubExecEvent(pW, pW->pEvt + (pW->nState * pW->EvtBuf[pW->six]));
				if (pW->cbNtc != NULL  &&  rc != pW->CurSts) {
					pW->cbNtc(AJCSTC_NTC_STS, rc, 0, pW->cbp);
				}
				pW->CurSts = rc;
				pW->six = ((pW->six + 1) % AJCSTC_MAX_EVT);
			}
			//	イベント実行中フラグ解除
			pW->fBusy = FALSE;
		}
	} while(0);

	return rc;
}
//==============================================================================================================//
//	待機中の全イベント破棄																						//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT VO 		WINAPI AjcStcPurge			(HAJCSTC pW)
{
	pW->six = pW->eix = 0;
}
//--------------------------------------------------------------------------------------------------------------//
//	イベント実行																								//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				pEvt	- イベントアクションテーブルの当該イベントエントリ・アドレス							//
//																												//
//	戻り値	：	遷移先状態コード（状態遷移しない場合は変化なし）												//
//				-1	- Busy																						//
//				-2	- Error																						//
//--------------------------------------------------------------------------------------------------------------//
static int		SubExecEvent(HAJCSTC pW, PCAJCSTC_EVT_NF pEvt)
{
	int		NewSts = pW->CurSts;
	PCAJCSTC_STSACT	pCom = pW->pCom;
	PCAJCSTC_STSACT	pRel = pW->pRel;

	//	イベントアクションテーブルの当該状態エントリアドレス設定
	pEvt += pW->CurSts;
	//	遷移先状態番号設定
	NewSts = pEvt->sts;
	//	状態遷移無しならば、他の状態から当該状態へ遷移時のアクションテーブルを無視
	if (NewSts == AJCSTC_NOCHG) {
		pCom = NULL;
	}
	//	状態遷移無し／同一状態ならば、当該状態から他の状態へ遷移時のアクションテーブルを無視
	if (NewSts == AJCSTC_NOCHG	||	NewSts == pW->CurSts) {
		pRel = NULL;
	}
	//----------------------------------------------------------------------------------------------------------//
	//	タイマ停止																								//
	//----------------------------------------------------------------------------------------------------------//
	//	イベントアクションテーブルのタイマ停止制御
	SubTimStop(pW, pEvt->tmc);
	//	状態遷移時の自状態におけるタイマ停止制御
	if (pRel != NULL) {
		SubTimStop(pW, (pRel + pW->CurSts)->tmc);
	}
	//	状態遷移時の遷移先におけるタイマ停止制御
	if (pCom != NULL) {
		SubTimStop(pW, (pCom + NewSts)->tmc);
	}
	//----------------------------------------------------------------------------------------------------------//
	//	プリセットファンクション実行																			//
	//----------------------------------------------------------------------------------------------------------//
	//----- イベントアクションテーブルのプリセットファンクション実行 -------------------------------------------//
	SubPreFunc(pW, pEvt->pft);
	//	状態遷移時の自状態におけるプリセットファンクション実行	
	if (pRel != NULL) {
		SubPreFunc(pW, (pRel + pW->CurSts)->pft);
	}
	//	状態遷移時の遷移先におけるプリセットファンクション実行	
	if (pCom != NULL) {
		SubPreFunc(pW, (pCom + NewSts)->pft);
	}
	//----------------------------------------------------------------------------------------------------------//
	//	タイマ起動																								//
	//----------------------------------------------------------------------------------------------------------//
	//	イベントアクションテーブルのタイマ起動制御	
	SubTimStart(pW, pEvt->tmc);
	//	状態遷移時の自状態におけるタイマ起動制御	
	if (pRel != NULL) {
		SubTimStart(pW, (pRel + pW->CurSts)->tmc);
	}
	//	状態遷移時の遷移先におけるタイマ起動制御	
	if (pCom != NULL) {
		SubTimStart(pW, (pCom + NewSts)->tmc);
	}

	return(NewSts != AJCSTC_NOCHG ? NewSts : pW->CurSts);
}
//--------------------------------------------------------------------------------------------------------------//
//	プリセットファンクション実行																				//
//--------------------------------------------------------------------------------------------------------------//
static	VO	SubPreFunc(HAJCSTC pW, const UB pft[AJCSTC_MAX_PSF])
{
	UI		i;

	if (pW->cbPsf != NULL) {
		for (i=0; i<AJCSTC_MAX_PSF; i++) {
			if (pft[i] != AJCSTC_NOPSF) {
				pW->cbPsf(pft[i], pW->CurSts, pW->cbp);
			}
		}
	}
}

//--------------------------------------------------------------------------------------------------------------//
//	タイマ起動制御																								//
//--------------------------------------------------------------------------------------------------------------//
static	VO	SubTimStart(HAJCSTC pW, UW tmc)
{
	UI		tid;
	UW		tmd;

	if (tmc != 0) {
		for (tid = 0; tmc != 0; tid++, tmc <<= 2) {
			//	タイマアクションコード設定
			tmd = (UW)(tmc & AJCSTC_TIM_CTL);
			//	単発タイマ／サイクリックタイマ起動ならば・・
			if (tmd == AJCSTC_TIM_START || tmd == AJCSTC_TIM_CYC) {
				//	サイクリックフラグ設定
				if (tmd == AJCSTC_TIM_CYC) pW->fCyclic[tid] = TRUE;
				else					   pW->fCyclic[tid] = FALSE;
				//	タイマ起動
				SetTimer(pW->hwnd, tid + 1, pW->TmInfo[tid].msTime, NULL);
				//	タイマ起動通知
				if (pW->cbNtc != NULL) {
					pW->cbNtc(AJCSTC_NTC_TIMSRT, tid, pW->TmInfo[tid].msTime, pW->cbp);
				}
			}
		}
	}
}
//--------------------------------------------------------------------------------------------------------------//
//	タイマ停止制御																								//
//--------------------------------------------------------------------------------------------------------------//
static	VO	SubTimStop(HAJCSTC pW, UW tmc)
{
	UI		tid;

	if (tmc != 0) {
		for (tid = 0; tmc != 0; tid++, tmc <<= 2) {
			//	タイマ停止ならば・・
			if ((tmc & AJCSTC_TIM_CTL) == AJCSTC_TIM_STOP) {
				//	タイマ停止
				KillTimer(pW->hwnd, tid + 1);
				//	タイマ停止通知
				if (pW->cbNtc != NULL) {
					pW->cbNtc(AJCSTC_NTC_TIMSTP, tid, 0, pW->cbp);
				}
			}
		}
	}
}

//--------------------------------------------------------------------------------------------------------------//
//																												//
//	ウインドプロシージャ																						//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	LRESULT CALLBACK 	WndStcHideProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	HAJCSTC	pW;
	UI		tid;

	switch (msg) {
		//------------------------------------------------------------------------------------------------------//
		case WM_CREATE:
		{	LPCREATESTRUCT p = (LPCREATESTRUCT)lParam;
			pW = p->lpCreateParams;
			//	ウインドにインスタンスワークを関連付ける
			MAjcSetWindowLong(hwnd, 0, (UX)pW);
			return 0;
		}
		//------------------------------------------------------------------------------------------------------//
		case WM_TIMER:
			pW = (HAJCSTC)MAjcGetWindowLong(hwnd, 0);
			//	タイマＩＤ設定
			tid = (UI)(wParam - 1);
			//	単発タイマならば、タイマ停止
			if (!pW->fCyclic[tid]) {
				KillTimer(hwnd, wParam);
			}
			//	タイムアウトイベント実行
			if (pW->TmInfo[tid].EvtCode != -1) {
				AjcStcExecEvent(pW, pW->TmInfo[tid].EvtCode);
			}
			return 0;
		//------------------------------------------------------------------------------------------------------//
		case WM_DESTROY:
			return 0;
	 }
	 return AjcDefWindowProc(hwnd, msg, wParam, lParam) ;
}
