﻿#include	"AjcInternal.h"

//**************************************************************************************************************//
//																												//
//	コンソール入出力																							//
//																												//
//**************************************************************************************************************//
//----- 標準出力／標準エラー情報 -------------------------------------------------------------------------------//
typedef struct {
	BOOL		fCon;				//	コンソール出力を示すフラグ
	UL			mode;				//	コンソール出力時のモード
	EAJCTEC		tec;				//	インダイレクト出力時のテキストエンコード
	BOOL		fBom;				//		  〃			  ＢＯＭ出力フラグ
	HANDLE		hdl;				//		  〃			  ディスクファイルハンドル
	HAJCFILE	*phFile;			//		  〃			  テキストファイルアクセスハンドルへのポインタ
	//	定数
	FILE		*pstd;				//	定数（stdout			/ stderr			）
	UL			hstd;				//	定数（STD_OUTPUT_HANDLE / STD_ERROR_HANDLE	）
} AJCSTDINFO, *PAJCSTDINFO;

//	モード設定なし時は、通常のprintf/wprintf実行する為、fCon = TRUE とする
static	AJCSTDINFO	StdOutInfo = {TRUE};	//	標準出力情報
static	AJCSTDINFO	StdErrInfo = {TRUE};	//	標準エラー情報

//--------------------------------------------------------------------------------------------------------------//
//																												//
//	プロセス終了時のコールバック（標準出力インダイレクトファイルのファイルオブジェクトをデタッチ）				//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	VO CALLBACK cbStdCleanup(VOP pHandle)
{
	HAJCFILE	*phFile = (HAJCFILE*)pHandle;
	AjcFDetach(*phFile);
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	コンソール出力時のモード設定																				//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	VO	SetConMode(PAJCSTDINFO pW, UL mode)
{
	if (pW->pstd != NULL) {
		if (pW->mode != mode) _setmode(_fileno(pW->pstd), pW->mode = mode);
	}
}

//==============================================================================================================//
//	標準出力／標準エラー モード設定																				//
//																												//
//	引　数：	pSTr - 出力する文字列																			//
//																												//
//	戻り値：	TRUE ：ＯＫ																						//
//				FALSE：エラー																					//
//==============================================================================================================//
static	BOOL SubSetStdMode(PAJCSTDINFO pW, EAJCTEC tec, BOOL fBom)
{
	BOOL		rc	= FALSE;
	HANDLE		h	= NULL;
	UL			m	= 0;
	VOP			vop;
	union {
		ULL sz;
		struct {UL sl, sh;} s;
	} u;

	if (h = GetStdHandle(pW->hstd)) {
		//	コンソール出力時のモード設定（何もしない）
		if (pW->fCon = (GetConsoleMode(h, &m) != 0)) {
			rc = TRUE;
		}
		//	インダイレクト出力時の設定
		else {
			//	初回 or ファイルハンドル変化？
			if (h != pW->hdl) {
				if (vop = AjcAllocInstance(sizeof(HAJCFILE), cbStdCleanup)) {
					//	追記の場合は、ＢＯＭ出力取り消し
					u.s.sl = GetFileSize(h, &u.s.sh);
					if (u.sz != 0) {
						fBom = FALSE;
					}
					//	ファイルハンドル設定
					pW->hdl = h;
					//	テキストファイルアクセスのアタッチ
					pW->phFile = (HAJCFILE*)vop;
					if ((*pW->phFile = AjcFAttachCreated(h, tec, fBom)) == NULL) {
						AjcReleaseInstance(vop);
						pW->hdl    = NULL;
						pW->phFile = NULL;
					}
				}
			}
			//	戻り値設定
			if (pW->phFile != NULL && *pW->phFile != NULL) {
				rc = TRUE;
			}
		}
	}
	return rc;
}
//----- AjcSetStdoutModeEx -------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI	 AjcSetStdoutModeEx(EAJCTEC tec, BOOL fBom)
{
	BOOL	rc = FALSE;

	if (StdOutInfo.pstd == NULL) {
		//	標準出力／標準エラー情報初期化
		StdOutInfo.pstd	= stdout;				StdErrInfo.pstd = stderr;
		StdOutInfo.hstd	= STD_OUTPUT_HANDLE;	StdErrInfo.hstd	= STD_ERROR_HANDLE;
		StdOutInfo.mode	= -1;					StdErrInfo.mode	= -1;

		//	コンソールモード設定
		rc	= SubSetStdMode(&StdOutInfo, tec, fBom);
		rc &= SubSetStdMode(&StdErrInfo, tec, fBom);
	}
	return rc;
}
//----- AjcSetStdoutMode ---------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI	 AjcSetStdoutMode(VO)
{
	return AjcSetStdoutModeEx(AJCTEC_MBC, FALSE);
}
//----- AjcSetStdMode ------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI	 AjcSetStdMode(EAJCTEC tec, BOOL fBom)
{
	return AjcSetStdoutModeEx(tec, fBom);
}


//==============================================================================================================//
//	標準出力へ１文字出力																						//
//																												//
//	引　数：	c - 出力する文字																				//
//																												//
//	戻り値：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcPutCA(BC c)
{
	BOOL		rc = FALSE;

	if (StdOutInfo.fCon) {
		SetConMode(&StdOutInfo, _O_TEXT);
		rc = (putchar(c) != EOF);
	}
	else {
		rc = AjcFPutCA(*StdOutInfo.phFile, c);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcPutCW(WC c)
{
	BOOL		rc = FALSE;

	if (StdOutInfo.fCon) {
		SetConMode(&StdOutInfo, _O_U8TEXT);
		rc = (putwchar(c) != WEOF);
	}
	else if (StdOutInfo.phFile) {
		rc = AjcFPutCW(*StdOutInfo.phFile,c);
	}
	return rc;
}
//==============================================================================================================//
//	標準出力へ文字列出力																						//
//																												//
//	引　数：	pSTr - 出力する文字列																			//
//																												//
//	戻り値：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcPutSA(C_BCP pStr)
{
	BOOL		rc = FALSE;

	if (pStr != NULL) {
		if (StdOutInfo.fCon) {
			SetConMode(&StdOutInfo, _O_TEXT);
			rc = (printf("%s", pStr) >= 0);
		}
		else {
			rc = AjcFPutSA(*StdOutInfo.phFile, pStr, -1);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcPutSW(C_WCP pStr)
{
	BOOL		rc = FALSE;

	if (pStr != NULL) {
		if (StdOutInfo.fCon) {
			SetConMode(&StdOutInfo, _O_U8TEXT);
			rc = (wprintf(L"%s", pStr) >= 0);
		}
		else if (StdOutInfo.phFile) {
			rc = AjcFPutSW(*StdOutInfo.phFile, pStr, -1);
		}
	}
	return rc;
}
//==============================================================================================================//
//	標準出力へ書式文字列出力																					//
//																												//
//	引　数：	pFmt - 書式文字列																				//
//																												//
//	戻り値：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcPrintFA(C_BCP pFmt, ...)
{
	BOOL	rc = FALSE;
	va_list vls;

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		if (StdOutInfo.fCon) {
			SetConMode(&StdOutInfo, _O_TEXT);
			rc = (vprintf(pFmt, vls) >= 0);
		}
		else if (StdOutInfo.phFile) {
			rc = AjcFVPrintFA(*StdOutInfo.phFile, pFmt, vls);
		}
		va_end	(vls);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcPrintFW(C_WCP pFmt, ...)
{
	BOOL	rc = FALSE;
	va_list vls;

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		if (StdOutInfo.fCon) {
			SetConMode(&StdOutInfo, _O_U8TEXT);
			rc = (vwprintf(pFmt, vls) >= 0);
		}
		else if (StdOutInfo.phFile) {
			rc = AjcFVPrintFW(*StdOutInfo.phFile, pFmt, vls);
		}
		va_end	(vls);
	}
	return rc;
}
//==============================================================================================================//
//	標準エラーへ１文字出力																						//
//																												//
//	引　数：	c - 出力する文字																				//
//																												//
//	戻り値：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcErrPutCA(BC c)
{
	BOOL		rc = FALSE;

	if (StdErrInfo.fCon) {
		SetConMode(&StdErrInfo, _O_TEXT);
		rc = (fputc(c, stderr) != EOF);
	}
	else {
		rc = AjcFPutCA(*StdErrInfo.phFile, c);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcErrPutCW(WC c)
{
	BOOL		rc = FALSE;

	if (StdErrInfo.fCon) {
		SetConMode(&StdErrInfo, _O_U8TEXT);
		rc = (fputwc(c, stderr) != WEOF);
	}
	else if (StdErrInfo.phFile) {
		rc = AjcFPutCW(*StdErrInfo.phFile,c);
	}
	return rc;
}
//==============================================================================================================//
//	標準エラーへ文字列出力																						//
//																												//
//	引　数：	pSTr - 出力する文字列																			//
//																												//
//	戻り値：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcErrPutSA(C_BCP pStr)
{
	BOOL		rc = FALSE;

	if (pStr != NULL) {
		if (StdErrInfo.fCon) {
			SetConMode(&StdErrInfo, _O_TEXT);
			rc = (fprintf(stderr, "%s", pStr) >= 0);
		}
		else {
			rc = AjcFPutSA(*StdErrInfo.phFile, pStr, -1);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcErrPutSW(C_WCP pStr)
{
	BOOL		rc = FALSE;

	if (pStr != NULL) {
		if (StdErrInfo.fCon) {
			SetConMode(&StdErrInfo, _O_U8TEXT);
			rc = (fwprintf(stderr, L"%s", pStr) >= 0);
		}
		else if (StdErrInfo.phFile) {
			rc = AjcFPutSW(*StdErrInfo.phFile, pStr, -1);
		}
	}
	return rc;
}
//==============================================================================================================//
//	標準エラーへ書式文字列出力																					//
//																												//
//	引　数：	pFmt - 書式文字列																				//
//																												//
//	戻り値：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT int	 WINAPI AjcErrPrintFA(C_BCP pFmt, ...)
{
	BOOL	rc = FALSE;
	va_list vls;

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		if (StdErrInfo.fCon) {
			SetConMode(&StdErrInfo, _O_TEXT);
			rc = (vfprintf(stderr, pFmt, vls) >= 0);
		}
		else if (StdErrInfo.phFile) {
			rc = AjcFVPrintFA(*StdErrInfo.phFile, pFmt, vls);
		}
		va_end	(vls);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcErrPrintFW(C_WCP pFmt, ...)
{
	BOOL	rc = FALSE;
	va_list vls;

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		if (StdErrInfo.fCon) {
			SetConMode(&StdErrInfo, _O_U8TEXT);
			rc = (vfwprintf(stderr, pFmt, vls) >= 0);
		}
		else if (StdErrInfo.phFile) {
			rc = AjcFVPrintFW(*StdErrInfo.phFile, pFmt, vls);
		}
		va_end	(vls);
	}
	return rc;
}
//==============================================================================================================//
//	ｍａｉｎ関数の引数をワイド文字に変換したメモリ確保（ワイド文字用）											//
//																												//
//	引　数：	argc - 引数の個数																				//
//				argv - 引数のポインタ配列のアドレス																//
//																												//
//	戻り値：	TRUE：ＯＫ , FALSE:Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI AjcAllocMainArgsW(int argc, char **argv[])
{
	BOOL	rc = TRUE;
	WCP		*pWArgs;
	int		i, wl;

	if (argv != NULL) {
		if (pWArgs = (WCP*)AJCMEM(sizeof(VOP) * argc)) {
			memset(pWArgs, 0, sizeof(VOP) * argc);
			for (i=0; i<argc; i++) {
				wl = MultiByteToWideChar(CP_ACP, 0, (*argv)[i], -1, NULL, 0);
				if (!(pWArgs[i] = AjcTAlloc(wl))) {rc = FALSE; break;}
				MultiByteToWideChar(CP_ACP, 0, (*argv)[i], -1, pWArgs[i], wl);
			}
			if (rc == TRUE) {
				*argv = (VOP)pWArgs;
			}
			else {
				for (i=0; i<argc && pWArgs[i]!=NULL; i++) {
					free(pWArgs[i]);
				}
				free(pWArgs);
			}
		}
		else rc = FALSE;
	}
	else rc = FALSE;

	return rc;
}
//==============================================================================================================//
//	AjcAllocMainArgs()で確保したメモリの開放（ワイド文字用）													//
//																												//
//	引　数：	argc - 引数の個数																				//
//				argv - 引数のポインタ配列のアドレス																//
//																												//
//	戻り値：	なし																							//
//==============================================================================================================//
AJCEXPORT	VO	WINAPI AjcFreeMainArgsW(int argc, char *argv[])
{
	int		i;

	if (argv != NULL) {
		for (i=0; i<argc; i++) {
			free(argv[i]);
		}
		free(argv);
	}
}
//==============================================================================================================//
//	コンソール情報取得																							//
//																												//
//	引　数：	pBufSize	- バッファサイズを格納するバッファのアドレス（不要時はNULL）						//
//				pRcWnd		- ウインド矩形を格納するバッファのアドレス	（不要時はNULL）						//
//				pMaxWndSize - 最大ウインドサイズを格納するバッファのアドレス（不要時はNULL）					//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI AjcGetConsoleScreenBufferInfo(LPSIZE pBufSize, LPRECT pRcWnd, LPSIZE pMaxWndSize)
{
	BOOL	rc = FALSE;
	CONSOLE_SCREEN_BUFFER_INFO info;

	if (rc = (GetConsoleScreenBufferInfo(GetStdHandle(STD_OUTPUT_HANDLE), &info) != 0)) {
		if (pBufSize != NULL) {
			pBufSize->cx = info.dwSize.X;
			pBufSize->cy = info.dwSize.Y;
		}
		if (pRcWnd != NULL) {
			pRcWnd->left   = info.srWindow.Left;
			pRcWnd->right  = info.srWindow.Right;
			pRcWnd->top	  = info.srWindow.Top;
			pRcWnd->bottom = info.srWindow.Bottom;
		}
		if (pMaxWndSize != NULL) {
			pMaxWndSize->cx = info.dwMaximumWindowSize.X;
			pMaxWndSize->cy = info.dwMaximumWindowSize.Y;
		}
	}
	return rc;
}
//==============================================================================================================//
//	コンソールウインド最大サイズ取得																			//
//																												//
//	引　数：	pCx - 横サイズ（文字数）を格納するバッファのアドレス（不要時はNULL）							//
//				pCy - 縦サイズ（文字数）を格納するバッファのアドレス（不要時はNULL）							//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI AjcGetConsoleMaxWndSize(int *pCx, int *pCy)
{
	BOOL	rc = FALSE;
	CONSOLE_SCREEN_BUFFER_INFO info;

	if (rc = (GetConsoleScreenBufferInfo(GetStdHandle(STD_OUTPUT_HANDLE), &info) != 0)) {
		if (pCx != NULL) *pCx = info.dwMaximumWindowSize.X;
		if (pCy != NULL) *pCy = info.dwMaximumWindowSize.Y;
	}
	return rc;
}
//==============================================================================================================//
//	コンソールバッファサイズ設定																				//
//																												//
//	引　数：	cx - 横サイズ（文字数）																			//
//				cy - 縦サイズ（文字数）																			//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI AjcSetConsoleBufSize(int cx, int cy)
{
	BOOL	rc = FALSE;
	COORD	cd;

	cd.X = cx;
	cd.Y = cy;
	rc = SetConsoleScreenBufferSize(GetStdHandle(STD_OUTPUT_HANDLE), cd);

	return rc;
}
//==============================================================================================================//
//	コンソールバッファサイズ取得																				//
//																												//
//	引　数：	pCx - 横文字数を格納するバッファのアドレス（不要時はNULL）										//
//				pCy - 縦文字数を格納するバッファのアドレス（不要時はNULL）										//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI AjcGetConsoleBufSize(int *pCx, int *pCy)
{
	BOOL	rc = FALSE;
	SIZE	BufSize;

	if (rc = AjcGetConsoleScreenBufferInfo(&BufSize, NULL, NULL)) {
		if (pCx != NULL) *pCx = BufSize.cx;
		if (pCy != NULL) *pCy = BufSize.cy;
	}
	return rc;
}
//==============================================================================================================//
//	コンソールスクリーン矩形設定																				//
//																												//
//	引　数：	left   - 左端文字位置																			//
//				top    - 上端文字位置																			//
//				right  - 右端文字位置																			//
//				bottom - 下端文字位置																			//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI AjcSetConsoleWndRect(int left, int top, int right, int bottom)
{
	BOOL		rc = FALSE;
	SMALL_RECT	rect;

	rect.Left	= left;
	rect.Right	= right;
	rect.Top	= top;
	rect.Bottom	= bottom;
	rc = SetConsoleWindowInfo(GetStdHandle(STD_OUTPUT_HANDLE), TRUE, &rect);

	return rc;
}
//==============================================================================================================//
//	コンソールスクリーン矩形取得																				//
//																												//
//	引　数：	pLeft	- 左端文字位置を格納するバッファのアドレス(不要時はNULL）								//
//				pTop	- 上端文字位置を格納するバッファのアドレス(不要時はNULL）								//
//				pRight	- 右端文字位置を格納するバッファのアドレス(不要時はNULL）								//
//				pBottom - 下端文字位置を格納するバッファのアドレス(不要時はNULL）								//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI AjcGetConsoleWndRect(int *pLeft, int *pTop, int *pRight, int *pBottom)
{
	BOOL	rc = FALSE;
	RECT	rect;

	if (rc = AjcGetConsoleScreenBufferInfo(NULL, &rect, NULL)) {
		if (pLeft	!= NULL) *pLeft	  = rect.left;
		if (pRight	!= NULL) *pRight  = rect.right;
		if (pTop	!= NULL) *pTop	  = rect.top;
		if (pBottom != NULL) *pBottom = rect.bottom;
	}
	return rc;
}
//==============================================================================================================//
//	コンソール表示色設定（フルカラー）																			//
//																												//
//	引　数：	ForeColor	- 前景色（－１：設定しない）														//
//				BackColor	- 背景色（－１：設定しない）														//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI AjcSetConsoleColor(COLORREF ForeColor, COLORREF BackColor)
{
	BOOL	rc	 = FALSE;
	HANDLE	hStd = GetStdHandle(STD_OUTPUT_HANDLE);
	UI		fix = 0, bix = 0;
	UW		att;
	CONSOLE_SCREEN_BUFFER_INFOEX info;

	if (ForeColor != -1 || BackColor != -1) {
		info.cbSize = sizeof(info);
		if (rc = (GetConsoleScreenBufferInfoEx(hStd, &info) != 0)) {
			if (ForeColor != -1) {
				if (info.wAttributes & FOREGROUND_INTENSITY) fix |= 0x08;
				if (info.wAttributes & FOREGROUND_RED	   ) fix |= 0x04;
				if (info.wAttributes & FOREGROUND_GREEN    ) fix |= 0x02;
				if (info.wAttributes & FOREGROUND_BLUE	   ) fix |= 0x01;
				info.ColorTable[fix] = ForeColor;
			}
			if (BackColor != -1) {
				if (info.wAttributes & BACKGROUND_INTENSITY) bix |= 0x08;
				if (info.wAttributes & BACKGROUND_RED	   ) bix |= 0x04;
				if (info.wAttributes & BACKGROUND_GREEN    ) bix |= 0x02;
				if (info.wAttributes & BACKGROUND_BLUE	   ) bix |= 0x01;
				info.ColorTable[bix] = BackColor;
			}
			info.srWindow.Bottom++;	// 設定する場合Bottomを+1する（WindowsAPIバグ）
			if (rc = (SetConsoleScreenBufferInfoEx(hStd, &info) != 0)) {
				att = 0;
				if (fix & 0x08) att |= FOREGROUND_INTENSITY;
				if (fix & 0x04) att |= FOREGROUND_RED;
				if (fix & 0x02) att |= FOREGROUND_GREEN;
				if (fix & 0x01) att |= FOREGROUND_BLUE;
				if (bix & 0x08) att |= BACKGROUND_INTENSITY;
				if (bix & 0x04) att |= BACKGROUND_RED;
				if (bix & 0x02) att |= BACKGROUND_GREEN;
				if (bix & 0x01) att |= BACKGROUND_BLUE;
				SetConsoleTextAttribute(hStd, att);
			}
		}
	}
	else {
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	コンソール表示色取得（フルカラー）																			//
//																												//
//	引　数：	pForeColor	- 前景色を格納するバッファのアドレス（不要時はNULL）								//
//				pBackColor	- 背景色を格納するバッファのアドレス（不要時はNULL）								//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
//	前景色と背景色
AJCEXPORT	BOOL	WINAPI AjcGetConsoleColor(LPCOLORREF pForeColor, LPCOLORREF pBackColor)
{
	BOOL	rc = FALSE;
	CONSOLE_SCREEN_BUFFER_INFOEX info;

	info.cbSize = sizeof(info);
	if (rc = (GetConsoleScreenBufferInfoEx(GetStdHandle(STD_OUTPUT_HANDLE), &info) != 0)) {
		if (pForeColor != NULL) {
			UI	fix = 0;
			if (info.wAttributes & FOREGROUND_INTENSITY) fix |= 0x08;
			if (info.wAttributes & FOREGROUND_RED	   ) fix |= 0x04;
			if (info.wAttributes & FOREGROUND_GREEN    ) fix |= 0x02;
			if (info.wAttributes & FOREGROUND_BLUE	   ) fix |= 0x01;
			*pForeColor = info.ColorTable[fix];
		}
		if (pBackColor != NULL) {
			UI	bix = 0;
			if (info.wAttributes & BACKGROUND_INTENSITY) bix |= 0x08;
			if (info.wAttributes & BACKGROUND_RED	   ) bix |= 0x04;
			if (info.wAttributes & BACKGROUND_GREEN    ) bix |= 0x02;
			if (info.wAttributes & BACKGROUND_BLUE	   ) bix |= 0x01;
			*pBackColor = info.ColorTable[bix];
		}
	}
	return rc;
}
//	前景色
AJCEXPORT	COLORREF	WINAPI AjcGetConsoleForeColor(VO)
{
	COLORREF	rc = -1;
	CONSOLE_SCREEN_BUFFER_INFOEX info;

	info.cbSize = sizeof(info);
	if (rc = (GetConsoleScreenBufferInfoEx(GetStdHandle(STD_OUTPUT_HANDLE), &info) != 0)) {
		UI	fix = 0;
		if (info.wAttributes & FOREGROUND_INTENSITY) fix |= 0x08;
		if (info.wAttributes & FOREGROUND_RED	   ) fix |= 0x04;
		if (info.wAttributes & FOREGROUND_GREEN    ) fix |= 0x02;
		if (info.wAttributes & FOREGROUND_BLUE	   ) fix |= 0x01;
		rc = info.ColorTable[fix];
	}
	return rc;
}
//	背景色
AJCEXPORT	COLORREF	WINAPI AjcGetConsoleBackColor(VO)
{
	COLORREF	rc = -1;
	CONSOLE_SCREEN_BUFFER_INFOEX info;

	info.cbSize = sizeof(info);
	if (rc = (GetConsoleScreenBufferInfoEx(GetStdHandle(STD_OUTPUT_HANDLE), &info) != 0)) {
		UI	bix = 0;
		if (info.wAttributes & BACKGROUND_INTENSITY) bix |= 0x08;
		if (info.wAttributes & BACKGROUND_RED	   ) bix |= 0x04;
		if (info.wAttributes & BACKGROUND_GREEN    ) bix |= 0x02;
		if (info.wAttributes & BACKGROUND_BLUE	   ) bix |= 0x01;
		rc = info.ColorTable[bix];
	}
	return rc;
}
//==============================================================================================================//
//	コンソール表示色設定（１６色）																				//
//																												//
//	引　数：	ForeColor	- 前景色（－１：設定しない）														//
//				BackColor	- 背景色（－１：設定しない）														//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI AjcSetConsole16Color(UI ForeColor, UI BackColor)
{
	BOOL	rc	 = FALSE;
	HANDLE	hStd = GetStdHandle(STD_OUTPUT_HANDLE);
	UI		fix = 0, bix = 0;
	UW		att;

	if (ForeColor != -1 || BackColor != -1) {
		if (ForeColor != -1) {
			fix = (ForeColor & 0x0F);
		}
		if (BackColor != -1) {
			bix = (BackColor & 0x0F);
		}
		att = 0;
		if (fix & 0x08) att |= FOREGROUND_INTENSITY;
		if (fix & 0x04) att |= FOREGROUND_RED;
		if (fix & 0x02) att |= FOREGROUND_GREEN;
		if (fix & 0x01) att |= FOREGROUND_BLUE;
		if (bix & 0x08) att |= BACKGROUND_INTENSITY;
		if (bix & 0x04) att |= BACKGROUND_RED;
		if (bix & 0x02) att |= BACKGROUND_GREEN;
		if (bix & 0x01) att |= BACKGROUND_BLUE;

		if (SetConsoleTextAttribute(hStd, att)) {
			rc = TRUE;
		}
	}
	else {
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	コンソール表示色取得（１６色）																				//
//																												//
//	引　数：	pForeColor	- 前景色を格納するバッファのアドレス（不要時はNULL）								//
//				pBackColor	- 背景色を格納するバッファのアドレス（不要時はNULL）								//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI AjcGetConsole16Color(UIP pForeColor, UIP pBackColor)
{
	BOOL	rc = FALSE;
	CONSOLE_SCREEN_BUFFER_INFOEX info;

	if (GetConsoleScreenBufferInfoEx(GetStdHandle(STD_OUTPUT_HANDLE), &info)) {
		if (pForeColor != NULL) {
			*pForeColor = 0;
			if (info.wAttributes & FOREGROUND_INTENSITY) *pForeColor |= 0x08;
			if (info.wAttributes & FOREGROUND_RED	   ) *pForeColor |= 0x04;
			if (info.wAttributes & FOREGROUND_GREEN    ) *pForeColor |= 0x02;
			if (info.wAttributes & FOREGROUND_BLUE	   ) *pForeColor |= 0x01;
		}
		if (pBackColor != NULL) {
			*pBackColor = 0;
			if (info.wAttributes & BACKGROUND_INTENSITY) *pBackColor |= 0x08;
			if (info.wAttributes & BACKGROUND_RED	   ) *pBackColor |= 0x04;
			if (info.wAttributes & BACKGROUND_GREEN    ) *pBackColor |= 0x02;
			if (info.wAttributes & BACKGROUND_BLUE	   ) *pBackColor |= 0x01;
		}
		rc = TRUE;
	}
	return rc;
}
//	前景色
AJCEXPORT	UI	WINAPI AjcGetConsoleFore16Color(VO)
{
	UI		rc = -1;
	CONSOLE_SCREEN_BUFFER_INFOEX info;

	info.cbSize = sizeof(info);
	if (rc = (GetConsoleScreenBufferInfoEx(GetStdHandle(STD_OUTPUT_HANDLE), &info) != 0)) {
		UI	fix = 0;
		if (info.wAttributes & FOREGROUND_INTENSITY) fix |= 0x08;
		if (info.wAttributes & FOREGROUND_RED	   ) fix |= 0x04;
		if (info.wAttributes & FOREGROUND_GREEN    ) fix |= 0x02;
		if (info.wAttributes & FOREGROUND_BLUE	   ) fix |= 0x01;
		rc = fix;
	}
	return rc;
}
//	背景色
AJCEXPORT	UI	WINAPI AjcGetConsoleBack16Color(VO)
{
	UI		rc = -1;
	CONSOLE_SCREEN_BUFFER_INFOEX info;

	info.cbSize = sizeof(info);
	if (rc = (GetConsoleScreenBufferInfoEx(GetStdHandle(STD_OUTPUT_HANDLE), &info) != 0)) {
		UI	bix = 0;
		if (info.wAttributes & BACKGROUND_INTENSITY) bix |= 0x08;
		if (info.wAttributes & BACKGROUND_RED	   ) bix |= 0x04;
		if (info.wAttributes & BACKGROUND_GREEN    ) bix |= 0x02;
		if (info.wAttributes & BACKGROUND_BLUE	   ) bix |= 0x01;
		rc = bix;
	}
	return rc;
}

//==============================================================================================================//
//	コンソール表示パレット選択																					//
//																												//
//	引　数：	ForePalette	- 前景色のパレット番号（０～１５：パレット番号，－１：設定しない）					//
//				BackPalette	- 背景色のパレット番号（０～１５：パレット番号，－１：設定しない）					//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI AjcSelConsolePalette(UI ForePalette, UI BackPalette)
{
	return AjcSetConsole16Color(ForePalette, BackPalette);
}
//==============================================================================================================//
//	コンソール表示パレット設定																					//
//																												//
//	引　数：	Palette	- 設定するパレット情報のアドレス														//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI AjcSetConsolePalette(const COLORREF Palette[16])
{
	BOOL	rc = FALSE;
	HANDLE	hStd = GetStdHandle(STD_OUTPUT_HANDLE);
	CONSOLE_SCREEN_BUFFER_INFOEX info;

	if (Palette != NULL) {
		info.cbSize = sizeof(info);
		if (GetConsoleScreenBufferInfoEx(hStd, &info) != 0) {
			//	パレット情報コピー
			memcpy(info.ColorTable, Palette, sizeof info.ColorTable);
			//	コンソールバッファ情報設定
			rc = (SetConsoleScreenBufferInfoEx(hStd, &info) != 0);
		}
	}
	return rc;
}

//==============================================================================================================//
//	コンソール表示パレット設定（パレット番号指定）																//
//																												//
//	引　数：	ix			- 設定するパレットの番号（０～１５）												//
//				color		- パレットに背ってする色															//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI AjcSetConsolePalByIx(UI ix, COLORREF color)
{
	BOOL	rc = FALSE;
	HANDLE	hStd = GetStdHandle(STD_OUTPUT_HANDLE);
	CONSOLE_SCREEN_BUFFER_INFOEX info;

	if (ix < 16) {
		info.cbSize = sizeof(info);
		if (GetConsoleScreenBufferInfoEx(hStd, &info) != 0) {
			//	指定パレット色設定
			info.ColorTable[ix] = color;
			//	コンソールバッファ情報設定
			rc = (SetConsoleScreenBufferInfoEx(hStd, &info) != 0);
		}
	}
	return rc;
}

//==============================================================================================================//
//	コンソール表示パレット取得																					//
//																												//
//	引　数：	Palette	- パレット色情報を格納するバッファのアドレス（不要時はNULL）							//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	UI		WINAPI AjcGetConsolePalette(COLORREF Palette[16])
{
	UI		rc = -1;
	UI		fix, bix;
	HANDLE	hStd = GetStdHandle(STD_OUTPUT_HANDLE);
	CONSOLE_SCREEN_BUFFER_INFOEX info;

	info.cbSize = sizeof(info);
	if (GetConsoleScreenBufferInfoEx(hStd, &info) != 0) {
		//	前景パレット番号設定
		fix = 0;
		if (info.wAttributes & FOREGROUND_INTENSITY) fix |= 0x08;
		if (info.wAttributes & FOREGROUND_RED	   ) fix |= 0x04;
		if (info.wAttributes & FOREGROUND_GREEN    ) fix |= 0x02;
		if (info.wAttributes & FOREGROUND_BLUE	   ) fix |= 0x01;
		//	背景パレット番号設定
		bix = 0;
		if (info.wAttributes & BACKGROUND_INTENSITY) bix |= 0x08;
		if (info.wAttributes & BACKGROUND_RED	   ) bix |= 0x04;
		if (info.wAttributes & BACKGROUND_GREEN    ) bix |= 0x02;
		if (info.wAttributes & BACKGROUND_BLUE	   ) bix |= 0x01;
		//	パレット番号合成（戻り値設定）
		rc = MAKELONG(fix, bix);
		//	パレット情報コピー
		if (Palette != NULL) {
			memcpy(Palette, info.ColorTable, sizeof info.ColorTable);
		}
	}
	return rc;
}
//==============================================================================================================//
//	コンソール表示パレット取得（パレット番号指定）																//
//																												//
//	引　数：	Palette	- パレット色情報を格納するバッファのアドレス（不要時はNULL）							//
//																												//
//	戻り値：	≠-1  - OK																						//
//				＝-1  - エラー																					//
//==============================================================================================================//
AJCEXPORT	COLORREF	WINAPI AjcGetConsolePalByIx(UI ix)
{
	COLORREF	rc = -1;
	HANDLE		hStd = GetStdHandle(STD_OUTPUT_HANDLE);
	CONSOLE_SCREEN_BUFFER_INFOEX info;

	if (ix < 16) {
		info.cbSize = sizeof(info);
		if (GetConsoleScreenBufferInfoEx(hStd, &info) != 0) {
			//	パレット情報設定
			rc = info.ColorTable[ix];
		}
	}
	return rc;
}
//==============================================================================================================//
//	コンソールのカーソル位置設定																				//
//																												//
//	引　数：	x	- 行位置（０～）																			//
//				y	- 桁位置（０～）																			//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI AjcSetConsoleCursor(int x, int y)
{
	BOOL	rc = FALSE;
	HANDLE	hStd = GetStdHandle(STD_OUTPUT_HANDLE);
	CONSOLE_SCREEN_BUFFER_INFOEX info;
	COORD	pos;

	if (x >= 0 && y >= 0) {
		info.cbSize = sizeof(info);
		if (GetConsoleScreenBufferInfoEx(hStd, &info) != 0) {
			if (x < info.srWindow.Right && y < info.srWindow.Bottom) {
				pos.X = x;
				pos.Y = y;
				rc = (SetConsoleCursorPosition(hStd, pos) != 0);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	コンソールのカーソル位置取得																				//
//																												//
//	引　数：	pX	- 行位置（０～）を格納するバッファのアドレス（不要時はNULL)									//
//				pY	- 桁位置（０～）を格納するバッファのアドレス（不要時はNULL)									//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI AjcGetConsoleCursor(int *pX, int *pY)
{
	BOOL	rc = FALSE;
	HANDLE	hStd = GetStdHandle(STD_OUTPUT_HANDLE);
	CONSOLE_SCREEN_BUFFER_INFOEX info;

	info.cbSize = sizeof(info);
	if (GetConsoleScreenBufferInfoEx(hStd, &info) != 0) {
		if (pX != NULL) *pX = info.dwCursorPosition.X;
		if (pY != NULL) *pY = info.dwCursorPosition.Y;
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	コマンドを実行し出力をファイルにリダイレクトする															//
//																												//
//	引　数：	pCmdLine	- コマンドラインも文字列															//
//				pOutFile	- インダイレクト出力するファイルのパス名（NULLの場合は、stdout)						//
//				msTime		- コマンド実行終了待ち時間 (INIFINIT:永久）											//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI AjcSystemA(C_BCP pCmdLine, C_BCP pOutFile, UI msTime)
{
	BOOL		rc = FALSE;
	UI			len;
	WCP			pCmdLineW	= NULL;
	WCP			pOutFileW	= NULL;

	if (pCmdLine != NULL) {
		len = MultiByteToWideChar(CP_ACP, 0, pCmdLine, -1, NULL, 0);
		if ((pCmdLineW = AjcTAllocW(len)) != NULL) {
			MultiByteToWideChar(CP_ACP, 0, pCmdLine, -1, pCmdLineW, len);
			if (pOutFile != NULL) {
				len = MultiByteToWideChar(CP_ACP, 0, pOutFile, -1, NULL, 0);
				if ((pOutFileW = AjcTAllocW(len)) != NULL) {
					MultiByteToWideChar(CP_ACP, 0, pOutFile, -1, pOutFileW, len);
					rc = AjcSystemW(pCmdLineW, pOutFileW, msTime);
				}
			}
			else {
				rc =  AjcSystemW(pCmdLineW, NULL, msTime);
			}
		}
	}
	if (pCmdLineW != NULL) AjcTFree(pCmdLineW);
	if (pOutFileW != NULL) AjcTFree(pOutFileW);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI AjcSystemW(C_WCP pCmdLine, C_WCP pOutFile, UI msTime)
{
	BOOL				rc	  = FALSE;
	HANDLE				hFile = INVALID_HANDLE_VALUE;
	HANDLE				hOut  = stdout;
	SECURITY_ATTRIBUTES	sa	  = {0};
	STARTUPINFO			si	  = {0};
	PROCESS_INFORMATION	pi	  = {0};
	UI					stl;
	WCP					pCmd  = NULL;

	do {
		if (pCmdLine == NULL) break;
		//	コマンドラインを動的メモリへコピー
		stl = (UI)wcslen(pCmdLine);
		if ((pCmd = AjcTAllocW(stl + 32)) == NULL) break;
		AjcSnPrintF(pCmd, stl + 32, L"Cmd.exe /c %s", pCmdLine);
		//	リダイレクトファイルハンドル生成
		if (pOutFile != NULL) {
			sa.nLength				= sizeof(sa);
			sa.lpSecurityDescriptor = NULL;
			sa.bInheritHandle		= TRUE;	//TRUEでハンドルを引き継ぐ
			hFile = CreateFile(pOutFile, GENERIC_READ | GENERIC_WRITE, FILE_SHARE_READ | FILE_SHARE_WRITE,&sa, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
			if (hFile == INVALID_HANDLE_VALUE) break;
			hOut = hFile;
		}
		//	標準入出力の指定
		si.cb			= sizeof(si);
		si.dwFlags		= STARTF_USESTDHANDLES;
		si.hStdInput	= stdin;
		si.hStdOutput	= hOut;
		si.hStdError	= hOut;

		//	別のプロセスで cmd.exe 実行
		if (!CreateProcess( NULL, pCmd, NULL, NULL, TRUE, CREATE_NO_WINDOW, NULL, NULL, &si, &pi)) break;
		//	起動したプロセスの終了を待つ
		WaitForSingleObject( pi.hProcess, msTime);
		//	プロセスのハンドルを閉じる
		CloseHandle(pi.hProcess);
		CloseHandle(pi.hThread);

		rc = TRUE;

	} while(0);

	//	ファイルハンドルを閉じる
	if (hFile != INVALID_HANDLE_VALUE) CloseHandle(hFile);

	//	コマンドメモリ解放
	if (pCmd != NULL) AjcMFree(pCmd);

	return rc;
}
