﻿#include	"AjcInternal.h"

//**************************************************************************************************************//
//																												//
//	文字列操作																									//
//																												//
//**************************************************************************************************************//

//--------------------------------------------------------------------------------------------------------------//
//	Ｓ－ＪＩＳ，ＥＵＣ文字コード範囲																			//
//--------------------------------------------------------------------------------------------------------------//
//----- ＳＪＩＳ １バイト目のコード範囲 ------------------------------------------------------------------------//
#define	SJH1_L		0x81
#define	SJH1_H		0x9F
#define	SJH2_L		0xE0
#define	SJH2_H		0xFC

#define	SJH1_NUM	(SJH1_H - SJH1_L + 1)
#define	SJH2_NUM	(SJH2_H - SJH2_L + 1)
#define	SJH_NUM		(SJH1_NUM + SJH2_NUM)

//----- ＳＪＩＳ ２バイト目のコード範囲 ------------------------------------------------------------------------//
#define	SJL1_L		0x40
#define	SJL1_H		0x7E
#define	SJL2_L		0x80
#define	SJL2_H		0xFC

#define	SJL1_NUM	(SJL1_H - SJL1_L + 1)
#define	SJL2_NUM	(SJL2_H - SJL2_L + 1)
#define	SJL_NUM		(SJL1_NUM + SJL2_NUM)

//----- ＥＵＣのコード範囲 -------------------------------------------------------------------------------------//
#define	EUC_3BYTE	0x8F
#define	EUC_KANA	0x8E
#define	EUC_L		0xA1
#define	EUC_H		0xFE

#define	EUC_NUM		(EUC_H - EUC_L + 1)

//----- 半角カタカナのコード範囲 -------------------------------------------------------------------------------//
#define	KANA_L		0xA1
#define	KANA_H		0xDF

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
//----- 対応する囲み文字取得 -----------------------------------------------------------------------------------//
static	BC	GetEncloseCharA(UB cl)
{
	BC	cr;

	switch (cl) {
		case '<':	cr = '>'; break;
		case '(':	cr = ')'; break;
		case '[':	cr = ']'; break;
		case '{':	cr = '}'; break;
		default:	cr = cl;  break;
	}
	return cr;
}
//--------------------------------------------------------------------------------------------------------------//
static	UW	GetEncloseCharW(UW cl)
{
	UW	cr;

	switch (cl) {
		case L'<':	cr = L'>';	break;
		case L'(':	cr = L')';	break;
		case L'[':	cr = L']';	break;
		case L'{':	cr = L'}';	break;
		case L'＜':	cr = L'＞';	break;
		case L'（':	cr = L'）';	break;
		case L'［':	cr = L'］';	break;
		case L'｛':	cr = L'｝';	break;
		case L'「':	cr = L'」';	break;
		case L'〔':	cr = L'〕';	break;
		case L'《':	cr = L'》';	break;
		case L'『':	cr = L'』';	break;
		case L'【':	cr = L'】';	break;
		default:	cr = cl;	break;
	}
	return cr;
}
//----- 囲み文字を詰める（バイト文字） -------------------------------------------------------------------------//
static	VO	SubPackEncloseA(BCP pBuf, C_BCP pEnc)
{
	if (pEnc != NULL) {
		BCP	p = pBuf;
		BCP	q;
		while (*p != 0) {
			//	囲み文字除去
			if (strchr(pEnc, *p) != NULL) {
				q = AjcStrStripEncloseA(p);
				p += strlen(p);
				if (p != q) {
					memmove(p, q, (strlen(p) + 1) * sizeof(BC));
				}
			}
			else p++;
		}
	}
}
//----- 囲み文字を詰める（ワイド文字） -------------------------------------------------------------------------//
static	VO	SubPackEncloseW(WCP pBuf, C_WCP pEnc)
{
	if (pEnc != NULL) {
		WCP	p = pBuf;
		WCP	q;
		while (*p != 0) {
			//	囲み文字除去
			if (wcschr(pEnc, *p) != NULL) {
				q = AjcStrStripEncloseW(p);
				p += wcslen(p);
				if (p != q) {
					memmove(p, q, (wcslen(p) + 1) * sizeof(WC));
				}
			}
			else p++;
		}
	}
}
//==============================================================================================================//
//	Function	:	文字列比較方法を指定して文字列を比較する（単純バイト比較）									//
//																												//
//	Argument	:	s1 - 文字列１のアドレス																		//
//					s2 - 文字列２のアドレス（検索する部分文字列）												//
//					CmpMode - 文字列比較方法																	//
//																												//
//	Return		:	< 0 - 比較索文字列１ < 比較索文字列２														//
//					= 0 - 比較索文字列１ = 比較索文字列２														//
//					> 0 - 比較索文字列１ > 比較索文字列２														//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	int		WINAPI	AjcStrCmpExA(C_BCP pStr1, C_BCP pStr2, EAJCCMPMODE CmpMode)
{
	int		rc = 0;

	switch (CmpMode) {
		default: // AJCCMP_EXACTE_WIDTH
			rc = mbscmp(pStr1, pStr2);
			break;

		case AJCCMP_IGNORE_WIDTH:
			rc = mbsicmp(pStr1, pStr2);
			break;

		case AJCCMP_MIX:
			if ((rc = mbsicmp(pStr1, pStr2)) == 0) {
				rc = mbscmp(pStr1, pStr2);
			}
			break;
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	int		WINAPI	AjcStrCmpExW(C_WCP pStr1, C_WCP pStr2, EAJCCMPMODE CmpMode)
{
	int		rc = 0;

	switch (CmpMode) {
		default: // AJCCMP_EXACT_WIDTH
			rc = wcscmp(pStr1, pStr2);
			break;

		case AJCCMP_IGNORE_WIDTH:
			rc = wcsicmp(pStr1, pStr2);
			break;

		case AJCCMP_MIX:
			if ((rc = wcsicmp(pStr1, pStr2)) == 0) {
				rc = wcscmp(pStr1, pStr2);
			}
			break;
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	文字列比較方法を指定して部分文字列を見つける（単純バイト比較）								//
//																												//
//	Argument	:	s1 - 文字列１のアドレス																		//
//					s2 - 文字列２のアドレス（検索する部分文字列）												//
//					fIgnoreWidth - 文字列比較方法(FALSE:英大小を区別する，TRUE:	英大小を区別しない）			//
//																												//
//	Return		:	≠ＮＵＬＬ：見つかった文字列のアドレス														//
//					＝ＮＵＬＬ：指定された部分文字列は見つからない												//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcStrStrExA(C_BCP s1, C_BCP s2, BOOL fIgnoreWidth)
{
	BCP		rc = NULL;

	if (fIgnoreWidth) rc = AjcStrIStrA(s1, s2);
	else			  rc = (BCP)strstr(s1, s2);

	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcStrStrExW(C_WCP s1, C_WCP s2, BOOL fIgnoreWidth)
{
	WCP		rc = NULL;

	if (fIgnoreWidth) rc = AjcStrIStrW(s1, s2);
	else			  rc = (WCP)wcsstr(s1, s2);

	return rc;
}
//==============================================================================================================//
//	Function	:	英大小文字を区別せず部分文字列を見つける													//
//																												//
//	Argument	:	s1 - 文字列１のアドレス																		//
//					s2 - 文字列２のアドレス（検索する部分文字列）												//
//																												//
//	Return		:	≠ＮＵＬＬ：見つかった文字列のアドレス														//
//					＝ＮＵＬＬ：指定された部分文字列は見つからない												//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcStrIStrA(C_BCP s1, C_BCP s2)
{
	BCP		rc = NULL;
	C_BCP	p  = s1;
	int		l1;
	int		l2;
	int		n;

	if (s1 != NULL && s2 != NULL) {
		l1 = (int)strlen(s1);
		l2 = (int)strlen(s2);
		if		(l2 == 0) {
			rc = (BCP)s1;
		}
		else if (l1 >= l2) {
			n = l1 - l2;
			while (n >= 0) {
				if (strnicmp(p, s2, l2) == 0) {
					rc = (BCP)p;
					break;
				}
				p++;
				n--;
			}
		}
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcStrIStrW(C_WCP s1, C_WCP s2)
{
	WCP		rc = NULL;
	C_WCP	p = s1;
	int		l1 = (int)wcslen(s1);
	int		l2 = (int)wcslen(s2);
	int		n;

	if (s1 != NULL && s2 != NULL) {
		if		(l2 == 0) {
			rc = (WCP)s1;
		}
		else if (l1 >= l2) {
			n = l1 - l2;
			while (n >= 0) {
				if (wcsnicmp(p, s2, l2) == 0) {
					rc = (WCP)p;
					break;
				}
				p++;
				n--;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	文字列比較方法を指定して部分文字列を見つける												//
//																												//
//	Argument	:	s1 - 文字列１のアドレス																		//
//					s2 - 文字列２のアドレス（検索する部分文字列）												//
//					fIgnoreWidth - 文字列比較方法(FALSE:英大小を区別する，TRUE:	英大小を区別しない）			//
//																												//
//	Return		:	≠ＮＵＬＬ：見つかった文字列のアドレス														//
//					＝ＮＵＬＬ：指定された部分文字列は見つからない												//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcMbsStrExA(C_BCP s1, C_BCP s2, BOOL fIgnoreWidth)
{
	BCP		rc = NULL;

	if (fIgnoreWidth) rc = AjcMbsIStrA(s1, s2);
	else			  rc = (BCP)strstr(s1, s2);

	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcMbsStrExW(C_WCP s1, C_WCP s2, BOOL fIgnoreWidth)
{
	WCP		rc = NULL;

	if (fIgnoreWidth) rc = AjcStrIStrW(s1, s2);
	else			  rc = (WCP)wcsstr(s1, s2);

	return rc;
}
//==============================================================================================================//
//	Function	:	英大小文字を区別せず部分文字列を見つける（マルチバイト用）									//
//																												//
//	Argument	:	p - 文字列１のアドレス																		//
//					q - 文字列２のアドレス																		//
//																												//
//	Return		:	≠ＮＵＬＬ：見つかった文字列のアドレス														//
//					＝ＮＵＬＬ：指定された部分文字列は見つからない												//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcMbsIStrA(C_BCP s1, C_BCP s2)
{
	BCP		rc = NULL;
	C_BCP	p  = s1;
	int		l1;
	int		l2;
	int		n;

	if (s1 != NULL && s2 != NULL) {
		l1 = (int)mbslen(s1);
		l2 = (int)mbslen(s2);
		if		(l2 == 0) {
			rc = (BCP)s1;
		}
		else if (l1 >= l2) {
			n = l1 - l2;
			while (n >= 0) {
				if (mbsnicmp(p, s2, l2) == 0) {
					rc = (BCP)p;
					break;
				}
				if (_ismbblead(*p) && _ismbbtrail(*(p + 1))) p += 2;
				else										 p++;
				n--;
			}
		}
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcMbsIStrW(C_WCP s1, C_WCP s2)
{
	return AjcStrIStrW(s1, s2);
}
//==============================================================================================================//
//	Function	:	複数の部分文字列を見つける																	//
//																												//
//	Argument	:	pSrc			- 非検索文字列のアドレス													//
//					pStr 			- 検索する文字列のアドレス（複数指定時はpDlmで分離）						//
//					pDlm			- 分離文字				 (NULL：分離無し）									//
//					pAny			- 任意の文字列とする文字（NULL：任意の文字を認識しない）					//
//					pEnc			- 囲み文字群			 (NULL：過去も文字無し）							//
//					fTrim			- 両端の空白除去フラグ														//
//					opt				- オプション（AJCSFO_XXXXX）												//
//					pInfo			- 見つかった位置情報を格納するバッファ（不要時はNULL）						//
//																												//
//	Return		:	≠ -1 - いずれかの文字列が見つかった（見つかった文字列のバイト数／文字数）					//
//					＝ -1 - いずれの文字列も見つからない														//
//==============================================================================================================//
static	int		SubStrFindExW(C_WCP pSrc, C_WCP pStr, C_WCP pDlm, C_WCP pAny, C_WCP pEnc, BOOL fTrim, UI opt, PAJCSTRFIND pInfo, BOOL fIgnoreWidth);
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	int		WINAPI	AjcStrFindA(C_BCP pSrc, C_BCP pStr, PAJCSTRFIND pInfo)
{
	return AjcStrFindExA(pSrc, pStr, ";", "*", "\"", TRUE, 0, pInfo);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	int		WINAPI	AjcStrIFindA(C_BCP pSrc, C_BCP pStr, PAJCSTRFIND pInfo)
{
	return AjcStrIFindExA(pSrc, pStr, ";", "*", "\"", TRUE, 0, pInfo);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	int		WINAPI	AjcStrFindExA(C_BCP pSrc, C_BCP pStr, C_BCP pDlm, C_BCP pAny, C_BCP pEnc, BOOL fTrim, UI opt, PAJCSTRFIND pInfo)
{
	int		rc = -1;
	WCP		pSrcW , pStrW, pDlmW, pAnyW, pEncW;
	do {
		BCF_ALLOC(pSrc);
		BCF_ALLOC(pStr);
		BCF_ALLOC(pDlm);
		BCF_ALLOC(pAny);
		BCF_ALLOC(pEnc);
		if ((rc = AjcStrFindExW(pSrcW, pStrW, pDlmW, pAnyW, pEncW, fTrim, opt, pInfo)) != -1) {
			pInfo->pos = WideCharToMultiByte(CP_ACP, 0, pSrcW			  , pInfo->pos, NULL, 0, NULL, NULL);
			pInfo->len = WideCharToMultiByte(CP_ACP, 0, pSrcW + pInfo->pos, pInfo->len, NULL, 0, NULL, NULL);
		}
	} while(0);
	if (pSrcW != NULL) AjcTFree(pSrcW);
	if (pStrW != NULL) AjcTFree(pStrW);
	if (pDlmW != NULL) AjcTFree(pDlmW);
	if (pAnyW != NULL) AjcTFree(pAnyW);
	if (pEncW != NULL) AjcTFree(pEncW);

	return rc;
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	int		WINAPI	AjcStrIFindExA(C_BCP pSrc, C_BCP pStr, C_BCP pDlm, C_BCP pAny, C_BCP pEnc, BOOL fTrim, UI opt, PAJCSTRFIND pInfo)
{
	int		rc = -1;
	WCP		pSrcW , pStrW, pDlmW, pAnyW, pEncW;
	do {
		BCF_ALLOC(pSrc);
		BCF_ALLOC(pStr);
		BCF_ALLOC(pDlm);
		BCF_ALLOC(pAny);
		BCF_ALLOC(pEnc);
		if ((rc = AjcStrIFindExW(pSrcW, pStrW, pDlmW, pAnyW, pEncW, fTrim, opt, pInfo)) != -1) {
			pInfo->pos = WideCharToMultiByte(CP_ACP, 0, pSrcW			  , pInfo->pos, NULL, 0, NULL, NULL);
			pInfo->len = WideCharToMultiByte(CP_ACP, 0, pSrcW + pInfo->pos, pInfo->len, NULL, 0, NULL, NULL);
		}
	} while(0);
	if (pSrcW != NULL) AjcTFree(pSrcW);
	if (pStrW != NULL) AjcTFree(pStrW);
	if (pDlmW != NULL) AjcTFree(pDlmW);
	if (pAnyW != NULL) AjcTFree(pAnyW);
	if (pEncW != NULL) AjcTFree(pEncW);

	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	int		WINAPI	AjcStrFindW(C_WCP pSrc, C_WCP pStr, PAJCSTRFIND pInfo)
{
	return AjcStrFindExW(pSrc, pStr, L";", L"*", L"\"", TRUE, 0, pInfo);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	int		WINAPI	AjcStrIFindW(C_WCP pSrc, C_WCP pStr, PAJCSTRFIND pInfo)
{
	return AjcStrIFindExW(pSrc, pStr, L";", L"*", L"\"", TRUE, 0, pInfo);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	int		WINAPI	AjcStrFindExW(C_WCP pSrc, C_WCP pStr, C_WCP pDlm, C_WCP pAny, C_WCP pEnc, BOOL fTrim, UI opt, PAJCSTRFIND pInfo)
{
	int		rc = -1;
	rc = SubStrFindExW(pSrc, pStr, pDlm, pAny, pEnc, fTrim, opt, pInfo, FALSE);
	return rc;
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	int		WINAPI	AjcStrIFindExW(C_WCP pSrc, C_WCP pStr, C_WCP pDlm, C_WCP pAny, C_WCP pEnc, BOOL fTrim, UI opt, PAJCSTRFIND pInfo)
{
	int		rc = -1;
	rc = SubStrFindExW(pSrc, pStr, pDlm, pAny, pEnc, fTrim, opt, pInfo, TRUE);
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
static	int		SubStrFindExW(C_WCP pSrc, C_WCP pStr, C_WCP pDlm, C_WCP pAny, C_WCP pEnc, BOOL fTrim, UI opt, PAJCSTRFIND pInfo, BOOL fIgnoreWidth)
{
	int		rc 		= -1;
	BOOL	fMatch	= FALSE;	//	一致フラグ
	C_WCP	pS;					//	パス名中のポインタ
	WCP		pT;					//	アスタリスクで分離したトークン
	WCP		pTop = NULL;		//	文字列バッファ先頭
	WCP		pBuf = NULL;		//	検索する文字列トークン
	WCP		pStrTokDlm = NULL;	//	文字列分離用ワーク
	WCP		pStrTokAny = NULL;	//		〃
	AJCSTRFIND info = {0};		//	見つかった位置情報
	AJCSTRFIND ifsv = {0};		//	直前の見つかった位置情報

	if (pSrc != NULL && pStr != NULL) {
		UI	stl;
		//	見つかった位置情報初期化
		info.pos = (opt & AJCSFO_HIGHPOS) ? 0 : INT_MAX;
		//	バッファ確保
		stl = (UI)wcslen(pStr);
		if (pTop = pBuf = AjcTAlloc(stl + 1)) {
			//	検索する文字列をバッファに転送
			wcscpy(pBuf, pStr);
			//	両端空白除去
			if (fTrim) {
				AjcStrTrimW(pBuf, pBuf, stl + 1);
			}
			//	検索する文字列は空文字列以外・・
			if (*pBuf != 0) {
				//	最初の検索する文字列設定（囲み文字(")は除去しない）
				if (pDlm != NULL && *pDlm != 0) {
					pBuf = AjcStrTokSExW(pBuf, pDlm, pEnc, &pStrTokDlm);
				}
				if (pBuf != NULL) {
					//	比較ループ
					do {
						//	見つかった位置未設定の旨、設定
						ifsv.pos = -1;
						//	検索する文字列の両端空白除去
						if (fTrim) {
							AjcStrTrimW(pBuf, pBuf, (UI)wcslen(pBuf) + 1);
						}
						//	非検索文字列ポインタ初期化
						pS = pSrc;
						//	アスタリスク(*)で分離する場合
						if (pAny != NULL && *pAny != 0) {
							pStrTokAny = NULL;
							//	最初のアスタリスク(*)で分離したトークン設定（囲み文字は除去しない）
							if ((pT = AjcStrTokSExW(pBuf, pAny, pEnc, &pStrTokAny)) != NULL) {
								fMatch = TRUE;
								do {
									//	両端空白除去
									if (fTrim) {
										AjcStrTrimW(pT, pT, (UI)wcslen(pT) + 1);
									}
									//	囲み文字を除去
									SubPackEncloseW(pT, pEnc);
									//	アスタリスク(*)で分離した文字列が空文字列以外ならば、文字列が含まれるかチェック
									if (*pT != 0) {
										WCP	pTmp;
										//	文字列が含まれる場合、見つかった文字列の直後へ非検索文字列ポインタ更新
										if (fIgnoreWidth) pTmp = AjcStrIStrW(pS, pT);
										else			  pTmp = wcsstr 	(pS, pT);
										if (pTmp != NULL) {
											pS = pTmp + (UI)wcslen(pT);
											//	見つかった先頭位置設定
											if (ifsv.pos == -1) ifsv.pos = (UI)(((UX)pTmp - (UX)pSrc) / sizeof(WC));
											//	見つかった長さ設定
											ifsv.len = (UI)((((UX)(pTmp + wcslen(pT)) - (UX)pSrc) / sizeof(WC)) - ifsv.pos);
										}
										//	文字列が含まれない場合、不一致
										else {
											fMatch = FALSE;
											break;
										}
									}
									//	アスタリスク(*)で分離した文字列が空文字列ならば一致とする
									else {
										//	NOP
									}
									//	次のアスタリスク(*)で分離した文字列設定（囲み文字は除去しない）
								} while (pT = AjcStrTokSEx(NULL, pAny, pEnc, &pStrTokAny));
							}
							//	最初のアスタリスク(*)で分離した文字列が空文字列ならば一致とする
							else {
								//	見つかった先頭位置設定（低アドレス優先）
								if (ifsv.pos == -1) ifsv.pos = (UI)(((UX)pS - (UX)pSrc) / sizeof(WC));
								//	見つかった長さ設定
								ifsv.len = (UI)(((UX)(pS - (UX)pSrc) / sizeof(WC)) - ifsv.pos);
								fMatch = TRUE;
							}
						}
						//	アスタリスク(*)で分離しない場合
						else {
							WCP	pTmp;		//	検索文字列が見つかった位置
							//	囲み文字を除去した検索する文字列を設定
							SubPackEncloseW(pBuf, pEnc);
							//	検索する文字列一致チェック
							if (fIgnoreWidth) pTmp = AjcStrIStrW(pS, pBuf);
							else			  pTmp = wcsstr 	(pS, pBuf);
							if (pTmp != NULL) {
								//	見つかった先頭位置設定（低アドレス優先）
								if (ifsv.pos == -1) ifsv.pos = (UI)(((UX)pTmp - (UX)pSrc) / sizeof(WC));
								//	見つかった長さ設定
								ifsv.len = (UI)((((UX)(pTmp + wcslen(pBuf)) - (UX)pSrc) / sizeof(WC)) - ifsv.pos);
								fMatch = TRUE;
							}
						}
						//	見つかった文字列の位置と長さ更新
						if (fMatch) {
							BOOL	fNeedChange = FALSE;
							if (ifsv.pos == info.pos) {
								if (opt & AJCSFO_SHORT	) fNeedChange = (ifsv.len < info.len);
								else					  fNeedChange = (ifsv.len > info.len);
							}
							else {
								if (opt & AJCSFO_HIGHPOS) fNeedChange = (ifsv.pos > info.pos);
								else					  fNeedChange = (ifsv.pos < info.pos);
							}
							if (fNeedChange) {
								memcpy(&info, &ifsv, sizeof info);
							}
							rc = TRUE;
						}
						//	次の分離文字列設定
						if (pDlm != NULL && *pDlm != 0) {
							pBuf = AjcStrTokREx(NULL, pDlm, pEnc, &pStrTokDlm);
							if (pBuf != NULL) ifsv.ix++;
						}
						else pBuf = NULL;
					} while (pBuf != NULL);
				}
				//	空文字列ならば、一致とする
				else {
					fMatch = TRUE;
				}
			}
			//	空文字列ならば、一致とする
			else {
				fMatch = TRUE;
			}
			//	バッファ解放
			AjcTFree(pTop);
		}
	}
	//	戻り情報設定
	if (rc == TRUE) {
		if (pInfo != NULL) {
			memcpy(pInfo, &info, sizeof(AJCSTRFIND));
		}
		rc = info.len;
	}
	else {
		rc = -1;
	}
	return rc;
}
//==============================================================================================================//
//	文字列内のトークンを検索																					//
//																												//
//	引　数	：	pStr - トークンを取得する文字列																	//
//				pDlm - 区切り文字セット																			//
//																												//
//	戻り値	：	≠ NULL - トークンへのポインタ																	//
//				＝ NULL - トークン無し																			//
//==============================================================================================================//
__declspec(thread) static BCP bpStrTokPtr = NULL;
__declspec(thread) static WCP wpStrTokPtr = NULL;

//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcStrTokA(BCP pStr, C_BCP pDlm)
{
	BCP		rc = NULL;

	rc = strtok_s(pStr, pDlm, &bpStrTokPtr);

	return rc;
}
//----- バイト文字（ポインタワーク指定）------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcStrTokSA(BCP pStr, C_BCP pDlm, BCP *ppContext)
{
	BCP		rc = NULL;

	if (ppContext != NULL) {
		rc = strtok_s(pStr, pDlm, &bpStrTokPtr);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcStrTokW(WCP pStr, C_WCP pDlm)
{
	WCP		rc = NULL;

	rc = wcstok_s(pStr, pDlm, &wpStrTokPtr);

	return rc;
}
//----- ワイド文字（ポインタワーク指定）------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcStrTokSW(WCP pStr, C_WCP pDlm, WCP *ppContext)
{
	WCP		rc = NULL;

	if (ppContext != NULL) {
		rc = wcstok_s(pStr, pDlm, ppContext);
	}
	return rc;
}
//==============================================================================================================//
//	マルチバイト文字列内のトークンを検索																		//
//																												//
//	引　数	：	pStr - トークンを取得する文字列																	//
//				pDlm - 区切り文字セット																			//
//																												//
//	戻り値	：	≠ NULL - トークンへのポインタ																	//
//				＝ NULL - トークン無し																			//
//==============================================================================================================//
__declspec(thread) static UBP bpMbsTokPtr = NULL;
__declspec(thread) static WCP wpMbsTokPtr = NULL;

//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BCP	WINAPI	AjcMbsTokA(BCP pMbc, C_BCP pDlm)
{
	BCP		rc = NULL;

	rc = _mbstok_s((UBP)pMbc, (UBP)pDlm, &bpMbsTokPtr);

	return rc;
}
//----- バイト文字（ポインタワーク指定）------------------------------------------------------------------------//
AJCEXPORT	BCP	WINAPI	AjcMbsTokSA(BCP pMbc, C_BCP pDlm, BCP *ppContext)
{
	BCP		rc = NULL;

	if (ppContext != NULL) {
		rc = _mbstok_s((UBP)pMbc, (UBP)pDlm, ppContext);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcMbsTokW(WCP pMbc, C_WCP pDlm)
{
	WCP		rc = NULL;

	rc = wcstok_s(pMbc, pDlm, &wpMbsTokPtr);

	return rc;
}
//----- ワイド文字（ポインタワーク指定）------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcMbsTokSW(WCP pMbc, C_WCP pDlm, WCP *ppContext)
{
	WCP		rc = NULL;

	if (ppContext != NULL) {
		rc = wcstok_s(pMbc, pDlm, ppContext);
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	区切り文字で区切られた文字列内の次のトークンを取得											//
//																												//
//	Argument	:	pStr - トークンを検出する文字列																//
//					pDlm - 区切り文字群																			//
//					pEnc - 囲み文字群（囲み文字が無い場合はNULL)												//
//																												//
//	Return		:	≠ＮＵＬＬ：見つかったトークンのアドレス													//
//					＝ＮＵＬＬ：トークン無し																	//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
__declspec(thread) static BCP pStrTokTopA = NULL;

AJCEXPORT	BCP		WINAPI	AjcStrTokExA(BCP pStr, C_BCP pDlm, C_BCP pEnc)
{
	return AjcStrTokSExA(pStr, pDlm, pEnc, &pStrTokTopA);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcStrTokRExA(BCP pStr, C_BCP pDlm, C_BCP pEnc, BCP *ppContext)
{
	BCP		rc;

	if ((rc = AjcStrTokSExA(pStr, pDlm, pEnc, ppContext)) != NULL) {
		SubPackEncloseA(rc, pEnc);
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcStrTokSExA(BCP pStr, C_BCP pDlm, C_BCP pEnc, BCP *ppContext)
{
	BCP		rc = NULL;
	BC		ce;

	//	NULL指定ならば、前回の次のトークンアドレス設定
	if (pStr == NULL) pStr = *ppContext;

	if (pStr != NULL && *pStr != 0) {
		//	戻り値＝トークン先頭
		rc = pStr;
		//	トークン設定
		while (*pStr != 0) {
			//	区切り文字ならば、文字列終端を設定
			if (pDlm != NULL && strchr(pDlm, *pStr) != NULL) {
				*pStr++ = 0;
				break;
			}
			//	区切り文字以外ならば、文字／囲み文字列をスキップ
			else {
				//	囲み文字ならば、囲み文字列をスキップ
				if (pEnc != NULL && strchr(pEnc, *pStr) != NULL) {
					//	囲み終端文字取得
					ce = GetEncloseCharA(*pStr);
					//	囲み文字スキップ
					if (_ismbblead(*pStr) && _ismbbtrail(*(pStr + 1))) pStr += 2;
					else											   pStr += 1;
					//	囲み文字列終端検索
					while (*pStr != ce && *pStr != 0) {
						if (_ismbblead(*pStr) && _ismbbtrail(*(pStr + 1))) pStr += 2;
						else											   pStr += 1;
					}
					//	囲み文字列終端ならば、囲み終端文字スキップ
					if (*pStr == ce) {
						pStr++;
					}
					//	文字列終端ならば、終了
					else {
						break;
					}
				}
				//	囲み文字以外ならば、当該文字をスキップ
				else {
					pStr++;
				}
			}
		}
		//	次のトークンアドレス設定
		if (pDlm != NULL) {
			while (*pStr != 0 && strchr(pDlm, *pStr) != NULL) {
				pStr++;
			}
		}
		*ppContext = pStr;
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
__declspec(thread) static WCP pStrTokTopW = NULL;

AJCEXPORT	WCP		WINAPI	AjcStrTokExW(WCP pStr, C_WCP pDlm, C_WCP pEnc)
{
	return AjcStrTokSExW(pStr, pDlm, pEnc, &pStrTokTopW);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcStrTokRExW(WCP pStr, C_WCP pDlm, C_WCP pEnc, WCP *ppContext)
{
	WCP		rc;

	if ((rc = AjcStrTokSExW(pStr, pDlm, pEnc, ppContext)) != NULL) {
		SubPackEncloseW(rc, pEnc);
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcStrTokSExW(WCP pStr, C_WCP pDlm, C_WCP pEnc, WCP *ppContext)
{
	WCP		rc = NULL;
	WC		ce;

	//	NULL指定ならば、前回の次のトークンアドレス設定
	if (pStr == NULL) pStr = *ppContext;

	if (pStr != NULL && *pStr != 0) {
		//	戻り値＝トークン先頭
		rc = pStr;
		//	トークン設定
		while (*pStr != 0) {
			//	区切り文字ならば、文字列終端を設定
			if (pDlm != NULL && wcschr(pDlm, *pStr) != NULL) {
				*pStr++ = 0;
				break;
			}
			//	区切り文字以外ならば、文字／囲み文字列をスキップ
			else {
				//	囲み文字ならば、囲み文字列をスキップ
				if (pEnc != NULL && wcschr(pEnc, *pStr) != NULL) {
					//	囲み終端文字取得
					ce = GetEncloseCharW(*pStr);
					//	囲み文字スキップ
					pStr++;
					//	囲み文字列終端検索
					while (*pStr != ce && *pStr != 0) {
						pStr++;
					}
					//	囲み文字列終端ならば、囲み終端文字スキップ
					if (*pStr == ce) {
						pStr++;
					}
					//	文字列終端ならば、終了
					else {
						break;
					}
				}
				//	囲み文字以外ならば、当該文字をスキップ
				else {
					pStr++;
				}
			}
		}
		//	次のトークンアドレス設定
		if (pDlm != NULL) {
			while (*pStr != 0 && wcschr(pDlm, *pStr) != NULL) {
				pStr++;
			}
		}
		*ppContext = pStr;
	}
	return rc;
}

//==============================================================================================================//
//	文字列を指定文字で囲む（文字列中の「指定文字」と '\' の直前には'\'が付加される）							//
//																												//
//	引　数	：	pStr  - ソース文字列のアドレス																	//
//				pBuf  - 文字列を囲み文字で囲んだ文字列を格納するバッファのアドレス（不要時はNULL)				//
//						「pBuf=NULL」とした場合は、囲み文字付加後の文字長・算出だけを行います。					//
//				lBuf  - 文字列を囲み文字で囲んだ文字列を格納するバッファの文字数								//
//				c	  - 文字列を囲む文字（括弧等の場合は、左側の文字を指定(右側自動生成)。０／￥は指定不可。）	//
//																												//
//	戻り値	：	≠０ - OK（囲み文字を付加した文字列の文字長）													//
//				＝０ - Error																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcStrEncloseA(C_BCP pStr, BCP pBuf, UI lBuf, UI c)
{
	UI		rc	= TRUE;
	UI		len = 0;
	BC		cl, cr;

	if (pStr != NULL) {
		//----- 囲み文字設定 ---------------------------------------------------------------------------------------//
		cl = c;
		cr = GetEncloseCharA(cl);
		if (cl == '\\' || cl == 0) rc = FALSE;
		//----- 先頭の囲み文字設定 ---------------------------------------------------------------------------------//
		if (pBuf != NULL  &&  lBuf > 1) {
			*pBuf++ = cl;
			lBuf--;
		}
		else if (pBuf != NULL) rc = FALSE;
		len++;
		//----- 文字列転送 -----------------------------------------------------------------------------------------//
		while (*pStr != 0) {
			if (_ismbblead(*pStr) && _ismbbtrail(*(pStr + 1))) {
				if (pBuf != NULL  &&  lBuf > 2) {
					memcpy(pBuf, pStr, 2);
					pBuf += 2;
					lBuf -= 2;
				}
				else if (pBuf != NULL) rc = FALSE;
				pStr += 2;
				len  += 2;
			}
			else if (*pStr == cr || *pStr == '\\') {
				if (pBuf != NULL  &&  lBuf > 2) {
					*pBuf++ = '\\';
					*pBuf++ = *pStr;
					lBuf -= 2;
				}
				else if (pBuf != NULL) rc = FALSE;
				pStr++;
				len += 2;
			}
			else {
				if (pBuf != NULL  &&  lBuf > 1) {
					*pBuf++ = *pStr;
					lBuf--;
				}
				else if (pBuf != NULL) rc = FALSE;
				pStr++;
				len++;
			}
		}
		//----- 末尾の囲み文字設定 ---------------------------------------------------------------------------------//
		if (pBuf != NULL  &&  lBuf > 1) {
			*pBuf++ = cr;
			lBuf--;
		}
		else if (pBuf != NULL) rc = FALSE;
		len++;
		//----- 文字列終端設定 -------------------------------------------------------------------------------------//
		if (pBuf != NULL  &&  lBuf >= 1) {
			*pBuf = 0;
		}
		else if (pBuf != NULL) rc = FALSE;
		//----- 戻り値設定 -----------------------------------------------------------------------------------------//
		if (rc != FALSE) {
			rc = len;
		}
	}
	else rc = 0;

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcStrEncloseW(C_WCP pStr, WCP pBuf, UI lBuf, UI c)
{
	BOOL	rc	= TRUE;
	int		len = 0;
	UW		cl, cr;

	if (pStr != NULL) {
		//----- 囲み文字設定 ---------------------------------------------------------------------------------------//
		cl = c;
		cr = GetEncloseCharW(cl);
		if (cl == L'\\' || cl == 0) rc = FALSE;
		//----- 先頭の囲み文字設定 ---------------------------------------------------------------------------------//
		if (pBuf != NULL  &&  lBuf > 1) {
			*pBuf++ = cl;
			lBuf--;
		}
		else if (pBuf != NULL) rc = FALSE;
		len++;
		//----- 文字列転送 -----------------------------------------------------------------------------------------//
		while (*pStr != 0) {
			if (*pStr == cr  ||  *pStr == L'\\') {
				if (pBuf != NULL  &&  lBuf > 2) {
					*pBuf++ = L'\\';
					*pBuf++ = *pStr;
					lBuf -= 2;
				}
				else if (pBuf != NULL) rc = FALSE;
				pStr++;
				len += 2;
			}
			else {
				if (pBuf != NULL  &&  lBuf > 1) {
					*pBuf++ = *pStr;
					lBuf--;
				}
				else if (pBuf != NULL) rc = FALSE;
				pStr++;
				len++;
			}
		}
		//----- 末尾の囲み文字設定 ---------------------------------------------------------------------------------//
		if (pBuf != NULL  &&  lBuf > 1) {
			*pBuf++ = cr;
			lBuf--;
		}
		else if (pBuf != NULL) rc = FALSE;
		len++;
		//----- 文字列終端設定 -------------------------------------------------------------------------------------//
		if (pBuf != NULL  &&  lBuf >= 1) {
			*pBuf = 0;
		}
		else if (pBuf != NULL) rc = FALSE;
		//----- 戻り値設定 -----------------------------------------------------------------------------------------//
		if (rc != FALSE) {
			rc = len;
		}
	}
	else rc = 0;

	return rc;
}
//==============================================================================================================//
//	文字列の囲み文字を取り除く																					//
//																												//
//	引　数	：	pStr  - 囲み文字を取り除く文字列のアドレス														//
//																												//
//	戻り値	：	≠NULL - 囲み文字列の次の文字のアドレス															//
//				＝NULL - Error																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcStrStripEncloseA(BCP pStr)
{
	BCP		rc = NULL;
	BCP		pDes;
	BC		cl, cr;

	do {
		if (pStr  == NULL ) break;		//	NULL
		if (*pStr == 0	  ) break;		//	空文字列
		if (*pStr == '\\' ) break;		//	囲み文字＝￥
		if (_ismbblead(*pStr) ) break;	//	先頭は全角文字

		cl = *pStr;
		cr = GetEncloseCharA(cl);

		pDes = pStr++;
		while (*pStr != cr	&&	*pStr != 0) {
			if (_ismbblead(*pStr) && _ismbbtrail(*(pStr + 1))) {
				*pDes++ = *pStr++;
				*pDes++ = *pStr++;
			}
			else if (*pStr == '\\') {
				pStr++;
				if (*pStr != 0) {
					*pDes++ = *pStr++;
				}
			}
			else {
				*pDes++ = *pStr++;
			}
		}
		*pDes = 0;
		if (*pStr == cr) rc = pStr + 1;
		else			 rc = pStr;

	} while(0);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcStrStripEncloseW(WCP pStr)
{
	WCP		rc = NULL;
	WCP		pDes;
	UW		cl, cr;

	do {
		if (pStr  == NULL ) break;	//	NULL
		if (*pStr == 0	  ) break;	//	空文字列
		if (*pStr == L'\\') break;	//	囲み文字＝￥

		cl = *pStr;
		cr = GetEncloseCharW(cl);

		pDes = pStr++;
		while (*pStr != cr	&&	*pStr != 0) {
			if (*pStr == L'\\') {
				pStr++;
				if (*pStr != 0) {
					*pDes++ = *pStr++;
				}
			}
			else {
				*pDes++ = *pStr++;
			}
		}
		*pDes = 0;
		if (*pStr == cr) rc = pStr + 1;
		else			 rc = pStr;

	} while(0);

	return rc;
}

//==============================================================================================================//
//	書式文字列生成																								//
//																												//
//	引　数	：	pBuf - 文字列を生成するバッファのアドレス														//
//				lBuf - 文字列を生成するバッファの文字数															//
//				pFmt - 書式文字列																				//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error（バッファオーバー）																//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcSnPrintFA(BCP pBuf, UI lBuf, C_BCP pFmt, ...)
{
	BOOL	rc = TRUE;
	va_list vls;

	if (pBuf != NULL && lBuf != 0 && pFmt != NULL) {
		if (lBuf != 0) {
			pBuf[lBuf - 1] = 0;
			va_start(vls, pFmt);
			_vsnprintf(pBuf, lBuf, pFmt, vls);
			va_end	(vls);
			if (pBuf[lBuf - 1] != 0) {
				pBuf[lBuf - 1] = 0;
				rc = FALSE;
			}
		}
	}
	else rc = FALSE;

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcSnPrintFW(WCP pBuf, UI lBuf, C_WCP pFmt, ...)
{
	BOOL	rc = TRUE;
	va_list vls;

	if (pBuf != NULL && lBuf != 0 && pFmt != NULL) {
		if (lBuf != 0) {
			pBuf[lBuf - 1] = 0;
			va_start(vls, pFmt);
			_vsnwprintf(pBuf, lBuf, pFmt, vls);
			va_end	(vls);
			if (pBuf[lBuf - 1] != 0) {
				pBuf[lBuf - 1] = 0;
				rc = FALSE;
			}
		}
	}
	else rc = FALSE;

	return rc;
}
//==============================================================================================================//
//	書式文字列生成（va_list指定）																				//
//																												//
//	引　数	：	pBuf - 文字列を生成するバッファのアドレス														//
//				lBuf - 文字列を生成するバッファの文字数															//
//				pFmt - 書式文字列																				//
//				vls  - 可変パラメタへのポインタ																	//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error（バッファオーバー）																//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcVSnPrintFA(BCP pBuf, UI lBuf, C_BCP pFmt, va_list vls)
{
	BOOL	rc = TRUE;

	if (pBuf != NULL && lBuf != 0 && pFmt != NULL) {
		if (lBuf != 0) {
			pBuf[lBuf - 1] = 0;
			_vsnprintf(pBuf, lBuf, pFmt, vls);
			if (pBuf[lBuf - 1] != 0) {
				pBuf[lBuf - 1] = 0;
				rc = FALSE;
			}
		}
	}
	else rc = FALSE;

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcVSnPrintFW(WCP pBuf, UI lBuf, C_WCP pFmt, va_list vls)
{
	BOOL	rc = TRUE;

	if (pBuf != NULL && lBuf != 0 && pFmt != NULL) {
		if (lBuf != 0) {
			pBuf[lBuf - 1] = 0;
			_vsnwprintf(pBuf, lBuf, pFmt, vls);
			if (pBuf[lBuf - 1] != 0) {
				pBuf[lBuf - 1] = 0;
				rc = FALSE;
			}
		}
	}
	else rc = FALSE;

	return rc;
}

//==============================================================================================================//
//	書式文字列生成と１０進数文字列の整数部で指定桁数毎に区切り文字を挿入する（ロケール情報を参照）				//
//																												//
//	引　数	：	pBuf - 文字列を生成するバッファのアドレス														//
//				lBuf - 文字列を生成するバッファの文字数															//
//				pFmt - 書式文字列																				//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error（バッファオーバー）																//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcSnPrtSepA(BCP pBuf, UI lBuf, C_BCP pFmt, ...)
{
	BOOL	rc = FALSE;

	va_list vls;

	if (pBuf != NULL && pFmt != NULL) {
		if (lBuf != 0) {
			pBuf[lBuf - 1] = 0;
			va_start(vls, pFmt);
			_vsnprintf(pBuf, lBuf, pFmt, vls);
			va_end	(vls);
			if (pBuf[lBuf - 1] == 0) {
				rc = AjcStrSepDecimalA(pBuf, lBuf);
			}
			else {
				pBuf[lBuf - 1] = 0;
				rc = FALSE;
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcSnPrtSepW(WCP pBuf, UI lBuf, C_WCP pFmt, ...)
{
	BOOL	rc = FALSE;

	va_list vls;

	if (pBuf != NULL && pFmt != NULL) {
		if (lBuf != 0) {
			pBuf[lBuf - 1] = 0;
			va_start(vls, pFmt);
			_vsnwprintf(pBuf, lBuf, pFmt, vls);
			va_end	(vls);
			if (pBuf[lBuf - 1] == 0) {
				rc = AjcStrSepDecimalW(pBuf, lBuf);
			}
			else {
				pBuf[lBuf - 1] = 0;
				rc = FALSE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	１０進数文字列の整数部で指定桁数毎に区切り文字を挿入する（ロケール情報参照）								//
//																												//
//	引　数	：	pStr - 区切り文字を挿入する１０進文字列のアドレス												//
//				lBuf - 区切り文字を挿入する１０進文字列バッファの文字数											//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error（バッファオーバー）																//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcStrSepDecimalA(BCP pStr, UI lBuf)
{
	BOOL	rc = FALSE;
	BC		cSep[64];
	BC		nSep[64];

	if (pStr != NULL) {
		do {
			if (GetLocaleInfoA(LOCALE_USER_DEFAULT, LOCALE_SLIST	, cSep, sizeof cSep) == 0) break;
			if (GetLocaleInfoA(LOCALE_USER_DEFAULT, LOCALE_SGROUPING, nSep, sizeof nSep) == 0) break;
			rc = AjcStrAnySepDecimalA(pStr, lBuf, cSep[0], atoi(nSep));
		} while(0);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcStrSepDecimalW(WCP pStr, UI lBuf)
{
	BOOL	rc = FALSE;
	WC		cSep[64];
	WC		nSep[64];

	if (pStr != NULL) {
		do {
			if (GetLocaleInfoW(LOCALE_USER_DEFAULT, LOCALE_SLIST	, cSep, sizeof cSep / 2) == 0) break;
			if (GetLocaleInfoW(LOCALE_USER_DEFAULT, LOCALE_SGROUPING, nSep, sizeof nSep / 2) == 0) break;
			rc = AjcStrAnySepDecimalW(pStr, lBuf, cSep[0], _wtoi(nSep));
		} while(0);
	}
	return rc;
}
//==============================================================================================================//
//	１０進数文字列の整数部で指定桁数毎に区切り文字を挿入する													//
//																												//
//	引　数	：	pStr - 区切り文字を挿入する１０進文字列のアドレス												//
//				lBuf - 区切り文字を挿入する１０進文字列バッファの文字数											//
//				cSep - 区切り文字																				//
//				nSep - 区切り桁数（０の場合は、区切り文字を挿入しない）											//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error（バッファオーバー）																//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcStrAnySepDecimalA(BCP pStr, UI lBuf, UI cSep, UI nSep)
{
	BOOL	rc = TRUE;
	BCP		pBuf = pStr;
	UI		lStr;
	int		len, six, eix, mod;

	if (pStr != NULL && lBuf != 0) {
		do {
			//----- 区切り桁数＝０の場合は、正常終了とする -----------------------------------------------------//
			if (nSep == 0) break;
			//----- 文字列長と、文字列インデクス設定 -----------------------------------------------------------//
			lStr = (UI)strlen(pStr);
			six  = 0;
			eix  = lStr + 1;
			//----- 先頭スペーススキップ -----------------------------------------------------------------------//
			while (MAjcIsBlankA(*pStr)) {
				pStr++; six++;
			}
			//----- 符号スキップ -------------------------------------------------------------------------------//
			if (*pStr == '+' || *pStr == '-') {
				pStr++; six++;
			}
			//----- 整数部桁数算出 -----------------------------------------------------------------------------//
			for (len = 0; _ismbcdigit(pStr[len]); len++);
			//----- 整数部が無い場合は、正常終了とする ---------------------------------------------------------//
			if (len == 0) break;
			//----- 先頭桁スキップ -----------------------------------------------------------------------------//
			pStr++; six++; len--;
			//----- 最初の区切り文字挿入位置までスキップ -------------------------------------------------------//
			mod = (len % nSep);
			pStr += mod;
			six  += mod;
			len  -= mod;
			//----- バッファオーバーチェック -------------------------------------------------------------------//
			if (lStr + (len / nSep) >= lBuf) {
				rc = FALSE;
				break;
			}
			//----- 区切り文字挿入 -----------------------------------------------------------------------------//
			while (len > 0) {
				memmove(&pBuf[six + 1], &pBuf[six], eix - six);
				pBuf[six] = cSep;
				lStr++;
				six  += (nSep + 1);
				eix++;
				len  -= nSep;
			}
		} while(0);
	}
	else rc = FALSE;

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcStrAnySepDecimalW(WCP pStr, UI lBuf, UI cSep, UI nSep)
{
	BOOL	rc = TRUE;
	WCP		pBuf = pStr;
	UI		lStr;
	int		len, six, eix, mod;

	if (pStr != NULL && lBuf != 0) {
		do {
			//----- 区切り桁数＝０の場合は、正常終了とする -----------------------------------------------------//
			if (nSep == 0) break;
			//----- 文字列長と、文字列インデクス設定 -----------------------------------------------------------//
			lStr = (UI)wcslen(pStr);
			six  = 0;
			eix  = lStr + 1;
			//----- 先頭スペーススキップ -----------------------------------------------------------------------//
			while (MAjcIsBlankW(*pStr)) {
				pStr++; six++;
			}
			//----- 符号スキップ -------------------------------------------------------------------------------//
			if (*pStr == L'+' || *pStr == L'-') {
				pStr++; six++;
			}
			//----- 整数部桁数算出 -----------------------------------------------------------------------------//
			for (len = 0; MAjcAscIsDigitW(pStr[len]); len++);
			//----- 整数部が無い場合は、正常終了とする ---------------------------------------------------------//
			if (len == 0) break;
			//----- 先頭桁スキップ -----------------------------------------------------------------------------//
			pStr++; six++; len--;
			//----- 最初の区切り文字挿入位置までスキップ -------------------------------------------------------//
			mod = (len % nSep);
			pStr += mod;
			six  += mod;
			len  -= mod;
			//----- バッファオーバーチェック -------------------------------------------------------------------//
			if (lStr + (len / nSep) >= lBuf) {
				rc = FALSE;
				break;
			}
			//----- 区切り文字挿入 -----------------------------------------------------------------------------//
			while (len > 0) {
				memmove(&pBuf[six + 1], &pBuf[six], (eix - six) * 2);
				pBuf[six] = cSep;
				lStr++;
				six  += (nSep + 1);
				eix++;
				len  -= nSep;
			}
		} while(0);
	}
	else rc = FALSE;

	return rc;
}
//==============================================================================================================//
//	１０進数文字列の整数部から区切り文字を削除する（ロケール情報参照）											//
//																												//
//	引　数	：	pStr - 区切り文字を削除する１０進文字列のアドレス												//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	VO		WINAPI	AjcStrRmvSepCharA(BCP pStr)
{
	BC		cSep[64];

	if (pStr != NULL) {
		if (GetLocaleInfoA(LOCALE_USER_DEFAULT, LOCALE_SLIST, cSep, sizeof cSep) != 0) {
			AjcStrAnyRmvSepCharA(pStr, cSep[0]);
		}
	}
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	VO		WINAPI	AjcStrRmvSepCharW(WCP pStr)
{
	WC		cSep[64];

	if (pStr != NULL) {
		if (GetLocaleInfoW(LOCALE_USER_DEFAULT, LOCALE_SLIST, cSep, sizeof cSep / 2) != 0) {
			AjcStrAnyRmvSepCharW(pStr, cSep[0]);
		}
	}
}
//==============================================================================================================//
//	１０進数文字列の整数部から区切り文字を削除する																//
//																												//
//	引　数	：	pStr - 区切り文字を削除する１０進文字列のアドレス												//
//				cSep - 区切り文字																				//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	VO		WINAPI	AjcStrAnyRmvSepCharA(BCP pStr, UI cSep)
{
	BCP		pBuf = pStr;
	int		six, eix;

	if (pStr != NULL) {
		//----- 文字列長と、文字列インデクス設定 -----------------------------------------------------------//
		six  = 0;
		eix  = (UI)strlen(pStr) + 1;
		//----- 先頭スペーススキップ -----------------------------------------------------------------------//
		while (MAjcIsBlankA((UB)*pStr)) {
			pStr++; six++;
		}
		//----- 符号スキップ -------------------------------------------------------------------------------//
		if (*pStr == '+' || *pStr == '-') {
			pStr++; six++;
		}
		//----- 区切り文字削除 -----------------------------------------------------------------------------//
		while (_ismbcdigit((UB)*pStr) || *pStr == cSep) {
			if (_ismbcdigit(*pStr)) {
				pStr++;
				six++;
			}
			else {
				memmove(&pBuf[six], &pBuf[six + 1], eix - six);
				eix--;
			}
		}
	}
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	VO		WINAPI	AjcStrAnyRmvSepCharW(WCP pStr, UI cSep)
{
	WCP		pBuf = pStr;
	int		six, eix;

	if (pStr != NULL) {
		//----- 文字列長と、文字列インデクス設定 -----------------------------------------------------------//
		six  = 0;
		eix  = (UI)wcslen(pStr) + 1;
		//----- 先頭スペーススキップ -----------------------------------------------------------------------//
		while (MAjcIsBlankW(*pStr)) {
			pStr++; six++;
		}
		//----- 符号スキップ -------------------------------------------------------------------------------//
		if (*pStr == '+' || *pStr == '-') {
			pStr++; six++;
		}
		//----- 区切り文字削除 -----------------------------------------------------------------------------//
		while (MAjcAscIsDigitW(*pStr) || *pStr == cSep) {
			if (MAjcAscIsDigitW(*pStr)) {
				pStr++;
				six++;
			}
			else {
				memmove(&pBuf[six], &pBuf[six + 1], (eix - six - 1) * sizeof(UW));
				eix--;
			}
		}
	}
}
//==============================================================================================================//
//	多重文字列→文字列ポインタ配列生成																			//
//																												//
//	引　数	：	pStr - 多重文字列のアドレス																		//
//				pnArr- 配列の要素数を格納するバッファのアドレス													//
//																												//
//	戻り値	：	≠NULL - 生成した文字列ポインタ配列のアドレス													//
//				＝NULL - 文字列なし／メモリ不足																	//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BCP	*	WINAPI	AjcMStrMakeArrayA(BCP pStr, UIP pnArr)
{
	BCP		*pArr = NULL;
	BCP		*p;
	UI		n;
	UI		i = 0;

	if (pStr != NULL) {
		n = AjcMStrCountA(pStr);
		if (pArr = p = (BCP*)AJCMEM((n + 1) * sizeof(BCP))) {
			for (i=0; i<n && pStr!=NULL; i++) {
				*p++ = pStr;
				pStr = AjcMStrNextPtrA(pStr);
			}
			*p = NULL;
		}
		if (pnArr != NULL) *pnArr = i;
	}
	return pArr;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	WCP	*	WINAPI	AjcMStrMakeArrayW(WCP pStr, UIP pnArr)
{
	WCP		*pArr = NULL;
	WCP		*p;
	UI		n;
	UI		i = 0;

	if (pStr != NULL) {
		n = AjcMStrCountW(pStr);
		if (pArr = p = (WCP*)AJCMEM((n + 1) * sizeof(WCP))) {
			for (i=0; i<n && pStr!=NULL; i++) {
				*p++ = pStr;
				pStr = AjcMStrNextPtrW(pStr);
			}
			*p = NULL;
		}
		if (pnArr != NULL) *pnArr = i;
	}
	return pArr;
}
//==============================================================================================================//
//	多重文字列中の文字列ポインタ配列開放																		//
//																												//
//	引　数	：	pArr - 文字列ポインタ配列のアドレス																//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	VO		WINAPI	AjcMStrReleaseArray(VOP pArr)
{
	if (pArr != NULL) {
		free(pArr);
	}
}

//==============================================================================================================//
//	多重文字列中の文字列・個数取得																				//
//																												//
//	引　数	：	pStr - 多重文字列のアドレス																		//
//																												//
//	戻り値	：	文字列個数																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcMStrCountA(C_BCP pStr)
{
	UI		rc = 0;
	UI		len;

	if (pStr != NULL && *pStr != 0) {
		do {
			len = (UI)strlen(pStr) + 1;
			rc++;
			pStr += len;
		} while (*pStr != 0);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcMStrCountW(C_WCP pStr)
{
	UI		rc = 0;
	UI		len;

	if (pStr != NULL && *pStr != 0) {
		do {
			len = (UI)wcslen(pStr) + 1;
			rc++;
			pStr += len;
		} while (*pStr != 0);
	}
	return rc;
}
//==============================================================================================================//
//	多重文字列の総文字数取得																					//
//																												//
//	引　数	：	pStr - 多重文字列のアドレス																		//
//																												//
//	戻り値	：	総文字数（終端文字を含む）																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcMStrTotalSizeA(C_BCP pStr)
{
	UI		rc = 1;
	UI		len;

	if (pStr != NULL && *pStr != 0) {
		do {
			len = (UI)strlen(pStr) + 1;
			rc	 += len;
			pStr += len;
		} while (*pStr != 0);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcMStrTotalSizeW(C_WCP pStr)
{
	UI		rc = 1;
	UI		len;

	if (pStr != NULL && *pStr != 0) {
		do {
			len = (UI)wcslen(pStr) + 1;
			rc	 += len;
			pStr += len;
		} while (*pStr != 0);
	}
	return rc;
}
//==============================================================================================================//
//	多重文字列の次の文字列ポインタ取得																			//
//																												//
//	引　数	：	pStr - 多重文字列のアドレス																		//
//																												//
//	戻り値	：	≠NULL - 次の文字列のアドレス																	//
//				＝NULL - 多重文字列の終端を検出																	//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcMStrNextPtrA(BCP pStr)
{
	BCP		pNxt = NULL;

	if (pStr != NULL && *pStr != 0) {
		pNxt = pStr + (strlen(pStr) + 1);
		if (*pNxt == 0) {
			pNxt = NULL;
		}
	}
	return pNxt;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcMStrNextPtrW(WCP pStr)
{
	WCP		pNxt = NULL;

	if (pStr != NULL && *pStr != 0) {
		pNxt = pStr + (wcslen(pStr) + 1);
		if (*pNxt == 0) {
			pNxt = NULL;
		}
	}
	return pNxt;
}
//==============================================================================================================//
//	Function	:	文字列１中の部分文字列を文字列２に置換														//
//																												//
//	Argument	:	pStr1 - 文字列１のアドレス																	//
//					lBuf1 - 文字列１が格納されているバッファのバイト数／文字数									//
//					loc   - 文字列１中の部分文字列のバイト位置／文字位置										//
//					len   - 文字列１中の部分文字列のバイト数／文字数											//
//					pStr2 - 文字列２のアドレス																	//
//					lStr2 - 文字列２の長さ（バイト数／文字数，－１指定時は自動算出）							//
//																												//
//	Return		:	≠ＮＵＬＬ：ＯＫ（置換された文字列１のアドレス）											//
//					＝ＮＵＬＬ：パラメタエラー（部分文字列は置換されません）									//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcRepPartStrA(BCP pStr1, UI lBuf, UI loc, UI len, C_BCP pStr2, UI lStr2)
{
	BCP		rc = NULL;
	UI		lStr1;
	BCP		p;

	do {
		//----- パラメタチェック（１）------//
		if (lBuf  == 0	 ) break;
		if (pStr1 == NULL) break;
		if (pStr2 == NULL) break;
		//----- 文字列１の長さ算出 ---------//
		for (p=pStr1, lStr1=0; lStr1 < lBuf && *p != 0; lStr1++, p++);
		if (*p != 0) break;
		//----- 文字列２の長さ算出 ---------//
		lStr2 = AjcStrAdjustLenA(pStr2, lStr2);
		//----- パラメタチェック（２）------//
		if ((loc + len) > lStr1) break;
		if (len < lStr2) {
			if ((lStr1 + (lStr2 - len)) > (lBuf - 1)) break;
		}
		//----- 部分文字列置換 -------------//
		if		(len > lStr2) {
			memmove((pStr1 + (loc + lStr2)), pStr1 + (loc + len), lStr1 - (loc + len) + 1);
		}
		else if (len < lStr2) {
			memmove(pStr1 + (loc + len + (lStr2 - len)), pStr1 + (loc + len), lStr1 - (loc + len) + 1);
		}
		memcpy(pStr1 + loc, pStr2, lStr2);
		rc = pStr1;
	} while(0);

	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcRepPartStrW(WCP pStr1, UI lBuf, UI loc, UI len, C_WCP pStr2, UI lStr2)
{
	WCP		rc = NULL;
	UI		lStr1;
	WCP		p;

	do {
		//----- パラメタチェック（１）------//
		if (lBuf  == 0	 ) break;
		if (pStr1 == NULL) break;
		if (pStr2 == NULL) break;
		//----- 文字列１の長さ算出 ---------//
		for (p=pStr1, lStr1=0; lStr1 < lBuf && *p != 0; lStr1++, p++);
		if (*p != 0) break;
		//----- 文字列２の長さ補正 ---------//
		lStr2 = AjcStrAdjustLenW(pStr2, lStr2);
		//----- パラメタチェック（２）------//
		if ((loc + len) > lStr1) break;
		if (len < lStr2) {
			if ((lStr1 + (lStr2 - len)) > (lBuf - 1)) break;
		}
		//----- 部分文字列置換 -------------//
		if		(len > lStr2) {
			memmove((pStr1 + (loc + lStr2)), pStr1 + (loc + len), (lStr1 - (loc + len) + 1) * 2);
		}
		else if (len < lStr2) {
			memmove(pStr1 + (loc + len + (lStr2 - len)), pStr1 + (loc + len), (lStr1 - (loc + len) + 1) * 2);
		}
		memcpy(pStr1 + loc, pStr2, (lStr2) * 2);
		rc = pStr1;
	} while(0);

	return rc;
}
//==============================================================================================================//
//	Function	:	文字列コピー																				//
//																												//
//	Argument	:	pBuf - 文字列コピー先のアドレス																//
//					lBuf - 文字列コピー先バッファのバイト数／文字数												//
//					pStr - コピーする文字列のアドレス															//
//																												//
//	Return		:	≠ NULL : コピー先文字列のアドレス( = pBuf )												//
//					＝ NULL : 文字列の長さがバッファを超えている／ヌルポインタ									//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcStrCpyA(BCP pBuf, UI lBuf, C_BCP pStr)
{
	BCP		rc = NULL;

	if (pBuf != NULL && pStr != NULL && strlen(pStr) < lBuf) {
		strcpy(pBuf, pStr);
		rc = pBuf;
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcStrCpyW(WCP pBuf, UI lBuf, C_WCP pStr)
{
	WCP		rc = NULL;

	if (pBuf != NULL && pStr != NULL && wcslen(pStr) < lBuf) {
		wcscpy(pBuf, pStr);
		rc = pBuf;
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	最大コピー文字数を指定して文字列コピー														//
//																												//
//	Argument	:	pBuf - 文字列コピー先のアドレス																//
//					lBuf - 文字列コピー先バッファのバイト数／文字数												//
//					pStr - コピーする文字列のアドレス															//
//					count- コピーする最大文字数																	//
//																												//
//	Return		:	≠ NULL : コピー先文字列のアドレス( = pBuf )												//
//					＝ NULL : 文字列の長さがバッファを超えている／ヌルポインタ									//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcStrNCpyA(BCP pBuf, UI lBuf, C_BCP pStr, UI n)
{
	BCP		rc = NULL;
	C_BCP	p;
	UI		svl,len;

	if (pBuf != NULL && pStr != NULL) {
		//	コピーバイト数算出
		for (p = pStr, svl = len = 0; *p != 0 && len < n; ) {
			svl = len;
			if (_ismbblead(*p) && *(p + 1) != 0) len += 2;
			else								 len += 1;
		}
		if (len > n) len = svl;
		//	指定文字数コピー
		if (len < lBuf) {
			memcpy(pBuf, pStr, len);
			pBuf[len] = 0;
			rc = pBuf;
		}
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcStrNCpyW(WCP pBuf, UI lBuf, C_WCP pStr, UI n)
{
	WCP		rc = NULL;
	UI		stl,len;

	if (pBuf != NULL && pStr != NULL) {
		//	コピー文字数算出
		stl = (UI)wcslen(pStr);
		len = __min(n, stl);
		//	指定文字数コピー
		if (len < lBuf) {
			wcsncpy(pBuf, pStr, len);
			rc = pBuf;
		}
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	文字列の末尾に文字列を追加																	//
//																												//
//	Argument	:	pStr1 - 文字列１のアドレス																	//
//					lBuf1 - 文字列１が格納されているバッファのバイト数／文字数									//
//					pStr2 - 文字列２のアドレス																	//
//																												//
//	Return		:	≠ＮＵＬＬ：ＯＫ（＝pStr1）																	//
//					＝ＮＵＬＬ：パラメタエラー（部分文字列は置換されません）									//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcStrCatA(BCP pStr1, UI lBuf, C_BCP pStr2)
{
	BCP		rc = NULL;

	if (pStr1 != NULL && pStr2 != NULL && (strlen(pStr1) + strlen(pStr2)) < lBuf) {
		strcat(pStr1, pStr2);
		rc = pStr1;
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcStrCatW(WCP pStr1, UI lBuf, C_WCP pStr2)
{
	WCP		rc = NULL;

	if (pStr1 != NULL && pStr2 != NULL && (wcslen(pStr1) + wcslen(pStr2)) < lBuf) {
		wcscat(pStr1, pStr2);
		rc = pStr1;
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	前部空白を除去																				//
//																												//
//	Argument	:	pStr - 空白を除去する文字列のアドレス														//
//					pBuf - 空白を除去した文字列を格納するバッファのアドレス										//
//					lBuf - バッファのバイト数／文字数															//
//																												//
//	Return		:	除去後の文字列のアドレス(=pBuf)																//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcStrLTrimA(BCP pStr, BCP pBuf, UI lBuf)
{
	UI		ix = 0;
	UI		len;

	if (pStr != NULL  &&  pBuf != NULL	&&	lBuf > 0) {
		//--- 前部空白スキップ -------------------------------//
		while (pStr[ix] != 0) {
			if		(pStr[ix] == 0x20						 ) ix++;
			else if (pStr[ix] == 0x81 && pStr[ix + 1] == 0x40) ix += 2;
			else break;
		}
		//--- 空白を除去した文字列をバッファに格納 -----------//
		len = (UI)strlen(&pStr[ix]) + 1;
		len = __min(len, lBuf);
		memmove(pBuf, &pStr[ix], len);
		pBuf[lBuf - 1] = 0;
	}
	return pBuf;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcStrLTrimW(WCP pStr, WCP pBuf, UI lBuf)
{
	UI		ix = 0;
	UI		len;

	if (pStr != NULL  &&  pBuf != NULL	&&	lBuf > 0) {
		//--- 前部空白スキップ -------------------------------//
		while (pStr[ix] != 0) {
			if (pStr[ix] == L' ' || pStr[ix] == L'　') ix++;
			else break;
		}
		//--- 空白を除去した文字列をバッファに格納 -----------//
		len = (UI)wcslen(&pStr[ix]) + 1;
		len = __min(len, lBuf);
		memmove(pBuf, &pStr[ix], len * 2);
		pBuf[lBuf - 1] = 0;
	}
	return pBuf;
}

//==============================================================================================================//
//	Function	:	後部の空白を除去																			//
//																												//
//	Argument	:	pStr - 空白を除去する文字列のアドレス														//
//					pBuf - 空白を除去した文字列を格納するバッファのアドレス										//
//					lBuf - バッファのバイト数／文字数															//
//																												//
//	Return		:	除去後の文字列のアドレス(=pBuf)																//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcStrRTrimA(BCP pStr, BCP pBuf, UI lBuf)
{
	UI		ix	= 0;
	UI		len = 0;

	if (pStr != NULL  &&  pBuf != NULL	&&	lBuf > 0) {
		//--- 後部空白の先頭位置検索 -------------------------//
		while (pStr[ix] != 0) {
			if (_ismbblead(pStr[ix]) && _ismbbtrail(pStr[ix + 1])) {
				if (pStr[ix] != 0x81 || pStr[ix + 1] != 0x40) {
					len = ix + 2;
				}
				ix += 2;
			}
			else {
				if (pStr[ix] != 0x20) {
					len = ix + 1;
				}
				ix++;
			}
		}
		//--- 空白を除去した文字列をバッファに格納 -----------//
		len = __min(len, lBuf - 1);
		memmove(pBuf, pStr, len);
		pBuf[len] = 0;
	}
	return pBuf;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcStrRTrimW(WCP pStr, WCP pBuf, UI lBuf)
{
	UI		ix	= 0;
	UI		len = 0;

	if (pStr != NULL  &&  pBuf != NULL	&&	lBuf > 0) {
		//--- 後部空白の先頭位置検索 -------------------------//
		while (pStr[ix] != 0) {
			if (pStr[ix] != L' ' && pStr[ix] != L'　') {
				len = ix + 1;
			}
			ix++;
		}
		//--- 空白を除去した文字列をバッファに格納 -----------//
		len = __min(len, lBuf - 1);
		memmove(pBuf, pStr, len * 2);
		pBuf[len] = 0;
	}
	return pBuf;
}

//==============================================================================================================//
//	Function	:	両端の空白を除去																			//
//																												//
//	Argument	:	pStr - 空白を除去する文字列のアドレス														//
//					pBuf - 空白を除去した文字列を格納するバッファのアドレス（pStrと重複可）						//
//					lBuf - バッファのバイト数／文字数															//
//																												//
//	Return		:	除去後の文字列のアドレス(=pBuf)																//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcStrTrimA (BCP pStr, BCP pBuf, UI lBuf)
{
	if (pStr != NULL  &&  pBuf != NULL	&&	lBuf > 0) {
		AjcStrLTrimA(pStr, pBuf, lBuf);
		AjcStrRTrimA(pBuf, pBuf, lBuf);
	}
	return pBuf;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcStrTrimW (WCP pStr, WCP pBuf, UI lBuf)
{
	if (pStr != NULL  &&  pBuf != NULL	&&	lBuf > 0) {
		AjcStrLTrimW(pStr, pBuf, lBuf);
		AjcStrRTrimW(pBuf, pBuf, lBuf);
	}
	return pBuf;
}
//==============================================================================================================//
//	Function	:	前部の不要な文字群を除去																	//
//																												//
//	Argument	:	pStr   - 文字を除去する文字列のアドレス														//
//					pBuf   - 文字を除去した文字列を格納するバッファのアドレス（pStrと重複可）					//
//					lBuf   - バッファのバイト数／文字数															//
//					pChars - 除去する文字の集合（文字列で指定）													//
//																												//
//	Return		:	除去後の文字列のアドレス(=pBuf)																//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcStrLTrimExA(BCP pStr, BCP pBuf, UI lBuf, C_BCP pChars)
{
	UI		ix = 0;
	UI		len;
	UI		c;

	if (pStr != NULL  &&  pBuf != NULL	&&	lBuf > 0 && pChars != NULL && *pChars != 0) {
		//--- 前部文字群スキップ -------------------------------//
		while (pStr[ix] != 0) {
			if (_ismbblead(pStr[ix]) &&  _ismbbtrail(pStr[ix + 1])) {
				c = ((pStr[ix] << 8) & 0xFF00) | ((pStr[ix + 1]) & 0x00FF);
				if (mbschr(pChars, c) != NULL) ix += 2;
				else						   break;
			}
			else {
				c = (UB)pStr[ix];
				if (mbschr(pChars, c) != NULL) ix++;
				else						   break;
			}
		}
		//--- 文字を除去した文字列をバッファに格納 -----------//
		len = (UI)strlen(&pStr[ix]) + 1;
		len = __min(len, lBuf);
		memmove(pBuf, &pStr[ix], len);
		pBuf[lBuf - 1] = 0;
	}
	return pBuf;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcStrLTrimExW(WCP pStr, WCP pBuf, UI lBuf, C_WCP pChars)
{
	UI		ix = 0;
	UI		len;

	if (pStr != NULL  &&  pBuf != NULL	&&	lBuf > 0 && pChars != NULL && *pChars != 0) {
		//--- 前部文字群スキップ -----------------------------//
		while (pStr[ix] != 0) {
			if (wcschr(pChars, pStr[ix])) ix++;
			else break;
		}
		//--- 文字を除去した文字列をバッファに格納 -----------//
		len = (UI)wcslen(&pStr[ix]) + 1;
		len = __min(len, lBuf);
		memmove(pBuf, &pStr[ix], len * 2);
		pBuf[lBuf - 1] = 0;
	}
	return pBuf;
}

//==============================================================================================================//
//	Function	:	後部の不要な文字群を除去																	//
//																												//
//	Argument	:	pStr   - 空白を除去する文字列のアドレス														//
//					pBuf   - 空白を除去した文字列を格納するバッファのアドレス									//
//					lBuf   - バッファのバイト数／文字数															//
//					pChars - 除去する文字の集合（文字列で指定）													//
//																												//
//	Return		:	除去後の文字列のアドレス(=pBuf)																//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcStrRTrimExA(BCP pStr, BCP pBuf, UI lBuf, C_BCP pChars)
{
	UI		ix	= 0;
	UI		len = 0;
	UI		c;

	if (pStr != NULL  &&  pBuf != NULL	&&	lBuf > 0 && pChars != NULL && *pChars != 0) {
		//--- 後部不要文字群の先頭位置検索 -------------------//
		while (pStr[ix] != 0) {
			if (_ismbblead(pStr[ix]) && _ismbbtrail(pStr[ix + 1])) {
				c = ((pStr[ix] << 8) & 0xFF00) | ((pStr[ix + 1]) & 0x00FF);
				if (mbschr(pChars, c) == NULL) {
					len = ix + 2;
				}
				ix += 2;
			}
			else {
				c = (UB)pStr[ix];
				if (!mbschr(pChars, c)) {
					len = ix + 1;
				}
				ix++;
			}
		}
		//--- 不要文字群を除去した文字列をバッファに格納 -----//
		len = __min(len, lBuf - 1);
		memmove(pBuf, pStr, len);
		pBuf[len] = 0;
	}
	return pBuf;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcStrRTrimExW(WCP pStr, WCP pBuf, UI lBuf, C_WCP pChars)
{
	UI		ix	= 0;
	UI		len = 0;

	if (pStr != NULL  &&  pBuf != NULL	&&	lBuf > 0 && pChars != NULL && *pChars != 0) {
		//--- 後部不要文字群の先頭位置検索 -------------------//
		while (pStr[ix] != 0) {
			if (wcschr(pChars, pStr[ix]) == NULL) {
				len = ix + 1;
			}
			ix++;
		}
		//--- 不要文字群を除去した文字列をバッファに格納 -----//
		len = __min(len, lBuf - 1);
		memmove(pBuf, pStr, len * 2);
		pBuf[len] = 0;
	}
	return pBuf;
}

//==============================================================================================================//
//	Function	:	両端の不要な文字群を除去																	//
//																												//
//	Argument	:	pStr   - 空白を除去する文字列のアドレス														//
//					pBuf   - 空白を除去した文字列を格納するバッファのアドレス									//
//					lBuf   - バッファのバイト数／文字数															//
//					pChars - 除去する文字の集合（文字列で指定）													//
//																												//
//	Return		:	除去後の文字列のアドレス(=pBuf)																//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcStrTrimExA (BCP pStr, BCP pBuf, UI lBuf, C_BCP pChars)
{
	if (pStr != NULL  &&  pBuf != NULL	&&	lBuf > 0 && pChars != NULL && *pChars != 0) {
		AjcStrLTrimExA(pStr, pBuf, lBuf, pChars);
		AjcStrRTrimExA(pBuf, pBuf, lBuf, pChars);
	}
	return pBuf;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcStrTrimExW (WCP pStr, WCP pBuf, UI lBuf, C_WCP pChars)
{
	if (pStr != NULL  &&  pBuf != NULL	&&	lBuf > 0 && pChars != NULL && *pChars != 0) {
		AjcStrLTrimExW(pStr, pBuf, lBuf, pChars);
		AjcStrRTrimExW(pBuf, pBuf, lBuf, pChars);
	}
	return pBuf;
}
//==============================================================================================================//
//	先頭と末尾の不要文字群を削除し、中間の連続した文字群を指定文字列に置き換える								//
//																												//
//	引　数	：	pStr	- 整形前文字列のアドレス																//
//				pBuf	- 整形した文字列を格納するバッファのアドレス											//
//				lBuf	- バッファのバイト数／文字数															//
//				pChars	- 文字群（文字列で指定）																//
//				pRepStr	- 置き変える文字列																		//
//																												//
//	戻り値	：	整形した文字列のアドレス（＝pBuf)																//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcStrShapingA(BCP pStr, BCP pBuf, UI lBuf, C_BCP pChars, C_BCP pRepStr)
{
	int		len 	= 0;
	int		ix		= 0;
	int		ixTop	= 0;
	UI		c;

	if (pStr != NULL  &&  pBuf != NULL	&&	lBuf > 0 && pChars != NULL && *pChars != 0 && pRepStr != NULL) {
		//----- pBufに文字列コピー ---------------------------------------//
		if (pStr != pBuf) {
			AjcSnPrintFA(pBuf, lBuf, "%s", pStr);
		}
		//----- 前後の文字群削除 -----------------------------------------//
		AjcStrTrimExA(pBuf, pBuf, lBuf, pChars);
		//----- 連続した文字群を１つにまとめる ---------------------------//
		while (pBuf[ix] != 0) {
			//	先頭文字チェック
			if (_ismbblead(pBuf[ix]) && _ismbbtrail(pBuf[ix + 1])) {
				c = ((pBuf[ix] << 8) & 0xFF00) | ((pBuf[ix + 1]) & 0x00FF);
				if (mbschr((UBP)pChars, c) != NULL) {
					len   = 2;
					ixTop = ix;
				}
				ix	 += 2;
			}
			else {
				c = (UB)pBuf[ix];
				if (mbschr(pChars, c) != NULL) {
					ixTop = ix;
					len   = 1;
				}
				ix++;
			}
			//	先頭文字が見つかったら終端検索し、文字列置換
			if (len != 0) {
				while (pBuf[ix] != 0) {
					if (_ismbblead(pBuf[ix]) && _ismbbtrail(pBuf[ix + 1])) {
						c = ((pBuf[ix] << 8) & 0xFF00) | ((pBuf[ix + 1]) & 0x00FF);
						if (mbschr((UBP)pChars, c) != NULL) {
							len += 2;
							ix	+= 2;
						}
						else break;
					}
					else {
						c = (UB)pBuf[ix];
						if (mbschr(pChars, c)) {
							len++;
							ix++;
						}
						else break;
					}
				}
				AjcRepPartStrA(pBuf, lBuf, ixTop, len, pRepStr, -1);
				ixTop = 0;
				len   = 0;
			}
		}
	}
	return pBuf;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcStrShapingW(WCP pStr, WCP pBuf, UI lBuf, C_WCP pChars, C_WCP pRepStr)
{
	int		len 	= 0;
	int		ix		= 0;
	int		ixTop	= 0;

	if (pStr != NULL  &&  pBuf != NULL	&&	lBuf > 0 && pChars != NULL && *pChars != 0 && pRepStr != NULL) {
		//----- pBufに文字列コピー ---------------------------------------//
		if (pStr != pBuf) {
			AjcSnPrintFW(pBuf, lBuf, L"%s", pStr);
		}
		//----- 前後の文字群削除 -----------------------------------------//
		AjcStrTrimExW(pBuf, pBuf, lBuf, pChars);
		//----- 連続した文字群を１つにまとめる ---------------------------//
		while (pBuf[ix] != 0) {
			//	先頭文字チェック
			if (wcschr(pChars, pBuf[ix]) != NULL) {
				ixTop = ix;
				len   = 1;
			}
			ix++;
			if (len != 0) {
				//	先頭文字が見つかったら終端検索し、文字列置換
				while (pBuf[ix] != 0) {
					if (wcschr(pChars, pBuf[ix]) != NULL) {
						len++;
						ix++;
					}
					else break;
				}
				AjcRepPartStrW(pBuf, lBuf, ixTop, len, pRepStr, -1);
				ixTop = 0;
				len   = 0;
			}
		}
	}
	return pBuf;
}
//==============================================================================================================//
//	Function	:	文字列中のESCシーケンスを除去																//
//																												//
//	Argument	:	pStr - ESCシーケンスを除去する文字列のアドレス												//
//					pBuf - ESCシーケンスを除去した文字列を格納するバッフアのアドレス							//
//					lBuf - ESCシーケンスを除去した文字列を格納するバッフアのバイト数／文字数					//
//																												//
//	Return		:	ESCシーケンスを除去した文字列を格納したバッファのアドレス（=pBuf）							//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BCP	WINAPI	AjcRemoveEscInStrA(C_BCP pStr, BCP pBuf, UI lBuf)
{
	BCP		rc	= pBuf;
	UI		len = 0;

	if (pBuf != NULL && lBuf != 0) {
		while (*pStr != 0 && len + 1 < lBuf) {
			//	ESCシーケンススキップ
			if (*pStr == '\x1B') {
				pStr++;
				do {
					if (_ismbblead(*pStr) && _ismbbtrail(*(pStr + 1))) pStr += 2;
					else											   pStr++;
				} while (*pStr != 0 && !MAjcAscIsAlphaA(*pStr));
				if (*pStr != 0 && MAjcAscIsAlphaA(*pStr)) {
					pStr++;
				}
			}
			//	テキスト文字格納
			if (*pStr != 0) {
				if (_ismbblead(*pStr) && _ismbbtrail(*(pStr + 1))) {
					if (len + 2 < lBuf) {
						*pBuf++ = *pStr++;
						*pBuf++ = *pStr++;
					}
					len  += 2;
				}
				else {
					*pBuf++ = *pStr++;
					len++;
				}
			}
		}
		*pBuf = 0;
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcRemoveEscInStrW(C_WCP pStr, WCP pBuf, UI lBuf)
{
	WCP		rc	= pBuf;
	UI		len = 0;

	if (pBuf != NULL && lBuf != 0) {
		while (*pStr != 0 && len + 1 < lBuf) {
			//	ESCシーケンススキップ
			if (*pStr == L'\x1B') {
				pStr++;
				do {
					pStr++;
				} while (*pStr != 0 && !MAjcAscIsAlphaW(*pStr));
				if (*pStr != 0 && MAjcAscIsAlphaW(*pStr)) {
					pStr++;
				}
			}
			//	テキスト文字格納
			if (*pStr != 0) {
				*pBuf++ = *pStr++;
				len++;
			}
		}
		*pBuf = 0;
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	マルチバイト(シフトＪＩＳ)文字列情報の取得													//
//																												//
//	Argument	:	pTop - Ｓ－ＪＩＳ文字列のアドレス（文字列終端(0x00)は不要）									//
//					ix	 - マルチバイト情報をチェックする、Ｓ－ＪＩＳ文字列中のバイト位置（０～）				//
//					pBuf - マルチバイト情報を格納するバッファのアドレス											//
//																												//
//	Return		:	TRUE  - ＯＫ																				//
//					FALSE - エラー																				//
//																												//
//	備　考		:	pBufには以下の情報が設定される。															//
//						pBuf->len : 「ix」の位置のバイトが、ｎバイト文字の構成要素であることを示す				//
//						pBuf->ix  : ｎバイト文字中のインデクス（０ ～ ｎ－１）									//
//						例えば、「pBuf->len=2」, 「pBuf->ix=0」である場合、２バイト文字の１バイト目を意味する	//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcStrChkMbcPart (C_BCP pTop, UI ix, PAJCMBCINFO pBuf)
{
	BOOL	rc = FALSE;

	if (pTop != NULL && *pTop != 0 && pBuf != NULL) {
		switch (mbsbtype(pTop, ix)) {
			default:
			case _MBC_SINGLE:	pBuf->len = 1; pBuf->ix = 0; break;
			case _MBC_LEAD:		pBuf->len = 2; pBuf->ix = 0; break;
			case _MBC_TRAIL:	pBuf->len = 2; pBuf->ix = 1; break;
		}
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	ＵＴＦ－８文字列情報の取得																	//
//																												//
//	Argument	:	pTop - ＵＴＦ－８文字列のアドレス（文字列終端(0x00)は不要）									//
//					ix	 - マルチバイト情報をチェックする、ＵＴＦ－８文字列中のバイト位置（０～）				//
//					pBuf - マルチバイト情報を格納するバッファのアドレス											//
//																												//
//	Return		:	TRUE  - ＯＫ																				//
//					FALSE - エラー																				//
//																												//
//	備　考		:	pBufには以下の情報が設定される。															//
//						pBuf->len : 「ix」の位置のバイトが、ｎバイト文字の構成要素であることを示す				//
//						pBuf->ix  : ｎバイト文字中のインデクス（０ ～ ｎ－１）									//
//						例えば、「pBuf->len=2」, 「pBuf->ix=0」である場合、２バイト文字の１バイト目を意味する	//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcStrChkUtf8Part(C_BCP pTop, UI ix, PAJCMBCINFO pBuf)
{
	BOOL	rc = FALSE;
	int		bix;
	UI		fix = 0;
	UI		mbc = 0;
	UB		hNib;

	if (pTop != NULL && pBuf != NULL) {
		//--- マルチバイト文字情報初期化（バイト文字を仮定） --//
		pBuf->len = 1;
		pBuf->ix  = 0;

		//--- 末尾のASCII文字／マルチバイト先頭検索 --//
		for (bix = ix; bix >= 0; bix--) {
			hNib = (UB)(pTop[bix] & 0xF0);

			if		((UB)pTop[bix] <= 0x7F						   ) {fix = bix; mbc = 1; break;}
			else if (hNib == 0xC0 || hNib == 0xD0				   ) {fix = bix; mbc = 2; break;}
			else if (hNib == 0xE0								   ) {fix = bix; mbc = 3; break;}
			else if ((UB)pTop[bix] >= 0xF0 && (UB)pTop[bix] <= 0xF7) {fix = bix; mbc = 4; break;}
			else if ((UB)pTop[bix] >= 0xF8 && (UB)pTop[bix] <= 0xFB) {fix = bix; mbc = 5; break;}
			else if ((UB)pTop[bix] >= 0xFC && (UB)pTop[bix] <= 0xFD) {fix = bix; mbc = 6; break;}
		}
		//--- マルチバイト文字追従 -------------------//
		while (fix <= ix) {
			hNib = (pTop[fix] & 0xF0);

			if		((UB)pTop[fix] <= 0x7F					   ) {		//	１バイト文字？
				if		(fix + 0 == ix) {pBuf->len = 1; pBuf->ix = 0; rc = TRUE; break;}
				fix++;
			}
			else if (hNib == 0xC0 || hNib == 0xD0		   ) {			//	２バイト文字？
				if		(fix + 0 == ix) {pBuf->len = 2; pBuf->ix = 0; rc = TRUE; break;}
				if		(fix + 1 == ix) {pBuf->len = 2; pBuf->ix = 1; rc = TRUE; break;}
				fix += 2;
			}
			else if (hNib == 0xE0						   ) {			//	３バイト文字？
				if		(fix + 0 == ix) {pBuf->len = 3; pBuf->ix = 0; rc = TRUE; break;}
				if		(fix + 1 == ix) {pBuf->len = 3; pBuf->ix = 1; rc = TRUE; break;}
				if		(fix + 2 == ix) {pBuf->len = 3; pBuf->ix = 2; rc = TRUE; break;}
				fix += 3;
			}
			else if ((UB)pTop[fix] >= 0xF0 && (UB)pTop[fix] <= 0xF7) {	//	４バイト文字？
				if		(fix + 0 == ix) {pBuf->len = 4; pBuf->ix = 0; rc = TRUE; break;}
				if		(fix + 1 == ix) {pBuf->len = 4; pBuf->ix = 1; rc = TRUE; break;}
				if		(fix + 2 == ix) {pBuf->len = 4; pBuf->ix = 2; rc = TRUE; break;}
				if		(fix + 3 == ix) {pBuf->len = 4; pBuf->ix = 3; rc = TRUE; break;}
				fix += 4;
			}
			else if ((UB)pTop[fix] >= 0xF8 && (UB)pTop[fix] <= 0xFB) {	//	５バイト文字？
				if		(fix + 0 == ix) {pBuf->len = 5; pBuf->ix = 0; rc = TRUE; break;}
				if		(fix + 1 == ix) {pBuf->len = 5; pBuf->ix = 1; rc = TRUE; break;}
				if		(fix + 2 == ix) {pBuf->len = 5; pBuf->ix = 2; rc = TRUE; break;}
				if		(fix + 3 == ix) {pBuf->len = 5; pBuf->ix = 3; rc = TRUE; break;}
				if		(fix + 4 == ix) {pBuf->len = 5; pBuf->ix = 4; rc = TRUE; break;}
				fix += 5;
			}
			else if ((UB)pTop[fix] >= 0xFC && (UB)pTop[fix] <= 0xFD) {	//	６バイト文字？
				if		(fix + 0 == ix) {pBuf->len = 6; pBuf->ix = 0; rc = TRUE; break;}
				if		(fix + 1 == ix) {pBuf->len = 6; pBuf->ix = 1; rc = TRUE; break;}
				if		(fix + 2 == ix) {pBuf->len = 6; pBuf->ix = 2; rc = TRUE; break;}
				if		(fix + 3 == ix) {pBuf->len = 6; pBuf->ix = 3; rc = TRUE; break;}
				if		(fix + 4 == ix) {pBuf->len = 6; pBuf->ix = 4; rc = TRUE; break;}
				if		(fix + 5 == ix) {pBuf->len = 6; pBuf->ix = 5; rc = TRUE; break;}
				fix += 6;
			}
			else {
				fix++;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	日本語ＥＵＣ文字列情報の取得																//
//												（文字列終端(0x00)は不要）										//
//	Argument	:	pTop - ＥＵＣ文字列のアドレス																//
//					ix	 - マルチバイト情報をチェックする、ＥＵＣ文字列中のバイト位置（０～）					//
//					pBuf - マルチバイト情報を格納するバッファのアドレス											//
//																												//
//	Return		:	TRUE  - ＯＫ																				//
//					FALSE - エラー																				//
//																												//
//	備　考		:	pBufには以下の情報が設定される。															//
//						pBuf->len : 「ix」の位置のバイトが、ｎバイト文字の構成要素であることを示す				//
//						pBuf->ix  : ｎバイト文字中のインデクス（０ ～ ｎ－１）									//
//						例えば、「pBuf->len=2」, 「pBuf->ix=0」である場合、２バイト文字の１バイト目を意味する	//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcStrChkEucPart (C_BCP pTop, UI ix, PAJCMBCINFO pBuf)
{
	BOOL	rc = FALSE;
	int		bix;
	UI		fix = 0;

	if (pTop != NULL && pBuf != NULL) {
		//--- マルチバイト文字情報初期化（バイト文字を仮定） --//
		pBuf->len = 1;
		pBuf->ix  = 0;

		//--- 末尾のASCII文字検索 -------------------------//
		for (bix = ix; bix >= 0; bix--) {
			if ((UB)pTop[bix] <= 0x7F) {
				fix = bix;
				break;
			}
		}
		//--- マルチバイト文字追従 ------------------------//
		while (fix <= ix) {
			if		(((UB)pTop[fix] >= 0xA1 && (UB)pTop[fix] <= 0xFE) || pTop[fix] == 0x8E) {	//	２バイト文字
				if		(fix + 0 == ix) {pBuf->len = 2; pBuf->ix = 0; break;}
				else if (fix + 1 == ix) {pBuf->len = 2; pBuf->ix = 1; break;}
				fix += 2;
			}
			else if ((UB)pTop[fix] == 0x8F) {													//	３バイト文字
				if		(fix + 0 == ix) {pBuf->len = 3; pBuf->ix = 0; break;}
				else if (fix + 1 == ix) {pBuf->len = 3; pBuf->ix = 1; break;}
				else if (fix + 2 == ix) {pBuf->len = 3; pBuf->ix = 2; break;}
				fix += 3;
			}
			else {																				//	１バイト文字（上記以外）
				if		(fix + 0 == ix) {pBuf->len = 1; pBuf->ix = 0; break;}
				fix++;
			}
		}
		//--- 戻り値設定 ----------------------------------//
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	マルチバイト(シフトＪＩＳ)文字のバイト数取得												//
//																												//
//	Argument	:	pTop - Ｓ－ＪＩＳ文字の先頭アドレス															//
//					len	 - Ｓ－ＪＩＳ文字列のバイト数（pTopから末尾までのバイト数，－１指定時は自動算出）		//
//																												//
//	Return		:	１～２	- マルチバイト(シフトＪＩＳ)文字のバイト数											//
//					０		- エラー（判定不可）																//
//==============================================================================================================//
AJCEXPORT	UI	WINAPI	AjcStrChkMbcLen (C_BCP pTop, UI len)
{
	UI		rc = 0;

	if (pTop != NULL && *pTop != 0) {
		len = AjcStrAdjustLenA(pTop, len);
		if (len >= 1) {
			if (MAjcIsLeadA(*pTop)) {		//	２バイト文字？
				if (len >= 2 && MAjcIsTrailA(*(pTop + 1))) rc = 2;
			} 
			else {							//	１バイト文字？
				rc = 1;
			}
		}
	}
	return rc;
}

//==============================================================================================================//
//	Function	:	マルチバイト(ＵＴＦ－８)文字のバイト数取得													//
//																												//
//	Argument	:	pTop - ＵＴＦ－８文字の先頭アドレス															//
//					len	 - ＵＴＦ－８文字列のバイト数（pTopから末尾までのバイト数，－１指定時は自動算出）		//
//																												//
//	Return		:	１～６	- マルチバイト(ＵＴＦ－８)文字のバイト数											//
//					０		- エラー（判定不可）																//
//==============================================================================================================//
AJCEXPORT	UI	WINAPI	AjcStrChkUtf8Len (C_BCP pTop, UI len)
{
	UI		rc = 0;
	UI		hNib;
	BC		c;

	if (pTop != NULL && *pTop != 0) {
		len = AjcStrAdjustLenA(pTop, len);
		if (len >= 1) {
			c = *pTop;
			hNib = (c & 0xF0);									//	上位ニブル値設定
			if		(hNib == 0xC0 || hNib == 0xD0) {			//	２バイト文字？
				if (len >= 2) rc = 2;
			}
			else if (hNib == 0xE0				   ) {			//	３バイト文字？
				if (len >= 3) rc = 3;
			}
			else if (c >= 0xF0 && c <= 0xF7 	   ) {			//	４バイト文字？
				if (len >= 4) rc = 4;
			}
			else if (c >= 0xF8 && c <= 0xFB 	   ) {			//	５バイト文字？
				if (len >= 5) rc = 5;
			}
			else if (c >= 0xFC && c <= 0xFD 	   ) {			//	６バイト文字？
				if (len >= 6) rc = 6;
			}
			else {												//	その他（１バイト文字）
				rc = 1;
			}
		}
	}
	return rc;
}

//==============================================================================================================//
//	Function	:	マルチバイト(ＥＵＣ)文字のバイト数取得														//
//																												//
//	Argument	:	pTop - ＥＵＣ文字の先頭アドレス																//
//					len	 - ＥＵＣ文字列のバイト数（pTopから末尾までのバイト数，－１指定時は自動算出）			//
//																												//
//	Return		:	１～３	- マルチバイト(シフトＪＩＳ)文字のバイト数											//
//					０		- エラー（判定不可）																//
//==============================================================================================================//
AJCEXPORT	UI	WINAPI	AjcStrChkEucLen (C_BCP pTop, UI len)
{
	UI		rc = 0;
	BC		c;

	if (pTop != NULL && *pTop != 0) {
		len = AjcStrAdjustLenA(pTop, len);
		if (len >= 1) {
			c = (UB)*pTop;
			if		((c >= 0xA1 && c <= 0xFE) || c == 0x8E) {		//	２バイト文字
				if (len >= 2) rc = 2;
			}
			else if (c == 0x8F) {									//	３バイト文字
				if (len >= 3) rc = 3;

			}
			else {													//	１バイト文字（上記以外）
				rc = 1;
			}
		}
	}
	return rc;
}

//==============================================================================================================//
//	Function	:	ワイド文字（リトルエンディアン）のワード数取得												//
//																												//
//	Argument	:	pTop - ワイド文字の先頭アドレス																//
//					len	 - ワイド文字列の文字数（pTopから末尾までの文字数，－１指定時は自動算出）				//
//																												//
//	Return		:	１～２	- ワイド文字（リトルエンディアン）のワード数										//
//					０		- エラー（判定不可）																//
//==============================================================================================================//
AJCEXPORT	UI	WINAPI	AjcStrChkU16LeLen (C_WCP pTop, UI len)
{
	UI		rc = 0;

	if (pTop != NULL && *pTop != 0) {
		len = AjcStrAdjustLenW(pTop, len);
		if (len >= 1) {
			if (MAjcIsLeadW(*pTop)) {								//	上位サロゲート？
				if (len >= 2 && MAjcIsTrailW(*(pTop + 1))) rc = 2;
			}
			else {													//	非サロゲート？
				rc = 1;
			}
		}
	}
	return rc;
}

//==============================================================================================================//
//	Function	:	ワイド文字（ビッグエンディアン）のワード数取得												//
//																												//
//	Argument	:	pTop - ワイド文字の先頭アドレス																//
//					len	 - ワイド文字列の文字数（pTopから末尾までの文字数，－１指定時は自動算出）				//
//																												//
//	Return		:	１～２	- ワイド文字（ビッグエンディアン）のワード数										//
//					０		- エラー（判定不可）																//
//==============================================================================================================//
AJCEXPORT	UI	WINAPI	AjcStrChkU16BeLen (C_WCP pTop, UI len)
{
	UI		rc = 0;

	if (pTop != NULL && *pTop != 0) {
		len = AjcStrAdjustLenW(pTop, len);
		if (len >= 1) {
			if (MAjcIsLeadW((WC)AjcExcWord((*pTop)))) {				//	上位サロゲート？
				if (len >= 2 && MAjcIsTrailW((WC)AjcExcWord((*(pTop + 1))))) rc = 2;
			}
			else {													//	非サロゲート？
				rc = 1;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	文字列長補正																				//
//																												//
//	Argument	:	pTxt - 文字列のアドレス																		//
//					len	 - ワイド文字列のバイト数（pTopから末尾までの文字数，－１指定時は自動算出）				//
//																												//
//	Return		:	１～２	- ワイド文字（ビッグエンディアン）のワード数										//
//					０		- エラー（判定不可）																//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcStrAdjustLenA(C_BCP pTxt, UI len)
{
	UI		rc = 0;

	if (pTxt != NULL) {
		if (len == -1) rc = (UI)strlen (pTxt);
		else		   rc = (UI)strnlen(pTxt, len);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcStrAdjustLenW(C_WCP pTxt, UI len)
{
	UI		rc = 0;

	if (pTxt != NULL) {
		if (len == -1) rc = (UI)wcslen (pTxt);
		else		   rc = (UI)wcsnlen(pTxt, len);
	}
	return rc;
}

/*==============================================================================================================*/
/*	Function	:	ＵＴＦ－８ → マルチバイト（シフトＪＩＳ）文字列変換										*/
/*																												*/
/*	Argument	:	pUtf8  - 変換するUTF-8コード文字列のアドレス												*/
/*					pMbc   - 変換したマルチバイト(シフトＪＩＳ)文字列を格納するバッファのアドレス(不要時はNULL)	*/
/*					lMbc   - 変換したマルチバイト(シフトＪＩＳ)文字列を格納するバッファのバイト数(不要時は０)	*/
/*																												*/
/*	Return		:	変換に必要なバッファサイズ（文字列終端(0x00)を含む）										*/
/*																												*/
/*	※　pMbc=NULL とした場合は、変換に必要なバッファサイズを返します。											*/
/*	※　出力バッファサイズが小さい場合でも、戻り値はＯＫを返します。											*/
/*		この場合、戻り値はバッファに出力したバイト数ではなく、変換に必要なバイト数を返します。					*/
/*==============================================================================================================*/
AJCEXPORT	int		WINAPI	AjcUtf8ToMbc(C_BCP pUtf8, BCP pMbc, UI lMbc)
{
	return AjcUtf8ToMbcEx(pUtf8, -1, pMbc, lMbc);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	int		WINAPI	AjcUtf8ToMbcEx(C_BCP pUtf8, UI lUtf8, BCP pMbc, UI lMbc)
{
	int		lBuf = 0;
	UI		lWid;
	WCP		pWid;

	if (pUtf8 != NULL) {
		//----- ＢＯＭスキップ ---------------------------------------------------//
		if (strlen(pUtf8) >= 3 && memcmp(pUtf8, "\xEF\xBB\xBF", 3) == 0) {
			pUtf8 += 3;
		}
		//----- 文字列変換 -------------------------------------------------------//
		lWid = MultiByteToWideChar(CP_UTF8, 0, pUtf8, lUtf8, NULL, 0);
		if (pWid = (WCP)AJCMEM(lWid * 2)) {
			MultiByteToWideChar(CP_UTF8, 0, pUtf8, lUtf8, pWid, lWid);
			lBuf = WideCharToMultiByte(CP_ACP, 0, pWid, lWid, NULL, 0, NULL, NULL);
			if (pMbc != NULL && lMbc != 0) {
				WideCharToMultiByte(CP_ACP, 0, pWid, lWid, pMbc, lMbc, NULL, NULL);
			}
			free(pWid);
		}
	}
	return lBuf;
}
/*==============================================================================================================*/
/*	Function	:	マルチバイト（シフトＪＩＳ） → ＵＴＦ８文字列変換											*/
/*																												*/
/*	Argument	:	pMbc   - マルチバイト（シフトＪＩＳ）コード文字列のアドレス									*/
/*					pUtf8  - 変換したＵＴＦ－８文字列を格納するバッファのアドレス（不要時はＮＵＬＬ）			*/
/*					lUtf8  - 変換したＵＴＦ－８文字列を格納するバッファのバイト数（不要時は０）					*/
/*																												*/
/*	Return		:	変換に必要なバッファサイズ（文字列終端(0x00)を含む）										*/
/*																												*/
/*	※　pUtf8=NULL とした場合は、変換に必要なバッファサイズを返します。											*/
/*	※　出力バッファサイズが小さい場合でも、戻り値はＯＫを返します。											*/
/*		この場合、戻り値はバッファに出力したバイト数ではなく、変換に必要なバイト数を返します。					*/
/*==============================================================================================================*/
AJCEXPORT	int		WINAPI	AjcMbcToUtf8(C_BCP pMbc, BCP pUtf8, UI lUtf8)
{
	return AjcMbcToUtf8Ex(pMbc, -1, pUtf8, lUtf8);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	int		WINAPI	AjcMbcToUtf8Ex(C_BCP pMbc, UI lMbc, BCP pUtf8, UI lUtf8)
{
	int		lBuf = 0;
	UI		lWid;
	WCP		pWid;

	if (pMbc != NULL) {
		lWid = MultiByteToWideChar(CP_ACP, 0, pMbc, lMbc, NULL, 0);
		if (pWid = (WCP)AJCMEM(lWid * 2)) {
			MultiByteToWideChar(CP_ACP, 0, pMbc, lMbc, pWid, lWid);
			lBuf = WideCharToMultiByte(CP_UTF8, 0, pWid, lWid, NULL, 0, NULL, NULL);
			if (pUtf8 != NULL && lUtf8 != 0) {
				WideCharToMultiByte(CP_UTF8, 0, pWid, lWid, pUtf8, lUtf8, NULL, NULL);
			}
			free(pWid);
		}
	}
	return lBuf;
}
/*==============================================================================================================*/
/*	Function	:	日本語ＥＵＣ → シフトＪＩＳ文字列変換														*/
/*																												*/
/*	Argument	:	pEuc   - 変換するＥＵＣコード文字列のアドレス												*/
/*					pSJis  - 変換したシフトＪＩＳ文字列を格納するバッファのアドレス（不要時はＮＵＬＬ）			*/
/*					lSJis  - 変換したシフトＪＩＳ文字列を格納するバッファのバイト数（不要時は０）				*/
/*																												*/
/*	Return		:	変換に必要なバッファサイズ（文字列終端(0x00)を含む）										*/
/*																												*/
/*	※　pSJis=NULL とした場合は、変換に必要なバッファサイズを返します。											*/
/*	※　出力バッファサイズが小さい場合でも、戻り値はＯＫを返します。											*/
/*		この場合、戻り値はバッファに出力したバイト数ではなく、変換に必要なバイト数を返します。					*/
/*==============================================================================================================*/

#define	SET_SJBUF(_C_)	if (lSJis >= 1) {	\
							*pSJis++ = _C_;	\
							lSJis--;		\
						}					\
						rc++
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	int		WINAPI	AjcEucToSJis(C_BCP pEuc , BCP pSJis, UI lSJis)
{
	return AjcEucToSJisEx(pEuc, -1, pSJis, lSJis);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	int		WINAPI	AjcEucToSJisEx(C_BCP pEuc , UI lEuc, BCP pSJis, UI lSJis)
{
	int		rc	= 0;
	UI		n;
	UB		h, l;

	if (pEuc != NULL) {

		if (lEuc  == -1  ) {rc = 1;}													/*	終端付き文字列ならば終端文字分初期化*/
		lEuc = AjcStrAdjustLenA(pEuc, lEuc);											/*	EUC文字列長補正						*/
		if (pSJis == NULL) {lSJis = 0;}													/*	出力ﾊﾞｯﾌｧなし時、ﾊﾞｯﾌｧ長=0とする	*/

		while (*pEuc != 0 && lEuc >= 1) {												/*	入力ＥＵＣコードデータループ		*/
			h = *pEuc;																	/*		ｈ＝上位バイト					*/
			if (lEuc >= 2) l = *(pEuc + 1);												/*		ｌ＝下位バイト					*/
			else		   l = 0;														/*		・								*/
			if		(h <= 0x7E) {														/*		ＡＳＣＩＩコード？				*/
				SET_SJBUF(h);															/*			出力データ＝入力データ		*/
				pEuc++;																	/*			入力データポインタ更新		*/
				lEuc--;																	/*			残入力文字列長更新			*/
			}
			else if (h == EUC_KANA && l >= KANA_L && l <= KANA_H) {						/*		カタカナ？						*/
				SET_SJBUF(l);															/*			出力データ＝下位バイト		*/
				pEuc += 2;																/*			入力データポインタ更新		*/
				lEuc -= 2;																/*			残入力文字列長更新			*/
			}
			else if (h >= EUC_L && h <= EUC_H && l >= EUC_L && l <= EUC_H) {			/*		全角コード？					*/
				n = ((h - EUC_L) * EUC_NUM) + (l - EUC_L);								/*			コード番号算出				*/
				if (n < (SJH1_NUM * SJL_NUM)) {											/*			SJIS上位=81～9F範囲内？		*/
					SET_SJBUF((BC)(SJH1_L + (n / (SJL_NUM))));							/*				出力上位バイト設定		*/
				}
				else {																	/*			SJIS上位=E0～FC範囲内？		*/
					SET_SJBUF((BC)(SJH2_L + (n / (SJL_NUM) - SJH1_NUM)));				/*				出力上位バイト設定		*/
				}
				n %= SJL_NUM;															/*			下位バイト分のコード番号算出*/
				if (n < SJL1_NUM) {														/*			SJIS下位=40～7E範囲内？		*/
					SET_SJBUF((BC)(SJL1_L + n));										/*				出力下位バイト設定		*/
				}
				else {																	/*			SJIS下位=80～FC範囲内？		*/
					SET_SJBUF((BC)(SJL1_L + n + 1));									/*				出力下位バイト設定		*/
				}
				pEuc += 2;																/*			入力データポインタ更新		*/
				lEuc -= 2;																/*			残入力文字列長更新			*/
			}
			else {																		/*		不当なＥＵＣコード？			*/
				SET_SJBUF('?');															/*			（文字スキップ）			*/
				pEuc++;																	/*			入力データポインタ更新		*/
				lEuc--;																	/*			残入力文字列長更新			*/
			}
		}

		if (lSJis >= 1) {																/*	出力バッファに空あり？				*/
			*pSJis = 0;																	/*		文字列終端設定					*/
			lSJis--;																	/*		出力バッファサイズ減算			*/
		}
	}
	return rc;
}
/*==============================================================================================================*/
/*	Function	:	シフトＪＩＳ → 日本語ＥＵＣ文字列変換														*/
/*																												*/
/*	Argument	:	pSJis  - 変換するシフトＪＩＳコード文字列のアドレス											*/
/*					pEuc   - 変換したＥＵＣ文字列を格納するバッファのアドレス（不要時はＮＵＬＬ）				*/
/*					lEuc   - 変換したＥＵＣ文字列を格納するバッファのバイト数（不要時は０）						*/
/*																												*/
/*	Return		:	変換に必要なバッファサイズ（文字列終端(0x00)を含む）										*/
/*																												*/
/*	※　pEuc=NULL とした場合は、変換に必要なバッファサイズを返します。											*/
/*	※　出力バッファサイズが小さい場合でも、戻り値はＯＫを返します。											*/
/*		この場合、戻り値はバッファに出力したバイト数ではなく、変換に必要なバイト数を返します。					*/
/*==============================================================================================================*/

#define	SET_EUBUF(_C_)	if (lEuc >= 1) {	\
							*pEuc++ = _C_;	\
							lEuc--;			\
						}					\
						rc++
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	int		WINAPI	AjcSJisToEuc(C_BCP pSJis, BCP pEuc , UI lEuc )
{
	return AjcSJisToEucEx(pSJis, -1, pEuc, lEuc);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	int		WINAPI	AjcSJisToEucEx(C_BCP pSJis, UI lSJis, BCP pEuc , UI lEuc )
{
	int		rc	= 0;
	UI		n;
	UB		h, l;

	if (pSJis != NULL) {

		if (lSJis == -1  ) {rc = 1;}													/*	終端付き文字列ならば終端文字分初期化*/
		lSJis = AjcStrAdjustLenA(pSJis, lSJis);											/*	SJIS文字列長補正					*/
		if (pEuc  == NULL) {lEuc = 0;}													/*	出力ﾊﾞｯﾌｧなし時、ﾊﾞｯﾌｧ長=0とする	*/

		while (*pSJis != 0 && lSJis >= 1) {												/*	入力ｼﾌﾄJISコードデータループ		*/
			h = (UB)*pSJis;																/*		ｈ＝上位バイト					*/
			if (lSJis >= 2) l = (UB)*(pSJis + 1);										/*		ｌ＝下位バイト					*/
			else			l = 0;														/*		・								*/
			if		(h <= 0x7E) {														/*		ＡＳＣＩＩコード？				*/
				SET_EUBUF(h);															/*			出力データ＝入力データ		*/
				pSJis++;																/*			入力データポインタ更新		*/
				lSJis--;																/*			残入力文字列長更新			*/
			}
			else if (h >= KANA_L && h <= KANA_H) {										/*		カタカナ？						*/
				SET_EUBUF(EUC_KANA);													/*			出力上位バイト設定			*/
				SET_EUBUF(h);															/*			出力データ＝下位バイト		*/
				pSJis++;																/*			入力データポインタ更新		*/
				lSJis--;																/*			残入力文字列長更新			*/
			}
			else if (((h >= SJH1_L && h <= SJH1_H) || (h >= SJH2_L && h <= SJH2_H)) &&	/*		全角コード？					*/
					 ((l >= SJL1_L && l <= SJL1_H) || (l >= SJL2_L && l <= SJL2_H))) {	/*		・								*/
				if (h < 0xF0) {															/*			変換可能コード？			*/
					if (h <= SJH1_H) n	= ( 		   h - SJH1_L ) * SJL_NUM;			/*				コード番号算出			*/
					else			 n	= (SJH1_NUM + (h - SJH2_L)) * SJL_NUM;			/*				・						*/
					if (l <= SJL1_H) n +=			  (l - SJL1_L);						/*				・						*/
					else			 n += (SJL1_NUM + (l - SJL2_L));					/*				・						*/
					SET_EUBUF((BC)(EUC_L + (n / EUC_NUM)));								/*				出力上位バイト設定		*/
					SET_EUBUF((BC)(EUC_L + (n % EUC_NUM)));								/*				出力下位バイト設定		*/
				}
				else {																	/*			変換不能コード？			*/
					SET_EUBUF(EUC_H);													/*				FEFE設定				*/
					SET_EUBUF(EUC_H);													/*				・						*/
				}
				pSJis += 2;																/*			入力データポインタ更新		*/
				lSJis -= 2;																/*			残入力文字列長更新			*/
			}
			else {																		/*		不当なｼﾌﾄJISコード？			*/
				SET_EUBUF('?');															/*			（文字スキップ）			*/
				pSJis++;																/*			入力データポインタ更新		*/
				lSJis--;																/*			残入力文字列長更新			*/
			}
		}

		if (lEuc >= 1) {																/*	出力バッファに空あり？				*/
			*pEuc = 0;																	/*		文字列終端設定					*/
			lEuc--;																		/*		出力バッファサイズ減算			*/
		}
	}
	return rc;
}

/*==============================================================================================================*/
/*	Function	:	文字列のコードチェック																		*/
/*																												*/
/*	Argument	:	pStr  - コードをチェックする文字列のアドレス												*/
/*																												*/
/*	Return		:	≠-1：文字列のコード種別（AJCMBC_SJIS / _EUC / _UTF8)										*/
/*					＝-1：判別不能																				*/
/*==============================================================================================================*/
AJCEXPORT	AJCMBCKIND		WINAPI	AjcStrChkCode(C_BCP pStr)
{
	AJCMBCKIND	rc = -1;
	int			i;
	int			MaxPer = 0;
	struct	{int per; AJCMBCKIND k;} s[4];

	memset(s, 0, sizeof s);

	if (pStr != NULL && strlen(pStr) >= 8) {
		s[0].per = AjcStrIsSJis(pStr);	s[0].k = AJCMBC_SJIS;
		s[1].per = AjcStrIsEuc (pStr);	s[1].k = AJCMBC_EUC;
		s[2].per = AjcStrIsUtf8(pStr);	s[2].k = AJCMBC_UTF8;
		for (i = 0; i < 4; i++) {
			if (s[i].per > MaxPer) {
				MaxPer = s[i].per;
				rc	   = s[i].k;
			}
		}
	}
	return rc;
}
/*==============================================================================================================*/
/*	Function	:	文字列のコードチェック（拡張版）															*/
/*																												*/
/*	Argument	:	pStr  - コードをチェックする文字列のアドレス												*/
/*					lStr  - コードをチェックする文字列のバイト数（-1：自動）									*/
/*																												*/
/*	Return		:	≠-1：文字列のコード種別（AJCMBC_SJIS / _EUC / _UTF8)										*/
/*					＝-1：判別不能																				*/
/*==============================================================================================================*/
AJCEXPORT	EAJCTEC		WINAPI	AjcStrChkCodeEx(C_BCP pStr, UI lStr)
{
	EAJCTEC rc	 = (EAJCTEC)-1;
	BCP		pBuf = NULL;
	int		perSJis = 0;
	int		perEuc	= 0;
	int		perUtf8 = 0;
	int		lpi 	= 0;

	if (pStr != NULL && lStr >= 8) {
		do {
			//	●UNICODEチェック
			//		UTF-16LE
			lpi = IS_TEXT_UNICODE_CONTROLS;
			IsTextUnicode(pStr, lStr, &lpi);
			if (lpi == IS_TEXT_UNICODE_CONTROLS) {
				rc = AJCTEC_UTF_16LE;
				break;
			}
			//		UTF-16BE
			lpi = IS_TEXT_UNICODE_REVERSE_CONTROLS;
			IsTextUnicode(pStr, lStr, &lpi);
			if (lpi == IS_TEXT_UNICODE_REVERSE_CONTROLS) {
				rc = AJCTEC_UTF_16BE;
				break;
			}
			//	●MBC(SJIS), EUC-J or UTF-8 チェック
			perSJis = AjcStrIsSJisEx(pStr, lStr);
			perEuc	= AjcStrIsEucEx (pStr, lStr);
			perUtf8 = AjcStrIsUtf8Ex(pStr, lStr);
			if		(perUtf8 > perSJis &&  perUtf8 > perEuc ) {
				rc = AJCTEC_UTF_8;
			}
			else if (perEuc  > perSJis &&  perEuc  > perUtf8) {
				rc = AJCTEC_EUC_J;
			}
			else {
				rc = AJCTEC_MBC;
			}
		} while(0);
	}
	return rc;
}
/*==============================================================================================================*/
/*	Function	:	ＪＩＳ文字列チェック																		*/
/*																												*/
/*	Argument	:	pStr  - JISコードであるかチェックする文字列のアドレス										*/
/*																												*/
/*	Return		:	JISコードである確率[%]																		*/
/*==============================================================================================================*/
AJCEXPORT	int		WINAPI	AjcStrIsJis (C_BCP pStr)
{
	int		rc = 0;

	if (pStr != NULL) {
		if (mbsstr(pStr, "\x1B\x28\x42") != NULL	||		//	ASCII			\x1B(B	ASCII開始
			mbsstr(pStr, "\x1B\x28\x4A") != NULL	||		//	JISローマ字		\x1B(J	JISローマ字の開始
			mbsstr(pStr, "\x1B\x28\x49") != NULL	||		//	JISカナ字		\x1B(I	半角カタカナの開始
			mbsstr(pStr, "\x1B\x24\x40") != NULL	||		//	旧JIS漢字		\x1B$@	漢字の開始(旧JIS漢字 JIS C 6226-1978)
			mbsstr(pStr, "\x1B\x24\x42") != NULL	||		//	新JIS漢字		\x1B$B	漢字の開始(新JIS漢字 JIS X 0208-1983 / 1990)
			mbsstr(pStr, "\x1B\x26\x40") != NULL	||		//	新JIS漢字		\x1B&@	漢字の開始(新JIS漢字 JIS X 0208-1990)
			mbsstr(pStr, "\x1B\x24\x44")) {					//	JIS補助漢字		\x1B$D	
			rc = 100;
		}
	}
	return rc;
}
/*==============================================================================================================*/
/*	Function	:	Ｓ－ＪＩＳ文字列チェック																	*/
/*																												*/
/*	Argument	:	pStr  - S-JISコードであるかチェックする文字列のアドレス										*/
/*					lStr  - 文字列の長さ（-1：自動）															*/
/*																												*/
/*	Return		:	S-JISコードである確率[%]																	*/
/*==============================================================================================================*/
AJCEXPORT	int		WINAPI	AjcStrIsSJis(C_BCP pStr)
{
	return AjcStrIsSJisEx(pStr, -1);
}
/*--------------------------------------------------------------------------------------------------------------*/
AJCEXPORT	int		WINAPI	AjcStrIsSJisEx(C_BCP pStr, UI lStr)
{
	int		rc	= 0;
	UI		len = 0;
	C_UBP	p, q;

	if (pStr != NULL) {
		if (lStr == -1) {
			lStr = (UI)strlen(pStr);
		}
		if (lStr > 0) {
			p = (C_UBP)pStr;
			while (len < lStr && *p != 0) {
				q = p + 1;
				if ((len + 1 < lStr) && _ismbblead(*p) && _ismbbtrail(*q)) {	//	●正常なS-JISマルチバイト文字
					if ((*p >= EUC_L && *p <= EUC_H) && (*q >= EUC_L && *q <= EUC_H)) {		//	ＥＵＣと重複する？
						p += 2; len += 2;
						rc += 3;
					}
					else if ((*p >= 0x80 && *p <= 0xBF) && (*q >= 0x80 && *q <= 0xBF)) {	//	UTF-8と重複する？
						p += 2;	len += 2;
						rc += 3;
					}
					else {																	//	重複しない
						p += 2;	len += 2;
						rc += 4;
					}
				}
				else if (*p >= KANA_L && *p <= KANA_H) {				//	●カナ文字
					p++;	len++;
					rc += 1;
				}
				else if (*p >= 0x00 && *p <= 0x7F) {					//	●ASCII文字
					p++;	len++;
					rc += 1;
				}
				else {													//	●その他
					p++;	len++;
				}
			}
			if (len != 0) {
				rc = (int)((ULL)rc * 100 / ((ULL)len * 2));
			}
			else rc = 0;
		}
		else rc = 0;
	}
	return rc;
}
/*==============================================================================================================*/
/*	Function	:	ＥＵＣ文字列チェック																		*/
/*																												*/
/*	Argument	:	pStr  - ＥＵＣコードであるかチェックする文字列のアドレス									*/
/*					lStr  - 文字列の長さ（-1：自動）															*/
/*																												*/
/*	Return		:	ＥＵＣコードである確率[%]																	*/
/*==============================================================================================================*/
AJCEXPORT	int		WINAPI	AjcStrIsEuc (C_BCP pStr)
{
	return AjcStrIsEucEx (pStr, -1);
}
/*--------------------------------------------------------------------------------------------------------------*/
AJCEXPORT	int		WINAPI	AjcStrIsEucEx (C_BCP pStr, UI lStr)
{
	int		rc	= 0;
	UI		len = 0;
	UI		i;
	C_UBP	p, q;

	if (pStr != NULL) {
		if (lStr == -1) {
			lStr = (UI)strlen(pStr);
		}
		if (lStr > 0 && lStr < 0x7FFFFFFF) {
			p = (C_UBP)pStr;
			while (len < lStr && *p != 0) {
				q = p + 1;
				if (len + 1 < lStr && *p == EUC_KANA && (*q >= KANA_L && *q <= KANA_H)) {
					p += 2;	len += 2;
					rc += 4;
				}
				else if (len + 2 < lStr && *p == EUC_3BYTE) {
					p++;	len++;
					for (i = 0; i < 2 && *p >= 0x80 && *p <= 0xBF && *p != 0; i++, p++, len++);
					if (i >= 2) rc += 6;
				}
				else if (len + 2 < lStr && (*p >= EUC_L && *p <= EUC_H) && (*q >= EUC_L && *q <= EUC_H)) {
					if		(_ismbblead(*p) && _ismbbtrail(*q)) {							//	MBC(SJIS)と重複する？
						p += 2;	len += 2;
						rc += 3;
					}
					else if ((*p >= 0x80 && *p <= 0xBF) && (*q >= 0x80 && *q <= 0xBF)) {	//	UTF-8と重複する？
						p += 2;	len += 2;
						rc += 3;
					}
					else {																	//	MBC(SJIS)／UTF-8と重複しない？
						p += 2;	len += 2;
						rc += 4;
					}
				}
				else if (*p >= 0x00 && *p <= 0x7F) {
					p++;	len++;
					rc += 1;
				}
				else {
					p++;	len++;
				}
			}
			if (len != 0) {
				rc = (int)((ULL)rc * 100 / ((ULL)len * 2));
			}
			else rc = 0;
		}
		else rc = 0;
	}
	return rc;
}
/*==============================================================================================================*/
/*	Function	:	ＵＴＦ－８文字列チェック																	*/
/*																												*/
/*	Argument	:	pStr  - ＵＴＦ－８コードであるかチェックする文字列のアドレス								*/
/*																												*/
/*	Return		:	ＵＴＦ－８コードである確率[%]																*/
/*==============================================================================================================*/
AJCEXPORT	int		WINAPI	AjcStrIsUtf8(C_BCP pStr)
{
	return AjcStrIsUtf8Ex(pStr, -1);
}
/*--------------------------------------------------------------------------------------------------------------*/
AJCEXPORT	int		WINAPI	AjcStrIsUtf8Ex(C_BCP pStr, UI lStr)
{
	int		rc = 0;
	UI		len = 0;
	C_UBP	p;

	if (pStr != NULL) {
		if (lStr == -1) {
			lStr = (UI)strlen(pStr);
		}
		if (lStr > 0 && lStr < 0x7FFFFFFF) {
			if (lStr >= 3 && memcmp(pStr, "\xEF\xBB\xBF", 3) == 0) {		//	BOM
				rc = 100;
			}
			else {
				p = (C_UBP)pStr;
				while (len < lStr && *p != 0) {
					UB		b4 = (UB)(*p & 0xF0);
					UB		b6 = (UB)(*p & 0xFC);
					UI		i;

					if		(b6 == 0xFC) {
						p++;	len++;
						for (i = 0; i < 5 && len < lStr && *p >= 0x80 && *p <= 0xBF && *p != 0; i++, p++, len++);
						if (i >= 5) rc += 12;
					}
					else if (b6 == 0xF8) {
						p++;	len++;
						for (i = 0; i < 4 && len < lStr && *p >= 0x80 && *p <= 0xBF && *p != 0; i++, p++, len++);
						if (i >= 4) rc += 10;
					}
					else if (b6 == 0xF0) {
						p++;	len++;
						for (i = 0; i < 3 && len < lStr && *p >= 0x80 && *p <= 0xBF && *p != 0; i++, p++, len++);
						if (i >= 3) rc += 8;
					}
					else if (b4 == 0xE0) {
						p++;	len++;
						for (i = 0; i < 2 && len < lStr && *p >= 0x80 && *p <= 0xBF && *p != 0; i++, p++, len++);
						if (i >= 2) rc += 6;
					}
					else if (b4 == 0xC0 || b4 == 0xD0) {
						p++;	len++;
						for (i = 0; i < 1 && len < lStr && *p >= 0x80 && *p <= 0xBF && *p != 0; i++, p++, len++);
						if (i >= 1) rc += 4;
					}
					else if (*p >= 0x00 && *p <= 0x7F) {
						p++;	len++;
						rc += 1;
					}
					else {
						p++;	len++;
					}
				}
				if (len != 0) {
					rc = (int)((ULL)rc * 100 / ((ULL)len * 2));
				}
				else rc = 0;
			}
		}
		else rc = 0;
	}
	return rc;
}
/*==============================================================================================================*/
/*	Function	:	文字コードチェック（インスタンス生成）														*/
/*																												*/
/*	Argument	:	BufSize  - 文字コードチェック用バッファのサイズ（１６～１０２４）							*/
/*					DefCode	 - デフォルト文字コード																*/
/*																												*/
/*	Return		:	≠NULL：ＯＫ（インスタンスハンドル）														*/
/*					＝NULL：エラー																				*/
/*==============================================================================================================*/
AJCEXPORT	HAJCMBC		WINAPI	AjcMbcCreate (UI BufSize, AJCMBCKIND DefCode)
{
	HAJCMBC		pW = NULL;

	if (pW = (HAJCMBC)AJCMEM(sizeof(AJCMBC))) {
		memset(pW, 0, sizeof(AJCMBC));
		pW->MbcDef	= DefCode;
		pW->BufSize	= __max(BufSize    , AJCMIN_MBCBUF);
		pW->BufSize	= __min(pW->BufSize, AJCMAX_MBCBUF);
		AjcMbcReset(pW);
	}
	return pW;
}
/*==============================================================================================================*/
/*	Function	:	文字コードチェック（リセット）																*/
/*																												*/
/*	Argument	:	pW		- インスタンスハンドル																*/
/*																												*/
/*	Return		:	TRUE - OK, FALSE - Error																	*/
/*==============================================================================================================*/
AJCEXPORT	BOOL		WINAPI	AjcMbcReset  (HAJCMBC pW)
{
	BOOL	rc = FALSE;

	if (pW != NULL) {
		pW->MbcKind	= pW->MbcDef;
		pW->MbcIx		= 0;
		memset(pW->MbcBuf, 0, sizeof pW->MbcBuf);
		rc = TRUE;
	}
	return rc;
}
/*==============================================================================================================*/
/*	Function	:	文字コードチェック（文字データ投与）														*/
/*																												*/
/*	Argument	:	pW		- インスタンスハンドル																*/
/*					pStr	- 投与する文字列のアドレス															*/
/*																												*/
/*	Return		:	TRUE - OK, FALSE - Error																	*/
/*==============================================================================================================*/
AJCEXPORT	BOOL		WINAPI	AjcMbcPutData(HAJCMBC pW, C_BCP pStr)
{
	BOOL	rc = FALSE;
	UI		i, stl;

	if (pW != NULL && pStr != NULL) {
		//--- バッファへ全角文字を蓄積 -------//
		stl = (UI)strlen(pStr);
		for (i = 0; i < stl; i++) {
		//	BC	prev, cmin;
		//	prev = (pW->MbcIx != 0) ? pW->MbcBuf[pW->MbcIx - 1] : pW->MbcBuf[pW->BufSize -1];
		//	if (_ismbblead(prev)) cmin = 0x40;
		//	else				  cmin = 0x80;
		//	if (pStr[i] >= cmin) {
				pW->MbcBuf[pW->MbcIx] = pStr[i];
				pW->MbcIx = ((pW->MbcIx + 1) % pW->BufSize);
		//	}
		}
		//--- 戻り値＝ＯＫ -------------------//
		rc = TRUE;
	}
	return rc;
}
/*==============================================================================================================*/
/*	Function	:	文字コードチェック（チェック）																*/
/*																												*/
/*	Argument	:	pW		- インスタンスハンドル																*/
/*																												*/
/*	Return		:	＞０：ＯＫ（文字コード種別）																*/
/*					≦０：エラー（判別不能）																	*/
/*==============================================================================================================*/
AJCEXPORT	AJCMBCKIND	WINAPI	AjcMbcCheck  (HAJCMBC pW)
{
	AJCMBCKIND	rc = 0;

	if (pW != NULL) {
		rc = AjcStrChkCode(pW->MbcBuf);
	}
	return rc;
}
/*==============================================================================================================*/
/*	Function	:	文字コードチェック（インスタンス消去）														*/
/*																												*/
/*	Argument	:	pW		- インスタンスハンドル																*/
/*																												*/
/*	Return		:	≠０：ＯＫ（文字コード種別）																*/
/*					＝０：エラー																				*/
/*==============================================================================================================*/
AJCEXPORT	VO			WINAPI	AjcMbcDelete (HAJCMBC pW)
{
	if (pW != NULL) {
		free(pW);
	}
}
//==============================================================================================================//
//	パスとファイル名を反対にした文字列を作成する（ex. "d:\sub1\x.txt" --> "x.txt;d:\sub1\"）					//
//																												//
//	引　数	：	pPath - ファイルパス名のアドレス																//
//				pBuf  - 反対にしたしたパス名文字列を格納するバッファのアドレス									//
//				lBuf  - 反対にしたしたパス名文字列を格納するバッファの文字数									//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcStrCnvRevPathA(BCP pPath, BCP pBuf, int lBuf)
{
	BOOL	rc = FALSE;
	BC		drv[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], fext[_MAX_EXT];

	if (pPath != NULL && pBuf != NULL && lBuf > 0) {
		_splitpath(pPath, drv, dir, fname, fext);
		AjcSnPrintFA(pBuf, lBuf, "%s%s;%s%s", fname, fext, drv, dir);
		rc = TRUE;
	}

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcStrCnvRevPathW(WCP pPath, WCP pBuf, int lBuf)
{
	BOOL	rc = FALSE;
	UW		drv[_MAX_DRIVE], dir[_MAX_DIR], fname[_MAX_FNAME], fext[_MAX_EXT];

	if (pPath != NULL && pBuf != NULL && lBuf > 0) {
		_wsplitpath(pPath, drv, dir, fname, fext);
		AjcSnPrintFW(pBuf, lBuf, L"%s%s;%s%s", fname, fext, drv, dir);
		rc = TRUE;
	}

	return rc;
}
//==============================================================================================================//
//	パスとファイル名を反対にしたパス名を元に戻す																//
//																												//
//	引　数	：	pRev  - 反対にしたパス名																		//
//				pBuf  - 元に戻したパス名文字列を格納するバッファのアドレス										//
//				lBuf  - 元に戻したパス名文字列を格納するバッファの文字数										//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcStrRetRevPathA(BCP pRev, BCP pBuf, int lBuf)
{
	BOOL	rc = FALSE;
	BCP		pTop, p;

	if (pRev != NULL && pBuf != NULL && lBuf > 0) {
		if (pTop = (BCP)AJCMEM(strlen(pRev) + 1)) {
			strcpy(pTop, pRev);
			//----- パス設定 ------------------------------------------------//
			 if (p = mbsstr(pTop, ";")) {
				p++;
				while (*p != 0	&&	lBuf > 1) {
					*pBuf++ = *p++;
					lBuf--;
				}
			}
			//----- ファイル名設定 ------------------------------------------//
			p = pTop;
			while (*p != ';'  &&  *p != 0  &&  lBuf > 1) {
				*pBuf++ = *p++;
				lBuf--;
			}
			//----- 文字列終端設定 ------------------------------------------//
			*pBuf = 0;
			//----- 一時バッファ解放 ----------------------------------------//
			free(pTop);
			//----- 戻り値=正常 ---------------------------------------------//
			rc = TRUE;
		}
	}

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcStrRetRevPathW(WCP pRev, WCP pBuf, int lBuf)
{
	BOOL	rc = FALSE;
	WCP		pTop, p;

	if (pRev != NULL && pBuf != NULL && lBuf > 0) {
		if (pTop = (WCP)AJCMEM((wcslen(pRev) + 1) * 2)) {
			wcscpy(pTop, pRev);
			//----- パス設定 ------------------------------------------------//
			 if (p = wcsstr(pTop, L";")) {
				p++;
				while (*p != 0	&&	lBuf > 1) {
					*pBuf++ = *p++;
					lBuf--;
				}
			}
			//----- ファイル名設定 ------------------------------------------//
			p = pTop;
			while (*p != L';'  &&  *p != 0	&&	lBuf > 1) {
				*pBuf++ = *p++;
				lBuf--;
			}
			//----- 文字列終端設定 ------------------------------------------//
			*pBuf = 0;
			//----- 一時バッファ解放 ----------------------------------------//
			free(pTop);
			//----- 戻り値=正常 ---------------------------------------------//
			rc = TRUE;
		}
	}

	return rc;
}
//==============================================================================================================//
//	文字列が全て同一文字で構成されているかチェック																//
//																												//
//	引　数	：	pPath - ファイルパス名のアドレス																//
//				pStr  - チェックする文字列のアドレス															//
//				c	  - チェックする文字																		//
//																												//
//	戻り値	：	TRUE  - 文字列は全て指定された文字で構成されている												//
//				FALSE - 文字列中に指定された文字以外が含まれている（空文字列を含む）							//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcStrSameAsAllCharA(C_BCP pStr, BC c)
{
	BOOL	rc = FALSE;

	if (pStr != NULL && *pStr != 0) {
		while (*pStr == c && *pStr != 0) {
			pStr++;
		}
		rc = (*pStr == 0);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcStrSameAsAllCharW(C_WCP pStr, WC c)
{
	BOOL	rc = FALSE;

	if (pStr != NULL && *pStr != 0) {
		while (*pStr == c && *pStr != 0) {
			pStr++;
		}
		rc = (*pStr == 0);
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	符号なし１０進数文字列→数値変換															//
//																												//
//	Argument	:	pStr  - 数値に変換する符号なし１０進数文字列のアドレス										//
//																												//
//	Return		:	変換した数値																				//
//==============================================================================================================//
//----- バイト１０進文字用（→ＵＩＮＴ）------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcDecToUIA(C_BCP pStr)
{
	ULL		rc = 0;

	while (MAjcIsBlankA((UB)*pStr)) pStr++;

	while (_ismbcdigit((UB)*pStr)) {
		rc *= 10;
		rc += (*pStr & 15);
		if (rc > UINT_MAX) {
			rc = UINT_MAX;
			break;
		}
		pStr++;
	}
	return (UI)rc;
}
//----- バイト１０進文字用（→ＵＬＯＮＧ ＬＯＮＧ）-------------------------------------------------------------//
AJCEXPORT	ULL		WINAPI	AjcDecToULLA(C_BCP pStr)
{
	ULL		rc = 0;

	while (MAjcIsBlankA((UB)*pStr)) pStr++;

	while (_ismbcdigit((UB)*pStr)) {
		rc *= 10;
		rc += (*pStr & 15);
		pStr++;
	}
	return rc;
}
//----- ワイド１０進文字用（→ＵＩＮＴ）------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcDecToUIW(C_WCP pStr)
{
	ULL		rc = 0;

	while (MAjcIsBlankW(*pStr)) pStr++;

	while (MAjcAscIsDigitW(*pStr)) {
		rc *= 10;
		rc += (*pStr & 15);
		if (rc > UINT_MAX) {
			rc = UINT_MAX;
			break;
		}
		pStr++;
	}
	return (UI)rc;
}
//----- ワイド１０進文字用（→ＵＬＯＮＧ ＬＯＮＧ）-------------------------------------------------------------//
AJCEXPORT	ULL		WINAPI	AjcDecToULLW(C_WCP pStr)
{
	ULL		rc = 0;

	while (MAjcIsBlankW(*pStr)) pStr++;

	while (MAjcAscIsDigitW(*pStr)) {
		rc *= 10;
		rc += (*pStr & 15);
		pStr++;
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	１６／８進数文字列→数値変換																//
//																												//
//	Argument	:	pStr  - 数値に変換する１６進／８進数文字列のアドレス										//
//																												//
//	Return		:	変換した数値																				//
//==============================================================================================================//
//----- バイト１６進文字用（→ＩＮＴ）--------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcHexToUIA(C_BCP pStr)
{
	UI		rc = 0;

	while (MAjcIsBlankA((UB)*pStr)) pStr++;

	if (*pStr == '0' && (*(pStr + 1) | 0x20) == 'x') {
		pStr += 2;
	}

	while (MAjcIsXDigit((UB)*pStr)) {
		rc <<= 4;
		if (_ismbcdigit(*pStr)) rc |= ( *pStr & 15);
		else					rc |= ((*pStr & 15) + 9);
		pStr++;
	}
	return rc;
}
//----- バイト１６進文字用（→ＬＯＮＧ ＬＯＮＧ）---------------------------------------------------------------//
AJCEXPORT	ULL		WINAPI	AjcHexToULLA(C_BCP pStr)
{
	ULL		rc = 0;

	while (MAjcIsBlankA((UB)*pStr)) pStr++;

	if (*pStr == '0' && (*(pStr + 1) | 0x20) == 'x') {
		pStr += 2;
	}

	while (MAjcIsXDigit((UB)*pStr)) {
		rc <<= (ULL)4;
		if (_ismbcdigit(*pStr)) rc |= ( *pStr & 15);
		else					rc |= ((*pStr & 15) + 9);
		pStr++;
	}
	return rc;
}
//----- バイト８進文字用（→ＩＮＴ）----------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcOctToUIA(C_BCP pStr)
{
	UI		rc = 0;

	while (MAjcIsBlankA((UB)*pStr)) pStr++;

	while (*pStr >= '0' && *pStr <= '7') {
		rc <<= 3;
		rc |= ( *pStr & 15);
		pStr++;
	}
	return rc;
}
//----- バイト８進文字用（→ＬＯＮＧ ＬＯＮＧ）-----------------------------------------------------------------//
AJCEXPORT	ULL		WINAPI	AjcOctToULLA(C_BCP pStr)
{
	ULL		rc = 0;

	while (MAjcIsBlankA((UB)*pStr)) pStr++;

	while (*pStr >= '0' && *pStr <= '7') {
		rc <<= 3;
		rc |= ( *pStr & 15);
		pStr++;
	}
	return rc;
}

//----- ワイド１６進文字用（→ＩＮＴ）--------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcHexToUIW(C_WCP pStr)
{
	UI		rc = 0;

	while (MAjcIsBlankW(*pStr)) pStr++;

	if (*pStr == L'0' && (*(pStr + 1) | 0x20) == L'x') {
		pStr += 2;
	}

	while (MAjcAscIsXDigitW(*pStr)) {
		rc <<= 4;
		if (MAjcAscIsDigitW(*pStr)) rc |= ( *pStr & 15);
		else						rc |= ((*pStr & 15) + 9);
		pStr++;
	}
	return rc;
}
//----- ワイド１６進文字用（→ＬＯＮＧ ＬＯＮＧ）---------------------------------------------------------------//
AJCEXPORT	ULL		WINAPI	AjcHexToULLW(C_WCP pStr)
{
	ULL		rc = 0;

	while (MAjcIsBlankW(*pStr)) pStr++;

	if (*pStr == L'0' && (*(pStr + 1) | 0x20) == L'x') {
		pStr += 2;
	}

	while (MAjcAscIsXDigitW(*pStr)) {
		rc <<= 4;
		if (MAjcAscIsDigitW(*pStr)) rc |= ( *pStr & 15);
		else						rc |= ((*pStr & 15) + 9);
		pStr++;
	}
	return rc;
}

//----- ワイド８進文字用（→ＩＮＴ）----------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcOctToUIW(C_WCP pStr)
{
	UI		rc = 0;

	while (MAjcIsBlankW(*pStr)) pStr++;

	while (*pStr >= L'0' && *pStr <= L'7') {
		rc <<= 3;
		rc |= ( *pStr & 15);
		pStr++;
	}
	return rc;
}
//----- ワイド８進文字用（→ＬＯＮＧ ＬＯＮＧ）-----------------------------------------------------------------//
AJCEXPORT	ULL		WINAPI	AjcOctToULLW(C_WCP pStr)
{
	ULL		rc = 0;

	while (MAjcIsBlankW(*pStr)) pStr++;

	while (*pStr >= L'0' && *pStr <= L'7') {
		rc <<= 3;
		rc |= ( *pStr & 15);
		pStr++;
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	１０／１６進数文字列→数値変換（１０／１６進数は自動判別）									//
//																												//
//	Argument	:	pStr  - 数値に変換する１０／１６進数文字列のアドレス（先頭が'0x'の場合は１６進）			//
//																												//
//	Return		:	変換した数値																				//
//==============================================================================================================//
//----- バイト文字用（→ＩＮＴ）--------------------------------------------------------------------------------//
AJCEXPORT	SI		WINAPI	AjcAscToIntA(C_BCP pStr)
{
	UI		rc = 0;

	while (MAjcIsBlankA((UB)*pStr)) pStr++;

	if (*pStr == '0' && (*(pStr + 1) | 0x20) == 'x') {
		rc = AjcHexToUIA(pStr);
	}
	else {
		rc = atoi(pStr);
	}

	return rc;
}
//----- バイト文字用（→ＬＯＮＧ ＬＯＮＧ）---------------------------------------------------------------------//
AJCEXPORT	SLL		WINAPI	AjcAscToLIntA(C_BCP pStr)
{
	ULL		rc = 0;

	while (MAjcIsBlankA((UB)*pStr)) pStr++;

	if (*pStr == '0' && (*(pStr + 1) | 0x20) == 'x') {
		rc = AjcHexToULLA(pStr);
	}
	else {
		rc = _atoi64(pStr);
	}

	return rc;
}
//----- ワイド文字用（→ＩＮＴ）--------------------------------------------------------------------------------//
AJCEXPORT	SI		WINAPI	AjcAscToIntW(C_WCP pStr)
{
	UI		rc = 0;

	while (MAjcIsBlankW(*pStr)) pStr++;

	if (*pStr == L'0' && (*(pStr + 1) | 0x20) == L'x') {
		rc = AjcHexToUIW(pStr);
	}
	else {
		rc = _wtoi(pStr);
	}

	return rc;
}
//----- ワイド文字用（→ＬＯＮＧ ＬＯＮＧ）---------------------------------------------------------------------//
AJCEXPORT	SLL		WINAPI	AjcAscToLIntW(C_WCP pStr)
{
	ULL		rc = 0;
	BC		tmp[64];

	while (MAjcIsBlankW(*pStr)) pStr++;

	if (*pStr == L'0' && (*(pStr + 1) | 0x20) == L'x') {
		rc = AjcHexToULLW(pStr);
	}
	else {
		WideCharToMultiByte(CP_ACP, 0, pStr, -1, tmp, 64, NULL, NULL);
		tmp[63] = 0;
		rc = _atoi64(tmp);
	}

	return rc;
}
//==============================================================================================================//
//	Function	:	実数文字列→数値変換																		//
//																												//
//	Argument	:	pStr  - 数値に変換する実数文字列のアドレス													//
//																												//
//	Return		:	変換した数値（実数）																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	double	WINAPI	AjcAscToRealA(C_BCP pStr)
{
	double	rc = 0.0;

	if (pStr != NULL) {
		while (MAjcIsBlankA(*pStr)) pStr++;

		if (strnicmp(pStr, "0x", 2) == 0) {
			rc = (double)AjcAscToLIntA(pStr);
		}
		else {
			rc = atof(pStr);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	double	WINAPI	AjcAscToRealW(C_WCP pStr)
{
	double	rc = 0.0;

	if (pStr != NULL) {
		while (MAjcIsBlankW(*pStr)) pStr++;

		if (wcsnicmp(pStr, L"0x", 2) == 0) {
			rc = (double)AjcAscToLIntW(pStr);
		}
		else {
			rc = _wtof(pStr);
		}
	}
	return rc;
}


//==============================================================================================================//
//	指定文字で区切られた複数の数値を得る																		//
//																												//
//	引　数	：	pStr - 区切られた数値群文字列のアドレス															//
//				sep	 - 区切り文字																				//
//				pTyp - 数値の個数とタイプを表す文字列のアドレス（i:int, l:long long, f:double）					//
//				...	 - 取得した数値を格納するバッファのアドレス群（numで指定した個数）							//
//																												//
//	戻り値	：	取得した数値の個数																				//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcGetSepNumbersA(C_BCP pStr, BC sep, C_BCP pTyp, ...)
{
	UI		rc = 0;
	union {UBP bp; UWP wp; UIP ip; ULLP lp; double *fp; VOP vp;} u;
	va_list vls;

	if (pStr != NULL && pTyp != NULL) {
		va_start(vls, pTyp);
		while (MAjcIsBlankA(*pStr)) pStr++;
		while (*pTyp != 0 && *pStr != 0) {
			u.vp = (VOP)va_arg(vls, VOP);
			if (*pStr != sep && u.vp != NULL) {
				switch (*pTyp) {
					case 'b': case 'B': *u.bp = (UB)AjcAscToIntA (pStr);	rc++; break;
					case 'w': case 'W': *u.wp = (UW)AjcAscToIntA (pStr);	rc++; break;
					case 'i': case 'I':	*u.ip =		AjcAscToIntA (pStr);	rc++; break;
					case 'l': case 'L':	*u.lp =		AjcAscToLIntA(pStr);	rc++; break;
					case 'f': case 'F':	*u.fp =		AjcAscToRealA(pStr);	rc++; break;
				}
			}
			pTyp++;
			while (*pStr != sep && *pStr != 0) pStr++;
			if (*pStr == sep) pStr++;
			while (MAjcIsBlankA(*pStr)) pStr++;
		}
		va_end(vls);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcGetSepNumbersW(C_WCP pStr, WC sep, C_WCP pTyp, ...)
{
	UI		rc = 0;
	union {UBP bp; UWP wp; UIP ip; ULLP lp; double *fp; VOP vp;} u;
	va_list vls;

	if (pStr != NULL && pTyp != NULL) {
		va_start(vls, pTyp);
		while (MAjcIsBlankW(*pStr)) pStr++;
		while (*pTyp != 0 && *pStr != 0) {
			u.vp = (VOP)va_arg(vls, VOP);
			if (*pStr != sep && u.vp != NULL) {
				switch (*pTyp) {
					case L'b': case L'B': *u.bp = (UB)AjcAscToIntW (pStr);	rc++; break;
					case L'w': case L'W': *u.wp = (UW)AjcAscToIntW (pStr);	rc++; break;
					case L'i': case L'I': *u.ip =	  AjcAscToIntW (pStr);	rc++; break;
					case L'l': case L'L': *u.lp =	  AjcAscToLIntW(pStr);	rc++; break;
					case L'f': case L'F': *u.fp =	  AjcAscToRealW(pStr);	rc++; break;
				}
			}
			pTyp++;
			while (*pStr != sep && *pStr != 0) pStr++;
			if (*pStr == sep) pStr++;
			while (MAjcIsBlankW(*pStr)) pStr++;
		}
		va_end(vls);
	}
	return rc;
}
//==============================================================================================================//
//	接頭語に続くパラメタ取得																					//
//																												//
//	引　数	：	pParam	- パラメタ文字列																		//
//				pPrefix	- パラメタ接頭語																		//
//				pVal	- パラメタ接頭語に続く文字列が表す数値を格納するバッファ		（不要時はNULL）		//
//				pBuf	- パラメタ接頭語に続く文字列を格納するバッファのアドレス		（不要時はNULL）		//
//				pBuf	- パラメタ接頭語に続く文字列を格納するバッファのバイト数／文字数						//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - 失敗（パラメタ文字列の先頭はパラメタ接頭語と一致しない）								//
//																												//
//	Ex.		AjcHexStrToUBA("/P123", "/P", &val, buf, sizeof buf); --> val=123, buf = "123"						//
//==============================================================================================================//
//----- バイト文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetAfterPrefixA(C_BCP pParam, C_BCP pPrefix, UIP pVal, BCP pBuf, UI lBuf)
{
	BOOL	rc = FALSE;
	UI		lParam, lPrefix;

	if (pParam != NULL && pPrefix != NULL) {
		lParam	= (UI)strlen(pParam);
		lPrefix = (UI)strlen(pPrefix);
		if (lParam >= lPrefix) {
			if (strnicmp(pParam, pPrefix, lPrefix) == 0) {	// 長さをバイト数としている為 strnicmp を使用（mbsnicmpは不可）
				if (pVal != NULL) {
					*pVal = AjcAscToIntA(pParam + lPrefix);
				}
				if (pBuf != NULL && lBuf > 0) {
					UI len = lParam - lPrefix + 1;
					len = __min(len, lBuf);
					strncpy(pBuf, pParam + lPrefix, len);
					pBuf[lBuf - 1] = 0;
				}
				rc = TRUE;
			}
		}
	}
	return rc;
}
//----- ワイド文字用 -------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcGetAfterPrefixW(C_WCP pParam, C_WCP pPrefix, UIP pVal, WCP pBuf, UI lBuf)
{
	BOOL	rc = FALSE;
	UI		lParam, lPrefix;

	if (pParam != NULL && pPrefix != NULL) {
		lParam	= (UI)wcslen(pParam);
		lPrefix = (UI)wcslen(pPrefix);
		if (lParam >= lPrefix) {
			if (wcsnicmp(pParam, pPrefix, lPrefix) == 0) {
				if (pVal != NULL) {
					*pVal = AjcAscToIntW(pParam + lPrefix);
				}
				if (pBuf != NULL && lBuf > 0) {
					UI len = lParam - lPrefix + 1;
					len = __min(len, lBuf);
					wcsncpy(pBuf, pParam + lPrefix, len);
					pBuf[lBuf - 1] = 0;
				}
				rc = TRUE;
			}
		}
	}
	return rc;
}

//==============================================================================================================//
//	Function	:	１６進文字列が表す数値を得る																//
//																												//
//	Argument	:	pRec - １６進文字列の先頭アドレス															//
//					nLen - １６進文字列の文字数																	//
//					pVal - １６進文字列が表す数値を格納するバッファのアドレス									//
//																												//
//	Return		:	TRUE ：ＯＫ																					//
//					FALSE：エラー（'0'～'F'以外の文字が存在する）												//
//==============================================================================================================//
//----- バイト文字用（ -> BYTE）--------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcHexStrToUBA(C_BCP pRec, UI nLen, UB *pVal)
{
	BOOL	rc = TRUE;

	if (pRec != NULL && pVal != NULL) {
		*pVal = 0;
		while (MAjcIsXDigitA((UB)*pRec) && nLen) {
			*pVal = ((*pVal) << 4) | (_ismbcdigit((UB)*pRec) ? (*pRec & 15) : (*pRec & 15) + 9);
			pRec++;
			nLen--;
		}
		if (nLen != 0) {
			rc = FALSE;
		}
	}
	else rc = FALSE;

	return rc;
}
//----- バイト文字用（ -> WORD）--------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcHexStrToUWA(C_BCP pRec, UI nLen, UW *pVal)
{
	BOOL	rc = TRUE;

	if (pRec != NULL && pVal != NULL) {
		*pVal = 0;
		while (MAjcIsXDigitA((UB)*pRec) && nLen) {
			*pVal = ((*pVal) << 4) | (_ismbcdigit((UB)*pRec) ? (*pRec & 15) : (*pRec & 15) + 9);
			pRec++;
			nLen--;
		}
		if (nLen != 0) {
			rc = FALSE;
		}
	}
	else rc = FALSE;

	return rc;
}
//----- バイト文字用（ -> UINT）--------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcHexStrToUIA(C_BCP pRec, UI nLen, UI *pVal)
{
	BOOL	rc = TRUE;

	if (pRec != NULL && pVal != NULL) {
		*pVal = 0;
		while (MAjcIsXDigitA((UB)*pRec) && nLen) {
			*pVal = ((*pVal) << 4) | (_ismbcdigit((UB)*pRec) ? (*pRec & 15) : (*pRec & 15) + 9);
			pRec++;
			nLen--;
		}
		if (nLen != 0) {
			rc = FALSE;
		}
	}
	else rc = FALSE;

	return rc;
}
//----- バイト文字用（ -> LONG LONG）---------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcHexStrToULLA(C_BCP pRec, UI nLen, ULL *pVal)
{
	BOOL	rc = TRUE;

	if (pRec != NULL && pVal != NULL) {
		*pVal = 0;
		while (MAjcIsXDigitA((UB)*pRec) && nLen) {
			*pVal = ((*pVal) << 4) | (_ismbcdigit((UB)*pRec) ? (*pRec & 15) : (*pRec & 15) + 9);
			pRec++;
			nLen--;
		}
		if (nLen != 0) {
			rc = FALSE;
		}
	}
	else rc = FALSE;

	return rc;
}
//----- ワイド文字用 ( -> BYTE)---------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcHexStrToUBW(C_WCP pRec, UI nLen, UB *pVal)
{
	BOOL	rc = TRUE;

	if (pRec != NULL && pVal != NULL) {
		*pVal = 0;
		while (MAjcAscIsXDigitW(*pRec) && nLen) {
			*pVal = ((*pVal) << 4) | (MAjcAscIsDigitW(*pRec) ? (*pRec & 15) : (*pRec & 15) + 9);
			pRec++;
			nLen--;
		}
		if (nLen != 0) {
			rc = FALSE;
		}
	}
	else rc = FALSE;

	return rc;
}

//----- ワイド文字用 ( -> WORD)---------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcHexStrToUWW(C_WCP pRec, UI nLen, UW *pVal)
{
	BOOL	rc = TRUE;

	if (pRec != NULL && pVal != NULL) {
		*pVal = 0;
		while (MAjcAscIsXDigitW(*pRec) && nLen) {
			*pVal = ((*pVal) << 4) | (MAjcAscIsDigitW(*pRec) ? (*pRec & 15) : (*pRec & 15) + 9);
			pRec++;
			nLen--;
		}
		if (nLen != 0) {
			rc = FALSE;
		}
	}
	else rc = FALSE;

	return rc;
}

//----- ワイド文字用 ( -> UINT)---------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcHexStrToUIW(C_WCP pRec, UI nLen, UI *pVal)
{
	BOOL	rc = TRUE;

	if (pRec != NULL && pVal != NULL) {
		*pVal = 0;
		while (MAjcAscIsXDigitW(*pRec) && nLen) {
			*pVal = ((*pVal) << 4) | (MAjcAscIsDigitW(*pRec) ? (*pRec & 15) : (*pRec & 15) + 9);
			pRec++;
			nLen--;
		}
		if (nLen != 0) {
			rc = FALSE;
		}
	}
	else rc = FALSE;

	return rc;
}
//----- ワイド文字用 ( -> LONG LONG) ---------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcHexStrToULLW(C_WCP pRec, UI nLen, ULL *pVal)
{
	BOOL	rc = TRUE;

	if (pRec != NULL && pVal != NULL) {
		*pVal = 0;
		while (MAjcAscIsXDigitW(*pRec) && nLen) {
			*pVal = ((*pVal) << 4) | (MAjcAscIsDigitW(*pRec) ? (*pRec & 15) : (*pRec & 15) + 9);
			pRec++;
			nLen--;
		}
		if (nLen != 0) {
			rc = FALSE;
		}
	}
	else rc = FALSE;

	return rc;
}
//==============================================================================================================//
//	Function	:	マルチバイトが分断されないようにマルチバイトテキストを抽出									//
//																												//
//	Argument	:	pBuf - マルチバイト文字列バッファアドレス													//
//					lBuf - マルチバイト文字列バッファのバイト数													//
//					pIx  - バッファ中の有効なデータのBYTE数(バッファ中の空領域の先頭Index)を示す変数のアドレス	//
//					mbk  - マルチバイト文字列の文字コード（AJCMBC_SJIS / AJCMBC_EUC / AJCMBC_UTF8）				//
//																												//
//	Return		:	≠ NULL - マルチバイト文字が分断されないように、完結したテキストのアドレス（動的なメモリ）	//
//					＝ NULL  - 完結したテキストなし／エラー														//
//==============================================================================================================//
static	VOP		SubExtractCompletedText (BCP pBuf, UI lBuf, UIP pIx, AJCMBCKIND mbk, BOOL fUnicode);
//----- バイト文字列を返す -------------------------------------------------------------------------------------//
AJCEXPORT	BCP  WINAPI	AjcExtractCompletedTextA(BCP pBuf, UI lBuf, UIP pIx, AJCMBCKIND mbk)
{
	return (BCP)SubExtractCompletedText(pBuf, lBuf, pIx, mbk, FALSE);
}
//----- ワイド文字列を返す -------------------------------------------------------------------------------------//
AJCEXPORT	WCP  WINAPI	AjcExtractCompletedTextW(BCP pBuf, UI lBuf, UIP pIx, AJCMBCKIND mbk)
{
	return (WCP)SubExtractCompletedText(pBuf, lBuf, pIx, mbk, TRUE);
}
//----- 共通サブ -----------------------------------------------------------------------------------------------//
static	VOP		SubExtractCompletedText (BCP pBuf, UI lBuf, UIP pIx, AJCMBCKIND mbk, BOOL fUnicode)
{
	BCP			rc = NULL;
	BCP			p;
	UI			i;
	AJCMBCINFO	mbi;		//	マルチバイト解析情報
	UI			lrx;		//	バッファ中の有効なテキスト長
	UI			len;		//	半端なバイトを除いたテキスト長
	UI			lHalf;		//	半端マルチバイト数
	BC			txtHalf[8];	//	半端マルチバイトテキスト

	if (pBuf != NULL && lBuf != 0 && pIx != NULL && *pIx != 0 && (mbk == AJCMBC_SJIS || mbk == AJCMBC_EUC || mbk == AJCMBC_UTF8)) {
		//	インデクスをバッファ長で制限する
		if (*pIx > lBuf) *pIx = lBuf;
		//	バッファ内有効バイト数設定
		lrx = *pIx;
		//	有効なデータサイズ設定（データ中に '\0' があるかチェック）
		for (i = 0, p = pBuf; i < lrx && *p != 0; i++, p++);
		//	有効なデータがある場合・・・
		if (i != 0) {
			lrx = i;
			//	文字列終端設定
			pBuf[lrx] = 0;
			//	有効なテキストバイト数設定
			switch (mbk) {
				default:
				case AJCMBC_SJIS:	AjcStrChkMbcPart (pBuf, lrx - 1, &mbi);	break;	//	シフトＪＩＳ
				case AJCMBC_EUC:	AjcStrChkEucPart (pBuf, lrx - 1, &mbi);	break;	//	ＥＵＣ　　　
				case AJCMBC_UTF8:	AjcStrChkUtf8Part(pBuf, lrx - 1, &mbi);	break;	//	ＵＴＦ－８　
			}
			//	半端バイト数設定
			lHalf = (mbi.ix + 1) % mbi.len;
			//	有効なテキスト長設定
			if (lHalf != 0) len = lrx - (mbi.ix + 1);	//	半端ありの場合は、半端バイト数を減算
			else			len = lrx;					//	半端なしの場合は、全体が有効テキスト
			//	末尾の半端マルチバイト退避
			memcpy(txtHalf, &pBuf[len], lHalf);
			txtHalf[lHalf] = 0;
			//	有効テキストの文字列終端設定
			pBuf[len] = 0;
			//	完結テキスト設定
			if (len > 0) {
				//----- UNICODE --------------------------------------------------------//
				if (fUnicode) {		//	UNICODE？
					//	● S-JIS	（S-JIS -> UNICODE 変換して通知）
					if (mbk == AJCMBC_SJIS) {
						WCP		pWcs;
						UI		lWcs;
						lWcs = MultiByteToWideChar(CP_ACP, 0, pBuf, len, NULL, 0);
						if (pWcs = (WCP)AJCMEM((lWcs + 1) * 2)) {
							MultiByteToWideChar(CP_ACP, 0, pBuf, len, pWcs, lWcs);
							pWcs[lWcs] = 0;
							rc = (VOP)pWcs;
						}
					}
					//	● EUC （EUC -> UNICODE 変換して通知）
					else if (mbk == AJCMBC_EUC) {
						BCP		pSJis;
						WCP		pWcs;
						UI		lSJis, lWcs;
						lSJis = AjcEucToSJis(pBuf, NULL, 0);
						if (pSJis = (BCP)AJCMEM(lSJis)) {
							AjcEucToSJis(pBuf, pSJis, lSJis);
							lWcs = MultiByteToWideChar(CP_ACP, 0, pSJis, -1, NULL, 0);
							if (pWcs = (WCP)AJCMEM(lWcs * 2)) {
								MultiByteToWideChar(CP_ACP, 0, pSJis, -1, pWcs, lWcs);
								rc = (VOP)pWcs;
							}
							free(pSJis);
						}
					}
					//	● UTF-8 （UTF-8 -> UNICODE 変換して通知）
					else if (mbk == AJCMBC_UTF8) {
						WCP		pWcs;
						UI		lWcs;
						lWcs = MultiByteToWideChar(CP_UTF8, 0, pBuf, -1, NULL, 0);
						if (pWcs = (WCP)AJCMEM(lWcs * 2)) {
							MultiByteToWideChar(CP_UTF8, 0, pBuf, -1, pWcs, lWcs + 1);
							rc = (VOP)pWcs;
						}
					}
				}
				//----- マルチバイト文字 -----------------------------------------------//
				else {					//	マルチバイト？
					//	● S-JIS （S-JISのまま通知）
					if (mbk == AJCMBC_SJIS) {
						BCP		pSJis;
						if (pSJis = (BCP)AJCMEM(len + 1)) {
							memcpy(pSJis, pBuf, len);
							pSJis[len] = 0;
							rc = (VOP)pSJis;
						}
					}
					//	● EUC （EUC -> S-JIS 変換して通知）
					else if (mbk == AJCMBC_EUC) {
						BCP		pSJis;
						UI		lSJis;
						lSJis = AjcEucToSJis(pBuf, NULL, 0);
						if (pSJis = (BCP)AJCMEM(lSJis)) {
							AjcEucToSJis(pBuf, pSJis, lSJis);
							rc = (VOP)pSJis;
						}
					}
					//	● UTF-8 (UTF-8 -> S-JIS 変換して通知）
					else if (mbk == AJCMBC_UTF8) {
						BCP		pSJis;
						UI		lSJis;
						lSJis = AjcUtf8ToMbc(pBuf, NULL, 0);
						if (pSJis = (BCP)AJCMEM(lSJis)) {
							AjcUtf8ToMbc(pBuf, pSJis, lSJis);
							rc = (VOP)pSJis;
						}
					}
				}
			}
			//	末尾の中途マルチバイトを先頭に設定
			memcpy(pBuf, txtHalf, lHalf);
			pBuf[lHalf] = 0;
			//	戻り値（バッファ中の有効バイト数）設定
			*pIx = lHalf;
		}
		//	有効なデータが無い場合（先頭が '\0' ）・・・
		else {
			*pIx = 0;
		}
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	Ｃ言語表記文字列をバイナリ化																//
//																												//
//	Argument	:	pTxt - Ｃ言語表記文字列のアドレス															//
//					pBuf - バイナリ化した文字列を格納するバッファのアドレス										//
//					lBuf - バイナリ化した文字列を格納するバッファのバイト数／文字数								//
//																												//
//	Return		:	バイナリ化した文字列の長さ																	//
//==============================================================================================================//
#define	CSTB_SETBUF(X)		if (pBuf != NULL) {*pBuf++ = X;}
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcCLangStrToBinA(C_BCP pTxt, BCP pBuf, UI lBuf)
{
	UI		len = 0;

	if (pTxt != NULL) {
		while (*pTxt != 0 && ((pBuf == NULL) || (pBuf != NULL && len < lBuf - 1))) {
			if (*pTxt == '\\') {
				pTxt++;
				if		(*pTxt == 'a')	{CSTB_SETBUF('\a');		len++; pTxt++;}
				else if (*pTxt == 'b')	{CSTB_SETBUF('\b');		len++; pTxt++;}
				else if (*pTxt == 'e')	{CSTB_SETBUF('\x1B');	len++; pTxt++;}		//	「\e」は本ＡＰＩ独自のエスケープ文字
				else if (*pTxt == 'f')	{CSTB_SETBUF('\f');		len++; pTxt++;}
				else if (*pTxt == 'n')	{CSTB_SETBUF('\n');		len++; pTxt++;}
				else if (*pTxt == 'r')	{CSTB_SETBUF('\r');		len++; pTxt++;}
				else if (*pTxt == 't')	{CSTB_SETBUF('\t');		len++; pTxt++;}
				else if (*pTxt == 'v')	{CSTB_SETBUF('\v');		len++; pTxt++;}
				else if (*pTxt == '\\') {CSTB_SETBUF('\\');		len++; pTxt++;}
				else if (*pTxt == '?')	{CSTB_SETBUF('?');		len++; pTxt++;}
				else if (*pTxt == '\'') {CSTB_SETBUF('\'');		len++; pTxt++;}
				else if (*pTxt == '"')  {CSTB_SETBUF('"');		len++; pTxt++;}
				else if (*pTxt == 'x')	{
					pTxt++;
					if (MAjcIsXDigitA(*pTxt)) {
						CSTB_SETBUF((BC)AjcHexToUIA(pTxt));
						len++;
						while (MAjcIsXDigitA(*pTxt)) {
							pTxt++;
						}
					}
				}
				else if (*pTxt >= '0' && *pTxt <= '7') {
					CSTB_SETBUF((BC)AjcOctToUIA(pTxt));
					len++;
					while (*pTxt >= '0' && *pTxt <= '7') {
						pTxt++;
					}
				}
			}
			else {
				CSTB_SETBUF(*pTxt); pTxt++;
				len++;
			}
		}
		CSTB_SETBUF(0);
	}
	return len;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcCLangStrToBinW(C_WCP pTxt, WCP pBuf, UI lBuf)
{
	UI		len = 0;

	if (pTxt != NULL) {
		while (*pTxt != 0 && ((pBuf == NULL) || (pBuf != NULL && len < lBuf - 1))) {
			if (*pTxt == L'\\') {
				pTxt++;
				if		(*pTxt == L'a')  {CSTB_SETBUF(L'\a');	len++; pTxt++;}
				else if (*pTxt == L'b')  {CSTB_SETBUF(L'\b');	len++; pTxt++;}
				else if (*pTxt == L'e')  {CSTB_SETBUF(L'\x1B');	len++; pTxt++;}		//	「\e」は本ＡＰＩ独自のエスケープ文字
				else if (*pTxt == L'f')  {CSTB_SETBUF(L'\f');	len++; pTxt++;}
				else if (*pTxt == L'n')  {CSTB_SETBUF(L'\n');	len++; pTxt++;}
				else if (*pTxt == L'r')  {CSTB_SETBUF(L'\r');	len++; pTxt++;}
				else if (*pTxt == L't')  {CSTB_SETBUF(L'\t');	len++; pTxt++;}
				else if (*pTxt == L'v')  {CSTB_SETBUF(L'\v');	len++; pTxt++;}
				else if (*pTxt == L'\\') {CSTB_SETBUF(L'\\');	len++; pTxt++;}
				else if (*pTxt == L'?')  {CSTB_SETBUF(L'?');	len++; pTxt++;}
				else if (*pTxt == L'\'') {CSTB_SETBUF(L'\'');	len++; pTxt++;}
				else if (*pTxt == L'"')  {CSTB_SETBUF(L'"');	len++; pTxt++;}
				else if (*pTxt == L'x')  {
					pTxt++;
					if (MAjcIsXDigitW(*pTxt)) {
						CSTB_SETBUF((WC)AjcHexToUIW(pTxt));
						len++;
						while (MAjcIsXDigitW(*pTxt)) {
							pTxt++;
						}
					}
				}
				else if (*pTxt >= '0' && *pTxt <= '7') {
					CSTB_SETBUF((WC)AjcOctToUIW(pTxt));
					len++;
					while (*pTxt >= '0' && *pTxt <= '7') {
						pTxt++;
					}
				}
			}
			else {
				CSTB_SETBUF(*pTxt); pTxt++;
				len++;
			}
		}
		CSTB_SETBUF(0);
	}
	return len;
}
//==============================================================================================================//
//	Function	:	バイナリ文字列をＣ言語表記文字列に変換														//
//																												//
//	Argument	:	pTxt - バイナリ化文字列のアドレス															//
//					pBuf - Ｃ言語表記文字列を格納するバッファのアドレス											//
//					lBuf - Ｃ言語表記文字列を格納するバッファのバイト数／文字数									//
//																												//
//	Return		:	Ｃ言語表記文字列の長さ																		//
//==============================================================================================================//
#define		MAbcSetStrA(STR) {	stl = (UI)strlen(STR); 						\
								if (pBuf != NULL) {							\
									if (len + stl < lBuf) {					\
										strcpy(pBuf, STR);					\
										pBuf += stl;						\
										len  += stl;						\
									} 										\
									else break;								\
								}											\
								else {										\
									len += stl;								\
								}											\
							 }

#define		MAbcSetStrW(STR) {	stl = (UI)wcslen(STR); 						\
								if (pBuf != NULL) {							\
									if (len + stl < lBuf) {					\
										wcscpy(pBuf, STR);					\
										pBuf += stl;						\
										len  += stl;						\
									} 										\
									else break;								\
								}											\
								else {										\
									len += stl;								\
								}											\
							 }
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcBinToCLangStrA(C_BCP pTxt, BCP pBuf, UI lBuf)
{
	UI		len = 0;
	UI		stl;

	if (pTxt != NULL) {
		while (*pTxt != 0 && ((pBuf == NULL) || (pBuf != NULL && len < lBuf - 1))) {
			if		(*pTxt == '\a'	) {MAbcSetStrA("\\a");	pTxt++;}
			else if (*pTxt == '\b'	) {MAbcSetStrA("\\b");	pTxt++;}
		//	else if (*pTxt == '\x1B') {MAbcSetStrA("\\e");	pTxt++;}
			else if (*pTxt == '\f'	) {MAbcSetStrA("\\f");	pTxt++;}
			else if (*pTxt == '\n'	) {MAbcSetStrA("\\n");	pTxt++;}
			else if (*pTxt == '\r'	) {MAbcSetStrA("\\r");	pTxt++;}
			else if (*pTxt == '\t'	) {MAbcSetStrA("\\t");	pTxt++;}
			else if (*pTxt == '\v'	) {MAbcSetStrA("\\v");	pTxt++;}
			else if (*pTxt == '\\'	) {MAbcSetStrA("\\\\"); pTxt++;}
			else if (*pTxt == '?'	) {MAbcSetStrA("\\?");	pTxt++;}
			else if (*pTxt == '\''	) {MAbcSetStrA("\\'");	pTxt++;}
			else if (*pTxt == '"'   ) {MAbcSetStrA("\\\""); pTxt++;}
			else if ((*pTxt > 0 && *pTxt < 0x20) || *pTxt == 0x7F) {
				BC	txt[8];
				AjcSnPrintFA(txt, 8, "\\x%02x", (UB)*pTxt);
				MAbcSetStrA(txt);
				pTxt++;
			}
			else {
				if (pBuf != NULL) *pBuf++ = *pTxt;
				pTxt++;
				len++;
			}
		}
		if (pBuf != NULL) *pBuf = 0;
	}
	return len;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcBinToCLangStrW(C_WCP pTxt, WCP pBuf, UI lBuf)
{
	UI		len = 0;
	UI		stl;

	if (pTxt != NULL) {
		while (*pTxt != 0 && ((pBuf == NULL) || (pBuf != NULL && len < lBuf - 1))) {
			if		(*pTxt == L'\a'  ) {MAbcSetStrW(L"\\a");  pTxt++;}
			else if (*pTxt == L'\b'  ) {MAbcSetStrW(L"\\b");  pTxt++;}
		//	else if (*pTxt == L'\x1B') {MAbcSetStrW(L"\\e");  pTxt++;}
			else if (*pTxt == L'\f'  ) {MAbcSetStrW(L"\\f");  pTxt++;}
			else if (*pTxt == L'\n'  ) {MAbcSetStrW(L"\\n");  pTxt++;}
			else if (*pTxt == L'\r'  ) {MAbcSetStrW(L"\\r");  pTxt++;}
			else if (*pTxt == L'\t'  ) {MAbcSetStrW(L"\\t");  pTxt++;}
			else if (*pTxt == L'\v'  ) {MAbcSetStrW(L"\\v");  pTxt++;}
			else if (*pTxt == L'\\'  ) {MAbcSetStrW(L"\\\\"); pTxt++;}
			else if (*pTxt == L'?'	 ) {MAbcSetStrW(L"\\?");  pTxt++;}
			else if (*pTxt == L'\''  ) {MAbcSetStrW(L"\\'");  pTxt++;}
			else if (*pTxt == L'"'   ) {MAbcSetStrW(L"\\\""); pTxt++;}
			else if ((*pTxt > 0 && *pTxt < 0x20) || *pTxt == 0x7F) {
				WC	txt[8];
				AjcSnPrintFW(txt, 8, L"\\x%04x", (UW)*pTxt);
				MAbcSetStrW(txt);
				pTxt++;
			}
			else {
				if (pBuf != NULL) *pBuf++ = *pTxt;
				pTxt++;
				len++;
			}
		}
		if (pBuf != NULL) *pBuf = 0;
	}
	return len;
}
//==============================================================================================================//
//	テキストの表示桁数を取得します（全角文字は２文字とします）													//
//																												//
//	引　数：	pText		- 表示桁数を算出するテキストのアドレス												//
//																												//
//	戻り値：	表示桁数																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI AjcTextLenA(C_BCP pText)
{
	return (UI)strlen(pText);
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI AjcTextLenW(C_WCP pText)
{
	UI		rc = 0;

	if (pText != NULL) {
		while (*pText != 0) {
			if (AjcIsBigCharW(pText)) rc += 2;
			else					  rc += 1;
			if (MAjcIsLeadW(*pText)) pText += 2;
			else					 pText++;
		}
	}
	return rc;
}

//==============================================================================================================//
//	文字の全角／半角チェック																					//
//																												//
//	引　数：	pTxt	- 文字の全角／半角チェックする文字列（１～２文字）										//
//																												//
//	戻り値：	TRUE  - 全角																					//
//				FALSE - 半角																					//
//==============================================================================================================//
extern	const UI  EastAsianWidthTableSize;
extern	const UB EastAsianWidthTable[];
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI AjcIsBigCharA(const BC txt[2])
{
	BOOL	rc = FALSE;

	if (txt != NULL && txt[0] != 0) {
		rc = (_ismbblead(txt[0]) != 0 && _ismbbtrail(txt[1]) != 0);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI AjcIsBigCharW(const WC txt[2])
{
	BOOL	rc = FALSE;
	UI		upt;
	int		ix;
	UB		msk;
	WC		c1, c2;

	if (txt != NULL && txt[0] != 0) {
		c1 = txt[0];
		c2 = txt[1];
		if (MAjcIsLeadW(c1) && MAjcIsTrailW(c2)) {
			upt = ((((UI)c1 & 0x03C0) + 0x0040) << 10)	|	//	uuuuu................
				  ((((UI)c1 & 0x003F)		  ) << 10)	|	//	.....xxxxxx..........
				  ((((UI)c2 & 0x03FF)		  ) 	 );		//	...........yyyyyyyyyy
		}
		else {
			upt = c1;
		}

		if (upt < (EastAsianWidthTableSize * 8)) {
			ix	= (upt / 8);
			msk = (UB)(0x01 << (upt % 8));
			rc = ((EastAsianWidthTable[ix] & msk) != 0);
		}
	}
	return rc;
}

//==============================================================================================================//
//	UNICODEコードポイント算出																					//
//																												//
//	引　数：	pTxt	- コードポイントを算出する文字列（１～２文字）											//
//																												//
//	戻り値：	TRUE  - 全角																					//
//				FALSE - 半角																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcTextToCodePointA(const BC txt[2])
{
	UI		rc = 0;
	BC		bc[2] = {0, 0};
	WC		wc[2] = {0, 0};
	BC		c1, c2;

	if (txt != NULL && txt[0] != 0) {
		c1 = txt[0];
		c2 = txt[1];
		if ((_ismbblead(c1) != 0 && _ismbbtrail(c2) != 0)) {
			bc[0] = c1;
			bc[1] = c2;
			MultiByteToWideChar(CP_ACP, 0, bc, 2, wc, 2);
		}
		else {
			bc[0] = c1;
			bc[1] = 0;
			MultiByteToWideChar(CP_ACP, 0, bc, 1, wc, 2);
		}
		rc = AjcTextToCodePointW(wc);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI	AjcTextToCodePointW(const WC txt[2])
{
	UI		rc = 0;
	WC		c1, c2;

	if (txt != NULL && txt[0] != 0) {
		c1 = txt[0];
		c2 = txt[1];
		if (MAjcIsLeadW(c1) && MAjcIsTrailW(c2)) {
			rc = ((((UI)c1 & 0x03C0) + 0x0040) << 10)  |	//	uuuuu................
				 ((((UI)c1 & 0x003F)		 ) << 10)  |	//	.....xxxxxx..........
				 ((((UI)c2 & 0x03FF)		 )		);		//	...........yyyyyyyyyy
		}
		else {
			rc = c1;
		}
	}
	return rc;
}
//==============================================================================================================//
//	UNICODEコードポイント→ 文字列変換																			//
//																												//
//	引　数：	cp			- コードポイント																	//
//				pTxt		- 文字列を格納するバッファのアドレス（２文字分）									//
//																												//
//	戻り値：	1 or 2	- 文字列のバイト数／文字数																//
//				0		- エラー																				//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcCodePointToTextA(UI cp, BC txt[2])
{
	UI		rc = 0;
	WC		wc[2];
	UI		len;

	if ((len = AjcCodePointToTextW(cp, wc)) != 0) {
		memset(txt, 0, 2);
		rc = WideCharToMultiByte(CP_ACP, 0, wc, len, txt, 2, NULL, NULL);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcCodePointToTextW(UI cp, WC txt[2])
{
	UI	rc = 0;

	if (txt != NULL && (cp & 0xFFE00000) == 0) {
		if ((cp & 0xFFFF0000) == 0) {
			txt[0] = (WC)cp;
			txt[1] = 0;
			rc = 1;
		}
		else {
			cp -= 0x00010000;
			txt[0] = (cp >> 10)   + 0xD800;
			txt[1] = (cp & 0x3FF) + 0xDC00;
			rc = 2;
		}
	}
	return rc;
}
//==============================================================================================================//
//	フォント情報を文字列化																						//
//																												//
//	引　数：	pLf			- フォント情報のアドレス															//
//				pTxt		- 文字列を格納するバッファのアドレス（不要時はNULL)									//
//				lTxt		- 文字列を格納するバッファの文字数													//
//				fKey		- TRUE: 文字列にキー(FN=, LF=)を含める												//
//																												//
//	戻り値：	≠０：文字列化するのに必要な文字数（終端(\0)を含む												//
//				＝０：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcLogFontToTextA(const LOGFONTA *pLf, BCP pTxt, UI lTxt, BOOL fKey)
{
	UI		rc		 = 0;
	BC		txt[512] = {0};

	if (pLf != NULL) {
		if (fKey) {
			AjcSnPrintFA(txt, 512, "FN=%s,LF=%d/%d/%d/%d/%d/%d/%d/%d/%d/%d/%d/%d/%d",
									  pLf->lfFaceName,	pLf->lfHeight,
														pLf->lfWidth,
														pLf->lfEscapement,
														pLf->lfOrientation,
														pLf->lfWeight,
														pLf->lfItalic,
														pLf->lfUnderline,
														pLf->lfStrikeOut,
														pLf->lfCharSet,
														pLf->lfOutPrecision,
														pLf->lfClipPrecision,
														pLf->lfQuality,
														pLf->lfPitchAndFamily
			);
		}
		else {
			AjcSnPrintFA(txt, 512, "%s,%d/%d/%d/%d/%d/%d/%d/%d/%d/%d/%d/%d/%d",
									  pLf->lfFaceName,	pLf->lfHeight,
														pLf->lfWidth,
														pLf->lfEscapement,
														pLf->lfOrientation,
														pLf->lfWeight,
														pLf->lfItalic,
														pLf->lfUnderline,
														pLf->lfStrikeOut,
														pLf->lfCharSet,
														pLf->lfOutPrecision,
														pLf->lfClipPrecision,
														pLf->lfQuality,
														pLf->lfPitchAndFamily
			);
		}
		if (pTxt != NULL && lTxt != 0) {
			AjcSnPrintFA(pTxt, lTxt, "%s", txt);
		}
		rc = (UI)strlen(txt) + 1;
	}
	return	rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcLogFontToTextW(const LOGFONTW *pLf, WCP pTxt, UI lTxt, BOOL fKey)
{
	UI		rc		 = 0;
	WC		txt[512] = {0};

	if (pLf != NULL) {
		if (fKey) {
			AjcSnPrintFW(txt, 512, L"FN=%s, LF=%d/%d/%d/%d/%d/%d/%d/%d/%d/%d/%d/%d/%d;",
									  pLf->lfFaceName,	pLf->lfHeight,
														pLf->lfWidth,
														pLf->lfEscapement,
														pLf->lfOrientation,
														pLf->lfWeight,
														pLf->lfItalic,
														pLf->lfUnderline,
														pLf->lfStrikeOut,
														pLf->lfCharSet,
														pLf->lfOutPrecision,
														pLf->lfClipPrecision,
														pLf->lfQuality,
														pLf->lfPitchAndFamily
			);
		}
		else {
			AjcSnPrintFW(txt, 512, L"%s, %d/%d/%d/%d/%d/%d/%d/%d/%d/%d/%d/%d/%d;",
									  pLf->lfFaceName,	pLf->lfHeight,
														pLf->lfWidth,
														pLf->lfEscapement,
														pLf->lfOrientation,
														pLf->lfWeight,
														pLf->lfItalic,
														pLf->lfUnderline,
														pLf->lfStrikeOut,
														pLf->lfCharSet,
														pLf->lfOutPrecision,
														pLf->lfClipPrecision,
														pLf->lfQuality,
														pLf->lfPitchAndFamily
			);
		}
		if (pTxt != NULL && lTxt != 0) {
			AjcSnPrintFW(pTxt, lTxt, L"%s", txt);
		}
		rc = (UI)wcslen(txt) + 1;
	}
	return	rc;
}
//==============================================================================================================//
//	文字列からフォント情報を作成																				//
//																												//
//	引　数：	pTxt		- フォント情報表現文字列															//
//				pLf			- フォント情報を格納するバッファのアドレス											//
//																												//
//	戻り値：	TRUE ：成功																						//
//				FALSE：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcTextToLogFontA(C_BCP pTxt, LPLOGFONTA pLf)
{
	BOOL	rc	 = FALSE;
	UI		i;
	C_BCP	p;

	if (pTxt != NULL && pLf != NULL) {
		//	キー(FN= / LF=)付きテキスト
		if ((p = strstr(pTxt, "FN=")) || (p = strstr(pTxt, "LF="))) {
			memset(pLf, 0, sizeof(LOGFONTA));
			//	フェース名設定
			if (p = strstr(pTxt, "FN=")) {
				for (i = 0, p+= 3; i < AJCTSIZE(pLf->lfFaceName) - 1 && *p != ',' && *p != 0; i++, p++) {
					pLf->lfFaceName[i] = *p;
				}
			}
			//	フォント情報設定
			if (p = strstr(pTxt, "LF=")) {
				p += 3;
				AjcGetSepNumbersA(p, '/', "iiiiibbbbbbbb",	&pLf->lfHeight			,
															&pLf->lfWidth			,
															&pLf->lfEscapement		,
															&pLf->lfOrientation		,
															&pLf->lfWeight			,
															&pLf->lfItalic			,
															&pLf->lfUnderline		,
															&pLf->lfStrikeOut		,
															&pLf->lfCharSet			,
															&pLf->lfOutPrecision	,
															&pLf->lfClipPrecision	,
															&pLf->lfQuality			,
															&pLf->lfPitchAndFamily);
			}
		}
		//	キー(FN= / LF=)無しテキスト
		else {
			memset(pLf, 0, sizeof(LOGFONTA));
			//	フェース名設定
			for (i = 0, p = pTxt; i < AJCTSIZE(pLf->lfFaceName) - 1 && *p != ',' && *p != 0; i++, p++) {
				pLf->lfFaceName[i] = *p;
			}
			//	空白とカンマ(,)スキップ
			while (*p == ',' || *p == ' ') {
				p++;
			}
			//	フォント情報設定
			if (*p != 0) {
				AjcGetSepNumbersA(p, '/', "iiiiibbbbbbbb",	&pLf->lfHeight			,
															&pLf->lfWidth			,
															&pLf->lfEscapement		,
															&pLf->lfOrientation		,
															&pLf->lfWeight			,
															&pLf->lfItalic			,
															&pLf->lfUnderline		,
															&pLf->lfStrikeOut		,
															&pLf->lfCharSet			,
															&pLf->lfOutPrecision	,
															&pLf->lfClipPrecision	,
															&pLf->lfQuality			,
															&pLf->lfPitchAndFamily);
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcTextToLogFontW(C_WCP pTxt, LPLOGFONTW pLf)
{
	BOOL	rc	 = FALSE;
	UI		i;
	C_WCP	p;

	if (pTxt != NULL && pLf != NULL) {
		//	キー(FN= / LF=)付きテキスト
		if ((p = wcsstr(pTxt, L"FN=")) || (p = wcsstr(pTxt, L"LF="))) {
			memset(pLf, 0, sizeof(LOGFONTW));
			//	フェース名設定
			if (p = wcsstr(pTxt, L"FN=")) {
				for (i = 0, p+= 3; i < AJCTSIZE(pLf->lfFaceName) - 1 && *p != L',' && *p != 0; i++, p++) {
					pLf->lfFaceName[i] = *p;
				}
			}
			//	フォント情報設定
			if (p = wcsstr(pTxt, L"LF=")) {
				p += 3;
				AjcGetSepNumbersW(p, '/', L"iiiiibbbbbbbb",	&pLf->lfHeight			,
															&pLf->lfWidth			,
															&pLf->lfEscapement		,
															&pLf->lfOrientation		,
															&pLf->lfWeight			,
															&pLf->lfItalic			,
															&pLf->lfUnderline		,
															&pLf->lfStrikeOut		,
															&pLf->lfCharSet			,
															&pLf->lfOutPrecision	,
															&pLf->lfClipPrecision	,
															&pLf->lfQuality			,
															&pLf->lfPitchAndFamily);
			}
		}
		//	キー(FN= / LF=)無しテキスト
		else {
			memset(pLf, 0, sizeof(LOGFONTW));
			//	フェース名設定
			for (i = 0, p = pTxt; i < AJCTSIZE(pLf->lfFaceName) - 1 && *p != L',' && *p != 0; i++, p++) {
				pLf->lfFaceName[i] = *p;
			}
			//	空白とカンマ(,)スキップ
			while (*p == L',' || *p == L' ') {
				p++;
			}
			//	フォント情報設定
			if (*p != 0) {
				AjcGetSepNumbersW(p, '/', L"iiiiibbbbbbbb",	&pLf->lfHeight			,
															&pLf->lfWidth			,
															&pLf->lfEscapement		,
															&pLf->lfOrientation		,
															&pLf->lfWeight			,
															&pLf->lfItalic			,
															&pLf->lfUnderline		,
															&pLf->lfStrikeOut		,
															&pLf->lfCharSet			,
															&pLf->lfOutPrecision	,
															&pLf->lfClipPrecision	,
															&pLf->lfQuality			,
															&pLf->lfPitchAndFamily);
			}
		}
	}
	return rc;
}
