﻿#include	"AjcInternal.h"

#define	INST_ID		0x617BCC05

#define	STBA_BUFSIZE	16384		//	エンコード判別の最大バイト数

//--------------------------------------------------------------------------------------------------------------//
//	ファイルポインタ																							//
//--------------------------------------------------------------------------------------------------------------//
typedef union {
	struct {UL l, h;}	s;
	ULL					ull;
} FPOINT, *PFPOINT;

//--------------------------------------------------------------------------------------------------------------//
//	改行情報																									//
//--------------------------------------------------------------------------------------------------------------//
typedef struct {
	UI		stl;			//	変換する改行コード文字列のバイト数
	BC		strA[4];		//	変換する改行コードのバイト文字列
	WC		strW[4];		//	変換する改行コードのUNICODE文字列
} LFINFO, *PLFINFO;
typedef const LFINFO *PCLFINFO;

//--------------------------------------------------------------------------------------------------------------//
//	ＢＯＭ																										//
//--------------------------------------------------------------------------------------------------------------//
static	const BC	bom_utf8   [3] = {0xEF, 0xBB, 0xBF};
static	const BC	bom_utf16le[2] = {0xFF, 0xFE};
static	const BC	bom_utf16be[2] = {0xFE, 0xFF};

//--------------------------------------------------------------------------------------------------------------//
//	最後にオープンしたファイルのテキストエンコード退避															//
//--------------------------------------------------------------------------------------------------------------//
__declspec(thread) static	EAJCTEC LastOpenedTec = AJCTEC_MBC;

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
static	HAJCFILE	SubFOpenW		(C_WCP pPath, EAJCTEC tec,			  HANDLE hF, UI sgare);
static	HAJCFILE	SubFCreateW		(C_WCP pPath, EAJCTEC tec, BOOL fBOM, HANDLE hF, UI share);
static	HAJCFILE	SubFAppendW		(C_WCP pPath, EAJCTEC tec, BOOL fBOM, HANDLE hF, UI share);
static	BOOL		SubFAppendW_SetTec(HAJCFILE pW);
static	BCP			SubFGetSByLineA	(HAJCFILE pW, BCP pBuf, UI lBuf);
static	BCP			SubFGetSByCharA	(HAJCFILE pW, BCP pBuf, UI lBuf);
static	WCP			SubFGetSByLineW	(HAJCFILE pW, WCP pBuf, UI lBuf);
static	WCP			SubFGetSByCharW	(HAJCFILE pW, WCP pBuf, UI lBuf);

static	UI			RevedWideCharToMultiByte(C_WCP pUni, UI lUni, BCP pMbc, UI lMbc);
static	EAJCTEC		GetTecByBOM(ULL fsize, BCP pDat, UI lDat, BOOL *prc_fExistBOM, ULL *prc_FIx, AJCFUPTR *prc_pInp);
static	EAJCTEC		GetTecByText(ULL fsize, BCP pDat, UI lDat);

//	文字列読み出し用サブ関数
static	VO			SetLfCode(HAJCFILE pW);

static	BOOL		GetLineA(HAJCFILE pW, BCP pBuf, UI lBuf);
static	BOOL		GetLineW(HAJCFILE pW, WCP pBuf, UI lBuf);

static	BOOL		SearchLFA(HAJCFILE pW, UI limit, UIP plLf, UIP plCtl);
static	BOOL		SearchLFW(HAJCFILE pW, UI limit, UIP plLf, UIP plCtl);

static	BOOL		SetToLineBufA(HAJCFILE pW, C_BCP pMbc, UI lMbc, BCP pBuf, UI lBuf);
static	BOOL		SetToLineBufW(HAJCFILE pW, C_WCP pUni, UI lUni, WCP pBuf, UI lBuf);


static	UI			CheckLFA(HAJCFILE pW, C_BCP pTxt, ULL lTxt, PLFINFO pLfInfo);
static	UI			CheckLFW(HAJCFILE pW, C_WCP pTxt, ULL lTxt, PLFINFO pLfInfo, BOOL fBE);

static	UI			ReadCharFromByteFile	(HAJCFILE pW, BC buf[8], UI bfl, BOOL *pfLF);
static	UI			ReadCharFromUnicodeFile	(HAJCFILE pW, WC buf[8], UI bfl, BOOL *pfLF);

static	UI			SetByteCharToBufA		(HAJCFILE pW, C_BCP pInp, UI lInp, BCP pBuf, UI lBuf);
static	UI			SetByteCharToBufW		(HAJCFILE pW, C_BCP pInp, UI lInp, WCP pBuf, UI lBuf);

static	UI			SetUnicodeCharToBufA	(HAJCFILE pW, C_WCP pInp, UI lInp, BCP pBuf, UI lBuf);
static	UI			SetUnicodeCharToBufW	(HAJCFILE pW, C_WCP pInp, UI lInp, WCP pBuf, UI lBuf);

static	VOP			GetFilePtrByBufA(HAJCFILE pW, ULLP pFIx);
static	VOP			GetFilePtrByBufW(HAJCFILE pW, ULLP pFIx);

//	文字列書き込み用サブ関数
static	BOOL		PutDataA	(HAJCFILE pW, C_BCP pDat, UI lDat);
static	BOOL		PutDataW	(HAJCFILE pW, C_WCP pDat, UI lDat);

static	BOOL		FlushBuf	(HAJCFILE pW);
static	VO			WriteBOM	(HAJCFILE pW);

static	BOOL		WriteMbcA	(HAJCFILE pW, C_BCP pTxt, UI lTxt);
static	BOOL		WriteMbcW	(HAJCFILE pW, C_WCP pTxt, UI lTxt);

static	BOOL		WriteUtf8A	(HAJCFILE pW, C_BCP pTxt, UI lTxt);
static	BOOL		WriteUtf8W	(HAJCFILE pW, C_WCP pTxt, UI lTxt);

static	BOOL		WriteEucA	(HAJCFILE pW, C_BCP pTxt, UI lTxt);
static	BOOL		WriteEucW	(HAJCFILE pW, C_WCP pTxt, UI lTxt);

static	BOOL		WriteUniA	(HAJCFILE pW, C_BCP pTxt, UI lTxt, BOOL fBE);
static	BOOL		WriteUniW	(HAJCFILE pW, C_WCP pTxt, UI lTxt, BOOL fBE);


//==============================================================================================================//
//	入力テキストファイルオープン																				//
//																												//
//	引　数	：	pPath	-	ファイルパス名																		//
//				tec		-	テキストエンコード種別																//
//																												//
//	戻り値	：	≠ NULL：ＯＫ（テキストファイルアクセスハンドル）												//
//				＝ NULL：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	HAJCFILE	WINAPI	AjcFOpenA	(C_BCP pPath, EAJCTEC tec)
{
	HAJCFILE	pW = NULL;
	WC			path[MAX_PATH];

	if (pPath != NULL) {
		MultiByteToWideChar(CP_ACP, 0, pPath, -1, path, MAX_PATH);
		path[MAX_PATH - 1] = 0;
		pW = SubFOpenW(path, tec, NULL, 0);
	}
	return pW;
}
//----- バイト文字（共有指定） ---------------------------------------------------------------------------------//
AJCEXPORT	HAJCFILE	WINAPI	AjcFOpenShareA(C_BCP pPath, EAJCTEC tec, UI share)
{
	HAJCFILE	pW = NULL;
	WC			path[MAX_PATH];

	if (pPath != NULL) {
		MultiByteToWideChar(CP_ACP, 0, pPath, -1, path, MAX_PATH);
		path[MAX_PATH - 1] = 0;
		pW = SubFOpenW(path, tec, NULL, share);
	}
	return pW;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	HAJCFILE	WINAPI	AjcFOpenW	(C_WCP pPath, EAJCTEC tec)
{
	HAJCFILE	pW	 = NULL;

	do {
		//	パス名指定チェック
		if (pPath == NULL) break;
		//	ファイルオープン
		pW = SubFOpenW(pPath, tec, NULL, 0);
	} while(0);

	return	pW;
}
//----- ワイド文字（共有指定）----------------------------------------------------------------------------------//
AJCEXPORT	HAJCFILE	WINAPI	AjcFOpenShareW(C_WCP pPath, EAJCTEC tec, UI share)
{
	HAJCFILE	pW	 = NULL;

	do {
		//	パス名指定チェック
		if (pPath == NULL) break;
		//	ファイルオープン
		pW = SubFOpenW(pPath, tec, NULL, share);
	} while(0);

	return	pW;
}
//----- 共通サブ -----------------------------------------------------------------------------------------------//
static	HAJCFILE	SubFOpenW	(C_WCP pPath, EAJCTEC tec, HANDLE hF, UI share)
{
	HAJCFILE	pW	  = NULL;
	BOOL		fErr  = FALSE;
	UL			hword = 0;
	UL			ftype = FILE_TYPE_UNKNOWN;

	do {
		//	有効なパス名／ハンドルが指定されているかチェック
		if (pPath == NULL && (hF == NULL || hF == INVALID_HANDLE_VALUE)) break;
		//	エンコード種別チェック
		if (!MAJCTEC_VALID(tec)) tec = AJCTEC_AUTO;
		//	インスタンスワーク生成
		if ((pW = (HAJCFILE)AJCMEM(sizeof(AJCFILE))) == NULL) break;
		memset(pW, 0, sizeof(AJCFILE));
		pW->iid 	 = INST_ID;
		//	テキストエンコード設定
		pW->tec = tec;
		//	改行モード初期化
		pW->LfCnv = AJCFLF_CRLF_TO_LF;
		//	ファイルオープン
		if (hF == NULL || hF == INVALID_HANDLE_VALUE) {
			if ((pW->hFile = CreateFileW(pPath, GENERIC_READ, share, NULL, OPEN_EXISTING, 0, NULL)) == INVALID_HANDLE_VALUE) {
				fErr = TRUE;
				break;
			}
		}
		else {
			pW->hFile = hF;
		}
		//	ファイル入力の旨、設定
		pW->fInp = TRUE;
		//	有効なファイルハンドルがチェック（ファイルタイプ取得）
		ftype = GetFileType(pW->hFile);
		if (ftype == FILE_TYPE_UNKNOWN && GetLastError() != NO_ERROR) {
			fErr = TRUE;
			break;
		}
		//	ディスクファイルならば、ＢＯＭチェックと自動判別
		if (ftype == FILE_TYPE_DISK) {
			EAJCTEC	tec_tmp;
			union {
				struct	{UI lo, hi;} s;
				ULL					 ull;
			} u;
			//	ファイルサイズ設定
			u.ull = 0;
			u.s.lo	= GetFileSize(pW->hFile, &u.s.hi);
			pW->FSizeA = u.ull;
			pW->FSizeW = pW->FSizeA / 2;
			//	空ファイル以外（ファイルサイズ≠０）の場合、ファイルマッピング
			if (pW->FSizeA != 0) {
				//	ファイルマッピングオブジェクト生成
				if ((pW->hMap = CreateFileMapping(pW->hFile, 0, PAGE_READONLY, 0, 0, NULL)) == NULL) {
					fErr = TRUE;
					break;
				}
				//	ファイルマッピング
				if ((pW->pInpTop.vop = (VOP)MapViewOfFile(pW->hMap, FILE_MAP_READ, 0, 0, 0)) == NULL) {
					fErr = TRUE;
					break;
				}
				//	BOMからテキストエンコード設定
				if ((tec_tmp = GetTecByBOM(pW->FSizeA, pW->pInpTop.bcp, (UI)(__min(pW->FSizeA, STBA_BUFSIZE)), &pW->fExistBOM, &pW->FIx, &pW->pInp)) == AJCTEC_ERROR) { // BOMなし？
					//	 BOMなしで自動設定ならば、テキストからエンコード設定
					if (pW->tec == AJCTEC_AUTO) {
						//	テキストデータからテキストエンコード設定
						pW->tec = GetTecByText(pW->FSizeA, pW->pInpTop.bcp, (UI)(__min(pW->FSizeA, STBA_BUFSIZE)));
					}
					//	BOMなしで自動設定以外ならば、指定エンコード
					else {
						//	NOP
					}
				}
				//	ＢＯＭ有りならば、当該テキストエンコード設定
				else {
					pW->tec = tec_tmp;
				}
				//	最後にオープンしたファイルのテキストエンコード退避
				LastOpenedTec = pW->tec;
				//	改行文字列設定
				SetLfCode(pW);
			}
			//	空ファイル（ファイルサイズ＝０）の場合、ファイルマッピングしない（エラーとなるためマッピング不可）
			else {
				//	NOP （以降、読み出し操作では、空ファイルの場合は処理を切り分ける）
			}
		}
		//	ディスクファイル以外ならばエラー
		else {
			fErr = TRUE;
			break;
		}

	} while(0);

	//	エラーならば、リソース解放
	if (fErr && pW != NULL) {
		if (pW->hFile		== INVALID_HANDLE_VALUE ||
			pW->hMap		== NULL 				||
			pW->pInpTop.vop == NULL) {
			if (pW->hMap  != NULL				 ) CloseHandle(pW->hMap);
			if (pW->hFile != INVALID_HANDLE_VALUE) CloseHandle(pW->hFile);
			free(pW);
			pW = NULL;
		}
	}
	return	pW;
}
//==============================================================================================================//
//	オープン済ファイルのハンドルを使用してインスタンス生成														//
//																												//
//	引　数	：	hFile	-	オープン済ファイルのハンドル														//
//				tec		-	テキストエンコード種別（AJCTEC_AUTOは指定不可）										//
//																												//
//	戻り値	：	≠ NULL：ＯＫ（テキストファイルアクセスハンドル）												//
//				＝ NULL：エラー																					//
//==============================================================================================================//
AJCEXPORT	HAJCFILE	WINAPI	AjcFAttachOpened	(HANDLE hFile, EAJCTEC tec)
{
	HAJCFILE	pW = NULL;

	pW = SubFOpenW(NULL, tec, hFile, 0);

	return	pW;
}
//==============================================================================================================//
//	文字列入力																									//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//				pBuf	-	読み出したテキストを格納するバッファのアドレス										//
//				lBuf	-	読み出したテキストを格納するバッファの文字数（文字列終端(0x00)を含む）				//
//																												//
//	戻り値	：	≠NULL ：ＯＫ（入力文字列のアドレス( = pBuf )）													//
//				＝NULL ：ＥＯＦ／エラー																			//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcFGetSA	(HAJCFILE pW, BCP pBuf, UI lBuf)
{
	return AjcFGetSExA(pW, pBuf, lBuf, 0, NULL);
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcFGetSW	(HAJCFILE pW, WCP pBuf, UI lBuf)
{
	return AjcFGetSExW(pW, pBuf, lBuf, 0, NULL);
}
//==============================================================================================================//
//	文字列入力（拡張）																							//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//				pBuf	-	読み出したテキストを格納するバッファのアドレス										//
//				lBuf	-	読み出したテキストを格納するバッファの文字数（文字列終端(0x00)を含む）				//
//				cbp		-	コールバックパラメタ																//
//				cbNtcRPos - 読み出し済バイト位置／文字位置 通知用コールバック関数（不要時はNULL）				//
//																												//
//	戻り値	：	≠NULL ：ＯＫ（入力文字列のアドレス( = pBuf )）													//
//				＝NULL ：ＥＯＦ／エラー																			//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcFGetSExA	(HAJCFILE pW, BCP pBuf, UI lBuf, UX cbp, BOOL (CALLBACK *cbNtcRPos)(ULL pos, UX cbp))
{
	BCP		rc = NULL;

	if (pW != NULL && pW->iid == INST_ID && pW->fInp && pBuf != NULL && lBuf >= 3) {
		//	コールバック関数設定
		pW->cbNtcRPos = cbNtcRPos;
		//	空ファイル以外の場合・・・
		if (pW->FSizeA != 0) {
			//	バッファに残留テキストがある場合は、現バッファ位置に対応するファイルポインタを設定
			if (pW->ixBuf < pW->BufChars) {
				//	バッファの読み出し済みデータからファイルポインタ設定
				pW->pInp.vop = GetFilePtrByBufA(pW, &pW->FIx);
				//	読み出しバッファ無効化
				pW->BufChars = pW->ixBuf = 0;
			}
			//	テキスト読み出し
			if (!AjcFEof(pW)) {		//	Not EOF?
				//	改行の認識あり？
				if (pW->LfCnv != AJCFLF_NONE) {
					if ((rc = SubFGetSByLineA(pW, pBuf, lBuf)) == NULL || *rc == 0) {	//	行単位で読み出し，失敗？
						 rc = SubFGetSByCharA(pW, pBuf, lBuf);							//		文字単位で読み出し
					}
				}
				//	改行の認識なし？
				else {
					BCP		rsu;
					BCP		pRem = pBuf;
					int		lRem = lBuf;
					int		stl;
					//	１行／１文字読み出し
					if ((rsu = SubFGetSByLineA(pW, pRem, lRem)) == NULL || *rsu == 0) {	//	行単位で読み出し，失敗？
						rsu = SubFGetSByCharA(pW, pRem, lRem);							//		文字単位で読み出し
					}
					//	読み出し成功ならば、バッファ満杯まで継続
					if (rc = rsu) {
						//	読み出したテキスト分バッファポインタ，バッファ長更新
						stl = (int)strlen(pRem);
						pRem += stl;
						lRem -= stl;
						//	バッファに空きあり ＆ ！ＥＯＦならば継続して読み出し
						while (lRem > 0 && rsu != NULL && *rsu != 0) {
							//	１行／１文字読み出し
							if ((rsu = SubFGetSByLineA(pW, pRem, lRem)) == NULL || *rsu == 0) {	//	行単位で読み出し，失敗？
								rsu = SubFGetSByCharA(pW, pRem, lRem);							//		文字単位で読み出し
							}
							//	読み出したテキスト分バッファポインタ，バッファ長更新
							if (rsu != NULL) {
								stl = (int)strlen(pRem);
								pRem += stl;
								lRem -= stl;
							}
						}
					}
				}
			}
		}
		//	空ファイルの場合・・・
		else {
			//	読み出し位置通知
			if (pW->cbNtcRPos != NULL) {
				pW->cbNtcRPos(0, pW->cbp);
			}
			//	戻り値＝EOF
			rc = NULL;
		}
	}
	return rc;
}
//- - - １行入力をトライ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -//
static	BCP		SubFGetSByLineA	(HAJCFILE pW, BCP pBuf, UI lBuf)
{
	BCP		rc = NULL;

	if (GetLineA(pW, pBuf, lBuf)) {
		//	コールバック無ならば、戻り値＝！ＥＯＦ
		if (pW->cbNtcRPos == NULL) {
			rc = pBuf;
		}
		//	コールバック有ならば、戻り値＝コールバックの指定に従う
		else {
			if (pW->cbNtcRPos(pW->FIx, pW->cbp)) rc = pBuf;	//	読み出し継続
			else								 rc = NULL;	//	読み出し中止（ＥＯＦを返す）
		}
	}

	return rc;
}
//- - - １文字毎に入力 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -//
static	BCP		SubFGetSByCharA	(HAJCFILE pW, BCP pBuf, UI lBuf)
{
	BCP		rc	 = NULL;		//	戻り値
	BOOL	fEnd = FALSE;		//	終了フラグ
	UI		lStr = 0;			//	読み出しバイト数／ワード数カウンタ
	UI		lOut;				//	バッファに設定した文字のバイト数

	if (!AjcFEof(pW)) {
		rc = pBuf;				//	戻り値初期化
		lBuf--;					//	文字列終端分減算
		while (lBuf != 0 && !fEnd) {
			switch (pW->tec) {
				//	バイト文字ファイル読み出し
				case AJCTEC_MBC:	case AJCTEC_UTF_8:	case AJCTEC_EUC_J:
				{	UI		lInp;
					BC		cInp[8];
					if ((lInp = ReadCharFromByteFile(pW, cInp, 8, &fEnd)) != 0) {
						if ((lOut = SetByteCharToBufA(pW, cInp, lInp, pBuf, lBuf)) != 0) {
							pBuf += lOut;
							lBuf -= lOut;
							lStr += lOut;
						}
						else {						//	バッファ満杯
							pW->pInp.bcp -= lInp;	//		１文字分バッファポインタ,ファイルポインタを戻す
							pW->FIx 	 -= lInp;
							fEnd = TRUE;
						}
					}
					else {							//	ＥＯＦに達した？
						if (lStr == 0) rc = NULL;	//		読み出しデータが無ければ、戻り値＝ＥＯＦ
						fEnd = TRUE;
					}
					break;
				}
				//	ワイド文字ファイル読み出し
				case AJCTEC_UTF_16LE:	case AJCTEC_UTF_16BE:
				{	UI		lInp;
					WC		cInp[8];
					if ((lInp = ReadCharFromUnicodeFile(pW, cInp, 8, &fEnd)) != 0) {
						if ((lOut = SetUnicodeCharToBufA(pW, cInp, lInp, pBuf, lBuf)) != 0) {
							pBuf += lOut;
							lBuf -= lOut;
							lStr += lOut;
						}
						else {						//	バッファ満杯
							pW->pInp.wcp -= lInp;	//		１文字分バッファポインタ,ファイルポインタを戻す
							pW->FIx 	 -= lInp;
							fEnd = TRUE;
						}
					}
					else {							//	ＥＯＦに達した？
						if (lStr == 0) rc = NULL;	//		読み出しデータが無ければ、戻り値＝ＥＯＦ
						fEnd = TRUE;
					}
					break;
				}
			}
			//	入力済みバイト位置通知
			if (pW->cbNtcRPos != NULL) {
				//	コールバックからの中止指示チェック
				if (!pW->cbNtcRPos(pW->FIx, pW->cbp)) {
					rc = NULL;		//	戻り値＝ＥＯＦを仮定
					break;			//	読み出し中止
				}
			}
		}
		//	文字列終端設定
		if (rc != NULL) *pBuf = 0;
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcFGetSExW	(HAJCFILE pW, WCP pBuf, UI lBuf, UX cbp, BOOL (CALLBACK *cbNtcRPos)(ULL pos, UX cbp))
{
	WCP		rc = NULL;

	if (pW != NULL && pW->iid == INST_ID && pW->fInp && pBuf != NULL && lBuf >= 3) {
		//	コールバック関数設定
		pW->cbNtcRPos = cbNtcRPos;
		//	空ファイル以外の場合・・・
		if (pW->FSizeA != 0) {
			//	バッファに残留テキストがある場合は、現バッファ位置に対応するファイルポインタを設定
			if (pW->ixBuf < pW->BufChars) {
				//	バッファの読み出し済みデータからファイルポインタ設定
				pW->pInp.vop = GetFilePtrByBufW(pW, &pW->FIx);
				//	読み出しバッファ無効化
				pW->BufChars = pW->ixBuf = 0;
			}
			//	テキスト読み出し
			if (!AjcFEof(pW)) {		//	Not EOF?
				//	改行の認識あり？
				if (pW->LfCnv != AJCFLF_NONE) {
					if ((rc = SubFGetSByLineW(pW, pBuf, lBuf)) == NULL || *rc == 0) {	//	行単位で読み出し，失敗？
						 rc = SubFGetSByCharW(pW, pBuf, lBuf);							//		文字単位で読み出し
					}
				}
				//	改行の認識なし？
				else {
					WCP		rsu;
					WCP		pRem = pBuf;
					int		lRem = lBuf;
					int		stl;
					//	１行／１文字読み出し
					if ((rsu = SubFGetSByLineW(pW, pRem, lRem)) == NULL || *rsu == 0) {	//	行単位で読み出し，失敗？
						rsu = SubFGetSByCharW(pW, pRem, lRem);							//		文字単位で読み出し
					}
					//	読み出し成功ならば、バッファ満杯まで継続
					if (rc = rsu) {
						//	読み出したテキスト分バッファポインタ，バッファ長更新
						stl = (int)wcslen(pRem);
						pRem += stl;
						lRem -= stl;
						//	バッファに空きあり ＆ ！ＥＯＦならば継続して読み出し
						while (lRem > 0 && rsu != NULL && *rsu != 0) {
							//	１行／１文字読み出し
							if ((rsu = SubFGetSByLineW(pW, pRem, lRem)) == NULL || *rsu == 0) {	//	行単位で読み出し，失敗？
								rsu = SubFGetSByCharW(pW, pRem, lRem);							//		文字単位で読み出し
							}
							//	読み出したテキスト分バッファポインタ，バッファ長更新
							if (rsu != NULL) {
								stl = (int)wcslen(pRem);
								pRem += stl;
								lRem -= stl;
							}
						}
					}
				}
			}
		}
		//	空ファイルの場合・・・
		else {
			//	読み出し位置通知
			if (pW->cbNtcRPos != NULL) {
				pW->cbNtcRPos(0, pW->cbp);
			}
			//	戻り値＝EOF
			rc = NULL;
		}
	}
	return rc;
}
//- - - １行入力をトライ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -//
static	WCP		SubFGetSByLineW	(HAJCFILE pW, WCP pBuf, UI lBuf)
{
	WCP		rc = NULL;

	if (GetLineW(pW, pBuf, lBuf)) {
		//	コールバック無ならば、戻り値＝！ＥＯＦ
		if (pW->cbNtcRPos == NULL) {
			rc = pBuf;
		}
		//	コールバック有ならば、戻り値＝コールバックの指定に従う
		else {
			if (pW->cbNtcRPos(pW->FIx, pW->cbp)) rc = pBuf;	//	読み出し継続
			else								 rc = NULL;	//	読み出し中止（ＥＯＦを返す）
		}
	}

	return rc;
}
//- - - １文字毎に入力 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -//
static	WCP		SubFGetSByCharW	(HAJCFILE pW, WCP pBuf, UI lBuf)
{
	WCP		rc = NULL;
	BOOL	fEnd = FALSE;
	UI		lStr = 0;			//	読み出しバイト数／ワード数カウンタ
	UI		lOut;				//	バッファに設定した文字のワード数

	if (!AjcFEof(pW)) {
		rc = pBuf;				//	戻り値初期化
		lBuf--;					//	文字列終端分減算
		while (lBuf != 0 && !fEnd) {
			switch (pW->tec) {
				//	バイト文字ファイル読み出し
				case AJCTEC_MBC:	case AJCTEC_UTF_8:	case AJCTEC_EUC_J:
				{	int		lInp;
					BC		cInp[8];
					if ((lInp = ReadCharFromByteFile(pW, cInp, 8, &fEnd)) != 0) {
						if ((lOut = SetByteCharToBufW(pW, cInp, lInp, pBuf, lBuf)) != 0) {
							pBuf += lOut;
							lBuf -= lOut;
							lStr += lOut;
						}
						else {						//	バッファ満杯
							pW->pInp.bcp -= lInp;	//		１文字分バッファポインタ,ファイルポインタを戻す
							pW->FIx 	 -= lInp;
							fEnd = TRUE;
						}
					}
					else {							//	ＥＯＦに達した？
						if (lStr == 0) rc = NULL;	//		読み出しデータが無ければ、戻り値＝ＥＯＦ
						fEnd = TRUE;
					}
					break;
				}
				//	ワイド文字ファイル読み出し
				case AJCTEC_UTF_16LE:	case AJCTEC_UTF_16BE:
				{	int		lInp;
					WC		cInp[8];
					if ((lInp = ReadCharFromUnicodeFile(pW, cInp, 8, &fEnd)) != 0) {
						if ((lOut = SetUnicodeCharToBufW(pW, cInp, lInp, pBuf, lBuf)) != 0) {
							pBuf += lOut;
							lBuf -= lOut;
							lStr += lOut;
						}
						else {						//	バッファ満杯
							pW->pInp.wcp -= lInp;	//		１文字分バッファポインタ,ファイルポインタを戻す
							pW->FIx 	 -= lInp;
							fEnd = TRUE;
						}
					}
					else {							//	ＥＯＦに達した？
						if (lStr == 0) rc = NULL;	//		読み出しデータが無ければ、戻り値＝ＥＯＦ
						fEnd = TRUE;
					}
					break;
				}
			}
			//	入力済み文字位置通知
			if (pW->cbNtcRPos != NULL) {
				//	コールバックからの中止指示チェック
				if (!pW->cbNtcRPos(pW->FIx, pW->cbp)) {
					rc	 = NULL;			//	戻り値＝ＥＯＦ
					fEnd = TRUE;
				}
			}
		}
		//	文字列終端設定
		if (rc != NULL) *pBuf = 0;
	}
	return rc;
}

//==============================================================================================================//
//	１文字入力																									//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//																												//
//	戻り値	：	≠ -1 ：ＯＫ（文字コード）																		//
//				＝ -1 ：ＥＯＦ／エラー																			//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI			WINAPI	AjcFGetCodeA	(HAJCFILE pW)
{
	UI		rc = -1;
	UI		rsu;

	if (pW != NULL && pW->iid == INST_ID && pW->fInp) {
		//	空ファイル以外の場合・・・
		if (pW->FSizeA != 0) {
			//	バッファ上の最後に読み出したバイトが、マルチバイト１バイト目ならば文字の先頭に戻す
			if (pW->ixBuf < pW->BufChars) {
				if (pW->ixBuf >= 1 && MAjcMbsBTypeA(pW->buf.mbc, pW->ixBuf - 1) == _MBC_LEAD) {
					pW->ixBuf--;
				}
			}
			//	１文字入力
			if ((rsu = AjcFGetCA(pW)) != -1) {					//	１バイト入力，ＥＯＦ以外？
				if (MAjcIsLeadA((BC)rsu)) {						//		全角１文字目？
					union {
						BC	bc[2];
						UW	uw;
					} u;
					u.bc[0] = (BC)rsu;							//			全角１文字目設定
					if ((rsu = AjcFGetCA(pW)) != -1) {			//			次バイト入力，ＥＯＦ以外？
						u.bc[1] = (BC)rsu;						//				全角２バイト目設定
						rc = u.uw;								//				戻り値＝全角コード
					}
				}
				else if (rsu != (UB)-1) {						//		バイト文字？
					rc = (UB)rsu;								//			戻り値＝バイトコード
				}
			}
		}
		//	空ファイルの場合・・・
		else {
			//	ＥＯＦを返す
			rc = -1;
		}
	}
	return	rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcFGetCodeW	(HAJCFILE pW)
{
	UI		rc = -1;
	UI		rsu;

	if (pW != NULL && pW->iid == INST_ID && pW->fInp) {
		//	空ファイル以外の場合・・・
		if (pW->FSizeA != 0) {
			//	バッファ上の最後に読み出したワードが、サロゲート１ワード目ならば文字の先頭に戻す
			if (pW->ixBuf < pW->BufChars) {
				if (pW->ixBuf >= 1 && MAjcIsLeadW(pW->buf.uni[pW->ixBuf - 1])) {
					pW->ixBuf--;
				}
			}
			//	１文字入力
			if ((rsu = AjcFGetCW(pW)) != (WC)-1) {				//	１ワード入力，ＥＯＦ以外？
				if (MAjcIsLeadW((WC)rsu)) {						//		サロゲート１ワード目？
					WC		wc[2];
					wc[0] = (WC)rsu;							//			サロゲート１ワード目設定
					if ((rsu = AjcFGetCW(pW)) != (WC)-1) {		//			次ワード入力，ＥＯＦ以外？
						wc[1] = (WC)rsu;						//				サロゲート２ワード目設定
						rc = AjcTextToCodePoint(wc);			//				戻り値＝サロゲート文字のコードポイント
					}
				}
				else if (rsu != (UW)-1) {						//		非サロゲート文字？
					rc = (WC)rsu;								//			戻り値＝非サロゲート文字のコードポイント
				}
			}
		}
		//	空ファイルの場合・・・
		else {
			//	ＥＯＦを返す
			rc = -1;
		}
	}
	return	rc;
}
//==============================================================================================================//
//	１バイト／１ワード入力																						//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//																												//
//	戻り値	：	≠ -1 ：ＯＫ（１バイト／１ワードデータ）														//
//				＝ -1 ：ＥＯＦ／エラー																			//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BC			WINAPI	AjcFGetCA	(HAJCFILE pW)
{
	BC		rc = -1;

	if (pW != NULL && pW->iid == INST_ID && pW->fInp) {
		//	空ファイル以外の場合・・・
		if (pW->FSizeA != 0) {
			if (pW->ixBuf < pW->BufChars) {
				rc = pW->buf.mbc[pW->ixBuf++];
			}
			else {
				pW->BufFIx = pW->FIx;
				if (SubFGetSByCharA(pW, pW->buf.mbc, AJCTF_MAX_BUFA)) {
					pW->ixBuf = 0;
					if (pW->BufChars = (UI)strlen(pW->buf.mbc)) {
						rc = (BC)pW->buf.mbc[pW->ixBuf++];
					}
				}
			}
		}
		//	空ファイルの場合・・・
		else {
			//	ＥＯＦを返す
			rc = -1;
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	WC			WINAPI	AjcFGetCW	(HAJCFILE pW)
{
	WC		rc = -1;

	if (pW != NULL && pW->iid == INST_ID && pW->fInp) {
		//	空ファイル以外の場合・・・
		if (pW->FSizeA != 0) {
			if (pW->ixBuf < pW->BufChars) {
				rc = pW->buf.uni[pW->ixBuf++];
			}
			else {
				pW->BufFIx = pW->FIx;
				if (SubFGetSByCharW(pW, pW->buf.uni, AJCTF_MAX_BUFW)) {
					pW->ixBuf = 0;
					if (pW->BufChars = (UI)wcslen(pW->buf.uni)) {
						rc = (WC)pW->buf.uni[pW->ixBuf++];
					}
				}
			}
		}
		//	空ファイルの場合・・・
		else {
			//	ＥＯＦを返す
			rc = -1;
		}
	}
	return rc;
}
//==============================================================================================================//
//	ファイル読み出しポイント取得																				//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//																												//
//	戻り値	：	≠ -1 ：現在のファイル読み出し位置（バイト位置／文字位置）										//
//				＝ -1 ：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	ULL		WINAPI	AjcFGetPointA(HAJCFILE pW)
{
	ULL		rc = -1;

	if (pW != NULL && pW->iid == INST_ID && pW->fInp) {
		//	空ファイル以外の場合・・・
		if (pW->FSizeA != 0) {
			if (pW->ixBuf < pW->BufChars) GetFilePtrByBufA(pW, &rc);
			else						  rc = pW->FIx;
		}
		//	空ファイルの場合・・・
		else {
			//	読み出し位置＝０を返す
			rc = 0;
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	ULL		WINAPI	AjcFGetPointW(HAJCFILE pW)
{
	ULL		rc = -1;

	if (pW != NULL && pW->iid == INST_ID && pW->fInp) {
		//	空ファイル以外の場合・・・
		if (pW->FSizeA != 0) {
			if (pW->ixBuf < pW->BufChars) GetFilePtrByBufW(pW, &rc);
			else						  rc = pW->FIx;
		}
		//	空ファイルの場合・・・
		else {
			//	読み出し位置＝０を返す
			rc = 0;
		}
	}
	return rc;
}
//==============================================================================================================//
//	ファイル読み出しバイトポイント取得																			//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//																												//
//	戻り値	：	≠ -1 ：現在のファイル読み出し位置（バイト位置／文字位置）										//
//				＝ -1 ：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	ULL		WINAPI	AjcFGetBytePointA(HAJCFILE pW)
{
	ULL		rc = -1;

	if ((rc = AjcFGetPointA(pW)) != -1) {
		//	空ファイル以外の場合・・・
		if (pW->FSizeA != 0) {
			if (pW->tec == AJCTEC_UTF_16LE || pW->tec == AJCTEC_UTF_16BE) {
				rc *= 2;
			}
		}
		//	空ファイルの場合・・・
		else {
			//	読み出し位置＝０を返す
			rc = 0;
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	ULL		WINAPI	AjcFGetBytePointW(HAJCFILE pW)
{
	ULL		rc = -1;

	if ((rc = AjcFGetPointW(pW)) != -1) {
		//	空ファイル以外の場合・・・
		if (pW->FSizeA != 0) {
			if (pW->tec == AJCTEC_UTF_16LE || pW->tec == AJCTEC_UTF_16BE) {
				rc *= 2;
			}
		}
		//	空ファイルの場合・・・
		else {
			//	読み出し位置＝０を返す
			rc = 0;
		}
	}
	return rc;
}
//==============================================================================================================//
//	ファイル読み出しポイント設定																				//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//				point	-	設定するファイル読み出し位置（バイト位置／文字位置）								//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcFSetPoint(HAJCFILE pW, ULL point)
{
	BOOL	rc = FALSE;

	if (pW != NULL && pW->iid == INST_ID && pW->fInp) {
		//	空ファイル以外の場合・・・
		if (pW->FSizeA != 0) {
			if (pW->tec == AJCTEC_MBC || pW->tec == AJCTEC_UTF_8 || pW->tec == AJCTEC_EUC_J) {
				point 		 = __min(pW->FSizeA, point);
				pW->FIx 	 = point;
				pW->pInp.bcp = pW->pInpTop.bcp + point;
			}
			else {
				point 		 = __min(pW->FSizeW, point);
				pW->FIx 	 = point;
				pW->pInp.wcp = pW->pInpTop.wcp + point;
			}
			pW->BufChars = 0;
			pW->ixBuf	 = 0;
			rc = TRUE;
		}
		//	空ファイルの場合・・・
		else {
			if (point == 0) rc = TRUE;
			else			rc = FALSE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	ファイル読み出しポイント退避																				//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//																												//
//	戻り値	：	≠ -1 ：退避したファイルの読み出し位置（バイト位置／文字位置）									//
//				＝ -1 ：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	ULL		WINAPI	AjcFSavePointA(HAJCFILE pW)
{
	ULL		rc = -1;

	if (pW != NULL && pW->iid == INST_ID && pW->fInp) {
		//	空ファイル以外の場合・・・
		if (pW->FSizeA != 0) {
			if (pW->ixBuf < pW->BufChars) {
				pW->Sv_pInp.vop = GetFilePtrByBufA(pW, &pW->Sv_FIx);
			}
			else {
				pW->Sv_pInp.vop = pW->pInp.vop;
				pW->Sv_FIx	= pW->FIx;
			}
			rc = pW->Sv_FIx;
		}
		//	空ファイルの場合・・・
		else {
			rc = 0;
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	ULL		WINAPI	AjcFSavePointW(HAJCFILE pW)
{
	ULL		rc = -1;

	if (pW != NULL && pW->iid == INST_ID && pW->fInp) {
		//	空ファイル以外の場合・・・
		if (pW->FSizeA != 0) {
			if (pW->ixBuf < pW->BufChars) {
				pW->Sv_pInp.vop = GetFilePtrByBufW(pW, &pW->Sv_FIx);
			}
			else {
				pW->Sv_pInp.vop = pW->pInp.vop;
				pW->Sv_FIx	= pW->FIx;
			}
			rc = pW->Sv_FIx;
		}
		//	空ファイルの場合・・・
		else {
			rc = 0;
		}
	}
	return rc;
}
//==============================================================================================================//
//	ファイル読み出しポイント回復																				//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//																												//
//	戻り値	：	TRUE ：ＯＫ																						//
//				FALSE：エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcFRecvPoint(HAJCFILE pW)
{
	BOOL	rc = FALSE;

	if (pW != NULL && pW->iid == INST_ID && pW->fInp) {
		//	空ファイル以外の場合・・・
		if (pW->FSizeA != 0) {
			if (pW->Sv_pInp.vop != NULL) {
				pW->FIx 	 = pW->Sv_FIx;
				pW->pInp.vop = pW->Sv_pInp.vop;
				pW->BufChars = 0;
				pW->ixBuf	 = 0;
				rc = TRUE;
			}
		}
		//	空ファイルの場合・・・
		else {
			rc = TRUE;
		}
	}
	return rc;
}

//==============================================================================================================//
//	ファイルＥＯＦチェック																						//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//																												//
//	戻り値	：	TRUE ：ＥＯＦ																					//
//				FALSE：ＥＯＦ以外																				//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcFEof(HAJCFILE pW)
{
	BOOL	rc = FALSE;

	if (pW != NULL && pW->iid == INST_ID && pW->fInp) {
		//	空ファイル以外の場合・・・
		if (pW->FSizeA != 0) {
			switch (pW->tec) {
				case AJCTEC_MBC:
				case AJCTEC_UTF_8:
				case AJCTEC_EUC_J:
					rc = (pW->FIx >= pW->FSizeA);
					break;
	
				case AJCTEC_UTF_16LE:
				case AJCTEC_UTF_16BE:
					rc = (pW->FIx >= pW->FSizeW);
					break;
			}
		}
		//	空ファイルの場合・・・
		else {
			//	ＥＯＦを返す
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	出力テキストファイル生成																					//
//																												//
//	引　数	：	pPath	-	ファイルパス名																		//
//				tec		-	テキストエンコード種別																//
//				fBOM	-	BOM出力フラグ																		//
//																												//
//	戻り値	：	≠ NULL：ＯＫ（テキストファイルアクセスハンドル）												//
//				＝ NULL：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	HAJCFILE	WINAPI	AjcFCreateA	(C_BCP pPath, EAJCTEC tec, BOOL fBOM)
{
	HAJCFILE	pW = NULL;
	WC			path[MAX_PATH];

	if (pPath != NULL) {
		MultiByteToWideChar(CP_ACP, 0, pPath, -1, path, MAX_PATH);
		path[MAX_PATH - 1] = 0;
		pW = SubFCreateW(path, tec, fBOM, NULL, 0);
	}
	return pW;
}
//----- バイト文字（共有指定）----------------------------------------------------------------------------------//
AJCEXPORT	HAJCFILE	WINAPI	AjcFCreateShareA(C_BCP pPath, EAJCTEC tec, BOOL fBOM, UI share)
{
	HAJCFILE	pW = NULL;
	WC			path[MAX_PATH];

	if (pPath != NULL) {
		MultiByteToWideChar(CP_ACP, 0, pPath, -1, path, MAX_PATH);
		path[MAX_PATH - 1] = 0;
		pW = SubFCreateW(path, tec, fBOM, NULL, share);
	}
	return pW;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	HAJCFILE	WINAPI	AjcFCreateW	(C_WCP pPath, EAJCTEC tec, BOOL fBOM)
{
	HAJCFILE	pW = NULL;

	do {
		//	パス名指定チェック
		if (pPath == NULL) break;
		//	ファイル生成
		pW = SubFCreateW(pPath, tec, fBOM, NULL, 0);
	} while(0);

	return	pW;
}
//----- ワイド文字（共有指定）----------------------------------------------------------------------------------//
AJCEXPORT	HAJCFILE	WINAPI	AjcFCreateShareW(C_WCP pPath, EAJCTEC tec, BOOL fBOM, UI share)
{
	HAJCFILE	pW = NULL;

	do {
		//	パス名指定チェック
		if (pPath == NULL) break;
		//	ファイル生成
		pW = SubFCreateW(pPath, tec, fBOM, NULL, share);
	} while(0);

	return	pW;
}
//----- 共通サブ -----------------------------------------------------------------------------------------------//
static	HAJCFILE	SubFCreateW	(C_WCP pPath, EAJCTEC tec, BOOL fBOM, HANDLE hF, UI share)
{
	HAJCFILE	pW	   = NULL;
	BOOL		fErr   = FALSE;
	UL			HiWord = 0;

	do {
		//	有効なパス名／ハンドルが指定されているかチェック
		if (pPath == NULL && (hF == NULL || hF == INVALID_HANDLE_VALUE)) break;
		//	エンコード種別チェック
		if (!MAJCTEC_VALID(tec)) tec = AJCTEC_AUTO;
		//	AUTOの場合、最後にオープンしたテキストエンコード設定
		if (tec == AJCTEC_AUTO) {
			tec = LastOpenedTec;
		}
		//	インスタンスワーク生成
		if ((pW 		 = (HAJCFILE)AJCMEM(sizeof(AJCFILE))) == NULL) break;
		memset(pW, 0, sizeof(AJCFILE));
		//	インスタンスワーク初期化
		pW->iid 	= INST_ID;
		//	改行モード初期化
		pW->LfCnv = AJCFLF_LF_TO_CRLF;
		//	テキストエンコード設定
		pW->tec 	= tec;
		//	ファイル出力モードの旨、設定
		pW->fInp	= FALSE;
		//	ＢＯＭ出力モード設定
		pW->fBomOut	= fBOM;
		//	出力テキスト半端バイト／半端ワードなしの旨設定
		pW->SvOutC	= 0;
		//	初期テキストバッファサイズ初期化
		pW->lFmtBuf = 2048;
		//	ファイルハンドル未指定ならば、ファイル生成
		if (hF == NULL || hF == INVALID_HANDLE_VALUE) {
			if ((pW->hFile = CreateFileW(pPath, GENERIC_WRITE, share, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL)) == INVALID_HANDLE_VALUE) {
				fErr = TRUE;
				break;
			}
		}
		//	ファイルハンドル指定ならば、ハンドル設定
		else {
			//	ファイルハンドル設定
			pW->hFile = hF;
			//	ファイルポインタが先頭以外ならば、ＢＯＭ出力取り消し
			HiWord = 0;
			if (SetFilePointer(hF, 0, &HiWord, FILE_CURRENT) != 0 || HiWord != 0) {
				pW->fBomOut = FALSE;
			}
		}
	} while(0);

	//	エラーならば、リソース開放
	if (fErr && pW != NULL) {
		free(pW);
		pW = NULL;
	}

	return	pW;
}
//==============================================================================================================//
//	追記モードでファイル生成																					//
//																												//
//	引　数	：	pPath	-	ファイルパス名																		//
//				tec		-	テキストエンコード種別																//
//				fBOM	-	BOM出力フラグ																		//
//																												//
//	戻り値	：	≠ NULL：ＯＫ（テキストファイルアクセスハンドル）												//
//				＝ NULL：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	HAJCFILE	WINAPI	AjcFAppendA	(C_BCP pPath, EAJCTEC tec, BOOL fBOM)
{
	HAJCFILE	pW = NULL;
	WC			path[MAX_PATH];

	if (pPath != NULL) {
		MultiByteToWideChar(CP_ACP, 0, pPath, -1, path, MAX_PATH);
		path[MAX_PATH - 1] = 0;
		pW = SubFAppendW(path, tec, fBOM, NULL, 0);
	}
	return pW;
}
//----- バイト文字（共有指定） ---------------------------------------------------------------------------------//
AJCEXPORT	HAJCFILE	WINAPI	AjcFAppendShareA(C_BCP pPath, EAJCTEC tec, BOOL fBOM, UI share)
{
	HAJCFILE	pW = NULL;
	WC			path[MAX_PATH];

	if (pPath != NULL) {
		MultiByteToWideChar(CP_ACP, 0, pPath, -1, path, MAX_PATH);
		path[MAX_PATH - 1] = 0;
		pW = SubFAppendW(path, tec, fBOM, NULL, share);
	}
	return pW;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	HAJCFILE	WINAPI	AjcFAppendW	(C_WCP pPath, EAJCTEC tec, BOOL fBOM)
{
	HAJCFILE	pW = NULL;

	do {
		//	パス名指定チェック
		if (pPath == NULL) break;
		//	ファイル生成
		pW = SubFAppendW(pPath, tec, fBOM, NULL, 0);
	} while(0);

	return	pW;
}
//----- ワイド文字（共有指定） ---------------------------------------------------------------------------------//
AJCEXPORT	HAJCFILE	WINAPI	AjcFAppendShareW(C_WCP pPath, EAJCTEC tec, BOOL fBOM, UI share)
{
	HAJCFILE	pW = NULL;

	do {
		//	パス名指定チェック
		if (pPath == NULL) break;
		//	ファイル生成
		pW = SubFAppendW(pPath, tec, fBOM, NULL, share);
	} while(0);

	return	pW;
}
//----- 共通サブ -----------------------------------------------------------------------------------------------//
static	HAJCFILE	SubFAppendW	(C_WCP pPath, EAJCTEC tec, BOOL fBOM, HANDLE hF, UI share)
{
	HAJCFILE	pW	   = NULL;
	BOOL		fErr   = FALSE;
	UL			HiWord = 0;
	UL			ftype  = FILE_TYPE_UNKNOWN;

	do {
		//	有効なパス名／ハンドルが指定されているかチェック
		if (pPath == NULL && (hF == NULL || hF == INVALID_HANDLE_VALUE)) break;
		//	エンコード種別チェック
		if (!MAJCTEC_VALID(tec)) tec = AJCTEC_AUTO;
		//	インスタンスワーク生成
		if ((pW 		 = (HAJCFILE)AJCMEM(sizeof(AJCFILE))) == NULL) break;
		memset(pW, 0, sizeof(AJCFILE));
		//	インスタンスワーク初期化
		pW->iid 	= INST_ID;
		//	改行モード初期化
		pW->LfCnv = AJCFLF_LF_TO_CRLF;
		//	テキストエンコード設定
		pW->tec 	= tec;
		//	ファイル出力モードの旨、設定
		pW->fInp	= FALSE;
		//	ＢＯＭ出力モード設定
		pW->fBomOut	= fBOM;
		//	出力テキスト半端バイト／半端ワードなしの旨設定
		pW->SvOutC	= 0;
		//	初期テキストバッファサイズ初期化
		pW->lFmtBuf = 2048;
		//	//	パス名が指定されている場合・・・
		if (pPath != NULL) {
			//	ファイルが存在する場合、ファイルを出力モードでオープンし、ファイルポインタを末尾に設定
			if (AjcPathIsFile(pPath)) {
				if ((pW->hFile = CreateFileW(pPath, GENERIC_READ | GENERIC_WRITE, share, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE) {
					//	ファイルエンコード設定
					if (SubFAppendW_SetTec(pW)) {
						SL	hi = 0;
						//	ファイルポインタを末尾に設定
						SetFilePointer(pW->hFile, 0, &hi, FILE_END);
					}
					//	エンコード設定時エラー
					else {
						fErr = TRUE;
						break;
					}
				}
				//	ファイルオープンエラー
				else {
					fErr = TRUE;
					break;
				}
			}
			//	ファイルが存在しない場合、ファイルを生成
			else {
				if ((pW->hFile = CreateFileW(pPath, GENERIC_WRITE, share, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE) {
					//	NOP
				}
				//	ファイル生成エラー
				else {
					fErr = TRUE;
					break;
				}
				//	ＡＵＴＯならば、最後にオープンしたファイルのテキストエンコード設定
				if (pW->tec == AJCTEC_AUTO) {
					pW->tec = LastOpenedTec;
				}
			}
		}
		//	パス名が指定されていない場合（ファイルハンドル指定の場合）・・・
		else {
			//	ファイルハンドル設定
			pW->hFile = hF;
			//	ファイルエンコード設定
			if (SubFAppendW_SetTec(pW)) {
				SL	hi = 0;
				//	ファイルポインタを末尾に設定
				SetFilePointer(pW->hFile, 0, &hi, FILE_END);
			}
			//	エンコード設定時エラー
			else {
				fErr = TRUE;
				break;
			}
		}
	} while(0);

	//	エラーならば、リソース開放
	if (fErr && pW != NULL) {
		free(pW);
		pW = NULL;
	}
	return	pW;
}
//----- テキストエンコード設定 ---------------------------------------------------------------------------------//
static	BOOL	SubFAppendW_SetTec(HAJCFILE pW)
{
	BOOL	rc = FALSE;
	UL		ftype = FILE_TYPE_UNKNOWN;
	BCP		pBuf  = NULL;
	union {
		struct {UL l, h;} s;
		ULL				  len;
	} fsize;

	do {
		//	有効なファイルハンドルがチェック（ファイルタイプ取得）
		ftype = GetFileType(pW->hFile);
		if (ftype == FILE_TYPE_UNKNOWN && GetLastError() != NO_ERROR) {
			break;
		}
		//	ディスクファイルならば・・・
		if (ftype == FILE_TYPE_DISK) {
			//	ＡＵＴＯ指定時は、テキストエンコード設定
			if (pW->tec == AJCTEC_AUTO) {
				UL		bytes = 0;
				if (pBuf = AjcTAllocA(STBA_BUFSIZE)) {
					SL		hi = 0;
					//	ファイルポインタを先頭に設定
					SetFilePointer(pW->hFile, 0, &hi, FILE_BEGIN);
					//	ファイルサイズ取得
					fsize.s.l = GetFileSize(pW->hFile, &fsize.s.h);
					//	ファイル先頭部分読み出し
					if (ReadFile(pW->hFile, pBuf, STBA_BUFSIZE, &bytes, NULL)) {
						//	読み出しデータありならば、ファイルエンコード設定とＢＯＭ出力無効化
						if (bytes != 0) {
							EAJCTEC	tec_tmp;
							//	BOMからテキストエンコード設定
							if ((tec_tmp = GetTecByBOM(fsize.len, pBuf, bytes, NULL, NULL, NULL)) == AJCTEC_ERROR) { // BOMなし？
								//	 BOMなしで自動設定ならば、テキストからエンコード設定
								if (pW->tec == AJCTEC_AUTO) {
									//	テキストデータからテキストエンコード設定
									pW->tec = GetTecByText(fsize.len, pBuf, bytes);
								}
								//	BOMなしで自動設定以外ならば、指定エンコード
								else {
									//	NOP
								}
							}
							//	ＢＯＭ有りならば、当該テキストエンコード設定
							else {
								pW->tec = tec_tmp;
							}
							//	ＢＯＭ出力無効化
							pW->fBomOut = FALSE;
						}
						//	ファイルサイズ＝０ならは、最後にオープンしたファイルのテキストエンコード設定
						else {
							pW->tec = LastOpenedTec;
						}
					}
					//	読み出しエラー
					else {
						break;
					}
				}
				//	バッファ確保失敗エラー
				else {
					break;
				}
			}
			//	戻り値＝正常
			rc = TRUE;
		}
		//	ディスクファイル以外ならばエラー
		else {
			break;
		}
	} while(0);

	//	読み出しバッファ解放
	if (pBuf != NULL) free(pBuf);

	return rc;
}
//==============================================================================================================//
//	生成済ファイルのハンドルを使用してファイル生成																//
//																												//
//	引　数	：	hFile	-	生成済ファイルのハンドル															//
//				tec		-	テキストエンコード種別																//
//				fBOM	-	BOM出力フラグ																		//
//																												//
//	戻り値	：	≠ NULL：ＯＫ（テキストファイルアクセスハンドル）												//
//				＝ NULL：エラー																					//
//==============================================================================================================//
AJCEXPORT	HAJCFILE	WINAPI	AjcFAttachCreated	(HANDLE hFile, EAJCTEC tec, BOOL fBOM)
{
	HAJCFILE	rc = NULL;

	if (hFile != NULL && hFile != INVALID_HANDLE_VALUE) {
		rc = SubFCreateW(NULL, tec, fBOM, hFile, 0);	//	hFile ≠ NULL (or INVALID_HANDLE_VALUE) の場合、shareモードは無視される
	}
	return rc;
}

//==============================================================================================================//
//	生成済ファイルのハンドルを使用してファイル追記																//
//																												//
//	引　数	：	hFile	-	生成済ファイルのハンドル															//
//				tec		-	テキストエンコード種別																//
//				fBOM	-	BOM出力フラグ																		//
//																												//
//	戻り値	：	≠ NULL：ＯＫ（テキストファイルアクセスハンドル）												//
//				＝ NULL：エラー																					//
//==============================================================================================================//
AJCEXPORT	HAJCFILE	WINAPI	AjcFAttachAppend	(HANDLE hFile, EAJCTEC tec, BOOL fBOM)
{
	HAJCFILE	rc = NULL;

	if (hFile != NULL && hFile != INVALID_HANDLE_VALUE) {
		rc = SubFAppendW(NULL, tec, fBOM, hFile, 0);	//	hFile ≠ NULL (or INVALID_HANDLE_VALUE) の場合、shareモードは無視される
	}
	return rc;
}

//==============================================================================================================//
//	文字列出力																									//
//																												//
//	引　数	：	hFile	-	テキストファイルアクセスハンドル													//
//				pText	-	出力する文字列のアドレス															//
//				lText	-	出力する文字列のバイト数／文字数（－１：自動算出）									//
//																												//
//	戻り値	：	TRUE ：ＯＫ																						//
//				FALSE：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcFPutSA	(HAJCFILE pW, C_BCP pText, UI lText)
{
	BOOL		rc = TRUE;

	if (pW != NULL && pW->iid == INST_ID && !pW->fInp && pText != NULL && lText != 0) {
		WriteBOM(pW);	//	ＢＯＭ出力
		switch (pW->tec) {
			case AJCTEC_MBC:		rc = WriteMbcA 	(pW, pText, lText);			break;
			case AJCTEC_UTF_8:		rc = WriteUtf8A	(pW, pText, lText);			break;
			case AJCTEC_EUC_J:		rc = WriteEucA	(pW, pText, lText);			break;
			case AJCTEC_UTF_16LE:	rc = WriteUniA	(pW, pText, lText, FALSE);	break;
			case AJCTEC_UTF_16BE:	rc = WriteUniA	(pW, pText, lText, TRUE);	break;
		}
	}
	return	rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcFPutSW	(HAJCFILE pW, C_WCP pText, UI lText)
{
	BOOL	rc = TRUE;

	if (pW != NULL && pW->iid == INST_ID && !pW->fInp && pText != NULL && lText != 0) {
		WriteBOM(pW);	//	ＢＯＭ出力
		switch (pW->tec) {
			case AJCTEC_MBC:		rc = WriteMbcW	(pW, pText, lText);			break;
			case AJCTEC_UTF_8:		rc = WriteUtf8W	(pW, pText, lText);			break;
			case AJCTEC_EUC_J:		rc = WriteEucW	(pW, pText, lText);			break;
			case AJCTEC_UTF_16LE:	rc = WriteUniW	(pW, pText, lText, FALSE);	break;
			case AJCTEC_UTF_16BE:	rc = WriteUniW	(pW, pText, lText, TRUE);	break;
		}
	}
	return	rc;
}

//==============================================================================================================//
//	１バイト／１ワード出力																						//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//				c		-	出力するバイト／ワードデータ														//
//																												//
//	戻り値	：	TRUE ：ＯＫ																						//
//				FALSE：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcFPutCA	(HAJCFILE pW, BC c)
{
	BOOL	rc = FALSE;

	if (pW != NULL && pW->iid == INST_ID && !pW->fInp) {
		//	ＢＯＭ出力
		WriteBOM(pW);
		//	マルチバイト１文字目退避あり？
		if (pW->SvOutC != 0) {
			BC		mbc[2];
			mbc[0] = (BC)((UB)pW->SvOutC);
			mbc[1] = c;
			pW->SvOutC = 0;
			rc = AjcFPutSA(pW, mbc, 2);
		}
		//	マルチバイト１文字目退避なし？
		else {
			if (MAjcIsLeadA(c)) {			//	マルチバイト１バイト目？
				pW->SvOutC = c;
				rc = TRUE;
			}
			else {							//	１バイト文字？
				rc = AjcFPutSA(pW, &c, 1);
			}
		}
		rc = TRUE;
	}
	return	rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcFPutCW	(HAJCFILE pW, WC c)
{
	BOOL	rc = FALSE;

	if (pW != NULL && pW->iid == INST_ID && !pW->fInp) {
		//	ＢＯＭ出力
		WriteBOM(pW);
		//	サロゲートペア１文字目退避あり？
		if (pW->SvOutC != 0) {
			WC		mwc[2];
			mwc[0] = (WC)pW->SvOutC;
			mwc[1] = c;
			pW->SvOutC = 0;
			rc = AjcFPutSW(pW, mwc, 2);
		}
		//	サロゲートペア１文字目退避なし？
		else {
			if (MAjcIsLeadW(c)) {			//	サロゲートペア１ワード目？
				pW->SvOutC = c;
				rc = TRUE;
			}
			else {							//	１ワード文字？
				rc = AjcFPutSW(pW, &c, 1);
			}
		}
	}
	return	rc;
}


//==============================================================================================================//
//	１文字出力																									//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//				c		-	出力する文字																		//
//																												//
//	戻り値	：	TRUE ：ＯＫ																						//
//				FALSE：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcFPutCodeA	(HAJCFILE pW, UI c)
{
	BOOL	rc = FALSE;
	union {
		BC	mbc[2];
		UW	uw;
	} u;

	if (pW != NULL && pW->iid == INST_ID && !pW->fInp) {
		//	ＢＯＭ出力
		WriteBOM(pW);
		//	文字出力
		u.uw = (UW)c;
		rc = AjcFPutSA(pW, u.mbc, 2);
	}
	return	rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcFPutCodeW	(HAJCFILE pW, UI c)
{
	BOOL	rc = FALSE;
	WC		uni[4] = {0};

	if (pW != NULL && pW->iid == INST_ID && !pW->fInp) {
		//	ＢＯＭ出力
		WriteBOM(pW);
		//	文字出力
		AjcCodePointToText(c, uni);
		rc = AjcFPutSW(pW, uni, -1);
	}
	return	rc;
}

//==============================================================================================================//
//	書式文字列出力（可変個パラメタ指定）																		//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//				pFmt	-	書式文字列																			//
//																												//
//	戻り値	：	TRUE ：ＯＫ																						//
//				FALSE：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcFPrintFA	(HAJCFILE pW, C_BCP pFmt, ...)
{
	int		rc = 0;
	va_list vls;

	if (pW != NULL && pW->iid == INST_ID && !pW->fInp && pFmt != NULL) {
		//	書式文字列出力
		va_start(vls, pFmt);
		rc = AjcFVPrintFA(pW, pFmt, vls);
		va_end	(vls);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcFPrintFW	(HAJCFILE pW, C_WCP pFmt, ...)
{
	BOOL	rc = FALSE;
	va_list vls;

	if (pW != NULL && pW->iid == INST_ID && !pW->fInp && pFmt != NULL) {
		//	書式文字列出力
		va_start(vls, pFmt);
		rc = AjcFVPrintFW(pW, pFmt, vls);
		va_end	(vls);
	}
	return rc;
}
//==============================================================================================================//
//	書式文字列出力（可変個パラメタのポインタ指定）																//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//				pFmt	-	書式文字列																			//
//				vls		-	可変個引数の先頭ポインタ															//
//																												//
//	戻り値	：	TRUE ：ＯＫ																						//
//				FALSE：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcFVPrintFA	(HAJCFILE pW, C_BCP pFmt, va_list vls)
{
	int		rc = 0;

	if (pW != NULL && pW->iid == INST_ID && !pW->fInp && pFmt != NULL) {
		//	書式文字列バッファ確保
		if (pW->pFmtBuf.bcp == NULL) {
			pW->pFmtBuf.bcp = (BCP)AJCMEM(pW->lFmtBuf * 2);	// バッファはワイド文字分で確保
		}
		//	書式文字列作成，ファイル出力
		if (pW->pFmtBuf.bcp != NULL) {
			//	書式文字列作成
			pW->pFmtBuf.bcp[pW->lFmtBuf - 1] = 0;
			_vsnprintf(pW->pFmtBuf.bcp, pW->lFmtBuf, pFmt, vls);
			if (pW->pFmtBuf.bcp[pW->lFmtBuf - 1] == 0) {
				rc = TRUE;
			}
			else {
				pW->pFmtBuf.bcp[pW->lFmtBuf - 1] = 0;
				rc = FALSE;
			}
			//	ファイル書き込み
			rc &= AjcFPutSA(pW, pW->pFmtBuf.bcp, -1);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcFVPrintFW	(HAJCFILE pW, C_WCP pFmt, va_list vls)
{
	BOOL	rc = FALSE;

	if (pW != NULL && pW->iid == INST_ID && !pW->fInp && pFmt != NULL) {
		//	書式文字列バッファ確保
		if (pW->pFmtBuf.wcp == NULL) {
			pW->pFmtBuf.wcp = (WCP)AJCMEM(pW->lFmtBuf * 2);
		}
		//	書式文字列作成，ファイル出力
		if (pW->pFmtBuf.wcp != NULL) {
			//	書式文字列作成
			pW->pFmtBuf.wcp[pW->lFmtBuf - 1] = 0;
			_vsnwprintf(pW->pFmtBuf.wcp, pW->lFmtBuf, pFmt, vls);
			if (pW->pFmtBuf.wcp[pW->lFmtBuf - 1] == 0) {
				rc = TRUE;
			}
			else {
				pW->pFmtBuf.wcp[pW->lFmtBuf - 1] = 0;
				rc = FALSE;
			}
			//	ファイル書き込み
			rc &= AjcFPutSW(pW, pW->pFmtBuf.wcp, -1);
		}
	}
	return rc;
}
//==============================================================================================================//
//	テキストファイル 出力データフラッシュ																		//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//																												//
//	戻り値	：	TRUE ：ＯＫ																						//
//				FALSE：エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcFFlush	(HAJCFILE pW)
{
	BOOL	rc = FALSE;

	if (pW != NULL && pW->iid == INST_ID && !pW->fInp) {
		FlushBuf(pW);
		rc = TRUE;
	}
	return	rc;
}

//==============================================================================================================//
//	テキストファイルクローズ																					//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//																												//
//	戻り値	：	TRUE ：ＯＫ																						//
//				FALSE：エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcFClose	(HAJCFILE pW)
{
	BOOL	rc = FALSE;

	if (pW != NULL && pW->iid == INST_ID) {
		//	出力データフラッシュ
		FlushBuf(pW);
		//	リソース解放
		if (pW->pFmtBuf.bcp != NULL 			   ) free(pW->pFmtBuf.bcp);
		if (pW->pInpTop.vop	!= NULL 			   ) UnmapViewOfFile(pW->pInpTop.vop);
		if (pW->hMap		!= NULL 			   ) CloseHandle	(pW->hMap		);
		if (pW->hFile		!= INVALID_HANDLE_VALUE) CloseHandle	(pW->hFile		);
		free(pW);
		rc = TRUE;
	}
	return	rc;
}
//==============================================================================================================//
//	テキストファイルをクローズしないでインスタンス解放															//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//																												//
//	戻り値	：	TRUE ：ＯＫ																						//
//				FALSE：エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcFDetach	(HAJCFILE pW)
{
	BOOL	rc = FALSE;

	if (pW != NULL && pW->iid == INST_ID) {
		//	出力データフラッシュ
		FlushBuf(pW);
		//	リソース解放（システムのファイルハンドルは除く）
		if (pW->pFmtBuf.bcp != NULL) free(pW->pFmtBuf.bcp);
		if (pW->pInpTop.vop	!= NULL) UnmapViewOfFile(pW->pInpTop.vop);
		if (pW->hMap		!= NULL) CloseHandle	(pW->hMap		);
		free(pW);
		rc = TRUE;
	}
	return	rc;
}
//==============================================================================================================//
//	改行コード変換モード設定																					//
//																												//
//	引　数	：	pW			-	テキストファイルアクセスハンドル												//
//				InpLfMode	- 	入力改行モード																	//
//																												//
//	戻り値	：	TRUE ：ＯＫ																						//
//				FALSE：エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL			WINAPI	AjcFSetLfConv (HAJCFILE  pW, AJCFLFCONV mode)
{
	BOOL	rc = FALSE;

	if (pW != NULL && pW->iid == INST_ID) {
		if (mode >= AJCFLFCONV_MIN && mode <= AJCFLFCONV_MAX) {
			//	改行コード変換モード設定
			pW->LfCnv = mode;
			//	改行文字列設定
			SetLfCode(pW);
			//	戻り値＝ＯＫ
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	改行コード変換モード取得																					//
//																												//
//	引　数	：	pW			-	テキストファイルアクセスハンドル												//
//				InpLfMode	- 	入力改行モード																	//
//																												//
//	戻り値	：	≠ 0 ：入力改行モード																			//
//				＝ 0 ：エラー																					//
//==============================================================================================================//
AJCEXPORT	AJCFLFCONV	WINAPI	AjcFGetLfConv (HAJCFILE  pW)
{
	AJCFLFCONV	rc = 0;

	if (pW != NULL && pW->iid == INST_ID) {
		rc = pW->LfCnv;
	}
	return rc;
}


//==============================================================================================================//
//	出力改行モード設定																							//
//																												//
//	引　数	：	pW			-	テキストファイルアクセスハンドル												//
//				InpLfMode	- 	入力改行モード																	//
//																												//
//	戻り値	：	TRUE ：ＯＫ																						//
//				FALSE：エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL			WINAPI	AjcFSetOutLfMode	(HAJCFILE pW, AJCFLFCONV mode)
{
	BOOL	rc = FALSE;

	if (pW != NULL && pW->iid == INST_ID) {
		if (mode >= AJCFOLF_MIN && mode <= AJCFOLF_MAX) {
			pW->LfCnv = mode;
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	出力改行モード取得																							//
//																												//
//	引　数	：	pW			-	テキストファイルアクセスハンドル												//
//																												//
//	戻り値	：	≠ 0 ：出力改行モード																			//
//				＝ 0 ：エラー																					//
//==============================================================================================================//
AJCEXPORT	AJCFLFCONV	WINAPI	AjcFGetOutLfMode	(HAJCFILE pW)
{
	AJCFLFCONV	rc = 0;

	if (pW != NULL && pW->iid == INST_ID) {
		rc = pW->LfCnv;
	}
	return rc;
}


//==============================================================================================================//
//	テキストエンコード種別取得																					//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//																												//
//	戻り値	：	テキストエンコード種別																			//
//==============================================================================================================//
AJCEXPORT	EAJCTEC		WINAPI	AjcFGetTec	(HAJCFILE pW)
{
	EAJCTEC		rc = AJCTEC_ERROR;

	if (pW != NULL && pW->iid == INST_ID) {
		rc = pW->tec;
	}
	return	rc;
}
//==============================================================================================================//
//	入力ファイルのＢＯＭ有無の取得																				//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//																												//
//	戻り値	：	TRUE  - 入力ファイルにＢＯＭあり																//
//				FALSE - 入力ファイルにＢＯＭなし																//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcFGetBOM	(HAJCFILE pW)
{
	BOOL	rc = FALSE;

	if (pW != NULL && pW->iid == INST_ID) {
		rc = pW->fExistBOM;
	}
	return	rc;
}
//==============================================================================================================//
//	ファイル入出力種別取得																						//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//																												//
//	戻り値	：	TRUE  - 入力モード																				//
//				FALSE - 出力モード																				//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcFIsModeInput(HAJCFILE pW)
{
	BOOL		rc = TRUE;

	if (pW != NULL && pW->iid == INST_ID) {
		rc = pW->fInp;
	}
	return	rc;
}
//==============================================================================================================//
//	書式文字列バッファサイズ設定																				//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//				len 	-	書式文字列バッファ長（１６以上）													//
//																												//
//	戻り値	：	テキストエンコード種別																			//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcFSetFmtLen	(HAJCFILE pW, UI len)
{
	BOOL	rc = FALSE;

	if (pW != NULL && pW->iid == INST_ID && len >= 16) {
		if (pW->pFmtBuf.bcp != NULL || len != pW->lFmtBuf) {
			//	前回のバッファ解放
			if (pW->pFmtBuf.bcp == NULL) free(pW->pFmtBuf.bcp);
			//	バッファ確保
			if ((pW->pFmtBuf.bcp = (BCP)AJCMEM(len * 2)) != NULL) {	// バッファはワイド文字分で確保
				pW->lFmtBuf = len;
				rc = TRUE;
			}
		}
	}
	return	rc;
}

//==============================================================================================================//
//	ハンドル取得																								//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//																												//
//	戻り値	：	≠NULL : ファイルハンドル																		//
//==============================================================================================================//
AJCEXPORT	HANDLE		WINAPI	AjcFGetFileHandle	(HAJCFILE pW)
{
	HANDLE	rc = NULL;

	if (pW != NULL && pW->iid == INST_ID) {
		rc = pW->hFile;
	}
	return	rc;
}

//==============================================================================================================//
//	エンコード種別設定ダイアログ																				//
//																												//
//	引　数	：	hOwner	-	オーナーウインドハンドル -----------------------------------（不要時はNULL)			//
//				pTitle	-	タイトル文字列 ---------------------------------------------（不要時はNULL)			//
//				hIcon	-	アイコンハンドル -------------------------------------------（不要時はNULL)			//
//				pInpTec	-	入力テキスト文字コードバッファのアドレス -------------------（不要時はNULL)			//
//				pOutTec	-	出力テキスト文字コードバッファのアドレス -------------------（不要時はNULL)			//
//				pOutBom	-	出力テキストファイルへのＢＯＭ出力フラグバッファのアドレス -（不要時はNULL)			//
//																												//
//	戻り値	：	TRUE ：ＯＫ																						//
//				FALSE：キャンセル																				//
//==============================================================================================================//
AJC_DLGPROC_DEF(SetTec);
#define	WM_SHOWGRAY		WM_APP + 100
//--------------------------------------------------------------------------------------------------------------//
typedef struct {
	BOOL		fUni;
	C_VOP		pTitle;
	HICON		hIcon;
	PEAJCTEC	pInpTec;
	PEAJCTEC	pOutTec;
	BOOL		*pOutBom;
} CBP_SET_TEC, *PCBP_SET_TEC;
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcFTecDialogA(HWND hOwner, C_BCP pTitle, HICON hIcon, PEAJCTEC pInpTec, PEAJCTEC pOutTec, BOOL *pOutBom)
{
	BOOL	rc = FALSE;
	CBP_SET_TEC	cbp;

	cbp.fUni	= FALSE;
	cbp.pTitle	= (C_VOP)pTitle;
	cbp.hIcon	= hIcon;
	cbp.pInpTec = pInpTec;
	cbp.pOutTec = pOutTec;
	cbp.pOutBom = pOutBom;
	if (DialogBoxParam(hDllInst, MAKEINTRESOURCE(IDD_SET_TEC), hOwner, AJC_DLGPROC_NAME(SetTec), (LPARAM)&cbp) == IDOK) {
		rc = TRUE;
	}

	return	rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcFTecDialogW(HWND hOwner, C_WCP pTitle, HICON hIcon, PEAJCTEC pInpTec, PEAJCTEC pOutTec, BOOL *pOutBom)
{
	BOOL	rc = FALSE;
	CBP_SET_TEC	cbp;

	cbp.fUni	= TRUE;
	cbp.pTitle	= (C_VOP)pTitle;
	cbp.hIcon	= hIcon;
	cbp.pInpTec = pInpTec;
	cbp.pOutTec = pOutTec;
	cbp.pOutBom = pOutBom;
	if (DialogBoxParam(hDllInst, MAKEINTRESOURCE(IDD_SET_TEC), hOwner, AJC_DLGPROC_NAME(SetTec), (LPARAM)&cbp) == IDOK) {
		rc = TRUE;
	}

	return	rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	エンコード種別設定ダイアログプロシージャ																	//
//--------------------------------------------------------------------------------------------------------------//
//----- ダイアログ初期化 ---------------------------------------------------------------------------------------//
AJC_DLGPROC(SetTec, WM_INITDIALOG )
{
	PCBP_SET_TEC	pW = (PCBP_SET_TEC)lParam;

	//	ダイアログ用ワークアドレス設定
	SetProp(hDlg, L"AjcSetTec", (HANDLE)lParam);

	//	タイトル設定
	if (pW->pTitle != NULL) {
		if (pW->fUni) SetWindowTextW(hDlg, (C_WCP)pW->pTitle);
		else		  SetWindowTextA(hDlg, (C_BCP)pW->pTitle);
	}
	else  SetWindowText(hDlg, LNGSEL(L"テキスト・エンコード設定", L"Set Text encoding"));

	//	アイコン設定
	if (pW->hIcon != NULL) {
		SendMessage(hDlg, WM_SETICON, ICON_SMALL, (LPARAM)pW->hIcon);
	}
	//	日英テキスト
	AjcSetDlgItemStr(hDlg, IDC_GRP_I_ENC, LNGSEL(L"入力ファイル エンコード" 	, L"Input File Encoding"	));
	AjcSetDlgItemStr(hDlg, IDC_RBT_I_MBC, szMyMbc													 );

	AjcSetDlgItemStr(hDlg, IDC_GRP_O_ENC, LNGSEL(L"出力ファイル エンコード" 	, L"Output File Encoding"));
	AjcSetDlgItemStr(hDlg, IDC_RBT_O_MBC, szMyMbc													 );
	AjcSetDlgItemStr(hDlg, IDC_CHK_O_BOM, LNGSEL(L"ＢＯＭを作成する"			, L"Output BOM"			));

	AjcSetDlgItemStr(hDlg, IDOK    , LNGSEL(L"ＯＫ" 	 , L"OK"	));
	AjcSetDlgItemStr(hDlg, IDCANCEL, LNGSEL(L"キャンセル", L"Cancel"));

	if (AjcGetLangId() != AJCLID_JPN) {
		ShowWindow(GetDlgItem(hDlg, IDC_RBT_I_EUC), SW_HIDE);
		ShowWindow(GetDlgItem(hDlg, IDC_RBT_O_EUC), SW_HIDE);
	}

	//	ラジオボタングループ化
	AjcSbcRadioBtns(GetDlgItem(hDlg, IDC_GRP_INPTEC));
	AjcSbcRadioBtns(GetDlgItem(hDlg, IDC_GRP_OUTTEC));

	//	入力エンコード初期値表示
	if (pW->pInpTec != NULL) {
		switch (*pW->pInpTec) {
			default:
			case AJCTEC_MBC:		AjcSetDlgItemUInt(hDlg, IDC_GRP_INPTEC, 0);	break;
			case AJCTEC_UTF_8:		AjcSetDlgItemUInt(hDlg, IDC_GRP_INPTEC, 1);	break;
			case AJCTEC_EUC_J:		AjcSetDlgItemUInt(hDlg, IDC_GRP_INPTEC, 2);	break;
			case AJCTEC_UTF_16LE:	AjcSetDlgItemUInt(hDlg, IDC_GRP_INPTEC, 3);	break;
			case AJCTEC_UTF_16BE:	AjcSetDlgItemUInt(hDlg, IDC_GRP_INPTEC, 4);	break;
			case AJCTEC_AUTO:		AjcSetDlgItemUInt(hDlg, IDC_GRP_INPTEC, 5);	break;
		}
	}
	else {
		AjcEnableDlgGroup(hDlg, IDC_GRP_I_ENC, FALSE, FALSE);
	}

	//	出力エンコード初期値表示
	if (pW->pOutTec != NULL) {
		switch (*pW->pOutTec) {
			default:
			case AJCTEC_MBC:		AjcSetDlgItemUInt(hDlg, IDC_GRP_OUTTEC, 0);	break;
			case AJCTEC_UTF_8:		AjcSetDlgItemUInt(hDlg, IDC_GRP_OUTTEC, 1);	break;
			case AJCTEC_EUC_J:		AjcSetDlgItemUInt(hDlg, IDC_GRP_OUTTEC, 2);	break;
			case AJCTEC_UTF_16LE:	AjcSetDlgItemUInt(hDlg, IDC_GRP_OUTTEC, 3);	break;
			case AJCTEC_UTF_16BE:	AjcSetDlgItemUInt(hDlg, IDC_GRP_OUTTEC, 4);	break;
		}
	}

	//	出力ＢＯＭ初期値表示
	if (pW->pOutBom != NULL) {
		AjcSetDlgItemChk(hDlg, IDC_CHK_O_BOM, *pW->pOutBom != FALSE);
	}

	//	グレー表示処理
	SendMessage(hDlg, WM_SHOWGRAY, 0, 0);

	return TRUE;
}
//----- ダイアログのグレー表示処理 -----------------------------------------------------------------------------//
AJC_DLGPROC(SetTec, WM_SHOWGRAY )
{
	PCBP_SET_TEC	pW = (PCBP_SET_TEC)GetProp(hDlg, L"AjcSetTec");

	//	入力設定グレー化
	if (pW->pInpTec != NULL) {
		AjcEnableDlgGroup(hDlg, IDC_GRP_I_ENC, TRUE , TRUE );
	}
	else {
		AjcEnableDlgGroup(hDlg, IDC_GRP_I_ENC, FALSE, FALSE);
	}

	//	出力設定グレー化
	if (pW->pOutTec == NULL && pW->pOutBom == NULL) {					//	出力ＥＮＣ and ＢＯＭ無効？
		AjcEnableDlgGroup(hDlg, IDC_GRP_O_ENC, FALSE, FALSE);
	}
	else {																//	出力ＥＮＣ or ＢＯＭ無効？
		AjcEnableDlgGroup(hDlg, IDC_GRP_O_ENC, TRUE , TRUE );
		if		(pW->pOutTec == NULL) {									//		出力ＥＮＣ無効？
			AjcEnableDlgGroup(hDlg, IDC_GRP_OUTTEC, FALSE, FALSE);		//			出力エンコード設定グレー化
		}
		else if (pW->pOutBom == NULL) {									//		出力ＢＯＭ無効？
			AjcEnableDlgItem(hDlg, IDC_CHK_O_BOM, FALSE);				//			出力ＢＯＭグレー化
		}
		else {															//		出力ＥＮＣ and ＢＯＭ有効？
			if (AjcGetDlgItemChk(hDlg, IDC_RBT_O_UTF8)	||				//			UTF-8 or UTF-16 チェック？
				AjcGetDlgItemChk(hDlg, IDC_RBT_O_U16LE) ||				//			・
				AjcGetDlgItemChk(hDlg, IDC_RBT_O_U16BE)) {				//			・
				AjcEnableDlgItem(hDlg, IDC_CHK_O_BOM, TRUE);			//				出力ＢＯＭグレー化解除
			}
			else {														//			UTF-8 and UTF-16 未チェック？
				AjcEnableDlgItem(hDlg, IDC_CHK_O_BOM, FALSE);			//				出力ＢＯＭグレー化
			}
		}
	}
	return TRUE;
}

//----- キャンセルボタン ---------------------------------------------------------------------------------------//
AJC_DLGPROC(SetTec, IDCANCEL )
{
	PCBP_SET_TEC	pW = (PCBP_SET_TEC)GetProp(hDlg, L"AjcSetTec");

	EndDialog(hDlg, IDCANCEL);

	return TRUE;
}

//----- ＯＫボタン ---------------------------------------------------------------------------------------------//
AJC_DLGPROC(SetTec, IDOK	 )
{
	PCBP_SET_TEC	pW = (PCBP_SET_TEC)GetProp(hDlg, L"AjcSetTec");

	if (pW->pInpTec != NULL) {
		switch (AjcGetDlgItemUInt(hDlg, IDC_GRP_INPTEC)) {
			case 0: *pW->pInpTec = AJCTEC_MBC;		break;
			case 1: *pW->pInpTec = AJCTEC_UTF_8;	break;
			case 2: *pW->pInpTec = AJCTEC_EUC_J;	break;
			case 3: *pW->pInpTec = AJCTEC_UTF_16LE;	break;
			case 4: *pW->pInpTec = AJCTEC_UTF_16BE;	break;
			case 5: *pW->pInpTec = AJCTEC_AUTO;		break;
		}
	}
	if (pW->pOutTec != NULL) {
		switch (AjcGetDlgItemUInt(hDlg, IDC_GRP_OUTTEC)) {
			case 0: *pW->pOutTec = AJCTEC_MBC;		break;
			case 1: *pW->pOutTec = AJCTEC_UTF_8;	break;
			case 2: *pW->pOutTec = AJCTEC_EUC_J;	break;
			case 3: *pW->pOutTec = AJCTEC_UTF_16LE;	break;
			case 4: *pW->pOutTec = AJCTEC_UTF_16BE;	break;
		}
	}
	if (pW->pOutBom != NULL) {
		*pW->pOutBom = AjcGetDlgItemChk(hDlg, IDC_CHK_O_BOM);
	}
	EndDialog(hDlg, IDOK);

	return TRUE;
}
//----- 入力エンコード設定 -------------------------------------------------------------------------------------//
AJC_DLGPROC(SetTec, IDC_GRP_INPTEC )
{
	//	グレー表示処理
	SendMessage(hDlg, WM_SHOWGRAY, 0, 0);

	return TRUE;
}
//----- 出力エンコード設定 -------------------------------------------------------------------------------------//
AJC_DLGPROC(SetTec, IDC_GRP_OUTTEC )
{
	//	グレー表示処理
	SendMessage(hDlg, WM_SHOWGRAY, 0, 0);

	return TRUE;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_DLGMAP_DEF(SetTec)
	AJC_DLGMAP_MSG(SetTec, WM_INITDIALOG	)
	AJC_DLGMAP_MSG(SetTec, WM_SHOWGRAY		)

	AJC_DLGMAP_CMD(SetTec, IDOK				)
	AJC_DLGMAP_CMD(SetTec, IDCANCEL			)
	AJC_DLGMAP_CMD(SetTec, IDC_GRP_INPTEC	)
	AJC_DLGMAP_CMD(SetTec, IDC_GRP_OUTTEC	)
AJC_DLGMAP_END



//--------------------------------------------------------------------------------------------------------------//
//	ユニコード（ビッグエンディアン）１文字（１～２ワード））をマルチバイトに変換								//
//																												//
//	戻り値	：	マルチバイト文字のバイト数																		//
//--------------------------------------------------------------------------------------------------------------//
static	UI		RevedWideCharToMultiByte(C_WCP pUni, UI lUni, BCP pMbc, UI lMbc)
{
	UI		rc	 = 0;
	UI		i;
	WC		tmp[8] = {0};

	for (i = 0; i < lUni && i < AJCTSIZE(tmp); i++) {
		tmp[i] = (WC)AjcExcWord(pUni[i]);
	}
	rc = WideCharToMultiByte(CP_ACP, 0, tmp, lUni, pMbc, lMbc, NULL, NULL);

	return rc;
}

//--------------------------------------------------------------------------------------------------------------//
//	ファイル先頭部分のＢＯＭから、テキストエンコードを得る														//
//																												//
//	引　数	;	fsize			- ファイルサイズ（バイト数）													//
//				pDat			- ファイル先頭部分のデータイメージへのポインタ									//
//				lDat			- ファイル先頭部分のデータイメージのバイト数									//
//				prc_fExistBOM	- ＢＯＭ有無を受け取るバッファへのポインタ										//
//				prc_FIx			- ファイル読み出し位置（インデクス）を受け取るバッファへのポインタ				//
//				prc_pInp		- ファイル読み出し位置（アドレス  ）を受け取るバッファへのポインタ				//
//																												//
//	戻り値	：	AJCTEC_ERROR：ＢＯＭなし																		//
//				その他		：テキストエンコード種別															//
//--------------------------------------------------------------------------------------------------------------//
static EAJCTEC GetTecByBOM(ULL fsize, BCP pDat, UI lDat, BOOL *prc_fExistBOM, ULL *prc_FIx, AJCFUPTR *prc_pInp)
{
	EAJCTEC		rc		  = AJCTEC_ERROR;
	UI			lBOM	  = 0;
	BOOL		fExistBOM = FALSE;
	ULL			FIx 	  = 0;
	AJCFUPTR	pInp;

	pInp.bcp = pDat;

	//	BOMチェック
	if		(fsize >= (sizeof bom_utf8	 ) && memcmp(pDat, bom_utf8   , sizeof bom_utf8   ) == 0) {	//	UTF-8
		rc 		  = AJCTEC_UTF_8;
		fExistBOM = TRUE;
		FIx 	  = 3;
		pInp.bcp += 3;
	}
	else if (fsize >= (sizeof bom_utf16le) && memcmp(pDat, bom_utf16le, sizeof bom_utf16le) == 0) {	//	UTF-16LE
		rc 		  = AJCTEC_UTF_16LE;
		fExistBOM = TRUE;
		FIx 	  = 1;
		pInp.wcp += 1;
	}
	else if (fsize >= (sizeof bom_utf16be) && memcmp(pDat, bom_utf16be, sizeof bom_utf16be) == 0) {	//	UTF-16BE
		rc 		  = AJCTEC_UTF_16BE;
		fExistBOM = TRUE;
		FIx 	  = 1;
		pInp.wcp += 1;
	}
	else {																							//	ＢＯＭなし
		//	ＢＯＭなしの旨、設定
		fExistBOM = FALSE;
		//	ファイルポインタを先頭に設定
		FIx 	  = 0;
		pInp.bcp  = pDat;
	}

	//	戻り情報設定
	if (prc_fExistBOM != NULL) *prc_fExistBOM = fExistBOM;
	if (prc_FIx 	  != NULL) *prc_FIx 	  = FIx;
	if (prc_pInp	  != NULL) *prc_pInp	  = pInp;

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	ファイル先頭部分のテキストから、テキストエンコードを得る													//
//																												//
//	引　数	;	fsize			- ファイルサイズ（バイト数）													//
//				pDat			- ファイル先頭部分のデータイメージへのポインタ									//
//				lDat			- ファイル先頭部分のデータイメージのバイト数									//
//																												//
//	戻り値	：	テキストエンコード種別																			//
//--------------------------------------------------------------------------------------------------------------//
static EAJCTEC GetTecByText(ULL fsize, BCP pDat, UI lDat)
{
	EAJCTEC 	rc		  = AJCTEC_MBC;	//	戻り値(S-JIS)

	do {
		//	BOMチェック
		if		(fsize >= (sizeof bom_utf8	 ) && memcmp(pDat, bom_utf8   , sizeof bom_utf8   ) == 0) {	//	UTF-8
			pDat += 3;
		}
		else if (fsize >= (sizeof bom_utf16le) && memcmp(pDat, bom_utf16le, sizeof bom_utf16le) == 0) {	//	UTF-16LE
			pDat += 2;
		}
		else if (fsize >= (sizeof bom_utf16be) && memcmp(pDat, bom_utf16be, sizeof bom_utf16be) == 0) {	//	UTF-16BE
			pDat += 2;;
		}
		else {																							//	ＢＯＭなし
			//	NOP
		}
		//	テキストエンコード種別設定
		{
			int		perSJis = 0;		//	S-JIS 確率
			int		perEuc	= 0;		//	EUC-J 確率
			int		perUtf8 = 0;		//	UTF-8 確率
			int		lpi 	= 0;		//	
			UI		len = (UI)(__min(fsize, STBA_BUFSIZE));
			if ((fsize & 1) == 0) {
				//	UTF-16LE
				lpi = IS_TEXT_UNICODE_CONTROLS;
				IsTextUnicode(pDat, len, &lpi);
				if (lpi == IS_TEXT_UNICODE_CONTROLS) {
					rc = AJCTEC_UTF_16LE;
					break;
				}
				//	UTF-16BE
				lpi = IS_TEXT_UNICODE_REVERSE_CONTROLS;
				IsTextUnicode(pDat, len, &lpi);
				if (lpi == IS_TEXT_UNICODE_REVERSE_CONTROLS) {
					rc = AJCTEC_UTF_16BE;
					break;
				}
			}
			//	MBC(SJIS), EUC-J or UTF-8 チェック
			perSJis = AjcStrIsSJisEx(pDat, len);
			perEuc	= AjcStrIsEucEx (pDat, len);
			perUtf8 = AjcStrIsUtf8Ex(pDat, len);
			if		(perUtf8 > perSJis &&  perUtf8 > perEuc ) {
				rc = AJCTEC_UTF_8;
			}
			else if (perEuc  > perSJis &&  perEuc  > perUtf8) {
				rc = AJCTEC_EUC_J;
			}
			else {
				rc = AJCTEC_MBC;
			}
		}
	} while(0);

	return rc;
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
//																												//
//	文字列読み出し用サブ関数																					//
//																												//
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
//--------------------------------------------------------------------------------------------------------------//
//	改行文字列設定																								//
//--------------------------------------------------------------------------------------------------------------//
static	VO		SetLfCode(HAJCFILE pW)
{
	//	改行文字列設定
	memset(pW->szLfA, 0, sizeof pW->szLfA);
	memset(pW->szLfW, 0, sizeof pW->szLfW);
	switch (pW->LfCnv) {
		case AJCFLF_NONE:														break;	//	変換無し
		case AJCFLF_LF_TO_CRLF:		pW->szLfA[0] = CC_LF;						break;	//	ＬＦ → ＣＲ，ＬＦ（ファイル書き込み時のデフォルト）
		case AJCFLF_LF_TO_CR:		pW->szLfA[0] = CC_LF;						break;	//	ＬＦ → ＣＲ
		case AJCFLF_CR_TO_CRLF:		pW->szLfA[0] = CC_CR;						break;	//	ＣＲ → ＣＲ，ＬＦ
		case AJCFLF_CR_TO_LF:		pW->szLfA[0] = CC_CR;						break;	//	ＣＲ → ＬＦ
		case AJCFLF_CRLF_TO_LF:		pW->szLfA[0] = CC_CR; pW->szLfA[1] = CC_LF;	break;	//	ＣＲ，ＬＦ → ＬＦ（ファイル読み出し時のデフォルト）
		case AJCFLF_CRLF_TO_CR:		pW->szLfA[0] = CC_CR; pW->szLfA[1] = CC_LF;	break;	//	ＣＲ，ＬＦ → ＣＲ
		case AJCFLF_LF:				pW->szLfA[0] = CC_LF;						break;	//	ＬＦを改行コードとし、変換無し
		case AJCFLF_CR:				pW->szLfA[0] = CC_CR;						break;	//	ＣＲを改行コードとし、変換無し
		case AJCFLF_LF_CRSKIP:		pW->szLfA[0] = CC_LF;						break;	//	ＬＦを改行コードとし、ＣＲは除去する
		case AJCFLF_CR_LFSKIP:		pW->szLfA[0] = CC_CR;						break;	//	ＣＲを改行コードとし、ＬＦは除去する
	}
	pW->szLfW[0] = pW->szLfA[0];
	pW->szLfW[1] = pW->szLfA[1];
	pW->szLfW[2] = pW->szLfA[2];
	//	UTF-16(BE)の場合、上下バイト交換
	if (pW->tec == AJCTEC_UTF_16BE) {
		pW->szLfW[0] = (WC)AjcExcWord(pW->szLfW[0]);
		pW->szLfW[1] = (WC)AjcExcWord(pW->szLfW[1]);
	}
}
//--------------------------------------------------------------------------------------------------------------//
//	行テキスト（改行コード／ＥＯＦまでのテキスト）をコード変換してバッファに格納								//
//																												//
//	引	数	：	pW		- インスタンスハンドル																	//
//				pBuf	- 行テキスト格納バッファのアドレス														//
//				lBuf	- 行テキスト格納バッファのバイト数／ワード数											//
//																												//
//	戻り値	：	TRUE  : 行テキストをコード変換してバッファに格納した											//
//				FALSE : 格納バッファ不足																		//
//--------------------------------------------------------------------------------------------------------------//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
static	BOOL	GetLineA(HAJCFILE pW, BCP pBuf, UI lBuf)
{
	BOOL	rc = FALSE;
	UI		i;

	switch (pW->tec) {
		//	●Ｓ－ＪＩＳ ファイル
		case AJCTEC_MBC:
		{	UI		lMbc[2];	//	[0] : 改行コードまでの長さ，[1] : 制御文字までの長さ
			BCP		pMbc;
			//	改行／ＥＯＦ／制御文字までのテキスト長算出
			if (SearchLFA(pW, lBuf, &lMbc[0], &lMbc[1])) {
				for (i = 0; i < 2 && !rc; i++) {
					if (lMbc[i] != 0) {
						pMbc = pW->pInp.bcp;
						//	改行までのテキスト長 ＜ バッファ長
						if		(lMbc[i] != 0 && lMbc[i] < lBuf) {
							//	テキストをバッファに格納
							if (SetToLineBufA(pW, pMbc, lMbc[i], pBuf, lBuf)) {
								//	ファイルポインタ更新
								pW->pInp.bcp += lMbc[i];
								pW->FIx 	 += lMbc[i];
								rc = TRUE;
							}
						}
					}
				}
			}
			break;
		}
		//	●ＵＦＦ－８ ファイル
		case AJCTEC_UTF_8:
		{	UI		lUtf8[2];	//	[0] : 改行コードまでの長さ，[1] : 制御文字までの長さ
			UI		lMbc;
			BCP		pMbc;
			//	改行／ＥＯＦ／制御文字までのテキスト長算出
			if (SearchLFA(pW, lBuf * 6, &lUtf8[0], &lUtf8[1])) {
				for (i = 0; i < 2 && !rc; i++) {
					if (lUtf8[i] != 0) {
						//	UTF8 -> SJIS 変換のテキスト長算出（～改行コード）
						if ((lMbc = AjcUtf8ToSJisEx(pW->pInp.bcp, lUtf8[i], NULL, 0)) != 0) {
							//	ＳＪＩＳテキスト長 ＜ バッファ長
							if (lMbc < lBuf && (pMbc = AjcTAllocA(lMbc))) {
								//	UTF8 -> SJIS 変換
								AjcUtf8ToSJisEx(pW->pInp.bcp, lUtf8[i], pMbc, lMbc);
								//	テキストをバッファに格納
								if (SetToLineBufA(pW, pMbc, lMbc, pBuf, lBuf)) {
									//	ファイルポインタ更新
									pW->pInp.bcp += lUtf8[i];
									pW->FIx 	 += lUtf8[i];
									rc = TRUE;
								}
								AjcTFree(pMbc);
							}
						}
					}
				}
			}
			break;
		}
		//	●ＥＵＣ－Ｊ ファイル
		case AJCTEC_EUC_J:
		{	UI		lEuc[2];	//	[0] : 改行コードまでの長さ，[1] : 制御文字までの長さ
			UI		lMbc;
			BCP		pMbc;
			//	改行／ＥＯＦ／制御文字までのテキスト長算出
			if (SearchLFA(pW, lBuf * 2, &lEuc[0], &lEuc[1])) {
				for (i = 0; i < 2 && !rc; i++) {
					if (lEuc[i] != 0) {
						//	EUC -> SJIS 変換のテキスト長算出
						if ((lMbc = AjcEucToSJisEx(pW->pInp.bcp, lEuc[i], NULL, 0)) != 0) {
							//	ＳＪＩＳテキスト長 ＜ バッファ長
							if (lMbc < lBuf && (pMbc = AjcTAllocA(lMbc))) {
								//	EUC -> SJIS 変換
								AjcEucToSJisEx(pW->pInp.bcp, lEuc[i], pMbc, lMbc);
								//	テキストをバッファに格納
								if (SetToLineBufA(pW, pMbc, lMbc, pBuf, lBuf)) {
									//	ファイルポインタ更新
									pW->pInp.bcp += lEuc[i];
									pW->FIx 	 += lEuc[i];
									rc = TRUE;
								}
								AjcTFree(pMbc);
							}
						}
					}
				}
			}
			break;
		}
		//	●ＵＴＦ－１６（ＬＥ） ファイル
		case AJCTEC_UTF_16LE:
		{	UI		lU16le[2];	//	[0] : 改行コードまでの長さ，[1] : 制御文字までの長さ
			UI		lMbc;
			BCP		pMbc;
			//	改行／ＥＯＦ／制御文字までのテキスト長算出
			if (SearchLFW(pW, lBuf * 2, &lU16le[0], &lU16le[1])) {
				for (i = 0; i < 2 && !rc; i++) {
					if (lU16le[i] != 0) {
						//	UNICODE -> SJIS 変換のテキスト長算出
						if ((lMbc = WideCharToMultiByte(CP_ACP, 0, pW->pInp.wcp, lU16le[i], NULL, 0, NULL, NULL)) != 0) {
							//	ＳＪＩＳテキスト長 ＜ バッファ長
							if (lMbc < lBuf && (pMbc = AjcTAllocA(lMbc))) {
								//	UNICODE -> SJIS 変換
								WideCharToMultiByte(CP_ACP, 0, pW->pInp.wcp, lU16le[i], pMbc, lMbc, NULL, NULL);
								//	テキストをバッファに格納
								if (SetToLineBufA(pW, pMbc, lMbc, pBuf, lBuf)) {
									//	ファイルポインタ更新
									pW->pInp.wcp += lU16le[i];
									pW->FIx 	 += lU16le[i];
									rc = TRUE;
								}
								AjcTFree(pMbc);
							}
						}
					}
				}
			}
			break;
		}
		//	●ＵＴＦ－１６（ＢＥ） ファイル
		case AJCTEC_UTF_16BE:
		{	UI		lU16be[2];	//	[0] : 改行コードまでの長さ，[1] : 制御文字までの長さ
			UI		j, lMbc;
			BCP		pMbc;
			WCP		pU16be, p, q;
			//	改行／ＥＯＦ／制御文字までのテキスト長算出
			if (SearchLFW(pW, lBuf * 2, &lU16be[0], &lU16be[1])) {
				for (i = 0; i < 2 && !rc; i++) {
					if (lU16be[i] != 0) {
						//	UNICODEバッファ確保
						if (pU16be = AjcTAllocW(lU16be[i])) {
							//	BE -> LE 変換
							for (j = 0, p = pU16be, q = pW->pInp.wcp; j < lU16be[i]; j++, p++, q++) {
								*p = (WC)AjcExcWord(*q);
							}
							//	UNICODE -> SJIS 変換のテキスト長算出
							if ((lMbc = WideCharToMultiByte(CP_ACP, 0, pU16be, lU16be[i], NULL, 0, NULL, NULL)) != 0) {
								//	ＳＪＩＳテキスト長 ＜ バッファ長
								if (lMbc < lBuf && (pMbc = AjcTAllocA(lMbc))) {
									//	UNICODE -> SJIS 変換
									WideCharToMultiByte(CP_ACP, 0, pU16be, lU16be[i], pMbc, lMbc, NULL, NULL);
									//	テキストをバッファに格納
									if (SetToLineBufA(pW, pMbc, lMbc, pBuf, lBuf)) {
										//	ファイルポインタ更新
										pW->pInp.wcp += lU16be[i];
										pW->FIx 	 += lU16be[i];
										rc = TRUE;
									}
									AjcTFree(pMbc);
								}
							}
							AjcTFree(pU16be);
						}
					}
				}
			}
			break;
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
static	BOOL	GetLineW(HAJCFILE pW, WCP pBuf, UI lBuf)
{
	BOOL	rc = FALSE;
	UI		i;

	switch (pW->tec) {
		//	●Ｓ－ＪＩＳ ファイル
		case AJCTEC_MBC:
		{	UI		lMbc[2];	//	[0] : 改行コードまでの長さ，[1] : 制御文字までの長さ
			UI		lUni;
			WCP		pUni;
			//	改行／ＥＯＦ／制御文字までのテキスト長算出
			if (SearchLFA(pW, lBuf * 2, &lMbc[0], &lMbc[1])) {
				for (i = 0; i < 2 && !rc; i++) {
					if (lMbc[i] != 0) {
						//	S-JIS -> UNICODE 変換のテキスト長算出
						if ((lUni = MultiByteToWideChar(CP_ACP, 0, pW->pInp.bcp, lMbc[i], NULL, 0)) != 0) {
							//	UNICODEテキスト長 ＜ バッファ長
							if (lUni < lBuf && (pUni = AjcTAllocW(lUni))) {
								//	S-JIS -> UNICODE 変換
								MultiByteToWideChar(CP_ACP, 0, pW->pInp.bcp, lMbc[i], pUni, lUni);
								//	テキストをバッファに格納
								if (SetToLineBufW(pW, pUni, lUni, pBuf, lBuf)) {
									//	ファイルポインタ更新
									pW->pInp.bcp += lMbc[i];
									pW->FIx 	 += lMbc[i];
									rc = TRUE;
								}
								AjcTFree(pUni);
							}
						}
					}
				}
			}
			break;
		}
		//	●ＵＦＦ－８ ファイル
		case AJCTEC_UTF_8:
		{	UI	lUtf8[2];	//	[0] : 改行コードまでの長さ，[1] : 制御文字までの長さ
			UI	lUni;
			WCP			   pUni;
			//	改行／ＥＯＦ／制御文字までのテキスト長算出
			if (SearchLFA(pW, lBuf * 6, &lUtf8[0], &lUtf8[1])) {
				for (i = 0; i < 2 && !rc; i++) {
					if (lUtf8[i] != 0) {
						//	UTF-8 -> UNICODE 変換のテキスト長算出
						if ((lUni = MultiByteToWideChar(CP_UTF8, 0, pW->pInp.bcp, lUtf8[i], NULL, 0)) != 0) {
							//	UNICODEテキスト長 ＜ バッファ長
							if (lUni < lBuf && (pUni = AjcTAllocW(lUni))) {
								//	UTF-8 -> UNICODE 変換
								MultiByteToWideChar(CP_UTF8, 0, pW->pInp.bcp, lUtf8[i], pUni, lUni);
								//	テキストをバッファに格納
								if (SetToLineBufW(pW, pUni, lUni, pBuf, lBuf)) {
									//	ファイルポインタ更新
									pW->pInp.bcp += lUtf8[i];
									pW->FIx 	 += lUtf8[i];
									rc = TRUE;
								}
								AjcTFree(pUni);
							}
						}
					}
				}
			}
			break;
		}
		//	●ＥＵＣ－Ｊ ファイル
		case AJCTEC_EUC_J:
		{	UI		lEuc[2];	//	[0] : 改行コードまでの長さ，[1] : 制御文字までの長さ
			UI		lMbc, lUni;
			BCP		pMbc;
			WCP		pUni;
			//	改行／ＥＯＦ／制御文字までのテキスト長算出
			if (SearchLFA(pW, lBuf * 2, &lEuc[0], &lEuc[1])) {
				for (i = 0; i < 2 && !rc; i++) {
					if (lEuc[i] != 0) {
						//	EUC -> S-JIS 変換のテキスト長取得
						if ((lMbc = AjcEucToSJisEx(pW->pInp.bcp, lEuc[i], NULL, 0)) != 0) {
							//	S-JISバッファ確保
							if (pMbc = AjcTAllocA(lMbc)) {
								//	EUC -> S-JIS 変換
								AjcEucToSJisEx(pW->pInp.bcp, lEuc[i], pMbc, lMbc);
								//	S-JIS -> UNICODE 変換のテキスト長算出
								if ((lUni = MultiByteToWideChar(CP_ACP, 0, pMbc, lMbc, NULL, 0)) != 0) {
									//	UNICODEテキスト長 ＜ バッファ長
									if (lUni < lBuf && (pUni = AjcTAllocW(lUni))) {
										//	S-JIS -> UNICODE 変換
										MultiByteToWideChar(CP_ACP, 0, pMbc, lMbc, pUni, lUni);
										//	テキストをバッファに格納
										if (SetToLineBufW(pW, pUni, lUni, pBuf, lBuf)) {
											//	ファイルポインタ更新
											pW->pInp.bcp += lEuc[i];
											pW->FIx 	 += lEuc[i];
											rc = TRUE;
										}
										AjcTFree(pUni);
									}
								}
								AjcTFree(pMbc);
							}
						}
					}
				}
			}
			break;
			}
		//	●ＵＴＦ－１６ ファイル
		case AJCTEC_UTF_16LE:
		case AJCTEC_UTF_16BE:
		{	UI		lU16[2];	//	[0] : 改行コードまでの長さ，[1] : 制御文字までの長さ
			//	改行／ＥＯＦ／制御文字までのテキスト長算出
			if (SearchLFW(pW, lBuf, &lU16[0], &lU16[1])) {
				for (i = 0; i < 2 && !rc; i++) {
					if (lU16[i] != 0) {
						//	UNICODEテキスト長 ＜ バッファ長
						if (lU16[i] < lBuf) {
							//	テキストをバッファに格納（含 BE->LE変換）
							if (SetToLineBufW(pW, pW->pInp.wcp, lU16[i], pBuf, lBuf)) {
								//	ファイルポインタ更新
								pW->pInp.wcp += lU16[i];
								pW->FIx 	 += lU16[i];
								rc = TRUE;
							}
						}
					}
				}
			}
			break;
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	改行コード／ＥＯＦまでのバイト数／ワード数取得																//
//																												//
//	引	数	：	pW		- インスタンスハンドル																	//
//				limit	- 最大検索バイト数／ワード数															//
//				plLf	- 改行コード(or EOF)までのバイト数／ワード数を格納するバッファのアドレス				//
//				plCtl	- 制御文字までのバイト数／ワード数を格納するバッファのアドレス							//
//																												//
//	戻り値	：	TRUE   : 改行コード／制御文字有り																//
//				FALSE  : 改行コード＆制御文字無し																//
//--------------------------------------------------------------------------------------------------------------//
//----- バイト文字ファイル -------------------------------------------------------------------------------------//
static	BOOL	SearchLFA(HAJCFILE pW, UI limit, UIP plLf, UIP plCtl)
{
	UI		rc	= FALSE;
	UI		lLf;
	UI		len   = 0;
	ULL		fix	  = pW->FIx;
	BCP		pFPtr = pW->pInp.bcp;

	//	戻り情報初期化
	*plLf  = 0;
	*plCtl = 0;

	//	改行コードの認識有りならば、改行コード検索
	if (pW->LfCnv != AJCFLF_NONE) {
		//	改行コードの長さ(1～2)設定
		lLf = (UI)strlen(pW->szLfA);
		//	改行コード／ＥＯＦ検索
		while (fix < pW->FSizeA && len < limit) {
			//	改行コードならば、改行コード分を加算し終了
			if (fix + lLf <= pW->FSizeA && strncmp(pFPtr, pW->szLfA, lLf) == 0) {
				*plLf = len + lLf;
				rc = TRUE;
				break;
			}
			//	改行コード以外ならば、１バイト加算
			else {
				//	制御文字ならば、位置を返す
				if (*plCtl == 0 && iscntrl((UB)*pFPtr)) {
					*plCtl = len + 1;
					rc = TRUE;
				}
				//	ポインタ，カウンタ更新
				fix++;
				pFPtr++;
				len++;
			}
		}
	}
	//	改行コード認識無しならば、制御コード検索
	else {
		while (fix < pW->FSizeA && len < limit) {
			//	制御コードならば、制御コード分を加算し終了
			if (fix + 1 <= pW->FSizeA && iscntrl((UB)*pFPtr)) {
				*plCtl = len + 1;
				rc = TRUE;
				break;
			}
			//	制御コード以外ならば、１バイト加算
			else {
				fix++;
				pFPtr++;
				len++;
			}
		}
	}
	//	ＥＯＦまでの長さチェック
	if (*plLf == 0 && fix == pW->FSizeA && len < limit) {
		*plLf = len;
		rc = TRUE;
	}
	return rc;
}
//----- ワイド文字ファイル -------------------------------------------------------------------------------------//
static	BOOL	SearchLFW(HAJCFILE pW, UI limit, UIP plLf, UIP plCtl)
{
	UI		rc	= 0;
	UI		lLf;
	UI		len   = 0;
	ULL		fix	  = pW->FIx;
	WCP		pFPtr = pW->pInp.wcp;

	//	戻り情報初期化
	*plLf  = 0;
	*plCtl = 0;

	//	改行コードの認識有りならば、改行コード検索
	if (pW->LfCnv != AJCFLF_NONE) {
		//	改行コードの長さ(1～2)設定
		lLf = (UI)wcslen(pW->szLfW);
		//	改行コード／ＥＯＦ検索
		while (fix < pW->FSizeW && len < limit) {
			//	改行コードならば、改行コード分を加算し終了
			if (fix + lLf <= pW->FSizeW && wcsncmp(pFPtr, pW->szLfW, lLf) == 0) {
				*plLf = len + lLf;
				rc = TRUE;
				break;
			}
			//	改行コード以外ならば、１ワード加算
			else {
				//	制御文字ならば、位置を返す
				if (*plCtl == 0 && iswcntrl(*pFPtr)) {
					*plCtl = len + 1;
					rc = TRUE;
				}
				//	ポインタ，カウンタ更新
				fix++;
				pFPtr++;
				len++;
			}
		}
	}
	//	改行コード認識無しならば、制御コード検索
	else {
		while (fix < pW->FSizeW && len < limit) {
			//	制御コードならば、制御コード分を加算し終了
			if (fix + 1 <= pW->FSizeW && iswcntrl(*pFPtr)) {
				*plCtl = len + 1;
				rc = TRUE;
				break;
			}
			//	制御コード以外ならば、１ワード加算
			else {
				fix++;
				pFPtr++;
				len++;
			}
		}
	}
	//	ＥＯＦまでの長さチェック
	if (*plLf == 0 && fix == pW->FSizeW && len < limit) {
		*plLf = len;
		rc = TRUE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	行テキストをバッファに格納																					//
//																												//
//	引	数	：	pW		- インスタンスハンドル																	//
//				pMbc	- ソーステキストポインタ																//
//				lMbc	- ソーステキストのバイト数／ワード数													//
//				pBuf	- バッファポインタ																		//
//				lBuf	- バッファのバイト数／ワード数															//
//																												//
//	戻り値	：	TRUE  : ＯＫ																					//
//				FALSE : 格納バッファ不足																		//
//--------------------------------------------------------------------------------------------------------------//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
static	BOOL	SetToLineBufA(HAJCFILE pW, C_BCP pMbc, UI lMbc, BCP pBuf, UI lBuf)
{
	BOOL	rc = FALSE;
	BC		src[3] = {0};
	BC		cnv[3] = {0};
	BC		del    =  0;
	UI		lsrc, lcnv;

	switch (pW->LfCnv) {
		case AJCFLF_NONE:																		break;	//	変換無し
		case AJCFLF_LF_TO_CRLF:	src[0] = CC_LF; 				cnv[0] = CC_CR; cnv[1] = CC_LF; break;	//	ＬＦ → ＣＲ，ＬＦ（ファイル書き込み時のデフォルト）
		case AJCFLF_LF_TO_CR:	src[0] = CC_LF; 				cnv[0] = CC_CR; 				break;	//	ＬＦ → ＣＲ
		case AJCFLF_CR_TO_CRLF:	src[0] = CC_CR; 				cnv[0] = CC_CR; cnv[1] = CC_LF; break;	//	ＣＲ → ＣＲ，ＬＦ
		case AJCFLF_CR_TO_LF:	src[0] = CC_CR; 				cnv[0] = CC_LF; 				break;	//	ＣＲ → ＬＦ
		case AJCFLF_CRLF_TO_LF:	src[0] = CC_CR; src[1] = CC_LF; cnv[0] = CC_LF; 				break;	//	ＣＲ，ＬＦ → ＬＦ（ファイル読み出し時のデフォルト）
		case AJCFLF_CRLF_TO_CR:	src[0] = CC_CR; src[1] = CC_LF; cnv[0] = CC_CR; 				break;	//	ＣＲ，ＬＦ → ＣＲ
		case AJCFLF_LF:																			break;	//	ＬＦを改行コードとし、変換無し
		case AJCFLF_CR:																			break;	//	ＣＲを改行コードとし、変換無し
		case AJCFLF_LF_CRSKIP:	del = CC_CR;													break;	//	ＬＦを改行コードとし、ＣＲは除去する
		case AJCFLF_CR_LFSKIP:	del = CC_LF;													break;	//	ＣＲを改行コードとし、ＬＦは除去する
	}
	lsrc = (UI)strlen(src);
	lcnv = (UI)strlen(cnv);

	while (lMbc > 0) {															//	残ソーステキストあり？
		if (*pMbc != del) {														//		削除する改行コード以外？
			if (lsrc != 0 && lMbc >= lsrc && strncmp(pMbc, src, lsrc) == 0) {	//			ソーステキストは変換要？
				if (lBuf > lcnv) {												//				バッファに変換コード格納可能？
					strncpy(pBuf, cnv, lcnv);									//					バッファに変換コード格納
					pMbc += lsrc;												//					ソーステキストポインタ更新
					lMbc -= lsrc;												//					ソーステキスト長更新
					pBuf += lcnv;												//					バッファポインタ更新
					lBuf -= lcnv;												//					バッファ長更新
				}
				else {															//				バッファに変換コード格納不可？
					break;														//					→終了（バッファオーバー）
				}
			}
			else {																//			ソーステキストは変換不要？
				*pBuf++ = *pMbc++;												//				ソーステキストをバッファに転送
				lMbc--;															//				ソーステキスト長更新
				lBuf--;															//				バッファ長更新
			}
		}
		else {																	//		削除する改行コード？
			pMbc++;																//			ソーステキストポインタ更新
			lMbc--;																//			ソーステキスト長更新
		}
	}
	if (lMbc == 0) {															//	全ソーステキストをバッファに転送済み？
		if (lBuf > 0) {															//		バッファに空きあり？
			*pBuf++ = 0;														//			バッファに文字列終端設定
			lBuf--;																//			バッファ長更新
			rc = TRUE;															//			戻り値＝ＯＫ
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
static	BOOL	SetToLineBufW(HAJCFILE pW, C_WCP pUni, UI lUni, WCP pBuf, UI lBuf)
{
	BOOL	rc = FALSE;
	WC		src[3] = {0};
	WC		cnv[3] = {0};
	WC		del    =  0;
	UI		lsrc, lcnv;

	switch (pW->LfCnv) {
		case AJCFLF_NONE:																		break;	//	変換無し
		case AJCFLF_LF_TO_CRLF:	src[0] = CC_LF; 				cnv[0] = CC_CR; cnv[1] = CC_LF; break;	//	ＬＦ → ＣＲ，ＬＦ（ファイル書き込み時のデフォルト）
		case AJCFLF_LF_TO_CR:	src[0] = CC_LF; 				cnv[0] = CC_CR; 				break;	//	ＬＦ → ＣＲ
		case AJCFLF_CR_TO_CRLF:	src[0] = CC_CR; 				cnv[0] = CC_CR; cnv[1] = CC_LF; break;	//	ＣＲ → ＣＲ，ＬＦ
		case AJCFLF_CR_TO_LF:	src[0] = CC_CR; 				cnv[0] = CC_LF; 				break;	//	ＣＲ → ＬＦ
		case AJCFLF_CRLF_TO_LF:	src[0] = CC_CR; src[1] = CC_LF; cnv[0] = CC_LF; 				break;	//	ＣＲ，ＬＦ → ＬＦ（ファイル読み出し時のデフォルト）
		case AJCFLF_CRLF_TO_CR:	src[0] = CC_CR; src[1] = CC_LF; cnv[0] = CC_CR; 				break;	//	ＣＲ，ＬＦ → ＣＲ
		case AJCFLF_LF:																			break;	//	ＬＦを改行コードとし、変換無し
		case AJCFLF_CR:																			break;	//	ＣＲを改行コードとし、変換無し
		case AJCFLF_LF_CRSKIP:	del = CC_CR;													break;	//	ＬＦを改行コードとし、ＣＲは除去する
		case AJCFLF_CR_LFSKIP:	del = CC_LF;													break;	//	ＣＲを改行コードとし、ＬＦは除去する
	}
	lsrc = (UI)wcslen(src);
	lcnv = (UI)wcslen(cnv);

	while (lUni > 0) {															//	残ソーステキストあり？
		if (*pUni != del) {														//		削除する改行コード以外？
			if (lsrc != 0 && lUni >= lsrc && wcsncmp(pUni, src, lsrc) == 0) {	//			ソーステキストは変換要？
				if (lBuf > lcnv) {												//				バッファに変換コード格納可能？
					wcsncpy(pBuf, cnv, lcnv);									//					バッファに変換コード格納
					pUni += lsrc;												//					ソーステキストポインタ更新
					lUni -= lsrc;												//					ソーステキスト長更新
					pBuf += lcnv;												//					バッファポインタ更新
					lBuf -= lcnv;												//					バッファ長更新
				}
				else {															//				バッファに変換コード格納不可？
					break;														//					→終了（バッファオーバー）
				}
			}
			else {																//			ソーステキストは変換不要？
				if (pW->tec == AJCTEC_UTF_16BE) *pBuf++ = AjcExcWord(*pUni++);	//				ソーステキストをバッファに転送
				else							*pBuf++ = *pUni++;
				lUni--;															//				ソーステキスト長更新
				lBuf--;															//				バッファ長更新
			}
		}
		else {																	//		削除する改行コード？
			pUni++;																//			ソーステキストポインタ更新
			lUni--;																//			ソーステキスト長更新
		}
	}
	if (lUni == 0) {															//	全ソーステキストをバッファに転送済み？
		if (lBuf > 0) {															//		バッファに空きあり？
			*pBuf++ = 0;														//			バッファに文字列終端設定
			lBuf--;																//			バッファ長更新
			rc = TRUE;															//			戻り値＝ＯＫ
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	改行チェック	・現在の位置が改行コードならばＴＲＵＥを返す。												//
//					・この時、バッファに空きがあれば改行コードをバッファに格納し、ファイルポインタを進める		//
//																												//
//	引	数	：	pW		- インスタンスハンドル																	//
//				pTxt	- 入力テキスト																			//
//				lTxt	- 入力テキストの長さ（バイト数／文字数）												//
//				pLfInfo	- 出力する改行情報を格納するバッファ													//
//																												//
//	戻り値	：	１～２：改行を認識	（認識した改行文字列のバイト数／文字数）									//
//							pLfInfo->stl - 変換する改行文字列のバイト数／文字数（≠０で改行認識を示す）			//
//								≠０：改行を認識し、変換することを示す											//
//								＝０：この改行文字列は読み飛ばすことを示す										//
//							pLfInfo->str - 変換する改行文字列													//
//				０	  ：改行未認識	（改行以外の文字）															//
//--------------------------------------------------------------------------------------------------------------//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
static	UI	CheckLFA(HAJCFILE pW, C_BCP pTxt, ULL lTxt, PLFINFO pLfInfo)
{
	UI		rc = 0;
	UB		c1, c2;

	memset(pLfInfo, 0, sizeof(LFINFO));

	if (lTxt >= 1) {

		//	テキスト先頭の文字コード設定
		c1 = *pTxt;
		if (lTxt >= 2) c2 = *(pTxt + 1);
		else		   c2 = 0;
		//	改行コード変換
		switch (pW->LfCnv) {
			case AJCFLF_NONE:												//	変換無し
				rc = 0;
				break;

			case AJCFLF_LF_TO_CRLF:											//	ＬＦ → ＣＲ，ＬＦ（ファイル書き込み時のデフォルト）
				if (c1 == CC_LF) {											//	ＬＦ？
					pLfInfo->stl	 = 2;									//		変換文字列長＝ ２
					pLfInfo->strA[0] = CC_CR;	pLfInfo->strW[0] = CC_CR;	//		変換文字列	＝ ＣＲ，ＬＦ
					pLfInfo->strA[1] = CC_LF;	pLfInfo->strW[1] = CC_LF;	//		・
					rc = 1;													//		認識した改行コード長＝１
				}
				break;

			case AJCFLF_LF_TO_CR:											//	ＬＦ → ＣＲ
				if (c1 == CC_LF) {											//	ＬＦ？
					pLfInfo->stl	 = 1;									//		変換文字列長＝ １
					pLfInfo->strA[0] = CC_CR;	pLfInfo->strW[0] = CC_CR;	//		変換文字列	＝ ＣＲ
					rc = 1;													//		認識した改行コード長＝１
				}
				break;

			case AJCFLF_CR_TO_CRLF:											//	ＣＲ → ＣＲ，ＬＦ
				if (c1 == CC_CR) {											//	ＣＲ？
					pLfInfo->stl	 = 2;									//		変換文字列長＝ １
					pLfInfo->strA[0] = CC_CR;	pLfInfo->strW[0] = CC_CR;	//		変換文字列	＝ ＣＲ，ＬＦ
					pLfInfo->strA[1] = CC_LF;	pLfInfo->strW[1] = CC_LF;	//		・
					rc = 1;													//		認識した改行コード長＝１
				}
				break;

			case AJCFLF_CR_TO_LF:											//	ＣＲ → ＬＦ
				if (c1 == CC_CR) {											//	ＣＲ？
					pLfInfo->stl	 = 1;									//		変換文字列長＝ １
					pLfInfo->strA[0] = CC_LF;	pLfInfo->strW[0] = CC_LF;	//		変換文字列	＝ ＬＦ
					rc = 1;													//		認識した改行コード長＝１
				}
				break;

			case AJCFLF_CRLF_TO_LF:											//	ＣＲ，ＬＦ → ＬＦ（ファイル読み出し時のデフォルト）
				if (c1 == CC_CR && c2 == CC_LF) {							//	ＣＲ，ＬＦ？
					pLfInfo->stl	 = 1;									//		変換文字列長＝ １
					pLfInfo->strA[0] = CC_LF;	pLfInfo->strW[0] = CC_LF;	//		変換文字列	＝ ＬＦ
					rc = 2;													//		認識した改行コード長＝２
				}
				break;

			case AJCFLF_CRLF_TO_CR:											//	ＣＲ，ＬＦ → ＣＲ
				if (c1 == CC_CR && c2 == CC_LF) {							//	ＣＲ，ＬＦ？
					pLfInfo->stl	 = 1;									//		変換文字列長＝ １
					pLfInfo->strA[0] = CC_CR;	pLfInfo->strW[0] = CC_CR;	//		変換文字列	＝ ＣＲ
					rc = 2;													//		認識した改行コード長＝２
				}
				break;

			case AJCFLF_LF:													//	ＬＦを改行コードとし、変換無し
				if (c1 == CC_LF) {											//		ＬＦ？
					rc = 1;													//			認識した改行コード長＝１
					pLfInfo->stl	 = 1;									//			変換文字列長＝ １
					pLfInfo->strA[0] = CC_LF;	pLfInfo->strW[0] = CC_LF;	//			変換文字列	＝ ＬＦ
				}
				break;

			case AJCFLF_CR:													//	ＣＲを改行コードとし、変換無し
				if (c1 == CC_CR) {											//		ＣＲ？
					pLfInfo->stl	 = 1;									//			変換文字列長＝ １
					pLfInfo->strA[0] = CC_CR;	pLfInfo->strW[0] = CC_CR;	//			変換文字列	＝ ＬＦ
					rc = 1;													//			認識した改行コード長＝１
				}
				break;

			case AJCFLF_LF_CRSKIP:											//	ＬＦを改行コードとし、ＣＲは除去する
				if		(c1 == CC_CR) {										//		ＣＲ？
					rc = 1;													//			認識した改行コード長＝１
				}
				else if (c1 == CC_LF) {										//		ＬＦ？
					pLfInfo->stl	 = 1;									//			変換文字列長＝ １
					pLfInfo->strA[0] = CC_LF;	pLfInfo->strW[0] = CC_LF;	//			変換文字列	＝ ＬＦ
					rc = 1;													//			認識した改行コード長＝１
				}
				break;

			case AJCFLF_CR_LFSKIP:											//	ＣＲを改行コードとし、ＬＦは除去する
				if		(c1 == CC_CR) {										//		ＣＲ？
					pLfInfo->stl	 = 1;									//			変換文字列長＝ １
					pLfInfo->strA[0] = CC_CR;	pLfInfo->strW[0] = CC_CR;	//			変換文字列	＝ ＬＦ
					rc = 1;													//			認識した改行コード長＝１
				}
				if (c1 == CC_LF) {											//		ＬＦ？
					rc = 1;													//			認識した改行コード長＝１
				}
				break;

		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
static	UI	CheckLFW(HAJCFILE pW, C_WCP pTxt, ULL lTxt, PLFINFO pLfInfo, BOOL fBE)
{
	UI		rc = 0;
	UW		c1, c2;
	BC		bc[2];

	memset(pLfInfo, 0, sizeof(LFINFO));

	if (lTxt >= 1) {
		//	テキスト先頭の文字コード設定
		c1 = *pTxt;
		if (lTxt >= 2) c2 = *(pTxt + 1);
		else		   c2 = 0;

		//	エンディアン変換
		if (fBE) {
			c1 = (WC)AjcExcWord(c1);
			c2 = (WC)AjcExcWord(c2);
		}
		//	改行コード変換
		if (pW->LfCnv == AJCFLF_CRLF_TO_LF || pW->LfCnv == AJCFLF_CRLF_TO_CR) {
			if (c1 == CC_CR && c2 == CC_LF) {
				bc[0] = (BC)c1;
				bc[1] = (BC)c2;
				rc = CheckLFA(pW, bc, 2, pLfInfo);
			}
		}
		else {
			if (c1 == CC_CR || c1 == CC_LF) {
				bc[0] = (BC)c1;
				rc = CheckLFA(pW, bc, 1, pLfInfo);
			}
		}
		//	UNICODE(BigEndian)ならば、改行文字をエンディアン変換
		if (fBE) {
			pLfInfo->strW[0] = (WC)AjcExcWord((UW)pLfInfo->strW[0]);
			pLfInfo->strW[1] = (WC)AjcExcWord((UW)pLfInfo->strW[1]);
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	バイトファイルから１文字読み出し（マルチバイト文字の場合は、複数バイト読み出す）							//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//				bfl		-	最大読み出しバイト数（バッファ空きバイト数）										//
//				pfLF	-	改行文字を示すフラグのバッファアドレス												//
//																												//
//	戻り値	：	≠０ ：読み出した文字のバイト数																	//
//				＝０ ：ＥＯＦ／エラー																			//
//--------------------------------------------------------------------------------------------------------------//
static	UI		ReadCharFromByteFile(HAJCFILE pW, BC buf[8], UI bfl, BOOL *pfLF)
{
	UI		rc	= 0;		//	戻り値（読み出した文字のバイト数）
	UI		lf;				//	改行文字列のバイト数
	LFINFO	lfInf;			//	改行情報

	//	改行文字フラグ初期化
	*pfLF = FALSE;

	while (pW->FIx < pW->FSizeA) {
		//	ＬＦならば、ＬＦを格納し終了（バッファ満杯時は読み出さないで終了）
		if (lf = CheckLFA(pW, pW->pInp.bcp, pW->FSizeA - pW->FIx, &lfInf)) {
			if (lfInf.stl <= bfl) {							//	改行文字をバッファに格納可能？
				pW->pInp.bcp += lf;							//		ファイルポインタ更新
				pW->FIx 	 += lf;							//		ファイルインデクス更新
				if (lfInf.stl == 0) {						//		読み飛ばし文字（ＣＲ／ＬＦ）？
					continue;								//			読み飛ばして継続
				}
				else {										//		有効な改行文字？
					strncpy(buf, lfInf.strA, lfInf.stl);	//			改行文字列を戻り情報に設定
					rc	  = lfInf.stl;						//			戻り値＝改行文字数
					*pfLF = TRUE;							//			改行文字の旨、フラグ設定
					break;									//			→終了
				}
			}
			else break;										//	バッファ満杯ならば終了
		}
		//	文字のバイト数判定，文字をバッファに格納
		switch (pW->tec) {
			case AJCTEC_MBC:
				rc = AjcStrChkMbcLen (pW->pInp.bcp, (UI)(pW->FSizeA - pW->FIx));
				break;

			case AJCTEC_UTF_8:
				rc = AjcStrChkUtf8Len(pW->pInp.bcp, (UI)(pW->FSizeA - pW->FIx));
				break;

			case AJCTEC_EUC_J:
				rc = AjcStrChkEucLen (pW->pInp.bcp, (UI)(pW->FSizeA - pW->FIx));
				break;
		}
		//	文字格納，ファイルポインタ更新
		if (rc != 0) {										//	ＥＯＦ以外？
			strncpy(buf, pW->pInp.bcp, rc);					//		文字格納
			pW->pInp.bcp += rc;								//		ファイルポインタ更新
			pW->FIx 	 += rc;								//		ファイルインデクス更新
		}
		else {												//	ＥＯＦ／ファイル終端は半端バイト？
			pW->FIx 	 = pW->FSizeA;
		}
		break;												//	終了
	}
	//	AJCFLF_LF_CRSKIP/AJCFLF_CR_LFSKIPの場合、後続の除去する改行文字を読み飛ばす
	if (pW->LfCnv == AJCFLF_LF_CRSKIP || pW->LfCnv == AJCFLF_CR_LFSKIP) {
		BC		cSkip;
		if (pW->LfCnv == AJCFLF_LF_CRSKIP) cSkip = 0x0D;
		else							   cSkip = 0x0A;
		while (pW->FIx < pW->FSizeA && *pW->pInp.bcp == cSkip) {
			pW->FIx++;
			pW->pInp.bcp++;
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	ＵＮＩＣＯＤＥファイルから１文字読み出し（サロゲートペア文字の場合は、複数ワード読み出す）					//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//				buf		-	読み出した文字を格納するバッファ													//
//				bfl		-	最大読み出しワード数（バッファ空きワード数）										//
//				pfLF	-	改行文字を示すフラグのバッファアドレス												//
//																												//
//	戻り値	：	≠０ ：読み出した文字のワード数（１～２）														//
//				＝０ ：ＥＯＦ／エラー																			//
//--------------------------------------------------------------------------------------------------------------//
static	UI		ReadCharFromUnicodeFile(HAJCFILE pW, WC buf[8], UI bfl, BOOL *pfLF)
{
	UI		rc	= 0;		//	戻り値（読み出した文字のバイト数）
	UI		lf;				//	改行文字列のバイト数
	LFINFO	lfInf;			//	改行情報

	//	改行文字フラグ初期化
	*pfLF = FALSE;

	while (pW->FIx < pW->FSizeA) {
		//	ＬＦならば、ＬＦを格納し終了（バッファ満杯時は読み出さないで終了）
		if (lf = CheckLFW(pW, pW->pInp.wcp, pW->FSizeW - pW->FIx, &lfInf, pW->tec == AJCTEC_UTF_16BE)) {
			if (lfInf.stl <= bfl) {							//	改行文字をバッファに格納可能？
				pW->pInp.wcp += lf;							//		ファイルポインタ更新
				pW->FIx 	 += lf;							//		ファイルインデクス更新
				if (lfInf.stl == 0) {						//		読み飛ばし文字（ＣＲ／ＬＦ）？
					continue;								//			読み飛ばして継続
				}
				else {										//		有効な改行文字？
					wcsncpy(buf, lfInf.strW, lfInf.stl);	//			改行文字列を戻り情報に設定
					rc	  = lfInf.stl;						//			戻り値＝改行文字数
					*pfLF = TRUE;							//			改行文字の旨、フラグ設定
					break;									//			→終了
				}
			}
			else break;										//	バッファ満杯ならば終了
		}
		//	文字のワード数判定，文字をバッファに格納
		switch (pW->tec) {
			case AJCTEC_UTF_16LE:
				rc = AjcStrChkU16LeLen(pW->pInp.wcp, (UI)(pW->FSizeW - pW->FIx));
				break;

			case AJCTEC_UTF_16BE:
				rc = AjcStrChkU16BeLen(pW->pInp.wcp, (UI)(pW->FSizeW - pW->FIx));
				break;
		}
		//	文字格納，ファイルポインタ更新
		if (rc != 0) {										//	ＥＯＦ以外？
			wcsncpy(buf, pW->pInp.wcp, rc);					//		文字格納
			pW->pInp.wcp += rc;								//		ファイルポインタ更新
			pW->FIx 	 += rc;								//		ファイルインデクス更新
		}
		else {												//	ＥＯＦ／ファイル終端は半端バイト？
			pW->FIx 	 = pW->FSizeW;
		}
		break;												//	終了
	}
	//	AJCFLF_LF_CRSKIP/AJCFLF_CR_LFSKIPの場合、後続の除去する改行文字を読み飛ばす
	if (pW->LfCnv == AJCFLF_LF_CRSKIP || pW->LfCnv == AJCFLF_CR_LFSKIP) {
		WC		cSkip;
		if (pW->LfCnv == AJCFLF_LF_CRSKIP) cSkip = 0x000D;
		else							   cSkip = 0x000A;
		if (pW->tec == AJCTEC_UTF_16BE) {
			cSkip = (WC)AjcExcWord((UW)cSkip);
		}
		while (pW->FIx < pW->FSizeW && *pW->pInp.wcp == cSkip) {
			pW->FIx++;
			pW->pInp.bcp++;
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	読み出したバイト文字をバッファに格納																		//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//				pInp	-	読み出したバイト文字																//
//				lInp	-	読み出したバイト文字のバイト数														//
//				pBuf	-	バッファポインタ																	//
//				lBuf	-	バッファサイズ（バッファの空バイト数／ワード数）									//
//																												//
//	戻り値	：	≠０ ：バッファに格納したバイト数／文字数														//
//				＝０ ：バッファ満杯																				//
//--------------------------------------------------------------------------------------------------------------//
//----- バイト文字バッファに格納 -------------------------------------------------------------------------------//
static	UI	SetByteCharToBufA(HAJCFILE pW, C_BCP pInp, UI lInp, BCP pBuf, UI lBuf)
{
	UI	rc = 0;

	switch (pW->tec) {
		case AJCTEC_MBC:
			if (lBuf >= lInp) {
				memcpy(pBuf, pInp, lInp);
				rc = lInp;
			}
			break;

		case AJCTEC_UTF_8:
		{	UI	lMbc;
			//	UTF-8 → SJIS 変換しバッファに格納
			if ((lMbc = AjcUtf8ToSJisEx(pInp, lInp, NULL, 0)) != 0) {
				if (lBuf >= lMbc) {
					AjcUtf8ToSJisEx(pInp, lInp, pBuf, lMbc);
					rc = lMbc;
				}
			}
			//	変換失敗時は、'?' を格納
			else if (lBuf >= 1) {
				*pBuf = '?';
				rc	  = 1;
			}
			break;
		}
		case AJCTEC_EUC_J:
		{	UI	lMbc;
			//	EUC → SJIS 変換しバッファに格納
			if ((lMbc = AjcEucToSJisEx(pInp, lInp, NULL, 0)) != 0) {
				if (lBuf >= lMbc) {
					AjcEucToSJisEx(pInp, lInp, pBuf, lMbc);
					rc = lMbc;
				}
			}
			//	変換失敗時は、'?' を格納
			else if (lBuf >= 1) {
				*pBuf = '?';
				rc	  = 1;
			}
			break;
		}
	}
	return rc;
}
//----- ワイド文字バッファに格納 -------------------------------------------------------------------------------//
static	UI	SetByteCharToBufW(HAJCFILE pW, C_BCP pInp, UI lInp, WCP pBuf, UI lBuf)
{
	UI	rc = 0;

	switch (pW->tec) {
		case AJCTEC_MBC:
		{	UI	lUni;
			//	MBC → UNICODE 変換しバッファに格納
			if ((lUni = MultiByteToWideChar(CP_ACP, 0, pInp, lInp, NULL, 0)) != 0) {
				if (lBuf >= lUni) {
					MultiByteToWideChar(CP_ACP, 0, pInp, lInp, pBuf, lUni);
					rc = lUni;
				}
			}
			//	変換失敗時は、'?' を格納
			else if (lBuf >= 1) {
				*pBuf = L'?';
				rc	  = 1;
			}
			break;
		}
		case AJCTEC_UTF_8:
		{	UI	lUni;
			//	UTF-8 → UNICODE 変換しバッファに格納，バッファ満杯ならば終了
			if ((lUni = MultiByteToWideChar(CP_UTF8, 0, pInp, lInp, NULL, 0)) != 0) {
				if (lBuf >= lUni) {
					MultiByteToWideChar(CP_UTF8, 0, pInp, lInp, pBuf, lUni);
					rc = lUni;
				}
			}
			//	変換失敗時は、'?' を格納
			else if (lBuf >= 1) {
				*pBuf = L'?';
				rc	  = 1;
			}
			break;
		}
		case AJCTEC_EUC_J:
		{	UI	lMbc;
			UI	lUni;
			BC	cMbc[8];
			//	EUC → SJIS 変換しバッファに格納
			if ((lMbc = AjcEucToSJisEx(pInp, lInp, cMbc, AJCTSIZE(cMbc))) != 0) {
				if ((lUni = MultiByteToWideChar(CP_ACP, 0, cMbc, lMbc, NULL, 0)) != 0) {
					if (lBuf >= lUni) {
						MultiByteToWideChar(CP_ACP, 0, cMbc, lMbc, pBuf, lUni);
						rc = lUni;
					}
				}
			}
			//	変換失敗時は、'?' を格納
			else if (lBuf >= 1) {
				*pBuf = L'?';
				rc	  = 1;
			}
			break;
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	読み出したワイド文字をバッファに格納																		//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//				pInp	-	読み出したバイト文字																//
//				lInp	-	読み出したバイト文字のバイト数														//
//				pBuf	-	バッファポインタ																	//
//				lBuf	-	バッファサイズ（バッファの空バイト数／ワード数）									//
//																												//
//	戻り値	：	≠０ ：バッファに格納したバイト数／文字数														//
//				＝０ ：バッファ満杯																				//
//--------------------------------------------------------------------------------------------------------------//
//----- バイト文字バッファに格納 -------------------------------------------------------------------------------//
static	UI	SetUnicodeCharToBufA(HAJCFILE pW, C_WCP pInp, UI lInp, BCP pBuf, UI lBuf)
{
	UI	rc = 0;

	switch (pW->tec) {
		case AJCTEC_UTF_16LE:
		{	UI	lMbc;
			//	UNICODE(LE) → SJIS 変換しバッファに格納
			if ((lMbc = WideCharToMultiByte(CP_ACP, 0, pInp, lInp, NULL, 0, NULL, NULL)) != 0) {
				if (lBuf >= lMbc) {
					WideCharToMultiByte(CP_ACP, 0, pInp, lInp, pBuf, lMbc, NULL, NULL);
					rc = lMbc;
				}
			}
			//	変換失敗時は、'?' を格納
			else if (lBuf >= 1) {
				*pBuf = '?';
				rc	  = 1;
			}
			break;
		}
		case AJCTEC_UTF_16BE:
		{	UI	lMbc;
			//	UNICODE(BE) → SJIS 変換しバッファに格納
			if ((lMbc = RevedWideCharToMultiByte(pInp, lInp, NULL, 0)) != 0) {
				if (lBuf >= lMbc) {
					RevedWideCharToMultiByte(pInp, lInp, pBuf, lMbc);
					rc = lMbc;
				}
			}
			//	変換失敗時は、'?' を格納
			else if (lBuf >= 1) {
				*pBuf = '?';
				rc	  = 1;
			}
			break;
		}
	}
	return rc;
}
//----- ワイド文字バッファに格納 -------------------------------------------------------------------------------//
static	UI	SetUnicodeCharToBufW(HAJCFILE pW, C_WCP pInp, UI lInp, WCP pBuf, UI lBuf)
{
	UI	rc = 0;

	switch (pW->tec) {
		case AJCTEC_UTF_16LE:
		{
			//	UNICODE(LE)をバッファに格納
			if (lBuf >= lInp) {
				wcsncpy(pBuf, pInp, lInp);
				rc = lInp;
			}
			break;
		}
		case AJCTEC_UTF_16BE:
		{
			//	UNICODE(BE) → UNICODE(LE) 変換しバッファに格納
			if (lBuf >= lInp) {
				UI		i;
				for (i = 0; i < lInp; i++) {
					*pBuf++ = (WC)AjcExcWord(*pInp++);
				}
				rc = lInp;
			}
			break;
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	バッファ内ロケーション(ixBuf)に対応するファイルポインタ取得（AjcFGetC()によるファイル読み出し位置を算出）	//
//																												//
//	引　数	：	pW		-	テキストファイルアクセスハンドル													//
//				pFIx	-	ファイルロケーション位置を格納するバッファのアドレス								//
//																												//
//	戻り値	：	バッファ内ロケーション(ixBuf)に対応するファイルポインタ											//
//--------------------------------------------------------------------------------------------------------------//
//----- バイトデータ -------------------------------------------------------------------------------------------//
static	VOP		GetFilePtrByBufA(HAJCFILE pW, ULLP pFIx)
{
	VOP		rc = NULL;
	UI		ix = pW->ixBuf;

	//	現バイトがマルチバイト先頭である場合は、１バイト戻す
	if (ix >= 1 && MAjcMbsBTypeA(pW->buf.mbc, ix - 1) == _MBC_LEAD) {
		ix--;
	}

	//	バイト文字→ファイルエンコード変換した際のバイト数からファイル位置を算出
	switch (pW->tec) {
		case AJCTEC_MBC:
			*pFIx = pW->BufFIx + ix;
			rc = pW->pInpTop.bcp + *pFIx;
			break;
		case AJCTEC_UTF_8:
			*pFIx = pW->BufFIx + AjcMbcToUtf8Ex(pW->buf.mbc, ix, NULL, 0);
			rc = pW->pInpTop.bcp + *pFIx;
			break;
		case AJCTEC_EUC_J:
			*pFIx = pW->BufFIx + AjcSJisToEucEx(pW->buf.mbc, ix, NULL, 0);
			rc = pW->pInpTop.bcp + *pFIx;
			break;
		case AJCTEC_UTF_16LE:
		case AJCTEC_UTF_16BE:
			*pFIx = pW->BufFIx + MultiByteToWideChar(CP_ACP, 0, pW->buf.mbc, ix, NULL, 0);
			rc = pW->pInpTop.wcp + *pFIx;
			break;
	}
	return rc;
}
//----- ワイドデータ -------------------------------------------------------------------------------------------//
static	VOP		GetFilePtrByBufW(HAJCFILE pW, ULLP pFIx)
{
	VOP		rc = NULL;
	UI		ix = pW->ixBuf;

	//	現ワードがサロゲートペア先頭である場合は、１ワード戻す
	if (ix >= 1 && MAjcIsLeadW(pW->buf.uni[ix - 1])) {
		ix--;
	}

	//	ワイド文字→ファイルエンコード変換した際のバイト数からファイル位置を算出
	switch (pW->tec) {
		case AJCTEC_MBC:
			*pFIx = pW->BufFIx + WideCharToMultiByte(CP_ACP , 0, pW->buf.uni, ix, NULL, 0, NULL, NULL);
			rc = pW->pInpTop.bcp + *pFIx;
			break;
		case AJCTEC_UTF_8:
			*pFIx = pW->BufFIx + WideCharToMultiByte(CP_UTF8, 0, pW->buf.uni, ix, NULL, 0, NULL, NULL);
			rc = pW->pInpTop.bcp + *pFIx;
			break;
		case AJCTEC_EUC_J:
		{	UI		lTmp;
			BCP		pTmp;
			lTmp = WideCharToMultiByte(CP_ACP, 0, pW->buf.uni, ix, NULL, 0, NULL, NULL);
			if (lTmp != 0 && (pTmp = malloc(lTmp))) {
				WideCharToMultiByte(CP_ACP, 0, pW->buf.uni, ix, pTmp, lTmp, NULL, NULL);
				*pFIx = pW->BufFIx + AjcSJisToEucEx(pTmp, lTmp, NULL, 0);
				rc = pW->pInpTop.bcp + *pFIx;
				free(pTmp);
			}
			else {
				//	ワーク確保不能時は、仮にバッファ先頭とする
				*pFIx = pW->BufFIx;
				rc = pW->pInpTop.bcp + *pFIx;
			}
			break;
		}
		case AJCTEC_UTF_16LE:
		case AJCTEC_UTF_16BE:
			*pFIx = pW->BufFIx + ix;
			rc = pW->pInpTop.wcp + *pFIx;
			break;
	}
	return rc;
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
//																												//
//	文字列書き込み用サブ関数																					//
//																												//
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	バッファにデータ格納																						//
//																												//
//--------------------------------------------------------------------------------------------------------------//
//----- バイトデータ -------------------------------------------------------------------------------------------//
static	BOOL	PutDataA(HAJCFILE pW, C_BCP pDat, UI lDat)
{
	BOOL	rc = TRUE;
	UI		rem = AJCTF_MAX_BUFA - pW->ixBuf;
	UI		len;
	UL		bytes;

	while (lDat != 0) {
		len = __min(lDat, rem);
		memcpy(&pW->buf.mbc[pW->ixBuf], pDat, len);
		pW->ixBuf += len;
		pDat	  += len;
		lDat	  -= len;
		rem 	  -= len;
		if (rem == 0) {
			if (WriteFile(pW->hFile, pW->buf.mbc, sizeof(pW->buf.mbc), &bytes, NULL) != 0 && bytes == sizeof(pW->buf.mbc)) {
				pW->ixBuf = 0;
				rem 	  = AJCTF_MAX_BUFA;
			}
			else {
				rc = FALSE;
				break;
			}
		}
	}
	pW->BufBytes = pW->ixBuf;
	return rc;
}
//----- ワイドデータ -------------------------------------------------------------------------------------------//
static	BOOL	PutDataW(HAJCFILE pW, C_WCP pDat, UI lDat)
{
	BOOL	rc = TRUE;
	UI		rem = AJCTF_MAX_BUFW - pW->ixBuf;
	UI		len;
	UL		bytes;

	while (lDat != 0) {
		len = __min(lDat, rem);
		memcpy(&pW->buf.uni[pW->ixBuf], pDat, len * 2);
		pW->ixBuf += len;
		pDat	  += len;
		lDat	  -= len;
		rem 	  -= len;
		if (rem == 0) {
			if (WriteFile(pW->hFile, pW->buf.mbc, sizeof(pW->buf.uni), &bytes, NULL) != 0 && bytes == sizeof(pW->buf.uni)) {
				pW->ixBuf = 0;
				rem 	  = AJCTF_MAX_BUFW;
			}
			else {
				rc = FALSE;
				break;
			}
		}
	}
	pW->BufBytes = pW->ixBuf * 2;
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	バッファフラッシュ																							//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	FlushBuf(HAJCFILE pW)
{
	BOOL	rc = TRUE;
	UL		bytes;

	if (pW->ixBuf != 0) {
		if (!(WriteFile(pW->hFile, pW->buf.mbc, pW->BufBytes, &bytes, NULL) != 0 && bytes == pW->BufBytes)) {
			rc = FALSE;
		}
		pW->ixBuf	 = 0;
		pW->BufBytes = 0;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	ＢＯＭ出力																									//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	VO	WriteBOM(HAJCFILE pW)
{
	UL		bytes;

	if (pW->fBomOut) {
		switch (pW->tec) {
			case AJCTEC_UTF_8:		WriteFile(pW->hFile, (VOP)bom_utf8	 , sizeof bom_utf8	 , &bytes, NULL);	break;
			case AJCTEC_UTF_16LE:	WriteFile(pW->hFile, (VOP)bom_utf16le, sizeof bom_utf16le, &bytes, NULL);	break;
			case AJCTEC_UTF_16BE:	WriteFile(pW->hFile, (VOP)bom_utf16be, sizeof bom_utf16be, &bytes, NULL);	break;
		}
		pW->fBomOut = FALSE;
	}
}

//--------------------------------------------------------------------------------------------------------------//
//																												//
//	ＭＢＣファイル書き込み																						//
//																												//
//--------------------------------------------------------------------------------------------------------------//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
static	BOOL		WriteMbcA  (HAJCFILE pW, C_BCP pTxt, UI lTxt)
{
	BOOL	rc = TRUE;
	UI		lf;				//	テキスト上の改行文字列長
	UI		icl;			//	入力MBC文字のバイト数
	LFINFO	lfInf;			//	改行情報
	BCP		pTmp = NULL;
	C_BCP	pMbc;
	UI		lMbc;
	UI		lMax = lTxt;	//	無限ループ抑止数（無限ループにはならないハズも、念のために抑止する）
	UI		lCnt = 0;		//	無限ループ抑止カウンタ

	if (lTxt != 0 && *pTxt != 0) {
		//	テキスト長自動設定／補正
		lTxt = AjcStrAdjustLenA(pTxt, lTxt);
		//	前回の半端バイトとテキストを動的バッファに格納
		if (pW->SvOutC != 0) {
			//	テキストを動的バッファに切り替える
			if (pTmp = AjcTAllocA(1 + lTxt + 1)) {
				*pTmp = (BC)pW->SvOutC;
				strncpy(pTmp + 1, pTxt, lTxt);
				*(pTmp + (1 + lTxt)) = 0;
				//	テキストポインタ切り替え
				pTxt = pTmp;
				lTxt++;
			}
			else rc = FALSE;
			pW->SvOutC = 0;
		}
		while (rc == TRUE && lCnt++ <= lMax && lTxt != 0 && *pTxt != 0) {
			//	改行文字列ならばならば、変換文字列をバッファに格納
			if (lf = CheckLFA(pW, pTxt, lTxt, &lfInf)) {
				rc = PutDataA(pW, lfInf.strA, lfInf.stl);
				pTxt += lf;
				lTxt -= lf;
				continue;	//	→改行文字列の後続文字列処理
			}
			//	制御コードならば、当該コードをバッファに格納
			else if (iscntrl((UB)*pTxt)) {
				if (!((pW->LfCnv == AJCFLF_LF_CRSKIP && *pTxt == CC_CR) || (pW->LfCnv == AJCFLF_CR_LFSKIP && *pTxt == CC_LF))) {
					rc = PutDataA(pW, pTxt, 1);
				}
				pTxt++;
				lTxt--;
				continue;	//	→制御コードの後続文字列処理
			}
			//	MBC文字長判定，テキスト末尾が半端バイトならば退避して終了
			if ((icl = AjcStrChkMbcLen(pTxt, lTxt)) == 0) {
				//	出力データ終端＝半端バイト
				pW->SvOutC = (UB)*pTxt;
				break;
			}
			//	制御コード検索
			for (pMbc = pTxt, lMbc = 0; lMbc < lTxt && !iscntrl((UB)*pMbc); pMbc++, lMbc++);
			//	制御コードが見つからない場合、最終半端文字を除いたバイト数設定
			if (lMbc >= lTxt || !iscntrl((UB)*pMbc)) {
				AJCMBCINFO mbi;
				if (AjcStrChkMbcPart(pTxt, lTxt - 1, &mbi)) lMbc = lTxt - (mbi.len - (mbi.ix + 1));
				else										lMbc = 1;
			}
			//	制御コードの直前まで／最終半端バイトを除いた部分をバッファに格納
			rc	  = PutDataA(pW, pTxt, lMbc);
			pTxt += lMbc;
			lTxt -= lMbc;
		}
		//	動的バッファ解放
		if (pTmp != NULL) free(pTmp);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
static	BOOL		WriteMbcW  (HAJCFILE pW, C_WCP pTxt, UI lTxt)
{
	BOOL	rc = TRUE;
	UI		lf;				//	テキスト上の改行文字列長
	UI		icl;			//	入力MBC文字のバイト数
	LFINFO	lfInf;			//	改行情報
	WCP		pTmp = NULL;
	C_WCP	pUni;
	BCP		pMbc;
	UI		lUni, lMbc;
	UI		lMax = lTxt;	//	無限ループ抑止数（無限ループにはならないハズも、念のために抑止する）
	UI		lCnt = 0;		//	無限ループ抑止カウンタ

	if (lTxt != 0 && *pTxt != 0) {
		//	テキスト長自動設定／補正
		lTxt = AjcStrAdjustLenW(pTxt, lTxt);
		//	前回の半端ワードとテキストを動的バッファに格納
		if (pW->SvOutC != 0) {
			//	前回の半端バイトをとテキストを動的バッファに格納
			if (pTmp = AjcTAllocW(1 + lTxt + 1)) {
				*pTmp = (BC)pW->SvOutC;
				wcsncpy(pTmp + 1, pTxt, lTxt);
				*(pTmp + (1 + lTxt)) = 0;
				//	テキストポインタ切り替え
				pTxt = pTmp;
				lTxt++;
			}
			else rc = FALSE;
			pW->SvOutC = 0;
		}
		while (rc == TRUE && lCnt++ <= lMax && lTxt != 0 && *pTxt != 0) {
			//	改行文字列ならば、変換文字列をバッファに格納
			if (lf = CheckLFW(pW, pTxt, lTxt, &lfInf, FALSE)) {
				if (!PutDataA(pW, lfInf.strA, lfInf.stl)) {
					rc = FALSE;
				}
				pTxt += lf;
				lTxt -= lf;
				continue;	//	→改行文字列の後続文字列処理
			}
			//	制御コードならば、当該コードをバッファに格納
			else if (iswcntrl(*pTxt)) {
				BC		bc = (BC)*pTxt;
				if (!((pW->LfCnv == AJCFLF_LF_CRSKIP && bc == CC_CR) || (pW->LfCnv == AJCFLF_CR_LFSKIP && bc == CC_LF))) {
					if (!PutDataA(pW, &bc, 1)) {
						rc = FALSE;
					}
				}
				pTxt++;
				lTxt--;
				continue;	//	→制御コードの後続文字列処理
			}
			//	UNICODE文字長判定，テキスト末尾が半端ワードならば退避して終了
			if ((icl = AjcStrChkU16LeLen(pTxt, lTxt)) == 0) {
				//	出力データ終端＝半端バイト
				pW->SvOutC = (UW)*pTxt;
				break;
			}
			//	制御コード検索
			for (pUni = pTxt, lUni = 0; lUni < lTxt && !iswcntrl(*pUni); pUni++, lUni++);
			//	制御コードが見つからない場合、最終半端ワードを除いたワード数設定
			if (lUni >= lTxt || !iswcntrl(*pUni)) {
				if (MAjcIsLeadW(*(pTxt + (lTxt - 1)))) lUni = lTxt - 1;
				else								   lUni = lTxt;
			}
			//	制御コードの直前まで／最終半端バイトを除いた部分をバッファに格納
			if ((lMbc = WideCharToMultiByte(CP_ACP, 0, pTxt, lUni, NULL, 0, NULL, NULL)) != 0) {
				if (pMbc = AjcTAllocA(lMbc)) {
					WideCharToMultiByte(CP_ACP, 0, pTxt, lUni, pMbc, lMbc, NULL, NULL);
					if (!PutDataA(pW, pMbc, lMbc)) {
						rc = FALSE;
					}
					pTxt += lUni;
					lTxt -= lUni;
					AjcTFree(pMbc);
				}
				else rc = FALSE;
			}
			else rc = FALSE;
		}
		//	動的バッファ解放
		if (pTmp != NULL) free(pTmp);
	}
	return rc;
}

//--------------------------------------------------------------------------------------------------------------//
//																												//
//	ＵＴＦ－８ファイル書き込み																					//
//																												//
//--------------------------------------------------------------------------------------------------------------//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
static	BOOL		WriteUtf8A (HAJCFILE pW, C_BCP pTxt, UI lTxt)
{
	BOOL	rc = TRUE;
	UI		lf;				//	テキスト上の改行文字列長
	UI		icl;			//	入力SJIS文字のバイト数
	LFINFO	lfInf;			//	改行情報
	BCP		pTmp = NULL;	//	テキスト切り替えバッファ
	C_BCP	pMbc;
	BCP		pUtf8;
	UI		lMbc, lUtf8;
	UI		lMax = lTxt;	//	無限ループ抑止数（無限ループにはならないハズも、念のために抑止する）
	UI		lCnt = 0;		//	無限ループ抑止カウンタ

	if (lTxt != 0 && *pTxt != 0) {
		//	テキスト長自動設定／補正
		lTxt = AjcStrAdjustLenA(pTxt, lTxt);
		//	前回の半端バイトとテキストを動的バッファに格納
		if (pW->SvOutC != 0) {
			//	テキストを動的バッファに切り替える
			if (pTmp = AjcTAllocA(1 + lTxt + 1)) {
				*pTmp = (BC)pW->SvOutC;
				strncpy(pTmp + 1, pTxt, lTxt);
				*(pTmp + (1 + lTxt)) = 0;
				//	テキストポインタ切り替え
				pTxt = pTmp;
				lTxt++;
			}
			else rc = FALSE;
			pW->SvOutC = 0;
		}
		while (rc == TRUE && lCnt++ <= lMax && lTxt != 0 && *pTxt != 0) {
			//	改行文字列ならば、変換文字列をバッファに格納
			if (lf = CheckLFA(pW, pTxt, lTxt, &lfInf)) {
				rc = PutDataA(pW, lfInf.strA, lfInf.stl);
				pTxt += lf;
				lTxt -= lf;
				continue;	//	→改行文字列の後続文字列処理
			}
			//	制御コードならば、当該コードをバッファに格納
			else if (iscntrl((UB)*pTxt)) {
				if (!((pW->LfCnv == AJCFLF_LF_CRSKIP && *pTxt == CC_CR) || (pW->LfCnv == AJCFLF_CR_LFSKIP && *pTxt == CC_LF))) {
					rc = PutDataA(pW, pTxt, 1);
				}
				pTxt++;
				lTxt--;
				continue;	//	→制御コードの後続文字列処理
			}
			//	MBC文字長判定，テキスト末尾が半端バイトならば退避して終了
			if ((icl = AjcStrChkMbcLen(pTxt, lTxt)) == 0) {
				//	出力データ終端＝半端バイト
				pW->SvOutC = (UB)*pTxt;
				break;
			}
			//	制御コード検索
			for (pMbc = pTxt, lMbc = 0; lMbc < lTxt && !iscntrl((UB)*pMbc); pMbc++, lMbc++);
			//	制御コードが見つからない場合、最終半端文字を除いたバイト数設定
			if (lMbc >= lTxt || !iscntrl((UB)*pMbc)) {
				AJCMBCINFO mbi;
				if (AjcStrChkMbcPart(pTxt, lTxt - 1, &mbi)) lMbc = lTxt - (mbi.len - (mbi.ix + 1));
				else										lMbc = 1;
			}
			//	制御コードの直前まで／最終半端バイトを除いた部分をバッファに格納
			if ((lUtf8 = AjcMbcToUtf8Ex(pTxt, lMbc, NULL, 0)) != 0) {
				if (pUtf8 = AjcTAllocA(lUtf8)) {
					AjcMbcToUtf8Ex(pTxt, lMbc, pUtf8, lUtf8);
					rc = PutDataA(pW, pUtf8, lUtf8);
					pTxt += lMbc;
					lTxt -= lMbc;
					AjcTFree(pUtf8);
				}
				else rc = FALSE;
			}
		}
		//	動的バッファ解放
		if (pTmp != NULL) free(pTmp);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
static	BOOL		WriteUtf8W (HAJCFILE pW, C_WCP pTxt, UI lTxt)
{
	BOOL	rc = TRUE;
	UI		lf;				//	テキスト上の改行文字列長
	UI		icl;			//	入力MBC文字のバイト数
	LFINFO	lfInf;			//	改行情報
	WCP		pTmp = NULL;
	C_WCP	pUni;
	BCP		pUtf8;
	UI		lUni, lUtf8;
	UI		lMax = lTxt;	//	無限ループ抑止数（無限ループにはならないハズも、念のために抑止する）
	UI		lCnt = 0;		//	無限ループ抑止カウンタ

	if (lTxt != 0 && *pTxt != 0) {
		//	テキスト長自動設定／補正
		lTxt = AjcStrAdjustLenW(pTxt, lTxt);
		//	前回の半端ワードとテキストを動的バッファに格納
		if (pW->SvOutC != 0) {
			//	前回の半端バイトをとテキストを動的バッファに格納
			if (pTmp = AjcTAllocW(1 + lTxt + 1)) {
				*pTmp = (WC)pW->SvOutC;
				wcsncpy(pTmp + 1, pTxt, lTxt);
				*(pTmp + (1 + lTxt)) = 0;
				//	テキストポインタ切り替え
				pTxt = pTmp;
				lTxt++;
			}
			else rc = FALSE;
			pW->SvOutC = 0;
		}
		while (rc == TRUE && lCnt++ <= lMax && lTxt != 0 && *pTxt != 0) {
			//	改行文字列ならば、変換文字列をバッファに格納
			if (lf = CheckLFW(pW, pTxt, lTxt, &lfInf, FALSE)) {
				rc = PutDataA(pW, lfInf.strA, lfInf.stl);
				pTxt += lf;
				lTxt -= lf;
				continue;	//	→改行文字列の後続文字列処理
			}
			//	制御コードならば、当該コードをバッファに格納
			else if (iswcntrl(*pTxt)) {
				BC		bc = (BC)*pTxt;
				if (!((pW->LfCnv == AJCFLF_LF_CRSKIP && bc == CC_CR) || (pW->LfCnv == AJCFLF_CR_LFSKIP && bc == CC_LF))) {
					rc = PutDataA(pW, &bc, 1);
				}
				pTxt++;
				lTxt--;
				continue;	//	→制御コードの後続文字列処理
			}
			//	UNICODE文字長判定，テキスト末尾が半端ワードならば退避して終了
			if ((icl = AjcStrChkU16LeLen(pTxt, lTxt)) == 0) {
				//	出力データ終端＝半端バイト
				pW->SvOutC = (UW)*pTxt;
				break;
			}
			//	制御コード検索
			for (pUni = pTxt, lUni = 0; lUni < lTxt && !iswcntrl(*pUni); pUni++, lUni++);
			//	制御コードが見つからない場合、最終半端ワードを除いたワード数設定
			if (lUni >= lTxt || !iswcntrl(*pUni)) {
				if (MAjcIsLeadW(*(pTxt + (lTxt - 1)))) lUni = lTxt - 1;
				else								   lUni = lTxt;
			}
			//	制御コードの直前まで／最終半端バイトを除いた部分をバッファに格納
			if ((lUtf8 = WideCharToMultiByte(CP_UTF8, 0, pTxt, lUni, NULL, 0, NULL, NULL)) != 0) {
				if (pUtf8 = AjcTAllocA(lUtf8)) {
					WideCharToMultiByte(CP_UTF8, 0, pTxt, lUni, pUtf8, lUtf8, NULL, NULL);
					rc = PutDataA(pW, pUtf8, lUtf8);
					pTxt += lUni;
					lTxt -= lUni;
					AjcTFree(pUtf8);
				}
				else rc = FALSE;
			}
			else rc = FALSE;
		}
		//	動的バッファ解放
		if (pTmp != NULL) free(pTmp);
	}
	return rc;
}

//--------------------------------------------------------------------------------------------------------------//
//																												//
//	ＥＵＣファイル書き込み																						//
//																												//
//--------------------------------------------------------------------------------------------------------------//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
static	BOOL		WriteEucA  (HAJCFILE pW, C_BCP pTxt, UI lTxt)
{
	BOOL	rc = TRUE;
	UI		lf;				//	テキスト上の改行文字列長
	UI		icl;			//	入力SJIS文字のバイト数
	LFINFO	lfInf;			//	改行情報
	BCP		pTmp = NULL;	//	テキスト切り替えバッファ
	C_BCP	pMbc;
	BCP		pEuc;
	UI		lMbc, lEuc;
	UI		lMax = lTxt;	//	無限ループ抑止数（無限ループにはならないハズも、念のために抑止する）
	UI		lCnt = 0;		//	無限ループ抑止カウンタ

	if (lTxt != 0 && *pTxt != 0) {
		//	テキスト長自動設定／補正
		lTxt = AjcStrAdjustLenA(pTxt, lTxt);
		//	前回の半端バイトとテキストを動的バッファに格納
		if (pW->SvOutC != 0) {
			//	テキストを動的バッファに切り替える
			if (pTmp = AjcTAllocA(1 + lTxt + 1)) {
				*pTmp = (BC)pW->SvOutC;
				strncpy(pTmp + 1, pTxt, lTxt);
				*(pTmp + (1 + lTxt)) = 0;
				//	テキストポインタ切り替え
				pTxt = pTmp;
				lTxt++;
			}
			else rc = FALSE;
			pW->SvOutC = 0;
		}
		while (rc == TRUE && lCnt++ <= lMax && lTxt != 0 && *pTxt != 0) {
			//	改行文字列ならば、変換文字列をバッファに格納
			if (lf = CheckLFA(pW, pTxt, lTxt, &lfInf)) {
				rc = PutDataA(pW, lfInf.strA, lfInf.stl);
				pTxt += lf;
				lTxt -= lf;
				continue;	//	→改行文字列の後続文字列処理
			}
			//	制御コードならば、当該コードをバッファに格納
			else if (iscntrl((UB)*pTxt)) {
				if (!((pW->LfCnv == AJCFLF_LF_CRSKIP && *pTxt == CC_CR) || (pW->LfCnv == AJCFLF_CR_LFSKIP && *pTxt == CC_LF))) {
					rc = PutDataA(pW, pTxt, 1);
				}
				pTxt++;
				lTxt--;
				continue;	//	→制御コードの後続文字列処理
			}
			//	MBC文字長判定，テキスト末尾が半端バイトならば退避して終了
			if ((icl = AjcStrChkMbcLen(pTxt, lTxt)) == 0) {
				//	出力データ終端＝半端バイト
				pW->SvOutC = (UB)*pTxt;
				break;
			}
			//	制御コード検索
			for (pMbc = pTxt, lMbc = 0; lMbc < lTxt && !iscntrl((UB)*pMbc); pMbc++, lMbc++);
			//	制御コードが見つからない場合、最終半端文字を除いたバイト数設定
			if (lMbc >= lTxt || !iscntrl((UB)*pMbc)) {
				AJCMBCINFO mbi;
				if (AjcStrChkMbcPart(pTxt, lTxt - 1, &mbi)) lMbc = lTxt - (mbi.len - (mbi.ix + 1));
				else										lMbc = 1;
			}
			//	制御コードの直前まで／最終半端バイトを除いた部分をバッファに格納
			if ((lEuc = AjcSJisToEucEx(pTxt, icl, NULL, 0)) != 0) {
				if (pEuc = AjcTAllocA(lEuc)) {
					AjcSJisToEucEx(pTxt, lMbc, pEuc, lEuc);
					rc = PutDataA(pW, pEuc, lEuc);
					pTxt += lMbc;
					lTxt -= lMbc;
					AjcTFree(pEuc);
				}
				else rc = FALSE;
			}
			else rc = FALSE;
		}
		//	動的バッファ解放
		if (pTmp != NULL) free(pTmp);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
static	BOOL		WriteEucW  (HAJCFILE pW, C_WCP pTxt, UI lTxt)
{
	BOOL	rc = TRUE;
	UI		lf;				//	テキスト上の改行文字列長
	UI		icl;			//	入力MBC文字のバイト数
	LFINFO	lfInf;			//	改行情報
	WCP		pTmp = NULL;
	C_WCP	pUni;
	BCP		pMbc;
	BCP		pEuc;
	UI		lUni, lMbc, lEuc;
	UI		lMax = lTxt;	//	無限ループ抑止数（無限ループにはならないハズも、念のために抑止する）
	UI		lCnt = 0;		//	無限ループ抑止カウンタ

	if (lTxt != 0 && *pTxt != 0) {
		//	テキスト長自動設定／補正
		lTxt = AjcStrAdjustLenW(pTxt, lTxt);
		//	前回の半端ワードとテキストを動的バッファに格納
		if (pW->SvOutC != 0) {
			//	前回の半端バイトをとテキストを動的バッファに格納
			if (pTmp = AjcTAllocW(1 + lTxt + 1)) {
				*pTmp = (BC)pW->SvOutC;
				wcsncpy(pTmp + 1, pTxt, lTxt);
				*(pTmp + (1 + lTxt)) = 0;
				//	テキストポインタ切り替え
				pTxt = pTmp;
				lTxt++;
			}
			else rc = FALSE;
			pW->SvOutC = 0;
		}
		while (rc == TRUE && lCnt++ <= lMax && lTxt != 0 && *pTxt != 0) {
			//	改行文字列ならば、変換文字列をバッファに格納
			if (lf = CheckLFW(pW, pTxt, lTxt, &lfInf, FALSE)) {
				rc = PutDataA(pW, lfInf.strA, lfInf.stl);
				pTxt += lf;
				lTxt -= lf;
				continue;	//	→改行文字列の後続文字列処理
			}
			//	制御コードならば、当該コードをバッファに格納
			else if (iswcntrl(*pTxt)) {
				BC		bc = (BC)*pTxt;
				if (!((pW->LfCnv == AJCFLF_LF_CRSKIP && bc == CC_CR) || (pW->LfCnv == AJCFLF_CR_LFSKIP && bc == CC_LF))) {
					rc = PutDataA(pW, &bc, 1);
				}
				pTxt++;
				lTxt--;
				continue;	//	→制御コードの後続文字列処理
			}
			//	UNICODE文字長判定，テキスト末尾が半端ワードならば退避して終了
			if ((icl = AjcStrChkU16LeLen(pTxt, lTxt)) == 0) {
				//	出力データ終端＝半端バイト
				pW->SvOutC = (UW)*pTxt;
				break;
			}
			//	制御コード検索
			for (pUni = pTxt, lUni = 0; lUni < lTxt && !iswcntrl(*pUni); pUni++, lUni++);
			//	制御コードが見つからない場合、最終半端ワードを除いたワード数設定
			if (lUni >= lTxt || !iswcntrl(*pUni)) {
				if (MAjcIsLeadW(*(pTxt + (lTxt - 1)))) lUni = lTxt - 1;
				else								   lUni = lTxt;
			}
			//	制御コードの直前まで／最終半端バイトを除いた部分をバッファに格納
			if ((lMbc = WideCharToMultiByte(CP_ACP, 0, pTxt, lUni, NULL, 0, NULL, NULL)) != 0) {
				if (pMbc = AjcTAllocA(lMbc)) {
					WideCharToMultiByte(CP_ACP, 0, pTxt, lUni, pMbc, lMbc, NULL, NULL);
					if ((lEuc = AjcSJisToEucEx(pMbc, lMbc, NULL, 0)) != 0) {
						if (pEuc = AjcTAllocA(lEuc)) {
							AjcSJisToEucEx(pMbc, lMbc, pEuc, lEuc);
							rc = PutDataA(pW, pEuc, lEuc);
							pTxt += lUni;
							lTxt -= lUni;
							AjcTFree(pEuc);
						}
						else rc = FALSE;
					}
					else rc = FALSE;
					AjcTFree(pMbc);
				}
				else rc = FALSE;
			}
			else rc = FALSE;
		}
		//	動的バッファ解放
		if (pTmp != NULL) free(pTmp);
	}
	return rc;
}

//--------------------------------------------------------------------------------------------------------------//
//																												//
//	ＵＮＩＣＯＤＥファイル書き込み																				//
//																												//
//--------------------------------------------------------------------------------------------------------------//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
static	BOOL		WriteUniA (HAJCFILE pW, C_BCP pTxt, UI lTxt, BOOL fBE)
{
	BOOL	rc = TRUE;
	UI		lf;				//	テキスト上の改行文字列長
	UI		icl;			//	入力MBC文字のバイト数
	LFINFO	lfInf;			//	改行情報
	BCP		pTmp = NULL;
	C_BCP	pMbc;
	WCP		pUni;
	UI		lMbc, lUni;
	UI		lMax = lTxt;	//	無限ループ抑止数（無限ループにはならないハズも、念のために抑止する）
	UI		lCnt = 0;		//	無限ループ抑止カウンタ

	if (lTxt != 0 && *pTxt != 0) {
		//	テキスト長自動設定／補正
		lTxt = AjcStrAdjustLenA(pTxt, lTxt);
		//	前回の半端バイトとテキストを動的バッファに格納
		if (pW->SvOutC != 0) {
			//	テキストを動的バッファに切り替える
			if (pTmp = AjcTAllocA(1 + lTxt + 1)) {
				*pTmp = (BC)pW->SvOutC;
				strncpy(pTmp + 1, pTxt, lTxt);
				*(pTmp + (1 + lTxt)) = 0;
				//	テキストポインタ切り替え
				pTxt = pTmp;
				lTxt++;
			}
			else rc = FALSE;
			pW->SvOutC = 0;
		}
		while (rc == TRUE && lCnt++ <= lMax && lTxt != 0 && *pTxt != 0) {
			//	改行文字列ならば、変換文字列をバッファに格納
			if (lf = CheckLFA(pW, pTxt, lTxt, &lfInf)) {
				if (fBE) {
					UI		i;
					for (i = 0; i < lfInf.stl; i++) {
						lfInf.strW[i] = (WC)AjcExcWord((UW)lfInf.strW[i]);
					}
				}
				rc = PutDataW(pW, lfInf.strW, lfInf.stl);
				pTxt += lf;
				lTxt -= lf;
				continue;	//	→改行文字列の後続文字列処理
			}
			//	制御コードならば、当該コードをバッファに格納
			else if (iscntrl((UB)*pTxt)) {
				if (!((pW->LfCnv == AJCFLF_LF_CRSKIP && *pTxt == CC_CR) || (pW->LfCnv == AJCFLF_CR_LFSKIP && *pTxt == CC_LF))) {
					WC		wc = ((fBE) ? ((WC)AjcExcWord((UW)*pTxt)) : ((WC)*pTxt));
					rc = PutDataW(pW, &wc, 1);
				}
				pTxt++;
				lTxt--;
				continue;	//	→制御コードの後続文字列処理
			}
			//	MBC文字長判定，テキスト末尾が半端バイトならば退避して終了
			if ((icl = AjcStrChkMbcLen(pTxt, lTxt)) == 0) {
				//	出力データ終端＝半端バイト
				pW->SvOutC = (UB)*pTxt;
				break;
			}
			//	制御コード検索
			for (pMbc = pTxt, lMbc = 0; lMbc < lTxt && !iscntrl((UB)*pMbc); pMbc++, lMbc++);
			//	制御コードが見つからない場合、最終半端文字を除いたバイト数設定
			if (lMbc >= lTxt || !iscntrl((UB)*pMbc)) {
				AJCMBCINFO mbi;
				if (AjcStrChkMbcPart(pTxt, lTxt - 1, &mbi)) lMbc = lTxt - (mbi.len - (mbi.ix + 1));
				else										lMbc = 1;
			}
			//	制御コードの直前まで／最終半端バイトを除いた部分をバッファに格納
			if ((lUni = MultiByteToWideChar(CP_ACP, 0, pTxt, lMbc, NULL, 0)) != 0) {
				if (pUni = AjcTAllocW(lUni)) {
					MultiByteToWideChar(CP_ACP, 0, pTxt, lMbc, pUni, lUni);
					if (fBE) {
						UI	i;
						for (i = 0; i < lUni; i++) {
							pUni[i] = (WC)AjcExcWord((UW)pUni[i]);
						}
					}
					rc = PutDataW(pW, pUni, lUni);
					pTxt += lMbc;
					lTxt -= lMbc;
					AjcTFree(pUni);
				}
				else rc = FALSE;
		}
			else rc = FALSE;
		}
		//	動的バッファ解放
		if (pTmp != NULL) free(pTmp);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
static	BOOL		WriteUniW (HAJCFILE pW, C_WCP pTxt, UI lTxt, BOOL fBE)
{
	BOOL	rc = TRUE;
	UI		lf;				//	テキスト上の改行文字列長
	UI		icl;			//	入力MBC文字のバイト数
	LFINFO	lfInf;			//	改行情報
	WCP		pTmp = NULL;
	C_WCP	pUni;
	UI		lUni;
	UI		lMax = lTxt;	//	無限ループ抑止数（無限ループにはならないハズも、念のために抑止する）
	UI		lCnt = 0;		//	無限ループ抑止カウンタ

	if (lTxt != 0 && *pTxt != 0) {
		//	テキスト長自動設定／補正
		lTxt = AjcStrAdjustLenW(pTxt, lTxt);
		//	前回の半端ワードとテキストを動的バッファに格納
		if (pW->SvOutC != 0) {
			//	前回の半端バイトをとテキストを動的バッファに格納
			if (pTmp = AjcTAllocW(1 + lTxt + 1)) {
				*pTmp = (WC)pW->SvOutC;
				wcsncpy(pTmp + 1, pTxt, lTxt);
				*(pTmp + (1 + lTxt)) = 0;
				//	テキストポインタ切り替え
				pTxt = pTmp;
				lTxt++;
			}
			else rc = FALSE;
			pW->SvOutC = 0;
		}
		while (rc == TRUE && lCnt++ <= lMax && lTxt != 0 && *pTxt != 0) {
			//	改行文字列ならば、変換文字列をバッファに格納
			if (lf = CheckLFW(pW, pTxt, lTxt, &lfInf, FALSE)) {
				if (fBE) {
					UI		i;
					for (i = 0; i < lfInf.stl; i++) {
						lfInf.strW[i] = (WC)AjcExcWord((UW)lfInf.strW[i]);
					}
				}
				rc = PutDataW(pW, lfInf.strW, lfInf.stl);
				pTxt += lf;
				lTxt -= lf;
				continue;	//	→改行文字列の後続文字列処理
			}
			//	制御コードならば、当該コードをバッファに格納
			else if (iswcntrl(*pTxt)) {
				WC	wc;
				if (!((pW->LfCnv == AJCFLF_LF_CRSKIP && *pTxt == CC_CR) || (pW->LfCnv == AJCFLF_CR_LFSKIP && *pTxt == CC_LF))) {
					if (fBE) wc = (WC)AjcExcWord((UW)*pTxt);
					else	 wc = *pTxt;
					rc = PutDataW(pW, &wc, 1);
				}
				pTxt++;
				lTxt--;
				continue;	//	→制御コードの後続文字列処理
			}
			//	UNICODE文字長判定，テキスト末尾が半端ワードならば退避して終了
			if ((icl = AjcStrChkU16LeLen(pTxt, lTxt)) == 0) {
				//	出力データ終端＝半端バイト
				pW->SvOutC = (UW)*pTxt;
				break;
			}
			//	制御コード検索
			for (pUni = pTxt, lUni = 0; lUni < lTxt && !iswcntrl(*pUni); pUni++, lUni++);
			//	制御コードが見つからない場合、最終半端ワードを除いたワード数設定
			if (lUni >= lTxt || !iswcntrl(*pUni)) {
				if (MAjcIsLeadW(*(pTxt + (lTxt - 1)))) lUni = lTxt - 1;
				else								   lUni = lTxt;
			}
			//	制御コードの直前まで／最終半端バイトを除いた部分をバッファに格納
			//	UNICODE(LE)をバッファに格納
			if (!fBE) {
				rc = PutDataW(pW, pTxt, lUni);
			}
			//	UNICODE(BE)をバッファに格納
			else {
				UI	i;
				WC	be;
				for (i = 0; i < lUni; i++) {
					be = (WC)AjcExcWord((UW)pTxt[i]);
					rc = PutDataW(pW, &be, 1);
				}
			}
			pTxt += lUni;
			lTxt -= lUni;
		}
		//	動的バッファ解放
		if (pTmp != NULL) free(pTmp);
	}
	return rc;
}

