﻿#include	"AjcInternal.h"

//**************************************************************************************************************//
//																												//
//	テキスト描画																								//
//																												//
//**************************************************************************************************************//

#define	INST_ID			0x624490B1
#define	IS_MY_INST(P)	(P != NULL && P->InstID == INST_ID)


//--------------------------------------------------------------------------------------------------------------//
//	フォント変更フラグ																							//
//--------------------------------------------------------------------------------------------------------------//
#define		AJCCFBE_NORMAL			0x00
#define		AJCCFBE_BOLD			0x01
#define		AJCCFBE_ITALIC			0x02
#define		AJCCFBE_ULINE			0x04
#define		AJCCFBE_BOLD_ITALIC		(AJCCFBE_BOLD	| AJCCFBE_ITALIC )
#define		AJCCFBE_BOLD_ULINE		(AJCCFBE_BOLD	| AJCCFBE_ULINE  )
#define		AJCCFBE_ITALIC_ULINE	(AJCCFBE_ITALIC | AJCCFBE_ULINE  )
#define		AJCCFBE_ALL				(AJCCFBE_BOLD	| AJCCFBE_ITALIC | AJCCFBE_ULINE)

//--------------------------------------------------------------------------------------------------------------//
//	描画イメージ情報																							//
//--------------------------------------------------------------------------------------------------------------//
static		HAJCAVL				hAvlImg = NULL;			//	イメージ情報（Bitmap / Icon）格納用ＡＶＬ木

//--------------------------------------------------------------------------------------------------------------//
//	作業領域																									//
//--------------------------------------------------------------------------------------------------------------//
static	VO		ChangeFontoToHdc	(HAJCTXO pW);
static	VO		ResetToDefaultColor	(HAJCTXO pW);

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	SubGetImgInfo(C_UTP pEsc, PTXOIMGINFO pInfo, BITMAP *pBm);
static	BOOL	SubRegistImage(BOOL fIcon, UX handle, UI id, int x, int y, int cx, int cy);

//==============================================================================================================//
//	起動時初期設定																								//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
BOOL	AjcTxoInit(VO)
{
	//	NOP

	return TRUE;
}
//==============================================================================================================//
//	終了時後処理																								//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
VO		AjcTxoEnd (VO)
{
	if (hAvlImg		!= NULL) {AjcAvlDelete(hAvlImg);					   hAvlImg		= NULL;	}
}
//==============================================================================================================//
//	インスタンスの生成																							//
//																												//
//	引　数	：	hdc		- デバイスコンテキスト・ハンドル														//
//				left 	- 左端描画開始位置																		//
//				top		- 上端描画開始位置																		//
//				defLsp 	- デフォルトの行間スペース（正数：ピクセル数，負数：文字高さのＮ％）					//
//																												//
//	戻り値	：	≠NULL - ＯＫ（インスタンスハンドル）															//
//				＝NULL - エラー																					//
//==============================================================================================================//
AJCEXPORT	HAJCTXO		WINAPI	AjcTxoCreate		(HDC hdc, int left, int top, int defLsp)
{
	HAJCTXO		pW = NULL;
	LOGFONT		lf;
	TEXTMETRIC	tm;

	if ((pW = (HAJCTXO)AJCMEM(sizeof(AJCTXO))) != NULL) {
		memset(pW, 0, sizeof *pW);
		pW->InstID			= INST_ID;
		pW->hdc				= hdc;
		pW->xLeft			= left;
		pW->yTop			= top;
		pW->Align			= AJCTXOALIGN_LEFT;
		pW->Palette[0]		= RGB(	 0,   0,   0);
		pW->Palette[1]		= RGB( 255,   0,   0);
		pW->Palette[2]		= RGB(	 0, 255,   0);
		pW->Palette[3]		= RGB( 255, 255,   0);
		pW->Palette[4]		= RGB(	 0,   0, 255);
		pW->Palette[5]		= RGB( 255,   0, 255);
		pW->Palette[6]		= RGB(	 0, 255, 255);
		pW->Palette[7]		= RGB( 255, 255, 255);

		//	描画位置初期化
		pW->x = 0;
		pW->y = 0;

		//	起動時のＤＣ属性退避
		if ((pW->hFontOrg = SelectObject(pW->hdc, GetStockObject(SYSTEM_FONT))) != NULL) {
			SelectObject(pW->hdc, pW->hFontOrg);
			pW->BkModeOrg	= GetBkMode   (pW->hdc);
			pW->BkColorOrg	= GetBkColor  (pW->hdc);
			pW->TxtColorOrg	= GetTextColor(pW->hdc);

			//	文字の高さ，幅設定
			GetTextMetrics(hdc, &tm);
			pW->cxChar = tm.tmAveCharWidth;
			pW->cyChar = tm.tmHeight;
			//	太字．斜字設定
			GetObject(pW->hFontOrg, sizeof lf, &lf);
			pW->fFont |= (lf.lfWeight == FW_BOLD) ? AJCCFBE_BOLD   : 0;
			pW->fFont |= (lf.lfItalic == TRUE	) ? AJCCFBE_ITALIC : 0;
			//	タブステップ数設定
			pW->TabStep = pW->cxChar * 4;
			//	行間スペース設定
			if (defLsp >= 0) pW->LspDef = defLsp;
			else			 pW->LspDef = abs(pW->cyChar * defLsp / 100);
			pW->Lsp = pW->LspDef;

			//	文字色，背景色のデフォルト値
			pW->TxtColorDef = RGB(0, 0, 0);		//	黒
			pW->BkColorDef	= -1;				//	透明

			//	描画色リセット
			ResetToDefaultColor(pW);
		}
		else {
			free(pW);
			pW = NULL;
		}
	}
	return pW;
}
//==============================================================================================================//
//	インスタンスの消去																							//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				fRestore - ＤＣ復旧フラグ																		//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTxoDelete		(HAJCTXO pW, BOOL fRestore)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		//	起動時のＤＣ属性回復
		if (fRestore) {
			SelectObject(pW->hdc, pW->hFontOrg	 );
			SetTextColor(pW->hdc, pW->TxtColorOrg);
			SetBkColor	(pW->hdc, pW->BkColorOrg );
			SetBkMode	(pW->hdc, pW->BkModeOrg  );
		}
		//	フォントオブジェクト破棄
		if (pW->hFontN	 != NULL) {DeleteObject(pW->hFontN	); pW->hFontN	= NULL;}
		if (pW->hFontB	 != NULL) {DeleteObject(pW->hFontB	); pW->hFontB	= NULL;}
		if (pW->hFontI	 != NULL) {DeleteObject(pW->hFontI	); pW->hFontI	= NULL;}
		if (pW->hFontU	 != NULL) {DeleteObject(pW->hFontU	); pW->hFontU	= NULL;}
		if (pW->hFontBI  != NULL) {DeleteObject(pW->hFontBI ); pW->hFontBI	= NULL;}
		if (pW->hFontBU  != NULL) {DeleteObject(pW->hFontBU ); pW->hFontBU	= NULL;}
		if (pW->hFontIU  != NULL) {DeleteObject(pW->hFontIU ); pW->hFontIU	= NULL;}
		if (pW->hFontBIU != NULL) {DeleteObject(pW->hFontBIU); pW->hFontBIU = NULL;}

		free(pW);
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	ＤＣ描画域のビットマップサイズ設定																			//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				cx, cy	 - ＤＣ描画スペース（ビットマップ）のサイズ												//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTxoSetDcBmpSize		(HAJCTXO pW, int cx, int cy)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		pW->szBmp.cx = cx;
		pW->szBmp.cy = cy;
		rc = TRUE;
	}
	return rc;
}

//==============================================================================================================//
//	文字詰め設定																								//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				Align	 - 文字詰め指定（AJCTXOALIGN_{LEFT/CENTER/RIGHT/TAB_RIGHT)）							//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL			WINAPI	AjcTxoSetAlign(HAJCTXO pW, AJCTXOALIGN Align)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		if (Align == AJCTXOALIGN_LEFT || Align == AJCTXOALIGN_CENTER || Align == AJCTXOALIGN_RIGHT || Align == AJCTXOALIGN_TAB_RIGHT) {
			pW->Align = Align;
			rc = TRUE;
		}
	}
	return rc;
}

//==============================================================================================================//
//	文字詰め取得																								//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	AJCTXOALIGN	WINAPI	AjcTxoGetAlign(HAJCTXO pW)
{
	AJCTXOALIGN	rc = (AJCTXOALIGN)0;

	if (IS_MY_INST(pW)) {
		rc = pW->Align;
	}
	return rc;
}

//==============================================================================================================//
//	描画位置設定																								//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				x, y, pPt	- 描画ピクセル位置																	//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- Ｘ，Ｙ指定 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTxoLocate		(HAJCTXO pW, int x, int y)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		pW->x = x;
		pW->y = y;
		rc = TRUE;
	}
	return rc;
}
//----- ＰＯＩＮＴ指定 -----------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTxoSetPos		(HAJCTXO pW, LPPOINT pPt)
{
	BOOL	rc = FALSE;

	if (pPt != NULL) {
		rc = AjcTxoLocate(pW, pPt->x, pPt->y);
	}
	return rc;
}
//==============================================================================================================//
//	描画位置取得																								//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				pPt		- 描画ピクセル位置を格納するバッファのアドレス											//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTxoGetPos		(HAJCTXO pW, LPPOINT pPt)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		if (pPt != NULL) {
			pPt->x = pW->x;
			pPt->y = pW->y;
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	書式テキスト描画																							//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				pFmt	- 書式テキストへのポインタ																//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTxoPrintFA		(HAJCTXO pW, C_BCP pFmt, ...)
{
	BOOL	rc = FALSE;
	va_list vls;
	BC		buf[2048];

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		_vsnprintf(buf, 1024, pFmt, vls);
		buf[2047] = 0;
		va_end	(vls);
		rc = AjcTxoTextOutA(pW, buf);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTxoPrintFW		(HAJCTXO pW, C_WCP pFmt, ...)
{
	BOOL	rc = FALSE;
	va_list vls;
	WC		buf[2048];

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		_vsnwprintf(buf, 1024, pFmt, vls);
		buf[2047] = 0;
		va_end	(vls);
		rc = AjcTxoTextOutW(pW, buf);
	}
	return rc;
}
//==============================================================================================================//
//	書式テキスト描画（描画位置指定）																			//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				x, y	- 描画位置																				//
//				pFmt	- 書式テキストへのポインタ																//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTxoPrintFExA		(HAJCTXO pW, int x, int y, C_BCP pFmt, ...)
{
	BOOL	rc = FALSE;
	va_list vls;
	BC		buf[2048];

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		_vsnprintf(buf, 1024, pFmt, vls);
		buf[2047] = 0;
		va_end	(vls);
		rc = AjcTxoTextOutExA(pW, x, y, buf);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTxoPrintFExW		(HAJCTXO pW, int x, int y, C_WCP pFmt, ...)
{
	BOOL	rc = FALSE;
	va_list vls;
	WC		buf[2048];

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		_vsnwprintf(buf, 1024, pFmt, vls);
		buf[2047] = 0;
		va_end	(vls);
		rc = AjcTxoTextOutExW(pW, x, y, buf);
	}
	return rc;
}
//==============================================================================================================//
//	書式テキスト描画（描画位置指定）																			//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				x, y	- 描画位置																				//
//				pFmt	- 書式テキストへのポインタ																//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTxoPrintFExExA		(HAJCTXO pW, int x, int y, int cx, C_BCP pFmt, ...)
{
	BOOL	rc = FALSE;
	va_list vls;
	BC		buf[2048];

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		_vsnprintf(buf, 1024, pFmt, vls);
		buf[2047] = 0;
		va_end	(vls);
		rc = AjcTxoTextOutExExA(pW, x, y, cx, buf);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTxoPrintFExExW		(HAJCTXO pW, int x, int y, int cx, C_WCP pFmt, ...)
{
	BOOL	rc = FALSE;
	va_list vls;
	WC		buf[2048];

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		_vsnwprintf(buf, 1024, pFmt, vls);
		buf[2047] = 0;
		va_end	(vls);
		rc = AjcTxoTextOutExExW(pW, x, y, cx, buf);
	}
	return rc;
}
//==============================================================================================================//
//	テキスト描画																								//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				pText	- 描画テキストへのポインタ																//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTxoTextOutA		(HAJCTXO pW, C_BCP pText)
{
	BOOL	rc = FALSE;
	POINT	pt;

	if (IS_MY_INST(pW)) {
		AjcTxoGetPos(pW, &pt);
		rc = AjcTxoTextOutExA(pW, pt.x, pt.y, pText);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTxoTextOutW		(HAJCTXO pW, C_WCP pText)
{
	BOOL	rc = FALSE;
	POINT	pt;

	if (IS_MY_INST(pW)) {
		AjcTxoGetPos(pW, &pt);
		rc = AjcTxoTextOutExW(pW, pt.x, pt.y, pText);
	}
	return rc;
}
//==============================================================================================================//
//	テキスト描画（描画位置指定）																				//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				x, y	- 描画位置																				//
//				pText	- 描画テキストへのポインタ																//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTxoTextOutExA	(HAJCTXO pW, int x, int y, C_BCP pText)
{
	return AjcTxoTextOutExExA(pW, x, y, 0, pText);
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTxoTextOutExW	(HAJCTXO pW,  int x, int y, C_WCP pText)
{
	return AjcTxoTextOutExExW(pW, x, y, 0, pText);
}
//==============================================================================================================//
//	テキスト描画（描画位置，テキスト描画域の幅指定）															//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				x, y	- 描画位置																				//
//				pText	- 描画テキストへのポインタ																//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTxoTextOutExExA	(HAJCTXO pW, int x, int y, int cx, C_BCP pText)
{
	BOOL	rc = FALSE;
	UI		len;
	WCP		pTmp;

	if (IS_MY_INST(pW) && pText != NULL) {
		len = AjcByteCharToWideChar(pText, NULL, 0);
		if (pTmp = AjcTAllocW(len)) {
			AjcByteCharToWideChar(pText, pTmp, len);
			rc = AjcTxoTextOutExExW(pW, x, y, cx, pTmp);
			free(pTmp);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTxoTextOutExExW	(HAJCTXO pW,  int x, int y, int cx, C_WCP pText)
{
	BOOL	rc = FALSE;
	C_WCP	pS, pE;						//	テキストポインタ
	SIZE	szFrame;					//	文字列描画枠のピクセルサイズ
	int		cxTxt, cyTxt;				//	テキスト描画域のサイズ
	UI		stl;
	int		left, temp;
	int		rem, add;
	int		nLine = 0;					//	テキストの行数
	UIP		pPixL = NULL;				//	各行のピクセル数
	int		ixL   = 0;					//	行インデクス
	int		xOfs  = 0;					//	中／右詰め時のオフセット
	BOOL	fTabRightDone  = FALSE;		//	ＴＡＢ以降右詰め済みフラグ
	BOOL	fRightOrCenter = FALSE;		//	Ｘ＝右隅／中央指定フラグ

	if (IS_MY_INST(pW) && pText != NULL) {
		do {
			//	テキストの行数設定
			if ((nLine = AjcTxoGetLineCountW(pW, pText)) == 0) break;
			//	各行のピクセル数バッファ確保
			if ((pPixL = AJCMEM(sizeof(UI) * nLine)) == NULL) break;
			//	各行のピクセル数とテキストの矩形サイズ設定
			AjcTxoGetExtent2W(pW, pText, &szFrame, pPixL, nLine);
			//	矩形の幅を指定値に変更
			if (cx != 0) {
				szFrame.cx = cx;
			}
			//	右隅／下隅／中央指定による、描画位置補正
			if (pW->szBmp.cx != 0 && pW->szBmp.cy != 0 &&
			   ((x >= AJCTXO_CORNER_MIN && x <= AJCTXO_CORNER_MAX) || (y >= AJCTXO_CORNER_MIN && y <= AJCTXO_CORNER_MAX))) {
				//	テキスト描画域のサイズ設定
				cxTxt = pW->szBmp.cx - pW->xLeft;
				cyTxt = pW->szBmp.cy - pW->yTop;
				//	Ｘ＝右隅指定時の位置補正
				if	   (x >= AJCTXO_RIGHT_MIN && x <= AJCTXO_RIGHT_MAX) {
					x = (cxTxt - szFrame.cx) + (x - AJCTXO_RIGHT);
					fRightOrCenter = TRUE;
				}
				//	Ｘ＝中央指定時の位置補正
				else if (x >= AJCTXO_CENTER_MIN && x <= AJCTXO_CENTER_MAX) {
					x = ((cxTxt - szFrame.cx) / 2) + (x - AJCTXO_CENTER);
					fRightOrCenter = TRUE;
				}
				//	Ｙ＝下隅指定時の位置補正
				if		(y >= AJCTXO_BOTTOM_MIN && y <= AJCTXO_BOTTOM_MAX) {
					y = (cyTxt - szFrame.cy) + (y - AJCTXO_BOTTOM);
				}
				//	Ｙ＝中央指定時の位置補正
				else if (y >= AJCTXO_CENTER_MIN && y <= AJCTXO_CENTER_MAX) {
					y = ((cyTxt - szFrame.cy) / 2) + (y - AJCTXO_CENTER);
				}
			}
			//	改行左端位置設定
			left = temp = x;
			//	テキストポインタ初期化
			pS = pE = pText;
			//	文字詰めによるＸオフセット設定
			switch (pW->Align) {
				default:
				case AJCTXOALIGN_LEFT:		xOfs = 0;								break;
				case AJCTXOALIGN_CENTER:	xOfs = ((szFrame.cx - pPixL[ixL]) / 2);	break;
				case AJCTXOALIGN_RIGHT:		xOfs =	(szFrame.cx - pPixL[ixL]);		break;
				case AJCTXOALIGN_TAB_RIGHT:	xOfs = 0;								break;
			}

			//	ＴＡＢ以降右詰め済みフラグクリアー
			fTabRightDone = FALSE;

			while (*pS != 0) {
				//	制御文字検索
				pE	= pS;
				stl = 0;
				while (*pE != 0 && !MAjcIsCntrl(*pE)) {
						pE++;
						stl++;	//	文字列長更新
				}
				//	制御文字直前までの文字列描画
				if (stl != 0) {
					SIZE	sz;
					//	文字列描画
					TextOutW(pW->hdc, pW->xLeft + x + xOfs, pW->yTop + y, pS, stl);
					//	ピクセル数算出し、Ｘ位置更新
					GetTextExtentPoint32W(pW->hdc, pS, stl, &sz);
					x	+= sz.cx;
				}
				//	制御文字アクション
				if (*pE != 0 && MAjcIsCntrl(*pE)) {
					switch (*pE) {
						case L'\x1B':								//	● '\x1B' : ＥＳＣ文字列によるＤＣ設定
							pE = AjcTxoEscOutW(pW, pE, TRUE);
							break;

						case L'\t':									//	● '\t'   : タブ
								//	●左詰め
								if (pW->Align == AJCTXOALIGN_LEFT) {
									//	タブステップ＝正数 and 右端／中央指定無し？
									if (pW->TabStep > 0 && !fRightOrCenter) {
										//	Ｘ位置更新（可変幅）
										rem = (x % pW->TabStep);
										if (rem == 0) add =  pW->TabStep;
										else		  add = (pW->TabStep - rem);
										if (add < pW->cxChar) add += pW->TabStep;
										x += add;
									}
									//	タブステップ＝負数 or 右端／中央指定あり？
									else {
										//	Ｘ位置更新（固定幅）
										x	+= abs(pW->TabStep);
									}
								}
								//	●ＴＡＢ以降を右詰め？
								else if (pW->Align == AJCTXOALIGN_TAB_RIGHT) {
									//	各行で１度だけタブ処理する
									if (!fTabRightDone) {
										fTabRightDone = TRUE;
										//	Ｘ位置更新
										x += abs(pW->TabStep);
										//	Ｘを右詰め位置に設定
										add = (szFrame.cx - pPixL[ixL]);
										x	+= add;
									}
								}
								//	●その他（中央合わせ／右詰め）？
								else {
										//	Ｘ位置更新（固定幅）
										x	+= abs(pW->TabStep);
								}
								//	'\t' スキップ
								pE++;
								break;

						case L'\r':									//	● '\r'   : Ｘ位置復帰
								//	左詰め時のみ復帰処理
								if (pW->Align == AJCTXOALIGN_LEFT) {
									//	復帰Ｘ位置設定
									temp = x = 0;
								}
								//	'\r' スキップ
								pE++;
								break;

						case L'\n':									//	● '\n'   : 改行
								//	行インデクス更新
								ixL++;
								//	Ｘ先頭位置設定
								x = temp;
								//	文字詰めによるＸオフセット設定
								switch (pW->Align) {
									default:
									case AJCTXOALIGN_LEFT:		xOfs = 0;								break;
									case AJCTXOALIGN_CENTER:	xOfs = ((szFrame.cx - pPixL[ixL]) / 2);	break;
									case AJCTXOALIGN_RIGHT:		xOfs =	(szFrame.cx - pPixL[ixL]);		break;
									case AJCTXOALIGN_TAB_RIGHT:	xOfs = 0;								break;
								}
								//	ＴＡＢ以降右詰め済みフラグクリアー
								fTabRightDone = FALSE;
								//	Ｘ先頭位置回復
								temp = left;
								//	行位置更新
								y += (pW->cyChar + pW->Lsp);
								//	行間スペースリセット
								pW->Lsp = pW->LspDef;
								//	'\n' スキップ
								pE++;
								break;

						case L'\f':									//	● '\f'   : １ピクセル進める
								//	１ピクセル進める
								x++;
								//	'\f' スキップ
								pE++;
								break;

						case L'\a':									//	● '\f'   : １０ピクセル進める
								//	１ピクセル進める
								x += 10;
								//	'\f' スキップ
								pE++;
								break;

						default:
								//	制御文字スキップ
								pE++;
								break;
					}
				}
				//	次の文字列先頭設定
				pS = pE;
			}
			//	描画位置退避
			pW->x = x;
			pW->y = y;

			rc = TRUE;
		} while(0);

		//	各行のピクセル数バッファ解放
		if (pPixL != NULL) free(pPixL);
	}
	return rc;
}
//==============================================================================================================//
//	ＥＳＣ描画（非公開）																						//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				pEsc	 - ＥＳＣシーケンス文字列																//
//				fDraw	 - 描画フラグ（TRUE - "\x1B[id・・z" によるイメージを描画する）							//
//																												//
//	戻り値	：	≠NULL - ESCシーケンスの直後のアドレス															//
//				＝NULL - ESC[ 以外																				//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcTxoEscOutA		(HAJCTXO pW, C_BCP pEsc, BOOL fDraw)
{
	BCP		rc = NULL;
	UI		lEsc = 0;
	C_BCP	pS	 = pEsc;
	C_BCP	pE;
	UI		len;
	WCP		pTmp;

	if (IS_MY_INST(pW)) {
		if (*pS == '\x1B') {
			do {
				//	'\x1B' スキップ
				pS++; lEsc++;
				//	ＥＳＣ末尾（英字）検索
				pE = pS;
				while (*pE != 0 && !MAjcAscIsAlphaW(*pE)) {
					pE++; lEsc++;
				}
				//	ＥＳＣ末尾（英字）スキップ
				if (MAjcAscIsAlphaW(*pE)) {
					pE++; lEsc++;
				}
				//	１つのＥＳＣ文字列の直後設定
				pS = pE;
			} while (*pS == '\x1B');
			//	戻り値＝ＥＳＣ文字列の直後
			rc = (BCP)pS;
			//	ＥＳＣ文字列をUNICODEに変換しＥＳＣ描画
			len = MultiByteToWideChar(CP_ACP, 0, pEsc, lEsc, NULL, 0);
			if (pTmp = AjcTAllocW(len + 1)) {
				AjcByteCharToWideChar(pEsc, pTmp, len);
				pTmp[len - 1] = 0;
				AjcTxoEscOutW(pW, pTmp, fDraw);
				free(pTmp);
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcTxoEscOutW		(HAJCTXO pW, C_WCP pEsc, BOOL fDraw)
{
	WCP		rc = NULL;
	C_WCP	pS = pEsc;
	C_WCP	pE;

	if (IS_MY_INST(pW)) {
		if (*pS == L'\x1B') {
			do {
				//	'\x1B' スキップ
				pS++;
				//	ＥＳＣ末尾（英字）検索
				pE = pS;
				while (*pE != 0 && !MAjcAscIsAlphaW(*pE)) {
					pE++;
				}
				//	ESC[・・・
				if (*pS == L'[') {
					//	'[' スキップ
					pS++;
					//	ＥＳＣ文字列解析
					if		(*pE == L'T') {							//	● ＥＳＣシーケンス末尾 = T - ボールド設定
						pW->fFont |= AJCCFBE_BOLD;
						ChangeFontoToHdc(pW);
						pW->NestBold++;
					}
					else if 	 (*pE == L't') {					//	● ＥＳＣシーケンス末尾 = t - ボールド解除
						if (pW->NestBold >= 1) {
							if (--pW->NestBold == 0) {
								pW->fFont &= ~AJCCFBE_BOLD;
								ChangeFontoToHdc(pW);
							}
						}
					}
					else if 	 (*pE == L'I') {					//	● ＥＳＣシーケンス末尾 = I - イタリック設定
						pW->fFont |= AJCCFBE_ITALIC;
						ChangeFontoToHdc(pW);
						pW->NestItalic++;
					}
					else if 	 (*pE == L'i') {					//	● ＥＳＣシーケンス末尾 = i - イタリック解除
						if (pW->NestItalic >= 1) {
							if (--pW->NestItalic == 0) {
								pW->fFont &= ~AJCCFBE_ITALIC;
								ChangeFontoToHdc(pW);
							}
						}
					}
					else if 	 (*pE == L'U') {					//	● ＥＳＣシーケンス末尾 = U - アンダーライン設定
						pW->fFont |= AJCCFBE_ULINE;
						ChangeFontoToHdc(pW);
						pW->NestULine++;
					}
					else if 	 (*pE == L'u') {					//	● ＥＳＣシーケンス末尾 = u - アンダーライン解除
						if (pW->NestULine >= 1) {
							if (--pW->NestULine == 0) {
								pW->fFont &= ~AJCCFBE_ULINE;
								ChangeFontoToHdc(pW);
							}
						}
					}
					else if 	 (*pE == L'N') {					//	● ＥＳＣシーケンス末尾 = N - ボールドとイタリック解除
						pW->fFont &= ~(AJCCFBE_BOLD | AJCCFBE_ITALIC | AJCCFBE_ULINE);
						ChangeFontoToHdc(pW);
						//	斜字、太字、下線ネストカウンタクリアー
						pW->NestBold = pW->NestItalic = pW->NestULine = 0;
					}
					else if 	 (*pE == L'L') {					//	● ＥＳＣシーケンス末尾 = L - 行間スペース
						if (*pS == L'L') {							//		・ESC[L
							pW->Lsp = pW->LspDef;
						}
						else {										//		・ESC[nnnL
							int	n = _wtoi(pS);
							pW->Lsp = pW->cyChar * n / 100;
						}
					}
					else if 	 (*pE == L'z') {					//	● ＥＳＣシーケンス末尾 = z - イメージ描画
						TXOIMGINFO	ImgInfo;
						BITMAP		bm;

						if (SubGetImgInfo(pS, &ImgInfo, &bm)) {
							//	イメージ描画
							if (fDraw) {
								//	描画位置が負数の場合、右端／下端から減算
								if (ImgInfo.x < 0) {
									ImgInfo.x = pW->szBmp.cx + ImgInfo.x;
								}
								if (ImgInfo.y < 0) {
									ImgInfo.y = pW->szBmp.cy + ImgInfo.y;
								}
								//	アイコン描画
								if (ImgInfo.fIcon) {
									DrawIconEx(pW->hdc, ImgInfo.x, ImgInfo.y, ImgInfo.u.hIcon,
												ImgInfo.cx, ImgInfo.cy, 0, NULL, DI_NORMAL);
								}
								//	ビットマップ描画
								else {
									HDC		mdc    = CreateCompatibleDC(pW->hdc);
									HBITMAP	hBmpSv = (HBITMAP)SelectObject(mdc, ImgInfo.u.hBmp);
									StretchBlt(pW->hdc, ImgInfo.x, ImgInfo.y, ImgInfo.cx  , ImgInfo.cy,
											   mdc    , 0      , 0      , bm.bmWidth, bm.bmHeight, SRCCOPY);
									SelectObject(mdc, hBmpSv);
									DeleteDC(mdc);
								}
							}
						}
					}
					else if (*pE == L'm') {							//	● ＥＳＣシーケンス末尾 = 'm' 
						while (MAjcAscIsDigitW(*pS)) {
							//	指定数値設定
							int n = _wtoi(pS);
							//	0		: 文字色，文字背景色，フォント リセット
							if		(n == 0) {
								//	描画色リセット
								ResetToDefaultColor(pW);
								//	フォントリセット
								pW->fFont = AJCCFBE_NORMAL;
								ChangeFontoToHdc(pW);
								//	斜字、太字、下線ネストカウンタクリアー
								pW->NestBold = pW->NestItalic = pW->NestULine = 0;
								//	数字(0)スキップ
								while (MAjcAscIsDigitW(*pS)) pS++;
							}
							//	1		: ボールド（太字）設定
							else if (n == 1) {
								pW->fFont |= AJCCFBE_BOLD;
								ChangeFontoToHdc(pW);
								pW->NestBold++;
								//	数字(1)スキップ
								while (MAjcAscIsDigitW(*pS)) pS++;
							}
							//	3		: イタリック（斜字）設定
							else if (n == 3) {
								pW->fFont |= AJCCFBE_ITALIC;
								ChangeFontoToHdc(pW);
								pW->NestItalic++;
								//	数字(3)スキップ
								while (MAjcAscIsDigitW(*pS)) pS++;
							}
							//	4		: アンダーライン（下線）設定
							else if (n == 4) {
								pW->fFont |= AJCCFBE_ULINE;
								ChangeFontoToHdc(pW);
								pW->NestULine++;
								//	数字(4)スキップ
								while (MAjcAscIsDigitW(*pS)) pS++;
							}
							//	9		: 文字色，文字背景色 リセット
							else if (n == 9) {
								SetBkMode	(pW->hdc, TRANSPARENT );
								SetTextColor(pW->hdc, pW->TxtColorDef);
								//	数字(9)スキップ
								while (MAjcAscIsDigitW(*pS)) pS++;
							}
							//	30 - 37 : 文字色設定
							else if (n >= 30 && n <= 37) {
								SetTextColor(pW->hdc, pW->Palette[n - 30]);
								//	数字(30 - 37)スキップ
								while (MAjcAscIsDigitW(*pS)) pS++;
							}
							//	38		: 文字色をＲＧＢ指定
							else if (n == 38) {
								COLORREF rgb;
								//	数字(38)と ';' スキップ
								while (MAjcAscIsDigitW(*pS)) pS++;
								if (*pS == L';') {
									pS++;
									n = _wtoi(pS);
									if (n == 2) {
										//	数字(2)と ';' スキップ
										while (MAjcAscIsDigitW(*pS)) pS++;
										if (*pS == L';') {
											WCP		p;
											pS++;
											if ((p = AjcTxoGetRgbByTextW(pS, &rgb)) != NULL) {
												pS = p;
												SetTextColor(pW->hdc, rgb);
											}
										}
									}
								}
							}
							//	39		: 文字色リセット
							else if (n == 39) {
								SetTextColor(pW->hdc, pW->TxtColorDef);
								//	数字(39)スキップ
								while (MAjcAscIsDigitW(*pS)) pS++;
							}
							//	40 - 47 : 文字背景色設定
							else if (n >= 40 && n <= 47) {
								SetBkMode (pW->hdc, OPAQUE);
								SetBkColor(pW->hdc, pW->Palette[n - 40]);
								//	数字(40 - 47)スキップ
								while (MAjcAscIsDigitW(*pS)) pS++;
							}
							//	48		: 文字背景色をＲＧＢ指定
							else if (n == 48) {
								COLORREF rgb;
								//	数字(48)と ';' スキップ
								while (MAjcAscIsDigitW(*pS)) pS++;
								if (*pS == L';') {
									pS++;
									n = _wtoi(pS);
									if (n == 2) {
										//	数字(2)と ';' スキップ
										while (MAjcAscIsDigitW(*pS)) pS++;
										if (*pS == L';') {
											WCP		p;
											pS++;
											if ((p = AjcTxoGetRgbByText(pS, &rgb)) != NULL) {
												pS = p;
												SetBkMode (pW->hdc, OPAQUE);
												SetBkColor(pW->hdc, rgb);
											}
										}
									}
								}
							}
							//	49		: 背景色リセット（透明）
							else if (n == 49) {
								SetBkMode (pW->hdc, TRANSPARENT);
								//	数字(49)スキップ
								while (MAjcAscIsDigitW(*pS)) pS++;
							}
							else {
								//	数字(??)スキップ
								while (MAjcAscIsDigitW(*pS)) pS++;
							}
							//	セミコロン(;)スキップ
							if (*pS == L';') pS++;
						}
					}
					//	ＥＳＣ末尾（英字）スキップ
					if (*pE != 0) {
						pE++;
					}
				}
				//	１つのＥＳＣ文字列の直後設定
				pS = pE;
			} while (*pS == L'\x1B');
			//	ＥＳＣ文字列の直後を返す
			rc = (WCP)pS;
		}
	}
	return rc;
}


//==============================================================================================================//
//	文字列の行数取得																							//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				pText	 - テキストへのポインタ																	//
//																												//
//	戻り値	：	≠0 - ＯＫ（行数）																				//
//				＝0 - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcTxoGetLineCountA		(HAJCTXO pW, C_BCP pText)
{
	UI		rc = 0;
	C_BCP	p;

	if (IS_MY_INST(pW) && pText != NULL) {
		rc = 1;
		p  = pText;
		while (*p != 0) {
			if (*p == '\n') {
				rc++;
			}
			p++;
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcTxoGetLineCountW		(HAJCTXO pW, C_WCP pText)
{
	UI		rc = 0;
	C_WCP	p;

	if (IS_MY_INST(pW) && pText != NULL) {
		rc = 1;
		p  = pText;
		while (*p != 0) {
			if (*p == L'\n') {
				rc++;
			}
			p++;
		}
	}
	return rc;
}
//==============================================================================================================//
//	文字列描画の矩形サイズ取得																					//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				pText	 - テキストへのポインタ																	//
//				pSzTxt	 - テキストの描画サイズを格納するバッファへのポインタ（不要時はNULL）					//
//																												//
//	戻り値	：	≠0 - ＯＫ（行数）																				//
//				＝0 - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcTxoGetExtentA		(HAJCTXO pW, C_BCP pText, LPSIZE pSzTxt)
{
	return AjcTxoGetExtentExA(pW, 0, 0, pText, pSzTxt);
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcTxoGetExtentW		(HAJCTXO pW, C_WCP pText, LPSIZE pSzTxt)
{
	return AjcTxoGetExtentExW(pW, 0, 0, pText, pSzTxt);
}
//==============================================================================================================//
//	文字列描画の矩形サイズ取得																					//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				pText	 - テキストへのポインタ																	//
//				pSzTxt	 - テキストの描画サイズを格納するバッファへのポインタ（不要時はNULL）					//
//																												//
//	戻り値	：	≠0 - ＯＫ（行数）																				//
//				＝0 - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcTxoGetExtent2A		(HAJCTXO pW, C_BCP pText, LPSIZE pSzTxt, UI LinePixs[], UI nLine)
{
	return AjcTxoGetExtent2ExA(pW, 0, 0, pText, pSzTxt, LinePixs, nLine);
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcTxoGetExtent2W		(HAJCTXO pW, C_WCP pText, LPSIZE pSzTxt, UI LinePixs[], UI nLine)
{
	return AjcTxoGetExtent2ExW(pW, 0, 0, pText, pSzTxt, LinePixs, nLine);
}
//==============================================================================================================//
//	文字列描画の矩形サイズ取得																					//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				x, y	 - 描画位置																				//
//				pText	 - テキストへのポインタ																	//
//				pSzTxt	 - テキストの描画サイズを格納するバッファへのポインタ（不要時はNULL）					//
//																												//
//	戻り値	：	≠0 - ＯＫ（行数）																				//
//				＝0 - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcTxoGetExtentExA		(HAJCTXO pW, int x, int y, C_BCP pText, LPSIZE pSzTxt)
{
	return AjcTxoGetExtent2ExA(pW, x, y, pText, pSzTxt, NULL, 0);
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcTxoGetExtentExW		(HAJCTXO pW, int x, int y, C_WCP pText, LPSIZE pSzTxt)
{
	return AjcTxoGetExtent2ExW(pW, x, y, pText, pSzTxt, NULL, 0);
}
//==============================================================================================================//
//	文字列描画の矩形サイズ取得（ＴＡＢ対応，各行の描画ピクセル取得）											//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				x, y	 - 描画位置																				//
//				pText	 - テキストへのポインタ																	//
//				pSzTxt	 - テキストの描画サイズを格納するバッファへのポインタ（不要時はNULL）					//
//				LinePixs - 各行のピクセル数を格納する配列バッファのアドレス										//
//				nLine	 - 各行のピクセル数を格納する配列バッファの要素数										//
//																												//
//	戻り値	：	≠0 - ＯＫ（行数）																				//
//				＝0 - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcTxoGetExtent2ExA		(HAJCTXO pW, int x, int y, C_BCP pText, LPSIZE pSzTxt, UI LinePixs[], UI nLine)
{
	BOOL	rc = FALSE;
	UI		len;
	WCP		pTmp;

	if (IS_MY_INST(pW)) {
		len = AjcByteCharToWideChar(pText, NULL, 0);
		if (pTmp = AjcTAllocW(len)) {
			AjcByteCharToWideChar(pText, pTmp, len);
			rc = AjcTxoGetExtent2ExW(pW, x, y, pTmp, pSzTxt, LinePixs, nLine);
			free(pTmp);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcTxoGetExtent2ExW		(HAJCTXO pW, int x, int y, C_WCP pText, LPSIZE pSzTxt, UI LinePixs[], UI nLine)
{
	UI			rc = 0;
	SIZE		sz;						//	エクステントサイズ
	C_WCP		pS, pE;					//	エクステント算出の先頭と末尾
	int			pix    = 0;				//	文字列の幅サイズ算出ワーク
	int			stl;					//	エクステント算出の文字列長
	int			lines  = 0;				//	行数
	int			width  = 0;				//	横ピクセル数
	int			height = 0;				//	縦ピクセル数
	int			left, temp;
	int			rem, add;
	HFONT		hFontSv;				//	フォント退避
	UI			BkModeSv;				//	背景モード退避
	COLORREF	BkColorSv;				//	背景色退避
	COLORREF	TxtColorSv;				//	文字色退避
	UI			fFontSv;				//	フォントフラグ退避
	int			LspSv;					//	行間スペース退避
	UI			ixLine = 0;				//	行インデクス
	BOOL		fTabRightDone  = FALSE;	//	TAB以降右詰め済みフラグ
	BOOL		fRightOrCenter = FALSE;	//	Ｘ＝右隅／中央指定フラグ

	if (IS_MY_INST(pW)) {
		//	各行のピクセル数バッファクリアー
		if (LinePixs != NULL && ixLine < nLine) {
			memset(LinePixs, 0, sizeof(UI) * nLine);
		}
		//	ＤＣ属性退避
		hFontSv	= (HFONT)SelectObject(pW->hdc, GetStockObject(SYSTEM_FONT));
		SelectObject(pW->hdc, (HGDIOBJ)hFontSv);
		BkModeSv	= GetBkMode   (pW->hdc);
		BkColorSv	= GetBkColor  (pW->hdc);
		TxtColorSv	= GetTextColor(pW->hdc);
		//	内部情報退避
		fFontSv	= pW->fFont;
		LspSv	= pW->Lsp;

		//	Ｘ＝右隅指定時、フラグ設定
		if	   (x >= AJCTXO_RIGHT_MIN && x <= AJCTXO_RIGHT_MAX) {
			fRightOrCenter = TRUE;
		}
		//	Ｘ＝中央指定時、フラグ設定
		else if (x >= AJCTXO_CENTER_MIN && x <= AJCTXO_CENTER_MAX) {
			fRightOrCenter = TRUE;
		}

		//	行数、高さを初期化
		lines  = 1;
		height = pW->cyChar;

		//	改行左端位置設定
		left = temp = x;
		//	テキストポインタ初期化
		pS = pE = pText;

		//	ＴＡＢ以降右詰め済みフラグクリアー
		fTabRightDone = FALSE;

		while (*pS != 0) {
			//	制御文字検索
			pE	= pS;
			stl = 0;
			while (*pE != 0 && !MAjcIsCntrl(*pE)) {
					pE++;
					stl++;	//	文字列長更新
			}
			//	制御文字直前の文字列までのピクセル数
			if (stl != 0) {
				//	Ｘ位置，ピクセル数更新
				GetTextExtentPoint32W(pW->hdc, pS, stl, &sz);
				x	+= sz.cx;
				pix += sz.cx;
			}
			//	制御文字アクション
			if (*pE != 0 && MAjcIsCntrl(*pE)) {
				switch (*pE) {
					case L'\x1B':									//	● '\x1B' : ＥＳＣ文字列によるＤＣ設定
						pE = AjcTxoEscOutW(pW, pE, FALSE);			//		FALSE:実際の描画は行わない
						break;

					case L'\t':										//	● '\t'   : タブ
						//	●左詰め
						if (pW->Align == AJCTXOALIGN_LEFT) {
							//	タブステップ＝正数 and 右端／中央指定無し？
							if (pW->TabStep > 0  && !fRightOrCenter) {
								//	Ｘ位置更新（可変幅）
								rem = (x % pW->TabStep);
								if (rem == 0) add = pW->TabStep;
								else		  add = pW->TabStep - rem;
								if (add < pW->cxChar) add += pW->TabStep;
								x	+= add;
								pix += add;
							}
							//	タブステップ＝負数 or 右端／中央指定あり？
							else {
								//	Ｘ位置更新（固定幅）
								add = abs(pW->TabStep);
								x	+= add;
								pix += add;
							}
						}
						//	●ＴＡＢ以降を右詰め？
						else if (pW->Align == AJCTXOALIGN_TAB_RIGHT) {
							//	各行で１度だけタブ処理する
							if (!fTabRightDone) {
								fTabRightDone = TRUE;
								//	Ｘ位置更新
								add = abs(pW->TabStep);
								x	+= add;
								pix += add;
							}
						}
						//	●その他（中央合わせ／右詰め）？
						else {
							//	Ｘ位置更新（固定幅）
							add = abs(pW->TabStep);
							x	+= add;
							pix += add;
						}
						//	'\t' スキップ
						pE++;
						break;

					case L'\r':										//	● '\r'   : Ｘ位置復帰
						//	左詰め時のみ復帰処理
						if (pW->Align == AJCTXOALIGN_LEFT) {
							//	最大幅更新
							width = __max(width, pix);
							pix   = 0;
							//	復帰Ｘ位置設定
							temp = x = 0;
						}
						//	'\r' スキップ
						pE++;
						break;

					case L'\n':										//	● '\n'   : 改行
						//	各行のピクセル数格納
						if (LinePixs != NULL && ixLine < nLine) {
							LinePixs[ixLine] = pix;
							ixLine++;
						}
						//	最大幅更新
						width = __max(width, pix);
						pix   = 0;
						//	高さ更新
						height += (pW->cyChar + pW->Lsp);
						//	行数更新
						lines++;
						//	ＴＡＢ以降右詰め済みフラグクリアー
						fTabRightDone = FALSE;
						//	Ｘ先頭位置設定
						x = temp;
						//	Ｘ先頭位置回復
						temp = left;
						//	行位置更新
						y += (pW->cyChar + pW->Lsp);
						//	行間スペースリセット
						pW->Lsp = pW->LspDef;
						//	'\n' スキップ
						pE++;
						break;

					default:
						//	制御文字スキップ
						pE++;
						break;
				}
			}
			//	次の文字列先頭設定
			pS = pE;
		}
		//	各行のピクセル数格納
		if (LinePixs != NULL && ixLine < nLine) {
			LinePixs[ixLine] = pix;
			ixLine++;
		}
		//	最大幅更新
		width = __max(width, pix);
		pix   = 0;
		//	内部情報回復
		pW->fFont = fFontSv;
		pW->Lsp   = LspSv;
		//	ＤＣ属性回復
		SelectObject(pW->hdc, hFontSv	);
		SetTextColor(pW->hdc, TxtColorSv);
		SetBkColor	(pW->hdc, BkColorSv );
		SetBkMode	(pW->hdc, BkModeSv	);

		//	戻り情報設定
		if (pSzTxt != NULL) {
			pSzTxt->cx = width;
			pSzTxt->cy = height;
		}
		//	戻り値＝行数
		rc = lines;
	}
	return rc;
}
//==============================================================================================================//
//	イメージ描画域の矩形サイズ取得																				//
//																												//
//	引　数	：	pText	 - テキストへのポインタ																	//
//				pSzTxt	 - テキストの描画サイズを格納するバッファへのポインタ（不要時はNULL）					//
//																												//
//	戻り値	：	TRUE  - ＯＫ（有効なイメージ描画あり）															//
//				FALSE - 有効なイメージ描画無し																	//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcTxoGetImgDrawSizeA(C_BCP pText, LPSIZE pSzImg)
{
	BOOL	rc = FALSE;
	UI		len;
	WCP		pTmp;

	if (pText != NULL && pSzImg != NULL) {
		len = AjcByteCharToWideChar(pText, NULL, 0);
		if (pTmp = AjcTAllocW(len)) {
			AjcByteCharToWideChar(pText, pTmp, len);
			rc = AjcTxoGetImgSpaceSize(pTmp, pSzImg);
			free(pTmp);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcTxoGetImgSpaceSize(C_WCP pText, LPSIZE pSzImg)
{
	BOOL	rc = FALSE;
	C_UTP	pS = NULL;
	C_UTP	pE = NULL;

	if (pText != NULL && pSzImg != NULL) {
		//	イメージ描画テキスト("\x1B[id・・z")から、ウインドサイズ補正
		pS = MAjcStrStr(pText, TEXT("\x1B"));
		while (pS != NULL) {
			TXOIMGINFO	ImgInfo = {0};
			int			iw = 0;
			int			ih = 0;
			pS++;
			pE = pS;
			while (*pE != 0 && !MAjcAscIsAlphaW(*pE)) {
				pE++;
			}
			if (*pE == L'z') {
				if (*pS == L'[') {
					pS++;
					if (SubGetImgInfo(pS, &ImgInfo, NULL)) {
						if (ImgInfo.x >= 0 && ImgInfo.y >= 0) {
							if (ImgInfo.x >= 0) iw = ImgInfo.x + ImgInfo.cx;
							if (ImgInfo.y >= 0) ih = ImgInfo.y + ImgInfo.cy;
							pSzImg->cx = __max(pSzImg->cx, iw);
							pSzImg->cy = __max(pSzImg->cy, ih);
							rc = TRUE;
						}
					}
				}
			}
			pS = MAjcStrStr(pS, TEXT("\x1B"));
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	イメージ描画情報取得（末尾が 'z' のエスケープシーケンスの解析）												//
//																												//
//	引　数	：	pEsc	- エスケープシーケンス文字列('\x1B['の直後 "<id>;<x>;<y>;<cx>;<cy>z")					//
//				pInfo	- イメージ情報を格納するバッファ														//
//				pBm		- ビットマップ情報格納バッファ（不要時はNULL）											//
//																												//
//	戻り値	：	TRUE	- ＯＫ（イメージ情報を設定した）														//
//				FALSE	- エラー																				//
//																												//
//	備　考	：	この関数は、AjcTipText.c からも参照																//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	SubGetImgInfo(C_UTP pEsc, PTXOIMGINFO pInfo, BITMAP *pBm)
{
	BOOL		rc = FALSE;
	C_UTP		pS = pEsc;
	TXOIMGINFO	node;
	UI		id, i;
	union {
		int		arr[4];
		struct {int x, y, cx, cy;} pos;
	} u;

	if (hAvlImg != NULL) {
		do {
			//	ＩＤ設定
			if (MAjcAscIsDigitW(*pS)) {
				id = _wtoi(pS);
				//	数字スキップ
				while (MAjcAscIsDigitW(*pS)) pS++;
				//	当該ＩＤのイメージ描画情報取得
				if (AjcAvlGetNode(hAvlImg, id, &node, sizeof node) != -1) {
					//	デフォルト値設定
					u.pos.x  = node.x;
					u.pos.y  = node.y;
					u.pos.cx = node.cx;
					u.pos.cy = node.cy;
					//	x, y, cx, cy設定
					for (i = 0; i < 4 && *pS == L';'; i++) {
						pS++;										//	セミコロン(;)スキップ
						if (MAjcAscIsDigitW(*pS) || *pS == L'-') {
							u.arr[i] = _wtoi(pS);					//	数値設定
							if (*pS == L'-') pS++;					//	'-'スキップ
							while (MAjcAscIsDigitW(*pS)) pS++;		//	数字スキップ
						}
						else break;
					}
					//	イメージサイズが負数ならば、ゼロ設定（原画サイズとする）
					if (u.pos.cx < 0) u.pos.cx = 0;
					if (u.pos.cy < 0) u.pos.cy = 0;
					//	アイコン情報算出
					if (node.fIcon) {
						ICONINFO	ic;
						if (GetIconInfo(node.u.hIcon, &ic)) {
							//	ゼロサイズ指定ならば、描画サイズ設定
							if (u.pos.cx == 0 || u.pos.cy == 0) {
								BITMAP	bm;
								if (ic.hbmColor != NULL) {
									if (GetObject(ic.hbmColor, sizeof bm, &bm)) {
										if (u.pos.cx == 0) u.pos.cx = bm.bmWidth;
										if (u.pos.cy == 0) u.pos.cy = bm.bmHeight;
									}
									else break;	//	ビットマップハンドルＮＧ
								}
								else if (ic.hbmMask != NULL) {
									if (GetObject(ic.hbmMask, sizeof bm, &bm)) {
										if (u.pos.cx == 0) u.pos.cx = bm.bmWidth;
										if (u.pos.cy == 0) u.pos.cy = bm.bmHeight / 2;
									}
									else break;	//	ビットマップハンドルＮＧ
								}
							}
						}
						else break;	//	アイコンハンドルＮＧ

						//	戻り情報設定
						pInfo->id		= id;
						pInfo->fIcon	= node.fIcon;
						pInfo->u.hIcon	= node.u.hIcon;
						pInfo->x		= u.pos.x;
						pInfo->y		= u.pos.y;
						pInfo->cx		= u.pos.cx;
						pInfo->cy		= u.pos.cy;
					}
					//	ビットマップ情報算出
					else {
						BITMAP	bm;
						if (GetObject(node.u.hBmp, sizeof(BITMAP), &bm) == sizeof(BITMAP)) {
							//	ゼロサイズ指定ならば、描画サイズ設定
							if (u.pos.cx == 0) u.pos.cx = bm.bmWidth;
							if (u.pos.cy == 0) u.pos.cy = bm.bmHeight;
							if (pBm != NULL) {
								memcpy(pBm, &bm, sizeof(BITMAP));
							}
						}
						else break;	//	ビットマップハンドルＮＧ

						//	戻り情報設定
						pInfo->id		= id;
						pInfo->fIcon	= node.fIcon;
						pInfo->u.hBmp	= node.u.hBmp;
						pInfo->x		= u.pos.x;
						pInfo->y		= u.pos.y;
						pInfo->cx		= u.pos.cx;
						pInfo->cy		= u.pos.cy;
					}
				}
				else break;		//	ＩＤ未登録
			}
			else break;			//	ＩＤなし

			//	戻り値＝ＯＫ
			rc = TRUE;
		} while(0);
	}
	return rc;
}

//==============================================================================================================//
//	パレットの設定																								//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				fRestore - ＤＣ復旧フラグ																		//
//																												//
//	戻り値	：	≠ NULL  - ＯＫ（ＲＧＢ表現文字列の直後のアドレス												//
//				＝ NULL  - エラー																				//
//==============================================================================================================//
//----- 全色設定 -----------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTxoSetPalette	(HAJCTXO pW, COLORREF rgb[8])
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		if (rgb != NULL) {
			memcpy(pW->Palette, rgb, sizeof(COLORREF) * 8);
			rc = TRUE;
		}
	}
	return rc;
}
//----- １色設定 -----------------------------------------------------------------------------------------------//
AJCEXPORT	COLORREF	WINAPI	AjcTxoSetPaletteItem(HAJCTXO pW, int ix, COLORREF rgb)
{
	COLORREF	rc = (COLORREF)-1;

	if (IS_MY_INST(pW)) {
		if (ix >= 0 && ix <= 7) {
			rc = pW->Palette[ix];
			pW->Palette[ix] = rgb;
		}
	}
	return rc;
}
//==============================================================================================================//
//	パレットの取得																								//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				fRestore - ＤＣ復旧フラグ																		//
//																												//
//	戻り値	：	≠ NULL  - ＯＫ（ＲＧＢ表現文字列の直後のアドレス												//
//				＝ NULL  - エラー																				//
//==============================================================================================================//
//----- 全色設定 -----------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTxoGetPalette	(HAJCTXO pW, COLORREF rgb[8])
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		if (rgb != NULL) {
			memcpy(rgb, pW->Palette, sizeof(COLORREF) * 8);
			rc = TRUE;
		}
	}
	return rc;
}
//----- １色設定 -----------------------------------------------------------------------------------------------//
AJCEXPORT	COLORREF	WINAPI	AjcTxoGetPaletteItem(HAJCTXO pW, int ix)
{
	COLORREF	rc = (COLORREF)-1;

	if (IS_MY_INST(pW)) {
		if (ix >= 0 && ix <= 7) {
			rc = pW->Palette[ix];
		}
	}
	return rc;
}
//==============================================================================================================//
//	テキストのデフォルト色設定																					//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				defColor - テキストのデフォルト色																//
//																												//
//	戻り値	：	≠ -1：前回のテキストのデフォルト色																//
//				＝ -1：エラー																					//
//==============================================================================================================//
AJCEXPORT	COLORREF	WINAPI	AjcTxoSetDefTextColor(HAJCTXO pW, COLORREF defColor)
{
	COLORREF	rc = (COLORREF)-1;

	if (IS_MY_INST(pW)) {
		rc = pW->TxtColorDef;
		pW->TxtColorDef = defColor;
		//	描画色リセット
		ResetToDefaultColor(pW);
	}
	return rc;
}
//==============================================================================================================//
//	背景のデフォルト色設定																						//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				defColor - テキスト背景デフォルト色（－１：透明）												//
//																												//
//	戻り値	：	≠ -1：前回のテキスト背景デフォルト色															//
//				＝ -1：エラー																					//
//==============================================================================================================//
AJCEXPORT	COLORREF	WINAPI	AjcTxoSetDefBkColor(HAJCTXO pW, COLORREF defColor)
{
	COLORREF	rc = (COLORREF)-1;

	if (IS_MY_INST(pW)) {
		rc = pW->BkColorDef;
		pW->BkColorDef = defColor;
		//	描画色リセット
		ResetToDefaultColor(pW);
	}
	return rc;
}
//==============================================================================================================//
//	文字サイズ取得																								//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				pSize	 - 文字サイズを格納するバッファへのポインタ												//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTxoGetCharSize	 (HAJCTXO pW, LPSIZE pSize)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		if (pSize != NULL) {
			pSize->cx = pW->cxChar;
			pSize->cy = pW->cyChar;
		}
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	行間スペース取得																							//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//																												//
//	戻り値	：	≠-1 ：成功（行間スペース）																		//
//				FALSE：エラー																					//
//==============================================================================================================//
AJCEXPORT	UI			WINAPI	AjcTxoGetLineSpace	 (HAJCTXO pW)
{
	UI	rc = -1;

	if (IS_MY_INST(pW)) {
		rc = pW->LspDef;
	}
	return rc;
}
//==============================================================================================================//
//	行間スペース設定																							//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				defLsp 	- デフォルトの行間スペース（正数：ピクセル数，負数：文字高さのＮ％）					//
//																												//
//	戻り値	：	≠-1 ：成功（行間スペース）																		//
//				＝-1 ：エラー																					//
//==============================================================================================================//
AJCEXPORT	UI			WINAPI	AjcTxoSetLineSpace	 (HAJCTXO pW, int defLsp)
{
	UI	rc = -1;

	if (IS_MY_INST(pW)) {
		rc = pW->LspDef;
		if (defLsp >= 0) pW->LspDef = defLsp;
		else			 pW->LspDef = abs(pW->cyChar * defLsp / 100);
		pW->Lsp = pW->LspDef;
	}
	return rc;
}

//==============================================================================================================//
//	タブステップ取得																							//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				defLsp 	- デフォルトの行間スペース（正数：ピクセル数，負数：文字高さのＮ％）					//
//																												//
//	戻り値	：	≠０ ：成功（タブステップ）																		//
//				＝０：エラー																					//
//==============================================================================================================//
AJCEXPORT	int			WINAPI	AjcTxoGetTabStep	 (HAJCTXO pW)
{
	int	rc = 0;

	if (IS_MY_INST(pW)) {
		rc = pW->TabStep;
	}
	return rc;
}
//==============================================================================================================//
//	タブステップ設定																							//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				TabStep	- タブステップ（ピクセル数，正数：可変幅，負数：固定幅）								//
//																												//
//	戻り値	：	≠０ ：成功（前回のタブステップ）																//
//				＝０：エラー																					//
//==============================================================================================================//
AJCEXPORT	int			WINAPI	AjcTxoSetTabStep	 (HAJCTXO pW, int TabStep)
{
	UI	rc = 0;

	if (IS_MY_INST(pW)) {
		if (TabStep != 0) {
			rc			= pW->TabStep;
			pW->TabStep = TabStep;
		}
	}
	return rc;
}
//==============================================================================================================//
//	文字背景モード取得																							//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//																												//
//	戻り値	：	≠０ ：成功（AJCTXO_{TRANSPARENT/OPAQUE}）														//
//				＝０：エラー																					//
//==============================================================================================================//
AJCEXPORT	AJCTXOBKMODE	WINAPI	AjcTxoGetBkMode	 (HAJCTXO pW)
{
	AJCTXOBKMODE	rc = (AJCTXOBKMODE)0;

	if (IS_MY_INST(pW)) {
		rc = (AJCTXOBKMODE)GetBkMode(pW->hdc);
	}
	return rc;
}
//==============================================================================================================//
//	文字背景モード設定																							//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				BkMode	- 文字背景モード（AJCTXO_{TRANSPARENT/OPAQUE}）											//
//																												//
//	戻り値	：	≠０ ：成功（文字背景モード）																	//
//				＝０ ：エラー																					//
//==============================================================================================================//
AJCEXPORT	AJCTXOBKMODE	WINAPI	AjcTxoSetBkMode	 (HAJCTXO pW, AJCTXOBKMODE BkMode)
{
	AJCTXOBKMODE	rc = (AJCTXOBKMODE)0;

	if (IS_MY_INST(pW)) {
		if (BkMode == AJCTXO_TRANSPARENT || BkMode == AJCTXO_OPAQUE) {
			rc = (AJCTXOBKMODE)SetBkMode(pW->hdc, (int)BkMode);
		}
	}
	return rc;
}

//==============================================================================================================//
//	ＥＳＣ文字列の列挙																							//
//																												//
//	引　数	：	pEsc	 - インスタンスハンドル																	//
//				fRestore - ＤＣ復旧フラグ																		//
//																												//
//	戻り値	：	検出したＥＳＣの数																				//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcTxoEnumEscA	(C_BCP pText, UX cbp, BOOL (CALLBACK *cbNtcEsc)(C_BCP pEsc, UI lEsc, BC cLast, UX cbp))
{
	UI		rc = 0;
	UI		lEsc;
	C_BCP	pEsc, pS;

	if (pText != NULL) {
		pS = pText;
		while (*pS != 0) {
			//	ＥＳＣならば、ＥＳＣ文字列解析
			if (*pS == '\x1B') {
				pEsc = pS;
				//	ＥＳＣ数更新
				rc++;
				//	ＥＳＣ長初期化
				lEsc = 0;
				//	"\x1B" スキップ
				lEsc++;
				pS++;
				//	ＥＳＣ末尾（英字）検索
				while (*pS != 0 && !MAjcAscIsAlphaA(*pS)) {
					lEsc++;
					pS++;
				}
				//	末尾が英字ならばＥＳＣ文字列通知
				if (MAjcAscIsAlphaA(*pS)) {
					//	英字を含めたＥＳＣ長設定
					lEsc++;
					//	ＥＳＣ文字列通知
					if (cbNtcEsc != NULL) {
						if (!cbNtcEsc(pEsc, lEsc, *pS, cbp)) {
							break;
						}
					}
					//	英字スキップ
					pS++;
				}
			}
			//	ＥＳＣ以外ならば、文字列ポインタ更新
			else {
				pS++;
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcTxoEnumEscW	(C_WCP pText, UX cbp, BOOL (CALLBACK *cbNtcEsc)(C_WCP pEsc, UI lEsc, WC cLast, UX cbp))
{
	UI		rc = 0;
	UI		lEsc;
	C_WCP	pEsc, pS;

	if (pText != NULL) {
		pS = pText;
		while (*pS != 0) {
			//	ＥＳＣならば、ＥＳＣ文字列解析
			if (*pS == L'\x1B') {
				pEsc = pS;
				//	ＥＳＣ数更新
				rc++;
				//	ＥＳＣ長初期化
				lEsc = 0;
				//	"\x1B" スキップ
				lEsc++;
				pS++;
				//	ＥＳＣ末尾（英字）検索
				while (*pS != 0 && !MAjcAscIsAlphaW(*pS)) {
					lEsc++;
					pS++;
				}
				//	末尾が英字ならばＥＳＣ文字列通知
				if (MAjcAscIsAlphaW(*pS)) {
					//	英字を含めたＥＳＣ長設定
					lEsc++;
					//	ＥＳＣ文字列通知
					if (cbNtcEsc != NULL) {
						if (!cbNtcEsc(pEsc, lEsc, *pS, cbp)) {
							break;
						}
					}
					//	英字スキップ
					pS++;
				}
			}
			//	ＥＳＣ以外ならば、文字列ポインタ更新
			else {
				pS++;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	ＲＧＢ表現文字列の解析																						//
//																												//
//	引　数	：	pText	 - ＲＧＢ表現文字列("r;g;b")へのポインタ												//
//				pBuf	 - ＲＧＢ値を格納するバッファのアドレス													//
//																												//
//	戻り値	：	≠ NULL  - ＯＫ（ＲＧＢ表現文字列の直後のアドレス）												//
//				＝ NULL  - エラー																				//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcTxoGetRgbByTextA	(C_BCP pText, COLORREF *pBuf)
{
	BCP			rc = NULL;
	C_BCP		pS = pText;
	UI			i;

	union {
		UI					 arr[3];
		struct {UI r, g, b;} s;
	} u;

	if (pText != NULL) {
		//	３つの数字（r;g;b）を取得
		for (i = 0; i < 3; i++) {
			//	数字取得
			if (MAjcAscIsDigitA(*pS)) {
				u.arr[i] = atoi(pS);
			}
			else {
				u.arr[i] = 0;
			}
			//	数字スキップ
			while (MAjcAscIsDigitA(*pS)) pS++;
			//	末尾(b)以外ならばセミコロン(;)スキップ
			if (i < 2) {
				if (*pS == ';') {
					pS++;
				}
				else break;	//	セミコロン(;)が無い場合はエラー
			}
		}
		//	ＲＧＢ設定
		if (i == 3) {
			if (pBuf != NULL) {
				*pBuf = RGB(u.s.r, u.s.g, u.s.b);
			}
			rc = (BCP)pS;
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcTxoGetRgbByTextW	(C_WCP pText, COLORREF *pBuf)
{
	WCP			rc = NULL;
	C_WCP		pS = pText;
	UI			i;

	union {
		UI					 arr[3];
		struct {UI r, g, b;} s;
	} u;

	if (pText != NULL) {
		//	３つの数字（r;g;b）を取得
		for (i = 0; i < 3; i++) {
			//	数字取得
			if (MAjcAscIsDigitW(*pS)) {
				u.arr[i] = _wtoi(pS);
			}
			else {
				u.arr[i] = 0;
			}
			//	数字スキップ
			while (MAjcAscIsDigitW(*pS)) pS++;
			//	末尾(b)以外ならばセミコロン(;)スキップ
			if (i < 2) {
				if (*pS == L';') {
					pS++;
				}
				else break;	//	セミコロン(;)が無い場合はエラー
			}
		}
		//	ＲＧＢ設定
		if (i == 3) {
			if (pBuf != NULL) {
				*pBuf = RGB(u.s.r, u.s.g, u.s.b);
			}
			rc = (WCP)pS;
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	フォント変更																								//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//																												//
//	戻り値	：	なし																							//
//--------------------------------------------------------------------------------------------------------------//
static	VO		ChangeFontoToHdc(HAJCTXO pW)
{
	LOGFONT		lf;

	switch (pW->fFont) {
		case AJCCFBE_NORMAL:		//	●ノーマル
			if (pW->hFontN == NULL) {
				if (GetObject((HGDIOBJ)pW->hFontOrg, sizeof lf, (VOP)&lf)) {
					lf.lfWeight    = FW_DONTCARE;
					lf.lfItalic    = FALSE;
					lf.lfUnderline = FALSE;
					pW->hFontN = CreateFontIndirect(&lf);
				}
			}
			if (pW->hFontN != NULL) {
				SelectObject(pW->hdc, pW->hFontN);
			}
			break;

		case AJCCFBE_BOLD:			//	●ボールド
			if (pW->hFontB == NULL) {
				if (GetObject((HGDIOBJ)pW->hFontOrg, sizeof lf, (VOP)&lf)) {
					lf.lfWeight    = FW_BOLD;
					lf.lfItalic    = FALSE;
					lf.lfUnderline = FALSE;
					pW->hFontB = CreateFontIndirect(&lf);
				}
			}
			if (pW->hFontB != NULL) {
				SelectObject(pW->hdc, pW->hFontB);
			}
			break;

		case AJCCFBE_ITALIC:		//	●イタリック
			if (pW->hFontI == NULL) {
				if (GetObject((HGDIOBJ)pW->hFontOrg, sizeof lf, (VOP)&lf)) {
					lf.lfWeight    = FW_DONTCARE;
					lf.lfItalic    = TRUE;
					lf.lfUnderline = FALSE;
					pW->hFontI = CreateFontIndirect(&lf);
				}
			}
			if (pW->hFontI != NULL) {
				SelectObject(pW->hdc, pW->hFontI);
			}
			break;

		case AJCCFBE_ULINE:			//	●アンダーライン
			if (pW->hFontU == NULL) {
				if (GetObject((HGDIOBJ)pW->hFontOrg, sizeof lf, (VOP)&lf)) {
					lf.lfWeight    = FW_DONTCARE;
					lf.lfItalic    = FALSE;
					lf.lfUnderline = TRUE;
					pW->hFontU = CreateFontIndirect(&lf);
				}
			}
			if (pW->hFontU != NULL) {
				SelectObject(pW->hdc, pW->hFontU);
			}
			break;

		case AJCCFBE_BOLD_ITALIC:	//	●ボールド＋イタリック
			if (pW->hFontBI == NULL) {
				if (GetObject((HGDIOBJ)pW->hFontOrg, sizeof lf, (VOP)&lf)) {
					lf.lfWeight    = FW_BOLD;
					lf.lfItalic    = TRUE;
					lf.lfUnderline = FALSE;
					pW->hFontBI = CreateFontIndirect(&lf);
				}
			}
			if (pW->hFontBI != NULL) {
				SelectObject(pW->hdc, pW->hFontBI);
			}
			break;

		case AJCCFBE_BOLD_ULINE:	//	●ボールド＋アンダーライン
			if (pW->hFontBU == NULL) {
				if (GetObject((HGDIOBJ)pW->hFontOrg, sizeof lf, (VOP)&lf)) {
					lf.lfWeight    = FW_BOLD;
					lf.lfItalic    = FALSE;
					lf.lfUnderline = TRUE;
					pW->hFontBU = CreateFontIndirect(&lf);
				}
			}
			if (pW->hFontBU != NULL) {
				SelectObject(pW->hdc, pW->hFontBU);
			}
			break;

		case AJCCFBE_ITALIC_ULINE:	//	●イタリック＋アンダーライン
			if (pW->hFontIU == NULL) {
				if (GetObject((HGDIOBJ)pW->hFontOrg, sizeof lf, (VOP)&lf)) {
					lf.lfWeight    = FW_DONTCARE;
					lf.lfItalic    = TRUE;
					lf.lfUnderline = TRUE;
					pW->hFontIU = CreateFontIndirect(&lf);
				}
			}
			if (pW->hFontIU != NULL) {
				SelectObject(pW->hdc, pW->hFontIU);
			}
			break;

		case AJCCFBE_ALL:			//	●ボールド＋イタリック＋アンダーライン
			if (pW->hFontBIU == NULL) {
				if (GetObject((HGDIOBJ)pW->hFontOrg, sizeof lf, (VOP)&lf)) {
					lf.lfWeight    = FW_BOLD;
					lf.lfItalic    = TRUE;
					lf.lfUnderline = TRUE;
					pW->hFontBIU = CreateFontIndirect(&lf);
				}
			}
			if (pW->hFontBIU != NULL) {
				SelectObject(pW->hdc, pW->hFontBIU);
			}
			break;
	}
}
//--------------------------------------------------------------------------------------------------------------//
//	描画色リセット																								//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//																												//
//	戻り値	：	なし																							//
//--------------------------------------------------------------------------------------------------------------//
static	VO	ResetToDefaultColor(HAJCTXO pW)
{
	//	テキスト色リセット
	SetTextColor(pW->hdc, pW->TxtColorDef);
	//	背景色リセット
	if (pW->BkColorDef == -1) {
		SetBkMode(pW->hdc, TRANSPARENT);
	}
	else {
		SetBkMode(pW->hdc, OPAQUE);
		SetBkColor(pW->hdc, pW->BkColorDef);
	}
}

//==============================================================================================================//
//	描画ビットマップイメージの登録																				//
//																												//
//	引　数	：	hBmp			- ビットマップハンドル															//
//				id				- 識別ＩＤ																		//
//				x, y, cx, cy	- デフォルトの描画位置															//
//				cx, cy			- デフォルトの描画サイズ（０：イメージサイズを採用）							//
//																												//
//	戻り値	：	TRUE   - ＯＫ																					//
//				FALSE  - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTxoRegistBitmap	(HBITMAP hBmp, UI id, int x, int y, int cx, int cy)
{
	BOOL	rc = FALSE;

	rc = SubRegistImage(FALSE, (UX)hBmp, id, x, y, cx, cy);

	return rc;
}
//==============================================================================================================//
//	描画アイコンイメージの登録																					//
//																												//
//	引　数	：	hIcon			- アイコンハンドル																//
//				id				- 識別ＩＤ																		//
//				x, y, cx, cy	- デフォルトの描画位置															//
//				cx, cy			- デフォルトの描画サイズ（０：イメージサイズを採用）							//
//																												//
//	戻り値	：	TRUE   - ＯＫ																					//
//				FALSE  - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTxoRegistIcon	(HICON hIcon, UI id, int x, int y, int cx, int cy)
{
	BOOL	rc = FALSE;

	rc = SubRegistImage(TRUE, (UX)hIcon, id, x, y, cx, cy);

	return rc;
}
//==============================================================================================================//
//	描画イメージの登録解除																						//
//																												//
//	引　数	：	id				- 識別ＩＤ																		//
//																												//
//	戻り値	：	TRUE   - ＯＫ																					//
//				FALSE  - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTxoUnregistImage	(UI id)
{
	BOOL	rc = FALSE;

	if (hAvlImg != NULL) {
		if (id == -1) {
			rc = AjcAvlDelAllNodes(hAvlImg);
		}
		else {
			rc = AjcAvlDelNode    (hAvlImg, id);
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	描画イメージの登録																							//
//																												//
//	引　数	：	fIcon			- イメージ種別（FALSE : Bitmap, TRUE : Icon）									//
//				handle			- イメージハンドル																//
//				id				- 識別ＩＤ																		//
//				x, y, cx, cy	- デフォルトの描画位置															//
//				cx, cy			- デフォルトの描画サイズ（０：イメージサイズを採用）							//
//																												//
//	戻り値	：	TRUE   - ＯＫ																					//
//				FALSE  - エラー																					//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	SubRegistImage(BOOL fIcon, UX handle, UI id, int x, int y, int cx, int cy)
{
	BOOL		rc = FALSE;
	TXOIMGINFO	node;

	if (hAvlImg == NULL) {
		hAvlImg = AjcAvlCreate(0, NULL, NULL);
	}
	if (hAvlImg != NULL && cx >= 0 && cy >= 0) {
		node.id		= id;
		node.fIcon	= fIcon;
		node.u.h	= handle;
		node.x		= x;
		node.y		= y;
		node.cx		= cx;
		node.cy		= cy;
		rc = AjcAvlInsOrRepNode(hAvlImg, id, &node, sizeof node);
	}
	return rc;
}
