﻿#include	"AjcInternal.h"
//**************************************************************************************************************//
//																												//
//	ツールチップテキスト表示（コントロールに関付けた チップテキスト表示）										//
//																												//
//**************************************************************************************************************//

#define		MY_CLASS		L"AjcTipCtrlClass"

#define		AJCTIPCTRL_ORGHWNDPROP	L"AJCTIPCTRLORGHWND"	//	オリジナルHWNDのプロパティ名（オリジナル自身の場合は無し）

//--------------------------------------------------------------------------------------------------------------//
//	タイマＩＤ																									//
//--------------------------------------------------------------------------------------------------------------//
enum {
	TID_CYC	= 1		,			//	周期監視タイマ
};

//--------------------------------------------------------------------------------------------------------------//
//	処理状態																									//
//--------------------------------------------------------------------------------------------------------------//
typedef enum {
	PST_IDLE		= 0		,	//	アイドル状態
	PST_DELAY				,	//	表示開始遅延中
	PST_SHOW				,	//	表示中
	PST_MOVE				,	//	コントロール間移動猶予中
} STATE, *PSTATE;

static	STATE pst = PST_IDLE;

//--------------------------------------------------------------------------------------------------------------//
//	サブクラス通知情報（ＡＶＬノード）																			//
//--------------------------------------------------------------------------------------------------------------//
typedef struct {
	HWND	hWndNtc;			//	通知先ウインドハンドル
	UI		msg;				//	通知メッセージコード
	UX		lParam;				//	LPARAM
} AVLSUBCLS, *PAVLSUBCLS;
typedef const AVLSUBCLS *PCAVLSUBCLS;

//--------------------------------------------------------------------------------------------------------------//
//	インスタンスワーク（ＡＶＬノード）																			//
//--------------------------------------------------------------------------------------------------------------//
typedef struct {
	HWND		hwnd;					//	コントロールのウインドハンドル
	BOOL		fUnicode;				//	UNICODEフラグ
	BOOL		fEnable;				//	表示許可／禁止フラグ
	HFONT		hFont;					//	フォントハンドル
	WCP			pText;					//	表示テキスト
	int			x, y;					//	チップテキスト表示位置
	int			w;						//	チップテキストの最小幅
	int			h;						//	チップテキストの高さ（０：文字の高さに合わせる）
	COLORREF	TextColor;				//	文字色（－１：デフォルト色）
	COLORREF	BkColor;				//	背景色（－１：デフォルト色）
	COLORREF	BorderColor;			//	外枠色（－１：デフォルト色，－２：外枠非表示）
	BOOL		fShowAlways;			//	非アクティブな状態時でも表示フラグ

	UI			msDelay;				//	表示開始遅延時間[ms]
	UI			msShow;					//	テキスト表示時間[ms]

	BOOL		fOnMouse;				//	マウスがこのコントロール上にあることを示すフラグ
	UI			stDelay;				//	表示遅延開始時刻[ms]
	UI			stShow;					//	表示開始時刻	[ms]

	BOOL		fCBWithBuf;				//	TRUE:コールバックの際に作業バッファを使用，FALSE：未使用
	UX			cbp;															//	コールバックパラメタ
	BOOL		(CALLBACK *cbChkCursor)(HWND hwnd, LPPOINT ptClient, UX cbp);	//	マウス位置確認用コールバック
	union {		
		VOP		vop;															//	チップテキスト取得用コールバック
		C_BCP	(CALLBACK *cbGetTextA) (HWND hwnd, BCP pBuf, UI lBuf, UX cbp);	//	・
		C_WCP	(CALLBACK *cbGetTextW) (HWND hwnd, WCP pBuf, UI lBuf, UX cbp);	//	・
		C_BCP	(CALLBACK *cbGetNoBufA)(HWND hwnd, UX cbp);						//	チップテキスト取得用コールバック（作業バッファ未使用）
		C_WCP	(CALLBACK *cbGetNoBufW)(HWND hwnd, UX cbp);						//	・
	}			u;
} AVLNODE, *PAVLNODE;

//--------------------------------------------------------------------------------------------------------------//
//	全てのチップテキスト表示許可／禁止																			//
//--------------------------------------------------------------------------------------------------------------//
typedef enum {
	TIPALL_DISABLE	= 0				,
	TIPALL_ENABLE	= 0x25A5CDF0	,
} TIPALL_STATE;

static	TIPALL_STATE	TipAllState	 = TIPALL_ENABLE;

//--------------------------------------------------------------------------------------------------------------//
//	作業領域																									//
//--------------------------------------------------------------------------------------------------------------//
extern	int					DefMsShow;								//	デフォルト表示時間(AjcTipTxt.c)

static	AJCTIP_POSMODE	TipPosMode = AJCTIP_POS_UNDER_CONTROL;		//	ツールチップの表示位置モード

static	ATOM				ClassTipCtl	 = 0;			//	クラスハンドル
static	HWND				hTipCtl		 = NULL;		//	ウインドハンドル
static	HAJCAVL				hAvlNode	 = NULL;		//	ＡＶＬハンドル（インスタンスワークノード）
static	HAJCAVL				hAvlSubC	 = NULL;		//	ＡＶＬハンドル（サブクラス通知情報ノード）
static	BOOL				fMultiThread = FALSE;		//	マルチスレッドフラグ
static	CRITICAL_SECTION	CritSec;					//	クリティカルセクション
static	int					DefMsDelay	 = 1000;		//	デフォルト 表示遅延時間[ms]
static	int					DefMsMove	 = 500;			//	デフォルト コントロール間移動猶予時間[ms]

static	BOOL				FlgShowForActive = FALSE;	//	TRUE:全てアクティブ時のみ表示, FALSE:個々の設定による表示
static	HWND				hCurCtl		 = NULL;		//	現在処理中のコントロール・ハンドル
static	HWND				hSavCtl		 = NULL;		//	直前処理中のコントロール・ハンドル
static	PAVLNODE			pCurNode	 = NULL;		//	現在処理中のコントロールのワーク
static	POINT				stPt		 = {0, 0};		//	チップテキスト表示開始時のカーソル位置
static	POINT				svPt		 = {0, 0};		//	マウスカーソル位置退避
static	UI		msMove = 500;							//	コントロール間移動猶予時間[ms]
static	UI		stMove;									//	コントロール間移動開始時刻[ms]

//--------------------------------------------------------------------------------------------------------------//
//	カーソルハンドル																							//
//--------------------------------------------------------------------------------------------------------------//
static	HCURSOR	hARROW;
static	HCURSOR	hIBEAM;
static	HCURSOR	hAPPSTARTING;
static	HCURSOR	hCROSS;
static	HCURSOR	hHAND;
static	HCURSOR	hHELP;
static	HCURSOR	hNO;
static	HCURSOR	hSIZEALL;
static	HCURSOR	hSIZENESW;
static	HCURSOR	hSIZENS;
static	HCURSOR	hSIZENWSE;
static	HCURSOR	hSIZEWE;
static	HCURSOR	hUPARROW;
static	HCURSOR	hWAIT;

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC_DEF(TipCtl		);
AJC_WNDPROC_DEF(SubClass	);

static	VO		ButtonDown(HWND hwnd);
static	VO		SubSetInstWork(HWND hCtrl, C_WCP pTxt, int msDelay, int msShow, HFONT hFont, COLORREF TextColor, COLORREF BkColor, COLORREF BorderColor, BOOL fUni);
static	HWND	GetTipCtrlWnd(HWND hwnd);
static	HWND	GetTipCtrlOrg(HWND hwnd);
static	VO		SetTipText(PAVLNODE pNode, C_WCP pTxt);
static	VO		SetTipTextByCallBack(PAVLNODE pNode);
static	BOOL	ShowTipText(PAVLNODE pNode, int x, int y);
static	VO		MoveTipText(PAVLNODE pNode, int x, int y);
static	VO		HideTipText(BOOL fDelay);
static	UI		GetInterval(UI SrtTime);
static	int		GetCursorOffset(VO);
static	UI		AdjustMsDelay(int msDelay);
static	UI		AdjustMsShow (int msShow);

//==============================================================================================================//
//	起動時初期設定																								//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- ＡＶＬノード消去用コールバック -------------------------------------------------------------------------//
static VO CALLBACK cbAvlRemove(UX key, VOP pNodeData, UI len, UI nest, UX cbp)
{
	PAVLNODE	pNode = (PAVLNODE)pNodeData;

	//	チップテキスト解放
	if (pNode->pText != NULL) {
		free(pNode->pText);
		pNode->pText = NULL;
	}
}

//--------------------------------------------------------------------------------------------------------------//
BOOL	AjcTipCtlInit(VO)
{
	BOOL		rc = FALSE;
	WNDCLASS	wndclass;

	do {
		//----- カーソルハンドル設定 --------------------------//
		hARROW			= LoadCursor(NULL, IDC_ARROW		);
		hIBEAM			= LoadCursor(NULL, IDC_IBEAM		);
		hAPPSTARTING	= LoadCursor(NULL, IDC_APPSTARTING	);
		hCROSS			= LoadCursor(NULL, IDC_CROSS		);
		hHAND			= LoadCursor(NULL, IDC_HAND			);
		hHELP			= LoadCursor(NULL, IDC_HELP			);
		hNO				= LoadCursor(NULL, IDC_NO			);
		hSIZEALL		= LoadCursor(NULL, IDC_SIZEALL		);
		hSIZENESW		= LoadCursor(NULL, IDC_SIZENESW		);
		hSIZENS			= LoadCursor(NULL, IDC_SIZENS		);
		hSIZENWSE		= LoadCursor(NULL, IDC_SIZENWSE		);
		hSIZEWE			= LoadCursor(NULL, IDC_SIZEWE		);
		hUPARROW		= LoadCursor(NULL, IDC_UPARROW		);
		hWAIT			= LoadCursor(NULL, IDC_WAIT			);

		//----- ＡＶＬインスタンス生成 ------------------------//
		hAvlNode = AjcAvlCreate(0, NULL, cbAvlRemove);
		hAvlSubC = AjcAvlCreate(0, NULL, NULL);

		//----- チップテキスト表示ウインドクラス --------------//
		wndclass.style			= CS_GLOBALCLASS;
		wndclass.lpfnWndProc	= AJC_WNDPROC_NAME(TipCtl);
		wndclass.cbClsExtra		= 0;
		wndclass.cbWndExtra		= sizeof(UX);
		wndclass.hInstance		= hDllInst;
		wndclass.hIcon			= NULL;
		wndclass.hCursor		= LoadCursor(NULL, IDC_ARROW);
		wndclass.hbrBackground	= (HBRUSH)GetStockObject(WHITE_BRUSH);
		wndclass.lpszMenuName	= NULL;
		wndclass.lpszClassName	= MY_CLASS;
		if ((ClassTipCtl = RegisterClass(&wndclass)) == 0) break;

		//----- チップコントロール生成 -----------------------//
		hTipCtl = CreateWindowEx (WS_EX_TOOLWINDOW,								// extended style
								  MY_CLASS,										// window class name
								  L"",											// window caption
								  WS_POPUP,										// window style
								  0,											// initial x position
								  0,											// initial y position
								  0,											// initial x size
								  0,											// initial y size
								  NULL,											// parent window handle
								  NULL,											// window menu handle
								  hDllInst,										// program instance handle
								  NULL);										// creation parameters

		//	ウインド生成失敗ならば終了
		if (hTipCtl == NULL) break;

		//----- ウインド非表示 --------------------------------//
		ShowWindow(hTipCtl, SW_HIDE);

		rc = TRUE;

	} while(0);

	if (rc == FALSE) {
		AjcTipCtlEnd();
	}

	return rc;
}
//==============================================================================================================//
//	終了時後処理																								//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
VO		AjcTipCtlEnd (VO)
{
	if (hTipCtl 	!= NULL) {DestroyWindow(hTipCtl);					   hTipCtl		= NULL; }
	if (ClassTipCtl != 0   ) {UnregisterClass((WCP)ClassTipCtl, hDllInst); ClassTipCtl	= 0;	}
	if (hAvlNode 	!= NULL) {AjcAvlDelete(hAvlNode);					   hAvlNode		= NULL; }
	if (hAvlSubC 	!= NULL) {AjcAvlDelete(hAvlSubC);					   hAvlSubC		= NULL; }
	if (fMultiThread	   ) {DeleteCriticalSection(&CritSec);			   fMultiThread = FALSE;}
}
//==============================================================================================================//
//	マルチスレッドの許可／禁止																					//
//																												//
//	引　数	：	fEnable - FALSE : マルチスレッド禁止															//
//						  TRUE	: マルチスレッド許可															//
//																												//
//	戻り値	：	前回の禁止／許可状態																			//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTipTextEnableMultiThread(BOOL fEnable)
{
	BOOL	rc = fMultiThread;

	if (fEnable) {
		if (!fMultiThread) {
			InitializeCriticalSection(&CritSec);
			fMultiThread = TRUE;
		}
	}
	else {
		if (fMultiThread) {
			fMultiThread = FALSE;
			DeleteCriticalSection(&CritSec);
		}
	}
	return rc;
}
//----- マルチスレッド ブロック --------------------------------------------------------------------------------//
VO	AjcTipTextEnter(VO)
{
	if (fMultiThread) {
		EnterCriticalSection(&CritSec);
	}
}
//----- マルチスレッド ブロック解除 ----------------------------------------------------------------------------//
VO	AjcTipTextLeave(VO)
{
	if (fMultiThread) {
		LeaveCriticalSection(&CritSec);
	}
}
//==============================================================================================================//
//	チップテキストの表示位置モード設定																			//
//																												//
//	引　数：	hCtrl		- チップテキストを表示するコントロールのウインドハンドル							//
//				PosMode		- チップテキストの表示位置モード													//
//																												//
//	戻り値：	前回のチップテキストの表示位置モード															//
//==============================================================================================================//
AJCEXPORT AJCTIP_POSMODE WINAPI AjcTipTextSetTipPosMode (AJCTIP_POSMODE PosMode)
{
	AJCTIP_POSMODE	rc = TipPosMode;

	if (PosMode == AJCTIP_POS_UNDER_CONTROL ||
		PosMode == AJCTIP_POS_UNDER_CURSOR) {
		TipPosMode = PosMode;
	}
	return rc;
}
//==============================================================================================================//
//	チップテキストの追加																						//
//																												//
//	引　数：	hCtrl		- チップテキストを表示するコントロールのウインドハンドル							//
//				pTxt		- 表示テキスト																		//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextAddA(HWND hCtrl, C_BCP pTxt)
{
	return AjcTipTextAddExA(hCtrl, pTxt, -1, -1, NULL, -1, -1, -1);
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextAddW(HWND hCtrl, C_WCP pTxt)
{
	return AjcTipTextAddExW(hCtrl, pTxt, -1, -1, NULL, -1, -1, -1);
}

//==============================================================================================================//
//	チップテキストの追加（拡張版）																				//
//																												//
//	引　数：	hCtrl		- チップテキストを表示するコントロールのウインドハンドル							//
//				pTxt		- 表示テキスト																		//
//				msDelay		- 表示遅延時間[ms]（－１：デフォルト値）											//
//				msShow		- 表示時間	  [ms]（－１：デフォルト値）											//
//				hFont		- フォントハンドル（NULL:デフォルトフォント）										//
//				TextColor	- 文字色（－１：黒）																//
//				BkColor		- 背景色（－１：クリーム色）														//
//				BorderColor	- 外枠色（－１：黒，－２：外枠非表示）												//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextAddExA(HWND hCtrl, C_BCP pTxt, int msDelay, int msShow, HFONT hFont, COLORREF TextColor, COLORREF BkColor, COLORREF BorderColor)
{
	BOOL	rc = FALSE;
	UI		len  = 0;
	WCP		pTmp = NULL;

	AjcTipTextEnter();

	if (IsWindow(hCtrl)) {
		hCtrl = GetTipCtrlWnd(hCtrl);
		if (pTxt != NULL) {
			len = MultiByteToWideChar(CP_ACP, 0, pTxt, -1, NULL, 0);
			if (len != 0 && (pTmp = (WCP)AJCMEM(len * 2))) {
				MultiByteToWideChar(CP_ACP, 0, pTxt, -1, pTmp, len);
				SubSetInstWork(hCtrl, pTmp, msDelay, msShow, hFont, TextColor, BkColor, BorderColor, FALSE);
				rc = TRUE;
			}
			if (pTmp != NULL) free(pTmp);
		}
		else {
			SubSetInstWork(hCtrl, NULL, msDelay, msShow, hFont, TextColor, BkColor, BorderColor, FALSE);
			rc = TRUE;
		}
	}

	AjcTipTextLeave();

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextAddExW(HWND hCtrl, C_WCP pTxt, int msDelay, int msShow, HFONT hFont, COLORREF TextColor, COLORREF BkColor, COLORREF BorderColor)
{
	BOOL			rc	  = FALSE;
	PAVLNODE		pNode = NULL;

	AjcTipTextEnter();

	if (IsWindow(hCtrl)) {
		hCtrl = GetTipCtrlWnd(hCtrl);
		SubSetInstWork(hCtrl, pTxt, msDelay, msShow, hFont, TextColor, BkColor, BorderColor, TRUE);
		rc = TRUE;
	}

	AjcTipTextLeave();

	return rc;
}
//----- インスタンスワーク設定 ---------------------------------------------------------------------------------//
static VO SubSetInstWork(HWND hCtrl, C_WCP pTxt, int msDelay, int msShow, HFONT hFont, COLORREF TextColor, COLORREF BkColor, COLORREF BorderColor, BOOL fUni)
{
	PAVLNODE	pNode;

	//	拡張コンボボックスならば、長テキストのチップ表示を禁止する
	if (GetProp(hCtrl, PN_SBCCBO) != NULL) {
		AjcSbcEnableLongTip(hCtrl, FALSE);
	}

	//	インスタンスワーク未登録ならば、インスタンスワーク登録とサブクラス化
	if ((pNode = (PAVLNODE)AjcAvlGetNodePtr(hAvlNode, (UX)hCtrl, NULL)) == NULL) {
		AVLNODE		node;
		PAVLSUBCLS	pSubC;
		memset(&node, 0, sizeof node);
		node.hwnd			= hCtrl;
		node.fUnicode		= fUni;
		node.fEnable		= TRUE;
		node.hFont			= hFont;
		SetTipText(&node, pTxt);
		node.TextColor		= TextColor;
		node.BkColor		= BkColor;
		node.BorderColor	= BorderColor;
		node.fShowAlways	= TRUE;
		node.msDelay		= msDelay;
		node.msShow			= msShow;
		//-- コールバックは無効とする --//
		node.cbp			= 0;
		node.u.vop			= NULL;
		node.cbChkCursor	= NULL;
		//	サブクラス化
		MAjcMmpSetSubclass(SubClass, hCtrl);
		//	ノード登録
		AjcAvlInsNode(hAvlNode, (UX)hCtrl, &node, sizeof node);
		//	サブクラス化通知情報が設定されていれば、サブクラス化(TRUE)を通知
		if ((pSubC = AjcAvlGetNodePtr(hAvlSubC, (UX)hCtrl, NULL)) != NULL) {
			if (IsWindow(pSubC->hWndNtc)) {
				SendMessage(pSubC->hWndNtc, pSubC->msg, TRUE, pSubC->lParam);
			}
		}
	}
	//	インスタンスワーク登録済みならば、インスタンスワーク書き換え
	else {
		pNode->hwnd			= hCtrl;
		pNode->fUnicode		= fUni;
		pNode->hFont		= hFont;
		SetTipText(pNode, pTxt);
		pNode->TextColor	= TextColor;
		pNode->BkColor		= BkColor;
		pNode->BorderColor	= BorderColor;
		pNode->msDelay		= msDelay;
		pNode->msShow		= msShow;
		pNode->stDelay		= 0;
		pNode->stShow		= 0;
		//-- コールバックは設定済み内容のままとする --//
	}
}
//==============================================================================================================//
//	チップテキストの追加（書式化）																				//
//																												//
//	引　数：	hCtrl		- チップテキストを表示するコントロールのウインドハンドル							//
//				pFmt		- 書式テキスト																		//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextAddFA(HWND hCtrl, C_BCP pFmt, ...)
{
	BOOL		rc = FALSE;
	va_list 	vls;
	BC			buf[2048];

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		AjcVSnPrintFA(buf, 2048, pFmt, vls);
		buf[2047] = 0;
		va_end	(vls);

		rc = AjcTipTextAddA(hCtrl, buf);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextAddFW(HWND hCtrl, C_WCP pFmt, ...)
{
	BOOL		rc = FALSE;
	va_list 	vls;
	WC			buf[2048];

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		AjcVSnPrintFW(buf, 2048, pFmt, vls);
		buf[2047] = 0;
		va_end	(vls);

		rc = AjcTipTextAddW(hCtrl, buf);
	}
	return rc;
}

//==============================================================================================================//
//	チップテキストの追加（拡張版，書式化）																		//
//																												//
//	引　数：	hCtrl		- チップテキストを表示するコントロールのウインドハンドル							//
//				pTxt		- 表示テキスト																		//
//				msDelay		- 表示遅延時間[ms]（－１：デフォルト値）											//
//				msShow		- 表示時間	  [ms]（－１：デフォルト値）											//
//				hFont		- フォントハンドル（NULL:デフォルトフォント）										//
//				TextColor	- 文字色（－１：黒）																//
//				BkColor		- 背景色（－１：クリーム色）														//
//				BorderColor	- 外枠色（－１：黒，－２：外枠非表示）												//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextAddFExA(HWND hCtrl, int msDelay, int msShow, HFONT hFont, COLORREF TextColor, COLORREF BackGround, COLORREF BorderColor, C_BCP pFmt, ...)
{
	BOOL		rc = FALSE;
	va_list 	vls;
	BC			buf[2048];

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		AjcVSnPrintFA(buf, 2048, pFmt, vls);
		buf[2047] = 0;
		va_end	(vls);

		rc = AjcTipTextAddExA(hCtrl, buf, msDelay, msShow, hFont, TextColor, BackGround, BorderColor);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextAddFExW(HWND hCtrl, int msDelay, int msShow, HFONT hFont, COLORREF TextColor, COLORREF BackGround, COLORREF BorderColor, C_WCP pFmt, ...)
{
	BOOL		rc = FALSE;
	va_list 	vls;
	WC			buf[2048];

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		AjcVSnPrintFW(buf, 2048, pFmt, vls);
		buf[2047] = 0;
		va_end	(vls);

		rc = AjcTipTextAddExW(hCtrl, buf, msDelay, msShow, hFont, TextColor, BackGround, BorderColor);
	}
	return rc;
}

//==============================================================================================================//
//	ツールチップの表示条件設定																					//
//																												//
//	引　数：	hCtrl			- チップテキストを表示するコントロールのウインドハンドル						//
//				fShowAlways		- TRUE	: 非アクティブな状態時でもツールチップを表示する						//
//								  FALSE : 非アクティブな状態時はツールチップを表示しない						//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//==============================================================================================================//
AJCEXPORT BOOL	 WINAPI AjcTipTextSetShowAlways(HWND hCtrl, BOOL fShowAlways)
{
	BOOL		rc = FALSE;
	PAVLNODE	pNode;

	AjcTipTextEnter();

	if (IsWindow(hCtrl)) {
		hCtrl = GetTipCtrlWnd(hCtrl);
		if ((pNode = (PAVLNODE)AjcAvlGetNodePtr(hAvlNode, (UX)hCtrl, NULL)) != NULL) {
			pNode->fShowAlways = fShowAlways;
			rc = TRUE;
		}
	}

	AjcTipTextLeave();

	return rc;
}
//==============================================================================================================//
//	ツールチップの表示条件取得																					//
//																												//
//	引　数：	hCtrl			- チップテキストを表示するコントロールのウインドハンドル						//
//																												//
//	戻り値：	ツールチップの表示条件																			//
//==============================================================================================================//
AJCEXPORT BOOL	 WINAPI AjcTipTextGetShowAlways(HWND hCtrl)
{
	BOOL		rc = FALSE;
	PAVLNODE	pNode;

	AjcTipTextEnter();

	if (IsWindow(hCtrl)) {
		hCtrl = GetTipCtrlWnd(hCtrl);
		if ((pNode = (PAVLNODE)AjcAvlGetNodePtr(hAvlNode, (UX)hCtrl, NULL)) != NULL) {
			rc = pNode->fShowAlways;
		}
	}

	AjcTipTextLeave();

	return rc;
}
//==============================================================================================================//
//	全てアクティブ時のみ表示 設定																				//
//																												//
//	引　数：	fShowForActive	- 	TRUE	: 全てアクティブな状態時のみ表示									//
//									FALSE 	: 個々の設定による表示												//
//																												//
//	戻り値：	なし																							//
//==============================================================================================================//
AJCEXPORT VO	 WINAPI AjcTipTextSetShowForActive(BOOL fShowForActive)
{
	AjcTipTextEnter();

	FlgShowForActive = fShowForActive;

	AjcTipTextLeave();
}
//==============================================================================================================//
//	全てアクティブ時のみ表示 取得																				//
//																												//
//	引　数：	なし																							//
//																												//
//	戻り値：	TRUE	: 全てアクティブな状態時のみ表示														//
//				FALSE 	: 個々の設定による表示																	//
//==============================================================================================================//
AJCEXPORT BOOL	 WINAPI AjcTipTextGetShowForActive(VO)
{
	BOOL	rc;

	AjcTipTextEnter();

	rc = FlgShowForActive;

	AjcTipTextLeave();

	return rc;
}
//==============================================================================================================//
//	コールバックの設定（作業バッファ使用）																		//
//																												//
//	引　数：	hCtrl			- チップテキストを表示するコントロールのウインドハンドル						//
//				cbChkMousePoint	- マウスポイントのチェック用コールバック関数（不要時はNULL)						//
//				cbGetText		- チップテキスト取得用コールバック関数		（不要時はNULL）					//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextSetCallBackA(HWND hCtrl, UX cbp, BOOL (CALLBACK *cbChkCursor)(HWND hCtrl, LPPOINT ptClient, UX cbp), C_BCP (CALLBACK *cbGetText)(HWND hCtrl, BCP pBuf, UI lBuf, UX cbp))
{
	BOOL		rc = FALSE;
	PAVLNODE	pNode;

	AjcTipTextEnter();

	if (IsWindow(hCtrl)) {
		hCtrl = GetTipCtrlWnd(hCtrl);
		if ((pNode = (PAVLNODE)AjcAvlGetNodePtr(hAvlNode, (UX)hCtrl, NULL)) != NULL) {
			pNode->fCBWithBuf = TRUE;	//	作業バッファ使用
			pNode->cbChkCursor	= cbChkCursor;
			pNode->cbp			= cbp;
			pNode->u.cbGetTextA = cbGetText;
			rc = TRUE;
		}
	}

	AjcTipTextLeave();

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextSetCallBackW(HWND hCtrl, UX cbp, BOOL (CALLBACK *cbChkCursor)(HWND hCtrl, LPPOINT ptClient, UX cbp), C_WCP (CALLBACK *cbGetText)(HWND hCtrl, WCP pBuf, UI lBuf, UX cbp))
{
	BOOL		rc = FALSE;
	PAVLNODE	pNode;

	AjcTipTextEnter();

	if (IsWindow(hCtrl)) {
		hCtrl = GetTipCtrlWnd(hCtrl);
		if ((pNode = (PAVLNODE)AjcAvlGetNodePtr(hAvlNode, (UX)hCtrl, NULL)) != NULL) {
			pNode->fCBWithBuf = TRUE;	//	作業バッファ使用
			pNode->cbChkCursor	= cbChkCursor;
			pNode->cbp			= cbp;
			pNode->u.cbGetTextW = cbGetText;
			rc = TRUE;
		}
	}

	AjcTipTextLeave();

	return rc;
}

//==============================================================================================================//
//	コールバックの設定（作業バッファ未使用）																	//
//																												//
//	引　数：	hCtrl			- チップテキストを表示するコントロールのウインドハンドル						//
//				cbChkMousePoint	- マウスポイントのチェック用コールバック関数（不要時はNULL)						//
//				cbGetText		- チップテキスト取得用コールバック関数		（不要時はNULL）					//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextSetCallBackNoBufA(HWND hCtrl, UX cbp, BOOL (CALLBACK *cbChkCursor)(HWND hCtrl, LPPOINT ptClient, UX cbp), C_BCP (CALLBACK *cbGetText)(HWND hCtrl, UX cbp))
{
	BOOL		rc = FALSE;
	PAVLNODE	pNode;

	AjcTipTextEnter();

	if (IsWindow(hCtrl)) {
		hCtrl = GetTipCtrlWnd(hCtrl);
		if ((pNode = (PAVLNODE)AjcAvlGetNodePtr(hAvlNode, (UX)hCtrl, NULL)) != NULL) {
			pNode->fCBWithBuf = FALSE;	//	作業バッファ未使用
			pNode->cbChkCursor	 = cbChkCursor;
			pNode->u.cbGetNoBufA = cbGetText;
			rc = TRUE;
		}
	}

	AjcTipTextLeave();

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextSetCallBackNoBufW(HWND hCtrl, UX cbp, BOOL (CALLBACK *cbChkCursor)(HWND hCtrl, LPPOINT ptClient, UX cbp), C_WCP (CALLBACK *cbGetText)(HWND hCtrl, UX cbp))
{
	BOOL		rc = FALSE;
	PAVLNODE	pNode;

	AjcTipTextEnter();

	if (IsWindow(hCtrl)) {
		hCtrl = GetTipCtrlWnd(hCtrl);
		if ((pNode = (PAVLNODE)AjcAvlGetNodePtr(hAvlNode, (UX)hCtrl, NULL)) != NULL) {
			pNode->fCBWithBuf = FALSE;	//	作業バッファ未使用
			pNode->cbChkCursor	 = cbChkCursor;
			pNode->u.cbGetNoBufW = cbGetText;
			rc = TRUE;
		}
	}

	AjcTipTextLeave();

	return rc;
}

//==============================================================================================================//
//	チップテキストの表示許可／禁止 設定																			//
//																												//
//	引　数：	hCtrl		- コントロールのウインドハンドル													//
//				fEnable		- 許可フラグ（TRUE:許可，FALSE:禁止）												//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//==============================================================================================================//
AJCEXPORT BOOL	 WINAPI AjcTipTextEnable(HWND hCtrl, BOOL fEnable)
{
	BOOL		rc = FALSE;
	PAVLNODE	pNode = NULL;

	AjcTipTextEnter();

	if (IsWindow(hCtrl)) {
		hCtrl = GetTipCtrlWnd(hCtrl);
		if ((pNode = (PAVLNODE)AjcAvlGetNodePtr(hAvlNode, (UX)hCtrl, NULL)) != NULL) {
			if (fEnable != 0) {			//	許可
				pNode->fEnable = TRUE;
			}
			else {						//	禁止
				pNode->fEnable = FALSE;
				//	現在処理中のコントロールならば、チップテキスト非表示
				if (hCtrl == hCurCtl) {
					//	現チップテキスト非表示
					AjcTipTextHide();
					//	現処理中のコントロール無しとする
					hCurCtl  = NULL;
					pCurNode = NULL;
					//	→アイドルモード
					pst = PST_IDLE;
				}
			}
			rc = TRUE;
		}
	}

	AjcTipTextLeave();

	return rc;
}
//==============================================================================================================//
//	チップテキストの表示許可／禁止状態取得																		//
//																												//
//	引　数：	hCtrl		- コントロールのウインドハンドル													//
//																												//
//	戻り値：	TRUE	- 表示許可状態																			//
//				FALSE	- 表示禁止状態																			//
//==============================================================================================================//
AJCEXPORT BOOL	 WINAPI AjcTipTextGetEnableState(HWND hCtrl)
{
	BOOL		rc = FALSE;
	PAVLNODE	pNode = NULL;

	AjcTipTextEnter();

	if (IsWindow(hCtrl)) {
		hCtrl = GetTipCtrlWnd(hCtrl);
		if ((pNode = (PAVLNODE)AjcAvlGetNodePtr(hAvlNode, (UX)hCtrl, NULL)) != NULL) {
			rc = pNode->fEnable;
		}
	}

	AjcTipTextLeave();

	return rc;
}
//==============================================================================================================//
//	全てのチップテキストの表示許可／禁止設定																	//
//																												//
//	引　数：	fEnable		- 表示許可／禁止フラグ（TRUE:表示，FALSE:非表示）									//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//==============================================================================================================//
AJCEXPORT VO	 WINAPI AjcTipTextEnableAll(BOOL fEnable)
{
	AjcTipTextEnter();

	TipAllState = (fEnable != 0) ? TIPALL_ENABLE : TIPALL_DISABLE;

	if (TipAllState == TIPALL_DISABLE) {
		//	現チップテキスト非表示
		AjcTipTextHide();
		//	現処理中のコントロール無しとする
		hCurCtl  = NULL;
		pCurNode = NULL;
		//	→アイドルモード
		pst = PST_IDLE;
	}

	AjcTipTextLeave();

}
//==============================================================================================================//
//	全てのチップテキストの表示許可／禁止状態取得																//
//																												//
//	引　数：	なし																							//
//																												//
//	戻り値：	全てのチップテキストの表示許可／禁止状（TRUE:許可，FALSE:禁止）									//
//==============================================================================================================//
AJCEXPORT BOOL	 WINAPI AjcTipTextGetEnableAllState(VO)
{
	BOOL	rc;

	AjcTipTextEnter();
	rc = (TipAllState == TIPALL_ENABLE);
	AjcTipTextLeave();

	return rc;
}
//==============================================================================================================//
//	チップテキストの関連付け消去																				//
//																												//
//	引　数：	hCtrl		- チップテキストを表示するコントロールのウインドハンドル							//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//==============================================================================================================//
AJCEXPORT BOOL	 WINAPI AjcTipTextRemove(HWND hCtrl)
{
	BOOL		rc = FALSE;
	PAVLNODE	pNode = NULL;
	PAVLSUBCLS	pSubC;

	AjcTipTextEnter();

	if (IsWindow(hCtrl)) {
		hCtrl = GetTipCtrlWnd(hCtrl);
		if ((pNode = (PAVLNODE)AjcAvlGetNodePtr(hAvlNode, (UX)hCtrl, NULL)) != NULL) {
			//	現在処理中のコントロールならば、チップテキスト非表示
			if (hCtrl == hCurCtl) {
				//	現チップテキスト非表示
				AjcTipTextHide();
				//	現処理中のコントロール無しとする
				hCurCtl  = NULL;
				pCurNode = NULL;
				//	→アイドルモード
				pst = PST_IDLE;
			}
			//	サブクラス解除
			MAjcMmpClrSubclass(SubClass, hCtrl);
			//	サブクラス化通知情報が設定されていれば、サブクラス解除(FALSE)を通知
			if ((pSubC = AjcAvlGetNodePtr(hAvlSubC, (UX)hCtrl, NULL)) != NULL) {
				if (IsWindow(pSubC->hWndNtc)) {
					SendMessage(pSubC->hWndNtc, pSubC->msg, FALSE, pSubC->lParam);
				}
			}
			//	当該コントロールのＡＶＬノード削除
			AjcAvlDelNode(hAvlNode, (UX)hCtrl);

			rc = TRUE;
		}
	}

	AjcTipTextLeave();

	return rc;
}
//==============================================================================================================//
//	全てのチップテキストの関連付け消去																			//
//																												//
//	引　数：	なし																							//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//==============================================================================================================//
AJCEXPORT BOOL	 WINAPI AjcTipTextRemoveAll(VO)
{
	BOOL		rc = FALSE;
	AVLNODE		node = {0};
	PAVLSUBCLS	pSubC;
	UX			key;
	HWND		hCtrl;

	AjcTipTextEnter();

	//	チップテキスト非表示
	AjcTipTextHide();
	//	現処理中のコントロール無しとする
	hCurCtl  = NULL;
	pCurNode = NULL;
	//	→アイドルモード
	pst = PST_IDLE;

	while (AjcAvlGetTopNode(hAvlNode, &key, &node, sizeof node) != -1) {
		hCtrl = GetTipCtrlWnd(node.hwnd);
		//	サブクラス解除
		MAjcMmpClrSubclass(SubClass, hCtrl);
		//	サブクラス化通知情報が設定されていれば、サブクラス解除(FALSE)を通知
		if ((pSubC = AjcAvlGetNodePtr(hAvlSubC, (UX)hCtrl, NULL)) != NULL) {
			if (IsWindow(pSubC->hWndNtc)) {
				SendMessage(pSubC->hWndNtc, pSubC->msg, FALSE, pSubC->lParam);
			}
		}
		//	当該コントロールのＡＶＬノード削除
		AjcAvlDelNode(hAvlNode, key);
	}
	rc = TRUE;

	AjcTipTextLeave();

	return rc;
}
//==============================================================================================================//
//	チップテキスト設定情報取得																					//
//																												//
//	引　数：	hCtrl		- コントロールのウインドハンドル													//
//				pInfoBuf	- チップテキスト設定情報を各野するバッファのアドレス（不要時はNULL)					//
//				pTxtBuf		- ツールチップテキストを格納するバッファのアドレス（不要時はNULL)					//
//				lTxtBuf		- ツールチップテキストを格納するバッファのバイト数／文字数							//
//																												//
//	戻り値：	≠-1	- 成功（テキストのバイト数／文字数）													//
//				＝-1	- 失敗																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT UI	 WINAPI AjcTipTextGetInfoA(HWND hCtrl, PAJCTIPINFO pInfoBuf, BCP pTxtBuf, UI lTxtBuf)
{
	UI			rc = -1;
	PAVLNODE	pNode;

	AjcTipTextEnter();

	if (IsWindow(hCtrl)) {
		hCtrl = GetTipCtrlWnd(hCtrl);
		if ((pNode = (PAVLNODE)AjcAvlGetNodePtr(hAvlNode, (UX)hCtrl, NULL)) != NULL) {
			if (pInfoBuf != NULL) {
				pInfoBuf->hCtrl			= pNode->hwnd;
				pInfoBuf->msDelay		= AdjustMsDelay(pNode->msDelay);
				pInfoBuf->msShow		= AdjustMsShow (pNode->msShow );
				pInfoBuf->hFont			= pNode->hFont;
				pInfoBuf->TextColor		= pNode->TextColor;
				pInfoBuf->BackGround	= pNode->BkColor;
				pInfoBuf->BorderColor	= pNode->BorderColor;
				pInfoBuf->WindowTime	= msMove;
				pInfoBuf->DefMsDelay	= DefMsDelay;
				pInfoBuf->hDefFont		= AjcTipTextGetDefFont();
				pInfoBuf->DefTextColor	= AjcTipTextGetDefTextColor();
				pInfoBuf->DefBackGround	= AjcTipTextGetDefBkColor();
				pInfoBuf->DefBorderColor= AjcTipTextGetDefBorderColor();
				pInfoBuf->DefMsShow		= AjcTipTextGetDefMsShow();
			}
			if (pNode->pText != NULL) {
				rc = WideCharToMultiByte(CP_ACP, 0, pNode->pText, -1, NULL, 0, NULL, NULL);
				if (pTxtBuf != NULL && lTxtBuf != 0) {
					memset(pTxtBuf, 0, lTxtBuf);
					WideCharToMultiByte(CP_ACP, 0, pNode->pText, -1, pTxtBuf, lTxtBuf, NULL, NULL);
				}
			}
		}
	}
	AjcTipTextLeave();

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT UI	 WINAPI AjcTipTextGetInfoW(HWND hCtrl, PAJCTIPINFO pInfoBuf, WCP pTxtBuf, UI lTxtBuf)
{
	UI			rc = -1;
	PAVLNODE	pNode;

	AjcTipTextEnter();

	if (IsWindow(hCtrl)) {
		hCtrl = GetTipCtrlWnd(hCtrl);
		if ((pNode = (PAVLNODE)AjcAvlGetNodePtr(hAvlNode, (UX)hCtrl, NULL)) != NULL) {
			if (pInfoBuf != NULL) {
				pInfoBuf->hCtrl			= pNode->hwnd;
				pInfoBuf->msDelay		= AdjustMsDelay(pNode->msDelay);
				pInfoBuf->msShow		= AdjustMsShow (pNode->msShow );
				pInfoBuf->hFont			= pNode->hFont;
				pInfoBuf->TextColor		= pNode->TextColor;
				pInfoBuf->BackGround	= pNode->BkColor;
				pInfoBuf->BorderColor	= pNode->BorderColor;
				pInfoBuf->WindowTime	= msMove;
				pInfoBuf->DefMsDelay	= DefMsDelay;
				pInfoBuf->hDefFont		= AjcTipTextGetDefFont();
				pInfoBuf->DefTextColor	= AjcTipTextGetDefTextColor();
				pInfoBuf->DefBackGround	= AjcTipTextGetDefBkColor();
				pInfoBuf->DefBorderColor= AjcTipTextGetDefBorderColor();
				pInfoBuf->DefMsShow		= AjcTipTextGetDefMsShow();
			}
			if (pNode->pText != NULL) {
				rc = (UI)wcslen(pNode->pText) + 1;
				if (pTxtBuf != NULL && lTxtBuf != 0) {
					memset(pTxtBuf, 0, lTxtBuf * 2);
					wcsncpy(pTxtBuf, pNode->pText, lTxtBuf - 1);
				}
			}
		}
	}
	AjcTipTextLeave();

	return rc;
}
//==============================================================================================================//
//	サブクラス化の通知情報設定																					//
//																												//
//	引　数：	hCtrl		- コントロールのウインドハンドル													//
//				hNtc		- 通知先ウインドハンドル															//
//				msg			- 通知メッセージコード																//
//				lParam		- 通知パラメタ																		//
//																												//
//	戻り値：	TRUE  - 成功																					//
//				FALSE - 失敗																					//
//==============================================================================================================//
AJCEXPORT BOOL	 WINAPI AjcTipTextSetNtcSubclass(HWND hCtrl, HWND hNtc, UI msg, UX lParam)
{
	BOOL		rc = FALSE;
	AVLSUBCLS	node;

	AjcTipTextEnter();	//	スレッド排他制御開始

	if (IsWindow(hCtrl)) {
		hCtrl = GetTipCtrlWnd(hCtrl);
		node.hWndNtc = hNtc;
		node.msg	 = msg;
		node.lParam  = lParam;
		rc = AjcAvlInsOrRepNode(hAvlSubC, (UX)hCtrl, &node, sizeof node);
	}

	AjcTipTextLeave();	//	スレッド排他制御終了

	return rc;
}
//==============================================================================================================//
//	サブクラス化の通知情報消去																					//
//																												//
//	引　数：	hCtrl		- コントロールのウインドハンドル													//
//																												//
//	戻り値：	TRUE  - 成功																					//
//				FALSE - 失敗																					//
//==============================================================================================================//
AJCEXPORT BOOL	 WINAPI AjcTipTextClrNtcSubclass(HWND hCtrl)
{
	BOOL	rc = FALSE;

	AjcTipTextEnter();	//	スレッド排他制御開始

	if (IsWindow(hCtrl)) {
		hCtrl = GetTipCtrlWnd(hCtrl);
		rc = AjcAvlDelNode(hAvlSubC, (UX)hCtrl);
	}

	AjcTipTextLeave();	//	スレッド排他制御終了

	return rc;
}
//==============================================================================================================//
//	コントロール間移動猶予時間設定																				//
//																												//
//	引　数：	msTime - コントロール間移動猶予時間[ms]															//
//																												//
//	戻り値：	なし																							//
//==============================================================================================================//
AJCEXPORT VO	 WINAPI AjcTipTextSetWindowTime(int msTime)
{
	AjcTipTextEnter();
	msMove = msTime;
	AjcTipTextLeave();
}
//==============================================================================================================//
//	コントロール間移動猶予時間取得																				//
//																												//
//	引　数：	なし																							//
//																												//
//	戻り値：	コントロール間移動猶予時間[ms]																	//
//==============================================================================================================//
AJCEXPORT int	 WINAPI AjcTipTextGetWindowTime(VO)
{
	int	rc;

	AjcTipTextEnter();
	rc = msMove;
	AjcTipTextLeave();
	return rc;
}
//==============================================================================================================//
//	デフォルト遅延時間の設定																					//
//																												//
//	引　数：	msDelay	- デフォルト遅延時間																	//
//																												//
//	戻り値：	なし																							//
//==============================================================================================================//
AJCEXPORT VO	 WINAPI AjcTipTextSetDefMsDelay(int msDelay)
{
	AjcTipTextEnter();	//	スレッド排他制御開始
	DefMsDelay = msDelay;
	AjcTipTextLeave();	//	スレッド排他制御終了
}
//==============================================================================================================//
//	デフォルト遅延時間の取得																					//
//																												//
//	引　数：	msDelay	- デフォルト遅延時間																	//
//																												//
//	戻り値：	なし																							//
//==============================================================================================================//
AJCEXPORT int	 WINAPI AjcTipTextGetDefMsDelay(VO)
{
	UI	rc;

	AjcTipTextEnter();	//	スレッド排他制御開始
	rc = DefMsDelay;
	AjcTipTextLeave();	//	スレッド排他制御終了
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	チップテキスト・コントロール ウインドプロシージャ															//
//																												//
//--------------------------------------------------------------------------------------------------------------//
//----- ウインド生成 -------------------------------------------------------------------------------------------//
AJC_WNDPROC(TipCtl, WM_CREATE	)
{
	return 0;
}
//----- ウインド破棄 -------------------------------------------------------------------------------------------//
AJC_WNDPROC(TipCtl, WM_DESTROY	)
{
	//	タイマ停止
	KillTimer(hwnd, TID_CYC);
	//	ウインドハンドルクリアー
	hTipCtl = NULL;

	return 0;
}
//----- タイマ -------------------------------------------------------------------------------------------------//
AJC_WNDPROC(TipCtl, WM_TIMER	)
{
	POINT	pt;					//	カーソル位置（スクリーン座標）
	POINT	pc;					//	カーソル位置（クライアント座標）
	RECT	r;					//	クライアント領域矩形（スクリーン座標）

	BOOL	fZOrder	= FALSE;	//	Ｚオーダー上位にチップ制御対象外のウインド有りを示すフラグ
	HWND	hwpt	= NULL;		//	Ｚオーダー上位ウインドのハンドル

	UI		pidWnd = 0;			//	Ｚオーダー上位ウインドのプロセスＩＤ
	UI		pidMe  = 0;			//	自プロセスＩＤ

	AjcTipTextEnter();

	//	自プロセスＩＤ設定
	pidMe = GetCurrentProcessId();

	//	●監視周期タイマ
	if (wParam == TID_CYC) {
		if (hCurCtl != NULL && pCurNode != NULL) {		//	対象コントロールハンドル有効？
			if (IsWindow(hCurCtl)) {					//		対象コントロールは有効なウインド？

				//	マウスカーソル位置取得
				GetCursorPos(&pt);
				//	Ｚオーダー上位のウインド取得
				if (hwpt = WindowFromPoint(pt)) {
					//	カーソル位置のウインドのプロセスＩＤ設定
					GetWindowThreadProcessId(hwpt, &pidWnd);
					//	Ｚオーダー上位のウインド有無を設定（自カスタムコントロールは除く）
					if (pidWnd == pidMe) {
						fZOrder = (hwpt != pCurNode->hwnd && (UX)GetProp(hwpt, AJCTIPFRIEND_PROPNAMEW) != AJCTIPFRIEND_ID);
					}
					else fZOrder = TRUE;
				}
				//	クライアント領域上のマウスカーソル位置設定
				memcpy(&pc, &pt, sizeof pc);
				MapWindowPoints(NULL, hCurCtl, &pc, 1);
				//	ウインド・クライアント領域の矩形取得
				GetClientRect(hCurCtl, &r);
				MapWindowPoints(hCurCtl, NULL, (LPPOINT)&r, 2);

				switch (pst) {
					//	●アイドル状態
					case PST_IDLE:
						//	カーソルがコントロール外に移動したら、カーソル滞在中フラグクリアーしタイマ停止
						if ((pt.x < r.left || pt.x >= r.right || pt.y < r.top || pt.y >= r.bottom) || (fZOrder) ||
							(pCurNode->cbChkCursor != NULL && !pCurNode->cbChkCursor(hCurCtl, &pc, pCurNode->cbp))) {
							pCurNode->fOnMouse = FALSE;									//	フラグクリアー
							hCurCtl  = NULL;											//	処理中のコントロール無し
							pCurNode = NULL;											//	・
							KillTimer(hwnd, TID_CYC);									//	タイマ停止
						}
						break;

					//	●表示開始遅延中
					case PST_DELAY:
						//	プロセスＩＤ取得
						if (!fZOrder && pt.x == svPt.x && pt.y == svPt.y) {				//	チップ制御対象ウインド上でカーソル停止中？
							//	遅延時間満了チェック									//		遅延時間満了？
							if (GetInterval(pCurNode->stDelay) >= AdjustMsDelay(pCurNode->msDelay)) {
								if (ShowTipText(pCurNode, pt.x, pt.y)) {				//			チップテキスト表示ＯＫ？
									pCurNode->stShow  = GetTickCount();					//				表示 開始時刻
									pst				  = PST_SHOW;						//				→表示中
								}
							}
						}
						else {															//	カーソル移動中？
							//	カーソルがコントロール外に移動したら、→アイドル状態
							if ((pt.x < r.left || pt.x >= r.right || pt.y < r.top || pt.y >= r.bottom) || (fZOrder) ||
								(pCurNode->cbChkCursor != NULL && !pCurNode->cbChkCursor(hCurCtl, &pc, pCurNode->cbp))) {
								pst = PST_IDLE;											//		→アイドル状態
							}
							//	カーソルがコントロール内ならば、遅延時間リセット
							else {
								pCurNode->stDelay = GetTickCount();						//		遅延時間リセット
							}
						}
						break;

					//	●表示中
					case PST_SHOW:
						//	カーソルがコントロール外に移動したら、カーソル滞在中フラグクリアー
						if ((pt.x < r.left || pt.x >= r.right || pt.y < r.top || pt.y >= r.bottom) || (fZOrder) ||
										(pCurNode->cbChkCursor != NULL && !pCurNode->cbChkCursor(hCurCtl, &pc, pCurNode->cbp))) {
							pCurNode->fOnMouse = FALSE;
						}
						//	表示時間満了 or カーソルがコントロール外に外れたら、テキスト非表示
						if (GetInterval(pCurNode->stShow) >= AdjustMsShow(pCurNode->msShow) || !pCurNode->fOnMouse) {
							HideTipText(TRUE);											//		チップテキスト非表示（遅延あり）
							stMove	= GetTickCount();									//		コントロール間移動猶予 開始時刻
							pst		= PST_MOVE;											//		→コントロール間移動猶予中
						}
						break;

					//	●コントロール間移動猶予中
					case PST_MOVE:
						if (GetInterval(stMove) >= msMove) {							//	コントロール間移動猶予時間満了？
							pst		= PST_IDLE;											//		→アイドル状態
						}
						break;

				}
				//	マウスカーソル位置退避
				svPt.x = pt.x;
				svPt.y = pt.y;
			}
			else {										//		対象コントロールは無効なウインド？
				//	ＡＶＬからノード削除
				AjcAvlDelNode(hAvlNode, (UX)hCurCtl);
				//	処理中のコントロール無しとする
				hCurCtl  = NULL;
				pCurNode = NULL;
				//	監視タイマ停止
				KillTimer(hwnd, TID_CYC);
				//	→アイドル状態
				pst = PST_IDLE;
			}
		}
	}

	AjcTipTextLeave();

	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDMAP_DEF(TipCtl)
	AJC_WNDMAP_MSG(TipCtl, WM_CREATE	)
	AJC_WNDMAP_MSG(TipCtl, WM_DESTROY	)
	AJC_WNDMAP_MSG(TipCtl, WM_TIMER		)
AJC_WNDMAP_END
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	サブクラス ウインドプロシージャ																				//
//																												//
//--------------------------------------------------------------------------------------------------------------//
//----- ウインド破棄 -------------------------------------------------------------------------------------------//
AJC_WNDPROC(SubClass, WM_DESTROY	)
{
	LRESULT	rc;

	AjcTipTextEnter();

	//	オリジナルウインドプロシージャ呼び出し
	rc = MAjcMmpCallOrgWndProc(SubClass);

	//	チップテキスト消去
	AjcTipTextRemove(hwnd);

	AjcTipTextLeave();

	return rc;
}
//----- マウス移動 ---------------------------------------------------------------------------------------------//
AJC_WNDPROC(SubClass, WM_MOUSEMOVE	)
{
	PAVLNODE	pNode;
	static HWND	SvhWnd = NULL;	//	直前のウインドハンドル

	AjcTipTextEnter();

	//	全チップ表示許可 ＆ マウスボタン離し 状態
	if ((TipAllState == TIPALL_ENABLE) && 
		!((GetKeyState(VK_LBUTTON) & 0x80) || (GetKeyState(VK_RBUTTON) & 0x80) || (GetKeyState(VK_MBUTTON) & 0x80))) {

		//	このコントロールのインスタンスワーク設定
		if (pNode = (PAVLNODE)AjcAvlGetNodePtr(hAvlNode, (UX)hwnd, NULL)) {
			BOOL	fValid = TRUE;
			POINT	pt;

			//	マウス位置設定
			pt.x = (SW)LOWORD(lParam);
			pt.y = (SW)HIWORD(lParam);

			do {
				//	表示禁止状態ならば無効とする
				if (!pNode->fEnable) {
					SvhWnd = NULL;
					break;
				}

				//	マウスポイントが有効かチェック（コールバック）
				if (pNode->cbChkCursor != NULL) {
					if (!pNode->cbChkCursor(hwnd, &pt, pNode->cbp)) {
						SvhWnd = NULL;
						break;
					}
				}

				//	新たなウインド／同一ウインドへ再突入の場合・・・
				if (SvhWnd != hwnd || !pNode->fOnMouse) {
					//	前のウインドに対応するノードの「マウス滞在フラグ」クリアー
					if (SvhWnd != NULL) {
						PAVLNODE	pOldNode;
						if (pOldNode = (PAVLNODE)AjcAvlGetNodePtr(hAvlNode, (UX)SvhWnd, NULL)) {
							pOldNode->fOnMouse = FALSE;
						}
					}
					//	新たなノードのマウス滞在フラグクリアー
					pNode->fOnMouse = FALSE;
					//	処理状態変更
					if		(pst == PST_DELAY) pst = PST_IDLE;	//	表示開始遅延中 → アイドル
					else if (pst == PST_SHOW ) pst = PST_MOVE;	//	表示中		   → コントロール間移動猶予中
					//	処理中のコントロール設定
					hCurCtl  = hwnd;
					pCurNode = pNode;
					//	ウインドハンドル退避
					SvhWnd = hwnd;
				}

				//	マウス位置取得（スクリーン座標）
				GetCursorPos(&pt);
				//	チップテキスト表示開始
				if (!pNode->fOnMouse) {										//	マウスがこのコントロール上に移動してきた？
					pNode->fOnMouse = TRUE;									//		マウスがこのコントロール上にある旨のフラグ設定
					switch (pst) {
						case PST_IDLE:										//		●アイドル状態
							if (AdjustMsDelay(pCurNode->msDelay) == 0) {	//				表示開始遅延時間 ＝ ０？
								if (ShowTipText(pNode, pt.x, pt.y)) {		//					チップテキスト表示ＯＫ？
									pNode->stShow  = GetTickCount();		//						表示開始時刻
									pst			   = PST_SHOW;				//						→表示中
								}
							}
							else {											//				表示開始遅延時間 ≠ ０？
								pNode->stDelay = GetTickCount();			//					遅延待ち開始時刻
								pst			   = PST_DELAY;					//					→表示遅延中
							}
							SetTimer(hTipCtl, TID_CYC, 100, NULL);			//				監視タイマ起動
							break;

						case PST_DELAY:										//		●表示開始遅延中
							break;

						case PST_SHOW:										//		●表示中
							break;

						case PST_MOVE:										//		●コントロール間移動猶予中
							if (ShowTipText(pNode, pt.x, pt.y)) {			//			チップテキスト表示ＯＫ？
								pNode->stShow  = GetTickCount();			//				表示開始時刻
								pst			   = PST_SHOW;					//				→表示中
							}
							break;
					}
				}
				else {														//	マウスがこのコントロール上で移動中？
					switch (pst) {
						case PST_IDLE:										//		●アイドル状態
							break;

						case PST_DELAY:										//		●表示開始遅延中
							pNode->stDelay = GetTickCount();				//			遅延時間リセット
							break;

						case PST_SHOW:										//		●表示中
						{
							//	チップウインド移動
							RECT	rect;
							if (AjcTipTextGetRect(&rect)) {	//	チップ表示中？
								int		rx = pt.x - stPt.x;
								int		ry = pt.y - stPt.y;
								double	len = sqrt((double)((rx * rx) + (ry * ry)));
								if (len > 40.0) {
									//	表示開始から１秒以内ならば、チップテキスト追従
									if (GetInterval(pCurNode->stShow) <= 1000) {
										//	チップテキスト移動
										if (len > 3.0) {
											MoveTipText(pNode, pt.x, pt.y);
											//	チップテキスト追従の為、カーソル位置更新
											stPt.x = pt.x;
											stPt.y = pt.y;
										}
									}
									//	表示開始から１秒を超えた場合は、チップテキストを消す
									else {
										HideTipText(FALSE);				//		チップテキスト非表示（遅延なし）
										stMove	= GetTickCount();		//		コントロール間移動猶予 開始時刻
										pst		= PST_MOVE;				//		→コントロール間移動猶予中
									}
								}
							}
							break;
						}
						case PST_MOVE:										//		●コントロール間移動猶予中
							break;
					}
				}
			} while(0);
		}
	}
	//	全チップ表示禁止 ／ マウスボタン押下 状態
	else {
		SvhWnd = NULL;
	}
	AjcTipTextLeave();

	return MAjcMmpCallOrgWndProc(SubClass);
}
//----- 左ボタン押下 -------------------------------------------------------------------------------------------//
AJC_WNDPROC(SubClass, WM_LBUTTONDOWN	)
{
	AjcTipTextEnter();

	//	チップ非表示
	if (TipAllState == TIPALL_ENABLE) {
		ButtonDown(hwnd);
	}

	AjcTipTextLeave();

	return MAjcMmpCallOrgWndProc(SubClass);
}
//----- 右ボタン押下 -------------------------------------------------------------------------------------------//
AJC_WNDPROC(SubClass, WM_RBUTTONDOWN	)
{
	AjcTipTextEnter();

	//	チップ非表示
	if (TipAllState == TIPALL_ENABLE) {
		ButtonDown(hwnd);
	}

	AjcTipTextLeave();

	return MAjcMmpCallOrgWndProc(SubClass);
}
//----- 中ボタン押下 -------------------------------------------------------------------------------------------//
AJC_WNDPROC(SubClass, WM_MBUTTONDOWN	)
{
	AjcTipTextEnter();

	//	チップ非表示
	if (TipAllState == TIPALL_ENABLE) {
		ButtonDown(hwnd);
	}

	AjcTipTextLeave();

	return MAjcMmpCallOrgWndProc(SubClass);
}
//----- ボタン押下処理（共通サブ）------------------------------------------------------------------------------//
static	VO	ButtonDown(HWND hwnd)
{
	//	現チップテキスト非表示
	AjcTipTextHide();
	//	現処理中のコントロール無しとする
	hCurCtl  = NULL;
	pCurNode = NULL;
	//	→アイドルモード
	pst = PST_IDLE;
}
//----- チップ表示切替通知 -------------------------------------------------------------------------------------//
AJC_WNDPROC(SubClass, WM_AJCTIP_CHANGED	)
{
	AjcTipTextEnter();

	//	→アイドル状態
	pst = PST_IDLE;

	AjcTipTextLeave();

	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDMAP_DEF(SubClass)
	AJC_WNDMAP_MSG(SubClass, WM_DESTROY		)
	AJC_WNDMAP_MSG(SubClass, WM_MOUSEMOVE	)
	AJC_WNDMAP_MSG(SubClass, WM_LBUTTONDOWN	)
	AJC_WNDMAP_MSG(SubClass, WM_RBUTTONDOWN	)
	AJC_WNDMAP_MSG(SubClass, WM_MBUTTONDOWN	)
	AJC_WNDMAP_RWM(SubClass, WM_AJCTIP_CHANGED, MSGSTR_NTC_TIPCHANGED)
AJC_WNDMAP_END



//--------------------------------------------------------------------------------------------------------------//
//																												//
//	チップコントロール対象ウインド取得																			//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	HWND	GetTipCtrlWnd(HWND hwnd)
{
	HWND	rc = hwnd;
	HWND	hw = NULL;

	if (hw = (HWND)GetProp(hwnd, AJCTIPCTRL_PROPNAMEW)) {
		SetProp(hw, AJCTIPCTRL_ORGHWNDPROP, (HANDLE)hwnd);
		rc = hw;
	}
	return rc;
}

//--------------------------------------------------------------------------------------------------------------//
//																												//
//	チップコントロール オリジナルウインド取得																	//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	HWND	GetTipCtrlOrg(HWND hwnd)
{
	HWND	rc = hwnd;
	HWND	hw = NULL;

	if (hw = (HWND)GetProp(hwnd, AJCTIPCTRL_ORGHWNDPROP)) {
		rc = hw;
	}
	return rc;
}

//--------------------------------------------------------------------------------------------------------------//
//																												//
//	チップテキスト設定																							//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static VO	SetTipText(PAVLNODE pNode, C_WCP pTxt)
{
	WC		WndTxt[1024];		//	自ウインドテキスト

	do {
		//	既存テキストと設定テキストが共にNULLならば終了
		if (pNode->pText == NULL && pTxt == NULL								   ) break;
		//	テキストが "@MyWndText@" ならば、自ウインドテキストをセットアップ
		if (pTxt != NULL && wcscmp(pTxt, L"@MyWndText@") == 0) {
			if (GetWindowText(pNode->hwnd, WndTxt, AJCTSIZE(WndTxt)) != 0) {
				WndTxt[AJCTSIZE(WndTxt) - 1] = 0;
				pTxt = WndTxt;
			}
		}
		//	既存テキストと設定テキストが同一ならば終了
		if (pNode->pText != NULL && pTxt != NULL && wcscmp(pNode->pText, pTxt) == 0) break;

		//	テキスト設定済ならば解放
		if (pNode->pText != NULL) {
			free(pNode->pText);
			pNode->pText = NULL;
		}
		//	チップテキスト設定
		if (pTxt != NULL) {
			UI	stl = (UI)wcslen(pTxt);
			if (stl != 0) {
				//	メモリ確保成功ならばテキスト設定
				if (pNode->pText = AJCMEM(stl * 2 + 2)) {
					wcscpy(pNode->pText, pTxt);
				}
			}
		}
	} while(0);
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	コールバックによるチップテキスト設定																		//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static VO	SetTipTextByCallBack(PAVLNODE pNode)
{
	if (pNode->u.vop != NULL) {		// コールバックあり
		//----- 作業バッファ使用 -----------------------------------------------//
		if (pNode->fCBWithBuf) {
			//	ワイド文字モード
			if (pNode->fUnicode) {
				C_WCP	pTxt = NULL;
				WCP		pBuf = NULL;
				UI		lBuf = 4096;
	
				if (pBuf = AJCMEM(lBuf * 2)) {
					//	チップテキスト取得
					memset(pBuf, 0, lBuf * 2);
					if ((pTxt = pNode->u.cbGetTextW(pNode->hwnd, pBuf, lBuf, pNode->cbp)) != NULL) {
						SetTipText(pNode, pTxt);
					}
					free(pBuf);
				}
			}
			//	バイト文字モード
			else {
				C_BCP	pTxt = NULL;
				BCP		pBuf = NULL;
				WCP		pTmp = NULL;
				UI		lTmp = 0;
				UI		lBuf = 4096;
	
				if (pBuf = AJCMEM(lBuf)) {
					//	チップテキスト取得
					memset(pBuf, 0, lBuf);
					if ((pTxt = pNode->u.cbGetTextA(pNode->hwnd, pBuf, lBuf, pNode->cbp)) != NULL) {
						//	ワイド文字に変換し、チップテキスト表示
						lTmp = MultiByteToWideChar(CP_ACP, 0, pTxt, -1, NULL, 0);
						if (lTmp != 0) {
							if (pTmp = AjcTAlloc(lTmp)) {
								MultiByteToWideChar(CP_ACP, 0, pTxt, -1, pTmp, lTmp);
								SetTipText(pNode, pTmp);
								free(pTmp);
							}
						}
					}
					free(pBuf);
				}
			}
		}
		//----- 作業バッファ未使用 ---------------------------------------------//
		else {
			//	ワイド文字モード
			if (pNode->fUnicode) {
				C_WCP	pTxt = NULL;
	
				if ((pTxt = pNode->u.cbGetNoBufW(pNode->hwnd, pNode->cbp)) != NULL) {
					SetTipText(pNode, pTxt);
				}
			}
			//	バイト文字モード
			else {
				C_BCP	pTxt = NULL;
				WCP		pTmp = NULL;
				UI		lTmp = 0;
	
				if ((pTxt = pNode->u.cbGetNoBufA(pNode->hwnd, pNode->cbp)) != NULL) {
					//	ワイド文字に変換し、チップテキスト表示
					lTmp = MultiByteToWideChar(CP_ACP, 0, pTxt, -1, NULL, 0);
					if (lTmp != 0) {
						if (pTmp = (WCP)AJCMEM(lTmp * 2)) {
							MultiByteToWideChar(CP_ACP, 0, pTxt, -1, pTmp, lTmp);
							SetTipText(pNode, pTmp);
							free(pTmp);
						}
					}
				}
			}
		}
	}
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	チップテキスト表示																							//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static BOOL	ShowTipText(PAVLNODE pNode, int x, int y)
{
	BOOL		rc		 = FALSE;
	HWND		hActWnd  = GetForegroundWindow();
	UL			pid 	 = 0;	//	プロセスＩＤ
	UL			tid 	 = 0;	//	スレッドＩＤ

	//	アクティブなウインドのプロセスＩＤ取得
	if (hActWnd != NULL) {
		tid = GetWindowThreadProcessId(hActWnd, &pid);
	}

	//	自スレッドにアクティブなウインド有り or 常時表示
	if ((pid == MyProcessId) || (pNode->fShowAlways && !FlgShowForActive)) {
		//	コールバックが設定されていれば、コールバックによるチップテキスト取得
		SetTipTextByCallBack(pNode);
		//	表示テキストがあればチップテキスト表示
		if (pNode->pText != NULL) {
			//	Ｙ位置をカーソルの高さ分加算
			y += GetCursorOffset();
			//	チップ位置をコントロール直下に補正
			if (TipPosMode == AJCTIP_POS_UNDER_CONTROL) {
				RECT r;
				//	コントロールの矩形位置取得
				GetWindowRect(pNode->hwnd, &r);
				//	表示位置がコントロール下端より下ならば、コントロール直下に訂正
				if (y > r.bottom + 2) {
				//	x += 14;
					y  = r.bottom + 2;
				}
			}
			//	チップテキスト表示
			AjcTipTextShowExInternal(x, y, 0, 0, pNode->pText, pNode->msShow, pNode->hFont,
									 pNode->TextColor, pNode->BkColor, pNode->BorderColor, pNode->hwnd);
			//	チップテキスト表示時のカーソル位置設定
			stPt.x = x;
			stPt.y = y;
			rc = TRUE;
		}
		//	表示テキストが無ければチップテキスト非表示
		else {
			HideTipText(FALSE);		//	非表示
			pst		= PST_IDLE;		//	→アイドル状態
		}
	}
	//	自スレッドにアクティブなウインド無し and アクティブ時のみ表示
	else {
		//	チップテキスト非表示
		HideTipText(FALSE);			//	非表示
		pst		= PST_IDLE;			//	→アイドル状態
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	チップテキスト移動																							//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static VO	MoveTipText(PAVLNODE pNode, int x, int y)
{
	//	Ｙ位置をカーソルの高さ分加算
	y += GetCursorOffset();
	//	チップ位置をコントロール直下に補正
	if (TipPosMode == AJCTIP_POS_UNDER_CONTROL) {
		RECT r;
		//	コントロールの矩形位置取得
		GetWindowRect(pNode->hwnd, &r);
		//	表示位置がコントロール下端より下ならば、コントロール直下に訂正
		if (y > r.bottom + 2) {
		//	x += 14;
			y  = r.bottom + 2;
		}
	}
	//	チップテキスト移動
	AjcTipTextMove(x, y);
}

//--------------------------------------------------------------------------------------------------------------//
//																												//
//	チップテキスト非表示																						//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static VO	HideTipText(BOOL fDelay)
{
	AjcTipTextHideEx(fDelay);
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	経過時間取得																								//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	UI	GetInterval(UI SrtTime)
{
	UI		rc = 0;
	UI		CurTime = GetTickCount();

	if (SrtTime <= CurTime) {
		rc = CurTime - SrtTime;
	}
	else {
		rc = (UI)(~SrtTime + 1) + CurTime;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	カーソルオフセット																							//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	int	GetCursorOffset(VO)
{
	int		rc = GetSystemMetrics(SM_CYCURSOR);
	HCURSOR	hCursor = GetCursor();

	if		(hCursor == hARROW		) rc = rc * 60 / 100;	//	標準矢印
	else if	(hCursor == hIBEAM		) rc = rc * 36 / 100;	//	Iビーム
	else if	(hCursor == hAPPSTARTING) rc = rc * 60 / 100;	//	標準矢印と小さな砂時計
	else if	(hCursor == hCROSS		) rc = rc * 60 / 100;	//	十字
	else if	(hCursor == hHAND		) rc = rc * 60 / 100;	//	手
	else if	(hCursor == hHELP		) rc = rc * 60 / 100;	//	矢印と疑問符
	else if	(hCursor == hNO			) rc = rc * 60 / 100;	//	スラッシュサークル
	else if	(hCursor == hSIZEALL	) rc = rc * 60 / 100;	//	北、南、東、西を指す四方の矢印	
	else if	(hCursor == hSIZENESW	) rc = rc * 60 / 100;	//	北東と南西を指す二重矢印
	else if	(hCursor == hSIZENS		) rc = rc * 60 / 100;	//	南北向きの二重矢印
	else if	(hCursor == hSIZENWSE	) rc = rc * 60 / 100;	//	北西と南東を指す二重矢印
	else if	(hCursor == hSIZEWE		) rc = rc * 60 / 100;	//	西と東を指す二重矢印
	else if	(hCursor == hUPARROW	) rc = rc * 60 / 100;	//	垂直矢印
	else if	(hCursor == hWAIT		) rc = rc * 60 / 100;	//	砂時計
	else 							  rc = rc * 60 / 100;	//	その他

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	表示遅延時間補正																							//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	UI		AdjustMsDelay(int msDelay)
{
	UI		rc;

	if (msDelay == -1) rc = DefMsDelay;
	else			   rc = msDelay;

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	表示時間補正																								//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	UI		AdjustMsShow(int msShow)
{
	UI		rc;

	if (msShow == -1) rc = DefMsShow;
	else			  rc = msShow;

	return rc;
}
