﻿#include	"AjcInternal.h"
//**************************************************************************************************************//
//																												//
//	ツールチップテキスト表示（コントロールに関付けない単純なチップテキスト表示）								//
//																												//
//**************************************************************************************************************//

#define		MY_CLASS		L"AjcTipTxt"

#define		WM_SETFOCUS_DELAY	(WM_APP + 100)		//	フォーカス移動用メッセージ

#define		MRG_X		40							//	左右マージン（文字幅のＮ％）
#define		MRG_Y		25							//	上下マージン（文字高のＮ％）

#define		IDC_BTN_COPY	5001					//	コピーボタンＩＤ

extern	VO	AjcTipTextEnter(VO);
extern	VO	AjcTipTextLeave(VO);

//--------------------------------------------------------------------------------------------------------------//
//	フォント変更情報																							//
//--------------------------------------------------------------------------------------------------------------//
#define		CFBE_NORMAL		0x00
#define		CFBE_BOLD		0x01
#define		CFBE_ITALIC		0x02
#define		CFBE_BOTH		(CFBE_BOLD | CFBE_ITALIC)

typedef struct {
	UI			flag;			//	ボールド，イタリックを示すフラグ
	HFONT		hFontOrg;		//	オリジナルフォント
	HFONT		hFontN;			//	ノーマルフォント
	HFONT		hFontB;			//	ボールドフォント
	HFONT		hFontI;			//	イタリックフォント
	HFONT		hFontBI;		//	ボールド＋イタリックフォント
} FONTCHGINFO, *PFONTCHGINFO;

//--------------------------------------------------------------------------------------------------------------//
//	タイマＩＤ																									//
//--------------------------------------------------------------------------------------------------------------//
enum {
	TID_SHOW	= 1		,		//	表示時間タイマ
	TID_CHK_CURSOR		,		//	カーソル位置監視タイマ
};

#define	TM_CHK_CURSOR	150		//	カーソル位置監視タイマの周期

//--------------------------------------------------------------------------------------------------------------//
//	描画イメージ情報																							//
//--------------------------------------------------------------------------------------------------------------//
typedef struct {
	UI		id;										//	識別ＩＤ
	BOOL	fIcon;									//	イメージ種別（FALSE : Bitmap, TRUE : Icon）
	union	{HBITMAP hBmp; HICON hIcon; UX h;} u;	//	イメージのハンドル
	int		x, y, cx, cy;							//	描画位置とサイズ
} TIPIMGINFO, *PTIPIMGINFO;
typedef const TIPIMGINFO *PCTIPIMGINFO;

//--------------------------------------------------------------------------------------------------------------//
//	作業領域																									//
//--------------------------------------------------------------------------------------------------------------//
static	ATOM		ClassTipTxt		= 0;					//	クラスハンドル
static	HWND		hWndTip			= NULL;					//	ウインドハンドル
static	HWND		hBtnCopy		= NULL;					//	コピーボタン
static	HWND		hToolTip		= NULL;					//	ボタン用ツールチップ
static	HFONT		hDefFont		= NULL;					//	デフォルト・フォントハンドル
static	HBRUSH		hDefBrush		= NULL;					//	デフォルト・ウインド背景ブラシハンドル
static	COLORREF	DefBkColor		= RGB(255, 255, 225);	//	デフォルト・ウインド背景色
static	COLORREF	DefTextColor	= RGB(0, 0, 0);			//	デフォルト・テキスト表示色
static	COLORREF	DefBorderColor	= RGB(0, 0, 0);			//	デフォルト・外枠表示色
		int			DefMsShow		= 5000;					//	デフォルト表示時間(AjcTipCtl.c で外部参照)
static	COLORREF	ColorKey		= RGB(254, 252, 253);	//	透明色

static	HAJCAVL		hAvlImg			= NULL;					//	イメージ情報格納用ＡＶＬ木（Bitmap / Icon）

static	HWND		SvhCaller		 = NULL;				//	呼び出し元のウインドハンドル
static	UI			MsgNtcTipChanged = 0;					//	チップ表示切替通知メッセージ

static	HWND		hWndFocus		= NULL;					//	直前のフォーカスウインド

static	BOOL		fShow			= FALSE;				//	表示中フラグ
static	BOOL		fShowBtnCopy	= FALSE;				//	コピーボタン表示状態
static	BOOL		fCloseMark		= FALSE;				//	クローズマーク(X)表示フラグ
static	HFONT		hTipFont		= NULL;					//	フォントハンドル
static	int			msShow			= 5000;					//	テキスト表示時間[ms]
static	WCP			pTipTxt			= NULL;					//	表示テキスト
static	UI			lTipTxt			= 0;					//	表示テキストの長さ
static	COLORREF	RgbTextColor	= RGB(	0,	 0,   0);	//	文字色
static	COLORREF	RgbBkColor		= RGB(255, 255, 225);	//	背景色
static	COLORREF	RgbBorderColor	= RGB(	0,	 0,   0);	//	外枠色（－２：外枠非表示）
static	UB			Alpha			= 255;					//	透明度（未使用）
static	int			cyChar			= 12;					//	文字の高さ
static	int			cxChar			= 8;					//	文字の幅
static	POINT		ptWnd			= {0, 0};				//	ウインド位置
static	SIZE		szWnd			= {0, 0};				//	ウインドサイズ
static	int			xTop, yTop;								//	描画開始位置

//	パレット色
static	COLORREF	Palette[8] = {RGB(	 0,	  0,   0),		//	0: 黒
								  RGB( 255,	  0,   0),		//	1: 赤
								  RGB(	 0, 255,   0),		//	2: 緑
								  RGB( 255, 255,   0),		//	3: 黄
								  RGB(	 0,	  0, 255),		//	4: 青
								  RGB( 255,	  0, 255),		//	5: 紫
								  RGB(	 0, 255, 255),		//	6: 水色
								  RGB( 255, 255, 255),		//	7: 白
};

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC_DEF(TipTxt		);
AJC_WNDPROC_DEF(BtnCopy		);

static	VO			MouseButtonDown(HWND hwnd);

static	VO			ShowTipTextWindow(VO);
static	VO			HideTipTextWindow(VO);
static	VO			ShowBtnCopy(VO);
static	VO			HideBtnCopy(VO);
static	VO			AjustWndPos(LPPOINT pPt, LPSIZE pSz);
static	BOOL		IsCursorOnWindow(VO);
static	BOOL		GetCharSize(HDC hdc, int *pWidth, int *pHeight);


//==============================================================================================================//
//	起動時初期設定																								//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
BOOL	AjcTipTxtInit(VO)
{
	BOOL		rc = FALSE;
	WNDCLASS	wndclass;
	LOGFONT		lf;

	do {
		//----- チップ表示切替通知メッセージコード設定 --------//
		MsgNtcTipChanged = RegisterWindowMessage(MSGSTR_NTC_TIPCHANGED);
		//----- デフォルト・ブラシ生成 ------------------------//
		hDefBrush = CreateSolidBrush(DefBkColor);
		//----- チップテキスト表示ウインドクラス --------------//
		wndclass.style			= CS_GLOBALCLASS;
		wndclass.lpfnWndProc	= AJC_WNDPROC_NAME(TipTxt);
		wndclass.cbClsExtra		= 0;
		wndclass.cbWndExtra		= sizeof(UX);
		wndclass.hInstance		= hDllInst;
		wndclass.hIcon			= NULL;
		wndclass.hCursor		= LoadCursor(NULL, IDC_ARROW);
		wndclass.hbrBackground	= hDefBrush;
		wndclass.lpszMenuName	= NULL;
		wndclass.lpszClassName	= MY_CLASS;
		if ((ClassTipTxt = RegisterClass(&wndclass)) == 0) break;

		//----- デフォルトフォント生成 ------------------------//
		lf.lfHeight=12;			lf.lfItalic=0;			lf.lfClipPrecision=2;
		lf.lfWidth=0;			lf.lfUnderline=0;		lf.lfQuality=1;
		lf.lfEscapement=0;		lf.lfStrikeOut=0;		lf.lfPitchAndFamily=VARIABLE_PITCH;
		lf.lfOrientation=0;		lf.lfCharSet=128;		wcscpy(lf.lfFaceName, L"MS UI Gothic");
		lf.lfWeight=400;		lf.lfOutPrecision=3;
		hDefFont = CreateFontIndirect(&lf);

		rc = TRUE;

	} while(0);

	if (rc == FALSE) {
		AjcTipTxtEnd();
	}

	return rc;
}
//==============================================================================================================//
//	終了時後処理																								//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
VO		AjcTipTxtEnd (VO)
{
	if (hWndTip 	!= NULL) {DestroyWindow(hWndTip);					   hWndTip		= NULL;	}
	if (ClassTipTxt != 0   ) {UnregisterClass((WCP)ClassTipTxt, hDllInst); ClassTipTxt	= 0;	}
	if (hDefFont	!= NULL) {DeleteObject(hDefFont);					   hDefFont 	= NULL; }
	if (hDefBrush	!= NULL) {DeleteObject(hDefBrush); 					   hDefBrush	= NULL; }
	if (pTipTxt 	!= NULL) {free(pTipTxt);							   pTipTxt		= NULL;	}
	if (hAvlImg		!= NULL) {AjcAvlDelete(hAvlImg);					   hAvlImg		= NULL;	}

}
//==============================================================================================================//
//	チップテキスト表示ウインドの生成																			//
//																												//
//	引　数：	なし																							//
//																												//
//	戻り値：	≠NULL	- 成功（ウインドハンドル）																//
//				＝NULL	- 失敗																					//
//==============================================================================================================//
AJCEXPORT HWND	 WINAPI AjcTipTextCreate(VO)
{
	if (hWndTip == NULL) {
		//----- チップテキストウインド生成 --------------------//
		hWndTip = CreateWindowEx(WS_EX_TOPMOST | WS_EX_TOOLWINDOW | WS_EX_LAYERED,		// extended style
								  MY_CLASS,												// window class name
								  L"",													// window caption
								  WS_POPUP,												// window style
								  0,													// initial x position
								  0,													// initial y position
								  0,													// initial x size
								  0,													// initial y size
								  NULL,													// parent window handle
								  NULL,													// window menu handle
								  hDllInst,												// program instance handle
								  NULL);												// creation parameters

		if (hWndTip != NULL) {
			//----- チップウインドはＺオーダー対象外とする ----//
			AJCTIP_SETFRIENDA(hWndTip);
			//----- チップウインド可視化 ----------------------//
			ShowWindow(hWndTip, SW_SHOW);
			//----- ボタン生成 --------------------------------//
			hBtnCopy = CreateWindow(L"BUTTON",								// window class name
									LNGSEL(L"コピー", L"Copy"),				// window caption
									WS_CHILD,								// window style
									0,										// initial x position
									0,										// initial y position
									0,										// initial x size
									0,										// initial y size
									hWndTip,								// parent window handle
									(HMENU)IDC_BTN_COPY,					// window menu handle
									hDllInst,								// program instance handle
									NULL);									// creation parameters
			ShowWindow(hBtnCopy, SW_HIDE);
			//----- コピーボタンをサブクラス化 ----------------//
			MAjcMmpSetSubclass(BtnCopy, hBtnCopy);
			//----- ツールチップ・コントロール生成 ------------//
			hToolTip = CreateWindowEx(WS_EX_TOPMOST,
									TOOLTIPS_CLASS,
									NULL,
									TTS_ALWAYSTIP,
									CW_USEDEFAULT,
									CW_USEDEFAULT,
									CW_USEDEFAULT,
									CW_USEDEFAULT,
									hWndTip, NULL,
									hDllInst, NULL);
			//----- ツール追加 --------------------------------//
			{	TOOLINFOA	ti;
				memset(&ti, 0, sizeof ti);
				ti.cbSize	= sizeof(TOOLINFOA);
				ti.hwnd		= hWndTip;
				ti.hinst	= hDllInst;
				ti.uFlags	= TTF_IDISHWND | TTF_SUBCLASS;
				ti.lpszText = LNGSEL("ツールチップテキストをクリップボードにコピーします。SHIFTキーを押しながらクリックした場合はツールチップイメージをコピーします。",
							 		 "Copy tooltip text to clipboard. Copy the tooltip image if you click with SHIFT key.");

				ti.uId = (UX)hBtnCopy;		SendMessageA(hToolTip, TTM_ADDTOOLA, 0, (LPARAM)&ti);
			}
			//----- ウインド非表示 ----------------------------//
			HideTipTextWindow();
		}
	}
	return hWndTip;
}
//==============================================================================================================//
//	チップテキスト表示ウインドの破棄																			//
//																												//
//	引　数：	なし																							//
//																												//
//	戻り値：	なし																							//
//==============================================================================================================//
AJCEXPORT VO	 WINAPI AjcTipTextDelete(VO)
{
	if (hWndTip != NULL) {
		DestroyWindow(hWndTip);
		hWndTip = NULL;
	}
}
//==============================================================================================================//
//	チップテキスト表示ウインドの破棄（コンソールアプリ用）														//
//																												//
//	引　数：	なし																							//
//																												//
//	戻り値：	なし																							//
//																												//
//	備　考：	コンソールアプリでは WM_DESTROY が発生しないため、ボタンウインドもここで破棄する				//
//==============================================================================================================//
AJCEXPORT VO	 WINAPI AjcTipTextDeleteForConApp(VO)
{
	HWND	hWndTipSv  = hWndTip;
	HWND	hBtnCopySv = hBtnCopy;

	hWndTip  = NULL;
	hBtnCopy = NULL;

	if (hWndTipSv != NULL) {
		//	コピーボタン破棄
		if (hBtnCopySv != NULL) {
			MAjcMmpClrSubclass(BtnCopy, hBtnCopySv);
			DestroyWindow(hBtnCopySv);
		}
		//	チップテキストウインド破棄
		DestroyWindow(hWndTipSv);
	}
}

//==============================================================================================================//
//	チップテキストの表示																						//
//																												//
//	引　数：	x, y	- ウインドの表示位置																	//
//				pTxt	- 表示テキスト																			//
//				msTime	- 表示時間（ms）																		//
//				hFont	- フォント(NULL:デフォルトフォント）													//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextShowA(int x, int y, C_BCP pTxt, int msTime, HFONT hFont)
{
	return AjcTipTextShowExA(x, y, 0, 0, pTxt, msTime, hFont, -1, -1, RGB(0,0,0));
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextShowW(int x, int y, C_WCP pTxt, int secTime, HFONT hFont)
{
	return AjcTipTextShowExW(x, y, 0, 0, pTxt, secTime, hFont, -1, -1, RGB(0,0,0));
}
//==============================================================================================================//
//	チップテキストの表示（拡張版）																				//
//																												//
//	引　数：	x, y		- ウインドの表示位置																//
//				minWidth	- ウインドの最小幅（ピクセル数，０の場合は、最大文字列長に合わせる）				//
//				height		- ウインドの高さ　（ピクセル数，０の場合は文字の高さ×行数に合わせる）				//
//				pTxt		- 表示テキスト																		//
//				msTime		- 表示時間（ms）																	//
//				hFont		- フォントハンドル（NULL:デフォルトフォント）										//
//				TextColor	- 文字色（－１：黒）																//
//				BackGround	- ウインド背景色（－１：デフォルト色）												//
//				BorderColor	- 外枠色（－１：黒，－２：外枠非表示）												//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextShowExA(int x, int y, int minWidth, int height, C_BCP pTxt, int msTime, HFONT hFont,
											 COLORREF TextColor, COLORREF BackGround, COLORREF BorderColor)
{
	BOOL	rc = FALSE;
	UI		len  = 0;
	WCP		pTmp = NULL;

	AjcTipTextEnter();	//	スレッド排他制御開始

	if (pTxt != NULL) {
		len = MultiByteToWideChar(CP_ACP, 0, pTxt, -1, NULL, 0);
		if (len != 0 && (pTmp = (WCP)AJCMEM(len * 2))) {
			MultiByteToWideChar(CP_ACP, 0, pTxt, -1, pTmp, len);
			rc = AjcTipTextShowExInternal(x, y, minWidth, height, pTmp, msTime, hFont, TextColor, BackGround, BorderColor, NULL);
		}
		if (pTmp != NULL) free(pTmp);
	}

	AjcTipTextLeave();	//	スレッド排他制御終了

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextShowExW(int x, int y, int minWidth, int height, C_WCP pTxt, int msTime, HFONT hFont,
									 COLORREF TextColor, COLORREF BackGround, COLORREF BorderColor)
{
	BOOL	rc = FALSE;

	AjcTipTextEnter();	//	スレッド排他制御開始

	rc = AjcTipTextShowExInternal(x, y, minWidth, height, pTxt, msTime, hFont, TextColor, BackGround, BorderColor, NULL);

	AjcTipTextLeave();	//	スレッド排他制御終了

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	チップテキストの表示（内部ファンクション）																	//
//																												//
//	引　数：	x, y		- ウインドの表示位置																//
//				width		- ウインドの幅　（０の場合は自動算出）												//
//				height		- ウインドの高さ（０の場合は自動算出）												//
//				pTxt		- 表示テキスト																		//
//				msTime		- 表示時間（ms）																	//
//				hFont		- フォントハンドル（NULL:デフォルトフォント）										//
//				TextColor	- 文字色（－１：黒）																//
//				BackGround	- ウインド背景色（－１：デフォルト色）												//
//				BorderColor	- 外枠色（－１：黒，－２：外枠非表示）												//
//				hCaller		- 呼び出し元のウインドハンドル（ツールチップサブクラスウインド）					//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//--------------------------------------------------------------------------------------------------------------//
extern	BOOL	SubTxoGetImgInfo(HAJCAVL hAvl, C_UTP pEsc, PTXOIMGINFO pInfo, BITMAP *pBm);

BOOL	 AjcTipTextShowExInternal(int x, int y, int width, int height, C_WCP pTxt, int msTime, HFONT hFont,
											 COLORREF TextColor, COLORREF BackGround, COLORREF BorderColor, HWND hCaller)
{
	BOOL			rc = FALSE;
	HDC				hdc;
	HFONT			hF;

	//	ウインド生成
	if (hWndTip == NULL) {
		AjcTipTextCreate();
	}

	if (hWndTip != NULL) {
		//	カーソル位置監視タイマ停止
		KillTimer(hWndTip, TID_CHK_CURSOR);
		//	テキスト表示
		if (pTxt != NULL && *pTxt != 0) {
			//	・フォント
			if (hFont == NULL) hTipFont = hDefFont;
			else			   hTipFont = hFont;
			//	・表示時間
			msShow	 	= (msTime == -1 ? DefMsShow : msTime);
			//	・表示テキスト
			lTipTxt		= (UI)wcslen(pTxt);
			if (pTipTxt != NULL) free(pTipTxt);
			pTipTxt		= (WCP)AJCMEM(lTipTxt * 2 + 2);
			if (pTipTxt != NULL) {
				//	表示テキスト設定
				wcscpy(pTipTxt, pTxt);
				//	・文字色
				RgbTextColor   = (TextColor   != -1) ? TextColor	: DefTextColor;
				//	・背景色
				RgbBkColor 	   = (BackGround  != -1) ? BackGround	: DefBkColor;
				//	・外枠色
				RgbBorderColor = (BorderColor != -1) ? BorderColor	: DefBorderColor;
				//	ＤＣ設定
				hdc = GetDC(hWndTip);
				//	フォント設定
				hF = (HFONT)SelectObject(hdc, hTipFont);
				//	文字サイズ設定
				GetCharSize(hdc, &cxChar, &cyChar);
				//	ウインドサイズ取得
				if (AjcTipTextGetSizeW(pTxt, hTipFont, RgbBorderColor != -2, &szWnd)) {
					//	描画開始位置初期化
					xTop = yTop = 0;
					//	ウインド幅，描画開始Ｘ位置設定
					//	・外枠ありの場合、Ｘ開始位置更新
					if (RgbBorderColor != -2) {
						xTop += 1;
					}
					//	・文字幅のＮ％の余白分Ｘ位置更新
					xTop += (cxChar * MRG_X 	/ 100);
					//	ウインド幅が指定された場合は、指定値設定
					if (height != 0) szWnd.cx = width;
					//	ウインド高さが指定された場合は、指定値設定
					if (height != 0) szWnd.cy = height;
					//	描画Ｙ位置設定
					yTop = (cyChar * MRG_Y 	/ 100);
					//		外枠ありの場合、Ｙ位置更新
					if (RgbBorderColor != -2 ) {
						yTop += 1;
					}
					//	ウインド位置設定
					ptWnd.x			= x;
					ptWnd.y			= y;
					//	ウインド表示位置補正
					AjustWndPos(&ptWnd, &szWnd);
					//	ウインド移動
					SetWindowPos(hWndTip, HWND_TOPMOST, ptWnd.x, ptWnd.y, szWnd.cx, szWnd.cy, SWP_NOACTIVATE);
					//	テキスト表示
					InvalidateRect(hWndTip, NULL, TRUE);
					//	ウインド表示
					ShowTipTextWindow();
					//	テキスト表示期間タイマ起動
					SetTimer(hWndTip, TID_SHOW, msShow, NULL);
					//	チップ表示切り替わりを通知
					if (SvhCaller != NULL && hCaller == NULL) {
						SendMessage(SvhCaller, MsgNtcTipChanged, 0, 0);
					}
					//	呼び出し元ウインドハンドル退避
					SvhCaller = hCaller;
					//	戻り値＝ＯＫ
					rc = TRUE;
				}
				//	フォント設定解除
				SelectObject(hdc, hF);
				//	ＤＣ解放
				ReleaseDC(hWndTip, hdc);
			}
		}
		else {
			//	表示テキストが無い場合は、ウインド非表示
			AjcTipTextHide();
		}
	}

	return rc;
}
//==============================================================================================================//
//	チップテキストをウインドの中央に表示																		//
//																												//
//	引　数：	hwnd	- ウインドのハンドル																	//
//				pTxt	- 表示テキスト																			//
//				msTime	- 表示時間（ms）																		//
//				hFont	- フォント(NULL:デフォルトフォント）													//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//==============================================================================================================//
static	BOOL	GetCenterRect(HWND hwnd, LPRECT pRect)
{
	BOOL	rc = FALSE;

	if		(hwnd == NULL) {
		POINT	pt;
		GetCursorPos(&pt);
		AjcGetMonitorInfoOfPoint(pt.x, pt.y, NULL, pRect);
		rc = TRUE;
	}
	else if (IsWindow(hwnd)) {
		GetClientRect(hwnd, pRect);
		MapWindowPoints(hwnd, NULL, (LPPOINT)pRect, 2);
		rc = TRUE;
	}
	return rc;
}
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextShowCenterA(HWND hwnd, C_BCP pTxt, int msTime, HFONT hFont)
{
	BOOL	rc = FALSE;
	SIZE	sz;
	RECT	r;
	int		wx, wy, ww, wh;

	AjcTipTextEnter();

	if (GetCenterRect(hwnd, &r)) {
		wx = r.left;  wy = r.top;
		ww = r.right - r.left;
		wh = r.bottom - r.top;
		if (AjcTipTextGetSizeA(pTxt, hFont, TRUE, &sz)) {
			rc = AjcTipTextShowExA(wx + (ww - sz.cx) / 2, wy + (wh - sz.cy) / 2, 0, 0, pTxt, msTime, hFont, -1, -1, RGB(0,0,0));
		}
	}

	AjcTipTextLeave();

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextShowCenterW(HWND hwnd, C_WCP pTxt, int msTime, HFONT hFont)
{
	BOOL	rc = FALSE;
	SIZE	sz;
	RECT	r;
	int		wx, wy, ww, wh;

	if (GetCenterRect(hwnd, &r)) {
		wx = r.left;  wy = r.top;
		ww = r.right - r.left;
		wh = r.bottom - r.top;
		if (AjcTipTextGetSizeW(pTxt, hFont, TRUE, &sz)) {
			rc = AjcTipTextShowExW(wx + (ww - sz.cx) / 2, wy + (wh - sz.cy) / 2, 0, 0, pTxt, msTime, hFont, -1, -1, RGB(0,0,0));
		}
	}
	return rc;
}
//==============================================================================================================//
//	チップテキストをウインドの中央に表示（拡張版）																//
//																												//
//	引　数：	x, y		- ウインドの表示位置																//
//				width		- ウインドの幅　（０の場合は自動算出）												//
//				height		- ウインドの高さ（０の場合は自動算出）												//
//				pTxt		- 表示テキスト																		//
//				msTime		- 表示時間（ms）																	//
//				hFont		- フォントハンドル（NULL:デフォルトフォント）										//
//				TextColor	- 文字色（－１：黒）																//
//				BackGround	- ウインド背景色（－１：デフォルト色）												//
//				BorderColor	- 外枠色（－１：黒，－２：外枠非表示）												//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 	WINAPI AjcTipTextShowCenterExA(HWND hwnd, int width, int height, C_BCP pTxt, int msTime, HFONT hFont, COLORREF TextColor, COLORREF BackGround, COLORREF BorderColor)
{
	BOOL	rc = FALSE;
	SIZE	sz;
	RECT	r;
	int		wx, wy, ww, wh;

	AjcTipTextEnter();

	//	ウインド生成
	if (hWndTip == NULL) {
		AjcTipTextCreate();
	}

	if (GetCenterRect(hwnd, &r)) {
		wx = r.left;  wy = r.top;
		ww = r.right - r.left;
		wh = r.bottom - r.top;
		if (AjcTipTextGetSizeA(pTxt, hFont, TRUE, &sz)) {
			rc = AjcTipTextShowExA(wx + (ww - sz.cx) / 2, wy + (wh - sz.cy) / 2, width, height, pTxt, msTime, hFont, TextColor, BackGround, BorderColor);
		}
	}

	AjcTipTextLeave();

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 	WINAPI AjcTipTextShowCenterExW(HWND hwnd, int width, int height, C_WCP pTxt, int msTime, HFONT hFont, COLORREF TextColor, COLORREF BackGround, COLORREF BorderColor)
{
	BOOL	rc = FALSE;
	SIZE	sz;
	RECT	r;
	int		wx, wy, ww, wh;

	if (GetCenterRect(hwnd, &r)) {
		wx = r.left;  wy = r.top;
		ww = r.right - r.left;
		wh = r.bottom - r.top;
		if (AjcTipTextGetSizeW(pTxt, hFont, TRUE, &sz)) {
			rc = AjcTipTextShowExW(wx + (ww - sz.cx) / 2, wy + (wh - sz.cy) / 2, width, height, pTxt, msTime, hFont, TextColor, BackGround, BorderColor);
		}
	}
	return rc;
}
//==============================================================================================================//
//	チップテキストにクローズ(X)マーク付加																		//
//																												//
//	引　数：	fMark		- クローズマーク表示フラグ（FALSE:表示しない，TRUE:表示する）						//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//==============================================================================================================//
AJCEXPORT VO	 	WINAPI AjcTipTextShowCloseMark(BOOL fMark)
{
	fCloseMark = (fMark != FALSE);
}
//==============================================================================================================//
//	チップテキストウインドの表示サイズ取得																		//
//																												//
//	引　数：	pTxt		- 表示テキスト																		//
//				hFont		- フォントハンドル（NULL:デフォルトフォント）										//
//				fBorder		- 外枠の有無																		//
//				pBuf		- チップテキスト表示サイズを格納するバッファのアドレス								//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextGetSizeA(C_BCP pTxt, HFONT hFont, BOOL fBorder, LPSIZE pBuf)
{
	BOOL	rc = FALSE;
	UI		len  = 0;
	WCP		pTmp = NULL;

	if (pTxt != NULL) {
		len = MultiByteToWideChar(CP_ACP, 0, pTxt, -1, NULL, 0);
		if (len != 0 && (pTmp = (WCP)AJCMEM(len * 2))) {
			MultiByteToWideChar(CP_ACP, 0, pTxt, -1, pTmp, len);
			rc = AjcTipTextGetSizeW(pTmp, hFont, fBorder, pBuf);
		}
		if (pTmp != NULL) free(pTmp);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTipTextGetSizeW(C_WCP pTxt, HFONT hFont, BOOL fBorder, LPSIZE pBuf)
{
	BOOL			rc  = FALSE;
	HDC				hdc = NULL;
	HFONT			hF  = NULL;
	SIZE			sz  = {0};
	int				ww, wh;
	HAJCTXO			hTxo;

	AjcTipTextEnter();

	//	ウインド生成
	if (hWndTip == NULL) {
		AjcTipTextCreate();
	}

	if (hWndTip != NULL && pBuf != NULL) {
		//	テキスト表示サイズ算出
		if (pTxt != NULL && *pTxt != 0) {
			AjcTipTextEnter();	//	スレッド排他制御開始
			do {
				SIZE	szImg;
				//	・フォント
				if (hFont == NULL) hTipFont = hDefFont;
				else			   hTipFont = hFont;
				//	ＤＣ設定
				hdc = GetDC(hWndTip);
				//	フォント設定
				hF = (HFONT)SelectObject(hdc, hTipFont);
				//	文字サイズ設定
				GetCharSize(hdc, &cxChar, &cyChar);
				//	文字列描画の矩形サイズ取得
				if (hTxo = AjcTxoCreate(hdc, xTop, yTop, -20)) {	//	行間＝文字高さの２０％
					AjcTxoGetExtentW(hTxo, pTxt, &sz);
					AjcTxoDelete(hTxo, TRUE);
				}
				//	イメージ描画テキスト("\x1B[id・・z")から、ウインドサイズ補正
				if (AjcTxoGetImgSpaceSizeW(pTxt, &szImg)) {
					sz.cx = __max(sz.cx, szImg.cx);
					sz.cy = __max(sz.cy, szImg.cy);
				}
				ww = sz.cx;
				wh = sz.cy;
				//	ウインド幅設定
				//		単独チップの場合、閉じるマーク（X)分広げる
				if (fCloseMark) {
					ww += 8;
				}
				//		外枠ありの場合、左右２ピクセル追加
				if (RgbBorderColor != -2 ) {
					ww += 2;
				}
				//		両端に文字幅のＮ％の余白
				ww	 += (cxChar * MRG_X * 2 / 100);
				pBuf->cx = ww;
				//	ウインド高さ設定
				//	外枠ありの場合、上下２ピクセル追加
				if (RgbBorderColor != -2 ) {
					wh	 += 2;
				}
				//	上下端に文字高さのＮ％の余白
				pBuf->cy = wh + (cyChar * MRG_Y * 2 / 100);
				//	フォント設定解除
				SelectObject(hdc, hF);
				//	ＤＣ解放
				ReleaseDC(hWndTip, hdc);
				rc = TRUE;
			} while (0);
			AjcTipTextLeave();	//	スレッド排他制御終了
		}
		else {
			//	表示テキストが無い場合は、サイズ＝０
			pBuf->cx = 0;
			pBuf->cy = 0;
			rc = TRUE;
		}
	}

	AjcTipTextLeave();

	return rc;
}
//==============================================================================================================//
//	チップテキストの移動																						//
//																												//
//	引　数：	x, y - 移動位置																					//
//																												//
//	戻り値：	なし																							//
//==============================================================================================================//
AJCEXPORT VO	 WINAPI AjcTipTextMove(int x, int y)
{
	AjcTipTextEnter();

	if (fShow) {
		//	ウインド表示位置補正
		ptWnd.x = x;
		ptWnd.y = y;
		AjustWndPos(&ptWnd, &szWnd);
		//	ウインド移動
		SetWindowPos(hWndTip, HWND_TOPMOST, ptWnd.x, ptWnd.y, szWnd.cx, szWnd.cy, SWP_NOACTIVATE);
		//	テキスト表示
		InvalidateRect(hWndTip, NULL, TRUE);
		//	ウインド表示
		ShowTipTextWindow();
	}

	AjcTipTextLeave();
}
//==============================================================================================================//
//	カーソルをチップテキスト上へ移動																			//
//																												//
//	引　数：	cm - 移動位置(AJCTIP_CM_CENT/LU/RU/LD/RD)														//
//																												//
//	戻り値：	なし																							//
//==============================================================================================================//
AJCEXPORT VO	 	WINAPI AjcTipTextMoveCursor(AJCTIP_CURMOVE cm)
{
	if (fShow) {
		switch (cm) {
			case AJCTIP_CM_CENT:	SetCursorPos(ptWnd.x + (szWnd.cx / 2) , ptWnd.y + (szWnd.cy / 2) ); break;	//	中心へ移動
			case AJCTIP_CM_LU:		SetCursorPos(ptWnd.x + 10			  , ptWnd.y + 10			 ); break;	//	左上へ移動
			case AJCTIP_CM_RU:		SetCursorPos(ptWnd.x + (szWnd.cx - 10), ptWnd.y + 10			 ); break;	//	右上へ移動
			case AJCTIP_CM_LD:		SetCursorPos(ptWnd.x + 10			  , ptWnd.y + (szWnd.cy - 10)); break;	//	左下へ移動
			case AJCTIP_CM_RD:		SetCursorPos(ptWnd.x + (szWnd.cx - 10), ptWnd.y + (szWnd.cy - 10)); break;	//	右下へ移動
		}
	}
}
//==============================================================================================================//
//	チップテキストの非表示																						//
//																												//
//	引　数：	なし																							//
//																												//
//	戻り値：	なし																							//
//==============================================================================================================//
AJCEXPORT VO	 WINAPI AjcTipTextHide(VO)
{
	AjcTipTextEnter();

	AjcTipTextHideEx(FALSE);

	AjcTipTextLeave();
}
//==============================================================================================================//
//	チップテキスト非表示（表示猶予指定）																		//
//																												//
//	引　数：	fDelay - 非表示遅延フラグ																		//
//							TRUE  : カーソル監視へ移行し、カーソルがチップウインド外ならば非表示				//
//							FALSE : 即時に非表示																//
//																												//
//	戻り値：	なし																							//
//==============================================================================================================//
AJCEXPORT VO	 WINAPI AjcTipTextHideEx(BOOL fDelay)
{
	AjcTipTextEnter();

	if (IsWindow(hWndTip)) {
		if (fShow) {
			//	表示タイマ停止
			KillTimer(hWndTip, TID_SHOW);
			if (fDelay) {
				//	カーソル位置監視タイマ起動
				SetTimer (hWndTip, TID_CHK_CURSOR, TM_CHK_CURSOR, NULL);
			}
			else {
				//	チップウインド非表示
				HideTipTextWindow();
			}
		}
	}

	AjcTipTextLeave();
}
//==============================================================================================================//
//	チップテキストウインド矩形情報取得																			//
//																												//
//	引　数：	pRect - チップテキストウインド矩形情報を格納するバッファのアドレス（不要時はNULL）				//
//																												//
//	戻り値：	TRUE  - 成功																					//
//				FALSE - 失敗（チップテキスト非表示中）															//
//==============================================================================================================//
AJCEXPORT BOOL	 WINAPI AjcTipTextGetRect(LPRECT pRect)
{
	BOOL	rc = FALSE;

	AjcTipTextEnter();

	if (fShow) {
		if (pRect != NULL) {
			GetWindowRect(hWndTip, pRect);
		}
		rc = TRUE;
	}

	AjcTipTextLeave();

	return rc;
}
//==============================================================================================================//
//	パレット色設定																								//
//																												//
//	引　数：	ix		- パレット番号（０～７）																//
//				color	- 設定する色																			//
//																												//
//	戻り値：	TRUE	- 成功																					//
//				FALSE	- 失敗																					//
//==============================================================================================================//
AJCEXPORT BOOL	 WINAPI AjcTipTextSetPalette(int ix, COLORREF color)
{
	BOOL	rc = FALSE;

	AjcTipTextEnter();

	if (ix >= 0 && ix <= 7) {
		AjcTipTextEnter();	//	スレッド排他制御開始
		Palette[ix] = color;
		rc = TRUE;
		AjcTipTextLeave();	//	スレッド排他制御終了
	}

	AjcTipTextLeave();

	return rc;
}
//==============================================================================================================//
//	パレット色取得																								//
//																												//
//	引　数：	ix		- パレット番号（０～７）																//
//																												//
//	戻り値：	パレット色																						//
//==============================================================================================================//
AJCEXPORT COLORREF	 WINAPI AjcTipTextGetPalette(int ix)
{
	COLORREF	rc = -1;
	AjcTipTextEnter();	//	スレッド排他制御開始
	if (ix >= 0 && ix <= 7) {
		rc = Palette[ix];
	}
	AjcTipTextLeave();	//	スレッド排他制御終了
	return rc;
}

//==============================================================================================================//
//	デフォルトのテキスト表示色設定																				//
//																												//
//	引　数：	color	- テキスト表示色																		//
//																												//
//	戻り値：	なし																							//
//==============================================================================================================//
AJCEXPORT VO	 WINAPI AjcTipTextSetDefTextColor(COLORREF color)
{
	AjcTipTextEnter();	//	スレッド排他制御開始
	DefTextColor = color;
	AjcTipTextLeave();	//	スレッド排他制御終了
}
//==============================================================================================================//
//	デフォルトのテキスト表示色取得																				//
//																												//
//	引　数：	なし																							//
//																												//
//	戻り値：	なし																							//
//==============================================================================================================//
AJCEXPORT COLORREF	 WINAPI AjcTipTextGetDefTextColor(VO)
{
	COLORREF	rc;
	AjcTipTextEnter();	//	スレッド排他制御開始
	rc = DefTextColor;
	AjcTipTextLeave();	//	スレッド排他制御終了
	return rc;
}
//==============================================================================================================//
//	デフォルトの外枠表示色設定																					//
//																												//
//	引　数：	color	- 外枠表示色																			//
//																												//
//	戻り値：	なし																							//
//==============================================================================================================//
AJCEXPORT VO	 WINAPI AjcTipTextSetDefBorderColor(COLORREF color)
{
	AjcTipTextEnter();	//	スレッド排他制御開始
	DefBorderColor = color;
	AjcTipTextLeave();	//	スレッド排他制御終了
}
//==============================================================================================================//
//	デフォルトの外枠表示色取得																					//
//																												//
//	引　数：	なし																							//
//																												//
//	戻り値：	外枠表示色																						//
//==============================================================================================================//
AJCEXPORT COLORREF	 WINAPI AjcTipTextGetDefBorderColor(VO)
{
	COLORREF	rc;
	AjcTipTextEnter();	//	スレッド排他制御開始
	rc = DefBorderColor;
	AjcTipTextLeave();	//	スレッド排他制御終了
	return rc;
}
//==============================================================================================================//
//	デフォルトのウインド背景色設定																				//
//																												//
//	引　数：	color	- ウインド背景色																		//
//																												//
//	戻り値：	なし																							//
//==============================================================================================================//
AJCEXPORT VO	 WINAPI AjcTipTextSetDefBkColor(COLORREF color)
{
	AjcTipTextEnter();	//	スレッド排他制御開始
	if (DefBkColor != color) {
		if (hDefBrush != NULL) {
			DeleteObject(hDefBrush);
		}
		DefBkColor = color;
		hDefBrush  = CreateSolidBrush(DefBkColor);
	}
	AjcTipTextLeave();	//	スレッド排他制御終了
}
//==============================================================================================================//
//	デフォルトのウインド背景色設定取得																			//
//																												//
//	引　数：	なし																							//
//																												//
//	戻り値：	なし																							//
//==============================================================================================================//
AJCEXPORT COLORREF	 WINAPI AjcTipTextGetDefBkColor(VO)
{
	COLORREF	rc;
	AjcTipTextEnter();	//	スレッド排他制御開始
	rc = DefBkColor;
	AjcTipTextLeave();	//	スレッド排他制御終了
	return rc;
}
//==============================================================================================================//
//	デフォルトフォントの設定																					//
//																												//
//	引　数：	hFont	- フォントハンドル																		//
//																												//
//	戻り値：	なし																							//
//==============================================================================================================//
AJCEXPORT VO	 WINAPI AjcTipTextSetDefFont(HFONT hFont)
{
	AjcTipTextEnter();	//	スレッド排他制御開始
	if (hDefFont	!= NULL) {
		DeleteObject(hDefFont);
		hDefFont = NULL; 
	}
	hDefFont = hFont;
	AjcTipTextLeave();	//	スレッド排他制御終了
}
//==============================================================================================================//
//	デフォルトフォントの取得																					//
//																												//
//	引　数：	hFont	- フォントハンドル																		//
//																												//
//	戻り値：	フォントハンドル																				//
//==============================================================================================================//
AJCEXPORT HFONT	 WINAPI AjcTipTextGetDefFont(VO)
{
	HFONT	rc;
	AjcTipTextEnter();	//	スレッド排他制御開始
	rc = hDefFont;
	AjcTipTextLeave();	//	スレッド排他制御終了
	return rc;
}
//==============================================================================================================//
//	デフォルト表示時間の設定																					//
//																												//
//	引　数：	msTime	- デフォルト表示時間																	//
//																												//
//	戻り値：	なし																							//
//==============================================================================================================//
AJCEXPORT VO	 WINAPI AjcTipTextSetDefMsShow(int msTime)
{
	AjcTipTextEnter();	//	スレッド排他制御開始
	DefMsShow = msTime;
	AjcTipTextLeave();	//	スレッド排他制御終了
}
//==============================================================================================================//
//	デフォルト表示時間の取得																					//
//																												//
//	引　数：	なし																							//
//																												//
//	戻り値：	デフォルト表示時間																				//
//==============================================================================================================//
AJCEXPORT int	 WINAPI AjcTipTextGetDefMsShow(VO)
{
	int		rc;
	AjcTipTextEnter();	//	スレッド排他制御開始
	rc = DefMsShow;
	AjcTipTextLeave();	//	スレッド排他制御終了
	return rc;
}
//==============================================================================================================//
//	デフォルト情報取得																							//
//																												//
//	引　数：	pTextColor	- テキスト描画色格納バッファ														//
//				pBackGround	- ウインド背景色格納バッファ														//
//				pBorderColor- 外枠描画色格納バッファ															//
//				pMsShow 	- 表示時間格納バッファ																//
//																												//
//	戻り値：	デフォルトのフォントハンドル																	//
//==============================================================================================================//
AJCEXPORT HFONT	 WINAPI AjcTipTextGetDefInfo(LPCOLORREF pTextColor, LPCOLORREF pBackGround, LPCOLORREF pBorderColor, int *pMsShow)
{
	if (pTextColor	 != NULL) *pTextColor	= RGB(0, 0, 0);
	if (pBackGround  != NULL) *pBackGround	= DefBkColor;
	if (pBorderColor != NULL) *pBorderColor = RGB(0, 0, 0);
	if (pMsShow 	 != NULL) *pMsShow		= DefMsShow;
	return	hDefFont;
}
//==============================================================================================================//
//	描画ビットマップイメージの登録																				//
//																												//
//	引　数	：	hBmp			- ビットマップハンドル															//
//				id				- 識別ＩＤ																		//
//				x, y, cx, cy	- デフォルトの描画位置															//
//				cx, cy			- デフォルトの描画サイズ（０：イメージサイズを採用）							//
//																												//
//	戻り値	：	TRUE   - ＯＫ																					//
//				FALSE  - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTipTextRegistBitmap	(HBITMAP hBmp, UI id, int x, int y, int cx, int cy)
{
	BOOL	rc = FALSE;

	rc = AjcTxoRegistBitmap	(hBmp, id, x, y, cx, cy);

	return rc;
}
//==============================================================================================================//
//	描画アイコンイメージの登録																					//
//																												//
//	引　数	：	hIcon			- アイコンハンドル																//
//				id				- 識別ＩＤ																		//
//				x, y, cx, cy	- デフォルトの描画位置															//
//				cx, cy			- デフォルトの描画サイズ（０：イメージサイズを採用）							//
//																												//
//	戻り値	：	TRUE   - ＯＫ																					//
//				FALSE  - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTipTextRegistIcon	(HICON hIcon, UI id, int x, int y, int cx, int cy)
{
	BOOL	rc = FALSE;

	rc = AjcTxoRegistIcon(hIcon, id, x, y, cx, cy);

	return rc;
}
//==============================================================================================================//
//	描画イメージの登録解除																						//
//																												//
//	引　数	：	id				- 識別ＩＤ																		//
//																												//
//	戻り値	：	TRUE   - ＯＫ																					//
//				FALSE  - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTipTextUnregistImage	(UI id)
{
	BOOL	rc = FALSE;

	rc = AjcTxoUnregistImage(id);

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	チップテキスト ウインドプロシージャ																			//
//																												//
//--------------------------------------------------------------------------------------------------------------//
//----- ウインド生成 -------------------------------------------------------------------------------------------//
AJC_WNDPROC(TipTxt, WM_CREATE	)
{
	SetLayeredWindowAttributes(hwnd, ColorKey, 0, LWA_COLORKEY);

	return 0;
}
//----- ウインド破棄 -------------------------------------------------------------------------------------------//
AJC_WNDPROC(TipTxt, WM_DESTROY	)
{
	//	コピーボタン破棄 
	if (hBtnCopy	!= NULL) {
		DestroyWindow(hBtnCopy);
		hBtnCopy = NULL;
	}
	//	チップテキスト解放
	if (pTipTxt != NULL) {free(pTipTxt); pTipTxt = NULL;}
	//	タイマ停止
	KillTimer(hwnd, TID_SHOW);
	KillTimer(hwnd, TID_CHK_CURSOR);
	//	ウインドハンドルクリアー
	hWndTip = NULL;

	return 0;
}
//----- タイマ -------------------------------------------------------------------------------------------------//
AJC_WNDPROC(TipTxt, WM_TIMER	)
{
	switch (wParam) {
		//	●表示時間タイマ
		case TID_SHOW:
			//	表示時間タイマ停止
			KillTimer(hwnd, TID_SHOW);
			//	カーソル位置監視タイマ起動
			SetTimer(hwnd, TID_CHK_CURSOR, TM_CHK_CURSOR, NULL);
			break;

		//	●カーソル位置監視タイマ
		case TID_CHK_CURSOR:
			//	カーソルがウインド外の場合、タイマ停止しウインド非表示
			if (!IsCursorOnWindow()) {
				KillTimer(hwnd, TID_CHK_CURSOR);
				HideTipTextWindow();
			}
			break;
	}
	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(TipTxt, WM_SETFOCUS	)
{
	//	直前のフォーカスウインド退避
	hWndFocus = (HWND)wParam;
	//	フォーカス移動用メッセージ送信（マウスのボタンイベントを有効とするため、メッセージ送出後にフォーカス移動する）
	PostMessage(hwnd, WM_SETFOCUS_DELAY, 0, 0);
	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(TipTxt, WM_SETFOCUS_DELAY)
{
	//	直前のウインドにフォーカスを戻す
	if (IsWindow(hWndFocus)) {
		SetFocus(hWndFocus);
		hWndFocus = NULL;
	}
	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(TipTxt, WM_LBUTTONDOWN	)
{
	//	（TOPMOSTウインド上でも）ツールチップウインドを前面に表示する為、一旦非表示後に再表示
	ShowWindow(hWndTip, SW_HIDE);
	ShowWindow(hWndTip, SW_SHOWNA);

	//	コピーボタン表示中ならば、コピーボタン非表示
	if (fShowBtnCopy) HideBtnCopy();
	//	コピーボタン非表示中ならば、ツールチップウインドとコピーボタン非表示
	else			  MouseButtonDown(hwnd);

	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(TipTxt, WM_RBUTTONDOWN	)
{
	//	（TOPMOSTウインド上でも）ツールチップウインドを前面に表示する為、一旦非表示後に再表示
	ShowWindow(hWndTip, SW_HIDE);
	ShowWindow(hWndTip, SW_SHOWNA);

	//	コピーボタン表示中ならば、コピーボタン非表示
	if (fShowBtnCopy) HideBtnCopy();
	//	コピーボタン非表示中ならば、ツールチップウインドとコピーボタン非表示
	else			  ShowBtnCopy();
	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(TipTxt, WM_MBUTTONDOWN	)
{
	//	WM_LBUTTONDOWNと同じ動作
	SendMessage(hwnd, WM_LBUTTONDOWN, wParam, lParam);

	return 0;
}
//----- ウインド描画 -------------------------------------------------------------------------------------------//
typedef struct {
	COLORREF	Border;
	COLORREF	BackGr;
} BBCOLOR, *PBBCOLOR;
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	BOOL CALLBACK cbNtcEsc(C_WCP pEsc, UI lEsc, WC cLast, UX cbp)
{
	PBBCOLOR	p	= (PBBCOLOR)cbp;
	C_WCP		pS	= pEsc;
	COLORREF	rgb = (COLORREF)-1;

	if (pS[0] == L'\x1B' && pS[1] == L'[' && (cLast == 'F' || cLast == L'B')) {
		pS += 2;
		if (AjcTxoGetRgbByText(pS, &rgb) == NULL) {
			if (MAjcAscIsDigitW(*pS)) {
				int	n = _wtoi(pS);
				rgb = Palette[n % 8];
			}
		}
		if (rgb != -1) {
			if (cLast == L'B') p->BackGr = rgb;
			else			   p->Border = rgb;
		}
	}
	return TRUE;
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJC_WNDPROC(TipTxt, WM_PAINT	)
{
	PAINTSTRUCT	ps;					//	ペイント構造体
	HDC			hdc;				//	デバイスコンテキスト
	HFONT		hF;					//	フォントハンドル退避
	HBRUSH		hSvBru, hBru;		//	ブラシハンドル
	HPEN		hSvPen, hPen;		//	ペンハンドル
	COLORREF	color;
	RECT		cr;
	FONTCHGINFO	Fi = {0};			//	フォント変更ワーク
	BBCOLOR		bbc;				//	ウインド背景色，外枠色
	HAJCTXO		hTxo;

	AjcTipTextEnter();

	hdc	  = BeginPaint(hwnd, &ps);
	GetClientRect(hwnd, &cr);
	//	フォント設定
	hF = (HFONT)SelectObject(hdc, Fi.hFontOrg = hTipFont);
	//	背景モード設定
	SetBkMode (hdc, TRANSPARENT);
	SetBkColor(hdc, RgbBkColor);
	//	文字色設定
	SetTextColor(hdc, RgbTextColor);
	//	ウインド背景色，外枠色設定初期化
	bbc.Border = RgbBorderColor;
	bbc.BackGr = RgbBkColor;

	//	ＥＳＣ列挙し、ウインド背景色，外枠色を設定
	AjcTxoEnumEscW(pTipTxt, (UX)&bbc, cbNtcEsc);

	//	背景色で塗りつぶし＆外枠描画
	hBru = CreateSolidBrush(bbc.BackGr);
	if (bbc.Border == -2) hPen = CreatePen(PS_SOLID, 1, bbc.BackGr);
	else				  hPen = CreatePen(PS_SOLID, 1, bbc.Border);
	hSvBru = (HBRUSH)SelectObject(hdc, hBru);
	hSvPen = (HPEN)SelectObject(hdc, hPen);
	Rectangle(hdc, cr.left, cr.top, cr.right, cr.bottom);
	DeleteObject(SelectObject(hdc, hSvBru));
	DeleteObject(SelectObject(hdc, hSvPen));
	//	四隅に丸みをつける（四隅に透明の正方形）
	hBru = CreateSolidBrush(	  ColorKey);
	hPen = CreatePen(PS_NULL , 1, ColorKey);
	hSvBru = (HBRUSH)SelectObject(hdc, hBru);
	hSvPen = (HPEN)SelectObject(hdc, hPen);
	Rectangle(hdc, cr.left	   , cr.top 	  , cr.left  + 4, cr.top	+ 4);
	Rectangle(hdc, cr.right - 3, cr.top 	  , cr.right + 1, cr.top	+ 4);
	Rectangle(hdc, cr.left	   , cr.bottom - 3, cr.left  + 4, cr.bottom + 1);
	Rectangle(hdc, cr.right - 3, cr.bottom - 3, cr.right + 1, cr.bottom + 1);
	DeleteObject(SelectObject(hdc, hSvBru));
	DeleteObject(SelectObject(hdc, hSvPen));
	//	四隅に丸みをつける（外枠）
	if (bbc.Border == -2) color = bbc.BackGr;
	else				  color = bbc.Border;
	//	左上															右上
	SetPixel(hdc, cr.left + 1, cr.top	 + 2, color);	SetPixel(hdc, cr.right - 3, cr.top	  + 1, color);
	SetPixel(hdc, cr.left + 2, cr.top	 + 1, color);	SetPixel(hdc, cr.right - 2, cr.top	  + 2, color);
	//	左下															右下
	SetPixel(hdc, cr.left + 1, cr.bottom - 3, color);	SetPixel(hdc, cr.right - 3, cr.bottom - 2, color);
	SetPixel(hdc, cr.left + 2, cr.bottom - 2, color);	SetPixel(hdc, cr.right - 2, cr.bottom - 3, color);
	//	四隅に丸みをつける（背景）
	//	左上															右上
	SetPixel(hdc, cr.left + 2, cr.top	 + 2, bbc.BackGr);	SetPixel(hdc, cr.right - 3, cr.top	  + 2, bbc.BackGr);
	//	左下															右下
	SetPixel(hdc, cr.left + 2, cr.bottom - 3, bbc.BackGr);	SetPixel(hdc, cr.right - 3, cr.bottom - 3, bbc.BackGr);
	//	単独ツールチップの場合、右上に 閉じるマーク(X)描画
	if (fCloseMark && SvhCaller == NULL) {
		hPen = (HPEN)SelectObject(hdc, CreatePen(PS_SOLID, 2, RGB(255, 0, 0)));
		MoveToEx(hdc, cr.right - 9, 3, NULL);
		LineTo	(hdc, cr.right - 4, 8);
		MoveToEx(hdc, cr.right - 4, 3, NULL);
		LineTo	(hdc, cr.right - 9, 8);
		DeleteObject(SelectObject(hdc, hPen));
	}
	//	文字列／イメージ描画
	if (hTxo = AjcTxoCreate(hdc, xTop, yTop, -20)) {					//	インスタンス生成（行間＝文字高さの２０％）
		RECT	r;
		GetWindowRect(hwnd, &r);
		AjcTxoSetDcBmpSize(hTxo, r.right - r.left, r.bottom - r.top);	//		ＤＣ描画スペースのビットマップサイズ設定
		AjcTxoSetDefTextColor(hTxo, RgbTextColor);						//		デフォルト文字色設定（リセット時のカラー）
		AjcTxoTextOut(hTxo, pTipTxt);									//		テキスト表示
		AjcTxoDelete(hTxo, TRUE);										//		インスタンス消去
	}

	EndPaint(hwnd, &ps);

	AjcTipTextLeave();

	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(TipTxt, WM_GETFONT	)
{
	HFONT		hFont;

	hFont = hTipFont;

	return (LRESULT)hFont;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(TipTxt, WM_SETFONT	)
{
	hTipFont = (HFONT)wParam;

	if (LOWORD(lParam) != 0) {
		InvalidateRect(hwnd, NULL, FALSE);
	}
	return 0;
}
//----- マウスボタン押下時の処理（共通）------------------------------------------------------------------------//
static	VO	MouseButtonDown(HWND hwnd)
{
	//	タイマ停止しウインド非表示
	KillTimer(hwnd, TID_SHOW	  );
	KillTimer(hwnd, TID_CHK_CURSOR);
	HideTipTextWindow();
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDMAP_DEF(TipTxt)
	AJC_WNDMAP_MSG(TipTxt, WM_CREATE		)
	AJC_WNDMAP_MSG(TipTxt, WM_DESTROY		)
	AJC_WNDMAP_MSG(TipTxt, WM_TIMER			)
	AJC_WNDMAP_MSG(TipTxt, WM_SETFOCUS		)
	AJC_WNDMAP_MSG(TipTxt, WM_SETFOCUS_DELAY)
	AJC_WNDMAP_MSG(TipTxt, WM_LBUTTONDOWN	)
	AJC_WNDMAP_MSG(TipTxt, WM_RBUTTONDOWN	)
	AJC_WNDMAP_MSG(TipTxt, WM_MBUTTONDOWN	)
	AJC_WNDMAP_MSG(TipTxt, WM_PAINT			)
	AJC_WNDMAP_MSG(TipTxt, WM_GETFONT		)
	AJC_WNDMAP_MSG(TipTxt, WM_SETFONT		)
AJC_WNDMAP_END
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	コピーボタン ウインドプロシージャ（サブクラス）																//
//																												//
//--------------------------------------------------------------------------------------------------------------//
#define		WM_COPYTIPIMAGE		(WM_APP + 100)
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(BtnCopy, WM_LBUTTONDOWN	)
{
	HGLOBAL	hGlobal;

	//	コピーボタン非表示
	HideBtnCopy();

	//	SHIFT, CTRL 未押下時は、チップテキストをクリップボードへコピー
	if		((wParam & (MK_SHIFT | MK_CONTROL)) == 0) {
		int		len;
		WCP		pTxt;
		if (len = (int)wcslen(pTipTxt)) {
			len++;
			if (hGlobal = GlobalAlloc(GHND | GMEM_SHARE, len * 2 + 2)) {
				pTxt = (WCP)GlobalLock(hGlobal);
				AjcRemoveEscInStrW(pTipTxt, pTxt, len);
				GlobalUnlock(hGlobal);
				OpenClipboard(hwnd);
				EmptyClipboard();
				SetClipboardData(CF_UNICODETEXT, hGlobal);
				CloseClipboard();
			}
		}
		//	チップウインド再表示
		ShowTipTextWindow();
	}
	//	SHIFT押下時は、チップイメージをクリップボードへコピー
	else if ((wParam & (MK_SHIFT | MK_CONTROL)) == MK_SHIFT) {
		//	ボタン部分の影ができる為、イメージのコピーを遅延させる
		SetTimer(hwnd, 100, 1, NULL);
	}
	return 0;	//	インターセプト
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(BtnCopy, WM_RBUTTONDOWN	)
{
	//	チップウインド再表示
	ShowTipTextWindow();

	return 0;	//	インターセプト
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(BtnCopy, WM_MBUTTONDOWN	)
{
	//	チップウインド再表示
	ShowTipTextWindow();

	return 0;	//	インターセプト
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(BtnCopy, WM_TIMER		)
{
	HBITMAP	hBmp = NULL;
	HBITMAP	hSav = NULL;
	HDC		hdc;
	HDC		hmd;
	RECT	r;

	KillTimer(hwnd, 100);

	if (hdc  = GetDC(hWndTip)) {
		if (hmd  = CreateCompatibleDC(hdc)) {
			if (hBmp = CreateCompatibleBitmap(hdc, szWnd.cx, szWnd.cy)) {
				//	ビットマップ作成
				hSav = (HBITMAP)SelectObject(hmd, hBmp);
				SetRect(&r, 0, 0, szWnd.cx, szWnd.cy);
				FillRect(hmd, &r, (HBRUSH)GetStockObject(WHITE_BRUSH));
				BitBlt(hmd, 0, 0, szWnd.cx, szWnd.cy, hdc, 0, 0, SRCCOPY);
				hBmp = (HBITMAP)SelectObject(hmd, hSav);
				//	クリップボードへ設定
				OpenClipboard(hwnd);
				EmptyClipboard();
				SetClipboardData(CF_BITMAP, hBmp);
				CloseClipboard();
			}
			DeleteDC(hmd);
		}
		ReleaseDC(hWndTip, hdc);
		//	チップウインド再表示
		ShowTipTextWindow();
	}
	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDMAP_DEF(BtnCopy)
	AJC_WNDMAP_MSG(BtnCopy, WM_LBUTTONDOWN	)
	AJC_WNDMAP_MSG(BtnCopy, WM_RBUTTONDOWN	)
	AJC_WNDMAP_MSG(BtnCopy, WM_MBUTTONDOWN	)
	AJC_WNDMAP_MSG(BtnCopy, WM_TIMER		)
AJC_WNDMAP_END
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	チップテキストウインド表示（対象が TOPMOST ウインドでも、その上に表示するために 一旦非表示後に再表示する）	//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	VO	ShowTipTextWindow(VO)
{
	ShowWindow(hWndTip, SW_HIDE);
	ShowWindow(hWndTip, SW_SHOWNA);
	fShow = TRUE;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	チップテキストウインド非表示																				//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	VO	HideTipTextWindow(VO)
{
	if (fShow) {
		//	コピーボタン非表示
		HideBtnCopy();
		//	チップテキストウインド非表示
		ShowWindow(hWndTip, SW_HIDE);
		fShow = FALSE;
	}
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	コピーボタン表示																							//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	VO	ShowBtnCopy(VO)
{
	int		bw, bh;
	int		x, y;
	int		len;
	POINT	pt;

	//	フォント設定
	len = (int)SendMessage(hBtnCopy, WM_GETTEXTLENGTH, 0, 0) + 2;
	if (cyChar > 12) {SendMessage(hBtnCopy, WM_SETFONT, (WPARAM)hDefFontVar12, 0); bh = 12	   + 4; bw = 6		* len + 4;}
	else			 {SendMessage(hBtnCopy, WM_SETFONT, (WPARAM)hTipFont	 , 0); bh = cyChar + 4; bw = cxChar * len + 4;}
	//	表示位置設定
	GetCursorPos(&pt); pt.x += 10;
	MapWindowPoints(NULL, hWndTip, &pt, 1);
	if (pt.x + bw > szWnd.cx) x = szWnd.cx - (bw + 1);
	else					  x = pt.x;
	if (pt.y + bh > szWnd.cy) y = szWnd.cy - (bh + 1);
	else					  y = pt.y;
	SetWindowPos(hBtnCopy, NULL, x, y, bw, bh, SWP_NOZORDER);
	//	コピーボタン表示
	ShowWindow(hBtnCopy, SW_SHOW);
	fShowBtnCopy = TRUE;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	コピーボタン非表示																							//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	VO	HideBtnCopy(VO)
{
	//	コピーボタン非表示
	ShowWindow(hBtnCopy, SW_HIDE);
	fShowBtnCopy = FALSE;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	ウインド表示位置補正	(pPt:表示位置，pSz:ウインドサイズ）													//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	VO	AjustWndPos(LPPOINT pPt, LPSIZE pSz)
{
	UI				i, nMon;
	AJCMONITORSINFO	ifMon;
	RECT			rcMon;				//	モニタ矩形ワーク
	RECT			rcWnd;				//	ウインド矩形
	int				sqr, sqrMax = 0;

	do {
		//	ウインド矩形情報設定
		rcWnd.left = pPt->x;	rcWnd.right  = pPt->x + pSz->cx;
		rcWnd.top  = pPt->y;	rcWnd.bottom = pPt->y + pSz->cy;
		//	マルチモニタ矩形情報取得
		AjcGetMonitorsRect(&rcMon);
		//	各モニタ矩形情報取得
		nMon = AjcGetMonitorsInfo(&ifMon);
		//	所属するモニタを検索
		for (i = 0; i < nMon; i++) {
			sqr = AjcGetDupRect(&rcWnd, &ifMon.rcMon[i], NULL);
			if (sqr > sqrMax) {
				sqrMax = sqr;
				memcpy(&rcMon, &ifMon.rcMon[i], sizeof rcMon);
			}
		}
		//	所属するモニタが無ければ中止
		if (sqrMax == 0) break;
		//	ウインドが右端を出る場合、Ｘを左に移動
		if (pPt->x + pSz->cx > rcMon.right) {
			pPt->x -= (pPt->x + pSz->cx - rcMon.right);
		}
		//	ウインドが左端を出る場合、Ｘを右に移動
		if (pPt->x < rcMon.left) {
			pPt->x += (rcMon.left - pPt->x);
		}
		//	ウインドが下端を出る場合、Ｙを上に移動
		if (pPt->y + pSz->cy > rcMon.bottom) {
			pPt->y -= (pPt->y + pSz->cy - rcMon.bottom);
		}
		//	ウインドが上端を出る場合、Ｙを下に移動
		if (pPt->y < rcMon.top) {
			pPt->y += (rcMon.top - pPt->y);
		}
	} while(0);
}

//--------------------------------------------------------------------------------------------------------------//
//																												//
//	カーソル位置チェック（TRUE:カーソルがウインド上にある，FALSE:カーソルはウインド外）							//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static BOOL	IsCursorOnWindow(VO)
{
	BOOL	rc = FALSE;
	POINT	pt;
	RECT	r;

	if (IsWindow(hWndTip)) {
		GetCursorPos(&pt);
		GetWindowRect(hWndTip, &r);
		if (pt.x >= r.left && pt.x < r.right && pt.y >= r.top && pt.y < r.bottom) {
			rc = TRUE;
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	文字サイズ取得																								//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	GetCharSize(HDC hdc, int *pWidth, int *pHeight)
{
	BOOL	rc;
	SIZE	cw;

	//	'W' の文字サイズ設定（この文字の高さを cyChar とする）
	cw.cx = 0;
	if (rc = (GetTextExtentPoint32W(hdc, L"W", 1, &cw) != 0)) {
		*pWidth  = cw.cx;
		*pHeight = cw.cy;
	}
	else {
		//	万が一の為、エラー時は適当な値をあてがう
		*pWidth  = 10;
		*pHeight = 20;
	}

	return rc;
}
