﻿#include	"AjcInternal.h"
#include	"AjcCtrl3dGraphicDef.h"
//**************************************************************************************************************//
//																												//
//	時計表示																									//
//																												//
//**************************************************************************************************************//
#define		WSX			200		//	時計ウインド幅
#define		WSY			200		//	時計ウインド高さ

#define		SSX			13		//	軸ウインドの幅
#define		SSY			13		//	軸ウインドの高さ

//==============================================================================================================//
//	現在時刻の設定／取得変数																					//
//==============================================================================================================//

//--------------------------------------------------------------------------------------------------------------//
//	作業領域																									//
//--------------------------------------------------------------------------------------------------------------//
static	ATOM		WndCls		= 0;				//	時計ウインドクラス
static	ATOM		SftCls		= 0;				//	軸ウインドクラス
static	COLORREF	BkGndColor	= RGB(0, 0, 255);	//	背景色
static	HBRUSH		hBruBkGnd	= NULL;				//	背景ブラシ	
static	HBRUSH		hBruCent	= NULL;				//	中心描画ブラシ
static	HPEN		hPenGray	= NULL;				//	灰色ペン
static	HPEN		hPenRed		= NULL;				//	赤色ペン
static	int			vcx = 98;						//	時計中心Ｘ	
static	int			vcy = 99;						//	時計中心Ｙ	
static	SL			TimeAdd		= 0;				//	時計を進める秒数

//--------------------------------------------------------------------------------------------------------------//
//	ノード																										//
//--------------------------------------------------------------------------------------------------------------//
typedef struct {
	UI		id;
	UI		state;
	WCP		pSect;
} NODE, *PNODE;

//--------------------------------------------------------------------------------------------------------------//
//	インスタンスワーク																							//
//--------------------------------------------------------------------------------------------------------------//
#define	WAT_INST_ID	0x716AC5DB

typedef struct {
	UI			InstID;					//	インスタンス識別ＩＤ
	UI			state;					//	現在の表示状態
	POINT		pt;						//	時計ウインド位置
	HAJCXQUE	hXQue;					//	表示状態退避キュー
	HWND		hOwner;					//	オーナーウインドハンドル
	HWND		hWnd;					//	時計ウインドハンドル
	HWND		hShaft;					//	中心軸ウインド
	WCP			pSect;					//	プロファイルセクション名

	BOOL		fCapture;				//	キャプチャ中フラグ
	BOOL		fChgSecHand;			//	秒針表示設定の直後を示すフラグ
	RECT		rcOrg;					//	ウインド移動開始時のウインド矩形
	POINT		ptOrg;					//	ウインド移動開始時のマウス位置
	UI			TmLbDown;				//	左ボタン押下時の時刻[ms]
	UI			   hh,	  mm,	 ss;
	UI			sv_hh, sv_mm, sv_ss;
} WRKWATCH, *PWRKWATCH;
//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC_DEF(Watch);
AJC_WNDPROC_DEF(Shaft);
static	BOOL	SubGetLocalTime	 (LPSYSTEMTIME pLt);
static	VO		SaveState		 (PNODE pNode);
static	VO		SaveAllState	 (PWRKWATCH pW);
static	VO		SaveWndPos		 (PWRKWATCH pW);
static	VO		ReleaseAll		 (PWRKWATCH pW);
static	PNODE	FindId			 (PWRKWATCH pW, UI id, BOOL *pfTop);
static	BOOL	CreateWatchWindow(PWRKWATCH pW, UI id, UI state);
static	VO		DestroyWatchWindow(PWRKWATCH pW);
static	HRGN	MakeRgnWatch	 (VO);

static	VO		WatchRotate		 (PWRKWATCH pW, double rot, POINT pt[], int n, int vcx, int vcy);

//==============================================================================================================//
//	起動時初期設定																								//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
BOOL	AjcWatInit(VO)
{
	BOOL		rc	= FALSE;
	WNDCLASSW	wndclass;

	do {
		//	ブラシ生成
		if ((hBruBkGnd = CreateSolidBrush(BkGndColor       )) == NULL) break;
		if ((hBruCent  = CreateSolidBrush(RGB(255, 127, 39))) == NULL) break;
		//	ペン生成
		if ((hPenGray  = CreatePen(PS_SOLID, 1, RGB(192, 192, 192))) == NULL) break;
		if ((hPenRed   = CreatePen(PS_SOLID, 1, RGB(255,   0,   0))) == NULL) break;

		//	時計クラス生成
		wndclass.style			= CS_NOCLOSE;
		wndclass.lpfnWndProc	= AJC_WNDPROC_NAME(Watch);
		wndclass.cbClsExtra		= 0;
		wndclass.cbWndExtra		= sizeof(PWRKWATCH);
		wndclass.hInstance		= hDllInst;
		wndclass.hIcon			= NULL;
		wndclass.hCursor		= LoadCursor(NULL, IDC_ARROW);;
		wndclass.hbrBackground	= hBruBkGnd;
		wndclass.lpszMenuName	= NULL;
		wndclass.lpszClassName	= L"_AjcWatchClass_";
		if ((WndCls = RegisterClassW(&wndclass)) == 0) break;

		//	軸クラス生成
		wndclass.style			= CS_NOCLOSE;
		wndclass.lpfnWndProc	= AJC_WNDPROC_NAME(Shaft);
		wndclass.cbClsExtra		= 0;
		wndclass.cbWndExtra		= sizeof(PWRKWATCH);
		wndclass.hInstance		= hDllInst;
		wndclass.hIcon			= NULL;
		wndclass.hCursor		= LoadCursor(NULL, IDC_ARROW);;
		wndclass.hbrBackground	= hBruCent;
		wndclass.lpszMenuName	= NULL;
		wndclass.lpszClassName	= L"_AjcShaftClass_";
		if ((SftCls = RegisterClassW(&wndclass)) == 0) break;


		rc = TRUE;
	} while(0);

	return rc;
}
//==============================================================================================================//
//	終了時後処理																								//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
VO		AjcWatEnd (VO)
{
	if (WndCls	  != 0	 ) UnregisterClass((UTP)WndCls, hDllInst);
	if (SftCls	  != 0	 ) UnregisterClass((UTP)SftCls, hDllInst);
	if (hBruBkGnd != NULL) DeleteObject(hBruBkGnd);
	if (hBruCent  != NULL) DeleteObject(hBruCent );
	if (hPenGray  != NULL) DeleteObject(hPenGray );
	if (hPenRed   != NULL) DeleteObject(hPenRed  );

}
//==============================================================================================================//
//	インスタンス生成																							//
//																												//
//	引　数	：	hWOner		- オーナーウインドハンドル															//
//				dummy		- 未使用																			//
//				pSect		- プロファイルセクション															//
//																												//
//	戻り値	：	≠NULL：成功（インスタンスハンドル）															//
//				＝NULL：失敗																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	HANDLE	WINAPI	AjcWatCreateA(HWND hOwner, UI dummy, C_BCP pSect)
{
	HANDLE	rc = NULL;
	UI		len;
	WCP		wcp;

	if (pSect != NULL) {
		len = AjcByteCharToWideChar(pSect, NULL, 0);
		if (wcp = AjcTAllocW(len)) {
			AjcByteCharToWideChar(pSect, wcp, len);
			rc = AjcWatCreateW(hOwner, dummy, wcp);
			free(wcp);
		}
	}
	else {
		rc = AjcWatCreateW(hOwner, dummy, NULL);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	HANDLE	WINAPI	AjcWatCreateW(HWND hOwner, UI dummy, C_WCP pSect)
{
	HANDLE		rc = NULL;
	PWRKWATCH	pW = NULL;
	RECT		mrc;
	int			w, h;


	if (pW = (PWRKWATCH)AJCMEM(sizeof(WRKWATCH))) {
		memset(pW, 0, sizeof(WRKWATCH));
		do {
			//	表示状態退避キュー生成
			pW->InstID = WAT_INST_ID;
			pW->state  = 0;
			pW->hOwner = hOwner;
			//	プライマリモニタの中央位置設定
			AjcGetPrimaryMonitorInfo(NULL, &mrc);
			w = mrc.right - mrc.left;
			h = mrc.bottom - mrc.top;
			pW->pt.x = (w - WSX) / 2;
			pW->pt.y = (h - WSY) / 2;
			//	プロファイルセクション名退避と、プロファイル読み出し
			if (pSect != NULL) {
				RECT		mrc;		//	モニタ矩形
				POINT		ptC;		//	ウインド中心
				UI			len;

				//	セクション名退避
				len = (UI)wcslen(pSect);
				if (pW->pSect = (WCP)AjcTAllocW(len + 1)) {
					wcscpy(pW->pSect, pSect);
				}
				else break;
				//	ウインド位置ロード
				pW->pt.x  = AjcGetProfileUInt(pW->pSect, L"AjcWatWndX" , 0);
				pW->pt.y  = AjcGetProfileUInt(pW->pSect, L"AjcWatWndY" , 0);
				//	ポイントが属するモニタ情報取得
				AjcGetMonitorInfoOfPoint(pW->pt.x, pW->pt.y, NULL, &mrc);
				ptC.x = pW->pt.x + (WSX / 2);
				ptC.y = pW->pt.y + (WSY / 2);
				//	ウインド中心がモニタ右端を出る場合、Ｘを左に移動
				if (ptC.x > mrc.right) {
					pW->pt.x -= (ptC.x - mrc.right);
				}
				//	ウインド中心がモニタ左端を出る場合、Ｘを右に移動
				if (ptC.x < mrc.left) {
					pW->pt.x += (mrc.left - ptC.x);
				}
				//	ウインド中心がモニタ下端を出る場合、Ｙを上に移動
				if (ptC.y > mrc.bottom) {
					pW->pt.y -= (ptC.y - mrc.bottom);
				}
				//	ウインド中心がモニタ上端を出る場合、Ｙを下に移動
				if (ptC.y < mrc.top) {
					pW->pt.y += (mrc.top - ptC.y);
				}
			}
			if ((pW->hXQue = AjcXQueCreate(0, NULL)) == NULL) break;
			//	戻り値設定
			rc = (HANDLE)pW;

		} while(0);

		//	失敗ならばリソース開放
		if (rc == NULL) {
			ReleaseAll(pW);
			free(pW);
			pW = NULL;
		}
	}
	return rc;
}
//==============================================================================================================//
//	インスタンス消去																							//
//																												//
//	引　数	：	hWat		- インスタンスハンドル																//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：失敗																						//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcWatDelete(HANDLE hWat)
{
	PWRKWATCH	pW = (PWRKWATCH)hWat;
	BOOL		rc = FALSE;

	if (pW != NULL && pW->InstID == WAT_INST_ID) {
		//	全ノード破棄し、プロファイルへ表示ステータス書き込み
		SaveAllState(pW);
		//	ウインド位置セーブ，セクションポインタ解放
		SaveWndPos(pW);
		//	ウインド破棄
		if (pW->hWnd != NULL) {
			DestroyWatchWindow(pW);
		}
		//	リソース開放
		ReleaseAll(pW);
		free(pW);
		pW = NULL;

		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	時計表示																									//
//																												//
//	引　数	：	hWat		- インスタンスハンドル																//
//				id			- 識別ＩＤ																			//
//				state		- 表示ステータス（ー１：無効）														//
//				pSect		- プロファイルセクション名（不要時はNULL）											//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcWatShowA(HANDLE hWat, UI id, UI state, C_BCP pSect)
{
	BOOL	rc = FALSE;
	UI		len;
	WCP		wcp;

	if (pSect != NULL) {
		len = AjcByteCharToWideChar(pSect, NULL, 0);
		if (wcp = AjcTAllocW(len)) {
			AjcByteCharToWideChar(pSect, wcp, len);
			rc = AjcWatShowW(hWat, id, state, wcp);
			free(wcp);
		}
	}
	else {
		rc = AjcWatShowW(hWat, id, state, NULL);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcWatShowW(HANDLE hWat, UI id, UI state, C_WCP pSect)
{
	PWRKWATCH	pW = (PWRKWATCH)hWat;
	BOOL		rc = FALSE;
	NODE		node;
	PNODE		pNode	= NULL;
	UTP			pTmp	= NULL;
	UI			len;

	if (pW != NULL && pW->InstID == WAT_INST_ID) {
		//	表示ステータス設定
		if (pSect != NULL) {
			pW->state = AjcGetProfileUInt(pSect, L"AjcWatState", state);
		}
		else pW->state = state;

		//	新時計ウインド生成
		if (CreateWatchWindow(pW, id, pW->state)) {
			do {
				//	ノードデータ作成
				node.id    = id;
				node.state = pW->state;
				if (pSect != NULL) {
					len = (UI)wcslen(pSect);
					if (pTmp = AjcTAllocW(len + 1)) {
						MAjcStrCpyW(pTmp, len + 1, pSect);
						node.pSect = pTmp;
						pTmp = NULL;
					}
					else break;
				}
				else node.pSect = NULL;

				//	同一ＩＤのノード削除
				if (pNode = FindId(pW, id, NULL)) {
					if (pNode->pSect != NULL) {
						AjcTFree(pNode->pSect);
					}
					AjcXQueRemove(pW->hXQue, pNode);
				}

				//	新ノード エンキュー
				if (AjcXQueEnqTop(pW->hXQue, &node, sizeof(node))) {
					rc = TRUE;
				}
			} while(0);
		}
	}
	//	リソース開放
	if (pTmp != NULL) free(pTmp);

	return rc;
}
//==============================================================================================================//
//	現在表示中の時計を非表示																					//
//																												//
//	引　数	：	hWat		- インスタンスハンドル																//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：失敗																						//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcWatHide(HANDLE hWat)
{
	PWRKWATCH	pW = (PWRKWATCH)hWat;
	BOOL		rc = FALSE;
	NODE		node;
	PNODE		pNode;

	if (pW != NULL && pW->InstID == WAT_INST_ID) {
		//	先頭ノード取り出し
		if (AjcXQueDeque(pW->hXQue, &node, sizeof(node)) != -1) {
			//	ステータスセーブ，セクションポインタ破棄
			SaveState(&node);
			//	先頭ノードポインタ取得
			if (pNode = (PNODE)AjcXQueTopNode(pW->hXQue, NULL)) {
				//	TOPMOST/TRANSPARENT表示状態が異なる場合、ウインド再生成
				if ((node.state & (AJCWATSS_TOPMOST | AJCWATSS_TRANSPARENT)) != (pNode->state & (AJCWATSS_TOPMOST | AJCWATSS_TRANSPARENT))) {
					//	ウインド再生成
					if (CreateWatchWindow(pW, pNode->id, pNode->state)) {
						//	表示状態，ウインドハンドル設定
						pW->state = pNode->state;
						rc = TRUE;
					}
				}
				//	TOPMOST表示状態が同じならば、表示状態設定
				else {
					pW->state = pNode->state;
					InvalidateRect(pW->hWnd, NULL, TRUE);
					rc = TRUE;
				}
				//	タイマ周期変更
				if ((node.state & AJCWATSS_SECHAND) != (pW->state & AJCWATSS_SECHAND)) {
					pW->fChgSecHand = TRUE;
					if (pW->state & AJCWATSS_SECHAND) SetTimer(pW->hWnd, 1,   300, NULL);
					else							  SetTimer(pW->hWnd, 1, 10000, NULL);
				}
			}
			//	ノード無しならば、ウインド破棄
			else {
				if (pW->hWnd != NULL) {
					DestroyWatchWindow(pW);
				}
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	時計非表示（ＩＤ指定）																						//
//																												//
//	引　数	：	hWat		- インスタンスハンドル																//
//				id			- 識別ＩＤ																			//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：失敗																						//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcWatHideByID(HANDLE hWat, UI id)
{
	PWRKWATCH	pW	  = (PWRKWATCH)hWat;
	BOOL		rc	  = FALSE;
	PNODE		pNode = NULL;
	BOOL		fTop  = FALSE;

	if (pW != NULL && pW->InstID == WAT_INST_ID) {
		if (pNode = FindId(pW, id, &fTop)) {
			//	先頭ノードならば、次ノード表示（表示中の時計を非表示）
			if (fTop) {
				AjcWatHide(hWat);
			}
			//	先頭ノード以外ならば、当該ノードエントリ削除
			else {
				//	ステータスセーブ，セクションポインタ破棄
				SaveState(pNode);
				//	ノード消去
				AjcXQueRemove(pW->hXQue, pNode);
			}
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	全時計ウインド非表示																						//
//																												//
//	引　数	：	hWat		- インスタンスハンドル																//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：失敗																						//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcWatHideAll(HANDLE hWat)
{
	PWRKWATCH	pW = (PWRKWATCH)hWat;
	BOOL		rc = FALSE;

	if (pW != NULL && pW->InstID == WAT_INST_ID) {
		//	マウスキャプチャ終了
		if (pW->fCapture) {
			ReleaseCapture();
			pW->fCapture = FALSE;
		}
		//	全ノード取り出し、プロファイルへ表示ステータス書き込み，セクションポインタ破棄
		SaveAllState(pW);
		//	時計ウインド破棄
		DestroyWatchWindow(pW);

		rc = TRUE;
	}
	return rc;
}

//==============================================================================================================//
//	表示状態設定																								//
//																												//
//	引　数	：	hWat		- インスタンスハンドル																//
//				state		- 表示状態																			//
//																												//
//	戻り値	：	≠-1：成功（前の表示状態）																		//
//				＝-1：失敗																						//
//==============================================================================================================//
AJCEXPORT	UI		WINAPI	AjcWatSetShowState(HANDLE hWat, UI id, UI state)
{
	PWRKWATCH	pW = (PWRKWATCH)hWat;
	UI			rc		  = -1;
	PNODE		pNode	  = NULL;
	BOOL		fTop	  = FALSE;
	UI			svs;

	if (pW != NULL && pW->InstID == WAT_INST_ID) {
		//	ノードポインタ取得
		if (pNode = FindId(pW, id, &fTop)) {
			//	先頭ノード（現在表示中）ならば、再表示
			if (fTop) {
				//	現表示状態退避
				svs = pW->state;
				//	TOPMOST/TRANSPARENT表示状態が異なる場合、ウインド再生成
				if ((pNode->state & (AJCWATSS_TOPMOST | AJCWATSS_TRANSPARENT)) != (state & (AJCWATSS_TOPMOST | AJCWATSS_TRANSPARENT))) {
					//	ウインド再生成
					if (CreateWatchWindow(pW, id, state)) {
						//	戻り値＝前の表示状態
						rc = pNode->state;
						//	表示状態，ウインドハンドル設定
						pW->state = pNode->state = state;
					}
				}
				//	TOPMOST&TRANSPARENT表示状態が同じ場合、表示状態設定
				else {
					//	戻り値＝前の表示状態
					rc = pW->state;
					//	表示状態設定
					pW->state = pNode->state = state;
					//	再描画
					InvalidateRect(pW->hWnd, NULL, TRUE);
				}
				//	タイマ周期変更
				if ((svs & AJCWATSS_SECHAND) != (pW->state & AJCWATSS_SECHAND)) {
					pW->fChgSecHand = TRUE;
					if (pW->state & AJCWATSS_SECHAND) SetTimer(pW->hWnd, 1,   300, NULL);
					else							  SetTimer(pW->hWnd, 1, 10000, NULL);
				}
			}
			//	先頭ノード以外（現在非表示中）ならば、ノードへステータス設定
			else {
				pNode->state = state;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	表示状態取得																								//
//																												//
//	引　数	：	hWat		- インスタンスハンドル																//
//				id			- 識別ＩＤ																			//
//																												//
//	戻り値	：	≠-1：成功（現在の表示状態）																	//
//				＝-1：失敗																						//
//==============================================================================================================//
AJCEXPORT	UI		WINAPI	AjcWatGetShowState(HANDLE hWat, UI id)
{
	PWRKWATCH	pW = (PWRKWATCH)hWat;
	UI			rc = -1;
	PNODE		pNode;
	BOOL		fTop  = FALSE;

	if (pW != NULL && pW->InstID == WAT_INST_ID) {
		if (pNode = FindId(pW, id, &fTop)) {
			rc = pNode->state;
		}
	}
	return rc;
}
//==============================================================================================================//
//	時計ウインドのハンドル取得																					//
//																												//
//	引　数	：	hWat		- インスタンスハンドル																//
//																												//
//	戻り値	：	≠NULL：成功（時計ウインドのハンドル）															//
//				＝NULL：失敗																					//
//==============================================================================================================//
AJCEXPORT	HWND	WINAPI	AjcWatGetHWnd(HANDLE hWat)
{
	PWRKWATCH	pW = (PWRKWATCH)hWat;
	HWND		rc = NULL;

	if (pW != NULL && pW->InstID == WAT_INST_ID) {
		rc = pW->hWnd;
	}
	return rc;
}
//==============================================================================================================//
//	加算秒数の設定（現在時刻の偽装）																			//
//																												//
//	引　数：	sec		- 加算秒数																				//
//																												//
//	戻り値：	なし																							//
//==============================================================================================================//
AJCEXPORT VO	 WINAPI AjcWatAdvance(SL sec)
{
	TimeAdd = sec;
}
//--------------------------------------------------------------------------------------------------------------//
//	現在時刻の取得（ローカルタイム）																			//
//																												//
//	引　数：	pLt		- 現在時刻を格納するバッファ															//
//																												//
//	戻り値：	なし																							//
//--------------------------------------------------------------------------------------------------------------//
static BOOL	 SubGetLocalTime(LPSYSTEMTIME pLt)
{
	BOOL	rc = FALSE;
	SL		ct;

	if (pLt != NULL) {
		GetLocalTime(pLt);							//	実際の現在時刻取得
		ct   = (SL)AjcSysTimeToTime1970(pLt);
		ct  += TimeAdd;								//	秒数加算
		AjcTime1970ToSysTime((UI)ct, pLt);			//	進めた時刻設定
		rc = TRUE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	表示状態 破棄，セーブ																						//
//--------------------------------------------------------------------------------------------------------------//
static	VO	SaveState(PNODE pNode)
{
	if (pNode->pSect != NULL) {
		AjcPutProfileUInt(pNode->pSect, L"AjcWatState" , pNode->state);
		AjcTFree((VOP)pNode->pSect);
	}
}
//--------------------------------------------------------------------------------------------------------------//
//	全ての表示状態 破棄，セーブ																					//
//--------------------------------------------------------------------------------------------------------------//
static	VO	SaveAllState(PWRKWATCH pW)
{
	NODE	node;

	while (AjcXQueDeque(pW->hXQue, &node, sizeof(node)) != -1) {
		SaveState(&node);
	}
}
//--------------------------------------------------------------------------------------------------------------//
//	ウインド位置セーブ																							//
//--------------------------------------------------------------------------------------------------------------//
static	VO	SaveWndPos(PWRKWATCH pW)
{
	RECT	wrc;

	if (IsWindow(pW->hWnd) && pW->pSect != NULL) {
		GetWindowRect(pW->hWnd, &wrc);
		AjcPutProfileSInt(pW->pSect, L"AjcWatWndX", wrc.left);
		AjcPutProfileSInt(pW->pSect, L"AjcWatWndY", wrc.top);
	}
}
//--------------------------------------------------------------------------------------------------------------//
//	リソース開放																								//
//--------------------------------------------------------------------------------------------------------------//
static	VO	ReleaseAll(PWRKWATCH pW)
{
	if (pW != NULL) {
		if (pW->pSect  != NULL) AjcTFree(pW->pSect);
		if (pW->hXQue  != NULL) AjcXQueDelete(pW->hXQue);
	}
}
//--------------------------------------------------------------------------------------------------------------//
//	ＩＤ検索																									//
//--------------------------------------------------------------------------------------------------------------//
static	PNODE	FindId(PWRKWATCH pW, UI id, BOOL *pfTop)
{
	PNODE	pNode = NULL;
	PNODE	pTop  = NULL;

	//	先頭ノードフラグクリアー
	if (pfTop != NULL) *pfTop = FALSE;

	if (pTop = pNode = (PNODE)AjcXQueTopNode(pW->hXQue, NULL)) {
		do {
			if (pNode->id == id) {
				break;
			}
		} while(pNode = (PNODE)AjcXQueNextNode(pW->hXQue, pNode, NULL));
		//	先頭ノードフラグ設定
		if (pfTop != NULL) *pfTop = (pTop == pNode);
	}
	return pNode;
}

//--------------------------------------------------------------------------------------------------------------//
//	チップ表示コールバック																						//
//--------------------------------------------------------------------------------------------------------------//
static C_UTP CALLBACK cbTipWatch(HWND hwnd, UTP pBuf, UI lBuf, UX id)
{
	PWRKWATCH	pW;
	PNODE		pNode;

	if (IsWindow(hwnd)) {
		pW	  = (PWRKWATCH)MAjcGetWindowLong(hwnd, 0);
		pNode = FindId(pW, (UI)id, NULL);
		if (pNode != NULL) {
			MAjcStrCpy(pBuf, lBuf, AJCLNGSEL(	 TEXT(" ・左ドラッグ ： ウインド移動"),
											 	 TEXT(" - L-Drag  : Move window")));

			if (!(pNode->state & AJCWATSS_DISLCLK)) {
				MAjcStrCat(pBuf, lBuf, AJCLNGSEL(TEXT("\n ・左クリック ： ウインド非表示"),
												 TEXT("\n - L-Click : Hide window")));
			}
			if (!(pNode->state & AJCWATSS_DISRCLK)) {
				MAjcStrCat(pBuf, lBuf, AJCLNGSEL(TEXT("\n ・右クリック ： ポップアップ メニュー"),
												 TEXT("\n - R-Click : Show pop-up menu")));
			}
		}
	}

	return pBuf;
}
//--------------------------------------------------------------------------------------------------------------//
//	時計ウインド・生成																							//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	CreateWatchWindow(PWRKWATCH pW, UI id, UI state)
{
	BOOL	rc   = FALSE;
	HRGN	hRgn = NULL;
	UI		exs = 0;
	SYSTEMTIME	lt;

	//	ウインドスタイル設定
	exs = WS_EX_TOOLWINDOW | WS_EX_LAYERED;
	if (state & AJCWATSS_TRANSPARENT) exs |= WS_EX_TRANSPARENT;
	if (state & AJCWATSS_TOPMOST    ) exs |= WS_EX_TOPMOST;

	//	旧ウインド破棄
	DestroyWatchWindow(pW);
	//	時計ウインド生成
	pW->state = state;
	if (pW->hWnd = CreateWindowEx(exs,					// extend style
						L"_AjcWatchClass_",			// window class name
						L"_AjcWatch_",				// window caption
						WS_POPUP,					// style
						pW->pt.x,					// initial x position
						pW->pt.y,					// initial y position
						WSX,						// initial x size
						WSY,						// initial y size
						pW->hOwner,					// parent window handle
						NULL,						// window menu handle
						hDllInst,					// program instance handle
						pW)) {						// creation parameters
		//	ツールチップ設定
		if (pW->state & AJCWATSS_TRANSPARENT) {
			AjcTipTextAdd(pW->hShaft, AJCLNGSEL(TEXT("アナログ時計"), TEXT("Analog Clock")));
			AjcTipTextSetCallBack(pW->hShaft, id, NULL, cbTipWatch);
		}
		else {
			AjcTipTextAdd(pW->hWnd, AJCLNGSEL(TEXT("アナログ時計"), TEXT("Analog Clock")));
			AjcTipTextSetCallBack(pW->hWnd, id, NULL, cbTipWatch);
		}
		//	現在時刻設定
		SubGetLocalTime(&lt);
		pW->hh = lt.wHour;
		pW->mm = lt.wMinute;
		pW->ss = lt.wSecond;
		//	ウインド表示
		ShowWindow(pW->hWnd, SW_SHOWNA);

		rc = TRUE;
	}

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	時計ウインド破棄																							//
//--------------------------------------------------------------------------------------------------------------//
static	VO	DestroyWatchWindow(PWRKWATCH pW)
{
	if (pW->hWnd   != NULL) {DestroyWindow(pW->hWnd  ); pW->hWnd   = NULL;}
	if (pW->hShaft != NULL) {DestroyWindow(pW->hShaft); pW->hShaft = NULL;}
}
//--------------------------------------------------------------------------------------------------------------//
//		時計外枠のリージョン生成																				//
//--------------------------------------------------------------------------------------------------------------//
static	HRGN	MakeRgnWatch(VO)
{
	HRGN	hRgn = NULL;
	double	r = WSX / 2;
	double	x, y, t;
	int		i;
	int		vertex[1] = {64};
	POINT	ptCir[64] = {0};

	for (i = 0, t = 0.0; i < 64; i++, t = (AJC_PAI / 32.0 * (double)i)) {
		y = (r * 0.95) * sin(t);
		x = (r * 0.95) * cos(t);
		ptCir[i].x = (int)(x + r);
		ptCir[i].y = (int)(y + r);
	}
	hRgn = CreatePolyPolygonRgn(ptCir, vertex, 1, WINDING) ;

	return hRgn;
}
//--------------------------------------------------------------------------------------------------------------//
//	時計・ウインド・プロシージャ																				//
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(Watch, WM_CREATE		)
{
	PWRKWATCH	pW = (PWRKWATCH)((LPCREATESTRUCT)lParam)->lpCreateParams;
	UI			exs = 0;
	RECT		r;
	HRGN		hRgn = NULL;

	MAjcSetWindowLong(hwnd, 0, (UX)pW);

	//	リージョン設定（おおまかな円でリージョン設定）
	if (hRgn = MakeRgnWatch()) {
		SetWindowRgn(hwnd, hRgn, FALSE);
		SetProp(hwnd, L"hRgn", (HANDLE)hRgn);
	}
	//	透明設定（文字盤と、リージョンから はみ出て見える部分を透明化）
	SetLayeredWindowAttributes(hwnd, BkGndColor, 0, LWA_COLORKEY);
	//	時分退避情報初期化
	pW->sv_hh = -1;
	pW->sv_mm = -1;
	pW->sv_ss = -1;

	//	中心軸ウインド生成
	exs = WS_EX_TOOLWINDOW;
	if (pW->state & AJCWATSS_TOPMOST) exs |= WS_EX_TOPMOST;

	GetWindowRect(hwnd, &r);
	pW->hShaft = CreateWindowEx(exs,				// extend style
						L"_AjcShaftClass_",			// window class name
						L"_AjcShaft_",				// window caption
						WS_POPUP,					// style
						r.left + vcx - (SSX / 2),	// initial x position
						r.top  + vcy - (SSY / 2),	// initial y position
						SSX,						// initial x size
						SSY,						// initial y size
						NULL,						// parent window handle
						NULL,						// window menu handle
						hDllInst,					// program instance handle
						pW);						// creation parameters

	if (pW->hShaft != NULL) {
		SetWindowPos(pW->hShaft, pW->hWnd, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);
		ShowWindow(pW->hShaft, SW_SHOWNA);
	}

	//	タイマ起動
	pW->fChgSecHand = TRUE;
	if (pW->state & AJCWATSS_SECHAND) SetTimer(hwnd, 1,   300, NULL);
	else							  SetTimer(hwnd, 1, 10000, NULL);

	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(Watch, WM_DESTROY		)
{
	PWRKWATCH	pW = (PWRKWATCH)MAjcGetWindowLong(hwnd, 0);
	HRGN		hRgn = NULL;

	//	リージョン解放
	if (hRgn = (HRGN)GetProp(hwnd, L"hRgn")) {
		DeleteObject(hRgn);
	}
	//	タイマ停止
	KillTimer(hwnd, 1);
	//	ウインド位置セーブ
	SaveWndPos(pW);

	//	軸ウインド破棄
	if (pW->hShaft != NULL) {
		DestroyWindow(pW->hShaft);
	}

	pW->hWnd = NULL;

	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(Watch, WM_TIMER			)
{
	PWRKWATCH	pW = (PWRKWATCH)MAjcGetWindowLong(hwnd, 0);
	SYSTEMTIME	lt;

	SubGetLocalTime(&lt);
	pW->hh = lt.wHour;
	pW->mm = lt.wMinute;
	pW->ss = lt.wSecond;
	if (pW->sv_ss != pW->ss) {
		pW->sv_hh = pW->hh;
		pW->sv_mm = pW->mm;
		pW->sv_ss = pW->ss;
		InvalidateRect(hwnd, NULL, FALSE);
	}
	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
enum {
	IDM_TOPMOST		=	5001,		//	最前面
	IDM_NOTOPMOST			,		//	非最前面
	IDM_TRANSPARENT			,		//	透明
	IDM_NOTRANSPARENT		,		//	非透明
	IDM_SECHAND				,		//	秒針表示
	IDM_NOSECHAND			,		//	秒針非表示
};

AJC_WNDPROC(Watch, WM_RBUTTONDOWN	)
{
	PWRKWATCH	pW = (PWRKWATCH)MAjcGetWindowLong(hwnd, 0);
	HMENU		hMenu;
	UI			id, state;
	POINT		pt;

	if (!(pW->state & AJCWATSS_DISRCLK)) {
		hMenu = CreatePopupMenu();

		if (pW->state & AJCWATSS_TOPMOST	) AppendMenu(hMenu, MFT_STRING, IDM_NOTOPMOST	  , AJCLNGSEL(L"非最前面(&M)"	, L"Non-Topmost(&M)"	  		));
		else								  AppendMenu(hMenu, MFT_STRING, IDM_TOPMOST 	  , AJCLNGSEL(L"最前面(&M)"  	, L"Topmost(&M)"		  		));

		if (pW->state & AJCWATSS_TRANSPARENT) AppendMenu(hMenu, MFT_STRING, IDM_NOTRANSPARENT , AJCLNGSEL(L"非透明(&T)"		, L"Non-Transparent(&T)" 		));
		else								  AppendMenu(hMenu, MFT_STRING, IDM_TRANSPARENT   , AJCLNGSEL(L"透明(&T)"		, L"Transparent(&T)"	  		));

		if (pW->state & AJCWATSS_SECHAND	) AppendMenu(hMenu, MFT_STRING, IDM_NOSECHAND	  , AJCLNGSEL(L"秒針非表示(&S)"	, L"Hide the second hand(&S)"	));
		else								  AppendMenu(hMenu, MFT_STRING, IDM_SECHAND 	  , AJCLNGSEL(L"秒針表示(&S)"	, L"Display the second hand(&S)"));

		GetCursorPos(&pt);
		id = AjcDgcTrackPopupMenuEx(hMenu, TPM_RETURNCMD | TPM_TOPALIGN, pt.x, pt.y, hwnd, NULL);
		DestroyMenu(hMenu		   );

		//	メニューで選択した処理を実行
		state = pW->state;
		switch (id) {
			case IDM_TOPMOST:		state |=  AJCWATSS_TOPMOST;		break;	//	最前面
			case IDM_NOTOPMOST:		state &= ~AJCWATSS_TOPMOST;		break;	//	非最前面
			case IDM_TRANSPARENT:	state |=  AJCWATSS_TRANSPARENT;	break;	//	透明
			case IDM_NOTRANSPARENT:	state &= ~AJCWATSS_TRANSPARENT;	break;	//	非透明
			case IDM_SECHAND:		state |=  AJCWATSS_SECHAND;		break;	//	秒針表示
			case IDM_NOSECHAND:		state &= ~AJCWATSS_SECHAND;		break;	//	秒針非表示
		}
		if (state != pW->state) {
			PNODE pNode = (PNODE)AjcXQueTopNode(pW->hXQue, NULL);
			AjcWatSetShowState(pW, pNode->id, state);
		}

	}
	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(Watch, WM_RBUTTONUP		)
{
	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(Watch, WM_LBUTTONDOWN	)
{
	PWRKWATCH	pW = (PWRKWATCH)MAjcGetWindowLong(hwnd, 0);

	//	ウインド位置退避
	GetWindowRect(hwnd, &pW->rcOrg);
	//	マウス移動開始位置退避
	GetCursorPos(&pW->ptOrg);
	//	マウスキャプチャ開始
	SetCapture(hwnd);
	pW->fCapture = TRUE;
	//	左ボタン押下時刻設定
	pW->TmLbDown = GetTickCount();

	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(Watch, WM_LBUTTONUP		)
{
	PWRKWATCH	pW	  = (PWRKWATCH)MAjcGetWindowLong(hwnd, 0);
	UI			tmCur = GetTickCount();
	RECT		r;
	POINT		pt;
	double		dist;

	//	ウインド位置退避
	GetWindowRect(hwnd, &r);
	pW->pt.x = r.left;
	pW->pt.y = r.top;

	//	マウスキャプチャ終了
	pW->fCapture = FALSE;
	ReleaseCapture();

	//	時計の中心からの距離が半径以内＆左ボタン押下から300ms以内ならば時計を非表示
	GetCursorPos(&pt);
	MapWindowPoints(NULL, hwnd, &pt, 1);
	dist = sqrt(pow((double)(pt.x - vcx), 2) + pow((double)(pt.y - vcy), 2));
	if (!(pW->state & AJCWATSS_DISLCLK) &&	//	左クリック許可
		dist < (double)WSX * 0.4		&&	//	中心から所定距離内
		(tmCur - pW->TmLbDown) < 300) {		//	左ボタン押下から所定時間内
		POINT	ptCur;
		GetCursorPos(&ptCur);
		if (ptCur.x == pW->ptOrg.x && ptCur.y == pW->ptOrg.y) {	//	ドラッグ無し？
			AjcWatHide(pW);
		}
	}
	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(Watch, WM_MOUSEMOVE		)
{
	PWRKWATCH	pW = (PWRKWATCH)MAjcGetWindowLong(hwnd, 0);
	POINT	ptCur;
	int		x, y;

	//	マウスキャプチャ中ならば、ウインド移動
	if (pW->fCapture) {
		GetCursorPos(&ptCur);
		x = pW->rcOrg.left + (ptCur.x - pW->ptOrg.x);
		y = pW->rcOrg.top  + (ptCur.y - pW->ptOrg.y);
		SetWindowPos(pW->hWnd  , NULL, x                  , y                  , 0, 0, SWP_NOSIZE);
		SetWindowPos(pW->hShaft, NULL, x + vcx - (SSX / 2), y + vcy - (SSY / 2), 0, 0, SWP_NOSIZE);
	}
	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(Watch, WM_PAINT			)
{
	PWRKWATCH	pW = (PWRKWATCH)MAjcGetWindowLong(hwnd, 0);
	PAINTSTRUCT	ps;
	HDC		hdc			= BeginPaint(hwnd, &ps);
	HDC		hmd			= CreateCompatibleDC(hdc);
	HBITMAP	hBmp		= NULL;
	HBITMAP	hSvBmp		= NULL;
	HBRUSH	hBru;
	HPEN	hPen;
	int		hl	 = 42;
	int		ml	 = 55;
	int		sl	 = 66;
	POINT	pth[5]	= {{vcx + 3, vcy + 13},{vcx - 3, vcy + 13},{vcx - 3, vcy - hl},{vcx, vcy - (hl + 3)},{vcx + 3, vcy - hl}};
	POINT	ptm[5]	= {{vcx + 3, vcy + 13},{vcx - 3, vcy + 13},{vcx - 3, vcy - ml},{vcx, vcy - (ml + 3)},{vcx + 3, vcy - ml}};
	POINT	pts[2]	= {{vcx    , vcy + 13},{vcx    , vcy - sl}};

	//	時計ビットマップ設定
	if (pW->state & AJCWATSS_TRANSPARENT) hBmp = LoadBitmap(hDllInst, MAKEINTRESOURCE(IDB_WATCH_T));
	else								  hBmp = LoadBitmap(hDllInst, MAKEINTRESOURCE(IDB_WATCH));
	hSvBmp = (HBITMAP)SelectObject(hmd, hBmp);

	//	ブラシ，ペン設定
	hBru = (HBRUSH)SelectObject(hmd, GetStockObject(BLACK_BRUSH));
	hPen = (HPEN  )SelectObject(hmd, hPenGray);
	//	短針描画
	WatchRotate(pW, ((360.0 / 12.0) * (double)pW->hh + (360.0 / (12.0 * 60.0)) * (double)pW->mm + (360.0 / (12.0 * 3600.0)) * (double)pW->ss), pth, 5, vcx, vcy);
	Polygon(hmd, pth, 5);
	//	長針描画
	WatchRotate(pW, ((360.0 / 60.0) * (double)pW->mm + (360.0 / 3600.0) * (double)pW->ss), ptm, 5, vcx, vcy);
	Polygon(hmd, ptm, 5);
	//	ブラシ，ペン回復
	SelectObject(hmd, hBru);
	SelectObject(hmd, hPen);
	//	秒針描画
	if (pW->fChgSecHand) {				//	秒針表示変更直後？
		pW->fChgSecHand = FALSE;
	}
	else {								//	秒針表示変更直後以外？
		if (pW->state & AJCWATSS_SECHAND) {
			WatchRotate(pW, ((360.0 / 60.0) * (double)pW->ss), pts, 2, vcx, vcy);
			hPen = (HPEN)SelectObject(hmd, hPenRed);
			MoveToEx(hmd, pts[0].x, pts[0].y, NULL);
			LineTo	(hmd, pts[1].x, pts[1].y);
			SelectObject(hmd, hPen);
		}
	}

	//	中心軸描画
	hBru = (HBRUSH)SelectObject(hmd, hBruCent);
	hPen = (HPEN  )SelectObject(hmd, GetStockObject(NULL_PEN));
	Ellipse(hmd, vcx - 6, vcy - 6, vcx + 6, vcy + 6);
	SelectObject(hmd, hPen);
	SelectObject(hmd, hBru);

	//	ウインド描画
	BitBlt(hdc, 0, 0, WSX, WSY, hmd, 0, 0, SRCCOPY);

	DeleteObject(SelectObject(hmd, hSvBmp));
	DeleteDC(hmd);

	EndPaint(hwnd, &ps);

	return 0;
}
//----- セッション終了 -----------------------------------------------------------------------------------------//
AJC_WNDPROC(Watch, WM_ENDSESSION	)
{
	PWRKWATCH	pW = (PWRKWATCH)MAjcGetWindowLong(hwnd, 0);

	//	全ノード破棄し、プロファイルへ表示ステータス書き込み
	SaveAllState(pW);
	//	ウインド位置セーブ
	SaveWndPos(pW);
	//	全リソース開放
	ReleaseAll(pW);

	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDMAP_DEF(Watch)
	AJC_WNDMAP_MSG(Watch, WM_CREATE				)
	AJC_WNDMAP_MSG(Watch, WM_DESTROY			)
	AJC_WNDMAP_MSG(Watch, WM_TIMER				)
	AJC_WNDMAP_MSG(Watch, WM_RBUTTONDOWN		)
	AJC_WNDMAP_MSG(Watch, WM_RBUTTONUP			)
	AJC_WNDMAP_MSG(Watch, WM_LBUTTONDOWN		)
	AJC_WNDMAP_MSG(Watch, WM_LBUTTONUP			)
	AJC_WNDMAP_MSG(Watch, WM_MOUSEMOVE			)
	AJC_WNDMAP_MSG(Watch, WM_PAINT				)
	AJC_WNDMAP_MSG(Watch, WM_ENDSESSION			)
AJC_WNDMAP_END
//--------------------------------------------------------------------------------------------------------------//
//	軸ウインドプロシージャ																						//
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(Shaft, WM_CREATE		)
{
	PWRKWATCH	pW = (PWRKWATCH)((LPCREATESTRUCT)lParam)->lpCreateParams;
	HRGN		hRgn = NULL;

	MAjcSetWindowLong(hwnd, 0, (UX)pW);

	//	軸を円形にするためのリージョン設定
	if (hRgn =  CreateEllipticRgn(0, 0, 13, 13)) {
		SetWindowRgn(hwnd, hRgn, FALSE);
		SetProp(hwnd, L"hRgn", (HANDLE)hRgn);
	}

	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(Shaft, WM_DESTROY		)
{
	PWRKWATCH	pW = (PWRKWATCH)MAjcGetWindowLong(hwnd, 0);
	HRGN		hRgn = NULL;

	//	リージョン解放
	if (hRgn = (HRGN)GetProp(hwnd, L"hRgn")) {
		DeleteObject(hRgn);
	}
	pW->hShaft = NULL;
	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(Shaft, WM_RBUTTONDOWN	)
{
	PWRKWATCH	pW = (PWRKWATCH)MAjcGetWindowLong(hwnd, 0);
	POINT	pt;
	pt.x = GET_X_LPARAM(lParam);
	pt.y = GET_Y_LPARAM(lParam);
	MapWindowPoints(hwnd, pW->hWnd, (LPPOINT)&pt, 1);
	SendMessage(pW->hWnd, WM_RBUTTONDOWN, wParam, MAKELONG(pt.x, pt.y));

	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(Shaft, WM_RBUTTONUP		)
{
	PWRKWATCH	pW = (PWRKWATCH)MAjcGetWindowLong(hwnd, 0);
	POINT	pt;
	pt.x = GET_X_LPARAM(lParam);
	pt.y = GET_Y_LPARAM(lParam);
	MapWindowPoints(hwnd, pW->hWnd, (LPPOINT)&pt, 1);
	SendMessage(pW->hWnd, WM_RBUTTONUP, wParam, MAKELONG(pt.x, pt.y));

	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(Shaft, WM_LBUTTONDOWN	)
{
	PWRKWATCH	pW = (PWRKWATCH)MAjcGetWindowLong(hwnd, 0);
	POINT	pt;
	pt.x = GET_X_LPARAM(lParam);
	pt.y = GET_Y_LPARAM(lParam);
	MapWindowPoints(hwnd, pW->hWnd, (LPPOINT)&pt, 1);
	SendMessage(pW->hWnd, WM_LBUTTONDOWN, wParam, MAKELONG(pt.x, pt.y));

	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(Shaft, WM_LBUTTONUP		)
{
	PWRKWATCH	pW = (PWRKWATCH)MAjcGetWindowLong(hwnd, 0);
	POINT	pt;
	pt.x = GET_X_LPARAM(lParam);
	pt.y = GET_Y_LPARAM(lParam);
	MapWindowPoints(hwnd, pW->hWnd, (LPPOINT)&pt, 1);
	SendMessage(pW->hWnd, WM_LBUTTONUP, wParam, MAKELONG(pt.x, pt.y));

	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDPROC(Shaft, WM_MOUSEMOVE		)
{
	PWRKWATCH	pW = (PWRKWATCH)MAjcGetWindowLong(hwnd, 0);
	POINT	pt;
	pt.x = GET_X_LPARAM(lParam);
	pt.y = GET_Y_LPARAM(lParam);
	MapWindowPoints(hwnd, pW->hWnd, (LPPOINT)&pt, 1);
	SendMessage(pW->hWnd, WM_MOUSEMOVE, wParam, MAKELONG(pt.x, pt.y));

	return 0;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_WNDMAP_DEF(Shaft)
	AJC_WNDMAP_MSG(Shaft, WM_CREATE				)
	AJC_WNDMAP_MSG(Shaft, WM_DESTROY			)
	AJC_WNDMAP_MSG(Shaft, WM_RBUTTONDOWN		)
	AJC_WNDMAP_MSG(Shaft, WM_RBUTTONUP			)
	AJC_WNDMAP_MSG(Shaft, WM_LBUTTONDOWN		)
	AJC_WNDMAP_MSG(Shaft, WM_LBUTTONUP			)
	AJC_WNDMAP_MSG(Shaft, WM_MOUSEMOVE			)
AJC_WNDMAP_END

//----- 回転処理 ------------------------------------------------------------------------------------------------//
static	VO		WatchRotate(PWRKWATCH pW, double rot, POINT pt[], int n, int vcx, int vcy)
{
	AJC2DVEC	v;
	int			i;

	for (i = 0; i < n; i++) {
		v.x = pt[i].x - vcx;
		v.y = pt[i].y - vcy;
		AjcV2dRotate(&v, rot, &v);
		pt[i].x = (int)(v.x + vcx);
		pt[i].y = (int)(v.y + vcy);
	}
}
