﻿#include	"AjcInternal.h"

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
static BCP MkKeyNameA(C_BCP pPrefix, C_BCP pName);
static WCP MkKeyNameW(C_WCP pPrefix, C_WCP pName);

//==============================================================================================================//
//	プロファイルからウインド位置をロード																		//
//																												//
//	引　数：	hwnd	   - ウインドハンドル																	//
//				pSec	   - プロファイルセクション名へのポインタ												//
//				pKeyPrefix - キープリフィックス名へのポインタ													//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcLoadWndPosA		 (HWND hwnd, C_BCP pSec)
{
	return AjcLoadWndPosExA(hwnd, pSec, NULL);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT BOOL	 WINAPI AjcLoadWndPosExA	 (HWND hwnd, C_BCP pSec, C_BCP pKeyPrefix)
{
	BOOL	rc = FALSE;
	RECT	r;
	int		x, y;

	if (pSec == NULL) pSec = "AjcWndPos";

	if (IsWindow(hwnd)) {
		if (rc = GetWindowRect(hwnd, &r)) {
			x  = AjcGetProfileSIntA(pSec, MkKeyNameA(pKeyPrefix, "AjcWndPX"), r.left);
			y  = AjcGetProfileSIntA(pSec, MkKeyNameA(pKeyPrefix, "AjcWndPY"), r.top);
			SetWindowPos(hwnd, NULL, x, y, 0, 0, SWP_NOSIZE);
			AjcMoveOutWndToOrigin(hwnd);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcLoadWndPosW		 (HWND hwnd, C_WCP pSec)
{
	return AjcLoadWndPosExW(hwnd, pSec, NULL);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT BOOL	 WINAPI AjcLoadWndPosExW	 (HWND hwnd, C_WCP pSec, C_WCP pKeyPrefix)
{
	BOOL	rc = FALSE;
	RECT	r;
	int		x, y;

	if (pSec == NULL) pSec = L"AjcWndPos";

	if (IsWindow(hwnd)) {
		if (rc = GetWindowRect(hwnd, &r)) {
			x  = AjcGetProfileSIntW(pSec, MkKeyNameW(pKeyPrefix, L"AjcWndPX"), r.left);
			y  = AjcGetProfileSIntW(pSec, MkKeyNameW(pKeyPrefix, L"AjcWndPY"), r.top);
			SetWindowPos(hwnd, NULL, x, y, 0, 0, SWP_NOSIZE);
			AjcMoveOutWndToOrigin(hwnd);
		}
	}
	return rc;
}
//==============================================================================================================//
//	プロファイルからウインド位置とサイズをロード																//
//																												//
//	引　数：	hwnd	  - ウインドハンドル																	//
//				pSec	   - プロファイルセクション名へのポインタ												//
//				pKeyPrefix - キープリフィックス名へのポインタ													//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcLoadWndRectA		 (HWND hwnd, C_BCP pSec)
{
	return AjcLoadWndRectExA(hwnd, pSec, NULL);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT BOOL	 WINAPI AjcLoadWndRectExA	 (HWND hwnd, C_BCP pSec, C_BCP pKeyPrefix)
{
	BOOL	rc = FALSE;
	RECT	r;
	int		x, y, cx, cy;

	if (pSec == NULL) pSec = "AjcWndRect";

	if (IsWindow(hwnd)) {
		if (rc = GetWindowRect(hwnd, &r)) {
			x  = AjcGetProfileSIntA(pSec, MkKeyNameA(pKeyPrefix, "AjcWndPX"), r.left);
			y  = AjcGetProfileSIntA(pSec, MkKeyNameA(pKeyPrefix, "AjcWndPY"), r.top);
			cx = AjcGetProfileSIntA(pSec, MkKeyNameA(pKeyPrefix, "AjcWndCX"), r.right - r.left);
			cy = AjcGetProfileSIntA(pSec, MkKeyNameA(pKeyPrefix, "AjcWndCY"), r.bottom - r.top);
			SetWindowPos(hwnd, NULL, x, y, cx, cy, 0);
			AjcMoveOutWndToOrigin(hwnd);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcLoadWndRectW		 (HWND hwnd, C_WCP pSec)
{
	return AjcLoadWndRectExW(hwnd, pSec, NULL);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT BOOL	 WINAPI AjcLoadWndRectExW	 (HWND hwnd, C_WCP pSec, C_WCP pKeyPrefix)
{
	BOOL	rc = FALSE;
	RECT	r;
	int		x, y, cx, cy;

	if (pSec == NULL) pSec = L"AjcWndRect";

	if (IsWindow(hwnd)) {
		if (rc = GetWindowRect(hwnd, &r)) {
			x  = AjcGetProfileSIntW(pSec, MkKeyNameW(pKeyPrefix, L"AjcWndPX"), r.left);
			y  = AjcGetProfileSIntW(pSec, MkKeyNameW(pKeyPrefix, L"AjcWndPY"), r.top);
			cx = AjcGetProfileSIntW(pSec, MkKeyNameW(pKeyPrefix, L"AjcWndCX"), r.right - r.left);
			cy = AjcGetProfileSIntW(pSec, MkKeyNameW(pKeyPrefix, L"AjcWndCY"), r.bottom - r.top);
			SetWindowPos(hwnd, NULL, x, y, cx, cy, 0);
			AjcMoveOutWndToOrigin(hwnd);
		}
	}
	return rc;
}
//==============================================================================================================//
//	プロファイルへウインド位置をセーブ																			//
//																												//
//	引　数：	hwnd	  - ウインドハンドル																	//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcSaveWndPosA		 (HWND hwnd, C_BCP pSec)
{
	return AjcSaveWndPosExA(hwnd, pSec, NULL);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT BOOL	 WINAPI AjcSaveWndPosExA	 (HWND hwnd, C_BCP pSec, C_BCP pKeyPrefix)
{
	BOOL	rc = FALSE;
	WINDOWPLACEMENT	wp;

	if (pSec == NULL) pSec = "AjcWndPos";

	if (IsWindow(hwnd)) {
		wp.length = sizeof(WINDOWPLACEMENT);
		if (rc = GetWindowPlacement(hwnd, &wp)) {
			AjcPutProfileSIntA(pSec, MkKeyNameA(pKeyPrefix, "AjcWndPX"), wp.rcNormalPosition.left);
			AjcPutProfileSIntA(pSec, MkKeyNameA(pKeyPrefix, "AjcWndPY"), wp.rcNormalPosition.top);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcSaveWndPosW		 (HWND hwnd, C_WCP pSec)
{
	return AjcSaveWndPosExW(hwnd, pSec, NULL);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT BOOL	 WINAPI AjcSaveWndPosExW	 (HWND hwnd, C_WCP pSec, C_WCP pKeyPrefix)
{
	BOOL	rc = FALSE;
	WINDOWPLACEMENT	wp;

	if (pSec == NULL) pSec = L"AjcWndPos";

	if (IsWindow(hwnd)) {
		wp.length = sizeof(WINDOWPLACEMENT);
		if (rc = GetWindowPlacement(hwnd, &wp)) {
			AjcPutProfileSIntW(pSec, MkKeyNameW(pKeyPrefix, L"AjcWndPX"), wp.rcNormalPosition.left);
			AjcPutProfileSIntW(pSec, MkKeyNameW(pKeyPrefix, L"AjcWndPY"), wp.rcNormalPosition.top);
		}
	}
	return rc;
}
//==============================================================================================================//
//	プロファイルへウインド位置とサイズをセーブ																	//
//																												//
//	引　数：	hwnd	  - ウインドハンドル																	//
//																												//
//	戻り値：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcSaveWndRectA		 (HWND hwnd, C_BCP pSec)
{
	return AjcSaveWndRectExA(hwnd, pSec, NULL);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT BOOL	 WINAPI AjcSaveWndRectExA	 (HWND hwnd, C_BCP pSec, C_BCP pKeyPrefix)
{
	BOOL	rc = FALSE;
	WINDOWPLACEMENT	wp;

	if (pSec == NULL) pSec = "AjcWndRect";

	if (IsWindow(hwnd)) {
		wp.length = sizeof(WINDOWPLACEMENT);
		if (rc = GetWindowPlacement(hwnd, &wp)) {
			AjcPutProfileSIntA(pSec, MkKeyNameA(pKeyPrefix, "AjcWndPX"), wp.rcNormalPosition.left);
			AjcPutProfileSIntA(pSec, MkKeyNameA(pKeyPrefix, "AjcWndPY"), wp.rcNormalPosition.top);
			AjcPutProfileSIntA(pSec, MkKeyNameA(pKeyPrefix, "AjcWndCX"), wp.rcNormalPosition.right - wp.rcNormalPosition.left);
			AjcPutProfileSIntA(pSec, MkKeyNameA(pKeyPrefix, "AjcWndCY"), wp.rcNormalPosition.bottom - wp.rcNormalPosition.top);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcSaveWndRectW		 (HWND hwnd, C_WCP pSec)
{
	return AjcSaveWndRectExW(hwnd, pSec, NULL);
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT BOOL	 WINAPI AjcSaveWndRectExW	 (HWND hwnd, C_WCP pSec, C_WCP pKeyPrefix)
{
	BOOL	rc = FALSE;
	WINDOWPLACEMENT	wp;

	if (pSec == NULL) pSec = L"AjcWndRect";

	if (IsWindow(hwnd)) {
		wp.length = sizeof(WINDOWPLACEMENT);
		if (rc = GetWindowPlacement(hwnd, &wp)) {
			AjcPutProfileSIntW(pSec, MkKeyNameW(pKeyPrefix, L"AjcWndPX"), wp.rcNormalPosition.left);
			AjcPutProfileSIntW(pSec, MkKeyNameW(pKeyPrefix, L"AjcWndPY"), wp.rcNormalPosition.top);
			AjcPutProfileSIntW(pSec, MkKeyNameW(pKeyPrefix, L"AjcWndCX"), wp.rcNormalPosition.right - wp.rcNormalPosition.left);
			AjcPutProfileSIntW(pSec, MkKeyNameW(pKeyPrefix, L"AjcWndCY"), wp.rcNormalPosition.bottom - wp.rcNormalPosition.top);
		}
	}
	return rc;
}
/*==============================================================================================================*/
/*	Function	:	ウインドの一部が隠れている場合ウインドをモニタ内へ移動する									*/
/*																												*/
/*	Argument	:	hwnd - ウインドハンドル																		*/
/*																												*/
/*	Return		:	TRUE  - OK																					*/
/*					FALSE - NG																					*/
/*==============================================================================================================*/
AJCEXPORT	BOOL	WINAPI	AjcMoveWndIntoMonitor (HWND hwnd)
{
	return AjcMoveWndRectIntoMonitor(hwnd, NULL);
}
/*==============================================================================================================*/
/*	Function	:	ウインド内矩形の一部が隠れている場合矩形をモニタ内へ移動する								*/
/*																												*/
/*	Argument	:	hwnd  - ウインドハンドル																	*/
/*					pRect - ウインド内の矩形領域（画面座標，ウインド全体の場合はNULL）							*/
/*																												*/
/*	Return		:	TRUE  - OK																					*/
/*					FALSE - NG																					*/
/*==============================================================================================================*/
AJCEXPORT	BOOL	WINAPI	AjcMoveWndRectIntoMonitor (HWND hwnd, LPRECT pRect)
{
	BOOL			rc = FALSE;
	UI				i, nMon;
	AJCMONITORSINFO	ifMon;				//	モニタ群矩形情報
	RECT			rcMon;				//	検索したモニタ矩形
	RECT			rcWnd;				//	ウインド矩形
	int				wx, wy, ww, wh;		//	ウインド位置，サイズ
	int				rx, ry, rw, rh;		//	ウインド内矩形の位置，サイズ
	int				sqr, sqrMax = 0;

	if (IsWindow(hwnd)) {
		//	ウインド矩形情報，幅，高さ設定
		GetWindowRect(hwnd, &rcWnd);
		wx = rcWnd.left;
		wy = rcWnd.top;
		ww = rcWnd.right  - rcWnd.left;
		wh = rcWnd.bottom - rcWnd.top;
		//	ウインド全体指定
		if (pRect == NULL) {
			pRect = &rcWnd;
		}
		//	各モニタ矩形情報取得
		nMon = AjcGetMonitorsInfo(&ifMon);
		//	ウインド内矩形の所属するモニタを検索
		for (i = 0; i < nMon; i++) {
			sqr = AjcGetDupRect(pRect, &ifMon.rcMon[i], NULL);
			if (sqr > sqrMax) {
				sqrMax = sqr;
				memcpy(&rcMon, &ifMon.rcMon[i], sizeof rcMon);
			}
		}
		//	所属モニタが無い場合は、ウインドが所属するモニタを検索
		if (sqrMax == 0 && pRect != &rcWnd) {
			for (i = 0; i < nMon; i++) {
				sqr = AjcGetDupRect(&rcWnd, &ifMon.rcMon[i], NULL);
				if (sqr > sqrMax) {
					sqrMax = sqr;
					memcpy(&rcMon, &ifMon.rcMon[i], sizeof rcMon);
				}
			}
		}
		//	所属するモニタあり？
		if (sqrMax != 0) {
			rx	= pRect->left;
			ry	= pRect->top;
			rw = pRect->right  - pRect->left;
			rh = pRect->bottom - pRect->top;
			//	ウインドが右端を出る場合、Ｘを左に移動
			if (rx + rw > rcMon.right) {
				wx -= (rx + rw - rcMon.right);
			}
			//	ウインドが左端を出る場合、Ｘを右に移動
			if (rx < rcMon.left) {
				wx += (rcMon.left - rx);
			}
			//	ウインドが下端を出る場合、Ｙを上に移動
			if (ry + rh > rcMon.bottom) {
				wy -= (ry + rh - rcMon.bottom);
			}
			//	ウインドが上端を出る場合、Ｙを下に移動
			if (ry < rcMon.top) {
				wy += (rcMon.top - ry);
			}
			//	モニタ内へ移動
			SetWindowPos(hwnd, NULL, wx, wy, 0, 0, SWP_NOSIZE);
		}
		//	所属モニタが無ければ、原点へ移動
		else {
			SetWindowPos(hwnd, NULL, 0, 0, 0, 0, SWP_NOSIZE);
		}
		//	戻り値＝正常
		rc = TRUE;
	}
	return rc;
}
/*==============================================================================================================*/
/*	Function	:	ウインド全体が隠れている場合 ウインドを原点へ移動する										*/
/*																												*/
/*	Argument	:	hwnd - ウインドハンドル																		*/
/*																												*/
/*	Return		:	TRUE  - OK																					*/
/*					FALSE - NG																					*/
/*==============================================================================================================*/
AJCEXPORT	BOOL	WINAPI	AjcMoveOutWndToOrigin (HWND hwnd)
{
	BOOL	rc = FALSE;
	UI		Bounds = 0;
	UI		Sqr    = 0;
	RECT	rcWnd;

	if (IsWindow(hwnd)) {
		GetWindowRect(hwnd, &rcWnd);
		if (!AjcIsSeeEntireInMonitors(&rcWnd, &Bounds, &Sqr)) {
			if (Bounds == 0) {
				SetWindowPos(hwnd, NULL, 0, 0, 0, 0, SWP_NOSIZE);
			}
		}
		rc = TRUE;
	}
	return rc;
}

/*==============================================================================================================*/
/*	Function	:	全モニタで矩形が見えているかチェックする													*/
/*																												*/
/*	Argument	:	hwnd	- ウインドハンドル																	*/
/*					pRect	- 矩形情報（画面座標）																*/
/*					pBounds - 見えているモニタ数を格納するバッファのアドレス（不要時はNULL）					*/
/*					pSqr	- 見えている総面積を格納するバッファのアドレス（不要時はNULL）						*/
/*																												*/
/*	Return		:	TRUE  - 全体が見える																		*/
/*					FALSE - 一部／全部が見えない																*/
/*==============================================================================================================*/
AJCEXPORT	BOOL	WINAPI	AjcIsSeeEntireInMonitors (LPRECT pRect, UIP pBounds, UIP pSqr)
{
	BOOL			rc	 = FALSE;		//	戻り値
	UI				nSee = 0;			//	見えているモニタ数
	BOOL			fSee;				//	当該モニタで見えていることを示すフラグ
	UI				sqr  = 0;			//	見えている面積
	UI				i, nMon;			//	モニタ数
	AJCMONITORSINFO	ifMon;				//	モニタ群矩形情報
	BOOL			fUL = FALSE;		//	左上が見えていることを示すフラグ
	BOOL			fUR = FALSE;		//	右上が見えていることを示すフラグ
	BOOL			fDL = FALSE;		//	左下が見えていることを示すフラグ
	BOOL			fDR = FALSE;		//	右下が見えていることを示すフラグ

	if (pRect != NULL) {
		//	各モニタ矩形情報取得
		nMon = AjcGetMonitorsInfo(&ifMon);
		//	ウインド内矩形の所属するモニタを検索
		for (i = 0; i < nMon; i++) {
			fSee = FALSE;
			//	上が見えているかチェック
			if (pRect->top >= ifMon.rcMon[i].top && pRect->top < ifMon.rcMon[i].bottom) {
				//	左が見えているかチェック
				if (pRect->left  >= ifMon.rcMon[i].left && pRect->left	< ifMon.rcMon[i].right) {
					fUL = TRUE; fSee = TRUE;
				}
				//	右が見えているかチェック
				if (pRect->right >= ifMon.rcMon[i].left && pRect->right < ifMon.rcMon[i].right) {
					fUR = TRUE; fSee = TRUE;
				}
			}
			//	下が見えているかチェック
			if (pRect->bottom >= ifMon.rcMon[i].top && pRect->bottom < ifMon.rcMon[i].bottom) {
				//	左が見えているかチェック
				if (pRect->left  >= ifMon.rcMon[i].left && pRect->left	< ifMon.rcMon[i].right) {
					fDL = TRUE; fSee = TRUE;
				}
				//	右が見えているかチェック
				if (pRect->right >= ifMon.rcMon[i].left && pRect->right < ifMon.rcMon[i].right) {
					fDR = TRUE; fSee = TRUE;
				}
			}
			//	見えているモニタ数と面積更新
			if (fSee) {
				nSee++;
				sqr += AjcGetDupRect(&ifMon.rcMon[i], pRect, NULL);
			}
		}
		//	またがっているモニタ数設定
		if (pBounds != NULL) {
			*pBounds = nSee;
		}
		//	総面積設定
		if (pSqr != NULL) {
			*pSqr = sqr;
		}
		//	全体が見えていれば、戻り値＝TRUE
		if (fUL && fUR && fDL && fDR) {
			rc = TRUE;
		}
	}
	return rc;
}
/*==============================================================================================================*/
/*	Function	:	隠れているウインドを掴める位置まで移動する（非推奨）										*/
/*																												*/
/*	Argument	:	hwnd - ウインドハンドル																		*/
/*																												*/
/*	Return		:	TRUE  - OK																					*/
/*					FALSE - NG																					*/
/*==============================================================================================================*/
AJCEXPORT	BOOL	WINAPI	AjcMoveWindowToGripped(HWND hwnd)
{
	BOOL	rc = FALSE;
	BOOL	fNeedMove = FALSE;
	HWND	hParent;
	RECT	rcParent;
	RECT	rcMyWnd, rcGrid;
	int		x;
	UX		ws	= MAjcGetWindowLong(hwnd, GWL_STYLE);
	UX		wes = MAjcGetWindowLong(hwnd, GWL_EXSTYLE);

	do {
		if (!IsWindow(hwnd)) break;
		//----- デスクトップ作業域／親ウインド・クライアント域のサイズ取得 -----//
		if (hParent = GetParent(hwnd)) {
			GetClientRect(hParent, &rcParent);
		}
		else {
			SystemParametersInfo(SPI_GETWORKAREA, 0, &rcParent, 0);
		}

		//----- 自ウインドサイズと、グリッド域（キャプションバーのセンタ域）サイズ取得 -----//
		GetWindowRect(hwnd, &rcMyWnd);
		memcpy(&rcGrid, &rcMyWnd, sizeof rcGrid);
		if (wes & WS_EX_TOOLWINDOW) rcGrid.bottom = rcGrid.top + GetSystemMetrics(SM_CYSMCAPTION);
		else						rcGrid.bottom = rcGrid.top + GetSystemMetrics(SM_CYCAPTION);
		rcGrid.left	 += __min((GetSystemMetrics(SM_CXSIZE) * 2), (rcGrid.right - rcGrid.left) / 2);
		rcGrid.right -= __min((GetSystemMetrics(SM_CXSIZE) * 4), rcGrid.right - rcGrid.left);

		//----- キャプションバーが上に隠れていれば、上端位置まで戻す -----//
		if (rcGrid.top	  < rcParent.top   ) {
			x = rcParent.top - rcGrid.top;
			rcMyWnd.top	   += x;
			rcMyWnd.bottom += x;
			fNeedMove = TRUE;
		}
		//----- キャプションバーが下に隠れていれば、下端位置まで戻す -----//
		if (rcGrid.bottom > rcParent.bottom) {
			x = rcGrid.bottom - rcParent.bottom;
			rcMyWnd.top	   -= x;
			rcMyWnd.bottom -= x;
			fNeedMove = TRUE;
		}
		//----- キャプションバーが左に隠れていれば、左端位置まで戻す -----//
		if (rcGrid.right < rcParent.left) {
			x = rcParent.left - rcGrid.right;
			rcMyWnd.left  += x;
			rcMyWnd.right += x;
			fNeedMove = TRUE;
		}
		//----- キャプションバーが右に隠れていれば、右端位置まで戻す -----//
		if (rcGrid.left > rcParent.right) {
			x = rcGrid.left - rcParent.right;
			rcMyWnd.left  -= x;
			rcMyWnd.right -= x;
			fNeedMove = TRUE;
		}
		//----- ウインド移動 -----//
		if (fNeedMove) {
			MoveWindow(hwnd, rcMyWnd.left, rcMyWnd.top, rcMyWnd.right - rcMyWnd.left,
														rcMyWnd.bottom - rcMyWnd.top, TRUE);
		}
		rc = TRUE;

	} while(0);

	return rc;
}
/*==============================================================================================================*/
/*	Function	:	隠れているトップレベルウインドウインドを掴める位置まで移動する（非推奨）					*/
/*																												*/
/*	Argument	:	hwnd  - ウインドハンドル																	*/
/*					ratio - タイトルバーの見えている部分の割合[%]												*/
/*							タイトルバーの見えている部分がratio以下ならばウインドを原点へ移動する				*/
/*																												*/
/*	Return		:	TRUE  - OK																					*/
/*					FALSE - NG																					*/
/*==============================================================================================================*/
typedef struct {
	int		ix;
	RECT	rcGrid;
	LPRECT	prcWork;
} AMVG_WORKAREA, *PAMVG_WORKAREA;
//--------------------------------------------------------------------------------------------------------------//
//	ディスプレィ情報通知
static	BOOL CALLBACK cbAmwgEnumDisplay(HMONITOR hMon, HDC hdcMon, LPRECT prcMon, LPARAM lParam)
{
	PAMVG_WORKAREA	pWA = (PAMVG_WORKAREA)lParam;
	HMONITOR		hMonitor = NULL;
	MONITORINFOEX	MonitorInfoEx;
	POINT			pt;

	pt.x = (prcMon->left + prcMon->right) / 2;
	pt.y = (prcMon->top + prcMon->bottom) / 2;
	hMonitor = MonitorFromPoint(pt, MONITOR_DEFAULTTONEAREST);
	memset(&MonitorInfoEx, 0, sizeof MonitorInfoEx);
	MonitorInfoEx.cbSize = sizeof(MonitorInfoEx);
	GetMonitorInfo(hMonitor, (LPMONITORINFO)&MonitorInfoEx);
	memcpy(&pWA->prcWork[pWA->ix], &MonitorInfoEx.rcWork, sizeof(RECT));
	pWA->ix++;

	return TRUE;
}
//--------------------------------------------------------------------------------------------------------------//
//	チャイルドウインド通知
static	BOOL CALLBACK cbAmwgEnumChild(HWND hw, LPARAM lParam)
{
	PAMVG_WORKAREA	pWA = (PAMVG_WORKAREA)lParam;
	RECT			wr;

	GetWindowRect(hw, &wr);
	if (AjcGetDupRect(&pWA->rcGrid, &wr, NULL) != 0) {
		if (wr.left > pWA->rcGrid.left && wr.right < pWA->rcGrid.right) {
			pWA->rcGrid.left  = __max(pWA->rcGrid.left , wr.right);
		}
	}
	return TRUE;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcMoveWindowToGrippedEx(HWND hwnd, UI ratio)
{
	BOOL			rc		 = FALSE;
	int				i, nDisplay = 0;
	AMVG_WORKAREA	AmvgWa;
	int				sqGrid, sqDup;
	int				per;
	UX				wes = MAjcGetWindowLong(hwnd, GWL_EXSTYLE);

	//	ワーククリアー
	memset(&AmvgWa, 0, sizeof AmvgWa);

	do {
		if (!IsWindow(hwnd)								 ) break;
		if (MAjcGetWindowLong(hwnd, GWL_STYLE) & WS_CHILD) break;
		//----- モニタ数取得 ---------------------------------------------------------------//
		nDisplay = GetSystemMetrics(SM_CMONITORS);
		//----- モニタ情報領域確保 ---------------------------------------------------------//
		if ((AmvgWa.prcWork = (LPRECT)AJCMEM(sizeof(RECT) * nDisplay)) == NULL) break;
		memset(AmvgWa.prcWork, 0, sizeof(RECT) * nDisplay);
		//----- 全ディスプレイの作業領域を取得 ---------------------------------------------//
		if (!EnumDisplayMonitors(NULL, NULL, cbAmwgEnumDisplay, (LPARAM)&AmvgWa)) break;
		//----- ウインドのタイトルバー矩形を得る -------------------------------------------//
		GetWindowRect(hwnd, &AmvgWa.rcGrid);
		if (wes & WS_EX_TOOLWINDOW) AmvgWa.rcGrid.bottom = AmvgWa.rcGrid.top + GetSystemMetrics(SM_CYSMCAPTION) - 1;
		else						AmvgWa.rcGrid.bottom = AmvgWa.rcGrid.top + GetSystemMetrics(SM_CYCAPTION) - 1;
		AmvgWa.rcGrid.right -= (GetSystemMetrics(SM_CXSIZE) * 3); // 右端のボタンを除外
		if (AmvgWa.rcGrid.right <= AmvgWa.rcGrid.left) break;
		//----- タイトルバーから子ウインドと重なる部分を除外する ---------------------------//
		EnumChildWindows(hwnd, cbAmwgEnumChild, (LPARAM)&AmvgWa);
		if (AmvgWa.rcGrid.right <= AmvgWa.rcGrid.left) break;
		//----- タイトルバー（グリッド部分）の面積算出 -------------------------------------//
		sqGrid = (AmvgWa.rcGrid.right - AmvgWa.rcGrid.left) * (AmvgWa.rcGrid.bottom - AmvgWa.rcGrid.top);
		//----- 全ディスプレイとウインドタイトルバーの重なる面積を求める -------------------//
		sqDup = 0;
		for (i = 0; i < nDisplay; i++) {
			sqDup += AjcGetDupRect(&AmvgWa.rcGrid, &AmvgWa.prcWork[i], NULL);
		}
		per = sqDup * 100 / sqGrid;
		//----- 重なっている面積が小さければ、原点へ移動 -----------------------------------//
		if ((UI)per < ratio) {
			SetWindowPos(hwnd, NULL, 0, 0, 0, 0, SWP_NOSIZE);
		}
		rc = TRUE;
	} while(0);

	if (AmvgWa.prcWork != NULL) free(AmvgWa.prcWork);

	return rc;
}
/*==============================================================================================================*/
/*	Function	:	ウインドを親ウインド／オーナーウインド／所属モニタ／プライマリモニタ中央へ移動する			*/
/*																												*/
/*	Argument	:	hwnd - ウインドハンドル																		*/
/*																												*/
/*	Return		:	TRUE  - OK																					*/
/*					FALSE - NG																					*/
/*==============================================================================================================*/
AJCEXPORT	BOOL	WINAPI	AjcMoveWindowToCenter(HWND hwnd)
{
	BOOL	rc = FALSE;
	int		sx, sy, scx, scy;
	int		wx, wy, wcx, wcy;
	HWND	hScr;
	RECT	rw, rs;

	if (IsWindow(hwnd)) {
		GetWindowRect(hwnd, &rw);
		if		(hScr = GetWindow(hwnd, GW_OWNER)) {
			//	ウインドのスクリーン座標，サイズ設定
			wx	= rw.left;
			wy	= rw.top;
			wcx = rw.right - rw.left;
			wcy = rw.bottom - rw.top;
			//	オーナーウインドのスクリーン座標，サイズ設定
			GetClientRect(hScr, &rs);
			MapWindowPoints(hScr, NULL, (LPPOINT)&rs, 2);
			sx	= rs.left;
			sy	= rs.top;
			scx = rs.right - rs.left;
			scy = rs.bottom - rs.top;
			//	ウインドをオーナーウインドの中央へ移動
			SetWindowPos(hwnd, NULL, sx + (scx - wcx) / 2, sy + (scy - wcy) / 2, 0, 0, SWP_NOSIZE);
		}
		else if (hScr = GetParent(hwnd)) {
			//	ウインドのクライアント座標，サイズ設定
			MapWindowPoints(NULL, hScr, (LPPOINT)&rw, 2);
			wx	= rw.left;
			wy	= rw.top;
			wcx = rw.right - rw.left;
			wcy = rw.bottom - rw.top;
			//	親ウインドのクライアント座標，サイズ設定
			GetClientRect(hScr, &rs);
			sx	= rs.left;
			sy	= rs.top;
			scx = rs.right - rs.left;
			scy = rs.bottom - rs.top;
			//	ウインドを親ウインドの中央へ移動
			SetWindowPos(hwnd, NULL, sx + (scx - wcx) / 2, sy + (scy - wcy) / 2, 0, 0, SWP_NOSIZE);
		}
		else {
			//	ウインドをモニタの中央へ移動
			AjcMoveWindowToCenterOfMonitor(hwnd);
		}
		rc = TRUE;
	}
	return rc;
}
/*==============================================================================================================*/
/*	Function	:	ウインドを画面中央へ移動する																*/
/*																												*/
/*	Argument	:	hwnd - ウインドハンドル																		*/
/*																												*/
/*	Return		:	TRUE  - OK																					*/
/*					FALSE - NG																					*/
/*==============================================================================================================*/
AJCEXPORT	BOOL	WINAPI	AjcMoveWindowToCenterOfScreen(HWND hwnd)
{
	BOOL	rc = FALSE;
	int		sx, sy, scx, scy;
	int		wx, wy, wcx, wcy;
	RECT	rw, rs;

	if (IsWindow(hwnd)) {
		//	ウインドの位置とサイズ設定
		GetWindowRect(hwnd, &rw);
		wx	= rw.left;
		wy	= rw.top;
		wcx = rw.right - rw.left;
		wcy = rw.bottom - rw.top;
		//	モニタ全体の位置とサイズ設定
		AjcGetMonitorsRect(&rs);
		sx	= rs.left;
		sy	= rs.top;
		scx = rs.right - rs.left;
		scy = rs.bottom - rs.top;
		//	ウインドをモニタ全体の中央へ移動
		SetWindowPos(hwnd, NULL, sx + (scx - wcx) / 2, sy + (scy - wcy) / 2, 0, 0, SWP_NOSIZE);
	}
	return rc;
}
/*==============================================================================================================*/
/*	Function	:	ウインドをモニタ中央へ移動する																*/
/*																												*/
/*	Argument	:	hwnd - ウインドハンドル																		*/
/*																												*/
/*	Return		:	TRUE  - OK																					*/
/*					FALSE - NG																					*/
/*==============================================================================================================*/
AJCEXPORT	BOOL	WINAPI	AjcMoveWindowToCenterOfMonitor(HWND hwnd)
{
	BOOL	rc = FALSE;
	int		sx, sy, scx, scy;
	int		wx, wy, wcx, wcy;
	RECT	rw, rs;

	if (IsWindow(hwnd)) {
		//	ウインドの位置とサイズ設定
		GetWindowRect(hwnd, &rw);
		wx	= rw.left;
		wy	= rw.top;
		wcx = rw.right - rw.left;
		wcy = rw.bottom - rw.top;
		//	モニタ矩形情報取得
		AjcGetMonitorInfoOfWindow(hwnd, &rs, NULL);
		//	モニタの位置とサイズ設定
		sx	= rs.left;
		sy	= rs.top;
		scx = rs.right - rs.left;
		scy = rs.bottom - rs.top;
		//	ウインドをモニタ中央へ移動
		SetWindowPos(hwnd, NULL, sx + (scx - wcx) / 2, sy + (scy - wcy) / 2, 0, 0, SWP_NOSIZE);

		rc = TRUE;
	}
	return rc;
}
/*==============================================================================================================*/
/*	Function	:	ウインドを他のウインドの中央へ移動する														*/
/*																												*/
/*	Argument	:	hwnd	- ウインドハンドル																	*/
/*					hWndScr - 他のウインドハンドル																*/
/*																												*/
/*	Return		:	TRUE  - OK																					*/
/*					FALSE - NG																					*/
/*==============================================================================================================*/
AJCEXPORT	BOOL	WINAPI	AjcMoveWindowToCenterOfWindow(HWND hwnd, HWND hWndScr)
{
	BOOL	rc = FALSE;
	RECT	rs, rw;
	int		sx, sy, scx, scy;
	int		wx, wy, wcx, wcy;

	if (IsWindow(hwnd) && IsWindow(hWndScr)) {
		GetClientRect(hWndScr, &rs);
		MapWindowPoints(hWndScr, NULL, (LPPOINT)&rs, 2);
		sx = rs.left;	scx = rs.right - rs.left;
		sy = rs.top;	scy = rs.bottom - rs.top;
		GetWindowRect(hwnd, &rw);
		wx = rw.left;	wcx = rw.right - rw.left;
		wy = rw.top;	wcy = rw.bottom - rw.top;
		SetWindowPos(hwnd, NULL, sx + (scx - wcx) / 2, sy + (scy - wcy) / 2, 0, 0, SWP_NOSIZE);
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	フォーカスされているウインドを取得する																		//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	≠ NULL - 成功（フォーカスされているウインドのハンドル）										//
//				＝ NULL - 失敗																					//
//==============================================================================================================//
AJCEXPORT	HWND WINAPI AjcGetFocusWindow(VO) 
{
	HWND	rc;
	DWORD ActiveThreadID;
	DWORD ActiveProcessID;

	ActiveThreadID = GetWindowThreadProcessId(GetForegroundWindow(), &ActiveProcessID);
	if (AttachThreadInput(GetCurrentThreadId(), ActiveThreadID, TRUE)) {
		rc = GetFocus();
		AttachThreadInput(GetCurrentThreadId(), ActiveThreadID, FALSE);
	}
	else {
		rc = GetFocus();
	}
	return rc;
}

//==============================================================================================================//
//	指定ウインドをフォーカスする																				//
//																												//
//	引　数	：	hwnd	- フォーカスするウインドのハンドル														//
//																												//
//	戻り値	：	TRUE  ： 成功																					//
//				FALSE ： 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL WINAPI AjcSetFocusWindow(HWND hwnd) 
{
	BOOL	rc = 0;
	DWORD	ActiveThreadID;
	DWORD	ActiveProcessID;

	if (IsWindow(hwnd)) {
		ActiveThreadID = GetWindowThreadProcessId(GetForegroundWindow(), &ActiveProcessID);
		if (AttachThreadInput(GetCurrentThreadId(), ActiveThreadID, TRUE)) {
			rc = SetForegroundWindow(hwnd);
		//	AttachThreadInput(GetCurrentThreadId(), ActiveThreadID, FALSE);
		}
		else {
			rc = SetForegroundWindow(hwnd);
		}
	}
	return (rc != 0);
}
//==============================================================================================================//
//	通常表示時のウインド矩形情報取得																			//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//				pRect	- 矩形情報を格納するバッファのアドレス													//
//																												//
//	戻り値	：	≠-1：ウインドの状態(SW_HIDE/SW_MINIMIZE/SW_SHOW/SW_SHOWNORMAL・・・)							//
//				＝-1：エラー																					//
//==============================================================================================================//
AJCEXPORT	int	WINAPI	AjcGetWindowNormalRect(HWND hwnd, LPRECT pRect)
{
	int				rc = -1;
	WINDOWPLACEMENT	wpl;

	if (IsWindow(hwnd)) {
		//	ウインド情報取得
		if (GetWindowPlacement(hwnd, &wpl)) {
			//	通常表示時の矩形情報設定
			if (pRect != NULL) {
				memcpy(pRect, &wpl.rcNormalPosition, sizeof(RECT));
			}
			rc = wpl.showCmd;
		}
	}
	return rc;
}
//==============================================================================================================//
//	スレッドの全トップレベルウインドへメッセージ送信															//
//																												//
//	引　数	：	msg		- メッセージコード																		//
//				wParam	- パラメタ																				//
//				lParam	- 　　〃																				//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
typedef struct {
	UI		msg;
	WPARAM	wParam;
	LPARAM	lParam;
	HWND	hWndPre;
	HWND	hWndPost;
} PARAM_SENDTHREAD, *PPARAM_SENDTHREAD;
//--------------------------------------------------------------------------------------------------------------//
static	BOOL CALLBACK cbSendThreadEnum(HWND hwnd, LPARAM lParam)
{
	PPARAM_SENDTHREAD p = (PPARAM_SENDTHREAD)lParam;

	if (GetParent(hwnd) == NULL) {
		SendMessage(hwnd, p->msg, p->wParam, p->lParam);
	}

	return TRUE;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	VO	WINAPI	AjcSendMessageToThreadWindows(UI msg, WPARAM wParam, LPARAM lParam)
{
	PARAM_SENDTHREAD	prm;

	prm.msg		  = msg;
	prm.wParam	  = wParam;
	prm.lParam	  = lParam;
	prm.hWndPre	  = NULL;
	prm.hWndPost  = NULL;
	EnumThreadWindows(GetCurrentThreadId(), cbSendThreadEnum, (LPARAM)&prm);
}
//==============================================================================================================//
//	スレッドの全トップレベルウインドと、指定された２つのウインドへへメッセージ送信								//
//																												//
//	引　数	：	msg		- メッセージコード																		//
//				wParam	- パラメタ																				//
//				lParam	- 　　〃																				//
//				hWndPre - 全スレッドウインドに先駆けてメセージを送信するウインドのハンドル	（不要時はNULL）	//
//				hWndPost- 全スレッドウインドの後にメセージを送信するウインドのハンドル		（不要時はNULL）	//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
static	BOOL CALLBACK cbSendThreadExEnum(HWND hwnd, LPARAM lParam)
{
	PPARAM_SENDTHREAD p = (PPARAM_SENDTHREAD)lParam;

	if (GetParent(hwnd) == NULL && hwnd != p->hWndPre && hwnd != p->hWndPost) {
		SendMessage(hwnd, p->msg, p->wParam, p->lParam);
	}

	return TRUE;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	VO	WINAPI	AjcSendMessageToThreadWindowsEx(UI msg, WPARAM wParam, LPARAM lParam, HWND hWndPre, HWND hWndPost)
{
	PARAM_SENDTHREAD	prm;

	//--- 'Pre'ウインドへメッセージ送信 --------------------------------------//
	if (hWndPre != NULL) {
		SendMessage(hWndPre , msg, wParam, lParam);
	}

	//--- Pre, Postウインド以外へメッセージ送信 ------------------------------//
	prm.msg		  = msg;
	prm.wParam	  = wParam;
	prm.lParam	  = lParam;
	prm.hWndPre	  = hWndPre;
	prm.hWndPost  = hWndPost;
	EnumThreadWindows(GetCurrentThreadId(), cbSendThreadExEnum, (LPARAM)&prm);

	//--- 'Post'ウインドへメッセージ送信 -------------------------------------//
	if (hWndPost != NULL) {
		SendMessage(hWndPost, msg, wParam, lParam);
	}
}
//==============================================================================================================//
//	スレッドの全トップレベルウインドを許可／禁止																//
//																												//
//	引　数	：	fEnable	- TRUE:許可，FALSE:禁止																	//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
static	BOOL CALLBACK cbEnableThreadEnum(HWND hwnd, LPARAM lParam)
{
	PPARAM_SENDTHREAD p = (PPARAM_SENDTHREAD)lParam;

	if (GetParent(hwnd) == NULL) {
		EnableWindow(hwnd, (BOOL)lParam);
	}

	return TRUE;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	VO	WINAPI	AjcEnableThreadWindows(BOOL fEnable)
{
	EnumThreadWindows(GetCurrentThreadId(), cbEnableThreadEnum, fEnable);
}

//==============================================================================================================//
//	Windowsイベント処理（ダイアログメッセージ処理なし）															//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	処理したメッセージ数																			//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI WINAPI AjcDoEventA(VO) 
{
	UI		rc = 0;
	MSG		msg;

	while (PeekMessageA(&msg, NULL, 0, 0, PM_REMOVE)) {
		do {
			TranslateMessage(&msg);
			DispatchMessageA(&msg);
		} while (0);
		rc++;
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI WINAPI AjcDoEventW(VO) 
{
	UI		rc = 0;
	MSG		msg;

	while (PeekMessageW(&msg, NULL, 0, 0, PM_REMOVE)) {
		do {
			TranslateMessage(&msg);
			DispatchMessageW(&msg);
		} while (0);
		rc++;
	}
	return rc;
}
//==============================================================================================================//
//	Windowsイベント処理（ダイアログメッセージ処理付き）															//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	処理したメッセージ数																			//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI WINAPI AjcDoEventExA(UI nDialog, ...) 
{
	UI		rc = 0;
	MSG		msg;
	UI		i;
	HWND	hdlg;
	va_list vls;

	va_start(vls, nDialog);

	while (PeekMessageA(&msg, NULL, 0, 0, PM_REMOVE)) {
		do {
			for (i = 0; i < nDialog; i++) {
				hdlg = va_arg(vls, HWND);
				if (IsWindow(hdlg)) {
					if (IsDialogMessageA(hdlg, &msg)) goto exit_struct;
				}
			}
			TranslateMessage(&msg);
			DispatchMessageA(&msg);
		} while (0);
exit_struct:;
		rc++;
	}

	va_end	(vls);

	return rc;
}

//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI WINAPI AjcDoEventExW(UI nDialog, ...) 
{
	UI		rc = 0;
	MSG		msg;
	UI		i;
	HWND	hdlg;
	va_list vls;

	va_start(vls, nDialog);

	while (PeekMessageW(&msg, NULL, 0, 0, PM_REMOVE)) {
		do {
			for (i = 0; i < nDialog; i++) {
				hdlg = va_arg(vls, HWND);
				if (IsWindow(hdlg)) {
					if (IsDialogMessageW(hdlg, &msg)) goto exit_struct;
				}
			}
			TranslateMessage(&msg);
			DispatchMessageW(&msg);
		} while (0);
exit_struct:;
		rc++;
	}

	va_end	(vls);

	return rc;
}

//==============================================================================================================//
//	ウインドハンドルから、実行ファイルのパス名取得																//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//				pBuf	- 実行ファイルのパス名を格納するバッファのアドレス	（不要時はNULL）					//
//				lBuf	- 実行ファイルのパス名を格納するバッファの文字数										//
//																												//
//	戻り値	：	≠NULL：インスタンスハンドル																	//
//				＝NULL：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT HINSTANCE	WINAPI	AjcGetExePathNameA(HWND hwnd, BCP pBuf, int lBuf)
{
	UI		ProcessId	= 0;
	HANDLE	hProcess	= NULL;
	HMODULE hModule		= NULL;
	UI		dummy		= 0;

	if (IsWindow(hwnd)) {
		do {
			GetWindowThreadProcessId(hwnd, &ProcessId);

			// プロセスハンドル
			if ((hProcess = OpenProcess(PROCESS_QUERY_INFORMATION | PROCESS_VM_READ, FALSE, ProcessId)) == NULL) {
				break;
			}

			// モジュールハンドル
			if(!EnumProcessModules(hProcess, &hModule, sizeof(HMODULE), &dummy)) {
				hModule = NULL;
				break;
			}

			// ファイル名(フルパス)
			if (pBuf != NULL  &&  lBuf > 0) {
				if(!GetModuleFileNameExA(hProcess, hModule, pBuf, lBuf)) {
					hModule = NULL;
					break;
				}
			}
		} while(0);

		CloseHandle(hProcess);
		//	hModuleはクローズしてはいけない
	}
	return hModule;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT HINSTANCE	WINAPI	AjcGetExePathNameW(HWND hwnd, WCP pBuf, int lBuf)
{
	UI		ProcessId	= 0;
	HANDLE	hProcess	= NULL;
	HMODULE hModule		= NULL;
	UI		dummy		= 0;

	if (IsWindow(hwnd)) {
		do {
			GetWindowThreadProcessId(hwnd, &ProcessId);

			// プロセスハンドル
			if ((hProcess = OpenProcess(PROCESS_QUERY_INFORMATION | PROCESS_VM_READ, FALSE, ProcessId)) == NULL) {
				break;
			}

			// モジュールハンドル
			if(!EnumProcessModules(hProcess, &hModule, sizeof(HMODULE), &dummy)) {
				hModule = NULL;
				break;
			}

			// ファイル名(フルパス)
			if (pBuf != NULL  &&  lBuf > 0) {
				if(!GetModuleFileNameExW(hProcess, hModule, pBuf, lBuf)) {
					hModule = NULL;
					break;
				}
			}
		} while(0);

		CloseHandle(hProcess);
		//	hModuleはクローズしてはいけない
	}
	return hModule;
}

//--------------------------------------------------------------------------------------------------------------//
//	接頭語とキー名を合成した名称を作成																			//
//																												//
//	引　数	：	pPrefix	- 接頭語文字列																			//
//				pName	- 名称																					//
//																												//
//	戻り値	：	合成した名称へのポインタ																		//
//--------------------------------------------------------------------------------------------------------------//
#define		MAX_KEY_LEN		256
__declspec(thread) static BC szKeyNameA[MAX_KEY_LEN];
__declspec(thread) static WC szKeyNameW[MAX_KEY_LEN];
//----- バイト文字 ---------------------------------------------------------------------------------------------//
static BCP MkKeyNameA(C_BCP pPrefix, C_BCP pName)
{
	szKeyNameA[0] = 0;

	if (pPrefix != NULL && pName != NULL) {
		AjcSnPrintFA(szKeyNameA, MAX_KEY_LEN, "%s_%s", pPrefix, pName);
	}
	else if (pPrefix != NULL) AjcSnPrintFA(szKeyNameA, MAX_KEY_LEN, "%s", pPrefix);
	else if (pName	 != NULL) AjcSnPrintFA(szKeyNameA, MAX_KEY_LEN, "%s", pName);

	return szKeyNameA;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
static WCP MkKeyNameW(C_WCP pPrefix, C_WCP pName)
{
	szKeyNameW[0] = 0;

	if (pPrefix != NULL && pName != NULL) {
		AjcSnPrintFW(szKeyNameW, MAX_KEY_LEN, L"%s_%s", pPrefix, pName);
	}
	else if (pPrefix != NULL) AjcSnPrintFW(szKeyNameW, MAX_KEY_LEN, L"%s", pPrefix);
	else if (pName	 != NULL) AjcSnPrintFW(szKeyNameW, MAX_KEY_LEN, L"%s", pName);

	return szKeyNameW;
}
//==============================================================================================================//
//	クラス情報設定																								//
//																												//
//	引　数	：	hwnd			- ウインドハンドル																//
//				ix				- 設定する値に対する 0 から始まるオフセット										//
//				data			- 設定値																		//
//																												//
//	戻り値	：	SetClassLongPtr()の戻り値																		//
//==============================================================================================================//
AJCEXPORT UX		WINAPI	AjcSetClassLongPtr(HWND hwnd, UI ix, UX data)
{
	UX		rc = 0;
	if (IsWindowUnicode(hwnd)) 
		rc = SetClassLongPtrW(hwnd, ix, data);
	else					   
		rc = SetClassLongPtrA(hwnd, ix, data);
	return rc;
}
//==============================================================================================================//
//	クラス情報取得																								//
//																												//
//	引　数	：	hwnd			- ウインドハンドル																//
//				ix				- 設定する値に対する 0 から始まるオフセット										//
//				data			- 設定値																		//
//																												//
//	戻り値	：	SetClassLongPtr()の戻り値																		//
//==============================================================================================================//
AJCEXPORT UX		WINAPI	AjcGetClassLongPtr(HWND hwnd, UI ix)
{
	UX		rc = 0;
	if (IsWindowUnicode(hwnd)) 
		rc = GetClassLongPtrW(hwnd, ix);
	else					   
		rc = GetClassLongPtrA(hwnd, ix);
	return rc;
}
//==============================================================================================================//
//	ウインドＬｏｎｇ情報設定																					//
//																												//
//	引　数	：	hwnd			- ウインドハンドル																//
//				ix				- 設定する値に対する 0 から始まるオフセット										//
//				data			- 設定値																		//
//																												//
//	戻り値	：	前回設定のウインドロング情報																	//
//==============================================================================================================//
AJCEXPORT UX	WINAPI	AjcSetWindowLongPtr(HWND hwnd, UI ix, UX data)
{
	UX		rc = 0;
	UI		styWin, styBtn;

	//	ウインドスタイルの設定？
	if (ix == GWL_STYLE) {
		//	サブクラス化したボタンならば、ウインドスタイルとサブクラス化したボタンスタイルを合成
		if (AjcDgcGetSbcButtonStyle(hwnd, NULL)) {
			//	サブクラス化したボタンスタイル設定（設定できるスタイルは限定される）
			styBtn = AjcDgcSetSbcButtonStyle(hwnd, (UI)(data & 0x0000FFFF));
			//	実際のウインドスタイル取得
			if (IsWindowUnicode(hwnd)) styWin = (UI)GetWindowLongPtrW(hwnd, ix);
			else					   styWin = (UI)GetWindowLongPtrA(hwnd, ix);
			//	実際のウインドスタイル設定（但し、ボタンスタイルは変更しない）
			if (IsWindowUnicode(hwnd)) SetWindowLongPtrW(hwnd, ix, (data & 0xFFFF0000) | (styWin & 0x0000FFFF));
			else					   SetWindowLongPtrA(hwnd, ix, (data & 0xFFFF0000) | (styWin & 0x0000FFFF));
			//	戻り値設定（ウインドスタイル（上位16Bit）とボタンスタイル（下位16Bit）の合成）
			rc = ((styWin & 0xFFFF0000) | styBtn);
		}
		//	サブクラス化したボタン以外ならば、ウインドスタイルをそのまま設定
		else {
			if (IsWindowUnicode(hwnd)) rc = SetWindowLongPtrW(hwnd, ix, data);
			else					   rc = SetWindowLongPtrA(hwnd, ix, data);
		}
	}
	//	ウインドスタイル以外の取得？
	else {
		if (IsWindowUnicode(hwnd)) rc = SetWindowLongPtrW(hwnd, ix, data);
		else					   rc = SetWindowLongPtrA(hwnd, ix, data);
	}
	return rc;
}

//==============================================================================================================//
//	ウインドＬｏｎｇ情報取得																					//
//																												//
//	引　数	：	hwnd			- ウインドハンドル																//
//				ix				- 設定する値に対する 0 から始まるオフセット										//
//																												//
//	戻り値	：	ウインドロング情報																				//
//==============================================================================================================//
AJCEXPORT UX	WINAPI	AjcGetWindowLongPtr(HWND hwnd, UI ix)
{
	UX		rc = 0;
	UI		styWin, styBtn;

	//	ウインドスタイルの取得？
	if (ix == GWL_STYLE) {
		//	ウインドスタイル取得
		if (IsWindowUnicode(hwnd)) styWin = (UI)GetWindowLongPtrW(hwnd, ix);
		else					   styWin = (UI)GetWindowLongPtrA(hwnd, ix);
		//	サブクラス化したボタンならば、ウインドスタイルとサブクラス化したボタンスタイルを合成
		if (AjcDgcGetSbcButtonStyle(hwnd, &styBtn)) {
			rc = (styWin & 0xFFFF0000) | styBtn;
		}
		//	サブクラス化したボタン以外ならば、ウインドスタイルをそのまま返す
		else {
			rc = styWin;
		}
	}
	//	ウインドスタイル以外の取得？
	else {
		if (IsWindowUnicode(hwnd)) rc = GetWindowLongPtrW(hwnd, ix);
		else					   rc = GetWindowLongPtrA(hwnd, ix);
	}
	return rc;
}

//==============================================================================================================//
//	デフォルトウインドプロシージャの実行																		//
//																												//
//	引　数	：	hwnd			- ウインドハンドル																//
//				msg				- メッセージ																	//
//				lParam, wParam	- パラメタ																		//
//																												//
//	戻り値	：	DefWindowProc()の戻り値																			//
//==============================================================================================================//
AJCEXPORT LRESULT	WINAPI	AjcDefWindowProc(HWND hwnd, UI msg, WPARAM wParam, LPARAM lParam)
{
	LRESULT	rc;
	if (IsWindowUnicode(hwnd)) 
		rc = DefWindowProcW(hwnd, msg, wParam, lParam);
	else					   
		rc = DefWindowProcA(hwnd, msg, wParam, lParam);
	return rc;
}
//==============================================================================================================//
//	ウインドプロシージャの呼び出し																				//
//																												//
//	引　数	：	wndproc			- ウインドプロシージャ															//
//				hwnd			- ウインドハンドル																//
//				msg				- メッセージ																	//
//				lParam, wParam	- パラメタ																		//
//																												//
//	戻り値	：	CallWindowProc()の戻り値																		//
//==============================================================================================================//
AJCEXPORT LRESULT	WINAPI	AjcCallWindowProc(WNDPROC wndproc, HWND hwnd, UI msg, WPARAM wParam, LPARAM lParam)
{
	LRESULT	rc;
	if (IsWindowUnicode(hwnd)) 
		rc = CallWindowProcW(wndproc, hwnd, msg, wParam, lParam);
	else					   
		rc = CallWindowProcA(wndproc, hwnd, msg, wParam, lParam);
	return rc;
}
//==============================================================================================================//
//	ウインド位置取得																							//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pX, pY		- ウインド位置を格納するバッファのアドレス											//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - 失敗																					//
//==============================================================================================================//
AJCEXPORT BOOL	WINAPI	AjcGetWindowPos(HWND hwnd, UIP pX, UIP pY)
{
	BOOL	rc = FALSE;
	RECT	r;

	if (IsWindow(hwnd) && pX != NULL && pY != NULL) {
		GetWindowRect(hwnd, &r);
		*pX = r.left;
		*pY = r.top;
		rc = TRUE;
	}
	return rc;
}
