﻿#include	"AjcInternal.h"
#include	"AjcXYModemL.h"

#define	INST_ID			0xBBCC569A
#define	IS_MY_INST(P)	(P != NULL && P->InstID == INST_ID)

/*--------------------------------------------------------------------------------------------------------------*/
/*	タイマ／リトライ情報																						*/
/*--------------------------------------------------------------------------------------------------------------*/
#define		TIMRX_REQ_INTERVAL		 1000			//	受信側：受信要求(NAK/C/G)送信インタバル	[ms]  ( t0 )
#define		TIMRX_WAI_REC			10000			//	受信側：レコード／ＥＯＴ受信待ちタイマ	[ms]  ( t1 )

#define		TIMTX_WAI_REQ			60000			//	送信側：受信要求(NAK/C/G)受信待ちタイマ	[ms]  ( t0 )
#define		TIMTX_WAI_ACK			 3000			//	送信側：ＡＣＫ受信待ちタイマ			[ms]  ( t1 )

#define		RTYRX_TX_ACK			3				//	受信側：ＡＣＫ再送回数			 ( r1 )
#define		RTYTX_TX_REC			3				//	送信側：レコード／ＥＯＴ再送回数 ( r1 )

/*--------------------------------------------------------------------------------------------------------------*/
/*	コンスタント																								*/
/*--------------------------------------------------------------------------------------------------------------*/
static	const UB	csC  [] = {'C'	 };
static	const UB	csG  [] = {'G'	 };
static	const UB	csACK[] = {CC_ACK};
static	const UB	csNAK[] = {CC_NAK};
static	const UB	csEOT[] = {CC_EOT};
static	const UB	csCAN[] = {CC_CAN, CC_CAN, CC_CAN};

/*--------------------------------------------------------------------------------------------------------------*/
/*	内部サブ関数																								*/
/*--------------------------------------------------------------------------------------------------------------*/
static VO	SubXymInit (HAJCXYM  pW, UX cbp,
						 VO		(CALLBACK *cbNotice )(UI knd, UX Param, UX cbp),
						 VO		(CALLBACK *cbGetFile)(PAJCXYMFILEINFOV pBuf, UX cbp),
						 VO		(CALLBACK *cbGetData)(VOP pBuf, UI lBuf, UIP pBytes, UX cbp),
						 VO		(CALLBACK *cbSend	)(C_VOP pTxD, UI lTxD, UX cbp));

static	VO	CALLBACK	cbPreFunc(UI pno, int sts, UX bcp);
static	VO	SetDatL  (HAJCXYM pW);
static	VO	DoEvent  (HAJCXYM pW, UI evt);
static	UL	OctToLong(C_UBP pStr);

/*==============================================================================================================*/
/*	インスタンス生成																							*/
/*																												*/
/*	引　数	：	cbp 	 - コールバックパラメタ																	*/
/*				cbNotice - イベント通知関数のアドレス（不要時はＮＵＬＬ）										*/
/*				cbGetFile- ファイル情報取得関数のアドレス														*/
/*				cbGetData- ファイルデータ読み出し関数のアドレス													*/
/*				cbSend	 - データ送信関数のアドレス																*/
/*																												*/
/*	戻り値	：	≠ＮＵＬＬ：ＯＫ（インスタンスハンドル（＝インスタンスワークアドレス））						*/
/*				＝ＮＵＬＬ：エラー																				*/
/*==============================================================================================================*/
/*----- バイト文字 ---------------------------------------------------------------------------------------------*/
AJCEXPORT HAJCXYM	WINAPI	AjcXymCreateA(UX cbp,
										  VO (CALLBACK *cbNotice )(UI knd, UX Param				, UX cbp),
										  VO (CALLBACK *cbGetFile)(PAJCXYMFILEINFOA pBuf		, UX cbp),
										  VO (CALLBACK *cbGetData)(VOP pBuf, UI lBuf, UIP pBytes, UX cbp),
										  VO (CALLBACK *cbSend	 )(C_VOP pTxD, UI lTxD			, UX cbp))
{
	HAJCXYM	pW;

	if ((pW = (HAJCXYM)AJCMEM(sizeof(AJCXYM))) != NULL) {
		SubXymInit(pW, cbp, cbNotice, (AJCXYMCB_GETFILEV)cbGetFile, cbGetData, cbSend);
		pW->fUnicode = FALSE;
	}

	return pW;
}
/*----- ワイド文字 ---------------------------------------------------------------------------------------------*/
AJCEXPORT HAJCXYM	WINAPI	AjcXymCreateW(UX cbp,
										  VO (CALLBACK *cbNotice )(UI knd, UX Param				, UX cbp),
										  VO (CALLBACK *cbGetFile)(PAJCXYMFILEINFOW pBuf		, UX cbp),
										  VO (CALLBACK *cbGetData)(VOP pBuf, UI lBuf, UIP pBytes, UX cbp),
										  VO (CALLBACK *cbSend	 )(C_VOP pTxD, UI lTxD			, UX cbp))
{
	HAJCXYM	pW;

	if ((pW = (HAJCXYM)AJCMEM(sizeof(AJCXYM))) != NULL) {
		SubXymInit(pW, cbp, cbNotice, (AJCXYMCB_GETFILEV)cbGetFile, cbGetData, cbSend);
		pW->fUnicode = TRUE;
	}

	return pW;
}
/*--------------------------------------------------------------------------------------------------------------*/
/*	インスタンス初期設定																						*/
/*																												*/
/*	引　数	：	pW		 - インスタンスワークのアドレス															*/
/*				cbp		 - コールバックパラメタ																	*/
/*				cbNotice - イベント通知関数のアドレス（不要時はＮＵＬＬ）										*/
/*				cbGetFile- ファイル情報取得関数のアドレス														*/
/*				cbGetData- ファイルデータ読み出し関数のアドレス													*/
/*				cbSend	 - データ送信関数のアドレス																*/
/*																												*/
/*	戻り値	：	なし																							*/
/*--------------------------------------------------------------------------------------------------------------*/
static VO	SubXymInit (HAJCXYM  pW, UX cbp,
						 VO		(CALLBACK *cbNotice )(UI knd, UX Param				, UX cbp),
						 VO		(CALLBACK *cbGetFile)(PAJCXYMFILEINFOV pBuf			, UX cbp),
						 VO		(CALLBACK *cbGetData)(VOP pBuf, UI lBuf, UIP pBytes	, UX cbp),
						 VO		(CALLBACK *cbSend	)(C_VOP pTxD, UI lTxD			, UX cbp))
{
	memset(pW, 0, sizeof *pW);
	pW->InstID 		= INST_ID;
	pW->hStc		= AjcStcCreate(XymStsTbl, NULL, XymEvtTbl[0], XYS_NUM, (UX)pW, cbPreFunc, NULL);
	pW->cbp			= cbp;
	pW->cbNotice	= cbNotice;
	pW->cbGetFile	= (AJCXYMCB_GETFILEV)cbGetFile;
	pW->cbGetData	= cbGetData;
	pW->cbSend		= cbSend;

	pW->tx_t0 = TIMTX_WAI_REQ;		pW->tx_r0 = 0;
	pW->tx_t1 = TIMTX_WAI_ACK;		pW->tx_r1 = RTYTX_TX_REC;

	pW->rx_t0 = TIMRX_REQ_INTERVAL;	pW->rx_r0 = 0;
	pW->rx_t1 = TIMRX_WAI_REC;		pW->rx_r1 = RTYRX_TX_ACK;
}
/*==============================================================================================================*/
/*	インスタンス消去																							*/
/*																												*/
/*	引　数	：	pW		 - インスタンスハンドル																	*/
/*																												*/
/*	戻り値	：	なし																							*/
/*==============================================================================================================*/
AJCEXPORT VO	WINAPI	AjcXymDelete(HAJCXYM pW)
{
	if (IS_MY_INST(pW)) {
		if (pW->hStc != NULL) {AjcStcDelete(pW->hStc); pW->hStc = NULL;}
		free(pW);
	}
}
/*==============================================================================================================*/
/*	ファイル送信開始																							*/
/*																												*/
/*	引　数	：	pW		 - インスタンスハンドル																	*/
/*				Protocol - プロトコル種別																		*/
/*																												*/
/*	戻り値	：	ＴＲＵＥ   : ＯＫ																				*/
/*				ＦＡＬＳＥ : エラー																				*/
/*==============================================================================================================*/
AJCEXPORT BOOL	WINAPI	AjcXymTxStart (HAJCXYM	pW, AJCXYMPROTOCOL Protocol)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		if (AjcStcGetCurrentState(pW->hStc) == AJCSTC_STS_IDLE	&&	ISXYP_OK(Protocol)	&&	pW->cbGetFile  &&  pW->cbGetData) {
			pW->bix = 0;																	//	受信バッファクリアー
			if (pW->knd == AJCXYP_YMODEM_G) {												//	Y-MODEM-G
				AjcStcSetTimerInfo(pW->hStc, XYTIM_WAIREQ, pW->tx_t0, XYE_TIMOUT_0_YG);		//	受信要求(NAK/C/G)受信待ちタイマ
				AjcStcSetTimerInfo(pW->hStc, XYTIM_WAIACK, pW->tx_t1, XYE_TIMOUT_1_YG);		//	ＡＣＫ受信待ちタイマ
			}
			else {																			//	Y-MODEM-G以外
				AjcStcSetTimerInfo(pW->hStc, XYTIM_WAIREQ, pW->tx_t0, XYE_TIMOUT_0);		//	受信要求(NAK/C/G)受信待ちタイマ
				AjcStcSetTimerInfo(pW->hStc, XYTIM_WAIACK, pW->tx_t1, XYE_TIMOUT_1);		//	ＡＣＫ受信待ちタイマ
			}
			pW->tim0 = pW->tx_t0;	pW->rty0 = pW->tx_r0;
			pW->tim1 = pW->tx_t1;	pW->rty1 = pW->tx_r1;
			pW->RtyCnt0 = pW->RtyCnt1 = 0;
			AjcStcSetCurrentState(pW->hStc, AJCSTC_STS_IDLE);	/*	→アイドル状態									*/
			pW->knd 	   = (UI)Protocol;						/*	プロトコル識別設定								*/
			SetDatL(pW);										/*	ﾌﾟﾛﾄｺﾙ種別による最大ﾃﾞｰﾀｻｲｽﾞとﾁｪｯｸﾊﾞｲﾄ数を設定	*/

			DoEvent(pW, XYE_REQ_TXSRT);							/*	ファイル送信開始イベント実行					*/

			rc = TRUE;
		}
	}
	return rc;
}

/*==============================================================================================================*/
/*	ファイル受信開始																							*/
/*																												*/
/*	引　数	：	pW		 - インスタンスハンドル																	*/
/*				Protocol - プロトコル種別																		*/
/*																												*/
/*	戻り値	：	ＴＲＵＥ   : ＯＫ																				*/
/*				ＦＡＬＳＥ : エラー																				*/
/*==============================================================================================================*/
AJCEXPORT BOOL	WINAPI	AjcXymRxStart (HAJCXYM	pW, AJCXYMPROTOCOL Protocol)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		if (AjcStcGetCurrentState(pW->hStc) == AJCSTC_STS_IDLE	&&	ISXYP_OK(Protocol)) {
			pW->bix = 0;																			//	受信バッファクリアー
			if (pW->knd == AJCXYP_YMODEM_G) {														//	Y-MODEM-G
				AjcStcSetTimerInfo(pW->hStc, XYTIM_REQCYC, pW->rx_t0, XYE_TIMOUT_0_YG);				//	受信要求(NAK/C/G)送信インタバル
				AjcStcSetTimerInfo(pW->hStc, XYTIM_WAIREC, pW->rx_t1, XYE_TIMOUT_1_YG);				//	レコード／ＥＯＴ受信待ちタイマ
			}
			else {																					//	Y-MODEM-G以外
				AjcStcSetTimerInfo(pW->hStc, XYTIM_REQCYC, pW->rx_t0, XYE_TIMOUT_0);				//	受信要求(NAK/C/G)送信インタバル
				AjcStcSetTimerInfo(pW->hStc, XYTIM_WAIREC, pW->rx_t1, XYE_TIMOUT_1);				//	レコード／ＥＯＴ受信待ちタイマ
			}
			pW->tim0 = pW->rx_t0;	pW->rty0 = pW->rx_r0;
			pW->tim1 = pW->rx_t1;	pW->rty1 = pW->rx_r1;
			pW->RtyCnt0 = pW->RtyCnt1 = 0;
			AjcStcSetCurrentState(pW->hStc, AJCSTC_STS_IDLE);	/*	→アイドル状態									*/
			pW->knd 	   = (UI)Protocol;						/*	プロトコル識別設定								*/
			SetDatL(pW);										/*	ﾌﾟﾛﾄｺﾙ種別による最大ﾃﾞｰﾀｻｲｽﾞ等を設定			*/

			DoEvent(pW, XYE_REQ_RXSRT);							/*	ファイル受信開始イベント実行					*/

			rc = TRUE;
		}
	}
	return rc;
}

/*==============================================================================================================*/
/*	送信時タイマ情報取得																						*/
/*																												*/
/*	引　数	：	pW		- インスタンスハンドル																	*/
/*				t0, r0	- 送信時：未使用																		*/
/*						　受信時：受信要求(NAK/C/G)送出インタバル[ms] と リトライ回数							*/
/*				t1, r1	- 送信時：ＡＣＫ／ＮＡＫ受信待ちタイマ[ms]	  と データ／ＥＯＴ再送限度回数				*/
/*						　受信時：データ／ＥＯＴ受信待ちタイマ[ms]	  と 再送要求(NAK)送出限度回数				*/
/*																												*/
/*	戻り値	：	なし																							*/
/*==============================================================================================================*/
AJCEXPORT	VO	WINAPI	AjcXymGetTxTimeInfo(HAJCXYM pW, UIP t0, UIP r0, UIP t1, UIP r1)
{
	if (IS_MY_INST(pW)) {
		if (t0 != NULL) *t0 = pW->tx_t0;
		if (r0 != NULL) *r0 = pW->tx_r0;
		if (t1 != NULL) *t1 = pW->tx_t1;
		if (r1 != NULL) *r1 = pW->tx_r1;
	}
}

/*==============================================================================================================*/
/*	送信時タイマ情報設定																						*/
/*																												*/
/*	引　数	：	pW		 - インスタンスハンドル																	*/
/*				t0, r0	- 送信時：未使用																		*/
/*						　受信時：受信要求(NAK/C/G)送出インタバル[ms] と リトライ回数							*/
/*				t1, r1	- 送信時：ＡＣＫ／ＮＡＫ受信待ちタイマ[ms]	  と データ／ＥＯＴ再送限度回数				*/
/*						　受信時：データ／ＥＯＴ受信待ちタイマ[ms]	  と 再送要求(NAK)送出限度回数				*/
/*																												*/
/*	戻り値	：	なし																							*/
/*==============================================================================================================*/
AJCEXPORT	VO	WINAPI	AjcXymSetTxTimeInfo(HAJCXYM pW, UI t0, UI r0, UI t1, UI r1)
{
	if (IS_MY_INST(pW)) {
		pW->tx_t0	= t0;
		pW->tx_r0	= r0;
		pW->tx_t1	= t1;
		pW->tx_r1	= r1;
	}
}
/*==============================================================================================================*/
/*	受信時タイマ情報取得																						*/
/*																												*/
/*	引　数	：	pW		- インスタンスハンドル																	*/
/*				t0, r0	- 送信時：未使用																		*/
/*						　受信時：受信要求(NAK/C/G)送出インタバル[ms] と リトライ回数							*/
/*				t1, r1	- 送信時：ＡＣＫ／ＮＡＫ受信待ちタイマ[ms]	  と データ／ＥＯＴ再送限度回数				*/
/*						　受信時：データ／ＥＯＴ受信待ちタイマ[ms]	  と 再送要求(NAK)送出限度回数				*/
/*																												*/
/*	戻り値	：	なし																							*/
/*==============================================================================================================*/
AJCEXPORT	VO	WINAPI	AjcXymGetRxTimeInfo(HAJCXYM pW, UIP t0, UIP r0, UIP t1, UIP r1)
{
	if (IS_MY_INST(pW)) {
		if (t0 != NULL) *t0 = pW->rx_t0;
		if (r0 != NULL) *r0 = pW->rx_r0;
		if (t1 != NULL) *t1 = pW->rx_t1;
		if (r1 != NULL) *r1 = pW->rx_r1;
	}
}

/*==============================================================================================================*/
/*	受信時タイマ情報設定																						*/
/*																												*/
/*	引　数	：	pW		 - インスタンスハンドル																	*/
/*				t0, r0	- 送信時：未使用																		*/
/*						　受信時：受信要求(NAK/C/G)送出インタバル[ms] と リトライ回数							*/
/*				t1, r1	- 送信時：ＡＣＫ／ＮＡＫ受信待ちタイマ[ms]	  と データ／ＥＯＴ再送限度回数				*/
/*						　受信時：データ／ＥＯＴ受信待ちタイマ[ms]	  と 再送要求(NAK)送出限度回数				*/
/*																												*/
/*	戻り値	：	なし																							*/
/*==============================================================================================================*/
AJCEXPORT	VO	WINAPI	AjcXymSetRxTimeInfo(HAJCXYM pW, UI t0, UI r0, UI t1, UI r1)
{
	if (IS_MY_INST(pW)) {
		pW->rx_t0	= t0;
		pW->rx_r0	= r0;
		pW->rx_t1	= t1;
		pW->rx_r1	= r1;
	}
}

/*==============================================================================================================*/
/*	ファイル転送中止																							*/
/*																												*/
/*	引　数	：	pW		 - インスタンスハンドル																	*/
/*																												*/
/*	戻り値	：	なし																							*/
/*==============================================================================================================*/
AJCEXPORT VO	WINAPI	AjcXymStop	   (HAJCXYM  pW)
{
	if (IS_MY_INST(pW)) {
		DoEvent(pW, XYE_REQ_STOP);
	}
}
/*==============================================================================================================*/
/*	シリアル回線からの受信データ投与																			*/
/*																												*/
/*	引　数	：	pW		 - インスタンスハンドル																	*/
/*				rxd 	 - 受信データバイト																		*/
/*																												*/
/*	戻り値	：	なし																							*/
/*==============================================================================================================*/
AJCEXPORT VO	WINAPI	AjcXymPutRxChar(HAJCXYM  pW, UI rxd)
{
	UI		evt = 0xFFFF;
	UI		i;
	UB		sum;

	if (IS_MY_INST(pW)) {
		//----- 送信中（ＹＭＯＤＥＭ－Ｇ以外）----------------------------------------------------------------------*/
		if		(ISXYM_TX) {
			/*------------------------------------------------------------------------------------------------------*/
			/*	ファイル送信時の処理																				*/
			/*------------------------------------------------------------------------------------------------------*/
			switch (rxd) {
				case 'C':															/*	●'Ｃ'受信					*/
								if (pW->knd != AJCXYP_YMODEM_G) {					/*		ＹＭＯＤＥＭ－Ｇ以外？	*/
									if (pW->knd == AJCXYP_XMODEM_SUM) {				/*		　XMODEM(SUM)？			*/
										pW->knd = AJCXYP_XMODEM_CRC;				/*			ﾌﾟﾛﾄｺﾙ変更(XM-CRC)	*/
										SetDatL(pW);								/*			ﾌﾟﾛﾄｺﾙ情報再設定	*/
										if (pW->cbNotice)							/*			ﾌﾟﾛﾄｺﾙ変更通知		*/
											pW->cbNotice(AJCXYN_PROTOCOL, pW->knd, pW->cbp);
									}
									evt = XYE_RXREQ;								/*		  「Ｃ受信」			*/
								}
								break;

				case 'G':															/*	●'Ｇ'受信					*/
								if (pW->knd == AJCXYP_YMODEM_G) {					/*		ＹＭＯＤＥＭ－Ｇ？		*/
									evt = XYE_RXREQ_YG;								/*			「'Ｇ'受信」		*/
								}
								break;

				case CC_ACK:														/*	●ＡＣＫ受信				*/
								if (IS_XMODEM)										/*		ＸＭＯＤＥＭ？			*/
									evt = XYE_RXACK_XM;								/*			「ACK受信(X-MODEM)」*/
								else												/*		ＹＭＯＤＥＭ？			*/
									evt = XYE_RXACK_YM;								/*			「ACK受信(Y-MODEM)」*/
								pW->RtyCnt1 = 0;									/*		ACK待ちﾘﾄﾗｲｶｳﾝﾀｸﾘｱｰ		*/
								break;

				case CC_NAK:														/*	●ＮＡＫ受信				*/
																					/*		受信要求待 && XMODEM？	*/
								if (AjcStcGetCurrentState(pW->hStc) == AJCSTC_STS_TX_RCVREQ && IS_XMODEM) {
									if (pW->knd != AJCXYP_XMODEM_SUM) {				/*		　XMODEM(SUM)以外？		*/
										pW->knd  = AJCXYP_XMODEM_SUM;				/*			ﾌﾟﾛﾄｺﾙ変更(XM-SUM)	*/
										SetDatL(pW);								/*			ﾌﾟﾛﾄｺﾙ情報再設定	*/
										if (pW->cbNotice)							/*			ﾌﾟﾛﾄｺﾙ変更通知		*/
											pW->cbNotice(AJCXYN_PROTOCOL, pW->knd, pW->cbp);
									}
									evt = XYE_RXREQ;								/*			「受信要求受信］	*/
								}
								else {												/*		受信要求待ち以外？		*/
									evt = XYE_RXNAK;								/*			「ＮＡＫ受信」	*/
								}
								break;

				case CC_CAN:														/*	●ＣＡＮ受信				*/
								evt = XYE_RXCAN;									/*		「ＣＡＮ受信」			*/
								break;
			}

		}
		//----- 送信中（ＹＭＯＤＥＭ－Ｇ）--------------------------------------------------------------------------*/
		else if (ISYMG_TX) {
			switch (rxd) {
				case CC_CAN:														/*	●ＣＡＮ受信				*/
								evt = XYE_RXCAN;									/*		「ＣＡＮ受信」			*/
								break;
			}
		}
		//----- 受信中 ---------------------------------------------------------------------------------------------*/
		else if (ISXYM_RX) {
			/*------------------------------------------------------------------------------------------------------*/
			/*	ファイル受信時の処理																				*/
			/*------------------------------------------------------------------------------------------------------*/
			if (pW->bix == 0) {												/*	レコード受信中以外？							*/
				switch (rxd) {
					case CC_SOH:											/*		●ＳＯＨ受信								*/
									pW->DatL = 128;							/*			データ長＝１２８						*/
									pW->RecL = 3 + pW->DatL + pW->ChkL;		/*			レコード長設定							*/
									pW->buf[pW->bix++] = (UB)rxd;			/*			バッファに受信バイト格納				*/
									break;

					case CC_STX:											/*		●ＳＴＸ受信								*/
									if (pW->knd == AJCXYP_XMODEM_SUM ||		/*			XMODEM(SUM/CRC)？						*/
										pW->knd == AJCXYP_XMODEM_CRC) {		/*			・										*/
										pW->knd  = AJCXYP_XMODEM_1K;		/*			ﾌﾟﾛﾄｺﾙ変更(XM-1K)						*/
										SetDatL(pW);						/*			ﾌﾟﾛﾄｺﾙ情報再設定						*/
									}
									pW->DatL = 1024;						/*			レコード長＝１Ｋ						*/
									pW->RecL = 3 + pW->DatL + pW->ChkL;		/*			レコード長設定							*/
									pW->buf[pW->bix++] = (UB)rxd;			/*			バッファに受信バイト格納				*/
									break;

					case CC_EOT:											/*		●ＥＯＴ受信								*/
									if (IS_XMODEM)							/*			ＸＭＯＤＥＭ？							*/
										evt = XYE_RXEOT_XM;					/*				「ＥＯＴ受信(XMODEM)」				*/
									else									/*			ＹＭＯＤＥＭ？							*/
										evt = XYE_RXEOT_YM;					/*				「ＥＯＴ受信(YMODEM)」				*/
									break;

					case CC_CAN:											/*		●ＣＡＮ受信								*/
									evt = XYE_RXCAN;						/*			「ＣＡＮ受信」							*/
									break;
				}
			}
			else {															/*	レコード受信中？								*/
				pW->buf[pW->bix++] = (UB)rxd;								/*		バッファに受信バイトを格納					*/
				if (pW->bix >= (3 + pW->DatL + pW->ChkL)) {					/*		全バイト受信完了？							*/
					if (pW->knd == AJCXYP_YMODEM_G) {						/*			ＹＭＯＤＥＭ－Ｇ？						*/
						if (pW->buf[1] ==  pW->rno			&&				/*				受信レコードチェックＯＫ？			*/
							pW->buf[2] == (UB)(~pW->rno)	&&				/*				・									*/
							AjcChkXMODEM_BE(&pW->buf[3], pW->DatL + 2)) {	/*				・									*/
							evt = XYE_RXREC_OK_YM;							/*					「レコード受信（OK,YMODEM）」	*/
						}
						else {												/*				受信レコードチェックＮＧ？			*/
							evt = XYE_RXREC_NG_YG;							/*					「ﾚｺｰﾄﾞ受信(NG,YMODEM-G)」		*/
						}
					}
					else {													/*			ＹＭＯＤＥＭ－Ｇ以外？					*/
						do {
							if (pW->knd == AJCXYP_XMODEM_SUM) {				/*				ＸＭＯＤＥＭ－ＳＵＭ？				*/
								sum = 0;									/*					チェックサム算出				*/
								for (i=3; i<131; i++) sum += pW->buf[i];	/*					・								*/
								if (sum != pW->buf[131]) {					/*					サムチェックＮＧ？				*/
									evt = XYE_RXREC_NG;						/*						「ﾚｺｰﾄﾞ受信(NG)」			*/
									break;									/*						→ＢＲＥＡＫ				*/
								}
							}
							else {											/*				ＸＭＯＤＥＭ－ＳＵＭ以外？			*/
								if (!AjcChkXMODEM_BE(&pW->buf[3], pW->DatL + 2)) {	/*			ＣＲＣチェックＮＧ？			*/
									evt = XYE_RXREC_NG;						/*						「ﾚｺｰﾄﾞ受信(NG)」			*/
									break;									/*						→ＢＲＥＡＫ				*/
								}
							}
							if (pW->buf[1] ==  (pW->rno - 1) &&				/*				１つ前のレコード番号？				*/
								pW->buf[2] == (UB)(~(pW->rno - 1))) {		/*				・									*/
								evt = XYE_RXREC_BF;							/*					「ﾚｺｰﾄﾞ受信(BF)」				*/
								break;										/*					→ＢＲＥＡＫ					*/
							}
							if (pW->buf[1] !=  pW->rno ||					/*				レコード番号ＮＧ？					*/
								pW->buf[2] != (UB)(~pW->rno)) {				/*				・									*/
								evt = XYE_RXREC_NG;							/*					「ﾚｺｰﾄﾞ受信(NG)」				*/
								break;										/*					→ＢＲＥＡＫ					*/
							}
							if (IS_XMODEM) evt = XYE_RXREC_OK_XM;			/*				すべてＯＫならば「ﾚｺｰﾄﾞ受信(OK)」	*/
							else		   evt = XYE_RXREC_OK_YM;			/*				・									*/
						} while(0);

					}/*if (pW->knd == AJCXYP_YMODEM_G)*/

					if (evt == XYE_RXREC_OK_XM || evt == XYE_RXREC_OK_YM) {	/*			「レコード受信(OK)」？					*/
						pW->RtyCnt1 = 0;									/*				REC待ちﾘﾄﾗｲｶｳﾝﾀｸﾘｱｰ					*/
						if (IS_YMODEM && pW->buf[1] == 0 && pW->buf[3] == 0) {	/*			YMODEM && RNO=0 && ﾃﾞｰﾀ先頭=0？		*/
							evt = XYE_RXREC_NUL;							/*					「ﾚｺｰﾄﾞ受信(ﾇﾙﾍｯﾀﾞ)」			*/
						}
					}
					pW->bix = 0;											/*			バッファインデクスリセット				*/

				}															/*		[END OF 全バイト受信完了]					*/
			}																/*	[END OF レコード受信中／非受信中]				*/
		}
		/*----------------------------------------------------------------------------------------------------------------------*/
		if (evt != 0xFFFF) {												/*	有効なイベントあり？							*/
			DoEvent(pW, evt);												/*		イベント実行								*/
		}
	}
}
/*==============================================================================================================*/
/*	シリアル回線への送信完了を通知（ＹＭＯＤＥＭ－Ｇによるファイル送信時のみコール要）							*/
/*																												*/
/*		　ＹＭＯＤＥＭ－Ｇの場合、すべてのファイルデータを送信後、送信完了を待って、ＥＯＴを送出しますが、		*/
/*		本モジュールでは、データ送信完了を検出しません。（送信完了待ちのタイムアウトも検出しません）			*/
/*		　本関数により、すべてのファイルデータの送信が完了したことを通知する必要があります。					*/
/*																												*/
/*	引　数	：	pW		 - インスタンスハンドル																	*/
/*																												*/
/*	戻り値	：	なし																							*/
/*==============================================================================================================*/
AJCEXPORT VO	WINAPI	AjcXymTxEnd    (HAJCXYM  pW)
{
	if (IS_MY_INST(pW)) {
		if (pW->knd == AJCXYP_YMODEM_G) {
			DoEvent(pW, XYE_TXEND);
		}
	}
}
/*==============================================================================================================*/
/*	処理状態取得																								*/
/*																												*/
/*	引　数	：	pW		 - インスタンスハンドル																	*/
/*																												*/
/*	戻り値	：	処理状態（０：アイドル状態，１～：通信中）														*/
/*==============================================================================================================*/
AJCEXPORT UI	WINAPI	AjcXymGetState(HAJCXYM	pW)
{
	UI		rc = 0;

	if (IS_MY_INST(pW)) {
		rc = AjcStcGetCurrentState(pW->hStc);
	}
	return rc;
}

/*--------------------------------------------------------------------------------------------------------------*/
/*	［共通］プロトコル種別による、最大データ長と、チェックバイト数を設定する。									*/
/*																												*/
/*	引　数	：	pW		 - インスタンスハンドル																	*/
/*																												*/
/*	戻り値	：	なし																							*/
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	SetDatL(HAJCXYM pW)
{
	switch (pW->knd) {
		case AJCXYP_XMODEM_SUM:	pW->MaxL =	128; pW->ChkL = 1;	pW->rno = 1; break;
		case AJCXYP_XMODEM_CRC:	pW->MaxL =	128; pW->ChkL = 2;	pW->rno = 1; break;
		case AJCXYP_XMODEM_1K:	pW->MaxL = 1024; pW->ChkL = 2;	pW->rno = 1; break;
		case AJCXYP_YMODEM_STD:	pW->MaxL = 1024; pW->ChkL = 2;	pW->rno = 0; break;
		case AJCXYP_YMODEM_G:	pW->MaxL = 1024; pW->ChkL = 2;	pW->rno = 0; break;
	}

}
/*--------------------------------------------------------------------------------------------------------------*/
/*	プリセットファンクション																					*/
/*																												*/
/*	引　数	：	pW		 - インスタンスハンドル																	*/
/*																												*/
/*	戻り値	：	なし																							*/
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_TX_REQ		(HAJCXYM pW)			/*	受信要求（NAK / C / G）送信								*/
{
	C_UBP		pTxd;
	UI 			len;

	switch (pW->knd) {
		case AJCXYP_XMODEM_SUM:	pTxd = csNAK; len = sizeof csNAK;	break;
		case AJCXYP_XMODEM_CRC:	pTxd = csC;   len = sizeof csC;		break;
		case AJCXYP_XMODEM_1K:	pTxd = csC;   len = sizeof csC;		break;
		case AJCXYP_YMODEM_STD:	pTxd = csC;   len = sizeof csC;		break;
		case AJCXYP_YMODEM_G:	pTxd = csG;   len = sizeof csG;		break;
	}
	pW->cbSend(pTxd, len, pW->cbp);
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_TX_ACK		(HAJCXYM pW)			/*	ＡＣＫ送信												*/
{
	pW->cbSend(csACK, sizeof csACK, pW->cbp);
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_TX_ACKNYG	(HAJCXYM pW)			/*	ＡＣＫ送信（ＹＭＯＤＥＭ－Ｇ以外）						*/
{
	if (pW->knd != AJCXYP_YMODEM_G) {
		pW->cbSend(csACK, sizeof csACK, pW->cbp);
	}
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_TX_NAK		(HAJCXYM pW)			/*	ＮＡＫ送信												*/
{
	if (pW->RtyCnt1++ >= pW->rty1) {				/*	リトライアウト？	*/
		DoEvent(pW, XYE_RTYOUT_1);					/*		ﾘﾄﾗｲｱｳﾄ			*/
	}
	else {											/*	リトライ範囲内？	*/
		pW->cbSend(csNAK, sizeof csNAK, pW->cbp);	/*		NAK送出			*/
	}
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_TX_EOT		(HAJCXYM pW)			/*	ＥＯＴ送信												*/
{
	pW->cbSend(csEOT, sizeof csEOT, pW->cbp);
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_TX_CAN		(HAJCXYM pW)			/*	ＣＡＮ送信												*/
{
	pW->cbSend(csCAN, sizeof csCAN, pW->cbp);
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_TX_ATT		(HAJCXYM pW)			/*	属性レコード送信										*/
{
	if (pW->DatL == 128) pW->buf[0] = CC_SOH;
	else				 pW->buf[0] = CC_STX;
	pW->buf[1] = 0x00;
	pW->buf[2] = 0xFF;
	AjcSetXMODEM_BE(&pW->buf[3], pW->DatL + 2);
	pW->RecL = pW->DatL + 5;
	pW->cbSend(pW->buf, pW->RecL, pW->cbp);
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_TX_REC		(HAJCXYM pW)			/*	データレコード送信										*/
{
	UI	i;
	UB	sum = 0;

	if (pW->DatL == 128) pW->buf[0] = CC_SOH;
	else				 pW->buf[0] = CC_STX;
	pW->buf[1] =  pW->rno;
	pW->buf[2] = (UB)(~pW->rno);
	if (pW->knd == AJCXYP_XMODEM_SUM) {
		for (i=3; i<131; i++) sum += pW->buf[i];
		pW->buf[131] = sum;
		pW->RecL = pW->DatL + 4;
	}
	else {
		AjcSetXMODEM_BE(&pW->buf[3], pW->DatL + 2);
		pW->RecL = pW->DatL + 5;
	}
	pW->cbSend(pW->buf, pW->RecL, pW->cbp);
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_TX_RTY		(HAJCXYM pW)			/*	レコード再送											*/
{
	if (pW->RtyCnt1++ >= pW->rty1) {				/*	リトライアウト？	*/
		DoEvent(pW, XYE_RTYOUT_1);					/*		ﾘﾄﾗｲｱｳﾄ			*/
	}
	else {											/*	リトライ範囲内？	*/
		pW->cbSend(pW->buf, pW->RecL, pW->cbp);		/*		レコード再送	*/
	}
}

/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_TX_FILE		(HAJCXYM pW)			/*	全ファイルデータ送信									*/
{
	UI		bytes;

	memset(&pW->buf[3], 0x1A, pW->MaxL);						/*	ファイルデータ取得							*/
	pW->cbGetData(&pW->buf[3], pW->MaxL, &bytes, pW->cbp);		/*	・											*/

	while (bytes != 0) {										/*		ＥＯＦまでループ						*/
		if (bytes <= 128) pW->DatL = 128;						/*			データ長設定						*/
		else			  pW->DatL = pW->MaxL;					/*			・									*/

		pW->rno++;												/*			ﾚｺｰﾄﾞ番号更新						*/
		pf_TX_REC(pW);											/*			１レコード送信						*/

		memset(&pW->buf[3], 0x1A, pW->MaxL);					/*			次のファイルデータ取得				*/
		pW->cbGetData(&pW->buf[3], pW->MaxL, &bytes, pW->cbp);	/*			・									*/
	}
}

/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_GET_FILE		(HAJCXYM pW)				/*	ファイル情報取得									*/
{
	AJCXYMFILEINFOV	fi;
	UI				ix;
	UB				str[64];

	
	pW->cbGetFile(&fi, pW->cbp);								/*	コールバック（ファイル情報取得）			*/

	if (fi.pFName != NULL) {									/*	ファイル名あり？							*/
		memset(&pW->buf[3], 0, 1024);							/*		バッファにファイル情報設定				*/
		ix = 3;													/*		・										*/
		if (pW->fUnicode) {										/*		・										*/
			WideCharToMultiByte(CP_ACP, 0, fi.pFName, -1,		/*		・										*/
				&pW->buf[ix], MAX_PATH, NULL, NULL);			/*		・										*/
		}														/*		・										*/
		else {													/*		・										*/
			strncpy(&pW->buf[ix], fi.pFName, MAX_PATH);			/*		・										*/
		}														/*		・										*/
		ix += (UI)(strlen(&pW->buf[ix]) + 1);					/*		・										*/
		strcat (&pW->buf[ix], _ui64toa(fi.size, str, 10));		/*		・										*/
		ix +=  (UI)strlen(&pW->buf[ix]);						/*		・										*/
		pW->buf[ix++] = ' ';									/*		・										*/
		strcat (&pW->buf[ix], _ui64toa(fi.time, str, 8));		/*		・										*/
		if (pW->DatL <= 128) pW->DatL = 128;					/*		属性データ長設定						*/
		else				 pW->DatL = 1024;					/*		・										*/

		if (IS_XMODEM)											/*		イベント（有効なファイル情報取得）実行	*/
			DoEvent(pW, XYE_FI_VALID_XM);						/*		・										*/
		else if (pW->knd == AJCXYP_YMODEM_STD)					/*		・										*/
			DoEvent(pW, XYE_FI_VALID_YM);						/*		・										*/
		else													/*		・										*/
			DoEvent(pW, XYE_FI_VALID_YG);						/*		・										*/
	}
	else {														/*	ファイル名なし？							*/
		memset(&pW->buf[3], 0, 128);							/*		空の属性データ設定						*/
		pW->DatL = 128;											/*		属性データ長＝１２８					*/

		if (IS_XMODEM)											/*		イベント（空のファイル情報取得）実行	*/
			DoEvent(pW, XYE_FI_NULL_XM);						/*		・										*/
		else if (pW->knd == AJCXYP_YMODEM_STD)					/*		・										*/
			DoEvent(pW, XYE_FI_NULL_YM);						/*		・										*/
		else													/*		・										*/
			DoEvent(pW, XYE_FI_NULL_YG);						/*		・										*/

	}
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_GET_DATA		(HAJCXYM pW)			/*	ファイルデータ取得										*/
{
	UI		bytes;

	memset(&pW->buf[3], 0x1A, pW->MaxL);
	pW->cbGetData(&pW->buf[3], pW->MaxL, &bytes, pW->cbp);

	if (bytes != 0) {
		if (bytes <= 128) pW->DatL = 128;
		else			  pW->DatL = pW->MaxL;
		DoEvent(pW, XYE_FDATA);
	}
	else {
		DoEvent(pW, XYE_EOF);
	}
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_INI_RNO		(HAJCXYM pW)			/*	レコード番号初期化										*/
{
	if (IS_XMODEM) pW->rno = 1;
	else		   pW->rno = 0;
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_UPD_RNO		(HAJCXYM pW)			/*	レコード番号更新										*/
{
	pW->rno++;
}
/*--------------------------------------------------------------------------------------------------------------*/
__declspec(thread) static WC TempFNameW[MAX_PATH] = {0};

static	VO	pf_N_FINFO		(HAJCXYM pW)			/*	ファイル情報通知										*/
{
	AJCXYMFILEINFOV	fi;
	size_t			ix;

	if (IS_XMODEM) {
		if (pW->fUnicode) fi.pFName = L"XMODEM.TMP";
		else			  fi.pFName =  "XMODEM.TMP";
		fi.size   = (ULL)0x7FFFFFFFFFFFFFFF;
		fi.time   = 0;
	}
	else {
		if (pW->fUnicode) {
			MultiByteToWideChar(CP_ACP, 0, (BCP)&pW->buf[3], -1, TempFNameW, MAX_PATH);
			fi.pFName = TempFNameW;
		}
		else			  {
			fi.pFName = &pW->buf[3];
		}
		ix = 3 + strlen(&pW->buf[3]) + 1;
		fi.size = _atoi64(&pW->buf[ix]);
		while (_ismbcdigit(pW->buf[ix])  && ix < pW->DatL + 3) {ix++;}
		while (MAjcIsBlankA(pW->buf[ix]) && ix < pW->DatL + 3) {ix++;}
		fi.time = OctToLong(&pW->buf[ix]);
	}
	if (pW->cbNotice) {
		pW->cbNotice(AJCXYN_RXFILEINFO, (UX)(&fi), pW->cbp);
	}
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_N_TXREC		(HAJCXYM pW)			/*	１レコード送信完了通知									*/
{
	if (pW->cbNotice)
		pW->cbNotice(AJCXYN_TXREC, pW->DatL, pW->cbp);
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_N_RXREC		(HAJCXYM pW)			/*	１レコード受信通知										*/
{
	if (pW->cbNotice) {
		if (pW->DatL == 128)
			pW->cbNotice(AJCXYN_RXREC_128, (UX)&pW->buf[3], pW->cbp);
		else
			pW->cbNotice(AJCXYN_RXREC_1K , (UX)&pW->buf[3], pW->cbp);
	}
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_N_RETRY		(HAJCXYM pW)			/*	再送発生通知											*/
{
	if (pW->RtyCnt1 <= pW->rty1) {					/*	リトライアウト見発生？	*/
		if (pW->cbNotice)
			pW->cbNotice(AJCXYN_RETRY, 0, pW->cbp);
	}
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_N_EOF		(HAJCXYM pW)			/*	ＥＯＦ通知												*/
{
	if (pW->cbNotice)
		pW->cbNotice(AJCXYN_EOF, 0, pW->cbp);
}

/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_N_END		(HAJCXYM pW)			/*	ファイル転送完了通知									*/
{
	if (pW->cbNotice)
		pW->cbNotice(AJCXYN_COMPLETE, 0, pW->cbp);
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_N_RX_CAN		(HAJCXYM pW)			/*	ファイル転送中止（ＣＡＮ受信					）通知	*/
{
	if (pW->cbNotice)
		pW->cbNotice(AJCXYN_RX_CAN, 0, pW->cbp);
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_N_USRSTP		(HAJCXYM pW)			/*	ファイル転送中止（ユーザからの中止要求			）通知	*/
{
	if (pW->cbNotice)
		pW->cbNotice(AJCXYN_USERSTOP, 0, pW->cbp);
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_N_TIMOUT		(HAJCXYM pW)			/*	ファイル転送中止（タイムアウト					）通知	*/
{
	if (pW->cbNotice)
		pW->cbNotice(AJCXYN_TIMEOUT, 0, pW->cbp);
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	pf_N_ABORT		(HAJCXYM pW)			/*	ファイル転送中止（YMODEM-Gで不正レコード受信	）通知	*/
{
	if (pW->cbNotice)
		pW->cbNotice(AJCXYN_ABORT, 0, pW->cbp);
}
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	CALLBACK	cbPreFunc(UI pno, int sts, UX cbp)
{
	HAJCXYM	pW = (HAJCXYM)cbp;

	switch (pno) {
		case AJCSTC_PF_TX_REQ:		pf_TX_REQ	(pW);	break;	/*	受信要求（NAK / C / G）送信						*/
		case AJCSTC_PF_TX_ACK:		pf_TX_ACK	(pW);	break;	/*	ＡＣＫ送信										*/
		case AJCSTC_PF_TX_ACKNYG:	pf_TX_ACKNYG(pW);	break;	/*	ＡＣＫ送信（ＹＭＯＤＥＭ－Ｇ以外）				*/
		case AJCSTC_PF_TX_NAK:		pf_TX_NAK	(pW);	break;	/*	ＮＡＫ送信										*/
		case AJCSTC_PF_TX_EOT:		pf_TX_EOT	(pW);	break;	/*	ＥＯＴ送信										*/
		case AJCSTC_PF_TX_CAN:		pf_TX_CAN	(pW);	break;	/*	ＣＡＮ送信										*/
		case AJCSTC_PF_TX_ATT:		pf_TX_ATT	(pW);	break;	/*	属性レコード送信								*/
		case AJCSTC_PF_TX_REC:		pf_TX_REC	(pW);	break;	/*	データレコード送信								*/
		case AJCSTC_PF_TX_FILE:		pf_TX_FILE	(pW);	break;	/*	全ファイルデータ送信							*/
		case AJCSTC_PF_TX_RTY:		pf_TX_RTY	(pW);	break;	/*	リトライチェック＆レコード再送					*/

		case AJCSTC_PF_GET_FILE:	pf_GET_FILE	(pW);	break;	/*	ファイル情報取得								*/
		case AJCSTC_PF_GET_DATA:	pf_GET_DATA	(pW);	break;	/*	ファイルデータ取得								*/

		case AJCSTC_PF_INI_RNO:		pf_INI_RNO	(pW);	break;	/*	レコード番号初期化								*/
		case AJCSTC_PF_UPD_RNO:		pf_UPD_RNO	(pW);	break;	/*	レコード番号更新								*/

		case AJCSTC_PF_N_FINFO:		pf_N_FINFO	(pW);	break;	/*	ファイル情報通知								*/
		case AJCSTC_PF_N_TXREC:		pf_N_TXREC	(pW);	break;	/*	１レコード送信完了通知							*/
		case AJCSTC_PF_N_RXREC:		pf_N_RXREC	(pW);	break;	/*	１レコード受信通知								*/
		case AJCSTC_PF_N_RETRY:		pf_N_RETRY	(pW);	break;	/*	再送発生通知									*/
		case AJCSTC_PF_N_EOF:		pf_N_EOF	(pW);	break;	/*	ＥＯＦ通知										*/

		case AJCSTC_PF_N_END:		pf_N_END	(pW);	break;	/*	ファイル転送完了通知							*/
		case AJCSTC_PF_N_RX_CAN:	pf_N_RX_CAN	(pW);	break;	/*	ファイル転送中止通知（ＣＡＮ受信			 ）	*/
		case AJCSTC_PF_N_USRSTP:	pf_N_USRSTP	(pW);	break;	/*	　〃　　　　　　通知（ユーザからの中止要求	 ）	*/
		case AJCSTC_PF_N_TIMOUT:	pf_N_TIMOUT	(pW);	break;	/*	　〃　　　　　　通知（タイムアウト			 ）	*/
		case AJCSTC_PF_N_ABORT:		pf_N_ABORT	(pW);	break;	/*	　〃　　　　　　通知（その他				 ）	*/
	}
}
/*--------------------------------------------------------------------------------------------------------------*/
/*	イベント実行																								*/
/*																												*/
/*	引　数	：	pW	   - インスタンスハンドル																	*/
/*				evt    - イベント番号																			*/
/*																												*/
/*	戻り値	：	なし																							*/
/*--------------------------------------------------------------------------------------------------------------*/
static	VO	DoEvent(HAJCXYM pW, UI evt)
{
	AjcStcExecEvent(pW->hStc, evt);
}
/*--------------------------------------------------------------------------------------------------------------*/
/*	８進文字列→数値変換																						*/
/*																												*/
/*	引　数	：	pW	   - インスタンスハンドル																	*/
/*				evt    - イベント番号																			*/
/*																												*/
/*	戻り値	：	なし																							*/
/*--------------------------------------------------------------------------------------------------------------*/
static	UL	OctToLong(C_UBP pStr)
{
	UL		rc = 0;

	while (*pStr >= '0' && *pStr <= '7') {
		rc <<= 3;
		rc |= ( *pStr & 15);
		pStr++;
	}
	return rc;
}

