﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Drawing.Design;
using System.Windows.Forms.Design;
using AjaraInit;

namespace CAjrCustCtrl
{
	//---------------------------------------------------------------------------------------------------------//
    [ToolboxBitmap(typeof(CAjr3DGraphic), "CAjr3dGraphic.bmp")]
	public partial class CAjr3DGraphic : UserControl
	{
		//----- イベント ----------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Occurs when dropping files.")]
		public event G3dEvtFileDrop OnFileDrop;

		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Occurs when dropping directories.")]
		public event G3dEvtDirDrop	OnDirDrop;

		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Occurs when Shift/Ctrl + right-clicked.")]
		public event G3dEvtRClick	OnRClick;

		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Occurs when you click on a plot point.")]
		public event G3dEvtPltLst	OnPltLst;


		private G3dCbkFileDrop	m_G3dCbkFileDrop;
		private G3dCbkDirDrop	m_G3dCbkDirDrop;
		private G3dCbkRClick	m_G3dCbkRClick;
		private G3dCbkPltLst	m_G3dCbkPltLst;

		//----- プロパティ値 ------------------------------//
		private	bool		m_EnablePopupMenu	= true;
		private bool		m_ShowDummyData		= true;
		private bool		m_2DMode			= false;
		private bool		m_EnableMesDraw		= false;

		private string		m_TitleText			= "";
		private bool		m_ToolTipShowAlways = true;
		private string		m_TipText			= "";
		private string[]	m_TipFilter 		= new string[16];

        private EAJCPLANEAXIS m_PlaneHoriAxis = EAJCPLANEAXIS.XP;
        private EAJCPLANEAXIS m_PlaneVertAxis = EAJCPLANEAXIS.YP;

        private Font m_TxoFont = new Font("MS UI Gothic", 9);

		//----- ワーク ------------------------------------//
		private IntPtr		m_hCtrl;
		private	bool		m_fFirst  = true;

		//----- コンストラクタ ---------------------------------------------------------------------------------//
		public CAjr3DGraphic()
		{
			InitializeComponent();

			//----- ライブラリ初期化 -------------------------------------------//
			CAjrInit.Init();

			//----- コントロールウインド生成 -----------------------------------//
			m_G3dCbkFileDrop	= new G3dCbkFileDrop(cbG3dFileDrop	);
			m_G3dCbkDirDrop	 	= new G3dCbkDirDrop	(cbG3dDirDrop	);
			m_G3dCbkRClick		= new G3dCbkRClick	(cbG3dRClick	);
			m_G3dCbkPltLst		= new G3dCbkPltLst	(cbG3dPltLst	);

			m_hCtrl = CAjrApi.AjcG3dCreateForMSIL(this.Handle, m_G3dCbkFileDrop,
															   m_G3dCbkDirDrop	,
															   m_G3dCbkRClick	,
															   m_G3dCbkPltLst	);
		}

		//------------------------------------------------------------------------------------------------------//
		//	コントロール内部イベント																			//
		//------------------------------------------------------------------------------------------------------//
		//----- 初期設定 ---------------------------------------------------------------------------------------//
		private void CAjr3DGraphic_Load(object sender, EventArgs e)
		{
		}
		//----- コントロール・サイズ変更 -----------------------------------------------------------------------//
		private void CAjr3DGraphic_Resize(object sender, EventArgs e)
		{
			//	コントロールのサイズ設定
			CAjrApi.AjcG3dSetProp_WndSize(m_hCtrl, (int)this.Size.Width, (int)this.Size.Height);

			//	初回のみの処理
			if (m_fFirst) {
				m_fFirst = false;
				//	デザインモード時ダミーデータ表示
				if (this.DesignMode) {
					GenDummyData();
				}
				//	２Ｄモード設定
				else {
					if (m_2DMode) {
						CAjrApi.AjcG3dSetProp_Set2dMode(m_hCtrl);
					}
				}
			}
		}
		//------------------------------------------------------------------------------------------------------//
		//	プロパティ																							//
		//------------------------------------------------------------------------------------------------------//
        //----- Fontプロパティを隠す ---------------------------------------------------------------------------//
        private Font vfontInit = new Font("MS UI Gothic", 9);
        [Browsable(false)]
        public override Font Font
        {
            set { base.Font = vfontInit; }
            get { return vfontInit; }
        }
        //----- FontTxoプロパティ ------------------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Font when TextOut()")]
		[Editor(typeof(FontTxoEditor), typeof(System.Drawing.Design.UITypeEditor))]
		public Font FontTxo
		{
            set {m_TxoFont = value;
                 CAjrApi.AjcG3dSetTextFont(m_hCtrl, value.ToHfont());
            }
			get {return m_TxoFont; }
		}
		//----- AllowDropプロパティを隠す ----------------------------------------------------------------------//
		private bool AllowDropInit = false;
		[Browsable(false)]
		public override bool AllowDrop
		{
			set { base.AllowDrop = AllowDropInit; }
			get { return AllowDropInit; }
		}
		//----- Enabledプロパティを変更 ------------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Enable / Disable the control")]
		public new bool Enabled
		{
			set {	if (base.Enabled == value) base.Enabled = !value;
					base.Enabled = value;
					CAjrApi.AjcEnableCtrlsInWnd(this.Handle, value);
				}
			get {	return base.Enabled;  }
		}
		//----- デザイン時のダミーデータ表示／非表示 -----------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Dummy data is shown at designing")]
		public bool ShowDummyData {
			set { m_ShowDummyData = value; if (this.DesignMode) GenDummyData(); }
			get { return m_ShowDummyData; }
		}
		//----- ２Ｄ／３Ｄモード識別（識別表示用）--------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("2D / 3D mode selection(This property can be set only at designing)")]
		public EDIMMODE	_DimMode {
			set {
				if (this.DesignMode) {
					//----- ２Ｄモード初期化 -----------------------------------------------------------------------//
					if		(value == EDIMMODE.MODE_2D) {
						double xc = this.CenterX;
						double yc = this.CenterY;
						double zc = this.CenterZ;
						double xr = this.RadiusX;
						double yr = this.RadiusY;
						double zr = this.RadiusZ;
						ViewVolumeRatio = 1.0;
						CAjrApi.AjcG2dInit(m_hCtrl, xc - xr, yc - yr,
												  xc + xr, yc + yr, 0xE400);
						this.Mode2D = true;
						GenDummyData();
					}
					//----- ３Ｄモード初期化 -----------------------------------------------------------------------//
					else if (value == EDIMMODE.MODE_3D) {
						double xc = this.CenterX;
						double yc = this.CenterY;
						double zc = this.CenterZ;
						double xr = this.RadiusX;
						double yr = this.RadiusY;
						double zr = this.RadiusZ;
						ViewVolumeRatio = 0.7;
						CAjrApi.AjcG3dInit(m_hCtrl, xc - xr, yc - yr, zc - zr,
												  xc + xr, yc + yr, zc + zr, 0x700F);
						this.Mode2D = false;
						GenDummyData();
					}
				}
			}
			get {return (m_2DMode ? EDIMMODE.MODE_2D : EDIMMODE.MODE_3D);}
		}
		//----- ２Ｄ／３Ｄモードフラグ（非表示プロパティ）------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[Browsable(false)]
		public bool Mode2D {
			set {m_2DMode = value;}
			get {return m_2DMode;}
		}
		//----- 方眼スケール表示スタイル -----------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Section scale display (Specifies a display-surface by 'ShowScale - XY / XZ / YZ' )")]
		public bool ShowRectScale {
			set {CAjrApi.AjcG3dSetSty_RECTSCALE(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetSty_RECTSCALE(m_hCtrl);}
		}
		//----- Ｘ軸スケール値表示スタイル ---------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("X axis scale value display")]
		public bool ShowScaleValueX {
			set {CAjrApi.AjcG3dSetSty_SCALEVALUE_X(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetSty_SCALEVALUE_X(m_hCtrl);}
		}
		//----- Ｙ軸スケール値表示スタイル ---------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Y axis scale value display")]
		public bool ShowScaleValueY {
			set {CAjrApi.AjcG3dSetSty_SCALEVALUE_Y(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetSty_SCALEVALUE_Y(m_hCtrl);}
		}
		//----- Ｚ軸スケール値表示スタイル ---------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Y axis scale value display")]
		public bool ShowScaleValueZ {
			set {CAjrApi.AjcG3dSetSty_SCALEVALUE_Z(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetSty_SCALEVALUE_Z(m_hCtrl);}
		}

		//----- 同心円スケール表示スタイル ---------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("concentric circle scale display (Specifies a display-surface by 'ShowScale - XY / XZ / YZ' )")]
		public bool ShowEllipseScale {
			set {CAjrApi.AjcG3dSetSty_ELPSSCALE(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetSty_ELPSSCALE(m_hCtrl);}
		}
		//----- ＸＹ平面にスケール表示スタイル -----------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Scale value display on XY plane")]
		public bool ShowScaleXY {
			set {CAjrApi.AjcG3dSetSty_SCALE_XY(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetSty_SCALE_XY(m_hCtrl);}
		}
		//----- ＸＺ平面にスケール表示スタイル -----------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Scale value display on XZ plane")]
		public bool ShowScaleXZ {
			set {CAjrApi.AjcG3dSetSty_SCALE_XZ(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetSty_SCALE_XZ(m_hCtrl);}
		}
		//----- ＹＺ平面にスケール表示スタイル -----------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Scale value display on YZ plane")]
		public bool ShowScaleYZ {
			set {CAjrApi.AjcG3dSetSty_SCALE_YZ(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetSty_SCALE_YZ(m_hCtrl);}
		}

		//----- 奥行表現スタイル -------------------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Enable depth expression")]
		public bool EnableDepthControl {
			set {CAjrApi.AjcG3dSetSty_NODEPTHCTRL(m_hCtrl, !value);	if (this.DesignMode) GenDummyData();}
			get {return !CAjrApi.AjcG3dGetSty_NODEPTHCTRL(m_hCtrl);}
		}
		//----- 外枠表示スタイル -------------------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Outer frame display")]
		public bool ShowBorder {
			set {CAjrApi.AjcG3dSetSty_NOBORDER(m_hCtrl, !value);}
			get {return !CAjrApi.AjcG3dGetSty_NOBORDER(m_hCtrl);}
		}
		//----- フィルタ表示スタイル ---------------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Filter check-box display")]
		public bool ShowFilter {
			set {CAjrApi.AjcG3dSetSty_NOFILTER(m_hCtrl, !value);}
			get {return !CAjrApi.AjcG3dGetSty_NOFILTER(m_hCtrl);}
		}
		//----- 視点設定／ドラッグ許可スタイル -----------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Permission of viewpoint setting by drag operation")]
		public bool EnableAngle {
			set {CAjrApi.AjcG3dSetSty_NOANGLE(m_hCtrl, !value);}
			get {return !CAjrApi.AjcG3dGetSty_NOANGLE(m_hCtrl);}
		}

		//----- 球体スケール表示スタイル -----------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Sphere scale display")]
		public bool ShowSphereScale {
			set {CAjrApi.AjcG3dSetSty_SPHERESCALE(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetSty_SPHERESCALE(m_hCtrl);}
		}
		//----- Ｘ軸表示スタイル -------------------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("X axis display")]
		public bool ShowAxisX {
			set {CAjrApi.AjcG3dSetSty_SHOWAXIS_X(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetSty_SHOWAXIS_X(m_hCtrl);}
		}
		//----- Ｙ軸表示スタイル -------------------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Y axis display")]
		public bool ShowAxisY {
			set {CAjrApi.AjcG3dSetSty_SHOWAXIS_Y(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetSty_SHOWAXIS_Y(m_hCtrl);}
		}
		//----- Ｚ軸表示スタイル -------------------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Z axis display")]
		public bool ShowAxisZ {
			set {CAjrApi.AjcG3dSetSty_SHOWAXIS_Z(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetSty_SHOWAXIS_Z(m_hCtrl);}
		}

		//----- 回転角度Ｘ 							(Rot.x) ----------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Rotation angle of X axis")]
		public double RotateX {
			set {CAjrApi.AjcG3dSetProp_Rot_x(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetProp_Rot_x(m_hCtrl);}
		}
		//----- 回転角度Ｙ 							(Rot.y) ----------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Rotation angle of Y axis")]
		public double RotateY {
			set {CAjrApi.AjcG3dSetProp_Rot_y(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetProp_Rot_y(m_hCtrl);}
		}
		//----- 回転角度Ｚ 							(Rot.z) ----------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Rotation angle of Z axis")]
		public double RotateZ {
			set {CAjrApi.AjcG3dSetProp_Rot_z(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetProp_Rot_z(m_hCtrl);}
		}
		//----- 中心位置Ｘ 							(Rot.x) ----------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Center position of X axis")]
		public double CenterX {
			set {	CAjrApi.AjcG3dSetProp_Cent_x(m_hCtrl, value);	
					if (m_2DMode) CAjrApi.AjcG3dSetProp_Set2dMode(m_hCtrl);
					if (this.DesignMode) GenDummyData();
				 }
			get {return CAjrApi.AjcG3dGetProp_Cent_x(m_hCtrl);}
		}
		//----- 中心位置Ｙ 							(Rot.y) ----------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Center position of Y axis")]
		public double CenterY {
			set {	CAjrApi.AjcG3dSetProp_Cent_y(m_hCtrl, value);
					if (m_2DMode) CAjrApi.AjcG3dSetProp_Set2dMode(m_hCtrl);
					if (this.DesignMode) GenDummyData();
				}
			get {return CAjrApi.AjcG3dGetProp_Cent_y(m_hCtrl);}
		}
		//----- 中心位置Ｚ 							(Rot.z) ----------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Center position of Z axis")]
		public double CenterZ {
			set {	CAjrApi.AjcG3dSetProp_Cent_z(m_hCtrl, value);
					if (m_2DMode) CAjrApi.AjcG3dSetProp_Set2dMode(m_hCtrl);
					if (this.DesignMode) GenDummyData();
				}
			get {return CAjrApi.AjcG3dGetProp_Cent_z(m_hCtrl);}
		}
		//----- 半径Ｘ 							(xr) -----------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Radius of X axis")]
		public double RadiusX {
			set {
					CAjrApi.AjcG3dSetProp_xr(m_hCtrl, value);
					if (this.DesignMode) GenDummyData();
				}
			get {return CAjrApi.AjcG3dGetProp_xr(m_hCtrl);}
		}
		//----- 半径Ｙ 							(yr) -----------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Radius of Y axis")]
		public double RadiusY {
			set {
					CAjrApi.AjcG3dSetProp_yr(m_hCtrl, value);
					if (this.DesignMode) GenDummyData();
				}
			get {return CAjrApi.AjcG3dGetProp_yr(m_hCtrl);}
		}
		//----- 半径Ｚ 							(zr) -----------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Radius of Z axis")]
		public double RadiusZ {
			set {
					CAjrApi.AjcG3dSetProp_zr(m_hCtrl, value);
					if (this.DesignMode) GenDummyData();
				}
			get {return CAjrApi.AjcG3dGetProp_zr(m_hCtrl);}
		}
		//----- ビューボリュームの割合			(ratio) --------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("view volume ratio")]
		public double ViewVolumeRatio {
			set {CAjrApi.AjcG3dSetProp_ratio(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetProp_ratio(m_hCtrl);}
		}
		//----- プロットデータのピクセルサイズ	(PlotSize) -----------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Pixel size of plot data")]
		public int PlotSize {
			set {CAjrApi.AjcG3dSetProp_PlotSize(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetProp_PlotSize(m_hCtrl);}
		}
		//----- 最終プロットデータのピクセルサイズ	(PlotSizeE) ------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Pixel size of last plot data")]
		public int PlotSizeE {
			set {CAjrApi.AjcG3dSetProp_PlotSizeE(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetProp_PlotSizeE(m_hCtrl);}
		}
		//----- プロットデータ結線					(PlotLine) -------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Displays line among the plot data")]
		public bool PlotLine {
			set {CAjrApi.AjcG3dSetProp_fPlotLine(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetProp_fPlotLine(m_hCtrl);}
		}
		//----- アスペクト１保持					(fAspect1) -------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Keeps aspect ratio at 1")]
		public bool Aspect1 {
			set {CAjrApi.AjcG3dSetProp_fAspect1(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetProp_fAspect1(m_hCtrl);}
		}
		//----- ファイル，フォルダ ドロップの許可／禁止 --------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Allow derop the file/folder")]
		public bool AcceptFiles
		{
			set {CAjrApi.AjcG3dSetProp_AcceptFiles(m_hCtrl, value);}
			get {return CAjrApi.AjcG3dGetProp_AcceptFiles(m_hCtrl);}
		}
		//----- ポップアップメニューの許可／禁止 --------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Enable popup menu by right click")]
		public bool EnablePopupMenu {
			set {m_EnablePopupMenu = value; CAjrApi.AjcG3dEnablePopupMenu(m_hCtrl, value);}
			get {return m_EnablePopupMenu;}
		}
		//----- ツールチップ表示条件 ---------------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Tooltip display condition. If false is set, the tooltip will not be displayed when inactive.")]
		public bool ToolTipShowAlways
		{
			set {
				m_ToolTipShowAlways = value;
				CAjrApi.AjcG3dSetTipShowAlways(m_hCtrl, m_ToolTipShowAlways);
				for (int i = 0; i < m_TipFilter.Length; i++) {
					CAjrApi.AjcG3dSetChkBoxTipShowAlways(m_hCtrl, i, m_ToolTipShowAlways);
				}
			}
			get {
				return m_ToolTipShowAlways;
			}
		}
		//----- ツールチップテキスト ---------------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Tool tip text of this control")]
		public string ToolTipText
		{
			set	{
					if (this.DesignMode)  m_TipText = value;
					else {
				 		StringBuilder s = new StringBuilder(2048);
				 		CAjrApi.AjcG3dGetTipTextForMSIL(m_hCtrl, s, (int)s.Capacity);
						if (value != s.ToString()) {
							CAjrApi.AjcG3dSetTipTextForMSIL(m_hCtrl, value);
							CAjrApi.AjcG3dSetTipShowAlways(m_hCtrl, m_ToolTipShowAlways);
						}
					}
				}
			get	{
					string rc = "";
					if (this.DesignMode) rc = m_TipText;
					else {
				 		StringBuilder s = new StringBuilder(2048);
				 		CAjrApi.AjcG3dGetTipTextForMSIL(m_hCtrl, s, (int)s.Capacity);
				 		rc = s.ToString();
				 	}
				 	return rc;
				}
		}
		//----- フィルタ・チェックボックスＮのツールチップ設定／取得 -------------------------------------------//
		private void SetToolTipFilter(string s, int ix)
		{
			if (this.DesignMode) m_TipFilter[ix] = s;
			else {
				if (s != GetToolTipFilter(ix)) {
					CAjrApi.AjcG3dSetChkBoxTipTextForMSIL(m_hCtrl, ix, s);
					CAjrApi.AjcG3dSetChkBoxTipShowAlways (m_hCtrl, ix, m_ToolTipShowAlways);
				}
			}
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		private string GetToolTipFilter(int ix)
		{
			string rc = "";
			if (this.DesignMode) rc = m_TipFilter[ix];
			else {
				StringBuilder s = new StringBuilder(2048);
				CAjrApi.AjcG3dGetChkBoxTipTextForMSIL(m_hCtrl, ix, s, (int)s.Capacity);
				rc = s.ToString();
			}
			return rc;
		}
		//----- フィルタ・チェックボックス( 0) のツールチップ --------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 0")]
		public string ToolTipFilter00
		{
			set {SetToolTipFilter(value, 0);}
			get {return GetToolTipFilter(0);}
		}
		//----- フィルタ・チェックボックス( 1) のツールチップ --------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 1")]
		public string ToolTipFilter01
		{
			set {SetToolTipFilter(value, 1);}
			get {return GetToolTipFilter(1);}
		}
		//----- フィルタ・チェックボックス( 2) のツールチップ --------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 2")]
		public string ToolTipFilter02
		{
			set {SetToolTipFilter(value, 2);}
			get {return GetToolTipFilter(2);}
		}
		//----- フィルタ・チェックボックス( 3) のツールチップ -------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 3")]
		public string ToolTipFilter03
		{
			set {SetToolTipFilter(value, 3);}
			get {return GetToolTipFilter(3);}
		}
		//----- フィルタ・チェックボックス( 4) のツールチップ --------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 4")]
		public string ToolTipFilter04
		{
			set {SetToolTipFilter(value, 4);}
			get {return GetToolTipFilter(4);}
		}
		//----- フィルタ・チェックボックス( 5) のツールチップ --------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 5")]
		public string ToolTipFilter05
		{
			set {SetToolTipFilter(value, 5);}
			get {return GetToolTipFilter(5);}
		}
		//----- フィルタ・チェックボックス( 6) のツールチップ --------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 6")]
		public string ToolTipFilter06
		{
			set {SetToolTipFilter(value, 6);}
			get {return GetToolTipFilter(6);}
		}
		//----- フィルタ・チェックボックス( 7) のツールチップ --------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 7")]
		public string ToolTipFilter07
		{
			set {SetToolTipFilter(value, 7);}
			get {return GetToolTipFilter(7);}
		}
		//----- フィルタ・チェックボックス( 8) のツールチップ --------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 8")]
		public string ToolTipFilter08
		{
			set {SetToolTipFilter(value, 8);}
			get {return GetToolTipFilter(8);}
		}
		//----- フィルタ・チェックボックス( 9) のツールチップ --------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 9")]
		public string ToolTipFilter09
		{
			set {SetToolTipFilter(value, 9);}
			get {return GetToolTipFilter(9);}
		}
		//----- フィルタ・チェックボックス(10) のツールチップ --------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 10")]
		public string ToolTipFilter10
		{
			set {SetToolTipFilter(value, 10);}
			get {return GetToolTipFilter(10);}
		}
		//----- フィルタ・チェックボックス(11) のツールチップ --------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 11")]
		public string ToolTipFilter11
		{
			set {SetToolTipFilter(value, 11);}
			get {return GetToolTipFilter(11);}
		}
		//----- フィルタ・チェックボックス(12) のツールチップ --------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 12")]
		public string ToolTipFilter12
		{
			set {SetToolTipFilter(value, 12);}
			get {return GetToolTipFilter(12);}
		}
		//----- フィルタ・チェックボックス(13) のツールチップ --------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 13")]
		public string ToolTipFilter13
		{
			set {SetToolTipFilter(value, 13);}
			get {return GetToolTipFilter(13);}
		}
		//----- フィルタ・チェックボックス(14) のツールチップ --------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 14")]
		public string ToolTipFilter14
		{
			set {SetToolTipFilter(value, 14);}
			get {return GetToolTipFilter(14);}
		}
		//----- フィルタ・チェックボックス(15) のツールチップ --------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 15")]
		public string ToolTipFilter15
		{
			set {SetToolTipFilter(value, 15);}
			get {return GetToolTipFilter(15);}
		}
		//----- 描画時間計測情報の許可／禁止プロパティ ---------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Enable the 'Display drawing time information' in popup-menu")]
		public bool EnableMesDraw
		{
			set { m_EnableMesDraw = value;
				  CAjrApi.AjcG3dEnableMesDraw(m_hCtrl, value); }
			get { return m_EnableMesDraw; }
		}
		//----- タイトルテキストプロパティ ---------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Title text. It is displayed in the upper right corner of the control.")]
		public string TitleText
		{
			set { m_TitleText = value;
				  SetTitleText(m_TitleText);}
			get { return m_TitleText; }
		}
		//----- 平面表示時の横軸の種別 -------------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Type of horizontal axis in plane display.(ZP, ZM cannot be specified in 2D mode)")]
		public EAJCPLANEAXIS PlaneHoriAxis
		{
			set { m_PlaneHoriAxis = value;
				  if (m_2DMode) {
					  CAjrApi.AjcG3dSetPlane(m_hCtrl, (int)m_PlaneHoriAxis, (int)m_PlaneVertAxis);
				  }
			}
			get { return m_PlaneHoriAxis; }
		}
		//----- 平面表示時の縦軸の種別 -------------------------------------------------------------------------//
		[System.ComponentModel.Category("Ajr3D-Graphic")]
		[System.ComponentModel.Description("Type of vertical axis in plane display.(ZP, ZM cannot be specified in 2D mode)")]
		public EAJCPLANEAXIS PlaneVertAxis
		{
			set { m_PlaneVertAxis = value;
				  if (m_2DMode) {
					  CAjrApi.AjcG3dSetPlane(m_hCtrl, (int)m_PlaneHoriAxis, (int)m_PlaneVertAxis);
				  }
			}
			get { return m_PlaneVertAxis; }
		}
		//------------------------------------------------------------------------------------------------------//
		//	メソッド																							//
		//------------------------------------------------------------------------------------------------------//
		//----- プロットデータ投与 -----------------------------------------------------------------------------//
		//	3D	//
		public void PutData(int id, double x, double y, double z)
		{
			if (m_2DMode) CAjrApi.AjcG3dPutPlotData(m_hCtrl, (int)id, x, y, 0.0);
			else		  CAjrApi.AjcG3dPutPlotData(m_hCtrl, (int)id, x, y, z);
		}
		public void PutData(int id, AJC3DVEC v)
		{
			if (m_2DMode) CAjrApi.AjcG3dPutPlotData(m_hCtrl, (int)id, v.x, v.y, 0.0);
			else		  CAjrApi.AjcG3dPutPlotData(m_hCtrl, (int)id, v.x, v.y, v.z);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		//	2D	//
		public void PutData(int id, double x, double y)
		{
			CAjrApi.AjcG2dPutPlotData(m_hCtrl, (int)id, x, y);
		}
		public void PutData(int id, AJC2DVEC v)
		{
			CAjrApi.AjcG2dPutPlotData(m_hCtrl, (int)id, v.x, v.y);
		}
		//----- レンジ設定 -------------------------------------------------------------------------------------//
		//	3D	//
		public void SetRange(double x1, double y1, double z1, double x2, double y2, double z2)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dSetRange(m_hCtrl, x1, y1, z1, x2, y2, z2);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		public void SetRange(AJC3DVEC v1, AJC3DVEC v2)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dSetRange(m_hCtrl, v1.x, v1.y, v1.z, v2.x, v2.y, v2.z);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		//	2D	//
		public void SetRange(double x1, double y1, double x2, double y2)
		{
			if (m_2DMode) {
				CAjrApi.AjcG2dSetRange(m_hCtrl, x1, y1, x2, y2);
			}
			else throw new InvalidOperationException("2D mode only");
		}
		public void SetRange(AJC2DVEC v1, AJC2DVEC v2)
		{
			if (m_2DMode) {
				CAjrApi.AjcG2dSetRange(m_hCtrl, v1.x, v1.y, v2.x, v2.y);
			}
			else throw new InvalidOperationException("2D mode only");
		}
		//----- レンジ自動調整 ---------------------------------------------------------------------------------//
		public void AdjustRange()
		{
			if (m_2DMode) CAjrApi.AjcG2dAdjustRange(m_hCtrl);
			else		  CAjrApi.AjcG3dAdjustRange(m_hCtrl);
		}
		//----- 中心位置設定 -----------------------------------------------------------------------------------//
		//	3D	//
		public void SetCenter(double xc, double yc, double zc)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dSetCenter(m_hCtrl, xc, yc, zc);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		public void SetCenter(AJC3DVEC vc)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dSetCenter(m_hCtrl, vc.x, vc.y, vc.z);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		//	2D	//
		public void SetCenter(double xc, double yc)
		{
			if (m_2DMode) {
				CAjrApi.AjcG2dSetCenter(m_hCtrl, xc, yc);
			}
			else throw new InvalidOperationException("2D mode only");
		}
		public void SetCenter(AJC2DVEC vc)
		{
			if (m_2DMode) {
				CAjrApi.AjcG2dSetCenter(m_hCtrl, vc.x, vc.y);
			}
			else throw new InvalidOperationException("2D mode only");
		}
		//----- 半径設定 ---------------------------------------------------------------------------------------//
		//	3D	//
		public void SetRadius(double xr, double yr, double zr)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dSetWidth(m_hCtrl, xr, yr, zr);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		public void SetRadius(AJC3DVEC r)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dSetWidth(m_hCtrl, r.x, r.y, r.z);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		//	2D	//
		public void SetRadius(double xr, double yr)
		{
			if (m_2DMode) {
				CAjrApi.AjcG2dSetWidth(m_hCtrl, xr, yr);
			}
			else throw new InvalidOperationException("2D mode only");
		}
		public void SetRadius(AJC2DVEC r)
		{
			if (m_2DMode) {
				CAjrApi.AjcG2dSetWidth(m_hCtrl, r.x, r.y);
			}
			else throw new InvalidOperationException("2D mode only");
		}
		//----- 各軸の半径を同一にする -------------------------------------------------------------------------//
		public void SetSameRadius()
		{
			CAjrApi.AjcG3dSetSameRangeWidth(m_hCtrl);
		}
		//----- ピクセル描画 -----------------------------------------------------------------------------------//
		//	3D	//
		public void Pixel(int id, double x, double y, double z, int pixelSize)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dPixel(m_hCtrl, (int)id, x, y, z, (int)pixelSize);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		public void Pixel(int id, AJC3DVEC v, int pixelSize)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dPixel(m_hCtrl, (int)id, v.x, v.y, v.z, (int)pixelSize);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		//	2D	//
		public void Pixel(int id, double x, double y, int pixelSize)
		{
			CAjrApi.AjcG2dPixel(m_hCtrl, (int)id, x, y, (int)pixelSize);
		}
		public void Pixel(int id, AJC2DVEC v, int pixelSize)
		{
			CAjrApi.AjcG2dPixel(m_hCtrl, (int)id, v.x, v.y, (int)pixelSize);
		}
		//----- ３Ｄライン描画(Line) ---------------------------------------------------------------------------//
		//	3D	(Line，座標指定)			//
		public void Line(int id, double x1, double y1, double z1, double x2, double y2, double z2)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dLine(m_hCtrl, (int)id, x1, y1, z1, x2, y2, z2);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		public void Line(int id, double x1, double y1, double z1, double x2, double y2, double z2, bool fArrow)
		{
			if (!m_2DMode) {
				if (fArrow) CAjrApi.AjcG3dArrow(m_hCtrl, (int)id, x1, y1, z1, x2, y2, z2);
				else		CAjrApi.AjcG3dLine (m_hCtrl, (int)id, x1, y1, z1, x2, y2, z2);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		//	3D	(Line，ベクトル指定)		//
		public void Line(int id, AJC3DVEC v1, AJC3DVEC v2)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dLine (m_hCtrl, (int)id, v1.x, v1.y, v1.z, v2.x, v2.y, v2.z);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		public void Line(int id, AJC3DVEC v1, AJC3DVEC v2, bool fArrow)
		{
			if (!m_2DMode) {
				if (fArrow) CAjrApi.AjcG3dArrow(m_hCtrl, (int)id, v1.x, v1.y, v1.z, v2.x, v2.y, v2.z);
				else		CAjrApi.AjcG3dLine (m_hCtrl, (int)id, v1.x, v1.y, v1.z, v2.x, v2.y, v2.z);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//----- ３Ｄライン描画(MoveTo) -------------------------------------------------------------------------//
		//	3D	(MoveTo，座標指定)			//
		public void MoveTo(int id, double x, double y, double z)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dMoveTo(m_hCtrl, (int)id, x, y, z);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		//	3D	(MoveTo，ベクトル指定)		//
		public void MoveTo(int id, AJC3DVEC v)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dMoveTo(m_hCtrl, (int)id, v.x, v.y, v.z);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//----- ３Ｄライン描画(LineTo) -------------------------------------------------------------------------//
		//	3D	(LineTo，座標指定)			//
		public void LineTo(int id, double x, double y, double z)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dLineTo(m_hCtrl, (int)id, x, y, z);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		public void LineTo(int id, double x, double y, double z, bool fArrow)
		{
			if (!m_2DMode) {
				if (fArrow) CAjrApi.AjcG3dArrowTo(m_hCtrl, (int)id, x, y, z);
				else		CAjrApi.AjcG3dLineTo (m_hCtrl, (int)id, x, y, z);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		//	3D	(LineTo，ベクトル指定)		//
		public void LineTo(int id, AJC3DVEC v)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dLineTo(m_hCtrl, (int)id, v.x, v.y, v.z);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		public void LineTo(int id, AJC3DVEC v, bool fArrow)
		{
			if (!m_2DMode) {
				if (fArrow) CAjrApi.AjcG3dArrowTo(m_hCtrl, (int)id, v.x, v.y, v.z);
				else		CAjrApi.AjcG3dLineTo (m_hCtrl, (int)id, v.x, v.y, v.z);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//----- ２Ｄライン描画(Line) ---------------------------------------------------------------------------//
		//	2D	(Line，座標指定)			//
		public void Line(int id, double x1, double y1, double x2, double y2)
		{
			CAjrApi.AjcG2dLine(m_hCtrl, (int)id, x1, y1, x2, y2);
		}
		public void Line(int id, double x1, double y1, double x2, double y2, bool fArrow)
		{
			if (fArrow) CAjrApi.AjcG2dArrow(m_hCtrl, (int)id, x1, y1, x2, y2);
			else		CAjrApi.AjcG2dLine (m_hCtrl, (int)id, x1, y1, x2, y2);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		//	2D	(Line，ベクトル指定)		//
		public void Line(int id, AJC2DVEC v1, AJC2DVEC v2)
		{
			CAjrApi.AjcG2dLine(m_hCtrl, (int)id, v1.x, v1.y, v2.x, v2.y);
		}
		public void Line(int id, AJC2DVEC v1, AJC2DVEC v2, bool fArrow)
		{
			if (fArrow) CAjrApi.AjcG2dArrow(m_hCtrl, (int)id, v1.x, v1.y, v2.x, v2.y);
			else		CAjrApi.AjcG2dLine (m_hCtrl, (int)id, v1.x, v1.y, v2.x, v2.y);
		}
		//----- ２Ｄライン描画(MoveTo) -------------------------------------------------------------------------//
		//	2D	(MoveTo，座標指定)			//
		public void MoveTo(int id, double x, double y)
		{
			CAjrApi.AjcG2dMoveTo(m_hCtrl, (int)id, x, y);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		//	2D	(MoveTo，ベクトル指定)		//
		public void MoveTo(int id, AJC2DVEC v)
		{
			CAjrApi.AjcG2dMoveTo(m_hCtrl, (int)id, v.x, v.y);
		}
		//----- ２Ｄライン描画(LineTo) -------------------------------------------------------------------------//
		//	2D	(LineTo，座標指定)			//
		public void LineTo(int id, double x, double y)
		{
			CAjrApi.AjcG2dLineTo(m_hCtrl, (int)id, x, y);
		}
		public void LineTo(int id, double x, double y, bool fArrow)
		{
			if (fArrow) CAjrApi.AjcG2dArrowTo(m_hCtrl, (int)id, x, y);
			else		CAjrApi.AjcG2dLineTo (m_hCtrl, (int)id, x, y);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		//	2D	(LineTo，ベクトル指定)		//
		public void LineTo(int id, AJC2DVEC v)
		{
			CAjrApi.AjcG2dLineTo(m_hCtrl, (int)id, v.x, v.y);
		}
		public void LineTo(int id, AJC2DVEC v, bool fArrow)
		{
			if (fArrow)	CAjrApi.AjcG2dArrowTo(m_hCtrl, (int)id, v.x, v.y);
			else		CAjrApi.AjcG2dLineTo (m_hCtrl, (int)id, v.x, v.y);
		}
		//----- 三角形描画 -------------------------------------------------------------------------------------//
		//	3D	//
		public void Triangle(int id, double x1, double y1, double z1, double x2, double y2, double z2, double x3, double y3, double z3)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dTriangle(m_hCtrl, (int)id, x1, y1, z1, x2, y2, z2, x3, y3, z3);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		public void Triangle(int id, AJC3DVEC v1, AJC3DVEC v2, AJC3DVEC v3)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dTriangle(m_hCtrl, (int)id, v1.x, v1.y, v1.z, v2.x, v2.y, v2.z, v3.x, v3.y, v3.z);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		//	2D	//
		public void Triangle(int id, double x1, double y1, double x2, double y2, double x3, double y3)
		{
			CAjrApi.AjcG2dTriangle(m_hCtrl, (int)id, x1, y1, x2, y2, x3, y3);
		}
		public void Triangle(int id, AJC2DVEC v1, AJC2DVEC v2, AJC2DVEC v3)
		{
			CAjrApi.AjcG2dTriangle(m_hCtrl, (int)id, v1.x, v1.y, v2.x, v2.y, v3.x, v3.y);
		}
		//----- 四角形描画 -------------------------------------------------------------------------------------//
		//	3D	//
		public void Square(int id, double x1, double y1, double z1, double x2, double y2, double z2, double x3, double y3, double z3, double x4, double y4, double z4)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dSquare(m_hCtrl, (int)id, x1, y1, z1, x2, y2, z2, x3, y3, z3, x4, y4, z4);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		public void Square(int id, AJC3DVEC v1, AJC3DVEC v2, AJC3DVEC v3, AJC3DVEC v4)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dSquare(m_hCtrl, (int)id,	v1.x, v1.y, v1.z, v2.x, v2.y, v2.z, v3.x, v3.y, v3.z, v4.x, v4.y, v4.z);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		//	2D	//
		public void Square(int id, double x1, double y1, double x2, double y2, double x3, double y3, double x4, double y4)
		{
			CAjrApi.AjcG2dSquare(m_hCtrl, (int)id, x1, y1, x2, y2, x3, y3, x4, y4);
		}
		public void Square(int id, AJC2DVEC v1, AJC2DVEC v2, AJC2DVEC v3, AJC2DVEC v4)
		{
			CAjrApi.AjcG2dSquare(m_hCtrl, (int)id,	v1.x, v1.y, v2.x, v2.y, v3.x, v3.y, v4.x, v4.y);
		}
		//----- 立方体／長方体描画 -----------------------------------------------------------------------------//
		//	3D	//
		public void Cube(int id, double xc, double yc, double zc, double xr, double yr, double zr, int division)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dCube(m_hCtrl, (int)id, xc, yc, zc, xr, yr, zr, (int)division);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		public void Cube(int id, AJC3DVEC vc, AJC3DVEC vr, int division)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dCube(m_hCtrl, (int)id, vc.x, vc.y, vc.z, vr.x, vr.y, vr.z, (int)division);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//----- 球／楕円球描画 ---------------------------------------------------------------------------------//
		//	3D	//
		public void Sphere(int id, double xc, double yc, double zc, double xr, double yr, double zr, int slice, int stack)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dSphere(m_hCtrl, (int)id, xc, yc, zc, xr, yr, zr, (int)slice, (int)stack);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		public void Sphere(int id,	AJC3DVEC vc, AJC3DVEC vr, int slice, int stack)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dSphere(m_hCtrl, (int)id, vc.x, vc.y, vc.z, vr.x, vr.y, vr.z, (int)slice, (int)stack);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//----- 任意の平面を設定 -------------------------------------------------------------------------------//
		public void SetAnyPlane(EAJCPLANEAXIS HoriAxis, EAJCPLANEAXIS VertAxis)
		{
			if (!m_2DMode || (HoriAxis != EAJCPLANEAXIS.ZP && HoriAxis != EAJCPLANEAXIS.ZM &&
							  VertAxis != EAJCPLANEAXIS.ZP && VertAxis != EAJCPLANEAXIS.ZM)) {
				if (!CAjrApi.AjcG3dSetPlane(m_hCtrl, (int)HoriAxis, (int)VertAxis)) {
					throw new InvalidOperationException("The same axis is assigned horizontally and vertically.");
				}
			}
			else {
				throw new InvalidOperationException("It is not possible to assign the Z axis in 2D mode.");
			}
		}
		//----- ３Ｄ空間上に任意の平面を定義 -------------------------------------------------------------------//
		//	3D	//
		public void DefPlane(int id, double xc, double yc, double zc, double xv, double yv, double zv, double xo, double yo, double zo)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dDefPlane(m_hCtrl, (int)id, xc, yc, zc, xv, yv, zv, xo, yo, zo);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void DefPlane(int id, AJC3DLVEC vl)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dDefPlane(m_hCtrl, (int)id, vl.p.x, vl.p.y, vl.p.z, vl.v.x, vl.v.y, vl.v.z, 0.0, 0.0, 0.0);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void DefPlane(int id, AJC3DLVEC vl, AJC3DVEC vo)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dDefPlane(m_hCtrl, (int)id, vl.p.x, vl.p.y, vl.p.z, vl.v.x, vl.v.y, vl.v.z, vo.x, vo.y, vo.z);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//----- 円／楕円描画 -----------------------------------------------------------------------------------//
		//	2D	//
		public void Ellipse(int id, double xc, double yc, double xr, double yr)
		{
			CAjrApi.AjcG2dEllipse(m_hCtrl, (int)id, xc, yc, xr, yr);
		}
		public void Ellipse(int id, AJC2DVEC vc, AJC2DVEC vr)
		{
			CAjrApi.AjcG2dEllipse(m_hCtrl, (int)id, vc.x, vc.y, vr.x, vr.y);
		}
		//----- 星形描画 ---------------------------------------------------------------------------------------//
		//	2D	//
		public void Star(int id, double xc, double yc, double r)
		{
			CAjrApi.AjcG2dStarEx(m_hCtrl, (int)id, xc  , yc, r  , 0.0, 5, 0.0, false);
		}
		public void Star(int id, AJC2DVEC vc, double r)
		{
			CAjrApi.AjcG2dStarEx(m_hCtrl, (int)id, vc.x, vc.y, r, 0.0, 5, 0.0, false);
		}
		public void Star(int id, double xc, double yc, double r1, double r2, int nVertex, double rot, bool fInsclibedLine)
		{
			CAjrApi.AjcG2dStarEx(m_hCtrl, (int)id, xc  , yc, r1, r2, nVertex, rot, fInsclibedLine);
		}
		public void Star(int id, AJC2DVEC vc, double r1, double r2, int nVertex, double rot, bool fInsclibedLine)
		{
            CAjrApi.AjcG2dStarEx(m_hCtrl, (int)id, vc.x, vc.y, r1, r2, nVertex, rot, fInsclibedLine);
		}
		//----- フィルタの設定 ---------------------------------------------------------------------------------//
		public void SetFilter(int ix, bool fEnable)
		{
			CAjrApi.AjcG3dSetFilter(m_hCtrl, (int)ix, fEnable);
		}
		//----- フィルタの取得 ---------------------------------------------------------------------------------//
		public bool GetFilter(int ix)
		{
			return CAjrApi.AjcG3dGetFilter(m_hCtrl, (int)ix);
		}
		//----- 最大プロット数設定 -----------------------------------------------------------------------------//
		public void SetMaxPlot(int ix, int n)
		{
			CAjrApi.AjcG3dSetProp_MaxPlot(m_hCtrl, (int)ix, (int)n);
		}
		//----- 最大プロット数取得 -----------------------------------------------------------------------------//
		public int GetMaxPlot(int ix)
		{
			return (int)CAjrApi.AjcG3dGetProp_MaxPlot(m_hCtrl, (int)ix);
		}
		//----- データ項目の前面表示色設定 ---------------------------------------------------------------------//
		public void SetItemColorP(int ix, Color color)
		{
			int r = color.R;
			int g = color.G;
			int b = color.B;

			CAjrApi.AjcG3dSetProp_rgbP(m_hCtrl, (int)ix, (int)(r | (g << 8) | (b << 16)));
		}
		//----- データ項目の前面表示色取得 ---------------------------------------------------------------------//
		public Color GetItemColorP(int ix)
		{
			int rgb = CAjrApi.AjcG3dGetProp_rgbP(m_hCtrl, (int)ix);
			Color c = Color.FromArgb((int)(rgb & 0xFF), (int)((rgb >> 8) & 0xFF), (int)((rgb >> 16) & 0xFF));

			return c;
		}
		//----- データ項目の後面表示色設定 ---------------------------------------------------------------------//
		public void SetItemColorN(int ix, Color color)
		{
			int r = color.R;
			int g = color.G;
			int b = color.B;

			CAjrApi.AjcG3dSetProp_rgbN(m_hCtrl, (int)ix, (int)(r | (g << 8) | (b << 16)));
		}
		//----- データ項目の後面表示色取得 ---------------------------------------------------------------------//
		public Color GetItemColorN(int ix)
		{
			int rgb = CAjrApi.AjcG3dGetProp_rgbN(m_hCtrl, (int)ix);
			Color c = Color.FromArgb((int)(rgb & 0xFF), (int)((rgb >> 8) & 0xFF), (int)((rgb >> 16) & 0xFF));

			return c;
		}
		//----- 視点設定 ---------------------------------------------------------------------------------------//
		public void SetAngle(double rtx, double rty, double rtz)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dSetAngle(m_hCtrl, rtx, rty, rtz);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void SetAngle(AJC3DVEC perspective)
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dSetAngleV(m_hCtrl, ref perspective);
			}
			else throw new InvalidOperationException("3D mode only");
		}

		//----- 視点をＸＹ平面に設定 ---------------------------------------------------------------------------//
		public void SetAngleXY()
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dSetAngleXY(m_hCtrl);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//----- 視点をＸＺ平面に設定 ---------------------------------------------------------------------------//
		public void SetAngleXZ()
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dSetAngleXZ(m_hCtrl);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//----- 視点をＹＺ平面に設定 ---------------------------------------------------------------------------//
		public void SetAngleYZ()
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dSetAngleYZ(m_hCtrl);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//----- 視点を３Ｄイメージに設定 -----------------------------------------------------------------------//
		public void SetAngle3D()
		{
			if (!m_2DMode) {
				CAjrApi.AjcG3dSetAngle3D(m_hCtrl);
			}
			else throw new InvalidOperationException("3D mode only");
		}
		//----- ドロップされたファイル名取得 -------------------------------------------------------------------//
		public string GetDroppedFile()
		{
			return CAjrApi.AjcG3dGetDroppedFile(m_hCtrl);
		}
		//----- ドロップされたディレクトリ名取得 ---------------------------------------------------------------//
		public string GetDroppedDir()
		{
			return CAjrApi.AjcG3dGetDroppedDir(m_hCtrl);
		}
		//----- タイトルテキストの表示 -------------------------------------------------------------------------//
		public void SetTitleText(string TitleText)
		{
			m_TitleText = TitleText;
			CAjrApi.AjcG3dSetTitleText(m_hCtrl, TitleText, Color.Empty, Color.Empty, (Font)null);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void SetTitleText(string TitleText, Color TextColor)
		{
			m_TitleText = TitleText;
			CAjrApi.AjcG3dSetTitleText(m_hCtrl, TitleText, TextColor, Color.Empty, null);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void SetTitleText(string TitleText, Color TextColor, Color BackColor)
		{
			m_TitleText = TitleText;
			CAjrApi.AjcG3dSetTitleText(m_hCtrl, TitleText, TextColor, BackColor, null);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void SetTitleText(string TitleText, Color TextColor, Color BackColor, Font TextFont)
		{
			m_TitleText = TitleText;
			CAjrApi.AjcG3dSetTitleText(m_hCtrl, TitleText, TextColor, BackColor, TextFont);
		}
		//----- プロファイルへ、フィルタ設定とプロパティ書き込み -----------------------------------------------//
		public void SaveToProfile(string section)
		{
			CAjrApi.AjcG3dSavePropEx(m_hCtrl, section);
		}
		//----- プロファイルから、フィルタ設定とプロパティ読み出し ---------------------------------------------//
		public void LoadFromProfile(string section)
		{
			CAjrApi.AjcG3dLoadPropEx(m_hCtrl, section, (IntPtr)0);
		}
		//----- ボーダー色で囲まれた部分の塗りつぶし -----------------------------------------------------------//
		public void FillB(int idFill, int idBorder, double x, double y)
		{
			CAjrApi.AjcG2dFillB(m_hCtrl, (int)idFill, (int)idBorder, x, y);
		}
		//----- 連続する白色部分の塗りつぶし -------------------------------------------------------------------//
		public void FillS(int idFill, double x, double y)
		{
			CAjrApi.AjcG2dFillS(m_hCtrl, (int)idFill, x, y);
		}
		//----- ピクセルの色取得 -------------------------------------------------------------------------------//
		public Color GetPixel(double x, double y)
		{
			int rgb = CAjrApi.AjcG2dGetPixel(m_hCtrl, x, y);
			Color c = Color.FromArgb((int)(rgb & 0xFF), (int)((rgb >> 8) & 0xFF), (int)((rgb >> 16) & 0xFF));

			return c;
		}
		//----- 表示停止／再開 ---------------------------------------------------------------------------------//
		public void Pause(bool fPause)
		{
			CAjrApi.AjcG3dPause(m_hCtrl, fPause);
		}
		//----- テキスト描画 -----------------------------------------------------------------------------------//
		public int TextOut(int x, int y, string text)
		{
            return CAjrApi.AjcG3dTextOutW(m_hCtrl, x, y, text);
		}
		//- - - 2D - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -//
		public int TextOut(AJC2DVEC v, string text)
		{
			return CAjrApi.AjcG3dTextOut2DForMSIL(m_hCtrl, EAJCTXOMD.RIGHT, v.x, v.y, text);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public int TextOut(EAJCTXOMD md, AJC2DVEC v, string text)
		{
			return CAjrApi.AjcG3dTextOut2DForMSIL(m_hCtrl, md, v.x, v.y, text);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public int TextOut(double x, double y, string text)
		{
			return CAjrApi.AjcG3dTextOut2DForMSIL(m_hCtrl, EAJCTXOMD.RIGHT, x, y, text);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public int TextOut(EAJCTXOMD md, double x, double y, string text)
		{
			return CAjrApi.AjcG3dTextOut2DForMSIL(m_hCtrl, md, x, y, text);
		}
		//- - - 3D - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -//
		public int TextOut(AJC3DVEC v, string text)
		{
			return CAjrApi.AjcG3dTextOut3DForMSIL(m_hCtrl, EAJCTXOMD.RIGHT, v.x, v.y, v.z, text);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public int TextOut(EAJCTXOMD md, AJC3DVEC v, string text)
		{
			return CAjrApi.AjcG3dTextOut3DForMSIL(m_hCtrl, md, v.x, v.y, v.z, text);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public int TextOut(double x, double y, double z, string text)
		{
			return CAjrApi.AjcG3dTextOut3DForMSIL(m_hCtrl, EAJCTXOMD.RIGHT, x, y, z, text);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public int TextOut(EAJCTXOMD md, double x, double y, double z, string text)
		{
			return CAjrApi.AjcG3dTextOut3DForMSIL(m_hCtrl, md, x, y, z, text);
		}
		//----- プロットデータ破棄 -----------------------------------------------------------------------------//
		public void PurgePlot(int id)
		{
			CAjrApi.AjcG3dClearPlot(m_hCtrl, id);
		}
		//- - - 全てのプロットデータ破棄 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -//
		public void PurgePlot()
		{
			for (int id = 0; id < 16; id++) {
				CAjrApi.AjcG3dClearPlot(m_hCtrl, id);
			}
		}
		//----- 図形描画データ破棄 -----------------------------------------------------------------------------//
		public void PurgeShape(int id)
		{
			CAjrApi.AjcG3dClearShape(m_hCtrl, (int)id);
		}
		//- - - 全ての図形描画データ破棄 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -//
		public void PurgeShape()
		{
			for (int id = 0; id < 16; id++) {
                CAjrApi.AjcG3dClearShape(m_hCtrl, id);
			}
		}
		//----- テキスト破棄 -----------------------------------------------------------------------------------//
		public bool PurgeText(int key)
		{
			return CAjrApi.AjcG3dClearText(m_hCtrl, key);
		}
		//- - - 全てのテキスト破棄 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -//
		public bool PurgeText()
		{
			return CAjrApi.AjcG3dClearAllText(m_hCtrl);
		}
		//----- 描画データ（図形データ，プロットデータ）破棄 ---------------------------------------------------//
		public void PurgeData(int id)
		{
			CAjrApi.AjcG3dClearPlot (m_hCtrl, id);
			CAjrApi.AjcG3dClearShape(m_hCtrl, id);
		}
		//- - - 全ての描画データ（図形データ，プロットデータ）破棄 - - - - - - - - - - - - - - - - - - - - - - -//
		public void PurgeData()
		{
			for (int id = 0; id < 16; id++) {
				CAjrApi.AjcG3dClearPlot (m_hCtrl, id);
				CAjrApi.AjcG3dClearShape(m_hCtrl, id);
			}
		}
		//----- 全てのデータ破棄 -------------------------------------------------------------------------------//
		public void Purge()
		{
			PurgePlot();
			PurgeShape();
			PurgeText();
		}

		//------------------------------------------------------------------------------------------------------//
		//	デザイン時のダミーデータ生成																		//
		//------------------------------------------------------------------------------------------------------//
		private void GenDummyData()
		{
			AJC3DVEC	v;
			double		t;

			this.Purge();
			if (m_ShowDummyData) {
				for (t = 10.0; t < 360.0; t += 30.0) {
					double r;
					r = Math.Max(this.RadiusX, this.RadiusY);
					r = Math.Max(this.RadiusZ, r);
					if (m_2DMode) r *= 0.8;
					v.x = r * Math.Cos(t * 3.14159 / 180.0) + this.CenterX;
					v.y = r * Math.Sin(t * 3.14159 / 180.0) + this.CenterY;
					v.z = 0;
					this.PutData(0, v);
				}
			}
		}
		//------------------------------------------------------------------------------------------------------//
		//	コールバック																						//
		//------------------------------------------------------------------------------------------------------//
		//----- コールバック関数（ファイルドロップ通知）--------------------------------------------------------//
		private void cbG3dFileDrop(int n)
		{
			G3dArgFileDrop eva = new G3dArgFileDrop();
			eva.n = n;
			if (OnFileDrop != null) {
				OnFileDrop(this, eva);
			}
		}
		//----- コールバック関数（フォルダドロップ通知）--------------------------------------------------------//
		private void cbG3dDirDrop(int n)
		{
			G3dArgDirDrop eva = new G3dArgDirDrop();
			eva.n = n;
			if (OnDirDrop != null) {
				OnDirDrop(this, eva);
			}
		}
		//----- コールバック関数（右クリック通知）--------------------------------------------------------------//
		private void cbG3dRClick(int x, int y, bool shift, bool ctrl)
		{
			G3dArgRClick eva = new G3dArgRClick();
			eva.x	  = x;
			eva.y	  = y;
			eva.shift = shift;
			eva.ctrl  = ctrl;
			if (OnRClick != null) {
				OnRClick(this, eva);
			}
		}
		//----- コールバック関数（プロットリスト通知）----------------------------------------------------------//
		private void cbG3dPltLst(int max, int num, IntPtr pItem)
		{
			G3dArgPltLst eva = new G3dArgPltLst();
			eva.max	  = max;
			eva.num   = num;
			eva.p	  = new PlotInfo[num];
			if (OnPltLst != null) {
				double x, y, z;
				for (int i = 0; i < num; i++) {
					eva.p[i].ix = CAjrApi.AjcG3dGetPlotEvtInfo(pItem, i, out x, out y, out z);
					eva.p[i].v.x = x;
					eva.p[i].v.y = y;
					eva.p[i].v.z = z;
				}
				OnPltLst(this, eva);
			}
		}
	}
	//==========================================================================================================//
	//	FontTxoプロパティ・エディタ	（ダイアログから、取り消し線や下線を隠す）									//
	//==========================================================================================================//
	internal class FontTxoEditor : UITypeEditor
	{
		//----- コンストラクタ ---------------------------------------------------------------------------------//
		public FontTxoEditor()
		{
		}
		//----- プロパティエディタとして、モーダル表示フォームを使用する事を通知 -------------------------------//
		public override UITypeEditorEditStyle GetEditStyle(ITypeDescriptorContext context)
		{
			return UITypeEditorEditStyle.Modal;
		}
		//----- 独自フォント設定 -------------------------------------------------------------------------------//
		public override object EditValue(ITypeDescriptorContext context, IServiceProvider provider, object value)
		{
			IWindowsFormsEditorService editorService = null;

			//----- MSDN UI 型エディターの実装 に基づいた記述
			if (provider != null) {
				editorService = provider.GetService(typeof(IWindowsFormsEditorService)) as IWindowsFormsEditorService;
			}

			//----- MSDN UI 型エディターの実装 に基づいた記述
			if (editorService == null) {
				return value;
			}
			
			//----- フォント表現文字列取得 ---------------------------------------------------------------------//
			Font f = (Font)value;
			FontDialog fd = new FontDialog();

			//	初期のフォント設定
			fd.Font = (Font)value;
			//	ポイントサイズの範囲
			fd.MaxSize = 144;
			fd.MinSize = 6;
			//	存在しないフォントやスタイルをユーザーが選択時、エラーメッセージを表示する
			fd.FontMustExist = true;
			//	横書きフォントだけを表示する
			fd.AllowVerticalFonts = false;
			//	色の選択
			fd.ShowColor = false;
			//	取り消し線、下線のオプション
			fd.ShowEffects = false;
			//	固定ピッチフォント以外も表示する
			fd.FixedPitchOnly = false;
			//	ベクタ フォントを選択できるようにする(デフォルト：True)
			fd.AllowVectorFonts = true;
			//	ダイアログを表示し、フォントを選択する
			if (fd.ShowDialog() != DialogResult.Cancel) {
				f = fd.Font;
			}
			return f;
		}
	}
}
