﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Drawing.Design;
using System.Windows.Forms.Design;
using AjaraInit;

namespace CAjrCustCtrl
{
	//----------------------------------------------------------------------------------------------------------//
	[ToolboxBitmap(typeof(CAjrBarGraph), "CAjrBarGraph.bmp")]
	public partial class CAjrBarGraph : UserControl
	{
		//----- イベント ----------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Occurs when the range changes.")]
		public event BarEvtRangeChanged OnRangeChanged;

		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Occurs when scrolling.")]
		public event BarEvtNtcScrollPos OnNtcScrollPos;

		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Occurs when dropping files.")]
		public event BarEvtFileDrop 	OnFileDrop;

		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Occurs when dropping directories.")]
		public event BarEvtDirDrop		OnDirDrop;

		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Occurs when Shift/Ctrl + right-clicked.")]
		public event BarEvtRClick		OnRClick;

		private BarCbkRangeChanged	m_BarCbkRangeChanged;
		private BarCbkNtcScrollPos	m_BarCbkNtcScrollPos;
		private BarCbkFileDrop		m_BarCbkFileDrop;
		private BarCbkDirDrop		m_BarCbkDirDrop;
		private BarCbkRClick		m_BarCbkRClick;

		//----- プロパティ値 ------------------------------//
		private	bool		m_EnablePopupMenu	= true;
		private string		m_DataTitle			= "Data-\nTitle";
		private string		m_UnitName			= "[UnitName]";
		private bool		m_ShowDummyData		= true;

		private string		m_TitleText			= "";
		private bool		m_ToolTipShowAlways = true;
		private string		m_TipText			= "";
		private string[]	m_TipFilter 		= new string[8];

        private Font        m_TxoFont = new Font("MS UI Gothic", 9);

		//----- ワーク ------------------------------------//
		private IntPtr		m_hCtrl;
		private bool		m_ShowScrollBar;
		private	bool		m_FirstResize = true;

		//----- コンストラクタ ---------------------------------------------------------------------------------//
		public CAjrBarGraph()
		{
			InitializeComponent();

			//----- ライブラリ初期化 -------------------------------------------//
			CAjrInit.Init();

			//----- 変数初期化 -------------------------------------------------//
			m_ShowScrollBar = true;

			//----- コントロールウインド生成 -----------------------------------//
			m_BarCbkRangeChanged = new BarCbkRangeChanged(cbBarRangeChanged);
			m_BarCbkNtcScrollPos = new BarCbkNtcScrollPos(cbBarNtcScrollPos);
			m_BarCbkFileDrop	 = new BarCbkFileDrop	 (cbBarFileDrop	);
			m_BarCbkDirDrop	 	 = new BarCbkDirDrop	 (cbBarDirDrop	);
			m_BarCbkRClick		 = new BarCbkRClick(cbBarRClick);

			m_hCtrl = CAjrApi.AjcBarCreateForMSIL(this.Handle, m_ShowScrollBar, m_BarCbkRangeChanged,
																				m_BarCbkNtcScrollPos,
																				m_BarCbkFileDrop	,
																				m_BarCbkDirDrop	 	,
																				m_BarCbkRClick		);

			//----- プロパティ初期値 -------------------------------------------//
			CAjrApi.AjcBarSetBarTtl(m_hCtrl, m_DataTitle);
			CAjrApi.AjcBarSetVUnit (m_hCtrl, m_UnitName);
		}

		//------------------------------------------------------------------------------------------------------//
		//	コントロール内部イベント																			//
		//------------------------------------------------------------------------------------------------------//
		//----- 初期設定 ---------------------------------------------------------------------------------------//
		private void CAjrBarGraph_Load(object sender, EventArgs e)
		{
		}
		//----- コントロール・サイズ変更 -----------------------------------------------------------------------//
		private void CAjrBarGraph_Resize(object sender, EventArgs e)
		{
			//	初回のみの処理
			if (m_FirstResize) {
				m_FirstResize = false;
				//	デザインモード時ダミーデータ表示
				if (this.DesignMode) GenDummyData();
			}
			//	コントロールのサイズ設定
			CAjrApi.AjcBarSetProp_WndSize(m_hCtrl, (int)this.Size.Width, (int)this.Size.Height);
		}
		//------------------------------------------------------------------------------------------------------//
		//	プロパティ																							//
		//------------------------------------------------------------------------------------------------------//
		//----- Fontプロパティを隠す ---------------------------------------------------------------------------//
		private Font vfontInit = new Font("MS UI Gothic", 9);
		[Browsable(false)]
		public override Font Font
		{
			set { base.Font = vfontInit; }
			get { return vfontInit; }
		}
        //----- FontTxoプロパティ ------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Font when TextOut()")]
		[Editor(typeof(FontTxoEditor), typeof(System.Drawing.Design.UITypeEditor))]
		public Font FontTxo
		{
            set {m_TxoFont = value;
                 CAjrApi.AjcBarSetTextFont(m_hCtrl, value.ToHfont());
            }
			get {return m_TxoFont; }
		}
		//----- AllowDropプロパティを隠す ----------------------------------------------------------------------//
		private bool AllowDropInit = false;
		[Browsable(false)]
		public override bool AllowDrop
		{
			set { base.AllowDrop = AllowDropInit; }
			get { return AllowDropInit; }
		}
		//----- Enabledプロパティを変更 ------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Enable / Disable the control")]
		public new bool Enabled
		{
			set {	if (base.Enabled == value) base.Enabled = !value;
					base.Enabled = value;
					CAjrApi.AjcEnableCtrlsInWnd(this.Handle, value);
				}
			get {	return base.Enabled;  }
		}
		//----- デザイン時のダミーデータ表示／非表示 -----------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Dummy data is shown at designing")]
		public bool ShowDummyData {
			set { m_ShowDummyData = value; if (this.DesignMode) GenDummyData(); }
			get { return m_ShowDummyData; }
		}
		//----- 外枠表示スタイル -------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Outer frame display")]
		public bool ShowBorder
		{
			set {CAjrApi.AjcBarSetSty_NOBORDER(m_hCtrl, !value);}
			get {return !CAjrApi.AjcBarGetSty_NOBORDER(m_hCtrl);}
		}

		//----- 折れ線グラフ・スタイル -------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Chart mode (false:Bar chart，true:Line chart)")]
		public bool LineChart
		{
			set { CAjrApi.AjcBarSetSty_LINEGRAPH(m_hCtrl, value); }
			get { return CAjrApi.AjcBarGetSty_LINEGRAPH(m_hCtrl); }
		}

		//----- フィルタ表示スタイル ---------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Filter check-box display")]
		public bool ShowFilter
		{
			set { CAjrApi.AjcBarSetSty_NOFILTER(m_hCtrl, !value); }
			get { return !CAjrApi.AjcBarGetSty_NOFILTER(m_hCtrl); }
		}

		//----- データタイトル ---------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Data title text (Text to display in lower left part, Input line feed by 'Shift+Enter')")]
		[Editor("System.ComponentModel.Design.MultilineStringEditor, System.Design, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b03f5f7f11d50a3a", 
			typeof(System.Drawing.Design.UITypeEditor))]
		public string DataTitle
		{
			set {m_DataTitle = value; CAjrApi.AjcBarSetBarTtl(m_hCtrl, value);}
			get {return m_DataTitle;}
		}

		//----- データ単位名 -----------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Data unit name (Text to display in upper left part")]
		public string UnitName
		{
			set {m_UnitName = value; CAjrApi.AjcBarSetVUnit(m_hCtrl, value);}
			get {return m_UnitName;}
		}

		//----- 低位レンジ値プロパティ -------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Low value in range")]
		public double RangeLow
		{
			set {CAjrApi.AjcBarSetProp_RngL(m_hCtrl, value);		if (this.DesignMode) GenDummyData();}
			get {return CAjrApi.AjcBarGetProp_RngL(m_hCtrl);}
		}

		//----- 高位レンジ値プロパティ -------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("High value in range")]
		public double RangeHigh
		{
			set { CAjrApi.AjcBarSetProp_RngH(m_hCtrl, value);		if (this.DesignMode) GenDummyData(); }
			get {return CAjrApi.AjcBarGetProp_RngH(m_hCtrl);}
		}

		//----- 棒グラフベース値 ------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Base value of bar chart")]
		public double Base
		{
			set { CAjrApi.AjcBarSetProp_Base(m_hCtrl, value);}
			get {return CAjrApi.AjcBarGetProp_Base(m_hCtrl);}
		}

		//----- 最大バッファサイズ・プロパティ -----------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Maximum number of buffering data")]
		public int MaxBuf
		{
			set { CAjrApi.AjcBarSetProp_MaxBuf(m_hCtrl, (int)value); if (this.DesignMode) GenDummyData(); }
			get {return  (int)CAjrApi.AjcBarGetProp_MaxBuf(m_hCtrl);}
		}

		//----- データ項目数プロパティ -------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Number of data items")]
		public int DataItems
		{
			set { CAjrApi.AjcBarSetProp_MaxItem(m_hCtrl, (int)value); if (this.DesignMode) GenDummyData(); }
			get {return  (int)CAjrApi.AjcBarGetProp_MaxItem(m_hCtrl);}
		}

		//----- スケール値表示域の幅プロパティ -----------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Width of the scale value indication area (pixels)")]
		public int ScaleWidth
		{
			set {CAjrApi.AjcBarSetProp_ScaleWidth(m_hCtrl, (int)value);}
			get {return  (int)CAjrApi.AjcBarGetProp_ScaleWidth(m_hCtrl);}
		}

		//----- 棒の幅プロパティ -------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Width of a bar (pixels)")]
		public int BarWidth
		{
			set {CAjrApi.AjcBarSetProp_BarWidth(m_hCtrl, (int)value);}
			get {return  (int)CAjrApi.AjcBarGetProp_BarWidth(m_hCtrl);}
		}

		//----- 棒域の幅プロパティ -----------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Displaying interval of bars / Line chart pixel (Pixels)")]
		public int BarFieldWidth
		{
			set {CAjrApi.AjcBarSetProp_MinWidth(m_hCtrl, (int)value);}
			get {return  (int)CAjrApi.AjcBarGetProp_MinWidth(m_hCtrl);}
		}

		//----- 棒タイトルの最大行数プロパティ -----------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Number of lines of a data-title text")]
		public int BarTitleLines
		{
			set {CAjrApi.AjcBarSetProp_BarTtlLines(m_hCtrl, (int)value);}
			get {return  (int)CAjrApi.AjcBarGetProp_BarTtlLines(m_hCtrl);}
		}

		//----- 文字の幅 ---------------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Charcter width (Pixels)")]
		public int CharWidth
		{
			get {return  (int)CAjrApi.AjcBarGetProp_CharWidth(m_hCtrl);}
		}

		//----- 文字の高さ -------------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Charcter height (Pixels)")]
		public int CharHeight
		{
			get {return  (int)CAjrApi.AjcBarGetProp_CharHeight(m_hCtrl);}
		}

		//----- ファイル，フォルダ ドロップの許可／禁止 --------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Allow derop the file/folder")]
		public bool AcceptFiles
		{
			set {CAjrApi.AjcBarSetProp_AcceptFiles(m_hCtrl, value);}
			get {return CAjrApi.AjcBarGetProp_AcceptFiles(m_hCtrl);}
		}

		//----- ポップアップメニューの許可／禁止 --------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Enable popup menu by right click")]
		public bool EnablePopupMenu {
			set {m_EnablePopupMenu = value; CAjrApi.AjcBarEnablePopupMenu(m_hCtrl, value);}
			get {return m_EnablePopupMenu;}
		}

		//----- ツールチップ表示条件 ---------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Tooltip display condition. If false is set, the tooltip will not be displayed when inactive.")]
		public bool ToolTipShowAlways
		{
			set {
				m_ToolTipShowAlways = value;
				CAjrApi.AjcBarSetTipShowAlways(m_hCtrl, m_ToolTipShowAlways);
				for (int i = 0; i < m_TipFilter.Length; i++) {
					CAjrApi.AjcBarSetChkBoxTipShowAlways(m_hCtrl, i, m_ToolTipShowAlways);
				}
			}
			get {
				return m_ToolTipShowAlways;
			}
		}
		//----- ツールチップテキスト ---------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Tool tip text of this control")]
		public string ToolTipText
		{
			set	{
					if (this.DesignMode)  m_TipText = value;
					else {
				 		StringBuilder s = new StringBuilder(2048);
				 		CAjrApi.AjcBarGetTipTextForMSIL(m_hCtrl, s, (int)s.Capacity);
						if (value != s.ToString()) {
							CAjrApi.AjcBarSetTipTextForMSIL(m_hCtrl, value);
							CAjrApi.AjcBarSetTipShowAlways (m_hCtrl, m_ToolTipShowAlways);
						}
					}
				}
			get	{
					string rc = "";
					if (this.DesignMode) rc = m_TipText;
					else {
				 		StringBuilder s = new StringBuilder(2048);
				 		CAjrApi.AjcBarGetTipTextForMSIL(m_hCtrl, s, (int)s.Capacity);
				 		rc = s.ToString();
				 	}
				 	return rc;
				}
		}

		//----- フィルタ・チェックボックスＮのツールチップ設定／取得 -------------------------------------------//
		private void SetToolTipFilter(string s, int ix)
		{
			if (this.DesignMode) m_TipFilter[ix] = s;
			else if (s != GetToolTipFilter(ix)) {
				CAjrApi.AjcBarSetChkBoxTipTextForMSIL(m_hCtrl, ix, s);
				CAjrApi.AjcBarSetChkBoxTipShowAlways (m_hCtrl, ix, m_ToolTipShowAlways);
			}
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		private string GetToolTipFilter(int ix)
		{
			string rc = "";
			if (this.DesignMode) rc = m_TipFilter[ix];
			else {
				StringBuilder s = new StringBuilder(2048);
				CAjrApi.AjcBarGetChkBoxTipTextForMSIL(m_hCtrl, ix, s, (int)s.Capacity);
				rc = s.ToString();
			}
			return rc;
		}

		//----- フィルタ・チェックボックス( 0)のツールチップ設定 -----------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 0")]
		public string ToolTipFilter0
		{
			set {SetToolTipFilter(value, 0);}
			get {return GetToolTipFilter(0);}
		}
		//----- フィルタ・チェックボックス( 1)のツールチップ設定 -----------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 1")]
		public string ToolTipFilter1
		{
			set {SetToolTipFilter(value, 1);}
			get {return GetToolTipFilter(1);}
		}
		//----- フィルタ・チェックボックス( 2)のツールチップ設定 -----------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 2")]
		public string ToolTipFilter2
		{
			set {SetToolTipFilter(value, 2);}
			get {return GetToolTipFilter(2);}
		}
		//----- フィルタ・チェックボックス( 3)のツールチップ設定 -----------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 3")]
		public string ToolTipFilter3
		{
			set {SetToolTipFilter(value, 3);}
			get {return GetToolTipFilter(3);}
		}
		//----- フィルタ・チェックボックス( 4)のツールチップ設定 -----------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 4")]
		public string ToolTipFilter4
		{
			set {SetToolTipFilter(value, 4);}
			get {return GetToolTipFilter(4);}
		}
		//----- フィルタ・チェックボックス( 5)のツールチップ設定 -----------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 5")]
		public string ToolTipFilter5
		{
			set {SetToolTipFilter(value, 5);}
			get {return GetToolTipFilter(5);}
		}
		//----- フィルタ・チェックボックス( 6)のツールチップ設定 -----------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 6")]
		public string ToolTipFilter6
		{
			set {SetToolTipFilter(value, 6);}
			get {return GetToolTipFilter(6);}
		}
		//----- フィルタ・チェックボックス( 7)のツールチップ設定 -----------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 7")]
		public string ToolTipFilter7
		{
			set {SetToolTipFilter(value, 7);}
			get {return GetToolTipFilter(7);}
		}
		//----- タイトルテキストプロパティ ---------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrBarGraph")]
		[System.ComponentModel.Description("Title text. It is displayed in the upper right corner of the control.")]
		public string TitleText
		{
			set { m_TitleText = value;
				  SetTitleText(m_TitleText);}
			get { return m_TitleText; }
		}

		//------------------------------------------------------------------------------------------------------//
		//	メソッド																							//
		//------------------------------------------------------------------------------------------------------//
		//----- データ投与 -------------------------------------------------------------------------------------//
		public void PutData(string BarCaption, double d1) {
			CAjrApi.AjcBarPutData1(m_hCtrl, BarCaption, d1);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void PutData(string BarCaption, double d1, double d2)
		{
			CAjrApi.AjcBarPutData2(m_hCtrl, BarCaption, d1, d2);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void PutData(string BarCaption, double d1, double d2, double d3)
		{
			CAjrApi.AjcBarPutData3(m_hCtrl, BarCaption, d1, d2, d3);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void PutData(string BarCaption, double d1, double d2, double d3, double d4)
		{
			CAjrApi.AjcBarPutData4(m_hCtrl, BarCaption, d1, d2, d3, d4);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void PutData(string BarCaption, double d1, double d2, double d3, double d4, double d5)
		{
			CAjrApi.AjcBarPutData5(m_hCtrl, BarCaption, d1, d2, d3, d4, d5);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void PutData(string BarCaption, double d1, double d2, double d3, double d4, double d5, double d6)
		{
			CAjrApi.AjcBarPutData6(m_hCtrl, BarCaption, d1, d2, d3, d4, d5, d6);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void PutData(string BarCaption, double d1, double d2, double d3, double d4, double d5, double d6, double d7)
		{
			CAjrApi.AjcBarPutData7(m_hCtrl, BarCaption, d1, d2, d3, d4, d5, d6, d7);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void PutData(string BarCaption, double d1, double d2, double d3, double d4, double d5, double d6, double d7, double d8)
		{
			CAjrApi.AjcBarPutData8(m_hCtrl, BarCaption, d1, d2, d3, d4, d5, d6, d7, d8);
		}
		//----- データ項目の表示色設定 -------------------------------------------------------------------------//
		public void SetItemColor(int ix, Color color)
		{
			int r = color.R;
			int g = color.G;
			int b = color.B;

			CAjrApi.AjcBarSetProp_rgb(m_hCtrl, (int)ix, (int)(r | (g << 8) | (b << 16)));
		}
		//----- データ項目の表示色取得 -------------------------------------------------------------------------//
		public Color GetItemColor(int ix)
		{
			int rgb = CAjrApi.AjcBarGetProp_rgb(m_hCtrl, (int)ix);
			Color c = Color.FromArgb((int)(rgb & 0xFF), (int)((rgb >> 8) & 0xFF), (int)((rgb >> 16) & 0xFF));

			return c;
		}
		//----- スクロール位置の設定 ---------------------------------------------------------------------------//
		public void SetScrollPos(int pos)
		{
			CAjrApi.AjcBarSetScrollPos(m_hCtrl, pos);
		}
		//----- スクロール位置の取得 ---------------------------------------------------------------------------//
		public int SetGcrollPos(int pos)
		{
			return CAjrApi.AjcBarGetScrollPos(m_hCtrl);
		}
		//----- フィルタの設定 ---------------------------------------------------------------------------------//
		public void SetFilter(int ix, bool fEnable)
		{
			CAjrApi.AjcBarSetFilter(m_hCtrl, (int)ix, fEnable);
		}
		//----- フィルタの取得 ---------------------------------------------------------------------------------//
		public bool GetFilter(int ix)
		{
			return CAjrApi.AjcBarGetFilter(m_hCtrl, (int)ix);
		}
		//----- 横線スタイル設定 -------------------------------------------------------------------------------//
		public void SetHoriLineStyle(int ix, Color color, int width, EBarLineStyle style)
		{
			int r = color.R;
			int g = color.G;
			int b = color.B;

			CAjrApi.AjcBarSetHLineAtt(m_hCtrl, (int)ix, (int)(r | (g << 8) | (b << 16)), width, (int)style);
		}
		//----- 横線描画位置 -----------------------------------------------------------------------------------//
		public void SetHoriLinePos(int ix, double pos)
		{
			CAjrApi.AjcBarSetHLinePos(m_hCtrl, (int)ix, pos);
		}
		//----- 横線描画の許可／禁止 ---------------------------------------------------------------------------//
		public void EnableHoriLine(int ix, bool fEnable)
		{
			CAjrApi.AjcBarEnableHLine(m_hCtrl, (int)ix, fEnable);
		}
		//----- ドロップされたファイル名取得 -------------------------------------------------------------------//
		public string GetDroppedFile()
		{
			return CAjrApi.AjcBarGetDroppedFile(m_hCtrl);
		}
		//----- ドロップされたディレクトリ名取得 ---------------------------------------------------------------//
		public string GetDroppedDir()
		{
			return CAjrApi.AjcBarGetDroppedDir(m_hCtrl);
		}
		//----- タイトルテキストの表示 -------------------------------------------------------------------------//
		public void SetTitleText(string TitleText)
		{
			m_TitleText = TitleText;
			CAjrApi.AjcBarSetTitleText(m_hCtrl, TitleText, Color.Empty, Color.Empty, (Font)null);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void SetTitleText(string TitleText, Color TextColor)
		{
			m_TitleText = TitleText;
			CAjrApi.AjcBarSetTitleText(m_hCtrl, TitleText, TextColor, Color.Empty, null);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void SetTitleText(string TitleText, Color TextColor, Color BackColor)
		{
			m_TitleText = TitleText;
			CAjrApi.AjcBarSetTitleText(m_hCtrl, TitleText, TextColor, BackColor, null);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void SetTitleText(string TitleText, Color TextColor, Color BackColor, Font TextFont)
		{
			m_TitleText = TitleText;
			CAjrApi.AjcBarSetTitleText(m_hCtrl, TitleText, TextColor, BackColor, TextFont);
		}
		//----- プロファイルへ、フィルタ設定とプロパティ書き込み -----------------------------------------------//
		public void SaveToProfile(string section)
		{
			CAjrApi.AjcBarSavePropEx(m_hCtrl, section);
		}
		//----- プロファイルから、フィルタ設定とプロパティ読み出し ---------------------------------------------//
		public void LoadFromProfile(string section)
		{
			CAjrApi.AjcBarLoadPropEx(m_hCtrl, section, (IntPtr)0);
		}
		//----- テキスト描画 -----------------------------------------------------------------------------------//
		public int TextOut(int x, int y, string text)
		{
            return CAjrApi.AjcBarTextOutW(m_hCtrl, x, y, text);
		}
		//----- テキスト削除 -----------------------------------------------------------------------------------//
		public bool PurgeText(int key)
		{
			return CAjrApi.AjcBarClearText(m_hCtrl, key);
		}
		//- - - 全てのテキスト破棄 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -//
		public bool PurgeText()
		{
			return CAjrApi.AjcBarClearAllText(m_hCtrl);
		}
		//----- データ破棄 -------------------------------------------------------------------------------------//
		public void PurgePlot()
		{
			CAjrApi.AjcBarPurge(m_hCtrl);
		}
		//----- 全てのデータ，描画テキスト破棄 -----------------------------------------------------------------//
		public void Purge()
		{
			CAjrApi.AjcBarPurge(m_hCtrl);
			CAjrApi.AjcBarClearAllText(m_hCtrl);
		}

		//------------------------------------------------------------------------------------------------------//
		//	デザイン時のダミーデータ生成																		//
		//------------------------------------------------------------------------------------------------------//
		private void GenDummyData()
		{
			int		i, n;

			if (this.DesignMode) {
				this.Purge();
				if (m_ShowDummyData) {
					Random cRandom = new System.Random();
					n = this.Width / (this.BarWidth * this.DataItems);
					for (i = 0; i < n; i++)
					{
						double	l, h, w, r, d1, d2, d3, d4, d5, d6, d7, d8;
						l = System.Math.Min(this.RangeLow, this.RangeHigh);
						h = System.Math.Max(this.RangeLow, this.RangeHigh);
						w = h - l;
						r = cRandom.NextDouble(); d1 = l + w * r;
						r = cRandom.NextDouble(); d2 = l + w * r;
						r = cRandom.NextDouble(); d3 = l + w * r;
						r = cRandom.NextDouble(); d4 = l + w * r;
						r = cRandom.NextDouble(); d5 = l + w * r;
						r = cRandom.NextDouble(); d6 = l + w * r;
						r = cRandom.NextDouble(); d7 = l + w * r;
						r = cRandom.NextDouble(); d8 = l + w * r;
						this.PutData(i.ToString(), d1, d2, d3, d4, d5, d6, d7, d8);
					}
				}
			}
		}

		//------------------------------------------------------------------------------------------------------//
		//	コールバック																						//
		//------------------------------------------------------------------------------------------------------//
		//----- コールバック関数（レンジ変化通知）--------------------------------------------------------------//
		private void cbBarRangeChanged(double low, double high)
		{
			BarArgRangeChanged eva = new BarArgRangeChanged();
			eva.low = low;
			eva.high = high;
			if (OnRangeChanged != null) {
				OnRangeChanged(this, eva);
			}
		}
		//----- コールバック関数（スクロール位置変化通知）------------------------------------------------------//
		private void cbBarNtcScrollPos(int pos)
		{
			BarArgNtcScrollPos eva = new BarArgNtcScrollPos();
			eva.pos = pos;
			if (OnNtcScrollPos != null) {
				OnNtcScrollPos(this, eva);
			}
		}
		//----- コールバック関数（ファイルドロップ通知）--------------------------------------------------------//
		private void cbBarFileDrop(int n)
		{
			BarArgFileDrop eva = new BarArgFileDrop();
			eva.n = n;
			if (OnFileDrop != null) {
				OnFileDrop(this, eva);
			}
		}
		//----- コールバック関数（フォルダドロップ通知）--------------------------------------------------------//
		private void cbBarDirDrop(int n)
		{
			BarArgDirDrop eva = new BarArgDirDrop();
			eva.n = n;
			if (OnDirDrop != null) {
				OnDirDrop(this, eva);
			}
		}
		//----- コールバック関数（右クリック通知）--------------------------------------------------------------//
		private void cbBarRClick(int x, int y, bool shift, bool ctrl)
		{
			BarArgRClick eva = new BarArgRClick();
			eva.x	  = x;
			eva.y	  = y;
			eva.shift = shift;
			eva.ctrl  = ctrl;
			if (OnRClick != null) {
				OnRClick(this, eva);
			}
		}
	}
	//==========================================================================================================//
	//	FontTxoプロパティ・エディタ	（ダイアログから、取り消し線や下線を隠す）									//
	//==========================================================================================================//
	internal class FontTxoEditor : UITypeEditor
	{
		//----- コンストラクタ ---------------------------------------------------------------------------------//
		public FontTxoEditor()
		{
		}
		//----- プロパティエディタとして、モーダル表示フォームを使用する事を通知 -------------------------------//
		public override UITypeEditorEditStyle GetEditStyle(ITypeDescriptorContext context)
		{
			return UITypeEditorEditStyle.Modal;
		}
		//----- 独自フォント設定 -------------------------------------------------------------------------------//
		public override object EditValue(ITypeDescriptorContext context, IServiceProvider provider, object value)
		{
			IWindowsFormsEditorService editorService = null;

			//----- MSDN UI 型エディターの実装 に基づいた記述
			if (provider != null) {
				editorService = provider.GetService(typeof(IWindowsFormsEditorService)) as IWindowsFormsEditorService;
			}

			//----- MSDN UI 型エディターの実装 に基づいた記述
			if (editorService == null) {
				return value;
			}
			
			//----- フォント表現文字列取得 ---------------------------------------------------------------------//
			Font f = (Font)value;
			FontDialog fd = new FontDialog();

			//	初期のフォント設定
			fd.Font = (Font)value;
			//	ポイントサイズの範囲
			fd.MaxSize = 144;
			fd.MinSize = 6;
			//	存在しないフォントやスタイルをユーザーが選択時、エラーメッセージを表示する
			fd.FontMustExist = true;
			//	横書きフォントだけを表示する
			fd.AllowVerticalFonts = false;
			//	色の選択
			fd.ShowColor = false;
			//	取り消し線、下線のオプション
			fd.ShowEffects = false;
			//	固定ピッチフォント以外も表示する
			fd.FixedPitchOnly = false;
			//	ベクタ フォントを選択できるようにする(デフォルト：True)
			fd.AllowVectorFonts = true;
			//	ダイアログを表示し、フォントを選択する
			if (fd.ShowDialog() != DialogResult.Cancel) {
				f = fd.Font;
			}
			return f;
		}
	}
}
