﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.Runtime.InteropServices;


//==============================================================================================================//
//																												//
//	複数のモジュールで使用する共通定義																			//
//																												//
//==============================================================================================================//
namespace CAjrCustCtrl
{
	//----------------------------------------------------------------------------------------------------------//
	//	定数																									//
	//----------------------------------------------------------------------------------------------------------//
	//----- テキスト出力時の右端／下端 -------------------------------------------------------------------------//
	public static class Txo {
		public const int Right	= 0x1FFF2000;
		public const int Bottom = 0x1FFF6000;
		public const int Center = 0x1FFFA000;
	}
	//----------------------------------------------------------------------------------------------------------//
	//	構造体／定数																							//
	//----------------------------------------------------------------------------------------------------------//
	//----- 言語種別 -------------------------------------------------------------------------------------------//
	public enum ELangId : int
	{
		English = 0, Japanese = 0x411
	}
	//----- ２Ｄベクトル ---------------------------------------------------------------------------------------//
	[StructLayout(LayoutKind.Sequential, Pack=1)]
	public struct AJC2DVEC {
		public AJC2DVEC(double x, double y)
		{
			this.x = x;
			this.y = y;
		}
		public double x, y;
	}
	//----- ２Ｄラインベクトル（始点と方向ベクトル）------------------------------------------------------------//
	[StructLayout(LayoutKind.Sequential, Pack=1)]
	public struct AJC2DLVEC {
		public AJC2DLVEC(AJC2DVEC p, AJC2DVEC v)
		{
			this.p.x = p.x;	this.v.x = v.x;
			this.p.y = p.y;	this.v.y = v.y;
		}
		public AJC2DVEC p, v;
	}
	//----- ３Ｄベクトル ---------------------------------------------------------------------------------------//
	[StructLayout(LayoutKind.Sequential, Pack=1)]
	public struct AJC3DVEC
	{
		public AJC3DVEC(double x, double y, double z)
		{
			this.x = x;
			this.y = y;
			this.z = z;
		}
		public double x, y, z;
	}
	//----- ３Ｄラインベクトル（始点と方向ベクトル）------------------------------------------------------------//
	[StructLayout(LayoutKind.Sequential, Pack=1)]
	public struct AJC3DLVEC
	{
		public AJC3DLVEC(AJC3DVEC p, AJC3DVEC v)
		{
			this.p.x = p.x;	this.v.x = v.x;
			this.p.y = p.y;	this.v.y = v.y;
			this.p.z = p.z;	this.v.z = v.z;
		}
		public AJC3DVEC p, v;
	}
	//----- ３Ｄラインポイント（始点と終点）-------------------------------------------------------------------//
	[StructLayout(LayoutKind.Sequential, Pack=1)]
	public struct AJC3DLINE
	{
		public AJC3DLINE(AJC3DVEC p1, AJC3DVEC p2)
		{
			this.p1.x = p1.x;	this.p2.x = p2.x;
			this.p1.y = p1.y;	this.p2.y = p2.y;
			this.p1.z = p1.z;	this.p2.z = p2.z;
		}
		public AJC3DVEC p1, p2;
	}
	//----- ３Ｄ三角形情報 -------------------------------------------------------------------------------------//
	[StructLayout(LayoutKind.Sequential, Pack=1)]
	public struct AJC3DTRI
	{
		public AJC3DTRI(AJC3DVEC p1, AJC3DVEC p2, AJC3DVEC p3)
		{
			this.p1.x = p1.x;	this.p1.y = p1.y;	this.p1.z = p1.z;
			this.p2.x = p2.x;	this.p2.y = p2.y;	this.p2.z = p2.z;
			this.p3.x = p3.x;	this.p3.y = p3.y;	this.p3.z = p3.z;
		}
		public AJC3DVEC p1, p2, p3;
	}
	//----- ３Ｄ行列 -------------------------------------------------------------------------------------------//
	[StructLayout(LayoutKind.Sequential, Pack=1)]
	public struct AJC3DMAT
	{
		public AJC3DMAT(double ini)
		{
			this.m = new double[3,3];
			this.m[0,0] = ini;	this.m[0,1] = ini;	this.m[0,2] = ini;	
			this.m[1,0] = ini;	this.m[1,1] = ini;	this.m[1,2] = ini;	
			this.m[2,0] = ini;	this.m[2,1] = ini;	this.m[2,2] = ini;	
		}
		public AJC3DMAT(double v00, double v01, double v02, double v10, double v11, double v12, double v20, double v21, double v22)
		{
			this.m = new double[3,3];
			this.m[0,0] = v00;	this.m[0,1] = v01;	this.m[0,2] = v02;	
			this.m[1,0] = v10;	this.m[1,1] = v11;	this.m[1,2] = v12;	
			this.m[2,0] = v20;	this.m[2,1] = v21;	this.m[2,2] = v22;	
		}
		public double[,]	m;
	}
	//----- ３Ｄ円の情報 ---------------------------------------------------------------------------------------//
	[StructLayout(LayoutKind.Sequential, Pack=1)]
	public struct AJC3DCIRINFO
	{
		public AJC3DLINE	lt1, lt2;			//	円に内接する２つの直線
		public AJC3DLINE	lc1, lc2;			//	内接線の中点からの垂線
		public AJC3DLVEC	lvc;				//	円の中心と法線
		public double		cr;					//	円の半径
	}
	//----- ３Ｄ球の情報 ---------------------------------------------------------------------------------------//
	[StructLayout(LayoutKind.Sequential, Pack=1)]
	public struct AJC3DSPHINFO {
		public AJC3DCIRINFO	cif1, cif2;			//	球に内接する２つの円
		public AJC3DLINE	ls1 , ls2;			//	内接円中心から球中心への直線
		public AJC3DVEC		sc;					//	球の中心
		public double		sr;					//	球の半径
	}
	//----- ３Ｄ／２Ｄ平面表示時の横軸／縦軸の種別 -------------------------------------------------------------//
	public enum EAJCPLANEAXIS {
		XP,										//	Ｘ軸昇順
		YP,										//	Ｙ軸 〃 
		ZP,										//	Ｚ軸 〃 
		XM,										//	Ｘ軸降順
		YM,										//	Ｙ軸 〃 
		ZM,										//	Ｚ軸 〃 
	}
	//----- ３Ｄ／２Ｄテキスト描画モード -----------------------------------------------------------------------//
	public enum EAJCTXOMD {
		RIGHT			,						//	点の右側
		LEFT			,						//	 〃 左側
		CENTER			,						//	 〃 中央
		BELLOW_RIGHT	,						//	点の下の右側
		BELLOW_LEFT		,						//	  〃	左側
		BELLOW_CENTER	,						//	  〃	中央
		ABOVE_RIGHT		,						//	点の上の右側
		ABOVE_LEFT		,						//	  〃	左側
		ABOVE_CENTER	,						//	  〃	中央
	}
	//----- コンソールアプリの終了ハンドラ・ルーチンに渡される定数 ---------------------------------------------//
	public enum EAJCEXITTYPE
	{
		CTRL_C_EVENT		= 0,
		CTRL_BREAK_EVENT	= 1,
		CTRL_CLOSE_EVENT	= 2,
		CTRL_LOGOFF_EVENT	= 5,
		CTRL_SHUTDOWN_EVENT = 6
	}

	//----- 位置情報 -------------------------------------------------------------------------------------------//
	[StructLayout(LayoutKind.Sequential, Pack=1)]
	public struct AJCPOINT {
		public int	x;
		public int	y;
	}
	//----- 矩形情報 -------------------------------------------------------------------------------------------//
	[StructLayout(LayoutKind.Sequential, Pack=1)]
	public struct AJCRECT {
		public int	left;
		public int	top;
		public int	right;
		public int	bottom;
	}
	//----- サイズ情報 -----------------------------------------------------------------------------------------//
	[StructLayout(LayoutKind.Sequential, Pack=1)]
	public struct AJCSIZE {
		public int	cx;
		public int	cy;
	}
	//----------------------------------------------------------------------------------------------------------//
	//	テキストファイル エンコード																				//
	//----------------------------------------------------------------------------------------------------------//
	//----- テキストエンコード ---------------------------------------------------------------------------------//
	public enum ETextEncode : int
	{
		TEC_MBC			= 0,		//	マルチバイト
		TEC_UTF_8		= 1,		//	ＵＴＦ－８
		TEC_EUC_J		= 2,		//	ＥＵＣ（日本語）
		TEC_UTF_16LE	= 3,		//	ＵＴＦ－１６ＬＥ
		TEC_UTF_16BE	= 4,		//	ＵＴＦ－１６ＢＥ
		TEC_AUTO		= 9,		//	ＡＵＴＯ（入力時のみ有効）
	}
	//----- ＢＯＭ出力（書き込み用）----------------------------------------------------------------------------//
	public enum EBomMode : int
	{
		NOT_WRITE_BOM	= 0,		//	ＢＯＭを書き込まない
		WRITE_BOM		= 1,		//	ＢＯＭを書き込む
	}
	//----------------------------------------------------------------------------------------------------------//
	//	テキストファイル 改行コード変換モード																	//
	//----------------------------------------------------------------------------------------------------------//
	public enum ETextLfConv : int
	{
		NONE			= 0	,		//	変換無し
		LF_TO_CRLF			,		//	ＬＦ → ＣＲ，ＬＦ（ファイル書き込み時のデフォルト）
		LF_TO_CR			,		//	ＬＦ → ＣＲ
		CR_TO_CRLF			,		//	ＣＲ → ＣＲ，ＬＦ
		CR_TO_LF			,		//	ＣＲ → ＬＦ
		CRLF_TO_LF			,		//	ＣＲ，ＬＦ → ＬＦ（ファイル読み出し時のデフォルト）
		CRLF_TO_CR			,		//	ＣＲ，ＬＦ → ＣＲ
		LF					,		//	ＬＦで改行，変換無し
		CR					,		//	ＣＲで改行，変換無し
		LF_CRSKIP			,		//	ＬＦで改行，ＣＲ除去
		CR_LFSKIP			,		//	ＣＲで改行，ＬＦ除去
	}

	//----------------------------------------------------------------------------------------------------------//
	//	２Ｄ／３Ｄグラフィック 構造体／定数																		//
	//----------------------------------------------------------------------------------------------------------//
	//----- プロット通知情報 -----------------------------------------------------------------------------------//
	public struct PlotInfo
	{
		public int		ix;
		public AJC3DVEC	v;
	}
	//----- 2D / 3D モード識別 ---------------------------------------------------------------------------------//
	public enum EDIMMODE : int
	{
		MODE_2D	= 2,
		MODE_3D = 3
	}
	//----------------------------------------------------------------------------------------------------------//
	//	棒グラフ 定数																							//
	//----------------------------------------------------------------------------------------------------------//
	//----- 線の属性 -------------------------------------------------------------------------------------------//
	public enum EBarLineStyle : int
	{
		Solid		= 0,		//	
		Dash		= 1,		//	-------
		Dot			= 2,		//	.......
		DashDot		= 3,		//	_._._._
		DashDotDot	= 4,		//	_.._.._
		NullLine	= 5,		//	
		InsideFrame = 6			//	
	}
	//----------------------------------------------------------------------------------------------------------//
	//	Ｃ言語プリコンパイル 定数																				//
	//----------------------------------------------------------------------------------------------------------//
	//----- プリプロセス文コメント化項目 -----------------------------------------------------------------------//
	public enum EPpcComItem : int {
		DEFINE		= 0x0001,			//	#define
		UNDEF		= 0x0002,			//	#undef
		INCLUDE		= 0x0004,			//	#include
		IFDEF		= 0x0008,			//	#ifdef
		IFNDEF		= 0x0010,			//	#ifdef
		IF			= 0x0020,			//	#if
		ELIF		= 0x0040,			//	#ifndef
		ELSE		= 0x0080,			//	#else
		ENDIF		= 0x0100,			//	#endif
		OTHERS		= 0x0200,			//	その他（#pragma等）

		COND		= (IFDEF | IFNDEF | IF | ELIF | ELSE | ENDIF),
		ALL			= (COND  | DEFINE | UNDEF),
	}
	//----- オプションフラグ -----------------------------------------------------------------------------------//
	public enum EPpcOption : int {
		NONE			= 0x0000,					//	オプション無し
		AUTOSRH			= 0x0001,					//	インクルードファイル自動検索フラグ
		ONCE			= 0x0002,					//	同一インクルードファイルを１回だけ読み出すフラグ
		GENALL			= 0x0004,					//	非生成部分（条件コンパイル＝偽の部分）もファイル出力する
		AUTOSRH_ONCE	= (AUTOSRH | ONCE  ),
		ONCE_GENALL		= (ONCE    | GENALL),
		AUTOSRH_GENALL	= (AUTOSRH | GENALL),
		ALL 			= (AUTOSRH | ONCE | GENALL),
	}
	//----- プリコンパイル結果 ---------------------------------------------------------------------------------//
	public enum EPpcResult : int {
		OK			= 0	,			//	ＯＫ
		NOFILE			,			//	ファイル無し
		MEMERR			,			//	メモリエラー
		STOP			,			//	中止
		NOTOKEN			,			//	トークン無し
		PARAM			,			//	パラメタエラー
		OUTFILE		= 99,			//	出力ファイル生成失敗
	}
	//----- 通知コード -----------------------------------------------------------------------------------------//
	public enum EPpcNtc : int {
		FILE_LNO		= 1	,		//	ファイル名，行番号通知							"ファイル名"		行＃				ネスト値（０～）
		SRH_START		,			//	インクルードファイル検索開始通知				"Incファイル名"		"検索フォルダ"		－
		SRH_DIR			,			//	インクルードファイル検索中のフォルダ通知		"フォルダパス"		－					－
		SRH_END			,			//	インクルードファイル検索終了通知				"Incファイル名"		－					－
		OPTSYM			,			//	プリプロセス用オプションシンボル通知			PCAJCPPCTKNNODE		－					－
		MACDEF			,			//	マクロ定義通知									PCAJCPPCTKNNODE		－					－
		MACREF			,			//	マクロ参照通知									PCAJCPPCTKNNODE		PCAJCPPCMACINFO		－
		OUTLOOP			,			//	トークンストリームをファイルへ出力ループ中		－					－					－
		SRCTEC			,			//	ソースファイルのエンコード通知					"ファイルパス"		エンコード			ＢＯＭ
//		TOKEN			,			//	ソースから読み出したトークン通知				トークンコード		"トークン文字列"	－
	}
	//----- エラーコード ---------------------------------------------------------------------------------------//
	public enum EPpcErr {
		//--- 一般エラー -----------//
		ERROR_OK			= 0	,	//	正常											－					－					－
		ERROR_SRC_OPEN			,	//	ソースファイルをオープンできません				"ファイルパス"		－					－
		ERROR_INC_OPEN			,	//	INCLUDEファイルをオープンできません				"ファイルパス"		行＃				"Includeパス"
		ERROR_NO_SYMBOL			,	//	#ifdef/#ifndefでシンボル名が指定されていません	"ファイルパス"		行＃				－
		ERROR_COND_NOTCLS		,	//	条件(#if～#endif)がクローズされていません		"ファイルパス"		行＃(=1固定)		－
		ERROR_COND_DEEP			,	//	条件(#if～#endif)のネストが深すぎます			"ファイルパス"		行＃				－
		ERROR_NOT_IN_IF			,	//	対応する「#if / #ifdef / #ifndef」がありません	"ファイルパス"		行＃				－
		ERROR_ELIF_IN_ELSE		,	//	「#else」条件中に「#elif」は記述できません		"ファイルパス"		行＃				－
		ERROR_ELSE_IN_ELSE		,	//	「#else」条件中に「#else」は記述できません		"ファイルパス"		行＃				－
		//--- defined --------------//
		DEFERR_INVALID	= 1000	,	//	definedの構文誤り								"ファイルパス"		行＃				－
		DEFERR_NEED_LP			,	//	defined'の後に左括弧 '(' が必要です				"ファイルパス"		行＃				－
		DEFERR_NEED_SYMBOL		,	//	defined'でシンボルが指定されていません			"ファイルパス"		行＃				－
		DEFERR_NEED_RP			,	//	シンボルの次に右括弧 ')' が必要です				"ファイルパス"		行＃				"シンボル"
		//--- MACRO ----------------//
		MACERR_NEED_LP	= 2000	,	//	マクロ名の後に左括弧が必要です					"ファイルパス"		行＃				"マクロ名"
		MACERR_NEED_RP			,	//	仮引数の後に右括弧が必要です					"ファイルパス"		行＃				"仮引数"
		MACERR_NEED_RP_C			,	//	仮引数の後に右括弧かカンマが必要です			"ファイルパス"		行＃				"仮引数"
		MACERR_MULTDEF			,	//	マクロ二重定義									"ファイルパス"		行＃				"マクロ名"
		MACERR_INV_NAME			,	//	不正なマクロ名です								"ファイルパス"		行＃				－
		MACERR_NO_NAME			,	//	マクロ名がありません							"ファイルパス"		行＃				－
		MACERR_NEST_OVER			,	//	マクロ展開ネストオーバー						－					－					－
		MACERR_ARG_LACK			,	//	マクロの引数が少なすぎる						"ファイルパス"		行＃				"マクロ名"
		MACERR_ARG_OVER			,	//	マクロの引数が多すぎる							"ファイルパス"		行＃				"マクロ名"
		//--- INCLUDE --------------//
		INCERR_NO_FILE	= 3000	,	//	Include ファイル名が指定されていません			"ファイルパス"		行＃				－
		INCERR_INV_FILE			,	//	Include ファイルの記述が不正です				"ファイルパス"		行＃				－
		INCERR_NOTCLS			,	//	Include ファイル名の後に '>' がありません		"ファイルパス"		行＃				－
		INCERR_NEST_OVER			,	//	Include ファイルのネストオーバー				"ファイルパス"		行＃				－
		INCERR_NOT_FOUND			,	//	Include ファイルが見つかりません				"ファイルパス"		行＃				"Includeファイル"
		//--- 数式評価エラー -------//
		SCLERR_INVSYL	= 5000	,	//	無効な語句										"ファイルパス"		行＃				"語句"
		SCLERR_DIVZERO			,	//	ゼロ除算										"ファイルパス"		行＃				－
		SCLERR_NOTCLS			,	//	括弧が閉じられていない							"ファイルパス"		行＃				－
		SCLERR_EXPRESSION		,	//	不当な数式表現									"ファイルパス"		行＃				"不正な語句"
		SCLERR_EOL				,	//	数式表現が途中で終了している					"ファイルパス"		行＃				－
		SCLERR_OVERNEST			,	//	数式表現のネストオーバー						"ファイルパス"		行＃				－
		//--- メモリ割り当て失敗 ---//
		ERROR_MEMALLOC	= 9999		//	メモリ割り当て失敗								－					－					－
	}
	//----------------------------------------------------------------------------------------------------------//
	//	 Ｃ言語字句分解 定数																					//
	//----------------------------------------------------------------------------------------------------------//
	//----- 機能フラグ -----------------------------------------------------------------------------------------//
	public enum ECtkOpt : uint {
		//	Bit 7 - 0
		CPLUSPLUS		=	0x00000001	,		//	Ｃ＋＋のトークンを含める
		INCSPACE		=	0x00000002	,		//	空白／改行／コメントもトークンに含める
		INCSYM_DOLLAR	=	0x00000004	,		//	シンボルの英字としてドルマーク($)を含める
		INCSYM_ATMARK	=	0x00000008	,		//	シンボルの英字としてアットマーク(@)を含める	
		INCSYM_MBSTR	=	0x00000010	,		//	シンボルの英字としてマルチバイト文字を含める
		APOST_NOESC		=	0x00000020	,		//	アポストロフィ(')で囲まれた文字列は、エスケース文字(\)を認識しない
		DOTSYMBOL		=	0x00000040	,		//	シンボル＋Dot(.)＋数字列を１つのシンボルとみなす(Ex. P3.12)
		ASMHEX			=	0x00000080	,		//	アセンブラ記法の16進数（末尾がH, ex 0FFH)を許可する

		//	Bit 31-24
		LBF_1K			=	0x01000000	,		//	テキストファイル行の読み出しバッファサイズ＝ ２KB
		LBF_2K			=	0x02000000	,		//			〃								  ＝ ４KB
		LBF_4K			=	0x04000000	,		//			〃								  ＝ ８KB
		LBF_8K			=	0x08000000	,		//			〃								  ＝ 16KB
		LBF_16K			=	0x10000000	,		//			〃								  ＝ 32KB
		LBF_32K			=	0x20000000	,		//			〃								  ＝ 64KB
		LBF_64K			=	0x40000000	,		//			〃								  ＝128KB
		LBF_128K		=	0x80000000	,		//			〃								  ＝256KB
	}
	//----- タブ幅 ---------------------------------------------------------------------------------------------//
	public enum ECtkTabWidth {
		TABW_2	=  2,
		TABW_4	=  4,
		TABW_8	=  8,
		TABW_16	=  16,
	}
	//----- トークン識別フラグ ---------------------------------------------------------------------------------//
	public enum ECtkFlg {
		PREPRO			=	0x0080,				//	プリプロセス文
		PPTOP			=	0x0040,				//	プリプロセス文の先頭トークン（＃）
		MACNAME			=	0x0020,				//	マクロ名
		MACWITHARG		=	0x0010,				//	引数付のマクロ名
		MACARG			=	0x0008,				//	マクロ引数' ( ・・・ ) '
		MACBODY			=	0x0004,				//	マクロボディ
		NXTSPC			=	0x0002,				//	トークンの次に空白ありを示すフラグ
		WIDECHAR		=	0x0001,				//	ワイド文字（L'X' / L"XXX"）フラグ
	}

	//----- エラーコード ---------------------------------------------------------------------------------------//
	public enum ECtkError {
		OK				 		=	0		,	//	ＯＫ
		NULLPTR					=	0x8000	,	//	ＮＵＬＬポインタが指定された
		ZEROSIZE				=	0x4000	,	//	バッファサイズがゼロ
		MEMALLOC				=	0x2000	,	//	メモリ割り当て失敗
		INVALID_OCTALNUMBER		=	0x1000	,	//	不正な８進数
		INVALID_HEXADECIMAL		=	0x0800	,	//	不正な１６進数
		INVALID_REALNUMBER		=	0x0400	,	//	不正な実数表現
		INVALID_SUFFIX			=	0x0200	,	//	不正なサフィックス
		INVALID_DELIMITER		=	0x0100	,	//	不正なデリミタ
		LFINSTR					=	0x0080	,	//	文字列中に改行が含まれている
		BUFOVER					=	0x0040	,	//	トークンバッファオーバー
		EOFINCOMMENT			=	0x0020	,	//	コメント中にＥＯＦ検出（ファイル内でコメントが閉じていない）
	}

	//----- トークン識別コード ---------------------------------------------------------------------------------//
	public enum ECtkKnd {
		USRSYM					=	0x0100	,	//	ユーザシンボル
		RSVSYM					=	0x0200	,	//	予約語
		VALUE					=	0x0400	,	//	数値定数
		STRING					=	0x0800	,	//	文字列
		PATHNAME				=	0x1000	,	//	ファイルパス名（#includeのみ)
		DELIMIT					=	0x2000	,	//	デリミタ
		SPACE					=	0x4000	,	//	空白／改行／コメント／行継続記号（￥）
		NEXT					=	0x8000	,	//	ユーザ定義用
	}

	//----- 数値定数のサフィックス・コード ---------------------------------------------------------------------//
	public enum ECtkSuf {
		SUF_NONE	=	0	,		//	サフィックスなし
		SUF_L				,		//	L
		SUF_UL				,		//	UL / LU
		SUF_LL				,		//	LL
		SUF_ULL				,		//	ULL
		SUF_FLOAT			,		//	F
		SUF_DOUBLE			,		//	D
		SUF_LDOUBLE					//	L
	}

	//----- トークンコード -------------------------------------------------------------------------------------//
	public enum ECtkCode {
		EOF			= 0							,	//	End Of File

		USR_NAME	= ECtkKnd.USRSYM			,	//	ユーザ定義名

		RSV_ASM		= ECtkKnd.RSVSYM			,	//	asm
		RSV_AUTO								,	//	auto
		RSV_BREAK								,	//	break
		RSV_CASE								,	//	case
		RSV_CATCH								,	//	catch		（C++）
		RSV_CHAR								,	//	char
		RSV_CLASS								,	//	class		（C++）
		RSV_CONST								,	//	const
		RSV_CONCAST								,	//	const_cast	（C++）
		RSV_CONTINUE							,	//	continue
		RSV_DEFAULT								,	//	default
		RSV_DEFINE								,	//	define		(プリプロセッサ）
		RSV_DEFINED								,	//	defined		(プリプロセッサ）
		RSV_DELETE								,	//	delete		（C++）
		RSV_DO									,	//	do
		RSV_DOUBLE								,	//	double
		RSV_DYNCAST								,	//	dynamic_cast（C++）
		RSV_ELIF								,	//	elif		(プリプロセッサ）
		RSV_ELSE								,	//	else		(プリプロセッサ兼用）
		RSV_ENDIF								,	//	endif		(プリプロセッサ）
		RSV_ENUM								,	//	enum
		RSV_EXTERN								,	//	extern
		RSV_FLOAT								,	//	float
		RSV_FOR									,	//	for
		RSV_FRIEND								,	//	friend		（C++）
		RSV_GOTO								,	//	goto
		RSV_IF									,	//	if			(プリプロセッサ兼用）
		RSV_IFDEF								,	//	ifdef		(プリプロセッサ）
		RSV_IFNDEF								,	//	ifndef		(プリプロセッサ）
		RSV_INCLUDE								,	//	include		(プリプロセッサ）
		RSV_INLINE								,	//	inline
		RSV_INT									,	//	int
		RSV_LONG								,	//	long
		RSV_NEW									,	//	new			（C++）
		RSV_OPERATOR							,	//	operator	（C++）
		RSV_PRAGMA								,	//	pragma		(プリプロセッサ）
		RSV_PRIVATE								,	//	private		（C++）
		RSV_PROTECTED							,	//	protected	（C++）
		RSV_PUBLIC								,	//	public		（C++）
		RSV_REGISTER							,	//	register
		RSV_REICAST								,	//	reinterpret_cast（C++）
		RSV_RETURN								,	//	return
		RSV_SHORT								,	//	short
		RSV_SIGNED								,	//	signed
		RSV_SIZEOF								,	//	sizeof
		RSV_STATIC								,	//	static
		RSV_STACAST								,	//	static_cast	（C++）
		RSV_STRUCT								,	//	struct
		RSV_SWITCH								,	//	switch
		RSV_TEMPLATE							,	//	template	（C++）
		RSV_THIS								,	//	this		（C++）
		RSV_THROW								,	//	tnrow		（C++）
		RSV_TRY									,	//	try			（C++）
		RSV_TYPEDEF								,	//	typedef
		RSV_UNDEF								,	//	undef
		RSV_UNION								,	//	union
		RSV_UNSIGNED							,	//	unsigned
		RSV_VIRTUAL								,	//	virtual		（C++）
		RSV_VOID								,	//	void
		RSV_VOLATILE							,	//	volatile
		RSV_WCHAR_T								,	//	wchar_t
		RSV_WHILE								,	//	while


		VAL_DECIMAL	= ECtkKnd.VALUE				,	//	nnn	
		VAL_HEXA								,	//	0xhhh
		VAL_OCTAL								,	//	0nnn
		VAL_REAL								,	//	nn.mm / nnnE[+-]mm
		VAL_INVALID								,	//	不完全な数値表現（エラー）

		VAL_HEXA_H								,	//	hhhH（アセンブラ記法）


		STR_QUOTE	= ECtkKnd.STRING			,	//	"..."
		STR_APOST								,	//	'...'
		STR_NEXT								,	//	その他の文字定数


		PATHNAME	= ECtkKnd.PATHNAME			,	//	#includeのヘッダファイルパス名

		DLM_LSPART	= ECtkKnd.DELIMIT			,	//	(
		DLM_LMPART								,	//	{
		DLM_LLPART								,	//	[
		DLM_RSPART								,	//	)
		DLM_RMPART								,	//	}
		DLM_RLPART								,	//	]

		DLM_SCOPE								,	//	::		(C++)
		DLM_IDDOT								,	//	.*		(C++)
		DLM_IDARROW								,	//	->*		(C++)

		DLM_LAND								,	//	&&
		DLM_LOR									,	//	||

		DLM_ARROW								,	//	->
		DLM_QUEST								,	//	?
		DLM_COLON								,	//	:
		DLM_SEMICOL								,	//	;
		DLM_COMMA								,	//	,
		DLM_VARG								,	//	...	
		DLM_DOT									,	//	.

		DLM_SHREQ								,	//	>>=	
		DLM_SHLEQ								,	//	<<=	
		DLM_SHR									,	//	>>
		DLM_SHL									,	//	<<

		DLM_EQEQ								,	//	==
		DLM_NOTEQ								,	//	!=
		DLM_LOEQ								,	//	<=
		DLM_HIEQ								,	//	>=
		DLM_LO									,	//	<
		DLM_HI									,	//	>

		DLM_PLUSEQ								,	//	+=
		DLM_MINUSEQ								,	//	-=
		DLM_MULTEQ								,	//	*=
		DLM_DIVEQ								,	//	/=
		DLM_MODEQ								,	//	%=
		DLM_ANDEQ								,	//	&=
		DLM_OREQ								,	//	|=
		DLM_XOREQ								,	//	^=

		DLM_SHARP2								,	//	##
		DLM_PLUS2								,	//	++
		DLM_MINUS2								,	//	--

		DLM_SHARP								,	//	#
		DLM_PLUS								,	//	+
		DLM_MINUS								,	//	-
		DLM_MULT								,	//	*
		DLM_DIV									,	//	/
		DLM_MOD									,	//	%
		DLM_AND									,	//	&
		DLM_OR									,	//	|
		DLM_XOR									,	//	^
		DLM_EQ									,	//	=
		DLM_NOT									,	//	~
		DLM_LNOT								,	//	!
		DLM_UNKNOWN								,	//	未定義のデリミタ
		DLM_NEXT								,	//	その他のデリミタ

		SPACE		= ECtkKnd.SPACE				,	//	空白／タブ列
		EOL										,	//	改行文字列
		COMMENT									,	//	コメント（／＊・・・＊／）
		LINECOMMENT								,	//	コメント（／／・・・）
		LINECONT								,	//	行継続記号（￥）
		NEXTSPACE								,	//	その他の空白
	}
	//----------------------------------------------------------------------------------------------------------//
	//	ファイル検索 定数																						//
	//----------------------------------------------------------------------------------------------------------//
	//----- ファイル属性 ---------------------------------------------------------------------------------------//
	public enum EFileAtt
	{
		_A_ARCH		= 0x20	,		//	アーカイブファイル
		_A_SUBDIR	= 0x10	,		//	サブディレクトリ
		_A_SYSTEM	= 0x04	,		//	システムファイル
		_A_HIDDEN	= 0x02	,		//	隠しファイル
		_A_RDONLY	= 0x01	,		//	読み出し専用ファイル
		_A_ALL		= 0x37	,		//	上記の全属性
	}
	//----------------------------------------------------------------------------------------------------------//
	//	汎用モジュール 定数																						//
	//----------------------------------------------------------------------------------------------------------//
	//----- Windows終了モード ----------------------------------------------------------------------------------//
	public enum EExitWindows 
	{
		EWX_LOGOFF		= 0,		//	ログオフ
		EWX_SHUTDOWN	= 1,		//	シャットダウン
		EWX_REBOOT		= 2,		//	リブート
		EWX_FORCE		= 4,		//	プロセス強制終了
		EWX_POWEROFF	= 8,		//	パワーオフ
	}
	//----- フォルダコピー通知 ---------------------------------------------------------------------------------//
	public enum ENtcDirCopy
	{
		SUCCESS		= 0,
		FAILURE		= 1,
		EXIST		= 2,
	}
	//----- ファイルコピー通知 ---------------------------------------------------------------------------------//
	public enum ENtcFileCopy
	{
		SUCCESS		= 0,
		FAILURE		= 1,
		EXIST		= 2,
	}

	//----------------------------------------------------------------------------------------------------------//
	//	シリアル通信 定数																						//
	//----------------------------------------------------------------------------------------------------------//
	//----- ＳＣＰ実行モード -----------------------------------------------------------------------------------//
	public enum EScpMode
	{
		NotificationByEvent	= 0,		//	イベントを通知する	（フォームアプリ）
		WaitingForEvent		= 1			//	イベントを待ち受ける（コンソールアプリ）
	}
	//----- 通信リソース ---------------------------------------------------------------------------------------//
	public enum ESelPort {
		SEL_COMPORT		= 0,
		SEL_MAILSLOT	= 1,
		SEL_SOCKET		= 2
	}
	//----- ＳＣＰポート状態 -----------------------------------------------------------------------------------//
	public enum EScpPortState
	{
		Closed		= 0,				//	クローズ状態
		Opened		= 1,				//	オープン状態
		OpenFailure	= 2,				//	オープン失敗
		PortChanged	= 3,				//	ポート番号が変化
		PortNumber	= 3,
		MySlotFail	= 4,				//	自メールスロット生成失敗
		TxFailure	= 5					//	送信失敗
	}
	//----- ＳＣＰチャンクデータの通知モード -------------------------------------------------------------------//
	public enum EScpChunkMode
	{
		BinaryData		= 0x01,			//	バイナリ・チャンクを通知
		TextData		= 0x02,			//	テキスト・チャンクを通知
		Both			= 0x03,			//	両方
	}
	//----- ＳＣＰデータビット数 -------------------------------------------------------------------------------//
	public enum EScpDataBits
	{
		DataBit_Default	= 0,			//	現在の設定値
		DataBit_7		= 7,			//	７ビット長
		DataBit_8		= 8				//	８ビット長
	}
	//----- ＳＣＰパリティビット -------------------------------------------------------------------------------//
	public enum EScpParity
	{
		DefaultParity	= 0		,		//	現在の設定値
		NoParity		= 'N'	,		//	パリティなし
		OddParity		= 'O'	,		//	奇数パリティ
		EvenParity		= 'E'	,		//	偶数パリティ
	}
	//----- ＳＣＰストップビット -------------------------------------------------------------------------------//
	public enum EScpStopBit
	{
		StopBit_Default	= 0		,		//	現在の設定値
		StopBit_1		= 1		,		//	１ビット
		StopBit_2		= 2				//	２ビット
	}
	//----- ＳＣＰイベントコード -------------------------------------------------------------------------------//
	public enum EScpEvt
	{
		EV_NOEVENT		= 0		,			//	イベント待ちタイムアウト
		EV_PORTSTATE	= 0x8000,			//	ポート状態通知
		EV_RXCHUNK		= 0x4000,			//	チャンクデータ受信通知
		EV_RXTEXT		= 0x2000,			//	テキスト受信通知
		EV_RXESC		= 0x1000,			//	ＥＳＣコード受信通知
		EV_RXCTRL		= 0x0800,			//	制御コード受信通知
		EV_RXPKT		= 0x0400,			//	パケットデータ受信通知
		EV_TXEMPTY		= 0x0200,			//	送信完了
		EV_RXNOPKT		= 0x0100,			//	パケット外テキスト通知

		EV_ERR			= 0x0080,			//	エラー通知
	//	EV_BREAK		= 0x0040,			//	BREAK検出通知
		EV_RING			= 0x0020,			//	RING変化通知
		EV_RLSD			= 0x0010,			//	RLSD変化通知
		EV_DSR			= 0x0008,			//	DSR変化通知
		EV_CTS			= 0x0004,			//	CTS変化通知

		EV_RXWORD14		= 0x0002,			//	バイトペアによるワード(14Bit値)受信通知
		EV_INVCHUNK		= 0x0001,			//	不正チャンクテキスト受信通知

		EV_SSEP			= (EV_RXTEXT	| EV_RXESC | EV_RXCTRL | EV_RXPKT),
		EV_DEFAULT_EVT	= (EV_SSEP		| EV_DSR   | EV_CTS),
		EV_DEFAULT_POST = (EV_PORTSTATE | EV_DEFAULT_EVT),

		EV_ALL			= ( EV_PORTSTATE |
							EV_RXCHUNK	 |
							EV_RXTEXT	 |
							EV_RXESC	 |
							EV_RXCTRL	 |
							EV_RXPKT	 |
							EV_TXEMPTY	 |
							EV_RXNOPKT	 |
							EV_ERR		 |
						//	EV_BREAK	 |
							EV_RING		 |
							EV_RLSD		 |
							EV_DSR		 |
							EV_CTS		 |
							EV_RXWORD14	 |
							EV_INVCHUNK
						  )
	}
	//----- エラーコード ---------------------------------------------------------------------------------------//
	public enum EScpErr
	{
		CE_RXOVER	= 0x0001,		//	受信キューオーバフロー
		CE_OVERRUN	= 0x0002,		//	オーバーランエラーを検出した
		CE_RXPARITY	= 0x0004,		//	パリティエラーを検出した
		CE_FRAME	= 0x0008,		//	フレーミングエラーを検出した
		CE_BREAK	= 0x0010,		//	ブレーク信号を検出した
		CE_TXFULL	= 0x0100,		//	送信キュー満杯
		CE_IOE		= 0x0400,		//	デバイスアクセス中にI/Oエラーを検出した
		
		RXERR		= 0x10000,		//	受信エラー
		TXERR		= 0x20000		//	送信エラー
	}
	//----- 受信テキストコード ---------------------------------------------------------------------------------//
	public enum EScpRxTextCode
	{
		SJIS			= 1	,				//	Ｓ－ＪＩＳ
		EUC				= 2	,				//	ＥＵＣ
		UTF8			= 3	,				//	ＵＴＦ－８
		AUTO			= 9	,				//	自動認識
	}
	//----- 送信テキストコード ---------------------------------------------------------------------------------//
	public enum EScpTxTextCode
	{
		SJIS			= 1	,				//	Ｓ－ＪＩＳ
		EUC				= 2	,				//	ＥＵＣ
		UTF8			= 3	,				//	ＵＴＦ－８
		AUTO			= 9	,				//	受信テキストコードに合わせる
	}
	//----- バイトペア受信順序 ---------------------------------------------------------------------------------//
	public enum EScpByteSeq
	{
		HIGH_BYTE_FIRST = 0,				//	High byte first
		LOW_BYTE_FIRST	= 1	,				//	Low byte first
	}

	//----------------------------------------------------------------------------------------------------------//
	//	ソケットサーバ 定数																						//
	//----------------------------------------------------------------------------------------------------------//
	//----- ＳＳＶ実行モード -----------------------------------------------------------------------------------//
	public enum ESsvMode
	{
		NotificationByEvent	= 0,		//	イベントを通知する
		WaitingForEvent		= 1			//	イベントを待ち受ける
	}
	//----- ＳＳＶチャンクデータの通知モード -------------------------------------------------------------------//
	public enum ESsvChunkMode
	{
		BinaryData		= 0x01,			//	バイナリ・チャンクを通知
		TextData		= 0x02,			//	テキスト・チャンクを通知
		Both			= 0x03,			//	両方
	}
	//----- ＳＳＶイベントコード -------------------------------------------------------------------------------//
	public enum ESsvEvt
	{
		EV_NOEVENT		= 0,				//	イベント無し

		EV_RXTEXT		= 0x00000001,		//	テキストデータ通知
		EV_RXESC		= 0x00000002,		//	ＥＳＣコードデータ通知
		EV_RXPKT		= 0x00000004,		//	パケットデータ通知
		EV_RXCTRL		= 0x00000008,		//	制御コード通知
		EV_RXNOPKT		= 0x00000010,		//	パケット外テキストデータ
		
		EV_RXCHUNK		= 0x00000100,		//	チャンクデータ受信通知
		EV_INVCHUNK		= 0x00000200,		//	不正チャンクテキスト受信通知
		EV_TXEMPTY		= 0x00000400,		//	送信完了通知
		EV_CONNECT		= 0x00000800,		//	接続通知
		EV_DISCONNECT	= 0x00001000,		//	切断通知
		EV_START		= 0x00002000,		//	サーバ開始通知
		EV_STOP			= 0x00004000,		//	サーバ終了通知
		EV_RXERR		= 0x00010000,		//	受信エラー通知
		EV_TXERR		= 0x00020000,		//	送信エラー通知
		EV_ERR			= 0x08000000,		//	エラー通知

		EV_SSEP			= (EV_RXTEXT  | EV_RXESC	  | EV_RXCTRL  | EV_RXPKT),
		EV_COMM			= (EV_RXCHUNK | EV_INVCHUNK   | EV_TXEMPTY),
		EV_GENERAL		= (EV_CONNECT | EV_DISCONNECT | EV_START   | EV_STOP),
		EV_ERRS			= (EV_RXERR   | EV_TXERR	  | EV_ERR),
		EV_CLIENT		= (EV_SSEP	  | EV_COMM 	  | EV_CONNECT | EV_DISCONNECT | EV_RXERR | EV_TXERR),
		EV_DEFAULT		= (EV_SSEP	  | EV_COMM 	  | EV_GENERAL | EV_ERRS),
		EV_ALL			= (EV_SSEP	  | EV_COMM 	  | EV_GENERAL | EV_ERRS | EV_RXNOPKT)
	}
	//----- エラーコード ---------------------------------------------------------------------------------------//
	public enum ESsvErrorCode {
		ERR_CREEVT			=  10,		//	終了通知用イベントオブジェクト生成失敗
		ERR_SOCKET			=  20,		//	接続要求待機用ソケット生成失敗(socket)
		ERR_BIND			=  30,		//	バインド失敗（bindt）
		ERR_LISTEN			=  40,		//	LISTEN失敗(listen)
		ERR_ACCEPT			=  50,		//	ACCEPT失敗（accept）
		ERR_ADDRINFO		=  60,		//	アドレス情報取得失敗（getaddrinfo）
		ERR_SOCKOPT			=  70,		//	ソケットオプション設定失敗(setsockopt)
		ERR_THREADPOWCTRL	=  80,		//	電源制御スレッド開始失敗
		ERR_THREADLISTEN	=  90,		//	接続待機スレッド開始失敗
		ERR_THREADCLIENT	= 100,		//	クライアント通信スレッド開始失敗
		ERR_CRESSEP			= 110,		//	ストリーム分離インスタンス生成失敗
		ERR_CREMBXTXD		= 120,		//	送信データ用メールボックス生成失敗
		ERR_TIMEOUT			= 130,		//	サーバ終了タイムアウト
		ERR_CRETHREADWORK	= 140,		//	クライアントスレッド用ワーク確保失敗
		ERR_CONNOVER		= 150,		//	クライアント接続数オーバー
		ERR_CREIXMAP		= 160,		//	インデクス割り当て用マップテーブル生成失敗
	}
	//----- 受信テキスト文字コード -----------------------------------------------------------------------------//
	public enum ESsvRxTextCode
	{
		SJIS			= 1	,				//	Ｓ－ＪＩＳ
		EUC				= 2	,				//	ＥＵＣ
		UTF8			= 3	,				//	ＵＴＦ－８
		AUTO			= 9	,				//	自動認識
	}
	//----- 送信テキスト文字コード -----------------------------------------------------------------------------//
	public enum ESsvTxTextCode
	{
		SJIS			= 1	,				//	Ｓ－ＪＩＳ
		EUC				= 2	,				//	ＥＵＣ
		UTF8			= 3	,				//	ＵＴＦ－８
		AUTO			= 9	,				//	受信テキストコードに合わせる
	}
	//----- アドレスファミリ -----------------------------------------------------------------------------------//
	public enum ESsvFamily
	{
		INET			= 0x02,				//	IPV4
		INET6			= 0x17,				//	IPV6
	}
	//----- サーバオプション -----------------------------------------------------------------------------------//
	public enum ESsvServOpt
	{
		NOOPT				= 0,			//	オプション無し
		SUP_SLEEP			= 0x01,			//	スリープ抑止
		SUP_DISPOFF			= 0x02,			//	ディスプレィOFF抑止
		SUP_ALL				= 0x03,			//	スリープ＆ディスプレィOFF抑止
	}

	//----------------------------------------------------------------------------------------------------------//
	//	ソケットクライアント 定数																				//
	//----------------------------------------------------------------------------------------------------------//
	//----- ＳＣＴ実行モード -----------------------------------------------------------------------------------//
	public enum ESctMode
	{
		NotificationByEvent	= 0,		//	イベントを通知する
		WaitingForEvent		= 1			//	イベントを待ち受ける
	}
	//----- ＳＣＴチャンクデータの扱い -------------------------------------------------------------------------//
	public enum ESctChunkMode
	{
		BinaryData		= 0x01,			//	バイナリ・チャンクを通知
		TextData		= 0x02,			//	テキスト・チャンクを通知
		Both			= 0x03,			//	両方
	}
	//----- ＳＣＴイベントコード -------------------------------------------------------------------------------//
	public enum ESctEvt
	{
		EV_NOEVENT		= 0,				//	イベント無し

		EV_RXTEXT		= 0x00000001,		//	テキストデータ通知
		EV_RXESC		= 0x00000002,		//	ＥＳＣコードデータ通知
		EV_RXPKT		= 0x00000004,		//	パケットデータ通知
		EV_RXCTRL		= 0x00000008,		//	制御コード通知
		EV_RXNOPKT		= 0x00000010,		//	パケット外テキストデータ

		EV_RXCHUNK		= 0x00000100,		//	チャンクデータ受信通知
		EV_INVCHUNK		= 0x00000200,		//	不正チャンクテキスト受信通知
		EV_TXEMPTY		= 0x00000400,		//	送信完了通知
		EV_CONNECT		= 0x00000800,		//	接続通知
		EV_DISCONNECT	= 0x00001000,		//	切断通知
		EV_CNFAIL		= 0x00002000,		//	接続失敗通知

		EV_RXERR		= 0x00010000,		//	受信エラー通知
		EV_TXERR		= 0x00020000,		//	送信エラー通知
		EV_ERR			= 0x08000000,		//	エラー通知

		EV_SSEP			= (EV_RXTEXT  | EV_RXESC	  | EV_RXCTRL  | EV_RXPKT),
		EV_COMM			= (EV_RXCHUNK | EV_INVCHUNK   | EV_TXEMPTY),
		EV_GENERAL		= (EV_CONNECT | EV_DISCONNECT | EV_CNFAIL),
		EV_ERRS			= (EV_RXERR   | EV_TXERR	  | EV_ERR),
		EV_CLIENT		= (EV_SSEP	  | EV_COMM 	  | EV_CONNECT | EV_DISCONNECT | EV_RXERR | EV_TXERR),
		EV_DEFAULT		= (EV_SSEP	  | EV_COMM 	  | EV_GENERAL | EV_ERRS),
		EV_ALL			= (EV_SSEP	  | EV_COMM 	  | EV_GENERAL | EV_ERRS | EV_RXNOPKT)
	}
	//----- エラーコード ---------------------------------------------------------------------------------------//
	public enum ESctErrorCode {
		ERR_CREEVT		=	 10	,			//	終了通知用イベントオブジェクト生成失敗
		ERR_SOCKET		=	 20	,			//	接続要求待機用ソケット生成失敗(socket)
		ERR_ADDRINFO	=	 30	,			//	アドレス情報取得失敗（getaddrinfo）
		ERR_SOCKOPT		=	 40	,			//	ソケットオプション設定失敗(setsockopt)
		ERR_SUBTHREAD	=	 50	,			//	クライアント通信サブスレッド開始失敗
		ERR_CRESSEP		=	 60	,			//	ストリーム分離インスタンス生成失敗
		ERR_CREMBXTXD	=	 70	,			//	送信データ用メールボックス生成失敗
		ERR_TIMEOUT		=	 80	,			//	サブスレッド終了タイムアウト
	}
	//----- 回線状態 -------------------------------------------------------------------------------------------//
	public enum ESctState {
		DISCONNECT		= 0	,				//	切断状態
		CONNECTING		= 1	,				//	接続中
		CONNECT			= 2	,				//	接続状態
		DISCONNECTING	= 3	,				//	切断中
	}
	//----- 受信テキスト文字コード -----------------------------------------------------------------------------//
	public enum ESctRxTextCode
	{
		SJIS			= 1	,				//	Ｓ－ＪＩＳ
		EUC				= 2	,				//	ＥＵＣ
		UTF8			= 3	,				//	ＵＴＦ－８
		AUTO			= 9	,				//	自動認識
	}
	//----- 送信テキスト文字コード -----------------------------------------------------------------------------//
	public enum ESctTxTextCode
	{
		SJIS			= 1	,				//	Ｓ－ＪＩＳ
		EUC				= 2	,				//	ＥＵＣ
		UTF8			= 3	,				//	ＵＴＦ－８
		AUTO			= 9	,				//	受信テキストコードに合わせる
	}
	//----- アドレスファミリ -----------------------------------------------------------------------------------//
	public enum ESctFamily
	{
		INET			= 0x02,				//	IPV4
		INET6			= 0x17,				//	IPV6
	}


	//----------------------------------------------------------------------------------------------------------//
	//	 文字列プール 定数																						//
	//----------------------------------------------------------------------------------------------------------//
	//----- 文字列比較パラメタ ---------------------------------------------------------------------------------//
	public enum ESplComp {
		RECOGNIZE_WIDTH	= 0,		//	英大小文字を区別する
		IGNORE_WIDTH	= 1,		//	英大小文字を区別しない
		ALPHABETIC		= 2,		//	同一時のみ英大小文字を区別する
	}

	//----- 部分文字列検索パラメタ -----------------------------------------------------------------------------//
	public enum ESplInStr {
		MATCHFIRST		= 1	,		//	先頭部分が文字列と一致
		INCLUDING			,		//	文字列を含む
	}

	//----- 文字列読み出し順 -----------------------------------------------------------------------------------//
	public enum ESplSeq {
		Ascending		= 0,		//	昇順( false )
		Descending		= 1,		//	降順( true	)
	}
	//----------------------------------------------------------------------------------------------------------//
	//	タイムチャート 定数																						//
	//----------------------------------------------------------------------------------------------------------//
	//----- 線の属性 -------------------------------------------------------------------------------------------//
	public enum ETchLineStyle
	{
		Solid		= 0,		//	
		Dash		= 1,		//	-------
		Dot			= 2,		//	.......
		DashDot		= 3,		//	_._._._
		DashDotDot	= 4,		//	_.._.._
		NullLine	= 5,		//	
		InsideFrame = 6			//	
	}
	//----- 波形補間表示種別 -----------------------------------------------------------------------------------//
	public enum ETchIpKind
	{
		Window		= 0,		//	ウインド表示データ
		All 		= 1,		//	全データ
		Latest 		= 2			//	最新データ
	}

	//----------------------------------------------------------------------------------------------------------//
	//	 レジストリアクセス																						//
	//----------------------------------------------------------------------------------------------------------//
	//----- レジストリトップキー（レジストリハイブ）------------------------------------------------------------//
	public enum EAjcHKey : uint
	{
		CLASSES_ROOT				=	(0x80000000),
		CURRENT_USER				=	(0x80000001),
		LOCAL_MACHINE				=	(0x80000002),
		USERS						=	(0x80000003),
		PERFORMANCE_DATA			=	(0x80000004),
		PERFORMANCE_TEXT			=	(0x80000050),
		PERFORMANCE_NLSTEXT 		=	(0x80000060),
		CURRENT_CONFIG				=	(0x80000005),
		DYN_DATA					=	(0x80000006),
		CURRENT_USER_LOCAL_SETTINGS =	(0x80000007),
	}
	//----- レジストリタイプ -----------------------------------------------------------------------------------//
	public enum ERegType
	{
		NONE						= 	0 ,		//	特定の型を持たない値
		SZ							= 	1 ,		//	文字列値
		EXPAND_SZ					= 	2 ,		//	展開可能な文字列値
		BINARY 						= 	3 ,		//	バイナリ値
		DWORD						= 	4 ,		//	３２ビット値（リトルエンディアン）
		DWORD_BIG_ENDIAN			= 	5 ,		//	３２ビット値（ビッグエンディアン）
		LINK						= 	6 ,		//	シンボリックリンク値
		MULTI_SZ					= 	7 ,		//	複数の文字列値
		RESOURCE_LIST			 	= 	8 ,		//	リソースマップ値
		FULL_RESOURCE_DESCRIPTOR 	= 	9 ,		//	ハードウェアリソースリスト値
		RESOURCE_REQUIREMENTS_LIST	=  10 ,		//	ネストされた一連の配列値
		QWORD						=  11 ,		//	６４ビット値（リトルエンディアン）
	}
	//----- プロファイル記録先		 ---------------------------------------------------------------------------//
	public enum EProfileDev
	{
		INIFILE						= 0,		//	INIファイル
		REGISTORY					= 1	,		//	レジストリ
	}
	//----- レジストリ書き込みモード ---------------------------------------------------------------------------//
	public enum ERegRecMode
	{
		NON_VOLATILE				= 0,		//	永続的なレジストリキー生成
		VOLATILE					= 1	,		//	一時的なレジストリキー生成
	}

	//----------------------------------------------------------------------------------------------------------//
	//	 コンソール																								//
	//----------------------------------------------------------------------------------------------------------//
	public enum ECInGetLineOpt {
		NONE		= 0,	//	初期ガイド表示
		F1GUIDE		= 1,	//	やり直しデータ位置表示
		SHOWREDO	= 2,	//	参照テキストデータ位置表示
		SHOWPREV	= 4,	//	入力したテキスト表示
		SHOWTEXT	= 8,	
		ALL			= (F1GUIDE | SHOWREDO | SHOWPREV | SHOWTEXT)
	}

	//----------------------------------------------------------------------------------------------------------//
	//	 ツールチップ																							//
	//----------------------------------------------------------------------------------------------------------//
	//----- マウスカーソル移動 ---------------------------------------------------------------------------------//
	public enum ETipCurMove {
		CENTER		= 0,	//	中心へ移動
		LU			= 1,	//	左上へ移動
		RU			= 2,	//	右上へ移動
		LD			= 3,	//	左下へ移動
		RD			= 4,	//	右下へ移動
	}
	//----------------------------------------------------------------------------------------------------------//
	//	 ウインド項目																							//
	//----------------------------------------------------------------------------------------------------------//
	//----- ドロップ種別 ---------------------------------------------------------------------------------------//
	public enum EDropMode {
		DIR				= 0x0001,			//	ディレクトリ
		FILE			= 0x0002,			//	ファイル
		BOTH			= (DIR | FILE),		//	ディレクトリとファイル
		CONNECT			= 0x0004,			//	全項目をセミコロン(;)で連結
		NOERRTIP		= 0x1000			//	目的のオブジェクト（ディレクトリ／ファイル）がドロップ
											//	されない場合エラーチップを表示しない
	}
	//----------------------------------------------------------------------------------------------------------//
	//	 全コントロールのセーブ／ロード オプション																//
	//----------------------------------------------------------------------------------------------------------//
	//----- セーブオプション -----------------------------------------------------------------------------------//
	public enum ESaveAllOpt : int {
		NoOption		= 0			,		//	指定無し
		ExcReadOnlyText	= 0x0001	,		//	読み出し専用テキストを除外
	}
	//----- ロードオプション -----------------------------------------------------------------------------------//
	public enum ELoadAllOpt : int {
		NoOption		= 0			,		//	指定無し
		RbtClickEvent	= 0x0001	,		//	ラジオボタンでClick()イベント生成する
	}
	//----------------------------------------------------------------------------------------------------------//
	//	 フォルダ構造コピー オプション																			//
	//----------------------------------------------------------------------------------------------------------//
	public enum EFscOpt : int
	{
		NONE			= 0,		//	オプション無し
		SAMETIME		= 0x01,		//	新規作成フォルダのタイムスタンプをコピー元と同じにする
		FORCETIME		= 0x02,		//	新規作成フォルダと既存フォルダのタイムスタンプをコピー元と同じにする
		MAKETOP			= 0x10,		//	転送元末尾フォルダ名をの転送先の先頭フォルダとして作成する
		ALLOWEXIST		= 0x20,		//	同一フォルダが存在してもエラーとしない
	}
	//----------------------------------------------------------------------------------------------------------//
	//	 ファイル群コピー オプション																			//
	//----------------------------------------------------------------------------------------------------------//
	public enum ECpyfOpt : int
	{
		NONE			= 0,		//	オプション無し
		CREATEALWAYS	= 0x01,		//	既にファイルが存在する場合、上書きする
		AJCCPF_WILDEXC	= 0x02,		//	ワイルドカードに一致するファイルを除外する
	}
	//----------------------------------------------------------------------------------------------------------//
	//	 ファイル突合せリスト オプション																		//
	//----------------------------------------------------------------------------------------------------------//
	public enum EFmlOpt : int
	{
		BOTH			= 0,		//	突き合せリストにファイルとディレクトリを含める
		NOFILE			= 0x01,		//	突き合せリストにファイルを含めない
		NODIR			= 0x02,		//	突き合せリストにディレクトリを含めない
	}

	//----------------------------------------------------------------------------------------------------------//
	//	 球面データ生成																							//
	//----------------------------------------------------------------------------------------------------------//
	//----- 球面データ生成パラメタ -----------------------------------------------------------------------------//
	[StructLayout(LayoutKind.Sequential, Pack=1)]
	public struct AJCSPD_PARAM {
		public double			cent_x;				//	球中心(x)
		public double			cent_y;				//	球中心(y)
		public double			cent_z;				//	球中心(z)
		public double			radius;				//	球半径
		public double			noise;				//	半径の誤差（±％）
		public double			xrot;				//	回転軸のＸ軸回転角度・最大値（０～最大値［度］）
		public double			yrot;				//	回転軸のＹ軸回転角度・最大値（０～最大値［度］）
		public double			pitch;				//	プロット点の回転角度・最大値（０～最大値［度］）
	}

	//----------------------------------------------------------------------------------------------------------//
	//	 ２Ｄ／３Ｄグラフィック・コールバックタイプ																//
	//----------------------------------------------------------------------------------------------------------//
	//----- ファイルドロップ通知 -------------------------------------------------------------------------------//
	public delegate void G3dCbkFileDrop(int n);
	public class		 G3dArgFileDrop : EventArgs {public int n;}
	public delegate void G3dEvtFileDrop(object sender, G3dArgFileDrop e);

	//----- フォルダドロップ通知 -------------------------------------------------------------------------------//
	public delegate void G3dCbkDirDrop(int n);
	public class		 G3dArgDirDrop : EventArgs {public int n;}
	public delegate void G3dEvtDirDrop(object sender, G3dArgDirDrop e);

	//----- 右クリック通知 -------------------------------------------------------------------------------------//
	public delegate void G3dCbkRClick(int x, int y, [MarshalAs(UnmanagedType.Bool)] bool shift, [MarshalAs(UnmanagedType.Bool)] bool ctrl);
	public class		 G3dArgRClick : EventArgs {public int x, y; public bool shift, ctrl;}
	public delegate void G3dEvtRClick(object sender, G3dArgRClick e);

	//----- プロットリスト通知 ---------------------------------------------------------------------------------//
	public delegate void G3dCbkPltLst(int max, int num, IntPtr pItem);
	public class		 G3dArgPltLst : EventArgs {public int max, num; public PlotInfo[] p;}
	public delegate void G3dEvtPltLst(object sender, G3dArgPltLst e);

	//----------------------------------------------------------------------------------------------------------//
	//	 棒グラフ・コールバックタイプ																			//
	//----------------------------------------------------------------------------------------------------------//
	//----- レンジ変化通知 -------------------------------------------------------------------------------------//
	public delegate void BarCbkRangeChanged(double low, double high);
	public class		 BarArgRangeChanged : EventArgs { public double low, high;}
	public delegate void BarEvtRangeChanged(object sender, BarArgRangeChanged e);

	//--- スクロール位置変化通知 -------------------------------------------------------------------------------//
	public delegate void BarCbkNtcScrollPos(int pos);
	public class		 BarArgNtcScrollPos : EventArgs { public int pos;}
	public delegate void BarEvtNtcScrollPos(object sender, BarArgNtcScrollPos e);

	//----- ファイルドロップ通知 -------------------------------------------------------------------------------//
	public delegate void BarCbkFileDrop(int n);
	public class		 BarArgFileDrop : EventArgs {public int n;}
	public delegate void BarEvtFileDrop(object sender, BarArgFileDrop e);

	//----- フォルダドロップ通知 -------------------------------------------------------------------------------//
	public delegate void BarCbkDirDrop(int n);
	public class		 BarArgDirDrop : EventArgs {public int n;}
	public delegate void BarEvtDirDrop(object sender, BarArgDirDrop e);

	//----- 右クリック通知 -------------------------------------------------------------------------------------//
	public delegate void BarCbkRClick(int x, int y, [MarshalAs(UnmanagedType.Bool)] bool shift, [MarshalAs(UnmanagedType.Bool)] bool ctrl);
	public class		 BarArgRClick : EventArgs {public int x, y; public bool shift, ctrl;}
	public delegate void BarEvtRClick(object sender, BarArgRClick e);

	//----------------------------------------------------------------------------------------------------------//
	//	 Ｃ言語プリコンパイル コールバックタイプ																//
	//----------------------------------------------------------------------------------------------------------//
	//----- いずれかのイベント発生通知 -------------------------------------------------------------------------//
	public delegate void   PpcCbkNtcAnyEvt(EPpcNtc EvtId);
	public class		   PpcArgNtcAnyEvt : EventArgs {public EPpcNtc EvtId;}
	public delegate void   PpcEvtNtcAnyEvt(object sender, PpcArgNtcAnyEvt e);

	//----- 現在処理中のファイル名，行番号通知 -----------------------------------------------------------------//
	public delegate void   PpcCbkNtcFileLno([MarshalAs(UnmanagedType.LPWStr)]string FileName, int lno, int nest);
	public class		   PpcArgNtcFileLno : EventArgs {public string FileName; public int lno; public int nest;}
	public delegate void   PpcEvtNtcFileLno(object sender, PpcArgNtcFileLno e);

	//----- インクルードファイル検索開始通知	-----------------------------------------------------------------//
	public delegate void   PpcCbkNtcSrhStart([MarshalAs(UnmanagedType.LPWStr)]string IncName, [MarshalAs(UnmanagedType.LPWStr)]string TopPath);
	public class		   PpcArgNtcSrhStart : EventArgs {public string IncName; public string TopPath;}
	public delegate void   PpcEvtNtcSrhStart(object sender, PpcArgNtcSrhStart e);

	//----- インクルードファイル検索フォルダ通知 ----------------------------------------------------------------//
	public delegate void   PpcCbkNtcSrhDir([MarshalAs(UnmanagedType.LPWStr)]string IncName, [MarshalAs(UnmanagedType.LPWStr)]string DirPath);
	public class		   PpcArgNtcSrhDir : EventArgs {public string IncName; public string DirPath;}
	public delegate void   PpcEvtNtcSrhDir(object sender, PpcArgNtcSrhDir e);

	//----- インクルードファイル検索終了通知 	-----------------------------------------------------------------//
	public delegate void   PpcCbkNtcSrhEnd([MarshalAs(UnmanagedType.LPWStr)]string IncName, [MarshalAs(UnmanagedType.Bool)]bool fFind);
	public class		   PpcArgNtcSrhEnd : EventArgs {public string IncName; public bool fFind;}
	public delegate void   PpcEvtNtcSrhEnd(object sender, PpcArgNtcSrhEnd e);

	//----- プリプロセス用オプションシンボル通知 ---------------------------------------------------------------//
	public delegate void   PpcCbkNtcOptSym([MarshalAs(UnmanagedType.LPWStr)]string FilePath, int lno, [MarshalAs(UnmanagedType.LPWStr)]string OptSym);
	public class		   PpcArgNtcOptSym : EventArgs {public string FilePath; public int lno; public string OptSym;}
	public delegate void   PpcEvtNtcOptSym(object sender, PpcArgNtcOptSym e);

	//----- マクロ定義通知 -------------------------------------------------------------------------------------//
	public delegate void   PpcCbkNtcMacDef([MarshalAs(UnmanagedType.LPWStr)]string FilePath, int lno, [MarshalAs(UnmanagedType.LPWStr)]string MacName);
	public class		   PpcArgNtcMacDef : EventArgs {public string FilePath; public int lno; public string MacName;}
	public delegate void   PpcEvtNtcMacDef(object sender, PpcArgNtcMacDef e);

	//----- マクロ参照通知 -------------------------------------------------------------------------------------//
	public delegate void   PpcCbkNtcMacRef([MarshalAs(UnmanagedType.LPWStr)]string FilePath, int lno, [MarshalAs(UnmanagedType.LPWStr)]string MacName);
	public class		   PpcArgNtcMacRef : EventArgs {public string FilePath; public int lno; public string MacName;}
	public delegate void   PpcEvtNtcMacRef(object sender, PpcArgNtcMacRef e);

	//----- トークンストリームをファイルへ出力中通知 -----------------------------------------------------------//
	public delegate void   PpcCbkNtcOutput();
	public class		   PpcArgNtcOutput : EventArgs {public int dummy;}
	public delegate void   PpcEvtNtcOutput(object sender, PpcArgNtcOutput e);

	//----- エラー通知 -----------------------------------------------------------------------------------------//
	public delegate void   PpcCbkNtcError(EPpcErr err, [MarshalAs(UnmanagedType.LPWStr)]string FilePath, int lno, [MarshalAs(UnmanagedType.LPWStr)]string param);
	public class		   PpcArgNtcError : EventArgs {public EPpcErr err; public string FilePath; public int lno; public string param;}
	public delegate void   PpcEvtNtcError(object sender, PpcArgNtcError e);

	//----- ソースファイルエンコード通知 -----------------------------------------------------------------------//
	public delegate void   PpcCbkNtcSrcTec([MarshalAs(UnmanagedType.LPWStr)]string FilePath, ETextEncode tec, EBomMode bom);
	public class		   PpcArgNtcSrcTec : EventArgs {public string FilePath; public ETextEncode tec; public EBomMode bom;}
	public delegate void   PpcEvtNtcSrcTec(object sender, PpcArgNtcSrcTec e);

	//----------------------------------------------------------------------------------------------------------//
	//	 Ｃ言語字句分解 コールバックタイプ																		//
	//----------------------------------------------------------------------------------------------------------//
	//----- １行読み出し ---------------------------------------------------------------------------------------//
	public delegate bool   CtkCbkGetS(IntPtr pBuf, int lBuf, IntPtr cbp);
	public class		   CtkArgGetS : EventArgs {public IntPtr pBuf; public int lBuf; public IntPtr cbp;}
	public delegate bool   CtkEvtGetS(object sender, CtkArgGetS e);

	//----------------------------------------------------------------------------------------------------------//
	//	 ファイル検索 コールバックタイプ																		//
	//----------------------------------------------------------------------------------------------------------//
	//----- ファイル／マイコンピュータ検索 ---------------------------------------------------------------------//
	[return: MarshalAs(UnmanagedType.Bool)]
	public delegate bool FsrCbkFindFile(int nest, [MarshalAs(UnmanagedType.LPWStr)]string PathName,
										   [MarshalAs(UnmanagedType.LPWStr)]string FileName, EFileAtt FileAtt, uint FTime, IntPtr cbp);
	public class FsrArgFindFile : EventArgs {
				public int		nest;
				public string	PathName;
				public string	FileName;
				public EFileAtt FileAtt;
				public uint		FTime;
				public IntPtr	cbp;
	}
	public delegate bool FsrEvtFindFile(object sender, FsrArgFindFile e);

	//----------------------------------------------------------------------------------------------------------//
	//	 ファイル／フォルダ操作 コールバックタイプ																//
	//----------------------------------------------------------------------------------------------------------//
	//----- フォルダコピー -------------------------------------------------------------------------------------//
	//	フォルダコピー結果通知
	[return: MarshalAs(UnmanagedType.Bool)]
	public delegate bool FopCbkScpNtc([MarshalAs(UnmanagedType.LPWStr)]string DirFrom, [MarshalAs(UnmanagedType.LPWStr)]string DirTo, ENtcDirCopy ntc, IntPtr cbp);
	//	フォルダコピー可否／名称変更問い合わせ
	[return: MarshalAs(UnmanagedType.LPWStr)]
	public delegate string FopCbkScpQry([MarshalAs(UnmanagedType.LPWStr)]string DirFront, [MarshalAs(UnmanagedType.LPWStr)]string DirNew, IntPtr cbp);

	//----- ファイルコピー -------------------------------------------------------------------------------------//
	//	ファイルコピー結果通知
	[return: MarshalAs(UnmanagedType.Bool)]
	public delegate bool FopCbkFcpNtc([MarshalAs(UnmanagedType.LPWStr)]string FileFrom, [MarshalAs(UnmanagedType.LPWStr)]string FileTo, ENtcFileCopy ntc, IntPtr cbp);
	//	ファイルコピー可否／名称変更問い合わせ
	[return: MarshalAs(UnmanagedType.LPWStr)]
	public delegate string FopCbkFcpQry([MarshalAs(UnmanagedType.LPWStr)]string FileFrom, [MarshalAs(UnmanagedType.LPWStr)]string FileTo,
										[MarshalAs(UnmanagedType.LPWStr)]string FileName, ref EFileAtt att, IntPtr cbp);
	//	ファイル進行状況通知
	[return: MarshalAs(UnmanagedType.LPWStr)]
	public delegate string FopCbkFcpPgs(long FileSize, long Copied, IntPtr cbp);

	//----- フォルダ／ファイル削除 -----------------------------------------------------------------------------//
	[return: MarshalAs(UnmanagedType.Bool)]
	public delegate bool FopCbkRmvDir([MarshalAs(UnmanagedType.LPWStr)]string PathName, uint ntc, IntPtr cbp);

	//----- 突合せリスト列挙 -----------------------------------------------------------------------------------//
	//	ファイル通知
	[return: MarshalAs(UnmanagedType.Bool)]
	public delegate bool FopCbkFmlQuery(int id, [MarshalAs(UnmanagedType.LPWStr)]string path, System.IO.FileAttributes att, [MarshalAs(UnmanagedType.Bool)] ref	bool fDiscard, IntPtr cbp);
	//	列挙通知
	[return: MarshalAs(UnmanagedType.Bool)]
	public delegate bool FopCbkFmlEnum ([MarshalAs(UnmanagedType.Bool)] bool fValid1, uint utc1, System.IO.FileAttributes att1, long size1, 
											[MarshalAs(UnmanagedType.LPWStr)]string path1, [MarshalAs(UnmanagedType.LPWStr)]string name1, [MarshalAs(UnmanagedType.LPWStr)]string tail1, 
										[MarshalAs(UnmanagedType.Bool)] bool fValid2, uint utc2, System.IO.FileAttributes att2, long size2,
											[MarshalAs(UnmanagedType.LPWStr)]string path2, [MarshalAs(UnmanagedType.LPWStr)]string name2, [MarshalAs(UnmanagedType.LPWStr)]string tail2, 
										IntPtr cbp);

	//----------------------------------------------------------------------------------------------------------//
	//	 ネイティブアクセス コールバックタイプ																	//
	//----------------------------------------------------------------------------------------------------------//
	//----- メモリエラー ---------------------------------------------------------------------------------------//
	public delegate void BinCbkOutOfMemory([MarshalAs(UnmanagedType.LPWStr)]string FileName, int lno, [MarshalAs(UnmanagedType.LPWStr)] string Version, IntPtr cbp);

	//----------------------------------------------------------------------------------------------------------//
	//	 数値入力・コールバックタイプ																			//
	//----------------------------------------------------------------------------------------------------------//
	//----- 整数値変化通知 -------------------------------------------------------------------------------------//
	public delegate void IvCbkNtcIntValue(int value);
	public class		 IvArgNtcIntValue : EventArgs { public int value;}
	public delegate void IvEvtNtcIntValue(object sender, IvArgNtcIntValue e);

	//----- 実数値変化通知 -------------------------------------------------------------------------------------//
	public delegate void IvCbkNtcRealValue(double value);
	public class		 IvArgNtcRealValue : EventArgs { public double value;}
	public delegate void IvEvtNtcRealValue(object sender, IvArgNtcRealValue e);

	//----- 右クリック通知 -------------------------------------------------------------------------------------//
	public delegate void IvCbkNtcRClick(int x, int y, [MarshalAs(UnmanagedType.Bool)] bool shift, [MarshalAs(UnmanagedType.Bool)] bool ctrl);
	public class		 IvArgNtcRClick : EventArgs { public int x, y; public bool shift, ctrl;}
	public delegate void IvEvtNtcRClick(object sender, IvArgNtcRClick e);

	//----------------------------------------------------------------------------------------------------------//
	//	 シリアル通信・コールバックタイプ																		//
	//----------------------------------------------------------------------------------------------------------//
	//----- ポート状態通知 -------------------------------------------------------------------------------------//
	public delegate void ScpCbkPortState(EScpPortState state, [MarshalAs(UnmanagedType.LPWStr)]string name);
	public class		 ScpArgPortState : EventArgs { public EScpPortState state; public string name;}
	public delegate void ScpEvtPortState(object sender, ScpArgPortState e);

	//----- チャンクテキスト受信通知 ---------------------------------------------------------------------------//
	public delegate void ScpCbkRxChunkTxt([MarshalAs(UnmanagedType.LPWStr)]string chunk);
	public class		 ScpArgRxChunkTxt : EventArgs { public string text;}
	public delegate void ScpEvtRxChunkTxt(object sender, ScpArgRxChunkTxt e);

	//----- チャンクバイナリ受信通知 ---------------------------------------------------------------------------//
	unsafe public delegate void ScpCbkRxChunkBin(void *p, int len);
	unsafe public class 		ScpArgRxChunkBin : EventArgs {public Byte[] bin;}
	unsafe public delegate void ScpEvtRxChunkBin(object sender, ScpArgRxChunkBin e);

	//----- テキスト受信通知 -----------------------------------------------------------------------------------//
	public delegate void ScpCbkRxText([MarshalAs(UnmanagedType.LPWStr)]string text);
	public class		 ScpArgRxText : EventArgs { public string text;}
	public delegate void ScpEvtRxText(object sender, ScpArgRxText e);

	//----- ＥＳＣ受信通知 -------------------------------------------------------------------------------------//
	public delegate void ScpCbkRxEsc([MarshalAs(UnmanagedType.LPWStr)]string esc);
	public class		 ScpArgRxEsc : EventArgs { public string esc;}
	public delegate void ScpEvtRxEsc(object sender, ScpArgRxEsc e);

	//----- 制御コード受信通知 ---------------------------------------------------------------------------------//
	public delegate void ScpCbkRxCtrl(char ctrl);
	public class		 ScpArgRxCtrl : EventArgs { public char ctrl;}
	public delegate void ScpEvtRxCtrl(object sender, ScpArgRxCtrl e);

	//----- パケット受信通知 -----------------------------------------------------------------------------------//
	unsafe public delegate void ScpCbkRxPacket(void *p, int len);
	unsafe public class 		ScpArgRxPacket : EventArgs {public Byte[] bin;}
	unsafe public delegate void ScpEvtRxPacket(object sender, ScpArgRxPacket e);

	//----- 送信完了通知 ---------------------------------------------------------------------------------------//
	public delegate void ScpCbkTxEmpty();
	public delegate void ScpEvtTxEmpty(object sender, EventArgs e);

	//----- パケット外データ受信通知 ---------------------------------------------------------------------------//
	public delegate void ScpCbkRxNoPkt([MarshalAs(UnmanagedType.LPWStr)]string text);
	public class		 ScpArgRxNoPkt: EventArgs {public string text;}
	public delegate void ScpEvtRxNoPkt(object sender, ScpArgRxNoPkt e);

	//----- バイトペアによるワード(14Bit値)受信通知 ------------------------------------------------------------//
	public delegate void ScpCbkRxWord14(int data);
	public class		 ScpArgRxWord14: EventArgs {public int data;}
	public delegate void ScpEvtRxWord14(object sender, ScpArgRxWord14 e);

	//----- 不正チャンクテキスト受信通知 ----------------------------------------------------------------------//
	unsafe public delegate void ScpCbkRxInvChunk(void *p, int len);
	unsafe public class 		ScpArgRxInvChunk : EventArgs {public Byte[] bin;}
	unsafe public delegate void ScpEvtRxInvChunk(object sender, ScpArgRxInvChunk e);

	//----- エラー通知 -----------------------------------------------------------------------------------------//
	public delegate void ScpCbkError(int err);
	public class		 ScpArgError : EventArgs {public EScpErr err;}
	public delegate void ScpEvtError(object sender, ScpArgError e);

	//----- ブレーク検出通知 -----------------------------------------------------------------------------------//
	public delegate void ScpCbkNtcBreak();
	public delegate void ScpEvtNtcBreak(object sender, EventArgs e);

	//----- RING検出通知 ---------------------------------------------------------------------------------------//
	public delegate void ScpCbkNtcRING(bool cts, bool dsr, bool rlsd, bool ring);
	public class		 ScpArgNtcRING : EventArgs {public bool cts, dsr, rlsd, ring;}
	public delegate void ScpEvtNtcRING(object sender, ScpArgNtcRING e);

	//----- RLSD検出通知 ---------------------------------------------------------------------------------------//
	public delegate void ScpCbkNtcRLSD(bool cts, bool dsr, bool rlsd, bool ring);
	public class		 ScpArgNtcRLSD : EventArgs {public bool cts, dsr, rlsd, ring;}
	public delegate void ScpEvtNtcRLSD(object sender, ScpArgNtcRLSD e);

	//----- DSR検出通知 ----------------------------------------------------------------------------------------//
	public delegate void ScpCbkNtcDSR(bool cts, bool dsr, bool rlsd, bool ring);
	public class		 ScpArgNtcDSR : EventArgs {public bool cts, dsr, rlsd, ring;}
	public delegate void ScpEvtNtcDSR(object sender, ScpArgNtcDSR e);

	//----- CTS検出通知 ----------------------------------------------------------------------------------------//
	public delegate void ScpCbkNtcCTS(bool cts, bool dsr, bool rlsd, bool ring);
	public class		 ScpArgNtcCTS : EventArgs {public bool cts, dsr, rlsd, ring;}
	public delegate void ScpEvtNtcCTS(object sender, ScpArgNtcCTS e);

	//----------------------------------------------------------------------------------------------------------//
	//	 ソケットサーバ・コールバックタイプ																		//
	//----------------------------------------------------------------------------------------------------------//
	//----- テキスト受信通知 -----------------------------------------------------------------------------------//
	public delegate void SsvCbkRxText(IntPtr hClient, [MarshalAs(UnmanagedType.LPWStr)]string text);
	public class		 SsvArgRxText : EventArgs	{public IntPtr hClient; public string text;}
	public delegate void SsvEvtRxText(object sender, SsvArgRxText e);

	//----- ＥＳＣ受信通知 -------------------------------------------------------------------------------------//
	public delegate void SsvCbkRxEsc(IntPtr hClient, [MarshalAs(UnmanagedType.LPWStr)]string esc);
	public class		 SsvArgRxEsc : EventArgs 	{public IntPtr hClient; public string esc;}
	public delegate void SsvEvtRxEsc(object sender, SsvArgRxEsc e);

	//----- 制御コード受信通知 ---------------------------------------------------------------------------------//
	public delegate void SsvCbkRxCtrl(IntPtr hClient, char ctrl);
	public class		 SsvArgRxCtrl : EventArgs 	{public IntPtr hClient; public char ctrl;}
	public delegate void SsvEvtRxCtrl(object sender, SsvArgRxCtrl e);

	//----- パケット受信通知 -----------------------------------------------------------------------------------//
	unsafe public delegate void SsvCbkRxPacket(IntPtr hClient, void *p, int len);
	unsafe public class SsvArgRxPacket : EventArgs	{public IntPtr hClient; public Byte[] bin;}
	unsafe public delegate void SsvEvtRxPacket(object sender, SsvArgRxPacket e);

	//----- パケット外テキスト受信通知 -------------------------------------------------------------------------//
	public delegate void SsvCbkRxNoPkt(IntPtr hClient, [MarshalAs(UnmanagedType.LPWStr)]string text);
	public class	 SsvArgRxNoPkt: EventArgs	{public IntPtr hClient; public string text;}
	public delegate void SsvEvtRxNoPkt(object sender, SsvArgRxNoPkt e);

	//----- 送信完了通知 ---------------------------------------------------------------------------------------//
	public delegate void SsvCbkTxEmpty(IntPtr hClient);
	public class		SsvArgTxEmpty: EventArgs	{public IntPtr hClient;}
	public delegate void SsvEvtTxEmpty(object sender, SsvArgTxEmpty e);

	//----- チャンクテキスト受信通知 ---------------------------------------------------------------------------//
	public delegate void SsvCbkRxChunkTxt(IntPtr hClient, [MarshalAs(UnmanagedType.LPWStr)]string chunk);
	public class		 SsvArgRxChunkTxt : EventArgs {public IntPtr hClient; public string text;}
	public delegate void SsvEvtRxChunkTxt(object sender, SsvArgRxChunkTxt e);

	//----- チャンクバイナリ受信通知 ---------------------------------------------------------------------------//
	unsafe public delegate void SsvCbkRxChunkBin(IntPtr hClient, void *p, int len);
	unsafe public class 		SsvArgRxChunkBin : EventArgs {public IntPtr hClient; public Byte[] bin;}
	unsafe public delegate void SsvEvtRxChunkBin(object sender, SsvArgRxChunkBin e);

	//----- 不正チャンクテキスト受信通知 ----------------------------------------------------------------------//
	unsafe public delegate void SsvCbkRxInvChunk(IntPtr hClient, void *p, int len);
	unsafe public class 		SsvArgRxInvChunk : EventArgs {public IntPtr hClient; public Byte[] bin;}
	unsafe public delegate void SsvEvtRxInvChunk(object sender, SsvArgRxInvChunk e);

	//----- 接続通知--------------------------------------------------------------------------------------------//
	public delegate void SsvCbkConnect(IntPtr hClient);
	public class		 SsvArgConnect : EventArgs	  {public IntPtr hClient;}
	public delegate void SsvEvtConnect(object sender, SsvArgConnect e);

	//----- 切断通知--------------------------------------------------------------------------------------------//
	public delegate void SsvCbkDisconnect(IntPtr hClient);
	public class		 SsvArgDisconnect : EventArgs {public IntPtr hClient;}
	public delegate void SsvEvtDisconnect(object sender, SsvArgDisconnect e);

	//----- サーバ開始通知--------------------------------------------------------------------------------------//
	public delegate void SsvCbkStart();
	public class		 SsvArgStart : EventArgs {public IntPtr dummy;}
	public delegate void SsvEvtStart(object sender, SsvArgStart e);

	//----- サーバ停止通知--------------------------------------------------------------------------------------//
	public delegate void SsvCbkStop();
	public class		 SsvArgStop : EventArgs {public IntPtr dummy;}
	public delegate void SsvEvtStop(object sender, SsvArgStop e);

	//----- 受信エラー通知 -------------------------------------------------------------------------------------//
	public delegate void SsvCbkRecvError(IntPtr hClient, [MarshalAs(UnmanagedType.Bool)]bool overlapped);
	public class		 SsvArgRecvError : EventArgs 			{public IntPtr hClient; public bool overlapped;}
	public delegate void SsvEvtRecvError(object sender, SsvArgRecvError e);

	//----- 送信エラー通知 -------------------------------------------------------------------------------------//
	public delegate void SsvCbkSendError(IntPtr hClient, [MarshalAs(UnmanagedType.Bool)]bool overlapped);
	public class		 SsvArgSendError : EventArgs 			{public IntPtr hClient; public bool overlapped;}
	public delegate void SsvEvtSendError(object sender, SsvArgSendError e);

	//----- その他のエラー通知 ---------------------------------------------------------------------------------//
	public delegate void SsvCbkGeneralError(ESsvErrorCode ErrorCode);
	public class		 SsvArgGeneralError : EventArgs	{public ESsvErrorCode ErrorCode;}
	public delegate void SsvEvtGeneralError(object sender, SsvArgGeneralError e);

	//----- クライアント列挙通知 -------------------------------------------------------------------------------//
	public delegate bool SsvCbkEnumClients(IntPtr pC, IntPtr cbp);
	public class		 SsvArgEnumClients : EventArgs	{public IntPtr hClient; public IntPtr param;}
	public delegate bool SsvEvtEnumClients(object sender, SsvArgEnumClients e);

	//----------------------------------------------------------------------------------------------------------//
	//	 ソケットクライアント・コールバックタイプ																//
	//----------------------------------------------------------------------------------------------------------//
	//----- テキスト受信通知 -----------------------------------------------------------------------------------//
	public delegate void SctCbkRxText([MarshalAs(UnmanagedType.LPWStr)]string text);
	public class		 SctArgRxText : EventArgs	{public string text;}
	public delegate void SctEvtRxText(object sender, SctArgRxText e);

	//----- ＥＳＣ受信通知 -------------------------------------------------------------------------------------//
	public delegate void SctCbkRxEsc([MarshalAs(UnmanagedType.LPWStr)]string esc);
	public class		 SctArgRxEsc : EventArgs 	{public string esc;}
	public delegate void SctEvtRxEsc(object sender, SctArgRxEsc e);

	//----- 制御コード受信通知 ---------------------------------------------------------------------------------//
	public delegate void SctCbkRxCtrl(char ctrl);
	public class		 SctArgRxCtrl : EventArgs 	{public char ctrl;}
	public delegate void SctEvtRxCtrl(object sender, SctArgRxCtrl e);

	//----- パケット受信通知 -----------------------------------------------------------------------------------//
	unsafe public delegate void SctCbkRxPacket(void *p, int len);
	unsafe public class SctArgRxPacket : EventArgs	{public Byte[] bin;}
	unsafe public delegate void SctEvtRxPacket(object sender, SctArgRxPacket e);

	//----- パケット外テキスト受信通知 -------------------------------------------------------------------------//
	public delegate void SctCbkRxNoPkt([MarshalAs(UnmanagedType.LPWStr)]string text);
	public class	 SctArgRxNoPkt: EventArgs	{public string text;}
	public delegate void SctEvtRxNoPkt(object sender, SctArgRxNoPkt e);

	//----- 送信完了通知 ---------------------------------------------------------------------------------------//
	public delegate void SctCbkTxEmpty();
	public delegate void SctEvtTxEmpty(object sender, EventArgs e);

	//----- チャンクテキスト受信通知 ---------------------------------------------------------------------------//
	public delegate void SctCbkRxChunkTxt([MarshalAs(UnmanagedType.LPWStr)]string chunk);
	public class		 SctArgRxChunkTxt : EventArgs {public string text;}
	public delegate void SctEvtRxChunkTxt(object sender, SctArgRxChunkTxt e);

	//----- チャンクバイナリ受信通知 ---------------------------------------------------------------------------//
	unsafe public delegate void SctCbkRxChunkBin(void *p, int len);
	unsafe public class 		SctArgRxChunkBin : EventArgs {public Byte[] bin;}
	unsafe public delegate void SctEvtRxChunkBin(object sender, SctArgRxChunkBin e);

	//----- 不正チャンクテキスト受信通知 ----------------------------------------------------------------------//
	unsafe public delegate void SctCbkRxInvChunk(void *p, int len);
	unsafe public class 		SctArgRxInvChunk : EventArgs {public Byte[] bin;}
	unsafe public delegate void SctEvtRxInvChunk(object sender, SctArgRxInvChunk e);

	//----- 接続通知--------------------------------------------------------------------------------------------//
	public delegate void SctCbkConnect();
	public delegate void SctEvtConnect(object sender, EventArgs e);

	//----- 切断通知--------------------------------------------------------------------------------------------//
	public delegate void SctCbkDisconnect();
	public delegate void SctEvtDisconnect(object sender, EventArgs e);

	//----- 接続失敗通知 ---------------------------------------------------------------------------------------//
	public delegate void SctCbkCnFail(int err);
	public class		 SctArgCnFail : EventArgs 			{public int err;}
	public delegate void SctEvtCnFail(object sender, SctArgCnFail e);

	//----- 受信エラー通知 -------------------------------------------------------------------------------------//
	public delegate void SctCbkRecvError([MarshalAs(UnmanagedType.Bool)]bool overlapped);
	public class		 SctArgRecvError : EventArgs 			{public bool overlapped;}
	public delegate void SctEvtRecvError(object sender, SctArgRecvError e);

	//----- 送信エラー通知 -------------------------------------------------------------------------------------//
	public delegate void SctCbkSendError([MarshalAs(UnmanagedType.Bool)]bool overlapped);
	public class		 SctArgSendError : EventArgs 			{public bool overlapped;}
	public delegate void SctEvtSendError(object sender, SctArgSendError e);

	//----- その他のエラー通知 ---------------------------------------------------------------------------------//
	public delegate void SctCbkGeneralError(ESctErrorCode ErrorCode);
	public class		 SctArgGeneralError : EventArgs	{public ESctErrorCode ErrorCode;}
	public delegate void SctEvtGeneralError(object sender, SctArgGeneralError e);

	//----------------------------------------------------------------------------------------------------------//
	//	 文字列プール コールバックタイプ																		//
	//----------------------------------------------------------------------------------------------------------//
	//----- 文字列通知 -----------------------------------------------------------------------------------------//
	public delegate bool SplCbkNtcStr([MarshalAs(UnmanagedType.LPWStr)]string str, IntPtr cbp);
	public class		 SplArgNtcStr : EventArgs { public string str; public IntPtr cbp;}
	public delegate bool SplEvtNtcStr(object sender, SplArgNtcStr e);

	//----------------------------------------------------------------------------------------------------------//
	//	 タイムチャート・コールバックタイプ																		//
	//----------------------------------------------------------------------------------------------------------//
	//----- レンジ変化通知 -------------------------------------------------------------------------------------//
	public delegate void TchCbkRangeChanged(double low, double high);
	public class		 TchArgRangeChanged : EventArgs { public double low, high;}
	public delegate void TchEvtRangeChanged(object sender, TchArgRangeChanged e);

	//--- スクロール位置変化通知 -------------------------------------------------------------------------------//
	public delegate void TchCbkNtcScrollPos(int pos);
	public class		 TchArgNtcScrollPos : EventArgs { public int pos;}
	public delegate void TchEvtNtcScrollPos(object sender, TchArgNtcScrollPos e);

	//----- ファイルドロップ通知 -------------------------------------------------------------------------------//
	public delegate void TchCbkFileDrop(int n);
	public class		 TchArgFileDrop : EventArgs {public int n;}
	public delegate void TchEvtFileDrop(object sender, TchArgFileDrop e);

	//----- フォルダドロップ通知 -------------------------------------------------------------------------------//
	public delegate void TchCbkDirDrop(int n);
	public class		 TchArgDirDrop : EventArgs {public int n;}
	public delegate void TchEvtDirDrop(object sender, TchArgDirDrop e);

	//----- 右クリック通知 -------------------------------------------------------------------------------------//
	public delegate void TchCbkRClick(int x, int y, [MarshalAs(UnmanagedType.Bool)] bool shift, [MarshalAs(UnmanagedType.Bool)] bool ctrl);
	public class		 TchArgRClick : EventArgs {public int x, y; public bool shift, ctrl;}
	public delegate void TchEvtRClick(object sender, TchArgRClick e);

	//----------------------------------------------------------------------------------------------------------//
	//	 ＶＴ１００コールバックタイプ																			//
	//----------------------------------------------------------------------------------------------------------//
	//----- ダブルクリック通知 ---------------------------------------------------------------------------------//
	public delegate void VthCbkNtcDblClk([MarshalAs(UnmanagedType.Bool)] bool shift, [MarshalAs(UnmanagedType.Bool)] bool ctrl);
	public class		 VthArgDblClk : EventArgs {public bool shift, ctrl;}
	public delegate void VthEvtNtcDblClk(object sender, VthArgDblClk e);

	//----- キー入力通知 ---------------------------------------------------------------------------------------//
	public delegate void VthCbkNtcKeyIn(int key, int rep);
	public class		 VthArgNtcKeyIn : EventArgs {public int key; public int rep;}
	public delegate void VthEvtNtcKeyIn(object sender, VthArgNtcKeyIn e);

	//----- 拡張キー押下通知 -----------------------------------------------------------------------------------//
	public delegate void VthCbkNtcVKeyIn(Keys key, int rep);
	public class		 VthArgNtcVKeyIn : EventArgs {public Keys vkey; public int rep;}
	public delegate void VthEvtNtcVKeyIn(object sender, VthArgNtcVKeyIn e);

	//----- 拡張キー離し通知 -----------------------------------------------------------------------------------//
	public delegate void VthCbkNtcVKeyOut(Keys key);
	public class		 VthArgNtcVKeyOut : EventArgs {public Keys vkey;}
	public delegate void VthEvtNtcVKeyOut(object sender, VthArgNtcVKeyOut e);

	//----- ファイルドロップ通知 -------------------------------------------------------------------------------//
	public delegate void VthCbkFileDrop(int n);
	public class		 VthArgFileDrop : EventArgs {public int n;}
	public delegate void VthEvtFileDrop(object sender, VthArgFileDrop e);

	//----- フォルダドロップ通知 -------------------------------------------------------------------------------//
	public delegate void VthCbkDirDrop(int n);
	public class		 VthArgDirDrop : EventArgs {public int n;}
	public delegate void VthEvtDirDrop(object sender, VthArgDirDrop e);

	//----- 文字サイズ変化通知 ---------------------------------------------------------------------------------//
	public delegate void VthCbkCharInfo(int LineHeight);
	public class		 VthArgCharInfo : EventArgs {public int LineHeight;}
	public delegate void VthEvtCharInfo(object sender, VthArgCharInfo e);

	//----- 右クリック通知 -------------------------------------------------------------------------------------//
	public delegate void VthCbkRClick(int x, int y, [MarshalAs(UnmanagedType.Bool)] bool shift, [MarshalAs(UnmanagedType.Bool)] bool ctrl);
	public class		 VthArgRClick : EventArgs {public int x, y; public bool shift, ctrl;}
	public delegate void VthEvtRClick(object sender, VthArgRClick e);

	//----------------------------------------------------------------------------------------------------------//
	//	 ＴｉｐＣｔｒｌコールバックタイプ																		//
	//----------------------------------------------------------------------------------------------------------//
	//----- チップテキスト取得要求 -----------------------------------------------------------------------------//
	[return: MarshalAs(UnmanagedType.LPWStr)]
	public delegate string			TipCbkNeedText(IntPtr hwnd, IntPtr cbp);

	//----- マウスカーソル位置の判定要求 -----------------------------------------------------------------------//
	[return: MarshalAs(UnmanagedType.Bool)]
	public delegate bool			TipCbkChkPoint(IntPtr hwnd, ref AJCPOINT ptClient, IntPtr cbp);

	//----------------------------------------------------------------------------------------------------------//
	//	 コンソール コールバックタイプ																			//
	//----------------------------------------------------------------------------------------------------------//
	//----- 行テキスト通知（内部文字列）------------------------------------------------------------------------//
	public delegate void	ConCbkNtcStr(int argc, [MarshalAs(UnmanagedType.LPWStr)]string argv, ConCbkNtcArgs cbp);

	//----- 行テキスト通知（引数リスト）------------------------------------------------------------------------//
	public delegate void	ConCbkNtcArgs(int argc, string[] argv);

	//----------------------------------------------------------------------------------------------------------//
	//	例外																									//
	//----------------------------------------------------------------------------------------------------------//

	//	ファイルオープンエラー
	[Serializable()]
	public class FileOpenFailureException : System.Exception
	{
		public	  FileOpenFailureException() : base() { }
		public	  FileOpenFailureException(string message) : base(message) { }
		public	  FileOpenFailureException(string message, System.Exception inner) : base(message, inner) { }
		protected FileOpenFailureException(System.Runtime.Serialization.SerializationInfo info,
						System.Runtime.Serialization.StreamingContext context) : base(info, context) { }
	}
	//	ファイル生成エラー
	[Serializable()]
	public class FileCreationFailureException : System.Exception
	{
		public	  FileCreationFailureException() : base() { }
		public	  FileCreationFailureException(string message) : base(message) { }
		public	  FileCreationFailureException(string message, System.Exception inner) : base(message, inner) { }
		protected FileCreationFailureException(System.Runtime.Serialization.SerializationInfo info,
						System.Runtime.Serialization.StreamingContext context) : base(info, context) { }
	}
	//	ファイルクローズエラー
	[Serializable()]
	public class FileCloseFailureException : System.Exception
	{
		public	  FileCloseFailureException() : base() { }
		public	  FileCloseFailureException(string message) : base(message) { }
		public	  FileCloseFailureException(string message, System.Exception inner) : base(message, inner) { }
		protected FileCloseFailureException(System.Runtime.Serialization.SerializationInfo info,
						System.Runtime.Serialization.StreamingContext context) : base(info, context) { }
	}
	//	ファイルアクセスエラー
	[Serializable()]
	public class FileAccessFailureException : System.Exception
	{
		public	  FileAccessFailureException() : base() { }
		public	  FileAccessFailureException(string message) : base(message) { }
		public	  FileAccessFailureException(string message, System.Exception inner) : base(message, inner) { }
		protected FileAccessFailureException(System.Runtime.Serialization.SerializationInfo info,
						System.Runtime.Serialization.StreamingContext context) : base(info, context) { }
	}
	//	登録失敗
	[Serializable()]
	public class RegistrationFailureException : System.Exception
	{
		public	  RegistrationFailureException() : base() { }
		public	  RegistrationFailureException(string message) : base(message) { }
		public	  RegistrationFailureException(string message, System.Exception inner) : base(message, inner) { }
		protected RegistrationFailureException(System.Runtime.Serialization.SerializationInfo info,
						System.Runtime.Serialization.StreamingContext context) : base(info, context) { }
	}
	//	リセット失敗
	[Serializable()]
	public class ResetFailedException : System.Exception
	{
		public	  ResetFailedException() : base() { }
		public	  ResetFailedException(string message) : base(message) { }
		public	  ResetFailedException(string message, System.Exception inner) : base(message, inner) { }
		protected ResetFailedException(System.Runtime.Serialization.SerializationInfo info,
						System.Runtime.Serialization.StreamingContext context) : base(info, context) { }
	}

}
