﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Drawing.Design;
using System.Windows.Forms.Design;
using AjaraInit;

namespace CAjrCustCtrl
{
	//----------------------------------------------------------------------------------------------------------//
	[ToolboxBitmap(typeof(CAjrInpValue), "CAjrInpValue.bmp")]
	public partial class CAjrInpValue : UserControl
	{
		//----- 外部へのイベント ------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Notifies the entered integer value.")]
		public event IvEvtNtcIntValue  OnNtcIntValue;

		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Notifies the entered real number.")]
		public event IvEvtNtcRealValue OnNtcRealValue;

		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Occurs when Shift/Ctrl + right-clicked.")]
		public event IvEvtNtcRClick    OnRClick;

		private IvCbkNtcIntValue	m_IvCbkNtcIntValue;
		private IvCbkNtcRealValue	m_IvCbkNtcRealValue;
		private IvCbkNtcRClick		m_IvCbkNtcRClick;

		//----- ワーク ------------------------------------//
		private IntPtr		m_hCtrl;
		private bool		m_ToolTipShowAlways = true;
		private string		m_TipText;
		private bool		m_EnableDefToolTip = false;
		private	bool		m_FirstResize = true;
		private	bool		m_RealMode = false;
		private	bool		m_Notify   = false;

		//----- コンストラクタ ---------------------------------------------------------------------------------//
		public CAjrInpValue()
		{
			InitializeComponent();

			//	ライブラリ初期化
			CAjrInit.Init();

			//	コントロールウインド生成
			m_IvCbkNtcIntValue	= new IvCbkNtcIntValue (cbIvNtcIntValue );
			m_IvCbkNtcRealValue = new IvCbkNtcRealValue(cbIvNtcRealValue);
			m_IvCbkNtcRClick	= new IvCbkNtcRClick   (cbIvNtcRClick	);

			m_hCtrl = CAjrApi.AjcIvCreateForMSIL(this.Handle, m_IvCbkNtcIntValue, m_IvCbkNtcRealValue, m_IvCbkNtcRClick);
		}

		//------------------------------------------------------------------------------------------------------//
		//	コントロール内部イベント																			//
		//------------------------------------------------------------------------------------------------------//
		//----- 初期設定 ---------------------------------------------------------------------------------------//
		private void CAjrInpValue_Load(object sender, EventArgs e)
		{

		}
		//----- コントロール・サイズ変更 -----------------------------------------------------------------------//
		private void CAjrInpValue_Resize(object sender, EventArgs e)
		{
			//	初回のみの処理
			if (m_FirstResize) {
				m_FirstResize = false;
			}
			//	コントロールのサイズ設定
			CAjrApi.AjcIvSetProp_WndSize(m_hCtrl, (int)this.Size.Width, (int)this.Size.Height);
		}

		//------------------------------------------------------------------------------------------------------//
		//	プロパティ																							//
		//------------------------------------------------------------------------------------------------------//
		//----- Fontプロパティを隠す ---------------------------------------------------------------------------//
		private Font vfontInit = new Font("MS UI Gothic", 9);
		[Browsable(false)]
		public override Font Font
		{
			set { base.Font = vfontInit; }
			get { return vfontInit; }
		}
		//----- AllowDropプロパティを隠す ----------------------------------------------------------------------//
		private bool AllowDropInit = false;
		[Browsable(false)]
		public override bool AllowDrop
		{
			set { base.AllowDrop = AllowDropInit; }
			get { return AllowDropInit; }
		}
		//----- Enabledプロパティを変更 ------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Enable / Disable the control")]
		public new bool Enabled
		{
			set {	if (base.Enabled == value) base.Enabled = !value;
					base.Enabled = value;
					CAjrApi.AjcEnableCtrlsInWnd(this.Handle, value);
				}
			get {	return base.Enabled;  }
		}
		//----- フォーカスによる自動数値編集		(AJCIVS_AUTOEDIT) ------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Becomes an input-state in the click of the text box.(true:Enable，false:Disable)")]
		public bool AutoEdit {
			set {CAjrApi.AjcIvSetSty_AUTOEDIT(m_hCtrl, value);}
			get {return CAjrApi.AjcIvGetSty_AUTOEDIT(m_hCtrl);}
		}
		//----- １６進表示							(AJCIVS_SHOWHEX) -------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Displaing by hexa-decimal(true: Hexa-decimal，false:Decimal)")]
		public bool ShowHexaDecimal {
			set {CAjrApi.AjcIvSetSty_SHOWHEX(m_hCtrl, value);}
			get {return CAjrApi.AjcIvGetSty_SHOWHEX(m_hCtrl);}
		}
		//----- 数値・直接入力時のブリンク			(AJCIVS_NOBLINK) -------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Blink in input-state (true:Blink，false:Not blink)")]
		public bool BlinkOnInput {
			set {CAjrApi.AjcIvSetSty_NOBLINK(m_hCtrl, !value);}
			get {return !CAjrApi.AjcIvGetSty_NOBLINK(m_hCtrl);}
		}
		//----- コントロール外枠非表示				(AJCTCS_NOBORDER) ------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Outer frame display")]
		public bool ShowBorder {
			set {CAjrApi.AjcIvSetSty_NOBORDER(m_hCtrl, !value);}
			get {return !CAjrApi.AjcIvGetSty_NOBORDER(m_hCtrl);}
		}
		//----- スライダ表示						(AJCIVS_NOSLD) ---------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Slider display")]
		public bool ShowSlider {
			set {CAjrApi.AjcIvSetSty_NOSLD(m_hCtrl, !value);}
			get {return !CAjrApi.AjcIvGetSty_NOSLD(m_hCtrl);}
		}
		//----- ボタン非表示						(AJCIVS_NOBTN) ---------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("OK-button display")]
		public bool ShowButton {
			set {CAjrApi.AjcIvSetSty_NOBTN(m_hCtrl, !value);}
			get {return !CAjrApi.AjcIvGetSty_NOBTN(m_hCtrl);}
		}
		//----- スピンボタン非表示					(AJCIVS_NOSPN) ---------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Spin button display")]
		public bool ShowSpinButton {
			set {CAjrApi.AjcIvSetSty_NOSPN(m_hCtrl, !value);}
			get {return !CAjrApi.AjcIvGetSty_NOSPN(m_hCtrl);}
		}
		//----- 整数部を３桁毎にカンマで区切る		(AJCIVS_SEPARATE) ------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Divides an integer part every 3 digits")]
		public bool Separate {
			set {CAjrApi.AjcIvSetSty_SEPARATE(m_hCtrl, value);}
			get {return CAjrApi.AjcIvGetSty_SEPARATE(m_hCtrl);}
		}
		//----- 実数モード							(AJCIVS_REALMODE) ------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Real number / Integer mode (Set this property first)")]
		public bool _RealMode {
			set {CAjrApi.AjcIvSetSty_REALMODE(m_hCtrl, m_RealMode = value);}
			get {return CAjrApi.AjcIvGetSty_REALMODE(m_hCtrl);}
		}
		//----- 数値の最少値 			(min) ------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Minimum input value")]
		public double MinValue {
			set {CAjrApi.AjcIvSetProp_min(m_hCtrl, value);}
			get {return CAjrApi.AjcIvGetProp_min(m_hCtrl);}
		}
		//----- 数値の最大値 			(max) ------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("maximum input value")]
		public double MaxValue {
			set {CAjrApi.AjcIvSetProp_max(m_hCtrl, value);}
			get {return CAjrApi.AjcIvGetProp_max(m_hCtrl);}
		}
		//----- 数値の最小単位			(SldUnit) --------------------------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Unit of value")]
		public double UnitValue {
			set {CAjrApi.AjcIvSetProp_SldUnit(m_hCtrl, value);}
			get {return CAjrApi.AjcIvGetProp_SldUnit(m_hCtrl);}
		}
		//----- スライダのページサイズ 	(SldPage) --------------------------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Slider page size")]
		public double SliderPageSize {
			set {CAjrApi.AjcIvSetProp_SldPage(m_hCtrl, value);}
			get {return CAjrApi.AjcIvGetProp_SldPage(m_hCtrl);}
		}
		//----- スピンボタンの増減値 	(SpnStep) --------------------------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Step by spin button")]
		public double SpinStep {
			set {CAjrApi.AjcIvSetProp_SpnStep(m_hCtrl, value);}
			get {return CAjrApi.AjcIvGetProp_SpnStep(m_hCtrl);}
		}
		//----- テキストの桁数 			(TextLen) --------------------------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Digits of nummeric text")]
		public int TextLength {
			set {CAjrApi.AjcIvSetProp_TextLen(m_hCtrl, (int)value);}
			get {return (int)CAjrApi.AjcIvGetProp_TextLen(m_hCtrl);}
		}
		//----- 小数部の桁数			(Precision) ------------------------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Digits of decimal fraction")]
		public int Precision {
			set {CAjrApi.AjcIvSetProp_Precision(m_hCtrl, (int)value);}
			get {return (int)CAjrApi.AjcIvGetProp_Precision(m_hCtrl);}
		}
		//----- １６進表示桁数			(HexLen) ---------------------------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Hexa-decimal length (Number of digits)")]
		public int HexLen {
			set {CAjrApi.AjcIvSetProp_HexLen(m_hCtrl, value);}
			get {return CAjrApi.AjcIvGetProp_HexLen(m_hCtrl);}
		}
		//----- コントロール外枠表示色	(BorderColor) ----------------------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Outer frame color")]
		public Color BorderColor {
			set {
				int r = value.R;
				int g = value.G;
				int b = value.B;
				CAjrApi.AjcIvSetProp_BorderColor(m_hCtrl, (r | (g << 8) | (b << 16)));
			}
			get {
				int rgb = (int)CAjrApi.AjcIvGetProp_BorderColor(m_hCtrl);
				Color c = Color.FromArgb((int)(rgb & 0xFF), (int)((rgb >> 8) & 0xFF), (int)((rgb >> 16) & 0xFF));
				return c;
			}
		}
		//----- 値設定時のイベント設定 -------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Raise a notification event when setting a value")]
		public bool NotifyWhenSet {
			set {m_Notify = value;}
			get {return m_Notify;}
		}
		//----- 実数値					(Value) ----------------------------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Initial real value")]
		public double Value {
			set {if (m_RealMode) CAjrApi.AjcIvSetRealValue(m_hCtrl, value, m_Notify);}
			get {return (m_RealMode ? CAjrApi.AjcIvGetRealValue(m_hCtrl) : 0.0);}
		}
		//----- 整数値					(IntValue) -------------------------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Initial integer value")]
		public int IntValue {
			set {if (!m_RealMode) CAjrApi.AjcIvSetIntValue(m_hCtrl, value, m_Notify);}
			get {return (!m_RealMode ? CAjrApi.AjcIvGetIntValue(m_hCtrl) : 0);}
		}
		//----- ツールチップ表示条件 ---------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Tooltip display condition. If false is set, the tooltip will not be displayed when inactive.")]
		public bool ToolTipShowAlways
		{
			set {
				m_ToolTipShowAlways = value;
				CAjrApi.AjcIvSetTipShowAlways(m_hCtrl, m_ToolTipShowAlways);
			}
			get {
				return m_ToolTipShowAlways;
			}
		}
		//----- ツールチップ文字列		(TipText) --------------------------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Tool tip text of this control")]
		public string ToolTipText {
			set	{
					if		(this.DesignMode) m_TipText = value;
					else {
				 		StringBuilder s = new StringBuilder(2048);
				 		CAjrApi.AjcIvGetTipTextForMSIL(m_hCtrl, s, s.Capacity);
						if (value != s.ToString()) {
							CAjrApi.AjcIvSetTipTextForMSIL(m_hCtrl, value);
							CAjrApi.AjcIvSetTipShowAlways (m_hCtrl, m_ToolTipShowAlways);
						}
					}
				}
			get	{
					string rc = "";
					if (this.DesignMode) rc = m_TipText;
					else {
				 		StringBuilder s = new StringBuilder(2048);
				 		CAjrApi.AjcIvGetTipTextForMSIL(m_hCtrl, s, s.Capacity);
				 		rc = s.ToString();
				 	}
				 	return rc;
				}
		}
		//----- テキスト				(Text) -----------------------------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Numeric expression text")]
		[Browsable(true)]
		override public string Text {
			set {
				CAjrApi.AjcIvSetText(m_hCtrl, value);
			}
			get {
				StringBuilder s = new StringBuilder(2048);
				CAjrApi.AjcIvGetText(m_hCtrl);
				return s.ToString();
			}
		}
		//----- デフォルトツールチップの許可／禁止 -------------------------------------------------------------//
		[System.ComponentModel.Category("AjrInputValue")]
		[System.ComponentModel.Description("Allow default tooltip")]
		public bool EnableDefToolTip {
			set {if (this.DesignMode) m_EnableDefToolTip = value;
				 else				  CAjrApi.AjcIvEnaDefTipText(m_hCtrl, m_EnableDefToolTip = value);}
			get {return  m_EnableDefToolTip;}
		}
		//------------------------------------------------------------------------------------------------------//
		//	メソッド																							//
		//------------------------------------------------------------------------------------------------------//
		//----- ToString ---------------------------------------------------------------------------------------//
		public override string ToString() 
		{
			string s;
			if (this._RealMode) s = this.Value.ToString();
			else				s = this.IntValue.ToString();
			return s;
		}
		//----- 値の設定 ---------------------------------------------------------------------------------------//
		public void SetValue(double value)
		{
			CAjrApi.AjcIvSetRealValue(m_hCtrl, value, false);
		}
		// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -//
		public void SetValue(double value, bool fNtc)
		{
			CAjrApi.AjcIvSetRealValue(m_hCtrl, value, fNtc);
		}
		//------------------------------------------------------------------------------------------------------//
		//	コールバック																						//
		//------------------------------------------------------------------------------------------------------//
		//----- コールバック関数（整数値通知）------------------------------------------------------------------//
		private void cbIvNtcIntValue(int value)
		{
			IvArgNtcIntValue eva = new IvArgNtcIntValue();
			eva.value = value;
			if (OnNtcIntValue != null) {
				OnNtcIntValue(this, eva);
			}
		}
		//----- コールバック関数（実数値通知）------------------------------------------------------------------//
		private void cbIvNtcRealValue(double value)
		{
			IvArgNtcRealValue eva = new IvArgNtcRealValue();
			eva.value = value;
			if (OnNtcRealValue != null) {
				OnNtcRealValue(this, eva);
			}
		}
		//----- コールバック関数（右クリック通知）--------------------------------------------------------------//
		private void cbIvNtcRClick(int x, int y, bool shift, bool ctrl)
		{
			IvArgNtcRClick eva = new IvArgNtcRClick();
			eva.x	  = x;
			eva.y	  = y;
			eva.shift = shift;
			eva.ctrl  = ctrl;
			if (OnRClick != null) {
				OnRClick(this, eva);
			}
		}
		//------------------------------------------------------------------------------------------------------//
		//	PreviewKeyDownイベントにより、数値編集中は矢印キーを無効化する（フォーカスを移動させない）			//
		//------------------------------------------------------------------------------------------------------//
		private void CAjrInpValue_PreviewKeyDown(object sender, PreviewKeyDownEventArgs e)
		{
			switch (e.KeyCode)
			{
				case Keys.Up:
				case Keys.Left:
				case Keys.Right:
				case Keys.Down:
				case Keys.Escape:
					if (CAjrApi.AjcIvGetEditState(m_hCtrl)) {
						e.IsInputKey = true;
					}
					break;
			}
		}
	}
}
