﻿using System;
using System.ComponentModel;
using System.Drawing;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.Runtime.InteropServices;
using AjaraInit;

namespace CAjrCustCtrl
{
	//----------------------------------------------------------------------------------------------------------//
	[ToolboxBitmap(typeof(CAjrSerialComPort), "CAjrSerialComPort.bmp")]
	public partial class CAjrSerialComPort : Component
	{
		//----- 外部へのイベント -------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Notifies that the communication port has changed.")]
		public event ScpEvtPortState		OnPortState ;

		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Notify incoming chunk text in real time.")]
		public event ScpEvtRxChunkTxt		OnRxChunkTxt;

		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Notify incoming chunk binary data in real time.")]
		public event ScpEvtRxChunkBin		OnRxChunkBin;

		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Notify the received text data.")]
		public event ScpEvtRxText			OnRxText	;

		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Notifies the received escape sequence.")]
		public event ScpEvtRxEsc			OnRxEsc 	;

		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Notify the received control code.")]
		public event ScpEvtRxCtrl			OnRxCtrl	;

		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Notify the received packet data.")]
		public event ScpEvtRxPacket			OnRxPacket	;

		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Notify the completion of transmission.")]
		public event ScpEvtTxEmpty			OnTxEmpty	;

		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Notifies the part of the received data other than the packet.")]
		public event ScpEvtRxNoPkt			OnRxNoPkt	;

		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Notifies the 14-bit value extracted from the received 2 bytes.")]
		public event ScpEvtRxWord14			OnRxWord14	;

		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Notifies that invalid text chunk data has been received.")]
		public event ScpEvtRxInvChunk		OnRxInvChunk;

		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Notifies that an error has been detected.")]
		public event ScpEvtError			OnError 	;

		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Notifies that the RING signal has changed.")]
		public event ScpEvtNtcRING			OnNtcRING	;

		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Notifies that the RLSD signal has changed.")]
		public event ScpEvtNtcRLSD			OnNtcRLSD	;

		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Notifies that the DSR signal has changed.")]
		public event ScpEvtNtcDSR			OnNtcDSR	;

		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Notifies that the CTS signal has changed.")]
		public event ScpEvtNtcCTS			OnNtcCTS	;

		private	ScpCbkPortState 		m_ScpCbkPortState;
		private	ScpCbkRxChunkTxt		m_ScpCbkRxChunkTxt;
		private	ScpCbkRxChunkBin		m_ScpCbkRxChunkBin;
		private	ScpCbkRxText			m_ScpCbkRxText;
		private	ScpCbkRxEsc 			m_ScpCbkRxEsc;
		private	ScpCbkRxCtrl			m_ScpCbkRxCtrl;
		private	ScpCbkRxPacket			m_ScpCbkRxPacket;
		private	ScpCbkTxEmpty			m_ScpCbkTxEmpty;
		private	ScpCbkRxNoPkt			m_ScpCbkRxNoPkt;
		private	ScpCbkRxWord14			m_ScpCbkRxWord14;
		private	ScpCbkRxInvChunk		m_ScpCbkRxInvChunk;
		private	ScpCbkError 			m_ScpCbkError;
		private	ScpCbkNtcRING			m_ScpCbkNtcRING;
		private	ScpCbkNtcRLSD			m_ScpCbkNtcRLSD;
		private	ScpCbkNtcDSR			m_ScpCbkNtcDSR;
		private	ScpCbkNtcCTS			m_ScpCbkNtcCTS;

		//----- コンテナモードフラグ ---------------------------------------------------------------------------//
		private	bool			m_fContainer = true;
		//----- 実行モード情報 ---------------------------------------------------------------------------------//
		private EScpMode		m_ScpMode	= EScpMode.NotificationByEvent;	//	実行モード
		private EScpChunkMode	m_ChunkMode	= EScpChunkMode.BinaryData;		//	チャンク種別
		//----- ワーク -----------------------------------------------------------------------------------------//
		private IntPtr			m_hScp;
		private string			m_ProfileSect=	"SerialPortSect";
		private	int				m_STX		 =	0x02;
		private	int				m_ETX		 =	0x03;
		private	int				m_DLE		 =	0x10;
		private	int				m_PktTimeout =	3000;
		private EScpRxTextCode	m_RxTextCode =	EScpRxTextCode.SJIS;
		private EScpTxTextCode	m_TxTextCode =	EScpTxTextCode.SJIS;

		//----- コンストラクタ（１） ---------------------------------------------------------------------------//
		unsafe public CAjrSerialComPort()
		{
			InitializeComponent();

			m_fContainer = false;							//	コンソールモード

			CAjrInit.Init();								//	ライブラリ初期化

			SubCreateSerialComPort();						//	シリアル通信インスタンス生成
		}

		//----- コンストラクタ（２） ---------------------------------------------------------------------------//
		unsafe public CAjrSerialComPort(IContainer container)
		{
			container.Add(this);

			InitializeComponent();

			m_fContainer = true;							//	コンテナモード

			CAjrInit.Init();								//	ライブラリ初期化

			SubCreateSerialComPort();						//	シリアル通信インスタンス生成

			this.Disposed  += OnUnloadMyControl;			//	終了イベント設定

		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		unsafe void SubCreateSerialComPort()
		{
			m_ScpCbkPortState		= new ScpCbkPortState		(cbScpPortState		);
			m_ScpCbkRxChunkTxt		= new ScpCbkRxChunkTxt		(cbScpRxChunkTxt	);
			m_ScpCbkRxChunkBin		= new ScpCbkRxChunkBin		(cbScpRxChunkBin	);
			m_ScpCbkRxText			= new ScpCbkRxText			(cbScpRxText 		);
			m_ScpCbkRxEsc			= new ScpCbkRxEsc			(cbScpRxEsc			);
			m_ScpCbkRxCtrl			= new ScpCbkRxCtrl			(cbScpRxCtrl 		);
			m_ScpCbkRxPacket		= new ScpCbkRxPacket		(cbScpRxPacket		);
			m_ScpCbkTxEmpty 		= new ScpCbkTxEmpty			(cbScpTxEmpty		);
			m_ScpCbkRxNoPkt 		= new ScpCbkRxNoPkt			(cbScpRxNoPkt		);
			m_ScpCbkRxWord14 		= new ScpCbkRxWord14		(cbScpRxWord14		);
			m_ScpCbkRxInvChunk 		= new ScpCbkRxInvChunk		(cbScpRxInvChunk	);
			m_ScpCbkError			= new ScpCbkError			(cbScpError			);
			m_ScpCbkNtcRING 		= new ScpCbkNtcRING			(cbScpNtcRING		);
			m_ScpCbkNtcRLSD 		= new ScpCbkNtcRLSD			(cbScpNtcRLSD		);
			m_ScpCbkNtcDSR			= new ScpCbkNtcDSR			(cbScpNtcDSR 		);
			m_ScpCbkNtcCTS			= new ScpCbkNtcCTS			(cbScpNtcCTS 		);

			m_hScp 	= CAjrApi.AjcScpCreateForMSIL(	m_ScpCbkPortState	,
													m_ScpCbkRxChunkTxt	,
													m_ScpCbkRxChunkBin	,
													m_ScpCbkRxText		,
													m_ScpCbkRxEsc		,
													m_ScpCbkRxCtrl		,
													m_ScpCbkRxPacket	,
													m_ScpCbkTxEmpty 	,
													m_ScpCbkRxNoPkt 	,
													m_ScpCbkRxWord14 	,
													m_ScpCbkRxInvChunk 	,
													m_ScpCbkError		,
													m_ScpCbkNtcRING 	,
													m_ScpCbkNtcRLSD 	,
													m_ScpCbkNtcDSR		,
													m_ScpCbkNtcCTS		);
		}
		//----- 終了処理 ---------------------------------------------------------------------------------------//
		private void OnUnloadMyControl(object sender, EventArgs e)
		{
			CAjrApi.AjcScpDeleteForMSIL(m_hScp);
		}
		//------------------------------------------------------------------------------------------------------//
		//	プロパティ																							//
		//------------------------------------------------------------------------------------------------------//
		//	シリアル通信初期化（_ProfileSection, _ScpModeとMySlotNameがセットされた時点で、シリアル通信の初期化を行う）
		private bool m_fProfSect = false;		//	プロファイルセクション名設定済フラグ
		private bool m_fScpMode  = false;		//	モード値				設定済フラグ
		private bool m_fMySlot	 = false;		//	自スロット名			設定済フラグ
		private bool m_fRmtHost  = false;		//	相手ホスト名			設定済フラグ
		private bool m_fRmtSlot  = false;		//	相手スロット名			設定済フラグ
		private bool m_fInited	 = false;		//	初期化済フラグ

		private void InitScp2()
		{
			if (!m_fInited) {													//	未初期化？
				if (m_fProfSect && m_fScpMode && m_fMySlot &&					//		各値設定済？
								   m_fRmtHost && m_fRmtSlot) {
					CAjrApi.AjcScpLoadParamFromProfile(m_hScp, m_ProfileSect);	//			プロファイル読み出し(通信パラメタ)
					CAjrApi.AjcScpCreateMySlotForMSIL(m_hScp);					//			自メールスロット生成(エラー未検知で暗黙生成)
					if (m_ScpMode == EScpMode.NotificationByEvent) {			//			フォームアプリ？
						CAjrApi.AjcScpSetModeForMSIL(m_hScp, true);				//				実行モード設定（OnXXXイベント使用）
					}
					else {														//			コンソールアプリ？
						CAjrApi.AjcScpSetModeForMSIL(m_hScp, false);			//				実行モード設定（WaitEvent()使用）
					}
					m_fInited = true;											//		   「初期化済」
				}
			}
		}
		//----- プロファイルセクション -------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Profile section which records a set value" +
										   "(This property can be set only at designing)")]
		public string _ProfileSection {
			set {m_ProfileSect = value;
				 CAjrApi.AjcScpSetProfileSectForMSIL(m_hScp, value);	//	プロファイルセクション名設定
				 m_fProfSect   = true;									// 「プロファイルセクション名設定済」
				 InitScp2();											//	ＳＣＰ２初期化
			}
			get {return CAjrApi.AjcScpGetProfileSectForMSIL(m_hScp);}
		}
		//----- 実行モード -------------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Event reception mode" +
										   "(This property can be set only at designing)")]
		public EScpMode _ScpMode {
			set {m_ScpMode	= value;				//	モード値設定
				 m_fScpMode = true;					// 「モード値設定済」
				 InitScp2();						//	ＳＣＰ２初期化
			}
			get {return m_ScpMode;}
		}
		//----- 自スロット名 -----------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Default local mail slot name")]
		public string MySlotName {
			set {CAjrApi.AjcScpSetMySlotForMSIL(m_hScp, value);		//	自スロット名設定
				 m_fMySlot = true;									// 「自スロット名設定済」
				 InitScp2();										//	ＳＣＰ２初期化
			}
			get {return CAjrApi.AjcScpGetMySlotForMSIL(m_hScp);}
		}
		//----- 相手コンピュータ名 -----------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Default remote computer name (When blank, it is a local computer)")]
		public string RemoteComputerName {
			set {CAjrApi.AjcScpSetRmtHostForMSIL(m_hScp, value);	//	相手ホスト名設定
				 m_fRmtHost = true;									// 「相手ホスト名設定済」
				 InitScp2();										//	ＳＣＰ２初期化
			}
			get {return CAjrApi.AjcScpGetRmtHostForMSIL(m_hScp);}
		}
		//----- 相手スロット名 ---------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Default remote mail slot name")]
		public string RemoteSlotName {
			set {CAjrApi.AjcScpSetRmtSlotForMSIL(m_hScp, value);	//	相手スロット名設定
				 m_fRmtSlot = true;									// 「相手スロット名設定済」
				 InitScp2();										//	ＳＣＰ２初期化
			}
			get {return CAjrApi.AjcScpGetRmtSlotForMSIL(m_hScp);}
		}
		//----- ＳＣＰチャンクデータの扱い ---------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Kind of reception chunk data" +
										   "(This property can be set only at designing)")]
		public EScpChunkMode ChunkMode {
			set {CAjrApi.AjcScpSetChunkMode(m_hScp, m_ChunkMode = value);}
			get {return m_ChunkMode;}
		}
		//----- パケット先頭コード(STX) ------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Packet starting code value (STX)")]
		public int STX {
			set {m_STX = (int)value;  CAjrApi.AjcScpSetPktCtrlCode(m_hScp, m_STX, 0, 0);}
			get {return (int)m_STX; }
		}
		//----- パケット末尾コード(ETX) ------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Packer ending code value (ETX)")]
		public int ETX {
			set {m_ETX = (int)value;  CAjrApi.AjcScpSetPktCtrlCode(m_hScp, 0, m_ETX, 0);}
			get {return (int)m_ETX; }
		}
		//----- パケット透過制御バイトコード(DLE) --------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Packet transparent control code (DLE)")]
		public int DLE {
			set {m_DLE = (int)value;  CAjrApi.AjcScpSetPktCtrlCode(m_hScp, 0, 0, m_DLE);}
			get {return (int)m_DLE; }
		}
		//----- パケット受信タイムアウトタイマ -----------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Packet reception timeout [ms]")]
		public int PktTimeout {
			set {m_PktTimeout = (int)value;  CAjrApi.AjcScpSetPktTimeout(m_hScp, m_PktTimeout);}
			get {return (int)m_PktTimeout; }
		}
		//----- 受信テキストの文字コード -----------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Received text code")]
		public EScpRxTextCode RxTextCode {
			set {m_RxTextCode = value;	CAjrApi.AjcScpSetRxTextCode(m_hScp, m_RxTextCode);
										CAjrApi.AjcScpSetTxTextCode(m_hScp, m_TxTextCode);}
			get {return m_RxTextCode; }
		}
		//----- 実際の受信テキストの文字コード（AUTO時は実際の受信文字コード）----------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Received actual text code")]
		public EScpRxTextCode ActualRxTextCode {
			get {return CAjrApi.AjcScpGetActualRxTextCode(m_hScp);}
		}
		//----- 送信テキストの文字コード -----------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Sending Text code")]
		public EScpTxTextCode TxTextCode {
			set {m_TxTextCode = value;	CAjrApi.AjcScpSetRxTextCode(m_hScp, m_RxTextCode);
										CAjrApi.AjcScpSetTxTextCode(m_hScp, m_TxTextCode);}
			get {return m_TxTextCode; }
		}
		//----- 実際の送信テキストの文字コード（AUTO時は受信文字コード）----------------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Sending actual text code")]
		public EScpTxTextCode ActualTxTextCode {
			get {return CAjrApi.AjcScpGetActualTxTextCode(m_hScp);}
		}
		//----- 自メールスロット生成状態 -----------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("My mail-slot creation status")]
		public bool IsCreatedMySlot {
			get {return CAjrApi.AjcScpMySlotIsCreated(m_hScp); }
		}

		//----- ポートオープン状態状態 -------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Communication port open status")]
		public bool IsOpened {
			get {return CAjrApi.AjcScpIsOpened(m_hScp) != 0 ? true : false;}
		}

		//----- 設定されている通信ポート番号 -------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Communication port number")]
		public int PortNo {
			get {return (int)CAjrApi.AjcScpGetSelectedPort(m_hScp);}
		}

		//----- バイトペアの受信順序 ---------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSerialCommunication")]
		[System.ComponentModel.Description("Receive order of byte pair upon reception of word (14 bit value)")]
		public EScpByteSeq ByteSeq {
			set {CAjrApi.AjcScpSetByteSeqRxWord14(m_hScp, value);}
			get {return (EScpByteSeq)CAjrApi.AjcScpGetByteSeqRxWord14(m_hScp);}
		}

		//------------------------------------------------------------------------------------------------------//
		//	メソッド																							//
		//------------------------------------------------------------------------------------------------------//
		//----- ＳＣＰ初期化 -----------------------------------------------------------------------------------//
		private bool m_ScpInited = false;
		public void Init()
		{
			if (!m_ScpInited) {
				//	OnXXXXXイベント定義に従い、イベントマスク設定
				if (m_fContainer) {
					SetEvtMaskByEvent();
				}
				//	フラグ＝初期化済み
				m_ScpInited = true;
			}
		}
		//----- 通信ポートオープン -----------------------------------------------------------------------------//
		public bool Open()
		{
			return CAjrApi.AjcScpOpenDefault(m_hScp);
		}
		//- - -  最初の通信リソースを指定してオープン - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public bool Open(ESelPort sel)
		{
			return CAjrApi.AjcScpOpenSelect(m_hScp, (int)sel);
		}
		//- - - ＣＯＭポート オープン - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public bool Open(int PortNo, int rate, EScpDataBits DataBits, EScpParity Parity, EScpStopBit StopBit)
		{
			return CAjrApi.AjcScpOpen(m_hScp, (int)PortNo, (int)rate, (int)DataBits, (int)Parity, (int)StopBit);
		}
		//- - - メールスロット オープン - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public bool Open(string RemoteHostName, string RemoteSlotName)
		{
			return CAjrApi.AjcScpOpenSlot(m_hScp, RemoteHostName, RemoteSlotName);
		}
		//- - - ソケット(TCP/IP) オープン - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public bool Open(string ServName, uint ServPortNo)
		{
			return CAjrApi.AjcScpOpenSock(m_hScp, ServName, ServPortNo);
		}
		//----- 通信ポートクローズ -----------------------------------------------------------------------------//
		public void Close()
		{
			CAjrApi.AjcScpClose(m_hScp);
		}
		//----- バイト送信 -------------------------------------------------------------------------------------//
		public void SendByte(int ByteCode)
		{
			CAjrApi.AjcScpSendChar(m_hScp, (byte)ByteCode);
		}
		//----- １文字送信 -------------------------------------------------------------------------------------//
		public void SendChar(char Character)
		{
			CAjrApi.AjcScpSendChar(m_hScp, (ushort)Character);
		}
		//----- １４ビットワード値（バイトペア）送信，Low Byte First -------------------------------------------//
		public void SendWord14LF(int data)
		{
			CAjrApi.AjcScpSendWord14LF(m_hScp, data);
		}
		//----- １４ビットワード値（バイトペア）送信，High Byte First ------------------------------------------//
		public void SendWord14HF(int data)
		{
			CAjrApi.AjcScpSendWord14HF(m_hScp, data);
		}
		//----- テキスト送信 -----------------------------------------------------------------------------------//
		public void SendText(string text)
		{
			CAjrApi.AjcScpSendText(m_hScp, text, (int)text.Length);
		}
		//----- バイナリ送信 -----------------------------------------------------------------------------------//
		public void SendBinary(Byte[] bin)
		{
			CAjrApi.AjcScpSendBinData(m_hScp, bin);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void SendBinary(IntPtr p, int len)
		{
			CAjrApi.AjcScpSendBinData(m_hScp, p, len);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		unsafe public void SendBinary(void *p, int len)
		{
			CAjrApi.AjcScpSendBinData(m_hScp, p, len);
		}
		//----- パケット送信 -----------------------------------------------------------------------------------//
		public int SendPacket(Byte[] bin)
		{
			int		rc = 0;
			if (bin != null) {
				rc = CAjrApi.AjcScpSendPacket(m_hScp, bin);
			}
			else {
				rc = CAjrApi.AjcScpSendPacket(m_hScp, (IntPtr)0, 0);
			}
			return rc;
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public int SendPacket(IntPtr p, int len)
		{
			return CAjrApi.AjcScpSendPacket(m_hScp, p, len);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		unsafe public int SendPacket(void* p, int len)
		{
			return CAjrApi.AjcScpSendPacket(m_hScp, p, len);
		}
		//----- ブレーク信号送出／停止 -------------------------------------------------------------------------//
		public void SendBreak(bool fActive)
		{
			CAjrApi.AjcScpSendBreak(m_hScp, fActive);
		}
		//----- ＤＴＲ信号設定 ---------------------------------------------------------------------------------//
		public void SetDTR(bool fActive)
		{
			CAjrApi.AjcScpSetDTR(m_hScp, fActive);
		}
		//----- ＲＴＳ信号設定 ---------------------------------------------------------------------------------//
		public void SetRTS(bool fActive)
		{
			CAjrApi.AjcScpSetRTS(m_hScp, fActive);
		}
		//----- ＤＳＲ信号取得 ---------------------------------------------------------------------------------//
		public bool GetDSR()
		{
			bool	rc = false;
			if ((CAjrApi.AjcScpGetSigState(m_hScp) & 0x0020) != 0) {
				rc = true;
			}
			return rc;
		}
		//----- ＣＴＳ信号取得 ---------------------------------------------------------------------------------//
		public bool GetCTS()
		{
			bool	rc = false;
			if ((CAjrApi.AjcScpGetSigState(m_hScp) & 0x0010) != 0) {
				rc = true;
			}
			return rc;
		}
		//----- ＲＬＳＤ信号取得 -------------------------------------------------------------------------------//
		public bool GetRLSD()
		{
			bool	rc = false;
			if ((CAjrApi.AjcScpGetSigState(m_hScp) & 0x0080) != 0) {
				rc = true;
			}
			return rc;
		}
		//----- ＲＩＮＧ信号取得 -------------------------------------------------------------------------------//
		public bool GetRING()
		{
			bool	rc = false;
			if ((CAjrApi.AjcScpGetSigState(m_hScp) & 0x0040) != 0) {
				rc = true;
			}
			return rc;
		}
		//----- 受信済データデータ破棄 -------------------------------------------------------------------------//
		public void PurgeRx()
		{
			CAjrApi.AjcScpPurgeRecvData(m_hScp);
		}
		//----- 送信待ちデータデータ破棄 -----------------------------------------------------------------------//
		public void PurgeTx()
		{
			CAjrApi.AjcScpPurgeSendData(m_hScp);
		}
		//----- 送受信データ破棄 -------------------------------------------------------------------------------//
		public void Purge()
		{
			CAjrApi.AjcScpPurgeAllData(m_hScp);
		}
		//----- ダイアログによる通信パラメタ設定 ---------------------------------------------------------------//
		public void SetParamByDialog()
		{
			CAjrApi.AjcScpDlgParamEasy(m_hScp, (IntPtr)0);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void SetParamByDialog(IntPtr hOwner)
		{
			CAjrApi.AjcScpDlgParamEasy(m_hScp, hOwner);
		}
		//----- ダイアログによる詳細な通信パラメタ設定 ---------------------------------------------------------//
		public void SetDetailParamByDialog()
		{
			CAjrApi.AjcScpDlgParamDetail(m_hScp, (IntPtr)0);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void SetDetailParamByDialog(IntPtr hOwner)
		{
			CAjrApi.AjcScpDlgParamDetail(m_hScp, hOwner);
		}
		//----- 通信パラメタ設定ダイアログによるＣＯＭポート，メールスロット，ソケット通信の選択許可／禁止 -----//
		public void EnablePortSelectionInDialog(bool fEnableComPort, bool fEnableMailSlot, bool fEnableSocket)
		{
			CAjrApi.AjcScpEnablePortSelectionEx(m_hScp, fEnableComPort, fEnableMailSlot, fEnableSocket);
		}
		//----- イベント待ち -----------------------------------------------------------------------------------//
		public EScpEvt WaitEvent(out Object EvtData, int msWaitTime)
		{
			IntPtr	wParam = (IntPtr)0;
			IntPtr	lParam = (IntPtr)0;
			IntPtr	pDat;
			int	len, param;
			EScpEvt	evt = (EScpEvt)0;

			//	出力情報初期化
			EvtData = null;

			if (CAjrApi.AjcScpWaitEvent(m_hScp, ref wParam, ref lParam, msWaitTime)) {
				CAjrApi.AjcScpGetEventData(m_hScp, lParam, out pDat, out len, out param);	//	イベントデータ取得

				evt = (EScpEvt)wParam;

				//----- ●ポートオープン状態通知 ---------------------------------------------------------------//
				if ((evt & EScpEvt.EV_PORTSTATE) != 0) {
					EvtData = param;
				}
				//----- ●チャンクデータ受信通知 ---------------------------------------------------------------//
				else if ((evt & EScpEvt.EV_RXCHUNK) != 0) {
					if (param == 0) {
						Byte[] BArr = new Byte[len];
						CAjrApi.AjcMemCopy(ref BArr[0], pDat, len);
						EvtData = BArr;
					}
					else {
						StringBuilder s = new StringBuilder(len + 1);
						CAjrApi.AjcWStrCopy(s, pDat);
						EvtData = s.ToString();
					}
   	   	   	 	}
				//----- ●テキスト受信通知 ---------------------------------------------------------------------//
				else if ((evt & EScpEvt.EV_RXTEXT) != 0) {
					StringBuilder s = new StringBuilder(len + 1);
					CAjrApi.AjcWStrCopy(s, pDat);
					EvtData = s.ToString();
				}
				//----- ●ＥＳＣコード受信通知 -----------------------------------------------------------------//
				else if ((evt & EScpEvt.EV_RXESC) != 0) {
					StringBuilder s = new StringBuilder(len + 1);
					CAjrApi.AjcWStrCopy(s, pDat);
					EvtData = s.ToString();
				}
				//----- ●制御コード受信通知 -------------------------------------------------------------------//
				else if ((evt & EScpEvt.EV_RXCTRL) != 0) {
					EvtData = CAjrApi.AjcMemPeekUW(pDat);
				}
				//----- ●パケットデータ受信通知 ---------------------------------------------------------------//
				else if ((evt & EScpEvt.EV_RXPKT) != 0) {
					Byte[] BArr = new Byte[len];
					CAjrApi.AjcMemCopy(ref BArr[0], pDat, len);
					EvtData = BArr;
				}
				//----- ●パケット外データ受信通知 -------------------------------------------------------------//
				else if ((evt & EScpEvt.EV_RXNOPKT) != 0) {
					StringBuilder s = new StringBuilder(len + 1);
					CAjrApi.AjcWStrCopy(s, pDat);
					EvtData = s.ToString();
				}
				//----- ●バイトペアによるワード(14Bit値）受信通知 ---------------------------------------------//
				else if ((evt & EScpEvt.EV_RXWORD14) != 0) {
					EvtData = CAjrApi.AjcMemPeekUW(pDat);
				}
				//----- ●その他 -------------------------------------------------------------------------------//
				else  {
					EvtData = param;
				}

				CAjrApi.AjcScpRelEventData(m_hScp, lParam);					//	イベントデータ開放
			}
			return evt;
		}
		//----- イベントマスク設定 -----------------------------------------------------------------------------//
		public void SetEvtMask(EScpEvt evt)
		{
			if (m_ScpMode == EScpMode.WaitingForEvent) {	//	コンソールアプリ？
				CAjrApi.AjcScpSetEvtMask(m_hScp, (int)evt);
			}
		}
		//----- イベントマスク取得 -----------------------------------------------------------------------------//
		public EScpEvt GetEvtMask()
		{
			EScpEvt	rc = 0;
			rc = (EScpEvt)CAjrApi.AjcScpGetEvtMask(m_hScp);
			return rc;
		}
		//----- ポートパス名取得 -------------------------------------------------------------------------------//
		public string GetPortPathName()
		{
			return CAjrApi.AjcScpGetPortPathNameForMSIL(m_hScp);
		}
		//----- ポート名取得 -----------------------------------------------------------------------------------//
		public string GetPortName()
		{
			return CAjrApi.AjcScpGetPortNameForMSIL(m_hScp);
		}
		//----- ＣＯＭポートデバイス名取得 ---------------------------------------------------------------------//
		public string GetPortDevName(string ComPortName)
		{
			return CAjrApi.AjcScpGetPortDevName(ComPortName);
		}
		//----- 自メールスロットパス名取得 ---------------------------------------------------------------------//
		public string GetMySlotPathName()
		{
			return CAjrApi.AjcScpGetMySlotPathNameForMSIL(m_hScp);
		}
		//----- 自メールスロット生成 ---------------------------------------------------------------------------//
		public bool CreateMySlot()
		{
			return CAjrApi.AjcScpCreateMySlot(m_hScp);
		}
		//----- 自メールスロット消去 ---------------------------------------------------------------------------//
		public void DeleteMySlot()
		{
			CAjrApi.AjcScpDeleteMySlot(m_hScp);
		}
		//----- メールスロット名情報設定 -----------------------------------------------------------------------//
		public void SetMailSlotNames(string MySlot, string RmtHost, string RmtSlot)
		{
			CAjrApi.AjcScpSetMailSlotNames(m_hScp, MySlot, RmtHost, RmtSlot);
		}
		//----- メールスロットの通信速度制限 -------------------------------------------------------------------//
		public void SetMailSlotSpeedLimit(bool fLimit, int KBps)
		{
			CAjrApi.AjcScpSetTxSpeedLimit(m_hScp, fLimit , KBps);
		}
		//----- ＳＣＰ消去 -------------------------------------------------------------------------------------//
		public void Delete() {
			CAjrApi.AjcScpDeleteForMSIL(m_hScp);
		}
		//------------------------------------------------------------------------------------------------------//
		//	内部サブ関数																						//
		//------------------------------------------------------------------------------------------------------//
		//----- SetEvtMaskByEvent		   ---------------------------------------------------------------------//
		private bool SetEvtMaskByEvent()
		{
			bool	rc;
			EScpEvt	evt = 0;

			if (OnPortState		 != null) evt |= EScpEvt.EV_PORTSTATE;		//	ポート状態通知
			if (OnRxChunkTxt	 != null) evt |= EScpEvt.EV_RXCHUNK;		//	チャンクテキスト受信通知
			if (OnRxChunkBin	 != null) evt |= EScpEvt.EV_RXCHUNK;		//	チャンクバイナリ受信通知
			if (OnRxText		 != null) evt |= EScpEvt.EV_RXTEXT;			//	テキスト受信通知
			if (OnRxEsc 		 != null) evt |= EScpEvt.EV_RXESC;			//	ＥＳＣ受信通知
			if (OnRxCtrl		 != null) evt |= EScpEvt.EV_RXCTRL;			//	制御コード受信通知
			if (OnRxPacket		 != null) evt |= EScpEvt.EV_RXPKT;			//	パケット受信通知
			if (OnTxEmpty		 != null) evt |= EScpEvt.EV_TXEMPTY;		//	送信完了通知
			if (OnRxNoPkt		 != null) evt |= EScpEvt.EV_RXNOPKT;		//	パケット外テキスト受信通知
			if (OnRxWord14		 != null) evt |= EScpEvt.EV_RXWORD14;		//	バイトペアによるワード(14Bit値)受信通知
			if (OnRxInvChunk	 != null) evt |= EScpEvt.EV_INVCHUNK;		//	不正チャンクテキスト受信通知
			if (OnError	 		 != null) evt |= EScpEvt.EV_ERR;			//	エラー通知
			if (OnNtcRING		 != null) evt |= EScpEvt.EV_RING;			//	RING変化通知
			if (OnNtcRLSD		 != null) evt |= EScpEvt.EV_RLSD;			//	RLSD変化通知
			if (OnNtcDSR		 != null) evt |= EScpEvt.EV_DSR;			//	DSR変化通知
			if (OnNtcCTS		 != null) evt |= EScpEvt.EV_CTS;			//	CTS変化通知

			rc = CAjrApi.AjcScpSetEvtMask(m_hScp, (int)evt);

			return rc;
		}
		//------------------------------------------------------------------------------------------------------//
		//	コールバック																						//
		//------------------------------------------------------------------------------------------------------//
		//----- コールバック関数（ポート状態通知）--------------------------------------------------------------//
		private void cbScpPortState(EScpPortState state, string name)
		{
			if (OnPortState != null) {
				ScpArgPortState eva = new ScpArgPortState();
				eva.state = state;
				eva.name  = name;
				OnPortState(this, eva);
			}
		}
		//----- コールバック関数（テキストチャンク受信通知）----------------------------------------------------//
		private void cbScpRxChunkTxt(string text)
		{
			if (OnRxChunkTxt != null) {
				ScpArgRxChunkTxt eva = new ScpArgRxChunkTxt();
				eva.text = text;
				OnRxChunkTxt(this, eva);
			}
		}
		//----- コールバック関数（バイナリチャンク受信通知）----------------------------------------------------//
		unsafe private void cbScpRxChunkBin(void *p, int len)
		{
			if (OnRxChunkBin != null && len != 0) {
				ScpArgRxChunkBin eva = new ScpArgRxChunkBin();
				eva.bin = new Byte[len];
				fixed (void *pBArr = &eva.bin[0]) {
					CAjrApi.AjcMemCopy(pBArr, p, len);
				}
				OnRxChunkBin(this, eva);
			}
		}
		//----- コールバック関数（テキスト受信通知）------------------------------------------------------------//
		private void cbScpRxText(string text)
		{
			if (OnRxText != null) {
				ScpArgRxText eva = new ScpArgRxText();
				eva.text  = text;
				OnRxText(this, eva);
			}
		}
		//----- コールバック関数（ＥＳＣ受信通知）--------------------------------------------------------------//
		private void cbScpRxEsc(string esc)
		{
			if (OnRxEsc != null) {
				ScpArgRxEsc eva = new ScpArgRxEsc();
				eva.esc  = esc;
				OnRxEsc(this, eva);
			}
		}
		//----- コールバック関数（制御コード受信通知）----------------------------------------------------------//
		private void cbScpRxCtrl(char ctrl)
		{
			if (OnRxCtrl != null) {
				ScpArgRxCtrl eva = new ScpArgRxCtrl();
				eva.ctrl  = ctrl;
				OnRxCtrl(this, eva);
			}
		}
		//----- コールバック関数（パケット受信通知）------------------------------------------------------------//
		unsafe private void cbScpRxPacket(void *p, int len)
		{
			if (OnRxPacket != null) {
				ScpArgRxPacket eva = new ScpArgRxPacket();
				if (len != 0) {
					eva.bin	= new Byte[len];
					fixed (void *pBArr = &eva.bin[0]) {
						CAjrApi.AjcMemCopy(pBArr, p, len);
					}
				}
				else {
					eva.bin = null;
				}
				OnRxPacket(this, eva);
			}
		}
		//----- コールバック関数（送信完了通知）----------------------------------------------------------------//
		private void cbScpTxEmpty()
		{
			if (OnTxEmpty != null) {
				EventArgs eva = new EventArgs();
				OnTxEmpty(this, eva);
			}
		}
		//----- コールバック関数（パケット外テキスト受信通知）--------------------------------------------------//
		private void cbScpRxNoPkt(string text)
		{
			if (OnRxNoPkt != null) {
				ScpArgRxNoPkt eva = new ScpArgRxNoPkt();
				eva.text  = text;
				OnRxNoPkt(this, eva);
			}
		}
		//----- コールバック関数（バイトペアによるワード(14Bit値）受信通知）------------------------------------//
		private void cbScpRxWord14(int data)
		{
			if (OnRxWord14 != null) {
				ScpArgRxWord14 eva = new ScpArgRxWord14();
				eva.data  = data;
				OnRxWord14(this, eva);
			}
		}
		//----- コールバック関数（不正チャンクテキスト受信通知）------------------------------------------------//
		unsafe private void cbScpRxInvChunk(void *p, int len)
		{
			if (OnRxInvChunk != null) {
				ScpArgRxInvChunk eva = new ScpArgRxInvChunk();
				eva.bin = new Byte[len];
				fixed (void *pBArr = &eva.bin[0]) {
					CAjrApi.AjcMemCopy(pBArr, p, len);
				}
				OnRxInvChunk(this, eva);
			}

		}
		//----- コールバック関数（エラー通知）------------------------------------------------------------------//
		private void cbScpError(int err)
		{
			if (OnError != null) {
				ScpArgError eva = new ScpArgError();
				eva.err = (EScpErr)err;
				OnError(this, eva);
			}
		}
		//----- コールバック関数（RING変化通知）----------------------------------------------------------------//
		private void cbScpNtcRING(bool cts, bool dsr,bool rlsd, bool ring)
		{
			if (OnNtcRING != null) {
				ScpArgNtcRING eva = new ScpArgNtcRING();
				eva.cts  = cts;
				eva.dsr  = dsr;
				eva.rlsd = rlsd;
				eva.ring = ring;
				OnNtcRING(this, eva);
			}
		}
		//----- コールバック関数（RLSD変化通知）----------------------------------------------------------------//
		private void cbScpNtcRLSD(bool cts, bool dsr,bool rlsd, bool ring)
		{
			if (OnNtcRLSD != null) {
				ScpArgNtcRLSD eva = new ScpArgNtcRLSD();
				eva.cts  = cts;
				eva.dsr  = dsr;
				eva.rlsd = rlsd;
				eva.ring = ring;
				OnNtcRLSD(this, eva);
			}
		}
		//----- コールバック関数（DSR変化通知）-----------------------------------------------------------------//
		private void cbScpNtcDSR(bool cts, bool dsr,bool rlsd, bool ring)
		{
			if (OnNtcDSR != null) {
				ScpArgNtcDSR eva = new ScpArgNtcDSR();
				eva.cts  = cts;
				eva.dsr  = dsr;
				eva.rlsd = rlsd;
				eva.ring = ring;
				OnNtcDSR(this, eva);
			}
		}
		//----- コールバック関数（CTS変化通知）-----------------------------------------------------------------//
		private void cbScpNtcCTS(bool cts, bool dsr,bool rlsd, bool ring)
		{
			if (OnNtcCTS != null) {
				ScpArgNtcCTS eva = new ScpArgNtcCTS();
				eva.cts  = cts;
				eva.dsr  = dsr;
				eva.rlsd = rlsd;
				eva.ring = ring;
				OnNtcCTS(this, eva);
			}
		}
	}
}
