﻿using System;
using System.ComponentModel;
using System.Drawing;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.Runtime.InteropServices;
using AjaraInit;

namespace CAjrCustCtrl
{
	//----------------------------------------------------------------------------------------------------------//
	[ToolboxBitmap(typeof(CAjrSockClient), "CAjrSockClient.bmp")]
	public partial class CAjrSockClient : Component
	{
		//----- 外部へのイベント -------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Notify the received text data.")]
		public event SctEvtRxText			OnRxText;

		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Notifies the received escape sequence.")]
		public event SctEvtRxEsc			OnRxEsc;

		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Notify the received control code.")]
		public event SctEvtRxCtrl			OnRxCtrl;

		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Notify the received packet data.")]
		public event SctEvtRxPacket			OnRxPacket;

		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Notifies the part of the received data other than the packet.")]
		public event SctEvtRxNoPkt			OnRxNoPkt;

		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Notify the completion of transmission.")]
		public event SctEvtTxEmpty			OnTxEmpty;

		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Notify incoming chunk text in real time.")]
		public event SctEvtRxChunkTxt		OnRxChunkTxt;

		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Notify incoming chunk binary data in real time.")]
		public event SctEvtRxChunkBin		OnRxChunkBin;

		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Notifies that invalid text chunk data has been received.")]
		public event SctEvtRxInvChunk		OnRxInvChunk;

		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Notifies you that you have connected to the server.")]
		public event SctEvtConnect			OnConnect;

		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Notifies connection with the server has been disconnected.")]
		public event SctEvtDisconnect		OnDisconnect;

		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Notifies connection error.")]
		public event SctEvtCnFail			OnCnFail;

		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Notifies that a reception error has been detected.")]
		public event SctEvtRecvError		OnRecvError;

		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Notifies that a transmission error has been detected.")]
		public event SctEvtSendError		OnSendError;

		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Notifies that a general error has been detected.")]
		public event SctEvtGeneralError		OnGeneralError;

		private SctCbkRxText				m_SctCbkRxText;
		private SctCbkRxEsc					m_SctCbkRxEsc;
		private SctCbkRxCtrl				m_SctCbkRxCtrl;
		private SctCbkRxPacket				m_SctCbkRxPacket;
		private SctCbkRxNoPkt				m_SctCbkRxNoPkt;
		private SctCbkTxEmpty				m_SctCbkTxEmpty;
		private SctCbkRxChunkTxt			m_SctCbkRxChunkTxt;
		private SctCbkRxChunkBin			m_SctCbkRxChunkBin;
		private SctCbkRxInvChunk			m_SctCbkRxInvChunk;
		private SctCbkConnect				m_SctCbkConnect;
		private SctCbkDisconnect			m_SctCbkDisconnect;
		private SctCbkCnFail				m_SctCbkCnFail;
		private SctCbkRecvError				m_SctCbkRecvError;
		private SctCbkSendError				m_SctCbkSendError;
		private SctCbkGeneralError			m_SctCbkGeneralError;

		//----- コンテナモードフラグ ---------------------------------------------------------------------------//
		private	bool			m_fContainer = true;
		//----- 実行モード情報 ---------------------------------------------------------------------------------//
		private ESctMode		m_SctMode	= ESctMode.NotificationByEvent;	//	実行モード
		private ESctChunkMode	m_ChunkMode	= ESctChunkMode.BinaryData;		//	チャンク種別

		//----- ワーク -----------------------------------------------------------------------------------------//
		private IntPtr			m_hSct;
		private	int				m_STX, m_ETX, m_DLE;
		private	int				m_PktTimeout;
		private ESctRxTextCode	m_RxTextCode =	ESctRxTextCode.SJIS;
		private ESctTxTextCode	m_TxTextCode =	ESctTxTextCode.SJIS;
		private	IntPtr			m_SvLParam	 =	(IntPtr)0;

		//----- コンストラクタ（１） ---------------------------------------------------------------------------//
		public CAjrSockClient()
		{
			InitializeComponent();

			m_fContainer = false;							//	コンソールモード

			//	初期化
			InitOnConstruct();
		}

		//----- コンストラクタ（２） ---------------------------------------------------------------------------//
		public CAjrSockClient(IContainer container)
		{
			container.Add(this);

			InitializeComponent();

			m_fContainer = true;							//	コンテナモード

			//	初期化
			InitOnConstruct();
		}
		//----- 初期化コード -----------------------------------------------------------------------------------//
		unsafe private void InitOnConstruct()
		{
			//	ライブラリ初期化
			CAjrInit.Init();

			//	変数初期化
			m_hSct 			= (IntPtr)0;
			m_STX			= 0x02;
			m_ETX			= 0x03;
			m_DLE			= 0x10;
			m_PktTimeout	= 3000;

			//	 サーバ生成
			m_SctCbkRxText		= new SctCbkRxText		(cbSctRxText		);
			m_SctCbkRxEsc		= new SctCbkRxEsc		(cbSctRxEsc			);
			m_SctCbkRxCtrl		= new SctCbkRxCtrl		(cbSctRxCtrl		);
			m_SctCbkRxPacket	= new SctCbkRxPacket	(cbSctRxPacket		);
			m_SctCbkRxNoPkt		= new SctCbkRxNoPkt		(cbSctRxNoPkt		);
			m_SctCbkTxEmpty		= new SctCbkTxEmpty		(cbSctTxEmpty		);
			m_SctCbkRxChunkTxt	= new SctCbkRxChunkTxt	(cbSctRxChunkTxt	);
			m_SctCbkRxChunkBin	= new SctCbkRxChunkBin	(cbSctRxChunkBin	);
			m_SctCbkRxInvChunk	= new SctCbkRxInvChunk	(cbSctRxInvChunk	);
			m_SctCbkConnect		= new SctCbkConnect		(cbSctConnect		);
			m_SctCbkDisconnect	= new SctCbkDisconnect	(cbSctDisconnect	);
			m_SctCbkCnFail		= new SctCbkCnFail		(cbSctCnFail		);
			m_SctCbkRecvError	= new SctCbkRecvError	(cbSctRecvError		);
			m_SctCbkSendError	= new SctCbkSendError	(cbSctSendError		);
			m_SctCbkGeneralError= new SctCbkGeneralError(cbSctGeneralError	);

			m_hSct = CAjrApi.AjcSctCreateForMSIL(m_SctCbkRxText		,
												 m_SctCbkRxEsc		,
												 m_SctCbkRxCtrl		,
												 m_SctCbkRxPacket	,
												 m_SctCbkRxNoPkt	,
												 m_SctCbkTxEmpty	,
												 m_SctCbkRxChunkTxt	,
												 m_SctCbkRxChunkBin	,
												 m_SctCbkRxInvChunk	,
												 m_SctCbkConnect	,
												 m_SctCbkDisconnect	,
												 m_SctCbkCnFail		,
												 m_SctCbkRecvError	,
												 m_SctCbkSendError	,
												 m_SctCbkGeneralError);

			//	終了イベント設定
			this.Disposed += OnUnloadMyControl;
		}
		//----- 終了処理 ---------------------------------------------------------------------------------------//
		private void OnUnloadMyControl(object sender, EventArgs e)
		{
			if (m_hSct != (IntPtr)0) {
				CAjrApi.AjcSctDeleteForMSIL(m_hSct);
			}
		}
		//------------------------------------------------------------------------------------------------------//
		//	プロパティ																							//
		//------------------------------------------------------------------------------------------------------//
		//----- 実行モード -------------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Event reception mode" +
										   "(This property can be set only at designing)")]
		public ESctMode _SctMode {
			set {m_SctMode = value;}
			get {return m_SctMode;}
		}
		//----- 回線状態 ---------------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Connection / Disconnection line state" )]
		public ESctState State {
			get {return CAjrApi.AjcSctGetState(m_hSct);}
		}
		//----- チャンクデータの扱い ---------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Kind of reception chunk data" )]
		public ESctChunkMode ChunkMode {
			set {CAjrApi.AjcSctSetChunkMode(m_hSct, m_ChunkMode = value);}
			get {return m_ChunkMode;}
		}
		//----- パケット先頭コード(STX) ------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Packet starting code value (STX)")]
		public int STX {
			set {m_STX = (int)value;  CAjrApi.AjcSctSetPktCtrlCode(m_hSct, m_STX, 0, 0);}
			get {return (int)m_STX; }
		}
		//----- パケット末尾コード(ETX) ------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Packer ending code value (ETX)")]
		public int ETX {
			set {m_ETX = (int)value;  CAjrApi.AjcSctSetPktCtrlCode(m_hSct, 0, m_ETX, 0);}
			get {return (int)m_ETX; }
		}
		//----- パケット透過制御バイトコード(DLE) --------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Packet transparent control code (DLE)")]
		public int DLE {
			set {m_DLE = (int)value;  CAjrApi.AjcSctSetPktCtrlCode(m_hSct, 0, 0, m_DLE);}
			get {return (int)m_DLE; }
		}
		//----- パケット受信タイムアウトタイマ -----------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Packet reception timeout [ms]")]
		public int PktTimeout {
			set {m_PktTimeout = (int)value;  CAjrApi.AjcSctSetPktTimeout(m_hSct, m_PktTimeout);}
			get {return (int)m_PktTimeout; }
		}
		//----- 受信テキストの文字コード -----------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Received text code")]
		public ESctRxTextCode RxTextCode {
			set {m_RxTextCode = value;	CAjrApi.AjcSctSetRxTextCode(m_hSct, m_RxTextCode);
										CAjrApi.AjcSctSetTxTextCode(m_hSct, m_TxTextCode);}
			get {return m_RxTextCode; }
		}
		//----- 実際の受信テキストの文字コード（AUTO時は実際の受信文字コード） ---------------------------------//
		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Received actual text code")]
		public ESctRxTextCode ActualRxTextCode {
			get {return CAjrApi.AjcSctGetActualRxTextCode(m_hSct);}
		}
		//----- 送信テキストの文字コード -----------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Sending Text code")]
		public ESctTxTextCode TxTextCode {
			set {m_TxTextCode = value;	CAjrApi.AjcSctSetRxTextCode(m_hSct, m_RxTextCode);
										CAjrApi.AjcSctSetTxTextCode(m_hSct, m_TxTextCode);}
			get {return m_TxTextCode; }
		}
		//----- 実際の送信テキストの文字コード（AUTO時は受信文字コード） ---------------------------------------//
		[System.ComponentModel.Category("AjrSockClient")]
		[System.ComponentModel.Description("Received actual text code")]
		public ESctTxTextCode ActualTxTextCode {
			get {return CAjrApi.AjcSctGetActualTxTextCode(m_hSct);}
		}
		//------------------------------------------------------------------------------------------------------//
		//	メソッド（初期化）																					//
		//------------------------------------------------------------------------------------------------------//
		private bool m_SctInited = false;
		public void Init()
		{
			if (!m_SctInited) {
				//	OnXXXXXイベント定義に従い、イベントマスク設定
				if (m_fContainer) {
				  SetEvtMaskByEvent();
				}
				//	フラグ＝初期化済み
				m_SctInited = true;
			}
		}
		//------------------------------------------------------------------------------------------------------//
		//	メソッド（クライアント操作）																		//
		//------------------------------------------------------------------------------------------------------//
		//----- １バイト送信 -----------------------------------------------------------------------------------//
		public void SendByte(byte ByteCode)
		{
			CAjrApi.AjcSctSendChar(m_hSct, ByteCode);
		}
		//----- １文字送信 -------------------------------------------------------------------------------------//
		public void SendChar(char Character)
		{
			CAjrApi.AjcSctSendChar(m_hSct, (ushort)Character);
		}
		//----- テキスト送信 -----------------------------------------------------------------------------------//
		public void SendText(string text)
		{
			CAjrApi.AjcSctSendText(m_hSct, text, (int)text.Length);
		}
		//----- バイナリ送信 -----------------------------------------------------------------------------------//
		public void SendBinary(Byte[] bin)
		{
			CAjrApi.AjcSctSendBinData(m_hSct, bin);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		unsafe public void SendBinary(void *p, int len)
		{
			CAjrApi.AjcSctSendBinData(m_hSct, p, len);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void SendBinary(IntPtr p, int len)
		{
			CAjrApi.AjcSctSendBinData(m_hSct, p, len);
		}
		//----- パケット送信 -----------------------------------------------------------------------------------//
		public int SendPacket(Byte[] bin)
		{
			int		rc = 0;
			if (bin != null) {
				rc = CAjrApi.AjcSctSendPacket(m_hSct, bin);
			}
			else {
				rc = CAjrApi.AjcSctSendPacket(m_hSct, (IntPtr)0, 0);
			}
			return rc;
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		unsafe public int SendPacket(void* p, int len)
		{
			return CAjrApi.AjcSctSendPacket(m_hSct, p, len);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public int SendPacket(IntPtr p, int len)
		{
			return CAjrApi.AjcSctSendPacket(m_hSct, p, len);
		}
		//----- 受信済データ破棄 -------------------------------------------------------------------------------//
		public void PurgeRx()
		{
			CAjrApi.AjcSctPurgeRecvData(m_hSct);
		}
		//----- 送信待ちデータ破棄 -----------------------------------------------------------------------------//
		public void PurgeTx()
		{
			CAjrApi.AjcSctPurgeSendData(m_hSct);
		}
		//----- 送受信データ破棄 -------------------------------------------------------------------------------//
		public void Purge()
		{
			CAjrApi.AjcSctPurgeAllData(m_hSct);
		}
		//----- 回線接続 ---------------------------------------------------------------------------------------//
		public void Connect(string Serv, uint PortNo)
		{
			Connect(Serv, PortNo, ESctFamily.INET);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void Connect(string Serv, uint PortNo, ESctFamily Family)
		{
			CAjrApi.AjcSctConnectForMSIL(m_hSct, Serv, PortNo.ToString(), Family, m_SctMode == ESctMode.NotificationByEvent ? true : false);
		}
		//----- 回線切断 ---------------------------------------------------------------------------------------//
		public void Disconnect()
		{
			CAjrApi.AjcSctDisconnectEx(m_hSct, 10000);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void Disconnect(int msTimeout)
		{
			CAjrApi.AjcSctDisconnectEx(m_hSct, msTimeout);
		}
		//------------------------------------------------------------------------------------------------------//
		//	メソッド（その他）																					//
		//------------------------------------------------------------------------------------------------------//
		//----- ＳＣＴ消去 -------------------------------------------------------------------------------------//
		public void Delete() {
			if (m_hSct != (IntPtr)0) {
				CAjrApi.AjcSctDeleteForMSIL(m_hSct);
				m_hSct = (IntPtr)0;
			}
		}
		//----- イベントマスク設定 -----------------------------------------------------------------------------//
		public void SetEvtMask(ESctEvt evt)
		{
			if (m_SctMode == ESctMode.WaitingForEvent) {	//	コンソールアプリ？
				CAjrApi.AjcSctSetEvtMask(m_hSct, evt);
			}
		}
		//----- イベントマスク取得 -----------------------------------------------------------------------------//
		public ESctEvt GetEvtMask()
		{
			ESctEvt	rc = 0;
			rc = (ESctEvt)CAjrApi.AjcSctGetEvtMask(m_hSct);
			return rc;
		}
		//----- イベント待ち -----------------------------------------------------------------------------------//
		public ESctEvt WaitEvent(out Object EvtData, int msWaitTime)
		{
			IntPtr	wParam	= (IntPtr)0;
			IntPtr	lParam	= (IntPtr)0;
			IntPtr	pDat;
			int		len, param;
			ESctEvt	evt 	= ESctEvt.EV_NOEVENT;
			IntPtr	pC		= (IntPtr)0;

			//	 出力情報クリアー
			EvtData = null; 			//	イベントデータ

			if (CAjrApi.AjcSctWaitEvent(m_hSct, ref wParam, ref lParam, msWaitTime)) {
				//	lParam 退避
				m_SvLParam = lParam;
				//	イベントデータ取得
				CAjrApi.AjcSctGetEventData(m_hSct, lParam, out pDat, out len, out param);
				//	イベントコード設定
				evt = (ESctEvt)wParam;

				//----- ●テキスト受信通知 ---------------------------------------------------------------------//
				if		((evt & ESctEvt.EV_RXTEXT) != 0) {
					StringBuilder s = new StringBuilder(len + 1);
					CAjrApi.AjcWStrCopy(s, pDat);
					EvtData 	= s.ToString();
				}
				//----- ●ＥＳＣコード受信通知 -----------------------------------------------------------------//
				else if ((evt & ESctEvt.EV_RXESC) != 0) {
					StringBuilder s = new StringBuilder(len + 1);
					CAjrApi.AjcWStrCopy(s, pDat);
					EvtData 	= s.ToString();
				}
				//----- ●制御コード受信通知 -------------------------------------------------------------------//
				else if ((evt & ESctEvt.EV_RXCTRL) != 0) {
					EvtData 	= CAjrApi.AjcMemPeekUW(pDat);
				}
				//----- ●パケットデータ受信通知 ---------------------------------------------------------------//
				else if ((evt & ESctEvt.EV_RXPKT) != 0) {
					Byte[] BArr = new Byte[len];
					CAjrApi.AjcMemCopy(ref BArr[0], pDat, len);
					EvtData 	= BArr;
				}
				//----- ●パケット外データ受信通知 -------------------------------------------------------------//
				else if ((evt & ESctEvt.EV_RXNOPKT) != 0) {
					StringBuilder s = new StringBuilder(len + 1);
					CAjrApi.AjcWStrCopy(s, pDat);
					EvtData 	= s.ToString();
				}
				//----- ●チャンクデータ受信通知 ---------------------------------------------------------------//
   	   	   		else if ((evt & ESctEvt.EV_RXCHUNK) != 0) {
					if (param == 0) {
						Byte[] BArr = new Byte[len];
						CAjrApi.AjcMemCopy(ref BArr[0], pDat, len);
						EvtData 	= BArr;
					}
					else {
						StringBuilder s = new StringBuilder(len + 1);
						CAjrApi.AjcWStrCopy(s, pDat);
						EvtData = s.ToString();
					}
				}
				//----- ●不正チャンクテキスト受信通知 ---------------------------------------------------------//
				else if ((evt & ESctEvt.EV_INVCHUNK) != 0) {
					Byte[] BArr = new Byte[len];
					CAjrApi.AjcMemCopy(ref BArr[0], pDat, len);
					EvtData 	= BArr;
				}
				//----- ●送信完了通知 -------------------------------------------------------------------------//
				else if ((evt & ESctEvt.EV_TXEMPTY) != 0) {
					EvtData 	= null;
				}
				//----- ●接続通知 -----------------------------------------------------------------------------//
				else if ((evt & ESctEvt.EV_CONNECT) != 0) {
					EvtData 	= null;
				}
				//----- ●切断通知 -----------------------------------------------------------------------------//
				else if ((evt & ESctEvt.EV_DISCONNECT) != 0) {
					EvtData 	= null;
				}
				//----- ●その他 -------------------------------------------------------------------------------//
				else  {
					EvtData 	= param;
				}

				//	イベントデータ開放
				CAjrApi.AjcSctRelEventData(m_hSct, m_SvLParam);
			}
			return evt;
		}
		//--------------------------------------------------------------------------------------------------//
		//	内部サブ関数																					//
		//--------------------------------------------------------------------------------------------------//
		//----- SetEvtMaskByEvent		   -----------------------------------------------------------------//
		private bool SetEvtMaskByEvent()
		{
			bool	rc;
			ESctEvt	evt = 0;

			if (OnRxText		 != null) evt |= ESctEvt.EV_RXTEXT;			//	テキストデータ通知
			if (OnRxEsc			 != null) evt |= ESctEvt.EV_RXESC;			//	ＥＳＣコードデータ通知
			if (OnRxCtrl		 != null) evt |= ESctEvt.EV_RXCTRL;			//	制御コード通知
			if (OnRxPacket		 != null) evt |= ESctEvt.EV_RXPKT;			//	パケットデータ通知
			if (OnRxNoPkt 		 != null) evt |= ESctEvt.EV_RXNOPKT;		//	パケット外テキストデータ
			if (OnRxChunkTxt	 != null) evt |= ESctEvt.EV_RXCHUNK;		//	パケットデータ通知
			if (OnRxChunkBin	 != null) evt |= ESctEvt.EV_RXCHUNK;		//	パケットデータ通知
			if (OnRxInvChunk	 != null) evt |= ESctEvt.EV_INVCHUNK;		//	チャンクデータ受信通知
			if (OnTxEmpty 		 != null) evt |= ESctEvt.EV_TXEMPTY;		//	送信完了通知
			if (OnConnect 		 != null) evt |= ESctEvt.EV_CONNECT;		//	不正チャンクテキスト受信通知
			if (OnDisconnect	 != null) evt |= ESctEvt.EV_DISCONNECT;		//	切断通知
			if (OnCnFail		 != null) evt |= ESctEvt.EV_CNFAIL;			//	接続失敗通知
			if (OnRecvError		 != null) evt |= ESctEvt.EV_RXERR;			//	受信エラー通知
			if (OnSendError		 != null) evt |= ESctEvt.EV_TXERR;			//	送信エラー通知

			rc = CAjrApi.AjcSctSetEvtMask(m_hSct, evt);
			return rc;
		}
		//------------------------------------------------------------------------------------------------------//
		//	コールバック																						//
		//------------------------------------------------------------------------------------------------------//
		//----- コールバック関数（テキスト受信通知）------------------------------------------------------------//
		private void cbSctRxText(string text)
		{
			if (OnRxText != null) {
				SctArgRxText eva = new SctArgRxText();
				eva.text	   = text;
				OnRxText(this, eva);
			}
		}
		//----- コールバック関数（ＥＳＣ受信通知）--------------------------------------------------------------//
		private void cbSctRxEsc(string esc)
		{
			if (OnRxEsc != null) {
				SctArgRxEsc eva = new SctArgRxEsc();
				eva.esc 	   = esc;
				OnRxEsc(this, eva);
			}
		}
		//----- コールバック関数（制御コード受信通知）----------------------------------------------------------//
		private void cbSctRxCtrl(char ctrl)
		{
			if (OnRxCtrl != null) {
				SctArgRxCtrl eva = new SctArgRxCtrl();
				eva.ctrl	   = ctrl;
				OnRxCtrl(this, eva);
			}
		}
		//----- コールバック関数（パケット受信通知）------------------------------------------------------------//
		unsafe private void cbSctRxPacket(void *p, int len)
		{
			if (OnRxPacket != null) {
				SctArgRxPacket eva = new SctArgRxPacket();
				if (len != 0) {
					eva.bin 	= new Byte[len];
					fixed (void *pBArr = &eva.bin[0]) {
						CAjrApi.AjcMemCopy(pBArr, p, len);
					}
				}
				else {
					eva.bin = null;
				}
				OnRxPacket(this, eva);
			}

		}
		//----- コールバック関数（パケット外テキスト受信通知）--------------------------------------------------//
		private void cbSctRxNoPkt(string text)
		{
			if (OnRxNoPkt != null) {
				SctArgRxNoPkt eva = new SctArgRxNoPkt();
				eva.text	   = text;
				OnRxNoPkt(this, eva);
			}
		}
		//----- コールバック関数（送信完了通知）----------------------------------------------------------------//
		private void cbSctTxEmpty()
		{
			if (OnTxEmpty != null) {
				EventArgs eva = new EventArgs();
				OnTxEmpty(this, eva);
			}
		}
		//----- コールバック関数（テキストチャンク受信通知）----------------------------------------------------//
		private void cbSctRxChunkTxt(string text)
		{
			if (OnRxChunkTxt != null) {
				SctArgRxChunkTxt eva = new SctArgRxChunkTxt();
				eva.text	   = text;
				OnRxChunkTxt(this, eva);
			}
		}
		//----- コールバック関数（バイナリチャンク受信通知）----------------------------------------------------//
		unsafe private void cbSctRxChunkBin(void *p, int len)
		{
			if (OnRxChunkBin != null) {
				SctArgRxChunkBin eva = new SctArgRxChunkBin();
				eva.bin = new Byte[len];
				fixed (void *pBArr = &eva.bin[0]) {
					CAjrApi.AjcMemCopy(pBArr, p, len);
				}
				OnRxChunkBin(this, eva);
			}
		}
		//----- コールバック関数（不正チャンクテキスト受信通知）------------------------------------------------//
		unsafe private void cbSctRxInvChunk(void *p, int len)
		{
			if (OnRxInvChunk != null) {
				SctArgRxInvChunk eva = new SctArgRxInvChunk();
				eva.bin 	= new Byte[len];
				fixed (void *pBArr = &eva.bin[0]) {
					CAjrApi.AjcMemCopy(pBArr, p, len);
				}
				OnRxInvChunk(this, eva);
			}
		}
		//----- コールバック関数（接続通知）--------------------------------------------------------------------//
		private void cbSctConnect()
		{
			if (OnConnect != null) {
				EventArgs eva = new EventArgs();
				OnConnect(this, eva);
			}
		}
		//----- コールバック関数（切断通知）--------------------------------------------------------------------//
		private void cbSctDisconnect()
		{
			if (OnDisconnect != null) {
				EventArgs eva = new EventArgs();
				OnDisconnect(this, eva);
			}
		}
		//----- コールバック関数（接続失敗通知）----------------------------------------------------------------//
		private void cbSctCnFail(int err)
		{
			if (OnCnFail != null) {
				SctArgCnFail eva = new SctArgCnFail();
				eva.err = err;
				OnCnFail(this, eva);
			}
		}
		//----- コールバック関数（受信エラー通知）--------------------------------------------------------------//
		private void cbSctRecvError(bool overlapped)
		{
			if (OnRecvError != null) {
				SctArgRecvError eva = new SctArgRecvError();
				eva.overlapped = overlapped;
				OnRecvError(this, eva);
			}
		}
		//----- コールバック関数（送信エラー通知）--------------------------------------------------------------//
		private void cbSctSendError(bool overlapped)
		{
			if (OnSendError != null) {
				SctArgSendError eva = new SctArgSendError();
				eva.overlapped = overlapped;
				OnSendError(this, eva);
			}
		}
		//----- コールバック関数（その他のエラー通知）----------------------------------------------------------//
		private void cbSctGeneralError(ESctErrorCode code)
		{
			if (OnGeneralError != null) {
				SctArgGeneralError eva = new SctArgGeneralError();
				eva.ErrorCode  = code;
				OnGeneralError(this, eva);
			}
		}
	}
}
