﻿using System;
using System.ComponentModel;
using System.Drawing;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.Runtime.InteropServices;
using AjaraInit;

namespace CAjrCustCtrl
{
	//----------------------------------------------------------------------------------------------------------//
	[ToolboxBitmap(typeof(CAjrSockServer), "CAjrSockServer.bmp")]
	public partial class CAjrSockServer : Component
	{
		//----- 外部へのイベント -------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Notify the received text data.")]
		public event SsvEvtRxText			OnRxText;

		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Notifies the received escape sequence.")]
		public event SsvEvtRxEsc			OnRxEsc;

		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Notify the received control code.")]
		public event SsvEvtRxCtrl			OnRxCtrl;

		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Notify the received packet data.")]
		public event SsvEvtRxPacket			OnRxPacket;

		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Notifies the part of the received data other than the packet.")]
		public event SsvEvtRxNoPkt			OnRxNoPkt;

		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Notify the completion of transmission.")]
		public event SsvEvtTxEmpty			OnTxEmpty;

		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Notify incoming chunk text in real time.")]
		public event SsvEvtRxChunkTxt		OnRxChunkTxt;

		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Notify incoming chunk binary data in real time.")]
		public event SsvEvtRxChunkBin		OnRxChunkBin;

		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Notifies that invalid text chunk data has been received.")]
		public event SsvEvtRxInvChunk		OnRxInvChunk;

		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Notifies you that you have connected to a client.")]
		public event SsvEvtConnect			OnConnect;

		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Notifies connection with a client has been disconnected.")]
		public event SsvEvtDisconnect		OnDisconnect;

		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Notifies that the server has started.")]
		public event SsvEvtStart			OnStart;

		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Notifies that the server has stopped.")]
		public event SsvEvtStop				OnStop;

		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Notifies that a reception error has been detected.")]
		public event SsvEvtRecvError		OnRecvError;

		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Notifies that a transmission error has been detected.")]
		public event SsvEvtSendError		OnSendError;

		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Notifies that a general error has been detected.")]
		public event SsvEvtGeneralError		OnGeneralError;

		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Clients enumeration notification.")]
		public event SsvEvtEnumClients		OnEnumClients;

		private SsvCbkRxText				m_SsvCbkRxText;
		private SsvCbkRxEsc					m_SsvCbkRxEsc;
		private SsvCbkRxCtrl				m_SsvCbkRxCtrl;
		private SsvCbkRxPacket				m_SsvCbkRxPacket;
		private SsvCbkRxNoPkt				m_SsvCbkRxNoPkt;
		private SsvCbkTxEmpty				m_SsvCbkTxEmpty;
		private SsvCbkRxChunkTxt			m_SsvCbkRxChunkTxt;
		private SsvCbkRxChunkBin			m_SsvCbkRxChunkBin;
		private SsvCbkRxInvChunk			m_SsvCbkRxInvChunk;
		private SsvCbkConnect				m_SsvCbkConnect;
		private SsvCbkDisconnect			m_SsvCbkDisconnect;
		private SsvCbkStart					m_SsvCbkStart;
		private SsvCbkStop					m_SsvCbkStop;
		private SsvCbkRecvError				m_SsvCbkRecvError;
		private SsvCbkSendError				m_SsvCbkSendError;
		private SsvCbkGeneralError			m_SsvCbkGeneralError;
		private SsvCbkEnumClients			m_SsvCbkEnumClients;

		//----- 実行モード情報 ---------------------------------------------------------------------------------//
		private ESsvMode		m_SsvMode	= ESsvMode.NotificationByEvent;	//	実行モード
		private ESsvChunkMode	m_ChunkMode	= ESsvChunkMode.BinaryData;		//	チャンク種別

		//----- クライアントデータ MAP	------------------------------------------------------------------------//
		private IDictionary<IntPtr, object> m_mapClientData = new Dictionary<IntPtr, object>();

		//----- ワーク -----------------------------------------------------------------------------------------//
		private IntPtr			m_hSsv;
		private	int				m_STX, m_ETX, m_DLE;
		private	int				m_PktTimeout;
		private ESsvRxTextCode	m_RxTextCode =	ESsvRxTextCode.SJIS;
		private ESsvTxTextCode	m_TxTextCode =	ESsvTxTextCode.SJIS;
		private	IntPtr			m_SvLParam	 =	(IntPtr)0;

		//----- コンストラクタ（１） ---------------------------------------------------------------------------//
		public CAjrSockServer()
		{
			InitializeComponent();

			//	初期化
			InitOnConstruct();
		}

		//----- コンストラクタ（２） ---------------------------------------------------------------------------//
		public CAjrSockServer(IContainer container)
		{
			container.Add(this);

			InitializeComponent();

			//	初期化
			InitOnConstruct();

		}
		//----- 初期化コード -----------------------------------------------------------------------------------//
		unsafe private void InitOnConstruct()
		{
			//	ライブラリ初期化
			CAjrInit.Init();

			//	変数初期化
			m_hSsv 			= (IntPtr)0;
			m_STX			= 0x02;
			m_ETX			= 0x03;
			m_DLE			= 0x10;
			m_PktTimeout	= 3000;

			//	 サーバ生成
			m_SsvCbkRxText		= new SsvCbkRxText		(cbSsvRxText		);
			m_SsvCbkRxEsc		= new SsvCbkRxEsc		(cbSsvRxEsc			);
			m_SsvCbkRxCtrl		= new SsvCbkRxCtrl		(cbSsvRxCtrl		);
			m_SsvCbkRxPacket	= new SsvCbkRxPacket	(cbSsvRxPacket		);
			m_SsvCbkRxNoPkt		= new SsvCbkRxNoPkt		(cbSsvRxNoPkt		);
			m_SsvCbkTxEmpty		= new SsvCbkTxEmpty		(cbSsvTxEmpty		);
			m_SsvCbkRxChunkTxt	= new SsvCbkRxChunkTxt	(cbSsvRxChunkTxt	);
			m_SsvCbkRxChunkBin	= new SsvCbkRxChunkBin	(cbSsvRxChunkBin	);
			m_SsvCbkRxInvChunk	= new SsvCbkRxInvChunk	(cbSsvRxInvChunk	);
			m_SsvCbkConnect		= new SsvCbkConnect		(cbSsvConnect		);
			m_SsvCbkDisconnect	= new SsvCbkDisconnect	(cbSsvDisconnect	);
			m_SsvCbkStart		= new SsvCbkStart		(cbSsvStart			);
			m_SsvCbkStop		= new SsvCbkStop		(cbSsvStop			);
			m_SsvCbkRecvError	= new SsvCbkRecvError	(cbSsvRecvError		);
			m_SsvCbkSendError	= new SsvCbkSendError	(cbSsvSendError		);
			m_SsvCbkGeneralError= new SsvCbkGeneralError(cbSsvGeneralError	);
			m_SsvCbkEnumClients	= new SsvCbkEnumClients	(cbSsvEnumClients	);

			m_hSsv = CAjrApi.AjcSsvCreateForMSIL(m_SsvCbkRxText		,
												 m_SsvCbkRxEsc		,
												 m_SsvCbkRxCtrl		,
												 m_SsvCbkRxPacket	,
												 m_SsvCbkRxNoPkt	,
												 m_SsvCbkTxEmpty	,
												 m_SsvCbkRxChunkTxt	,
												 m_SsvCbkRxChunkBin	,
												 m_SsvCbkRxInvChunk	,
												 m_SsvCbkConnect	,
												 m_SsvCbkDisconnect	,
												 m_SsvCbkRecvError	,
												 m_SsvCbkSendError	,
												 m_SsvCbkStart		,
												 m_SsvCbkStop		,
												 m_SsvCbkGeneralError,
												 m_SsvCbkEnumClients);

			//	終了イベント設定
			this.Disposed += OnUnloadMyControl;
		}
		//----- 終了処理 ---------------------------------------------------------------------------------------//
		private void OnUnloadMyControl(object sender, EventArgs e)
		{
			if (m_hSsv != (IntPtr)0) {
				CAjrApi.AjcSsvDeleteForMSIL(m_hSsv);
			}
		}
		//------------------------------------------------------------------------------------------------------//
		//	プロパティ																							//
		//------------------------------------------------------------------------------------------------------//
		//----- 実行モード -------------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Event reception mode" +
										   "(This property can be set only at designing)")]
		public ESsvMode _SsvMode {
			set {m_SsvMode = value;}
			get {return m_SsvMode;}
		}
		//----- サーバオプション -------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Power control options")]
		public ESsvServOpt SsvOption {
			set {CAjrApi.AjcSsvSetOpt(m_hSsv, value);}
			get {return CAjrApi.AjcSsvGetOpt(m_hSsv);}
		}
		//----- チャンクデータの扱い ---------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Kind of reception chunk data" )]
		public ESsvChunkMode ChunkMode {
			set {m_ChunkMode = value; CAjrApi.AjcSsvSetChunkMode(m_hSsv, value);}
			get {return m_ChunkMode;}
		}
		//----- パケット先頭コード(STX) ------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Packet starting code value (STX)")]
		public int STX {
			set {m_STX = (int)value;  CAjrApi.AjcSsvSetPktCtrlCode(m_hSsv, m_STX, 0, 0);}
			get {return (int)m_STX; }
		}
		//----- パケット末尾コード(ETX) ------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Packer ending code value (ETX)")]
		public int ETX {
			set {m_ETX = (int)value;  CAjrApi.AjcSsvSetPktCtrlCode(m_hSsv, 0, m_ETX, 0);}
			get {return (int)m_ETX; }
		}
		//----- パケット透過制御バイトコード(DLE) --------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Packet transparent control code (DLE)")]
		public int DLE {
			set {m_DLE = (int)value;  CAjrApi.AjcSsvSetPktCtrlCode(m_hSsv, 0, 0, m_DLE);}
			get {return (int)m_DLE; }
		}
		//----- パケット受信タイムアウトタイマ -----------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Packet reception timeout [ms]")]
		public int PktTimeout {
			set {m_PktTimeout = (int)value;  CAjrApi.AjcSsvSetPktTimeout(m_hSsv, m_PktTimeout);}
			get {return (int)m_PktTimeout; }
		}
		//----- 受信テキストの文字コード -----------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Received text code")]
		public ESsvRxTextCode RxTextCode {
			set {m_RxTextCode = value;	CAjrApi.AjcSsvSetRxTextCode(m_hSsv, m_RxTextCode);
										CAjrApi.AjcSsvSetTxTextCode(m_hSsv, m_TxTextCode);}
			get {return m_RxTextCode; }
		}
		//----- 送信テキストの文字コード -----------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Sending Text code")]
		public ESsvTxTextCode TxTextCode {
			set {m_TxTextCode = value;	CAjrApi.AjcSsvSetRxTextCode(m_hSsv, m_RxTextCode);
										CAjrApi.AjcSsvSetTxTextCode(m_hSsv, m_TxTextCode);}
			get {return m_TxTextCode; }
		}
		//----- 接続クライアント数 -----------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Number of clients")]
		public int Clients {
			get {	int rc;
				 rc = CAjrApi.AjcSsvGetClientCount(m_hSsv);
				 return rc;
			}
		}
		//----- サーバ起動状態 ---------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrSockServer")]
		[System.ComponentModel.Description("Server Start Status")]
		public bool IsStarted {
			get {	bool rc;
				 rc = CAjrApi.AjcSsvIsStarted(m_hSsv);
				 return rc;
			}
		}
		//------------------------------------------------------------------------------------------------------//
		//	メソッド（クライアント操作）																		//
		//------------------------------------------------------------------------------------------------------//
		//----- １バイト送信 -----------------------------------------------------------------------------------//
		public void SendByte(IntPtr hClient, byte ByteCode)
		{
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				CAjrApi.AjcSsvSendChar(hClient, ByteCode);
			}
			else {throw new ArgumentException("Invalid client handle.");}
		}
		//----- １文字送信 -------------------------------------------------------------------------------------//
		public void SendChar(IntPtr hClient, char Character)
		{
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				CAjrApi.AjcSsvSendChar(hClient, (ushort)Character);
			}
			else {throw new ArgumentException("Invalid client handle.");}
		}
		//----- テキスト送信 -----------------------------------------------------------------------------------//
		public void SendText(IntPtr hClient, string text)
		{
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				CAjrApi.AjcSsvSendText(hClient, text, (int)text.Length);
			}
			else {throw new ArgumentException("Invalid client handle.");}
		}
		//----- バイナリ送信 -----------------------------------------------------------------------------------//
		public void SendBinary(IntPtr hClient, Byte[] bin)
		{
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				CAjrApi.AjcSsvSendBinData(hClient, bin);
			}
			else {throw new ArgumentException("Invalid client handle.");}
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		unsafe public void SendBinary(IntPtr hClient, void *p, int len)
		{
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				CAjrApi.AjcSsvSendBinData(hClient, p, len);
			}
			else {throw new ArgumentException("Invalid client handle.");}
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void SendBinary(IntPtr hClient, IntPtr p, int len)
		{
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				CAjrApi.AjcSsvSendBinData(hClient, p, len);
			}
			else {throw new ArgumentException("Invalid client handle.");}
		}
		//----- パケット送信 -----------------------------------------------------------------------------------//
		public int SendPacket(IntPtr hClient, Byte[] bin)
		{
			int		rc = 0;
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				if (bin != null) {
					rc = CAjrApi.AjcSsvSendPacket(hClient, bin);
				}
				else {
					rc = CAjrApi.AjcSsvSendPacket(hClient, (IntPtr)0, 0);
				}
			}
			else {throw new ArgumentException("Invalid client handle.");}
			return rc;
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		unsafe public int SendPacket(IntPtr hClient, void* p, int len)
		{
			int		rc = 0;
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				rc = CAjrApi.AjcSsvSendPacket(hClient, p, len);
			}
			else {throw new ArgumentException("Invalid client handle.");}
			return rc;
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public int SendPacket(IntPtr hClient, IntPtr p, int len)
		{
			int		rc = 0;
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				rc = CAjrApi.AjcSsvSendPacket(hClient, p, len);
			}
			else {throw new ArgumentException("Invalid client handle.");}
			return rc;
		}
		//----- 受信済データ破棄 -------------------------------------------------------------------------------//
		public void PurgeRx(IntPtr hClient)
		{
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				CAjrApi.AjcSsvPurgeRecvData(hClient);
			}
			else {throw new ArgumentException("Invalid client handle.");}
		}
		//----- 送信待ちデータ破棄 -----------------------------------------------------------------------------//
		public void PurgeTx(IntPtr hClient)
		{
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				CAjrApi.AjcSsvPurgeSendData(hClient);
			}
			else {throw new ArgumentException("Invalid client handle.");}
		}
		//----- 送受信データ破棄 -------------------------------------------------------------------------------//
		public void Purge(IntPtr hClient)
		{
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				CAjrApi.AjcSsvPurgeAllData(hClient);
			}
			else {throw new ArgumentException("Invalid client handle.");}
		}
		//----- クライアントにデータを関連付け -----------------------------------------------------------------//
		public void SetClientData(IntPtr hClient, object obj)
		{
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				if (m_mapClientData.ContainsKey(hClient)) {
					m_mapClientData[hClient] = obj;
				}
				else {
					m_mapClientData.Add(hClient, obj);
				}
			}
			else {throw new ArgumentException("Invalid client handle.");}

		}
		//----- クライアントに関連付けられたデータ取得 ---------------------------------------------------------//
		public void GetClientData(IntPtr hClient, out object obj)
		{
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				if (m_mapClientData.ContainsKey(hClient)) {
					obj = m_mapClientData[hClient];
				}
				else {throw new ArgumentException("Client data is not registered.");}
			}
			else {throw new ArgumentException("Invalid client handle.");}
		}
		//----- クライアント接続順序番号取得 -------------------------------------------------------------------//
		public int GetSeqNo(IntPtr hClient)
		{
			int		rc = 0;
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				rc = CAjrApi.AjcSsvGetSeqNo(hClient);
			}
			else {throw new ArgumentException("Invalid client handle.");}
			return rc;
		}
		//----- クライアントのインデクス値取得 -----------------------------------------------------------------//
		public int GetIndex(IntPtr hClient)
		{
			int		rc = 0;
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				rc = CAjrApi.AjcSsvGetIndex(hClient);
			}
			else {throw new ArgumentException("Invalid client handle.");}
			return rc;
		}
		//----- クライアントのＩＰアドレス文字列取得 -----------------------------------------------------------//
		public string GetIpAddrStr(IntPtr hClient)
		{
			string	rc = null;
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				rc = CAjrApi.AjcSsvGetIpAddrStr(hClient);
			}
			else {throw new ArgumentException("Invalid client handle.");}
			return rc;
		}
		//----- クライアントスレッドのスレッドＩＤ取得 ---------------------------------------------------------//
		public uint GetThreadId(IntPtr hClient)
		{
			uint	rc = 0;
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				rc = CAjrApi.AjcSsvGetThreadId(hClient);
			}
			else {throw new ArgumentException("Invalid client handle.");}
			return rc;
		}
		//----- クライアント切断 -------------------------------------------------------------------------------//
		public void Disconnect(IntPtr hClient)
		{
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				CAjrApi.AjcSsvDisconnect(hClient);
			}
			else {throw new ArgumentException("Invalid client handle.");}
		}
		//----- 実際の受信テキストの文字コード（AUTO時は実際の受信文字コード） ---------------------------------//
		public ESsvRxTextCode GetActualRxTextCode(IntPtr hClient)
		{
			ESsvRxTextCode	rc = 0;
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				rc = CAjrApi.AjcSsvGetActualRxTextCode(hClient);
			}
			else {throw new ArgumentException("Invalid client handle.");}
			return rc;
		}
		//----- 実際の送信テキストの文字コード（AUTO時は受信文字コード） ---------------------------------------//
		public ESsvTxTextCode GetActualTxTextCode(IntPtr hClient)
		{
			ESsvTxTextCode	rc = 0;
			if (CAjrApi.AjcSsvIsClientValid(m_hSsv, hClient)) {
				rc = CAjrApi.AjcSsvGetActualTxTextCode(hClient);
			}
			else {throw new ArgumentException("Invalid client handle.");}
			return rc;
		}
		//------------------------------------------------------------------------------------------------------//
		//	メソッド（その他）																					//
		//------------------------------------------------------------------------------------------------------//
		//----- サーバ開始 -------------------------------------------------------------------------------------//
		public void Start(uint PortNo)
		{
			Start(PortNo, ESsvFamily.INET, 10);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void Start(uint PortNo, int MaxClients)
		{
			Start(PortNo, ESsvFamily.INET, MaxClients);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void Start(uint PortNo, ESsvFamily AddressFamily, int MaxClients)
		{
			//	イベント設定（OnXXXXX イベント)
			if (m_SsvMode == ESsvMode.NotificationByEvent) {
				SetEvtMaskByEvent();
			}
			//	サーバ開始
			CAjrApi.AjcSsvStartForMSIL(m_hSsv, PortNo.ToString(), AddressFamily, MaxClients, m_SsvMode == ESsvMode.NotificationByEvent ? true : false);
		}
		//----- サーバ停止 -------------------------------------------------------------------------------------//
		public void Stop()
		{
			CAjrApi.AjcSsvStop(m_hSsv, 100000);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void Stop(int msTimeout)
		{
			CAjrApi.AjcSsvStop(m_hSsv, msTimeout);
		}
		//----- クライアント列挙 -------------------------------------------------------------------------------//
		public	int EnumClients(IntPtr cbp)
		{
			return CAjrApi.AjcSsvEnumClientsForMSIL(m_hSsv, cbp);
		}
		//----- ＳＳＶ消去 -------------------------------------------------------------------------------------//
		public void Delete() {
			if (m_hSsv != (IntPtr)0) {
				CAjrApi.AjcSsvDeleteForMSIL(m_hSsv);
				m_hSsv = (IntPtr)0;
			}
		}
		//----- イベントマスク設定 -----------------------------------------------------------------------------//
		public void SetEvtMask(ESsvEvt evt)
		{
			if (m_SsvMode == ESsvMode.WaitingForEvent) {	//	コンソールアプリ？
				CAjrApi.AjcSsvSetEvtMask(m_hSsv, (int)evt);
			}
		}
		//----- イベントマスク取得 -----------------------------------------------------------------------------//
		public ESsvEvt GetEvtMask()
		{
			ESsvEvt	rc = 0;
			rc = (ESsvEvt)CAjrApi.AjcSsvGetEvtMask(m_hSsv);
			return rc;
		}
		//----- イベント待ち -----------------------------------------------------------------------------------//
		public ESsvEvt WaitEvent(out Object EvtData, int msWaitTime, out IntPtr hClient)
		{
			IntPtr	wParam	= (IntPtr)0;
			IntPtr	lParam	= (IntPtr)0;
			IntPtr	pDat;
			int		len, param;
			ESsvEvt	evt 	= ESsvEvt.EV_NOEVENT;
			IntPtr	pC		= (IntPtr)0;

			//	 出力情報クリアー
			EvtData = null; 			//	イベントデータ
			hClient = (IntPtr)0;		//	クライアントハンドル

			if (m_SvLParam == (IntPtr)0) {
				if (CAjrApi.AjcSsvWaitEvent(m_hSsv, ref wParam, ref lParam, msWaitTime)) {
					//	lParam 退避
					m_SvLParam = lParam;
					//	イベントデータ取得
					CAjrApi.AjcSsvGetEventData(m_hSsv, lParam, out pDat, out len, out param);
					//	クライアント・ハンドル取得
					if (CAjrApi.AjcSsvGetClient(m_hSsv, lParam, ref pC)) {
						hClient = pC;
					}
					//	イベントコード設定
					evt = (ESsvEvt)wParam;

					//----- ●テキスト受信通知 ---------------------------------------------------------------------//
					if		((evt & ESsvEvt.EV_RXTEXT) != 0) {
						StringBuilder s = new StringBuilder(len + 1);
						CAjrApi.AjcWStrCopy(s, pDat);
						EvtData 	= s.ToString();
						hClient 	= pC;
					}
					//----- ●ＥＳＣコード受信通知 -----------------------------------------------------------------//
					else if ((evt & ESsvEvt.EV_RXESC) != 0) {
						StringBuilder s = new StringBuilder(len + 1);
						CAjrApi.AjcWStrCopy(s, pDat);
						EvtData 	= s.ToString();
						hClient 	= pC;
					}
					//----- ●制御コード受信通知 -------------------------------------------------------------------//
					else if ((evt & ESsvEvt.EV_RXCTRL) != 0) {
						EvtData 	= CAjrApi.AjcMemPeekUW(pDat);
						hClient 	= pC;
					}
					//----- ●パケットデータ受信通知 ---------------------------------------------------------------//
					else if ((evt & ESsvEvt.EV_RXPKT) != 0) {
						Byte[] BArr = new Byte[len];
						CAjrApi.AjcMemCopy(ref BArr[0], pDat, len);
						EvtData 	= BArr;
						hClient 	= pC;
					}
					//----- ●パケット外データ受信通知 -------------------------------------------------------------//
					else if ((evt & ESsvEvt.EV_RXNOPKT) != 0) {
						StringBuilder s = new StringBuilder(len + 1);
						CAjrApi.AjcWStrCopy(s, pDat);
						EvtData 	= s.ToString();
						hClient 	= pC;
					}
					//----- ●チャンクデータ受信通知 ---------------------------------------------------------------//
   	   	   	   		else if ((evt & ESsvEvt.EV_RXCHUNK) != 0) {
						if (param == 0) {
							Byte[] BArr = new Byte[len];
							CAjrApi.AjcMemCopy(ref BArr[0], pDat, len);
							EvtData 	= BArr;
						}
						else {
							StringBuilder s = new StringBuilder(len + 1);
							CAjrApi.AjcWStrCopy(s, pDat);
							EvtData = s.ToString();
						}
						hClient 	= pC;
					}
					//----- ●不正チャンクテキスト受信通知 ---------------------------------------------------------//
					else if ((evt & ESsvEvt.EV_INVCHUNK) != 0) {
						Byte[] BArr = new Byte[len];
						CAjrApi.AjcMemCopy(ref BArr[0], pDat, len);
						EvtData 	= BArr;
						hClient 	= pC;
					}
					//----- ●送信完了通知 -------------------------------------------------------------------------//
					else if ((evt & ESsvEvt.EV_TXEMPTY) != 0) {
						EvtData 	= null;
						hClient 	= pC;
					}
					//----- ●接続通知 -----------------------------------------------------------------------------//
					else if ((evt & ESsvEvt.EV_CONNECT) != 0) {
						EvtData 	= null;
						hClient 	= pC;
					}
					//----- ●切断通知 -----------------------------------------------------------------------------//
					else if ((evt & ESsvEvt.EV_DISCONNECT) != 0) {
						EvtData 	= null;
						if (m_mapClientData.ContainsKey(hClient)) {
							EvtData = m_mapClientData[hClient];
							m_mapClientData.Remove(hClient);
						}
						hClient 	= pC;
					}
					//----- ●サーバ開始通知 -----------------------------------------------------------------------//
					else if ((evt & ESsvEvt.EV_START) != 0) {
						EvtData 	= null;
						m_mapClientData.Clear();
					}
					//----- ●サーバ停止通知 -----------------------------------------------------------------------//
					else if ((evt & ESsvEvt.EV_STOP) != 0) {
						EvtData 	= null;
						m_mapClientData.Clear();
					}
					//----- ●その他 -------------------------------------------------------------------------------//
					else  {
						EvtData 	= param;
					}
				}
				else {	//	タイムアウト
					m_SvLParam = (IntPtr)0;
				}
			}
			else {		//	m_SvLParam != 0
				{throw new InvalidOperationException("EndEvent() is not executed after WaitEvent().");}
			}
			return evt;
		}
		//----- イベントの終了 ---------------------------------------------------------------------------------//
		public void EndEvent()
		{
			if (m_SvLParam != (IntPtr)0) {
				CAjrApi.AjcSsvRelEventData(m_hSsv, m_SvLParam);
				m_SvLParam = (IntPtr)0;
			}
		}
		//--------------------------------------------------------------------------------------------------//
		//	内部サブ関数																					//
		//--------------------------------------------------------------------------------------------------//
		//----- SetEvtMaskByEvent		   -----------------------------------------------------------------//
		private bool SetEvtMaskByEvent()
		{
			bool	rc;
			ESsvEvt	evt = 0;

			if (OnRxText		 != null) evt |= ESsvEvt.EV_RXTEXT;			//	テキストデータ通知
			if (OnRxEsc			 != null) evt |= ESsvEvt.EV_RXESC;			//	ＥＳＣコードデータ通知
			if (OnRxCtrl		 != null) evt |= ESsvEvt.EV_RXCTRL;			//	制御コード通知
			if (OnRxPacket		 != null) evt |= ESsvEvt.EV_RXPKT;			//	パケットデータ通知
			if (OnRxNoPkt 		 != null) evt |= ESsvEvt.EV_RXNOPKT;		//	パケット外テキストデータ
			if (OnRxChunkTxt	 != null) evt |= ESsvEvt.EV_RXCHUNK;		//	パケットデータ通知
			if (OnRxChunkBin	 != null) evt |= ESsvEvt.EV_RXCHUNK;		//	パケットデータ通知
			if (OnRxInvChunk	 != null) evt |= ESsvEvt.EV_INVCHUNK;		//	チャンクデータ受信通知
			if (OnTxEmpty 		 != null) evt |= ESsvEvt.EV_TXEMPTY;		//	送信完了通知
			if (OnConnect 		 != null) evt |= ESsvEvt.EV_CONNECT;		//	不正チャンクテキスト受信通知
			if (OnDisconnect	 != null) evt |= ESsvEvt.EV_DISCONNECT;		//	切断通知
			if (OnStart			 != null) evt |= ESsvEvt.EV_START;			//	サーバ開始通知
			if (OnStop			 != null) evt |= ESsvEvt.EV_STOP;			//	サーバ停止通知
			if (OnRecvError		 != null) evt |= ESsvEvt.EV_RXERR;			//	受信エラー通知
			if (OnSendError		 != null) evt |= ESsvEvt.EV_TXERR;			//	送信エラー通知
			if (OnGeneralError	 != null) evt |= ESsvEvt.EV_ERR;			//	エラー通知

			rc = CAjrApi.AjcSsvSetEvtMask(m_hSsv, (int)evt);
			return rc;
		}
		//------------------------------------------------------------------------------------------------------//
		//	コールバック																						//
		//------------------------------------------------------------------------------------------------------//
		//----- コールバック関数（テキスト受信通知）------------------------------------------------------------//
		private void cbSsvRxText(IntPtr hClient, string text)
		{
			if (OnRxText != null) {
				SsvArgRxText eva = new SsvArgRxText();
				eva.hClient    = hClient;
				eva.text	   = text;
				OnRxText(this, eva);
			}
		}
		//----- コールバック関数（ＥＳＣ受信通知）--------------------------------------------------------------//
		private void cbSsvRxEsc(IntPtr hClient, string esc)
		{
			if (OnRxEsc != null) {
				SsvArgRxEsc eva = new SsvArgRxEsc();
				eva.hClient    = hClient;
				eva.esc 	   = esc;
				OnRxEsc(this, eva);
			}
		}
		//----- コールバック関数（制御コード受信通知）----------------------------------------------------------//
		private void cbSsvRxCtrl(IntPtr hClient, char ctrl)
		{
			if (OnRxCtrl != null) {
				SsvArgRxCtrl eva = new SsvArgRxCtrl();
				eva.hClient    = hClient;
				eva.ctrl	   = ctrl;
				OnRxCtrl(this, eva);
			}
		}
		//----- コールバック関数（パケット受信通知）------------------------------------------------------------//
		unsafe private void cbSsvRxPacket(IntPtr hClient, void *p, int len)
		{
			if (OnRxPacket != null) {
				SsvArgRxPacket eva = new SsvArgRxPacket();
				if (len != 0) {
					eva.hClient = hClient;
					eva.bin 	= new Byte[len];
					fixed (void *pBArr = &eva.bin[0]) {
						CAjrApi.AjcMemCopy(pBArr, p, len);
					}
				}
				else {
					eva.bin = null;
				}
				OnRxPacket(this, eva);
			}

		}
		//----- コールバック関数（パケット外テキスト受信通知）--------------------------------------------------//
		private void cbSsvRxNoPkt(IntPtr hClient, string text)
		{
			if (OnRxNoPkt != null) {
				SsvArgRxNoPkt eva = new SsvArgRxNoPkt();
				eva.hClient    = hClient;
				eva.text	   = text;
				OnRxNoPkt(this, eva);
			}
		}
		//----- コールバック関数（送信完了通知）----------------------------------------------------------------//
		private void cbSsvTxEmpty(IntPtr hClient)
		{
			if (OnTxEmpty != null) {
				SsvArgTxEmpty eva = new SsvArgTxEmpty();
				eva.hClient    = hClient;
				OnTxEmpty(this, eva);
			}
		}
		//----- コールバック関数（テキストチャンク受信通知）----------------------------------------------------//
		private void cbSsvRxChunkTxt(IntPtr hClient, string text)
		{
			if (OnRxChunkTxt != null) {
				SsvArgRxChunkTxt eva = new SsvArgRxChunkTxt();
				eva.hClient    = hClient;
				eva.text	   = text;
				OnRxChunkTxt(this, eva);
			}
		}
		//----- コールバック関数（バイナリチャンク受信通知）----------------------------------------------------//
		unsafe private void cbSsvRxChunkBin(IntPtr hClient, void *p, int len)
		{
			if (OnRxChunkBin != null) {
				SsvArgRxChunkBin eva = new SsvArgRxChunkBin();
				eva.hClient    = hClient;
				eva.bin = new Byte[len];
				fixed (void *pBArr = &eva.bin[0]) {
					CAjrApi.AjcMemCopy(pBArr, p, len);
				}
				OnRxChunkBin(this, eva);
			}
		}
		//----- コールバック関数（不正チャンクテキスト受信通知）------------------------------------------------//
		unsafe private void cbSsvRxInvChunk(IntPtr hClient, void *p, int len)
		{
			if (OnRxInvChunk != null) {
				SsvArgRxInvChunk eva = new SsvArgRxInvChunk();
				eva.hClient = hClient;
				eva.bin 	= new Byte[len];
				fixed (void *pBArr = &eva.bin[0]) {
					CAjrApi.AjcMemCopy(pBArr, p, len);
				}
				OnRxInvChunk(this, eva);
			}
		}
		//----- コールバック関数（接続通知）--------------------------------------------------------------------//
		private void cbSsvConnect(IntPtr hClient)
		{
			if (OnConnect != null) {
				SsvArgConnect eva = new SsvArgConnect();
				eva.hClient    = hClient;
				OnConnect(this, eva);
			}
		}
		//----- コールバック関数（切断通知）--------------------------------------------------------------------//
		private void cbSsvDisconnect(IntPtr hClient)
		{
			if (OnDisconnect != null) {
				SsvArgDisconnect eva = new SsvArgDisconnect();
				eva.hClient    = hClient;
				OnDisconnect(this, eva);
			}
		}
		//----- コールバック関数（受信エラー通知）--------------------------------------------------------------//
		private void cbSsvRecvError(IntPtr hClient, bool overlapped)
		{
			if (OnRecvError != null) {
				SsvArgRecvError eva = new SsvArgRecvError();
				eva.hClient    = hClient;
				eva.overlapped = overlapped;
				OnRecvError(this, eva);
			}
		}
		//----- コールバック関数（送信エラー通知）--------------------------------------------------------------//
		private void cbSsvSendError(IntPtr hClient, bool overlapped)
		{
			if (OnSendError != null) {
				SsvArgSendError eva = new SsvArgSendError();
				eva.hClient    = hClient;
				eva.overlapped = overlapped;
				OnSendError(this, eva);
			}
		}
		//----- コールバック関数（その他のエラー通知）----------------------------------------------------------//
		private void cbSsvGeneralError(ESsvErrorCode code)
		{
			if (OnGeneralError != null) {
				SsvArgGeneralError eva = new SsvArgGeneralError();
				eva.ErrorCode  = code;
				OnGeneralError(this, eva);
			}
		}
		//----- コールバック関数（サーバ開始通知）--------------------------------------------------------------//
		private void cbSsvStart()
		{
			if (OnDisconnect != null) {
				SsvArgStart eva = new SsvArgStart();
				OnStart(this, eva);
			}
		}
		//----- コールバック関数（サーバ停止通知）--------------------------------------------------------------//
		private void cbSsvStop()
		{
			if (OnStop != null) {
				SsvArgStop eva = new SsvArgStop();
				OnStop(this, eva);
			}
		}
		//----- コールバック関数（クライアント列挙）------------------------------------------------------------//
		private bool cbSsvEnumClients(IntPtr hClient, IntPtr cbp)
		{
			bool	rc = false;

			if (OnEnumClients != null) {
				SsvArgEnumClients eva = new SsvArgEnumClients();
				eva.hClient  = hClient;
				eva.param	 = cbp;
				rc = OnEnumClients(this, eva);
			}
			return rc;
		}
	}
}
