﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Runtime.InteropServices;
using System.IO;
using AjaraInit;

namespace CAjrCustCtrl
{
	[ToolboxBitmap(typeof(CAjrTextFile), "CAjrTextFile.bmp")]
	public partial class CAjrTextFile : Component
	{
		//======================================================================================================//
		//	コンストラクタ																						//
		//======================================================================================================//
		//	コンストラクタ（１）
		public CAjrTextFile()
		{
			InitializeComponent();
			InitOnConstruct();
		}
		//	コンストラクタ（２）
		public CAjrTextFile(IContainer container)
		{
			container.Add(this);
			InitializeComponent();
			InitOnConstruct();
		}
		//	ライブラリ初期化
		private void InitOnConstruct()
		{
			CAjrInit.Init();
		}
		//======================================================================================================//
		//	ワーク																								//
		//======================================================================================================//
		private IntPtr			m_hFile 	= (IntPtr)0;
		private	ETextEncode		m_EncodeR	= ETextEncode.TEC_MBC;
		private	ETextEncode		m_EncodeW	= ETextEncode.TEC_MBC;
		private	EBomMode		m_BOM		= EBomMode.NOT_WRITE_BOM;
		private	ETextLfConv		m_LfConvR	= ETextLfConv.LF_TO_CRLF;
		private	ETextLfConv		m_LfConvW	= ETextLfConv.CRLF_TO_LF;
		private FileStream		m_FileStream= null;

		//======================================================================================================//
		//	プロパティ																							//
		//======================================================================================================//
		[System.ComponentModel.Category("AjrTextFile")]
		[System.ComponentModel.Description("Text encode at reading")]
		public ETextEncode TextEncodeAtRead
		{
			set {
					if (value != ETextEncode.TEC_MBC		&&
						value != ETextEncode.TEC_UTF_8		&&
						value != ETextEncode.TEC_EUC_J		&&
						value != ETextEncode.TEC_UTF_16LE	&&
						value != ETextEncode.TEC_UTF_16BE	&&
						value != ETextEncode.TEC_AUTO) {
						throw new ArgumentException("Invalid property TextEncodeAtRead=" + ((int)value).ToString() + ".");
					}
					m_EncodeR = value;
				}

			get {
					ETextEncode rc;
					if (m_hFile != (IntPtr)0 && CAjrApi.AjcFIsModeInput(m_hFile)) {
						rc = CAjrApi.AjcFGetTec(m_hFile);
					}
					else rc = m_EncodeR;
					return rc;
				}
		}

		[System.ComponentModel.Category("AjrTextFile")]
		[System.ComponentModel.Description("Text encode at writing.\nIf 'TEC_AUTO' is specified, it will be the text encoding of the last opened file.")]
		public ETextEncode TextEncodeAtWrite
		{
			set {
					if (value != ETextEncode.TEC_MBC		&&
						value != ETextEncode.TEC_UTF_8		&&
						value != ETextEncode.TEC_EUC_J		&&
						value != ETextEncode.TEC_UTF_16LE	&&
						value != ETextEncode.TEC_UTF_16BE	&&
						value != ETextEncode.TEC_AUTO) {
						throw new ArgumentException("Invalid property TextEncodeAtWrite=" + ((int)value).ToString() + ".");
					}
					m_EncodeW = value;
				}

			get {
					ETextEncode rc;
					if (m_hFile != (IntPtr)0 && !CAjrApi.AjcFIsModeInput(m_hFile)) {
						rc = CAjrApi.AjcFGetTec(m_hFile);
					}
					else rc = m_EncodeW;
					return rc;
				}
		}

		[System.ComponentModel.Category("AjrTextFile")]
		[System.ComponentModel.Description("BOM output at writing")]
		public EBomMode BOM
		{
			set {
					if (value != EBomMode.NOT_WRITE_BOM &&
						value != EBomMode.WRITE_BOM) {
						throw new ArgumentException("Invalid property BOM=" + ((int)value).ToString() + ".");
					}
					m_BOM = value;
				}

			get {
					return m_BOM;
				}
		}

		[System.ComponentModel.Category("AjrTextFile")]
		[System.ComponentModel.Description("Line feed code conversion mode when file reading a text file.")]
		public ETextLfConv TextLfConvAtRead
		{
			set {
					if (value != ETextLfConv.NONE		&&
						value != ETextLfConv.LF_TO_CRLF	&&
						value != ETextLfConv.LF_TO_CR	&&
						value != ETextLfConv.CR_TO_CRLF	&&
						value != ETextLfConv.CR_TO_LF	&&
						value != ETextLfConv.CRLF_TO_LF	&&
						value != ETextLfConv.CRLF_TO_CR	&&
						value != ETextLfConv.CR			&&
						value != ETextLfConv.LF			&&
						value != ETextLfConv.CR_LFSKIP	&&
						value != ETextLfConv.LF_CRSKIP) {
						throw new ArgumentException("Invalid property TextLfConvAtRead=" + ((int)value).ToString() + ".");
					}
					m_LfConvR = value;
				}

			get {
				return m_LfConvR;
				}
		}

		[System.ComponentModel.Category("AjrTextFile")]
		[System.ComponentModel.Description("Line feed code conversion mode when file writing a text file.")]
		public ETextLfConv TextLfConvAtWrite
		{
			set {
					if (value != ETextLfConv.NONE		&&
						value != ETextLfConv.LF_TO_CRLF	&&
						value != ETextLfConv.LF_TO_CR	&&
						value != ETextLfConv.CR_TO_CRLF	&&
						value != ETextLfConv.CR_TO_LF	&&
						value != ETextLfConv.CRLF_TO_LF	&&
						value != ETextLfConv.CRLF_TO_CR	&&
						value != ETextLfConv.LF 		&&
						value != ETextLfConv.CR			&&
						value != ETextLfConv.CR_LFSKIP	&&
						value != ETextLfConv.LF_CRSKIP) {
						throw new ArgumentException("Invalid property TextLfConvAtWrite=" + ((int)value).ToString() + ".");
					}
					m_LfConvW = value;
			}
			get {
				return m_LfConvW;
			}
		}
		//======================================================================================================//
		//	メソッド																							//
		//======================================================================================================//
		//----- Open					------------------------------------------------------------------------//
		public void Open(string FilePath)
		{
			Open(FilePath, m_EncodeR);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void Open(string FilePath, ETextEncode TextEncode)
		{
			if (m_hFile == (IntPtr)0) {
				if ((m_hFile = CAjrApi.AjcFOpen(FilePath, TextEncode)) != (IntPtr)0) {
					CAjrApi.AjcFSetLfConv(m_hFile, m_LfConvR);
				}
				else {
					throw new FileOpenFailureException(FilePath);
				}
			}
			else throw new InvalidOperationException("File already open/created.");
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void Open(string FilePath, ETextEncode TextEncode, System.IO.FileShare share)
		{
			if (m_hFile == (IntPtr)0) {
				if ((m_hFile = CAjrApi.AjcFOpenShare(FilePath, TextEncode, share)) != (IntPtr)0) {
					CAjrApi.AjcFSetLfConv(m_hFile, m_LfConvR);
				}
				else {
					throw new FileOpenFailureException(FilePath);
				}
			}
			else throw new InvalidOperationException("File already open/created.");
		}
		//----- GetS					------------------------------------------------------------------------//
		public string GetS()
		{
			string rc = "";

			if (m_hFile != (IntPtr)0 && CAjrApi.AjcFIsModeInput(m_hFile)) {
				rc = CAjrApi.AjcFGetS(m_hFile, 1024);
			}
			else throw new InvalidOperationException("File is not open.");

			return rc;
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public string GetS(int MaxLength)
		{
			string rc = "";

			if (m_hFile != (IntPtr)0 && CAjrApi.AjcFIsModeInput(m_hFile)) {
				rc = CAjrApi.AjcFGetS(m_hFile, MaxLength);
			}
			else throw new InvalidOperationException("File is not open.");

			return rc;
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public bool GetS(IntPtr pBuf, int lBuf)
		{
			bool rc = false;

			if (m_hFile != (IntPtr)0 && CAjrApi.AjcFIsModeInput(m_hFile)) {
				rc = CAjrApi.AjcFGetS(m_hFile, pBuf, lBuf);
			}
			else throw new InvalidOperationException("File is not open.");

			return rc;
		}

		//----- GetC					------------------------------------------------------------------------//
		public char GetC()
		{
			char rc = '\xFFFF';

			if (m_hFile != (IntPtr)0 && CAjrApi.AjcFIsModeInput(m_hFile)) {
				rc = CAjrApi.AjcFGetC(m_hFile);
			}
			else throw new InvalidOperationException("File is not open.");

			return rc;
		}
		//----- SavePoint				------------------------------------------------------------------------//
		public long SavePoint()
		{
			long rc = -1;
			if (m_hFile != (IntPtr)0 && CAjrApi.AjcFIsModeInput(m_hFile)) {
				rc = CAjrApi.AjcFSavePoint(m_hFile);
			}
			else throw new InvalidOperationException("File is not open.");

			return rc;
		}
		//----- RecvPoint				------------------------------------------------------------------------//
		public void RecvPoint()
		{
			if (m_hFile != (IntPtr)0 && CAjrApi.AjcFIsModeInput(m_hFile)) {
				CAjrApi.AjcFRecvPoint(m_hFile);
			}
			else throw new InvalidOperationException("File is not open.");
		}
		//----- GetPoint				------------------------------------------------------------------------//
		public long GetPoint()
		{
			long rc = -1;
			if (m_hFile != (IntPtr)0 && CAjrApi.AjcFIsModeInput(m_hFile)) {
				rc = CAjrApi.AjcFGetPoint(m_hFile);
			}
			else throw new InvalidOperationException("File is not open.");
			return rc;
		}
		//----- SetPoint				------------------------------------------------------------------------//
		public void SetPoint(long point)
		{
			if (m_hFile != (IntPtr)0 && CAjrApi.AjcFIsModeInput(m_hFile)) {
				CAjrApi.AjcFSetPoint(m_hFile, point);
			}
			else throw new InvalidOperationException("File is not open.");
		}
		//----- GetBytePoint				------------------------------------------------------------------------//
		public long GetBytePoint()
		{
			long rc = -1;
			if (m_hFile != (IntPtr)0 && CAjrApi.AjcFIsModeInput(m_hFile)) {
				rc = CAjrApi.AjcFGetBytePoint(m_hFile);
			}
			else throw new InvalidOperationException("File is not open.");
			return rc;
		}
		//----- IsEof					------------------------------------------------------------------------//
		public void IsEof()
		{
			if (m_hFile != (IntPtr)0 && CAjrApi.AjcFIsModeInput(m_hFile)) {
				CAjrApi.AjcFEof(m_hFile);
			}
			else throw new InvalidOperationException("File is not open.");
		}
		//----- Create					------------------------------------------------------------------------//
		public void Create(string FilePath)
		{
			Create(FilePath, (ETextEncode)m_EncodeW, m_BOM);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void Create(string FilePath, ETextEncode TextEncode)
		{
			Create(FilePath, TextEncode, m_BOM);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void Create(string FilePath, ETextEncode TextEncode, EBomMode BomMode)
		{
			if (m_hFile == (IntPtr)0) {
				if ((m_hFile = CAjrApi.AjcFCreate(FilePath, TextEncode, BomMode)) == (IntPtr)0) {
					throw new FileCreationFailureException(FilePath);
				}
				CAjrApi.AjcFSetLfConv(m_hFile, m_LfConvW);
			}
			else throw new InvalidOperationException("File is not closed.");
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void Create(string FilePath, ETextEncode TextEncode, EBomMode BomMode, System.IO.FileShare  share)
		{
			if (m_hFile == (IntPtr)0) {
				if ((m_hFile = CAjrApi.AjcFCreateShare(FilePath, TextEncode, BomMode, share)) == (IntPtr)0) {
					throw new FileCreationFailureException(FilePath);
				}
				CAjrApi.AjcFSetLfConv(m_hFile, m_LfConvW);
			}
			else throw new InvalidOperationException("File is not closed.");
		}
		//----- Append					------------------------------------------------------------------------//
		public void Append(string FilePath)
		{
			Append(FilePath, (ETextEncode)m_EncodeW, m_BOM);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void Append(string FilePath, ETextEncode TextEncode)
		{
			Append(FilePath, TextEncode, m_BOM);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void Append(string FilePath, ETextEncode TextEncode, EBomMode BomMode)
		{
			if (m_hFile == (IntPtr)0) {
				if ((m_hFile = CAjrApi.AjcFAppend(FilePath, TextEncode, BomMode)) == (IntPtr)0) {
					throw new FileCreationFailureException(FilePath);
				}
				CAjrApi.AjcFSetLfConv(m_hFile, m_LfConvW);
			}
			else throw new InvalidOperationException("File is not closed.");
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void Append(string FilePath, ETextEncode TextEncode, EBomMode BomMode, System.IO.FileShare  share)
		{
			if (m_hFile == (IntPtr)0) {
				if ((m_hFile = CAjrApi.AjcFAppendShare(FilePath, TextEncode, BomMode, share)) == (IntPtr)0) {
					throw new FileCreationFailureException(FilePath);
				}
				CAjrApi.AjcFSetLfConv(m_hFile, m_LfConvW);
			}
			else throw new InvalidOperationException("File is not closed.");
		}
		//----- PutS					------------------------------------------------------------------------//
		public void PutS(string str)
		{
			PutS(str, str.Length);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void PutS(string str, int len)
		{
			if (m_hFile != (IntPtr)0 && !CAjrApi.AjcFIsModeInput(m_hFile)) {
				if (!CAjrApi.AjcFPutS(m_hFile, str, len)) {
					throw new FileAccessFailureException("File writing failure.");
				}
			}
			else throw new InvalidOperationException("File is not created.");
		}
		//----- PutC					------------------------------------------------------------------------//
		public void PutC(char c)
		{
			if (m_hFile != (IntPtr)0 && !CAjrApi.AjcFIsModeInput(m_hFile)) {
				if (!CAjrApi.AjcFPutC(m_hFile, c)) {
					throw new FileAccessFailureException("File writing failure.");
				}
			}
			else throw new InvalidOperationException("File is not created.");
		}
		//----- PutFormat				------------------------------------------------------------------------//
		public bool PutFormat(string format, params object[] args)
		{
			bool   rc = false;
			string s;

			if (m_hFile != (IntPtr)0 && !CAjrApi.AjcFIsModeInput(m_hFile)) {
				s = CAjrApi.MakeFormatString(format, args);
				if (!CAjrApi.AjcFPutS(m_hFile, s, s.Length)) {
					throw new FileAccessFailureException("File writing failure.");
				}
			}
			else throw new InvalidOperationException("File is not created.");

			return rc;
		}
		//----- Flush					------------------------------------------------------------------------//
		public void Flush()
		{
			if (m_hFile != (IntPtr)0 && !CAjrApi.AjcFIsModeInput(m_hFile)) {
				if (!CAjrApi.AjcFFlush(m_hFile)) {
					throw new FileAccessFailureException("File flush failure.");
				}
			}
			else throw new InvalidOperationException("File is not created.");
		}
		//----- Close					------------------------------------------------------------------------//
		public void Close()
		{
			//	通常のクローズ
			if (m_FileStream == null) {
				if (m_hFile != (IntPtr)0) {
					if (CAjrApi.AjcFClose(m_hFile)) {
						m_hFile = (IntPtr)0;
					}
					else throw new FileCloseFailureException("File close failure.");
				}
				else throw new InvalidOperationException("File is not open or not created.");
			}
			//	ファイルストリームによるクローズ
			else {
				//	ファイルストリームクローズ，破棄
				m_FileStream.Close();
				m_FileStream.Dispose();
				m_FileStream = null;
				//	テキストファイルアクセスへのアタッチ解除
				if (m_hFile != (IntPtr)0) {
					if (CAjrApi.AjcFDetach(m_hFile)) {
						m_hFile = (IntPtr)0;
					}
					else throw new FileCloseFailureException("File close failure.");
				}
				else throw new InvalidOperationException("File is not open or not created.");
			}
		}
		//----- AttachOpenedFile				----------------------------------------------------------------//
		public void AttachOpenedFile(IntPtr FileHandle, ETextEncode tec)
		{
			if (m_hFile == (IntPtr)0) {
				if ((m_hFile = CAjrApi.AjcFAttachOpened(FileHandle, tec)) == (IntPtr)0) {
					throw new FileAccessFailureException("File attach opened file  failure.");
				}
				CAjrApi.AjcFSetLfConv(m_hFile, m_LfConvW);
			}
			else throw new InvalidOperationException("File is activated.");
		}
		//----- AttachCreatedFile				----------------------------------------------------------------//
		public void AttachCreatedFile(IntPtr FileHandle, ETextEncode tec, bool fBOM)
		{
			if (m_hFile == (IntPtr)0) {
				if ((m_hFile = CAjrApi.AjcFAttachCreated(FileHandle, tec, fBOM)) == (IntPtr)0) {
					throw new FileAccessFailureException("File attach created file  failure.");
				}
				CAjrApi.AjcFSetLfConv(m_hFile, m_LfConvW);
			}
			else throw new InvalidOperationException("File is activated.");
		}
		//----- Detach					------------------------------------------------------------------------//
		public void Detach()
		{
			if (m_hFile != (IntPtr)0) {
				if (CAjrApi.AjcFDetach(m_hFile)) {
					m_hFile = (IntPtr)0;
				}
				else throw new FileCloseFailureException("File detach failure.");
			}
			else throw new InvalidOperationException("File is not open or not created.");
		}
	}
}
