﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Drawing.Design;
using System.Windows.Forms.Design;
using AjaraInit;

namespace CAjrCustCtrl
{
	//----------------------------------------------------------------------------------------------------------//
	[ToolboxBitmap(typeof(CAjrTimeChart), "CAjrTimeChart.bmp")]
	public partial class CAjrTimeChart : UserControl
	{
		//----- 外部へのイベント --------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Occurs when the range changes.")]
		public event TchEvtRangeChanged OnRangeChanged;

		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Occurs when scrolling.")]
		public event TchEvtNtcScrollPos OnNtcScrollPos;

		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Occurs when dropping files.")]
		public event TchEvtFileDrop 	OnFileDrop;

		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Occurs when dropping directories.")]
		public event TchEvtDirDrop		OnDirDrop;

		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Occurs when Shift/Ctrl + right-clicked.")]
		public event TchEvtRClick		OnRClick;

		private TchCbkRangeChanged	m_TchCbkRangeChanged;
		private TchCbkNtcScrollPos	m_TchCbkNtcScrollPos;
		private TchCbkFileDrop		m_TchCbkFileDrop;
		private TchCbkDirDrop		m_TchCbkDirDrop;
		private TchCbkRClick		m_TchCbkRClick;

		//----- プロパティ値 ------------------------------//
		private	bool		m_EnablePopupMenu	= true;
		private bool		m_ShowDummyData		= true;
		private bool		m_EnableMesDraw		= false;

		private string		m_TitleText			= "";
		private bool		m_ToolTipShowAlways = true;
		private string		m_TipText			= "";
		private string[]	m_TipFilter 		= new string[16];

        private Font        m_TxoFont = new Font("MS UI Gothic", 9);

		//----- ワーク ------------------------------------//
		private IntPtr		m_hCtrl;
		private bool		m_ShowScrollBar;
		private	bool		m_FirstResize = true;


		//----- コンストラクタ ---------------------------------------------------------------------------------//
		public CAjrTimeChart()
		{
			InitializeComponent();

			//	ライブラリ初期化
			CAjrInit.Init();

			//	変数初期化
			m_ShowScrollBar = true;

			//	コントロールウインド生成
			m_TchCbkRangeChanged = new TchCbkRangeChanged(cbTchRangeChanged);
			m_TchCbkNtcScrollPos = new TchCbkNtcScrollPos(cbTchNtcScrollPos);
			m_TchCbkFileDrop	 = new TchCbkFileDrop	 (cbTchFileDrop);
			m_TchCbkDirDrop 	 = new TchCbkDirDrop	 (cbTchDirDrop);
			m_TchCbkRClick		 = new TchCbkRClick 	 (cbTchRClick);

			m_hCtrl = CAjrApi.AjcTchCreateForMSIL(this.Handle, m_ShowScrollBar, m_TchCbkRangeChanged,
																				m_TchCbkNtcScrollPos,
																				m_TchCbkFileDrop	,
																				m_TchCbkDirDrop 	,
																				m_TchCbkRClick		);
		}

		//------------------------------------------------------------------------------------------------------//
		//	コントロール内部イベント																			//
		//------------------------------------------------------------------------------------------------------//
		//----- 初期設定 ---------------------------------------------------------------------------------------//
		private void CAjrTimeChart_Load(object sender, EventArgs e)
		{
		}
		//----- コントロール・サイズ変更 -----------------------------------------------------------------------//
		private void CAjrTimeChart_Resize(object sender, EventArgs e)
		{
			//	初回のみの処理
			if (m_FirstResize) {
				m_FirstResize = false;
				//	デザインモード時ダミーデータ表示
				if (this.DesignMode) GenDummyData();
			}
			//	コントロールのサイズ設定
			CAjrApi.AjcTchSetProp_WndSize(m_hCtrl, (int)this.Size.Width, (int)this.Size.Height);
		}
		//------------------------------------------------------------------------------------------------------//
		//	プロパティ																							//
		//------------------------------------------------------------------------------------------------------//
		//----- Fontプロパティを隠す ---------------------------------------------------------------------------//
		private Font vfontInit = new Font("MS UI Gothic", 9);
		[Browsable(false)]
		public override Font Font
		{
			set { base.Font = vfontInit; }
			get { return vfontInit; }
		}
        //----- FontTxoプロパティ ------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Font when TextOut()")]
		[Editor(typeof(FontTxoEditor), typeof(System.Drawing.Design.UITypeEditor))]
		public Font FontTxo
		{
            set {m_TxoFont = value;
                 CAjrApi.AjcTchSetTextFont(m_hCtrl, value.ToHfont());
            }
			get {return m_TxoFont; }
		}
		//----- AllowDropプロパティを隠す ----------------------------------------------------------------------//
		private bool AllowDropInit = false;
		[Browsable(false)]
		public override bool AllowDrop
		{
			set { base.AllowDrop = AllowDropInit; }
			get { return AllowDropInit; }
		}
		//----- Enabledプロパティを変更 ------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Enable / Disable the control")]
		public new bool Enabled
		{
			set {	if (base.Enabled == value) base.Enabled = !value;
					base.Enabled = value;
					CAjrApi.AjcEnableCtrlsInWnd(this.Handle, value);
				}
			get {	return base.Enabled;  }
		}

		//----- デザイン時のダミーデータ表示／非表示 -----------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Dummy data is shown at designing")]
		public bool ShowDummyData {
			set { m_ShowDummyData = value; if (this.DesignMode) GenDummyData(); }
			get { return m_ShowDummyData; }
		}
		//----- 外枠表示スタイル -------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Outer frame display")]
		public bool ShowBorder {
			set {CAjrApi.AjcTchSetSty_NOBORDER(m_hCtrl, !value);}
			get {return !CAjrApi.AjcTchGetSty_NOBORDER(m_hCtrl);}
		}
		//----- スケールライン表示スタイル ---------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Scale line disaply")]
		public bool ShowScaleLine
		{
			set { CAjrApi.AjcTchSetSty_NOSCALELINE(m_hCtrl, !value); }
			get { return !CAjrApi.AjcTchGetSty_NOSCALELINE(m_hCtrl); }
		}
		//----- スケール値表示スタイル -------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Scale value display")]
		public bool ShowScaleValue
		{
			set { CAjrApi.AjcTchSetSty_NOSCALEVALUE(m_hCtrl, !value); }
			get { return !CAjrApi.AjcTchGetSty_NOSCALEVALUE(m_hCtrl); }
		}
		//----- フィルタ表示スタイル ---------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Filter check box display")]
		public bool ShowFilter
		{
			set { CAjrApi.AjcTchSetSty_NOFILTER(m_hCtrl, !value); }
			get { return !CAjrApi.AjcTchGetSty_NOFILTER(m_hCtrl); }
		}

		//----- 低位レンジ値プロパティ -------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Low value in range")]
		public double RangeLow
		{
			set {CAjrApi.AjcTchSetProp_RngL(m_hCtrl, value);		if (this.DesignMode) GenDummyData();}
			get {return CAjrApi.AjcTchGetProp_RngL(m_hCtrl);}
		}

		//----- 高位レンジ値プロパティ -------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("High value in range")]
		public double RangeHigh
		{
			set {CAjrApi.AjcTchSetProp_RngH(m_hCtrl, value);		if (this.DesignMode) GenDummyData();}
			get {return CAjrApi.AjcTchGetProp_RngH(m_hCtrl);}
		}

		//----- 横軸表示幅プロパティ ---------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Vartical scale line display")]
		public int HoriScaleWidth
		{
			set {CAjrApi.AjcTchSetProp_TmScale(m_hCtrl, (int)value);}
			get {return (int)CAjrApi.AjcTchGetProp_TmScale(m_hCtrl);}
		}

		//----- 最大バッファサイズ・プロパティ -----------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Maximum number of buffering data")]
		public int MaxBuf
		{
			set {CAjrApi.AjcTchSetProp_MaxBuf(m_hCtrl, (int)value);		if (this.DesignMode) GenDummyData();}
			get {return  (int)CAjrApi.AjcTchGetProp_MaxBuf(m_hCtrl);}
		}

		//----- データ項目数プロパティ -------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Number of data items")]
		public int DataItems
		{
			set {CAjrApi.AjcTchSetProp_MaxItem(m_hCtrl, (int)value);		if (this.DesignMode) GenDummyData();}
			get {return  (int)CAjrApi.AjcTchGetProp_MaxItem(m_hCtrl);}
		}

		//----- 平均化個数プロパティ ---------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Number of averaging items")]
		public int AverageNumber
		{
			set { CAjrApi.AjcTchSetProp_AveNum(m_hCtrl, (int)value);		if (this.DesignMode) GenDummyData(); }
			get {return  (int)CAjrApi.AjcTchGetProp_AveNum(m_hCtrl);}
		}

		//----- 最大結線長プロパティ ---------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("When the difference of two pieces of data is this above the parameter value, it doesn't do a connection line(0:Always does a connection.)")]
		public double MaxLineDist
		{
			set { CAjrApi.AjcTchSetMaxLineDist(m_hCtrl, value); }
			get { return (int)CAjrApi.AjcTchGetMaxLineDist(m_hCtrl); }
		}

		//----- ファイル，フォルダ ドロップの許可／禁止 --------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Allow derop the file/folder")]
		public bool AcceptFiles
		{
			set {CAjrApi.AjcTchSetProp_AcceptFiles(m_hCtrl, value);}
			get {return CAjrApi.AjcTchGetProp_AcceptFiles(m_hCtrl);}
		}

		//----- ポップアップメニューの許可／禁止 --------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Enable popup menu by right click")]
		public bool EnablePopupMenu {
			set {m_EnablePopupMenu = value; CAjrApi.AjcTchEnablePopupMenu(m_hCtrl, value);}
			get {return m_EnablePopupMenu;}
		}

		//----- ツールチップ表示条件 ---------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Tooltip display condition. If false is set, the tooltip will not be displayed when inactive.")]
		public bool ToolTipShowAlways
		{
			set {
				m_ToolTipShowAlways = value;
				CAjrApi.AjcTchSetTipShowAlways(m_hCtrl, m_ToolTipShowAlways);
				for (int i = 0; i < m_TipFilter.Length; i++) {
					CAjrApi.AjcTchSetChkBoxTipShowAlways(m_hCtrl, i, m_ToolTipShowAlways);
				}
			}
			get {
				return m_ToolTipShowAlways;
			}
		}
		//----- ツールチップテキスト ---------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Tool tip text of this control")]
		public string ToolTipText
		{
			set	{
					if (this.DesignMode)  m_TipText = value;
					else {
				 		StringBuilder s = new StringBuilder(2048);
				 		CAjrApi.AjcTchGetTipTextForMSIL(m_hCtrl, s, (int)s.Capacity);
						if (value != s.ToString()) {
							CAjrApi.AjcTchSetTipTextForMSIL(m_hCtrl, value);
							CAjrApi.AjcTchSetTipShowAlways (m_hCtrl, m_ToolTipShowAlways);
						}
					}
				}
			get	{
					string rc = "";
					if (this.DesignMode) rc = m_TipText;
					else {
				 		StringBuilder s = new StringBuilder(2048);
				 		CAjrApi.AjcTchGetTipTextForMSIL(m_hCtrl, s, (int)s.Capacity);
				 		rc = s.ToString();
				 	}
				 	return rc;
				}
		}

		//----- フィルタ・チェックボックスＮのツールチップ設定／取得 -------------------------------------------//
		private void SetToolTipFilter(string s, int ix)
		{
			if (this.DesignMode) m_TipFilter[ix] = s;
			else if (s != GetToolTipFilter(ix))	 {
				CAjrApi.AjcTchSetChkBoxTipTextForMSIL(m_hCtrl, ix, s);
				CAjrApi.AjcTchSetChkBoxTipShowAlways (m_hCtrl, ix, m_ToolTipShowAlways);
			}
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		private string GetToolTipFilter(int ix)
		{
			string rc = "";
			if (this.DesignMode) rc = m_TipFilter[ix];
			else {
				StringBuilder s = new StringBuilder(2048);
				CAjrApi.AjcTchGetChkBoxTipTextForMSIL(m_hCtrl, ix, s, (int)s.Capacity);
				rc = s.ToString();
			}
			return rc;
		}

		//----- フィルタ・チェックボックス( 0)のツールチップ設定 -----------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 0")]
		public string ToolTipFilter0
		{
			set {SetToolTipFilter(value, 0);}
			get {return GetToolTipFilter(0);}
		}
		//----- フィルタ・チェックボックス( 1)のツールチップ設定 -----------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 1")]
		public string ToolTipFilter1
		{
			set {SetToolTipFilter(value, 1);}
			get {return GetToolTipFilter(1);}
		}
		//----- フィルタ・チェックボックス( 2)のツールチップ設定 -----------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 2")]
		public string ToolTipFilter2
		{
			set {SetToolTipFilter(value, 2);}
			get {return GetToolTipFilter(2);}
		}
		//----- フィルタ・チェックボックス( 3)のツールチップ設定 -----------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 3")]
		public string ToolTipFilter3
		{
			set {SetToolTipFilter(value, 3);}
			get {return GetToolTipFilter(3);}
		}
		//----- フィルタ・チェックボックス( 4)のツールチップ設定 -----------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 4")]
		public string ToolTipFilter4
		{
			set {SetToolTipFilter(value, 4);}
			get {return GetToolTipFilter(4);}
		}
		//----- フィルタ・チェックボックス( 5)のツールチップ設定 -----------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 5")]
		public string ToolTipFilter5
		{
			set {SetToolTipFilter(value, 5);}
			get {return GetToolTipFilter(5);}
		}
		//----- フィルタ・チェックボックス( 6)のツールチップ設定 -----------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 6")]
		public string ToolTipFilter6
		{
			set {SetToolTipFilter(value, 6);}
			get {return GetToolTipFilter(6);}
		}
		//----- フィルタ・チェックボックス( 7)のツールチップ設定 -----------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Tool tip text of filter check-box 7")]
		public string ToolTipFilter7
		{
			set {SetToolTipFilter(value, 7);}
			get {return GetToolTipFilter(7);}
		}
		//----- 波形補間表示種別プロパティ ---------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Kind of interpolation view")]
		public ETchIpKind IpKnd
		{
			set { CAjrApi.AjcTchSetProp_IpKnd(m_hCtrl, (int)value); }
			get { return (ETchIpKind)CAjrApi.AjcTchGetProp_IpKnd(m_hCtrl); }
		}
		//----- データ補間表示時の最大データ数プロパティ -------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Number of interpolation data")]
		public int IpNum
		{
			set { CAjrApi.AjcTchSetProp_IpNum(m_hCtrl, (int)value); }
			get { return (int)CAjrApi.AjcTchGetProp_IpNum(m_hCtrl); }
		}
		//----- データ補間の幅プロパティ -----------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Width of interpolation")]
		public int IpWidth
		{
			set { CAjrApi.AjcTchSetProp_IpWidth(m_hCtrl, (int)value); }
			get { return (int)CAjrApi.AjcTchGetProp_IpWidth(m_hCtrl); }
		}
		//----- 描画時間計測情報の許可／禁止プロパティ ---------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Enable the 'Display drawing time information' in popup-menu")]
		public bool EnableMesDraw
		{
			set { m_EnableMesDraw = value;
				  CAjrApi.AjcTchEnableMesDraw(m_hCtrl, value); }
			get { return m_EnableMesDraw; }
		}
		//----- タイトルテキストプロパティ ---------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrTimeChart")]
		[System.ComponentModel.Description("Title text. It is displayed in the upper right corner of the control.")]
		public string TitleText
		{
			set { m_TitleText = value;
				  SetTitleText(m_TitleText);}
			get { return m_TitleText; }
		}
		//------------------------------------------------------------------------------------------------------//
		//	メソッド																							//
		//------------------------------------------------------------------------------------------------------//
		//----- グラフ表示停止 ---------------------------------------------------------------------------------//
		public void Stop()
		{
			CAjrApi.AjcTchStop(m_hCtrl);
		}
		//----- グラフ開始 -------------------------------------------------------------------------------------//
		public void Start()
		{
			CAjrApi.AjcTchStart(m_hCtrl);
		}
		//----- データ投与 -------------------------------------------------------------------------------------//
		public void PutData(double d1) {
			CAjrApi.AjcTchPutData1(m_hCtrl, d1);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void PutData(double d1, double d2) {
			CAjrApi.AjcTchPutData2(m_hCtrl, d1, d2);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void PutData(double d1, double d2, double d3) {
			CAjrApi.AjcTchPutData3(m_hCtrl, d1, d2, d3);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void PutData(double d1, double d2, double d3, double d4) {
			CAjrApi.AjcTchPutData4(m_hCtrl, d1, d2, d3, d4);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void PutData(double d1, double d2, double d3, double d4, double d5) {
			CAjrApi.AjcTchPutData5(m_hCtrl, d1, d2, d3, d4, d5);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void PutData(double d1, double d2, double d3, double d4, double d5, double d6) {
			CAjrApi.AjcTchPutData6(m_hCtrl, d1, d2, d3, d4, d5, d6);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void PutData(double d1, double d2, double d3, double d4, double d5, double d6, double d7) {
			CAjrApi.AjcTchPutData7(m_hCtrl, d1, d2, d3, d4, d5, d6, d7);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void PutData(double d1, double d2, double d3, double d4, double d5, double d6, double d7, double d8) {
			CAjrApi.AjcTchPutData8(m_hCtrl, d1, d2, d3, d4, d5, d6, d7, d8);
		}
		//----- データ項目のオフセット値設定 -------------------------------------------------------------------//
		public void SetItemOffset(int ix, double offset)
		{
			CAjrApi.AjcTchSetProp_ofs(m_hCtrl, (int)ix, offset);
		}
		//----- データ項目のオフセット値取得 -------------------------------------------------------------------//
		public double GetItemOffset(int ix)
		{
			return CAjrApi.AjcTchGetProp_ofs(m_hCtrl, (int)ix);
		}
		//----- データ項目の表示色設定 -------------------------------------------------------------------------//
		public void SetItemColor(int ix, Color color)
		{
			int r = color.R;
			int g = color.G;
			int b = color.B;

			CAjrApi.AjcTchSetProp_rgb(m_hCtrl, (int)ix, (int)(r | (g << 8) | (b << 16)));
		}
		//----- データ項目の表示色取得 -------------------------------------------------------------------------//
		public Color GetItemColor(int ix)
		{
			int rgb = CAjrApi.AjcTchGetProp_rgb(m_hCtrl, (int)ix);
			Color c = Color.FromArgb((int)(rgb & 0xFF), (int)((rgb >> 8) & 0xFF), (int)((rgb >> 16) & 0xFF));

			return c;
		}
		//----- グラフレンジの自動調整 -------------------------------------------------------------------------//
		public void AdjustRange()
		{
			CAjrApi.AjcTchAdjustRange(m_hCtrl);
		}
		//----- スクロール位置の設定 ---------------------------------------------------------------------------//
		public void SetScrollPos(int pos)
		{
			CAjrApi.AjcTchSetScrollPos(m_hCtrl, pos);
		}
		//----- スクロール位置の取得 ---------------------------------------------------------------------------//
		public int GetGcrollPos()
		{
			return CAjrApi.AjcTchGetScrollPos(m_hCtrl);
		}
		//----- フィルタの設定 ---------------------------------------------------------------------------------//
		public void SetFilter(int ix, bool fEnable)
		{
			CAjrApi.AjcTchSetFilter(m_hCtrl, (int)ix, fEnable);
		}
		//----- フィルタの取得 ---------------------------------------------------------------------------------//
		public bool GetFilter(int ix)
		{
			return CAjrApi.AjcTchGetFilter(m_hCtrl, (int)ix);
		}
		//----- 横線スタイル設定 -------------------------------------------------------------------------------//
		public void SetHoriLineStyle(int ix, Color color, int width, ETchLineStyle style)
		{
			int r = color.R;
			int g = color.G;
			int b = color.B;

			CAjrApi.AjcTchSetHLineAtt(m_hCtrl, (int)ix, (int)(r | (g << 8) | (b << 16)), width, style);
		}
		//----- 横線描画位置 -----------------------------------------------------------------------------------//
		public void SetHoriLinePos(int ix, double pos)
		{
			CAjrApi.AjcTchSetHLinePos(m_hCtrl, (int)ix, pos);
		}
		//----- 横線描画の許可／禁止 ---------------------------------------------------------------------------//
		public void EnableHoriLine(int ix, bool fEnable)
		{
			CAjrApi.AjcTchEnableHLine(m_hCtrl, (int)ix, fEnable);
		}
		//----- 縦線描画 ---------------------------------------------------------------------------------------//
		public void SetVertLine(Color color, int width, ETchLineStyle style)
		{
			int r = color.R;
			int g = color.G;
			int b = color.B;

			CAjrApi.AjcTchSetVLine(m_hCtrl, (int)(r | (g << 8) | (b << 16)), width, style);
		}
		//----- 縦線描画の許可／禁止 ---------------------------------------------------------------------------//
		public void EnableVertLine(bool fEnable)
		{
			CAjrApi.AjcTchEnableVLine(m_hCtrl, fEnable);
		}
		//----- ドロップされたファイル名取得 -------------------------------------------------------------------//
		public string GetDroppedFile()
		{
			return CAjrApi.AjcTchGetDroppedFile(m_hCtrl);
		}
		//----- ドロップされたディレクトリ名取得 ---------------------------------------------------------------//
		public string GetDroppedDir()
		{
			return CAjrApi.AjcTchGetDroppedDir(m_hCtrl);
		}
		//----- タイトルテキストの表示 -------------------------------------------------------------------------//
		public void SetTitleText(string TitleText)
		{
			m_TitleText = TitleText;
			CAjrApi.AjcTchSetTitleText(m_hCtrl, TitleText, Color.Empty, Color.Empty, (Font)null);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void SetTitleText(string TitleText, Color TextColor)
		{
			m_TitleText = TitleText;
			CAjrApi.AjcTchSetTitleText(m_hCtrl, TitleText, TextColor, Color.Empty, null);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void SetTitleText(string TitleText, Color TextColor, Color BackColor)
		{
			m_TitleText = TitleText;
			CAjrApi.AjcTchSetTitleText(m_hCtrl, TitleText, TextColor, BackColor, null);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void SetTitleText(string TitleText, Color TextColor, Color BackColor, Font TextFont)
		{
			m_TitleText = TitleText;
			CAjrApi.AjcTchSetTitleText(m_hCtrl, TitleText, TextColor, BackColor, TextFont);
		}

		//----- プロファイルへ、永続情報書き込み ---------------------------------------------------------------//
		public void SavePermInfoToProfile()
		{
			CAjrApi.AjcTchSavePermInfo(m_hCtrl);
		}

		//----- プロファイルから、永続情報読み出し -------------------------------------------------------------//
		public void LoadPermInfoFromProfile(string section)
		{
			CAjrApi.AjcTchLoadPermInfo(m_hCtrl, section, this.Name);
		}

		//----- プロファイルへ、フィルタ設定とプロパティ書き込み -----------------------------------------------//
		public void SaveToProfile(string section)
		{
			CAjrApi.AjcTchSavePropEx(m_hCtrl, section);
		}
		//----- プロファイルから、フィルタ設定とプロパティ読み出し ---------------------------------------------//
		public void LoadFromProfile(string section)
		{
			CAjrApi.AjcTchLoadPropEx(m_hCtrl, section, (IntPtr)0);
		}

		//----- 表示停止／再開 ---------------------------------------------------------------------------------//
		public void Pause(bool fPause)
		{
			CAjrApi.AjcTchPause(m_hCtrl, fPause);
		}

		//----- プロット周期計測値の表示 -----------------------------------------------------------------------//
		public void MesPeriShow(bool fShow)
		{
			CAjrApi.AjcTchMesPeriShow(m_hCtrl, fShow);
		}

		//----- プロット周期計測のリセット ---------------------------------------------------------------------//
		public void MesPeriReset()
		{
			CAjrApi.AjcTchMesPeriReset(m_hCtrl);
		}
		//----- テキスト描画 -----------------------------------------------------------------------------------//
		public int TextOut(int x, int y, string text)
		{
            return CAjrApi.AjcTchTextOutW(m_hCtrl, x, y, text);
		}
		//----- テキスト削除 -----------------------------------------------------------------------------------//
		public bool PurgeText(int key)
		{
			return CAjrApi.AjcTchClearText(m_hCtrl, key);
		}
		//- - - 全てのテキスト破棄 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -//
		public bool PurgeText()
		{
			return CAjrApi.AjcTchClearAllText(m_hCtrl);
		}
		//----- データ破棄 -------------------------------------------------------------------------------------//
		public void PurgePlot()
		{
			CAjrApi.AjcTchPurge(m_hCtrl);
		}
		//----- 全てのデータ，描画テキスト破棄 -----------------------------------------------------------------//
		public void Purge()
		{
			CAjrApi.AjcTchPurge(m_hCtrl);
			CAjrApi.AjcTchClearAllText(m_hCtrl);
		}

		//------------------------------------------------------------------------------------------------------//
		//	デザイン時のダミーデータ生成																		//
		//------------------------------------------------------------------------------------------------------//
		private void GenDummyData()
		{
			if (this.DesignMode) {
				int 	i, n;
				double	t;
				this.Purge();
				if (m_ShowDummyData) {
					n = this.Width * this.AverageNumber;
					for (i = 0, t = 0.0; i < n; i++, t += 2.0) {
						double	l, h, c, r, d1, d2, d3, d4, d5, d6, d7, d8;
						l = System.Math.Min(this.RangeLow, this.RangeHigh);
						h = System.Math.Max(this.RangeLow, this.RangeHigh);
						c = (h + l) / 2.0;
						r = (h - l) / 2.0;
						r *= 0.8;
						d1 = c + Math.Sin((t +	 0.0) * 3.14159 / 180.0) * r;
						d2 = c + Math.Sin((t +	20.0) * 3.14159 / 180.0) * r;
						d3 = c + Math.Sin((t +	40.0) * 3.14159 / 180.0) * r;
						d4 = c + Math.Sin((t +	60.0) * 3.14159 / 180.0) * r;
						d5 = c + Math.Sin((t +	80.0) * 3.14159 / 180.0) * r;
						d6 = c + Math.Sin((t + 100.0) * 3.14159 / 180.0) * r;
						d7 = c + Math.Sin((t + 120.0) * 3.14159 / 180.0) * r;
						d8 = c + Math.Sin((t + 140.0) * 3.14159 / 180.0) * r;
						this.PutData(d1, d2, d3, d4, d5, d6, d7, d8);
					}
				}
			}
		}

		//------------------------------------------------------------------------------------------------------//
		//	コールバック																						//
		//------------------------------------------------------------------------------------------------------//
		//----- コールバック関数（レンジ変化通知）--------------------------------------------------------------//
		private void cbTchRangeChanged(double low, double high)
		{
			TchArgRangeChanged eva = new TchArgRangeChanged();
			eva.low = low;
			eva.high = high;
			if (OnRangeChanged != null) {
				OnRangeChanged(this, eva);
			}
		}
		//----- コールバック関数（スクロール位置変化通知）------------------------------------------------------//
		private void cbTchNtcScrollPos(int pos)
		{
			TchArgNtcScrollPos eva = new TchArgNtcScrollPos();
			eva.pos = pos;
			if (OnNtcScrollPos != null) {
				OnNtcScrollPos(this, eva);
			}
		}
		//----- コールバック関数（ファイルドロップ通知）--------------------------------------------------------//
		private void cbTchFileDrop(int n)
		{
			TchArgFileDrop eva = new TchArgFileDrop();
			eva.n = n;
			if (OnFileDrop != null) {
				OnFileDrop(this, eva);
			}
		}
		//----- コールバック関数（フォルダドロップ通知）--------------------------------------------------------//
		private void cbTchDirDrop(int n)
		{
			TchArgDirDrop eva = new TchArgDirDrop();
			eva.n = n;
			if (OnDirDrop != null) {
				OnDirDrop(this, eva);
			}
		}
		//----- コールバック関数（右クリック通知）--------------------------------------------------------------//
		private void cbTchRClick(int x, int y, bool shift, bool ctrl)
		{
			TchArgRClick eva = new TchArgRClick();
			eva.x	  = x;
			eva.y	  = y;
			eva.shift = shift;
			eva.ctrl  = ctrl;
			if (OnRClick != null) {
				OnRClick(this, eva);
			}
		}
	}
	//==========================================================================================================//
	//	FontTxoプロパティ・エディタ	（ダイアログから、取り消し線や下線を隠す）									//
	//==========================================================================================================//
	internal class FontTxoEditor : UITypeEditor
	{
		//----- コンストラクタ ---------------------------------------------------------------------------------//
		public FontTxoEditor()
		{
		}
		//----- プロパティエディタとして、モーダル表示フォームを使用する事を通知 -------------------------------//
		public override UITypeEditorEditStyle GetEditStyle(ITypeDescriptorContext context)
		{
			return UITypeEditorEditStyle.Modal;
		}
		//----- 独自フォント設定 -------------------------------------------------------------------------------//
		public override object EditValue(ITypeDescriptorContext context, IServiceProvider provider, object value)
		{
			IWindowsFormsEditorService editorService = null;

			//----- MSDN UI 型エディターの実装 に基づいた記述
			if (provider != null) {
				editorService = provider.GetService(typeof(IWindowsFormsEditorService)) as IWindowsFormsEditorService;
			}

			//----- MSDN UI 型エディターの実装 に基づいた記述
			if (editorService == null) {
				return value;
			}
			
			//----- フォント表現文字列取得 ---------------------------------------------------------------------//
			Font f = (Font)value;
			FontDialog fd = new FontDialog();

			//	初期のフォント設定
			fd.Font = (Font)value;
			//	ポイントサイズの範囲
			fd.MaxSize = 144;
			fd.MinSize = 6;
			//	存在しないフォントやスタイルをユーザーが選択時、エラーメッセージを表示する
			fd.FontMustExist = true;
			//	横書きフォントだけを表示する
			fd.AllowVerticalFonts = false;
			//	色の選択
			fd.ShowColor = false;
			//	取り消し線、下線のオプション
			fd.ShowEffects = false;
			//	固定ピッチフォント以外も表示する
			fd.FixedPitchOnly = false;
			//	ベクタ フォントを選択できるようにする(デフォルト：True)
			fd.AllowVectorFonts = true;
			//	ダイアログを表示し、フォントを選択する
			if (fd.ShowDialog() != DialogResult.Cancel) {
				f = fd.Font;
			}
			return f;
		}
	}
}
