﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Drawing.Design;
using System.Windows.Forms.Design;
using AjaraInit;

namespace CAjrCustCtrl
{
	//----------------------------------------------------------------------------------------------------------//
	[ToolboxBitmap(typeof(CAjrVT100), "CAjrVT100.bmp")]
	public partial class CAjrVT100 : UserControl
	{
		//----- 外部へのイベント -------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Occurs when double-clicked.")]
		public event VthEvtNtcDblClk OnNtcDblClk;

		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Notifies the key input.")]
		public event VthEvtNtcKeyIn  OnNtcKeyIn;

		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Notifies the press vartual key.")]
		public event VthEvtNtcVKeyIn OnNtcVKeyIn;

		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Notifies the release vartual key.")]
		public event VthEvtNtcVKeyOut OnNtcVKeyOut;

		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Occurs when dropping files.")]
		public event VthEvtFileDrop  OnFileDrop;

		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Occurs when dropping directories.")]
		public event VthEvtDirDrop	 OnDirDrop;

		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Notifies that the font or size has changed and the character information has changed.")]
		public event VthEvtCharInfo  OnCharInfo;

		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Occurs when Shift/Ctrl + right-clicked.")]
		public event VthEvtRClick 	 OnRClick;

		private VthCbkNtcDblClk		m_VthCbkNtcDblClk;
		private VthCbkNtcKeyIn		m_VthCbkNtcKeyIn;
		private VthCbkNtcVKeyIn		m_VthCbkNtcVKeyIn;
		private VthCbkNtcVKeyOut	m_VthCbkNtcVKeyOut;
		private VthCbkFileDrop		m_VthCbkFileDrop;
		private VthCbkDirDrop		m_VthCbkDirDrop;
		private VthCbkCharInfo		m_VthCbkCharInfo;
		private VthCbkRClick		m_VthCbkRClick;

		//----- プロパティ値 ------------------------------//
		private string		m_TitleText			= "";
		private bool		m_ToolTipShowAlways = true;
		private	bool		m_EnablePopupMenu	= true;
        private string      m_Font = "P: FN=MS Gothic, LF=12/0/0/0/0/0/0/0/128/3/2/1/49; 1; 0x8031,MS Gothic; 0x8032,MS UI Gothic";
		private bool		m_ShowDummyData		= true;
		private string		m_TipText			= "";
		private	bool		m_FirstResize		= true;
		private	bool		m_LFActInPopupMenu	= true;
		private string		m_StrFindInfoSect	= "_DefaultStrFindInfoSect_";
		private Keys		m_StrFindKey		= Keys.F3;

		//----- VRamHeight, MaxLines 競合調整 -------------//
		private	bool		m_Fixed_VRamHeight_And_MaxLines = false;
		private	int			m_SvVRamHeight = -1;
		private	int			m_SvMaxLines   = -1;

		//----- ワーク -----------------------------------------------------------------------------------------//
		private IntPtr		m_hCtrl;

		//----- コンストラクタ ---------------------------------------------------------------------------------//
		public CAjrVT100()
		{
			InitializeComponent();

			//	ライブラリ初期化
			CAjrInit.Init();

			//	コントロールウインド生成
			m_VthCbkNtcDblClk	= new VthCbkNtcDblClk (cbVthNtcDblClk );
			m_VthCbkNtcKeyIn	= new VthCbkNtcKeyIn  (cbVthNtcKeyIn  );
			m_VthCbkNtcVKeyIn	= new VthCbkNtcVKeyIn (cbVthNtcVKeyIn );
			m_VthCbkNtcVKeyOut	= new VthCbkNtcVKeyOut(cbVthNtcVKeyOut);
			m_VthCbkFileDrop	= new VthCbkFileDrop  (cbVthFileDrop  );
			m_VthCbkDirDrop	 	= new VthCbkDirDrop	  (cbVthDirDrop	  );
			m_VthCbkCharInfo	= new VthCbkCharInfo  (cbVthCharInfo  );
			m_VthCbkRClick		= new VthCbkRClick	  (cbVthRClick	  );

			m_hCtrl = CAjrApi.AjcVthCreateForMSIL(this.Handle, m_VthCbkNtcDblClk ,
															   m_VthCbkNtcKeyIn	 ,
															   m_VthCbkNtcVKeyIn ,
															   m_VthCbkNtcVKeyOut,
															   m_VthCbkFileDrop	 ,
															   m_VthCbkDirDrop	 ,
															   m_VthCbkCharInfo	 ,
															   m_VthCbkRClick	 );

			//	フォント設定
			FontVt100 = m_Font;
		}
		//------------------------------------------------------------------------------------------------------//
		//	コントロール内部イベント																			//
		//------------------------------------------------------------------------------------------------------//
		//----- 初期設定 ---------------------------------------------------------------------------------------//
		private void CAjrVT100_Load(object sender, EventArgs e)
		{
		}
		//----- コントロール・サイズ変更 -----------------------------------------------------------------------//
		private void CAjrVT100_Resize(object sender, EventArgs e)
		{
			//	初回のみの処理
			if (m_FirstResize) {
				m_FirstResize = false;
				//	デザインモード時ダミーデータ表示
				if (this.DesignMode) GenDummyData();
				//	デフォルトフォント設定（実行時のみ）
				else CAjrApi.AjcVthSetDefFont(m_hCtrl, this.Font.ToHfont());
				//	初期プロパティ退避
				CAjrApi.AjcVthSaveIniPropForMSIL(m_hCtrl);
			}
			//	コントロールのサイズ設定
			CAjrApi.AjcVthSetProp_WndSize(m_hCtrl, (int)this.Size.Width, (int)this.Size.Height);
		}
		//------------------------------------------------------------------------------------------------------//
		//	VRamHeight, MaxLines の競合調整	（初回は、VRamHeight, MaxLinesの順で設定することを保証する）		//
		//------------------------------------------------------------------------------------------------------//
		private bool IsFixed_VRamHeight_And_MaxLines()
		{
			bool	rc = false;
			if (!m_Fixed_VRamHeight_And_MaxLines) {
				if (m_SvVRamHeight != -1 && m_SvMaxLines != -1) {
					CAjrApi.AjcVthSetProp_VramH   (m_hCtrl, m_SvVRamHeight);
					CAjrApi.AjcVthSetProp_MaxLines(m_hCtrl, m_SvMaxLines  );
					if (this.DesignMode) GenDummyData();
					m_Fixed_VRamHeight_And_MaxLines = true;
				}
			}
			else rc = true;

			return rc;
		}

		//------------------------------------------------------------------------------------------------------//
		//	プロパティ																							//
		//------------------------------------------------------------------------------------------------------//
		//----- Fontプロパティを隠す ---------------------------------------------------------------------------//
		private Font vfontInit = new Font("MS UI Gothic", 9);
		[Browsable(false)]
		public override Font Font
		{
			set { base.Font = vfontInit; }
			get { return vfontInit; }
		}
		//----- AllowDropプロパティを隠す ----------------------------------------------------------------------//
		private bool AllowDropInit = false;
		[Browsable(false)]
		public override bool AllowDrop
		{
			set { base.AllowDrop = AllowDropInit; }
			get { return AllowDropInit; }
		}
		//----- Enabledプロパティを変更 ------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Enable / Disable the control")]
		public new bool Enabled
		{
			set {	if (base.Enabled == value) base.Enabled = !value;
					base.Enabled = value;
					CAjrApi.AjcEnableCtrlsInWnd(this.Handle, value);
				}
			get {	return base.Enabled;  }
		}
		//----- デザイン時のダミーデータ表示／非表示 -----------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Dummy data is shown at designing")]
		public bool ShowDummyData {
			set {m_ShowDummyData = value; if (this.DesignMode) GenDummyData();}
			get {return m_ShowDummyData;}
		}
		//----- 独自のフォントプロパティ -----------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[Description("Font")]
		[Editor(typeof(FontVt100Editor), typeof(System.Drawing.Design.UITypeEditor))]
		public string FontVt100
		{
			set {
				m_Font = value;
				CAjrApi.AjcVthSetProp_Font(m_hCtrl, m_Font);
			}
			get {
				return m_Font;
			}
		}

		//----- ＶＲＡＭ桁数 		(VramW) --------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Digits of VRAM width")]
		public int VRamWidth {
			set {CAjrApi.AjcVthSetProp_VramW(m_hCtrl, (int)value); if (this.DesignMode) GenDummyData();}
			get {return (int)CAjrApi.AjcVthGetProp_VramW(m_hCtrl);}
		}
		//----- ＶＲＡＭ行数		(VramH) --------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Lines of VRAM height")]
		public int VRamHeight {
			set {
					m_SvVRamHeight = value;
					if (IsFixed_VRamHeight_And_MaxLines()) {
						CAjrApi.AjcVthSetProp_VramH(m_hCtrl, value);
						if (this.DesignMode) GenDummyData();
					}
			}
			get {
					int		rc;
					if (!m_Fixed_VRamHeight_And_MaxLines) rc = m_SvVRamHeight;
					else								  rc = CAjrApi.AjcVthGetProp_VramH(m_hCtrl);
					return rc;
			}
		}
		//----- キャレットの高さ	(CaretH) -------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Caret height")]
		public int CaretHeight {
			set {CAjrApi.AjcVthSetProp_CaretH(m_hCtrl, (int)value);}
			get {return (int)CAjrApi.AjcVthGetProp_CaretH(m_hCtrl);}
		}
		//----- キャレット表示／非表示 -------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Caret display")]
		public bool ShowCaret {
			set {CAjrApi.AjcVthShowCaret(m_hCtrl, value);}
		}
		//----- 最大行数			(MaxLines) -----------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Buffering lines(Values less than VRamHeight cannot be set)")]
		public int MaxLines {
			set {
					m_SvMaxLines = value;
					if (IsFixed_VRamHeight_And_MaxLines()) {
						CAjrApi.AjcVthSetProp_MaxLines(m_hCtrl, value);
						if (this.DesignMode) GenDummyData();
					}
			}
			get {
					int		rc;
					if (!m_Fixed_VRamHeight_And_MaxLines) rc = m_SvMaxLines;
					else								  rc = CAjrApi.AjcVthGetProp_MaxLines(m_hCtrl);
					return rc;
			}
		}
		//----- タブ・スキップ数	(TabStep) ------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Width of TAB")]
		public int TabStep {
			set {CAjrApi.AjcVthSetProp_TabStep(m_hCtrl, (int)value);}
			get {return (int)CAjrApi.AjcVthGetProp_TabStep(m_hCtrl);}
		}
		//----- 行間スペース		(LSpace) -------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("line spacing pixels")]
		public int LineSpace {
			set {CAjrApi.AjcVthSetProp_LSpace(m_hCtrl, (int)value);}
			get {return (int)CAjrApi.AjcVthGetProp_LSpace(m_hCtrl);}
		}
		//----- 一時保留サイズ		(PendSize) -----------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Data buffering size (bytes)")]
		public int PendingBufferSize {
			set {CAjrApi.AjcVthSetProp_PendSize(m_hCtrl, (int)value);}
			get {return (int)CAjrApi.AjcVthGetProp_PendSize(m_hCtrl);}
		}
		//----- 文字の幅			(CharWidth) ----------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Charcter width (Pixels)")]
		public int CharWidth {
			get {return (int)CAjrApi.AjcVthGetProp_CharWidth(m_hCtrl);}
		}
		//----- 文字の高さ			(CharHeight) ---------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Charcter height (Pixels)")]
		public int CharHeight {
			get {return (int)CAjrApi.AjcVthGetProp_CharHeight(m_hCtrl);}
		}
		//----- 行の高さ			(LineHeight) ---------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Line height (Pixels)")]
		public int LineHeight {
			get {return (int)CAjrApi.AjcVthGetProp_LineHeight(m_hCtrl);}
		}
		//----- 文字色 -----------------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Text color (Palette number(0-7))")]
		public int ForegroundColor
		{
			set {CAjrApi.AjcVthSetColor(m_hCtrl, (int)value); if (this.DesignMode) GenDummyData();}
			get {return (int)CAjrApi.AjcVthGetColor(m_hCtrl);}
		}
		//----- 文字背景色 -------------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Text background color (Palette number(0-7))")]
		public int BackgroundColor
		{
			set {CAjrApi.AjcVthSetBkColor(m_hCtrl, (int)value); if (this.DesignMode) GenDummyData();}
			get {return (int)CAjrApi.AjcVthGetBkColor(m_hCtrl);}
		}
		//----- ウインドの背景色 -------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Window background color (Palette number(0-7))")]
		public int WndBkColor
		{
			set {CAjrApi.AjcVthSetWndBkColor(m_hCtrl, (int)value); if (this.DesignMode) GenDummyData();}
			get {return (int)CAjrApi.AjcVthGetWndBkColor(m_hCtrl);}
		}
		//----- ファイル，フォルダ ドロップの許可／禁止 --------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Allow derop the file/folder")]
		public bool AcceptFiles
		{
			set {CAjrApi.AjcVthSetProp_AcceptFiles(m_hCtrl, value);}
			get {return CAjrApi.AjcVthGetProp_AcceptFiles(m_hCtrl);}
		}
		//----- ポップアップメニューの許可／禁止 --------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Enable popup menu by right click")]
		public bool EnablePopupMenu {
			set {m_EnablePopupMenu = value; CAjrApi.AjcVthEnablePopupMenu(m_hCtrl, value);}
			get {return m_EnablePopupMenu;}
		}
		//----- ツールチップ表示条件 ---------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Tooltip display condition. If false is set, the tooltip will not be displayed when inactive.")]
		public bool ToolTipShowAlways
		{
			set {
				m_ToolTipShowAlways = value;
				CAjrApi.AjcVthSetTipShowAlways(m_hCtrl, m_ToolTipShowAlways);
			}
			get {
				return m_ToolTipShowAlways;
			}
		}
		//----- ファイル出力コントロール表示／非表示(AJCVTHS_LOGFILE	) --------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("File output control display")]
		public bool EnableLogFile {
			set {CAjrApi.AjcVthSetSty_LOGFILE(m_hCtrl, value);}
			get {return CAjrApi.AjcVthGetSty_LOGFILE(m_hCtrl);}
		}
		//----- 固定ピッチ							(AJCVTHS_FIXEDPITCH	) --------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Display characters at a fixed pitch. (Only when fixed pitch font is set)")]
		public bool FixedPitch
		{
			set {CAjrApi.AjcVthSetSty_FIXEDPITCH(m_hCtrl, value);}
			get {return CAjrApi.AjcVthGetSty_FIXEDPITCH(m_hCtrl);}
		}
		//----- 高速表示							(AJCVTHS_FAST		) --------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Executes drawing processing at high speed. Text color and background color cannot be set.")]
		public bool Fast
		{
			set {CAjrApi.AjcVthSetSty_FAST(m_hCtrl, value);}
			get {return CAjrApi.AjcVthGetSty_FAST(m_hCtrl);}
		}
		//----- 画面クリアーボタン許可				(AJCVTHS_NOCLSBTN	) --------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Allow screen clear button.")]
		public bool EnableClsBtn
		{
			set {CAjrApi.AjcVthSetSty_NOCLSBTN(m_hCtrl, !value);}
			get {return !CAjrApi.AjcVthGetSty_NOCLSBTN(m_hCtrl);}
		}
		//----- フォント固定						(AJCVTHS_FIXEDFONT	) --------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Font to fixed. (Don't include font settings in pop-up menus)")]
		public bool FixedFont
		{
			set {CAjrApi.AjcVthSetSty_FIXEDFONT(m_hCtrl, value);}
			get {return CAjrApi.AjcVthGetSty_FIXEDFONT(m_hCtrl);}
		}
		//----- 改行動作固定						(AJCVTHS_FIXEDLF	) --------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Line feed action to fixed. (No L/F setting items in popup menu)")]
		public bool FixedLF
		{
			set {CAjrApi.AjcVthSetSty_FIXEDLF(m_hCtrl, value);}
			get {return CAjrApi.AjcVthGetSty_FIXEDLF(m_hCtrl);}
		}
		//----- 縦スクロールバー表示／非表示		(AJCVTHS_NOVSCROLL	) --------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Show vertical scrollbar.")]
		public bool ShowVScroll
		{
			set {CAjrApi.AjcVthSetSty_NOVSCROLL(m_hCtrl, !value);}
			get {return !CAjrApi.AjcVthGetSty_NOVSCROLL(m_hCtrl);}
		}
		//----- 横スクロールバー表示／非表示		(AJCVTHS_NOHSCROLL	) --------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Show horizontal scrollbar.")]
		public bool ShowHScroll
		{
			set {CAjrApi.AjcVthSetSty_NOHSCROLL(m_hCtrl, !value);}
			get {return !CAjrApi.AjcVthGetSty_NOHSCROLL(m_hCtrl);}
		}
		//----- 自動スクロールの許可／禁止			(AJCVTHS_NOSCRL		) --------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Enable auto scroll (It is scrolled to the end for each drawing)")]
		public bool EnableAutoScroll {
			set {CAjrApi.AjcVthSetSty_NOSCRL(m_hCtrl, !value);}
			get {return !CAjrApi.AjcVthGetSty_NOSCRL(m_hCtrl);}
		}
		//----- コントロール外枠の非表示／表示		(AJCVTHS_NOBORDER	) --------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Outer frame display")]
		public bool ShowBorder {
			set {CAjrApi.AjcVthSetSty_NOBORDER(m_hCtrl, !value);}
			get {return !CAjrApi.AjcVthGetSty_NOBORDER(m_hCtrl);}
		}
		//----- スクロールアウトの許可／禁止		(AJCVTHS_NOSCRLOUT	) --------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Enable scroll out")]
		public bool EnableScrollOut {
			set {CAjrApi.AjcVthSetSty_NOSCRLOUT(m_hCtrl, !value);}
			get {return !CAjrApi.AjcVthGetSty_NOSCRLOUT(m_hCtrl);}
		}
		//----- ＣＲで改行							(AJCVTHS_CRCTRL		) --------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Line feed by C/R(0x0D)")]
		public bool LineFeedByCR {
			set {CAjrApi.AjcVthSetSty_CRCTRL(m_hCtrl, value);}
			get {return CAjrApi.AjcVthGetSty_CRCTRL(m_hCtrl);}
		}
		//----- ＬＦで改行							(AJCVTHS_LFCTRL		) --------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Line feed by L/F(0x0A)")]
		public bool LineFeedByLF {
			set {CAjrApi.AjcVthSetSty_LFCTRL(m_hCtrl, value);}
			get {return CAjrApi.AjcVthGetSty_LFCTRL(m_hCtrl);}
		}
		//----- ＶＲＡＭとその他の領域分離			(AJCVTHS_SEPARATE	) --------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Distinguishes between the display of VRAM / Buffering data (Displays scrolled out line darkly)")]
		public bool Separate {
			set {CAjrApi.AjcVthSetSty_SEPARATE(m_hCtrl, value);}
			get {return CAjrApi.AjcVthGetSty_SEPARATE(m_hCtrl);}
		}
		//----- ツールチップテキスト ---------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Tool tip text of this control")]
		public string ToolTipText
		{
			set	{
					if (this.DesignMode) m_TipText = value;
					else {
				 		StringBuilder s = new StringBuilder(2048);
				 		CAjrApi.AjcVthGetTipTextForMSIL(m_hCtrl, s, (int)s.Capacity);
						if (value != s.ToString()) {
							CAjrApi.AjcVthSetTipTextForMSIL(m_hCtrl, value);
							CAjrApi.AjcVthSetTipShowAlways(m_hCtrl, m_ToolTipShowAlways);
						}
					}
				}
			get	{
					string rc = "";
					if (this.DesignMode) rc = m_TipText;
					else {
				 		StringBuilder s = new StringBuilder(2048);
				 		CAjrApi.AjcVthGetTipTextForMSIL(m_hCtrl, s, (int)s.Capacity);
				 		return s.ToString();
				 	}
				 	return rc;
				}
		}
		//----- バッファに格納されている行数 -------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Line count")]
		public int LineCount
		{
			get {return (int)CAjrApi.AjcVthGetLineCount(m_hCtrl);}
		}
		//----- 横スクロール位置 -------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Horizontal scroll position")]
		public int ScrollPosH
		{
			set {CAjrApi.AjcVthSetHScrollPos(m_hCtrl, value);}
			get {return (int)CAjrApi.AjcVthGetHScrollPos(m_hCtrl);}
		}
		//----- 縦スクロール位置 -------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Vartical scroll position")]
		public int ScrollPosV
		{
			set {CAjrApi.AjcVthSetVScrollPos(m_hCtrl, value);}
			get {return (int)CAjrApi.AjcVthGetVScrollPos(m_hCtrl);}
		}
		//----- 表示ウインド行数 -------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Window height(Lines)")]
		public int WndHeight
		{
			get {return (int)CAjrApi.AjcVthGetProp_WndHeight(m_hCtrl);}
		}
		//----- 表示ウインド桁数 -------------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Window width(Chars)")]
		public int WndWidth
		{
			get {return (int)CAjrApi.AjcVthGetProp_WndWidth(m_hCtrl);}
		}
		//----- カーソル行位置（バッファ上の位置）--------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Cursor position in buffer")]
		public int CursouLine
		{
			get {	int		line, pos;
					CAjrApi.AjcVthGetCursorPosInfo(m_hCtrl, out line, out pos);
					return line;
				}
		}
		//----- カーソル桁位置（バッファ上の文字位置） ---------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Cursor position in buffer")]
		public int CursorPos
		{
			get {	int		line, pos;
					CAjrApi.AjcVthGetCursorPosInfo(m_hCtrl, out line, out pos);
					return pos;
				}
		}
		//----- 改行動作の設定をポップアップメニューに含める／含めない -----------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Include 'Line feed action' in pop-up menu")]
		public bool LFActInPopupMenu
		{
			set {m_LFActInPopupMenu = value;
				 CAjrApi.AjcVthSetLFActInPopupMenu(m_hCtrl, value);}
			get {return m_LFActInPopupMenu;}
		}
		//----- タイトルテキストプロパティ ---------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("Title text. It is displayed in the upper right corner of the control.")]
		public string TitleText
		{
			set { m_TitleText = value;
				  SetTitleText(m_TitleText);}
			get { return m_TitleText; }
		}
		//----- 文字列検索情報永続化セクション ------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("The name of the profile section in which to record string search information.")]
		public string StrFindInfoSect
		{
			set
			{
				m_StrFindInfoSect = value;
				CAjrApi.AjcVthSetFindProfileSectW(m_hCtrl, value);
			}
			get { return m_StrFindInfoSect; }
		}
		//----- 文字列検索の操作キー ----------------------------------------------------------------------------//
		[System.ComponentModel.Category("AjrVT100")]
		[System.ComponentModel.Description("String search operation key")]
		public Keys StrFindKey
		{
			set
			{
				m_StrFindKey = value;
				CAjrApi.AjcVthSetFindKey(m_hCtrl, value);
			}
			get { return m_StrFindKey; }
		}
		//------------------------------------------------------------------------------------------------------//
		//	メソッド																							//
		//------------------------------------------------------------------------------------------------------//
		//----- Focus()メソッドを変更 --------------------------------------------------------------------------//
		public new bool Focus()
		{
			bool rc;
			rc = CAjrApi.AjcVthSetFocus(m_hCtrl);
			return rc;
		}
		//----- １バイト描画 -----------------------------------------------------------------------------------//
		public void PutByte(int c)
		{
			CAjrApi.AjcVthPutChar(m_hCtrl, (byte)c);
		}
		//----- １文字描画 -------------------------------------------------------------------------------------//
		public void PutChar(int c)
		{
			CAjrApi.AjcVthPutChar(m_hCtrl, (ushort)c);
		}
		//----- 文字列描画 -------------------------------------------------------------------------------------//
		public void PutText(string text)
		{
			CAjrApi.AjcVthPutText(m_hCtrl, text, -1);
		}
		//----- 文字列描画 -------------------------------------------------------------------------------------//
		public void PutText(string text, int len)
		{
			CAjrApi.AjcVthPutText(m_hCtrl, text, len);
		}
		//----- 書式文字列描画 ---------------------------------------------------------------------------------//
		public void PutFormat(string format, params object[] args)
		{
			CAjrApi.AjcVthPutText(m_hCtrl, CAjrApi.MakeFormatString(format, args), -1);
		}
		//----- タイムスタンプ描画 -----------------------------------------------------------------------------//
		public void PrintTimeStamp()
		{
			CAjrApi.AjcVthTimeStamp(m_hCtrl);
		}
		//----- １６進ダンプ描画 -------------------------------------------------------------------------------//
		public void PrintHexDump(Byte[] bin)
		{
			CAjrApi.AjcVthHexDump(m_hCtrl, bin);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void PrintHexDump(IntPtr pData, int lData)
		{
			CAjrApi.AjcVthHexDump(m_hCtrl, pData, lData);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		unsafe public void PrintHexDump(void *pData, int lData)
		{
			CAjrApi.AjcVthHexDump(m_hCtrl, pData, lData);
		}
		//----- カーソル位置設定 -------------------------------------------------------------------------------//
		public void Locate(int line, int col)
		{
			CAjrApi.AjcVthLocate(m_hCtrl, (int)line, (int)col);
		}
		//----- 表示色設定				(rgb[]) ----------------------------------------------------------------//
		public void SetPaletteColor(int ix, Color color) {
			int r = color.R;
			int g = color.G;
			int b = color.B;
			CAjrApi.AjcVthSetProp_rgb(m_hCtrl, (int)ix, (int)(r | (g << 8) | (b << 16)));
		}
		//----- 表示色取得				(rgb[]) ----------------------------------------------------------------//
		public Color GetPaletteColor(int ix) {
			int rgb = CAjrApi.AjcVthGetProp_rgb(m_hCtrl, (int)ix);
			Color c = Color.FromArgb((int)(rgb & 0xFF), (int)((rgb >> 8) & 0xFF), (int)((rgb >> 16) & 0xFF));

			return c;
		}
		//----- 部分テキスト選択 -------------------------------------------------------------------------------//
		public void Select(int slp, int scp, int elp, int ecp)
		{
			CAjrApi.AjcVthSelect(m_hCtrl, slp, scp, elp, ecp);
		}
		//----- 全テキスト選択 ---------------------------------------------------------------------------------//
		public void SelectAll()
		{
			CAjrApi.AjcVthSelectAll(m_hCtrl);
		}
		//----- 選択テキストをクリップボードへコピー -----------------------------------------------------------//
		public void Copy()
		{
			CAjrApi.AjcVthCopyText(m_hCtrl);
		}
		//----- 全テキスト クリアー ----------------------------------------------------------------------------//
		public void Purge()
		{
			CAjrApi.AjcVthClearAllText(m_hCtrl);
		}
		//----- ドロップされたファイル名取得 -------------------------------------------------------------------//
		public string GetDroppedFile()
		{
			return CAjrApi.AjcVthGetDroppedFile(m_hCtrl);
		}
		//----- ドロップされたディレクトリ名取得 ---------------------------------------------------------------//
		public string GetDroppedDir()
		{
			return CAjrApi.AjcVthGetDroppedDir(m_hCtrl);
		}
		//----- タイトルテキストの表示 -------------------------------------------------------------------------//
		public void SetTitleText(string TitleText)
		{
			m_TitleText = TitleText;
			CAjrApi.AjcVthSetTitleText(m_hCtrl, TitleText, Color.Empty, Color.Empty, (Font)null);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void SetTitleText(string TitleText, Color TextColor)
		{
			m_TitleText = TitleText;
			CAjrApi.AjcVthSetTitleText(m_hCtrl, TitleText, TextColor, Color.Empty, null);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void SetTitleText(string TitleText, Color TextColor, Color BackColor)
		{
			m_TitleText = TitleText;
			CAjrApi.AjcVthSetTitleText(m_hCtrl, TitleText, TextColor, BackColor, null);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public void SetTitleText(string TitleText, Color TextColor, Color BackColor, Font TextFont)
		{
			m_TitleText = TitleText;
			CAjrApi.AjcVthSetTitleText(m_hCtrl, TitleText, TextColor, BackColor, TextFont);
		}
		//----- プロファイルへプロパティ書き込み ---------------------------------------------------------------//
		public void SaveToProfile(string section)
		{
			CAjrApi.AjcVthSaveProp(m_hCtrl, section);
		}
		//----- プロファイルからプロパティ読み出し -------------------------------------------------------------//
		public void LoadFromProfile(string section)
		{
			CAjrApi.AjcVthLoadProp(m_hCtrl, section, (IntPtr)0);
		}
		//----- プロファイルへフォント情報書き込み -------------------------------------------------------------//
		public void SavePermInfoToProfile()
		{
			CAjrApi.AjcVthSavePermInfo(m_hCtrl);
		}
		//----- プロファイルからフォント情報読み出し -----------------------------------------------------------//
		public void LoadPermInfoFromProfile(string section)
		{
			CAjrApi.AjcVthLoadPermInfo(m_hCtrl, section, this.Name);
		}
		//----- 行テキスト取得 ---------------------------------------------------------------------------------//
		public string GetLineText(int pos)
		{
			int		vw = VRamWidth + 1;
			StringBuilder s = new StringBuilder(vw);
			CAjrApi.AjcVthGetLineText(m_hCtrl, pos, s, vw);
			return s.ToString();
		}
		//----- ダブルクリック行テキスト取得 -------------------------------------------------------------------//
		public string GetDblCkickedLineText()
		{
			int len = 0;

			len = CAjrApi.AjcVthGetDblClickedLine(m_hCtrl);
			StringBuilder s = new StringBuilder((int)len);
			CAjrApi.AjcVthGetDblClickedLine(m_hCtrl, s, len);

			return s.ToString();
		}
		//----- ダブルクリック行位置取得 -----------------------------------------------------------------------//
		public int GetDblCkickedLinePos()
		{
			int		line, pos;
			CAjrApi.AjcVthGetDblClickedLineEx(m_hCtrl, out line, out pos);
			return line;
		}
		//----- 下方向文字列検索 -------------------------------------------------------------------------------//
		public int SearchBelow(string str, char delimiter)
		{
			return 	CAjrApi.AjcVthSearchBelow(m_hCtrl, str, (int)delimiter);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public int SearchBelow(string str, string dlm, string any, bool fTrim, bool fIgnoreWidth)
		{
			return 	CAjrApi.AjcVthSearchBelowEx(m_hCtrl, str, dlm, any, fTrim, fIgnoreWidth);
		}
		//----- 上方向文字列検索 -------------------------------------------------------------------------------//
		public int SearchAbove(string str, char delimiter)
		{
			return CAjrApi.AjcVthSearchAbove(m_hCtrl, str, (int)delimiter);
		}
		//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
		public int SearchAbove(string str, string dlm, string any, bool fTrim, bool fIgnoreWidth)
		{
			return CAjrApi.AjcVthSearchAboveEx(m_hCtrl, str, dlm, any, fTrim, fIgnoreWidth);
		}
		//----- 表示停止／再開 ---------------------------------------------------------------------------------//
		public void Pause(bool fPause)
		{
			CAjrApi.AjcVthPause(m_hCtrl, fPause);
		}
		//----- フォント設定 -----------------------------------------------------------------------------------//
		public void SetFont(Font font)
		{
			CAjrApi.AjcVthSetFontForMSIL(m_hCtrl, font.ToHfont());
		}
		//----- フォント取得 -----------------------------------------------------------------------------------//
		public Font GetFont()
		{
			IntPtr	hFont = CAjrApi.AjcVthGetFontForMSIL(m_hCtrl);
			Font font = Font.FromHfont(hFont);
			return font;
		}
		//------------------------------------------------------------------------------------------------------//
		//	デザイン時のダミーデータ生成																		//
		//------------------------------------------------------------------------------------------------------//
		private void GenDummyData()
		{
			if (this.DesignMode) {
				this.Purge();
				if (m_ShowDummyData) {
					this.PutText("ABCDEFGHIJKLMNOPQRSTUVWXYZ\r\n");
					this.PutText("0123456789\r\n");
					if (CAjrApi.AjcGetLangId() == ELangId.Japanese) {
						this.PutText("\n");
						this.PutText("ＶＴ１００エミュレーションウインド\n");
						this.PutText("任意のサイズでスクリーンを設定し、ログ表示等に使用します。\n");
						this.PutText("カーソル位置を指定して、グラフィカルに描画することもできます。\n");
					}
				}
			}
		}
		//------------------------------------------------------------------------------------------------------//
		//	コールバック																						//
		//------------------------------------------------------------------------------------------------------//
		//----- コールバック関数（ダブルクリック通知）----------------------------------------------------------//
		private void cbVthNtcDblClk(bool shift, bool ctrl)
		{
			VthArgDblClk eva = new VthArgDblClk();
			eva.shift = shift;
			eva.ctrl  = ctrl;
			if (OnNtcDblClk != null) {
				OnNtcDblClk(this, eva);
			}
		}
		//----- コールバック関数（キー入力通知）----------------------------------------------------------------//
		private void cbVthNtcKeyIn(int key, int rep)
		{
			VthArgNtcKeyIn eva = new VthArgNtcKeyIn();
			eva.key = key;
			eva.rep = rep;
			if (OnNtcKeyIn != null) {
				OnNtcKeyIn(this, eva);
			}
		}
		//----- コールバック関数（拡張キー押下通知, int RepCnt）------------------------------------------------//
		private void cbVthNtcVKeyIn(Keys vkey, int rep)
		{
			VthArgNtcVKeyIn eva = new VthArgNtcVKeyIn();
			eva.vkey = vkey;
			eva.rep  = rep;
			if (OnNtcVKeyIn != null) {
				OnNtcVKeyIn(this, eva);
			}
		}
		//----- コールバック関数（拡張キー離し通知）------------------------------------------------------------//
		private void cbVthNtcVKeyOut(Keys vkey)
		{
			VthArgNtcVKeyOut eva = new VthArgNtcVKeyOut();
			eva.vkey = vkey;
			if (OnNtcVKeyOut != null) {
				OnNtcVKeyOut(this, eva);
			}
		}
		//----- コールバック関数（ファイルドロップ通知）--------------------------------------------------------//
		private void cbVthFileDrop(int n)
		{
			VthArgFileDrop eva = new VthArgFileDrop();
			eva.n = n;
			if (OnFileDrop != null) {
				OnFileDrop(this, eva);
			}
		}
		//----- コールバック関数（フォルダドロップ通知）--------------------------------------------------------//
		private void cbVthDirDrop(int n)
		{
			VthArgDirDrop eva = new VthArgDirDrop();
			eva.n = n;
			if (OnDirDrop != null) {
				OnDirDrop(this, eva);
			}
		}
		//----- コールバック関数（文字サイズ変化通知）----------------------------------------------------------//
		private void cbVthCharInfo(int LineHeight)
		{
			VthArgCharInfo eva = new VthArgCharInfo();
			eva.LineHeight = LineHeight;
			if (OnCharInfo != null) {
				OnCharInfo(this, eva);
			}
		}
		//----- コールバック関数（右クリック通知）--------------------------------------------------------------//
		private void cbVthRClick(int x, int y, bool shift, bool ctrl)
		{
			VthArgRClick eva = new VthArgRClick();
			eva.x	  = x;
			eva.y	  = y;
			eva.shift = shift;
			eva.ctrl  = ctrl;
			if (OnRClick != null) {
				OnRClick(this, eva);
			}
		}
	}

	//==========================================================================================================//
	//	FontVt100プロパティ・エディタ																			//
	//==========================================================================================================//
	internal class FontVt100Editor : UITypeEditor
	{
		//----- コンストラクタ ---------------------------------------------------------------------------------//
		public FontVt100Editor()
		{
		}
		//----- プロパティエディタとして、モーダル表示フォームを使用する事を通知 -------------------------------//
		public override UITypeEditorEditStyle GetEditStyle(ITypeDescriptorContext context)
		{
			return UITypeEditorEditStyle.Modal;
		}
		//----- 独自フォント設定 -------------------------------------------------------------------------------//
		public override object EditValue(ITypeDescriptorContext context, IServiceProvider provider, object value)
		{
			IWindowsFormsEditorService editorService = null;

			//----- MSDN UI 型エディターの実装 に基づいた記述
			if (provider != null) {
				editorService = provider.GetService(typeof(IWindowsFormsEditorService)) as IWindowsFormsEditorService;
			}

			//----- MSDN UI 型エディターの実装 に基づいた記述
			if (editorService == null) {
				return value;
			}
			
			//----- フォント表現文字列取得 ---------------------------------------------------------------------//
			string s;
			s = CAjrApi.AjcVthGetProp_Font((string)value);
			return s;
		}
	}
}
