﻿//
//	SW_3DGraphic1.c
//

#include	<AjrCstXX.h>
#include	<math.h>
#include	<tchar.h>
#include	"resource.h"

//--------------------------------------------------------------------------------------------------------------//
//	タイマＩＤ																									//
//--------------------------------------------------------------------------------------------------------------//
#define		TID_PLOT_PERIOD		1		//	プロットデータ生成周期

//--------------------------------------------------------------------------------------------------------------//
//	ワーク																										//
//--------------------------------------------------------------------------------------------------------------//
static	HINSTANCE		hInst;							//	ＤＬＬインスタンスハンドル
static	HWND			hDlgMain;						//	ダイアログボックスハンドル
static	SIZE			szDlg;							//	ダイアログの最小サイズ
static	HWND			hWndG3d;						//	３Ｄグラフコントロールのウインドハンドル
static	HWND			hWndVth;						//	ログ表示コントロールのウインドハンドル
static	BOOL			fPlot = FALSE;					//	プロット中を示すフラグ
static	RECT			rcG3d, rcVth;					//	初期のウインド矩形
static	BOOL			fTest1 = FALSE;
static	BOOL			fTest2 = FALSE;
static	BOOL			fTest3 = FALSE;

//	球の中心と半径
#define		CX		0.2
#define		CY		0.4
#define		CZ		0.1
#define		R		2.0

//	描画色ＩＤ
#define		ID_PLOT1		0		//	プロット点１
#define		ID_PLOT2		1		//	プロット点２
#define		ID_BALL			3		//	算出した球
#define		ID_INS1			4		//	球の内接円１
#define		ID_INS2			12		//	球の内接円２

#define		ID_TRACK1		8		//	軌道円１
#define		ID_TRACK2		9		//	軌道円２

#define		ID_2DGRID		6		//	２Ｄ方眼
#define		ID_2DSHAPE		7		//	２Ｄ図形
#define		ID_3DSHAPE		5		//	２Ｄ図形
#define		ID_TXTPOINT		1		//	テキスト描画ポイント

//	プロット点演算パラメタ
static	AJCSPD_PARAM	prm[2] = {
//		  vCent 	   radius  noise  xrot	yrot  pitch
		{{CX, CY, CZ}, R,	   3.0,   0.5,	0.8,  9.0},
		{{CX, CY, CZ}, R,	   3.0,   0.6,	0.7,  8.5},
};
//	プロット点演算インスタンスハンドル
HAJCSPD		hSpd[2] = {NULL};


//	プロットデータ収集バッファ情報
#define	MAX_PLOT	4
typedef struct {
	UI			id1, id2, id3;							//	描画色ＩＤ
	double		dist;									//	プロット間の最低距離
	UI			num;									//	プロット収集個数
	UI			cnt;									//	プロットデータカウンタ
	AJC3DVEC	plt[MAX_PLOT];							//	プロット点収集バッファ
	AJC3DVEC	cent;									//	中心
	double		radius;									//	半径
	AJC3DVEC	vec;									//	法線
} PLOTINFO, *PPLOTINFO;

//	球算出用プロットバッファ
//									   id1		  id2		id3 	  dist		 num
static	PLOTINFO	PltBall		=	 { ID_INS1	, ID_INS2,	ID_BALL,  R * 0.6,	 4	 };

//	軌道円算出用プロットバッファ
//									   id1		  id2		id3 	  dist		 num
static	PLOTINFO	PltCir[2]	=	{{ ID_TRACK1, 0 	 ,	0	   ,  R * 0.4,	 3	 },
									 { ID_TRACK2, 0 	 ,	0	   ,  R * 0.4,	 3	 }};

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
AJC_DLGPROC_DEF(Main);

static	VO		SetControlSize(HWND hDlg);
static	C_UTP	CALLBACK cbGetTipText(HWND hCtrl, UTP pBuf, UI lBuf, UX cbp);
static	VO		BallCorrectAndShow(PCAJC3DVEC pVec, PPLOTINFO pBuf);
static	VO		CirCorrectAndShow (PCAJC3DVEC pVec, PPLOTINFO pBuf);
static	VO		DrawInscribedCircle(PCAJC3DCIRINFO pCif, int id);
static	VO		SetStyleToCheckBox(VO);
static	VO		SetCheckBoxToStyle(VO);

//==============================================================================================================//
//																												//
//	W i n M a i n																								//
//																												//
//==============================================================================================================//
int WINAPI AjcWinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance, UTP szCmdLine, int iCmdShow)
{
	MSG 	msg;
	RECT	rect;

	hInst = hInstance;

	//----- メイン・ダイアログオープン -----------------//
	hDlgMain = CreateDialog(hInst, MAKEINTRESOURCE(IDD_DLGMAIN), NULL, AJC_DLGPROC_NAME(Main));
	//----- ウインド最小サイズ退避 ---------------------//
	GetWindowRect(hDlgMain, &rect);
	szDlg.cx = rect.right - rect.left;
	szDlg.cy = rect.bottom - rect.top;
	//----- ダイアログ表示 -----------------------------//
	ShowWindow(hDlgMain, SW_SHOW);

	//----- メッセージループ ---------------------------//
	while (GetMessage(&msg, NULL, 0, 0)) {
		do {
			if (IsDialogMessage(hDlgMain, &msg)) break;
			TranslateMessage(&msg);
			DispatchMessage (&msg);
		} while (0);
	}

	return (int)msg.wParam ;
}
//==============================================================================================================//
//																												//
//	ダイアログ・プロシージャ																					//
//																												//
//==============================================================================================================//
//----- ダイアログ初期化 ---------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, WM_INITDIALOG 	)
{
	hDlgMain = hDlg;
	hWndG3d = GetDlgItem(hDlgMain, IDC_3DGRAPH);
	hWndVth = GetDlgItem(hDlgMain, IDC_VTH_LOG);
	//----- スタイルチェックボックスのツールチップ設定 -----------//
	AjcTipTextAdd(GetDlgItem(hDlg, IDC_CHK_NODEPTHCTRL) , TEXT("遠近表現（原点の手前側と向う側の色分け）禁止")); 
	AjcTipTextAdd(GetDlgItem(hDlg, IDC_CHK_NOFILTER   ) , TEXT("データ表示／非表示用チェックボックス非表示")); 
	AjcTipTextAdd(GetDlgItem(hDlg, IDC_CHK_NOBORDER   ) , TEXT("外枠非表示")); 
	AjcTipTextAdd(GetDlgItem(hDlg, IDC_CHK_NOANGLE	  ) , TEXT("ドラッグによる回転禁止")); 
	AjcTipTextAdd(GetDlgItem(hDlg, IDC_CHK_AXIS_X	  ) , TEXT("Ｘ軸表示")); 
	AjcTipTextAdd(GetDlgItem(hDlg, IDC_CHK_AXIS_Y	  ) , TEXT("Ｙ軸表示")); 
	AjcTipTextAdd(GetDlgItem(hDlg, IDC_CHK_AXIS_Z	  ) , TEXT("Ｚ軸表示")); 
	AjcTipTextAdd(GetDlgItem(hDlg, IDC_CHK_RECTSCALE  ) , TEXT("方眼スケール表示（表示面はSCALE_XY/XZ/YZで指定）")); 
	AjcTipTextAdd(GetDlgItem(hDlg, IDC_CHK_ELPSSCALE  ) , TEXT("同心円スケール表示（表示面はSCALE_XY/XZ/YZで指定）")); 
	AjcTipTextAdd(GetDlgItem(hDlg, IDC_CHK_SPHERESCALE) , TEXT("球体スケール表示")); 
	AjcTipTextAdd(GetDlgItem(hDlg, IDC_CHK_SCALE_XY   ) , TEXT("ＸＹ平面にスケール表示（スケールはRECTSCALE, ELPSSCALEで指定）")); 
	AjcTipTextAdd(GetDlgItem(hDlg, IDC_CHK_SCALE_XZ   ) , TEXT("ＸＺ平面にスケール表示（スケールはRECTSCALE, ELPSSCALEで指定）")); 
	AjcTipTextAdd(GetDlgItem(hDlg, IDC_CHK_SCALE_YZ   ) , TEXT("ＹＺ平面にスケール表示（スケールはRECTSCALE, ELPSSCALEで指定）")); 
	AjcTipTextAdd(GetDlgItem(hDlg, IDC_CHK_VAL_X	  ) , TEXT("Ｘ軸の値表示")); 
	AjcTipTextAdd(GetDlgItem(hDlg, IDC_CHK_VAL_Y	  ) , TEXT("Ｙ軸の値表示")); 
	AjcTipTextAdd(GetDlgItem(hDlg, IDC_CHK_VAL_Z	  ) , TEXT("Ｚ軸の値表示")); 
	//----- フィルタチェックボックスのツールチップ設定 -----------//
	AjcG3dSetChkBoxTipText(hWndG3d, ID_PLOT1 , TEXT("プロット点１")); 
	AjcG3dSetChkBoxTipText(hWndG3d, ID_PLOT2 , TEXT("プロット点２")); 
	AjcG3dSetChkBoxTipText(hWndG3d, ID_BALL  , TEXT("算出した球"  )); 
	AjcG3dSetChkBoxTipText(hWndG3d, ID_INS1  , TEXT("球の内接円１")); 
	AjcG3dSetChkBoxTipText(hWndG3d, ID_INS2  , TEXT("球の内接円２")); 
	AjcG3dSetChkBoxTipText(hWndG3d, ID_TRACK1, TEXT("軌道円１"	  )); 
	AjcG3dSetChkBoxTipText(hWndG3d, ID_TRACK2, TEXT("軌道円２"	  )); 
	//----- 初期のウインド矩形設定 -------------------------------//
	GetWindowRect(hWndG3d, &rcG3d); MapWindowPoints(NULL, hDlg, (LPPOINT)&rcG3d, 2);
	GetWindowRect(hWndVth, &rcVth); MapWindowPoints(NULL, hDlg, (LPPOINT)&rcVth, 2);
	SetControlSize(hDlg);
	//----- チップテキスト表示設定（状況依存で球情報表示）--------//
	AjcTipTextAddEx(hWndG3d, TEXT(""), -1, -1, NULL, -1, -1, -1);
	AjcTipTextSetCallBack(hWndG3d, 0, NULL, cbGetTipText);
	//----- タイトル設定 -----------------------------------------//
	AjcG3dSetTitleText(hWndG3d, TEXT(" 本来の球中心=(0.2, 0.4, 0.1), 半径=2.0 "), -1, -1, NULL);
	//----- ３Ｄグラフモード設定 ---------------------------------//
	AjcG3dInitV(hWndG3d, NULL, NULL, NULL, (AJC3DGS_AXIS | AJC3DGS_SCALEVALUE));
	//----- プロットデータ演算初期化 -----------------------------//
	hSpd[0] = AjcSpdCreate(0);
	hSpd[1] = AjcSpdCreate(0);
	//----- ウインドスタイル値をチェックボックスに設定 -----------//
	SetStyleToCheckBox();
	//----- プロット数初期化 -------------------------------------//
	AjcSetDlgItemUInt(hDlg, IDC_INP_PLOTNUM0, 100);
	AjcSetDlgItemUInt(hDlg, IDC_INP_PLOTNUM1, 100);
	//----- 設定値ロード -----------------------------------------//
	AjcDlgItemSetPermAtt(hDlg, IDC_TXT_3DGRAPH, AJCCTL_PSEL_EXCLUDE); // GetTextは除外
	AjcLoadAllControlSettings(hDlg, TEXT("Settings"), AJCOPT2(AJCCTL_SELACT_, CHKEXCLUDE, NTCINP));
	//	チェックボックスからスタイルを設定
	SetCheckBoxToStyle();
	//	プロット線描画チェックボックス反映
	SendMessage(hDlg, WM_COMMAND, MAKELONG(IDC_CHK_PLOTLINE, BN_CLICKED), (WPARAM)hDlg);
	//----- グラフレンジ設定 -------------------------------------//
	AjcG3dSetCenter(hWndG3d, CX, CY, CZ);
	AjcG3dSetWidth (hWndG3d, R, R, R);
	//----- 視点を３Ｄイメージのアングルに設定 -------------------//
	AjcG3dSetAngle3D(hWndG3d);

	return TRUE;
}
//----- ウインド破棄 -------------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, WM_DESTROY		)
{
	//----- 設定値セーブ -----------------------------------------//
	AjcSaveAllControlSettings(hDlg);
	AjcDelAllCtrlPermAtt(hDlg); // リソース解放 
	//----- プロットデータ演算終了 -------------------------------//
	AjcSpdDelete(hSpd[0]);
	AjcSpdDelete(hSpd[1]);
	//----- プログラム終了 ---------------------------------------//
	PostQuitMessage(0);
	return TRUE;
}
//----- サイズ変更中 -------------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, WM_SIZING			)
{
	LPRECT	pRect = (LPRECT)lParam;
	SIZE	sz;

	sz.cx = pRect->right - pRect->left;
	sz.cy = pRect->bottom - pRect->top;
	
	if (sz.cx < szDlg.cx) pRect->right	= pRect->left + szDlg.cx;
	if (sz.cy < szDlg.cy) pRect->bottom = pRect->top  + szDlg.cy;

	return TRUE;
}
//----- サイズ変更 ---------------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, WM_SIZE			)
{
	SetControlSize(hDlg);

	return TRUE;
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
static	VO		SetControlSize(HWND hDlg)
{
	RECT	r;
	int		w, h;

	GetClientRect(hDlg, &r);
	w = r.right - r.left;
	h = r.bottom - r.top;
	MoveWindow(hWndG3d, rcG3d.left, 0		 , w - rcG3d.left - 4	   , h			   - 4, FALSE);
	MoveWindow(hWndVth, rcVth.left, rcVth.top, rcVth.right - rcVth.left, h - rcVth.top - 4, FALSE);
	InvalidateRect(hDlg, NULL, TRUE);
}
//----- COMMAND ------------------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, WM_COMMAND		)
{
	BOOL		rc = FALSE;
	int 		cmd = LOWORD(wParam);
	UI			ix;
	AJC3DGPROP	prop;

	//----- スタイル設定チェックボックス操作 -------------------------------------------------//
	if (cmd >= IDC_CHK_RECTSCALE && cmd <= IDC_CHK_AXIS_Z && HIWORD(wParam) == BN_CLICKED) {
		//	チェックボックスの内容をスタイルに反映
		SetCheckBoxToStyle();
		rc = TRUE;
	}
	//----- プロット数設定操作 -----------------------------------------------------------------//
	if (cmd >= IDC_INP_PLOTNUM0 && cmd <= IDC_INP_PLOTNUM1 && HIWORD(wParam) == AJCIVN_INTVALUE) {
		ix = cmd - IDC_INP_PLOTNUM0;
		AjcG3dGetProp(hWndG3d, &prop);
		prop.Item[ix].MaxPlot = (UI)lParam;
		AjcG3dSetProp(hWndG3d, &prop);
		rc = TRUE;
	}

	return rc;
}
//----- WM_TIMER -----------------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, WM_TIMER		)
{
	UI			id;
	AJC3DVEC	v[2];

	switch (wParam) {
		case TID_PLOT_PERIOD:						//	●プロットデータ生成周期
			for (id = 0; id < 2; id++) {
				//	ランダムなプロットデータ生成
				AjcSpdCalcV(hSpd[id], &v[id]);
				//	プロットデータ投与
				AjcG3dPutPlotDataV(hWndG3d, id, &v[id]);
				//	球データ収集と算出した球の表示
				BallCorrectAndShow(&v[id], &PltBall);
				//	軌道円データ収集と算出した軌道円の表示
				CirCorrectAndShow (&v[id], &PltCir[id]);
			}
			//	ログ表示
			AjcVthTimeStamp(hWndVth);
			AjcVthPrintF(hWndVth, TEXT(": x1=%6.2f, y1=%6.2f, z1=%6.2f | x2=%6.2f, y2=%6.2f, z2=%6.2f\n"),
																 v[0].x, v[0].y, v[0].z, v[1].x, v[1].y, v[1].z);
			break;
	}
	return TRUE;
}
//----- プロットライン・チェックボックス -----------------------------------------------------------------------//
AJC_DLGPROC(Main, IDC_CHK_PLOTLINE)
{
	AJC3DGPROP	prop;

	if (HIWORD(wParam) == BN_CLICKED) {
		AjcG3dGetProp(hWndG3d, &prop);
		prop.fPlotLine = AjcGetDlgItemChk(hDlg, IDC_CHK_PLOTLINE);
		AjcG3dSetProp(hWndG3d, &prop);
	}
	return TRUE;
}
//----- サンプリング周期入力 -----------------------------------------------------------------------------------//
AJC_DLGPROC(Main, IDC_INP_PERIOD)
{
	if (HIWORD(wParam) == AJCIVN_INTVALUE) {
		if (fPlot) {
			SetTimer(hDlgMain, TID_PLOT_PERIOD, AjcGetDlgItemUInt(hDlg, IDC_INP_PERIOD), NULL);
		}
	}
	return TRUE;
}
//----- ノイズ値入力（AjcLoadAllControlSettings()で、AJCCTL_SELACT_NTCINP 指定の為、初期化時にも実行される）----//
AJC_DLGPROC(Main, IDC_INP_NOISE	)
{
	if (HIWORD(wParam) == AJCIVN_REALVALUE) {
		prm[0].noise = prm[1].noise = *((double *)lParam);
		AjcSpdSetParam(hSpd[0], &prm[0]);
		AjcSpdSetParam(hSpd[1], &prm[1]);
	}
	return TRUE;
}
//----- プロット開始／停止ボタン -------------------------------------------------------------------------------//
AJC_DLGPROC(Main, IDC_CMD_START_STOP)
{
	if (fPlot) {
		fPlot = FALSE;
		KillTimer(hDlg, TID_PLOT_PERIOD);
		AjcSetDlgItemStr(hDlg, IDC_CMD_START_STOP, TEXT("プロット開始"));
	}
	else {
		fPlot = TRUE;
		SetTimer(hDlgMain, TID_PLOT_PERIOD, AjcGetDlgItemUInt(hDlg, IDC_INP_PERIOD), NULL);
		AjcSetDlgItemStr(hDlg, IDC_CMD_START_STOP, TEXT("プロット停止"));
	}
	return TRUE;
}
//----- テスト１ボタン -----------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, IDC_CMD_TEST1 	)
{

	if (!fTest1) {
		AjcG3dCube	   (hWndG3d, ID_3DSHAPE, +0.1, +0.2, +0.3,	0.2, 0.3, 0.4, 8);
		AjcG3dSphere   (hWndG3d, ID_3DSHAPE, +0.7, +0.6, +0.8,	0.2, 0.3, 0.4, 12, 8);
		AjcG3dMoveTo   (hWndG3d, ID_3DSHAPE, -0.6,	0.2,  0.3);
		AjcG3dLineTo   (hWndG3d, ID_3DSHAPE, -0.3, -0.2, -0.1);
		AjcG3dTriangle (hWndG3d, ID_3DSHAPE, -0.8, -0.7, -0.6,	-0.4, -0.3, -0.5,  -0.4, -0.5, -0.3);
		AjcG3dSquare   (hWndG3d, ID_3DSHAPE,  0.0, -0.1,  0.6,	-0.2, -0.5,  0.6,  -0.7, -0.5,	0.3, -0.6,-0.3, 0.1);
		fTest1 = TRUE;
	}
	else {
		AjcG3dClearShape(hWndG3d, ID_3DSHAPE);
		fTest1 = FALSE;
	}
	return TRUE;
}
//----- テスト２ボタン -----------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, IDC_CMD_TEST2 	)
{
	UI		idAll = -1;
	double	x, y;

	if (!fTest2) {
		AjcG3dDefPlane(hWndG3d, idAll, 0.2, 0.1, 0.0,	0.6738, -0.5510, 0.4924,  0.1, 0.0, 0.0);
		AjcG2dRectangle(hWndG3d, ID_2DGRID, -0.9, -0.9,  +0.9, +0.9);
		for (x = -0.8; x < -0.01; x += 0.2) AjcG2dLine(hWndG3d, ID_2DGRID, x, -0.9, x, +0.9);
		for (x =  0.2; x <	0.89; x += 0.2) AjcG2dLine(hWndG3d, ID_2DGRID, x, -0.9, x, +0.9);
		for (y = -0.8; y < -0.01; y += 0.2) AjcG2dLine(hWndG3d, ID_2DGRID, -0.9, y, +0.9, y);
		for (y =  0.2; y <	0.89; y += 0.2) AjcG2dLine(hWndG3d, ID_2DGRID, -0.9, y, +0.9, y);
		AjcG2dLine	  (hWndG3d, ID_2DSHAPE, -0.9,  0.0,	+0.9,	0.0);
		AjcG2dLine	  (hWndG3d, ID_2DSHAPE,	0.0, -0.9,	 0.0, +0.9);
		AjcG2dTriangle(hWndG3d, ID_2DSHAPE, -0.6, -0.7,	-0.3, -0.3,  -0.1, -0.7);
		AjcG2dSquare  (hWndG3d, ID_2DSHAPE,	0.2,  0.3,	0.4, 0.7,	0.9, 0.7,  0.7, 0.3);
		AjcG2dPixel   (hWndG3d, ID_2DSHAPE,	0.3, -0.5,	2);
		AjcG2dEllipse (hWndG3d, ID_2DSHAPE,	0.3, -0.5,	0.4, 0.3);
		AjcG2dPixel   (hWndG3d, ID_2DSHAPE, 0.6, 0.8, 3);
		fTest2 = TRUE;
	}
	else {
		AjcG3dClearShape(hWndG3d, ID_2DGRID);
		AjcG3dClearShape(hWndG3d, ID_2DSHAPE);
		fTest2 = FALSE;
	}
	return TRUE;
}
//----- テスト３ボタン -----------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, IDC_CMD_TEST3 	)
{
	AJC3DVEC	v = {1, 1, 1};

	if (!fTest3) {
		AjcG3dPixelV  (hWndG3d, ID_TXTPOINT, &v, 3);
		AjcG3dTextOutV(hWndG3d, AJCG3DTXOMD_BELLOW_CENTER, &v, TEXT("\x1B[1;31m    ↑    \n(1, 1, 1)"));
		fTest3 = TRUE;
	}
	else {
		AjcG3dClearShape(hWndG3d, ID_TXTPOINT);
		AjcG3dClearAllText(hWndG3d);
		fTest3 = FALSE;
	}
	return TRUE;
}
//----- クリアーボタン -----------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, IDC_CMD_CLEAR 	)
{
	AjcG3dClear(hWndG3d);
	fTest1 = fTest2 = fTest3 = FALSE;
	return TRUE;
}
//----- GetText ボタン -----------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, IDC_CMD_GETTEXT	)
{
	UT		txt[1024];

	GetWindowText(hWndG3d, txt, 1024);
	AjcSetDlgItemStr(hDlg, IDC_TXT_3DGRAPH, txt);
	return TRUE;
}
//----- SetText ボタン -----------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, IDC_CMD_SETTEXT	)
{
	UT		txt[1024];

	AjcGetDlgItemStr(hDlg, IDC_TXT_3DGRAPH, txt, 1024);
	SetWindowText(hWndG3d, txt);
	return TRUE;
}
//----- Enable ボタン -----------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, IDC_CMD_ENABLE	)
{
	EnableWindow(hWndG3d, TRUE);
	return TRUE;
}
//----- Disable ボタン -----------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, IDC_CMD_DISABLE	)
{
	EnableWindow(hWndG3d, FALSE);
	return TRUE;
}
//----- ３Ｄグラフコントロール ---------------------------------------------------------------------------------//
AJC_DLGPROC(Main, IDC_3DGRAPH		)
{
	POINT	 pt;

	switch (HIWORD(wParam)) {
		case AJC3DGN_RCLICK:							//	●右クリック通知
		{	PAJC3DGRCLK	p = (PAJC3DGRCLK)lParam;
			UT			txt[64] = {0};
			AjcSnPrintF(txt, 64, TEXT("%s%s右クリック発生(x = %d, y = %d)"), p->fShift ? TEXT("Shift+") : TEXT(""),
																			 p->fCtrl  ? TEXT("Ctrl+" ) : TEXT(""),
																			 p->x, p->y);
			GetCursorPos(&pt);
			AjcTipTextShow(pt.x, pt.y + 20, txt, 3000, NULL);
			break;
		}
		case AJC3DGN_DROPDIR:							//	●ディレクトリドロップ
		{	UT		path[MAX_PATH];
			UT		txt[4096];
			MAjcStrCpy(txt, AJCTSIZE(txt), TEXT("-- Dir dropped --\n"));
			while (AjcG3dGetDroppedDir(hWndG3d, path)) {
				MAjcStrCat(txt, AJCTSIZE(txt), path);
				MAjcStrCat(txt, AJCTSIZE(txt), TEXT("\n"));
			}
			GetCursorPos(&pt);
			AjcTipTextShow(pt.x, pt.y + 20, txt, 3000, NULL);
			break;
		}
		case AJC3DGN_DROPFILE:							//	●ファイルドロップ
		{	UT		path[MAX_PATH];
			UT		txt[4096];
			MAjcStrCpy(txt, AJCTSIZE(txt), TEXT("-- File dropped --\n"));
			while (AjcG3dGetDroppedFile(hWndG3d, path)) {
				MAjcStrCat(txt, AJCTSIZE(txt), path);
				MAjcStrCat(txt, AJCTSIZE(txt), TEXT("\n"));
			}
			GetCursorPos(&pt);
			AjcTipTextShow(pt.x, pt.y + 20, txt, 3000, NULL);
			break;
		}
		case AJC3DGN_PLOTLIST:							//	●プロットリスト通知
		{	PCAJC3DGPLOTLIST p = (PCAJC3DGPLOTLIST)lParam;
			static UT txt[128];
			AjcSnPrintF(txt, AJCTSIZE(txt), TEXT("x=%.2f, y=%.2f, z=%.2f"), p->lst[0].vec.x, p->lst[0].vec.y, p->lst[0].vec.z);
			GetCursorPos(&pt);
			AjcTipTextShow(pt.x, pt.y + 20, txt, 3000, NULL);
			break;
		}
	}
	return TRUE;
}
//----- 「Cancel」ボタン ---------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, IDCANCEL			)
{
	DestroyWindow(hDlg);
	return TRUE;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_DLGMAP_DEF(Main)
	AJC_DLGMAP_MSG(Main, WM_INITDIALOG	   )
	AJC_DLGMAP_MSG(Main, WM_SIZING		   )
	AJC_DLGMAP_MSG(Main, WM_SIZE		   )
	AJC_DLGMAP_MSG(Main, WM_DESTROY 	   )
	AJC_DLGMAP_MSG(Main, WM_COMMAND 	   )
	AJC_DLGMAP_MSG(Main, WM_TIMER		   )

	AJC_DLGMAP_CMD(Main, IDC_CHK_PLOTLINE  )
	AJC_DLGMAP_CMD(Main, IDC_INP_PERIOD	   )
	AJC_DLGMAP_CMD(Main, IDC_INP_NOISE	   )
	AJC_DLGMAP_CMD(Main, IDC_CMD_START_STOP)
	AJC_DLGMAP_CMD(Main, IDC_CMD_TEST1	   )
	AJC_DLGMAP_CMD(Main, IDC_CMD_TEST2	   )
	AJC_DLGMAP_CMD(Main, IDC_CMD_TEST3	   )
	AJC_DLGMAP_CMD(Main, IDC_CMD_CLEAR	   )
	AJC_DLGMAP_CMD(Main, IDC_CMD_GETTEXT   )
	AJC_DLGMAP_CMD(Main, IDC_CMD_SETTEXT   )
	AJC_DLGMAP_CMD(Main, IDC_CMD_ENABLE    )
	AJC_DLGMAP_CMD(Main, IDC_CMD_DISABLE   )
	AJC_DLGMAP_CMD(Main, IDC_3DGRAPH	   )
	AJC_DLGMAP_CMD(Main, IDCANCEL		   )
AJC_DLGMAP_END

//--------------------------------------------------------------------------------------------------------------//
//	チップテキストコールバック																					//
//--------------------------------------------------------------------------------------------------------------//
static	C_UTP	CALLBACK cbGetTipText(HWND hCtrl, UTP pBuf, UI lBuf, UX cbp)
{
	//	チップテキスト（球情報）設定
	AjcSnPrintF(pBuf, lBuf,
					TEXT("球算出情報：\n")
					TEXT("  中心 = (%6.2f, %6.2f, %6.2f )\n")
					TEXT("  半径 =  %.2f"), PltBall.cent.x, PltBall.cent.y, PltBall.cent.z, PltBall.radius);
	return pBuf;
}
//--------------------------------------------------------------------------------------------------------------//
//	球算出用プロット点の収集と描画																				//
//--------------------------------------------------------------------------------------------------------------//
static	VO		BallCorrectAndShow(PCAJC3DVEC pVec, PPLOTINFO pBuf)
{
	UI		i;

	//	プロット間の最低距離チェック
	for (i = 0; i < pBuf->cnt; i++) {
		if (AjcV3dDistP2P(pVec, &pBuf->plt[i]) < pBuf->dist) {
			break;
		}
	}
	//	プロット点の収集
	if (i >= pBuf->cnt) {
		memcpy(&pBuf->plt[i], pVec, sizeof(AJC3DVEC));
		pBuf->cnt++;
	}
	//	球の算出と表示
	if (pBuf->cnt >= pBuf->num) {
		AJC3DSPHINFO	sph;
		if ((pBuf->radius = AjcV3dCalcSphereVEx(&pBuf->plt[0], &pBuf->plt[1], &pBuf->plt[2], &pBuf->plt[3],
																					 &pBuf->cent, &sph)) != -1) {
			//	２つの平面円の角度が30度以上ならば球描画
			double t = AjcV3dTheta(&sph.cif1.lvc.v, &sph.cif2.lvc.v);
			if (t > 30 && t < 180 - 30) {
				//	前回の球表示をクリアー
				AjcG3dClearShape(hWndG3d, pBuf->id3);
				//	球表示
				AjcG3dSphereV(hWndG3d, pBuf->id3, &pBuf->cent, pBuf->radius, pBuf->radius, pBuf->radius, 8, 8);
				//	前回の内接円表示をクリアー
				AjcG3dClearShape(hWndG3d, pBuf->id1);
				AjcG3dClearShape(hWndG3d, pBuf->id2);
				//	２つの内接円描画
				if (AjcGetDlgItemChk(hDlgMain, IDC_CHK_BALLINFO)) {
					//	球中心点描画
					AjcG3dPixelV(hWndG3d, pBuf->id1, &pBuf->cent, 3);
					AjcG3dPixelV(hWndG3d, pBuf->id2, &pBuf->cent, 3);
					//	球に内接する２つの円の情報を描画
					DrawInscribedCircle(&sph.cif1, pBuf->id1);
					DrawInscribedCircle(&sph.cif2, pBuf->id2);
					//	内接円中心から球中心への垂線
					AjcG3dLineV(hWndG3d, pBuf->id1, &sph.cif1.lvc.p, &pBuf->cent);
					AjcG3dLineV(hWndG3d, pBuf->id2, &sph.cif2.lvc.p, &pBuf->cent);
				}
			}
		}
		//	球の情報クリアー
		pBuf->cnt = 0;
		memset(pBuf->plt, 0, sizeof pBuf->plt);
	}
}
//--------------------------------------------------------------------------------------------------------------//
//	軌道円算出用プロット点の収集と描画																			//
//--------------------------------------------------------------------------------------------------------------//
static	VO		CirCorrectAndShow(PCAJC3DVEC pVec, PPLOTINFO pBuf)
{
	UI		i;

	//	プロット間の最低距離チェック
	for (i = 0; i < pBuf->cnt; i++) {
		if (AjcV3dDistP2P(pVec, &pBuf->plt[i]) < pBuf->dist) {
			break;
		}
	}
	//	プロット点の収集
	if (i >= pBuf->cnt) {
		memcpy(&pBuf->plt[i], pVec, sizeof(AJC3DVEC));
		pBuf->cnt++;
	}
	
	//	軌道円の算出と表示
	if (pBuf->cnt >= pBuf->num) {
		AJC3DCIRINFO	cif;
		if ((pBuf->radius = AjcV3dCalcCircleVEx(&pBuf->plt[0], &pBuf->plt[1], &pBuf->plt[2],
																		 &pBuf->cent, &pBuf->vec , &cif)) != -1) {
			if (AjcGetDlgItemChk(hDlgMain, IDC_CHK_CIRINFO)) {
				//	前の軌道円描画クリアー
				AjcG3dClearShape(hWndG3d, pBuf->id1);
				//	３点描画
				AjcG3dPixelV(hWndG3d, pBuf->id1, &pBuf->plt[0], 4);
				AjcG3dPixelV(hWndG3d, pBuf->id1, &pBuf->plt[1], 4);
				AjcG3dPixelV(hWndG3d, pBuf->id1, &pBuf->plt[2], 4);
				//	内接円描画
				DrawInscribedCircle(&cif, pBuf->id1);
			}
		}
		//	軌道円の情報クリアー
		pBuf->cnt = 0;
		memset(pBuf->plt, 0, sizeof pBuf->plt);
	}
}
//--------------------------------------------------------------------------------------------------------------//
//	球算出の内接円／軌道円の描画																				//
//--------------------------------------------------------------------------------------------------------------//
static	VO		DrawInscribedCircle(PCAJC3DCIRINFO pCif, int id)
{
	//	円に内接する２つの直線と端点
	AjcG3dLineV (hWndG3d, id, &pCif->lt1.p1, &pCif->lt1.p2);
	AjcG3dLineV (hWndG3d, id, &pCif->lt2.p1, &pCif->lt2.p2);
	AjcG3dPixelV(hWndG3d, id, &pCif->lt1.p1, 4);
	AjcG3dPixelV(hWndG3d, id, &pCif->lt1.p2, 4);
	AjcG3dPixelV(hWndG3d, id, &pCif->lt2.p1, 4);
	AjcG3dPixelV(hWndG3d, id, &pCif->lt2.p2, 4);
	//	内接線中点からの垂線
	AjcG3dLineV (hWndG3d, id, &pCif->lc1.p1, &pCif->lc1.p2);
	AjcG3dLineV (hWndG3d, id, &pCif->lc2.p1, &pCif->lc2.p2);
	//	内接円と内接円の中心点
	AjcG3dDefPlaneV(hWndG3d, id, &pCif->lvc, NULL);
	AjcG2dEllipse  (hWndG3d, id, 0, 0, pCif->cr, pCif->cr);
	AjcG2dPixel    (hWndG3d, id, 0, 0, 4);
}

//--------------------------------------------------------------------------------------------------------------//
//	ウインドスタイル値をチェックボックスに設定																	//
//--------------------------------------------------------------------------------------------------------------//
static	VO	SetStyleToCheckBox(VO)
{
	UI		sty;

	sty = (UI)MAjcGetWindowLong(hWndG3d, GWL_STYLE);
	AjcSetDlgItemChk(hDlgMain, IDC_CHK_NODEPTHCTRL , (sty & AJC3DGS_NODEPTHCTRL  ) != 0);
	AjcSetDlgItemChk(hDlgMain, IDC_CHK_NOBORDER    , (sty & AJC3DGS_NOBORDER	 ) != 0);
	AjcSetDlgItemChk(hDlgMain, IDC_CHK_NOFILTER    , (sty & AJC3DGS_NOFILTER	 ) != 0);
	AjcSetDlgItemChk(hDlgMain, IDC_CHK_NOANGLE	   , (sty & AJC3DGS_NOANGLE 	 ) != 0);
	AjcSetDlgItemChk(hDlgMain, IDC_CHK_AXIS_X	   , (sty & AJC3DGS_SHOWAXIS_X	 ) != 0);
	AjcSetDlgItemChk(hDlgMain, IDC_CHK_AXIS_Y	   , (sty & AJC3DGS_SHOWAXIS_Y	 ) != 0);
	AjcSetDlgItemChk(hDlgMain, IDC_CHK_AXIS_Z	   , (sty & AJC3DGS_SHOWAXIS_Z	 ) != 0);
	AjcSetDlgItemChk(hDlgMain, IDC_CHK_RECTSCALE   , (sty & AJC3DGS_RECTSCALE	 ) != 0);
	AjcSetDlgItemChk(hDlgMain, IDC_CHK_ELPSSCALE   , (sty & AJC3DGS_ELPSSCALE	 ) != 0);
	AjcSetDlgItemChk(hDlgMain, IDC_CHK_SPHERESCALE , (sty & AJC3DGS_SPHERESCALE  ) != 0);
	AjcSetDlgItemChk(hDlgMain, IDC_CHK_LINE_XY	   , (sty & AJC3DGS_SCALE_XY	 ) != 0);
	AjcSetDlgItemChk(hDlgMain, IDC_CHK_LINE_XZ	   , (sty & AJC3DGS_SCALE_XZ	 ) != 0);
	AjcSetDlgItemChk(hDlgMain, IDC_CHK_LINE_YZ	   , (sty & AJC3DGS_SCALE_YZ	 ) != 0);
	AjcSetDlgItemChk(hDlgMain, IDC_CHK_VAL_X	   , (sty & AJC3DGS_SCALEVALUE_X ) != 0);
	AjcSetDlgItemChk(hDlgMain, IDC_CHK_VAL_Y	   , (sty & AJC3DGS_SCALEVALUE_Y ) != 0);
	AjcSetDlgItemChk(hDlgMain, IDC_CHK_VAL_Z	   , (sty & AJC3DGS_SCALEVALUE_Z ) != 0);
}
//--------------------------------------------------------------------------------------------------------------//
//	チェックボックスの設定をスタイルに反映																		//
//--------------------------------------------------------------------------------------------------------------//
static	VO	SetCheckBoxToStyle(VO)
{
	UI		sty;

	sty = (UI)MAjcGetWindowLong(hWndG3d, GWL_STYLE);
	sty &= 0xFFFF0000;
	if (AjcGetDlgItemChk(hDlgMain, IDC_CHK_NODEPTHCTRL )) sty |= AJC3DGS_NODEPTHCTRL;
	if (AjcGetDlgItemChk(hDlgMain, IDC_CHK_NOBORDER    )) sty |= AJC3DGS_NOBORDER;
	if (AjcGetDlgItemChk(hDlgMain, IDC_CHK_NOFILTER    )) sty |= AJC3DGS_NOFILTER;
	if (AjcGetDlgItemChk(hDlgMain, IDC_CHK_NOANGLE	   )) sty |= AJC3DGS_NOANGLE;
	if (AjcGetDlgItemChk(hDlgMain, IDC_CHK_AXIS_X	   )) sty |= AJC3DGS_SHOWAXIS_X;
	if (AjcGetDlgItemChk(hDlgMain, IDC_CHK_AXIS_Y	   )) sty |= AJC3DGS_SHOWAXIS_Y;
	if (AjcGetDlgItemChk(hDlgMain, IDC_CHK_AXIS_Z	   )) sty |= AJC3DGS_SHOWAXIS_Z;
	if (AjcGetDlgItemChk(hDlgMain, IDC_CHK_RECTSCALE   )) sty |= AJC3DGS_RECTSCALE;
	if (AjcGetDlgItemChk(hDlgMain, IDC_CHK_ELPSSCALE   )) sty |= AJC3DGS_ELPSSCALE;
	if (AjcGetDlgItemChk(hDlgMain, IDC_CHK_SPHERESCALE )) sty |= AJC3DGS_SPHERESCALE;
	if (AjcGetDlgItemChk(hDlgMain, IDC_CHK_SCALE_XY    )) sty |= AJC3DGS_SCALE_XY;
	if (AjcGetDlgItemChk(hDlgMain, IDC_CHK_SCALE_XZ    )) sty |= AJC3DGS_SCALE_XZ;
	if (AjcGetDlgItemChk(hDlgMain, IDC_CHK_SCALE_YZ    )) sty |= AJC3DGS_SCALE_YZ;
	if (AjcGetDlgItemChk(hDlgMain, IDC_CHK_VAL_X	   )) sty |= AJC3DGS_SCALEVALUE_X;
	if (AjcGetDlgItemChk(hDlgMain, IDC_CHK_VAL_Y	   )) sty |= AJC3DGS_SCALEVALUE_Y;
	if (AjcGetDlgItemChk(hDlgMain, IDC_CHK_VAL_Z	   )) sty |= AJC3DGS_SCALEVALUE_Z;
	MAjcSetWindowLong(hWndG3d, GWL_STYLE, sty);
}
