﻿//
//	SW_StateCtrl.c
//

#include	<AjrCstXX.h>
#include	<math.h>
#include	<tchar.h>
#include	<mmsystem.h>

#include	"resource.h"

#define		SOUND_TIMES		3		//	鳴音回数

//--------------------------------------------------------------------------------------------------------------//
//	ワーク																										//
//--------------------------------------------------------------------------------------------------------------//
static	HINSTANCE		hInst;				//	ＤＬＬインスタンスハンドル
static	HWND			hDlgMain;			//	ダイアログボックスハンドル
static	HWND			hVth;				//	ＶＴＨウインド
static	HAJCSTC 		hStc;				//	状態遷移制御ハンドル
static	UI				CntSound;			//	鳴音回数カウンタ
static	UI				SrtTick;			//	開始時刻
static	UI				msTimer;			//	設定時間[ms]
static	UI				svSec;				//	経過秒数退避
static	HGLOBAL			hWavData;			//	Wavデータリソース
static	VOP				pWavData;

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
AJC_DLGPROC_DEF(Main);
static	VO		Log(C_UTP pFmt, ...);

//--------------------------------------------------------------------------------------------------------------//
//	状態																										//
//--------------------------------------------------------------------------------------------------------------//
enum {
	AJCSTC_STS_IDLE 	,			//	0: 待機状態
	AJCSTC_STS_ACTIVE	,			//	1: タイマ動作中
	AJCSTC_STS_RINGING	,			//	2: 鳴音中

	STS_NUM 						//	状態数
};

//--------------------------------------------------------------------------------------------------------------//
//	イベント																									//
//--------------------------------------------------------------------------------------------------------------//
enum {
	EVT_START	,					//	0: ＳＴＡＲＴボタン押下
	EVT_TMO_T0	,					//	1: タイムアウト監視タイマ(t0)タイムアウト
	EVT_TMO_T1	,					//	2: キッチンタイマ(t1)タイムアウト
	EVT_TMO_T2	,					//	3: 鳴音インタバルタイマ(t2)タイムアウト
	EVT_OVER	,					//	4: 鳴音回数満了
	EVT_STOP	,					//	5: ＳＴＯＰボタン押下

	EVT_NUM 						//	イベント数
};
//--------------------------------------------------------------------------------------------------------------//
//	プリセットファンクション																					//
//--------------------------------------------------------------------------------------------------------------//
enum {
	AJCSTC_PF_CNT_INI 	,			//	鳴音回数初期化
	AJCSTC_PF_SET_TIM	,			//	キッチンタイマ時間設定
	AJCSTC_PF_DSP_LAP	,			//	経過時間表示
	AJCSTC_PF_SOUND 	,			//	鳴音／鳴音回数満了チェック
	AJCSTC_PF_QUIET 	,			//	鳴音停止
	AJCSTC_PF_ENA_SRT	,			//	スタートボタン許可
	AJCSTC_PF_DIS_SRT	,			//	スタートボタン禁止
	AJCSTC_PF_ENA_STP	,			//	ストップボタン許可
	AJCSTC_PF_DIS_STP	,			//	ストップボタン禁止

	PF_NUM							//	プリセットファンクション数
};

#define TM	AJCTM
#define SM	AJCSM
#define PF	AJCPF

//--------------------------------------------------------------------------------------------------------------//
//	他の状態→自状態へ遷移時 (IDLE状態へ遷移時、STARTボタン許可，STOPボタン禁止，鳴音停止，全ﾀｲﾏ停止)			//
//--------------------------------------------------------------------------------------------------------------//
static const AJCSTC_STSACT ComTbl[STS_NUM] = {
//					 t0  t1  t2  t3 	 Pre-set functions
/* 0:IDLE	 */ { TM( R , R , R , _), PF( ENA_SRT , DIS_STP , QUIET   , __		, __	  , __		)},
/* 1:ACTIVE  */ { TM( _ , _ , _ , _), PF( __	  , __		, __	  , __		, __	  , __		)},
/* 2:RINGING */ { TM( _ , _ , _ , _), PF( __	  , __		, __	  , __		, __	  , __		)},
};
//--------------------------------------------------------------------------------------------------------------//
//	自状態→他の状態へ遷移時 (IDLE状態から他の状態へ遷移時、STARTボタン禁止，STOPボタン許可)					//
//--------------------------------------------------------------------------------------------------------------//
static const AJCSTC_STSACT RelTbl[STS_NUM] = {
//					 t0  t1  t2  t3 	 Pre-set functions
/* 0:IDLE	 */ { TM( _ , _ , _ , _), PF( DIS_SRT , ENA_STP , __	  , __		, __	  , __		)},
/* 1:ACTIVE  */ { TM( _ , _ , _ , _), PF( __	  , __		, __	  , __		, __	  , __		)},
/* 2:RINGING */ { TM( _ , _ , _ , _), PF( __	  , __		, __	  , __		, __	  , __		)},
};
//--------------------------------------------------------------------------------------------------------------//
//	状態遷移表																									//
//--------------------------------------------------------------------------------------------------------------//
static	const AJCSTC_EVT_NF  EvtTbl[ EVT_NUM ][ STS_NUM ] = {
//----- [ 0] - ＳＴＡＲＴボタン押下 ----------------------------------------------------------------------------//
//				  NextState 	   t0  t1  t2  t3 	   Pre-set functions
/* 0:IDLE	 */{{SM( ACTIVE  ), TM( C , S , _ , _), PF( CNT_INI , SET_TIM , DSP_LAP , __	  , __		, __	  )},
/* 1:ACTIVE  */ {SM( __ 	 ), TM( _ , _ , _ , _), PF( __		, __	  , __		, __	  , __		, __	  )},
/* 2:RINGING */ {SM( __ 	 ), TM( _ , _ , _ , _), PF( __		, __	  , __		, __	  , __		, __	  )}},

//----- [ 1] - 監視タイマ(t0)タイムアウト ----------------------------------------------------------------------//
//				  NextState  	  t0  t1  t2  t3 	   Pre-set functions
/* 0:IDLE	 */{{SM( __ 	 ), TM( _ , _ , _ , _), PF( __		, __	  , __		, __	  , __		, __	  )},
/* 1:ACTIVE  */ {SM( __ 	 ), TM( _ , _ , _ , _), PF( DSP_LAP , __	  , __		, __	  , __		, __	  )},
/* 2:RINGING */ {SM( __ 	 ), TM( _ , _ , _ , _), PF( __		, __	  , __		, __	  , __		, __	  )}},

//----- [ 2] - キッチンタイマ(t1)タイムアウト ------------------------------------------------------------------//
//				  NextState 	   t0  t1  t2  t3 	   Pre-set functions
/* 0:IDLE	 */{{SM( __ 	 ), TM( _ , _ , _ , _), PF( __		, __	  , __		, __	  , __		, __	  )},
/* 1:ACTIVE  */ {SM( RINGING ), TM( R , _ , C , _), PF( DSP_LAP , SOUND   , __		, __	  , __		, __	  )},
/* 2:RINGING */ {SM( __ 	 ), TM( _ , _ , _ , _), PF( __		, __	  , __		, __	  , __		, __	  )}},

//----- [ 3] - 鳴音インタバルタイマ(t2)タイムアウト ------------------------------------------------------------//
//				  NextState 	   t0  t1  t2  t3 	   Pre-set functions
/* 0:IDLE	 */{{SM( __ 	 ), TM( _ , _ , _ , _), PF( __		, __	  , __		, __	  , __		, __	  )},
/* 1:ACTIVE  */ {SM( __ 	 ), TM( _ , _ , _ , _), PF( __		, __	  , __		, __	  , __		, __	  )},
/* 2:RINGING */ {SM( __ 	 ), TM( _ , _ , _ , _), PF( __		, SOUND   , __		, __	  , __		, __	  )}},

//----- [ 4] - 鳴音回数満了 ------------------------------------------------------------------------------------//
//				  NextState 	   t0  t1  t2  t3 	   Pre-set functions
/* 0:IDLE	 */{{SM( __ 	 ), TM( _ , _ , _ , _), PF( __		, __	  , __		, __	  , __		, __	  )},
/* 1:ACTIVE  */ {SM( __ 	 ), TM( _ , _ , _ , _), PF( __		, __	  , __		, __	  , __		, __	  )},
/* 2:RINGING */ {SM( IDLE	 ), TM( _ , _ , _ , _), PF( __		, __	  , __		, __	  , __		, __	  )}},

//----- [ 5] - ＳＴＯＰボタン押下 ------------------------------------------------------------------------------//
//				  NextState 	   t0  t1  t2  t3 	   Pre-set functions
/* 0:IDLE	 */{{SM( __ 	 ), TM( _ , _ , _ , _), PF( __		, __	  , __		, __	  , __		, __	  )},
/* 1:ACTIVE  */ {SM( IDLE	 ), TM( _ , _ , _ , _), PF( __		, __	  , __		, __	  , __		, __	  )},
/* 2:RINGING */ {SM( IDLE	 ), TM( _ , _ , _ , _), PF( __		, __	  , __		, __	  , __		, __	  )}},
};
//--------------------------------------------------------------------------------------------------------------//
//	プリセットファンクション実行　コールバック関数																//
//																												//
//	引　数	：	pno    - プリセットファンクション番号															//
//				sts    - 現在の状態コード																		//
//				cbp    - コールバックパラメタ（未使用） 														//
//																												//
//	戻り値	：	なし																							//
//--------------------------------------------------------------------------------------------------------------//
static VO CALLBACK cbPsf(UI pno, int sts, UX cbp)
{
	switch (pno) {
		//------------------------------------------------------------------------------------------------------//
		case AJCSTC_PF_CNT_INI 	:			//	鳴音回数初期化
			//	ログ表示
			Log(TEXT("   Func: CNT_INI   - 鳴音回数初期化\n"));
			//	鳴音回数カウンタクリアー
			CntSound = 0;
			break;
		//------------------------------------------------------------------------------------------------------//
		case AJCSTC_PF_SET_TIM 	:			//	キッチンタイマ時間設定
		{	UI	tm = AjcGetDlgItemUInt(hDlgMain, IDC_INP_MINUTE) * 60000 + AjcGetDlgItemUInt(hDlgMain, IDC_INP_SECOND) * 1000;
			//	ログ表示
			Log(TEXT("   Func: SET_TIM   - キッチンタイマ時間設定(t1=%dms)\n"), tm);
			//	タイマ情報設定
			SrtTick = GetTickCount();
			AjcStcSetTimerInfo(hStc, 1, tm, EVT_TMO_T1);
			break;
		}
		//------------------------------------------------------------------------------------------------------//
		case AJCSTC_PF_DSP_LAP	:			//	経過時間表示
		{	UI	tm, second, minute;
			UT	txt[128];
			//	ログ表示（このログは頻繁に出力されるため割愛する）
			//	Log(TEXT("   Func: DSP_LAP   - 経過時間表示\n"));
			//	経過時間表示
			tm = GetTickCount() - SrtTick;
			second = tm / 1000;
			minute = second / 60;
			if (second == 0 || second != svSec) {
				AjcSnPrintF(txt, AJCTSIZE(txt), TEXT("%02d : %02d"), minute, second % 60);
				AjcSetDlgItemStr(hDlgMain, IDC_TXT_LAP, txt);
			}
			svSec = second;
			break;
		}
		//------------------------------------------------------------------------------------------------------//
		case AJCSTC_PF_SOUND 	:			//	鳴音
		{	UT	txt[128];
			//	鳴音回数更新
			CntSound++;
			//	ログ表示
			AjcSnPrintF(txt, AJCTSIZE(txt), TEXT("鳴音(%d)"), CntSound);
			Log(TEXT("   Func: SOUND     - %s\n"), txt);
			//	鳴音回数未了
			if (CntSound <= SOUND_TIMES) {
				RECT	r;
				int		w, h;
				SIZE	sz;
				//	ウインド中央にツールチップで「鳴音」を表示
				GetWindowRect(hDlgMain, &r);
				w = r.right - r.left;
				h = r.bottom - r.top;
				AjcTipTextGetSize(txt, NULL, TRUE, &sz);
				AjcTipTextShow(r.left + (w - sz.cx) / 2, r.top + (h - sz.cy) / 2, txt, 1000, NULL);
				//	ビープ音
				PlaySound((UTP)pWavData, NULL, SND_MEMORY | SND_ASYNC);
			}
			//	鳴音回数満了
			else {
				AjcStcExecEvent(hStc, EVT_OVER);
			}
			break;
		}
		//------------------------------------------------------------------------------------------------------//
		case AJCSTC_PF_QUIET 	:			//	鳴音停止
			//	ログ表示
			Log(TEXT("   Func: QUIET     - 鳴音停止\n"));
			//	ツールチップ消去
			AjcTipTextHide();
			//	鳴音停止
			PlaySound(NULL, NULL, 0);
			break;
		//------------------------------------------------------------------------------------------------------//
		case AJCSTC_PF_ENA_SRT 	:			//	スタートボタン許可
			//	ログ表示
			Log(TEXT("   Func: ENA_SRT   - スタートボタン許可\n"));
			//	スタートボタン許可
			AjcEnableDlgGroup(hDlgMain, IDC_GRP_START, TRUE, TRUE);
			break;
		//------------------------------------------------------------------------------------------------------//
		case AJCSTC_PF_DIS_SRT 	:			//	スタートボタン禁止
			//	ログ表示
			Log(TEXT("   Func: DIS_SRT   - スタートボタン禁止\n"));
			//	スタートボタン禁止
			AjcEnableDlgGroup(hDlgMain, IDC_GRP_START, FALSE, FALSE);
			break;
		//------------------------------------------------------------------------------------------------------//
		case AJCSTC_PF_ENA_STP 	:			//	ストップボタン許可
			//	ログ表示
			Log(TEXT("   Func: ENA_STP   - ストップボタン許可\n"));
			//	ストップボタン許可
			AjcEnableDlgItem(hDlgMain, IDC_CMD_STOP, TRUE);
			break;
		//------------------------------------------------------------------------------------------------------//
		case AJCSTC_PF_DIS_STP 	:			//	ストップボタン禁止
			//	ログ表示
			Log(TEXT("   Func: DIS_STP   - ストップボタン禁止\n"));
			//	ストップボタン禁止
			AjcEnableDlgItem(hDlgMain, IDC_CMD_STOP, FALSE);
			break;
	}
}
//--------------------------------------------------------------------------------------------------------------//
//	状態遷移通知　コールバック関数																				//
//																												//
//	引　数	：	ntc    - 通知コード 																			//
//				p1, p2 - 通知パラメタ																			//
//				cbp    - コールバックパラメタ（未使用） 														//
//																												//
//	戻り値	：	なし																							//
//--------------------------------------------------------------------------------------------------------------//
static C_UTP	pSts[STS_NUM] = {_T("IDLE   (待機状態)    "),
								 _T("ACTIVE (タイマ動作中)"),
								 _T("RINGING(鳴音中)      "),
				};
static C_UTP	pEvt[EVT_NUM] = {_T("START (ＳＴＡＲＴボタン押下)")  ,
								 _T("TMO_T0(タイムアウト監視タイマ(t0)タイムアウト)"),
								 _T("TMO_T1(キッチンタイマ(t1)タイムアウト)"),
								 _T("TMO_T2(鳴音インタバルタイマ(t2)タイムアウト)"),
								 _T("OVER  (鳴音回数満了)"),
								 _T("STOP  (ＳＴＯＰボタン押下)"),
				};

static	VO CALLBACK cbNtc(AJCSTC_NTC ntc, UL p1, UL p2, UX cbp)
{
	switch (ntc) {
		case AJCSTC_NTC_EVT:	Log(TEXT("状態 = %s, イベント = %s\n"  ), pSts[p1], pEvt[p2]);	break;
		case AJCSTC_NTC_STS:	Log(TEXT("   状態 遷移 --> %s\n"	   ), pSts[p1]			);	break;
		case AJCSTC_NTC_TIMSRT: Log(TEXT("   タイマ %d START (%dms)\n" ), p1, p2			);	break;
		case AJCSTC_NTC_TIMSTP: Log(TEXT("   タイマ %d STOP\n"		   ), p1				);	break;
	}
}

//==============================================================================================================//
//																												//
//	W i n M a i n																								//
//																												//
//==============================================================================================================//
int WINAPI AjcWinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance, UTP szCmdLine, int iCmdShow)
{
	MSG 	msg;

	hInst = hInstance;

	//	メイン・ダイアログオープン	
	hDlgMain = CreateDialog(hInst, MAKEINTRESOURCE(IDD_DLGMAIN), NULL, AJC_DLGPROC_NAME(Main));
	//	ダイアログ表示	
	ShowWindow(hDlgMain, SW_SHOW);

	//	メッセージループ	
	while (GetMessage(&msg, NULL, 0, 0)) {
		do {
			if (IsDialogMessage(hDlgMain, &msg)) break;
			TranslateMessage(&msg);
			DispatchMessage (&msg);
		} while (0);
	}

	return (int)msg.wParam ;
}
//==============================================================================================================//
//																												//
//	ダイアログ・プロシージャ																					//
//																												//
//==============================================================================================================//
//----- ダイアログ初期化 ---------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, WM_INITDIALOG 	)
{
	hDlgMain = hDlg;
	hVth	 = GetDlgItem(hDlgMain, IDC_VTH);

	//	Wavリソースロード
	hWavData = LoadResource(hInst,
					FindResource(hInst, MAKEINTRESOURCE(IDR_WAV1), TEXT("WAV")));
	pWavData = LockResource(hWavData);

	//	チップテキストのフォント設定
	AjcTipTextSetDefFont((HFONT)SendMessage(hDlg, WM_GETFONT, 0, 0));

	//	状態遷移制御インスタンス生成	
	hStc = AjcStcCreate(ComTbl, RelTbl, EvtTbl[0], STS_NUM, 0, cbPsf, cbNtc);
	//	タイマにイベントを関連付け
	AjcStcSetTimerInfo(hStc, 0,  100, EVT_TMO_T0); 	//	0: 監視タイマ
	AjcStcSetTimerInfo(hStc, 2, 5000, EVT_TMO_T2); 	//	2: 鳴音インタバルタイマ

	return TRUE;
}
//----- ウインド破棄 -------------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, WM_DESTROY		)
{
	//		状態遷移制御インスタンス解放
	AjcStcDelete(hStc);
	//	WAVリソース解放
	FreeResource(hWavData);
	//	プログラム終了
	PostQuitMessage(0);
	return TRUE;
}
//----- STARTボタン --------------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, IDC_CMD_START		)
{
	//	ログクリアー
	AjcVthClear(hVth);
	//	スタートボタン押下イベント
	AjcStcExecEvent(hStc, EVT_START);
	return TRUE;
}
//----- STOPボタン ---------------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, IDC_CMD_STOP		)
{
	//	ストップボタン押下イベント
	AjcStcExecEvent(hStc, EVT_STOP);
	return TRUE;
}
//----- ウインドクローズ ---------------------------------------------------------------------------------------//
AJC_DLGPROC(Main, IDCANCEL			)
{
	//	ストップボタン押下イベント
	AjcStcExecEvent(hStc, EVT_STOP);
	//	ウインドクローズ
	DestroyWindow(hDlg);
	return TRUE;
}
//--------------------------------------------------------------------------------------------------------------//
AJC_DLGMAP_DEF(Main)
	AJC_DLGMAP_MSG(Main, WM_INITDIALOG	   )
	AJC_DLGMAP_MSG(Main, WM_DESTROY 	   )
	AJC_DLGMAP_CMD(Main, IDC_CMD_START	   )
	AJC_DLGMAP_CMD(Main, IDC_CMD_STOP	   )
	AJC_DLGMAP_CMD(Main, IDCANCEL		   )
AJC_DLGMAP_END
//--------------------------------------------------------------------------------------------------------------//
//	ログ出力																									//
//--------------------------------------------------------------------------------------------------------------//
static	VO		Log(C_UTP pFmt, ...)
{
	static	BOOL fNeedLF = FALSE;
	static	UT	 svTxt[1024];
	va_list 	 vls;
	UT			 txt[1024] = {0};


	va_start(vls, pFmt);
	AjcVSnPrintF(txt, 1024, pFmt, vls);
	txt[1023] = 0;
	va_end	(vls);

	AjcVthTimeStamp(hVth);
	AjcVthPutText(hVth, TEXT(" "), -1);
	if (MAjcStrCmp(svTxt, txt) != 0) {
		if (fNeedLF) {
			AjcVthPutText(hVth, TEXT("\n"), -1);
			fNeedLF = FALSE;
		}
		AjcVthPutText(hVth, TEXT(" "), -1);
		AjcVthPutText(hVth, txt, -1);
		MAjcStrCpy(svTxt, AJCTSIZE(svTxt), txt);
	}
	else {
		AjcVthPutText(hVth, TEXT("・・・同上・・・\r"), -1);
	}
}

