document.addEventListener('DOMContentLoaded', () => {
    const verse1Input = document.getElementById('verse1');
    const verse2Input = document.getElementById('verse2');
    const visualizer = document.getElementById('visualizer');

    function updateVisualizer() {
        const text1 = verse1Input.value;
        const text2 = verse2Input.value;

        // Split by lines
        const lines1 = text1.split('\n');
        const lines2 = text2.split('\n');

        // Determine the maximum number of lines
        const maxLines = Math.max(lines1.length, lines2.length);

        if (maxLines === 0 || (lines1.length === 1 && lines1[0] === '' && lines2.length === 1 && lines2[0] === '')) {
            visualizer.innerHTML = '<p class="placeholder-text">歌詞を入力すると、ここに比較が表示されます。</p>';
            return;
        }

        let html = '';

        for (let i = 0; i < maxLines; i++) {
            const line1 = lines1[i] || '';
            const line2 = lines2[i] || '';

            // Custom counting logic
            const count1 = countCharacters(line1);
            const count2 = countCharacters(line2);
            const diff = Math.abs(count1 - count2);

            const isMatch = (count1 === count2);
            const statusClass = isMatch ? 'status-match' : 'status-mismatch';

            html += `
            <div class="comparison-row">
                <div class="verse-line">
                    <label>1</label>
                    <span class="text">${escapeHtml(line1)}</span>
                    <span class="count">${count1}</span>
                </div>
                <div class="verse-line">
                    <label>2</label>
                    <span class="text">${escapeHtml(line2)}</span>
                    <span class="count">${count2}</span>
                </div>
                <div class="status-indicator ${statusClass}" title="${isMatch ? '文字数一致' : '文字数差: ' + diff}"></div>
            </div>
            `;
        }

        visualizer.innerHTML = html;
    }

    function countCharacters(text) {
        if (!text) return 0;

        // 1. Remove ignored characters (spaces, brackets, punctuation, symbols)
        // Includes: Space, Full-width Space, (), （）, Punctuation, Nakaguro, Ellipsis
        let cleanText = text.replace(/[\s\u3000\(\)（）「」『』\[\]【】、。,.!?;:！？・…‥]/g, "");

        // 2. Remove small characters that should not be counted individually
        // ぁぃぅぇぉゃゅょ (User specified)
        // Note: These are combined with the previous char to form 1 mora/sound, so we just remove them to count the pair as 1.
        // "きゃ" -> "き" (len 1). "きっ" -> "きっ" (len 2) because "っ" is not removed.
        cleanText = cleanText.replace(/[ぁぃぅぇぉゃゅょ]/g, "");

        return cleanText.length;
    }

    function escapeHtml(text) {
        return text
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;")
            .replace(/ /g, "&nbsp;"); // Maintain spaces
    }

    // Add event listeners
    verse1Input.addEventListener('input', updateVisualizer);
    verse2Input.addEventListener('input', updateVisualizer);

    // active line highlighting logic
    function updateActiveLine(event) {
        const textarea = event.target;
        const value = textarea.value;
        const selectionStart = textarea.selectionStart;

        // Calculate line number (0-indexed)
        const lineNumber = value.substr(0, selectionStart).split("\n").length - 1;

        highlightRow(lineNumber);
    }

    function highlightRow(index) {
        const rows = visualizer.querySelectorAll('.comparison-row');
        rows.forEach((row, i) => {
            if (i === index) {
                row.classList.add('active-row');
                // Optional: Scroll into view if needed, but might be jumpy
                row.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
            } else {
                row.classList.remove('active-row');
            }
        });
    }

    // Attach listeners for cursor movement
    ['click', 'keyup', 'focus'].forEach(evt => {
        verse1Input.addEventListener(evt, updateActiveLine);
        verse2Input.addEventListener(evt, updateActiveLine);
    });

    // Also update highlight when input changes (in case line count changes)
    verse1Input.addEventListener('input', (e) => {
        // Wait for visualizer to rebuild, then highlight
        requestAnimationFrame(() => updateActiveLine(e));
    });
    verse2Input.addEventListener('input', (e) => {
        requestAnimationFrame(() => updateActiveLine(e));
    });

    // Goriawase Logic
    const goriawaseBtn = document.getElementById('goriawaseBtn');

    goriawaseBtn.addEventListener('click', () => {
        const text1 = verse1Input.value;
        const text2 = verse2Input.value;
        const lines1 = text1.split('\n');
        const lines2 = text2.split('\n');
        const maxLines = Math.max(lines1.length, lines2.length);

        let newLines1 = [...lines1];
        let newLines2 = [...lines2];

        for (let i = 0; i < maxLines; i++) {
            let line1 = newLines1[i] || '';
            let line2 = newLines2[i] || '';

            const count1 = countCharacters(line1);
            const count2 = countCharacters(line2);

            if (count1 === count2) continue;

            const diff = Math.abs(count1 - count2);

            if (count1 < count2) {
                // Extend line 1
                newLines1[i] = extendLine(line1, diff);
            } else {
                // Extend line 2
                newLines2[i] = extendLine(line2, diff);
            }
        }

        // Update Textareas
        verse1Input.value = newLines1.join('\n');
        verse2Input.value = newLines2.join('\n');

        // Trigger visualizer update
        updateVisualizer();
    });

    function extendLine(line, count) {
        if (!line || count <= 0) return line;

        // Regex for ignored chars (including spaces)
        // Note: We use this to find "valid" chars to determine vowel.
        const ignoreRegex = /[\s\u3000\(\)（）「」『』\[\]【】、。,.!?;:！？・…‥]/;

        // Helper to find last valid character index
        function getLastValidIndex(str, upTo = str.length - 1) {
            for (let i = upTo; i >= 0; i--) {
                if (!ignoreRegex.test(str[i])) {
                    return i;
                }
            }
            return -1;
        }

        const lastIdx = getLastValidIndex(line);
        if (lastIdx === -1) {
            // No valid characters found, just append 'あ'
            return line + 'あ'.repeat(count);
        }

        const lastChar = line[lastIdx];

        // Handling 'ん' case
        if (lastChar === 'ん') {
            // Find the character before 'ん' to determine vowel
            // "みかん！" -> lastIdx is 'ん'
            const prevIdx = getLastValidIndex(line, lastIdx - 1);
            if (prevIdx !== -1) {
                const prevChar = line[prevIdx];
                const vowel = getVowel(prevChar);
                // Insert vowels BEFORE 'ん'
                // "みかん" -> "みか" + vowels + "ん"
                // "みかん！" -> "みか" + vowels + "ん" + "！"
                // line.slice(0, lastIdx) gets everything before 'ん'
                // line.slice(lastIdx) gets 'ん' and everything after (punctuation)
                return line.slice(0, lastIdx) + vowel.repeat(count) + line.slice(lastIdx);
            } else {
                // 'ん' is the only valid char, just duplicate 'ん'
                // "ん" -> "ん" + "んん"
                // insert before 'ん' (at lastIdx)
                return line.slice(0, lastIdx) + 'ん'.repeat(count) + line.slice(lastIdx);
            }
        }

        // Standard case (not 'ん')
        const vowel = getVowel(lastChar);
        // Insert AFTER the last valid character (before any trailing spaces/punctuation)
        // "いいへんさ　" -> lastIdx is 'さ' (index 4)
        // slice(0, 5) -> "いいへんさ"
        // slice(5) -> "　"
        // Result: "いいへんさ" + "ああ" + "　"
        return line.slice(0, lastIdx + 1) + vowel.repeat(count) + line.slice(lastIdx + 1);
    }

    function getVowel(char) {
        // Hiragana Vowel Map
        const a_col = 'あかさたなはまやらわがざだばぱぁゃ';
        const i_col = 'いきしちにひみりゐぎじぢびぴぃ';
        const u_col = 'うくすつぬふむゆるぐずづぶぷぅゅっ'; // Treat 'tsu' small as 'u' sound? or just 't'? User said "ki-t" is 2 chars. Usually extended as "u" or silence. Let's use 'う' for 'っ'.
        const e_col = 'えけせてねへめれゑげぜでべぺぇ';
        const o_col = 'おこそとのほもよろをごぞどぼぽぉょ';
        const n_col = 'ん';

        if (a_col.includes(char)) return 'あ';
        if (i_col.includes(char)) return 'い';
        if (u_col.includes(char)) return 'う';
        if (e_col.includes(char)) return 'え';
        if (o_col.includes(char)) return 'お';
        if (n_col.includes(char)) return 'ん';

        // Katakana Vowel Map (Simple mapping by converting to Hiragana code range or explicit list)
        // Check Katakana ranges
        const code = char.charCodeAt(0);
        if (code >= 0x30A1 && code <= 0x30F6) {
            // Simple Katakana mapping is tedious to list all.
            // Maybe just return 'ー' (long vowel) for Katakana/Kanji as safe fallback?
            // User example was Hiragana.
            // Let's try to be smart for Katakana too if easy.
            // Actually, easiest is to assume 'ー' for anything else.
            return 'ー';
        }

        // Kanji or others -> 'ー'
        return 'ー';
    }

    // Initial call
    updateVisualizer();
});
