/*
 * _dcs.h
 *
 * Copyright 2019-2026 Ichiji Tadokoro. All Rights Reserved.
 */

#ifndef __DCS_H_

#define __DCS_H_

#include <limits.h>
#include <stdint.h>
#include <stdarg.h>
#include "_dcs_inter.h"
#include "ll_list.h"

/* V2.10: UTF-16 全角が ASCII でもある場合、ASCII を保つ */
#define _KEEP_ASCII		1

#define _MAX_JIS_OFFSET		32
#define _MAX_SJIS_OFFSET	2
#define _MAX_EUC_OFFSET		3
#define _MAX_UNIBE_OFFSET	4
#define _MAX_UNILE_OFFSET	4
#define _MAX_UTF8_OFFSET	4
#define _MAX_OFFSET			_MAX_JIS_OFFSET

#define _SHIFT_OUT		'B'

#define _ZEN_TILDE_0		0x20	/* '~' */
#define _ZEN_TILDE_1		0x3e
#define _UYEN				0xa5	/* '¥' */

#define _MIN_VAL		(INT_MIN / 2)
#define _MAX_VAL		(INT_MAX / 2)

#define _DEFINE_BINARY_RANGE		0

#define _BINARY_RANGE(bsize, ssize) \
	(((double)bsize) / ((double)ssize * ssize) >= 0.0000005)

#define _BAD_RANGE(mbad, sbad) \
		((double)(mbad) * 5 < (double)(sbad))

#define _ESC_CHAR			0x1b
#define _JIS_ROMAN_MAX	0x7f

/* UTF32コード指定 _uni.eto_...(), ...to_uni.e() */
#define _UTYPE_FROM_OTHER		000000001
#define _UTYPE_FROM_UTF32		000000002
#define _UTYPE_TO_OTHER		000000004
#define _UTYPE_TO_UTF32		000000010
#define _UTYPE_FROM_OTHER_TO_OTHER	(_UTYPE_FROM_OTHER|_UTYPE_TO_OTHER)
#define _UTYPE_FROM_OTHER_TO_UTF32	(_UTYPE_FROM_OTHER|_UTYPE_TO_UTF32)
#define _UTYPE_FROM_UTF32_TO_OTHER		(_UTYPE_FROM_UTF32|_UTYPE_TO_OTHER)
#define _UTYPE_FROM_UTF32_TO_UTF32		(_UTYPE_FROM_UTF32|_UTYPE_TO_UTF32)

typedef int64_t	dcskey_t;

typedef struct code_info {
	int type;
	int nd_type;
	dcskey_t zen_cnt;
	bool maybe;
	dcskey_t bad_cnt;
	dcskey_t zen_kana_cnt;
	dcskey_t han_kana_cnt;
	dcskey_t tan_han_kana_cnt;
	dcskey_t blen;
} CodeInfo;

#define _I_SJIS		0
#define _I_SJIS1		1
#define _I_SJIS2		2
#define _I_EUC		3
#define _I_EUC1		4
#define _I_EUC2		5
#define _I_EUC3		6
#define _I_EUC4		7
#define _I_UTF8		8
#define _I_UTF16BE	9
#define _I_UTF16LE	10
#define _I_UTF32BE	11
#define _I_UTF32LE	12
#define _I_MAX		13

/* データ・タイプ */
#define _DTYPE_NON			0000000000 /* 未定義 */
#define _DTYPE_BUFFER		0000000010 /* バッファー */
#define _DTYPE_STREAM		0000000020 /* ストリーム */
#define _DTYPE_FILE_OPEN	0000000040 /* ファイル・オープン */
#define _DTYPE_FILE			0000000060 /* ファイル */
#define _DTYPE_NATIVE		0000000070 /* すべてのネイティヴ・データ・タイプ */
#define _DTYPE_JNI_FLAG		0000000100 /* Java ネイティブ・インターフェース */
#define _DTYPE_JNI_BUFFER	0000000110 /* バッファー */
#define _DTYPE_JNI_FILE		0000000160 /* ファイル */
#define _DTYPE_JNI_STREAM	0000000300 /* ストリーム(ネイティブのフラグはない) */
#define _DTYPE_ALL			0000000370 /* すべてのデータ・タイプ */

/* データ・タイプ取得 */
#define _get_data_type(dcs)				((dcs)->_type & _DTYPE_ALL)

/* データ・タイプ設定 */
#define _set_data_type(dcs, type) { \
		if (((dcs)->_type & _DTYPE_ALL) == 0) { (dcs)->_type |= (type & _DTYPE_ALL); } }

/* データ・タイプ・クリア */
#define _clear_data_type(dcs, type)		((dcs)->_type &= ~_DTYPE_ALL)

/* データ・タイプがファイル */
#define _is_type_file(dcs)				(((dcs)->_type & _DTYPE_FILE) == _DTYPE_FILE)

/* データ・タイプがストリーム */
#define _is_type_stream(dcs)			(((dcs)->_type & _DTYPE_STREAM) == _DTYPE_STREAM)

/* データ・タイプがバッファ */
#define _is_type_buffer(dcs)			(((dcs)->_type & _DTYPE_BUFFER) == _DTYPE_BUFFER)

/* データ・タイプがJNIファイル */
#define _is_type_jni_file(dcs)			(((dcs)->_type & _DTYPE_JNI_FILE) == _DTYPE_JNI_FILE)

/* データ・タイプがJNIストリーム(ネイティブのフラグはない) */
#define _is_type_jni_stream(dcs)		(((dcs)->_type & _DTYPE_JNI_STREAM) == _DTYPE_JNI_STREAM)

/* データ・タイプがJNIバッファ */
#define _is_type_jni_buffer(dcs)		(((dcs)->_type & _DTYPE_JNI_BUFFER) == _DTYPE_JNI_BUFFER)

/* 処理ステータス設定、リターン */
#define DCS_RETURN(dcs, _dcs_status_)		return((dcs)->sts = (_dcs_status_))
/* 処理ステータス設定、リターン */
#define DCS_RETURN_COUNT(dcs, _dcs_status_, cnt)		{ (dcs)->sts = (_dcs_status_); return (cnt); }

#define _dcs_push_back(dcs, c)	(*(dcs)->_bcp++ = (c))

#define _dcs_getc(dcs)	((dcs)->_bcp > (dcs->_btp)? --*(dcs)->_bcp: \
		_is_type_buffer(dcs)? \
		(dcs)->_sp >= (dcs)->_ep? EOF: *(dcs)->_sp++: _is_type_stream(dcs)? getc((dcs)->_sfp): \
		(*((JniJudgedStreamInfo *)(dcs)->_jni_info)->jenv)->CallIntMethod(((JniJudgedStreamInfo *)(dcs)->_jni_info)->jenv, \
				((JniJudgedStreamInfo *)(dcs)->_jni_info)->jistream, ((JniJudgedStreamInfo *)(dcs)->_jni_info)->mid_char))

#define _add_buffer(list, data, size) \
	(ll_add(ll_last(list), data, size))

#define check_ll_buf(dcs, sp, ep, max_length, clist, tlist, binfo, tbinfo) { \
if ((sp) + (max_length) > (ep)) { \
/* 残りが最大長に満たない時 */ \
	if (((tlist) = ll_next(clist)) != NULL) { \
		/* 長さを調整 */ \
		(binfo)->length -= ((ep) - (sp)); \
		/* NOCHAR オフセットを更新 */ \
		(dcs)->_no_char_offset += ((binfo)->buf + (binfo)->length) - ((binfo)->buf + (binfo)->offset); \
		/* 次のバッファを取得 */ \
		(tbinfo) = ll_get(tlist); \
		/* 残りを次の先頭にコピー */ \
		memcpy((tbinfo)->buf + (tbinfo)->offset - ((ep) - (sp)), (sp), (ep) - (sp)); \
		/* オフセットを調整 */ \
		(tbinfo)->offset -= (ep) - (sp); \
		(binfo) = (tbinfo); \
		(clist) = (tlist);  \
		(sp) = (dcs)->_no_char_ptr = (binfo)->buf + (binfo)->offset; \
		(ep) = (binfo)->buf + (binfo)->length; \
	} \
} }

/* sts = 0 => CS_UEXPEOF */
#define _check_buf(dp, tbuf, tsize, wbuf, sts) {	\
	if (dp - tbuf < tsize - _MAX_JIS_OFFSET) {	\
		sts = 0;	\
	} else {	\
		wbuf = malloc(tsize * 2);	\
		if (wbuf == NULL) {	\
			sts = DCS_EALLOC;	\
		} else {	\
			tsize *= 2;	\
			memcpy(wbuf, tbuf, dp - tbuf);	\
			dp = wbuf + (dp - tbuf);	\
			free(tbuf);	\
			tbuf = wbuf;	\
			sts = 0;	\
		}	\
	} }

/* 出力バッファ・サイズを超える時、ライター出力(エラー時、break) */
#define check_info_buf_break(dcs, dp, dbuf, dsize, sts) {	\
	if (dp - dbuf >= dsize) { \
	/* バッファ・サイズに達した時 */ \
		sts = (dcs->_writer == NULL? \
				DCS_SUCCESS: dcs->_writer((const char *)dbuf, dp - dbuf, dcs->_user_info)); \
		if (sts == DCS_SUCCESS) { \
			dp = dbuf; \
		} else { \
			break; \
		} \
	} }

/* 出力バッファ・サイズを超える時、ライター出力(エラー時、goto) */
#define check_info_buf_goto(dcs, dp, dbuf, dsize, sts, label) {	\
	if (dp - dbuf >= dsize) { \
	/* バッファ・サイズに達した時 */ \
		sts = (dcs->_writer == NULL? \
				DCS_SUCCESS: dcs->_writer((const char *)dbuf, dp - dbuf, dcs->_user_info)); \
		if (sts == DCS_SUCCESS) { \
			dp = dbuf; \
		} else { \
			goto label; \
		}	\
	} }

/* ブレイク時、出力バッファの残りをライター出力 */
#define check_info_buf_rem(dcs, dp, dbuf, sts) {	\
	if (dcs->_writer != NULL && dp - dbuf > 0) {	\
	/* 出力バッファの残りがある時 */ \
		sts = dcs->_writer((const char *)dbuf, dp - dbuf, dcs->_user_info);	\
	} else { \
		sts = DCS_SUCCESS; \
	} }

int _convert_buf(Dcs *dcs, char *message, size_t ssize, int type);
int _tcompare(const void *info1, const void *info2);

/* 改行変換 STYPE, DTYPE(ascii, unibe, unile)*/
#define cnvln_stype_to_jis(dp, c, sp, ep, ltype, STYPE) \
		cnvln_stype_to_dtype(dp, c, sp, ep, ltype, STYPE, ascii)
#define cnvln_stype_to_sjis(dp, c, sp, ep, ltype, STYPE) \
		cnvln_stype_to_dtype(dp, c, sp, ep, ltype, STYPE, ascii)
#define cnvln_stype_to_euc(dp, c, sp, ep, ltype, STYPE) \
		cnvln_stype_to_dtype(dp, c, sp, ep, ltype, STYPE, ascii)
#define cnvln_stype_to_unibe(dp, c, sp, ep, ltype, STYPE) \
		cnvln_stype_to_dtype(dp, c, sp, ep, ltype, STYPE, unibe)
#define cnvln_stype_to_unile(dp, c, sp, ep, ltype, STYPE) \
		cnvln_stype_to_dtype(dp, c, sp, ep, ltype, STYPE, unile)
#define cnvln_stype_to_utf8(dp, c, sp, ep, ltype, STYPE) \
		cnvln_stype_to_dtype(dp, c, sp, ep, ltype, STYPE, ascii)
#define cnvln_stype_to_uni32be(dp, c, sp, ep, ltype, STYPE) \
		cnvln_stype_to_dtype(dp, c, sp, ep, ltype, STYPE, uni32be)
#define cnvln_stype_to_uni32le(dp, c, sp, ep, ltype, STYPE) \
		cnvln_stype_to_dtype(dp, c, sp, ep, ltype, STYPE, uni32le)

#define cnvln_stype_to_dtype(dp, c, sp, ep, ltype, STYPE, DTYPE)	{ \
		int _tc_; \
		if (c == '\n') { \
			cnvln_ln_to_dtype(dp, c, ltype, DTYPE); \
		} else if (c == '\r') { \
			cnvln_next_ ## STYPE (dp, _tc_, sp, ep); \
			cnvln_cr_to_dtype(dp, c, ltype, _tc_,  DTYPE ); \
		} else { \
			cnvln_to_ ## DTYPE (dp, c); \
		} }

#define cnvln_to_ascii(dp, c) { \
		*dp++ = c; }

#define cnvln_to_utf8(dp, c)	cnvln_to_ascii(dp, c)

#define cnvln_to_unibe(dp, c) { \
		*dp++ = 0; \
		*dp++ = c; }

#define cnvln_to_unile(dp, c) { \
		*dp++ = c; \
		*dp++ = 0; }

#define cnvln_to_uni32be(dp, c) { \
		*dp++ = 0; \
		*dp++ = 0; \
		*dp++ = 0; \
		*dp++ = c; }

#define cnvln_to_uni32le(dp, c) { \
		*dp++ = c; \
		*dp++ = 0; \
		*dp++ = 0; \
		*dp++ = 0; }

#define cnvln_next_jis(dp, next_char, sp, ep) \
		cnvln_next_ascii(dp, next_char, sp, ep)

#define cnvln_next_sjis(dp, next_char, sp, ep) \
		cnvln_next_ascii(dp, next_char, sp, ep)

#define cnvln_next_euc(dp, next_char, sp, ep) \
		cnvln_next_ascii(dp, next_char, sp, ep)

#define cnvln_next_utf8(dp, next_char, sp, ep) \
		cnvln_next_ascii(dp, next_char, sp, ep)

#define cnvln_next_ascii(dp, next_char, sp, ep) { \
	if (sp < ep && *sp == '\n') { \
		sp++; \
		next_char = '\n'; \
	} else { \
		next_char = 0; \
	} }

#define cnvln_next_unibe(dp, next_char, sp, ep) { \
	if (sp + 1 < ep && *sp == 0 && *(sp + 1) == '\n') { \
		sp++; \
		sp++; \
		_tc_ = '\n'; \
	} else { \
		_tc_ = 0; \
	} }

#define cnvln_next_unile(dp, next_char, sp, ep) { \
	if (sp + 1 < ep && *sp == '\n' && *(sp + 1) == 0) { \
		sp++; \
		sp++; \
		_tc_ = '\n'; \
	} else { \
		_tc_ = 0; \
	} }

#define cnvln_next_uni32be(dp, next_char, sp, ep) { \
	if (sp + 3 < ep && *sp == 0 && *(sp + 1) == 0 && *(sp + 2) == 0 && *(sp + 3) == '\n') { \
		sp += 4; \
		_tc_ = '\n'; \
	} else { \
		_tc_ = 0; \
	} }

#define cnvln_next_uni32le(dp, next_char, sp, ep) { \
	if (sp + 3 < ep && *sp == '\n' && *(sp + 1) == 0 && *(sp + 2) == 0 && *(sp + 3) == 0) { \
		sp += 4; \
		_tc_ = '\n'; \
	} else { \
		_tc_ = 0; \
	} }

#define cnvln_ln_to_dtype(dp, c, ltype, DTYPE) \
	if (((ltype) & LTYPE_ALL) == LTYPE_NON) { \
		cnvln_to_ ## DTYPE (dp, c); \
	} else if (((ltype) & LTYPE_ALL) == LTYPE_LF) { \
		cnvln_to_ ## DTYPE (dp, '\n'); \
	} else if (((ltype) & LTYPE_ALL) == LTYPE_CR) { \
		cnvln_to_ ## DTYPE (dp, '\r'); \
	} else if (((ltype) & LTYPE_ALL) == LTYPE_CRLF) { \
		cnvln_to_ ## DTYPE (dp, '\r'); \
		cnvln_to_ ## DTYPE (dp, '\n'); \
	}

#define cnvln_cr_to_dtype(dp, c, ltype, next_char, DTYPE) \
	if (((ltype) & LTYPE_ALL) == LTYPE_NON) { \
		cnvln_to_ ## DTYPE (dp, c); \
		if (next_char != 0) { \
			cnvln_to_ ## DTYPE (dp, next_char); \
		} \
	} else if (((ltype) & LTYPE_ALL) == LTYPE_LF) { \
		cnvln_to_ ## DTYPE (dp, '\n'); \
	} else if (((ltype) & LTYPE_ALL) == LTYPE_CR) { \
		cnvln_to_ ## DTYPE (dp, '\r'); \
	} else if (((ltype) & LTYPE_ALL) == LTYPE_CRLF) { \
		cnvln_to_ ## DTYPE (dp, '\r'); \
		cnvln_to_ ## DTYPE (dp, '\n'); \
	}

#define _euc_to_unibe(dcs, euc) \
		(((dcs)->_euc2unibe_table00 == _euc2unibe2004_table00)? \
			/* サロゲート文字があるテーブルの時 */ \
		((euc) < 0x10000UL? ((unsigned long *)(dcs)->_euc2unibe_table00)[euc]: \
				((unsigned long *)(dcs)->_euc2unibe_table8f)[(euc)&0xffff]): \
		((euc) < 0x10000UL? ((unsigned short *)(dcs)->_euc2unibe_table00)[euc]: \
				((unsigned short *)(dcs)->_euc2unibe_table8f)[(euc)&0xffff]))

/* UT8 の簡易コピー（改行文字の編集のみ）  */
#define _utf8_copy(dcs, dp, sp, ep, ltype, c) { \
	if (sp >= ep) { \
		break; \
	} \
	c = *sp++; \
	if ((c & 0x80) == 0) { \
		cnvln_stype_to_dtype(dp, c, sp, ep, ltype, ascii, ascii); \
	} else { \
		*dp++ = c++; \
	}}

/* UTF16-BE の簡易コピー（改行文字の編集のみ）  */
#define _utf16be_copy(dcs, dp, sp, ep, ltype, c) { \
	if (sp + 1 >= ep) { \
		if (sp < ep) { \
			sts = DCS_EUEXPEOD; \
		} \
		break; \
	} \
	if (*sp == 0 && (*(sp + 1) & 0x80) == 0) { \
		c = *(sp + 1); \
		sp += 2; \
		cnvln_stype_to_dtype(dp, c, sp, ep, ltype, unibe, unibe); \
	} else { \
		*dp++ = *sp; \
		*dp++ = *(sp + 1); \
		sp += 2; \
	}}

/* UTF16-LE の簡易コピー（改行文字の編集のみ）  */
#define _utf16le_copy(dcs, dp, sp, ep, ltype, c) { \
	if (sp + 1 >= ep) { \
		if (sp < ep) { \
			sts = DCS_EUEXPEOD; \
		} \
		break; \
	} \
	if ((*sp & 0x80) == 0 && *(sp + 1) == 0) { \
		c = *sp; \
		sp += 2; \
		cnvln_stype_to_dtype(dp, c, sp, ep, ltype, unile, unile); \
	} else { \
		*dp++ = *sp; \
		*dp++ = *(sp + 1); \
		sp += 2; \
	}}

/* UTF32-BE の簡易コピー（改行文字の編集のみ）  */
#define _utf32be_copy(dcs, dp, sp, ep, ltype, c) { \
	if (sp + 3 >= ep) { \
		if (sp < ep) { \
			sts = DCS_EUEXPEOD; \
		} \
		break; \
	} \
	if (*sp == 0 && *(sp + 1) == 0) { \
	/* UTF16 値の時 */ \
		if (*(sp + 2) == 0 && (*(sp + 3) & 0x80) == 0) { \
			c = *(sp + 3); \
			sp += 4; \
			cnvln_stype_to_dtype(dp, c, sp, ep, ltype, uni32be, uni32be); \
		} else { \
			sp += 2; \
			*dp++ = 0; \
			*dp++ = 0; \
			*dp++ = *sp++; \
			*dp++ = *sp++; \
		} \
	} else { \
	/* UTF32 値の時 */ \
		*dp++ = *sp++; \
		*dp++ = *sp++; \
		*dp++ = *sp++; \
		*dp++ = *sp++; \
	}}

/* UTF32-LE の簡易コピー（改行文字の編集のみ）  */
#define _utf32le_copy(dcs, dp, sp, ep, ltype, c) { \
	if (sp + 3 >= ep) { \
		if (sp < ep) { \
			sts = DCS_EUEXPEOD; \
		} \
		break; \
	} \
	if (*(sp + 2) == 0 && *(sp + 3) == 0) { \
	/* UTF16 値の時 */ \
		if ((*sp & 0x80) == 0 && *(sp + 1) == 0) { \
			c = *sp; \
			sp += 4; \
			cnvln_stype_to_dtype(dp, c, sp, ep, ltype, uni32le, uni32le); \
		} else { \
			*dp++ = *sp++; \
			*dp++ = *sp++; \
			*dp++ = 0; \
			*dp++ = 0; \
			sp += 2; \
		} \
	} else { \
		/* UTF32 値の時 */ \
		*dp++ = *sp++; \
		*dp++ = *sp++; \
		*dp++ = *sp++; \
		*dp++ = *sp++; \
	}}

/* バッファー確保 */
int _v1_fill_buffer(Dcs *dcs, const char *sbuf, size_t size, int stype);

/* バッファー設定 */
void _v1_set_buffer(char **dbuf, size_t *dsize);

int _dcs_copy(Dcs *dcs, int stype);
int _add_bom(Dcs *dcs, int dtype);
int _delete_bom(Dcs *dcs, int dtype);
int _eucto_euc(Dcs *dcs);
int _eucto_jis(Dcs *dcs);
int _eucto_sjis(Dcs *dcs);
int _eucto_unibe(Dcs *dcs, bool is_bom, int utf_type);
int _eucto_unile(Dcs *dcs, bool is_bom, int utf_type);
int _eucto_utf8(Dcs *dcs, bool is_bom);
int _jisto_euc(Dcs *dcs);
int _jisto_jis(Dcs *dcs);
int _jisto_sjis(Dcs *dcs);
int _jisto_unibe(Dcs *dcs, bool is_bom, int utf_type);
int _jisto_unile(Dcs *dcs, bool is_bom, int utf_type);
int _jisto_utf8(Dcs *dcs, bool is_bom);
int _sjisto_euc(Dcs *dcs);
int _sjisto_jis(Dcs *dcs);
int _sjisto_sjis(Dcs *dcs);
int _sjisto_unibe(Dcs *dcs, bool is_bom, int utf_type);
int _sjisto_unile(Dcs *dcs, bool is_bom, int utf_type);
int _sjisto_utf8(Dcs *dcs, bool is_bom);
int _unibeto_euc(Dcs *dcs, int utf_type);
int _unibeto_jis(Dcs *dcs, int utf_type);
int _unibeto_sjis(Dcs *dcs, int utf_type);
int _unibeto_unibe(Dcs *dcs, bool is_bom, int utf_type);
int _unibeto_unile(Dcs *dcs, bool is_bom, int utf_type);
int _unibeto_utf8(Dcs *dcs, bool is_bom, int utf_type);
int _unileto_euc(Dcs *dcs, int utf_type);
int _unileto_jis(Dcs *dcs, int utf_type);
int _unileto_sjis(Dcs *dcs, int utf_type);
int _unileto_unibe(Dcs *dcs, bool is_bom, int utf_type);
int _unileto_unile(Dcs *dcs, bool is_bom, int utf_type);
int _unileto_utf8(Dcs *dcs, bool is_bom, int utf_type);
int _utf8to_euc(Dcs *dcs);
int _utf8to_jis(Dcs *dcs);
int _utf8to_sjis(Dcs *dcs);
int _utf8to_unibe(Dcs *dcs, bool is_bom, int utf_type);
int _utf8to_unile(Dcs *dcs, bool is_bom, int utf_type);
int _utf8to_utf8(Dcs *dcs, bool is_bom);

#endif /* __DCS_H_ */
