/*
 * _dcs_inter.h
 *
 * Copyright 2019-2026 Ichiji Tadokoro. All Rights Reserved.
 */

#ifndef __DCS_INTER_H_
#define __DCS_INTER_H_

#ifdef __cplusplus
extern "C" {
#endif

#include "dcs.h"
#include "_han_zen.h"

/* ＤＣＳバージョン */
#define DCS_INTER_VERSION			((float)3.00)

#define _I_SRG		0x02

#define BIT_JIS2_SIDE2		0x01

/* Charset 定義ビット(_unibe_def_table.h マスク値) */
/* UTF, EX 以外は未実装 */
#define CSB_UTF16			0000001
#define CSB_UTF8			0000002
#define CSB_UTF				0000003
#define CSB_JIS				0000010
#define CSB_SJIS			0000020
#define CSB_EUC				0000040
#define CSB_EUC1			0000100
#define CSB_EUC2			0000200
#define CSB_EUC3			0000400
#undef  CSB_EUC4
#define CSB_OTHER			0000770
#define CSB_ALL				0000777
#define CSB_EX				0001000
#define CSB_EX2				0002000
#define CSB_UTFEX			(CSB_EX | CSB_UTF)
#define CSB_UTFEX2		(CSB_EX2 | CSB_UTF)

#define CSB_MAC_D			0000020 /* 濁点 */
#define CSB_MAC_H			0000040 /* 半濁点 */
#define CSB_MAC_BO			0000100 /* '̈'分音符（ウムラウト）*/
#define CSB_MAC_TO			0000200 /* '̆'短音符（ブレーヴェ）*/

#define CSB_HAN2_D			0010000 /* 濁点（半角全角変換） */
#define CSB_HAN2_H			0020000 /* 半濁点（半角全角変換） */

#define HAN_DAKUTEN				0xff9e /* 'ﾞ' */
#define HAN_HANDAKUTEN		0xff9f /* 'ﾟ' */

#define MAC_HK_U        0x3046 /* 'う' */
#define MAC_HK_UD       0x3094 /* 'ゔ' */
#define MAC_KK_U        0x30a6 /* 'ウ' */
#define MAC_KK_UD       0x30f4 /* 'ヴ' */
#define MAC_KK_WA       0x30ef /* 'ワ' */
#define MAC_KK_WAD      0x30f7 /* 'ヷ' */
#define MAC_KK_I        0x30f0 /* 'ヰ' */
#define MAC_KK_ID       0x30f8 /* 'ヸ' */
#define MAC_KK_E        0x30f1 /* 'ヱ' */
#define MAC_KK_ED       0x30f9 /* 'ヹ' */
#define MAC_KK_WO       0x30f2 /* 'ヲ' */
#define MAC_KK_WOD      0x30fa /* 'ヺ' */

#define MAC_KP_E      0x0415 /* 'Е' */
#define MAC_KP_EY      0x0401 /* 'Ё' */
#define MAC_KP_ES      0x0435 /* 'е' */
#define MAC_KP_EYS      0x0451 /* 'ё' */

#define MAC_KP_II      0x0418 /* 'И' */
#define MAC_KP_I      0x0419 /* 'Й' */
#define MAC_KP_IIS      0x0438 /* 'и' */
#define MAC_KP_IS      0x0439 /* 'й' */

#define MAC_DAKUTEN		0x3099	/* '゙' */
#define MAC_HANDAKUTEN		0x309a	/* '゚' */
#define MAC_BUNONPU		0x0308	/* '̈'分音符（ウムラウト）*/
#define MAC_TANONPU		0x0306	/* '̆'短音符（ブレーヴェ）*/

#define DCS_ND_JIS_SIDE1	01000000000
#define DCS_ND_JIS_SIDE2	02000000000
#define DCS_ND_JIS21		(DCS_ND_JIS2 | DCS_ND_JIS_SIDE1)
#define DCS_ND_JIS22		(DCS_ND_JIS2 | DCS_ND_JIS_SIDE2)

/* JISシフト・コード・タイプ */
#define STYPE_UNKNOWN 		-1
#define STYPE_ASCII 			0 /* 1B 28 42: ESC ( B */
#define STYPE_KANA 			1 /* 1B 28 49: ESC ( I */
#define STYPE_LATINI 		2 /* 1B 28 4A: ESC ( J */
#define STYPE_C6226 			3 /* 1B 24 40: ESC $ @ */
#define STYPE_X0208_1983 	4 /* 1B 24 42: ESC $ B */
#define STYPE_X0208_1990 	5 /* 1B 26 40 1B 24 42: ESC & @ ESC $ B */
#define STYPE_X0212 			6 /* 1B 24 28 44: ESC $ ( D */
#define STYPE_X0213_2000_1 	7 /* 1B 24 28 4F: ESC $ ( O */
#define STYPE_X0213_2004 	8 /* 1B 24 28 51: ESC $ ( Q */
#define STYPE_X0213_2000_2 	9 /* 1B 24 28 50: ESC $ ( P */

/* コード変換テーブル */
extern const uint16_t _jis2unibe50221_table01[];
extern const uint16_t _jis2unibe50221_table02[];
extern const size_t _jis2unibe50221_table02_offset;
extern const uint32_t _jis2unibe2022jp_table[];
extern const uint32_t _jis2unibe2022jp2004_table01[];
extern const uint32_t _jis2unibe2022jp2004_table02[];
extern const uint32_t _jis2unibe2022jp3_table01[];
extern const uint32_t _jis2unibe2022jp3_table02[];
extern const uint16_t _sjis2unibe_table[];
extern const uint16_t _sjis2unibe932_table[];
extern const uint32_t _sjis2unibe2004_table[];
extern const uint16_t _euc2unibecommon_table00[];
extern const uint16_t _euc2unibe20932_table00[];
extern const uint16_t _euc2unibe20932_table8f[];
extern const size_t _euc2unibe20932_table8f_offset;
extern const uint16_t _euc2unibe51932_table00[];
extern const uint16_t _euc2unibems_table00[];
extern const uint16_t _euc2unibems_table8f[];
extern const size_t _euc2unibems_table8f_offset;
extern const uint32_t _euc2unibe2004_table00[];
extern const uint32_t _euc2unibe2004_table8f[];
extern const size_t _euc2unibe2004_table8f_offset;
extern const uint16_t _unibe2jis50221_table[];
extern const char _unibe2jis50221_table_bits[];
extern const uint16_t _unibe2jis2022jp2004_table[];
extern const uint16_t _unibe2jis2022jp2004_table_srg[];
extern const char _unibe2jis2022jp2004_table_bits[];
extern const char _unibe2jis2022jp2004_table_srg_bits[];
extern const uint16_t _unibe2jis2022jp_table[];
extern const char _unibe2jis2022jp_table_bits[];
extern const uint16_t _unibe2jis2022jp3_table[];
extern const uint16_t _unibe2jis2022jp3_table_srg[];
extern const char _unibe2jis2022jp3_table_bits[];
extern const char _unibe2jis2022jp3_table_srg_bits[];
extern const uint16_t _unibe2sjis_table[];
extern const uint16_t _unibe2sjis932_table[];
extern const uint16_t _unibe2sjis2004_table[];
extern const uint16_t _unibe2sjis2022jp2004_table_srg[];
extern const uint16_t _unibe2euccommon_table[];
extern const uint32_t _unibe2euc20932_table[];
extern const uint16_t _unibe2euc51932_table[];
extern const uint32_t _unibe2eucms_table[];
extern const uint32_t _unibe2euc2004_table[];
extern const uint32_t _unibe2euc2004_table_srg[];
extern const uint16_t _unibe_def_table[];

extern const size_t _jis2unibe50221_table01_size;
extern const size_t _jis2unibe50221_table02_size;
extern const size_t _jis2unibe2022jp2004_table01_size;
extern const size_t _jis2unibe2022jp2004_table02_size;
extern const size_t _jis2unibe2022jp2004_table02_offset;
extern const size_t _jis2unibe2022jp_table_size;
extern const size_t _jis2unibe2022jp3_table01_size;
extern const size_t _jis2unibe2022jp3_table02_size;
extern const size_t _jis2unibe2022jp3_table02_offset;
extern const size_t _sjis2unibe_table_size;
extern const size_t _sjis2unibe932_table_size;
extern const size_t _sjis2unibe2004_table_size;
extern const size_t _euc2unibecommon_table00_size;
extern const size_t _euc2unibe20932_table00_size;
extern const size_t _euc2unibe20932_table8f_size;
extern const size_t _euc2unibe51932_table00_size;
extern const size_t _euc2unibems_table00_size;
extern const size_t _euc2unibems_table8f_size;
extern const size_t _euc2unibe2004_table00_size;
extern const size_t _euc2unibe2004_table8f_size;
extern const size_t _unibe2jis50221_table_size;
extern const size_t _unibe2jis50221_table_bits_size;
extern const size_t _unibe2jis2022jp_table_size;
extern const size_t _unibe2jis2022jp_table_bits_size;
extern const size_t _unibe2jis2022jp2004_table_size;
extern const size_t _unibe2jis2022jp2004_table_srg_size;
extern const size_t _unibe2jis2022jp2004_table_bits_size;
extern const size_t _unibe2jis2022jp2004_table_srg_bits_size;
extern const size_t _unibe2jis2022jp3_table_size;
extern const size_t _unibe2jis2022jp3_table_srg_size;
extern const size_t _unibe2jis2022jp3_table_bits_size;
extern const size_t _unibe2jis2022jp3_table_srg_bits_size;
extern const size_t _unibe2sjis_table_size;
extern const size_t _unibe2sjis932_table_size;
extern const size_t _unibe2sjis2004_table_size;
extern const size_t _unibe2euccommon_table_size;
extern const size_t _unibe2euc20932_table_size;
extern const size_t _unibe2euc51932_table_size;
extern const size_t _unibe2eucms_table_size;
extern const size_t _unibe2euc2004_table_size;
extern const size_t _unibe2euc2004_table_srg_size;
extern const size_t _unibe_def_table_size;

#define set_jis2022jp_to(dcs) { \
		(dcs)->_jis2unibe_table = (uint16_t *)_jis2unibe2022jp_table; \
		(dcs)->_jis2unibe_table_size = _jis2unibe2022jp_table_size; \
		(dcs)->_jis2unibe_table_offset = 0; }

#define is_to_jis2022jp(dcs) \
		((dcs)->_unibe2jis_table == (uint32_t *)_unibe2jis2022jp_table)

#define set_to_jis2022jp(dcs) { \
		(dcs)->_unibe2jis_table = (uint16_t *)_unibe2jis2022jp_table; \
		(dcs)->_unibe2jis_table_size = _unibe2jis2022jp_table_size; \
		(dcs)->_unibe2jis_table_bits = (char *)_unibe2jis2022jp_table_bits; \
		(dcs)->_unibe2jis_table_bits_size = _unibe2jis2022jp_table_bits_size; \
		(dcs)->_unibe2jis_table_srg[_I_SRG] = NULL; \
		(dcs)->_unibe2jis_table_srg_size[_I_SRG] = 0; \
		(dcs)->_unibe2jis_table_srg_bits[_I_SRG] = NULL; \
		(dcs)->_unibe2jis_table_srg_bits_size[_I_SRG] = 0; }

#define is_jis50221_to(dcs) \
		((dcs)->_jis2unibe_table == (uint16_t *)_jis2unibe50221_table01 \
			|| (dcs)->_jis2unibe_table == (uint16_t *)_jis2unibe50221_table02)

#define set_jis5022101_to(dcs) { \
		(dcs)->_jis2unibe_table = (uint16_t *)_jis2unibe50221_table01; \
		(dcs)->_jis2unibe_table_size = _jis2unibe50221_table01_size; \
		(dcs)->_jis2unibe_table_offset = 0; }

#define set_jis5022102_to(dcs) { \
		(dcs)->_jis2unibe_table = (uint16_t *)_jis2unibe50221_table02; \
		(dcs)->_jis2unibe_table_size = _jis2unibe50221_table02_size; \
		(dcs)->_jis2unibe_table_offset = _jis2unibe50221_table02_offset; }

#define is_to_jis50221(dcs) \
		((dcs)->_unibe2jis_table == (uint32_t *)_unibe2jis50221_table)

#define set_to_jis50221(dcs) { \
		(dcs)->_unibe2jis_table = (uint16_t *)_unibe2jis50221_table; \
		(dcs)->_unibe2jis_table_size = _unibe2jis50221_table_size; \
		(dcs)->_unibe2jis_table_bits = (char *)_unibe2jis50221_table_bits; \
		(dcs)->_unibe2jis_table_bits_size = _unibe2jis50221_table_bits_size; \
		(dcs)->_unibe2jis_table_srg[_I_SRG] = NULL; \
		(dcs)->_unibe2jis_table_srg_size[_I_SRG] = 0; \
		(dcs)->_unibe2jis_table_srg_bits[_I_SRG] = NULL; \
		(dcs)->_unibe2jis_table_srg_bits_size[_I_SRG] = 0; }

#define set_jis2022jp301_to(dcs) { \
		(dcs)->_jis2unibe_table = (uint32_t *)_jis2unibe2022jp3_table01; \
		(dcs)->_jis2unibe_table_size = _jis2unibe2022jp3_table01_size; \
		(dcs)->_jis2unibe_table_offset = 0; }

#define set_jis2022jp302_to(dcs) { \
		(dcs)->_jis2unibe_table = (uint32_t *)_jis2unibe2022jp3_table02; \
		(dcs)->_jis2unibe_table_size = _jis2unibe2022jp3_table02_size; \
		(dcs)->_jis2unibe_table_offset = _jis2unibe2022jp3_table02_offset; }

#define is_to_jis2022jp3(dcs) \
		((dcs)->_unibe2jis_table == _unibe2jis2022jp3_table)

#define set_to_jis2022jp3(dcs) { \
		(dcs)->_unibe2jis_table = (uint16_t *)_unibe2jis2022jp3_table; \
		(dcs)->_unibe2jis_table_size = _unibe2jis2022jp3_table_size; \
		(dcs)->_unibe2jis_table_bits \
								= (char *)_unibe2jis2022jp3_table_bits; \
		(dcs)->_unibe2jis_table_bits_size \
								= _unibe2jis2022jp3_table_bits_size; \
		(dcs)->_unibe2jis_table_srg[_I_SRG] \
								= (uint16_t *)_unibe2jis2022jp3_table_srg; \
		(dcs)->_unibe2jis_table_srg_size[_I_SRG] \
								= _unibe2jis2022jp3_table_srg_size; \
		(dcs)->_unibe2jis_table_srg_bits[_I_SRG] \
							= (char *)_unibe2jis2022jp3_table_srg_bits; \
		(dcs)->_unibe2jis_table_srg_bits_size[_I_SRG] \
								= _unibe2jis2022jp3_table_srg_bits_size; }

#define is_jis2022jp2004_to(dcs) \
		((dcs)->_jis2unibe_table == (uint32_t *)_jis2unibe2022jp2004_table01 \
			|| (dcs)->_jis2unibe_table == (uint32_t *)_jis2unibe2022jp2004_table02)

#define set_jis2022jp200401_to(dcs) { \
		(dcs)->_jis2unibe_table = (uint32_t *)_jis2unibe2022jp2004_table01; \
		(dcs)->_jis2unibe_table_size = _jis2unibe2022jp2004_table01_size; \
		(dcs)->_jis2unibe_table_offset = 0; }

#define set_jis2022jp200402_to(dcs) { \
		(dcs)->_jis2unibe_table = (uint32_t *)_jis2unibe2022jp2004_table02; \
		(dcs)->_jis2unibe_table_size = _jis2unibe2022jp2004_table02_size; \
		(dcs)->_jis2unibe_table_offset = _jis2unibe2022jp2004_table02_offset; }

#define is_to_jis2022jp2004(dcs) \
		((dcs)->_unibe2jis_table == (uint16_t *)_unibe2jis2022jp2004_table)

#define set_to_jis2022jp2004(dcs) { \
		(dcs)->_unibe2jis_table = (uint16_t *)_unibe2jis2022jp2004_table; \
		(dcs)->_unibe2jis_table_size = _unibe2jis2022jp2004_table_size; \
		(dcs)->_unibe2jis_table_bits \
								= (char *)_unibe2jis2022jp2004_table_bits; \
		(dcs)->_unibe2jis_table_bits_size \
								= _unibe2jis2022jp2004_table_bits_size; \
		(dcs)->_unibe2jis_table_srg[_I_SRG] \
								= (uint16_t *)_unibe2jis2022jp2004_table_srg; \
		(dcs)->_unibe2jis_table_srg_size[_I_SRG] \
								= _unibe2jis2022jp2004_table_srg_size; \
		(dcs)->_unibe2jis_table_srg_bits[_I_SRG] \
								= (char *)_unibe2jis2022jp2004_table_srg_bits; \
		(dcs)->_unibe2jis_table_srg_bits_size[_I_SRG] \
								= _unibe2jis2022jp2004_table_srg_bits_size; }

#define set_sjis932_to(dcs) { \
		(dcs)->_sjis2unibe_table = (uint16_t *)_sjis2unibe932_table; \
		(dcs)->_sjis2unibe_table_size = _sjis2unibe932_table_size;}

#define set_sjis_to(dcs) { \
		(dcs)->_sjis2unibe_table = (uint16_t *)_sjis2unibe_table; \
		(dcs)->_sjis2unibe_table_size = _sjis2unibe_table_size;}

#define set_sjis2004_to(dcs) { \
		(dcs)->_sjis2unibe_table = (uint16_t *)_sjis2unibe2004_table; \
		(dcs)->_sjis2unibe_table_size = _sjis2unibe2004_table_size;}

#define is_to_sjis2004(dcs) \
		((dcs)->_unibe2sjis_table == _unibe2sjis2004_table)

#define set_to_sjis932(dcs) { \
		(dcs)->_unibe2sjis_table = (uint16_t *)_unibe2sjis932_table; \
		(dcs)->_unibe2sjis_table_size = _unibe2sjis932_table_size;}

#define set_to_sjis(dcs) { \
		(dcs)->_unibe2sjis_table = (uint16_t *)_unibe2sjis_table; \
		(dcs)->_unibe2sjis_table_size = _unibe2sjis_table_size;}

#define set_to_sjis2004(dcs) { \
		(dcs)->_unibe2sjis_table = (uint16_t *)_unibe2sjis2004_table; \
		(dcs)->_unibe2sjis_table_size = _unibe2sjis2004_table_size;}

#define set_euccommon_to(dcs) { \
		(dcs)->_euc2unibe_table00 = (uint16_t *)_euc2unibecommon_table00; \
		(dcs)->_euc2unibe_table8f = NULL; \
		(dcs)->_euc2unibe_table8f_offset = 0; \
		(dcs)->_euc2unibe_table00_size = _euc2unibecommon_table00_size; \
		(dcs)->_euc2unibe_table8f_size = 0; }

#define is_to_euc2004(dcs) \
		((dcs)->_lunibe2euc_table == _unibe2euc2004_table)

#define set_to_euccommon(dcs) { \
		(dcs)->_unibe2euc_table = (uint16_t *)_unibe2euccommon_table; \
		(dcs)->_lunibe2euc_table = NULL; \
		(dcs)->_unibe2euc_table_size = _unibe2euccommon_table_size; \
		(dcs)->_lunibe2euc_table_size = 0; }

#define set_euc20932_to(dcs) { \
		(dcs)->_euc2unibe_table00 = (uint16_t *)_euc2unibe20932_table00; \
		(dcs)->_euc2unibe_table8f = (uint16_t *)_euc2unibe20932_table8f; \
		(dcs)->_euc2unibe_table8f_offset = _euc2unibe20932_table8f_offset; \
		(dcs)->_euc2unibe_table00_size = _euc2unibe20932_table00_size; \
		(dcs)->_euc2unibe_table8f_size = _euc2unibe20932_table8f_size; }

#define set_to_euc20932(dcs) { \
		(dcs)->_unibe2euc_table = NULL; \
		(dcs)->_lunibe2euc_table = (uint32_t *)_unibe2euc20932_table; \
		(dcs)->_unibe2euc_table_size = 0; \
		(dcs)->_lunibe2euc_table_size = _unibe2euc20932_table_size; }

#define set_euc51932_to(dcs) { \
		(dcs)->_euc2unibe_table00 = (uint16_t *)_euc2unibe51932_table00; \
		(dcs)->_euc2unibe_table8f = NULL; \
		(dcs)->_euc2unibe_table8f_offset = 0; \
		(dcs)->_euc2unibe_table00_size = _euc2unibe51932_table00_size; \
		(dcs)->_euc2unibe_table8f_size = 0; }

#define set_to_euc51932(dcs) { \
		(dcs)->_unibe2euc_table = (uint16_t *)_unibe2euc51932_table; \
		(dcs)->_lunibe2euc_table = NULL; \
		(dcs)->_unibe2euc_table_size = _unibe2euc51932_table_size; \
		(dcs)->_lunibe2euc_table_size = 0; }

#define set_eucms_to(dcs) { \
		(dcs)->_euc2unibe_table00 = _euc2unibems_table00; \
		(dcs)->_euc2unibe_table8f = _euc2unibems_table8f; \
		(dcs)->_euc2unibe_table8f_offset = _euc2unibems_table8f_offset; \
		(dcs)->_euc2unibe_table00_size = _euc2unibems_table00_size; \
		(dcs)->_euc2unibe_table8f_size = _euc2unibems_table8f_size; }

#define set_euc2004_to(dcs) { \
		(dcs)->_euc2unibe_table00 = _euc2unibe2004_table00; \
		(dcs)->_euc2unibe_table8f = _euc2unibe2004_table8f; \
		(dcs)->_euc2unibe_table8f_offset = _euc2unibe2004_table8f_offset; \
		(dcs)->_euc2unibe_table00_size = _euc2unibe2004_table00_size; \
		(dcs)->_euc2unibe_table8f_size = _euc2unibe2004_table8f_size; }

#define set_to_eucms(dcs) { \
		(dcs)->_unibe2euc_table = NULL; \
		(dcs)->_lunibe2euc_table = (uint32_t *)_unibe2eucms_table; \
		(dcs)->_unibe2euc_table_size = 0; \
		(dcs)->_lunibe2euc_table_size = _unibe2eucms_table_size; }

#define set_to_euc2004(dcs) { \
		(dcs)->_unibe2euc_table = NULL; \
		(dcs)->_lunibe2euc_table = (uint32_t *)_unibe2euc2004_table; \
		(dcs)->_unibe2euc_table_size = 0; \
		(dcs)->_lunibe2euc_table_size = _unibe2euc2004_table_size; }

#define set_euc_table_to(dcs, ctype) { \
	if ((ctype) == DCS_EUC) { \
		set_euccommon_to(dcs); \
	} else if ((ctype) == DCS_EUC1) { \
		set_euc20932_to(dcs); \
	} else if ((ctype) == DCS_EUC2) { \
		set_euc51932_to(dcs); \
	} else if ((ctype) == DCS_EUC3) { \
		set_eucms_to(dcs); \
	} else if ((ctype) == DCS_EUC4) { \
		set_euc2004_to(dcs); \
	}}

#define set_to_euc_table(dcs, ctype) { \
	if ((ctype) == DCS_EUC) { \
		set_to_euccommon(dcs); \
	} else if ((ctype) == DCS_EUC1) { \
		set_to_euc20932(dcs); \
	} else if ((ctype) == DCS_EUC2) { \
		set_to_euc51932(dcs); \
	} else if ((ctype) == DCS_EUC3) { \
		set_to_eucms(dcs); \
	} else if ((ctype) == DCS_EUC4) { \
		set_to_euc2004(dcs); \
	}}

/*=================================================================*/

#define reset_no_char(dcs, sp) { \
		(dcs)->_no_char_ptr = (unsigned char *)(sp); \
		(dcs)->_no_char_offset = 0; \
		(dcs)->_f_no_char_offset = (dcs)->_l_no_char_offset = -1; \
		(dcs)->_no_char_count = 0; \
		(dcs)->_f_no_char = (dcs)->_l_no_char = -1; }

#define set_no_char_ascii(dcs, dp, c, sp) { \
		if (*(dp) == NO_CHAR \
		 && (c) != NO_CHAR) { \
			if ((dcs)->_f_no_char_offset < 0) { \
				(dcs)->_f_no_char_offset = (dcs)->_no_char_offset \
											+ ((sp) - (dcs)->_no_char_ptr); \
				(dcs)->_f_no_char = (c); \
			} \
			(dcs)->_l_no_char_offset = (dcs)->_no_char_offset \
										+ ((sp) - (dcs)->_no_char_ptr); \
			(dcs)->_l_no_char = (c); \
			(dcs)->_no_char_count++; \
		} }

#define set_no_char_jis(dcs, dp, c, sp) { \
		if (*(dp) == NO_CHAR1 \
		 && *((dp) + 1) == NO_CHAR2 \
		 && ((c) != NO_CHAR)) { \
			if ((dcs)->_f_no_char_offset < 0) { \
				(dcs)->_f_no_char_offset = (dcs)->_no_char_offset \
											+ ((sp) - (dcs)->_no_char_ptr); \
				(dcs)->_f_no_char = (c); \
			} \
			(dcs)->_l_no_char_offset = (dcs)->_no_char_offset \
										+ ((sp) - (dcs)->_no_char_ptr); \
			(dcs)->_l_no_char = (c); \
			(dcs)->_no_char_count++; \
		} }

#define set_no_char_unibe(dcs, dp, c, sp) { \
		if (*(dp) == 0 \
		 && *((dp) + 1) == NO_CHAR \
		 && (c) != NO_CHAR) { \
			if ((dcs)->_f_no_char_offset < 0) { \
				(dcs)->_f_no_char_offset = (dcs)->_no_char_offset \
											+ ((sp) - (dcs)->_no_char_ptr); \
				(dcs)->_f_no_char = (c); \
			} \
			(dcs)->_l_no_char_offset = (dcs)->_no_char_offset \
										+ ((sp) - (dcs)->_no_char_ptr); \
			(dcs)->_l_no_char = (c); \
			(dcs)->_no_char_count++; \
		} }

#define set_no_char_unile(dcs, dp, c, sp) { \
		if (*(dp) == NO_CHAR \
		 && *((dp) + 1) == 0 \
		 && (((c)>>8)&0xff) != NO_CHAR) { \
			if ((dcs)->_f_no_char_offset < 0) { \
				(dcs)->_f_no_char_offset = (dcs)->_no_char_offset \
										+ ((sp) - (dcs)->_no_char_ptr); \
				(dcs)->_f_no_char = (c); \
			} \
			(dcs)->_l_no_char_offset = (dcs)->_no_char_offset \
										+ ((sp) - (dcs)->_no_char_ptr); \
			(dcs)->_l_no_char = (c); \
			(dcs)->_no_char_count++; \
		} }

#define set_no_char_utf8(dcs, dp, c, sp) \
		set_no_char_ascii(dcs, dp, c, sp)

/*=================================================================*/

/* JISコード(16bit)の判定 */
#define is_jis_short(dcs, jis) \
	(0 <= (jis)-(dcs)->_jis2unibe_table_offset \
	 && (jis)-(dcs)->_jis2unibe_table_offset < (dcs)->_jis2unibe_table_size \
	 && ((dcs)->_jis2unibe_table == _jis2unibe50221_table? \
		((uint16_t *)(dcs)->_jis2unibe_table)[(jis)-(dcs)->_jis2unibe_table_offset]: \
		((uint32_t *)(dcs)->_jis2unibe_table)[(jis)-(dcs)->_jis2unibe_table_offset]) != 0)

/* SJISコード(16bit)の判定 */
#define is_sjis_short(dcs, sjis) \
	(0 <= (sjis) && (sjis) < (dcs)->_sjis2unibe_table_size? \
			(dcs)->_sjis2unibe_table==_sjis2unibe2004_table? \
			((uint32_t *)(dcs)->_sjis2unibe_table)[sjis] != 0: \
			((uint16_t *)(dcs)->_sjis2unibe_table)[sjis] != 0: 0)

#define is_euc_surrogate(dcs, euc) \
	(((dcs)->_euc2unibe_table00 == _euc2unibe2004_table00)? \
	/* サロゲート文字があるテーブルの時 */ \
	((euc) >= 0? (euc) < (dcs)->_euc2unibe_table00_size? \
		((uint32_t *)(dcs)->_euc2unibe_table00)[euc]: \
		((euc & ~0xffffUL) == 0x8f0000UL && ((euc)&0xffff) - (dcs)->_euc2unibe_table8f_offset < (dcs)->_euc2unibe_table8f_size \
		 && ((uint32_t *)(dcs)->_euc2unibe_table8f)[((euc)&0xffff) - (dcs)->_euc2unibe_table8f_offset]): 0): 0)

/* EUCコード(24bit)の判定 */
#define is_euc_long(dcs, euc) \
	(((dcs)->_euc2unibe_table00 == _euc2unibe2004_table00)? \
	/* サロゲート文字があるテーブルの時 */ \
	((euc) >= 0? (euc) < (dcs)->_euc2unibe_table00_size? \
		( ((uint32_t *)(dcs)->_euc2unibe_table00)[euc] != 0? 1: 0): \
		((euc & ~0xffffUL) == 0x8f0000UL && ((euc)&0xffff) - (dcs)->_euc2unibe_table8f_offset < (dcs)->_euc2unibe_table8f_size \
		 && ((uint32_t *)(dcs)->_euc2unibe_table8f)[((euc)&0xffff) - (dcs)->_euc2unibe_table8f_offset] != 0? 1: 0): 0): \
	((euc) >= 0? (euc) < (dcs)->_euc2unibe_table00_size? \
		( ((uint16_t *)(dcs)->_euc2unibe_table00)[euc] != 0? 1: 0): \
		((euc & ~0xffffUL) == 0x8f0000UL && ((euc)&0xffff) - (dcs)->_euc2unibe_table8f_offset < (dcs)->_euc2unibe_table8f_size \
		 && ((uint16_t *)(dcs)->_euc2unibe_table8f)[((euc)&0xffff) - (dcs)->_euc2unibe_table8f_offset] != 0? 1: 0): 0))

/* EUCJP-MSコード(8bit(0x80-0x9f))の判定 */
#define is_eucms_byte(euc) \
	((euc) >= 0 && (euc) < _euc2unibems_table00_size \
		 && _euc2unibems_table00[euc] != 0)

/* Unicode(BE) 判定 */
#define is_unibe_short(dcs, uni)	 \
	((uni) < 0 || (uni) >= (dcs)->_unibe_def_table_size \
	 || ((dcs)->_unibe_def_table[uni]&CSB_UTFEX) == 0? 0: 1)

/* 標準 Unicode(BE) 判定 */
#define is_unibestd_short(dcs, uni)	 \
	((uni) < 0 || (uni) >= (dcs)->_unibe_def_table_size \
	 || ((dcs)->_unibe_def_table[uni]&CSB_UTFEX) != CSB_UTF? 0: 1)

/* 拡張 Unicode(BE) 判定 */
#define is_unibeex_short(dcs, uni)	 \
	((uni) < 0 || (uni) >= (dcs)->_unibe_def_table_size \
	 || ((dcs)->_unibe_def_table[uni]&CSB_UTFEX) != CSB_UTFEX? 0: 1)

/* 拡張 Unicode(BE)2 判定 */
#define is_unibeex2_short(dcs, uni)	 \
	((uni) < 0 || (uni) >= (dcs)->_unibe_def_table_size \
	 || ((dcs)->_unibe_def_table[uni]&CSB_UTFEX2) != CSB_UTFEX2? 0: 1)

/* Unicode(LE) 判定 */
#define is_unile_short(dcs, uni)	 \
		((uni) < 0 || (uni) >= (dcs)->_unibe_def_table_size \
		 || ((dcs)->_unibe_def_table[unile_to_unibe_short(uni)]&CSB_UTFEX) == 0? 0: 1)

/* 標準 Unicode(LE) 判定 */
#define is_unilestd_short(dcs, uni)	 \
		((uni) < 0 || (uni) >= (dcs)->_unibe_def_table_size \
		 || ((dcs)->_unibe_def_table[unile_to_unibe_short(uni)]&CSB_UTFEX) != CSB_UTF? 0: 1)

/* 拡張 Unicode(LE) 判定 */
#define is_unileex_short(dcs, uni)	 \
		((uni) < 0 || (uni) >= (dcs)->_unibe_def_table_size \
		 || ((dcs)->_unibe_def_table[unile_to_unibe_short(uni)]&CSB_UTFEX) != CSB_UTFEX? 0: 1)

/* 拡張 Unicode(LE)2 判定 */
#define is_unileex2_short(dcs, uni)	 \
		((uni) < 0 || (uni) >= (dcs)->_unibe_def_table_size \
		 || ((dcs)->_unibe_def_table[unile_to_unibe_short(uni)]&CSB_UTFEX2) != CSB_UTFEX2? 0: 1)

/* Unicode(BE)サロゲート・ペア上位ビットの判定 */
#define is_unibe_surrogate_high(uni)	 \
		(0xd800 <= (uni) && (uni) <= 0xdbff)

/* Unicode(BE)サロゲート・ペア下位ビットの判定 */
#define is_unibe_surrogate_low(uni)	 \
		(0xdc00 <= (uni) && (uni) <= 0xdfff)

/* Unicode(LE)サロゲート・ペア上位ビットの判定 */
#define is_unile_surrogate_high(uni)	 \
		is_unibe_surrogate_high(unile_to_unibe_short(uni))

/* Unicode(LE)サロゲート・ペア下位ビットの判定 */
#define is_unile_surrogate_low(uni)	 \
		is_unibe_surrogate_low(unile_to_unibe_short(uni))

/* UTF8-MACの判定（is_unibe_short, is_unibe_zenkana 判定済み） */
#define is_unibe_mac(dcs, uni, mac) \
		(((dcs)->_unibe_def_table[uni] & (mac)) == 0? 0: 1)

#define NFD_TO_NFC_SWITCH(dcs, ch, _ch_, sp, inc) \
		if (_ch_ ==  MAC_DAKUTEN && is_unibe_mac(dcs, ch, CSB_MAC_D)) { \
			switch(ch) { \
			case MAC_HK_U: /* 'う' */ \
				ch = MAC_HK_UD; /* 'ゔ' */ \
				break; \
			case MAC_KK_U: /* 'ウ' */ \
				ch = MAC_KK_UD; /* 'ヴ' */ \
				break; \
			case MAC_KK_WA: /* 'ワ' */ \
				ch = MAC_KK_WAD; /* 'ヷ' */ \
				break; \
			case MAC_KK_I: /* 'ヰ' */ \
				ch = MAC_KK_ID; /* 'ヸ' */ \
				break; \
			case MAC_KK_E: /* 'ヱ' */ \
				ch = MAC_KK_ED; /* 'ヹ' */ \
				break; \
			case MAC_KK_WO: /* 'ヲ' */ \
				ch = MAC_KK_WOD; /* 'ヺ' */ \
				break; \
			default: \
				(ch)++; \
				break; \
			} \
			(sp) += inc; \
		} else if (_ch_ ==  MAC_HANDAKUTEN && is_unibe_mac(dcs, ch, CSB_MAC_H)) { \
			(ch) += 2; \
			(sp) += inc; \
		} else if (_ch_ ==  MAC_TANONPU && is_unibe_mac(dcs, ch, CSB_MAC_TO)) { \
			if ((ch) == MAC_KP_IIS) { /* 'и' */ \
				(ch) = MAC_KP_IS; /* 'й' */ \
				(sp) += inc; \
			} else if ((ch) == MAC_KP_II) { /* 'И' */ \
				(ch) = MAC_KP_I; /* 'Й' */ \
				(sp) += inc; \
			} \
		} else if (_ch_ ==  MAC_BUNONPU && is_unibe_mac(dcs, ch, CSB_MAC_BO)) { \
			if ((ch) == MAC_KP_ES) { /* 'е' */ \
				(ch) = MAC_KP_EYS; /* 'ё' */; \
				(sp) += inc; \
			} else if ((ch) == MAC_KP_E) { /* 'Е' */ \
				(ch) = MAC_KP_EY; /* 'Ё' */ \
				(sp) += inc; \
			} }

#define UTF16BE_NFD_TO_NFC(dcs, ch, sp, ep) { \
	unsigned short _ch_; \
	if (dcs_is_utfmac_to(dcs) && dcs_utfmac_exists(dcs) && (sp) + 2 <= (ep)) { \
		if (is_unibe_mac(dcs, ch, CSB_MAC_D | CSB_MAC_H | CSB_MAC_TO | CSB_MAC_BO)) { \
		/* UTF8-MAC 短音符（ブレーヴェ）、分音符（ウムラウト）の時 */ \
			_ch_ = (((*(sp)&0xff)<<8)|(*((sp)+1)&0xff)); \
			NFD_TO_NFC_SWITCH(dcs, ch, _ch_, sp, 2) \
		} \
	} }

#define UTF16LE_NFD_TO_NFC(dcs, ch, sp, ep) { \
	unsigned short _ch_; \
	if (dcs_is_utfmac_to(dcs) && dcs_utfmac_exists(dcs) && (sp) + 2 <= (ep)) { \
		if (is_unibe_mac(dcs, ch, CSB_MAC_D | CSB_MAC_H | CSB_MAC_TO | CSB_MAC_BO)) { \
		/* UTF8-MAC 濁点、半濁点の時 */ \
			_ch_ = (((*((sp)+1)&0xff)<<8)|(*(sp)&0xff)); \
			NFD_TO_NFC_SWITCH(dcs, ch, _ch_, sp, 2) \
		} \
	} }

#define UTF32BE_NFD_TO_NFC(dcs, ch, sp, ep) { \
	unsigned short _ch_; \
	if (dcs_is_utfmac_to(dcs) && dcs_utfmac_exists(dcs) && (sp) + 4 <= (ep)) { \
		if (is_unibe_mac(dcs, ch, CSB_MAC_D | CSB_MAC_H | CSB_MAC_TO | CSB_MAC_BO)) { \
		/* UTF8-MAC 短音符（ブレーヴェ）、分音符（ウムラウト）の時 */ \
			_ch_ = (((*(sp)&0xff)<<24)|((*((sp)+1)&0xff)<<16)|((*((sp)+2)&0xff)<<8)|(*((sp)+3)&0xff)); \
			NFD_TO_NFC_SWITCH(dcs, ch, _ch_, sp, 4) \
		} \
	} }

#define UTF32LE_NFD_TO_NFC(dcs, ch, sp, ep) { \
	unsigned short _ch_; \
	if (dcs_is_utfmac_to(dcs) && dcs_utfmac_exists(dcs) && (sp) + 4 <= (ep)) { \
		if (is_unibe_mac(dcs, ch, CSB_MAC_D | CSB_MAC_H | CSB_MAC_TO | CSB_MAC_BO)) { \
		/* UTF8-MAC 濁点、半濁点の時 */ \
			_ch_ = (((*((sp)+3)&0xff)<<24)|((*((sp)+2)&0xff)<<16)|((*((sp)+1)&0xff)<<8)|(*(sp)&0xff)); \
			NFD_TO_NFC_SWITCH(dcs, ch, _ch_, sp, 4) \
		} \
	} }

#define UTF8_NFD_TO_NFC_2BYTES(dcs, ch, sp, ep) { \
	unsigned short _ch_, _c_, _c2_; \
	if (dcs_is_utfmac_to(dcs) && dcs_utfmac_exists(dcs) && (sp) + 2 <= (ep)) { \
		if (is_unibe_mac(dcs, ch, CSB_MAC_TO | CSB_MAC_BO)) { \
		/* UTF8-MAC 濁点、半濁点の時 */ \
			_c_ = *(sp); \
			_c2_ = *(sp + 1); \
			if ((_c_ & 0xe0) == 0xc0 && (_c2_ & 0xc0) == 0x80) { \
			/* 110xxxxx 10xxxxxx: ２バイト文字 */ \
				_ch_ = ((_c_&0x1f)<<6|(_c2_&0x3f)); \
				NFD_TO_NFC_SWITCH(dcs, ch, _ch_, sp, 2) \
			} \
		} \
	} }

#define UTF8_NFD_TO_NFC(dcs, ch, sp, ep) { \
	unsigned short _ch_, _c_, _c2_, _c3_; \
	if (dcs_is_utfmac_to(dcs) && dcs_utfmac_exists(dcs) && (sp) + 3 <= (ep)) { \
		if (is_unibe_mac(dcs, ch, CSB_MAC_D | CSB_MAC_H)) { \
		/* UTF8-MAC 濁点、半濁点の時 */ \
			_c_ = *(sp); \
			_c2_ = *(sp + 1); \
			_c3_ = *(sp + 2); \
			if ((_c_ & 0xf0) == 0xe0 && (_c2_ & 0xc0) == 0x80 && (_c3_ & 0xc0) == 0x80) { \
			/* 1110xxxx 10xxxxxx 10xxxxxx: ３バイト文字 */ \
				_ch_ = ((_c_&0x0f)<<12|(_c2_&0x3f)<<6|(_c3_&0x3f)); \
				NFD_TO_NFC_SWITCH(dcs, ch, _ch_, sp, 3) \
			} \
		} \
	} }

/*=================================================================*/

#define JIS_HAN_TO_ZEN(dcs, ch, sp, ep, is_wa_vu) { \
	unsigned short _ch_; \
	if (is_unibe_kana(ch) && dcs_is_hankana_to(dcs)) { \
		if ((sp) + 1 <= (ep)) { \
			_ch_ = jis_kana_to_unibe((*sp)&0xff); \
		} else { \
			_ch_ = 0; \
		} \
		HAN_TO_ZEN_SWITCH(dcs, ch, _ch_, sp, 1, is_wa_vu) \
	} }

#define SJIS_HAN_TO_ZEN(dcs, ch, sp, ep, is_wa_vu) { \
	unsigned short _ch_; \
	if (is_unibe_kana(ch) && dcs_is_hankana_to(dcs)) { \
		if ((sp) + 1 <= (ep)) { \
			_ch_ = sjis_kana_to_unibe((*sp)&0xff); \
		} else { \
			_ch_ = 0; \
		} \
		HAN_TO_ZEN_SWITCH(dcs, ch, _ch_, sp, 1, is_wa_vu) \
	} }

#define EUC_HAN_TO_ZEN(dcs, ch, sp, ep, is_wa_vu) { \
	unsigned short _ch_; \
	if (is_unibe_kana(ch) && dcs_is_hankana_to(dcs)) { \
		if ((sp) + 2 <= (ep)) { \
			_ch_ = sjis_kana_to_unibe(*(sp+1)&0xff); \
		} else { \
			_ch_ = 0; \
		} \
		HAN_TO_ZEN_SWITCH(dcs, ch, _ch_, sp, 2, is_wa_vu) \
	} }

#define UTF8_HAN_TO_ZEN(dcs, ch, sp, ep, is_wa_vu) { \
	unsigned short _ch_, _c_, _c2_, _c3_; \
	if (is_unibe_kana(ch) && dcs_is_hankana_to(dcs)) { \
		_ch_ = 0; \
		if ((sp) + 3 <= (ep)) { \
			_c_ = *(sp); \
			_c2_ = *(sp + 1); \
			_c3_ = *(sp + 2); \
			if ((_c_ & 0xf0) == 0xe0 && (_c2_ & 0xc0) == 0x80 && (_c3_ & 0xc0) == 0x80) { \
			/* 1110xxxx 10xxxxxx 10xxxxxx: ３バイト文字 */ \
				_ch_ = ((_c_&0x0f)<<12|(_c2_&0x3f)<<6|(_c3_&0x3f)); \
			} \
		} \
		HAN_TO_ZEN_SWITCH(dcs, ch, _ch_, sp, 3, is_wa_vu) \
	} }

#define UTF16BE_HAN_TO_ZEN(dcs, ch, sp, ep, is_wa_vu) { \
	unsigned short _ch_; \
	if (is_unibe_kana(ch) && dcs_is_hankana_to(dcs)) { \
		if ((sp) + 2 <= (ep)) { \
			_ch_ = (((*(sp)&0xff)<<8)|(*((sp)+1)&0xff)); \
		} else { \
			_ch_ = 0; \
		} \
		HAN_TO_ZEN_SWITCH(dcs, ch, _ch_, sp, 2, is_wa_vu) \
	} }

#define UTF16LE_HAN_TO_ZEN(dcs, ch, sp, ep, is_wa_vu) { \
	unsigned short _ch_; \
	if (is_unibe_kana(ch) && dcs_is_hankana_to(dcs)) { \
		if ((sp) + 2 <= (ep)) { \
			_ch_ = (((*((sp)+1)&0xff)<<8)|(*(sp)&0xff)); \
		} else { \
			_ch_ = 0; \
		} \
		HAN_TO_ZEN_SWITCH(dcs, ch, _ch_, sp, 2, is_wa_vu) \
	} }

#define UTF32BE_HAN_TO_ZEN(dcs, ch, sp, ep, is_wa_vu) { \
	unsigned short _ch_; \
	if (is_unibe_kana(ch) && dcs_is_hankana_to(dcs)) { \
		if ((sp) + 4 <= (ep)) { \
			_ch_ = (((*(sp)&0xff)<<24)|((*((sp)+1)&0xff)<<16)|((*((sp)+2)&0xff)<<8)|(*((sp)+3)&0xff)); \
		} else { \
			_ch_ = 0; \
		} \
		HAN_TO_ZEN_SWITCH(dcs, ch, _ch_, sp, 4, is_wa_vu) \
	} }

#define UTF32LE_HAN_TO_ZEN(dcs, ch, sp, ep, is_wa_vu) { \
	unsigned short _ch_; \
	if (is_unibe_kana(ch) && dcs_is_hankana_to(dcs)) { \
		if ((sp) + 4 <= (ep)) { \
			_ch_ = (((*((sp)+3)&0xff)<<24)|((*((sp)+2)&0xff)<<16)|((*((sp)+1)&0xff)<<8)|(*(sp)&0xff)); \
		} else { \
			_ch_ = 0; \
		} \
		HAN_TO_ZEN_SWITCH(dcs, ch, _ch_, sp, 4, is_wa_vu) \
	} }

#define HAN_TO_ZEN_SWITCH(dcs, ch, _ch_, sp, inc, is_wa_vu) { \
	if (_ch_ ==  HAN_DAKUTEN) { /* 半角カナ濁点 */ \
		_ch_ = _f_han_to_zen_d(ch); \
		if (_ch_ == ch || ((_ch_ == UTFBE_ZEN_30F7 || _ch_ == UTFBE_ZEN_30FA) && !(is_wa_vu))) { \
		/* 変換されなかった時、又は'ヷ'、'ヺ'に変換出来ない時 */ \
			ch = _f_han_to_zen(ch); \
		} else { \
			ch = _ch_; \
			sp += inc; \
		} \
	} else if (_ch_ ==  HAN_HANDAKUTEN ) { /* 半角カナ半濁点 */ \
		_ch_ = _f_han_to_zen_h(ch); \
		if (_ch_ == ch) { \
			ch = _f_han_to_zen(ch); \
		} else { \
			ch = _ch_; \
			sp += inc; \
		} \
	} else { \
		ch = _f_han_to_zen(ch); \
	} }

/*=================================================================*/

#define _ESC_CHAR		0x1b
#define _SO_CHAR		0x0e	/* JIS半角カナシフトOUT */
#define _SI_CHAR		0x0f	/* JIS半角カナシフトIN */
#define _JIS_ASCII_MAX	0x7f

/* JIS シフト・イン／アウト */
/* ASII */
#define _check_jis_to_ascii(jis, shift_flag) { \
	if ((shift_flag) != STYPE_ASCII) { \
		*(jis)++ = _ESC_CHAR;		/* エスケープ(0x1b) */ \
		*(jis)++ = '(';			/* JISシフトOUT */ \
		*(jis)++ = 'B';			/* ASCII */ \
		(shift_flag) = STYPE_ASCII;	/* シフトOUT */ \
	} }

/* 半角かな */
#define _check_jis_to_kana(jis, shift_flag) { \
	if ((shift_flag) != STYPE_KANA) { \
		*(jis)++ = _ESC_CHAR;		/* エスケープ(0x1b) */ \
		*(jis)++ = '(';			/* JISシフトOUT */ \
		*(jis)++ = 'I';			/* JIS半角カナ */ \
		(shift_flag) = STYPE_KANA; \
	} }

/* 全角 83JIS */
#define _check_jis_to_zen(jis, shift_flag) { \
	if ((shift_flag) != STYPE_X0208_1983) { \
		*(jis)++ = _ESC_CHAR;		/* エスケープ(0x1b) */ \
		*(jis)++ = '$';			/* JISシフトIN */ \
		*(jis)++ = 'B';			/* 83JIS */ \
		(shift_flag) = STYPE_X0208_1983;	/* シフトIN */ \
	} }

/*=================================================================*/

/* JISコード（文字列）からSJISコードへの変換（変換不能文字位置設定） */
#define jis_to_sjis_char_sp(dcs, sjis, jis, sp, bytes) \
	jis_to_sjis_short_sp(dcs, sjis, char_to_short(jis), sp, bytes);

/* JISコード(16bit)からSJISコードへの変換 */
#define jis_to_sjis_short_sp(dcs, sjis, jis, sp, bytes) { \
	jis_to_unibe_short_sp(dcs, sjis, jis, sp); \
	unibe_to_sjis_char_sp(dcs, sjis, sjis, sp, bytes); \
}

/* JISコード（文字列）からEUCへの変換 */
#define jis_to_euc_char_sp(dcs, euc, jis, sp, bytes) \
	jis_to_euc_short_sp(dcs, uni, char_to_short(jis), sp, bytes)

/* JISコード(24bit)からEUCへの変換 */
#define jis_to_euc_short_sp(dcs, euc, jis, sp, bytes) { \
		jis_to_unibe_short_sp(dcs, euc, jis, sp); \
		unibe_to_euc_char_sp(dcs, euc, euc, sp, bytes); \
	}

/* JISコード（文字列）からUnicode(BE)への変換 */
#define jis_to_unibe_char_sp(dcs, uni, jis, sp) \
	jis_to_unibe_short_sp(dcs, jis, char_to_short(jis), sp)

/* JISコード(16bit)からUnicode(BE)への変換 */
#define jis_to_unibe_short_sp(dcs, uni, jis, sp) { \
	uint32_t _ch_; \
	if ((dcs)->_jis2unibe_table == _jis2unibe50221_table01 \
	 || (dcs)->_jis2unibe_table == _jis2unibe50221_table02) { \
		_ch_ = ((uint16_t *)(dcs)->_jis2unibe_table)[(jis)-(dcs)->_jis2unibe_table_offset]; \
		if (_ch_ == 0) { \
			*(uni) = 0; \
			*((uni)+1) = NO_CHAR; \
			set_no_char_unibe(dcs, uni, jis, sp); \
		} else { \
			*(uni) = (_ch_ >> 8); \
			*((uni)+1) = (_ch_ & 0xff); \
		} \
	} else { \
	/* サロゲート文字があるテーブルの時 */ \
		jis_to_surlb_long_sp(dcs, uni, jis, sp); \
	} }

/* JISコード(16bit)から UTFサロゲート・ペアへの変換 */ \
/* 呼び出し側で is_unibe_surrogate_high などで出力バイト数を判定すること */ \
#define jis_to_surlb_long_sp(dcs, surro, jis, sp) { \
		uint32_t _lch_ = ((uint32_t *)(dcs)->_jis2unibe_table)[(jis)-(dcs)->_jis2unibe_table_offset]; \
		if (_lch_ == 0) { \
		/* 不正文字の時 */ \
			*(surro) = 0; \
			*((surro)+1) = NO_CHAR; \
			set_no_char_unibe(dcs, surro, jis, sp); \
		} else if ((_lch_ & ~0xffff) == 0) { \
		/* サロゲートのデコードではない時 */ \
			*(surro) = (_lch_>>8); \
			*((surro) + 1) = _lch_; \
		} else { \
			uint16_t hsg = (uint16_t)(((_lch_ -= 0x10000) >> 10) | 0Xd800); \
			uint16_t lsg = (uint16_t)(((_lch_) & 0x3ff) | 0xdc00); \
			*(surro) = (hsg>>8); \
			*((surro) + 1) = hsg; \
			*((surro) + 2) = (lsg>>8); \
			*((surro) + 3) = lsg; \
		} }

/* JISコード(16bit)から UTFサロゲート・コードへの変換 */ \
/* 呼び出し側で is_unibe_surrogate_high などで出力バイト数を判定すること */ \
#define jis_to_sur32lb_long_sp(dcs, surro, jis, sp) { \
		uint32_t _lch_ = ((uint32_t *)(dcs)->_jis2unibe_table)[(jis)-(dcs)->_jis2unibe_table_offset]; \
		if (_lch_ == 0) { \
		/* 不正文字の時 */ \
			*(surro) = 0; \
			*((surro) + 1) = 0; \
			*((surro) + 2) = 0; \
			*((surro) + 3) = NO_CHAR; \
			set_no_char_unibe(dcs, surro, jis, sp); \
		} else if ((_lch_ & ~0xffff) == 0) { \
		/* サロゲートのデコードではない時 */ \
			*(surro) = 0; \
			*((surro) + 1) = 0; \
			*((surro) + 2) = (_lch_>>8); \
			*((surro) + 3) = _lch_; \
		} else { \
			*(surro) = (_lch_>>24); \
			*((surro) + 1) = (_lch_>>16); \
			*((surro) + 2) = (_lch_>>8); \
			*((surro) + 3) = _lch_; \
		} }

/* JISコード(16bit)から UTFサロゲート・ペアへの変換 */ \
/* 呼び出し側で is_unibe_surrogate_high などで出力バイト数を判定すること */ \
#define jis_to_surle_long_sp(dcs, surro, jis, sp) { \
		uint32_t _lch_ = ((uint32_t *)(dcs)->_jis2unibe_table)[(jis)-(dcs)->_jis2unibe_table_offset]; \
		if (_lch_ == 0) { \
		/* 不正文字の時 */ \
			*(surro) = NO_CHAR; \
			*((surro)+1) = 0; \
			set_no_char_unibe(dcs, surro, jis, sp); \
		} else if ((_lch_ & ~0xffff) == 0) { \
		/* サロゲートのデコードではない時 */ \
			*(surro) = _lch_; \
			*((surro) + 1) = (_lch_>>8); \
		} else { \
			uint16_t hsg = (uint16_t)(((_lch_ -= 0x10000) >> 10) | 0Xd800); \
			uint16_t lsg = (uint16_t)(((_lch_) & 0x3ff) | 0xdc00); \
			*(surro) = hsg; \
			*((surro) + 1) = (hsg>>8); \
			*((surro) + 2) = lsg; \
			*((surro) + 3) = (lsg>>8); \
		} }

/* JISコード(16bit)から UTFサロゲート・コードへの変換 */ \
/* 呼び出し側で is_unibe_surrogate_high などで出力バイト数を判定すること */ \
#define jis_to_sur32le_long_sp(dcs, surro, jis, sp) { \
		uint32_t _lch_ = ((uint32_t *)(dcs)->_jis2unibe_table)[(jis)-(dcs)->_jis2unibe_table_offset]; \
		if (_lch_ == 0) { \
		/* 不正文字の時 */ \
			*(surro) = NO_CHAR; \
			*((surro) + 1) = 0; \
			*((surro) + 2) = 0; \
			*((surro) + 3) = 0; \
			set_no_char_unibe(dcs, surro, jis, sp); \
		} else if ((_lch_ & ~0xffff) == 0) { \
		/* サロゲートのデコードではない時 */ \
			*(surro) = _lch_; \
			*((surro) + 1) = (_lch_>>8); \
			*((surro) + 2) = 0; \
			*((surro) + 3) = 0; \
		} else { \
			*(surro) = _lch_; \
			*((surro) + 1) = (_lch_>>8); \
			*((surro) + 2) = (_lch_>>16); \
			*((surro) + 3) = (_lch_>>24); \
		} }

/* JISコード（文字列）からUnicode(LE)への変換 */
#define jis_to_unile_char_sp(dcs, uni, jis, sp) \
	jis_to_unile_short_sp(dcs, uni, char_to_short(jis), sp)

/* JISコード(16bit)からUnicode(LE)への変換 */
#define jis_to_unile_short_sp(dcs, uni, jis, sp) { \
	uint16_t _ch_; \
	if ((dcs)->_jis2unibe_table == _jis2unibe50221_table01 \
	 || (dcs)->_jis2unibe_table == _jis2unibe50221_table02) { \
		_ch_ = ((uint16_t *)(dcs)->_jis2unibe_table)[(jis)-(dcs)->_jis2unibe_table_offset]; \
		if (_ch_ == 0) { \
			*(uni) = NO_CHAR; \
			*((uni)+1) = 0; \
			set_no_char_unile(dcs, uni, jis, sp); \
		} else { \
			*(uni) = (_ch_ & 0xff); \
			*((uni)+1) = (_ch_ >> 8); \
		} \
	} else { \
	/* サロゲート文字があるテーブルの時 */ \
		jis_to_surle_long_sp(dcs, uni, jis, sp); \
	} }

/* JISコード（文字列）からUTF-8への変換 */
#define jis_to_utf8_char_sp(dcs, utf8, jis, sp, bytes) \
	jis_to_utf8_short_sp(dcs, utf8, char_to_short(jis), sp, bytes)

/* JISコード(16bit)からUTF-8への変換 */
#define jis_to_utf8_short_sp(dcs, utf8, jis, sp, bytes) { \
	jis_to_unibe_short_sp(dcs, utf8, jis, sp); \
	unibe_to_utf8_char(utf8, utf8, bytes); \
}

/* JISコード(16bit)からUnicode(BE)への変換 */
#define jis_to_uni32be_short_sp(dcs, uni, jis, sp) { \
	uint16_t _ch_; \
	if ((dcs)->_jis2unibe_table == _jis2unibe50221_table01 \
	 || (dcs)->_jis2unibe_table == _jis2unibe50221_table02) { \
		_ch_ = ((uint16_t *)(dcs)->_jis2unibe_table)[(jis)-(dcs)->_jis2unibe_table_offset]; \
		if (_ch_ == 0) { \
			*(uni) = 0; \
			*((uni)+1) = 0; \
			*((uni)+2) = 0; \
			*((uni)+3) = NO_CHAR; \
			set_no_char_unibe(dcs, uni+2, jis, sp); \
		} else { \
			*(uni) = 0; \
			*((uni)+1) = 0; \
			*((uni)+2) = (_ch_ >> 8); \
			*((uni)+3) = (_ch_ & 0xff); \
		} \
	} else { \
	/* サロゲート文字があるテーブルの時 */ \
		jis_to_sur32lb_long_sp(dcs, uni, jis, sp); \
	} }

/* JISコード(16bit)からUnicode(LE)への変換 */
#define jis_to_uni32le_short_sp(dcs, uni, jis, sp) { \
	uint16_t _ch_; \
	if ((dcs)->_jis2unibe_table == _jis2unibe50221_table01 \
	 || (dcs)->_jis2unibe_table == _jis2unibe50221_table02) { \
		_ch_ = ((uint16_t *)(dcs)->_jis2unibe_table)[(jis)-(dcs)->_jis2unibe_table_offset]; \
		if (_ch_ == 0) { \
			*(uni) = NO_CHAR; \
			*((uni)+1) = 0; \
			*((uni)+2) = 0; \
			*((uni)+3) = 0; \
			set_no_char_unile(dcs, uni, jis, sp); \
		} else { \
			*(uni) = (_ch_ & 0xff); \
			*((uni)+1) = (_ch_ >> 8); \
			*((uni)+2) = 0; \
			*((uni)+3) = 0; \
		} \
	} else { \
	/* サロゲート文字があるテーブルの時 */ \
		jis_to_sur32le_long_sp(dcs, uni, jis, sp); \
	} }

	/* SJISコードからEUCへの変換 */
#define sjis_to_euc_long_sp(dcs, euc, sjis, sp, bytes) { \
		sjis_to_unibe_long_sp(dcs, euc, sjis, sp, bytes); \
		unibe_to_euc_char_sp(dcs, euc, euc, sp, bytes); \
	}

/* SJISコードからUnicode(BE)への変換 */
#define sjis_to_unibe_long_sp(dcs, uni, sjis, sp, bytes) { \
		if ((dcs)->_sjis2unibe_table == _sjis2unibe2004_table) { \
		/* サロゲート文字があるテーブルの時 */ \
			sjis_to_surrogate_be_long_sp(dcs, uni, sjis, sp, bytes); \
		} else { \
			unsigned long _ch_ = ((uint16_t *)(dcs)->_sjis2unibe_table)[(sjis)]; \
			(bytes) = 2; \
			if (_ch_ == 0) { \
				*(uni) = 0; \
				*((uni)+1) = NO_CHAR; \
				set_no_char_unibe(dcs, uni, sjis, sp); \
			} else { \
				*(uni) = (_ch_ >> 8); \
				*((uni)+1) = (_ch_ & 0xff); \
			} \
		}}

/* SJISコード(16bit)からUnicode(BE)への変換 */
#define sjis_to_unibe32_long_sp(dcs, uni, sjis, sp, bytes) { \
	if ((dcs)->_sjis2unibe_table == _sjis2unibe2004_table) { \
		/* サロゲート文字があるテーブルの時 */ \
			sjis_to_surrogate32_long_sp(dcs, uni, sjis, sp, bytes); \
	} else { \
		unsigned long _ch_ = ((uint16_t *)(dcs)->_sjis2unibe_table)[sjis]; \
		(bytes) = 2; \
		if (_ch_ == 0) { \
			*(uni) = 0; \
			*((uni)+1) = NO_CHAR; \
			set_no_char_unibe(dcs, uni, sjis, sp); \
		} else { \
			*(uni) = (_ch_ >> 8); \
			*((uni)+1) = (_ch_ & 0xff); \
		} \
	}}

/* SJISコードからサロゲート・ペア(BE)への変換 */
#define sjis_to_surrogate_be_long_sp(dcs, surro, sjis, sp, bytes) { \
	uint32_t _lch_, _hcd_, _lcd_; \
	(bytes) = 2; \
	_lch_ = ((uint32_t *)(dcs)->_sjis2unibe_table)[(sjis)]; \
	if (_lch_ == 0) { \
	/* 不正文字の時 */ \
		*(surro) = 0; \
		*((surro)+1) = NO_CHAR; \
		set_no_char_unibe(dcs, surro, sjis, sp); \
	} else if ((_lch_ & ~0xffff) == 0) { \
	/* サロゲートのデコードではない時 */ \
		*(surro) = (_lch_>>8); \
		*((surro) + 1) = _lch_; \
	} else { \
		surrogate_cd_to_unibe(_hcd_, _lcd_, _lch_); \
		*(surro) = (_hcd_>>8); \
		*((surro) + 1) = _hcd_; \
		*((surro) + 2) = (_lcd_>>8); \
		*((surro) + 3) = _lcd_; \
		(bytes) = 4; \
	} }

/* SJISコード(16bit)から UTFサロゲート・ペアへの変換 */ \
/* 呼び出し側で is_unibe_surrogate_high などで出力バイト数を判定すること */ \
#define sjis_to_surrogate32_long_sp(dcs, surro, sjis, sp, bytes) { \
		uint32_t _lch_ = ((uint32_t *)(dcs)->_sjis2unibe_table)[sjis]; \
		(bytes) = 2; \
		if (_lch_ == 0) { \
		/* 不正文字の時 */ \
			*(surro) = 0; \
			*((surro)+1) = NO_CHAR; \
			set_no_char_unibe(dcs, surro, sjis, sp); \
		} else if ((_lch_ & ~0xffff) == 0) { \
		/* サロゲートのデコードではない時 */ \
			*(surro) = (_lch_>>8); \
			*((surro) + 1) = _lch_; \
		} else { \
			*(surro) = 0; \
			*((surro) + 1) = (_lch_>>16); \
			*((surro) + 2) = (_lch_>>8); \
			*((surro) + 3) = _lch_; \
			(bytes) = 4; \
		} }

/* SJISコードからUnicode(LE)への変換 */
#define sjis_to_unile_long_sp(dcs, uni, sjis, sp, bytes) { \
		if ((dcs)->_sjis2unibe_table == _sjis2unibe2004_table) { \
		/* サロゲート文字があるテーブルの時 */ \
			sjis_to_surrogate_le_long_sp(dcs, uni, sjis, sp, bytes); \
		} else { \
			unsigned long _ch_ = ((uint16_t *)(dcs)->_sjis2unibe_table)[sjis]; \
			(bytes) = 2; \
			if (_ch_ == 0) { \
				*(uni) = NO_CHAR; \
				*((uni)+1) = 0; \
				set_no_char_unibe(dcs, uni, sjis, sp); \
			} else { \
				*(uni) = (_ch_ & 0xff); \
				*((uni)+1) = (_ch_ >> 8); \
			} \
		}}

/* SJISコードからサロゲート・ペア(LE)への変換 */
#define sjis_to_surrogate_le_long_sp(dcs, surro, sjis, sp, bytes) { \
		uint32_t _lch_, _hcd_, _lcd_; \
		(bytes) = 2; \
		_lch_ = ((uint32_t *)(dcs)->_sjis2unibe_table)[sjis]; \
		if (_lch_ == 0) { \
		/* 不正文字の時 */ \
			*(surro) = NO_CHAR; \
			*((surro)+1) = 0; \
			set_no_char_unibe(dcs, surro, sjis, sp); \
		} else if ((_lch_ & ~0xffff) == 0) { \
		/* サロゲートのデコードではない時 */ \
			*(surro) = _lch_; \
			*((surro) + 1) = (_lch_>>8); \
		} else { \
			surrogate_cd_to_unibe(_hcd_, _lcd_, _lch_); \
			*(surro) = _hcd_; \
			*((surro) + 1) = (_hcd_>>8); \
			*((surro) + 2) = _lcd_; \
			*((surro) + 3) = (_lcd_>>8); \
			(bytes) = 4; \
		} }

/* SJISコードからUnicode(32BE)への変換 */
#define sjis_to_uni32be_long_sp(dcs, uni, sjis, sp) { \
	if ((dcs)->_sjis2unibe_table == _sjis2unibe2004_table) { \
	/* サロゲート文字があるテーブルの時 */ \
		sjis_to_surrogate_32be_long_sp(dcs, dp, sjis, sp); \
	} else { \
		unsigned long _ch_ = ((uint16_t *)(dcs)->_sjis2unibe_table)[sjis]; \
		if (_ch_ == 0) { \
			*(uni) = 0; \
			*((uni)+1) = 0; \
			*((uni)+2) = 0; \
			*((uni)+3) = NO_CHAR; \
			set_no_char_unibe(dcs, uni+2, sjis, sp); \
		} else { \
			*(uni) = 0; \
			*((uni)+1) = 0; \
			*((uni)+2) = (_ch_ >> 8); \
			*((uni)+3) = (_ch_ & 0xff); \
		} \
	}}

/* SJISコードからサロゲート・コード(32BE)への変換 */
#define sjis_to_surrogate_32be_long_sp(dcs, surro, sjis, sp) { \
			uint32_t _lch_; \
			(bytes) = 2; \
			_lch_ = ((uint32_t *)(dcs)->_sjis2unibe_table)[sjis]; \
			if (_lch_ == 0) { \
			/* 不正文字の時 */ \
				*(surro) = 0; \
				*((surro) + 1) = 0; \
				*((surro) + 2) = 0; \
				*((surro) + 3) = NO_CHAR; \
				set_no_char_unibe(dcs, dp, sjis, sp); \
			} else if ((_lch_ & ~0xffff) == 0) { \
			/* サロゲートのデコードではない時 */ \
				*(surro) = 0; \
				*((surro) + 1) = 0; \
				*((surro) + 2) = (_lch_>>8); \
				*((surro) + 3) = _lch_; \
			} else { \
				*(surro) = (_lch_>>24); \
				*((surro) + 1) = (_lch_>>16); \
				*((surro) + 2) = (_lch_>>8); \
				*((surro) + 3) = _lch_; \
			} }

/* SJISコードからUnicode(32LE)への変換 */
#define sjis_to_uni32le_long_sp(dcs, uni, sjis, sp) { \
	(bytes) = 2; \
	if ((dcs)->_sjis2unibe_table == _sjis2unibe2004_table) { \
	/* サロゲート文字があるテーブルの時 */ \
		sjis_to_surrogate_32le_long_sp(dcs, dp, sjis, sp); \
	} else { \
		unsigned long _ch_ = ((uint16_t *)(dcs)->_sjis2unibe_table)[sjis]; \
		if (_ch_ == 0) { \
			*(uni) = NO_CHAR; \
			*((uni) + 1) = 0; \
			*((uni) + 2) = 0; \
			*((uni) + 3) = 0; \
			set_no_char_unibe(dcs, dp, sjis, sp); \
		} else { \
			*(uni) = (_ch_ & 0xff); \
			*((uni) + 1) = (_ch_ >> 8); \
			*((uni) + 2) = 0; \
			*((uni) + 3) = 0; \
		} \
	}}

/* SJISコード(16bit)からUTF-8への変換 */
#define sjis_to_utf8_long_sp(dcs, utf8, sjis, sp, bytes) { \
		sjis_to_unibe32_long_sp(dcs, utf8, sjis, sp, bytes); \
		unibe_to_utf8_char(utf8, utf8, bytes); \
	}

/* SJISコードからサロゲート・コード(32LE)への変換 */
#define sjis_to_surrogate_32le_long_sp(dcs, surro, sjis, sp) { \
			uint32_t _lch_ = ((uint32_t *)(dcs)->_sjis2unibe_table)[sjis]; \
			if (_lch_ == 0) { \
			/* 不正文字の時 */ \
				*(surro) = NO_CHAR; \
				*((surro) + 1) = 0; \
				*((surro) + 2) = 0; \
				*((surro) + 3) = 0; \
				set_no_char_unibe(dcs, dp, sjis, sp); \
			} else if ((_lch_ & ~0xffff) == 0) { \
			/* サロゲートのデコードではない時 */ \
				*(surro) = _lch_; \
				*((surro) + 1) = (_lch_>>8); \
				*((surro) + 2) = 0; \
				*((surro) + 3) = 0; \
			} else { \
				*(surro) = _lch_; \
				*((surro) + 1) = (_lch_>>8); \
				*((surro) + 2) = (_lch_>>16); \
				*((surro) + 3) = (_lch_>>24); \
			} }

/* SJISコード（文字列）からUTF-8への変換 */
#define sjis_to_utf8_char_sp(dcs, utf8, sjis, sp, bytes) \
	sjis_to_utf8_short_sp(dcs, utf8, char_to_short(sjis), sp, bytes)

/* SJISコード(16bit)からUTF-8への変換 */
#define sjis_to_utf8_short_sp(dcs, utf8, sjis, sp, bytes) { \
	sjis_to_unibe_long_sp(dcs, utf8, sjis, sp, bytes); \
	unibe_to_utf8_char(utf8, utf8, bytes); \
}

/* EUCコード(24bit)からJISへの変換 */
#define euc_to_jis_long_sp(dcs, jis, euc, sp, bytes) { \
		euc_to_unibe_long_sp(dcs, jis, euc, sp, bytes); \
		unibe_to_jis_char_sp(dcs, jis, jis, sp); \
	}

/* EUCコード(24bit)からSJISへの変換 */
#define euc_to_sjis_long_sp(dcs, sjis, euc, sp, bytes) { \
		euc_to_unibe_long_sp(dcs, sjis, euc, sp, bytes); \
		unibe_to_sjis_char_sp(dcs, sjis, sjis, sp, bytes); \
	}

/* EUCコード(24bit)からEUCコード(24bit)への変換 */
#define euc_to_euc_long_sp(dcs, deuc, seuc, sp, bytes) { \
		euc_to_unibe_long_sp(dcs, deuc, seuc, sp, bytes); \
		unibe_to_euc_char_sp(dcs, deuc, deuc, sp, bytes); \
	}

/* EUCコード(24bit)からUnicode(BE)への変換 */
#define euc_to_unibe_long_sp(dcs, uni, euc, sp, bytes) { \
	uint16_t _ch_; \
	if ((dcs)->_euc2unibe_table00 == _euc2unibe2004_table00) { \
		/* サロゲート文字があるテーブルの時 */ \
			euc_to_surrogate_be_long_sp(dcs, uni, euc, sp, bytes); \
	} else { \
		bytes = 2; \
		if ((euc) < 0x10000UL) { \
			_ch_ = ((uint16_t *)(dcs)->_euc2unibe_table00)[euc]; \
			if (_ch_ == 0) { \
				*(uni) = 0; \
				*((uni)+1) = NO_CHAR; \
				set_no_char_unibe(dcs, uni, euc, sp); \
			} else { \
				*(uni) = (_ch_ >> 8); \
				*((uni)+1) = (_ch_ & 0xff); \
			} \
		} else if (((euc) & 0xffff) - (dcs)->_euc2unibe_table8f_offset >= (dcs)->_euc2unibe_table8f_size) { \
			*(uni) = 0; \
			*((uni)+1) = NO_CHAR; \
			set_no_char_unibe(dcs, uni, euc, sp); \
		} else { \
			_ch_ = ((uint16_t *)(dcs)->_euc2unibe_table8f)[((euc)&0xffff) - (dcs)->_euc2unibe_table8f_offset]; \
			if (_ch_ == 0) { \
				*(uni) = 0; \
				*((uni)+1) = NO_CHAR; \
				set_no_char_unibe(dcs, uni, euc, sp); \
			} else { \
				*(uni) = (_ch_ >> 8); \
				*((uni)+1) = (_ch_ & 0xff); \
			} \
		} \
	}}

/* EUCコード(24bit)からUnicode(BE)への変換 */
#define euc_to_unibe32_long_sp(dcs, uni, euc, sp, bytes) { \
	uint16_t _ch_; \
	if ((dcs)->_euc2unibe_table00 == _euc2unibe2004_table00) { \
		/* サロゲート文字があるテーブルの時 */ \
			euc_to_surrogate32_long_sp(dcs, uni, euc, sp, bytes); \
	} else { \
		bytes = 2; \
		if ((euc) < 0x10000UL) { \
			_ch_ = ((uint16_t *)(dcs)->_euc2unibe_table00)[euc]; \
			if (_ch_ == 0) { \
				*(uni) = 0; \
				*((uni)+1) = NO_CHAR; \
				set_no_char_unibe(dcs, uni, euc, sp); \
			} else { \
				*(uni) = (_ch_ >> 8); \
				*((uni)+1) = (_ch_ & 0xff); \
			} \
		} else if (((euc) & 0xffff) - (dcs)->_euc2unibe_table8f_offset >= (dcs)->_euc2unibe_table8f_size) { \
			*(uni) = 0; \
			*((uni)+1) = NO_CHAR; \
			set_no_char_unibe(dcs, uni, euc, sp); \
		} else { \
			_ch_ = ((uint16_t *)(dcs)->_euc2unibe_table8f)[((euc)&0xffff) - (dcs)->_euc2unibe_table8f_offset]; \
			if (_ch_ == 0) { \
				*(uni) = 0; \
				*((uni)+1) = NO_CHAR; \
				set_no_char_unibe(dcs, uni, euc, sp); \
			} else { \
				*(uni) = (_ch_ >> 8); \
				*((uni)+1) = (_ch_ & 0xff); \
			} \
		} \
	}}

/* EUCコード(24bit)からUnicode(LE)への変換 */
#define euc_to_unile_long_sp(dcs, uni, euc, sp, bytes) { \
		uint16_t _ch_; \
	if ((dcs)->_euc2unibe_table00 == _euc2unibe2004_table00) { \
		/* サロゲート文字があるテーブルの時 */ \
			euc_to_surrogate_le_long_sp(dcs, uni, euc, sp, bytes); \
	} else { \
		bytes = 2; \
		if ((euc) < 0x10000UL) { \
			_ch_ = ((uint16_t *)(dcs)->_euc2unibe_table00)[euc]; \
			if (_ch_ == 0) { \
				*(uni) = NO_CHAR; \
				*((uni) + 1) = 0; \
				set_no_char_unile(dcs, uni, euc, sp); \
			} else { \
				*(uni) = (_ch_ & 0xff); \
				*((uni) + 1) = (_ch_ >> 8); \
			} \
		} else if (((euc) & 0xffff) - (dcs)->_euc2unibe_table8f_offset >= (dcs)->_euc2unibe_table8f_size) { \
			*(uni) = NO_CHAR; \
			*((uni) + 1) = 0; \
			set_no_char_unile(dcs, uni, euc, sp); \
		} else { \
			_ch_ = ((uint16_t *)(dcs)->_euc2unibe_table8f)[((euc) & 0xffff) - (dcs)->_euc2unibe_table8f_offset]; \
			if (_ch_ == 0) { \
				*(uni) = NO_CHAR; \
				*((uni) + 1) = 0; \
				set_no_char_unile(dcs, uni, euc, sp); \
			} else { \
				*(uni) = (_ch_ & 0xff); \
				*((uni) + 1) = (_ch_ >> 8); \
			} \
		} \
	}}

	/* EUCコード(16bit)から UTFサロゲート・ペアへの変換 */ \
	/* 呼び出し側で is_unibe_surrogate_high などで出力バイト数を判定すること */ \
	#define euc_to_surrogate_be_long_sp(dcs, surro, euc, sp, bytes) { \
			uint32_t _lch_, _hcd_, _lcd_; \
			if ((euc) < 0x10000UL) { \
				_lch_ = ((uint32_t *)(dcs)->_euc2unibe_table00)[(euc)]; \
			} else { \
				_lch_ = ((uint32_t *)(dcs)->_euc2unibe_table8f)[((euc) & 0xffff) - (dcs)->_euc2unibe_table8f_offset]; \
			} \
			(bytes) = 2; \
			if (_lch_ == 0) { \
			/* 不正文字の時 */ \
				*(surro) = 0; \
				*((surro)+1) = NO_CHAR; \
				set_no_char_unibe(dcs, surro, euc, sp); \
			} else if ((_lch_ & ~0xffff) == 0) { \
			/* サロゲートのデコードではない時 */ \
				*(surro) = (_lch_>>8); \
				*((surro) + 1) = _lch_; \
			} else { \
				surrogate_cd_to_unibe(_hcd_, _lcd_, _lch_); \
				*(surro) = (_hcd_>>8); \
				*((surro) + 1) = _hcd_; \
				*((surro) + 2) = (_lcd_>>8); \
				*((surro) + 3) = _lcd_; \
				(bytes) = 4; \
			} }

	/* EUCコード(16bit)から UTFサロゲート・ペアへの変換 */ \
	/* 呼び出し側で is_unibe_surrogate_high などで出力バイト数を判定すること */ \
	#define euc_to_surrogate_le_long_sp(dcs, surro, euc, sp, bytes) { \
			uint32_t _lch_, _hcd_, _lcd_; \
			if ((euc) < 0x10000UL) { \
				_lch_ = ((uint32_t *)(dcs)->_euc2unibe_table00)[(euc)]; \
			} else { \
				_lch_ = ((uint32_t *)(dcs)->_euc2unibe_table8f)[((euc) & 0xffff) - (dcs)->_euc2unibe_table8f_offset]; \
			} \
			(bytes) = 2; \
			if (_lch_ == 0) { \
			/* 不正文字の時 */ \
				*(surro) = NO_CHAR; \
				*((surro)+1) = 0; \
				set_no_char_unibe(dcs, surro, euc, sp); \
			} else if ((_lch_ & ~0xffff) == 0) { \
			/* サロゲートのデコードではない時 */ \
				*(surro) = _lch_; \
				*((surro) + 1) = (_lch_>>8); \
			} else { \
				surrogate_cd_to_unibe(_hcd_, _lcd_, _lch_); \
				*(surro) = _hcd_; \
				*((surro) + 1) = (_hcd_>>8); \
				*((surro) + 2) = _lcd_; \
				*((surro) + 3) = (_lcd_>>8); \
				(bytes) = 4; \
			} }

	/* EUCコード(16bit)から UTFサロゲート・ペアへの変換 */ \
	/* 呼び出し側で is_unibe_surrogate_high などで出力バイト数を判定すること */ \
	#define euc_to_surrogate32_long_sp(dcs, surro, euc, sp, bytes) { \
			uint32_t _lch_; \
			if ((euc) < 0x10000UL) { \
				_lch_ = ((uint32_t *)(dcs)->_euc2unibe_table00)[(euc)]; \
			} else { \
				_lch_ = ((uint32_t *)(dcs)->_euc2unibe_table8f)[(euc & 0xffff) - (dcs)->_euc2unibe_table8f_offset]; \
			} \
			(bytes) = 2; \
			if (_lch_ == 0) { \
			/* 不正文字の時 */ \
				*(surro) = 0; \
				*((surro)+1) = NO_CHAR; \
				set_no_char_unibe(dcs, surro, euc, sp); \
			} else if ((_lch_ & ~0xffff) == 0) { \
			/* サロゲートのデコードではない時 */ \
				*(surro) = (_lch_>>8); \
				*((surro) + 1) = _lch_; \
			} else { \
				*(surro) = 0; \
				*((surro) + 1) = (_lch_>>16); \
				*((surro) + 2) = (_lch_>>8); \
				*((surro) + 3) = _lch_; \
				(bytes) = 4; \
			} }

/* EUCコード(24bit)からUnicode(32BE)への変換 */
#define euc_to_uni32be_long_sp(dcs, uni, euc, sp) { \
	uint32_t _ch_; \
	if ((euc) < 0x10000UL) { \
		if ((dcs)->_euc2unibe_table00 == _euc2unibe2004_table00) { \
			_ch_ = ((uint32_t *)(dcs)->_euc2unibe_table00)[euc]; \
		} else { \
			_ch_ = ((uint16_t *)(dcs)->_euc2unibe_table00)[euc]; \
		} \
		if (_ch_ == 0) { \
			*(uni) = 0; \
			*((uni)+1) = 0; \
			*((uni)+2) = 0; \
			*((uni)+3) = NO_CHAR; \
			set_no_char_unibe(dcs, uni+2, euc, sp); \
		} else { \
			*(uni)     = (_ch_ >> 24); \
			*((uni)+1) = (_ch_ >> 16); \
			*((uni)+2) = (_ch_ >> 8); \
			*((uni)+3) = 	_ch_; \
		} \
	} else if (((euc) & 0xffff) - (dcs)->_euc2unibe_table8f_offset >= (dcs)->_euc2unibe_table8f_size) { \
		*(uni) = 0; \
		*((uni)+1) = 0; \
		*((uni)+2) = 0; \
		*((uni)+3) = NO_CHAR; \
		set_no_char_unibe(dcs, uni+2, euc, sp); \
	} else { \
		if ((dcs)->_euc2unibe_table8f == _euc2unibe2004_table8f) { \
			_ch_ = ((uint32_t *)(dcs)->_euc2unibe_table8f)[((euc)&0xffff) - (dcs)->_euc2unibe_table8f_offset]; \
		} else { \
			_ch_ = ((uint16_t *)(dcs)->_euc2unibe_table8f)[((euc)&0xffff) - (dcs)->_euc2unibe_table8f_offset]; \
		} \
		if (_ch_ == 0) { \
			*(uni) = 0; \
			*((uni)+1) = 0; \
			*((uni)+2) = 0; \
			*((uni)+3) = NO_CHAR; \
			set_no_char_unibe(dcs, uni+2, euc, sp); \
		} else { \
			*(uni)     = (_ch_ >> 24); \
			*((uni)+1) = (_ch_ >> 16); \
			*((uni)+2) = (_ch_ >> 8); \
			*((uni)+3) = 	_ch_; \
		} \
	}}

/* EUCコード(24bit)からUnicode(32LE)への変換 */
#define euc_to_uni32le_long_sp(dcs, uni, euc, sp) { \
		uint32_t _ch_; \
	if ((euc) < 0x10000UL) { \
		if ((dcs)->_euc2unibe_table00 == _euc2unibe2004_table00) { \
			_ch_ = ((uint32_t *)(dcs)->_euc2unibe_table00)[euc]; \
		} else { \
			_ch_ = ((uint16_t *)(dcs)->_euc2unibe_table00)[euc]; \
		} \
		if (_ch_ == 0) { \
			*(uni) = NO_CHAR; \
			*((uni) + 1) = 0; \
			*((uni) + 2) = 0; \
			*((uni) + 3) = 0; \
			set_no_char_unile(dcs, uni, euc, sp); \
		} else { \
			*(uni)       =  _ch_; \
			*((uni) + 1) = (_ch_ >> 8); \
			*((uni) + 2) = (_ch_ >> 16); \
			*((uni) + 3) = (_ch_ >> 24); \
		} \
	} else if (((euc) & 0xffff) - (dcs)->_euc2unibe_table8f_offset >= (dcs)->_euc2unibe_table8f_size) { \
		*(uni) = NO_CHAR; \
		*((uni) + 1) = 0; \
		*((uni) + 2) = 0; \
		*((uni) + 3) = 0; \
		set_no_char_unile(dcs, uni, euc, sp); \
	} else { \
		if ((dcs)->_euc2unibe_table8f == _euc2unibe2004_table8f) { \
			_ch_ = ((uint32_t *)(dcs)->_euc2unibe_table8f)[((euc)&0xffff) - (dcs)->_euc2unibe_table8f_offset]; \
		} else { \
			_ch_ = ((uint16_t *)(dcs)->_euc2unibe_table8f)[((euc)&0xffff) - (dcs)->_euc2unibe_table8f_offset]; \
		} \
		if (_ch_ == 0) { \
			*(uni) = NO_CHAR; \
			*((uni) + 1) = 0; \
			*((uni) + 2) = 0; \
			*((uni) + 3) = 0; \
			set_no_char_unile(dcs, uni, euc, sp); \
		} else { \
			*(uni)       =  _ch_; \
			*((uni) + 1) = (_ch_ >> 8); \
			*((uni) + 2) = (_ch_ >> 16); \
			*((uni) + 3) = (_ch_ >> 24); \
		} \
	}}

/* EUCコード(24bit)からUTF-8への変換 */
#define euc_to_utf8_long_sp(dcs, utf8, euc, sp, bytes) { \
		euc_to_unibe32_long_sp(dcs, utf8, euc, sp, bytes); \
		unibe_to_utf8_char(utf8, utf8, bytes); \
}

/* Unicode(BE)（文字列）からJISコードへの変換 */
#define unibe_to_jis_char_sp(dcs, jis, uni, sp)  \
	unibe_to_jis_short_sp(dcs, jis, unibe_char_to_short(uni), sp)

/* Unicode(BE)(16bit)からJISコードへの変換 */
#define unibe_to_jis_short_sp(dcs, jis, _uni, sp) { \
		uint16_t _ch_ = (uint16_t)(_uni); \
	_ch_ = (_ch_ >= (dcs)->_unibe2jis_table_size? 0: \
		(dcs)->_unibe2jis_table[_ch_]); \
	if (_ch_ == 0) { \
		*(jis) = NO_CHAR1; \
		*((jis) + 1) = NO_CHAR2; \
		set_no_char_jis(dcs, jis, _uni, sp); \
	} else { \
		*(jis) = (_ch_ >> 8); \
		*((jis) + 1) = (_ch_ & 0xff); \
	}}

/* Unicode(BE)（文字列）からSJISコードへの変換 */
#define unibe_to_sjis_char_sp(dcs, sjis, uni, sp, bytes) \
	unibe_to_sjis_short_sp(dcs, sjis, unibe_char_to_short(uni), sp, bytes)

/* Unicode(BE)(16bit)からSJISコードへの変換 */
#define unibe_to_sjis_short_sp(dcs, sjis, uni, sp, bytes) { \
	uint16_t _ch_ = (uint16_t)(uni); \
	uint32_t _lch_; \
	if (0xd800 <= _ch_ && _ch_ < 0xe000) { \
	/*  サロゲート・ペアの時  */ \
		uint32_t _hsg_ = _ch_; \
		uint32_t _lsg_ = unibe_char_to_short(sjis + 2); \
		uint64_t _lcd_ = (0x10000 + \
				(((_hsg_ & ~0xd800)<<10) | (_lsg_ & ~0xdc00))); \
		uint32_t _lcd2_ = (_lcd_ >> 16); \
		uint32_t _lcd3_ = (_lcd_ & 0xffff); \
		_lch_ = (_lcd2_ != _I_SRG \
			|| _lcd3_>=(dcs)->_unibe2sjis_table_srg_size[_I_SRG]? 0: \
				(dcs)->_unibe2sjis_table_srg[_lcd2_][_lcd3_]); \
		*(sjis) = (_lch_ >> 8); \
		*((sjis) + 1) = (_lch_ & 0xff); \
		(bytes) = 2; \
	} else { \
		_ch_ = (_ch_ >= (dcs)->_unibe2sjis_table_size? 0: \
			(dcs)->_unibe2sjis_table[_ch_]); \
		if (_ch_ == 0) { \
			*(sjis) = NO_CHAR; \
			set_no_char_ascii(dcs, sjis, uni, sp); \
			(bytes) = 1; \
		} else if ((_ch_ & 0xff00) == 0) { \
			*(sjis) = (_ch_ & 0xff); \
			(bytes) = 1; \
		} else { \
			*(sjis) = (_ch_ >> 8); \
			*((sjis) + 1) = (_ch_ & 0xff); \
			(bytes) = 2; \
		} \
	}}

/* Unicode(BE)サロゲート・ペアから SJIS への変換 */ \
#define unibe_surrogate_to_sjis_sp(dcs, sjis, hsg, lsg, sp) { \
	int bytes; \
	uint32_t _hsg_ = (unsigned)(hsg); \
	uint32_t _lsg_ = (unsigned)(lsg); \
	uint64_t _lcd_ = (0x10000 + \
		(((_hsg_ & ~0xd800)<<10) | (_lsg_ & ~0xdc00))); \
	uint32_t _lcd2_ = (_lcd_ >> 16); \
	uint32_t _lcd3_ = (_lcd_ & 0xffff); \
	uint32_t _cd_ = (_lcd2_ != _I_SRG \
			|| _lcd3_>=(dcs)->_unibe2sjis_table_srg_size[_I_SRG]? 0: \
					(dcs)->_unibe2sjis_table_srg[_lcd2_][_lcd3_]); \
	if (_cd_ == 0) { \
		if (!dcs_is_suppress(dcs)) { \
			unibe_to_sjis_short_sp(dcs, sjis, 0, sp, bytes); \
			(sjis) += bytes; \
				} \
		} else { \
			*(sjis)++ = ((_cd_)>>8); \
			*(sjis)++ = (_cd_); \
		} }

/* Unicode(BE)（文字列）からEUCコードへの変換 */
#define unibe_to_euc_char_sp(dcs, euc, uni, sp, bytes) \
	unibe_to_euc_short_sp(dcs, euc, unibe_char_to_short(uni), sp, bytes)

#define _set_euc(dcs, lch, ch) \
	lch = ((dcs)->_lunibe2euc_table == NULL? \
			ch >= (dcs)->_unibe2euc_table_size? 0: \
	(dcs)->_unibe2euc_table[ch]: ch >= (dcs)->_lunibe2euc_table_size? 0: \
	(dcs)->_lunibe2euc_table[ch]);

/* Unicode(BE)(16bit)からEUCコードへの変換 */
#define unibe_to_euc_short_sp(dcs, euc, uni, sp, bytes) { \
	uint16_t _ch_ = (uint16_t)(uni); \
	uint32_t _lch_; \
	if (0xd800 <= _ch_ && _ch_ < 0xe000) { \
	/*  サロゲート・ペアの時  */ \
		uint32_t _hsg_ = _ch_; \
		uint32_t _lsg_ = unibe_char_to_short(euc + 2); \
		uint64_t _lcd_ = (0x10000 + \
				(((_hsg_ & ~0xd800)<<10) | (_lsg_ & ~0xdc00))); \
		uint32_t _lcd2_ = (_lcd_ >> 16); \
		uint32_t _lcd3_ = (_lcd_ & 0xffff); \
		_lch_ = (_lcd2_ != _I_SRG \
			|| _lcd3_>=(dcs)->_unibe2euc_table_srg_size[_I_SRG]? 0: \
				(dcs)->_unibe2euc_table_srg[_lcd2_][_lcd3_]); \
		*(euc) = (_lch_ >> 8); \
		*((euc) + 1) = (_lch_ & 0xff); \
		(bytes) = 2; \
	} else { \
		_set_euc(dcs, _lch_, _ch_); \
	} \
	if (_lch_ == 0) { \
		*(euc) = NO_CHAR; \
		set_no_char_ascii(dcs, euc, _ch_, sp); \
		(bytes) = 1; \
	} else if ((_lch_ & ~0xff) == 0) { \
		*(euc) = (_lch_ & 0xff); \
		(bytes) = 1; \
	} else if ((_lch_ & ~0xffff) == 0) { \
		*(euc) = (_lch_ >> 8); \
		*((euc) + 1) = (_lch_ & 0xff); \
		(bytes) = 2; \
	} else { \
		*(euc) = (_lch_ >> 16); \
		*((euc) + 1) = ((_lch_ >> 8)&0xff); \
		*((euc) + 2) = (_lch_ & 0xff); \
		(bytes) = 3; \
	}}

/* Unicode(LE)（文字列）からJISコードへの変換 */
#define unile_to_jis_char_sp(dcs, jis, uni, sp)	 \
	unibe_to_jis_short_sp(dcs, jis, unile_to_unibe_char(uni), sp)

/* Unicode(LE)(16bit)からJISコードへの変換 */
#define unile_to_jis_short_sp(dcs, jis, uni, sp) \
	unibe_to_jis_short_sp(dcs, jis, unile_to_unibe_short(uni), sp)

/* Unicode(LE)（文字列）からSJISコードへの変換 */
#define unile_to_sjis_char_sp(dcs, sjis, uni, sp, bytes) \
	unibe_to_sjis_short_sp(dcs, sjis, unile_to_unibe_char(uni), sp, bytes)

/* Unicode(LE)(16bit)からSJISコードへの変換 */
#define unile_to_sjis_short_sp(dcs, sjis, uni, sp, bytes) \
		unibe_to_sjis_short_sp(dcs, sjis, unile_to_unibe_short(uni), sp, bytes)

/* Unicode(LE)（文字列）からEUCコードへの変換 */
#define unile_to_euc_char_sp(dcs, euc, uni, sp, bytes) \
	unibe_to_euc_short_sp(dcs, euc, unile_to_unibe_char(uni), sp, bytes)

/* Unicode(LE)(16bit)からEUCコードへの変換 */
#define unile_to_euc_short_sp(dcs, euc, uni, sp, bytes)	 \
	unibe_to_euc_short_sp(dcs, euc, unile_to_unibe_short(uni), sp, bytes)

	/* Unicode(BE)サロゲート・ペアから EUC への変換 */ \
	#define unibe_surrogate_to_euc_sp(dcs, euc, hsg, lsg, sp) { \
		int bytes; \
		uint32_t _hsg_ = (unsigned)(hsg); \
		uint32_t _lsg_ = (unsigned)(lsg); \
		uint64_t _lcd_ = (0x10000 + \
				(((_hsg_ & ~0xd800)<<10) | (_lsg_ & ~0xdc00))); \
		uint32_t _lcd2_ = (_lcd_ >> 16); \
		uint32_t _lcd3_ = (_lcd_ & 0xffff); \
		uint32_t _cd_ = (_lcd2_ != _I_SRG \
			|| _lcd3_>=(dcs)->_unibe2euc_table_srg_size[_I_SRG]? 0: \
				(dcs)->_unibe2euc_table_srg[_lcd2_][_lcd3_]); \
		if (_cd_ == 0) { \
			if (!dcs_is_suppress(dcs)) { \
				unibe_to_euc_short_sp(dcs, euc, 0, sp, bytes); \
				(euc) += bytes; \
			} \
		} else { \
			if ((_cd_ & 0xff0000) == 0x8f0000) { \
				*(euc)++ = ((_cd_)>>16); \
			} \
			*(euc)++ = ((_cd_)>>8); \
			*(euc)++ = (_cd_); \
		} }

/* SJISコード（文字列）からJISコードへの変換 */
#define judge_sjis_to_jis_short_sp(dcs, dp, c, c2, sp, shift_flag) { \
	int bytes; \
	sjis_to_unibe_long_sp(dcs, dp, c, sp, bytes); \
	c2 = unibe_char_to_short(dp + 2); \
	judge_unibe_to_jis_short_sp(dcs, \
			dp, unibe_char_to_short(dp), c2, sp, shift_flag); }

/* EUCコード（文字列）からJISコードへの変換 */
#define judge_euc_to_jis_short_sp(dcs, dp, c, c2, sp, shift_flag) { \
	int bytes; \
	euc_to_unibe_long_sp(dcs, dp, c, sp, bytes); \
	c2 = 	unibe_char_to_short(dp + 2); \
	judge_unibe_to_jis_short_sp(dcs, \
			dp, unibe_char_to_short(dp), c2, sp, shift_flag); }

/* Unicode(BE)(16bit)からJISコードへの変換 */
#define judge_unibe_to_jis_short_sp(dcs, jis, uni, uni2, sp, shift_flag) { \
	uint16_t _ch2_, _ch_ = (uint16_t)(uni); \
	if (_ch_ == NO_CHAR) { \
	/* 不正文字の時 */ \
		if (!dcs_is_suppress(dcs)) { \
			/* shift_flag が STYPE_X0208_1983 でなければシフトコードを出力 */ \
			_check_jis_to_zen(jis, shift_flag); \
			_ch2_ = (uint16_t)(uni2); \
			unibe_to_jis_short_sp(dcs, jis, _ch2_, sp); \
			(jis) += 2; \
		} \
	} else if (is_unibe_surrogate_high(_ch_)) { \
	/* Unicode(BE)サロゲート・ペア上位ビットの時 */ \
		_ch2_ = (uint16_t)(uni2); \
		if (is_unibe_surrogate_low(_ch2_)) { \
		/* Unicode(BE)サロゲート・ペア下位ビットの時 */ \
			/* Unicode(BE)サロゲート・ペアから JIS への変換 */ \
			unibe_surrogate_to_jis_sp(dcs, jis, _ch_, _ch2_, sp, shift_flag); \
			uni2 = -1; /* サロゲート・ペアを処理 */ \
		} else { \
			if (!dcs_is_suppress(dcs)) { \
			/* shift_flag が STYPE_X0208_1983 でなければシフトコードを出力 */ \
				_check_jis_to_zen(jis, shift_flag); \
				unibe_to_jis_short_sp(dcs, jis, 0, sp); \
				(jis) += 2; \
			} \
		} \
	} else { \
		_ch2_ = (_ch_ >= (dcs)->_unibe2jis_table_size? 0: \
				(dcs)->_unibe2jis_table[_ch_]); \
		/* 変換 JIS コード・チェック */ \
		_check_conv_jis_sp(dcs, jis, _ch2_, _ch_, sp, shift_flag); \
	} }

/* 変換 JIS コード・チェック */
#define _check_conv_jis_sp(dcs, jis, ch2, ch, sp, shift_flag) { \
	uint16_t __ch = ch; \
	uint16_t __ch2 = ch2; \
	if ((__ch2) == 0 || (__ch) >= dcs->_unibe2jis_table_bits_size) { \
	/* 不正文字の時 */ \
		if (!dcs_is_suppress(dcs)) { \
			/* shift_flag が STYPE_X0208_1983 でなければシフトコードを出力 */ \
			_check_jis_to_zen(jis, shift_flag); \
			unibe_to_jis_short_sp(dcs, jis, __ch, sp); \
			(jis) += 2; \
		} \
	} else { \
		if (shift_flag != dcs->_unibe2jis_table_bits[__ch]) { \
		/* シフト・コードが変わった時 */ \
			put_shift_code(jis, dcs->_unibe2jis_table_bits[__ch]); \
			(shift_flag) = dcs->_unibe2jis_table_bits[__ch]; \
		} \
		if (__ch2 >= 0x100) { \
			*(jis)++ = ((__ch2)>>8); \
		} \
		*(jis)++ = (__ch2); \
	} }

/* Unicode(BE)サロゲート・ペアから JIS への変換 */ \
#define unibe_surrogate_to_jis_sp(dcs, jis, hsg, lsg, sp, shift_flag) { \
	uint32_t _hsg_ = (unsigned)(hsg); \
	uint32_t _lsg_ = (unsigned)(lsg); \
	uint64_t _lcd_ = (0x10000 + \
			(((_hsg_ & ~0xd800)<<10) | (_lsg_ & ~0xdc00))); \
	uint32_t _lcd2_ = (_lcd_ >> 16); \
	uint32_t _lcd3_ = (_lcd_ & 0xffff); \
	uint32_t _jcd_ = (_lcd2_ != _I_SRG \
		|| _lcd3_>=(dcs)->_unibe2jis_table_srg_size[_I_SRG]? 0: \
			(dcs)->_unibe2jis_table_srg[_lcd2_][_lcd3_]); \
	if (_jcd_ == 0) { \
		if (!dcs_is_suppress(dcs)) { \
			/* shift_flag が STYPE_X0208_1983 でなければシフトコードを出力 */ \
			_check_jis_to_zen(jis, shift_flag); \
			unibe_to_jis_short_sp(dcs, jis, 0, sp); \
			(jis) += 2; \
		} \
	} else { \
		if (shift_flag != dcs->_unibe2jis_table_srg_bits[_I_SRG][_lcd3_]) { \
		/* シフト・コードが変わった時 */ \
			put_shift_code(jis, dcs->_unibe2jis_table_srg_bits[_I_SRG][_lcd3_]); \
			(shift_flag) = dcs->_unibe2jis_table_srg_bits[_I_SRG][_lcd3_]; \
		} \
		*(jis)++ = ((_jcd_)>>8); \
		*(jis)++ = (_jcd_); \
	} }

#define put_shift_code(jis, stype) { \
		switch(stype) { \
		case STYPE_ASCII:	/* 1B 28 42: ESC ( B */ \
			*(jis)++ = _ESC_CHAR; \
			*(jis)++ = '('; \
			*(jis)++ = 'B'; \
			break; \
		case STYPE_KANA:	/* 1B 28 49: ESC ( I */ \
			*(jis)++ = _ESC_CHAR; \
			*(jis)++ = '('; \
			*(jis)++ = 'I'; \
			break; \
		case STYPE_LATINI:	/* 1B 28 4A: ESC ( J */ \
			*(jis)++ = _ESC_CHAR; \
			*(jis)++ = '('; \
			*(jis)++ = 'J'; \
			break; \
		case STYPE_C6226:	/* 1B 24 40: ESC $ @ */ \
			*(jis)++ = _ESC_CHAR; \
			*(jis)++ = '$'; \
			*(jis)++ = '@'; \
			break; \
		case STYPE_X0208_1983:	/* 1B 24 42: ESC $ B */ \
			*(jis)++ = _ESC_CHAR; \
			*(jis)++ = '$'; \
			*(jis)++ = 'B'; \
			break; \
		case STYPE_X0212:			/* 1B 24 28 44: ESC $ ( D */ \
			*(jis)++ = _ESC_CHAR; \
			*(jis)++ = '$'; \
			*(jis)++ = '('; \
			*(jis)++ = 'D'; \
			break; \
		case STYPE_X0213_2000_1:	/* 1B 24 28 4F: ESC $ ( O */ \
			*(jis)++ = _ESC_CHAR; \
			*(jis)++ = '$'; \
			*(jis)++ = '('; \
			*(jis)++ = 'O'; \
			break; \
		case STYPE_X0213_2004:	/* 1B 24 28 51: ESC $ ( Q */ \
			*(jis)++ = _ESC_CHAR; \
			*(jis)++ = '$'; \
			*(jis)++ = '('; \
			*(jis)++ = 'Q'; \
			break; \
		case STYPE_X0213_2000_2:	/* 1B 24 28 50: ESC $ ( P */ \
			*(jis)++ = _ESC_CHAR; \
			*(jis)++ = '$'; \
			*(jis)++ = '('; \
			*(jis)++ = 'P'; \
			break; \
		} }

/* Unicode(LE)（文字列）からJISコードへの変換 */
#define unile_to_jis_char(dcs, jis, uni)	 \
	unibe_to_jis_short(dcs, jis, unile_to_unibe_char(uni))

/* Unicode(LE)(16bit)からJISコードへの変換 */
#define unile_to_jis_short(dcs, jis, uni) \
	unibe_to_jis_short(dcs, jis, unile_to_unibe_short(uni))

/* Unicode(LE)（文字列）からSJISコードへの変換 */
#define unile_to_sjis_char(dcs, sjis, uni, bytes) \
	unibe_to_sjis_short(dcs, sjis, unile_to_unibe_char(uni), bytes)

/* Unicode(LE)(16bit)からSJISコードへの変換 */
#define unile_to_sjis_short(dcs, sjis, uni, bytes) \
		unibe_to_sjis_short(dcs, sjis, unile_to_unibe_short(uni), bytes)

/* Unicode(LE)（文字列）からEUCコードへの変換 */
#define unile_to_euc_char(dcs, euc, uni, bytes) \
	unibe_to_euc_short(dcs, euc, unile_to_unibe_char(uni), bytes)

/* Unicode(LE)(16bit)からEUCコードへの変換 */
#define unile_to_euc_short(dcs, euc, uni, bytes)	 \
	unibe_to_euc_short(dcs, euc, unile_to_unibe_short(uni), bytes)

#ifdef __cplusplus
}
#endif

#endif /* __DCS_INTER_H_ */

