/*
 * dcs.h
 *
 * Copyright 2019-2026 Ichiji Tadokoro. All Rights Reserved.
 */

#ifndef _DCS_H_
#define _DCS_H_

#ifdef __cplusplus
extern "C" {
#endif

/* ＤＣＳバージョン */
#define DCS_VERSION			((float)3.00)

/* ＤＣＳバージョン情報 */
#define DCS_VERSION_INFO \
	"dcs(Detect Characters Set) Version %.2f(2026/02/16)\n\
Copyright (C) 2019-2026 Ichiji Tadokoro.\n\
This is free software.\n\
You are free to redistribute it as long as you do not change the content.\n\
There is no guarantee regarding usage.\n\
"

#include <stdbool.h>
#include <stddef.h>
#include <stdio.h>
#include <stdint.h>

#define DCS_SEPARATE_CHAR	';' /* 文字列の区切り文字 */

#ifndef min
#define min(a, b)	((a) < (b)? (a): (b))
#define max(a, b)	((a) > (b)? (a): (b))
#endif

/* 処理ステータス */
#define DCS_SUCCESS			0 /* 処理成功 */
#define DCS_EOF				-1 /* 入力ファイルの終わり */
#define DCS_ENOTEXISTS		-2 /* ファイルは存在せず */
#define DCS_ENOTFILE		-3 /* 通常ファイルではない */
#define DCS_EALLOC			-4 /* メモリー・アロケート・エラー */
#define DCS_EROPEN			-5 /* 入力ファイル・オープン・エラー */
#define DCS_EWOPEN			-6 /* 出力ファイル・オープン・エラー */
#define DCS_EREAD			-7 /* 入力エラー */
#define DCS_EWRITE			-8 /* 出力エラー */

#define DCS_EIOMODE			-10 /* 入出力モード誤り */
#define DCS_EIOFDRANGE		-11 /* ファイル記述子範囲誤り */
#define DCS_EIONOTOPEN		-12 /* 未オープン・ファイルへのアクセス */
#define DCS_EIOMAX			-13 /* 上限を超えたファイル・オープン */

#define DCS_ECTYPE			-20 /* コード・タイプ誤り */
#define DCS_EBOM			-21 /* BOM 不正（未使用） */
#define DCS_EUEXPEOD		-22 /* 予期せぬデータの終わり */
#define DCS_ECHAR			-23 /* コード値エラー */
#define DCS_ELTYPE			-24 /* 改行コード・タイプ誤り */

#define DCS_ECANOBS			-30 /* 監視者による取り消し */

/* コード・タイプ（コード判定ステータス）*/
#define DCS_UNKNOWN				-1 /* コード・タイプ不明 */
#define DCS_EMPTY				0 /* 空データ */
#define DCS_ASCII				1 /* ASCII */
#define DCS_JIS					2 /* JIS ISO-2022-JP */
#define DCS_JIS1					3 /* JIS CP50221 */
#define DCS_JIS2				4 /* JIS ISO-2022-JP-2004 */
#define DCS_JIS3				5 /* JIS ISO-2022-JP-3 */
#define DCS_SJIS				6 /* Shift_JIS */
#define DCS_SJIS1				7 /* SJIS CP932 */
#define DCS_SJIS2				8 /* Shift_JIS-2004 */
#define DCS_EUC					9 /* EUC-JP */
#define DCS_EUC1				10 /* EUC CP20932 */
#define DCS_EUC2				11 /* EUC CP51932 */
#define DCS_EUC3				12  /* eucJP-ms */
#define DCS_EUC4				13 /* eucJP-2004 */
#define DCS_UTF8				20 /* UTF-8 */
#define DCS_UTF8_BOM			21 /* UTF-8（BOM付）*/
#define DCS_UTF16BE				22 /* UTF-16 ビッグ・エンディアン */
#define DCS_UTF16BE_BOM			23 /* UTF-16 ビッグ・エンディアン（BOM付）*/
#define DCS_UTF16LE				24 /* UTF-16 リトル・エンディアン・ */
#define DCS_UTF16LE_BOM			25 /* UTF-16 リトル・エンディアン（BOM付）*/
#define DCS_UTF32BE				26 /* UTF-32 ビッグ・エンディアン */
#define DCS_UTF32BE_BOM			27 /* UTF-32 ビッグ・エンディアン（BOM付）*/
#define DCS_UTF32LE				28 /* UTF-32 リトル・エンディアン・ */
#define DCS_UTF32LE_BOM			29 /* UTF-32 リトル・エンディアン（BOM付）*/
#define DCS_NORMAL				00000000077	/* 確定コード・タイプの最大値 */

/* 各コード範囲判定 */
#define IS_JIS(stype)	(DCS_JIS <= (stype) && (stype) <= DCS_JIS3)
#define IS_SJIS(stype)	(DCS_SJIS <= (stype) && (stype) <= DCS_SJIS2)
#define IS_EUC(stype)	(DCS_EUC <= (stype) && (stype) <= DCS_EUC4)
#define IS_UTF(stype)	(DCS_UTF8 <= (stype) && (stype) <= DCS_UTF32LE_BOM)

/* 確定コード・タイプ */
#define DCS_CONFIRMED_CODE_TYPE(sts)	(DCS_UNKNOWN < (sts) && (sts) <= DCS_NORMAL)

/* 未確定コード・タイプ（マスク値） */
#define DCS_ND_ASCII						000000000100 /* ASCII */
#define DCS_ND_JIS							000000000200 /* JIS ISO-2022-JP */
#define DCS_ND_JIS1							000000000400 /* JIS CP50221 */
#define DCS_ND_JIS2							000000001000 /* JIS ISO-2022-JP-2004 */
#define DCS_ND_JIS3							000000002000 /* JIS ISO-2022-JP-3 */
#define DCS_ND_JIS_ALL					000000003600 /* JIS ALL */
#define DCS_ND_SJIS							000000004000 /* Shift_JIS */
#define DCS_ND_SJIS1						000000010000 /* SJIS CP932 */
#define DCS_ND_SJIS2						000000020000 /* Shift_JIS-2004 */
#define DCS_ND_SJIS_ALL				000000034000 /* SJIS ALL */
#define DCS_ND_EUC							000000040000 /* EUC-JP */
#define DCS_ND_EUC1							000000100000 /* EUC CP20932 */
#define DCS_ND_EUC2							000000200000 /* EUC CP51932 */
#define DCS_ND_EUC3							000000400000 /* eucJP-ms */
#define DCS_ND_EUC4							000001000000 /* eucJP-2004 */
#define DCS_ND_EUC_ALL					000001740000 /* EUC ALL */
#define DCS_ND_UTF8							000010000000 /* UTF-8 */
#define DCS_ND_UTF8_BOM				000020000000 /* UTF-8（BOM付）*/
#define DCS_ND_UTF16BE					000040000000 /* UTF-16 ビッグ・エンディアン */
#define DCS_ND_UTF16BE_BOM		000100000000 /* UTF-16 ビッグ・エンディアン（BOM付）*/
#define DCS_ND_UTF16LE					000200000000 /* UTF-16 リトル・エンディアン */
#define DCS_ND_UTF16LE_BOM		000400000000 /* UTF-16 リトル・エンディアン（BOM付）*/
#define DCS_ND_UTF32BE					001000000000 /* UTF-32 ビッグ・エンディアン */
#define DCS_ND_UTF32BE_BOM		002000000000 /* UTF-32 ビッグ・エンディアン（BOM付）*/
#define DCS_ND_UTF32LE					004000000000 /* UTF-32 リトル・エンディアン */
#define DCS_ND_UTF32LE_BOM		010000000000 /* UTF-32 リトル・エンディアン（BOM付）*/
#define DCS_ND_UTF_BOM			(DCS_ND_UTF8_BOM|DCS_ND_UTF16BE_BOM|DCS_ND_UTF16LE_BOM| \
						DCS_ND_UTF32BE_BOM|DCS_ND_UTF32LE_BOM) /* UTF（BOM付）*/
#define DCS_ND_ALL				017777777700 /* ND ALL */

/* 変換監視継続、停止ステータス */
#define DCS_OBS_STOP			-1		/* 変換監視停止 */
#define DCS_OBS_CONTINUE		0		/* 変換監視継続 */

/* 改行コード・タイプ */
#define LTYPE_NON			000000000 /* 改行なし */
#define LTYPE_LF			000000001 /* UNIX系 */
#define LTYPE_CR			000000002 /* Mac OS9 */
#define LTYPE_CRLF			000000004 /* Win */
#define LTYPE_ALL			000000007 /* すべての改行 */

/* BOM 追加／削除判定 */
#define CBOM_NON		0	/* BOM 追加削除なし */
#define CBOM_ADD		1	/* BOM 追加 */
#define CBOM_DELETE		-1	/* BOM 削除 */

/* 未定義コード表示値 */
#define NO_CHAR				'?'

/* 未定義コード表示値(JIS) */
#define NO_CHAR1			0x21	/* 第１バイト */
#define NO_CHAR2			0x29	/* 第２バイト */

/* UTF-16BE BOM */
#define UTF16_B0			0xfe
#define UTF16_B1			0xff

/* UTF-8 BOM */
#define UTF8_B0				0xef
#define UTF8_B1				0xbb
#define UTF8_B2				0xbf

/* UTF-16 置換文字 */
#define REP_CHARBE			0xfffd
#define REP_CHARLE			0xfdff

/* JIS全角仮名領域 */
#define JIS_HKANA_B			0x2421 /* 開始平仮名 */
#define JIS_HKANA_E			0x2473 /* 終了平仮名 */
#define JIS_KKANA_B			0x2521 /* 開始片仮名 */
#define JIS_KKANA_E			0x2573 /* 終了片仮名 */

/* SJIS全角仮名領域 */
#define SJIS_HKANA_B		0x829f /* 開始平仮名 */
#define SJIS_HKANA_E		0x82f1 /* 終了平仮名 */
#define SJIS_KKANA_B		0x8340 /* 開始片仮名 */
#define SJIS_KKANA_E		0x8393 /* 終了片仮名 */

/* EUC全角仮名領域 */
#define EUC_HKANA_B			0xa4a1 /* 開始平仮名 */
#define EUC_HKANA_E			0xa4f3 /* 終了平仮名 */
#define EUC_KKANA_B			0xa5a1 /* 開始片仮名 */
#define EUC_KKANA_E			0xa5f3 /* 終了片仮名 */

/* Unicode(BE)全角仮名領域 */
#define UNIBE_HKANA_B		0x3041 /* 開始平仮名 */
#define UNIBE_HKANA_E		0x309f /* 終了平仮名 */
#define UNIBE_KKANA_B		0x30a1 /* 開始片仮名 */
#define UNIBE_KKANA_E		0x30ff /* 終了片仮名 */
#define UNIBE_ODORIJI_B		0x3031 /* 開始範囲外踊り字（〱〲〳〴〵） */
#define UNIBE_ODORIJI_E		0x3035 /* 終了範囲外踊り字 */

/* バッファー情報 */
#define _SBUF_SIZE		(1024 * 8)

typedef struct buffer_info {
	unsigned char buf[_SBUF_SIZE];
	int32_t offset;
	int32_t length;
	int32_t j_offset;
	int32_t s_offset;
	int32_t e_offset;
	int32_t u8_offset;
	int32_t u16_offset;
	int32_t u32_offset;
} BufferInfo;

#define _SURROGATE_RANGE_SIZE	4
#define _PUSH_BACK_SIZE	16

/* DCS 情報*/
typedef struct _detect_charset_ {
	/* バッファー情報 */
	BufferInfo _buffer_info;
	/* コード変換テーブル */
	const void *_jis2unibe_table;
	const void *_sjis2unibe_table;
	const void *_euc2unibe_table00;
	const void *_euc2unibe_table8f;
	const uint16_t *_unibe2jis_table;
	const char *_unibe2jis_table_bits;
	const uint16_t *_unibe2jis_table_srg[_SURROGATE_RANGE_SIZE];
	const char *_unibe2jis_table_srg_bits[_SURROGATE_RANGE_SIZE];
	const uint16_t *_unibe2sjis_table;
	const uint32_t *_unibe2sjis_table_srg[_SURROGATE_RANGE_SIZE];
	const uint16_t *_unibe2euc_table;
	const uint32_t *_lunibe2euc_table;
	const uint32_t *_unibe2euc_table_srg[_SURROGATE_RANGE_SIZE];
	const uint16_t *_unibe_def_table;
	/* コード変換テーブル・サイズ */
	size_t _jis2unibe_table_offset;
	size_t _jis2unibe_table_size;
	size_t _sjis2unibe_table_size;
	size_t _euc2unibe_table00_size;
	size_t _euc2unibe_table8f_offset;
	size_t _euc2unibe_table8f_size;
	size_t _unibe2jis_table_size;
	size_t _unibe2jis_table_bits_size;
	size_t _unibe2jis_table_srg_size[_SURROGATE_RANGE_SIZE];
	size_t _unibe2jis_table_srg_bits_size[_SURROGATE_RANGE_SIZE];
	size_t _unibe2sjis_table_size;
	size_t _unibe2sjis_table_srg_size[_SURROGATE_RANGE_SIZE];
	size_t _unibe2euc_table_size;
	size_t _lunibe2euc_table_size;
	size_t _unibe2euc_table_srg_size[_SURROGATE_RANGE_SIZE];
	size_t _unibe_def_table_size;
	/* 入力ファイル・ポインター */
	FILE *_sfp;
	/* 入力バッファー・ポインター */
	unsigned char *_sp;
	unsigned char *_ep;
	/* プッシュ・バック・バッファー・ポインター */
	short _bbuf[_PUSH_BACK_SIZE];
	short *_bcp;
	short *_btp;
	/* 処理タイプ（改行コード・タイプのみ） */
	int _type;
	/* ファイル・アクセス時のエラー番号 */
	int _dcs_errno;
	/* サロゲート文字有無判定 */
	bool _is_surrogate;
	/* UTF8-MAC文字有無判定 */
	bool _utfmac_exists;
	/* UTF8-MAC文字変換 */
	bool _utfmac_to;
	/* 半角カナ全角文変換 */
	bool _hankana_to;
	/* 不正文字出力抑止の判定 */
	bool _is_suppress;
	/* 変換不能コードの開始／終了位置と件数 */
	long _no_char_offset;
	unsigned char *_no_char_ptr;
	long _f_no_char_offset;
	long _l_no_char_offset;
	int _no_char_count;
	int64_t _f_no_char;
	int64_t _l_no_char;
	/* 変換用データ */
	void *_block;
	/* JNI 情報 */
	void *_jni_info;
	/* ユーザー情報 */
	void *_user_info;
	/* 変換用ライター */
	int (*_writer)(const char *buf, size_t size, void *user_info);
	/* 変換用オブザーバー */
	int (*_observer)(int sts, void *user_info);
	/* 処理ステータス */
	int sts;
	/* Dcs バージョン */
	float version;
} Dcs;

/* 改行コード・タイプ取得 */
#define get_ltype(dcs)				((dcs)->_type & LTYPE_ALL)
/* 改行コード・タイプ設定 */
#define set_ltype(dcs, ltype) { \
	if ((ltype) == LTYPE_NON) { \
		(dcs)->_type &= ~LTYPE_ALL; \
	} else if (((ltype) & ~LTYPE_ALL) == 0) { \
		if (((ltype) & LTYPE_ALL) == LTYPE_LF \
		 || ((ltype) & LTYPE_ALL) == LTYPE_CR \
		 || ((ltype) & LTYPE_ALL) == LTYPE_CRLF) { \
			(dcs)->_type = ((dcs)->_type & ~LTYPE_ALL) | (ltype); \
		} \
	} }

/* 不正文字出力抑止の判定 */
#define dcs_is_suppress(dcs)			((dcs)->_is_suppress)
/* 不正文字出力抑止の設定 */
#define dcs_set_suppress(dcs, flag)		((dcs)->_is_suppress = flag)

/* 出力コードのUTF8-MACから１バイト変換の判定 */
#define dcs_is_utfmac_to(dcs)	((dcs)->_utfmac_to)

/* 出力コードをUTF8-MACから１バイトに変換 */
#define dcs_set_utfmac_to(dcs, flag)	((dcs)->_utfmac_to = (flag))

/* 入力コードのUTF8-MAC有無判定 */
#define dcs_utfmac_exists(dcs)		((dcs)->_utfmac_exists)

/* 出力コードの半角カナから全角変換の判定 */
#define dcs_is_hankana_to(dcs)	((dcs)->_hankana_to)

/* 出力コードを半角カナから全角に変換 */
#define dcs_set_hankana_to(dcs, flag)	((dcs)->_hankana_to = (flag))

/* 不正文字数取得 */
#define get_no_char_count(dcs)		((dcs)->_no_char_count)
/* 先頭不正文字取得 */
#define get_first_no_char(dcs) \
							((dcs)->_no_char_count==0? 0: (dcs)->_f_no_char)
/* 最終不正文字取得 */
#define get_last_no_char(dcs) \
							((dcs)->_no_char_count==0? 0: (dcs)->_l_no_char)
/* 先頭不正文字位置取得 */
#define get_first_no_char_offset(dcs)	((dcs)->_f_no_char_offset)
/* 最終不正文字位置取得 */
#define get_last_no_char_offset(dcs)	((dcs)->_l_no_char_offset)

/* 半角カナ・コード判定 */
#define is_jis_kana(c)		((c)>=0x21&&(c)<=0x5f)		/* JIS半角カナ */
#define is_sjis_kana(c)		((c)>=0xa1&&(c)<=0xdf)		/* SJIS半角カナ */
#define is_unibe_kana(c)	((c)>=0xff61&&(c)<=0xff9f)	/* Unicose(BE)半角カナ */

/*-半角カナ・コード変換------------------------------*/

/* JIS半角カナ・コードからSJISコードへの変換 */
#define jis_kana_to_sjis(c)	((c) | 0x80)
/* JIS半角カナ・コードからUnicode(BE)コードへの変換 */
#define jis_kana_to_unibe(c)	(0xff00 | (((c)+0x40) & 0xff))

/* SJIS半角カナ・コードからJISコードへの変換 */
#define sjis_kana_to_jis(c)	((c) & ~0x80)
/* SJIS半角カナ・コードからUnicode(BE)コードへの変換 */
#define sjis_kana_to_unibe(c)	(0xff00 | (((c) | 0x80) - 0x40))

/* Unicode(BE)半角カナ・コードからJISコードへの変換 */
#define unibe_kana_to_jis(c)	(((c)-0x40) & 0x7f)
/* Unicode(BE)半角カナ・コードからSJISコードへの変換 */
#define unibe_kana_to_sjis(c)	((((c)-0x40) | 0x80) & 0xff)

/* 変換不能コードの判定 */
#define is_jis_no_char(buf)		(*(buf) == NO_CHAR1 && *((buf)+1) == NO_CHAR2)
#define is_sjis_no_char(buf)	(*(buf) == NO_CHAR)
#define is_euc_no_char(buf)		is_sjis_no_char(buf)
#define is_utf8_no_char(buf)	is_sjis_no_char(buf)
#define is_unibe_no_char(buf)	(*(buf) == 0 && *((buf)+1) == NO_CHAR)
#define is_unile_no_char(buf)	(*(buf) == NO_CHAR && *((buf)+1) == 0)
#define is_uni32be_no_char(buf)	(*(buf) == 0 && *((buf)+1) == 0 && *((buf)+2) == 0 && *((buf)+3) == NO_CHAR)
#define is_uni32le_no_char(buf)	(*(buf) == NO_CHAR && *((buf)+1) == 0 && *((buf)+2) == 0 && *((buf)+3) == 0)

#define is_jis_no_short(c)	\
						((((c)>>8)&0xff) == NO_CHAR1 && ((c)&0xff) == NO_CHAR2)
#define is_sjis_no_short(c)	(((c)&0xff) == NO_CHAR)
#define is_euc_no_short(c)	is_sjis_no_short(c)
#define is_utf8_no_short(c)	is_sjis_no_short(c)
#define is_unibe_no_short(c)	((c) == NO_CHAR)
#define is_unile_no_short(c)	((c) == (NO_CHAR<<8))

/* 全角仮名・コード判定 */
/* JIS全角仮名 */
#define is_jis_zenkana(c)		((JIS_HKANA_B <= (c) && (c) <= JIS_HKANA_E) \
								|| (JIS_KKANA_B <= (c) && (c) <= JIS_KKANA_E))
/* SJIS全角仮名 */
#define is_sjis_zenkana(c)		((SJIS_HKANA_B <= (c) && (c) <= SJIS_HKANA_E) \
								|| (SJIS_KKANA_B <= (c) && (c) <= SJIS_KKANA_E))
/* EUC全角仮名 */
#define is_euc_zenkana(c)		((EUC_HKANA_B <= (c) && (c) <= EUC_HKANA_E) \
								|| (EUC_KKANA_B <= (c) && (c) <= EUC_KKANA_E))
/* Unicode(BE)全角仮名 */
#define is_unibe_zenkana(c)	((UNIBE_HKANA_B <= (c) && (c) <= UNIBE_HKANA_E) \
							|| (UNIBE_KKANA_B <= (c) && (c) <= UNIBE_KKANA_E) \
							|| (UNIBE_ODORIJI_B <= (c) && (c) <= UNIBE_ODORIJI_E))

/* UTF-32(BE)の判定 */
#define is_unibe_long(c)	((c) > 0xffff && (c) <= 0x0010FFFF)

/* 変換後も ASCII である出力タイプの判定 */
#define remains_ascii(dtype)	((dtype) < DCS_UTF16BE || (dtype) == DCS_UTF8)

/* サロゲートペア上位不正コード */
#define is_illegal_surrogate_hi(u16b)	(0xd8c0 <= (u16b) && (u16b) <= 0xdbff)

/*===========================================================================*/

/* 16bit数値から文字列への変換 */
#define short_to_char(buf, c)		(*(buf)=((c)>>8), *((buf)+1)=(c))

/* 文字列から16bit数値への変換 */
#define char_to_short(buf)	(((*(buf) & 0xff) << 8) | (*((buf)+1) & 0xff))

/* 文字列から32bit数値への変換 */
#define unibe_char_to_long(buf)	\
	(((*(buf) & 0xff) << 24) \
			| ((*((buf)+1) & 0xff) << 16) \
			| ((*((buf)+2) & 0xff) << 8) | (*((buf)+3) & 0xff))

/* Unicode(BE)（文字列）からUnicode(BE)(16bit)への変換 */
#define unibe_char_to_short(unibe)	char_to_short(unibe)

/* Unicode(LE)（文字列）からUnicode(LE)(16bit)への変換 */
#define unile_char_to_short(unile)	char_to_short(unile)

/* Unicode(BE)（文字列）からUnicode(LE)への変換 */
#define unibe_to_unile_char(unibe) \
		((*((unibe)+1)&0xff)<<8)|(*(unibe)&0xff)

/* Unicode(LE)（文字列）からUnicode(BE)への変換 */
#define unile_to_unibe_char(unile)	unibe_to_unile_char(unile)

/* Unicode(BE)(16bit)からUnicode(LE)への変換 */
#define unibe_to_unile_short(unibe)	 \
	((((unibe)&0xff)<<8)|((unibe)>>8))

/* Unicode(LE)(16bit)からUnicode(LB)への変換 */
#define unile_to_unibe_short(unile)	 unibe_to_unile_short(unile)

/* UTF32(BE)（文字列）からUTF32(LE)への変換 */
#define utf32be_to_utf32le_char(utf32be) \
	(((*((utf32be)+3)&0xff)<<24)|((*((utf32be)+2)&0xff)<<16)|((*((utf32be)+1)&0xff)<<8)|(*(utf32be)&0xff))

/* UTF32(LE)（文字列）からUTF32(BE)への変換 */
#define utf32le_to_utf32be_char(utf32le)	utf32be_to_utf32le_char(utf32le)

/* UTF32(BE)(32bit)からUTF32(LE)への変換 */
#define utf32be_to_utf32le_long(utf32be)	 \
	((((utf32be)&0xff)<<24)|(((utf32be)&0xff00)<<8)|(((utf32be)&0xff0000)>>8)|(((utf32be)&0xff000000)>>24))

/* UTF32(LE)(32bit)からUTF32(LB)への変換 */
#define utf32le_to_utf32be_long(utf32le)	 utf32be_to_utf32le_long(utf32le)

/*===========================================================================*/

/* UTF32(BE)(32bit)から(char)へのコピー */
#define utf32be_long_to_char(dp, utf32be)	 \
	(*(dp)++ = (((utf32be)>>24)&0xff), *(dp)++ = (((utf32be)>>16)&0xff), \
			*(dp)++ = (((utf32be)>>8)&0xff), *(dp)++ = ((utf32be)&0xff))

/* UTF32(BE)(32bit)からUTF32(LE)(char)への変換 */
#define utf32be_long_to_utf32le_char(dp, utf32be)	 \
	(*(dp)++ = ((utf32be)&0xff), *(dp)++ = (((utf32be)>>8)&0xff), \
			*(dp)++ = (((utf32be)>>16)&0xff), *(dp)++ = (((utf32be)>>24)&0xff))

/* UTF32(LE)(32bit)から(char)へのコピー */
#define utf32le_long_to_char(dp, utf32le)	 \
		utf32be_long_to_char(dp, utf32le)

/* UTF32(BL)(32bit)からUTF32(LB)(char)への変換 */
#define utf32le_long_to_utf32be_char(dp, utf32le)	 \
		utf32be_long_to_utf32le_char(dp, utf32le)

/*===========================================================================*/

/* Unicode(LE)（文字列）からUTF-8への変換 */
#define unile_to_utf8_char(utf8, uni, bytes) \
	unibe_to_utf8_short(utf8, bytes==4? \
			unile_to_unibe_lchar(uni): unile_to_unibe_char(uni), bytes)

/* Unicode(LE)(16bit)からUTF-8への変換 */
#define unile_to_utf8_short(utf8, uni, bytes) \
	unibe_to_utf8_short(utf8, bytes==4? \
			unile_to_unibe_long(uni): unile_to_unibe_short(uni), bytes)

/* Unicode(BE)（文字列）からUTF-8への変換 */
#define unibe_to_utf8_char(utf8, uni, bytes) \
		unibe_to_utf8_short(utf8, bytes==4? \
				unibe_char_to_long(uni): unibe_char_to_short(uni), bytes)

/* Unicode(BE)(16bit)からUTF-8への変換 */
/*if (c <= 0x007f) 0xxxxxxx: １バイト文字 */
/*} else if (c <= 0x7ff) 110xxxxx 10xxxxxx: ２バイト文字 */
/*} else if (c <= 0xffff) 1110xxxx 10xxxxxx 10xxxxxx: ３バイト文字 */
/*} else 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx: ４バイト文字 */
#define unibe_to_utf8_short(utf8, uni, bytes) { \
	unsigned long _ch_ = (unsigned long)(uni); \
	if (_ch_ <= 0x007f) { \
		*(utf8) = _ch_; \
		(bytes) = 1; \
	} else if (_ch_ <= 0x7ff) { \
		*(utf8) = ((0x06 << 5) | ((_ch_ >> 6) & 0x1f)); \
		*((utf8) + 1) = ((0x02 << 6) | (_ch_ & 0x3f)); \
		(bytes) = 2; \
	} else if (_ch_ <= 0xffff) { \
		*(utf8) = ((0x0e << 4) | ((_ch_ >> 12) & 0x0f)); \
		*((utf8) + 1) = ((0x02 << 6) | ((_ch_ >> 6) & 0x3f)); \
		*((utf8) + 2) = ((0x02 << 6) | (_ch_ & 0x3f)); \
		(bytes) = 3; \
	} else { \
		*(utf8) = ((0x1e << 3) | ((_ch_ >> 18) & 0x07)); \
		*((utf8) + 1) = ((0x02 << 6) | ((_ch_ >> 12) & 0x3f)); \
		*((utf8) + 2) = ((0x02 << 6) | ((_ch_ >> 6) & 0x3f)); \
		*((utf8) + 3) = ((0x02 << 6) | (_ch_ & 0x3f)); \
		(bytes) = 4; \
	} }

/* UTF-8(3byte)からUnicode(BE)への変換 */
/* 0xxxxxxx: １バイト文字 */
/* 110xxxxx 10xxxxxx: ２バイト文字 */
/* 1110xxxx 10xxxxxx 10xxxxxx: ３バイト文字 */
/* 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx: ４バイト文字 */
#define utf8_to_unibe_4bytes(uni, c, c2, c3, c4, bytes, sts) { \
	if (((c) & 0x80) == 0) { \
		*(uni) = 0; \
		*((uni) + 1) = c; \
		(bytes) = 1; \
		sts = DCS_SUCCESS; \
	} else if (c2 < 0) { \
		(bytes) = 1; \
		sts = DCS_EUEXPEOD; \
	} else { \
		unsigned short _ch_; \
		unsigned long _lch_; \
		if ((c & 0xe0) == 0xc0) { \
			if ((c2 & 0xc0) == 0x80) { \
				_ch_ = (((c & 0x1f) << 6) | (c2 & 0x3f)); \
				*(uni) = ((_ch_ >> 8)); \
				*((uni) + 1) = _ch_; \
				(bytes) = 2; \
				sts = DCS_SUCCESS; \
			} else { \
				(bytes) = 1; \
				sts = DCS_ECHAR; \
			} \
		} else { \
			if (c3 < 0) { \
				(bytes) = 2; \
				sts = DCS_EUEXPEOD; \
			} else if ((c & 0xf0) == 0xe0 \
					&& (c2 & 0xc0) == 0x80 \
					&& (c3 & 0xc0) == 0x80) { \
				_ch_ = ((c & 0x0f) << 12 | ((c2 & 0x3f) << 6) | (c3 & 0x3f)); \
				*(uni) = (_ch_ >> 8); \
				*((uni) + 1) = _ch_; \
				(bytes) = 3; \
				sts = DCS_SUCCESS; \
			} else if (c4 < 0) { \
				(bytes) = 2; \
				sts = DCS_EUEXPEOD; \
			} else { \
				if (((c) & 0xf8) == 0xf0 \
						&& ((c2) & 0xc0) == 0x80 \
						&& ((c3) & 0xc0) == 0x80 \
						&& ((c4) & 0xc0) == 0x80) { \
					_lch_ = (((c & 07) << 18)|((c2 & 0x3f) << 12)|((c3 & 0x3f) << 6)|(c4 & 0x3f)); \
					_lch_ -= 0x10000; \
					_ch_ = ((_lch_ >> 10)|0xd800); \
					*(uni) = (_ch_ >> 8); \
					*((uni) + 1) = _ch_; \
					_ch_ = ((_lch_ & 0x3ff)|0xdc00); \
					*((uni) + 2) = (_ch_ >> 8); \
					*((uni) + 3) = _ch_; \
					(bytes) = 4; \
					sts = DCS_SUCCESS; \
				} else { \
					(bytes) = 2; \
					sts = DCS_ECHAR; \
				} \
			} \
		}}

/* Unicode(BE)サロゲート・ペアからUTF-8への変換 */
/* 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx: ４バイト文字 */
#define unibe_surrogate_to_utf8(utf8, hcd, lcd) { \
	uint16_t _hsg_ = (uint16_t)(hcd); \
	uint16_t _lsg_ = (uint16_t)(lcd); \
	uint32_t _lcd_ = (0x10000 + (((_hsg_ & ~0xd800)<<10) | (_lsg_ & ~0xdc00))); \
	unibe_surrogate_cd_to_utf8(utf8, _lcd_); }

/* Unicode(BE)サロゲート・コードからUTF-8への変換 */
/* 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx: ４バイト文字 */
#define unibe_surrogate_cd_to_utf8(utf8, scd) { \
	*(utf8) = (0xf0 | ((scd) >> 18)); \
	*((utf8) + 1) = (0x80|(((scd) >> 12) & 0x3f)); \
	*((utf8) + 2) = (0x80|(((scd) >>  6) & 0x3f)); \
	*((utf8) + 3) = (0x80|((scd) & 0x3f)); }

/* Unicode(BE)サロゲート・ペアからコードへの変換 */
#define unibe_surrogate_to_cd(dcd, hcd, lcd) { \
	unsigned short _hsg_ = (unsigned short)(hcd); \
	unsigned short _lsg_ = (unsigned short)(lcd); \
	dcd = (0x10000 + (((_hsg_ & ~0xd800)<<10) | (_lsg_ & ~0xdc00))); }

/* Unicode(BE)サロゲート・コードからペアへの変換 */
#define surrogate_cd_to_unibe(hcd, lcd, scd) { \
	unsigned long _tcd_ = (unsigned long)(scd) - 0x10000; \
	hcd = ((_tcd_ / 0x400 ) + 0xd800); \
	lcd = ((_tcd_ % 0x400 ) + 0xdc00); }


/* バージョン取得 */
float dcs_get_version();

/* エラー番号取得 */
#define dcs_get_errno(dcs)				((dcs)->_dcs_errno)
/* エラー・メッセージ取得 */
int dcs_get_strerror(Dcs *dcs, int err_no, char *str, size_t size, int ctype);

/* Dcs 生成 */
Dcs *create_dcs();
/* Dcs 解放 */
void release_dcs(Dcs *dcs);

/* Dcs コード判定（バッファー） */
int dcs_judges_buffer(Dcs *dcs, int *atype, int *ln_type, const char *buffer, size_t size, int stype);
/* Dcs コード判定（ストリーム） */
int dcs_judges_stream(Dcs *dcs, int *atype, int *ln_type, FILE *sfp, int stype);
/* Dcs コード判定（ファイル） */
int dcs_judges_file(Dcs *dcs, int *atype, int *ln_type, const char *spath, int stype);
/* Dcs コード変換（バッファー） */
int dcs_convert_buffer(Dcs *dcs, const char *buffer, size_t size, int stype, int dtype, bool is_forced,
		int (*observer)(int sts, void *user_info), int (*writer)(const char *buf, size_t size, void *user_info), void *user_info);
/* Dcs コード変換（ストリーム） */
int dcs_convert_stream(Dcs *dcs, FILE *sfp, int stype, int dtype, bool is_forced,
		int (*observer)(int sts, void *user_info), int (*writer)(const char *buf, size_t size, void *user_info), void *user_info);
/* Dcs コード変換（ファイル） */
int dcs_convert_file(Dcs *dcs, const char *spath, int stype, int dtype, bool is_forced,
		int (*observer)(int sts, void *user_info), int (*writer)(const char *buf, size_t size, void *user_info), void *user_info);

/* BOM 追加／削除判定 */
int dcs_cmp_bom(int stype, int dtype);

/* 処理ステータスからメッセージを取得 */
int dcs_statusto_message(Dcs *dcs, char *str, int sts, int ctype);
/* コード・タイプから名前を取得 */
int dcs_typeto_name(char *name, int ctype);
/* コード名からタイプを取得 */
int dcs_nameto_type(const char *name);
/* 不特定のコード名を取得 */
int dcs_multi_type_names(char *buf, size_t size, int ctype);
/* すべてのコード名を取得 */
int dcs_all_type_names(char *buf, size_t size);
/* すべてのコード名の長さを取得 */
int dcs_all_type_names_length();

#ifdef __cplusplus
}
#endif

#endif /* _DCS_H_ */

