/*
 * dcs_judges_convert.c
 *
 * Copyright 2019-2026 Ichiji Tadokoro. All Rights Reserved.
 */

#include <errno.h>
#include "_jdcs.h"
#include "_dcs.h"

/* 改行コード・タイプ UNIBE, UNILE 無効 */
#define judge_unicode(cinfo, tltype) \
	((cinfo)->type >= DCS_UTF16BE || (cinfo)->type <= DCS_UTF32LE? 0: tltype)

/* EOD の前 */
#define FILL_BUFFER_AND_RETURN(dcs, ltype, atype, surrog_type, sts, is_keep)	{ \
	if ((is_keep || (ltype) != NULL) && DCS_CONFIRMED_CODE_TYPE(sts)) { \
		int	_sts_; \
		CHECK_SURROGATE(dcs, surrog_type, sts); \
		_sts_ = _fill_buffer(dcs, sts); \
		if (_sts_ != DCS_SUCCESS) { return _sts_; }} \
	CHECKLN_SETA_AND_RETURN(dcs, ltype, atype, surrog_type, sts); }

#define CHECK_SURROGATE(dcs, surrog_type, sts) { \
	if (!dcs->_is_surrogate) { \
		dcs->_is_surrogate = (surrog_type != 0 && sts <= DCS_NORMAL \
	 && (((surrog_type & DCS_ND_UTF8) && sts == DCS_UTF8) \
		|| ((surrog_type & DCS_ND_UTF16BE) && sts == DCS_UTF16BE) \
		|| ((surrog_type & DCS_ND_UTF16LE) && sts == DCS_UTF16LE) \
		|| ((surrog_type & DCS_ND_UTF32BE) && sts == DCS_UTF32BE) \
		|| ((surrog_type & DCS_ND_UTF32LE) && sts == DCS_UTF32LE) \
		|| ((surrog_type & DCS_ND_JIS2) && sts == DCS_JIS2) \
		|| ((surrog_type & DCS_ND_SJIS2) && sts == DCS_SJIS2) \
		|| ((surrog_type & DCS_ND_EUC4) && sts == DCS_EUC4))); } }

/* EOD の後 */
#define CHECKLN_SETA_AND_RETURN(dcs, ltype, atype, surrog_type, sts)	{ \
	CHECK_SURROGATE(dcs, surrog_type, sts); \
	if ((ltype) != NULL) { *ltype = _check_ln(dcs, sts); } \
	if ((atype) != NULL) { *(atype) = (sts); } return (sts); }

#define SETA_AND_RETURN(dcs, atype, sts)	{ \
	if ((atype) != NULL) { *(atype) = (sts); } return (sts); }

int _dcs_judges(Dcs *dcs, int *atype, int *ltype, int stype, bool is_forced, bool is_keep);
int _dcs_convert(Dcs *dcs, int stype, int dtype, bool is_forced);
int _check_ln(Dcs *dcs, int stype);
int _fill_buffer(Dcs *dcs, int stype);

/*
 * コード判定（バッファー）
 */
int dcs_judges_buffer(Dcs *dcs, int *atype, int *ltype, const char *buffer, size_t size, int stype) {
	int sts;
	dcs->_sp = (unsigned char *)buffer;
	dcs->_ep = (unsigned char *)buffer + size;
	_set_data_type(dcs, _DTYPE_BUFFER);
	sts = _dcs_judges(dcs, atype, ltype, stype, false, ltype != NULL);
	_clear_data_type(dcs, _DTYPE_BUFFER);
	ll_clear(dcs->_block);
	DCS_RETURN(dcs, sts);
}

/*
 * コード判定（ストリ−ム）
 */
int dcs_judges_stream(Dcs *dcs, int *atype, int *ltype, FILE *sfp, int stype) {
	int sts;

	dcs->_sfp = sfp;
	_set_data_type(dcs, _DTYPE_STREAM);
	sts = _dcs_judges(dcs, atype, ltype, stype, false, ltype != NULL);
	_clear_data_type(dcs, _DTYPE_STREAM);
	ll_clear(dcs->_block);
	DCS_RETURN(dcs, sts);
}

/*
 * コード判定（ファイル）
 */
int dcs_judges_file(Dcs *dcs, int *atype, int *ltype, const char *spath, int stype) {
	int sts;
	FILE *fp = fopen(spath, "rb");
	if (fp == NULL) {
		dcs->_dcs_errno = errno;
		sts = DCS_EROPEN;
		DCS_RETURN(dcs, sts);
	}
	_set_data_type(dcs, _DTYPE_FILE);
	sts = dcs_judges_stream(dcs, atype, ltype, fp, stype);
	_clear_data_type(dcs, _DTYPE_FILE);
	fclose(fp);
	DCS_RETURN(dcs, sts);
}

/*
 * コード変換（バッファー）
 */
int dcs_convert_buffer(Dcs *dcs, const char *buffer, size_t size, int stype, int dtype, bool is_forced,
		int (*observer)(int sts, void *user_info), int (*writer)(const char *buf, size_t size, void *user_info), void *user_info) {
	int sts;
	dcs->_sp = (unsigned char *)buffer;
	dcs->_ep = (unsigned char *)buffer + size;
	dcs->_writer = writer;
	dcs->_observer = observer;
	dcs->_user_info = user_info;
	_set_data_type(dcs, _DTYPE_BUFFER);
	sts = _dcs_convert(dcs, stype, dtype, is_forced);
	_clear_data_type(dcs, _DTYPE_BUFFER);
	ll_clear(dcs->_block);
	DCS_RETURN(dcs, sts);
}

/*
 * コード変換（ストリーム）
 */
int dcs_convert_stream(Dcs *dcs, FILE *sfp, int stype, int dtype, bool is_forced,
		int (*observer)(int sts, void *user_info), int (*writer)(const char *buf, size_t size, void *user_info), void *user_info) {
	int sts;
	dcs->_sfp = sfp;
	dcs->_writer = writer;
	dcs->_observer = observer;
	dcs->_user_info = user_info;
	_set_data_type(dcs, _DTYPE_STREAM);
	sts = _dcs_convert(dcs, stype, dtype, is_forced);
	_clear_data_type(dcs, _DTYPE_STREAM);
	ll_clear(dcs->_block);
	DCS_RETURN(dcs, sts);
}

/*
 * コード変換（ファイル）
 */
int dcs_convert_file(Dcs *dcs, const char *spath, int stype, int dtype, bool is_forced,
		int (*observer)(int sts, void *user_info), int (*writer)(const char *buf, size_t size, void *user_info), void *user_info) {
	int sts;
	FILE *fp = fopen(spath, "rb");
	if (fp == NULL) {
		dcs->_dcs_errno = errno;
		sts = DCS_EROPEN;
		DCS_RETURN(dcs, sts);
	}
	_set_data_type(dcs, _DTYPE_FILE);
	sts = dcs_convert_stream(dcs, fp, stype, dtype, is_forced, observer, writer, user_info);
	_clear_data_type(dcs, _DTYPE_FILE);
	fclose(fp);
	dcs->_dcs_errno = errno;
	DCS_RETURN(dcs, sts);
}

// BufferInfo dcs->_buffer_info;

/* Push Back バッファーから */ \
#define restore_buf(dcs, sp) { \
		int _c; \
		/* Push Back バッファーから */ \
		while(dcs->_bcp > dcs->_btp) { \
			if ((_c = *--dcs->_bcp) < 0) { \
				break; \
			} \
			*sp++ = _c; \
		} }

#define get_one_buffer(dcs, sp, ep, alen, rlen) { \
	/* Push Back バッファーから */ \
	restore_buf(dcs, sp); \
	if (_is_type_buffer(dcs)) { \
	/* バッファーの時（JNI共通） */ \
		rlen = min(ep - sp, dcs->_ep - dcs->_sp); \
		if (ep - sp < dcs->_ep - dcs->_sp) { \
			rlen = ep - sp; \
		} else { \
			rlen = dcs->_ep - dcs->_sp; \
			/* 長さが等しくても EOF */ \
			is_eof = true; \
		} \
		memcpy(sp, dcs->_sp, rlen); \
		dcs->_sp += rlen; \
		sp += rlen; \
	} else if (_is_type_stream(dcs)) { \
		/* ストリームの時（ファイル、JNIファイル共通） */ \
		rlen = fread(sp, 1, ep - sp, dcs->_sfp); \
		if (feof(dcs->_sfp)) { \
			is_eof = true; \
		} else if(ferror(dcs->_sfp)) { \
			dcs->_dcs_errno = errno; \
			ll_clear((ll_list *)dcs->_block); \
			return DCS_EREAD; \
		} \
		sp += rlen; \
	} else {/* if (_is_type_jni_stream(dcs)) */ \
	/* JNI ストリームの時 */ \
		rlen = min(ep - sp, alen); \
		rlen = (*jni_info->jenv)->CallIntMethod( \
							jni_info->jenv, \
							jni_info->jistream, \
							jni_info->mid_buf, \
							jni_info->jbuf, \
							0, \
							rlen); \
		/* IOException とは限らない */ \
		if ((*jni_info->jenv)->ExceptionCheck(jni_info->jenv)) { \
			(*jni_info->jenv)->ExceptionClear(jni_info->jenv); \
			ll_clear((ll_list *)dcs->_block); \
			return DCS_EREAD; \
		} \
		if (rlen < 0) { \
			is_eof = true; \
		} else { \
			(*jni_info->jenv)->GetByteArrayRegion(jni_info->jenv, jni_info->jbuf, 0, rlen, (jbyte *)sp); \
			sp += rlen; \
		} \
	} }

#define _scompare(tinfo1, tinfo2) ( \
			tinfo1->zen_cnt >= tinfo2->zen_cnt \
				&& tinfo1->bad_cnt <= tinfo2->bad_cnt \
				&& tinfo1->zen_kana_cnt >= tinfo2->zen_kana_cnt \
				&& (tinfo1->zen_cnt > tinfo2->zen_cnt \
						|| tinfo1->bad_cnt < tinfo2->bad_cnt \
						|| tinfo1->zen_kana_cnt > tinfo2->zen_kana_cnt)? -1: \
			tinfo1->zen_cnt <= tinfo2->zen_cnt \
			 && tinfo1->bad_cnt >= tinfo2->bad_cnt \
			 && tinfo1->zen_kana_cnt <= tinfo2->zen_kana_cnt \
			 && (tinfo1->zen_cnt < tinfo2->zen_cnt \
					 || tinfo1->bad_cnt > tinfo2->bad_cnt \
					 || tinfo1->zen_kana_cnt < tinfo2->zen_kana_cnt)? 1: \
			tinfo1->zen_cnt == tinfo2->zen_cnt \
			 && tinfo1->bad_cnt == tinfo2->bad_cnt \
			 && tinfo1->zen_kana_cnt == tinfo2->zen_kana_cnt \
			 && ((IS_JIS(tinfo1->type) && IS_JIS(tinfo2->type)) \
					|| (IS_JIS(tinfo1->type) && IS_JIS(tinfo2->type)) \
					|| (IS_SJIS(tinfo1->type) && IS_SJIS(tinfo2->type)) \
					|| (IS_EUC(tinfo1->type) && IS_EUC(tinfo2->type)))? tinfo1->type - tinfo2->type: \
	tinfo1->bad_cnt != 0 && tinfo2->bad_cnt == 0? 1: \
		tinfo1->bad_cnt == 0 && tinfo2->bad_cnt != 0? -1: \
		tinfo1->zen_cnt > tinfo1->bad_cnt && tinfo2->bad_cnt > tinfo2->zen_cnt? -1: \
		tinfo1->zen_cnt < tinfo1->bad_cnt && tinfo2->bad_cnt < tinfo2->zen_cnt? 1: \
				tinfo1->zen_cnt > 0 && tinfo2->zen_cnt > 0? ( \
		(double)tinfo1->tan_han_kana_cnt/tinfo1->zen_cnt < (double)tinfo2->tan_han_kana_cnt/tinfo2->zen_cnt? -1: \
		(double)tinfo1->tan_han_kana_cnt/tinfo1->zen_cnt > (double)tinfo2->tan_han_kana_cnt/tinfo2->zen_cnt? 1: \
		(double)tinfo1->han_kana_cnt/tinfo1->zen_cnt < (double)tinfo2->han_kana_cnt/tinfo2->zen_cnt? -1: \
		(double)tinfo1->han_kana_cnt/tinfo1->zen_cnt > (double)tinfo2->han_kana_cnt/tinfo2->zen_cnt? 1: \
		tinfo1->bad_cnt + tinfo1->tan_han_kana_cnt != tinfo2->bad_cnt + tinfo2->tan_han_kana_cnt? \
				tinfo1->bad_cnt + tinfo1->tan_han_kana_cnt - tinfo2->bad_cnt - tinfo2->tan_han_kana_cnt: \
		tinfo1->bad_cnt != tinfo2->bad_cnt? tinfo1->bad_cnt - tinfo2->bad_cnt: 0): \
		tinfo1->bad_cnt + tinfo1->tan_han_kana_cnt != tinfo2->bad_cnt + tinfo2->tan_han_kana_cnt? \
				tinfo1->bad_cnt + tinfo1->tan_han_kana_cnt - tinfo2->bad_cnt - tinfo2->tan_han_kana_cnt: \
				tinfo1->bad_cnt != tinfo2->bad_cnt? tinfo1->bad_cnt - tinfo2->bad_cnt: 0)

/* 共通コード判定 */
int _dcs_judges(Dcs *dcs, int *atype, int *ltype, int stype, bool is_forced, bool is_keep) {
	int i, clen, three_bytes;
	int c, c2, c3, c4, ch;
	unsigned long lch;
	unsigned char *sp, *ep, *tp, *zp, *han_kana_p;
	int ecnt;
	int err_no;
	int32_t maxb;
	int normal_type;
	unsigned flag_btype, flag_btype_all;
	int32_t ubacnt, ulacnt, ubacnt_all, ulacnt_all;
	int32_t uba_ulz_cnt, ula_ubz_cnt;
	int dtype;
	int ub_mac_cnt, ul_mac_cnt, u8_mac_cnt, u32b_mac_cnt, u32l_mac_cnt;
	int ub_surrog_cnt, ul_surrog_cnt, u8_surrog_cnt;
	long long _source_size;
	bool is_ascii, is_all_ascii, is_eof, is_jis_kana, is_bad_jis_kana;
	bool is_utfmac_d;
	unsigned int type_surrogate;
	bool is_u8_surrogate, is_ub_surrogate, is_ul_surrogate, is_u32b_surrogate, is_u32l_surrogate;
	int retsts;
	ll_list *clist = ll_first((ll_list *)dcs->_block);
	CodeInfo code_info[_I_MAX];
	CodeInfo *sjis_info = &code_info[_I_SJIS];
	CodeInfo *sjis1_info = &code_info[_I_SJIS1];
	CodeInfo *sjis2_info = &code_info[_I_SJIS2];
	CodeInfo *euc_info = &code_info[_I_EUC];
	CodeInfo *euc1_info = &code_info[_I_EUC1];
	CodeInfo *euc2_info = &code_info[_I_EUC2];
	CodeInfo *euc3_info = &code_info[_I_EUC3];
	CodeInfo *euc4_info = &code_info[_I_EUC4];
	CodeInfo *u8_info = &code_info[_I_UTF8];
	CodeInfo *ub_info = &code_info[_I_UTF16BE];
	CodeInfo *ul_info = &code_info[_I_UTF16LE];
	CodeInfo *u32b_info = &code_info[_I_UTF32BE];
	CodeInfo *u32l_info = &code_info[_I_UTF32LE];
	CodeInfo *cinfo_sp[] = {
			sjis_info, euc_info, u8_info, ub_info, u32b_info,
	};
	CodeInfo *cinfo[] = {
			&code_info[0], &code_info[1], &code_info[2],
			&code_info[3], &code_info[4], &code_info[5],
			&code_info[6], &code_info[7], &code_info[8],
			&code_info[9], &code_info[10], &code_info[11], &code_info[12],
	};
	int types[] = {
			DCS_SJIS, DCS_SJIS1, DCS_SJIS2,
			DCS_EUC, DCS_EUC1, DCS_EUC2, DCS_EUC3, DCS_EUC4,
			DCS_UTF8, DCS_UTF16BE, DCS_UTF16LE,
			DCS_UTF32BE, DCS_UTF32LE,
	};
	int nd_types[] = {
			DCS_ND_SJIS, DCS_ND_SJIS1, DCS_ND_SJIS2,
			DCS_ND_EUC, DCS_ND_EUC1, DCS_ND_EUC2, DCS_ND_EUC3, DCS_ND_EUC4,
			DCS_ND_UTF8, DCS_ND_UTF16BE, DCS_ND_UTF16LE,
			DCS_ND_UTF32BE, DCS_ND_UTF32LE,
	};
	JniJudgedStreamInfo * jni_info = (JniJudgedStreamInfo *)dcs->_jni_info;
	int rlen, alen = (_is_type_jni_stream(dcs)?
			(*jni_info->jenv)->GetArrayLength(jni_info->jenv, jni_info->jbuf): 0);

	if (atype != NULL) {
		*atype = DCS_UNKNOWN;
	}
	if (ltype != NULL) {
		*ltype = LTYPE_NON;
	}
	memset(&dcs->_buffer_info, 0, sizeof(dcs->_buffer_info));
	dcs->_is_surrogate = dcs->_utfmac_exists = false;
	dcs->_bcp = dcs->_btp = dcs->_bbuf; // プッシュ・バック・バッファーをリセット
	sp = dcs->_buffer_info.buf;
	*sp++ = c = _dcs_getc(dcs);	// 第１バイト
	if (c == EOF) {
		err_no = errno;
		if (_is_type_file(dcs) && ferror(dcs->_sfp)) {
		/* ファイル・アクセス・エラーの時 */
			dcs->_dcs_errno = err_no;
			ll_clear((ll_list *)dcs->_block);
			return DCS_EREAD;
		}
		if ((atype) != NULL) {
			*(atype) = (DCS_EMPTY);
		}
		return (DCS_EMPTY);
	}
	*sp++ = c2 = _dcs_getc(dcs);	// 第２バイト
	if (c == 0 && c2 == 0) {
	// UTF32BE BOM の可能性あり
		*sp++ = c = _dcs_getc(dcs);	// 第３バイト
		if (c == EOF) {
			c2 = EOF;
		} else {
			*sp++ = c2 = _dcs_getc(dcs);	// 第４バイト
		}
		if (c == UTF16_B0 && c2 == UTF16_B1) {
		/* UTFBE BOM がある時 */
			FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_UTF32BE_BOM, is_keep);
		}
	} else if (c == UTF16_B0 && c2 == UTF16_B1) {
		/* UTFBE BOM がある時 */
			FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_UTF16BE_BOM, is_keep);
	} else if (c == UTF16_B1 && c2 == UTF16_B0) {
	/* UTF16LE BOM がある時 */
		c = _dcs_getc(dcs);		// 第３バイト
		c2 = _dcs_getc(dcs);	// 第４バイト
		if (c == 0 && c2 == 0) {
		// UTF32LE BOM の時
			retsts = DCS_UTF32LE_BOM;
		} else {
			// 読み過ぎの２バイトを戻す
			_dcs_push_back(dcs, c2);
			_dcs_push_back(dcs, c);
			retsts = DCS_UTF16LE_BOM;
		}
		FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, retsts, is_keep);
	}
	c3 = 0;
	if (c == UTF8_B0 && c2 == UTF8_B1) {
	/* UTF8 BOM 先頭２バイトの時 */
		*sp++ = c3 = _dcs_getc(dcs);
		if (c3 == UTF8_B2) {
		/* UTF8 BOM がある時 */
			FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_UTF8_BOM, is_keep);
		}
	}
	if (c2 == EOF || c3 == EOF) {
		err_no = errno;
		if (_is_type_file(dcs) && ferror(dcs->_sfp)) {
		/* ファイル・アクセス・エラーの時 */
			dcs->_dcs_errno = err_no;
			ll_clear((ll_list *)dcs->_block);
			return DCS_EREAD;
		}
		sp--;
	}
	memset(code_info, 0, sizeof(code_info));

	for(i=0; i<sizeof(code_info)/sizeof(code_info[0]); i++) {
		code_info[i].type = types[i];
		code_info[i].nd_type = nd_types[i];
	}
	set_jis2022jp_to(dcs);
	set_to_jis2022jp(dcs);
	set_sjis_to(dcs);
	set_to_sjis(dcs);
	set_euccommon_to(dcs);
	set_to_euccommon(dcs);
//
	is_eof = is_bad_jis_kana = false;
	is_all_ascii = true;
	ub_mac_cnt = ul_mac_cnt = u8_mac_cnt = 0;
	u32b_mac_cnt = u32l_mac_cnt = 0;
	ub_surrog_cnt = ul_surrog_cnt = u8_surrog_cnt = 0;
	ubacnt_all = ulacnt_all = uba_ulz_cnt = ula_ubz_cnt = 0;
	_source_size = 0;
	flag_btype_all = 0;
	type_surrogate = 0;
	is_u8_surrogate = is_ub_surrogate = is_ul_surrogate
			= is_u32b_surrogate = is_u32l_surrogate = false;
//
	for(;;) {
		is_ascii = true;
		ubacnt = ulacnt = 0;
		flag_btype = (DCS_ND_ALL & ~(DCS_ND_JIS_ALL | DCS_ND_UTF_BOM));
		ep = dcs->_buffer_info.buf + sizeof(dcs->_buffer_info.buf) - _MAX_OFFSET;
		get_one_buffer(dcs, sp, ep, alen, rlen);
		ep = sp;
		dcs->_buffer_info.length = ep - dcs->_buffer_info.buf;
		if (is_keep) {
		/* バッファーを保持する時 */
			if ((clist = _add_buffer(clist, &dcs->_buffer_info, sizeof(dcs->_buffer_info))) == NULL) {
				ll_clear((ll_list *)dcs->_block);
				return DCS_EALLOC;
			}
		}
		if (is_forced) {
		// 強制指示の時
			if (is_eof) {
				CHECKLN_SETA_AND_RETURN(dcs, ltype, atype, 0, stype);
			}
			dcs->_buffer_info.offset = _MAX_OFFSET;
			dcs->_buffer_info.s_offset = _MAX_OFFSET;
			dcs->_buffer_info.e_offset = _MAX_OFFSET;
			dcs->_buffer_info.u8_offset  = _MAX_OFFSET;
			dcs->_buffer_info.u16_offset = _MAX_OFFSET;
			dcs->_buffer_info.u32_offset = _MAX_OFFSET;
			sp = dcs->_buffer_info.buf + _MAX_OFFSET;
			continue;
		}
/* JIS, SJIS の解析	 --------------------------------------------------------*/
		sp = han_kana_p = dcs->_buffer_info.buf + dcs->_buffer_info.s_offset;
		sjis_info->blen = 0;
		_source_size += (ep - sp);
//
		while(sp < ep) {
			c = *sp++;
			if ((c & ~0x7f) == 0) {
			/* サイン・ビットが立っていない時 */
				if (c == 0) {
				/* UTF16 のASCII がある時 */
					if (((sp - dcs->_buffer_info.buf) & 01) == 0) {
					/* UTF16LE の ASCII がある時 */
						ulacnt++;
						ch = *(sp - 2);
						if (is_unibestd_short(dcs, ch)) {
						/* UTF16BE の全角が有効の時 */
							ula_ubz_cnt++;
						}
					} else {
					/* UTF16BE の ASCII がある時 */
						if (sp >= ep) {
						/* 次のバイトがない時 */
							break;
						}
						ubacnt++;
						ch = *sp;
						if (is_unibestd_short(dcs, ch)) {
						/* UTF16LE の全角が有効の時 */
							uba_ulz_cnt++;
						}
						sp++;
					}
					continue;
				}
				if ((c & ~0x1f) == 0) {
				/* 制御文字の時 */
					if (c == _ESC_CHAR) {	/* エスケープ(0x1b) */
						if (sp + 2 > ep) {
							sp += 2;
						} else {
							c2 = *sp++;
							if (c2 == '$') {	/* JIS漢字SHIFT IN */
								c3 = *sp++;
								switch(c3) {
								case '@':	/* 78JIS */
								case 'B':	/* 83JIS */
								flag_btype |= DCS_ND_JIS;
									break;
								case '(':	/* JIS X 0213 */
									if (sp + 1 > ep) {
										sp++;
									} else {
										c3 = *sp++;
										switch(c3) {
										case 'D': /* 補助漢字 */
											flag_btype |= DCS_ND_JIS1;
											break;
										case 'Q':	/* 第1面 */
											flag_btype |= DCS_ND_JIS2;
											type_surrogate |= DCS_ND_JIS2;
											if (!_BINARY_RANGE(sjis_info->bad_cnt, _source_size)) {
											/* バイナリー域ではない時（SJISで代用） */
												FILL_BUFFER_AND_RETURN(dcs, ltype, atype, type_surrogate, DCS_JIS2, is_keep);
											}
											/* no break */
										case 'O': /* JIS X 0213:2000第1面 */
										case 'P':	/* 第2面 */
											flag_btype |= DCS_ND_JIS3;
											break;
										default:
											sp--;
											break;
										}
									}
									break;
								default:
									sp--;
									break;
								}
							} else if (c2 == '(') {	/* JIS漢字SHIFT OUT */
								c3 = *sp++;
								switch(c3) {
								case 'J':	/* JISローマ字 */
								case 'H':	/* JISローマ字 */
								case 'B':	/* ASCII */
									flag_btype |= DCS_ND_JIS;
									break;
								case 'I':	/* JIS半角カナ */
									flag_btype |= DCS_ND_JIS;
									break;
								default:
									sp--;
									break;
								}
							} else {
								sp--;
							}
						}
					} else if (c == _SO_CHAR && !is_bad_jis_kana) {
						/* JIS半角カナ SHIFT OUT */
						is_jis_kana = false;
						for(; sp < ep;) {
							c = *sp++;
							if (is_jis_kana(c)) {
								is_jis_kana = true;
							} else {
								if (c == _SI_CHAR){
									if (is_jis_kana) {
										flag_btype |= (DCS_ND_JIS | DCS_ND_JIS1);
									}
								} else {
									is_bad_jis_kana = true;
									sp--;
								}
								break;
							}
						}
					}
				}
				continue;
			} else {
			/* サイン・ビットが立っている時 */
				/* ASCII をクリア */
				is_ascii = is_all_ascii = false;
				if (is_sjis_kana(c)) {
				/* 半角カナの時 */
					sjis_info->han_kana_cnt++;
					sjis_info->zen_cnt++;
					if (han_kana_p < sp - 2
					 && (sp >= ep || (c2 = *sp, !is_sjis_kana(c2)))) {
					/* 半角カナが単一の時 */
						sjis_info->tan_han_kana_cnt++;
					}
					han_kana_p = sp - 1;
				} else if (sp >= ep) {
				/* 泣き別れの時 */
					if (is_eof) {
						sjis_info->bad_cnt += 2;
					}
					sjis_info->blen = 1;
					break;
				} else {
					c2 = *sp++;
					ch = ((c << 8)|c2);
					if (is_sjis_short(dcs, ch)) {
					/* SJIS 全角の時 */
						if (is_sjis_zenkana(ch)) {
						/* SJIS 全角かなの時 */
							sjis_info->zen_kana_cnt += 2;
						}
						sjis_info->zen_cnt += 2;
					} else {
						set_sjis932_to(dcs);
						if (is_sjis_short(dcs, ch)) {
						/* SJIS 全角の時 */
							sjis1_info->zen_cnt += 2;
						} else {
							sjis1_info->bad_cnt += 2;
						}
						set_sjis2004_to(dcs);
						if (is_sjis_short(dcs, ch)) {
						/* SJIS 全角の時 */
							sjis2_info->zen_cnt += 2;
							type_surrogate |= DCS_ND_SJIS2;
						} else {
							sjis2_info->bad_cnt += 2;
						}
						sjis_info->bad_cnt += 2;
						set_sjis_to(dcs);
					}
				}
			}
		} /* while(sp < ep) */
		if (is_eof) {
			// CP932
			sjis1_info->zen_cnt += sjis_info->zen_cnt;
			sjis1_info->zen_kana_cnt += sjis_info->zen_kana_cnt;
			sjis1_info->han_kana_cnt += sjis_info->han_kana_cnt;
			sjis1_info->tan_han_kana_cnt += sjis_info->tan_han_kana_cnt;
			// Shift_JIS-2004
			sjis2_info->zen_cnt += sjis_info->zen_cnt;
			sjis2_info->zen_kana_cnt += sjis_info->zen_kana_cnt;
			sjis2_info->han_kana_cnt += sjis_info->han_kana_cnt;
			sjis2_info->tan_han_kana_cnt += sjis_info->tan_han_kana_cnt;
			if (!is_all_ascii) {
			/* ASCII ではない時 */
				flag_btype &= ~DCS_ND_ASCII;
			}
			/* 全角の数が等しい時は、共通コードを採る */
			if (stype != DCS_SJIS2) {
			/* SJIS2 の指示がない時 */
				if (sjis2_info->zen_cnt == sjis_info->zen_cnt) {
					flag_btype &= ~DCS_ND_SJIS2;
				}
			}
			if (!_BINARY_RANGE(sjis_info->bad_cnt, _source_size)) {
			/* バイナリー域ではない時（SJISを兼用） */
				flag_btype_all |= flag_btype;
				if ((flag_btype_all & DCS_ND_JIS_ALL) != 0) {
					stype = ((flag_btype_all & DCS_ND_JIS2) != 0? DCS_JIS2:
							(flag_btype_all & DCS_ND_JIS3) != 0? DCS_JIS3:
							(flag_btype_all & DCS_ND_JIS1) != 0? DCS_JIS1: DCS_JIS);
					CHECKLN_SETA_AND_RETURN(dcs, ltype, atype, 0, stype);
				}
				if (is_forced) {
					CHECKLN_SETA_AND_RETURN(dcs, ltype, atype, type_surrogate, stype);
				}
			}
			// JIS を解除
			flag_btype &= ~DCS_ND_JIS_ALL;
		}
		if (ubacnt > 0 || ulacnt > 0) {
		/* UTF16 の ASCII があった時 */
			is_ascii = is_all_ascii = false;
			ubacnt_all += ubacnt;
			ulacnt_all += ulacnt;
		}
		{
/* EUC の解析 ---------------------------------------------------------------*/
			sp = han_kana_p = dcs->_buffer_info.buf + dcs->_buffer_info.e_offset;
			if (is_ascii) {
			/* ASCII だった時 */
				zp = ep;
				ep = sp + _MAX_EUC_OFFSET;
			} else {
			/* ASCII ではなかった時 */
				/* ASCII, JIS をクリア */
				flag_btype &= ~(DCS_ND_ASCII | DCS_ND_JIS_ALL);
				euc_info->blen = u8_info->blen = 0;
				zp = NULL;
			}
//
			while(sp < ep) {
				if ((c = *sp++) <= 0x7f) {
					continue;
				}
				three_bytes = 0;
				if (c == 0x8e) {
					if ((sp >= ep)) {
						if (is_eof) {
							euc_info->bad_cnt += 2;
							euc1_info->bad_cnt += 2;
							euc2_info->bad_cnt += 2;
							euc3_info->bad_cnt += 2;
							euc4_info->bad_cnt += 2;
						}
						euc_info->blen = 1;
						break;
					} else {
						c2 = *sp++;
						if (is_sjis_kana(c2)) {
							euc_info->han_kana_cnt++;
							euc_info->zen_cnt += 2;
							if (han_kana_p < sp - 4 && (sp + 1 >= ep
							 || !(*sp == 0x8e && is_sjis_kana(*(sp + 1))))) {
							/* 半角カナが単一の時 */
								euc_info->tan_han_kana_cnt += 2;
							}
							han_kana_p = sp - 2;
						} else {
							euc_info->bad_cnt += 2;
						}
					}
					continue;
				} else if (c == 0x8f) {
					if (sp + 2 > ep) {
						if (is_eof) {
							euc1_info->bad_cnt += 3;
							euc3_info->bad_cnt += 3;
							euc4_info->bad_cnt += 3;
						}
						euc_info->blen = 2 - (ep - sp);
						break;
					}
					three_bytes = 1;
					c = ((c << 8) | *sp++);
				} else if (sp >= ep) {
					if (is_eof) {
						euc_info->bad_cnt += 2;
						euc1_info->bad_cnt += 2;
						euc2_info->bad_cnt += 2;
						euc3_info->bad_cnt += 2;
						euc4_info->bad_cnt += 2;
					}
					euc_info->blen = 1;
					break;
				}
				c = ((c << 8) | *sp++);
				if (is_euc_long(dcs, c)) {
					if (is_euc_zenkana(c)) {
						euc_info->zen_kana_cnt += 2;
					}
					euc_info->zen_cnt += 2 + three_bytes;
				} else {
					euc_info->bad_cnt += 2 + three_bytes;
					set_euc20932_to(dcs);
					if (is_euc_long(dcs, c)) {
						euc1_info->zen_cnt += 2 + three_bytes;
					} else {
						euc1_info->bad_cnt += 2 + three_bytes;
					}
					if (three_bytes == 0) {
					/* 0x8fxxxx コードではない時 */
						set_euc51932_to(dcs);
						if (is_euc_long(dcs, c)) {
							euc2_info->zen_cnt += 2;
						} else {
							euc2_info->bad_cnt += 2;
						}
					} else {
						euc2_info->bad_cnt += 2 + three_bytes;
					}
					set_eucms_to(dcs);
					if (is_euc_long(dcs, c)) {
						/* 拡張 Unicode(BE)2 判定 */
						if (!is_unibeex_short(dcs, _euc_to_unibe(dcs, c))) {
							euc3_info->zen_cnt += 2 + three_bytes;
						}
					} else {
						euc3_info->bad_cnt += 2 + three_bytes;
					}
					set_euc2004_to(dcs);
					if (is_euc_surrogate(dcs, c)) {
						type_surrogate |= DCS_ND_EUC4;
						set_to_euc2004(dcs);
					}
					if (is_euc_long(dcs, c)) {
						euc4_info->zen_cnt += 2 + three_bytes;
					} else {
						euc4_info->bad_cnt += 2 + three_bytes;
					}
					set_euccommon_to(dcs);
				} /* if (is_euc_long(dcs, c)) */
			} /* while(sp < ep) */
			if (is_eof) {
				/* 共通コードのカウントを他のコードに加える */
				if (euc_info->zen_cnt != 0) {
					euc1_info->zen_cnt += euc_info->zen_cnt;
					euc2_info->zen_cnt += euc_info->zen_cnt;
					euc3_info->zen_cnt += euc_info->zen_cnt;
					euc4_info->zen_cnt += euc_info->zen_cnt;
				}
				if (euc_info->zen_kana_cnt != 0) {
					euc1_info->zen_kana_cnt += euc_info->zen_kana_cnt;
					euc2_info->zen_kana_cnt += euc_info->zen_kana_cnt;
					euc3_info->zen_kana_cnt += euc_info->zen_kana_cnt;
					euc4_info->zen_kana_cnt += euc_info->zen_kana_cnt;
				}
				if (euc_info->han_kana_cnt != 0) {
					euc1_info->han_kana_cnt += euc_info->han_kana_cnt;
					euc2_info->han_kana_cnt += euc_info->han_kana_cnt;
					euc3_info->han_kana_cnt += euc_info->han_kana_cnt;
					euc4_info->han_kana_cnt += euc_info->han_kana_cnt;
				}
				/* 共通コードが泣き別れの時は、全てが泣き別れ */
				/* 全角の数が等しい時は、共通コードを採る */
				if (stype != DCS_EUC1) {
				/* EUC1 の指示がない時 */
					if (euc1_info->zen_cnt == euc_info->zen_cnt) {
						flag_btype &= ~DCS_ND_EUC1;
					}
				}
				if (stype != DCS_EUC2) {
				/* EUC2 の指示がない時 */
					if (euc2_info->zen_cnt == euc_info->zen_cnt) {
						flag_btype &= ~DCS_ND_EUC2;
					}
				}
				if (stype != DCS_EUC3) {
				/* EUC3 の指示がない時 */
					if (euc3_info->zen_cnt == euc_info->zen_cnt) {
						flag_btype &= ~DCS_ND_EUC3;
					} else if (euc3_info->zen_cnt > euc_info->zen_cnt) {
						if ((flag_btype & DCS_ND_EUC1) != 0) {
							if (euc1_info->zen_cnt == euc3_info->zen_cnt
							 && euc1_info->bad_cnt == euc3_info->bad_cnt) {
							/* EUC1, EUC3 の区別が付かない時 */
								flag_btype &= ~DCS_ND_EUC3;
							}
						}
						if ((flag_btype & DCS_ND_EUC2) != 0) {
							if (euc2_info->zen_cnt == euc3_info->zen_cnt
							 && euc2_info->bad_cnt == euc3_info->bad_cnt) {
							/* EUC2, EUC3 の区別が付かない時 */
								flag_btype &= ~DCS_ND_EUC3;
							}
						}
					}
				}
				if (stype != DCS_EUC4) {
				/* EUC4 の指示がない時 */
					if (euc4_info->zen_cnt == euc_info->zen_cnt) {
						flag_btype &= ~DCS_ND_EUC4;
					} else if (euc4_info->zen_cnt > euc_info->zen_cnt) {
						if ((flag_btype & DCS_ND_EUC1) != 0) {
							if (euc1_info->zen_cnt == euc4_info->zen_cnt
							 && euc1_info->bad_cnt == euc4_info->bad_cnt) {
							/* EUC1, EUC4 の区別が付かない時 */
								flag_btype &= ~DCS_ND_EUC4;
							}
						}
						if ((flag_btype & DCS_ND_EUC2) != 0) {
							if (euc2_info->zen_cnt == euc4_info->zen_cnt
							 && euc2_info->bad_cnt == euc4_info->bad_cnt) {
							/* EUC2, EUC4 の区別が付かない時 */
								flag_btype &= ~DCS_ND_EUC4;
							}
						}
						if ((flag_btype & DCS_ND_EUC3) != 0) {
							if (euc3_info->zen_cnt == euc4_info->zen_cnt
							 && euc3_info->bad_cnt == euc4_info->bad_cnt) {
							/* EUC3, EUC4 の区別が付かない時 */
								flag_btype &= ~DCS_ND_EUC4;
							}
						}
					}
				}
			}
/* UTF-8 の解析	------------------------------------------------------------*/
			sp = dcs->_buffer_info.buf + dcs->_buffer_info.u8_offset;
			is_utfmac_d = false;
			ecnt = 0;
//
			while(sp < ep) {
				tp = sp;
				c = *sp++;
				if ((c & 0x80) == 0) {
				/* 0xxxxxxx: １バイト文字 */
					continue;
				}
				if (sp >= ep) {
					if (is_eof) {
						u8_info->bad_cnt += 2;
					}
					u8_info->blen = 1;
					break;
				}
				c2 = *sp++;
				if ((c & 0xe0) == 0xc0) {
					if ((c2 & 0xc0) == 0x80) {
					/* 110xxxxx 10xxxxxx: ２バイト文字 */
						ch = ((c&0x1f)<<6|(c2&0x3f));
						if (is_unibe_short(dcs, ch)) {
							u8_info->zen_cnt += 2;
							if (is_unibe_mac(dcs, ch, CSB_MAC_BO | CSB_MAC_TO)) {
							/* UTF8-MAC 短音符（ブレーヴェ）、分音符（ウムラウト）の時 */
								is_utfmac_d = true;
							} else if (is_utfmac_d) {
								if (ch == MAC_BUNONPU || ch == MAC_TANONPU) {
									u8_mac_cnt += 4;
								}
								is_utfmac_d = false;
							}
						} else {
							u8_info->bad_cnt += 2;
						}
					} else {
						u8_info->bad_cnt += 2;
					}
				} else {
					if (sp >= ep) {
						if (is_eof) {
							u8_info->bad_cnt += 3;
						}
						u8_info->blen = 2;
						break;
					}
					c3 = *sp++;
					if ((c & 0xf0) == 0xe0) {
						if ((c2 & 0xc0) == 0x80 && (c3 & 0xc0) == 0x80) {
						/* 1110xxxx 10xxxxxx 10xxxxxx: ３バイト文字 */
							ch = ((c&0x0f)<<12|(c2&0x3f)<<6|(c3&0x3f));
							if (is_unibe_short(dcs, ch)) {
								u8_info->zen_cnt += 3;
								if (is_unibe_zenkana(ch)) {
								/* 全角仮名の時 */
									u8_info->zen_kana_cnt += 3;
									if (is_unibe_mac(dcs, ch, CSB_MAC_D | CSB_MAC_H)) {
									/* UTF8-MAC 濁点、半濁点の時 */
										is_utfmac_d = true;
									} else if (is_utfmac_d) {
										if (ch == MAC_DAKUTEN || ch == MAC_HANDAKUTEN) {
											u8_mac_cnt += 6;
										}
										is_utfmac_d = false;
									}
								} else if(is_unibe_kana(ch)) {
									u8_info->han_kana_cnt++;
								}
							} else {
								u8_info->bad_cnt += 3;
							}
						} else {
							u8_info->bad_cnt += 3;
						}
					} else {
						if (sp >= ep) {
							if (is_eof) {
								u8_info->bad_cnt += 4;
							}
							u8_info->blen = 3;
							break;
						}
						c4 = *sp++;
						if ((c & 0xf8) == 0xf0 && (c2 & 0xc0) == 0x80 && (c3 & 0xc0) == 0x80 && (c4 & 0xc0) == 0x80) {
						/* 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx: ４バイト文字 */
							/* サロゲート文字 */
							u8_info->zen_cnt += 4;
							u8_surrog_cnt += 4;
						} else {
							ecnt++;
							sp = tp + 1;
							continue;
						}
					}
				}
			} /* while(sp < ep) */
			u8_info->bad_cnt += ecnt;
			if (is_eof) {
				if (u8_info->zen_cnt == 0) {
					/* ASCII 以外の文字が無かった時 */
					flag_btype &= ~DCS_ND_UTF8;
				}
			}
		} /* if (is_ascii) */
		if (zp != NULL) {
		/* 保存位置が設定されていた(is_ascii)時 */
			euc_info->blen = u8_info->blen = sjis_info->blen;
			ep = zp;
		}
/* UTF-16 の解析	 -----------------------------------------------------------*/
		sp = dcs->_buffer_info.buf + dcs->_buffer_info.u16_offset;
		ub_info->blen = 0; /* ul_info 共通 */
//
		while(sp < ep) {
			c = *sp++;
			if (sp >= ep) {
				if (is_eof) {
					ub_info->bad_cnt += 2;
					ul_info->bad_cnt += 2;
				}
				ub_info->blen = 1;
				break;
			}
			c2 = *sp++;
			/* Big Endian */
			ch = ((c << 8) | c2);

			if (is_unibestd_short(dcs, ch)) {
				if (is_unibe_zenkana(ch)) {
					ub_info->zen_kana_cnt += 2;
				} else if(is_unibe_kana(ch)) {
					ub_info->han_kana_cnt++;
				}
				ub_info->zen_cnt += 2;
				if (is_unibe_mac(dcs, ch, CSB_MAC_D | CSB_MAC_H | CSB_MAC_BO | CSB_MAC_TO)) {
				/* UTF8-MAC 濁点、半濁点の時 */
					if (sp + 2 <= ep) {
						c3 = (((*sp&0xff)<<8)|(*(sp+1)&0xff));
						if (c3 == MAC_DAKUTEN || c3 == MAC_HANDAKUTEN || c3 == MAC_BUNONPU || c3 == MAC_TANONPU) {
							ub_info->zen_cnt += 2;
							ub_mac_cnt += 4;
							sp += 2;
						}
					}
				}
			} else if (is_unibe_surrogate_high(ch)) {
				if (sp + 2 <= ep) {
					c3 = (((*sp&0xff)<<8)|(*(sp+1)&0xff));
					if (is_unibe_surrogate_low(c3)) {
						ub_info->zen_cnt += 4;
						ub_surrog_cnt += 4;
						sp += 2;
					}
				}
			} else if (ch == REP_CHARBE) {
			/* 置換文字の時 */
				ub_info->zen_cnt += 2;
			} else {
				ub_info->bad_cnt += 2;
			}
			/* Little Endian */
			ch = ((c2 << 8) | c);
			if (is_unibestd_short(dcs, ch)) {
				if (is_unibe_zenkana(ch)) {
					ul_info->zen_kana_cnt += 2;
				} else if(is_unibe_kana(ch)) {
					ul_info->han_kana_cnt++;
				}
				ul_info->zen_cnt += 2;
				if (is_unibe_mac(dcs, ch, CSB_MAC_D | CSB_MAC_H | CSB_MAC_BO | CSB_MAC_TO)) {
				/* UTF8-MAC 濁点、半濁点の時 */
					if (sp + 2 <= ep) {
						c3 = (((*(sp+1)&0xff)<<8)|(*sp&0xff));
						if (c3 == MAC_DAKUTEN || c3 == MAC_HANDAKUTEN || c3 == MAC_BUNONPU || c3 == MAC_TANONPU) {
							ul_info->zen_cnt += 2;
							ul_mac_cnt += 4;
							sp += 2;
						}
					}
				}
			} else if (is_unibe_surrogate_high(ch)) {
				if (sp + 2 <= ep) {
					c3 = (((*(sp+1)&0xff)<<8)|(*sp&0xff));
					if (is_unibe_surrogate_low(c3)) {
						ul_info->zen_cnt += 4;
						ul_surrog_cnt += 4;
						sp += 2;
					}
				}
			} else if (ch == REP_CHARLE) {
			/* 置換文字の時 */
				ul_info->zen_cnt += 2;
			} else {
				ul_info->bad_cnt += 2;
			}
		} /* while(sp < ep) */
/* UTF-32 の解析	 -----------------------------------------------------------*/
		sp = dcs->_buffer_info.buf + dcs->_buffer_info.u32_offset;
		u32b_info->blen = 0; /* u32l_info 共通 */
//
		while(sp < ep) {
			c = *sp++;
			if (sp + 2 >= ep) {
				if (is_eof) {
					u32b_info->bad_cnt += 4;
					u32l_info->bad_cnt += 4;
				}
				u32b_info->blen = ep - sp + 1;
				break;
			}
			c2 = *sp++;
			c3 = *sp++;
			c4 = *sp++;
			/* Big Endian */
			lch = ((c << 24) | (c2 << 16) | (c3 << 8) | c4);

			if (is_unibestd_short(dcs, lch)) {
				if (is_unibe_zenkana(lch)) {
					u32b_info->zen_kana_cnt += 4;
				} else if(is_unibe_kana(lch)) {
					u32b_info->han_kana_cnt++;
				}
				u32b_info->zen_cnt += 4;
				if (is_unibe_mac(dcs, lch, CSB_MAC_D | CSB_MAC_H | CSB_MAC_BO | CSB_MAC_TO)) {
				/* UTF8-MAC 濁点、半濁点の時 */
					if (sp + 4 <= ep) {
						c3 = (((*sp&0xff)<<24)|((*(sp+1)&0xff)<<16)|((*(sp+2)&0xff)<<8)|(*(sp+3)&0xff));
						if (c3 == MAC_DAKUTEN || c3 == MAC_HANDAKUTEN || c3 == MAC_BUNONPU || c3 == MAC_TANONPU) {
							u32b_info->zen_cnt += 4;
							u32b_mac_cnt += 8;
							sp += 4;
						}
					}
				}
			} else if (lch < 0x10000) {
				u32b_info->zen_cnt += 4;
			} else if (is_unibe_long(lch)) {
				u32b_info->zen_cnt += 4;
				is_u32b_surrogate = true;
			} else {
				u32b_info->bad_cnt += 4;
			}
			/* Little Endian */
			lch = ((c4 << 24) | (c3 << 16) | (c2 << 8) | c);
			if (is_unibestd_short(dcs, lch)) {
				if (is_unibe_zenkana(lch)) {
					u32l_info->zen_kana_cnt += 4;
				} else if(is_unibe_kana(lch)) {
					u32l_info->han_kana_cnt++;
				}
				u32l_info->zen_cnt += 4;
				if (is_unibe_mac(dcs, lch, CSB_MAC_D | CSB_MAC_H | CSB_MAC_BO | CSB_MAC_TO)) {
				/* UTF8-MAC 濁点、半濁点の時 */
					if (sp + 4 <= ep) {
						c3 = (((*(sp+3)&0xff)<<24)|((*(sp+2)&0xff)<<16)|((*(sp+1)&0xff)<<8)|(*sp&0xff));
						if (c3 == MAC_DAKUTEN || c3 == MAC_HANDAKUTEN || c3 == MAC_BUNONPU || c3 == MAC_TANONPU) {
							u32l_info->zen_cnt += 4;
							u32l_mac_cnt += 8;
							sp += 4;
						}
					}
				}
			} else if (lch < 0x10000) {
				u32l_info->zen_cnt += 4;
			} else if (is_unibe_long(lch)) {
				u32l_info->zen_cnt += 4;
				is_u32l_surrogate = true;
			} else {
				u32l_info->bad_cnt += 4;
			}
		} /* while(sp < ep) */
		if (ub_surrog_cnt > 0) {
			is_ub_surrogate = true;
		}
		if (ul_surrog_cnt > 0) {
			is_ul_surrogate = true;
		}
		if (u8_surrog_cnt > 0) {
			is_u8_surrogate = true;
		}
		if (ub_mac_cnt > 0 || ul_mac_cnt > 0 || u8_mac_cnt > 0 || u32b_mac_cnt > 0 || u32l_mac_cnt > 0) {
			dcs->_utfmac_exists = true;
		}
		if (is_eof) {
			if (ubacnt_all > 0 || ulacnt_all > 0) {
			/* UTF16 の ASCII がある時 */
			} else if (is_all_ascii) {
			/* すべて ASCII だった時 */
				if ((ub_info->bad_cnt > 0
				 && ul_info->bad_cnt > 0)
				 || (ub_info->zen_cnt == 0 && ul_info->zen_cnt == 0)) {
				/* UTF16 の全角が無かった時 */
					FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_ASCII, is_keep);
				}
			}
			if (ub_info->bad_cnt > ul_info->bad_cnt) {
			/* UTF16BE の不正文字が多い時 */
				/* UTF16LE の全角を優先する */
				ub_info->zen_cnt -= uba_ulz_cnt;
				ub_info->zen_cnt -= ul_surrog_cnt;
			} else if (ul_info->bad_cnt > ub_info->bad_cnt) {
			/* UTF16LE の不正文字が多い時 */
				/* UTF16BE の全角を優先する */
				ul_info->zen_cnt -= ula_ubz_cnt;
				ul_info->zen_cnt -= ub_surrog_cnt;
			} else if (ub_surrog_cnt > ul_surrog_cnt) {
			/* UTF16BE のサロゲート文字が多い時 */
				ub_info->zen_cnt += ub_surrog_cnt;
			} else if (ul_surrog_cnt > ub_surrog_cnt) {
			/* UTF16LE のサロゲート文字が多い時 */
				ul_info->zen_cnt += ul_surrog_cnt;
			}
		}
		if (flag_btype == 0) {
		/* 該当なしの時 */
			FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_UNKNOWN, is_keep);
		}
		if (is_u8_surrogate) {
			type_surrogate |= DCS_ND_UTF8;
		}
		if (is_ub_surrogate) {
			type_surrogate |= DCS_ND_UTF16BE;
		}
		if (is_ul_surrogate) {
			type_surrogate |= DCS_ND_UTF16LE;
		}
		if (is_u32b_surrogate) {
			type_surrogate |= DCS_ND_UTF32BE;
		}
		if (is_u32l_surrogate) {
			type_surrogate |= DCS_ND_UTF32LE;
		}
		/* 不特定コード・タイプが一意の場合、コード・タイプを決定する */
		switch(flag_btype) {
		case DCS_ND_ASCII:
			FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_ASCII, is_keep);
		case DCS_ND_SJIS:
			FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_SJIS, is_keep);
		case DCS_ND_SJIS1:
			FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_SJIS1, is_keep);
		case DCS_ND_SJIS2:
			FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_SJIS2, is_keep);
		case DCS_ND_EUC:
			FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_EUC, is_keep);
		case DCS_ND_EUC1:
			FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_EUC1, is_keep);
		case DCS_ND_EUC2:
			FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_EUC2, is_keep);
		case DCS_ND_EUC3:
			FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_EUC3, is_keep);
		case DCS_ND_EUC4:
			FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_EUC4, is_keep);
		case DCS_ND_UTF16BE:
			dcs->_is_surrogate = is_ub_surrogate;
			FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_UTF16BE, is_keep);
		case DCS_ND_UTF16LE:
			dcs->_is_surrogate = is_ul_surrogate;
			FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_UTF16LE, is_keep);
		case DCS_ND_UTF8:
			dcs->_is_surrogate = is_u8_surrogate;
			FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_UTF8, is_keep);
		case DCS_ND_UTF32BE:
			dcs->_is_surrogate = is_u32b_surrogate;
			FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_UTF32BE, is_keep);
		case DCS_ND_UTF32LE:
			dcs->_is_surrogate = is_u32l_surrogate;
			FILL_BUFFER_AND_RETURN(dcs, ltype, atype, 0, DCS_UTF32LE, is_keep);
		default:
			break;
		}
		flag_btype_all |= flag_btype;
		if (!is_eof) {
		/* EOF ではない時 */
			/* 最大値を取得 */
			maxb = 0;
			for(i=0; i<sizeof(cinfo_sp)/sizeof(cinfo_sp[0]); i++) {
				if ((flag_btype & cinfo_sp[i]->nd_type) != 0) {
				/* 処理対象の時 */
					maxb = cinfo_sp[i]->blen;
					for(i++; i<sizeof(cinfo_sp)/sizeof(cinfo_sp[0]); i++) {
						if ((flag_btype & cinfo_sp[i]->nd_type) != 0) {
						/* 処理対象の時 */
							if (maxb < cinfo_sp[i]->blen) {
								maxb = cinfo_sp[i]->blen;
							}
						}
					}
				}
			}
			dcs->_buffer_info.offset = _MAX_OFFSET;
			dcs->_buffer_info.s_offset = _MAX_OFFSET - sjis_info->blen;
			dcs->_buffer_info.e_offset = _MAX_OFFSET - euc_info->blen;
			dcs->_buffer_info.u8_offset  = _MAX_OFFSET - u8_info->blen;
			dcs->_buffer_info.u16_offset = _MAX_OFFSET - ub_info->blen;
			dcs->_buffer_info.u32_offset = _MAX_OFFSET - u32b_info->blen;
			/* 残りを先頭へコピー */
			memcpy(dcs->_buffer_info.buf + _MAX_OFFSET - maxb, ep - maxb, maxb);
			sp = dcs->_buffer_info.buf + _MAX_OFFSET;
			/* 続きを読み込む */
			continue;
		}
		/* これより、全て表示指定と返却値が別れる */
		if (!is_all_ascii) {
			flag_btype_all &= ~DCS_ND_ASCII;
		}
		if (atype != NULL) {
		/* 全て表示指定ありの時 */
			*atype = flag_btype_all;
		}
		for(i=0; i<sizeof(code_info)/sizeof(code_info[0]); i++) {
			if ((flag_btype_all & code_info[i].nd_type) == 0
			 || code_info[i].zen_cnt == 0) {
				/* 無効データの件数を無効とする */
				code_info[i].zen_cnt = _MIN_VAL;
				code_info[i].bad_cnt = _MAX_VAL;
				flag_btype_all &= ~code_info[i].nd_type;
//			} else {
//				/* ソート第２キー：全角＋全角かなー１文字半角カナ */
//				code_info[i].key = code_info[i].zen_cnt
//						+ code_info[i].zen_kana_cnt;
			}
		}
		// shell sort
		CodeInfo *tmp;
		int si, sj, slen = sizeof(cinfo)/sizeof(cinfo[0]);
		int gap = slen / 2;
		// 大きなギャップから始めて、段階的に縮小
		for (gap = slen / 2; gap > 0; gap /= 2) {
			// このギャップサイズで「ギャップ付き」挿入ソートを実行
			for (si = gap; si < slen; si++) {
				// 現在の要素を正しく配置
				tmp = cinfo[si];
				sj = si;
				//  tmp より大きい要素をシフトしてスペースを確保
				while (sj >= gap && _scompare(cinfo[sj - gap], tmp) > 0) {
					cinfo[sj] = cinfo[sj - gap];
					sj -= gap;
				}
				// tmp を正しい位置に配置
				cinfo[sj] = tmp;
			}
		}
		normal_type = 0;
		clen = sizeof(cinfo)/sizeof(cinfo[0]);
		for(i=0; i<clen; i++) {
			if (cinfo[i]->type == stype) {
				/* 指示にマッチしたタイプの時 */
				CHECKLN_SETA_AND_RETURN(dcs, ltype, atype, type_surrogate, cinfo[i]->type);
			}
			if (cinfo[i]->zen_cnt > 0) {
				if (((normal_type & DCS_ND_SJIS_ALL) == 0 || (cinfo[i]->nd_type & DCS_ND_SJIS_ALL) == 0)
				 && ((normal_type & DCS_ND_EUC_ALL) == 0 || (cinfo[i]->nd_type & DCS_ND_EUC_ALL) == 0)) {
				// SJIS, EUC のコードが設定されていない時
					normal_type |= cinfo[i]->nd_type;
				} else {
				// 無効コードを詰める
					memmove(&cinfo[i], &cinfo[i + 1], sizeof(cinfo[0]) * (sizeof(cinfo)/sizeof(cinfo[0]) - (i + 1)));
					i--;
					clen--;
					continue;
				}
			}
			if (cinfo[i]->bad_cnt > 0) {
				/* 不正文字がある時 */
				/* 又は泣き別れの時 */
				break;
			}
		}
#if _DEFINE_BINARY_RANGE
		if (_BINARY_RANGE(cinfo[0]->bad_cnt, _source_size)) {
		/* バイナリー域の時 */
			SETA_AND_RETURN(dcs, atype, DCS_UNKNOWN);
		}
#endif
		if (_BAD_RANGE(cinfo[0]->bad_cnt, cinfo[1]->bad_cnt)) {
			CHECKLN_SETA_AND_RETURN(dcs, ltype, atype, type_surrogate,
					is_all_ascii? cinfo[0]->nd_type | DCS_ND_ASCII:
							cinfo[0]->zen_cnt == cinfo[1]->zen_cnt
							&& cinfo[0]->han_kana_cnt == cinfo[1]->han_kana_cnt
							&& (cinfo[0]->zen_kana_cnt == cinfo[1]->zen_kana_cnt
							|| cinfo[1]->bad_cnt == 0)?
									cinfo[0]->nd_type | cinfo[1]->nd_type: cinfo[0]->type);
		}
		if ((cinfo[0]->zen_cnt > cinfo[1]->zen_cnt
		  && cinfo[0]->bad_cnt <= cinfo[1]->bad_cnt
		  && cinfo[0]->zen_kana_cnt >= cinfo[1]->zen_kana_cnt)
			||(cinfo[0]->zen_cnt >= cinfo[1]->zen_cnt
			&& cinfo[0]->bad_cnt < cinfo[1]->bad_cnt
			&& cinfo[0]->zen_kana_cnt >= cinfo[1]->zen_kana_cnt)
			||(cinfo[0]->zen_cnt >= cinfo[1]->zen_cnt
			&& cinfo[0]->bad_cnt <= cinfo[1]->bad_cnt
			&& cinfo[0]->zen_kana_cnt > cinfo[1]->zen_kana_cnt)) {
		/* 全角、正文字、全角カナが少なくなく、何れかが多い時 */
			CHECKLN_SETA_AND_RETURN(dcs, ltype, atype, type_surrogate,
					is_all_ascii? cinfo[0]->nd_type | DCS_ND_ASCII:
							cinfo[0]->zen_cnt == cinfo[1]->zen_cnt
							&& cinfo[0]->han_kana_cnt == cinfo[1]->han_kana_cnt
							&& (cinfo[0]->zen_kana_cnt == cinfo[1]->zen_kana_cnt
							|| cinfo[1]->bad_cnt == 0)?
									cinfo[0]->nd_type | cinfo[1]->nd_type: cinfo[0]->type);
		}
		// 以上で flag_btype_all 設定完了のため normal_type に反映
		normal_type &= flag_btype_all;
		if ((cinfo[0]->type == DCS_UTF16BE || cinfo[0]->type == DCS_UTF16LE)
		 && (flag_btype_all & DCS_ND_ASCII) != 0
		 && (flag_btype_all & (DCS_ND_UTF16BE | DCS_ND_UTF16LE)) != 0) {
		/* ASCII と Unicode の時 */
			CHECKLN_SETA_AND_RETURN(dcs, ltype, atype, type_surrogate,
					(flag_btype_all & (DCS_ND_ASCII | DCS_ND_UTF16BE | DCS_ND_UTF16LE | DCS_ND_UTF32BE | DCS_ND_UTF32LE)));
		}
		if (cinfo[0]->bad_cnt <= cinfo[1]->bad_cnt
		 && cinfo[0]->tan_han_kana_cnt <= cinfo[1]->tan_han_kana_cnt) {
		// 不正文字が多くない時
		// かつ、単一半角カナが多くない時
			CHECKLN_SETA_AND_RETURN(dcs, ltype, atype, type_surrogate, cinfo[0]->type);
		}
		if (cinfo[0]->zen_kana_cnt > cinfo[1]->zen_kana_cnt
		 || (cinfo[0]->tan_han_kana_cnt == 0 && cinfo[1]->tan_han_kana_cnt > 0)) {
		/* 全角仮名が多い時 */
		/* 又は、単一半角カナが無く、かつ後続に単一半角カナがある時 */
			stype = (cinfo[0]->type == stype? cinfo[0]->type:
					flag_btype_all == cinfo[0]->nd_type
					 || normal_type == cinfo[0]->nd_type? cinfo[0]->type: normal_type);
			CHECKLN_SETA_AND_RETURN(dcs, ltype, atype, type_surrogate, stype);
		}
#if _KEEP_ASCII
		/* ASCII を保つ */
		dtype = (flag_btype_all & DCS_ND_ASCII);
#else
		dtype = 0;
#endif
		for(i=0;; i++) {
			if (cinfo[i]->type == stype) {
			/* 指示にマッチしたタイプの時 */
				CHECKLN_SETA_AND_RETURN(dcs, ltype, atype, type_surrogate, stype);
			}
			switch(cinfo[i]->type) {
			case DCS_JIS:
				dtype |= DCS_ND_JIS;
				break;
			case DCS_JIS1:
				dtype |= DCS_ND_JIS1;
				break;
			case DCS_JIS2:
				dtype |= DCS_ND_JIS2;
				break;
			case DCS_JIS3:
				dtype |= DCS_ND_JIS3;
				break;
			case DCS_SJIS:
				dtype |= DCS_ND_SJIS;
				break;
			case DCS_SJIS1:
				dtype |= DCS_ND_SJIS1;
				break;
			case DCS_SJIS2:
				dtype |= DCS_ND_SJIS2;
				break;
			case DCS_EUC:
				dtype |= DCS_ND_EUC;
				break;
			case DCS_EUC1:
				dtype |= DCS_ND_EUC1;
				break;
			case DCS_EUC2:
				dtype |= DCS_ND_EUC2;
				break;
			case DCS_EUC3:
				dtype |= DCS_ND_EUC3;
				break;
			case DCS_EUC4:
				dtype |= DCS_ND_EUC4;
				break;
			case DCS_UTF16BE:
				dtype |= DCS_ND_UTF16BE;
				break;
			case DCS_UTF16LE:
				dtype |= DCS_ND_UTF16LE;
				break;
			case DCS_UTF8:
				dtype |= DCS_ND_UTF8;
				break;
			case DCS_UTF32BE:
				dtype |= DCS_ND_UTF32BE;
				break;
			case DCS_UTF32LE:
				dtype |= DCS_ND_UTF32LE;
				break;
			}
			if (i + 1 >= clen) {
			/* 最後のデータの時 */
				break;
			}
			if (cinfo[i]->zen_cnt > cinfo[i+1]->zen_cnt) {
			/* 全角が多い時 */
				break;
			}
			if (cinfo[i]->bad_cnt < cinfo[i+1]->bad_cnt) {
			/* 不正文字が少ない時 */
				break;
			}
		}
		if (IS_SJIS(stype)) {
		// SJIS の指示がある時
			if ((dtype & DCS_ND_SJIS_ALL) != 0) {
			// SJIS の可能性がある時
				switch(dtype & DCS_ND_SJIS_ALL) {
				case DCS_ND_SJIS:
					dtype = DCS_SJIS;
					break;
				case DCS_ND_SJIS1:
					dtype = DCS_SJIS1;
					break;
				case DCS_ND_SJIS2:
					dtype = DCS_SJIS2;
					break;
				}
			}
		} else if (IS_EUC(stype)) {
		// EUC の指示がある時
			if ((dtype & DCS_ND_EUC_ALL) != 0) {
			// EUC の可能性がある時
				switch(dtype & DCS_ND_EUC_ALL) {
				case DCS_ND_EUC:
					dtype = DCS_EUC;
					break;
				case DCS_ND_EUC1:
					dtype = DCS_EUC1;
					break;
				case DCS_ND_EUC2:
					dtype = DCS_EUC2;
					break;
				case DCS_ND_EUC3:
					dtype = DCS_EUC3;
					break;
				case DCS_ND_EUC4:
					dtype = DCS_EUC4;
					break;
				}
			}
		}
		CHECKLN_SETA_AND_RETURN(dcs, ltype, atype, 0, dtype);
	} /* for(;;) */
} /* _dcs_judges */

#define _LN_CHECK_AND_RERURN(ltype) { \
	if ((ltype) == LTYPE_ALL) { \
		return LTYPE_ALL; \
	} }

int _check_ln(Dcs *dcs, int stype) {
	int c, c2, ltype = 0;
	const unsigned char *sp, *ep;
	ll_list *tlist, *clist;
	BufferInfo *tbinfo, *binfo;

	// 先頭からすべての改行が見つかるかEOFになるまで
	clist = ll_first((ll_list *)dcs->_block);
	binfo = ll_get(clist);
	sp = binfo->buf + binfo->offset;;
	ep = binfo->buf + binfo->length;

	if (stype == DCS_UTF16BE || stype == DCS_UTF16BE_BOM) {
	// UTf-16BE の時
		for(;;) {
			/* 残りが最大長に満たない時、次の入力バッファに移動 */
			check_ll_buf(dcs, sp, ep, 4, clist, tlist, binfo, tbinfo);
			if (sp + 1 >= ep) {
				return ltype;
			}
			c = *sp++;
			c2 = *sp++;
			if (c == 0) {
				if (c2 == '\n') {
					_LN_CHECK_AND_RERURN(ltype |= LTYPE_LF);
				} else if (c2 == '\r') {
					if (sp + 1 >= ep) {
						ltype |= LTYPE_CR;
						return ltype;
					}
					c = *sp++;
					c2 = *sp++;
					if (c == 0 && c2 == '\n') {
						_LN_CHECK_AND_RERURN(ltype |= LTYPE_CRLF);
					} else {
						_LN_CHECK_AND_RERURN(ltype |= LTYPE_CR);
						sp -= 2;
					}
				}
			}
		}
	} else if (stype == DCS_UTF16LE || stype == DCS_UTF16LE_BOM) {
	// UTf-16LE の時
		for(;;) {
			/* 残りが最大長に満たない時、次の入力バッファに移動 */
			check_ll_buf(dcs, sp, ep, 4, clist, tlist, binfo, tbinfo);
			if (sp + 1 >= ep) {
				return ltype;
			}
			c = *sp++;
			c2 = *sp++;
			if (c2 == 0) {
				if (c == '\n') {
					_LN_CHECK_AND_RERURN(ltype |= LTYPE_LF);
				} else if (c == '\r') {
					if (sp + 1 >= ep) {
						ltype |= LTYPE_CR;
						return ltype;
					}
					c = *sp++;
					c2 = *sp++;
					if (c == '\n' && c2 == 0) {
						_LN_CHECK_AND_RERURN(ltype |= LTYPE_CRLF);
					} else {
						_LN_CHECK_AND_RERURN(ltype |= LTYPE_CR);
						sp -= 2;
					}
				}
			}
		}
	} else if (stype == DCS_UTF32BE || stype == DCS_UTF32BE_BOM) {
	// UTf-32BE の時
		for(;;) {
			/* 残りが最大長に満たない時、次の入力バッファに移動 */
			check_ll_buf(dcs, sp, ep, 8, clist, tlist, binfo, tbinfo);
			if (sp + 3 >= ep) {
				return ltype;
			}
			c = *sp++;
			c += *sp++;
			c += *sp++;
			c2 = *sp++;
			if (c == 0) {
				if (c2 == '\n') {
					_LN_CHECK_AND_RERURN(ltype |= LTYPE_LF);
				} else if (c2 == '\r') {
					if (sp + 3 >= ep) {
						ltype |= LTYPE_CR;
						return ltype;
					}
					c = *sp++;
					c += *sp++;
					c += *sp++;
					c2 = *sp++;
					if (c == 0 && c2 == '\n') {
						_LN_CHECK_AND_RERURN(ltype |= LTYPE_CRLF);
					} else {
						_LN_CHECK_AND_RERURN(ltype |= LTYPE_CR);
						sp -= 4;
					}
				}
			}
		}
	} else if (stype == DCS_UTF32LE || stype == DCS_UTF32LE_BOM) {
	// UTf-32LE の時
		for(;;) {
			/* 残りが最大長に満たない時、次の入力バッファに移動 */
			check_ll_buf(dcs, sp, ep, 8, clist, tlist, binfo, tbinfo);
			if (sp + 3 >= ep) {
				return ltype;
			}
			c = *sp++;
			c2 = *sp++;
			c2 += *sp++;
			c2 += *sp++;
			if (c2 == 0) {
				if (c == '\n') {
					_LN_CHECK_AND_RERURN(ltype |= LTYPE_LF);
				} else if (c == '\r') {
					if (sp + 3 >= ep) {
						ltype |= LTYPE_CR;
						return ltype;
					}
					c = *sp++;
					c2 = *sp++;
					c2 += *sp++;
					c2 += *sp++;
					if (c == '\n' && c2 == 0) {
						_LN_CHECK_AND_RERURN(ltype |= LTYPE_CRLF);
					} else {
						_LN_CHECK_AND_RERURN(ltype |= LTYPE_CR);
						sp -= 4;
					}
				}
			}
		}
	} else {
	// 改行文字が１バイトの時
		if (!DCS_CONFIRMED_CODE_TYPE(stype)) {
		/* 未確定コードがあった時 */
			return DCS_ECTYPE;
		}
		for(;;) {
			/* 残りが最大長に満たない時、次の入力バッファに移動 */
			check_ll_buf(dcs, sp, ep, 2, clist, tlist, binfo, tbinfo);
			if (sp >= ep) {
				return ltype;
			}
			c = *sp++;
			if (c == '\n') {
				_LN_CHECK_AND_RERURN(ltype |= LTYPE_LF);
			} else if (c == '\r') {
				if (sp >= ep) {
					return ltype |= LTYPE_CR;
				}
				c = *sp++;
				if (c == '\n') {
					_LN_CHECK_AND_RERURN(ltype |= LTYPE_CRLF);
				} else {
					sp--;
					_LN_CHECK_AND_RERURN(ltype |= LTYPE_CR);
				}
			}
		}
	}
} /* _check_ln */

/* 共通コード変換 */
int _dcs_convert(Dcs *dcs, int stype, int dtype, bool is_forced) {
	int sts = DCS_SUCCESS;
	int utf_type, estype;
	int cmp;
	bool is_edit;

	estype = _dcs_judges(dcs, NULL, NULL, stype, is_forced, true);

	if (dcs->_observer != NULL
	 && dcs->_observer(estype, dcs->_user_info) != DCS_OBS_CONTINUE) {
	/* 処理中止の時 */
		return estype;
	}
	if (!DCS_CONFIRMED_CODE_TYPE(estype)) {
	/* 入力タイプが確定できなかった時 */
		return estype;
	}
	if (estype == DCS_ASCII) {
		/* SJIS とみなす */
		estype = DCS_SJIS;
	}
	if (dtype == DCS_ASCII) {
		/* SJIS とみなす */
		dtype = DCS_SJIS;
	}
	/* UTF-8-MAC 編集
	 * 又は半角カナ全角変換
	 * 又は改行編集を編集とみなす
	 */
	is_edit = ((dcs_is_utfmac_to(dcs) && dcs_utfmac_exists(dcs))
			 || (dcs_is_hankana_to(dcs)
				|| get_ltype(dcs) != LTYPE_NON));
	if (!is_edit && estype == dtype) {
	/* 編集ではない時
	 * かつ入出力タイプが等しい時 */
		sts = _dcs_copy(dcs, estype);
	} else {
		cmp = dcs_cmp_bom(estype, dtype);
		switch(estype) {
		case DCS_JIS:
			set_jis2022jp_to(dcs);
			break;
		case DCS_JIS1:
			set_jis5022101_to(dcs);
			break;
		case DCS_JIS2:
			set_jis2022jp200401_to(dcs);
			break;
		case DCS_JIS3:
			set_jis2022jp301_to(dcs);
			break;
		case DCS_SJIS:
			set_sjis_to(dcs);
			break;
		case DCS_SJIS1:
			set_sjis932_to(dcs);
			break;
		case DCS_SJIS2:
			set_sjis2004_to(dcs);
			break;
		case DCS_EUC:
			set_euccommon_to(dcs);
			break;
		case DCS_EUC1:
			set_euc20932_to(dcs);
			break;
		case DCS_EUC2:
			set_euc51932_to(dcs);
			break;
		case DCS_EUC3:
			set_eucms_to(dcs);
			break;
		case DCS_EUC4:
			set_euc2004_to(dcs);
			break;
		}
		switch(dtype) {
		case DCS_JIS:
			if (dcs->_is_surrogate) {
				set_to_jis2022jp3(dcs);
			} else {
				set_to_jis2022jp(dcs);
			}
			break;
		case DCS_JIS1:
			set_to_jis50221(dcs);
			break;
		case DCS_JIS2:
			set_to_jis2022jp2004(dcs);
			break;
		case DCS_JIS3:
			set_to_jis2022jp3(dcs);
			break;
		case DCS_SJIS:
			set_to_sjis(dcs);
			break;
		case DCS_SJIS1:
			set_to_sjis932(dcs);
			break;
		case DCS_SJIS2:
			set_to_sjis2004(dcs);
			break;
		case DCS_EUC:
			set_to_euccommon(dcs);
			break;
		case DCS_EUC1:
			set_to_euc20932(dcs);
			break;
		case DCS_EUC2:
			set_to_euc51932(dcs);
			break;
		case DCS_EUC3:
			set_to_eucms(dcs);
			break;
		case DCS_EUC4:
			set_to_euc2004(dcs);
			break;
		}
		utf_type = _UTYPE_FROM_OTHER;
		switch(estype) {
		case DCS_JIS:
		case DCS_JIS1:
		case DCS_JIS2:
		case DCS_JIS3:
			switch(dtype) {
			case DCS_JIS:
			case DCS_JIS1:
			case DCS_JIS2:
			case DCS_JIS3:
				sts = _jisto_jis(dcs);
				break;
			case DCS_SJIS:
			case DCS_SJIS1:
			case DCS_SJIS2:
				sts = _jisto_sjis(dcs);
				break;
			case DCS_EUC:
			case DCS_EUC1:
			case DCS_EUC2:
			case DCS_EUC3:
			case DCS_EUC4:
				sts = _jisto_euc(dcs);
				break;
			case DCS_UTF8:
			case DCS_UTF8_BOM:
				sts = _jisto_utf8(dcs, dtype==DCS_UTF8_BOM);
				break;
			case DCS_UTF16BE:
			case DCS_UTF16BE_BOM:
				sts = _jisto_unibe(dcs, dtype==DCS_UTF16BE_BOM, _UTYPE_FROM_OTHER_TO_OTHER);
				break;
			case DCS_UTF16LE:
			case DCS_UTF16LE_BOM:
				sts = _jisto_unile(dcs, dtype==DCS_UTF16LE_BOM, _UTYPE_FROM_OTHER_TO_OTHER);
				break;
			case DCS_UTF32BE:
			case DCS_UTF32BE_BOM:
				sts = _jisto_unibe(dcs, dtype==DCS_UTF32BE_BOM, _UTYPE_FROM_OTHER_TO_UTF32);
				break;
			case DCS_UTF32LE:
			case DCS_UTF32LE_BOM:
				sts = _jisto_unile(dcs, dtype==DCS_UTF32LE_BOM, _UTYPE_FROM_OTHER_TO_UTF32);
				break;
			}
			break;
		case DCS_SJIS:
		case DCS_SJIS1:
		case DCS_SJIS2:
			switch(dtype) {
			case DCS_JIS:
			case DCS_JIS1:
			case DCS_JIS2:
			case DCS_JIS3:
				sts = _sjisto_jis(dcs);
				break;
			case DCS_SJIS:
			case DCS_SJIS1:
			case DCS_SJIS2:
				sts = _sjisto_sjis(dcs);
				break;
			case DCS_EUC:
			case DCS_EUC1:
			case DCS_EUC2:
			case DCS_EUC3:
			case DCS_EUC4:
				sts = _sjisto_euc(dcs);
				break;
			case DCS_UTF8:
			case DCS_UTF8_BOM:
				sts = _sjisto_utf8(dcs, dtype==DCS_UTF8_BOM);
				break;
			case DCS_UTF16BE:
			case DCS_UTF16BE_BOM:
				sts = _sjisto_unibe(dcs, dtype==DCS_UTF16BE_BOM, _UTYPE_FROM_OTHER_TO_OTHER);
				break;
			case DCS_UTF16LE:
			case DCS_UTF16LE_BOM:
				sts = _sjisto_unile(dcs, dtype==DCS_UTF16LE_BOM, _UTYPE_FROM_OTHER_TO_OTHER);
				break;
			case DCS_UTF32BE:
			case DCS_UTF32BE_BOM:
				sts = _sjisto_unibe(dcs, dtype==DCS_UTF32BE_BOM, _UTYPE_FROM_OTHER_TO_UTF32);
				break;
			case DCS_UTF32LE:
			case DCS_UTF32LE_BOM:
				sts = _sjisto_unile(dcs, dtype==DCS_UTF32LE_BOM, _UTYPE_FROM_OTHER_TO_UTF32);
				break;
			}
			break;
		case DCS_EUC:
		case DCS_EUC1:
		case DCS_EUC2:
		case DCS_EUC3:
		case DCS_EUC4:
			switch(dtype) {
			case DCS_JIS:
			case DCS_JIS1:
			case DCS_JIS2:
			case DCS_JIS3:
				sts = _eucto_jis(dcs);
				break;
			case DCS_SJIS:
			case DCS_SJIS1:
			case DCS_SJIS2:
				sts = _eucto_sjis(dcs);
				break;
			case DCS_EUC:
			case DCS_EUC1:
			case DCS_EUC2:
			case DCS_EUC3:
			case DCS_EUC4:
				sts = _eucto_euc(dcs);
				break;
			case DCS_UTF8:
			case DCS_UTF8_BOM:
				sts = _eucto_utf8(dcs, dtype==DCS_UTF8_BOM);
				break;
			case DCS_UTF16BE:
			case DCS_UTF16BE_BOM:
				sts = _eucto_unibe(dcs, dtype==DCS_UTF16BE_BOM, _UTYPE_FROM_OTHER_TO_OTHER);
				break;
			case DCS_UTF16LE:
			case DCS_UTF16LE_BOM:
				sts = _eucto_unile(dcs, dtype==DCS_UTF16LE_BOM, _UTYPE_FROM_OTHER_TO_OTHER);
				break;
			case DCS_UTF32BE:
			case DCS_UTF32BE_BOM:
				sts = _eucto_unibe(dcs, dtype==DCS_UTF32BE_BOM, _UTYPE_FROM_OTHER_TO_UTF32);
				break;
			case DCS_UTF32LE:
			case DCS_UTF32LE_BOM:
				sts = _eucto_unile(dcs, dtype==DCS_UTF32LE_BOM, _UTYPE_FROM_OTHER_TO_UTF32);
				break;
			}
			break;
		case DCS_UTF8:
		case DCS_UTF8_BOM:
			switch(dtype) {
			case DCS_JIS:
			case DCS_JIS1:
			case DCS_JIS2:
			case DCS_JIS3:
				sts = _utf8to_jis(dcs);
				break;
			case DCS_SJIS:
			case DCS_SJIS1:
			case DCS_SJIS2:
				sts = _utf8to_sjis(dcs);
				break;
			case DCS_EUC:
			case DCS_EUC1:
			case DCS_EUC2:
			case DCS_EUC3:
			case DCS_EUC4:
				sts = _utf8to_euc(dcs);
				break;
			case DCS_UTF8:
			case DCS_UTF8_BOM:
				if (is_edit) {
					sts = _utf8to_utf8(dcs, cmp == CBOM_ADD);
				} else if (cmp == CBOM_ADD) {
					sts = _add_bom(dcs, dtype);
				} else {
					sts = _delete_bom(dcs, dtype);
				}
				break;
			case DCS_UTF16BE:
			case DCS_UTF16BE_BOM:
				sts = _utf8to_unibe(dcs, dtype==DCS_UTF16BE_BOM, _UTYPE_FROM_OTHER_TO_OTHER);
				break;
			case DCS_UTF16LE:
			case DCS_UTF16LE_BOM:
				sts = _utf8to_unile(dcs, dtype==DCS_UTF16LE_BOM, _UTYPE_FROM_OTHER_TO_OTHER);
				break;
			case DCS_UTF32BE:
			case DCS_UTF32BE_BOM:
				sts = _utf8to_unibe(dcs, dtype==DCS_UTF32BE_BOM, _UTYPE_FROM_OTHER_TO_UTF32);
				break;
			case DCS_UTF32LE:
			case DCS_UTF32LE_BOM:
				sts = _utf8to_unile(dcs, dtype==DCS_UTF32LE_BOM, _UTYPE_FROM_OTHER_TO_UTF32);
				break;
			}
			break;
		case DCS_UTF32BE:
		case DCS_UTF32BE_BOM:
			utf_type = _UTYPE_FROM_UTF32;
			/* no break */
		case DCS_UTF16BE:
		case DCS_UTF16BE_BOM:
			switch(dtype) {
			case DCS_JIS:
			case DCS_JIS1:
			case DCS_JIS2:
			case DCS_JIS3:
				sts = _unibeto_jis(dcs, utf_type | _UTYPE_TO_OTHER);
				break;
			case DCS_SJIS:
			case DCS_SJIS1:
			case DCS_SJIS2:
				sts = _unibeto_sjis(dcs, utf_type | _UTYPE_TO_OTHER);
				break;
			case DCS_EUC:
			case DCS_EUC1:
			case DCS_EUC2:
			case DCS_EUC3:
			case DCS_EUC4:
				sts = _unibeto_euc(dcs, utf_type | _UTYPE_TO_OTHER);
				break;
			case DCS_UTF8:
			case DCS_UTF8_BOM:
				sts = _unibeto_utf8(dcs, dtype==DCS_UTF8_BOM, utf_type | _UTYPE_TO_OTHER);
				break;
			case DCS_UTF16BE:
			case DCS_UTF16BE_BOM:
				if (is_edit || estype == DCS_UTF32BE || estype == DCS_UTF32BE_BOM) {
					sts = _unibeto_unibe(dcs, dtype==DCS_UTF16BE_BOM, utf_type | _UTYPE_TO_OTHER);
				} else if (cmp == CBOM_ADD) {
					sts = _add_bom(dcs, dtype);
				} else {
					sts = _delete_bom(dcs, dtype);
				}
				break;
			case DCS_UTF16LE:
			case DCS_UTF16LE_BOM:
				sts = _unibeto_unile(dcs, dtype==DCS_UTF16LE_BOM, utf_type | _UTYPE_TO_OTHER);
				break;
			case DCS_UTF32BE:
			case DCS_UTF32BE_BOM:
				if (is_edit || estype == DCS_UTF16BE || estype == DCS_UTF16BE_BOM) {
					sts = _unibeto_unibe(dcs, dtype==DCS_UTF32BE_BOM, utf_type | _UTYPE_TO_UTF32);
				} else if (cmp == CBOM_ADD) {
					sts = _add_bom(dcs, dtype);
				} else {
					sts = _delete_bom(dcs, dtype);
				}
				break;
			case DCS_UTF32LE:
			case DCS_UTF32LE_BOM:
				sts = _unibeto_unile(dcs, dtype==DCS_UTF32LE_BOM, utf_type | _UTYPE_TO_UTF32);
				break;
			}
			break;
		case DCS_UTF32LE:
		case DCS_UTF32LE_BOM:
			utf_type = _UTYPE_FROM_UTF32;
			/* no break */
		case DCS_UTF16LE:
		case DCS_UTF16LE_BOM:
			switch(dtype) {
			case DCS_JIS:
			case DCS_JIS1:
			case DCS_JIS2:
			case DCS_JIS3:
				sts = _unileto_jis(dcs, utf_type | _UTYPE_TO_OTHER);
				break;
			case DCS_SJIS:
			case DCS_SJIS1:
			case DCS_SJIS2:
				sts = _unileto_sjis(dcs, utf_type | _UTYPE_TO_OTHER);
				break;
			case DCS_EUC:
			case DCS_EUC1:
			case DCS_EUC2:
			case DCS_EUC3:
			case DCS_EUC4:
				sts = _unileto_euc(dcs, utf_type | _UTYPE_TO_OTHER);
				break;
			case DCS_UTF8:
			case DCS_UTF8_BOM:
				sts = _unileto_utf8(dcs, dtype==DCS_UTF8_BOM, utf_type | _UTYPE_TO_OTHER);
				break;
			case DCS_UTF16BE:
			case DCS_UTF16BE_BOM:
				sts = _unileto_unibe(dcs, dtype==DCS_UTF16BE_BOM, utf_type | _UTYPE_TO_OTHER);
				break;
			case DCS_UTF16LE:
			case DCS_UTF16LE_BOM:
				if (is_edit || estype == DCS_UTF32LE || estype == DCS_UTF32LE_BOM) {
					sts = _unileto_unile(dcs, dtype==DCS_UTF16LE_BOM, utf_type | _UTYPE_TO_OTHER);
				} else if (cmp == CBOM_ADD) {
					sts = _add_bom(dcs, dtype);
				} else {
					sts = _delete_bom(dcs, dtype);
				}
				break;
			case DCS_UTF32BE:
			case DCS_UTF32BE_BOM:
				sts = _unileto_unibe(dcs, dtype==DCS_UTF32BE_BOM, utf_type | _UTYPE_TO_UTF32);
				break;
			case DCS_UTF32LE:
			case DCS_UTF32LE_BOM:
				if (is_edit || estype == DCS_UTF16LE || estype == DCS_UTF16LE_BOM) {
					sts = _unileto_unile(dcs, dtype==DCS_UTF32LE_BOM, utf_type | _UTYPE_TO_UTF32);
				} else if (cmp == CBOM_ADD) {
					sts = _add_bom(dcs, dtype);
				} else {
					sts = _delete_bom(dcs, dtype);
				}
				break;
			}
			break;
		}
	}
	return sts;
} /* _dcs_convert */

/* バッファー確保 */
int _fill_buffer(Dcs *dcs, int stype) {
	int c, c2, c3, c4;
	unsigned short ch;
	uint64_t lch, lch2;
	unsigned char *sp, *ep;
	bool is_eof = false;
	bool is_utfmac_d;
	ll_list *tlist, *clist = dcs->_block;
	JniJudgedStreamInfo * jni_info = (JniJudgedStreamInfo *)dcs->_jni_info;
	size_t rlen, alen = (_is_type_jni_stream(dcs)?
			(*jni_info->jenv)->GetArrayLength(jni_info->jenv, jni_info->jbuf): 0);

	for(;;) {
		memset(&dcs->_buffer_info, 0, sizeof(dcs->_buffer_info));
		dcs->_buffer_info.offset = _MAX_OFFSET;
		sp = dcs->_buffer_info.buf + _MAX_OFFSET;
		ep = dcs->_buffer_info.buf + sizeof(dcs->_buffer_info.buf) - _MAX_OFFSET;
		get_one_buffer(dcs, sp, ep, alen, rlen);
		ep = sp;
		dcs->_buffer_info.length = ep - dcs->_buffer_info.buf;
		if ((clist = _add_buffer(clist, &dcs->_buffer_info, sizeof(dcs->_buffer_info))) == NULL) {
			ll_clear((ll_list *)dcs->_block);
			return DCS_EALLOC;
		}
		if (is_eof) {
			if ((!dcs->_is_surrogate || !dcs->_utfmac_exists)
			 && DCS_UTF16BE <= stype && stype <= DCS_UTF32LE_BOM) {
			/* サロゲートの判定が必要な時 */
				BufferInfo *binfo, *tbinfo;

				clist = ll_first((ll_list *)dcs->_block);
				binfo = ll_get(clist);
				sp = binfo->buf + binfo->offset;;
				ep = binfo->buf + binfo->length;
				if (stype == DCS_UTF16BE || stype == DCS_UTF16BE_BOM) {
				/* UTF16BE の時 */
					for(;;) {
						/* 残りが最大長に満たない時、次の入力バッファに移動 */
						check_ll_buf(dcs, sp, ep, 4, clist, tlist, binfo, tbinfo);
						if (sp >= ep) {
							break;
						}
						ch = unibe_char_to_short(sp);
						if (is_unibe_surrogate_high(ch)) {
							dcs->_is_surrogate = true;
							if (dcs->_utfmac_exists) {
								goto loop_end;
							}
						} else {
							if (is_unibe_mac(dcs, ch, CSB_MAC_D | CSB_MAC_H | CSB_MAC_BO | CSB_MAC_TO)) {
							/* UTF8-MAC 濁点、半濁点の時 */
								if (sp + 4 <= ep) {
									c3 = (((*(sp+2)&0xff)<<8)|(*(sp+3)&0xff));
									if (c3 == MAC_DAKUTEN || c3 == MAC_HANDAKUTEN || c3 == MAC_BUNONPU || c3 == MAC_TANONPU) {
										dcs->_utfmac_exists = true;
										if (dcs->_is_surrogate) {
											goto loop_end;
										}
										sp += 2;
									}
								}
							}
						}
						sp += 2;
					}
				} else if (stype == DCS_UTF16LE || stype == DCS_UTF16LE_BOM) {
				/* UTF16LE の時 */
					for(;;) {
						/* 残りが最大長に満たない時、次の入力バッファに移動 */
						check_ll_buf(dcs, sp, ep, 4, clist, tlist, binfo, tbinfo);
						if (sp >= ep) {
							break;
						}
						ch = unile_to_unibe_char(sp);
						if (is_unibe_surrogate_high(ch)) {
							dcs->_is_surrogate = true;
							if (dcs->_utfmac_exists) {
								goto loop_end;
							}
						} else {
							if (is_unibe_mac(dcs, ch, CSB_MAC_D | CSB_MAC_H | CSB_MAC_BO | CSB_MAC_TO)) {
							/* UTF8-MAC 濁点、半濁点の時 */
								if (sp + 4 <= ep) {
									c3 = (((*(sp+3)&0xff)<<8)|(*(sp+2)&0xff));
									if (c3 == MAC_DAKUTEN || c3 == MAC_HANDAKUTEN || c3 == MAC_BUNONPU || c3 == MAC_TANONPU) {
										dcs->_utfmac_exists = true;
										if (dcs->_is_surrogate) {
											goto loop_end;
										}
										sp += 2;
									}
								}
							}
						}
						sp += 2;
					}
				} else if (stype == DCS_UTF8 || stype == DCS_UTF8_BOM) {
				/* UTF8 の時 */
					is_utfmac_d = false;
					for(;;) {
						/* 残りが最大長に満たない時、次の入力バッファに移動 */
						check_ll_buf(dcs, sp, ep, 4, clist, tlist, binfo, tbinfo);
						if (sp >= ep) {
							break;
						}
						c = *sp++;
						if ((c & 0x80) == 0 || c == _UYEN) {
						/* 0xxxxxxx: １バイト文字 */
						} else if (sp >= ep) {
							return DCS_EUEXPEOD;
						} else {
							c2 = *sp++;
							if ((c & 0xe0) == 0xc0 && (c2 & 0xc0) == 0x80) {
							/* 110xxxxx 10xxxxxx: ２バイト文字 */
								ch = ((c&0x1f)<<6|(c2&0x3f));
								if (is_unibe_short(dcs, ch)) {
									if (is_unibe_mac(dcs, ch, CSB_MAC_BO | CSB_MAC_TO)) {
									/* UTF8-MAC の時 */
										is_utfmac_d = true;
									} else if (is_utfmac_d) {
										if (ch == MAC_DAKUTEN || ch == MAC_TANONPU) {
											dcs->_utfmac_exists = true;
											if (dcs->_is_surrogate) {
												goto loop_end;
											}
										}
										is_utfmac_d = false;
									}
								}
							} else if (sp >= ep) {
								return DCS_EUEXPEOD;
							} else {
								c3 = *sp++;
								if ((c & 0xf0) == 0xe0 && (c2 & 0xc0) == 0x80 && (c3 & 0xc0) == 0x80) {
								/* 1110xxxx 10xxxxxx 10xxxxxx: ３バイト文字 */
									ch = ((c&0x0f)<<12|(c2&0x3f)<<6|(c3&0x3f));
									if (is_unibe_short(dcs, ch)) {
										if (is_unibe_zenkana(ch)) {
										/* 全角仮名の時 */
											if (is_unibe_mac(dcs, ch, CSB_MAC_D | CSB_MAC_H)) {
											/* UTF8-MAC の時 */
												is_utfmac_d = true;
											} else if (is_utfmac_d) {
												if (ch == MAC_DAKUTEN || ch == MAC_HANDAKUTEN) {
													dcs->_utfmac_exists = true;
													if (dcs->_is_surrogate) {
														goto loop_end;
													}
												}
												is_utfmac_d = false;
											}
										}
									}
								} else if (sp >= ep) {
									return DCS_EUEXPEOD;
								} else {
									c4 = *sp++;
									/* 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx: ４バイト文字 */
									if (((c) & 0xf8) == 0xf0 && ((c2) & 0xc0) == 0x80 && ((c3) & 0xc0) == 0x80 && ((c4) & 0xc0) == 0x80) {
										lch = (((c & 07) << 18)|((c2 & 0x3f) << 12)|((c3 & 0x3f) << 6)|(c4 & 0x3f));
										lch -= 0x10000;
										ch = ((lch >> 10)|0xd800);
										if (is_unibe_surrogate_high(ch)) {
											dcs->_is_surrogate = true;
											if (dcs->_utfmac_exists) {
												goto loop_end;
											}
										}
									}
								}
							}
						}
					}
				} else if (stype == DCS_UTF32BE || stype == DCS_UTF32BE_BOM) {
				/* UTF32BE の時 */
					for(;;) {
						/* 残りが最大長に満たない時、次の入力バッファに移動 */
						check_ll_buf(dcs, sp, ep, 4, clist, tlist, binfo, tbinfo);
						if (sp >= ep) {
							break;
						}
						lch = unibe_char_to_long(sp);
						if ((lch & 0xffff0000) != 0) {
							dcs->_is_surrogate = true;
							if (dcs->_utfmac_exists) {
								goto loop_end;
							}
						} else {
							if (is_unibe_mac(dcs, lch, CSB_MAC_D | CSB_MAC_H | CSB_MAC_BO | CSB_MAC_TO)) {
							/* UTF8-MAC 濁点、半濁点の時 */
								if (sp + 8 <= ep) {
									lch2 = unibe_char_to_long(sp+4);
									if (lch2 == MAC_DAKUTEN || lch2 == MAC_HANDAKUTEN || lch2 == MAC_BUNONPU || lch2 == MAC_TANONPU) {
										dcs->_utfmac_exists = true;
										if (dcs->_is_surrogate) {
											goto loop_end;
										}
										sp += 4;
									}
								}
							}
						}
						sp += 4;
					}
				} else { // if (stype == DCS_UTF32LE || stype == DCS_UTF32LE_BOM)
				/* UTF32LE の時 */
					for(;;) {
						/* 残りが最大長に満たない時、次の入力バッファに移動 */
						check_ll_buf(dcs, sp, ep, 4, clist, tlist, binfo, tbinfo);
						if (sp >= ep) {
							break;
						}
						lch = utf32le_to_utf32be_char(sp);
						if ((lch & 0xffff0000) != 0) {
							dcs->_is_surrogate = true;
							if (dcs->_utfmac_exists) {
								goto loop_end;
							}
						} else {
							if (is_unibe_mac(dcs, lch, CSB_MAC_D | CSB_MAC_H | CSB_MAC_BO | CSB_MAC_TO)) {
							/* UTF8-MAC 濁点、半濁点の時 */
								if (sp + 8 <= ep) {
									lch2 = utf32le_to_utf32be_char(sp+4);
									if (lch2 == MAC_DAKUTEN || lch2 == MAC_HANDAKUTEN || lch2 == MAC_BUNONPU || lch2 == MAC_TANONPU) {
										dcs->_utfmac_exists = true;
										if (dcs->_is_surrogate) {
											goto loop_end;
										}
										sp += 4;
									}
								}
							}
						}
						sp += 4;
					}
				}
			}
loop_end:
			return DCS_SUCCESS;
		}
	}
} /* _fill_buffer */

