/*
 * dcs_typeto_name.c
 *
 * Copyright 2019-2026 Ichiji Tadokoro. All Rights Reserved.
 */

#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include "_dcs.h"

static char *_buffer, *_bptr;

typedef struct _type_name {
	int mtype;
	int type;
	char *name;
} TypeName;

TypeName _typeNames[] = {
		{ 0,0, "Empty", },
		{ DCS_ND_ASCII, DCS_ASCII, "ASCII", },
		{ DCS_ND_JIS, DCS_JIS, "JIS(ISO-2022-JP)", },
		{ DCS_ND_JIS1, DCS_JIS1, "JIS(CP50221)", },
		{ DCS_ND_JIS2, DCS_JIS2, "JIS2(ISO-2022-2004)", },
		{ DCS_ND_JIS3, DCS_JIS3, "JIS3(ISO-2022-JP-3)", },
		{ DCS_ND_SJIS, DCS_SJIS, "Shift_JIS(CP932)", },
		{ DCS_ND_SJIS2, DCS_SJIS2, "Shift_JIS-2004", },
		{ DCS_ND_EUC, DCS_EUC, "EUC", },
		{ DCS_ND_EUC1, DCS_EUC1, "EUC1(CP20932)", },
		{ DCS_ND_EUC2, DCS_EUC2, "EUC2(CP51932)", },
		{ DCS_ND_EUC3, DCS_EUC3, "EUC3(EUC-MS)", },
		{ DCS_ND_EUC4, DCS_EUC4, "EUC4(eucJP-2004)", },
		{ 0, 0, "DUMMY1", },
		{ 0, 0, "DUMMY2", },
		{ DCS_ND_UTF8, DCS_UTF8, "UTF8", },
		{ DCS_ND_UTF8_BOM, DCS_UTF8_BOM, "UTF8_BOM", },
		{ DCS_ND_UTF16BE, DCS_UTF16BE, "UTF16BE", },
		{ DCS_ND_UTF16BE_BOM, DCS_UTF16BE_BOM, "UTF16BE_BOM", },
		{ DCS_ND_UTF16LE, DCS_UTF16LE, "UTF16LE", },
		{ DCS_ND_UTF16LE_BOM, DCS_UTF16LE_BOM, "UTF16LE_BOM", },
		{ DCS_ND_UTF32BE, DCS_UTF32BE, "UTF32BE", },
		{ DCS_ND_UTF32BE_BOM, DCS_UTF32BE_BOM, "UTF32BE_BOM", },
		{ DCS_ND_UTF32LE, DCS_UTF32LE, "UTF32LE", },
		{ DCS_ND_UTF32LE_BOM, DCS_UTF32LE_BOM, "UTF32LE_BOM", },
};

size_t _typeNamesSize = sizeof(_typeNames) / sizeof(_typeNames[0]);

typedef struct _status_message {
	int sts;
	char *message;
} SatusMessage;

SatusMessage _satusMessage[] = {
		{ DCS_SUCCESS,  "SUCCESS:処理成功", },
		{ DCS_EOF,      "EOF:入力ファイルの終わり", },
		{ DCS_ENOTEXISTS,   "ENOTEXISTS:ファイルは存在せず", },
		{ DCS_ENOTFILE, "ENOTFILE:通常ファイルではない", },
		{ DCS_EALLOC,   "EALLOC:メモリー・アロケート・エラー", },
		{ DCS_EROPEN,   "EROPEN:入力ファイル・オープン・エラー", },
		{ DCS_EWOPEN,   "EWOPEN:出力ファイル・オープン・エラー", },
		{ DCS_EREAD,    "EREAD:入力エラー", },
		{ DCS_EWRITE,   "EWRITE:出力エラー", },

		{ DCS_EIOMODE,  "EIOMODE:入出力モード誤り", },
		{ DCS_EIOFDRANGE,   "EIOFDRANGE:ファイル記述子範囲誤り", },
		{ DCS_EIONOTOPEN,   "EIONOTOPEN:未オープン・ファイルへのアクセス", },
		{ DCS_EIOMAX,   "EIOMAX:上限を超えたファイル・オープン", },

		{ DCS_ECTYPE,   "ECTYPE:コード・タイプ誤り", },
		{ DCS_EBOM, "EBOM:BOM 不正（未使用）", },
		{ DCS_EUEXPEOD, "EUEXPEOD:予期せぬデータの終わり", },
		{ DCS_ECHAR,    "ECHAR:コード値エラー", },
		{ DCS_ELTYPE,   "ELTYPE:改行コード・タイプ誤り", },
};

size_t _satusMessageSize = sizeof(_satusMessage) / sizeof(_satusMessage[0]);

/* 処理ステータスからメッセージを取得 */
int dcs_statusto_message(Dcs *dcs, char *message, int sts, int ctype) {
	size_t ssize;
	int j;

	for(j=DCS_ASCII;j<_satusMessageSize;j++) {
    	if (_satusMessage[j].sts == sts) {
    		strcpy(message, _satusMessage[j].message);
    		break;
    	}
    }
    if (j >= _satusMessageSize) {
        strcpy(message, "Unknown");
    }
	ssize = strlen(message);
	if (dcs != NULL) {
		ssize = _convert_buf(dcs, message, ssize, ctype); /* サイズ・チェックなし！ */
	}
	return ssize;
}

/* バッファー変換出力 */
int _convert_buf_writer(const char *buf, size_t size, void *user_info) {
	memcpy(_bptr, buf, size);
	_bptr += size;
	*_bptr = 0;
	return DCS_SUCCESS;
}

int _convert_buf(Dcs *dcs, char *message, size_t ssize, int ctype) {
	char str[256];

	_buffer = _bptr = str;

	switch(ctype) {
	case DCS_JIS:
		/* 返却値チェックなし */
		dcs_convert_buffer(dcs, message, ssize, DCS_UTF8, DCS_JIS, false,
				NULL, _convert_buf_writer, NULL);
		break;
	case DCS_SJIS:
		/* 返却値チェックなし */
		dcs_convert_buffer(dcs, message, ssize, DCS_UTF8, DCS_SJIS, false,
				NULL, _convert_buf_writer, NULL);
		break;
	case DCS_EUC:
		/* 返却値チェックなし */
		dcs_convert_buffer(dcs, message, ssize, DCS_UTF8, DCS_EUC, false,
				NULL, _convert_buf_writer, NULL);
		break;
	case DCS_UTF8:
	default:
		return ssize;
	}
	memcpy(message, _buffer, _bptr - _buffer);
	return (_bptr - _buffer);
}

/* コード・タイプから名前を取得 */
int dcs_typeto_name(char *name, int ctype) {
	int j;
	for(j=0; ; j++) {
		if (j >= _typeNamesSize) {
			strcpy(name, "Unknown");
			break;
		}
		if (_typeNames[j].type == ctype) {
			strcpy(name, _typeNames[j].name);
			break;
		}
	}
	return strlen(name);
}

/* コード名からタイプを取得 */
int dcs_nameto_type(const char *name) {
	int j;
	const char *p;
	char *tp, str[256];

	p = name;
	tp = str;
	while(tp - str < sizeof(str) - 1) {
		if (*p == 0) {
			break;
		}
		*tp++ = toupper(*p);
		p++;
	}
	*tp = 0;
	for(j=DCS_ASCII; j < _typeNamesSize; j++) {
		if (strcmp(str, _typeNames[j].name) == 0) {
			return _typeNames[j].type;
		}
	}
	return DCS_ECTYPE;
}

/* 不特定のコード名を取得 */
int dcs_multi_type_names(char *names, size_t size, int ctype) {
    int j, len;
    char *p;
    if ((size) <= 0) {
        return 0;
    }
    if ((ctype) <= DCS_NORMAL) {
        return dcs_typeto_name((names), (ctype));
    }
    p = (names);
    for(j=DCS_ASCII;j<_typeNamesSize;j++) {
        if ((_typeNames[j].mtype & ctype) != 0) {
            len = strlen(_typeNames[_typeNames[j].type].name);
            if ((p - (names)) + len > (size)) {
            /* ... + XXX */
            /* サイズを超えた時 */
                break;
            }
            strcpy(p, _typeNames[_typeNames[j].type].name);
            p += len;
            if ((p - (names)) + 1 > size) {
            /* サイズを超えた時 */
                break;
            }
            *p++ = DCS_SEPARATE_CHAR;
        }
    }
    if (p > (names) && *(p - 1) == DCS_SEPARATE_CHAR) {
        p--;
    }
    *p = 0;
    return(p - (names));
}

/* すべてのコード名を取得 */
int dcs_all_type_names(char *names, size_t size) {
	int j, len;
	char *p;

	if (size <= 0) {
		return 0;
	}
	p = names;
    for(j=DCS_ASCII;j<_typeNamesSize;j++) {
    	if (_typeNames[j].type == 0) {
    		continue;
    	}
		len = strlen(_typeNames[j].name);
		if ((p - names) + len > size) {
		/* サイズを超えた時 */
			break;
		}
		strcpy(p, _typeNames[j].name);
		p += len;
		if ((p - names) + 1 > size) {
		/* サイズを超えた時 */
			break;
		}
		*p++ = DCS_SEPARATE_CHAR;
	}
    if (p > (names) && *(p - 1) == DCS_SEPARATE_CHAR) {
        p--;
    }
    *p = 0;
	return(p - names); /* ヌル文字を含まない */
}
/* すべてのコード名の長さを取得 */
int dcs_all_type_names_length() {
	size_t len = 0;
	int j;

	for(j=DCS_ASCII; j < _typeNamesSize; j++) {
		if (_typeNames[j].type > 0) {
			/* + 1: 区切り文字、最後はヌル文字 */
			len += strlen(_typeNames[j].name) + 1;
		}
	}
	return len; /* ヌル文字を含む */
}
