/*
 * dcs_main.c
 *
 * Copyright 2019-2025 Ichiji Tadokoro. All Rights Reserved.
 */

#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include "dcs.h"

/* 終了ステータス */
#define STS_SUCCESS		0	/* 成功 */
#define STS_WARNING		1	/* 警告 */
#define STS_ERROR		2	/* エラー */

int _exec_sts;

/* 標準入力を有効とする */
#define _EFFECT_STDIN		1
/* 標準出力を有効とする */
#define _EFFECT_STDOUT		1
/* コピーを行わない */
#define _NO_COPY			0

const char _STDIN_NAME_[] = "STDIN(-)";
const char _STDOUT_NAME_[] = "STDOUT";

char *_pname;
int _option;
int _sltype, _dltype;
FILE *_gfp;
Dcs *_dcs;

static char *_buffer, *_bptr;

#define FLAG_STYPE		000000000001
#define FLAG_DTYPE		000000000002
#define FLAG_DPATH		000000000004
#define FLAG_UTFMAC_TO	000000000010
#define FLAG_FORCED		000000000020
#define FLAG_TEST		000000000040
#define FLAG_VERBOSE	000000000100
#define FLAG_SUPPRESS	000000000200
#define FLAG_MULTI_IN	000000000400
#define FLAG_CTYPE		000000001000
#define FLAG_STYPE_ALL	000000002000
#define FLAG_SLTYPE		000000004000
#define FLAG_DLTYPE		000000010000
#define FLAG_ENGLISH	000000020000
#define FLAG_NPPATH		000000040000
#define FLAG_PPATH		000000100000
#define FLAG_SUPPRESS_DEST	000000200000
#define FLAG_HANKANA_TO	000000400000
#define FLAG_VERSION		010000000000
#define FLAG_HELP		020000000000

int _dstat;
char *_sname;
char *_dname = (char *)_STDOUT_NAME_;
char _dfname[PATH_MAX];
char *_oname;

#define _DST_NOTHING		0
#define _DST_DIR			1
#define _DST_FILE		2
#define _DST_OTHER		9

const char *_io_val_jp[] = {
	"入力",
	"出力",
	"表示",
};

const char *_io_val_en[] = {
	"input",
	"output",
	"view",
};

#define _I_IN		0
#define _I_OUT		1
#define _I_VIEW		2

const char **_io_val;

const char _usage_fmt_jp[] =
"形式１：%s [-s type] ([-A][(-h|-H)][-l][-q] |\n\
                       [ -d type [-o dest][-L values][-Q][-m][-k][-f][-t] ])\n\
            [-C type][-E][-v][src...]\n\
形式２：%s ( --help [-E] | --version )\n\
-s:   入力タイプ\n\
type(最右正式名の type は大文字／小文字区別なし)\n\
    j,     ISO-2022-JP      - JIS\n\
    j1,    CP50221          - JIS\n\
    j2,    ISO-2022-JP-2004 - JIS\n\
    j3,    ISO-2022-JP-3    - JIS\n\
    s,     CP932            - Shift JIS\n\
    s1,    Shift_JIS\n\
    s2,    Shift_JIS-2004\n\
    e,     EUC-JP\n\
    e1,    CP20932          - EUC-JP\n\
    e2,    CP51932          - EUC-JP\n\
    e3,    eucJP-ms\n\
    e4,    EUC-JIS-2004\n\
    u8,    UTF-8\n\
    u8m,   UTF-8-BOM\n\
    ub,    u16b,  UTF-16BE\n\
    ubm,   u16bm, UTF-16BE-BOM\n\
    ul,    u16l,  UTF-16LE\n\
    ulm,   u16lm, UTF-16LE-BOM\n\
    u32b,  UTF-32BE\n\
    u32bm, UTF-32BE-BOM\n\
    u32l,  UTF-32LE\n\
    u32lm, UTF-32LE-BOM\n\
-A:   可能性のある入力タイプをすべて表示\n\
-h:   ファイル名を非表示\n\
-H:   ファイル名を表示\n\
-l:   改行文字判定\n\
-q:   入力タイプの注釈 \"?\"を抑止\n\
-d:   出力タイプ（処理がコードの判定から変換に変わる）\n\
-o:   出力先（指定がなければ画面に表示する）\n\
dest: 出力先名（ファイル又はディレクトリ）\n\
-L:   改行文字変換\n\
values: 変換で付加する値\n\
    a   - ライン・フィード\n\
    d   - キャリッジ・リターン\n\
-Q:   不正文字\"?\"出力を抑止\n\
-m:   日本語UTF8-MACを１文字に変換\n\
-k:   半角カナを全角に変換\n\
-f:   変換時の入力タイプを強制\n\
-t:   テスト（出力せず）\n\
-C:   表示タイプ\n\
-E:   メッセージを英語表示\n\
-v:   詳細情報\n\
src:  入力ファイル\n\
--help:     ヘルプを表示して終了\n\
--version:  バージョン情報を表示して終了\n\
\n\
注：オプション -d:  出力タイプの指定により、コード判定から変換に処理が変わる\n\
    オプション -o:  出力先はファイルがあれば同名でも上書きし、ディレクトリがあれ\n\
                    ばその中に入力と同名で出力し、なければファイルとして出力する\n\
例：%s -l foo.txt                    - foo.txt の文字コードと改行コードを表示\n\
    %s -l -scp932 foo.txt            - 上記に加え、入力判定が未確定で候補に\n\
                                        SJIS があれば \"cp932\" と確定表示\n\
    %s -du8 -o dest -Lda foo.txt     - foo.txt の文字コードを UTF-8 に変換して\n\
                    dest に出力し、改行文字は CR/LF の２バイトに変換する\n\
    %s -du8 -o dest -Lda -ss foo.txt - 上記に加え、入力判定が未確定で候補に\n\
                    SJIS があれば SJIS として変換する\n\
DCS バージョン：%.2f\n\
";

const char _usage_fmt_en[] =
"Usage 1: %s [-s type] ([-A][(-h|-H)][-l][-q] |\n\
                        [ -d type [-o dest][-L values][-Q][-m][-k][-f][-t] ])\n\
             [-C type][-E][-v][src...]\n\
Usage 2: %s (--help [-E] | --version)\n\
-s: Input type\n\
type(The rightmost formal name type is case insensitive)\n\
    j,     ISO-2022-JP      - JIS\n\
    j1,    CP50221          - JIS\n\
    j2,    ISO-2022-JP-2004 - JIS\n\
    j3,    ISO-2022-JP-3    - JIS\n\
    s,     CP932            - Shift JIS\n\
    s1,    Shift_JIS\n\
    s2,    Shift_JIS-2004\n\
    e,     EUC-JP\n\
    e1,    CP20932          - EUC-JP\n\
    e2,    CP51932          - EUC-JP\n\
    e3,    eucJP-ms\n\
    e4,    EUC-JIS-2004\n\
    u8,    UTF-8\n\
    u8m,   UTF-8-BOM\n\
    ub,    u16b,  UTF-16BE\n\
    ubm,   u16bm, UTF-16BE-BOM\n\
    ul,    u16l,  UTF-16LE\n\
    ulm,   u16lm, UTF-16LE-BOM\n\
    u32b,  UTF-32BE\n\
    u32bm, UTF-32BE-BOM\n\
    u32l,  UTF-32LE\n\
    u32lm, UTF-32LE-BOM\n\
-A: Show all possible input types\n\
-h: Don't display the file name.\n\
-H: Display the file name\n\
-l: Newline character judgment\n\
-q: Suppress input type annotations \"?\"\n\
-d: Output type (processing changes from code determination to conversion)\n\
-o: Output destination (displayed on the screen if not specified)\n\
dest: output destination name (file or directory)\n\
-L: Newline character conversion\n\
values: Value to be added by conversion\n\
    a  - line feed\n\
    d  - carriage return\n\
-Q: Suppressing the output of illegal characters \"?\"\n\
-m: Convert Japanese UTF8-MAC to 1 character\n\
-k: Convert half-width katakana to full-width\n\
-f: Force input type during conversion\n\
-t: test (without output)\n\
-C: Display type\n\
-E: Display message in English\n\
-v: More info\n\
src: Input file\n\
--help:     Show help and exit\n\
--version:  Display version information and exit\n\
\n\
Note:\n\
    Option -d: Processing changes from code judgment to code conversion\n\
                    depending on the output type specification\n\
    Option -o: If there is a file, the output destination will be overwritten\n\
                    with the same name, if there is a directory, it will be\n\
                    output with the same name as the input file, otherwise it\n\
                    will be output as a file\n\
Example:\n\
    %s -l foo.txt                    - Displays the character code and line\n\
                    feed code of foo.txt\n\
    %s -l -scp932 foo.txt            - In addition to the above, if the input\n\
                    judgment is unconfirmed and there is SJIS as a candidate,\n\
                    \"cp932\" is confirmed and displayed.\n\
    %s -du8 -o dest -Lda foo.txt     - Converts the character code of foo.txt\n\
                    to UTF-8 and outputs it to dest Newline character is 2 bytes\n\
                    of CR/LF\n\
    %s -du8 -o dest -Lda -ss foo.txt - In addition to the above, if the input\n\
                    judgment is undecided and there is SJIS as a candidate, it\n\
                    will be converted as SJIS.\
DCS version: %.2f\n\
";

enum _err_no {
	ERR_UNKNOWN,
	ERR_ILLEGAL_OPTIONL,
	ERR_ILLEGAL_OPTION,
	ERR_TYPE_OPTION_NOTHING,
	ERR_ILLEGAL_TYPE_OPTION,
#if !_EFFECT_STDIN
	ERR_SNAME_NOTHING,
#endif
	ERR_SFILE_NOTHING,
	ERR_SFILE_NOTFILE,
	ERR_DTYPE_NOTHING,
#if !_EFFECT_STDOUT
	ERR_DNAME_NOTHING,
#endif
	ERR_DNAME_VALUE_NOTHING,
	ERR_TYPES_EQUALS,
	ERR_DFILE_NOTHING,
	ERR_DNAME_NOTDIR,
	ERR_DNAME_NOTHING,

	ERR_TEST,
	ERR_MKDIR,
	ERR_FILE_STAT,
	ERR_FILE_OPEN,
	ERR_FILE_READ,
	ERR_FILE_WRITE,
	ERR_MALLOC,
	ERR_REAL_TYPES_EQUALS,
	ERR_DECIDED,
	ERR_DECIDED_SKIP,
	ERR_NAME,
	ERR_EXEC,
	ERR_BOM,
	ERR_UEXPEOF,
	ERR_ILLEGALCHAR,
	ERR_MULTI_DTYPE,
	ERR_NOCHAR,
	ERR_LTYPE_VALUE_NOTHING,
	ERR_ILLEGAL_LTYPE_VALUE,
	ERR_LTYPE_MIXED,
	ERR_DECIDED_SKIP_LN,
	ERR_MULTI_SLTYPE,
};

const char *_messages_jp[] = {
	"%s: 不明なエラーです - %d(%d: %s) - %s\n",
	"%s: 次のオプションは不適当です - %s\n",
	"%s: 次のオプションは不適当です - -%c\n",
	"%s: %sタイプのオプションを指定して下さい - -%c\n",
	"%s: 次のファイル・タイプは不適当です - -%c %s\n",
#if !_EFFECT_STDIN
	"%s: 入力ファイル名を指定して下さい\n",
#endif
	"%s: 入力ファイルが存在しません - %s\n",
	"%s: 入力ファイルは通常ファイルではありません - %s\n",
	"%s: 出力タイプ(-d)を指定して下さい\n",
#if !_EFFECT_STDOUT
	"%s: 出力ファイル名(-o)を指定して下さい\n",
#endif
	"%s: 出力先(dest)を指定して下さい - -%c\n",
	"%s: 入出力タイプが同じです - %s\n",
	"%s: 出力先ディレクトリが存在しません(%s)\n",
	"%s: 複数入力に対する出力先はディレクトリを指定して下い(%s)\n",
	"%s: 複数入力に対する出力先ディレクトリが存在しません(%s)\n",

	"テスト・モードです\n",
	"%s: 出力先ディレクトリを作成できません(%d: %s) - %s\n",
	"%s: このファイルは存在しません(%d: %s) - %s\n",
	"%s: ファイル・オープン・エラー(%d: %s) - %s\n",
	"%s: ファイル入力エラー(%d: %s) - %s\n",
	"%s: ファイル出力エラー(%d: %s) - %s\n",
	"%s: メモリー・アロケート・エラー\n",
	"実入力タイプが出力タイプと同じです - %s %s コピーします\n",
	"入力タイプを確定できないので %s と見なします - %s\n",
	"%s: 入力タイプを確定できないので処理をスキップします - %s: %s\n",
	"%s: %s%s\n",
	"%s(%s) -> %s(%s) %s\n",
	"%s: BOM が不正です - %s\n",
	"%s: 予期せぬファイルの終わり - %s\n",
	"%s: 文字コードが不正です - %s\n",
	"%s: オプション '-A', '-d' が競合しています - '-d' を採ります\n",
	"変換できない文字がありました（%d 文字：先頭 %d %04x：末尾 %d %04x）- %s:%s => %s:%s\n",
	"%s: 出力する改行文字を指定して下さい - -%c\n",
	"%s: 次の改行文字の値は不適当です - -%c %c\n",
	"改行タイプが混在しています - %s: %s\n",
	"入力タイプを確定できないので改行処理をスキップします - %s\n",
	"%s: オプション '-A', '-l' が競合しています - '-A' を採ります\n",
};

const char *_messages_en[] = {
	"%s: Unknown error - %d(%d: %s) - %s\n",
	"%s: The following options are inappropriate - %s\n",
	"%s: The following options are inappropriate - -%c\n",
	"%s: Specify %s type options - -%c\n",
	"%s: The following file types are incorrect - -%c %s\n",
	#if !_EFFECT_STDIN
	"%s: Specify the input file name\n",
	#endif
	"%s: Input file does not exist - %s\n",
	"%s: Input file is not a regular file - %s\n",
	"%s: Specify output type (-d)\n",
	#if !_EFFECT_STDOUT
	"%s: Specify the output file name (-o)\n",
	#endif
	"%s: Specify the output destination (dest) - -%c\n",
	"%s: I/O type is the same - %s\n",
	"%s: Output destination directory does not exist (%s)\n",
	"%s: Specify a directory as the output destination for multiple inputs (%s)\n",
	"%s: Output destination directory does not exist for multiple inputs (%s)\n",

	"Test mode\n",
	"%s: Unable to create output directory (%d: %s) - %s\n",
	"%s: This file does not exist (%d: %s) - %s\n",
	"%s: File open error (%d: %s) - %s\n",
	"%s: File input error (%d: %s) - %s\n",
	"%s: File output error (%d: %s) - %s\n",
	"%s: Memory allocate error\n",
	"The actual input type is the same as the output type - %s %s copy\n",
	"The input type cannot be determined, so consider it as %s - %s\n",
	"%s: Skip processing because input type cannot be determined - %s: %s\n",
	"%s: %s%s\n",
	"%s(%s) -> %s(%s) %s\n",
	"%s: Illegal BOM - %s\n",
	"%s: Unexpected EOF - %s\n",
	"%s: Character code is invalid - %s\n",
	"%s: Options '-A', '-d' conflict - take '-d'\n",
	"Some characters could not be converted (%d characters: start %d %04x: end %d %04x) - %s:%s => %s:%s\n",
	"%s: Specify the newline character to output - -L\n",
	"%s: The value of this newline character is inappropriate - %c\n",
	"Mixed line feed types - %s: %s\n",
	"Since the input type cannot be determined, line feed processing is skipped - %s\n",
	"%s: Option '-A', '-l' conflicts - take '-A'\n",
};

enum i_ope_no {
	OPE_COPY,
	OPE_ADD_BOM,
	OPE_DELETE_BOM,
	OPE_CONV,
};

const char *_ope_msg_jp[] = {
	"コピーします...",
	"BOM を追加します...",
	"BOM を削除します...",
	"コード変換します...",
};

const char *_ope_msg_en[] = {
	"Copy...",
	"Add a BOM...",
	"Delete the BOM...",
	"Code conversion...",
};

const char *_usage_fmt;
const char **_messages;
const char **_ope_msg;

/* 入力タイプ・デフォルト */
int _dstype = DCS_ASCII;
/* 出力時の判定入力タイプ */
int _stype;
/* 出力時の出力タイプ */
int _dtype;
/* 出力時の入力タイプを強制 */
bool _is_forced = false;
/* 表示タイプ */
int _dcs_ctype;

/* アーギュメント・チェック */
char **_check_arg(bool *is_error, int *argcp, char *argv[], bool is_exec);
/* タイプ名取得 */
char *_get_type_name(char *str, int type);
/* ファイル・タイプ・チェック */
int _check_type(const char *type);
/* ファイル・チェック */
int _check_stat(const char *path);
/* 変換監視 */
int _observer(int sts, void *user_info);
/* 変換出力 */
int _writer(const char *buf, size_t size, void *user_info);
/* １ファイル実行 */
int _execute(const char *fname);
/* 操作内容判定 */
int _check_ope(int stype, int dtype);
/* 文字列コードを表示タイプに変換 */
char *change_locale(char *dest, const char *src);
/* 	エラー・チェック */
int _check_error(int sts);
/* ストリーム取得 */
int _get_stream(FILE **fp, const char *dpath);
/* バッファー変換出力 */
int _buf_writer(const char *buf, size_t size, void *user_info);

char _cbuf[4096];
char *_stdin[] = { "-" };
FILE *_ofp;

int main(int argc, char *argv[]) {
	int tsts, sts = STS_SUCCESS;
	char **fnames;
	bool is_error;

	_ofp = stdout;
	_usage_fmt = _usage_fmt_jp;
	_messages = _messages_jp;
	_io_val = _io_val_jp;
	_ope_msg = _ope_msg_jp;
	/* アーギュメント・チェック */
	_check_arg(&is_error, &argc, argv, false);
	if ((_option & FLAG_ENGLISH) != 0) {
	/* 英語表示の時 */
		_usage_fmt = _usage_fmt_en;
		_messages = _messages_en;
		_io_val = _io_val_en;
		_ope_msg = _ope_msg_en;
	}
	fnames = _check_arg(&is_error, &argc, argv, true);

	if ((_option & FLAG_VERBOSE) != 0) {
		if ((_option & FLAG_TEST) != 0) {
			/* "テスト・モードです\n", */
			fprintf(_gfp, change_locale(_cbuf, _messages[ERR_TEST]));
		}
	}
	if ((_option & FLAG_SUPPRESS_DEST) != 0) {
		dcs_set_suppress(_dcs, true);
	}
	if (argc >= 2) {
	/* 入力ファイル名が複数ある時 */
		_option |= FLAG_MULTI_IN;
	}
	if (sts == DCS_SUCCESS) {
		if ((_option & FLAG_DTYPE) != 0) {
		/* 出力タイプがある時 */
			if ((_option & FLAG_DLTYPE) != 0) {
			/* 改行処理がある時 */
				set_ltype(_dcs, _dltype);
			}
		}
		/* cygwin -O で *fnames == NULL 不正 */
		while(argc-- > 0) { /* *fnames != NULL) { */
			tsts = _execute(*fnames);
			if (sts < tsts) {
				sts = tsts;
			}
			fnames++;
		}
		if (_ofp == stdout) {
		/* 出力ファイル名がない時 */
			fflush(stdout);
		} else {
		/* 出力ファイル名がある時 */
			if (_ofp != NULL) {
				fclose(_ofp);
			}
		}
	}
	if (sts != 0) {
		fflush(stderr);
	}
	release_dcs(_dcs);
	return(sts);
} /* main */

/* アーギュメント・チェック */
char **_check_arg(bool *pis_error, int *argcp, char *argv[], bool is_exec)
{
	bool is_error = false;
	int c, tc, argc = *argcp;
	char *targ;
	char *p, *sp, str[1024], *ep, eval[256];

	_gfp = stdout;
	sp = str;
	sp += sprintf(sp, "--------------------------------------------------------------------------------\n");
	/* プログラム名の抽出 */
	if ((_pname = strrchr(argv[0], '/')) == NULL) {
		_pname = strrchr(argv[0], '\\');
	}
	if (_pname == NULL) {
		_pname = argv[0];
	} else {
		_pname++;
	}
	if ((p = strrchr(_pname, '.')) != NULL) {
		if ((*++p == 'E' || *p == 'e')
		 && (*++p == 'X' || *p == 'x')
		 && (*++p == 'E' || *p == 'e')
		 && *(p + 1) == 0) {
			*(p - 3) = 0;
		}
	}
	/* 引数チェック */
	_option = _sltype = _dltype = 0;
	while(--argc && (*(++argv)[0] == '-')) {
	/* オプションが続いているあいだ繰り返す */
		if (!strcmp(argv[0], "-")) {
		/* 標準入力の時 */
			break;
		} else if (!strncmp(argv[0], "--", 2)) {
			if (!strcmp(argv[0], "--")) {
			/* 明示的なオプションの終わりの時 */
				argc--;
				argv++;
				break;
			} else if (!strcmp(argv[0], "--help")) {
			/* ヘルプの時 */
				_option |= FLAG_HELP;
				continue;
			} else if (!strcmp(argv[0], "--version")) {
				/* バージョン情報の時 */
				_option |= FLAG_VERSION;
				continue;
			} else {
				/* "%s: 次のオプションは不適当です - %s\n" */
				sp += sprintf(sp, _messages[ERR_ILLEGAL_OPTIONL], _pname, argv[0]);
				is_error = true;
				continue;
			}
		}
		targ = *argv;
		while((c = *++argv[0]) != 0) {
			/* オプション付きフラグ */
			switch (c) {
			case 's':
			case 'd':
			case 'o':
			case 'C':
			case 'L':
				if (!*(p = ++argv[0])) {
				/* フラグの後に文字がない時 */
					if (argc > 1 && *(p = (argv + 1)[0]) != '-') {
					/* 次の文字列が有効の時 */
						*argv = targ;
						argc--;
						argv++;
						targ = *argv;
					} else {
						p = NULL;
					}
				}
				if (p != NULL) {
				/* 有効なオプションがあった時 */
					while(*(++argv[0]))
						;
				}
				argv[0]--;
				break;
			}
			/* すべてのオプション */
			switch (c) {
			case 's':
				if (p == NULL) {
					/* "%s: %sタイプ・オプションを指定して下さい - -%c\n" */
					sp += sprintf(sp, _messages[ERR_TYPE_OPTION_NOTHING], _pname, _io_val[_I_IN], c);
					is_error = true;
				} else {
					if ((_dstype = _check_type(p)) < 0) {
						/* "%s: 次のファイル・タイプは不適当です - -%c %s\n" */
						sp += sprintf(sp, _messages[ERR_ILLEGAL_TYPE_OPTION], _pname, c, p);
						is_error = true;
					} else {
						_option |= FLAG_STYPE;
					}
				}
				break;
			case 'd':
				if (p == NULL) {
				/* 有効なオプションがなかった時 */
					/* "%s: %sタイプ・オプションを指定して下さい - -%c\n" */
					sp += sprintf(sp, _messages[ERR_TYPE_OPTION_NOTHING], _pname, _io_val[_I_OUT], c);
					is_error = true;
				} else {
					if ((_dtype = _check_type(p)) < 0) {
						/* "%s: 次のファイル・タイプは不適当です - -%c %s\n" */
						sp += sprintf(sp, _messages[ERR_ILLEGAL_TYPE_OPTION], _pname, c, p);
						is_error = true;
#if _NO_COPY
					} else if (_dtype == _dstype) {
						/* "%s: 入出力タイプが同じです - %s\n" */
						sp += sprintf(sp, _messages[ERR_TYPES_EQUALS], _pname, p);
						is_error = true;
#endif
					}
				}
				/* ERR_DTYPE_NOTHING を回避するため */
				_option |= FLAG_DTYPE;
				break;
			case 'C':
				if (p == NULL) {
					/* "%s: %sタイプ・オプションを指定して下さい - -%c\n" */
					sp += sprintf(sp, _messages[ERR_TYPE_OPTION_NOTHING], _pname, _io_val[_I_VIEW], c);
					is_error = true;
				} else {
					if ((_dcs_ctype = _check_type(p)) < 0) {
						/* "%s: 次のファイル・タイプは不適当です - -%c %s\n" */
						sp += sprintf(sp, _messages[ERR_ILLEGAL_TYPE_OPTION], _pname, c, p);
						is_error = true;
					} else {
						_option |= FLAG_CTYPE;
					}
				}
				break;
			case 'o':
				if (p == NULL) {
					/* "%s: 出力先(dest)を指定して下さい - -%c\n" */
					sp += sprintf(sp, _messages[ERR_DNAME_VALUE_NOTHING], _pname, c);
					is_error = true;
				} else {
					_dstat = _check_stat(p);
					_dname = p;
					_option |= FLAG_DPATH;
				}
				break;
			case 'l':
				_option |= FLAG_SLTYPE;
				break;
			case 'L':
				if (p == NULL) {
					/* "%s: 出力する改行文字を指定して下さい - -L\n" */
					sp += sprintf(sp, _messages[ERR_LTYPE_VALUE_NOTHING], _pname, c);
					is_error = true;
				} else {
					tc = c;
					_dltype = 0;
					*eval = 0;
					ep = eval;
					for(; *p; p++) {
						if ((c = tolower(*p)) == 'a') {
							_dltype |= LTYPE_LF;
						} else if (c == 'd') {
							_dltype |= LTYPE_CR;
						} else {
							if (strchr(eval, c) == NULL) {
								/* "%s: 次の改行文字の値は不適当です - %c\n" */
								sp += sprintf(sp, _messages[ERR_ILLEGAL_LTYPE_VALUE], _pname, tc, *p);
								*ep++ = c;
								*ep = 0;
							}
							is_error = true;
						}
					}
					if ((_dltype & (LTYPE_CR|LTYPE_LF)) == (LTYPE_CR|LTYPE_LF)) {
						_dltype = LTYPE_CRLF;
					}
					_option |= FLAG_DLTYPE;
				}
				break;
			case 'm':
				_option |= FLAG_UTFMAC_TO;
				break;
			case 'k':
				_option |= FLAG_HANKANA_TO;
				break;
			case 'f':
				_is_forced = true;
				_option |= FLAG_FORCED;
				break;
			case 't':
				_option |= FLAG_TEST;
				break;
			case 'q':
				_option |= FLAG_SUPPRESS;
				break;
			case 'Q':
				_option |= FLAG_SUPPRESS_DEST;
				break;
			case 'v':
				_option |= FLAG_VERBOSE;
				break;
			case 'A':
				_option |= FLAG_STYPE_ALL;
				break;
			case 'E':
				_option |= FLAG_ENGLISH;
				break;
			case 'h':
				_option |= FLAG_NPPATH;
				break;
			case 'H':
				_option |= FLAG_PPATH;
				break;
			default:
				/* "%s: 次のオプションは不適当です - -%c\n" */
				sp += sprintf(sp, _messages[ERR_ILLEGAL_OPTION], _pname, c);
				is_error = true;
				break;
			}
		}
		/* *arg の位置を戻す */
		*argv = targ;
	}
	if (argc < 1) {
		argv++;
	}
	if ((_option & FLAG_DTYPE) == 0) {
	/* 出力タイプがない時 */
		if ((_option & (FLAG_DPATH|FLAG_DLTYPE)) != 0) {
		/* 出力ファイル名がある時
		 * 又は改行編集がある時 */
	   		/* "%s: 出力タイプ(-d)を指定して下さい\n" */
			sp += sprintf(sp, _messages[ERR_DTYPE_NOTHING], _pname);
			is_error = true;
		}
		if ((_option & FLAG_TEST) != 0) {
		/* テスト・モードがある時 */
			_option &= ~FLAG_TEST;
		}
	} else {
	/* 出力タイプがある時 */
		if ((_option & FLAG_DPATH) == 0) {
		/* 出力ファイル名がない時 */
#if !_EFFECT_STDOUT
			/* "%s: 出力ファイル名(-o)を指定して下さい\n" */
			sp += sprintf(sp, _messages[ERR_DNAME_NOTHING], _pname);
			is_error = true;
#endif
			_gfp = stderr;
		} else {
		/* 出力ファイル名がある時 */
			if (argc >= 2) {
			/* 入力ファイル名が複数ある時 */
				if (_dstat != _DST_DIR) {
				/* 出力ファイル属性がディレクトリではない時 */
					if (_dstat == _DST_NOTHING) {
					/* 出力ファイルが存在しない時 */
				   		/* "%s: 複数入力に対する出力先ディレクトリが存在しません(%s)\n" */
						sp += sprintf(sp, _messages[ERR_DNAME_NOTHING], _pname, _dname);
					} else {
				   		/* "%s: 複数入力に対する出力先はディレクトリを指定して下い(%s)\n" */
						sp += sprintf(sp, _messages[ERR_DNAME_NOTDIR], _pname, _dname);
					}
					is_error = true;
				}
			}
		}
	}
	if (argc <= 0) {
		argc = 1;
		argv = _stdin;
	}
	if (is_exec) {
		_dcs = create_dcs();
		if (_dcs == NULL) {
			fprintf(stderr, _messages[ERR_MALLOC], _pname, _sname);
			exit(STS_ERROR);
		}
		if (is_error) {
			fprintf(stderr, change_locale(_cbuf, _usage_fmt), _pname, _pname, _pname, _pname, _pname, _pname, dcs_get_version());
			fprintf(stderr, change_locale(_cbuf, str));
			exit(STS_ERROR);
		} else if ((_option & FLAG_HELP) != 0) {
		/* ヘルプの時 */
			fprintf(stdout, change_locale(_cbuf, _usage_fmt), _pname, _pname, _pname, _pname, _pname, _pname, dcs_get_version());
			exit(STS_SUCCESS);
		} else if ((_option & FLAG_VERSION) != 0) {
		/* バージョン情報の時 */
			fprintf(stdout, change_locale(_cbuf, DCS_VERSION_INFO), dcs_get_version()); fflush(stdout);
			exit(STS_SUCCESS);
		} else {
			if ((_option & FLAG_DTYPE) != 0) {
			/* 出力タイプがある時 */
				if ((_option & FLAG_STYPE_ALL) != 0) {
				/* 入力タイプをすべて表示の時 */
					/* "%s: オプション '-A', '-d' が競合しています - '-d' を採ります\n", */
					fprintf(_gfp, _messages[ERR_MULTI_DTYPE], _pname);
					_option &= ~FLAG_STYPE_ALL;
				}
				if ((_option & FLAG_UTFMAC_TO) != 0) {
				/* UTF8-MACを１バイト文字に変換する時 */
					dcs_set_utfmac_to(_dcs, true);
				}
				if ((_option & FLAG_HANKANA_TO) != 0) {
				/* 半角カナを全角に変換する時 */
					dcs_set_hankana_to(_dcs, true);
				}
			} else if ((_option & FLAG_SLTYPE) != 0) {
			/* 改行判定がある時 */
				if ((_option & FLAG_STYPE_ALL) != 0) {
				/* 入力タイプをすべて表示の時 */
					/* "%s: オプション '-A', '-l' が競合しています - '-A' を採ります\n", */
					fprintf(_gfp, _messages[ERR_MULTI_SLTYPE], _pname);
					_option &= ~FLAG_SLTYPE;
				}
			}
		}
		*pis_error = is_error;
		*argcp = argc;
	} /* if (is_exec) */
	return argv;
} /* _check_arg */

typedef struct _type_value {
	int type;
	char *value;
} TypeValue;

/* ファイル・タイプ・チェック */
int _check_type(const char *type) {
	const char *p = type;
	bool is_utf32;
	int idx, min, max, cmp;

	TypeValue type_value[] = {
		{ DCS_EUC1, "CP20932", },
		{ DCS_JIS1,  "CP50221", },
		{ DCS_EUC2, "CP51932", },
		{ DCS_SJIS1, "CP932", },
		{ DCS_EUC4, "EUC-JIS-2004", },
		{ DCS_EUC,  "EUC-JP", },
		{ DCS_EUC3, "eucJP-ms", },
		{ DCS_JIS, "ISO-2022-JP", },
		{ DCS_JIS2, "ISO-2022-JP-2004", },
		{ DCS_JIS3, "ISO-2022-JP-3", },
		{ DCS_SJIS, "Shift_JIS", },
		{ DCS_SJIS2, "Shift_JIS-2004", },
		{ DCS_UTF16BE,     "UTF-16BE", },
		{ DCS_UTF16BE_BOM, "UTF-16BE-BOM", },
		{ DCS_UTF16LE,     "UTF-16LE", },
		{ DCS_UTF16LE_BOM, "UTF-16LE-BOM", },
		{ DCS_UTF32BE,     "UTF-32BE", },
		{ DCS_UTF32BE_BOM, "UTF-32BE-BOM", },
		{ DCS_UTF32LE,     "UTF-32LE", },
		{ DCS_UTF32LE_BOM, "UTF-32LE-BOM", },
		{ DCS_UTF8,        "UTF-8", },
		{ DCS_UTF8_BOM,    "UTF-8-BOM", },
	};

	switch(*p) {
	case 'j':
		p++;
		switch(*p) {
		case 0:
			return DCS_JIS;
		case '1':
			if (*++p == 0) {
				return DCS_JIS1;
			}
			break;
		case '2':
			if (*++p == 0) {
				return DCS_JIS2;
			}
			break;
		case '3':
			if (*++p == 0) {
				return DCS_JIS3;
			}
			break;
		default:
			break;
		}
		break;
	case 's':
		p++;
		switch(*p) {
		case 0:
			return DCS_SJIS;
		case '1':
			if (*++p == 0) {
				return DCS_SJIS1;
			}
			break;
		case '2':
			if (*++p == 0) {
				return DCS_SJIS2;
			}
			break;
		default:
			break;
		}
		break;
	case 'e':
		p++;
		switch(*p) {
		case 0:
			return DCS_EUC;
		case '1':
			if (*++p == 0) {
				return DCS_EUC1;
			}
			break;
		case '2':
			if (*++p == 0) {
				return DCS_EUC2;
			}
			break;
		case '3':
			if (*++p == 0) {
				return DCS_EUC3;
			}
			break;
		case '4':
			if (*++p == 0) {
				return DCS_EUC4;
			}
			break;
		default:
			break;
		}
		break;
	case 'u':
		p++;
		is_utf32 = false;
		if (*p == '3' && *(p + 1) == '2' && (*(p + 2) < '0' || *(p + 2) > '9')) {
			is_utf32 = true;
			p += 2;
		} else if (*p == '1' && *(p + 1) == '6' && (*(p + 2) < '0' || *(p + 2) > '9')) {
				p += 2;
		}
		switch(*p) {
		case 'b':
			if (*++p == 0) {
				return (is_utf32?  DCS_UTF32BE: DCS_UTF16BE);
			}
			if (*p == 'm'
			 && *++p == 0) {
				return (is_utf32?  DCS_UTF32BE_BOM: DCS_UTF16BE_BOM);
			}
			break;
		case 'l':
			if (*++p == 0) {
				return (is_utf32?  DCS_UTF32LE: DCS_UTF16LE);
			}
			if (*p == 'm'
			 && *++p == 0) {
				return (is_utf32?  DCS_UTF32LE_BOM: DCS_UTF16LE_BOM);
			}
			break;
		case '8':
			if (*++p == 0) {
				return DCS_UTF8;
			}
			if (*p == 'm'
			 && *++p == 0) {
				return DCS_UTF8_BOM;
			}
			break;
		default:
			break;
		}
		break;
	default:
		break;
	}
	/* 正式名の判定 */
	p = type;
	min = 0;
	max = sizeof(type_value) / sizeof(type_value[0]);

	while(min < max) {
		idx = (max - min) / 2 + min;
		/* 大文字／小文字区別なし */
		cmp = strcasecmp(type_value[idx].value, p);
		if (cmp < 0) {
			min = idx + 1;
		} else if (cmp > 0) {
			max = idx;
		} else {
			return type_value[idx].type;
		}
	}
	/* while(min < max) の条件を抜けた場合  */
	if (strcasecmp(type_value[min].value, p) == 0) {
		return type_value[min].type;
	}
	return -1;
} /* _check_type */

typedef struct _type_name_info {
	int type;
	char *name;
} TypeNameInfo;

#define TN_MAYBE		"?"
#define TN_EMPTY		"Empty"
#define TN_UNKNOWN		"Unknown"
#define TN_ASCII		"ascii"
#define TN_JIS			"iso-2022-jp"
#define TN_JIS1			"cp50221"
#define TN_JIS2			"iso-2022-jp-2004"
#define TN_JIS3			"iso-2022-jp-3"
#define TN_SJIS			"shift_jis"
#define TN_SJIS1			"cp932"
#define TN_SJIS2			"shift_jis-2004"
#define TN_EUC			"euc-jp"
#define TN_EUC1			"cp20932"
#define TN_EUC2			"cp51932"
#define TN_EUC3			"eucJP-ms"
#define TN_EUC4			"euc-jis-2004"
#define TN_UTF16BE		"utf-16be"
#define TN_UTF16BE_BOM	"utf-16be_bom"
#define TN_UTF16LE		"utf-16le"
#define TN_UTF16LE_BOM	"utf-16le_bom"
#define TN_UTF8			"utf-8"
#define TN_UTF8_BOM		"utf-8_bom"
#define TN_UTF32BE		"utf-32be"
#define TN_UTF32BE_BOM	"utf-32be_bom"
#define TN_UTF32LE		"utf-32le"
#define TN_UTF32LE_BOM	"utf-32le_bom"

/* タイプ名取得 */
char *_get_type_name(char *str, int type) {
	char *p;
	static TypeNameInfo info[] = {
			{ DCS_ND_ASCII, 	TN_ASCII, },
			{ DCS_ND_JIS, 		TN_JIS, },
			{ DCS_ND_JIS1, 		TN_JIS1, },
			{ DCS_ND_JIS2, 		TN_JIS2, },
			{ DCS_ND_JIS3, 		TN_JIS3, },
			{ DCS_ND_SJIS,		TN_SJIS, },
			{ DCS_ND_SJIS1,		TN_SJIS1, },
			{ DCS_ND_SJIS2,		TN_SJIS2, },
			{ DCS_ND_EUC,		TN_EUC, },
			{ DCS_ND_EUC1,		TN_EUC1, },
			{ DCS_ND_EUC2,		TN_EUC2, },
			{ DCS_ND_EUC3,		TN_EUC3, },
			{ DCS_ND_EUC4,		TN_EUC4, },
			{ DCS_ND_UTF16BE,	TN_UTF16BE, },
			{ DCS_ND_UTF16LE,	TN_UTF16LE, },
			{ DCS_ND_UTF8,		TN_UTF8, },
			{ DCS_ND_UTF32BE,	TN_UTF32BE, },
			{ DCS_ND_UTF32LE,	TN_UTF32LE, },
	};
	switch(type) {
	case DCS_UNKNOWN:
		strcpy(str, TN_UNKNOWN);
		break;
	case DCS_EMPTY:
		strcpy(str, TN_EMPTY);
		break;
	case DCS_ASCII:
		strcpy(str, TN_ASCII);
		break;
	case DCS_JIS:
		strcpy(str, TN_JIS);
		break;
	case DCS_JIS1:
		strcpy(str, TN_JIS1);
		break;
	case DCS_JIS2:
		strcpy(str, TN_JIS2);
		break;
	case DCS_JIS3:
		strcpy(str, TN_JIS3);
		break;
	case DCS_SJIS:
		strcpy(str, TN_SJIS);
		break;
	case DCS_SJIS1:
		strcpy(str, TN_SJIS1);
		break;
	case DCS_SJIS2:
		strcpy(str, TN_SJIS2);
		break;
	case DCS_EUC:
		strcpy(str, TN_EUC);
		break;
	case DCS_EUC1:
		strcpy(str, TN_EUC1);
		break;
	case DCS_EUC2:
		strcpy(str, TN_EUC2);
		break;
	case DCS_EUC3:
		strcpy(str, TN_EUC3);
		break;
	case DCS_EUC4:
		strcpy(str, TN_EUC4);
		break;
	case DCS_UTF16BE:
		strcpy(str, TN_UTF16BE);
		break;
	case DCS_UTF16BE_BOM:
		strcpy(str, TN_UTF16BE_BOM);
		break;
	case DCS_UTF16LE:
		strcpy(str, TN_UTF16LE);
		break;
	case DCS_UTF16LE_BOM:
		strcpy(str, TN_UTF16LE_BOM);
		break;
	case DCS_UTF8:
		strcpy(str, TN_UTF8);
		break;
	case DCS_UTF8_BOM:
		strcpy(str, TN_UTF8_BOM);
		break;
	case DCS_UTF32BE:
		strcpy(str, TN_UTF32BE);
		break;
	case DCS_UTF32BE_BOM:
		strcpy(str, TN_UTF32BE_BOM);
		break;
	case DCS_UTF32LE:
		strcpy(str, TN_UTF32LE);
		break;
	case DCS_UTF32LE_BOM:
		strcpy(str, TN_UTF32LE_BOM);
		break;
	case DCS_ND_ASCII:
		sprintf(str, "%s%s", TN_ASCII, (_option & FLAG_SUPPRESS)? "": TN_MAYBE);
		break;
	case DCS_ND_JIS:
		sprintf(str, "%s%s", TN_JIS, (_option & FLAG_SUPPRESS)? "": TN_MAYBE);
		break;
	case DCS_ND_JIS1:
		sprintf(str, "%s%s", TN_JIS1, (_option & FLAG_SUPPRESS)? "": TN_MAYBE);
		break;
	case DCS_ND_JIS2:
		sprintf(str, "%s%s", TN_JIS2, (_option & FLAG_SUPPRESS)? "": TN_MAYBE);
		break;
	case DCS_ND_JIS3:
		sprintf(str, "%s%s", TN_JIS3, (_option & FLAG_SUPPRESS)? "": TN_MAYBE);
		break;
	case DCS_ND_SJIS:
		sprintf(str, "%s%s", TN_SJIS, (_option & FLAG_SUPPRESS)? "": TN_MAYBE);
		break;
	case DCS_ND_SJIS1:
		sprintf(str, "%s%s", TN_SJIS1, (_option & FLAG_SUPPRESS)? "": TN_MAYBE);
		break;
	case DCS_ND_SJIS2:
		sprintf(str, "%s%s", TN_SJIS2, (_option & FLAG_SUPPRESS)? "": TN_MAYBE);
		break;
	case DCS_ND_EUC:
		sprintf(str, "%s%s", TN_EUC, (_option & FLAG_SUPPRESS)? "": TN_MAYBE);
		break;
	case DCS_ND_EUC1:
		sprintf(str, "%s%s", TN_EUC1, (_option & FLAG_SUPPRESS)? "": TN_MAYBE);
		break;
	case DCS_ND_EUC2:
		sprintf(str, "%s%s", TN_EUC2, (_option & FLAG_SUPPRESS)? "": TN_MAYBE);
		break;
	case DCS_ND_EUC3:
		sprintf(str, "%s%s", TN_EUC3, (_option & FLAG_SUPPRESS)? "": TN_MAYBE);
		break;
	case DCS_ND_EUC4:
		sprintf(str, "%s%s", TN_EUC4, (_option & FLAG_SUPPRESS)? "": TN_MAYBE);
		break;
	case DCS_ND_UTF16BE:
		sprintf(str, "%s%s", TN_UTF16BE, (_option & FLAG_SUPPRESS)? "": TN_MAYBE);
		break;
	case DCS_ND_UTF16LE:
		sprintf(str, "%s%s", TN_UTF16LE, (_option & FLAG_SUPPRESS)? "": TN_MAYBE);
		break;
	case DCS_ND_UTF8:
		sprintf(str, "%s%s", TN_UTF8, (_option & FLAG_SUPPRESS)? "": TN_MAYBE);
		break;
	case DCS_ND_UTF32BE:
		sprintf(str, "%s%s", TN_UTF32BE, (_option & FLAG_SUPPRESS)? "": TN_MAYBE);
		break;
	case DCS_ND_UTF32LE:
		sprintf(str, "%s%s", TN_UTF32LE, (_option & FLAG_SUPPRESS)? "": TN_MAYBE);
		break;
	default:
		if (type > DCS_NORMAL) {
			int i;
			*str = 0;
			p = str;
			for(i=0; i<sizeof(info)/sizeof(info[0]); i++) {
				if ((info[i].type & type) != 0) {
					p += sprintf(p, "%s, ", info[i].name);
				}
			}
			if (p > str) {
				/* 最後の ", " を削除 */
				*(p - 2) = 0;
			} else {
				strcpy(str, "Unknown");
			}
		} else {
			strcpy(str, "Unknown");
		}
		break;
	}
	return str;
} /* _get_type_name */

/* １ファイル実行 */
int _execute(const char *fname) {
	int sts, sttype, rval, astype, ln_type;
	char str[512], str2[512];

	if (strcmp(fname, "-") == 0) {
	/* 標準入力の時 */
		_sname = (char *)_STDIN_NAME_;
	} else {
		sttype = _check_stat(fname);

		if (sttype != _DST_FILE) {
			/* "%s: 入力ファイルが存在しません - %s\n" */
			/* "%s: 入力ファイルは通常ファイルではありません - %s\n" */
			fprintf(stderr, change_locale(_cbuf, _messages[sttype==_DST_NOTHING? ERR_SFILE_NOTHING: ERR_SFILE_NOTFILE]), _pname, fname);
			return STS_ERROR;
		}
		_sname = (char *)fname;
	}
	if ((_option & FLAG_DTYPE) == 0) {
	/* 出力タイプがない時（出力しない時） */
		if (strcmp(fname, "-") == 0) {
		/* 標準入力の時 */
			_sname = (char *)_STDIN_NAME_;
			rval = dcs_judges_stream(_dcs, (_option & FLAG_STYPE_ALL)==0? NULL: &astype, (_option & FLAG_SLTYPE)==0? NULL: &ln_type, stdin, _dstype);
		} else {
			_sname = (char *)fname;
			rval = dcs_judges_file(_dcs, (_option & FLAG_STYPE_ALL)==0? NULL: &astype, (_option & FLAG_SLTYPE)==0? NULL: &ln_type, fname, _dstype);
		}
		if (rval < DCS_UNKNOWN) {
		/* 異常の時 */
			return _check_error(rval);
		}
		if ((_option & FLAG_STYPE_ALL) != 0) {
		/* 入力タイプをすべて表示の時 */
			rval = astype;
		}
		if (!DCS_CONFIRMED_CODE_TYPE(rval)) {
		/* 入力タイプが確定できなかった時 */
			if ((_option & FLAG_VERBOSE) != 0) {
				/* "入力タイプを確定できないので %s と見なします - %s\n", */
				fprintf(_gfp, change_locale(_cbuf, _messages[ERR_DECIDED]), _get_type_name(str, rval), _sname);
			}
			sts = STS_WARNING;
		}
		*str2 = 0;
		if ((_option & FLAG_SLTYPE) != 0) {
		/* 改行処理がある時 */
			if (rval == DCS_UNKNOWN) {
				strcpy(str2, " CR,LF-unknown");
				sts = STS_WARNING;
			} else {
				switch(ln_type) {
				case DCS_ECTYPE:
				/* 文字タイプを特定できない時 */
					if ((_option & FLAG_VERBOSE) != 0) {
						/* "%s: 入力タイプを確定できないので改行処理をスキップします - %s: %s\n", */
						fprintf(_gfp, change_locale(_cbuf, _messages[ERR_DECIDED_SKIP_LN]), _sname);
					}
					strcpy(str2, " CR,LF-unknown");
					sts = STS_WARNING;
					break;
				default:
					if ((ln_type & LTYPE_LF) != 0) {
						strcat(str2, " LF,");
					}
					if ((ln_type & LTYPE_CR) != 0) {
						strcat(str2, *str2 == 0? " CR,": "CR,");
					}
					if ((ln_type & LTYPE_CRLF) != 0) {
						strcat(str2, *str2 == 0? " CR/LF,": "CR/LF,");
					}
					if (*str2 == 0) {
						strcpy(str2, " LF,CR-nothing");
					} else {
						/* 末尾の ',' を削除 */
						*(str2 + strlen(str2) - 1) = 0;
					}
					if (ln_type != LTYPE_LF
					 && ln_type != LTYPE_CR
					 && ln_type != LTYPE_CRLF
					 && ln_type != LTYPE_NON) {
					/* 改行文字タイプが混在している時 */
						if ((_option & FLAG_VERBOSE) != 0) {
							/* "改行タイプが混在しています - %s: %s\n", */
							fprintf(_gfp, change_locale(_cbuf, _messages[ERR_LTYPE_MIXED]), _sname, str2);
						}
						sts = STS_WARNING;
					}
					break;
				}
			}
		}
		if (((_option & FLAG_MULTI_IN) != 0
		 && (_option & FLAG_NPPATH) == 0)
		 || (_option & FLAG_PPATH) != 0) {
        /* 入力ファイルが複数、かつファイル名非表示指示がない時 */
        /* 又はファイル名表示指示がある時 */
			/* "%s: %s%s\n", */
			fprintf(_gfp, change_locale(_cbuf, _messages[ERR_NAME]), _sname, _get_type_name(str, rval), str2);fflush(_gfp);
		} else {
			fprintf(_gfp, "%s%s\n", _get_type_name(str, rval), str2);fflush(_gfp);
		}
		return sts;
	} else {
	/* 出力タイプがある時（出力する時） */
		_exec_sts = STS_SUCCESS;
		if (strcmp(fname, "-") == 0) {
		/* 標準入力の時 */
			_sname = (char *)_STDIN_NAME_;
			rval = dcs_convert_stream(_dcs, stdin, _dstype, _dtype, _is_forced,
					_observer, (_option & FLAG_TEST)==0? _writer: NULL, NULL);
								/* テストの時は、writer == NULL */
		} else {
			_sname = (char *)fname;
			rval = dcs_convert_file(_dcs, fname, _dstype, _dtype, _is_forced,
					_observer, (_option & FLAG_TEST)==0? _writer: NULL, NULL);
								/* テストの時は、writer == NULL */
		}
		if (rval < DCS_UNKNOWN) {
		/* 異常の時 */
			return _check_error(rval);
		}
		if (get_no_char_count(_dcs) > 0) {
			if ((_option & FLAG_VERBOSE) != 0) {
/* "変換できない文字がありました（%d 文字：先頭 %d %04x：末尾 %d %04x）\n", */
				fprintf(_gfp, change_locale(_cbuf, _messages[ERR_NOCHAR]),
						get_no_char_count(_dcs), get_first_no_char_offset(_dcs) + 1, get_first_no_char(_dcs),
						get_last_no_char_offset(_dcs) + 1, get_last_no_char(_dcs),
						_get_type_name(str, _stype), _sname, _get_type_name(str2, _dtype), _dname);
			}
			_exec_sts = STS_WARNING;
		}
	}
	return _exec_sts;
} /* _execute */

/* 変換監視 */
int _observer(int sts, void *user_info) {
	char str[256], str2[256], str3[256];
	int i_ope;

	if (!DCS_CONFIRMED_CODE_TYPE(sts)) {
	/* 入力タイプが確定できなかった時 */
		/* "%s: 入力タイプを確定できないので処理をスキップします - %s: %s\n", */
		fprintf(_gfp, change_locale(_cbuf, _messages[ERR_DECIDED_SKIP]), _pname, _get_type_name(str, sts), _sname);
		_exec_sts = STS_ERROR;
		return DCS_OBS_STOP;
	}
	if ((_option & FLAG_VERBOSE) != 0) {
		i_ope = _check_ope(sts, _dtype);
		/* "%s(%s) -> %s(%s) %s\n" */
		fprintf(_gfp, change_locale(_cbuf, _messages[ERR_EXEC]),
			_sname, _get_type_name(str, sts),
			_dname, _get_type_name(str2, _dtype),
			change_locale(str3, _ope_msg[i_ope]));
	}
	if ((_option & FLAG_TEST) == 0) {
	/* テスト・モードではない時 */
		if ((_option & FLAG_DPATH) != 0) {
		/* 出力ファイル名がある時 */
			if (_get_stream(&_ofp, _dname) == DCS_SUCCESS) {
				if (_dstat != _DST_DIR) {
				/* 出力ファイル属性がディレクトリではない時 */
					_oname  = _dname;
				}
			} else {
			/* "%s: ファイル・オープン・エラー(%d: %s) - %s\n" */
				fprintf(stderr, change_locale(_cbuf, _messages[ERR_FILE_OPEN]), _pname, errno, strerror(errno), _dname);
				_exec_sts = STS_ERROR;
				return DCS_OBS_STOP;
			}
		}
	}
	_stype = sts;
	return DCS_OBS_CONTINUE;
}

int _check_ope(int stype, int dtype) {
	int cmp;

	if (stype == dtype
	 || (DCS_EUC <= stype && stype <= DCS_EUC4
	 && DCS_EUC <= dtype && dtype <= DCS_EUC4)) {
	/* 入出力のコードが等しい時 */
		return OPE_COPY;
	} else if ((cmp = dcs_cmp_bom(stype, dtype)) == CBOM_ADD) {
	/* 同一 UTF コードで BOM 追加の時 */
		return OPE_ADD_BOM;
	} else if (cmp == CBOM_DELETE) {
	/* 同一 UTF コードで BOM 削除の時 */
		return OPE_DELETE_BOM;
	} else {
		return OPE_CONV;
	}
}

/* 変換出力 */
int _writer(const char *buf, size_t size, void *user_info) {
	if (fwrite(buf, 1, size, _ofp) != size) {
		/* "%s: ファイル出力エラー(%d: %s) - %s\n" */
		fprintf(stderr, change_locale(_cbuf, _messages[ERR_FILE_WRITE]), _pname, errno, strerror(errno), _dname);
		return DCS_EWRITE;
	}
	return DCS_SUCCESS;
}

/* バッファー変換出力 */
int _buf_writer(const char *buf, size_t size, void *user_info) {
	memcpy(_bptr, buf, size);
	_bptr += size;
	*_bptr = 0;
	return DCS_SUCCESS;
}

/* ファイル・チェック */
int _check_stat(const char *path) {
    struct stat stbuf;

	if (stat(path, &stbuf) != 0) {
		return _DST_NOTHING;
	}
	return(S_ISREG(stbuf.st_mode)? _DST_FILE: S_ISDIR(stbuf.st_mode)? _DST_DIR: _DST_OTHER);
}

/* ストリーム取得 */
int _get_stream(FILE **fp, const char *dpath) {
	char *p, *pp;

	if (_dstat == _DST_DIR) {
		p = strrchr(_sname, '/');
		if (p == NULL) {
			p = strrchr(_sname, '\\');
		}
		if (p == NULL) {
			p = _sname;
		} else {
			p++;
		}
		sprintf(_dfname, "%s/%s", dpath, p);
		pp = _dfname;
	} else {
		pp = (char *)dpath;
	}
	_oname = pp;
	if (pp == _STDOUT_NAME_) {
		*fp = stdout;
	} else if ((*fp = fopen(pp, "wb")) == NULL) {
		/* "%s: ファイル・オープン・エラー(%d: %s) - %s\n" */
		fprintf(stderr, change_locale(_cbuf, _messages[ERR_FILE_OPEN]), _pname, errno, strerror(errno), _oname);
		return DCS_EWOPEN;
	}
	return DCS_SUCCESS;
} /* _get_stream */

/* 文字列コードを表示タイプに変換 */
char *change_locale(char *dest, const char *src) {
	/* デフォルトは UTF-8 */
	int locale = DCS_UTF8;
	size_t ssize = strlen(src);
	char str[128];

	if ((_option & FLAG_CTYPE) == 0) {
	/* 表示タイプ指示がない時 */
		char *p, *tp, *lenv = getenv("LANG");

		if (lenv == NULL) {
		/* LANG 環境変数がない時 */
			if ((p = getenv("TEMP")) == NULL) {
				p = getenv("TMP");
			}
			if (p != NULL
			 && isalpha(*p)
			 && *++p == ':'
			 && *++p == '\\') {
			/* TMP 環境変数がある時
			 * かつ、DOSドライブ表現の時
			 */
				lenv = "ms932";
			}
		}
		if (lenv != NULL) {
		/* LANG 環境変数がある時 */
			tp = str;
			for(p = lenv; *p; p++) {
				*tp++ = tolower(*p);
			}
			*tp = 0;
			if ((tp = strstr(str, "iso")) != NULL) {
				if (strstr(tp, "2022") != NULL) {
					locale = DCS_JIS;
				} else if (strstr(tp, "8859") != NULL) {
					locale = DCS_SJIS;
				}
			} else if (strstr(str, "euc") != NULL
			 || strstr(str, "ujis") != NULL) {
				locale = DCS_EUC;
			} else if (strstr(str, "ms") != NULL
			 || strstr(str, "windows") != NULL) {
				locale = DCS_SJIS;
			}
		}
	} else {
		locale = _dcs_ctype;
	}
	_buffer = _bptr = dest;

	switch(locale) {
	case DCS_JIS:
		/* 返却値チェックなし */
		dcs_convert_buffer(_dcs, src, ssize, DCS_UTF8, DCS_JIS, false, NULL, _buf_writer, NULL);
		break;
	case DCS_SJIS:
		/* 返却値チェックなし */
		dcs_convert_buffer(_dcs, src, ssize, DCS_UTF8, DCS_SJIS, false, NULL, _buf_writer, NULL);
		break;
	case DCS_EUC:
		/* 返却値チェックなし */
		dcs_convert_buffer(_dcs, src, ssize, DCS_UTF8, DCS_EUC, false, NULL, _buf_writer, NULL);
		break;
	case DCS_UTF8:
	default:
		strcpy(dest, src);
		break;
	}
	return dest;
} /* change_locale */

int _check_error(int sts) {
	switch(sts) {
/* コード判定エラー */
	case DCS_EALLOC:
		/* "%s: メモリー・アロケート・エラー - %s\n" */
		fprintf(stderr, change_locale(_cbuf, _messages[ERR_MALLOC]), _pname);
		return STS_ERROR;
	case DCS_ENOTEXISTS:
		/* "%s: このファイルは存在しません(%d: %s) - %s\n" */
		fprintf(stderr, change_locale(_cbuf, _messages[ERR_FILE_STAT]), _pname, errno, strerror(errno), _sname);
		return STS_ERROR;
	case DCS_EROPEN:
		/* "%s: ファイル・オープン・エラー(%d: %s) - %s\n" */
		fprintf(stderr, change_locale(_cbuf, _messages[ERR_FILE_OPEN]), _pname, errno, strerror(errno), _sname);
		return STS_ERROR;
	case DCS_EWOPEN:
		/* "%s: ファイル・オープン・エラー(%d: %s) - %s\n" */
		fprintf(stderr, change_locale(_cbuf, _messages[ERR_FILE_OPEN]), _pname, errno, strerror(errno), _oname);
		return STS_ERROR;
	case DCS_EREAD:
		/* "%s: ファイル入力エラー(%d: %s) - %s\n" */
		fprintf(stderr, change_locale(_cbuf, _messages[ERR_FILE_READ]), _pname, errno, strerror(errno), _sname);
		return STS_ERROR;
/* コード変換エラー */
	case DCS_EBOM:
		/* "%s: BOM が不正です - %s\n" */
		fprintf(stderr, change_locale(_cbuf, _messages[ERR_BOM]), _pname, _sname);
		return STS_ERROR;
	case DCS_EUEXPEOD:
		/* "%s: 予期せぬファイルの終わり - %s\n" */
		fprintf(stderr, change_locale(_cbuf, _messages[ERR_UEXPEOF]), _pname, _sname);
		return STS_ERROR;
	case DCS_ECHAR:
		/* "%s: 文字コードが不正です - %s\n" */
		fprintf(stderr, change_locale(_cbuf, _messages[ERR_ILLEGALCHAR]), _pname, _sname);
		return STS_ERROR;
	default:
		fprintf(stderr, change_locale(_cbuf, _messages[ERR_UNKNOWN]), _pname, sts, dcs_get_errno(_dcs), strerror(dcs_get_errno(_dcs)), _sname);
		return STS_ERROR;
	}
}

