/*
 * Dcs.java
 *
 * Copyright 2019-2026 Ichiji Tadokoro. All Rights Reserved.
 */

package rabbit.dcs;

import java.io.InputStream;

/**
 * &nbsp;&nbsp;&nbsp;&nbsp;<font size=+1><code>Dcs</code></font> は文字コード判定と文字コード変換用の DCS ライブラリーにアクセスするための JNI API です。アクセス方法はすべてパブリックなので、それ以外の <font size=+1><code>Dcs</code></font> のメソッドや変数、定数を参照する必要はありません。<!--
 *  -->不要になったら close() を呼び出して、この <font size=+1><code>Dcs</code></font> のすべてのリソースを破棄します。以下に対応するコードを示します。<br>
<br>
対応文字コード<br>
<pre>
	ISO-2022-JP
	CP50221(JIS)
	ISO-2022-JP-2004
	ISO-2022-JP-3
	Shift_JIS
	CP932(SJIS)
	Shift_JIS-2004
	EUC-JP
	CP20932(EUC-JP)
	CP51932(EUC-JP)
	eucJP-ms
	eucJP-2004
	UTF-8
	UTF-16BE
	UTF-16LE
	UTF-32BE
	UTF-32LE
	（BOM、サロゲート文字、日本語 UTF8-MAC 対応）
</pre>
<br>
<br>
<font size=+1><b>バッファーによるコード判定とコード変換の例</b></font><br>
<pre><code>
	package rabbit.hellodcs;

	import java.io.UnsupportedEncodingException;

	import rabbit.dcs.ConvertObserver;
	import rabbit.dcs.ConvertWriter;
	import rabbit.dcs.Dcs;
	import rabbit.dcs.DcsException;
	import rabbit.dcs.DcsStatus;

	public class HelloDcs {

		public HelloDcs() {
		}

		public static void main(String[] args) {
			Dcs dcs;
			try {
				byte[] buf = new String("皆さん、今日は！").getBytes("euc-jp");
				// Dcs 生成
				dcs = new Dcs();
				// バッファーによるコード判定
				dcs.judgesBuffer(null, null, buf, buf.length, Dcs.UNKNOWN);
				System.out.println(String.format("judgesBuffer sts: %d %s", dcs.sts, DcsStatus.typeToName(dcs.sts)));
				// 変換観察
				ConvertObserver observer = new ConvertObserver() {
					&#64;Override
					public int observe(int sts, Object userInfo) {
						System.out.println(String.format("observe sts: %d %s", sts, DcsStatus.typeToName(dcs.sts)));
						return Dcs.OBS_CONTINUE;
					}};
				// 変換書き込み
				ConvertWriter writer = new ConvertWriter() {
					&#64;Override
					public int write(byte[] buffer, int length, Object userInfo) {
						try {
							System.out.println(new String(buffer, "utf-8"));
						} catch (UnsupportedEncodingException e) {
							e.printStackTrace();
							return Dcs.EWRITE;
						}
						return Dcs.SUCCESS;
					}};
				// バッファーによるコード変換
				dcs.convertBuffer(buf, buf.length, Dcs.UNKNOWN, Dcs.UTF8, false, observer, writer, null);
				// Dcs クローズ
				dcs.close();
			} catch (DcsException | UnsupportedEncodingException e) {
				e.printStackTrace();
				System.exit(1);
			}
			System.exit(0);
		}
	}
</code>
</pre>
<br>
<br>
実行結果<br>
<pre>
	judgesBuffer sts: 9 EUC-JP
	observe sts: 9 EUC-JP
	皆さん、今日は！
</pre>
<br>
<br>
<font size=+1><b>文字コード判定と変換用のメソッド</b></font><br>
 *  <BR>
 *  &nbsp;&nbsp;&nbsp;&nbsp;文字コードの判定、変換には入力データの形式（バッファー、ストリーム、ファイル）により３つ、計６つのメソッドがあります。<br>
<br>
<pre>
文字コード判定
	public int judgesBuffer(int[] atype, int[] lnType, byte[] buf, int length, int stype)
	public int judgesStream(int[] atype, int[] lnType, InputStream istream, int stype)
	public int judgesFile(int[] atype, int[] lnType, String path, int stype)

文字コード変換
	public int convertBuffer(byte[] buf, int length, int stype, int dtype, boolean isForced,
                                ConvertObserver observer, ConvertWriter writer, Object userInfo)
	public int convertStream(InputStream istream, int stype, int dtype, boolean isForced,
                                ConvertObserver observer, ConvertWriter writer, Object userInfo)
	public int convertFile(String path, int stype, int dtype, boolean isForced,
                                ConvertObserver observer, ConvertWriter writer, Object userInfo)
</pre>
<br>
<br>
<font size=+1><b>変換不能文字用のメソッド</b></font><br
 *  ><br
 *  >
 * &nbsp;&nbsp;&nbsp;&nbsp;文字コード変換では、変換後の文字コードがない場合は変換不能として扱われ、それらの個数や先頭、末尾の値あるいは位置を以下のメソッドで知ることができます。<br>
<br>
<pre>
変換不能文字の扱い
	public int getNoCharCount()
	public long getFirstNoChar()
	public long getLastNoChar()
	public int getFirstNoCharIndex()
	public int getLastNoCharIndex()


</pre>
<font size=+1><b>改行コード判定と設定用のメソッド</b></font><br
 *  ><br
 *  >
&nbsp;&nbsp;&nbsp;&nbsp;バッファによるコード変換 convertBuffer() 等で改行コードの編集を行なう場合、あらかじめ
setLtype() で置き換えるコードを設定しておきます。初期値は LTYPE_NON で、改行コードの編集は行ないません。<br
 *  >
<pre><code>
改行コード・タイプの取得と設定
	public int getLtype()
	public void setLtype(int ltype)
</code></pre>
<br>
<br>
<font size=+1><b>出力コードの半角カナから全角への変換の判定と設定用のメソッド</b></font><br
 *  ><br
 *  >
&nbsp;&nbsp;&nbsp;&nbsp;バッファによるコード変換 convertBuffer() 等で半角カナから全角への変換の真偽を変更する場合、あらかじめ
setHankanaTo() で設定しておきます。初期値は偽で、半角カナから全角への変換を行ないません。<br
 *  >
<pre><code>
半角カナから全角への変換の判定と設定
	public boolean isHankanaTo()
	public void setHankanaTo(boolean flag)
</code></pre>
<br>
<br>
<font size=+1><b>日本語 UTF8-MAC １文字変換の真偽判定と設定及び、UTF8-MAC コード有無判定用のメソッド</b></font><br
 *  ><br
 *  >
&nbsp;&nbsp;&nbsp;&nbsp;バッファによるコード変換  convertBuffer() 等で日本語 UTF8-MAC １文字変換の真偽を変更する場合、あらかじめ
setUtfmacTo() で設定しておきます。初期値は偽で、日本語 UTF8-MAC １文字変換を行ないません。<br
 *  >
<pre><code>
日本語 UTF8-MAC １文字変換の判定と設定
	public boolean isUtfmacTo()
	public void setUtfmacTo(bool flag)
</code>
</pre><br>
&nbsp;&nbsp;&nbsp;&nbsp;バッファによるコード判定 judgesBuffer() やバッファによるコード変換 convertBuffer() 等の後に、入力データの日本語 UTF8-MAC コード有無の判定を行ないます。<br
 *  >
<pre><code>
入力データの日本語 UTF8-MAC コード有無の判定
	public boolean utfmacExists()
</code>
</pre>
<br>
<br>
<font size=+1><b>不正文字の出力抑止の真偽判定と設定用のメソッド</b></font><br
 *  ><br
 *  >
&nbsp;&nbsp;&nbsp;&nbsp;バッファによるコード変換  convertBuffer() 等で不正文字の出力抑止の真偽を変更する場合、あらかじめ
setSuppress() で設定しておきます。初期値は偽で、不正文字を出力します。<br
 *  >
<pre><code>

不正文字出力抑止の判定と設定
	public boolean isSuppress()
	public void setSuppress(bool isSuppress)
</code>
</pre>
<br>
<font size=+1><b>その他の文字コード判定と設定用のメソッド</b></font><br
 *  ><br
 *  >
&nbsp;&nbsp;&nbsp;&nbsp;ここからは Dcs API を使用するために必須ではありませんが、文字コードを扱う上で有用なものが定義されています。<br
 *  >
<pre><code>

半角カナ判定
	public static boolean isJisKana(int c)
	public static boolean isSjisKana(int c)
	public static boolean isUnibeKana(int c)

Unicode, UTF-8 変換
	public static int unibeToUtf8Short(byte[] dbuf, int scd)
	public static int utf8ToUnibeBytes(byte[] dbuf, int doffset, byte[] sbuf, int[] soffset)

Unicodeサロゲート文字、UTF-8 変換
	public void unibeSurrogateToUtf8(byte[] dbuf, int hcd, int lcd)
	public void unibeSurrogateToCd(int[] dcd, int hcd, int lcd)

半角カナ変換
	public static int jisKanaToSjis(int c)
	public static int jisKanaToUnibe(int c)
	public static int sjisKanaToJis(int c)
	public static int sjisKanaToUnibe(int c)
	public static int unibeKanaToJis(int c)
	public static int unibeKanaToSjis(int c)
	</code></pre>
 *
 * @author Ichiji Tadokoro
 */
public class Dcs implements DcsStatus, AutoCloseable {

	/**
	 * この <font size=+1>Dcs</font> のバージョンです。
	 *
	 * @see #nversion
	 * @see #getVersion
	 */
	public static final float CVERSION = ((float)3.00);

	/**
	 * ネイティブ Dcs API のバージョンです。Dcs クラスのロード時に設定されます。
	 *
	 * @see #CVERSION
	 * @see #getVersion
	 */
	public static float nversion;

	/**
	 * 最後の処理ステータスです。
	 */
	public int sts;

	/**
	 *  確定コード・タイプの範囲内であることを判定します。
	 * @param sts 判定を行なう処理ステータス
	 * @return 指定したステータスが確定コード・タイプの範囲内であれば真、それ以外は偽を返します。
	 * @since 2.0
	 */
	public static boolean CONFIRMED_CODE_TYPE(int sts) {
		return (EMPTY < sts && sts <= NORMAL);
	}

	long _hdcsInfo;

	static final int _LINUX = 0;

	static final int _WINDOWS = 1;

	static final boolean _32BITS = false;

	static int _os_value = (System.getProperty("os.name").toLowerCase().startsWith("windows")? _WINDOWS: _LINUX);

	/**
	 * LINUXバージョン判定
	 *
	 * @return このクラスがLINUXバージョンの時は真、それ以外は偽を返します。
	 * @since 1.1
	 */
	public static boolean isLinux() {
		return (_os_value == _LINUX);
	}

	/**
	 * Windowsバージョン判定
	 *
	 * @return このクラスがWindowsバージョンの時は真、それ以外は偽を返します。
	 * @since 2.0
	 */
	public static boolean isWindows() {
		return (_os_value == _WINDOWS);
	}

	/**
	 * アーキテクチャのビット数判定
	 *
	 * @return このクラスのアーキテクチャ・ビット数を返します。
	 * @since 1.1
	 */
	public static int getBits() {
		return(_32BITS? 32: 64);
	}

	/**
	 * <font size=+1><code>System.loadLibrary</code></font> でネイティブ Dcs API jdcs
	 * をロードします。
	 */
	static {
		String jpath = (isLinux()? "dcs": _32BITS? "jdcsw32": "jdcsw64");
		try {
			System.loadLibrary(jpath);
			nversion = getVersion();
		} catch(UnsatisfiedLinkError e) {
			e.printStackTrace();
		}
	}

	/**
	 * ネイティブ Dcs を生成します。必要なくなれば Dcs.close() を呼び出して Dcs の持つすべてのリソースを破棄して下さい。何らかの理由で Dcs が生成されない場合は DcsException がスローされます。
	 *
	 * @throws DcsException 何らかの理由で Dcs が生成されない場合は DcsException がスローされます。
	 */
	public Dcs() throws DcsException {
		_createDcs();
		if (_hdcsInfo == 0) {
			throw new DcsException("ネイティブ Dcs を生成できません");
		}
	}

    /**
	 * この Dcs の持つすべてのリソースを破棄します。
	 *
	 * @see finalize
	 */
	@Override
	public void close() {
		try {
			synchronized (this) {
				if (_hdcsInfo != 0) {
					_releaseDcs();
				}
			}
		} catch(Exception e) {
			e.printStackTrace();
		}
	}

	/*
	 * ネイティブ Dcs を生成します。アプリケーションで呼び出す事はありません。
	 */
	public native int _createDcs();

	/*
	 * この Dcs の持つすべてのリソースを破棄します。アプリケーションで呼び出す事はありません。
	 *
	 * @see close
	 */
	public native void _releaseDcs();

	/**
	 * ネイティブ Dcs API のバージョンを取得します。<font size=+1>{@link #nversion
	 * nversion}</font> と同値です。
	 *
	 * @return ネイティブ Dcs API のバージョン
	 *
	 * @see #CVERSION
	 * @see #nversion
	 */
	public native static float getVersion();

	/**
	 *  JIS半角カナの判定
	 *
	 * @param c 真偽を判定する文字コード
	 * @return 指定した文字コードがこの文字コードセットの半角カナに含まれる時は真、それ以外は偽を返します。
	 */
	public static boolean isJisKana(int c) {
		/* JIS半角カナ */
		return ((c)>=0x21&&(c)<=0x5f);
	}

	/**
	 *  SJIS半角カナの判定
	 *
	 * @param c 真偽を判定する文字コード
	 * @return 指定した文字コードがこの文字コードセットの半角カナに含まれる時は真、それ以外は偽を返します。
	 */
	public static boolean isSjisKana(int c) {
		/* SJIS半角カナ */
		return ((c)>=0xa1&&(c)<=0xdf);
	}

	/**
	 *  Unicose(BE)半角カナの判定
	 *
	 * @param c 真偽を判定する文字コード
	 * @return 指定した文字コードがこの文字コードセットの半角カナに含まれる時は真、それ以外は偽を返します。
	 */
	public static boolean isUnibeKana(int c) {
		/* Unicose(BE)半角カナ */
		return ((c)>=0xff61&&(c)<=0xff9f);
	}

	/**
	 *  JIS全角仮名の判定
	 *
	 * @param c 真偽を判定する文字コード
	 * @return 指定した文字コードがこの文字コードセットの全角平仮名、又は全角片仮名に含まれる時は真、それ以外は偽を返します。
	 */
	public static boolean isJisZenkana(int c) {
		/* JIS全角仮名 */
		return (JIS_HKANA_B <= (c) && (c) <= JIS_HKANA_E
				|| JIS_KKANA_B <= (c) && (c) <= JIS_KKANA_E);
	}

	/**
	 *  シフトJIS全角仮名の判定
	 *
	 * @param c 真偽を判定する文字コード
	 * @return 指定した文字コードがこの文字コードセットの全角平仮名、又は全角片仮名に含まれる時は真、それ以外は偽を返します。
	 */
	public static boolean isSjisZenkana(int c) {
		/* シフトJIS全角仮名 */
		return (SJIS_HKANA_B <= (c) && (c) <= SJIS_HKANA_E
				|| SJIS_KKANA_B <= (c) && (c) <= SJIS_KKANA_E);
	}

	/**
	 *  EUC全角仮名の判定
	 *
	 * @param c 真偽を判定する文字コード
	 * @return 指定した文字コードがこの文字コードセットの全角平仮名、又は全角片仮名に含まれる時は真、それ以外は偽を返します。
	 */
	public static boolean isEucZenkana(int c) {
		/* EUC全角仮名 */
		return (EUC_HKANA_B <= (c) && (c) <= EUC_HKANA_E
				|| EUC_KKANA_B <= (c) && (c) <= EUC_KKANA_E);
	}

	/**
	 *  Unicode(BE)全角仮名の判定
	 *
	 * @param c 真偽を判定する文字コード
	 * @return 指定した文字コードがこの文字コードセットの全角平仮名、又は全角片仮名に含まれる時は真、それ以外は偽を返します。
	 */
	public static boolean isUnibeZenkana(int c) {
		/* Unicode(BE)全角仮名 */
		return (UNIBE_HKANA_B <= (c) && (c) <= UNIBE_HKANA_E
				|| UNIBE_KKANA_B <= (c) && (c) <= UNIBE_KKANA_E);
	}

	/**
	 * 変換後も ASCII である出力タイプの判定
	 *
	 * @param dtype 真偽を判定する出力タイプ
	 * @return ASCII から変換を掛けても ASCII のまま変わらない出力タイプの時は真、それ以外は偽を返します。
	 * @since 2.11
	 */
	public static boolean remainsAscii(int dtype) {
		return(dtype < UTF16BE || dtype == UTF8);
	}

    /**
     *  Unicode(BE)(16bit)からUTF-8への変換。出力バッファのサイズ・チェックはしないため十分な領域を確保する必要があります。
     *
	 * @param dbuf 変換後の文字コードを格納するバッファ
	 * @param scd 変換前の文字コード
	 * @return dbuf へ格納したバイト数を返します。
	 */
	public static int unibeToUtf8Short(byte[] dbuf, int scd) {
    	if (scd <= 0x007f) {
    		dbuf[0] = (byte) scd;
    		return 1;
    	} else if (scd <= 0x7ff) {
    		dbuf[0] = (byte) ((0x06<<5)|(scd>>6));
    		dbuf[1] = (byte) ((0x02<<6)|(scd&0x3f));
    		return 2;
    	} else	{
    		dbuf[0] = (byte) ((0x0e<<4)|((scd>>12)&0x0f));
    		dbuf[1] = (byte) ((0x02<<6)|((scd>>6)&0x3f));
    		dbuf[2] = (byte) ((0x02<<6)|(scd&0x3f));
    		return 3;
    	}
    }

	/**
	 *  UTF-8コードからUnicode(BE)への変換。入力バイト数が４バイトならサロゲート文字であり出力も４バイトとなり、
	 *  その他の出力はすべて２バイトです。出力バッファのサイズ・チェックはしないため十分な領域を確保する必要があります。
	 *
	 * @param dbuf 変換後の文字コードを格納するバッファ
	 * @param doffset dbuf へ格納する開始位置
	 * @param sbuf 変換前の文字コードを取り出すバッファ
	 * @param soffset sbuf から取り出す開始位置。処理したバイト数が加算されます。
	 * @return 変換が成功した時は SUCCESS、不正なコードがあった時は
	 *  ECHAR、コードの途中で終わっている時は EUEXPEOD を返します。
	 */
	public static int utf8ToUnibeBytes(byte[] dbuf, int doffset, byte[] sbuf, int[] soffset) {
		int sts;
		int c, c2, c3, c4, ch;
		long lch;

		if (soffset[0] >= sbuf.length) {
			sts = EUEXPEOD;
		} else {
			c = sbuf[soffset[0]++];
			if ((c & 0x80) == 0) {
				/* 0xxxxxxx: １バイト文字 */
				dbuf[0] = 0;
				dbuf[1] = (byte) c;
				sts = SUCCESS;
			} else if (soffset[0] >= sbuf.length) {
				sts = EUEXPEOD;
			} else {
				c2 = sbuf[soffset[0]++];
				if ((c & 0xe0) == 0xc0) {
					if ((c2 & 0xc0) == 0x80) {
					/* 110xxxxx 10xxxxxx: ２バイト文字 */
						ch = ((c&0x1f)<<6|(c2&0x3f));
						dbuf[doffset] = (byte) ((ch>>8));
						dbuf[doffset + 1] = (byte) ch;
						sts = SUCCESS;
					} else {
						sts = ECHAR;
					}
				} else if (soffset[0] >= sbuf.length) {
					sts = EUEXPEOD;
				} else {
					c3 = sbuf[soffset[0]++];
					if ((c & 0xf0) == 0xe0 && (c2 & 0xc0) == 0x80 && (c3 & 0xc0) == 0x80) {
						/* 1110xxxx 10xxxxxx 10xxxxxx: ３バイト文字 */
						ch = ((c&0x0f)<<12|(c2&0x3f)<<6|(c3&0x3f));
						dbuf[doffset] = (byte) ((ch>>8));
						dbuf[doffset + 1] = (byte) ch;
						sts = SUCCESS;
					} else if (soffset[0] >= sbuf.length) {
						sts = EUEXPEOD;
					} else {
						c4 = sbuf[soffset[0]++];
						if (((c) & 0xf8) == 0xf0 && ((c2) & 0xc0) == 0x80 && ((c3) & 0xc0) == 0x80 && ((c4) & 0xc0) == 0x80) {
						/* 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx: ４バイト文字 */
							lch = (((c & 0x07) << 18)|((c2 & 0x3f) << 12)|((c3 & 0x3f) << 6)|(c4 & 0x3f));
							lch -= 0x10000;
							ch = (int)(((lch >> 10)|0xd800)&0x7fffffff);
							dbuf[doffset] = (byte) ((ch>>8));
							dbuf[doffset + 1] = (byte) ch;
							ch = (int)(((lch & 0x3ff)|0xdc00)&0x7fffffff);
							dbuf[doffset + 2] = (byte) ((ch>>8));
							dbuf[doffset + 3] = (byte) ch;
							sts = SUCCESS;
						} else {
							sts = ECHAR;
						}
					}
				}
			}
		}
		return sts;
	}

	/** Unicode(BE)サロゲート・ペアからUTF-8への変換。
	 *  出力は４バイトとなりますが、出力バッファのサイズ・チェックはしないため十分な領域を確保する必要があります。
	 * @param dbuf 変換後の文字コードを格納するバッファ
	 * @param hcd 変換前の上位文字コード
	 * @param lcd 変換前の下位文字コード
	 * @since 2.0
	 */
	public void unibeSurrogateToUtf8(byte[] dbuf, int hcd, int lcd) {
		long _lcd_ = (0x10000 + (((hcd & ~0xd800)<<10) | (lcd & ~0xdc00)));
		dbuf[0] = (byte) (0xf0 | (_lcd_ >> 18));
		dbuf[1] = (byte) (0x80 |((_lcd_ >> 12) & 0x3f));
		dbuf[2] = (byte) (0x80 |((_lcd_ >>  6) & 0x3f));
		dbuf[3] = (byte) (0x80 |(_lcd_ & 0x3f));
	}

	/**
	 *  Unicode(BE)サロゲート・ペアからコードへの変換。
	 * @param dcd 変換後の文字コードを格納するバッファ
	 * @param hcd 変換前の上位文字コード
	 * @param lcd 変換前の下位文字コード
	 * @since 2.0
	 */
	public void unibeSurrogateToCd(int[] dcd, int hcd, int lcd) {
		dcd[0] = (0x10000 + (((hcd & ~0xd800)<<10) | (lcd & ~0xdc00)));
	}

	/*-半角カナ・コード変換------------------------------*/

	/**
	 *  JIS半角カナ・コードからシフトJISコードへの変換
	 *
	 * @param c 変換対象の文字コード
	 * @return 変換後の文字コード。該当がない時は不正値を返します。
	 */
	public static int jisKanaToSjis(int c) {
		return ((c) | 0x80);
	}

	/**
	 *  JIS半角カナ・コードからUnicode(BE)コードへの変換
	 *
	 * @param c 変換対象の文字コード
	 * @return 変換後の文字コード。該当がない時は不正値を返します。
	 */
	public static int jisKanaToUnibe(int c) {
		return (0xff00 | (((c)+0x40) & 0xff));
	}


	/**
	 *  シフトJIS半角カナ・コードからJISコードへの変換
	 *
	 * @param c 変換対象の文字コード
	 * @return 変換後の文字コード。該当がない時は不正値を返します。
	 */
	public static int sjisKanaToJis(int c) {
		return ((c) & ~0x80);
	}

	/**
	 *  シフトJIS半角カナ・コードからUnicode(BE)コードへの変換
	 *
	 * @param c 変換対象の文字コード
	 * @return 変換後の文字コード。該当がない時は不正値を返します。
	 */
	public static int sjisKanaToUnibe(int c) {
		return (0xff00 | (((c) | 0x80) - 0x40));
	}


	/**
	 *  Unicode(BE)半角カナ・コードからJISコードへの変換
	 *
	 * @param c 変換対象の文字コード
	 * @return 変換後の文字コード。該当がない時は不正値を返します。
	 */
	public static int unibeKanaToJis(int c) {
		return (((c)-0x40) & 0x7f);
	}

	/**
	 *  Unicode(BE)半角カナ・コードからシフトJISコードへの変換
	 *
	 * @param c 変換対象の文字コード
	 * @return 変換後の文字コード。該当がない時は不正値を返します。
	 */
	public static int unibeKanaToSjis(int c) {
		return ((((c)-0x40) | 0x80) & 0xff);
	}

	/**
	 *  &nbsp;&nbsp;&nbsp;&nbsp;入力バッファ buffer、入力サイズ length のデータから文字コードの判定を行ないます。<br>
	 *  &nbsp;&nbsp;&nbsp;&nbsp;atype は未確定の文字コード・タイプ（CONFIRMED_CODE_TYPE(sts)が偽）がある時にはその値、
	 *  一意に決まればリターン値と同じになり、エラーの場合は UNKNOWN(-1) が設定されます。
	 *  不要ならば null を設定します。<br>
	 *  &nbsp;&nbsp;&nbsp;&nbsp;lnType は 入力データの改行コード・タイプであり、LTYPE_NON, LTYPE_LF, LTYPE_CR, LTYPE_CRLF, LTYPE_ALL
	 *  の何れかが混在していた時はそれらの OR 値が設定されます。
	 *  不要ならば null を設定します。<br>
	 *  &nbsp;&nbsp;&nbsp;&nbsp;stype は文字コードが一意に決まらなかった場合に、同一の文字コード・タイプがあれば、その値に決定します。
	 *  不要ならば UNKNOWN を設定します。<br>
	 *
	 * @param atype 未確定の文字コード・タイプ
	 * @param lnType 改行コード・タイプ
	 * @param buffer 入力バッファ
	 * @param length 入力サイズ
	 * @param stype 入力データの文字コード・タイプ
	 * @return 判定した文字コード・タイプを返します。
	 * メモリー・アロケート・エラーの時は EALLOC。
	 * @since 2.0
	 */
	public native int judgesBuffer(int[] atype, int[] lnType, byte[] buffer, int length, int stype);

	/**
	 *  入力ストリーム istream のデータから文字コードの判定を行ない、istream は呼び出し側でクローズします。<br>
	 *  &nbsp;&nbsp;&nbsp;&nbsp;atype は未確定の文字コード・タイプ（CONFIRMED_CODE_TYPE(sts)が偽）がある時にはその値、
	 *  一意に決まればリターン値と同じになり、エラーの場合は UNKNOWN(-1) が設定されます。
	 *  不要ならば null を設定します。<br>
	 *  &nbsp;&nbsp;&nbsp;&nbsp;lnType は 入力データの改行コード・タイプであり、LTYPE_NON, LTYPE_LF, LTYPE_CR, LTYPE_CRLF, LTYPE_ALL
	 *  の何れかが混在していた時はそれらの OR 値が設定されます。
	 *  不要ならば null を設定します。<br>
	 *  &nbsp;&nbsp;&nbsp;&nbsp;stype は文字コードが一意に決まらなかった場合に、同一の文字コード・タイプがあれば、その値に決定します。
	 *  不要ならば UNKNOWN を設定します。<br>
	 *
	 * @param atype 未確定の文字コード・タイプ
	 * @param lnType 改行コード・タイプ
	 * @param istream 入力ストリーム
	 * @param stype 入力データの文字コード・タイプ
	 * @return 判定した文字コード・タイプを返します。
	 * メモリー・アロケート・エラーの時は EALLOC、
	 * ファイル入力エラーの時は EREAD。
	 * @since 2.0
	 */
	public native int judgesStream(int[] atype, int[] lnType, InputStream istream, int stype);

	/**
	 *  &nbsp;&nbsp;&nbsp;&nbsp;入力ファイル path のデータから文字コードの判定を行ないます。<br>
	 *  &nbsp;&nbsp;&nbsp;&nbsp;atype は未確定の文字コード・タイプ（CONFIRMED_CODE_TYPE(sts)が偽）がある時にはその値、
	 *  一意に決まればリターン値と同じになり、エラーの場合は UNKNOWN(-1) が設定されます。
	 *  不要ならば null を設定します。<br>
	 *  &nbsp;&nbsp;&nbsp;&nbsp;lnType は 入力データの改行コード・タイプであり、LTYPE_NON, LTYPE_LF, LTYPE_CR, LTYPE_CRLF, LTYPE_ALL
	 *  の何れかが混在していた時はそれらの OR 値が設定されます。
	 *  不要ならば null を設定します。<br>
	 *  &nbsp;&nbsp;&nbsp;&nbsp;stype は文字コードが一意に決まらなかった場合に、同一の文字コード・タイプがあれば、その値に決定します。
	 *  不要ならば UNKNOWN を設定します。<br>
	 *
	 * @param atype 未確定の文字コード・タイプ
	 * @param lnType 改行コード・タイプ
	 * @param path 入力ファイル・パス
	 * @param stype 入力データの文字コード・タイプ
	 * @return 判定した文字コード・タイプを返します。
	 * メモリー・アロケート・エラーの時は EALLOC、
	 * ファイル・オープン・エラーの時は EROPEN、
	 * ファイル入力エラーの時は EREAD。
	 * @since 2.0
	 */
	public int judgesFile(int[] atype, int[] lnType, String path, int stype) {
		return judgesFile(atype, lnType, path.getBytes(), stype);
	}

	/**
	 *  &nbsp;&nbsp;&nbsp;&nbsp;入力ファイル path のデータから文字コードの判定を行ないます。<br>
	 *  &nbsp;&nbsp;&nbsp;&nbsp;atype は未確定の文字コード・タイプ（CONFIRMED_CODE_TYPE(sts)が偽）がある時にはその値、
	 *  一意に決まればリターン値と同じになり、エラーの場合は UNKNOWN(-1) が設定されます。
	 *  不要ならば null を設定します。<br>
	 *  &nbsp;&nbsp;&nbsp;&nbsp;lnType は 入力データの改行コード・タイプであり、LTYPE_NON, LTYPE_LF, LTYPE_CR, LTYPE_CRLF, LTYPE_ALL
	 *  の何れかが混在していた時はそれらの OR 値が設定されます。
	 *  不要ならば null を設定します。<br>
	 *  &nbsp;&nbsp;&nbsp;&nbsp;stype は文字コードが一意に決まらなかった場合に、同一の文字コード・タイプがあれば、その値に決定します。
	 *  不要ならば UNKNOWN を設定します。<br>
	 *
	 * @param atype 未確定の文字コード・タイプ
	 * @param lnType 改行コード・タイプ
	 * @param apath 入力ファイル・パスのバイト配列
	 * @param stype 入力データの文字コード・タイプ
	 * @return 判定した文字コード・タイプを返します。
	 * メモリー・アロケート・エラーの時は EALLOC、
	 * ファイル・オープン・エラーの時は EROPEN、
	 * ファイル入力エラーの時は EREAD。
	 * @since 2.0
	 */
	public native int judgesFile(int[] atype, int[] lnType, byte[] apath, int stype);

	/**
	 *  引き数で与えられた文字コード変換で起きる BOM の追加、削除の判定を行います。同じ文字コードであれば追加や削除になり、それ以外は非となります。UTF16BE から UTF16BE_BOM への変換は CBOM_ADD、UTF16BE から UTF16LE_BOM への変換は CBOM_NON を返します。
	 *
	 * @param stype 変換前の文字コード
	 * @param dtype 変換後の文字コード
	 * @return BOM 追加／削除の判定値（CBOM_NON、CBOM_ADD、CBOM_DELETE の何れか）を返します。
	 */
	public native static int cmpBom(int stype, int dtype);

    /**
	 *  &nbsp;&nbsp;&nbsp;&nbsp;入力バッファ buffer、入力サイズ length のデータを指定された文字コードに変換します。
     *  初めに入力データの文字コード判定を行ない、
     *  一意に決定した場合は指定された dtype の文字コードに変換します。<br>
     *  &nbsp;&nbsp;&nbsp;&nbsp;stype は judgesBuffer() 等のコード判定だけを行なうメソッドと同様に、
     *  文字コードが一意に決まらなかった場合に、
     *  同一の文字コード・タイプがあれば、その値に決定します。
     *  不要ならば UNKNOWN を設定します。isForced が真ならば、Dcs.UNKNOWN 以外の stype の値が強制的に採られます。ただし、stype や実際の文字コードが JIS や BOM 付きの UTF の場合は無効となります。<br>
     *  &nbsp;&nbsp;&nbsp;&nbsp;observer のメソッド observe() は変換前のコード判定が終了した時点で、渡される判定結果 sts により必ず１回呼び出されます。
     *  継続ならば OBS_CONTINUE、中止する時は OBS_STOP で戻ります。
     *  致命的なエラーや入力データの文字コード・タイプが確定できなかった（CONFIRMED_CODE_TYPE(sts)が偽）場合は判定結果が
     *  convertBuffer() の返却値となり、observe() が OBS_CONTINUE を返しても継続されません。
     *  変換結果をファイルに出力する場合は observe() がそのファイルをオープンし、convertBuffer() の返った後、呼び出し側でクローズします。
     *  observer のメソッド observe() が不要ならば null を設定します。<br>
     *  &nbsp;&nbsp;&nbsp;&nbsp;writer のメソッド write() は変換中に編集バッファがフルになる度に呼び出され、
     *  渡される出力バッファ buffer、出力サイズ length により実際の出力を行ないます。
     *  出力が成功したら SUCCESS、ファイル出力エラーなら EWRITE、
     *  メモリー・アロケート・エラーならば EALLOC で戻ります。
     *  write() の返却値が SUCCESS ならば継続され、それ以外の値なら中断されます。
     *  最後の呼び出しの場合、これらは convertBuffer() の返却値となります。
     *  writer のメソッド write() が不要ならば null を設定します。<br>
     *  userInfo は observe(), write() に渡すユーザー情報です。不要ならば null を設定します。<br>
     *
     * @param buffer 入力バッファ
	 * @param length 入力サイズ
	 * @param stype 入力データの文字コード・タイプ
	 * @param dtype 出力データの文字コード・タイプ
	 * @param isForced 入力データの文字コード・タイプを強制的に stype とする
	 * @param observer コード判定結果の監視を行なうインターフェース
	 * @param writer コード変換データの出力を行なうインターフェース
	 * @param userInfo ユーザー情報
	 * @return 変換が成功した場合は SUCCESS、
	 * 判定終了時に observer のメソッド observe() を呼び出して中止した場合は ECANOBS、
	 * コード判定エラー（CONFIRMED_CODE_TYPE(sts)が偽）の時はその値を返します。
	 * メモリー・アロケート・エラーの時は EALLOC。
	 * @since 2.0
	 */
	public native int convertBuffer(byte[] buffer, int length, int stype, int dtype, boolean isForced, ConvertObserver observer, ConvertWriter writer, Object userInfo);

    /**
	 *  &nbsp;&nbsp;&nbsp;&nbsp;入力ストリーム istream のデータを指定された文字コードに変換します。<br>
	 *  &nbsp;&nbsp;&nbsp;&nbsp;istream は呼び出し側でクローズします。入力エラーの時は EREAD を返します。<br>
	 *  &nbsp;&nbsp;&nbsp;&nbsp;その他の機能は convertBuffer() と同じです。<br>
     *
	 * @param istream 入力ストリーム
	 * @param stype 入力データの文字コード・タイプ
	 * @param dtype 出力データの文字コード・タイプ
	 * @param isForced 入力データの文字コード・タイプを強制的に stype とする
	 * @param observer コード判定結果の監視を行なうインターフェース
	 * @param writer コード変換データの出力を行なうインターフェース
	 * @param userInfo ユーザー情報
	 * @return 変換が成功した場合は SUCCESS、
	 * 判定終了時に observer のメソッド observe() を呼び出して中止した場合は ECANOBS、
	 * コード判定エラー（CONFIRMED_CODE_TYPE(sts)が偽）の時はその値を返します。
	 * メモリー・アロケート・エラーの時は EALLOC、
	 * 入力エラーの時は EREAD。
	 * @since 2.0
	 */
	public native int convertStream(InputStream istream, int stype, int dtype, boolean isForced, ConvertObserver observer, ConvertWriter writer, Object userInfo);

    /**
     *  &nbsp;&nbsp;&nbsp;&nbsp;入力ファイル path のデータを指定された文字コードに変換します。<br>
     *  &nbsp;&nbsp;&nbsp;&nbsp;ファイル・オープン・エラーの時は EROPEN、入力エラーの時は EREAD を返します。<br>
     *  &nbsp;&nbsp;&nbsp;&nbsp;その他の機能は convertBuffer() と同じです。<br>
     *
	 * @param path 入力ファイル・パス
	 * @param stype 入力データの文字コード・タイプ
	 * @param dtype 出力データの文字コード・タイプ
	 * @param isForced 入力データの文字コード・タイプを強制的に stype とする
	 * @param observer コード判定結果の監視を行なうインターフェース
	 * @param writer コード変換データの出力を行なうインターフェース
	 * @param userInfo ユーザー情報
	 * @return 変換が成功した場合は SUCCESS、
	 * 判定終了時に observer のメソッド observe() を呼び出して中止した場合は ECANOBS、
	 * コード判定エラー（CONFIRMED_CODE_TYPE(sts)が偽）の時はその値を返します。
	 * メモリー・アロケート・エラーの時は EALLOC、
	 * ファイル・オープン・エラーの時は EROPEN、
	 * 入力エラーの時は EREAD。
	 * @since 2.0
	 */
	public int convertFile(String path, int stype, int dtype, boolean isForced, ConvertObserver observer, ConvertWriter writer, Object userInfo) {
		return convertFile(path.getBytes(), stype, dtype, isForced, observer, writer, userInfo);
	}

    /**
     *  &nbsp;&nbsp;&nbsp;&nbsp;入力ファイル path のデータを指定された文字コードに変換します。<br>
     *  &nbsp;&nbsp;&nbsp;&nbsp;ファイル・オープン・エラーの時は EROPEN、入力エラーの時は EREAD を返します。<br>
     *  &nbsp;&nbsp;&nbsp;&nbsp;その他の機能は convertBuffer() と同じです。<br>
     *
	 * @param apath 入力ファイル・パスのバイト配列
	 * @param stype 入力データの文字コード・タイプ
	 * @param dtype 出力データの文字コード・タイプ
	 * @param isForced 入力データの文字コード・タイプを強制的に stype とする
	 * @param observer コード判定結果の監視を行なうインターフェース
	 * @param writer コード変換データの出力を行なうインターフェース
	 * @param userInfo ユーザー情報
	 * @return 変換が成功した場合は SUCCESS、
	 * 判定終了時に observer のメソッド observe() を呼び出して中止した場合は ECANOBS、
	 * コード判定エラー（CONFIRMED_CODE_TYPE(sts)が偽）の時はその値を返します。
	 * メモリー・アロケート・エラーの時は EALLOC、
	 * ファイル・オープン・エラーの時は EROPEN、
	 * 入力エラーの時は EREAD。
	 * @since 2.0
	 */
	public native int convertFile(byte[] apath, int stype, int dtype, boolean isForced, ConvertObserver observer, ConvertWriter writer, Object userInfo);

    /**
     *  直前に行なったコード変換で、変換できなかった文字数を返します。
     *
	 * @return 変換ができなかった文字数を返します。
	 */
	public native int getNoCharCount();

    /**
     *  直前に行なったコード変換で、変換できなかった最初の文字を返します。
     *
	 * @return 変換ができなかった最初の文字を返します。すべて変換できた時はゼロ
	 */
	public native long getFirstNoChar();

    /**
     *  直前に行なったコード変換で、変換できなかった最後の文字を返します。
     *
	 * @return 変換ができなかった最後の文字を返します。すべて変換できた時はゼロ
	 */
	public native long getLastNoChar();

    /**
     *  直前に行なったコード変換で、変換できなかった最初のバイト位置を返します。
     *
	 * @return 変換ができなかった最初のバイト位置を返します。すべて変換できた時は -1
	 */
	public native int getFirstNoCharIndex();

    /**
     *  直前に行なったコード変換で、変換できなかった最後のバイト位置を返します。
     *
	 * @return 変換ができなかった最後のバイト位置を返します。すべて変換できた時は -1
	 */
	public native int getLastNoCharIndex();

    /**
     *  改行コード・タイプ取得
     *
	 * @return 設定されている改行コード・タイプを返します（LTYPE_NON, LTYPE_LF, LTYPE_CR, LTYPE_CRLF の何れか。初期値は LTYPE_NON であり、改行コードの編集を行わない）。
	 * @since 1.1
	 */
	public native int getLtype();

	/**
     *  改行コード・タイプ設定
     *
	 * @param ltype 設定する改行コード・タイプ（LTYPE_NON, LTYPE_LF, LTYPE_CR, LTYPE_CRLF の何れか）
	 * @since 1.1
	 */
	public native void setLtype(int ltype);

    /**
     *  出力コードの半角カナから全角への変換の判定。初期値は偽であり、半角カナから全角への変換を行いません。
     *
	 * @return 半角カナから全角への変換の真偽を返します。
	 * @since 3.0
	 */
	public native boolean isHankanaTo();

	/**
     *  半角カナから全角への変換の設定
     *
	 * @param flag 出力コードの半角カナから全角への変換の真偽を設定します。
	 * @since 3.0
	 */
	public native void setHankanaTo(boolean flag);

    /**
     *  日本語 UTF8-MAC １文字変換の判定。初期値は偽であり、日本語 UTF8-MACの１文字変換を行いません。
     *
	 * @return 日本語 UTF8-MACの１文字変換の真偽を返します。
	 * @since 2.1
	 */
	public native boolean isUtfmacTo();

	/**
     *  日本語 UTF8-MAC １文字変換の設定
     *
	 * @param flag 日本語 UTF8-MAC １文字変換の真偽を設定します。
	 * @since 2.1
	 */
	public native void setUtfmacTo(boolean flag);

    /**
     *  入力データの日本語 UTF8-MAC コード有無の判定
     *
	 * @return 入力データの日本語 UTF8-MAC コードの有無を返します。
	 * @since 2.1
	 */
	public native boolean utfmacExists();

    /**
     *  不正文字出力抑止の判定。
     *  convertBuffer() 等によるコード変換時に発生する不正文字抑止の真偽を判定します。
     *  初期値は偽であり、不正文字を出力します。
     *
	 * @return 設定されている不正文字出力抑止の真偽を返します。
	 * @since 2.0
	 */
	public native boolean isSuppress();

	/**
     *  不正文字出力抑止の設定。
     *  convertBuffer() 等によるコード変換時に発生する不正文字出力抑止の真偽を設定します。
     *  出力を抑止する時は真、それ以外は偽。
     *
	 * @param isSuppress 不正文字出力抑止の真偽を設定します。
	 * @since 2.0
	 */
	public native void setSuppress(boolean isSuppress);
}
