/*
 * DcsStatus.java
 *
 * Copyright 2019-2026 Ichiji Tadokoro. All Rights Reserved.
 */

package rabbit.dcs;

import java.util.Hashtable;

/**
 *
 * &nbsp;&nbsp;&nbsp;&nbsp;<font size=+1><code>DcsStatus</code></font> は文字コードのタイプと処理結果ステータスを定義したインターフェースです。<font size=¥+1><code>Dcs</code></font> で実装されています。
 *
 * @author Ichiji Tadokoro
 *
 */
public interface DcsStatus {

	/* コード・タイプ */

	/**
	 * コード判定の結果が不明であることを表します。
	 */
	public static final int UNKNOWN			= -1;

	/**
	 * 空データであることを表します。
	 */
	public static final int EMPTY			= 0;

	/**
	 * コードが ASCII であることを表します。
	 */
	public static final int ASCII			= 1;

	/**
	 * コードが JIS(ISO-2022-JP) であることを表します。
	 */
	public static final int JIS				= 2;

	/**
	 * コードが JIS(CP50221) であることを表します。
	 */
	public static final int JIS1				= 3;

	/**
	 * コードが JIS(ISO-2022-JP-2004) であることを表します。
	 */
	public static final int JIS2				= 4;

	/**
	 * コードが JIS(ISO-2022-JP-3) であることを表します。
	 */
	public static final int JIS3				= 5;

	/**
	 * コードがシフトJIS(Shift_JIS) であることを表します。
	 */
	public static final int SJIS				= 6;

	/**
	 * コードがシフトJIS(CP932) であることを表します。
	 */
	public static final int SJIS1			= 7;

	/**
	 * コードがシフトJIS(Shift_JIS-2004) であることを表します。
	 */
	public static final int SJIS2				= 8;

	/**
	 * コードが EUC であることを表します。
	 */
	public static final int EUC				= 9;

	/**
	 * コードが EUC-PC20932 であることを表します。
	 */
	public static final int EUC1				= 10;

	/**
	 * コードが EUC-PC51932 であることを表します。
	 */
	public static final int EUC2				= 11;

	/**
	 * コードが EUC-MS であることを表します。
	 */
	public static final int EUC3				= 12;

	/**
	 * コードが EUC-JIS-2004 であることを表します。
	 */
	public static final int EUC4				= 13;

	/**
	 * コードが UTF8（BOM 無し）であることを表します。
	 */
	public static final int UTF8				= 20;

	/**
	 * コードが UTF8（BOM 付き）であることを表します。
	 */
	public static final int UTF8_BOM			= 21;

	/**
	 * コードが UTF16-BE（BOM 無し）であることを表します。
	 */
	public static final int UTF16BE			= 22;

	/**
	 * コードが UTF16-BE（BOM 付き）であることを表します。
	 */
	public static final int UTF16BE_BOM	= 23;

	/**
	 * コードが UTF16-LE（BOM 無し）であることを表します。
	 */
	public static final int UTF16LE			= 24;

	/**
	 * コードが UTF16-LE（BOM 付き）であることを表します。
	 */
	public static final int UTF16LE_BOM	= 25;

	/**
	 * コードが UTF32-BE（BOM 無し）であることを表します。
	 */
	public static final int UTF32BE			= 26;

	/**
	 * コードが UTF32-BE（BOM 付き）であることを表します。
	 */
	public static final int UTF32BE_BOM	= 27;

	/**
	 * コードが UTF32-LE（BOM 無し）であることを表します。
	 */
	public static final int UTF32LE			= 28;

	/**
	 * コードが UTF32-LE（BOM 付き）であることを表します。
	 */
	public static final int UTF32LE_BOM	= 29;

	/**
	 * 確定コード・タイプの最大値です。
	 */
	public static final int NORMAL		= 000077;

	/**
	 * コードが ASCII らしいことを表します。
	 */
	public static final int ND_ASCII		= 000000000100;
	/**
	 * コードが JIS(ISO-2022-JP) らしいことを表します。
	 */
	public static final int ND_JIS		= 000000000200;
	/**
	 * コードが JIS(CP50221) らしいことを表します。
	 */
	public static final int ND_JIS1		= 000000000400;
	/**
	 * コードが JIS(ISO-2022-JP-2004) らしいことを表します。
	 */
	public static final int ND_JIS2		= 000000001000;
	/**
	 * コードが JIS(ISO-2022-JP-3) らしいことを表します。
	 */
	public static final int ND_JIS3		= 000000002000;
	/**
	 * コードがシフトJIS(Shift_JIS) らしいことを表します。
	 */
	public static final int ND_SJIS		= 000000004000;
	/**
	 * コードがシフトJIS(CP932) らしいことを表します。
	 */
	public static final int ND_SJIS1		= 000000010000;
	/**
	 * コードがシフトJIS(Shift_JIS-2004) らしいことを表します。
	 */
	public static final int ND_SJIS2		= 000000020000;
	/**
	 * コードが EUC らしいことを表します。
	 */
	public static final int ND_EUC		= 000000040000;
	/**
	 * コードが EUC-PC20932 らしいことを表します。
	 */
	public static final int ND_EUC1		= 000000100000;
	/**
	 * コードが EUC-PC51932 らしいことを表します。
	 */
	public static final int ND_EUC2		= 000000200000;
	/**
	 * コードが EUC-MS らしいことを表します。
	 */
	public static final int ND_EUC3		= 000000400000;
	/**
	 * コードが eucJP-2004 らしいことを表します。
	 */
	public static final int ND_EUC4		= 000001000000;
	/**
	 * コードが UTF8（BOM 無し）らしいことを表します。
	 */
	public static final int ND_UTF8		= 000010000000;
	/**
	 * コードが UTF8（BOM 付き）らしいことを表します。
	 */
	public static final int ND_UTF8_BOM		= 000020000000;
	/**
	 * コードが UTF16-BE（BOM 無し）らしいことを表します。
	 */
	public static final int ND_UTF16BE	= 000040000000;
	/**
	 * コードが UTF16-BE（BOM 付き）らしいことを表します。
	 */
	public static final int ND_UTF16BE_BOM	= 000100000000;
	/**
	 * コードが UTF16-LE（BOM 無し）らしいことを表します。
	 */
	public static final int ND_UTF16LE	= 000200000000;
	/**
	 * コードが UTF16-LE（BOM 付き）らしいことを表します。
	 */
	public static final int ND_UTF16LE_BOM	= 000400000000;
	/**
	 * コードが UTF32-BE（BOM 無し）らしいことを表します。
	 */
	public static final int ND_UTF32BE	= 001000000000;
	/**
	 * コードが UTF32-BE（BOM 付き）らしいことを表します。
	 */
	public static final int ND_UTF32BE_BOM	= 002000000000;
	/**
	 * コードが UTF32-LE（BOM 無し）らしいことを表します。
	 */
	public static final int ND_UTF32LE	= 004000000000;
	/**
	 * コードが UTF32-LE（BOM 付き）らしいことを表します。
	 */
	public static final int ND_UTF32LE_BOM	= 010000000000;
	/**
	 * 未確定コード・タイプの総和値です。
	 */
	public static final int ND_ALL		= 017777777700;

	/**
	 * 変換監視停止ステータス
	 */
	public static final int OBS_STOP			=	-1;
	/**
	 * 変換監視継続ステータス
	 */
	public static final int OBS_CONTINUE	=	0;

	/** 改行コードなしの値です。 */
	public static final int LTYPE_NON	= 000000000;
	/** 改行コード・タイプ(LF)です。 */
	public static final int LTYPE_LF		= 000000001; /* UNIX系 */
	/** 改行コード・タイプ(CR)です。 */
	public static final int LTYPE_CR		= 000000002; /* Mac OS9 */
	/** 改行コード・タイプ(CR/LF)です。 */
	public static final int LTYPE_CRLF	= 000000004; /* Win */
	/** 改行コード・タイプの総和値です。 */
	public static final int LTYPE_ALL	= 000000007;

	/** 変換不能コードの変換後（JIS以外）の代替え半角文字の値です。代替え文字は '?' */
	public static final int NO_CHAR		= '?';

	/** 変換不能コードの変換後（JIS）の代替え全角文字第１バイトの値です。代替え文字は '？' */
	public static final int NO_CHAR1		= 0x21;
	/** 変換不能コードの変換後（JIS）の代替え全角文字第２バイトの値です。代替え文字は '？' */
	public static final int NO_CHAR2		= 0x29;
	/** 変換不能コードの変換後（JIS）の代替え全角文字に対する UTF-16BE の値です。 */
	public static final int JIS_NO_CHAR		= 0xff1f;

	/** UTF-16BE BOM 第１バイト */
	public static final int UTF16_B0			= 0xfe;
	/** UTF-16BE BOM 第２バイト */
	public static final int UTF16_B1			= 0xff;

	/** UTF-8 BOM 第１バイト */
	public static final int UTF8_B0			= 0xef;
	/** UTF-8 BOM 第２バイト */
	public static final int UTF8_B1			= 0xbb;
	/** UTF-8 BOM 第３バイト */
	public static final int UTF8_B2			= 0xbf;

	/** Unicode 置換え文字 */
	public static final int REP_CHAR			= 0xFFFD;
	/* 全角仮名領域 */
	/** JIS全角平仮名開始コード */
	public static final int JIS_HKANA_B = 0x2421;
	/** JIS全角平仮名終了コード */
	public static final int JIS_HKANA_E = 0x2473;
	/** JIS全角片仮名開始コード */
	public static final int JIS_KKANA_B = 0x2521;
	/** JIS全角片仮名終了コード */
	public static final int JIS_KKANA_E = 0x2573;

	/** シフトJIS全角平仮名開始コード */
	public static final int SJIS_HKANA_B = 0x829f;
	/** シフトJIS全角平仮名終了コード */
	public static final int SJIS_HKANA_E = 0x82f1;
	/** シフトJIS全角片仮名開始コード */
	public static final int SJIS_KKANA_B = 0x8340;
	/** シフトJIS全角片仮名終了コード */
	public static final int SJIS_KKANA_E = 0x8393;

	/** EUC全角平仮名開始コード */
	public static final int EUC_HKANA_B = 0xa4a1;
	/** EUC全角平仮名終了コード */
	public static final int EUC_HKANA_E = 0xa4f3;
	/** EUC全角片仮名開始コード */
	public static final int EUC_KKANA_B = 0xa5a1;
	/** EUC全角片仮名終了コード */
	public static final int EUC_KKANA_E = 0xa5f3;

	/** UTF-16BE全角平仮名開始コード */
	public static final int UNIBE_HKANA_B = 0x3041;
	/** UTF-16BE全角平仮名終了コード */
	public static final int UNIBE_HKANA_E = 0x3093;
	/** UTF-16BE全角片仮名開始コード */
	public static final int UNIBE_KKANA_B = 0x30a1;
	/** UTF-16BE全角片仮名終了コード */
	public static final int UNIBE_KKANA_E = 0x30f3;

	/* BOM 追加／削除判定 */
	/** BOM 追加削除なし */
	public static final int CBOM_NON			= 0;
	/** BOM 追加 */
	public static final int CBOM_ADD			= 1;
	/** BOM 削除 */
	public static final int CBOM_DELETE		= -1;

	class TypeNameInfo {
		int type;
		String name;
		TypeNameInfo(int type, String name) {
			this.type = type;
			this.name = name;
		}
	}

	static final String TN_UNKNOWN = "Unknown";
	static final String TN_EMPTY = "Empty";
	static final String TN_ASCII = "ASCII";
	static final String TN_JIS = "JIS(ISO-2022-JP)";
	static final String TN_JIS1 = "JIS(CP50221)";
	static final String TN_JIS2 = "JIS(ISO-2022-JP-2004)";
	static final String TN_JIS3 = "JIS(ISO-2022-JP-3)";
	static final String TN_SJIS = "SJIS(CP932)";
	static final String TN_SJIS2 = "SJIS(Shift_JIS-2004)";
	static final String TN_EUC = "EUC-JP";
	static final String TN_EUC1 = "EUC1(PC20932)";
	static final String TN_EUC2 = "EUC2(PC51932)";
	static final String TN_EUC3 = "EUC3(EUC-MS)";
	static final String TN_EUC4 = "EUC4(EUC-JIS-2004)";
	static final String TN_UTF8 = "UTF8";
	static final String TN_UTF8_BOM = "UTF8_BOM";
	static final String TN_UTF16BE = "UTF16BE";
	static final String TN_UTF16BE_BOM = "UTF16BE_BOM";
	static final String TN_UTF16LE = "UTF16LE";
	static final String TN_UTF16LE_BOM = "UTF16LE_BOM";
	static final String TN_UTF32BE = "UTF32BE";
	static final String TN_UTF32BE_BOM = "UTF32BE_BOM";
	static final String TN_UTF32LE = "UTF32LE";
	static final String TN_UTF32LE_BOM = "UTF32LE_BOM";

static final Hashtable<String, Integer>[] _hNameToType = new Hashtable[1];

static final Hashtable<String, Integer>[] _hCharsetNameToType = new Hashtable[1];

public static void _createNameToType() {
	if (_hNameToType[0] != null) {
		return;
	}
	_hNameToType[0] = new Hashtable();
	TypeNameInfo[] info = {
		new TypeNameInfo(ASCII, TN_ASCII),
		new TypeNameInfo(JIS, TN_JIS),
		new TypeNameInfo(JIS1, TN_JIS1),
		new TypeNameInfo(JIS2, TN_JIS2),
		new TypeNameInfo(JIS3, TN_JIS3),
		new TypeNameInfo(SJIS, TN_SJIS),
		new TypeNameInfo(SJIS2, TN_SJIS2),
		new TypeNameInfo(EUC, TN_EUC),
		new TypeNameInfo(EUC1, TN_EUC1),
		new TypeNameInfo(EUC2, TN_EUC2),
		new TypeNameInfo(EUC3, TN_EUC3),
		new TypeNameInfo(EUC4, TN_EUC4),
		new TypeNameInfo(UTF8, TN_UTF8),
		new TypeNameInfo(UTF16BE, TN_UTF16BE),
		new TypeNameInfo(UTF16LE, TN_UTF16LE),
		new TypeNameInfo(UTF32BE, TN_UTF32BE),
		new TypeNameInfo(UTF32LE, TN_UTF32LE),
	};
	int i, len = info.length;
	for(i=0;i<len;i++) {
		_hNameToType[0].put(info[i].name, info[i].type);
	}
}

/**
 *  コード名をコード・タイプに変換します。
 *
 * @param name 変換対象のコード名
 * @return 変換したコード・タイプ
 */
	public static int nameToType(String name) {
		if (_hNameToType[0] == null) {
			_createNameToType();
		}
		Integer type = _hNameToType[0].get(name);
		return(type == null? UNKNOWN: type);
	}

	public static void _createCharsetNameToType() {
		if (_hCharsetNameToType[0] != null) {
			return;
		}
		_hCharsetNameToType[0] = new Hashtable();
		TypeNameInfo[] info = {
			new TypeNameInfo(ASCII, TN_ASCII),
			new TypeNameInfo(JIS, TN_JIS),
			new TypeNameInfo(JIS1, TN_JIS1),
			new TypeNameInfo(JIS2, TN_JIS2),
			new TypeNameInfo(JIS3, TN_JIS3),
			new TypeNameInfo(SJIS, TN_SJIS),
			new TypeNameInfo(SJIS2, TN_SJIS2),
			new TypeNameInfo(EUC, TN_EUC),
			new TypeNameInfo(EUC1, TN_EUC1),
			new TypeNameInfo(EUC2, TN_EUC2),
			new TypeNameInfo(EUC3, TN_EUC3),
			new TypeNameInfo(EUC4, TN_EUC4),
			new TypeNameInfo(UTF8, TN_UTF8),
			new TypeNameInfo(UTF16BE, TN_UTF16BE),
			new TypeNameInfo(UTF16LE, TN_UTF16LE),
			new TypeNameInfo(UTF32BE, TN_UTF32BE),
			new TypeNameInfo(UTF32LE, TN_UTF32LE),
		};
		int i, len = info.length;
		for(i=0;i<len;i++) {
			_hCharsetNameToType[0].put(info[i].name, info[i].type);
		}
	}

	/**
	 *  キャラクター・セット名をコード・タイプに変換します。
	 *
	 * @param name 変換対象のキャラクター・セット名
	 * @return 変換したコード・タイプ
	 */
	public static int charsetNameToType(String name) {
		if (_hCharsetNameToType[0] == null) {
			_createCharsetNameToType();
		}
		Integer type = _hCharsetNameToType[0].get(name);
		return(type == null? UNKNOWN: type);
	}

	/**
	 *  コード・タイプをコード名に変換します。
	 *
	 * @param type 変換対象のコード・タイプ
	 * @return 変換したコード名
	 */
	public static String typeToName(int type) {
		switch(type) {
		case EMPTY:
			return TN_EMPTY;
		case ASCII:
			return TN_ASCII;
		case JIS:
	   return TN_JIS;
		case JIS1:
			return TN_JIS1;
		case JIS2:
			return TN_JIS2;
		case JIS3:
			return TN_JIS3;
		case SJIS:
			return TN_SJIS;
		case SJIS2:
			return TN_SJIS2;
		case EUC:
			return TN_EUC;
		case EUC1:
			return TN_EUC1;
		case EUC2:
			return TN_EUC2;
		case EUC3:
			return TN_EUC3;
		case EUC4:
			return TN_EUC4;
		case UTF8:
			return TN_UTF8;
		case UTF8_BOM:
			return TN_UTF8_BOM;
		case UTF16BE:
			return TN_UTF16BE;
		case UTF16BE_BOM:
			return TN_UTF16BE_BOM;
		case UTF16LE:
			return TN_UTF16LE;
		case UTF16LE_BOM:
			return TN_UTF16LE_BOM;
		case UTF32BE:
			return TN_UTF32BE;
		case UTF32BE_BOM:
			return TN_UTF32BE_BOM;
		case UTF32LE:
			return TN_UTF32LE;
		case UTF32LE_BOM:
			return TN_UTF32LE_BOM;
		default:
			if (type > NORMAL) {
			// 複合の時
				TypeNameInfo[] info = {
					new TypeNameInfo(ND_ASCII, TN_ASCII),
					new TypeNameInfo(ND_JIS, TN_JIS),
					new TypeNameInfo(ND_JIS1, TN_JIS1),
					new TypeNameInfo(ND_JIS2, TN_JIS2),
					new TypeNameInfo(ND_JIS3, TN_JIS3),
					new TypeNameInfo(ND_SJIS, TN_SJIS),
					new TypeNameInfo(ND_SJIS2, TN_SJIS2),
					new TypeNameInfo(ND_EUC, TN_EUC),
					new TypeNameInfo(ND_EUC1, TN_EUC1),
					new TypeNameInfo(ND_EUC2, TN_EUC2),
					new TypeNameInfo(ND_EUC3, TN_EUC3),
					new TypeNameInfo(ND_EUC4, TN_EUC4),
					new TypeNameInfo(ND_UTF8, TN_UTF8),
					new TypeNameInfo(ND_UTF16BE, TN_UTF16BE),
					new TypeNameInfo(ND_UTF16LE, TN_UTF16LE),
					new TypeNameInfo(ND_UTF32BE, TN_UTF32BE),
					new TypeNameInfo(ND_UTF32LE, TN_UTF32LE),
				};
				StringBuffer sb = new StringBuffer();
				int i, len = info.length;
				for(i=0;i<len;i++) {
					if ((type & info[i].type) != 0) {
						sb.append(info[i].name).append(',');
					}
				}
				sb.setLength(sb.length() - 1);
				return sb.toString();
			}
			return TN_UNKNOWN;
		}
	}

	/**
	 *  コード・タイプをキャラクター・セット名に変換します。
	 *
	 * @param type 変換対象のコード・タイプ
	 * @return 変換したキャラクター・セット名
	 */
	public static String typeToCharsetName(int type) {
		switch(type) {
		case ASCII:
			return "ASCII";
		case JIS:
	   return "ISO-2022-JP";
		case JIS1:
	   return "CP50221";
		case JIS2:
	   return "ISO-2022-JP-2004";
		case JIS3:
	   return "ISO-2022-JP-3";
		case SJIS:
	   return "CP932";
		case SJIS2:
			return "Shift_JIS-2004";
		case EUC:
	   return "EUC-JP";
		case EUC1:
	   return "PC20932";
		case EUC2:
	   return "PC51932";
		case EUC3:
			return "eucJP-ms";
		case EUC4:
			return "eucJP-2004";
		case UTF8:
		case UTF8_BOM:
			return "UTF8";
		case UTF16BE:
		case UTF16BE_BOM:
			return "UTF16-BE";
		case UTF16LE:
		case UTF16LE_BOM:
			return "UTF16-LE";
		case UTF32BE:
		case UTF32BE_BOM:
			return "UTF32-BE";
		case UTF32LE:
		case UTF32LE_BOM:
			return "UTF32-LE";
		default:
			return "Unknown";
		}
	}

	/**
	 *  コード・タイプを DCS のコード・タイプ名に変換します。
	 *
	 * @param type 変換対象のコード・タイプ
	 * @return 変換したコード・タイプ名
	 */
	public static String typeToTypeName(int type) {
		switch(type) {
		case ASCII:
			return "ascii";
		case JIS:
	   return "j";
		case JIS1:
	   return "j1";
		case JIS2:
	   return "j2";
		case JIS3:
	   return "j3";
		case SJIS:
			return "s";
		case SJIS2:
			return "s2";
		case EUC:
			return "e";
		case EUC1:
			return "e1";
		case EUC2:
			return "e2";
		case EUC3:
			return "e3";
		case EUC4:
			return "e4";
		case UTF8:
			return "u8";
		case UTF8_BOM:
			return "u8m";
		case UTF16BE:
			return "ub";
		case UTF16BE_BOM:
			return "ubm";
		case UTF16LE:
			return "ul";
		case UTF16LE_BOM:
			return "ulm";
		case UTF32BE:
			return "u32b";
		case UTF32BE_BOM:
			return "u32bm";
		case UTF32LE:
			return "u32l";
		case UTF32LE_BOM:
			return "u32lm";
		default:
			return "Unknown";
		}
	}

	/* ステータス */

	/**
	 * 処理成功ステータス。
	 */
	public static final int SUCCESS			= 0;

	/**
	 * データの終わりに達しことを表します。
	 */
	public static final int EOF				= -1;

	/**
	 * 対象のファイルが存在しないことを表します。
	 */
	public static final int ENOTEXISTS		= -2;

	/**
	 * 対象のファイルが通常ファイルではないことを表します。
	 */
	public static final int ENOTFILE		= -3;

	/**
	 * メモリー・アロケート・エラーを表します。
	 */
	public static final int EALLOC			= -4;

	/**
	 * 入力ファイルのオープン・エラーを表します。
	 */
	public static final int EROPEN			= -5;

	/**
	 * 出力ファイルのオープン・エラーを表します。
	 */
	public static final int EWOPEN			= -6;

	/**
	 * 入力ファイルの読み込みエラーを表します。
	 */
	public static final int EREAD			= -7;

	/**
	 * 出力ファイルの書き出しエラーを表します。
	 */
	public static final int EWRITE			= -8;

	/**
	 * 入出力モード誤り（未使用）
	 */
	public static final int EIOMODE		= -10;
	/**
	 * ファイル記述子範囲誤り（未使用）
	 */
	public static final int EIOFDRANGE		= -11;
	/**
	 * 未オープン・ファイルへのアクセス（未使用）
	 */
	public static final int EIONOTOPEN		= -12;
	/**
	 * 上限を超えたファイル・オープン（未使用）
	 */
	public static final int EIOMAX			= -13;

	/**
	 * 文字コード・タイプの値が不正であることを表します。
	 */
	public static final int ECTYPE		= -20;

	/**
	 * BOM の値が不正であることを表します。
	 */
	public static final int EBOM			= -21;

	/**
	 * 予期せぬデータの終わりを表します。
	 */
	public static final int EUEXPEOD		= -22;

	/**
	 * データに不正文字があることを表します。
	 */
	public static final int ECHAR			= -23;

	/**
	 * 改行コード・タイプの値が不正であることを表します。
	 */
	public static final int ELTYPE			= -24;

	/**
	 * 出力バッファが・フルであることを表します。
	 */
	public static final int EFULL			= -25;
}

